// Package core is generated by github.com/omeid/go-resources
package core

import (
	"bytes"
	"errors"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"time"
)

// FileSystem is an http.FileSystem implementation.
type FileSystem struct {
	files map[string]File
}

// String returns the content of the file as string.
func (fs *FileSystem) String(name string) (string, bool) {
	if filepath.Separator != '/' && strings.IndexRune(name, filepath.Separator) >= 0 ||
		strings.Contains(name, "\x00") {
		return "", false
	}

	file, ok := fs.files[name]

	if !ok {
		return "", false
	}

	return string(file.data), true
}

// Open implements http.FileSystem.Open
func (fs *FileSystem) Open(name string) (http.File, error) {
	if filepath.Separator != '/' && strings.IndexRune(name, filepath.Separator) >= 0 ||
		strings.Contains(name, "\x00") {
		return nil, errors.New("http: invalid character in file path")
	}
	file, ok := fs.files[name]
	if !ok {
		files := []os.FileInfo{}
		for path, file := range fs.files {
			if strings.HasPrefix(path, name) {
				fi := file.fi
				files = append(files, &fi)
			}
		}

		if len(files) == 0 {
			return nil, os.ErrNotExist
		}

		//We have a directory.
		return &File{
			fi: FileInfo{
				isDir: true,
				files: files,
			}}, nil
	}
	file.Reader = bytes.NewReader(file.data)
	return &file, nil
}

// File implements http.File
type File struct {
	*bytes.Reader
	data []byte
	fi   FileInfo
}

// Close is a noop-closer.
func (f *File) Close() error {
	return nil
}

// Readdir implements http.File.Readdir
func (f *File) Readdir(count int) ([]os.FileInfo, error) {
	return nil, os.ErrNotExist
}

// Stat implements http.Stat.Readdir
func (f *File) Stat() (os.FileInfo, error) {
	return &f.fi, nil
}

// FileInfo implements the os.FileInfo interface.
type FileInfo struct {
	name    string
	size    int64
	mode    os.FileMode
	modTime time.Time
	isDir   bool
	sys     interface{}

	files []os.FileInfo
}

// Name implements os.FileInfo.Name
func (f *FileInfo) Name() string {
	return f.name
}

// Size implements os.FileInfo.Size
func (f *FileInfo) Size() int64 {
	return f.size
}

// Mode implements os.FileInfo.Mode
func (f *FileInfo) Mode() os.FileMode {
	return f.mode
}

// ModTime implements os.FileInfo.ModTime
func (f *FileInfo) ModTime() time.Time {
	return f.modTime
}

// IsDir implements os.FileInfo.IsDir
func (f *FileInfo) IsDir() bool {
	return f.isDir
}

// Readdir implements os.FileInfo.Readdir
func (f *FileInfo) Readdir(count int) ([]os.FileInfo, error) {
	return f.files, nil
}

// Sys returns the underlying value.
func (f *FileInfo) Sys() interface{} {
	return f.sys
}

var FS *FileSystem

func init() {
	FS = &FileSystem{
		files: map[string]File{
			"/tmp/LICENSES": {
				data: []byte{
					0x4f, 0x50, 0x45, 0x4e, 0x20, 0x53, 0x4f, 0x55, 0x52, 0x43, 0x45, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x53, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e,
					0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76, 0x65, 0x72, 0x0a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d, 0x69,
					0x6e, 0x64, 0x73, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34, 0x2d, 0x32,
					0x30, 0x31, 0x35, 0x2c, 0x20, 0x4d, 0x61, 0x74, 0x74, 0x20, 0x42, 0x75,
					0x74, 0x63, 0x68, 0x65, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x4d, 0x61,
					0x74, 0x74, 0x20, 0x46, 0x61, 0x72, 0x69, 0x6e, 0x61, 0x0a, 0x0a, 0x50,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73,
					0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e,
					0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66,
					0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f,
					0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65,
					0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65,
					0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c,
					0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c,
					0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20,
					0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f,
					0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x69, 0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73,
					0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a,
					0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64,
					0x20, 0x69, 0x6e, 0x0a, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69,
					0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61,
					0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53,
					0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41,
					0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f,
					0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c,
					0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a,
					0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43,
					0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43,
					0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41,
					0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47,
					0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c,
					0x20, 0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44,
					0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49,
					0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20,
					0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c,
					0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53,
					0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55,
					0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43,
					0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49,
					0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55,
					0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x0a,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72,
					0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76, 0x65, 0x72,
					0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74,
					0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x70, 0x72, 0x69,
					0x67, 0x0a, 0x0a, 0x0a, 0x53, 0x70, 0x72, 0x69, 0x67, 0x0a, 0x43, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20,
					0x32, 0x30, 0x31, 0x33, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d,
					0x69, 0x6e, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65,
					0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20,
					0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72,
					0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70,
					0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e,
					0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f,
					0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d,
					0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f,
					0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20,
					0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20,
					0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a,
					0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f,
					0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75,
					0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64,
					0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63,
					0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x0a, 0x61,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72,
					0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c,
					0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56,
					0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52,
					0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49,
					0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49,
					0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e,
					0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41,
					0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52,
					0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e,
					0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54,
					0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a,
					0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c,
					0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42,
					0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43,
					0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48,
					0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20,
					0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f,
					0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53,
					0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46,
					0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20,
					0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f,
					0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52,
					0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49,
					0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e,
					0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73,
					0x2f, 0x73, 0x70, 0x72, 0x69, 0x67, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47,
					0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c, 0x69, 0x2f, 0x67, 0x6f, 0x75,
					0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x56,
					0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30, 0x2c, 0x20,
					0x4a, 0x61, 0x6e, 0x75, 0x61, 0x72, 0x79, 0x20, 0x32, 0x30, 0x30, 0x34,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e,
					0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x6c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x2f, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x54, 0x45, 0x52, 0x4d, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43,
					0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x52, 0x45, 0x50, 0x52, 0x4f,
					0x44, 0x55, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x2c, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x44, 0x49, 0x53, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x49, 0x4f,
					0x4e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x31, 0x2e, 0x20, 0x44, 0x65, 0x66,
					0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20,
					0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x61, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x65, 0x64, 0x20,
					0x62, 0x79, 0x20, 0x53, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x31, 0x20, 0x74, 0x68, 0x72, 0x6f, 0x75, 0x67, 0x68, 0x20, 0x39, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x64, 0x6f, 0x63, 0x75,
					0x6d, 0x65, 0x6e, 0x74, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x22, 0x20,
					0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x6f, 0x72, 0x20, 0x65, 0x6e,
					0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69,
					0x7a, 0x65, 0x64, 0x20, 0x62, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x74, 0x68, 0x61,
					0x74, 0x20, 0x69, 0x73, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x69, 0x6e,
					0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x4c,
					0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x22,
					0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x75, 0x6e, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x63, 0x74, 0x69, 0x6e, 0x67, 0x20,
					0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x74, 0x68,
					0x65, 0x72, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x69, 0x65, 0x73, 0x20,
					0x74, 0x68, 0x61, 0x74, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c,
					0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f,
					0x6c, 0x6c, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72, 0x20,
					0x61, 0x72, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x63, 0x6f,
					0x6d, 0x6d, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20,
					0x74, 0x68, 0x61, 0x74, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2e,
					0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72,
					0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x63, 0x6f, 0x6e,
					0x74, 0x72, 0x6f, 0x6c, 0x22, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x73, 0x20,
					0x28, 0x69, 0x29, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f, 0x77, 0x65,
					0x72, 0x2c, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x20, 0x6f, 0x72,
					0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6e, 0x61, 0x67,
					0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x75, 0x63,
					0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2c, 0x20, 0x77, 0x68,
					0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63, 0x6f, 0x6e,
					0x74, 0x72, 0x61, 0x63, 0x74, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65,
					0x2c, 0x20, 0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x29, 0x20, 0x6f, 0x77,
					0x6e, 0x65, 0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66, 0x20, 0x66,
					0x69, 0x66, 0x74, 0x79, 0x20, 0x70, 0x65, 0x72, 0x63, 0x65, 0x6e, 0x74,
					0x20, 0x28, 0x35, 0x30, 0x25, 0x29, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x6f,
					0x72, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x73, 0x68, 0x61, 0x72, 0x65, 0x73, 0x2c, 0x20,
					0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x69, 0x29, 0x20, 0x62, 0x65, 0x6e,
					0x65, 0x66, 0x69, 0x63, 0x69, 0x61, 0x6c, 0x20, 0x6f, 0x77, 0x6e, 0x65,
					0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x75, 0x63,
					0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x22, 0x20, 0x28,
					0x6f, 0x72, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x72, 0x22, 0x29, 0x20, 0x73,
					0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x61, 0x6e,
					0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c, 0x20,
					0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74,
					0x69, 0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x78,
					0x65, 0x72, 0x63, 0x69, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x67, 0x72, 0x61,
					0x6e, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65,
					0x22, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x72,
					0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x20, 0x66, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6b, 0x69, 0x6e, 0x67, 0x20,
					0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x73,
					0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x2c, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x66,
					0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x73, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x22, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x22, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x72, 0x65,
					0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x72, 0x6f, 0x6d,
					0x20, 0x6d, 0x65, 0x63, 0x68, 0x61, 0x6e, 0x69, 0x63, 0x61, 0x6c, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x66,
					0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x72, 0x20,
					0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x6f, 0x66, 0x20, 0x61, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x69, 0x6c, 0x65,
					0x64, 0x20, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x63, 0x6f, 0x64,
					0x65, 0x2c, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x74, 0x65, 0x64,
					0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x63, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d,
					0x65, 0x64, 0x69, 0x61, 0x20, 0x74, 0x79, 0x70, 0x65, 0x73, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x57, 0x6f, 0x72, 0x6b,
					0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x66,
					0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x68, 0x69, 0x70, 0x2c,
					0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20,
					0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x66,
					0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x6d, 0x61, 0x64, 0x65, 0x20, 0x61, 0x76,
					0x61, 0x69, 0x6c, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x61, 0x73, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x63, 0x61,
					0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x69, 0x73, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64,
					0x20, 0x69, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x74, 0x74, 0x61, 0x63,
					0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x77,
					0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x61,
					0x6e, 0x20, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x20, 0x69, 0x73,
					0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x70, 0x65, 0x6e, 0x64, 0x69,
					0x78, 0x20, 0x62, 0x65, 0x6c, 0x6f, 0x77, 0x29, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x22, 0x20,
					0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x77, 0x68, 0x65,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x2c,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x62, 0x61, 0x73,
					0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x64, 0x65,
					0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x29, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x66, 0x6f, 0x72, 0x20, 0x77, 0x68, 0x69, 0x63, 0x68, 0x20, 0x74,
					0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x64, 0x69,
					0x74, 0x6f, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x72, 0x65, 0x76, 0x69, 0x73,
					0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x61, 0x6e, 0x6e, 0x6f, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x65, 0x6c, 0x61, 0x62, 0x6f,
					0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x20,
					0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69,
					0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x2c,
					0x20, 0x61, 0x73, 0x20, 0x61, 0x20, 0x77, 0x68, 0x6f, 0x6c, 0x65, 0x2c,
					0x20, 0x61, 0x6e, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c,
					0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x75, 0x74,
					0x68, 0x6f, 0x72, 0x73, 0x68, 0x69, 0x70, 0x2e, 0x20, 0x46, 0x6f, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65,
					0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c,
					0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x20,
					0x77, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x72,
					0x65, 0x6d, 0x61, 0x69, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x73, 0x65, 0x70, 0x61, 0x72, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x66, 0x72,
					0x6f, 0x6d, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x65, 0x72, 0x65, 0x6c,
					0x79, 0x20, 0x6c, 0x69, 0x6e, 0x6b, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x62,
					0x69, 0x6e, 0x64, 0x20, 0x62, 0x79, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x29,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x69, 0x6e, 0x74, 0x65,
					0x72, 0x66, 0x61, 0x63, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74,
					0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x6d, 0x65, 0x61, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x6f, 0x72,
					0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73,
					0x68, 0x69, 0x70, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69,
					0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20, 0x76, 0x65,
					0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74,
					0x6f, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20,
					0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x65, 0x72,
					0x65, 0x6f, 0x66, 0x2c, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73,
					0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c,
					0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x75, 0x62,
					0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x62, 0x79, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x61, 0x6e, 0x20, 0x69,
					0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c, 0x20, 0x6f, 0x72,
					0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74, 0x69, 0x74,
					0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x20, 0x6f,
					0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f, 0x66, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65,
					0x72, 0x2e, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70,
					0x75, 0x72, 0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x22, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x74,
					0x65, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x65,
					0x61, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x20, 0x6f, 0x66, 0x20, 0x65, 0x6c, 0x65, 0x63, 0x74, 0x72, 0x6f, 0x6e,
					0x69, 0x63, 0x2c, 0x20, 0x76, 0x65, 0x72, 0x62, 0x61, 0x6c, 0x2c, 0x20,
					0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e, 0x20, 0x63,
					0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x73, 0x65, 0x6e, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x6f, 0x72, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x74, 0x73, 0x20, 0x72,
					0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e,
					0x67, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69,
					0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x6e, 0x20, 0x65, 0x6c, 0x65, 0x63,
					0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63, 0x20, 0x6d, 0x61, 0x69, 0x6c, 0x69,
					0x6e, 0x67, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x73, 0x2c, 0x20, 0x73, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x63, 0x6f,
					0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
					0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x20, 0x74, 0x72, 0x61, 0x63, 0x6b,
					0x69, 0x6e, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x73, 0x20,
					0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d, 0x61, 0x6e,
					0x61, 0x67, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72, 0x20,
					0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f, 0x66,
					0x2c, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65,
					0x20, 0x6f, 0x66, 0x20, 0x64, 0x69, 0x73, 0x63, 0x75, 0x73, 0x73, 0x69,
					0x6e, 0x67, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x69, 0x6d, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x2c, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x63, 0x6f, 0x6e,
					0x73, 0x70, 0x69, 0x63, 0x75, 0x6f, 0x75, 0x73, 0x6c, 0x79, 0x20, 0x6d,
					0x61, 0x72, 0x6b, 0x65, 0x64, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68,
					0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x64, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x62,
					0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x61, 0x73,
					0x20, 0x22, 0x4e, 0x6f, 0x74, 0x20, 0x61, 0x20, 0x43, 0x6f, 0x6e, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x22, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c, 0x20, 0x6f,
					0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74, 0x69,
					0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x6e, 0x20,
					0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f, 0x66, 0x20, 0x77, 0x68,
					0x6f, 0x6d, 0x20, 0x61, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x68, 0x61, 0x73, 0x20, 0x62, 0x65,
					0x65, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76, 0x65, 0x64, 0x20,
					0x62, 0x79, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20,
					0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x75,
					0x62, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x6c, 0x79, 0x20, 0x69,
					0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65, 0x64, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x32, 0x2e, 0x20,
					0x47, 0x72, 0x61, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x43, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2e, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68, 0x65, 0x72, 0x65,
					0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x73, 0x20, 0x74, 0x6f,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x20, 0x70, 0x65, 0x72, 0x70, 0x65,
					0x74, 0x75, 0x61, 0x6c, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x77, 0x6f, 0x72, 0x6c, 0x64, 0x77, 0x69, 0x64, 0x65, 0x2c, 0x20, 0x6e,
					0x6f, 0x6e, 0x2d, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x76, 0x65,
					0x2c, 0x20, 0x6e, 0x6f, 0x2d, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c,
					0x20, 0x72, 0x6f, 0x79, 0x61, 0x6c, 0x74, 0x79, 0x2d, 0x66, 0x72, 0x65,
					0x65, 0x2c, 0x20, 0x69, 0x72, 0x72, 0x65, 0x76, 0x6f, 0x63, 0x61, 0x62,
					0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64,
					0x75, 0x63, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x65, 0x70, 0x61, 0x72, 0x65,
					0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x6f, 0x66, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x6c, 0x79,
					0x20, 0x64, 0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x2c, 0x20, 0x70, 0x75,
					0x62, 0x6c, 0x69, 0x63, 0x6c, 0x79, 0x20, 0x70, 0x65, 0x72, 0x66, 0x6f,
					0x72, 0x6d, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76,
					0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20,
					0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x6f, 0x72,
					0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x33, 0x2e, 0x20, 0x47, 0x72, 0x61,
					0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x50, 0x61, 0x74, 0x65, 0x6e, 0x74,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20, 0x53, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x65, 0x61, 0x63,
					0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f,
					0x72, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61,
					0x6e, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61,
					0x20, 0x70, 0x65, 0x72, 0x70, 0x65, 0x74, 0x75, 0x61, 0x6c, 0x2c, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x6f, 0x72, 0x6c, 0x64, 0x77,
					0x69, 0x64, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x65, 0x78, 0x63,
					0x6c, 0x75, 0x73, 0x69, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x2d, 0x63,
					0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x72, 0x6f, 0x79, 0x61, 0x6c,
					0x74, 0x79, 0x2d, 0x66, 0x72, 0x65, 0x65, 0x2c, 0x20, 0x69, 0x72, 0x72,
					0x65, 0x76, 0x6f, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x28, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x61,
					0x73, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e,
					0x29, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x6b, 0x65,
					0x2c, 0x20, 0x68, 0x61, 0x76, 0x65, 0x20, 0x6d, 0x61, 0x64, 0x65, 0x2c,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20,
					0x6f, 0x66, 0x66, 0x65, 0x72, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x65, 0x6c,
					0x6c, 0x2c, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x2c, 0x20, 0x69, 0x6d, 0x70,
					0x6f, 0x72, 0x74, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f, 0x74, 0x68,
					0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x20, 0x74, 0x72, 0x61, 0x6e, 0x73,
					0x66, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68, 0x65, 0x72,
					0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x6e, 0x6c, 0x79, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65,
					0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x63, 0x6c, 0x61, 0x69,
					0x6d, 0x73, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x61, 0x62, 0x6c,
					0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x79, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x72, 0x65,
					0x20, 0x6e, 0x65, 0x63, 0x65, 0x73, 0x73, 0x61, 0x72, 0x69, 0x6c, 0x79,
					0x20, 0x69, 0x6e, 0x66, 0x72, 0x69, 0x6e, 0x67, 0x65, 0x64, 0x20, 0x62,
					0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63, 0x6f, 0x6d, 0x62, 0x69, 0x6e,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x69, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x69, 0x63, 0x68, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x77, 0x61, 0x73,
					0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x2e, 0x20,
					0x49, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x69, 0x6e, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x65, 0x20, 0x70,
					0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x74, 0x69, 0x67, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e, 0x73, 0x74,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x20,
					0x28, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x61,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x72, 0x6f, 0x73, 0x73,
					0x2d, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f,
					0x75, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x20, 0x6c, 0x61, 0x77, 0x73, 0x75, 0x69, 0x74, 0x29,
					0x20, 0x61, 0x6c, 0x6c, 0x65, 0x67, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68,
					0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x20, 0x43,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65, 0x64,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x74, 0x69, 0x74,
					0x75, 0x74, 0x65, 0x73, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x79, 0x20, 0x70, 0x61,
					0x74, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e, 0x66, 0x72, 0x69, 0x6e, 0x67,
					0x65, 0x6d, 0x65, 0x6e, 0x74, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x6e, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20,
					0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x74, 0x65, 0x72, 0x6d,
					0x69, 0x6e, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x61, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x61,
					0x74, 0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69, 0x74, 0x69,
					0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x34, 0x2e, 0x20,
					0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20,
					0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68,
					0x65, 0x72, 0x65, 0x6f, 0x66, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x6d, 0x65, 0x64, 0x69, 0x75, 0x6d, 0x2c, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66,
					0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x2c, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6d, 0x65, 0x65, 0x74, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x61, 0x29, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x67, 0x69, 0x76, 0x65, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x72, 0x65,
					0x63, 0x69, 0x70, 0x69, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x72, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x44, 0x65,
					0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x73, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x28, 0x62, 0x29, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x75,
					0x73, 0x74, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x65, 0x64, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x61, 0x72, 0x72, 0x79,
					0x20, 0x70, 0x72, 0x6f, 0x6d, 0x69, 0x6e, 0x65, 0x6e, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6e, 0x67,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x63, 0x68,
					0x61, 0x6e, 0x67, 0x65, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x73, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x28, 0x63, 0x29, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69, 0x6e, 0x2c,
					0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59,
					0x6f, 0x75, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x2c, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x2c, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74,
					0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d, 0x61, 0x72, 0x6b, 0x2c,
					0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x66,
					0x72, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75,
					0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x64, 0x6f, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x70, 0x65, 0x72, 0x74, 0x61,
					0x69, 0x6e, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x61,
					0x72, 0x74, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x28, 0x64, 0x29, 0x20, 0x49, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x73, 0x20, 0x61, 0x20, 0x22, 0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45, 0x22,
					0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x61,
					0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74,
					0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x20, 0x74, 0x68, 0x65, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x44,
					0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x20, 0x61,
					0x20, 0x72, 0x65, 0x61, 0x64, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x74,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x61, 0x69,
					0x6e, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x73, 0x75, 0x63,
					0x68, 0x20, 0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x2c, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67,
					0x20, 0x74, 0x68, 0x6f, 0x73, 0x65, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20, 0x6e,
					0x6f, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x70, 0x65, 0x72, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x6f, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x69, 0x6e,
					0x20, 0x61, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x73, 0x74, 0x20, 0x6f, 0x6e,
					0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x6c, 0x61, 0x63, 0x65, 0x73, 0x3a,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x4e, 0x4f,
					0x54, 0x49, 0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x61, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x75,
					0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x72, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c,
					0x20, 0x69, 0x66, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64,
					0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20, 0x6f, 0x72,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x64, 0x69, 0x73,
					0x70, 0x6c, 0x61, 0x79, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x74,
					0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65,
					0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x73, 0x2c, 0x20, 0x69, 0x66, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68, 0x65,
					0x72, 0x65, 0x76, 0x65, 0x72, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x74,
					0x68, 0x69, 0x72, 0x64, 0x2d, 0x70, 0x61, 0x72, 0x74, 0x79, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x6e, 0x6f, 0x72, 0x6d, 0x61,
					0x6c, 0x6c, 0x79, 0x20, 0x61, 0x70, 0x70, 0x65, 0x61, 0x72, 0x2e, 0x20,
					0x54, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x73,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x66, 0x6f,
					0x72, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x61, 0x6c, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65, 0x73,
					0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x6f, 0x20, 0x6e,
					0x6f, 0x74, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20, 0x59,
					0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x61, 0x64, 0x64, 0x20, 0x59,
					0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x61, 0x74, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x73, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x44, 0x65, 0x72,
					0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x61,
					0x6c, 0x6f, 0x6e, 0x67, 0x73, 0x69, 0x64, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x73,
					0x20, 0x61, 0x6e, 0x20, 0x61, 0x64, 0x64, 0x65, 0x6e, 0x64, 0x75, 0x6d,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f, 0x54, 0x49,
					0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x72, 0x6f, 0x6d,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x61, 0x6c, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x63,
					0x61, 0x6e, 0x6e, 0x6f, 0x74, 0x20, 0x62, 0x65, 0x20, 0x63, 0x6f, 0x6e,
					0x73, 0x74, 0x72, 0x75, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x73, 0x20, 0x6d, 0x6f, 0x64, 0x69,
					0x66, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x61, 0x64,
					0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x73, 0x74, 0x61,
					0x74, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x59, 0x6f,
					0x75, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69,
					0x64, 0x65, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61,
					0x6c, 0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x66, 0x66, 0x65, 0x72, 0x65,
					0x6e, 0x74, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x74,
					0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x72,
					0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c,
					0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x72,
					0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x75, 0x63,
					0x68, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x20, 0x61, 0x20,
					0x77, 0x68, 0x6f, 0x6c, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69,
					0x64, 0x65, 0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x75, 0x73, 0x65,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x72,
					0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65,
					0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x65, 0x73, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x35, 0x2e, 0x20, 0x53, 0x75, 0x62, 0x6d, 0x69,
					0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x43, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x20,
					0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x65,
					0x78, 0x70, 0x6c, 0x69, 0x63, 0x69, 0x74, 0x6c, 0x79, 0x20, 0x73, 0x74,
					0x61, 0x74, 0x65, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73,
					0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x61,
					0x6c, 0x6c, 0x79, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65,
					0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x73,
					0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x79,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x73, 0x68, 0x61,
					0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x74,
					0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x4e, 0x6f, 0x74, 0x77, 0x69, 0x74, 0x68, 0x73, 0x74, 0x61,
					0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x74, 0x68, 0x69, 0x6e, 0x67,
					0x20, 0x68, 0x65, 0x72, 0x65, 0x69, 0x6e, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x73, 0x75, 0x70, 0x65, 0x72, 0x73, 0x65, 0x64, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x65, 0x70,
					0x61, 0x72, 0x61, 0x74, 0x65, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20,
					0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x68, 0x61, 0x76, 0x65,
					0x20, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x65, 0x64, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x72, 0x65, 0x67, 0x61, 0x72, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x36, 0x2e, 0x20, 0x54, 0x72, 0x61, 0x64, 0x65,
					0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2e, 0x20, 0x54, 0x68, 0x69, 0x73, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x64, 0x6f, 0x65, 0x73,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x6f,
					0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x72, 0x61,
					0x64, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x61, 0x6d,
					0x65, 0x73, 0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d, 0x61, 0x72,
					0x6b, 0x73, 0x2c, 0x20, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x20,
					0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72,
					0x6f, 0x64, 0x75, 0x63, 0x74, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65,
					0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x61, 0x73, 0x20, 0x72, 0x65, 0x71,
					0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x72, 0x65,
					0x61, 0x73, 0x6f, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x63, 0x75, 0x73, 0x74, 0x6f, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x75,
					0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69,
					0x62, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x69, 0x6e,
					0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e,
					0x74, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f, 0x54,
					0x49, 0x43, 0x45, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x37, 0x2e, 0x20, 0x44, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69,
					0x6d, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x57, 0x61, 0x72, 0x72, 0x61,
					0x6e, 0x74, 0x79, 0x2e, 0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20,
					0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20,
					0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c,
					0x61, 0x77, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e,
					0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69,
					0x64, 0x65, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x20, 0x28, 0x61, 0x6e, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65,
					0x73, 0x20, 0x69, 0x74, 0x73, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x29, 0x20, 0x6f, 0x6e, 0x20,
					0x61, 0x6e, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42,
					0x41, 0x53, 0x49, 0x53, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f,
					0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20, 0x4f, 0x46, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x65, 0x69,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72, 0x65, 0x73, 0x73,
					0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6d,
					0x70, 0x6c, 0x69, 0x65, 0x64, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x61, 0x72, 0x72, 0x61, 0x6e,
					0x74, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6f, 0x66, 0x20, 0x54, 0x49, 0x54, 0x4c, 0x45, 0x2c, 0x20, 0x4e,
					0x4f, 0x4e, 0x2d, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x2c, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x6f, 0x72,
					0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x41, 0x52,
					0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50,
					0x4f, 0x53, 0x45, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x72, 0x65,
					0x20, 0x73, 0x6f, 0x6c, 0x65, 0x6c, 0x79, 0x20, 0x72, 0x65, 0x73, 0x70,
					0x6f, 0x6e, 0x73, 0x69, 0x62, 0x6c, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x64, 0x65, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20,
					0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x70,
					0x70, 0x72, 0x6f, 0x70, 0x72, 0x69, 0x61, 0x74, 0x65, 0x6e, 0x65, 0x73,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x75, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x6f,
					0x72, 0x20, 0x72, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x75, 0x6d, 0x65,
					0x20, 0x61, 0x6e, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72,
					0x69, 0x73, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61,
					0x74, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x59, 0x6f, 0x75,
					0x72, 0x20, 0x65, 0x78, 0x65, 0x72, 0x63, 0x69, 0x73, 0x65, 0x20, 0x6f,
					0x66, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x38, 0x2e, 0x20, 0x4c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x4c, 0x69, 0x61, 0x62, 0x69,
					0x6c, 0x69, 0x74, 0x79, 0x2e, 0x20, 0x49, 0x6e, 0x20, 0x6e, 0x6f, 0x20,
					0x65, 0x76, 0x65, 0x6e, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x6e,
					0x64, 0x65, 0x72, 0x20, 0x6e, 0x6f, 0x20, 0x6c, 0x65, 0x67, 0x61, 0x6c,
					0x20, 0x74, 0x68, 0x65, 0x6f, 0x72, 0x79, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x6f, 0x72, 0x74, 0x20, 0x28, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x65, 0x67, 0x6c, 0x69, 0x67,
					0x65, 0x6e, 0x63, 0x65, 0x29, 0x2c, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72,
					0x61, 0x63, 0x74, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65,
					0x72, 0x77, 0x69, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x75, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71, 0x75,
					0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c,
					0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20, 0x28,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x61, 0x73, 0x20, 0x64, 0x65, 0x6c, 0x69,
					0x62, 0x65, 0x72, 0x61, 0x74, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x67,
					0x72, 0x6f, 0x73, 0x73, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6e, 0x65, 0x67, 0x6c, 0x69, 0x67, 0x65, 0x6e, 0x74, 0x20, 0x61,
					0x63, 0x74, 0x73, 0x29, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67, 0x72, 0x65,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77, 0x72, 0x69,
					0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x20, 0x62, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6c, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x59,
					0x6f, 0x75, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67,
					0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e,
					0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74,
					0x2c, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x2c, 0x20,
					0x73, 0x70, 0x65, 0x63, 0x69, 0x61, 0x6c, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x61,
					0x6c, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x65, 0x71,
					0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x64, 0x61, 0x6d, 0x61,
					0x67, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x63,
					0x68, 0x61, 0x72, 0x61, 0x63, 0x74, 0x65, 0x72, 0x20, 0x61, 0x72, 0x69,
					0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x73, 0x20, 0x61, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x75, 0x74, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20,
					0x69, 0x6e, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x74, 0x6f,
					0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x28, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x20, 0x66, 0x6f,
					0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x67, 0x6f,
					0x6f, 0x64, 0x77, 0x69, 0x6c, 0x6c, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x73, 0x74, 0x6f, 0x70, 0x70,
					0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x75, 0x74, 0x65,
					0x72, 0x20, 0x66, 0x61, 0x69, 0x6c, 0x75, 0x72, 0x65, 0x20, 0x6f, 0x72,
					0x20, 0x6d, 0x61, 0x6c, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x65, 0x72, 0x63,
					0x69, 0x61, 0x6c, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x20,
					0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x65, 0x73, 0x29, 0x2c, 0x20,
					0x65, 0x76, 0x65, 0x6e, 0x20, 0x69, 0x66, 0x20, 0x73, 0x75, 0x63, 0x68,
					0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x61, 0x73, 0x20, 0x62,
					0x65, 0x65, 0x6e, 0x20, 0x61, 0x64, 0x76, 0x69, 0x73, 0x65, 0x64, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f, 0x73, 0x73, 0x69,
					0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x39, 0x2e, 0x20, 0x41, 0x63, 0x63, 0x65, 0x70,
					0x74, 0x69, 0x6e, 0x67, 0x20, 0x57, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74,
					0x79, 0x20, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x61, 0x6c, 0x20, 0x4c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74,
					0x79, 0x2e, 0x20, 0x57, 0x68, 0x69, 0x6c, 0x65, 0x20, 0x72, 0x65, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6e, 0x67, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74,
					0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2c, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x6d, 0x61, 0x79, 0x20, 0x63, 0x68, 0x6f, 0x6f, 0x73, 0x65, 0x20, 0x74,
					0x6f, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67,
					0x65, 0x20, 0x61, 0x20, 0x66, 0x65, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x2c,
					0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x61, 0x6e, 0x63, 0x65, 0x20,
					0x6f, 0x66, 0x20, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x2c, 0x20,
					0x77, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x2c, 0x20, 0x69, 0x6e,
					0x64, 0x65, 0x6d, 0x6e, 0x69, 0x74, 0x79, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20,
					0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x6f, 0x62,
					0x6c, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e,
					0x64, 0x2f, 0x6f, 0x72, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20,
					0x63, 0x6f, 0x6e, 0x73, 0x69, 0x73, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x69, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20,
					0x48, 0x6f, 0x77, 0x65, 0x76, 0x65, 0x72, 0x2c, 0x20, 0x69, 0x6e, 0x20,
					0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x6f, 0x62, 0x6c, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x2c, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20,
					0x61, 0x63, 0x74, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f,
					0x77, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x73, 0x6f,
					0x6c, 0x65, 0x20, 0x72, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x73, 0x69, 0x62,
					0x69, 0x6c, 0x69, 0x74, 0x79, 0x2c, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6f,
					0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x6f, 0x74,
					0x68, 0x65, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f, 0x6e, 0x6c,
					0x79, 0x20, 0x69, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x67, 0x72,
					0x65, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x64, 0x65, 0x6d, 0x6e,
					0x69, 0x66, 0x79, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64,
					0x65, 0x66, 0x65, 0x6e, 0x64, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x68,
					0x6f, 0x6c, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68, 0x61, 0x72,
					0x6d, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x75, 0x72, 0x72,
					0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74, 0x65,
					0x64, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e, 0x73, 0x74, 0x2c, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x72, 0x65, 0x61, 0x73, 0x6f,
					0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x79,
					0x6f, 0x75, 0x72, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x69, 0x6e,
					0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x77,
					0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x20, 0x6f, 0x72, 0x20, 0x61,
					0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x6c, 0x69,
					0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2e, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x45, 0x4e, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x45, 0x52, 0x4d,
					0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54,
					0x49, 0x4f, 0x4e, 0x53, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x41, 0x50, 0x50,
					0x45, 0x4e, 0x44, 0x49, 0x58, 0x3a, 0x20, 0x48, 0x6f, 0x77, 0x20, 0x74,
					0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75, 0x72, 0x20, 0x77,
					0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x54, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75, 0x72, 0x20,
					0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x61, 0x74, 0x74, 0x61, 0x63, 0x68,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x6f, 0x69,
					0x6c, 0x65, 0x72, 0x70, 0x6c, 0x61, 0x74, 0x65, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x73, 0x20, 0x65, 0x6e, 0x63,
					0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x62, 0x72, 0x61,
					0x63, 0x6b, 0x65, 0x74, 0x73, 0x20, 0x22, 0x5b, 0x5d, 0x22, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x61, 0x63, 0x65,
					0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x79, 0x6f, 0x75, 0x72, 0x20,
					0x6f, 0x77, 0x6e, 0x20, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x79,
					0x69, 0x6e, 0x67, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x2e, 0x20, 0x28, 0x44, 0x6f, 0x6e, 0x27, 0x74, 0x20,
					0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x62, 0x72, 0x61, 0x63, 0x6b, 0x65,
					0x74, 0x73, 0x21, 0x29, 0x20, 0x20, 0x54, 0x68, 0x65, 0x20, 0x74, 0x65,
					0x78, 0x74, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62, 0x65,
					0x20, 0x65, 0x6e, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x70, 0x70, 0x72, 0x6f, 0x70, 0x72,
					0x69, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63,
					0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x73, 0x79, 0x6e, 0x74, 0x61,
					0x78, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x2e, 0x20, 0x57,
					0x65, 0x20, 0x61, 0x6c, 0x73, 0x6f, 0x20, 0x72, 0x65, 0x63, 0x6f, 0x6d,
					0x6d, 0x65, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x6f,
					0x72, 0x20, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x6e, 0x61, 0x6d, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x75, 0x72, 0x70,
					0x6f, 0x73, 0x65, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x61, 0x6d, 0x65, 0x20, 0x22, 0x70,
					0x72, 0x69, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x70, 0x61, 0x67, 0x65, 0x22,
					0x20, 0x61, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x66, 0x6f, 0x72, 0x20, 0x65, 0x61, 0x73, 0x69, 0x65, 0x72, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69,
					0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x69, 0x72, 0x64, 0x2d, 0x70, 0x61,
					0x72, 0x74, 0x79, 0x20, 0x61, 0x72, 0x63, 0x68, 0x69, 0x76, 0x65, 0x73,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x5b, 0x79, 0x79, 0x79, 0x79, 0x5d, 0x20, 0x5b,
					0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x5d,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x64, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x32, 0x2e, 0x30, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x22, 0x29, 0x3b, 0x0a, 0x20, 0x20, 0x20,
					0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6e, 0x6f, 0x74, 0x20,
					0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x69, 0x6e, 0x20,
					0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x6d, 0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x61,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x74, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70,
					0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68,
					0x65, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x73, 0x2f, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x2d, 0x32,
					0x2e, 0x30, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73,
					0x73, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62,
					0x79, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65,
					0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67, 0x72, 0x65,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77, 0x72, 0x69,
					0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20,
					0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53, 0x49, 0x53, 0x2c, 0x0a,
					0x20, 0x20, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c,
					0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72,
					0x65, 0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70, 0x6c, 0x69,
					0x65, 0x64, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x53, 0x65, 0x65, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x66,
					0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69,
					0x66, 0x69, 0x63, 0x20, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65,
					0x20, 0x67, 0x6f, 0x76, 0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61,
					0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e,
					0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c, 0x69, 0x2f,
					0x67, 0x6f, 0x75, 0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x61, 0x70, 0x65, 0x78, 0x2f, 0x6c, 0x6f, 0x67, 0x0a,
					0x0a, 0x0a, 0x28, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32,
					0x30, 0x31, 0x35, 0x20, 0x54, 0x4a, 0x20, 0x48, 0x6f, 0x6c, 0x6f, 0x77,
					0x61, 0x79, 0x63, 0x68, 0x75, 0x6b, 0x20, 0x26, 0x6c, 0x74, 0x3b, 0x74,
					0x6a, 0x40, 0x74, 0x6a, 0x68, 0x6f, 0x6c, 0x6f, 0x77, 0x61, 0x79, 0x63,
					0x68, 0x75, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x61, 0x26, 0x67, 0x74, 0x3b,
					0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
					0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67,
					0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65,
					0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f,
					0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x0a,
					0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74,
					0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28,
					0x74, 0x68, 0x65, 0x0a, 0x27, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x27, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c,
					0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74,
					0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x0a,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65,
					0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69,
					0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70,
					0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f,
					0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69,
					0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x74, 0x6f, 0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65,
					0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f,
					0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x20, 0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73,
					0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f,
					0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f,
					0x0a, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76,
					0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68,
					0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63,
					0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62,
					0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44,
					0x20, 0x27, 0x41, 0x53, 0x20, 0x49, 0x53, 0x27, 0x2c, 0x20, 0x57, 0x49,
					0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49,
					0x4e, 0x44, 0x2c, 0x0a, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20,
					0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55,
					0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45,
					0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52,
					0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x0a, 0x4d,
					0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43,
					0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45,
					0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52,
					0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x0a, 0x49, 0x4e,
					0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48,
					0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48,
					0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52,
					0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53,
					0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x43, 0x4c, 0x41, 0x49, 0x4d,
					0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49,
					0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41,
					0x43, 0x54, 0x2c, 0x0a, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20,
					0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41,
					0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c,
					0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49,
					0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e,
					0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20,
					0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49,
					0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69,
					0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x70, 0x65,
					0x78, 0x2f, 0x6c, 0x6f, 0x67, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49,
					0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x67, 0x68, 0x6f, 0x64, 0x73, 0x73, 0x2f, 0x79, 0x61, 0x6d, 0x6c,
					0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49, 0x54, 0x29,
					0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34, 0x20, 0x53, 0x61, 0x6d,
					0x20, 0x47, 0x68, 0x6f, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64,
					0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68,
					0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61,
					0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73,
					0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
					0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65,
					0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68,
					0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c,
					0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a,
					0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a,
					0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69,
					0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58,
					0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49,
					0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20,
					0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50,
					0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e,
					0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48,
					0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41,
					0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e,
					0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41,
					0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45,
					0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a,
					0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28,
					0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x32, 0x20, 0x54, 0x68, 0x65, 0x20,
					0x47, 0x6f, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x2e, 0x20,
					0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72,
					0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20,
					0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62,
					0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75,
					0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x0a, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20,
					0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f,
					0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63,
					0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a,
					0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61,
					0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74,
					0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x0a, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74,
					0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73,
					0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a, 0x69, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20,
					0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69,
					0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a,
					0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66,
					0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63, 0x2e,
					0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d,
					0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74, 0x73, 0x0a, 0x63, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x73, 0x20, 0x6d,
					0x61, 0x79, 0x20, 0x62, 0x65, 0x20, 0x75, 0x73, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x73, 0x65, 0x20, 0x6f, 0x72,
					0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f, 0x74, 0x65, 0x20, 0x70, 0x72, 0x6f,
					0x64, 0x75, 0x63, 0x74, 0x73, 0x20, 0x64, 0x65, 0x72, 0x69, 0x76, 0x65,
					0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69,
					0x63, 0x20, 0x70, 0x72, 0x69, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74,
					0x74, 0x65, 0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53,
					0x0a, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20,
					0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42,
					0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54,
					0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52,
					0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53,
					0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49,
					0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53,
					0x45, 0x20, 0x41, 0x52, 0x45, 0x20, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41,
					0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x0a, 0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20,
					0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54,
					0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20,
					0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x0a,
					0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45,
					0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c,
					0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54,
					0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44,
					0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53,
					0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53,
					0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53,
					0x45, 0x2c, 0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20,
					0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20,
					0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54,
					0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48,
					0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x20, 0x43, 0x41, 0x55, 0x53, 0x45,
					0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59,
					0x0a, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48,
					0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43,
					0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c,
					0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49,
					0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48,
					0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56,
					0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x68, 0x6f, 0x64,
					0x73, 0x73, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x6a, 0x65, 0x73, 0x73, 0x65, 0x76, 0x64, 0x6b, 0x2f,
					0x67, 0x6f, 0x2d, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x0a, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29,
					0x20, 0x32, 0x30, 0x31, 0x32, 0x20, 0x4a, 0x65, 0x73, 0x73, 0x65, 0x20,
					0x76, 0x61, 0x6e, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x4b, 0x69, 0x65, 0x62,
					0x6f, 0x6f, 0x6d, 0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64,
					0x2e, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65,
					0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64,
					0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61,
					0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64,
					0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68,
					0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65, 0x74, 0x3a,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64,
					0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69,
					0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65,
					0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69,
					0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64,
					0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a,
					0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66,
					0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63, 0x2e,
					0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d,
					0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x6f, 0x72, 0x73, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65, 0x20, 0x75,
					0x73, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f, 0x72,
					0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f, 0x74,
					0x65, 0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20, 0x64,
					0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20,
					0x70, 0x72, 0x69, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65,
					0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
					0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56,
					0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f,
					0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f,
					0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x0a, 0x22,
					0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54,
					0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44,
					0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46,
					0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55,
					0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20,
					0x41, 0x52, 0x45, 0x20, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d,
					0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x0a,
					0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20,
					0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e,
					0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x0a, 0x53, 0x50,
					0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50,
					0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e,
					0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20, 0x44,
					0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49,
					0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x20,
					0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c,
					0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52,
					0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55,
					0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52,
					0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57,
					0x45, 0x56, 0x45, 0x52, 0x20, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x54,
					0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f,
					0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47,
					0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41,
					0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53,
					0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f,
					0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45,
					0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x65, 0x73, 0x73, 0x65, 0x76,
					0x64, 0x6b, 0x2f, 0x67, 0x6f, 0x2d, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x0a,
					0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x6f, 0x68, 0x6f, 0x2f,
					0x67, 0x6f, 0x64, 0x6f, 0x74, 0x65, 0x6e, 0x76, 0x0a, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29,
					0x20, 0x32, 0x30, 0x31, 0x33, 0x20, 0x4a, 0x6f, 0x68, 0x6e, 0x20, 0x42,
					0x61, 0x72, 0x74, 0x6f, 0x6e, 0x0a, 0x0a, 0x4d, 0x49, 0x54, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64,
					0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68,
					0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61,
					0x69, 0x6e, 0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73,
					0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x0a, 0x22, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
					0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65,
					0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68,
					0x2c, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c,
					0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20,
					0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a,
					0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65,
					0x0a, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69,
					0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a, 0x45, 0x58,
					0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46, 0x49,
					0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20,
					0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50,
					0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x0a, 0x4e,
					0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x0a, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41,
					0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x4f, 0x46, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e,
					0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57, 0x49, 0x54, 0x48, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41,
					0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a,
					0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x6f, 0x68, 0x6f, 0x2f, 0x67, 0x6f, 0x64,
					0x6f, 0x74, 0x65, 0x6e, 0x76, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49,
					0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x6b, 0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f, 0x73, 0x2f, 0x6f,
					0x73, 0x65, 0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31,
					0x32, 0x20, 0x54, 0x68, 0x65, 0x20, 0x47, 0x6f, 0x20, 0x41, 0x75, 0x74,
					0x68, 0x6f, 0x72, 0x73, 0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65,
					0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75,
					0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20,
					0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d,
					0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c,
					0x20, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74,
					0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20,
					0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65,
					0x74, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63,
					0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74,
					0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76,
					0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e,
					0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72,
					0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69,
					0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f,
					0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f,
					0x76, 0x65, 0x0a, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65,
					0x72, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61,
					0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20,
					0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20, 0x70, 0x72,
					0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20,
					0x74, 0x68, 0x65, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e,
					0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e,
					0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c,
					0x65, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x69, 0x74, 0x73, 0x0a, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x73, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65, 0x20,
					0x75, 0x73, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f,
					0x72, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f,
					0x74, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20,
					0x64, 0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d,
					0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x73,
					0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x70, 0x72, 0x69, 0x6f,
					0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54,
					0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44,
					0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59,
					0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52,
					0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49,
					0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x0a, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x20,
					0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20,
					0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20,
					0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f,
					0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x0a, 0x4f, 0x57, 0x4e, 0x45,
					0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42,
					0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49,
					0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44, 0x45,
					0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x0a, 0x53, 0x50, 0x45, 0x43, 0x49, 0x41,
					0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59,
					0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55,
					0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47,
					0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c,
					0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50,
					0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54, 0x45,
					0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53, 0x45,
					0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53,
					0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x0a, 0x44, 0x41, 0x54,
					0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49, 0x54,
					0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e, 0x45,
					0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50, 0x54,
					0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45, 0x52,
					0x20, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x54, 0x48, 0x45, 0x4f, 0x52,
					0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c,
					0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53,
					0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x49,
					0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f, 0x55,
					0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x20,
					0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20, 0x4f,
					0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x43,
					0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45,
					0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x6b, 0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f, 0x73, 0x2f,
					0x6f, 0x73, 0x65, 0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49,
					0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x6f, 0x6c, 0x65, 0x6b, 0x75, 0x6b, 0x6f, 0x6e, 0x6b, 0x6f, 0x2f,
					0x74, 0x73, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34, 0x20,
					0x62, 0x79, 0x20, 0x4f, 0x6c, 0x65, 0x6b, 0x75, 0x20, 0x4b, 0x6f, 0x6e,
					0x6b, 0x6f, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79,
					0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72,
					0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72,
					0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e,
					0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73,
					0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65,
					0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e,
					0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69,
					0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75,
					0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f,
					0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c,
					0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f,
					0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20,
					0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77,
					0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e,
					0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20,
					0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20,
					0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x0a, 0x61, 0x6c, 0x6c,
					0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73,
					0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70,
					0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e,
					0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44,
					0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20,
					0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53,
					0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20,
					0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49,
					0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46,
					0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53,
					0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54,
					0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f,
					0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e,
					0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20,
					0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20,
					0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41, 0x55,
					0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50,
					0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45,
					0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41,
					0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c,
					0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f,
					0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49,
					0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47,
					0x53, 0x20, 0x49, 0x4e, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6f, 0x6c,
					0x65, 0x6b, 0x75, 0x6b, 0x6f, 0x6e, 0x6b, 0x6f, 0x2f, 0x74, 0x73, 0x0a,
					0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x67, 0x2f, 0x65,
					0x72, 0x72, 0x6f, 0x72, 0x73, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30,
					0x31, 0x35, 0x2c, 0x20, 0x44, 0x61, 0x76, 0x65, 0x20, 0x43, 0x68, 0x65,
					0x6e, 0x65, 0x79, 0x20, 0x3c, 0x64, 0x61, 0x76, 0x65, 0x40, 0x63, 0x68,
					0x65, 0x6e, 0x65, 0x79, 0x2e, 0x6e, 0x65, 0x74, 0x3e, 0x0a, 0x41, 0x6c,
					0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73,
					0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e,
					0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69,
					0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72,
					0x65, 0x20, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20,
					0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65,
					0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f,
					0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x0a, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69,
					0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72,
					0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x0a, 0x20, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c,
					0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61,
					0x69, 0x6d, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x0a, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61,
					0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a,
					0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45,
					0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50,
					0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45,
					0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x0a, 0x41, 0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x2c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45,
					0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53,
					0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45,
					0x0a, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e,
					0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54,
					0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43,
					0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c,
					0x44, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x0a, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e,
					0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49,
					0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x20, 0x53, 0x50, 0x45, 0x43,
					0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41,
					0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45,
					0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x0a, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c,
					0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55,
					0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x0a,
					0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x20, 0x4c, 0x4f,
					0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x44,
					0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46,
					0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49,
					0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55,
					0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56,
					0x45, 0x52, 0x0a, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x54, 0x48, 0x45,
					0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43,
					0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x4f, 0x52, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e,
					0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20,
					0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55,
					0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45,
					0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44,
					0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53,
					0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53,
					0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a,
					0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x67, 0x2f, 0x65, 0x72, 0x72, 0x6f,
					0x72, 0x73, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x64,
					0x73, 0x75, 0x62, 0x68, 0x61, 0x73, 0x2f, 0x67, 0x6f, 0x2d, 0x65, 0x6c,
					0x61, 0x73, 0x74, 0x69, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x0a, 0x0a,
					0x0a, 0x4d, 0x49, 0x54, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x53, 0x75, 0x62,
					0x68, 0x61, 0x73, 0x20, 0x44, 0x61, 0x6e, 0x64, 0x61, 0x70, 0x61, 0x6e,
					0x69, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
					0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20,
					0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65,
					0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73,
					0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67,
					0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61,
					0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20,
					0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61,
					0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d,
					0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73,
					0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64,
					0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20,
					0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62,
					0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64,
					0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70,
					0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70,
					0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68,
					0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69,
					0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73,
					0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74,
					0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77,
					0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f,
					0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73,
					0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73,
					0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a,
					0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75,
					0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f,
					0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a,
					0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45,
					0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57,
					0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b,
					0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53,
					0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c,
					0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42,
					0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54,
					0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20,
					0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49,
					0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53,
					0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46,
					0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49,
					0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53,
					0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54,
					0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59,
					0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52,
					0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49,
					0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54,
					0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52,
					0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20,
					0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d,
					0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20,
					0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f,
					0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53,
					0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x64,
					0x73, 0x75, 0x62, 0x68, 0x61, 0x73, 0x2f, 0x67, 0x6f, 0x2d, 0x65, 0x6c,
					0x61, 0x73, 0x74, 0x69, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x0a, 0x0a,
					0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x61, 0x74, 0x6f, 0x72, 0x69,
					0x2f, 0x67, 0x6f, 0x2e, 0x75, 0x75, 0x69, 0x64, 0x0a, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29,
					0x20, 0x32, 0x30, 0x31, 0x33, 0x2d, 0x32, 0x30, 0x31, 0x36, 0x20, 0x62,
					0x79, 0x20, 0x4d, 0x61, 0x78, 0x69, 0x6d, 0x20, 0x42, 0x75, 0x62, 0x6c,
					0x69, 0x73, 0x20, 0x3c, 0x62, 0x40, 0x63, 0x6f, 0x64, 0x65, 0x6d, 0x6f,
					0x6e, 0x6b, 0x65, 0x79, 0x2e, 0x72, 0x75, 0x3e, 0x0a, 0x0a, 0x50, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62,
					0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x0a,
					0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73,
					0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20,
					0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69,
					0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69,
					0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e,
					0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55,
					0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44,
					0x0a, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x0a,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x4f,
					0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f,
					0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57, 0x49, 0x54,
					0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44,
					0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e,
					0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x61, 0x74, 0x6f, 0x72, 0x69, 0x2f,
					0x67, 0x6f, 0x2e, 0x75, 0x75, 0x69, 0x64, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x74, 0x67, 0x75, 0x6c, 0x61, 0x63, 0x73, 0x69, 0x2f,
					0x77, 0x72, 0x61, 0x70, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31,
					0x34, 0x2c, 0x20, 0x42, 0x72, 0x61, 0x6e, 0x64, 0x6f, 0x6e, 0x20, 0x54,
					0x68, 0x6f, 0x6d, 0x73, 0x6f, 0x6e, 0x0a, 0x41, 0x6c, 0x6c, 0x20, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76,
					0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79,
					0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x0a, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74,
					0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f,
					0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d,
					0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64,
					0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x0a, 0x20,
					0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e,
					0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72,
					0x2e, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20,
					0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75,
					0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x0a,
					0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65,
					0x72, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61,
					0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x6f, 0x74, 0x68, 0x65,
					0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20,
					0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49,
					0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49,
					0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42,
					0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49,
					0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55,
					0x54, 0x4f, 0x52, 0x53, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x20, 0x41, 0x4e, 0x44, 0x0a, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50,
					0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c,
					0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c,
					0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x0a, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46,
					0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54,
					0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50,
					0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55,
					0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x0a, 0x44, 0x49,
					0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e,
					0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48,
					0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59,
					0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52,
					0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55,
					0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42,
					0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x4e, 0x59, 0x20, 0x44,
					0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52,
					0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e,
					0x54, 0x41, 0x4c, 0x2c, 0x20, 0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c,
					0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c,
					0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45,
					0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45,
					0x53, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47,
					0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49,
					0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52,
					0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46,
					0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20,
					0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53, 0x45, 0x52,
					0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x0a, 0x4c, 0x4f, 0x53, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x44, 0x41, 0x54, 0x41,
					0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53,
					0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53,
					0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49,
					0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x20,
					0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f,
					0x4e, 0x0a, 0x41, 0x4e, 0x59, 0x20, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59,
					0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49,
					0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52,
					0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47,
					0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45,
					0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20,
					0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x0a, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49,
					0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48,
					0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e,
					0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x74, 0x67, 0x75, 0x6c, 0x61, 0x63, 0x73, 0x69, 0x2f, 0x77, 0x72,
					0x61, 0x70, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69,
					0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x67, 0x6a, 0x73, 0x6f, 0x6e, 0x0a,
					0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49, 0x54, 0x29, 0x0a,
					0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28,
					0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x4a, 0x6f, 0x73, 0x68,
					0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64,
					0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68,
					0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61,
					0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73,
					0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
					0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20,
					0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65,
					0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68,
					0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c,
					0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x0a,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20,
					0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a, 0x73, 0x75, 0x62, 0x6a,
					0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69,
					0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58,
					0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46, 0x49,
					0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20,
					0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50,
					0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e,
					0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52,
					0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x49, 0x4e, 0x20, 0x41,
					0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a, 0x43, 0x4f, 0x4e, 0x4e,
					0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41,
					0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a,
					0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x67,
					0x6a, 0x73, 0x6f, 0x6e, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e,
					0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
					0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x6d, 0x61, 0x74, 0x63,
					0x68, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49, 0x54,
					0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x4a, 0x6f,
					0x73, 0x68, 0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a, 0x50, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62,
					0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20,
					0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73,
					0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20,
					0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69,
					0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69,
					0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a, 0x73, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e,
					0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55,
					0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20,
					0x4f, 0x52, 0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f,
					0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a, 0x43, 0x4f,
					0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54,
					0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44,
					0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e,
					0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c,
					0x2f, 0x6d, 0x61, 0x74, 0x63, 0x68, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47,
					0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e, 0x69, 0x6e, 0x2f,
					0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a, 0x0a, 0x43, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x32, 0x30, 0x31, 0x31,
					0x2d, 0x32, 0x30, 0x31, 0x36, 0x20, 0x43, 0x61, 0x6e, 0x6f, 0x6e, 0x69,
					0x63, 0x61, 0x6c, 0x20, 0x4c, 0x74, 0x64, 0x2e, 0x0a, 0x0a, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x56, 0x65, 0x72,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30, 0x20, 0x28, 0x74, 0x68,
					0x65, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x22, 0x29,
					0x3b, 0x0a, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6e, 0x6f,
					0x74, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x69,
					0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x59, 0x6f, 0x75, 0x20, 0x6d,
					0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x61, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x74, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
					0x77, 0x77, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x6f, 0x72,
					0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x2f, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x2d, 0x32, 0x2e, 0x30, 0x0a, 0x0a,
					0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69,
					0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69,
					0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72,
					0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69,
					0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x0a, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x20, 0x22,
					0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53, 0x49, 0x53,
					0x2c, 0x0a, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72, 0x65,
					0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70, 0x6c, 0x69, 0x65,
					0x64, 0x2e, 0x0a, 0x53, 0x65, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20,
					0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x20, 0x67, 0x6f, 0x76,
					0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x6c,
					0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x75,
					0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x66, 0x6f,
					0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x69, 0x6c, 0x65,
					0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x65,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x47, 0x6f, 0x20, 0x66, 0x72, 0x6f, 0x6d,
					0x20, 0x43, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x6c, 0x69, 0x62, 0x79, 0x61, 0x6d, 0x6c, 0x2c, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x75, 0x73, 0x0a, 0x61, 0x72, 0x65, 0x20, 0x73, 0x74,
					0x69, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x76, 0x65, 0x72, 0x65, 0x64, 0x20,
					0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x20, 0x6f, 0x72, 0x69,
					0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x61, 0x70,
					0x69, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65, 0x6d,
					0x69, 0x74, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x70, 0x61, 0x72, 0x73, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x61, 0x64, 0x65, 0x72, 0x63,
					0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x73, 0x63, 0x61, 0x6e,
					0x6e, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x77, 0x72, 0x69, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x68, 0x2e, 0x67, 0x6f, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x70, 0x72, 0x69, 0x76,
					0x61, 0x74, 0x65, 0x68, 0x2e, 0x67, 0x6f, 0x0a, 0x0a, 0x43, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32,
					0x30, 0x30, 0x36, 0x20, 0x4b, 0x69, 0x72, 0x69, 0x6c, 0x6c, 0x20, 0x53,
					0x69, 0x6d, 0x6f, 0x6e, 0x6f, 0x76, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64,
					0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68,
					0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61,
					0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73,
					0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
					0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20,
					0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65,
					0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68,
					0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c,
					0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x0a, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20,
					0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x64, 0x6f, 0x0a, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a,
					0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69,
					0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58,
					0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49,
					0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20,
					0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50,
					0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e,
					0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48,
					0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41,
					0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e,
					0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41,
					0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45,
					0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a,
					0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e, 0x69, 0x6e,
					0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a, 0x0a, 0x42,
					0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d, 0x69,
					0x6e, 0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76, 0x65, 0x72, 0x0a, 0x0a,
					0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d,
					0x69, 0x6e, 0x64, 0x73, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34, 0x2d,
					0x32, 0x30, 0x31, 0x35, 0x2c, 0x20, 0x4d, 0x61, 0x74, 0x74, 0x20, 0x42,
					0x75, 0x74, 0x63, 0x68, 0x65, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x4d,
					0x61, 0x74, 0x74, 0x20, 0x46, 0x61, 0x72, 0x69, 0x6e, 0x61, 0x0a, 0x0a,
					0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69,
					0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61,
					0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f,
					0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20,
					0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64,
					0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68,
					0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22,
					0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72,
					0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20,
					0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79,
					0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62,
					0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72,
					0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73,
					0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20,
					0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a,
					0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x64, 0x20, 0x69, 0x6e, 0x0a, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70,
					0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74,
					0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49,
					0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22,
					0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59,
					0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44,
					0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52,
					0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20,
					0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20,
					0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52,
					0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c,
					0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e,
					0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e,
					0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c,
					0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47,
					0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42,
					0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20,
					0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e,
					0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54,
					0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49,
					0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f,
					0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20,
					0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57,
					0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e,
					0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65,
					0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76, 0x65,
					0x72, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49,
					0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69,
					0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73,
					0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x70, 0x72,
					0x69, 0x67, 0x0a, 0x0a, 0x0a, 0x53, 0x70, 0x72, 0x69, 0x67, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29,
					0x20, 0x32, 0x30, 0x31, 0x33, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72,
					0x6d, 0x69, 0x6e, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69,
					0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72,
					0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c,
					0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61,
					0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69,
					0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73,
					0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75,
					0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f,
					0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69,
					0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74,
					0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74,
					0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c,
					0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72,
					0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c,
					0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c,
					0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c,
					0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c,
					0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74,
					0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66,
					0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65,
					0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f,
					0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65,
					0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20,
					0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x0a,
					0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61,
					0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f,
					0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53,
					0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50,
					0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c,
					0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49,
					0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c,
					0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53,
					0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54,
					0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50,
					0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55,
					0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f,
					0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e,
					0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45,
					0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45,
					0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f,
					0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47,
					0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a,
					0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57,
					0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e,
					0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52,
					0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49,
					0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20,
					0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46,
					0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45,
					0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c,
					0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45,
					0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64,
					0x73, 0x2f, 0x73, 0x70, 0x72, 0x69, 0x67, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c, 0x69, 0x2f, 0x67, 0x6f,
					0x75, 0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30, 0x2c,
					0x20, 0x4a, 0x61, 0x6e, 0x75, 0x61, 0x72, 0x79, 0x20, 0x32, 0x30, 0x30,
					0x34, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77,
					0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x2f,
					0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x2f, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x54, 0x45, 0x52, 0x4d, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x52, 0x45, 0x50, 0x52,
					0x4f, 0x44, 0x55, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x2c, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x44, 0x49, 0x53, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x49,
					0x4f, 0x4e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x31, 0x2e, 0x20, 0x44, 0x65,
					0x66, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65,
					0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65, 0x2c,
					0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x61, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x53, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x31, 0x20, 0x74, 0x68, 0x72, 0x6f, 0x75, 0x67, 0x68, 0x20, 0x39,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x22,
					0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x6f, 0x72, 0x20, 0x65,
					0x6e, 0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72,
					0x69, 0x7a, 0x65, 0x64, 0x20, 0x62, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x74, 0x68,
					0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x69,
					0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22,
					0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74, 0x69, 0x74, 0x79,
					0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x6e, 0x69, 0x6f, 0x6e, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x63, 0x74, 0x69, 0x6e, 0x67,
					0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x74,
					0x68, 0x65, 0x72, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x69, 0x65, 0x73,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f,
					0x6c, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72,
					0x6f, 0x6c, 0x6c, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72,
					0x20, 0x61, 0x72, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x63,
					0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79,
					0x2e, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75,
					0x72, 0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x63, 0x6f,
					0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x22, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x73,
					0x20, 0x28, 0x69, 0x29, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f, 0x77,
					0x65, 0x72, 0x2c, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x20, 0x6f,
					0x72, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6e, 0x61,
					0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2c, 0x20, 0x77,
					0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63, 0x6f,
					0x6e, 0x74, 0x72, 0x61, 0x63, 0x74, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73,
					0x65, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x29, 0x20, 0x6f,
					0x77, 0x6e, 0x65, 0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66, 0x20,
					0x66, 0x69, 0x66, 0x74, 0x79, 0x20, 0x70, 0x65, 0x72, 0x63, 0x65, 0x6e,
					0x74, 0x20, 0x28, 0x35, 0x30, 0x25, 0x29, 0x20, 0x6f, 0x72, 0x20, 0x6d,
					0x6f, 0x72, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e,
					0x64, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x68, 0x61, 0x72, 0x65, 0x73, 0x2c,
					0x20, 0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x69, 0x29, 0x20, 0x62, 0x65,
					0x6e, 0x65, 0x66, 0x69, 0x63, 0x69, 0x61, 0x6c, 0x20, 0x6f, 0x77, 0x6e,
					0x65, 0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x22, 0x20,
					0x28, 0x6f, 0x72, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x72, 0x22, 0x29, 0x20,
					0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x61,
					0x6e, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c,
					0x20, 0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e,
					0x74, 0x69, 0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65,
					0x78, 0x65, 0x72, 0x63, 0x69, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x67, 0x72,
					0x61, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x53, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x22, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70,
					0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6b, 0x69, 0x6e, 0x67,
					0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x20,
					0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x2c,
					0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x6f, 0x75,
					0x72, 0x63, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e,
					0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x22, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x22, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61,
					0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x72,
					0x65, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x72, 0x6f,
					0x6d, 0x20, 0x6d, 0x65, 0x63, 0x68, 0x61, 0x6e, 0x69, 0x63, 0x61, 0x6c,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x72, 0x61, 0x6e, 0x73,
					0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x72,
					0x20, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x6f, 0x66, 0x20, 0x61, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65,
					0x20, 0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x69, 0x6c,
					0x65, 0x64, 0x20, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x63, 0x6f,
					0x64, 0x65, 0x2c, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20,
					0x6d, 0x65, 0x64, 0x69, 0x61, 0x20, 0x74, 0x79, 0x70, 0x65, 0x73, 0x2e,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x57, 0x6f, 0x72,
					0x6b, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x6f,
					0x66, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x68, 0x69, 0x70,
					0x2c, 0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69, 0x6e,
					0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x6f, 0x72, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x6d, 0x61, 0x64, 0x65, 0x20, 0x61,
					0x76, 0x61, 0x69, 0x6c, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x75, 0x6e, 0x64,
					0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x73, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x63,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x74, 0x68, 0x61,
					0x74, 0x20, 0x69, 0x73, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x64, 0x20, 0x69, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x74, 0x74, 0x61,
					0x63, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x77, 0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
					0x61, 0x6e, 0x20, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x20, 0x69,
					0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x70, 0x65, 0x6e, 0x64,
					0x69, 0x78, 0x20, 0x62, 0x65, 0x6c, 0x6f, 0x77, 0x29, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x44, 0x65, 0x72, 0x69, 0x76,
					0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x22,
					0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x77, 0x68,
					0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75,
					0x72, 0x63, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63,
					0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x2c, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x62, 0x61,
					0x73, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x64,
					0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x29,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x77, 0x68, 0x69, 0x63, 0x68, 0x20,
					0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x64,
					0x69, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x72, 0x65, 0x76, 0x69,
					0x73, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x61, 0x6e, 0x6e, 0x6f, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x65, 0x6c, 0x61, 0x62,
					0x6f, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6f, 0x72,
					0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66,
					0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74,
					0x2c, 0x20, 0x61, 0x73, 0x20, 0x61, 0x20, 0x77, 0x68, 0x6f, 0x6c, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x61,
					0x6c, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x75,
					0x74, 0x68, 0x6f, 0x72, 0x73, 0x68, 0x69, 0x70, 0x2e, 0x20, 0x46, 0x6f,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73,
					0x65, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x20, 0x77, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x72, 0x65, 0x6d, 0x61, 0x69, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x73, 0x65, 0x70, 0x61, 0x72, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x66,
					0x72, 0x6f, 0x6d, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x65, 0x72, 0x65,
					0x6c, 0x79, 0x20, 0x6c, 0x69, 0x6e, 0x6b, 0x20, 0x28, 0x6f, 0x72, 0x20,
					0x62, 0x69, 0x6e, 0x64, 0x20, 0x62, 0x79, 0x20, 0x6e, 0x61, 0x6d, 0x65,
					0x29, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x69, 0x6e, 0x74,
					0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x2c, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76,
					0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20,
					0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2e, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x6f,
					0x72, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72,
					0x73, 0x68, 0x69, 0x70, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20, 0x76,
					0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x64, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x74, 0x6f, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x65,
					0x72, 0x65, 0x6f, 0x66, 0x2c, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69,
					0x73, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x61,
					0x6c, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x75,
					0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x69, 0x6e, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x62, 0x79,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x61, 0x6e, 0x20,
					0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c, 0x20, 0x6f,
					0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74, 0x69,
					0x74, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x20,
					0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f, 0x66,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e,
					0x65, 0x72, 0x2e, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x22, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74,
					0x74, 0x65, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d,
					0x65, 0x61, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x20, 0x6f, 0x66, 0x20, 0x65, 0x6c, 0x65, 0x63, 0x74, 0x72, 0x6f,
					0x6e, 0x69, 0x63, 0x2c, 0x20, 0x76, 0x65, 0x72, 0x62, 0x61, 0x6c, 0x2c,
					0x20, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e, 0x20,
					0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x73, 0x65, 0x6e, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x6f, 0x72, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x74, 0x73, 0x20,
					0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69,
					0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c,
					0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x6e, 0x20, 0x65, 0x6c, 0x65,
					0x63, 0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63, 0x20, 0x6d, 0x61, 0x69, 0x6c,
					0x69, 0x6e, 0x67, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x73, 0x2c, 0x20, 0x73,
					0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65,
					0x6d, 0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x20, 0x74, 0x72, 0x61, 0x63,
					0x6b, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x73,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d, 0x61,
					0x6e, 0x61, 0x67, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72,
					0x20, 0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f,
					0x66, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66, 0x6f,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73,
					0x65, 0x20, 0x6f, 0x66, 0x20, 0x64, 0x69, 0x73, 0x63, 0x75, 0x73, 0x73,
					0x69, 0x6e, 0x67, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x69, 0x6d, 0x70, 0x72,
					0x6f, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x2c, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20,
					0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x63, 0x6f,
					0x6e, 0x73, 0x70, 0x69, 0x63, 0x75, 0x6f, 0x75, 0x73, 0x6c, 0x79, 0x20,
					0x6d, 0x61, 0x72, 0x6b, 0x65, 0x64, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74,
					0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x64, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x65, 0x64,
					0x20, 0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x20,
					0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x61,
					0x73, 0x20, 0x22, 0x4e, 0x6f, 0x74, 0x20, 0x61, 0x20, 0x43, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x22, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x22, 0x20, 0x73, 0x68, 0x61,
					0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c, 0x20,
					0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74,
					0x69, 0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x6e,
					0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f, 0x66, 0x20, 0x77,
					0x68, 0x6f, 0x6d, 0x20, 0x61, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x68, 0x61, 0x73, 0x20, 0x62,
					0x65, 0x65, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72,
					0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73,
					0x75, 0x62, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x6c, 0x79, 0x20,
					0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65, 0x64,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x32, 0x2e,
					0x20, 0x47, 0x72, 0x61, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x43, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2e, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d,
					0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68, 0x65, 0x72,
					0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x73, 0x20, 0x74,
					0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x20, 0x70, 0x65, 0x72, 0x70,
					0x65, 0x74, 0x75, 0x61, 0x6c, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x77, 0x6f, 0x72, 0x6c, 0x64, 0x77, 0x69, 0x64, 0x65, 0x2c, 0x20,
					0x6e, 0x6f, 0x6e, 0x2d, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x76,
					0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x2d, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65,
					0x2c, 0x20, 0x72, 0x6f, 0x79, 0x61, 0x6c, 0x74, 0x79, 0x2d, 0x66, 0x72,
					0x65, 0x65, 0x2c, 0x20, 0x69, 0x72, 0x72, 0x65, 0x76, 0x6f, 0x63, 0x61,
					0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f,
					0x64, 0x75, 0x63, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x65, 0x70, 0x61, 0x72,
					0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x6f, 0x66, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x6c,
					0x79, 0x20, 0x64, 0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x2c, 0x20, 0x70,
					0x75, 0x62, 0x6c, 0x69, 0x63, 0x6c, 0x79, 0x20, 0x70, 0x65, 0x72, 0x66,
					0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x20, 0x74, 0x68, 0x65, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x6f,
					0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x33, 0x2e, 0x20, 0x47, 0x72,
					0x61, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x50, 0x61, 0x74, 0x65, 0x6e,
					0x74, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20, 0x53,
					0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f,
					0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x65, 0x61,
					0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x6f, 0x72, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72,
					0x61, 0x6e, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x61, 0x20, 0x70, 0x65, 0x72, 0x70, 0x65, 0x74, 0x75, 0x61, 0x6c, 0x2c,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x6f, 0x72, 0x6c, 0x64,
					0x77, 0x69, 0x64, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x65, 0x78,
					0x63, 0x6c, 0x75, 0x73, 0x69, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x2d,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x72, 0x6f, 0x79, 0x61,
					0x6c, 0x74, 0x79, 0x2d, 0x66, 0x72, 0x65, 0x65, 0x2c, 0x20, 0x69, 0x72,
					0x72, 0x65, 0x76, 0x6f, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x28, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20,
					0x61, 0x73, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x65, 0x63, 0x74, 0x69, 0x6f,
					0x6e, 0x29, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x6b,
					0x65, 0x2c, 0x20, 0x68, 0x61, 0x76, 0x65, 0x20, 0x6d, 0x61, 0x64, 0x65,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x75, 0x73, 0x65, 0x2c,
					0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x65,
					0x6c, 0x6c, 0x2c, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x2c, 0x20, 0x69, 0x6d,
					0x70, 0x6f, 0x72, 0x74, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f, 0x74,
					0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x20, 0x74, 0x72, 0x61, 0x6e,
					0x73, 0x66, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68, 0x65,
					0x72, 0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x6f, 0x73,
					0x65, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x63, 0x6c, 0x61,
					0x69, 0x6d, 0x73, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x61, 0x62,
					0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x79, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x72,
					0x65, 0x20, 0x6e, 0x65, 0x63, 0x65, 0x73, 0x73, 0x61, 0x72, 0x69, 0x6c,
					0x79, 0x20, 0x69, 0x6e, 0x66, 0x72, 0x69, 0x6e, 0x67, 0x65, 0x64, 0x20,
					0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x65,
					0x20, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63, 0x6f, 0x6d, 0x62, 0x69,
					0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x69, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x69, 0x63, 0x68,
					0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x77, 0x61,
					0x73, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x2e,
					0x20, 0x49, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x69, 0x6e, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x65, 0x20,
					0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x74, 0x69, 0x67,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e, 0x73,
					0x74, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79,
					0x20, 0x28, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20,
					0x61, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x72, 0x6f, 0x73,
					0x73, 0x2d, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x20, 0x6f, 0x72, 0x20, 0x63,
					0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x20, 0x6c, 0x61, 0x77, 0x73, 0x75, 0x69, 0x74,
					0x29, 0x20, 0x61, 0x6c, 0x6c, 0x65, 0x67, 0x69, 0x6e, 0x67, 0x20, 0x74,
					0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x20,
					0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x74, 0x69,
					0x74, 0x75, 0x74, 0x65, 0x73, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x79, 0x20, 0x70,
					0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e, 0x66, 0x72, 0x69, 0x6e,
					0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x6e,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20,
					0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x74, 0x65, 0x72,
					0x6d, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x61, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64,
					0x61, 0x74, 0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69, 0x74,
					0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x34, 0x2e,
					0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79,
					0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74,
					0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x6d, 0x65, 0x64, 0x69, 0x75, 0x6d, 0x2c, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x6f, 0x64, 0x69,
					0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65,
					0x20, 0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x66,
					0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65,
					0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x65, 0x65, 0x74, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x61, 0x29, 0x20, 0x59,
					0x6f, 0x75, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x67, 0x69, 0x76, 0x65,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x72,
					0x65, 0x63, 0x69, 0x70, 0x69, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x72,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x44,
					0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x73, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x28, 0x62, 0x29, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d,
					0x75, 0x73, 0x74, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x65, 0x64, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x61, 0x72, 0x72,
					0x79, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x69, 0x6e, 0x65, 0x6e, 0x74, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6e,
					0x67, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x63,
					0x68, 0x61, 0x6e, 0x67, 0x65, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x63, 0x29, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69, 0x6e,
					0x2c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x75,
					0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x59, 0x6f, 0x75, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x2c, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e,
					0x74, 0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d, 0x61, 0x72, 0x6b,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20,
					0x66, 0x72, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x75,
					0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x78, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x64, 0x6f, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x70, 0x65, 0x72, 0x74,
					0x61, 0x69, 0x6e, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70,
					0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72,
					0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x73, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x28, 0x64, 0x29, 0x20, 0x49, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x65, 0x73, 0x20, 0x61, 0x20, 0x22, 0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45,
					0x22, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20,
					0x61, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x69,
					0x74, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x20, 0x6d, 0x75, 0x73, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x20,
					0x61, 0x20, 0x72, 0x65, 0x61, 0x64, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61,
					0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x61,
					0x69, 0x6e, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x2c, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e,
					0x67, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20,
					0x6e, 0x6f, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x70, 0x65, 0x72, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x6f,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x73, 0x74, 0x20, 0x6f,
					0x6e, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c,
					0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x6c, 0x61, 0x63, 0x65, 0x73,
					0x3a, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x4e,
					0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x61, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x72,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x20, 0x69, 0x66, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65,
					0x64, 0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20, 0x6f,
					0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x64, 0x69,
					0x73, 0x70, 0x6c, 0x61, 0x79, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61,
					0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44,
					0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x73, 0x2c, 0x20, 0x69, 0x66, 0x20, 0x61, 0x6e, 0x64, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68,
					0x65, 0x72, 0x65, 0x76, 0x65, 0x72, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20,
					0x74, 0x68, 0x69, 0x72, 0x64, 0x2d, 0x70, 0x61, 0x72, 0x74, 0x79, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x6e, 0x6f, 0x72, 0x6d,
					0x61, 0x6c, 0x6c, 0x79, 0x20, 0x61, 0x70, 0x70, 0x65, 0x61, 0x72, 0x2e,
					0x20, 0x54, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74,
					0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f, 0x54, 0x49, 0x43,
					0x45, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x66,
					0x6f, 0x72, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65,
					0x73, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x6f, 0x20,
					0x6e, 0x6f, 0x74, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20,
					0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x61, 0x64, 0x64, 0x20,
					0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x61, 0x74, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x73, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x44, 0x65,
					0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20,
					0x61, 0x6c, 0x6f, 0x6e, 0x67, 0x73, 0x69, 0x64, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x61,
					0x73, 0x20, 0x61, 0x6e, 0x20, 0x61, 0x64, 0x64, 0x65, 0x6e, 0x64, 0x75,
					0x6d, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f, 0x54,
					0x49, 0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x72, 0x6f,
					0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c, 0x20,
					0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x61, 0x6c, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20,
					0x63, 0x61, 0x6e, 0x6e, 0x6f, 0x74, 0x20, 0x62, 0x65, 0x20, 0x63, 0x6f,
					0x6e, 0x73, 0x74, 0x72, 0x75, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x73, 0x20, 0x6d, 0x6f, 0x64,
					0x69, 0x66, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x61,
					0x64, 0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x73, 0x74,
					0x61, 0x74, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x59,
					0x6f, 0x75, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x70, 0x72, 0x6f, 0x76,
					0x69, 0x64, 0x65, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x61, 0x6c, 0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x66, 0x66, 0x65, 0x72,
					0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20,
					0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20,
					0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x59, 0x6f, 0x75,
					0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x20, 0x61,
					0x20, 0x77, 0x68, 0x6f, 0x6c, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x6f, 0x76,
					0x69, 0x64, 0x65, 0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x75, 0x73,
					0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70,
					0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73,
					0x65, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x65, 0x73, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x35, 0x2e, 0x20, 0x53, 0x75, 0x62, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x43, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e,
					0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x65, 0x78, 0x70, 0x6c, 0x69, 0x63, 0x69, 0x74, 0x6c, 0x79, 0x20, 0x73,
					0x74, 0x61, 0x74, 0x65, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69,
					0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e,
					0x61, 0x6c, 0x6c, 0x79, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x74,
					0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62,
					0x79, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x73, 0x68,
					0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20,
					0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x77, 0x69, 0x74, 0x68, 0x73, 0x74,
					0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61,
					0x62, 0x6f, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x74, 0x68, 0x69, 0x6e,
					0x67, 0x20, 0x68, 0x65, 0x72, 0x65, 0x69, 0x6e, 0x20, 0x73, 0x68, 0x61,
					0x6c, 0x6c, 0x20, 0x73, 0x75, 0x70, 0x65, 0x72, 0x73, 0x65, 0x64, 0x65,
					0x20, 0x6f, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72,
					0x6d, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x65,
					0x70, 0x61, 0x72, 0x61, 0x74, 0x65, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x6d, 0x65, 0x6e, 0x74,
					0x20, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x68, 0x61, 0x76,
					0x65, 0x20, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x65, 0x64, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x72, 0x65, 0x67, 0x61, 0x72,
					0x64, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x36, 0x2e, 0x20, 0x54, 0x72, 0x61, 0x64,
					0x65, 0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2e, 0x20, 0x54, 0x68, 0x69, 0x73,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x64, 0x6f, 0x65,
					0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x74,
					0x6f, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x72,
					0x61, 0x64, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x61,
					0x6d, 0x65, 0x73, 0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d, 0x61,
					0x72, 0x6b, 0x73, 0x2c, 0x20, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65,
					0x20, 0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x70,
					0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x61, 0x73, 0x20, 0x72, 0x65,
					0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x72,
					0x65, 0x61, 0x73, 0x6f, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x63, 0x75, 0x73, 0x74, 0x6f, 0x6d, 0x61, 0x72, 0x79, 0x20,
					0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x73, 0x63, 0x72,
					0x69, 0x62, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x69,
					0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x65,
					0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f,
					0x54, 0x49, 0x43, 0x45, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x37, 0x2e, 0x20, 0x44, 0x69, 0x73, 0x63, 0x6c, 0x61,
					0x69, 0x6d, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x57, 0x61, 0x72, 0x72,
					0x61, 0x6e, 0x74, 0x79, 0x2e, 0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73,
					0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79,
					0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20,
					0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69,
					0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x76,
					0x69, 0x64, 0x65, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x20, 0x28, 0x61, 0x6e, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x73, 0x20, 0x69, 0x74, 0x73, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x29, 0x20, 0x6f, 0x6e,
					0x20, 0x61, 0x6e, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20,
					0x42, 0x41, 0x53, 0x49, 0x53, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52,
					0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20, 0x4f, 0x46,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x65,
					0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72, 0x65, 0x73,
					0x73, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69,
					0x6d, 0x70, 0x6c, 0x69, 0x65, 0x64, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x61, 0x72, 0x72, 0x61,
					0x6e, 0x74, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x6f, 0x66, 0x20, 0x54, 0x49, 0x54, 0x4c, 0x45, 0x2c, 0x20,
					0x4e, 0x4f, 0x4e, 0x2d, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2c, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x6f,
					0x72, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x41,
					0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52,
					0x50, 0x4f, 0x53, 0x45, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x72,
					0x65, 0x20, 0x73, 0x6f, 0x6c, 0x65, 0x6c, 0x79, 0x20, 0x72, 0x65, 0x73,
					0x70, 0x6f, 0x6e, 0x73, 0x69, 0x62, 0x6c, 0x65, 0x20, 0x66, 0x6f, 0x72,
					0x20, 0x64, 0x65, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x69, 0x6e, 0x67,
					0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x70, 0x70, 0x72, 0x6f, 0x70, 0x72, 0x69, 0x61, 0x74, 0x65, 0x6e, 0x65,
					0x73, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x75, 0x73, 0x69, 0x6e, 0x67, 0x20,
					0x6f, 0x72, 0x20, 0x72, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x75, 0x6d,
					0x65, 0x20, 0x61, 0x6e, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x72, 0x69, 0x73, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x59, 0x6f,
					0x75, 0x72, 0x20, 0x65, 0x78, 0x65, 0x72, 0x63, 0x69, 0x73, 0x65, 0x20,
					0x6f, 0x66, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x38, 0x2e, 0x20, 0x4c, 0x69, 0x6d, 0x69, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x4c, 0x69, 0x61, 0x62,
					0x69, 0x6c, 0x69, 0x74, 0x79, 0x2e, 0x20, 0x49, 0x6e, 0x20, 0x6e, 0x6f,
					0x20, 0x65, 0x76, 0x65, 0x6e, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75,
					0x6e, 0x64, 0x65, 0x72, 0x20, 0x6e, 0x6f, 0x20, 0x6c, 0x65, 0x67, 0x61,
					0x6c, 0x20, 0x74, 0x68, 0x65, 0x6f, 0x72, 0x79, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20,
					0x69, 0x6e, 0x20, 0x74, 0x6f, 0x72, 0x74, 0x20, 0x28, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x65, 0x67, 0x6c, 0x69,
					0x67, 0x65, 0x6e, 0x63, 0x65, 0x29, 0x2c, 0x20, 0x63, 0x6f, 0x6e, 0x74,
					0x72, 0x61, 0x63, 0x74, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68,
					0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x75, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71,
					0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70,
					0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20,
					0x28, 0x73, 0x75, 0x63, 0x68, 0x20, 0x61, 0x73, 0x20, 0x64, 0x65, 0x6c,
					0x69, 0x62, 0x65, 0x72, 0x61, 0x74, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x67, 0x72, 0x6f, 0x73, 0x73, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x6e, 0x65, 0x67, 0x6c, 0x69, 0x67, 0x65, 0x6e, 0x74, 0x20,
					0x61, 0x63, 0x74, 0x73, 0x29, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67, 0x72,
					0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77, 0x72,
					0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x6f, 0x72, 0x20, 0x62, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x74, 0x6f, 0x20,
					0x59, 0x6f, 0x75, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x6d, 0x61,
					0x67, 0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69,
					0x6e, 0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63,
					0x74, 0x2c, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x2c,
					0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x61, 0x6c, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x69, 0x64, 0x65, 0x6e, 0x74,
					0x61, 0x6c, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x65,
					0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x64, 0x61, 0x6d,
					0x61, 0x67, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x61, 0x63, 0x74, 0x65, 0x72, 0x20, 0x61, 0x72,
					0x69, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x73, 0x20, 0x61, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x75, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6f, 0x72,
					0x20, 0x69, 0x6e, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x74,
					0x6f, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x28, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x20,
					0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x20, 0x66,
					0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x67,
					0x6f, 0x6f, 0x64, 0x77, 0x69, 0x6c, 0x6c, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x73, 0x74, 0x6f, 0x70,
					0x70, 0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x75, 0x74,
					0x65, 0x72, 0x20, 0x66, 0x61, 0x69, 0x6c, 0x75, 0x72, 0x65, 0x20, 0x6f,
					0x72, 0x20, 0x6d, 0x61, 0x6c, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x65, 0x72,
					0x63, 0x69, 0x61, 0x6c, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73,
					0x20, 0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x65, 0x73, 0x29, 0x2c,
					0x20, 0x65, 0x76, 0x65, 0x6e, 0x20, 0x69, 0x66, 0x20, 0x73, 0x75, 0x63,
					0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f,
					0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x61, 0x73, 0x20,
					0x62, 0x65, 0x65, 0x6e, 0x20, 0x61, 0x64, 0x76, 0x69, 0x73, 0x65, 0x64,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f, 0x73, 0x73,
					0x69, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x2e,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x39, 0x2e, 0x20, 0x41, 0x63, 0x63, 0x65,
					0x70, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x57, 0x61, 0x72, 0x72, 0x61, 0x6e,
					0x74, 0x79, 0x20, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x4c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69,
					0x74, 0x79, 0x2e, 0x20, 0x57, 0x68, 0x69, 0x6c, 0x65, 0x20, 0x72, 0x65,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6e, 0x67,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76,
					0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20,
					0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2c, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x6d, 0x61, 0x79, 0x20, 0x63, 0x68, 0x6f, 0x6f, 0x73, 0x65, 0x20,
					0x74, 0x6f, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x68, 0x61, 0x72,
					0x67, 0x65, 0x20, 0x61, 0x20, 0x66, 0x65, 0x65, 0x20, 0x66, 0x6f, 0x72,
					0x2c, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x61, 0x6e, 0x63, 0x65,
					0x20, 0x6f, 0x66, 0x20, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x2c,
					0x20, 0x77, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x2c, 0x20, 0x69,
					0x6e, 0x64, 0x65, 0x6d, 0x6e, 0x69, 0x74, 0x79, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x6f,
					0x62, 0x6c, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61,
					0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x63, 0x6f, 0x6e, 0x73, 0x69, 0x73, 0x74, 0x65, 0x6e, 0x74, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x69, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e,
					0x20, 0x48, 0x6f, 0x77, 0x65, 0x76, 0x65, 0x72, 0x2c, 0x20, 0x69, 0x6e,
					0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x6f, 0x62, 0x6c, 0x69, 0x67, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79,
					0x20, 0x61, 0x63, 0x74, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20,
					0x6f, 0x77, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x73,
					0x6f, 0x6c, 0x65, 0x20, 0x72, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x73, 0x69,
					0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2c, 0x20, 0x6e, 0x6f, 0x74, 0x20,
					0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x6f, 0x72, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f, 0x6e,
					0x6c, 0x79, 0x20, 0x69, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x67,
					0x72, 0x65, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x64, 0x65, 0x6d,
					0x6e, 0x69, 0x66, 0x79, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x64, 0x65, 0x66, 0x65, 0x6e, 0x64, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x68, 0x6f, 0x6c, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68, 0x61,
					0x72, 0x6d, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x75, 0x72,
					0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x63,
					0x6c, 0x61, 0x69, 0x6d, 0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74,
					0x65, 0x64, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e, 0x73, 0x74, 0x2c, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x72, 0x65, 0x61, 0x73,
					0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66, 0x20,
					0x79, 0x6f, 0x75, 0x72, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x69,
					0x6e, 0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20,
					0x77, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x20, 0x6f, 0x72, 0x20,
					0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x6c,
					0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x45, 0x4e, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x45, 0x52,
					0x4d, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49,
					0x54, 0x49, 0x4f, 0x4e, 0x53, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x41, 0x50,
					0x50, 0x45, 0x4e, 0x44, 0x49, 0x58, 0x3a, 0x20, 0x48, 0x6f, 0x77, 0x20,
					0x74, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75, 0x72, 0x20,
					0x77, 0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x54, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75, 0x72,
					0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x61, 0x74, 0x74, 0x61, 0x63,
					0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77,
					0x69, 0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x6f,
					0x69, 0x6c, 0x65, 0x72, 0x70, 0x6c, 0x61, 0x74, 0x65, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x73, 0x20, 0x65, 0x6e,
					0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x62, 0x72,
					0x61, 0x63, 0x6b, 0x65, 0x74, 0x73, 0x20, 0x22, 0x5b, 0x5d, 0x22, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x61, 0x63,
					0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x79, 0x6f, 0x75, 0x72,
					0x20, 0x6f, 0x77, 0x6e, 0x20, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66,
					0x79, 0x69, 0x6e, 0x67, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x20, 0x28, 0x44, 0x6f, 0x6e, 0x27, 0x74,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x62, 0x72, 0x61, 0x63, 0x6b,
					0x65, 0x74, 0x73, 0x21, 0x29, 0x20, 0x20, 0x54, 0x68, 0x65, 0x20, 0x74,
					0x65, 0x78, 0x74, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62,
					0x65, 0x20, 0x65, 0x6e, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x70, 0x70, 0x72, 0x6f, 0x70,
					0x72, 0x69, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x63, 0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x73, 0x79, 0x6e, 0x74,
					0x61, 0x78, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x2e, 0x20,
					0x57, 0x65, 0x20, 0x61, 0x6c, 0x73, 0x6f, 0x20, 0x72, 0x65, 0x63, 0x6f,
					0x6d, 0x6d, 0x65, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x6e, 0x61, 0x6d,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69,
					0x70, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x75, 0x72,
					0x70, 0x6f, 0x73, 0x65, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x61, 0x6d, 0x65, 0x20, 0x22,
					0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x70, 0x61, 0x67, 0x65,
					0x22, 0x20, 0x61, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x65, 0x61, 0x73, 0x69, 0x65, 0x72,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x64, 0x65, 0x6e, 0x74,
					0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x69, 0x72, 0x64, 0x2d, 0x70,
					0x61, 0x72, 0x74, 0x79, 0x20, 0x61, 0x72, 0x63, 0x68, 0x69, 0x76, 0x65,
					0x73, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x5b, 0x79, 0x79, 0x79, 0x79, 0x5d, 0x20,
					0x5b, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72,
					0x5d, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e,
					0x20, 0x32, 0x2e, 0x30, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x22, 0x29, 0x3b, 0x0a, 0x20, 0x20,
					0x20, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6e, 0x6f, 0x74,
					0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x69, 0x6e,
					0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x6d, 0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x20,
					0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x74,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x74, 0x74,
					0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x70, 0x61, 0x63,
					0x68, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x73, 0x2f, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x2d,
					0x32, 0x2e, 0x30, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x55, 0x6e, 0x6c, 0x65,
					0x73, 0x73, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20,
					0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c,
					0x65, 0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67, 0x72,
					0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77, 0x72,
					0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x20, 0x22, 0x41,
					0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53, 0x49, 0x53, 0x2c,
					0x0a, 0x20, 0x20, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53,
					0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44,
					0x2c, 0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70,
					0x72, 0x65, 0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70, 0x6c,
					0x69, 0x65, 0x64, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x53, 0x65, 0x65, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20,
					0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63,
					0x69, 0x66, 0x69, 0x63, 0x20, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67,
					0x65, 0x20, 0x67, 0x6f, 0x76, 0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c, 0x69,
					0x2f, 0x67, 0x6f, 0x75, 0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a, 0x42,
					0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x70, 0x65, 0x78, 0x2f, 0x6c, 0x6f, 0x67,
					0x0a, 0x0a, 0x0a, 0x28, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x29, 0x0a, 0x0a, 0x43, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20,
					0x32, 0x30, 0x31, 0x35, 0x20, 0x54, 0x4a, 0x20, 0x48, 0x6f, 0x6c, 0x6f,
					0x77, 0x61, 0x79, 0x63, 0x68, 0x75, 0x6b, 0x20, 0x26, 0x6c, 0x74, 0x3b,
					0x74, 0x6a, 0x40, 0x74, 0x6a, 0x68, 0x6f, 0x6c, 0x6f, 0x77, 0x61, 0x79,
					0x63, 0x68, 0x75, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x61, 0x26, 0x67, 0x74,
					0x3b, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
					0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20,
					0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65,
					0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73,
					0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67,
					0x0a, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61,
					0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20,
					0x28, 0x74, 0x68, 0x65, 0x0a, 0x27, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x27, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61,
					0x6c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67,
					0x0a, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d,
					0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x75, 0x73,
					0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64,
					0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20,
					0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62,
					0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64,
					0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70,
					0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x6f, 0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70,
					0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68,
					0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69,
					0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73,
					0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74,
					0x6f, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77,
					0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f,
					0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73,
					0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73,
					0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x20,
					0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75,
					0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f,
					0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a,
					0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45,
					0x44, 0x20, 0x27, 0x41, 0x53, 0x20, 0x49, 0x53, 0x27, 0x2c, 0x20, 0x57,
					0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b,
					0x49, 0x4e, 0x44, 0x2c, 0x0a, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c,
					0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42,
					0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54,
					0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x0a,
					0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49,
					0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53,
					0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46,
					0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x0a, 0x49,
					0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53,
					0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54,
					0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59,
					0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52,
					0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x43, 0x4c, 0x41, 0x49,
					0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54,
					0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x41, 0x43, 0x54, 0x2c, 0x0a, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52,
					0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20,
					0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d,
					0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20,
					0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f,
					0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53,
					0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x70,
					0x65, 0x78, 0x2f, 0x6c, 0x6f, 0x67, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47,
					0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x67, 0x68, 0x6f, 0x64, 0x73, 0x73, 0x2f, 0x79, 0x61, 0x6d,
					0x6c, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49, 0x54,
					0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34, 0x20, 0x53, 0x61,
					0x6d, 0x20, 0x47, 0x68, 0x6f, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68,
					0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65,
					0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74,
					0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74,
					0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d,
					0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
					0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65,
					0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73,
					0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73,
					0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62,
					0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62,
					0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74,
					0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50,
					0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20,
					0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e,
					0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e,
					0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45,
					0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48,
					0x45, 0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a,
					0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x32, 0x20, 0x54, 0x68, 0x65,
					0x20, 0x47, 0x6f, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x2e,
					0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20,
					0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52,
					0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e,
					0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73,
					0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69,
					0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72,
					0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e,
					0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x73,
					0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d,
					0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74,
					0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73,
					0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20, 0x20,
					0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e,
					0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73,
					0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x0a, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73,
					0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69,
					0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72,
					0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72,
					0x69, 0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65,
					0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e,
					0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69, 0x74, 0x68, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f,
					0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63,
					0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61,
					0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74, 0x73, 0x0a, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x73, 0x20,
					0x6d, 0x61, 0x79, 0x20, 0x62, 0x65, 0x20, 0x75, 0x73, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x73, 0x65, 0x20, 0x6f,
					0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f, 0x74, 0x65, 0x20, 0x70, 0x72,
					0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20, 0x64, 0x65, 0x72, 0x69, 0x76,
					0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x0a, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66,
					0x69, 0x63, 0x20, 0x70, 0x72, 0x69, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69,
					0x74, 0x74, 0x65, 0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73,
					0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50,
					0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52,
					0x53, 0x0a, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c,
					0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20,
					0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49,
					0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45,
					0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53,
					0x53, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54,
					0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f,
					0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x20, 0x44, 0x49, 0x53, 0x43, 0x4c,
					0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47,
					0x48, 0x54, 0x0a, 0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53,
					0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43,
					0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c,
					0x20, 0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c,
					0x0a, 0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58,
					0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20,
					0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41,
					0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49,
					0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55,
					0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45,
					0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52,
					0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42,
					0x53, 0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45,
					0x53, 0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55,
					0x53, 0x45, 0x2c, 0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52,
					0x20, 0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52,
					0x20, 0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e,
					0x54, 0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20,
					0x48, 0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x20, 0x43, 0x41, 0x55, 0x53,
					0x45, 0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e,
					0x59, 0x0a, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57,
					0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f,
					0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49,
					0x43, 0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49,
					0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47,
					0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52,
					0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54,
					0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44,
					0x56, 0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49,
					0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69,
					0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x68, 0x6f,
					0x64, 0x73, 0x73, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x0a, 0x0a, 0x0a, 0x42,
					0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x65, 0x73, 0x73, 0x65, 0x76, 0x64, 0x6b,
					0x2f, 0x67, 0x6f, 0x2d, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x0a, 0x0a, 0x0a,
					0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63,
					0x29, 0x20, 0x32, 0x30, 0x31, 0x32, 0x20, 0x4a, 0x65, 0x73, 0x73, 0x65,
					0x20, 0x76, 0x61, 0x6e, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x4b, 0x69, 0x65,
					0x62, 0x6f, 0x6f, 0x6d, 0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65,
					0x64, 0x2e, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73,
					0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66,
					0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f,
					0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f,
					0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20,
					0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65,
					0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74,
					0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c,
					0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65, 0x74,
					0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f,
					0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c,
					0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61,
					0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52,
					0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79,
					0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72,
					0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c,
					0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65,
					0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20,
					0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e,
					0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69, 0x74, 0x68, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f,
					0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63,
					0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61,
					0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74, 0x73, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x73, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65, 0x20,
					0x75, 0x73, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f,
					0x72, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f,
					0x74, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20,
					0x64, 0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63,
					0x20, 0x70, 0x72, 0x69, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74,
					0x65, 0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
					0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f,
					0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x0a,
					0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49,
					0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55,
					0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45,
					0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43,
					0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20,
					0x46, 0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43,
					0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45,
					0x20, 0x41, 0x52, 0x45, 0x20, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49,
					0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x0a, 0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f,
					0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42,
					0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c,
					0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49,
					0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x0a, 0x53,
					0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d,
					0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f,
					0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20,
					0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43,
					0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20,
					0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20,
					0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54,
					0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b,
					0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45,
					0x2c, 0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50,
					0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42,
					0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45,
					0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f,
					0x57, 0x45, 0x56, 0x45, 0x52, 0x20, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44,
					0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x0a,
					0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43,
					0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49,
					0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53,
					0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57,
					0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49,
					0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49,
					0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50,
					0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47,
					0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x65, 0x73, 0x73, 0x65,
					0x76, 0x64, 0x6b, 0x2f, 0x67, 0x6f, 0x2d, 0x66, 0x6c, 0x61, 0x67, 0x73,
					0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x6f, 0x68, 0x6f,
					0x2f, 0x67, 0x6f, 0x64, 0x6f, 0x74, 0x65, 0x6e, 0x76, 0x0a, 0x0a, 0x0a,
					0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63,
					0x29, 0x20, 0x32, 0x30, 0x31, 0x33, 0x20, 0x4a, 0x6f, 0x68, 0x6e, 0x20,
					0x42, 0x61, 0x72, 0x74, 0x6f, 0x6e, 0x0a, 0x0a, 0x4d, 0x49, 0x54, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a, 0x0a, 0x50, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68,
					0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65,
					0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74,
					0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x0a, 0x22,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74,
					0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d,
					0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
					0x68, 0x2c, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73,
					0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73,
					0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62,
					0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62,
					0x65, 0x0a, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74,
					0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50,
					0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x20, 0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x0a,
					0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x0a, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20,
					0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x4f, 0x46,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e,
					0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e,
					0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57, 0x49, 0x54, 0x48,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45,
					0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a,
					0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x6f, 0x68, 0x6f, 0x2f, 0x67, 0x6f,
					0x64, 0x6f, 0x74, 0x65, 0x6e, 0x76, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47,
					0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x6b, 0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f, 0x73, 0x2f,
					0x6f, 0x73, 0x65, 0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30,
					0x31, 0x32, 0x20, 0x54, 0x68, 0x65, 0x20, 0x47, 0x6f, 0x20, 0x41, 0x75,
					0x74, 0x68, 0x6f, 0x72, 0x73, 0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76,
					0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79,
					0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a,
					0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74,
					0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64,
					0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f,
					0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d,
					0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20,
					0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65,
					0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f,
					0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x0a, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65,
					0x72, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72,
					0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x0a, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77,
					0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d,
					0x65, 0x72, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x20, 0x74, 0x68, 0x65, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20,
					0x4e, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67,
					0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x69, 0x74, 0x73, 0x0a, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x6f, 0x72, 0x73, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65,
					0x20, 0x75, 0x73, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64,
					0x6f, 0x72, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d,
					0x6f, 0x74, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73,
					0x20, 0x64, 0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f,
					0x6d, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x70, 0x72, 0x69,
					0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a,
					0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45,
					0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50,
					0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45,
					0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x0a, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x2c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45,
					0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53,
					0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x0a,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45,
					0x20, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e,
					0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54,
					0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43,
					0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x0a, 0x4f, 0x57, 0x4e,
					0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49,
					0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44,
					0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44,
					0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x0a, 0x53, 0x50, 0x45, 0x43, 0x49,
					0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52,
					0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51,
					0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49,
					0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20,
					0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54,
					0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53,
					0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x20, 0x4c, 0x4f, 0x53,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x0a, 0x44, 0x41,
					0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49,
					0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e,
					0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50,
					0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45,
					0x52, 0x20, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x54, 0x48, 0x45, 0x4f,
					0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54,
					0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49,
					0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49,
					0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20,
					0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f,
					0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20,
					0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55,
					0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a,
					0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x6b, 0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f, 0x73,
					0x2f, 0x6f, 0x73, 0x65, 0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47,
					0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x6f, 0x6c, 0x65, 0x6b, 0x75, 0x6b, 0x6f, 0x6e, 0x6b, 0x6f,
					0x2f, 0x74, 0x73, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34,
					0x20, 0x62, 0x79, 0x20, 0x4f, 0x6c, 0x65, 0x6b, 0x75, 0x20, 0x4b, 0x6f,
					0x6e, 0x6b, 0x6f, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73,
					0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62,
					0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66,
					0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65,
					0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69,
					0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63,
					0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65,
					0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65,
					0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64,
					0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74,
					0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69,
					0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c,
					0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20,
					0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d,
					0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65,
					0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73,
					0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61,
					0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63,
					0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20,
					0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72,
					0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f,
					0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c,
					0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61,
					0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x0a, 0x61, 0x6c,
					0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20,
					0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20,
					0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49,
					0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c,
					0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52,
					0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45,
					0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45,
					0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47,
					0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d,
					0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f,
					0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45,
					0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52,
					0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50,
					0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49,
					0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e,
					0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54,
					0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41,
					0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f,
					0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44,
					0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c,
					0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41,
					0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45,
					0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52,
					0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f,
					0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54,
					0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e,
					0x47, 0x53, 0x20, 0x49, 0x4e, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x45, 0x4e, 0x44, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6f,
					0x6c, 0x65, 0x6b, 0x75, 0x6b, 0x6f, 0x6e, 0x6b, 0x6f, 0x2f, 0x74, 0x73,
					0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x67, 0x2f,
					0x65, 0x72, 0x72, 0x6f, 0x72, 0x73, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32,
					0x30, 0x31, 0x35, 0x2c, 0x20, 0x44, 0x61, 0x76, 0x65, 0x20, 0x43, 0x68,
					0x65, 0x6e, 0x65, 0x79, 0x20, 0x3c, 0x64, 0x61, 0x76, 0x65, 0x40, 0x63,
					0x68, 0x65, 0x6e, 0x65, 0x79, 0x2e, 0x6e, 0x65, 0x74, 0x3e, 0x0a, 0x41,
					0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65,
					0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73,
					0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69,
					0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76,
					0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61,
					0x72, 0x65, 0x20, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x2a, 0x20, 0x52,
					0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
					0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72,
					0x65, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x0a, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c,
					0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61,
					0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66,
					0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70,
					0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61,
					0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x0a, 0x20,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20,
					0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69,
					0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a,
					0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44,
					0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f,
					0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44,
					0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x0a, 0x41, 0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49,
					0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41,
					0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52,
					0x45, 0x0a, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44,
					0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f,
					0x4c, 0x44, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x0a, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49,
					0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43,
					0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x20, 0x53, 0x50, 0x45,
					0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c,
					0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53,
					0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x0a, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e,
					0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54,
					0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52,
					0x0a, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x20, 0x4c,
					0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20,
					0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f,
					0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53,
					0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52,
					0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45,
					0x56, 0x45, 0x52, 0x0a, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x54, 0x48,
					0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41,
					0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x4f, 0x52,
					0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45,
					0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e,
					0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59,
					0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45,
					0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53,
					0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e,
					0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x67, 0x2f, 0x65, 0x72, 0x72,
					0x6f, 0x72, 0x73, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72,
					0x64, 0x73, 0x75, 0x62, 0x68, 0x61, 0x73, 0x2f, 0x67, 0x6f, 0x2d, 0x65,
					0x6c, 0x61, 0x73, 0x74, 0x69, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x0a,
					0x0a, 0x0a, 0x4d, 0x49, 0x54, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x53, 0x75,
					0x62, 0x68, 0x61, 0x73, 0x20, 0x44, 0x61, 0x6e, 0x64, 0x61, 0x70, 0x61,
					0x6e, 0x69, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79,
					0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72,
					0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72,
					0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e,
					0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73,
					0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65,
					0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e,
					0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69,
					0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75,
					0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f,
					0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c,
					0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f,
					0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20,
					0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77,
					0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e,
					0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20,
					0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20,
					0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c,
					0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73,
					0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70,
					0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e,
					0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44,
					0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20,
					0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53,
					0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20,
					0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49,
					0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46,
					0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53,
					0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54,
					0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f,
					0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e,
					0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20,
					0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20,
					0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41, 0x55,
					0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50,
					0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45,
					0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41,
					0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c,
					0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f,
					0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49,
					0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47,
					0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72,
					0x64, 0x73, 0x75, 0x62, 0x68, 0x61, 0x73, 0x2f, 0x67, 0x6f, 0x2d, 0x65,
					0x6c, 0x61, 0x73, 0x74, 0x69, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x0a,
					0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x61, 0x74, 0x6f, 0x72,
					0x69, 0x2f, 0x67, 0x6f, 0x2e, 0x75, 0x75, 0x69, 0x64, 0x0a, 0x0a, 0x0a,
					0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43,
					0x29, 0x20, 0x32, 0x30, 0x31, 0x33, 0x2d, 0x32, 0x30, 0x31, 0x36, 0x20,
					0x62, 0x79, 0x20, 0x4d, 0x61, 0x78, 0x69, 0x6d, 0x20, 0x42, 0x75, 0x62,
					0x6c, 0x69, 0x73, 0x20, 0x3c, 0x62, 0x40, 0x63, 0x6f, 0x64, 0x65, 0x6d,
					0x6f, 0x6e, 0x6b, 0x65, 0x79, 0x2e, 0x72, 0x75, 0x3e, 0x0a, 0x0a, 0x50,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73,
					0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e,
					0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66,
					0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f,
					0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65,
					0x0a, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65,
					0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c,
					0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c,
					0x69, 0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20,
					0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x0a,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f,
					0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73,
					0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a,
					0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64,
					0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69,
					0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61,
					0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53,
					0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41,
					0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f,
					0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c,
					0x0a, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43,
					0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52, 0x43,
					0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41,
					0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e,
					0x44, 0x0a, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47,
					0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45,
					0x0a, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44,
					0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49,
					0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a,
					0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c,
					0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53,
					0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55,
					0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43,
					0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57, 0x49,
					0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55,
					0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x61, 0x74, 0x6f, 0x72, 0x69,
					0x2f, 0x67, 0x6f, 0x2e, 0x75, 0x75, 0x69, 0x64, 0x0a, 0x0a, 0x0a, 0x42,
					0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x67, 0x75, 0x6c, 0x61, 0x63, 0x73, 0x69,
					0x2f, 0x77, 0x72, 0x61, 0x70, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30,
					0x31, 0x34, 0x2c, 0x20, 0x42, 0x72, 0x61, 0x6e, 0x64, 0x6f, 0x6e, 0x20,
					0x54, 0x68, 0x6f, 0x6d, 0x73, 0x6f, 0x6e, 0x0a, 0x41, 0x6c, 0x6c, 0x20,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72,
					0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72,
					0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74,
					0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x0a, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65,
					0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20,
					0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f,
					0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x0a,
					0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65,
					0x72, 0x2e, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e,
					0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64,
					0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76,
					0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x0a, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77,
					0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d,
					0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x6f, 0x74, 0x68,
					0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73,
					0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48,
					0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20,
					0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20,
					0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52,
					0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53,
					0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42,
					0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53,
					0x22, 0x20, 0x41, 0x4e, 0x44, 0x0a, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58,
					0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49,
					0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x0a,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f,
					0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49,
					0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20,
					0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50,
					0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x0a, 0x44,
					0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49,
					0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53,
					0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50,
					0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45,
					0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42,
					0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x4e, 0x59, 0x20,
					0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49,
					0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44, 0x45,
					0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x20, 0x53, 0x50, 0x45, 0x43, 0x49, 0x41,
					0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59,
					0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55,
					0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47,
					0x45, 0x53, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c,
					0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50,
					0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54, 0x45,
					0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53, 0x45,
					0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x0a, 0x4c, 0x4f, 0x53, 0x53,
					0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x44, 0x41, 0x54,
					0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49, 0x54,
					0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e, 0x45,
					0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50, 0x54,
					0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45, 0x52,
					0x20, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x4f, 0x4e, 0x0a, 0x41, 0x4e, 0x59, 0x20, 0x54, 0x48, 0x45, 0x4f, 0x52,
					0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c,
					0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53,
					0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x49,
					0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f, 0x55,
					0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x0a, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x20,
					0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20, 0x4f,
					0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x43,
					0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45,
					0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x74, 0x67, 0x75, 0x6c, 0x61, 0x63, 0x73, 0x69, 0x2f, 0x77,
					0x72, 0x61, 0x70, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74,
					0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x67, 0x6a, 0x73, 0x6f, 0x6e,
					0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49, 0x54, 0x29,
					0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x4a, 0x6f, 0x73,
					0x68, 0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a, 0x50, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68,
					0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65,
					0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74,
					0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74,
					0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d,
					0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
					0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73,
					0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73,
					0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a, 0x73, 0x75, 0x62,
					0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62,
					0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74,
					0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50,
					0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f,
					0x52, 0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x49, 0x4e, 0x20,
					0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e,
					0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a, 0x43, 0x4f, 0x4e,
					0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45,
					0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a,
					0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f,
					0x67, 0x6a, 0x73, 0x6f, 0x6e, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49,
					0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x6d, 0x61, 0x74,
					0x63, 0x68, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49,
					0x54, 0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x4a,
					0x6f, 0x73, 0x68, 0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a, 0x50,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73,
					0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e,
					0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66,
					0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f,
					0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65,
					0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e,
					0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65,
					0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x73, 0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c,
					0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c,
					0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20,
					0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x66, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f,
					0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a, 0x73,
					0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a,
					0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64,
					0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69,
					0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61,
					0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53,
					0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41,
					0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f,
					0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c,
					0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a,
					0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43,
					0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43,
					0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41,
					0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47,
					0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53,
					0x20, 0x4f, 0x52, 0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44,
					0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x49,
					0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20,
					0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c,
					0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53,
					0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55,
					0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a, 0x43,
					0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49,
					0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55,
					0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c,
					0x6c, 0x2f, 0x6d, 0x61, 0x74, 0x63, 0x68, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e, 0x69, 0x6e,
					0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x32, 0x30, 0x31,
					0x31, 0x2d, 0x32, 0x30, 0x31, 0x36, 0x20, 0x43, 0x61, 0x6e, 0x6f, 0x6e,
					0x69, 0x63, 0x61, 0x6c, 0x20, 0x4c, 0x74, 0x64, 0x2e, 0x0a, 0x0a, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x56, 0x65,
					0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30, 0x20, 0x28, 0x74,
					0x68, 0x65, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x22,
					0x29, 0x3b, 0x0a, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6e,
					0x6f, 0x74, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20,
					0x69, 0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e, 0x63,
					0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x59, 0x6f, 0x75, 0x20,
					0x6d, 0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x61,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x74, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
					0x77, 0x77, 0x77, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x6f,
					0x72, 0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x2f,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x2d, 0x32, 0x2e, 0x30, 0x0a,
					0x0a, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71, 0x75,
					0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c,
					0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f,
					0x72, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x0a, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64,
					0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x20,
					0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53, 0x49,
					0x53, 0x2c, 0x0a, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c,
					0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72,
					0x65, 0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70, 0x6c, 0x69,
					0x65, 0x64, 0x2e, 0x0a, 0x53, 0x65, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63,
					0x20, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x20, 0x67, 0x6f,
					0x76, 0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x0a,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x70, 0x6f, 0x72, 0x74,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x47, 0x6f, 0x20, 0x66, 0x72, 0x6f,
					0x6d, 0x20, 0x43, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x6c, 0x69, 0x62, 0x79, 0x61, 0x6d, 0x6c, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x75, 0x73, 0x0a, 0x61, 0x72, 0x65, 0x20, 0x73,
					0x74, 0x69, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x76, 0x65, 0x72, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x20, 0x6f, 0x72,
					0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x70, 0x69, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65,
					0x6d, 0x69, 0x74, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x70, 0x61, 0x72, 0x73, 0x65, 0x72, 0x63, 0x2e, 0x67,
					0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x61, 0x64, 0x65, 0x72,
					0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x73, 0x63, 0x61,
					0x6e, 0x6e, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x68, 0x2e, 0x67, 0x6f,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x70, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x65, 0x68, 0x2e, 0x67, 0x6f, 0x0a, 0x0a, 0x43, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20,
					0x32, 0x30, 0x30, 0x36, 0x20, 0x4b, 0x69, 0x72, 0x69, 0x6c, 0x6c, 0x20,
					0x53, 0x69, 0x6d, 0x6f, 0x6e, 0x6f, 0x76, 0x0a, 0x0a, 0x50, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68,
					0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65,
					0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74,
					0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74,
					0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d,
					0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
					0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x0a, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73,
					0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73,
					0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x6f, 0x0a, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62,
					0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62,
					0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74,
					0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50,
					0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20,
					0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e,
					0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e,
					0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45,
					0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48,
					0x45, 0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a,
					0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e, 0x69,
					0x6e, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a, 0x0a,
					0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d,
					0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76, 0x65, 0x72, 0x0a,
					0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72,
					0x6d, 0x69, 0x6e, 0x64, 0x73, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34,
					0x2d, 0x32, 0x30, 0x31, 0x35, 0x2c, 0x20, 0x4d, 0x61, 0x74, 0x74, 0x20,
					0x42, 0x75, 0x74, 0x63, 0x68, 0x65, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x4d, 0x61, 0x74, 0x74, 0x20, 0x46, 0x61, 0x72, 0x69, 0x6e, 0x61, 0x0a,
					0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72,
					0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20,
					0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74,
					0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66,
					0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75,
					0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f,
					0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72,
					0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c,
					0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e,
					0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
					0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61,
					0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x65, 0x64, 0x20, 0x69, 0x6e, 0x0a, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f,
					0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73,
					0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54,
					0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20,
					0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20,
					0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54,
					0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e,
					0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f,
					0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49,
					0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54,
					0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44,
					0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45,
					0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55,
					0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49,
					0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20,
					0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41,
					0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f,
					0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49,
					0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20,
					0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c,
					0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f,
					0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43,
					0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52,
					0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a,
					0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20,
					0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49,
					0x4e, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74,
					0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76,
					0x65, 0x72, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61,
					0x73, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x70,
					0x72, 0x69, 0x67, 0x0a, 0x0a, 0x0a, 0x53, 0x70, 0x72, 0x69, 0x67, 0x0a,
					0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43,
					0x29, 0x20, 0x32, 0x30, 0x31, 0x33, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65,
					0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64,
					0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68,
					0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61,
					0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73,
					0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
					0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65,
					0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68,
					0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c,
					0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a,
					0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a,
					0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x0a, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69,
					0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58,
					0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49,
					0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20,
					0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50,
					0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e,
					0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48,
					0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41,
					0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e,
					0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41,
					0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x0a, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a,
					0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e,
					0x64, 0x73, 0x2f, 0x73, 0x70, 0x72, 0x69, 0x67, 0x0a, 0x0a, 0x0a, 0x42,
					0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c, 0x69, 0x2f, 0x67,
					0x6f, 0x75, 0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30,
					0x2c, 0x20, 0x4a, 0x61, 0x6e, 0x75, 0x61, 0x72, 0x79, 0x20, 0x32, 0x30,
					0x30, 0x34, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77,
					0x77, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x6f, 0x72, 0x67,
					0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x2f, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x54, 0x45, 0x52, 0x4d, 0x53, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x52, 0x45, 0x50,
					0x52, 0x4f, 0x44, 0x55, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x2c, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x44, 0x49, 0x53, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54,
					0x49, 0x4f, 0x4e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x31, 0x2e, 0x20, 0x44,
					0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d,
					0x65, 0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d,
					0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65,
					0x2c, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x61, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x65,
					0x64, 0x20, 0x62, 0x79, 0x20, 0x53, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x31, 0x20, 0x74, 0x68, 0x72, 0x6f, 0x75, 0x67, 0x68, 0x20,
					0x39, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72,
					0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x6f, 0x72, 0x20,
					0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f,
					0x72, 0x69, 0x7a, 0x65, 0x64, 0x20, 0x62, 0x79, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x74,
					0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x22, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74, 0x69, 0x74,
					0x79, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x6e, 0x69, 0x6f, 0x6e, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x63, 0x74, 0x69, 0x6e,
					0x67, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x69, 0x65,
					0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72,
					0x6f, 0x6c, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74,
					0x72, 0x6f, 0x6c, 0x6c, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f,
					0x72, 0x20, 0x61, 0x72, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20,
					0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74,
					0x79, 0x2e, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70,
					0x75, 0x72, 0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x22, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x73, 0x20, 0x28, 0x69, 0x29, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f,
					0x77, 0x65, 0x72, 0x2c, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x20,
					0x6f, 0x72, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68,
					0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x69, 0x72, 0x65,
					0x63, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6e,
					0x61, 0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2c, 0x20,
					0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x61, 0x63, 0x74, 0x20, 0x6f, 0x72, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69,
					0x73, 0x65, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x29, 0x20,
					0x6f, 0x77, 0x6e, 0x65, 0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66,
					0x20, 0x66, 0x69, 0x66, 0x74, 0x79, 0x20, 0x70, 0x65, 0x72, 0x63, 0x65,
					0x6e, 0x74, 0x20, 0x28, 0x35, 0x30, 0x25, 0x29, 0x20, 0x6f, 0x72, 0x20,
					0x6d, 0x6f, 0x72, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61,
					0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x68, 0x61, 0x72, 0x65, 0x73,
					0x2c, 0x20, 0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x69, 0x29, 0x20, 0x62,
					0x65, 0x6e, 0x65, 0x66, 0x69, 0x63, 0x69, 0x61, 0x6c, 0x20, 0x6f, 0x77,
					0x6e, 0x65, 0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x22,
					0x20, 0x28, 0x6f, 0x72, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x72, 0x22, 0x29,
					0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20,
					0x61, 0x6e, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61,
					0x6c, 0x20, 0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45,
					0x6e, 0x74, 0x69, 0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x65, 0x78, 0x65, 0x72, 0x63, 0x69, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x67,
					0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x53, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x22, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x73, 0x68, 0x61,
					0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x70, 0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6b, 0x69, 0x6e,
					0x67, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65,
					0x2c, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f,
					0x6e, 0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x22, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x22, 0x20, 0x66,
					0x6f, 0x72, 0x6d, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65,
					0x61, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20,
					0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x72,
					0x6f, 0x6d, 0x20, 0x6d, 0x65, 0x63, 0x68, 0x61, 0x6e, 0x69, 0x63, 0x61,
					0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x72, 0x61, 0x6e,
					0x73, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f,
					0x72, 0x20, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69,
					0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x69,
					0x6c, 0x65, 0x64, 0x20, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x63,
					0x6f, 0x64, 0x65, 0x2c, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x74,
					0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x73, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x20, 0x74, 0x79, 0x70, 0x65, 0x73,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x57, 0x6f,
					0x72, 0x6b, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65,
					0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20,
					0x6f, 0x66, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x68, 0x69,
					0x70, 0x2c, 0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69,
					0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x6f, 0x72, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74,
					0x20, 0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x6d, 0x61, 0x64, 0x65, 0x20,
					0x61, 0x76, 0x61, 0x69, 0x6c, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x75, 0x6e,
					0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x73, 0x20, 0x69, 0x6e, 0x64, 0x69,
					0x63, 0x61, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x74, 0x68,
					0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x74, 0x74,
					0x61, 0x63, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x77, 0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x28, 0x61, 0x6e, 0x20, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x20,
					0x69, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x70, 0x65, 0x6e,
					0x64, 0x69, 0x78, 0x20, 0x62, 0x65, 0x6c, 0x6f, 0x77, 0x29, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x77,
					0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65,
					0x63, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x2c, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x62,
					0x61, 0x73, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x28, 0x6f, 0x72, 0x20,
					0x64, 0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d,
					0x29, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x77, 0x68, 0x69, 0x63, 0x68,
					0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65,
					0x64, 0x69, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x72, 0x65, 0x76,
					0x69, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x61, 0x6e, 0x6e, 0x6f,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x65, 0x6c, 0x61,
					0x62, 0x6f, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6f,
					0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69,
					0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e,
					0x74, 0x2c, 0x20, 0x61, 0x73, 0x20, 0x61, 0x20, 0x77, 0x68, 0x6f, 0x6c,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e,
					0x61, 0x6c, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61,
					0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x68, 0x69, 0x70, 0x2e, 0x20, 0x46,
					0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f,
					0x73, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x65, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x72, 0x65, 0x6d, 0x61, 0x69, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x73, 0x65, 0x70, 0x61, 0x72, 0x61, 0x62, 0x6c, 0x65, 0x20,
					0x66, 0x72, 0x6f, 0x6d, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x65, 0x72,
					0x65, 0x6c, 0x79, 0x20, 0x6c, 0x69, 0x6e, 0x6b, 0x20, 0x28, 0x6f, 0x72,
					0x20, 0x62, 0x69, 0x6e, 0x64, 0x20, 0x62, 0x79, 0x20, 0x6e, 0x61, 0x6d,
					0x65, 0x29, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x69, 0x6e,
					0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x2c,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77,
					0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f,
					0x72, 0x73, 0x68, 0x69, 0x70, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x69, 0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20,
					0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x64, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68,
					0x65, 0x72, 0x65, 0x6f, 0x66, 0x2c, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x69, 0x73, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e,
					0x61, 0x6c, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73,
					0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x62,
					0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x61, 0x6e,
					0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c, 0x20,
					0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74,
					0x69, 0x74, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74,
					0x20, 0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f,
					0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77,
					0x6e, 0x65, 0x72, 0x2e, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x22, 0x73, 0x75, 0x62, 0x6d, 0x69,
					0x74, 0x74, 0x65, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x6d, 0x65, 0x61, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x20, 0x6f, 0x66, 0x20, 0x65, 0x6c, 0x65, 0x63, 0x74, 0x72,
					0x6f, 0x6e, 0x69, 0x63, 0x2c, 0x20, 0x76, 0x65, 0x72, 0x62, 0x61, 0x6c,
					0x2c, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e,
					0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x73, 0x65, 0x6e, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x74, 0x73,
					0x20, 0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x20,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69,
					0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x6e, 0x20, 0x65, 0x6c,
					0x65, 0x63, 0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63, 0x20, 0x6d, 0x61, 0x69,
					0x6c, 0x69, 0x6e, 0x67, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x73, 0x2c, 0x20,
					0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20,
					0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x73, 0x79, 0x73, 0x74,
					0x65, 0x6d, 0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x20, 0x74, 0x72, 0x61,
					0x63, 0x6b, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
					0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d,
					0x61, 0x6e, 0x61, 0x67, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f,
					0x72, 0x20, 0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20,
					0x6f, 0x66, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66,
					0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f,
					0x73, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x64, 0x69, 0x73, 0x63, 0x75, 0x73,
					0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x69, 0x6d, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x63,
					0x6f, 0x6e, 0x73, 0x70, 0x69, 0x63, 0x75, 0x6f, 0x75, 0x73, 0x6c, 0x79,
					0x20, 0x6d, 0x61, 0x72, 0x6b, 0x65, 0x64, 0x20, 0x6f, 0x72, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x64, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67,
					0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20,
					0x61, 0x73, 0x20, 0x22, 0x4e, 0x6f, 0x74, 0x20, 0x61, 0x20, 0x43, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x22,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x22, 0x20, 0x73, 0x68,
					0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c,
					0x20, 0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e,
					0x74, 0x69, 0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f,
					0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f, 0x66, 0x20,
					0x77, 0x68, 0x6f, 0x6d, 0x20, 0x61, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x68, 0x61, 0x73, 0x20,
					0x62, 0x65, 0x65, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76, 0x65,
					0x64, 0x20, 0x62, 0x79, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f,
					0x72, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x73, 0x75, 0x62, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x6c, 0x79,
					0x20, 0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x32,
					0x2e, 0x20, 0x47, 0x72, 0x61, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63,
					0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72,
					0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x20, 0x70, 0x65, 0x72,
					0x70, 0x65, 0x74, 0x75, 0x61, 0x6c, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x6f, 0x72, 0x6c, 0x64, 0x77, 0x69, 0x64, 0x65, 0x2c,
					0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x73, 0x69,
					0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x2d, 0x63, 0x68, 0x61, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x72, 0x6f, 0x79, 0x61, 0x6c, 0x74, 0x79, 0x2d, 0x66,
					0x72, 0x65, 0x65, 0x2c, 0x20, 0x69, 0x72, 0x72, 0x65, 0x76, 0x6f, 0x63,
					0x61, 0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x72, 0x65, 0x70, 0x72,
					0x6f, 0x64, 0x75, 0x63, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x65, 0x70, 0x61,
					0x72, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x6f, 0x66, 0x2c, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x63,
					0x6c, 0x79, 0x20, 0x64, 0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x2c, 0x20,
					0x70, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x6c, 0x79, 0x20, 0x70, 0x65, 0x72,
					0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x20, 0x74, 0x68, 0x65,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x44, 0x65, 0x72,
					0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x73, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x33, 0x2e, 0x20, 0x47,
					0x72, 0x61, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x50, 0x61, 0x74, 0x65,
					0x6e, 0x74, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20,
					0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x65,
					0x61, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67,
					0x72, 0x61, 0x6e, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x61, 0x20, 0x70, 0x65, 0x72, 0x70, 0x65, 0x74, 0x75, 0x61, 0x6c,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x6f, 0x72, 0x6c,
					0x64, 0x77, 0x69, 0x64, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x65,
					0x78, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f,
					0x2d, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x72, 0x6f, 0x79,
					0x61, 0x6c, 0x74, 0x79, 0x2d, 0x66, 0x72, 0x65, 0x65, 0x2c, 0x20, 0x69,
					0x72, 0x72, 0x65, 0x76, 0x6f, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74,
					0x20, 0x61, 0x73, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x65, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x29, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61,
					0x6b, 0x65, 0x2c, 0x20, 0x68, 0x61, 0x76, 0x65, 0x20, 0x6d, 0x61, 0x64,
					0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x75, 0x73, 0x65,
					0x2c, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x20, 0x74, 0x6f, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x2c, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x2c, 0x20, 0x69,
					0x6d, 0x70, 0x6f, 0x72, 0x74, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x20, 0x74, 0x72, 0x61,
					0x6e, 0x73, 0x66, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68,
					0x65, 0x72, 0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x6f,
					0x73, 0x65, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x73, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x61,
					0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x79,
					0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61,
					0x72, 0x65, 0x20, 0x6e, 0x65, 0x63, 0x65, 0x73, 0x73, 0x61, 0x72, 0x69,
					0x6c, 0x79, 0x20, 0x69, 0x6e, 0x66, 0x72, 0x69, 0x6e, 0x67, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x61, 0x6c, 0x6f, 0x6e,
					0x65, 0x20, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63, 0x6f, 0x6d, 0x62,
					0x69, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x69, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x69, 0x63,
					0x68, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x77,
					0x61, 0x73, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64,
					0x2e, 0x20, 0x49, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x69, 0x6e, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x65,
					0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x74, 0x69,
					0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e,
					0x73, 0x74, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74,
					0x79, 0x20, 0x28, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67,
					0x20, 0x61, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x72, 0x6f,
					0x73, 0x73, 0x2d, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x20, 0x6f, 0x72, 0x20,
					0x63, 0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x6c, 0x61, 0x69, 0x6d,
					0x20, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x6c, 0x61, 0x77, 0x73, 0x75, 0x69,
					0x74, 0x29, 0x20, 0x61, 0x6c, 0x6c, 0x65, 0x67, 0x69, 0x6e, 0x67, 0x20,
					0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x61,
					0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74,
					0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x74,
					0x69, 0x74, 0x75, 0x74, 0x65, 0x73, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63,
					0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x79, 0x20,
					0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e, 0x66, 0x72, 0x69,
					0x6e, 0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x2c, 0x20, 0x74, 0x68, 0x65,
					0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74,
					0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x74, 0x65,
					0x72, 0x6d, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x61, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x64, 0x61, 0x74, 0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69,
					0x74, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x34,
					0x2e, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61,
					0x79, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76,
					0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20,
					0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x20, 0x69, 0x6e, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x6d, 0x65, 0x64, 0x69, 0x75, 0x6d, 0x2c, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x6f, 0x64,
					0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x20, 0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x65, 0x65, 0x74, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x61, 0x29, 0x20,
					0x59, 0x6f, 0x75, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x67, 0x69, 0x76,
					0x65, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20,
					0x72, 0x65, 0x63, 0x69, 0x70, 0x69, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f,
					0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x73, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x28, 0x62, 0x29, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x65, 0x64, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x61, 0x72,
					0x72, 0x79, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x69, 0x6e, 0x65, 0x6e, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x74, 0x61, 0x74, 0x69,
					0x6e, 0x67, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x63, 0x29, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69,
					0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x2c, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2c, 0x20, 0x70, 0x61, 0x74, 0x65,
					0x6e, 0x74, 0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d, 0x61, 0x72,
					0x6b, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73,
					0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x78, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61,
					0x74, 0x20, 0x64, 0x6f, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x70, 0x65, 0x72,
					0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65,
					0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x73, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x28, 0x64, 0x29, 0x20, 0x49, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x65, 0x73, 0x20, 0x61, 0x20, 0x22, 0x4e, 0x4f, 0x54, 0x49, 0x43,
					0x45, 0x22, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x69, 0x6c, 0x65,
					0x20, 0x61, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x69, 0x74, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x6e, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59,
					0x6f, 0x75, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x20, 0x61, 0x20, 0x72, 0x65, 0x61, 0x64, 0x61, 0x62, 0x6c, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x63, 0x6f, 0x6e, 0x74,
					0x61, 0x69, 0x6e, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x2c, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69,
					0x6e, 0x67, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f,
					0x20, 0x6e, 0x6f, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x70, 0x65, 0x72, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74,
					0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x2c, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x73, 0x74, 0x20,
					0x6f, 0x6e, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x6c, 0x61, 0x63, 0x65,
					0x73, 0x3a, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20,
					0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x61, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f,
					0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x69, 0x66, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x64, 0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20,
					0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x64,
					0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x69, 0x66, 0x20, 0x61, 0x6e, 0x64,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77,
					0x68, 0x65, 0x72, 0x65, 0x76, 0x65, 0x72, 0x20, 0x73, 0x75, 0x63, 0x68,
					0x20, 0x74, 0x68, 0x69, 0x72, 0x64, 0x2d, 0x70, 0x61, 0x72, 0x74, 0x79,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x6e, 0x6f, 0x72,
					0x6d, 0x61, 0x6c, 0x6c, 0x79, 0x20, 0x61, 0x70, 0x70, 0x65, 0x61, 0x72,
					0x2e, 0x20, 0x54, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e,
					0x74, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f, 0x54, 0x49,
					0x43, 0x45, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x61, 0x72, 0x65, 0x20,
					0x66, 0x6f, 0x72, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73,
					0x65, 0x73, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x61, 0x6e, 0x64, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x6f,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x61, 0x64, 0x64,
					0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x61, 0x74,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x73, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x44,
					0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c,
					0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67, 0x73, 0x69, 0x64, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20,
					0x61, 0x73, 0x20, 0x61, 0x6e, 0x20, 0x61, 0x64, 0x64, 0x65, 0x6e, 0x64,
					0x75, 0x6d, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f,
					0x54, 0x49, 0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x72,
					0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c,
					0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73,
					0x20, 0x63, 0x61, 0x6e, 0x6e, 0x6f, 0x74, 0x20, 0x62, 0x65, 0x20, 0x63,
					0x6f, 0x6e, 0x73, 0x74, 0x72, 0x75, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x73, 0x20, 0x6d, 0x6f,
					0x64, 0x69, 0x66, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20,
					0x61, 0x64, 0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x73,
					0x74, 0x61, 0x74, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x6f, 0x20,
					0x59, 0x6f, 0x75, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x61, 0x6c, 0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x66, 0x66, 0x65,
					0x72, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65, 0x2c,
					0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x59, 0x6f,
					0x75, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x20,
					0x61, 0x20, 0x77, 0x68, 0x6f, 0x6c, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x75,
					0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65,
					0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69,
					0x73, 0x65, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x65, 0x73, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x35, 0x2e, 0x20, 0x53, 0x75, 0x62,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x43,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x2e, 0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x65, 0x78, 0x70, 0x6c, 0x69, 0x63, 0x69, 0x74, 0x6c, 0x79, 0x20,
					0x73, 0x74, 0x61, 0x74, 0x65, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77,
					0x69, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f,
					0x6e, 0x61, 0x6c, 0x6c, 0x79, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74,
					0x74, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x62, 0x79, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x73,
					0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c,
					0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x77, 0x69, 0x74, 0x68, 0x73,
					0x74, 0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x74, 0x68, 0x69,
					0x6e, 0x67, 0x20, 0x68, 0x65, 0x72, 0x65, 0x69, 0x6e, 0x20, 0x73, 0x68,
					0x61, 0x6c, 0x6c, 0x20, 0x73, 0x75, 0x70, 0x65, 0x72, 0x73, 0x65, 0x64,
					0x65, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65,
					0x72, 0x6d, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73,
					0x65, 0x70, 0x61, 0x72, 0x61, 0x74, 0x65, 0x20, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x6d, 0x65, 0x6e,
					0x74, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x68, 0x61,
					0x76, 0x65, 0x20, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x65, 0x64, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x72, 0x65, 0x67, 0x61,
					0x72, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x36, 0x2e, 0x20, 0x54, 0x72, 0x61,
					0x64, 0x65, 0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2e, 0x20, 0x54, 0x68, 0x69,
					0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x64, 0x6f,
					0x65, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74,
					0x72, 0x61, 0x64, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e,
					0x61, 0x6d, 0x65, 0x73, 0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d,
					0x61, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63,
					0x65, 0x20, 0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x20,
					0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x20, 0x6e, 0x61, 0x6d, 0x65,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x61, 0x73, 0x20, 0x72,
					0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x72, 0x65, 0x61, 0x73, 0x6f, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x63, 0x75, 0x73, 0x74, 0x6f, 0x6d, 0x61, 0x72, 0x79,
					0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x73, 0x63,
					0x72, 0x69, 0x62, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63,
					0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74,
					0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e,
					0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x37, 0x2e, 0x20, 0x44, 0x69, 0x73, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x57, 0x61, 0x72,
					0x72, 0x61, 0x6e, 0x74, 0x79, 0x2e, 0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73,
					0x73, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62,
					0x79, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65,
					0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x28, 0x61, 0x6e, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69,
					0x64, 0x65, 0x73, 0x20, 0x69, 0x74, 0x73, 0x20, 0x43, 0x6f, 0x6e, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x29, 0x20, 0x6f,
					0x6e, 0x20, 0x61, 0x6e, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x20, 0x42, 0x41, 0x53, 0x49, 0x53, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72, 0x65,
					0x73, 0x73, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x69, 0x6d, 0x70, 0x6c, 0x69, 0x65, 0x64, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x61, 0x72, 0x72,
					0x61, 0x6e, 0x74, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x54, 0x49, 0x54, 0x4c, 0x45, 0x2c,
					0x20, 0x4e, 0x4f, 0x4e, 0x2d, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47,
					0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2c, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x6f, 0x72, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x41, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50,
					0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55,
					0x52, 0x50, 0x4f, 0x53, 0x45, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61,
					0x72, 0x65, 0x20, 0x73, 0x6f, 0x6c, 0x65, 0x6c, 0x79, 0x20, 0x72, 0x65,
					0x73, 0x70, 0x6f, 0x6e, 0x73, 0x69, 0x62, 0x6c, 0x65, 0x20, 0x66, 0x6f,
					0x72, 0x20, 0x64, 0x65, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x69, 0x6e,
					0x67, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x61, 0x70, 0x70, 0x72, 0x6f, 0x70, 0x72, 0x69, 0x61, 0x74, 0x65, 0x6e,
					0x65, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x75, 0x73, 0x69, 0x6e, 0x67,
					0x20, 0x6f, 0x72, 0x20, 0x72, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x75,
					0x6d, 0x65, 0x20, 0x61, 0x6e, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x72, 0x69, 0x73, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63,
					0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x59,
					0x6f, 0x75, 0x72, 0x20, 0x65, 0x78, 0x65, 0x72, 0x63, 0x69, 0x73, 0x65,
					0x20, 0x6f, 0x66, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x38, 0x2e, 0x20, 0x4c, 0x69, 0x6d, 0x69, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x4c, 0x69, 0x61,
					0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2e, 0x20, 0x49, 0x6e, 0x20, 0x6e,
					0x6f, 0x20, 0x65, 0x76, 0x65, 0x6e, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x6e, 0x6f, 0x20, 0x6c, 0x65, 0x67,
					0x61, 0x6c, 0x20, 0x74, 0x68, 0x65, 0x6f, 0x72, 0x79, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x69, 0x6e, 0x20, 0x74, 0x6f, 0x72, 0x74, 0x20, 0x28, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x65, 0x67, 0x6c,
					0x69, 0x67, 0x65, 0x6e, 0x63, 0x65, 0x29, 0x2c, 0x20, 0x63, 0x6f, 0x6e,
					0x74, 0x72, 0x61, 0x63, 0x74, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74,
					0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x75, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65,
					0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70,
					0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77,
					0x20, 0x28, 0x73, 0x75, 0x63, 0x68, 0x20, 0x61, 0x73, 0x20, 0x64, 0x65,
					0x6c, 0x69, 0x62, 0x65, 0x72, 0x61, 0x74, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x67, 0x72, 0x6f, 0x73, 0x73, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6e, 0x65, 0x67, 0x6c, 0x69, 0x67, 0x65, 0x6e, 0x74,
					0x20, 0x61, 0x63, 0x74, 0x73, 0x29, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67,
					0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77,
					0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x62, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x74, 0x6f,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x6d,
					0x61, 0x67, 0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x64, 0x69, 0x72, 0x65,
					0x63, 0x74, 0x2c, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74,
					0x2c, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x61, 0x6c, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x69, 0x64, 0x65, 0x6e,
					0x74, 0x61, 0x6c, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x73,
					0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x64, 0x61,
					0x6d, 0x61, 0x67, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x63, 0x68, 0x61, 0x72, 0x61, 0x63, 0x74, 0x65, 0x72, 0x20, 0x61,
					0x72, 0x69, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x73, 0x20, 0x61, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x75, 0x74, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6f,
					0x72, 0x20, 0x69, 0x6e, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x28, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x20,
					0x66, 0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x67, 0x6f, 0x6f, 0x64, 0x77, 0x69, 0x6c, 0x6c, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x73, 0x74, 0x6f,
					0x70, 0x70, 0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x75,
					0x74, 0x65, 0x72, 0x20, 0x66, 0x61, 0x69, 0x6c, 0x75, 0x72, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6c, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x65,
					0x72, 0x63, 0x69, 0x61, 0x6c, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x65, 0x73, 0x29,
					0x2c, 0x20, 0x65, 0x76, 0x65, 0x6e, 0x20, 0x69, 0x66, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x61, 0x73,
					0x20, 0x62, 0x65, 0x65, 0x6e, 0x20, 0x61, 0x64, 0x76, 0x69, 0x73, 0x65,
					0x64, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f, 0x73,
					0x73, 0x69, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x6f, 0x66, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x39, 0x2e, 0x20, 0x41, 0x63, 0x63,
					0x65, 0x70, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x57, 0x61, 0x72, 0x72, 0x61,
					0x6e, 0x74, 0x79, 0x20, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x4c, 0x69, 0x61, 0x62, 0x69, 0x6c,
					0x69, 0x74, 0x79, 0x2e, 0x20, 0x57, 0x68, 0x69, 0x6c, 0x65, 0x20, 0x72,
					0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6e,
					0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2c, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x63, 0x68, 0x6f, 0x6f, 0x73, 0x65,
					0x20, 0x74, 0x6f, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x68, 0x61,
					0x72, 0x67, 0x65, 0x20, 0x61, 0x20, 0x66, 0x65, 0x65, 0x20, 0x66, 0x6f,
					0x72, 0x2c, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x61, 0x6e, 0x63,
					0x65, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74,
					0x2c, 0x20, 0x77, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x2c, 0x20,
					0x69, 0x6e, 0x64, 0x65, 0x6d, 0x6e, 0x69, 0x74, 0x79, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65,
					0x72, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20,
					0x6f, 0x62, 0x6c, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x69, 0x73, 0x74, 0x65, 0x6e, 0x74,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x69, 0x73, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2e, 0x20, 0x48, 0x6f, 0x77, 0x65, 0x76, 0x65, 0x72, 0x2c, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x69, 0x6e, 0x67, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x6f, 0x62, 0x6c, 0x69, 0x67, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61,
					0x79, 0x20, 0x61, 0x63, 0x74, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72,
					0x20, 0x6f, 0x77, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20,
					0x73, 0x6f, 0x6c, 0x65, 0x20, 0x72, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x73,
					0x69, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2c, 0x20, 0x6e, 0x6f, 0x74,
					0x20, 0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f,
					0x6e, 0x6c, 0x79, 0x20, 0x69, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61,
					0x67, 0x72, 0x65, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x64, 0x65,
					0x6d, 0x6e, 0x69, 0x66, 0x79, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x64, 0x65, 0x66, 0x65, 0x6e, 0x64, 0x2c, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x68, 0x6f, 0x6c, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68,
					0x61, 0x72, 0x6d, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74,
					0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x75,
					0x72, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72, 0x20,
					0x63, 0x6c, 0x61, 0x69, 0x6d, 0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72,
					0x74, 0x65, 0x64, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e, 0x73, 0x74, 0x2c,
					0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x72, 0x65, 0x61,
					0x73, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66,
					0x20, 0x79, 0x6f, 0x75, 0x72, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74,
					0x69, 0x6e, 0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x75, 0x63, 0x68,
					0x20, 0x77, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x20, 0x6f, 0x72,
					0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20,
					0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x45, 0x4e, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x45,
					0x52, 0x4d, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x44,
					0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x41,
					0x50, 0x50, 0x45, 0x4e, 0x44, 0x49, 0x58, 0x3a, 0x20, 0x48, 0x6f, 0x77,
					0x20, 0x74, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75, 0x72,
					0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x54, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75,
					0x72, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x61, 0x74, 0x74, 0x61,
					0x63, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62,
					0x6f, 0x69, 0x6c, 0x65, 0x72, 0x70, 0x6c, 0x61, 0x74, 0x65, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x73, 0x20, 0x65,
					0x6e, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x62,
					0x72, 0x61, 0x63, 0x6b, 0x65, 0x74, 0x73, 0x20, 0x22, 0x5b, 0x5d, 0x22,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x61,
					0x63, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x79, 0x6f, 0x75,
					0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69,
					0x66, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x20, 0x28, 0x44, 0x6f, 0x6e, 0x27,
					0x74, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x62, 0x72, 0x61, 0x63,
					0x6b, 0x65, 0x74, 0x73, 0x21, 0x29, 0x20, 0x20, 0x54, 0x68, 0x65, 0x20,
					0x74, 0x65, 0x78, 0x74, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20,
					0x62, 0x65, 0x20, 0x65, 0x6e, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x70, 0x70, 0x72, 0x6f,
					0x70, 0x72, 0x69, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x73, 0x79, 0x6e,
					0x74, 0x61, 0x78, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x2e,
					0x20, 0x57, 0x65, 0x20, 0x61, 0x6c, 0x73, 0x6f, 0x20, 0x72, 0x65, 0x63,
					0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x61, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x65,
					0x20, 0x6f, 0x72, 0x20, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x6e, 0x61,
					0x6d, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x65, 0x73, 0x63, 0x72,
					0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x75,
					0x72, 0x70, 0x6f, 0x73, 0x65, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x61, 0x6d, 0x65, 0x20,
					0x22, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x70, 0x61, 0x67,
					0x65, 0x22, 0x20, 0x61, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x65, 0x61, 0x73, 0x69, 0x65,
					0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x64, 0x65, 0x6e,
					0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x69, 0x72, 0x64, 0x2d,
					0x70, 0x61, 0x72, 0x74, 0x79, 0x20, 0x61, 0x72, 0x63, 0x68, 0x69, 0x76,
					0x65, 0x73, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x5b, 0x79, 0x79, 0x79, 0x79, 0x5d,
					0x20, 0x5b, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65,
					0x72, 0x5d, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f,
					0x6e, 0x20, 0x32, 0x2e, 0x30, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x22, 0x29, 0x3b, 0x0a, 0x20,
					0x20, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6e, 0x6f,
					0x74, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x69,
					0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e,
					0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61,
					0x74, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x74,
					0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x70, 0x61,
					0x63, 0x68, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x73, 0x2f, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x2d, 0x32, 0x2e, 0x30, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x55, 0x6e, 0x6c,
					0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62,
					0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67,
					0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77,
					0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x20, 0x22,
					0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53, 0x49, 0x53,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e,
					0x44, 0x2c, 0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78,
					0x70, 0x72, 0x65, 0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70,
					0x6c, 0x69, 0x65, 0x64, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x53, 0x65, 0x65,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65,
					0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61,
					0x67, 0x65, 0x20, 0x67, 0x6f, 0x76, 0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x6c, 0x69, 0x6d, 0x69,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c,
					0x69, 0x2f, 0x67, 0x6f, 0x75, 0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a,
					0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x70, 0x65, 0x78, 0x2f, 0x6c, 0x6f,
					0x67, 0x0a, 0x0a, 0x0a, 0x28, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x29, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29,
					0x20, 0x32, 0x30, 0x31, 0x35, 0x20, 0x54, 0x4a, 0x20, 0x48, 0x6f, 0x6c,
					0x6f, 0x77, 0x61, 0x79, 0x63, 0x68, 0x75, 0x6b, 0x20, 0x26, 0x6c, 0x74,
					0x3b, 0x74, 0x6a, 0x40, 0x74, 0x6a, 0x68, 0x6f, 0x6c, 0x6f, 0x77, 0x61,
					0x79, 0x63, 0x68, 0x75, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x61, 0x26, 0x67,
					0x74, 0x3b, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79,
					0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72,
					0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72,
					0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e,
					0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73,
					0x20, 0x28, 0x74, 0x68, 0x65, 0x0a, 0x27, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x27, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65,
					0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e,
					0x67, 0x0a, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69,
					0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x75,
					0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f,
					0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c,
					0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f,
					0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20,
					0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77,
					0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e,
					0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20,
					0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20,
					0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c,
					0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73,
					0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70,
					0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e,
					0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44,
					0x45, 0x44, 0x20, 0x27, 0x41, 0x53, 0x20, 0x49, 0x53, 0x27, 0x2c, 0x20,
					0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20,
					0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49,
					0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46,
					0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53,
					0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54,
					0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f,
					0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e,
					0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x0a,
					0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20,
					0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55,
					0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50,
					0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45,
					0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x43, 0x4c, 0x41,
					0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x41, 0x43, 0x54, 0x2c, 0x0a, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c,
					0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f,
					0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49,
					0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x0a,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47,
					0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61,
					0x70, 0x65, 0x78, 0x2f, 0x6c, 0x6f, 0x67, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x67, 0x68, 0x6f, 0x64, 0x73, 0x73, 0x2f, 0x79, 0x61,
					0x6d, 0x6c, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49,
					0x54, 0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34, 0x20, 0x53,
					0x61, 0x6d, 0x20, 0x47, 0x68, 0x6f, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62,
					0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20,
					0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73,
					0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20,
					0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69,
					0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69,
					0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e,
					0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55,
					0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f,
					0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f,
					0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54,
					0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44,
					0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54,
					0x48, 0x45, 0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e,
					0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x32, 0x20, 0x54, 0x68,
					0x65, 0x20, 0x47, 0x6f, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73,
					0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a,
					0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69,
					0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66,
					0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73,
					0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69,
					0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20,
					0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69,
					0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75,
					0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x0a, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69,
					0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64,
					0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f,
					0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65,
					0x72, 0x69, 0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x0a,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69, 0x74, 0x68,
					0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20,
					0x6f, 0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e,
					0x63, 0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e,
					0x61, 0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74, 0x73, 0x0a,
					0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x73,
					0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65, 0x20, 0x75, 0x73, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x73, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f, 0x74, 0x65, 0x20, 0x70,
					0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20, 0x64, 0x65, 0x72, 0x69,
					0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x0a, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69,
					0x66, 0x69, 0x63, 0x20, 0x70, 0x72, 0x69, 0x6f, 0x72, 0x20, 0x77, 0x72,
					0x69, 0x74, 0x74, 0x65, 0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f,
					0x52, 0x53, 0x0a, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45,
					0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45,
					0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c,
					0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d,
					0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52,
					0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d,
					0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45,
					0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50, 0x41, 0x52,
					0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50,
					0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x20, 0x44, 0x49, 0x53, 0x43,
					0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e,
					0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c,
					0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49,
					0x47, 0x48, 0x54, 0x0a, 0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52,
					0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45,
					0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c,
					0x2c, 0x0a, 0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45,
					0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49,
					0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28,
					0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42,
					0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54,
					0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55,
					0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55,
					0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f,
					0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43,
					0x45, 0x53, 0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20,
					0x55, 0x53, 0x45, 0x2c, 0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f,
					0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f,
					0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49,
					0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29,
					0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x20, 0x43, 0x41, 0x55,
					0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41,
					0x4e, 0x59, 0x0a, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52,
					0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28,
					0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45,
					0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41,
					0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20,
					0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41,
					0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x68,
					0x6f, 0x64, 0x73, 0x73, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x0a, 0x0a, 0x0a,
					0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x6f, 0x68, 0x6f, 0x2f, 0x67, 0x6f,
					0x64, 0x6f, 0x74, 0x65, 0x6e, 0x76, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32,
					0x30, 0x31, 0x33, 0x20, 0x4a, 0x6f, 0x68, 0x6e, 0x20, 0x42, 0x61, 0x72,
					0x74, 0x6f, 0x6e, 0x0a, 0x0a, 0x4d, 0x49, 0x54, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65,
					0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20,
					0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72,
					0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70,
					0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e,
					0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f,
					0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d,
					0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x0a, 0x22, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x74, 0x6f,
					0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20,
					0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x0a,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20,
					0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20,
					0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f,
					0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20, 0x66, 0x75,
					0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64,
					0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63,
					0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x0a, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72,
					0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c,
					0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56,
					0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a, 0x45, 0x58, 0x50, 0x52,
					0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49,
					0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49,
					0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e,
					0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41,
					0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52,
					0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x0a, 0x4e, 0x4f, 0x4e,
					0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54,
					0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c,
					0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x0a, 0x4c, 0x49, 0x41, 0x42,
					0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43,
					0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48,
					0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20,
					0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x4f, 0x46, 0x20, 0x43, 0x4f,
					0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53,
					0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46,
					0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20,
					0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f,
					0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52,
					0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49,
					0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x0a, 0x45,
					0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x6a, 0x6f, 0x68, 0x6f, 0x2f, 0x67, 0x6f, 0x64, 0x6f, 0x74,
					0x65, 0x6e, 0x76, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6b,
					0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f, 0x73, 0x2f, 0x6f, 0x73, 0x65,
					0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x32, 0x20,
					0x54, 0x68, 0x65, 0x20, 0x47, 0x6f, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f,
					0x72, 0x73, 0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e,
					0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65,
					0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64,
					0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61,
					0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64,
					0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68,
					0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65, 0x74, 0x3a,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64,
					0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c,
					0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a,
					0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20,
					0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75,
					0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65,
					0x0a, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d,
					0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64,
					0x2f, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61,
					0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76,
					0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68,
					0x65, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d,
					0x65, 0x20, 0x6f, 0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20,
					0x49, 0x6e, 0x63, 0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x6e, 0x61, 0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74,
					0x73, 0x0a, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f,
					0x72, 0x73, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65, 0x20, 0x75, 0x73,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x73,
					0x65, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f, 0x74, 0x65,
					0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20, 0x64, 0x65,
					0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x0a, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x73, 0x70, 0x65,
					0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x70, 0x72, 0x69, 0x6f, 0x72, 0x20,
					0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49,
					0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49,
					0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42,
					0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49,
					0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55,
					0x54, 0x4f, 0x52, 0x53, 0x0a, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x20, 0x41, 0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50,
					0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c,
					0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c,
					0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46,
					0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54,
					0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50,
					0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55,
					0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x20, 0x44, 0x49,
					0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e,
					0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48,
					0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59,
					0x52, 0x49, 0x47, 0x48, 0x54, 0x0a, 0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54,
					0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49,
					0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45,
					0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54,
					0x41, 0x4c, 0x2c, 0x0a, 0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c,
					0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e,
					0x54, 0x49, 0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53,
					0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c,
					0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d,
					0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f,
					0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20,
					0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47,
					0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53, 0x45, 0x52, 0x56,
					0x49, 0x43, 0x45, 0x53, 0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f,
					0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c,
					0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b,
					0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53,
					0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f,
					0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x20, 0x43,
					0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e,
					0x20, 0x41, 0x4e, 0x59, 0x0a, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53,
					0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54,
					0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20,
					0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29,
					0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f,
					0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46,
					0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20,
					0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44,
					0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
					0x6b, 0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f, 0x73, 0x2f, 0x6f, 0x73,
					0x65, 0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70,
					0x6b, 0x67, 0x2f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x73, 0x0a, 0x0a, 0x0a,
					0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63,
					0x29, 0x20, 0x32, 0x30, 0x31, 0x35, 0x2c, 0x20, 0x44, 0x61, 0x76, 0x65,
					0x20, 0x43, 0x68, 0x65, 0x6e, 0x65, 0x79, 0x20, 0x3c, 0x64, 0x61, 0x76,
					0x65, 0x40, 0x63, 0x68, 0x65, 0x6e, 0x65, 0x79, 0x2e, 0x6e, 0x65, 0x74,
					0x3e, 0x0a, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a,
					0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69,
					0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66,
					0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a,
					0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75,
					0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73,
					0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x0a, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74,
					0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73,
					0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x0a, 0x2a, 0x20,
					0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72,
					0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20,
					0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x2c, 0x0a, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73,
					0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69,
					0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x2f,
					0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74,
					0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69,
					0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f,
					0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20,
					0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x0a, 0x41, 0x4e, 0x44, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49,
					0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55,
					0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45,
					0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43,
					0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43,
					0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45,
					0x20, 0x41, 0x52, 0x45, 0x0a, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49,
					0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20,
					0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x0a, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54,
					0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20,
					0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x20,
					0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45,
					0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c,
					0x0a, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54,
					0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44,
					0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53,
					0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53,
					0x20, 0x4f, 0x52, 0x0a, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53,
					0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53,
					0x45, 0x2c, 0x20, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20,
					0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20,
					0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54,
					0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48,
					0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x0a, 0x43, 0x41, 0x55, 0x53, 0x45,
					0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48,
					0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43,
					0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x0a, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x28, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c,
					0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49,
					0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48,
					0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56,
					0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x67, 0x2f,
					0x65, 0x72, 0x72, 0x6f, 0x72, 0x73, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47,
					0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x73, 0x61, 0x74, 0x6f, 0x72, 0x69, 0x2f, 0x67, 0x6f, 0x2e,
					0x75, 0x75, 0x69, 0x64, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31,
					0x33, 0x2d, 0x32, 0x30, 0x31, 0x36, 0x20, 0x62, 0x79, 0x20, 0x4d, 0x61,
					0x78, 0x69, 0x6d, 0x20, 0x42, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x20, 0x3c,
					0x62, 0x40, 0x63, 0x6f, 0x64, 0x65, 0x6d, 0x6f, 0x6e, 0x6b, 0x65, 0x79,
					0x2e, 0x72, 0x75, 0x3e, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65,
					0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20,
					0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72,
					0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70,
					0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e,
					0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f,
					0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d,
					0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x0a, 0x22, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x74, 0x6f,
					0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20,
					0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x0a,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20,
					0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20,
					0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f,
					0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20, 0x66, 0x75,
					0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64,
					0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63,
					0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x0a, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72,
					0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c,
					0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56,
					0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a, 0x45, 0x58, 0x50, 0x52,
					0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49,
					0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49,
					0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e,
					0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41,
					0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52,
					0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x0a, 0x4e, 0x4f, 0x4e,
					0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54,
					0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c,
					0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x0a, 0x4c, 0x49, 0x41, 0x42,
					0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43,
					0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48,
					0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20,
					0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x4f, 0x46, 0x20, 0x43, 0x4f,
					0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53,
					0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46,
					0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20,
					0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f,
					0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52,
					0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49,
					0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e,
					0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x73, 0x61, 0x74, 0x6f, 0x72, 0x69, 0x2f, 0x67, 0x6f, 0x2e, 0x75,
					0x75, 0x69, 0x64, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74,
					0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x67, 0x6a, 0x73, 0x6f, 0x6e,
					0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49, 0x54, 0x29,
					0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x4a, 0x6f, 0x73,
					0x68, 0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a, 0x50, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68,
					0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65,
					0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74,
					0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74,
					0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d,
					0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
					0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73,
					0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73,
					0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a, 0x73, 0x75, 0x62,
					0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62,
					0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74,
					0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50,
					0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f,
					0x52, 0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x49, 0x4e, 0x20,
					0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e,
					0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a, 0x43, 0x4f, 0x4e,
					0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45,
					0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a,
					0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f,
					0x67, 0x6a, 0x73, 0x6f, 0x6e, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49,
					0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d,
					0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x6d, 0x61, 0x74,
					0x63, 0x68, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49,
					0x54, 0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x4a,
					0x6f, 0x73, 0x68, 0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a, 0x50,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73,
					0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e,
					0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66,
					0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f,
					0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65,
					0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e,
					0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65,
					0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x73, 0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c,
					0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c,
					0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20,
					0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x66, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f,
					0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a, 0x73,
					0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a,
					0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c,
					0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64,
					0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69,
					0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61,
					0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53,
					0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41,
					0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f,
					0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c,
					0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a,
					0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43,
					0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43,
					0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41,
					0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47,
					0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53,
					0x20, 0x4f, 0x52, 0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44,
					0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x49,
					0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20,
					0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c,
					0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53,
					0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55,
					0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a, 0x43,
					0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49,
					0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55,
					0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c,
					0x6c, 0x2f, 0x6d, 0x61, 0x74, 0x63, 0x68, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e, 0x69, 0x6e,
					0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x32, 0x30, 0x31,
					0x31, 0x2d, 0x32, 0x30, 0x31, 0x36, 0x20, 0x43, 0x61, 0x6e, 0x6f, 0x6e,
					0x69, 0x63, 0x61, 0x6c, 0x20, 0x4c, 0x74, 0x64, 0x2e, 0x0a, 0x0a, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x56, 0x65,
					0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30, 0x20, 0x28, 0x74,
					0x68, 0x65, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x22,
					0x29, 0x3b, 0x0a, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6e,
					0x6f, 0x74, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20,
					0x69, 0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e, 0x63,
					0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x59, 0x6f, 0x75, 0x20,
					0x6d, 0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x61,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x74, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
					0x77, 0x77, 0x77, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x6f,
					0x72, 0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x2f,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x2d, 0x32, 0x2e, 0x30, 0x0a,
					0x0a, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71, 0x75,
					0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c,
					0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f,
					0x72, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x0a, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64,
					0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x20,
					0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53, 0x49,
					0x53, 0x2c, 0x0a, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c,
					0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72,
					0x65, 0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70, 0x6c, 0x69,
					0x65, 0x64, 0x2e, 0x0a, 0x53, 0x65, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63,
					0x20, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x20, 0x67, 0x6f,
					0x76, 0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x0a,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x70, 0x6f, 0x72, 0x74,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x47, 0x6f, 0x20, 0x66, 0x72, 0x6f,
					0x6d, 0x20, 0x43, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x6c, 0x69, 0x62, 0x79, 0x61, 0x6d, 0x6c, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x75, 0x73, 0x0a, 0x61, 0x72, 0x65, 0x20, 0x73,
					0x74, 0x69, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x76, 0x65, 0x72, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x20, 0x6f, 0x72,
					0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x70, 0x69, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65,
					0x6d, 0x69, 0x74, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x70, 0x61, 0x72, 0x73, 0x65, 0x72, 0x63, 0x2e, 0x67,
					0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x61, 0x64, 0x65, 0x72,
					0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x73, 0x63, 0x61,
					0x6e, 0x6e, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x68, 0x2e, 0x67, 0x6f,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x70, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x65, 0x68, 0x2e, 0x67, 0x6f, 0x0a, 0x0a, 0x43, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20,
					0x32, 0x30, 0x30, 0x36, 0x20, 0x4b, 0x69, 0x72, 0x69, 0x6c, 0x6c, 0x20,
					0x53, 0x69, 0x6d, 0x6f, 0x6e, 0x6f, 0x76, 0x0a, 0x0a, 0x50, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68,
					0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65,
					0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74,
					0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61,
					0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74,
					0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d,
					0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
					0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x0a, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73,
					0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73,
					0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x6f, 0x0a, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62,
					0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54,
					0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62,
					0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74,
					0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50,
					0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20,
					0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55,
					0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d,
					0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54,
					0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20,
					0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e,
					0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e,
					0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45,
					0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48,
					0x45, 0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a,
					0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e, 0x69,
					0x6e, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a, 0x0a,
					0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d,
					0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76, 0x65, 0x72, 0x0a,
					0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72,
					0x6d, 0x69, 0x6e, 0x64, 0x73, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34,
					0x2d, 0x32, 0x30, 0x31, 0x35, 0x2c, 0x20, 0x4d, 0x61, 0x74, 0x74, 0x20,
					0x42, 0x75, 0x74, 0x63, 0x68, 0x65, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x4d, 0x61, 0x74, 0x74, 0x20, 0x46, 0x61, 0x72, 0x69, 0x6e, 0x61, 0x0a,
					0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72,
					0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20,
					0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74,
					0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66,
					0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75,
					0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f,
					0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74,
					0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72,
					0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c,
					0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e,
					0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
					0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61,
					0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x65, 0x64, 0x20, 0x69, 0x6e, 0x0a, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f,
					0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73,
					0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54,
					0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20,
					0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20,
					0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54,
					0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e,
					0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f,
					0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49,
					0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54,
					0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44,
					0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45,
					0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55,
					0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49,
					0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20,
					0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41,
					0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f,
					0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49,
					0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20,
					0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c,
					0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f,
					0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43,
					0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52,
					0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a,
					0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20,
					0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49,
					0x4e, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74,
					0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x65, 0x6d, 0x76,
					0x65, 0x72, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x4d, 0x61,
					0x73, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x2f, 0x73, 0x70,
					0x72, 0x69, 0x67, 0x0a, 0x0a, 0x0a, 0x53, 0x70, 0x72, 0x69, 0x67, 0x0a,
					0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43,
					0x29, 0x20, 0x32, 0x30, 0x31, 0x33, 0x20, 0x4d, 0x61, 0x73, 0x74, 0x65,
					0x72, 0x6d, 0x69, 0x6e, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64,
					0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68,
					0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61,
					0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73,
					0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63,
					0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74,
					0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
					0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65,
					0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68,
					0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
					0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c,
					0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a,
					0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a,
					0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e,
					0x0a, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20,
					0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69,
					0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58,
					0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50,
					0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49,
					0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20,
					0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50,
					0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e,
					0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56,
					0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48,
					0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41,
					0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f,
					0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f,
					0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e,
					0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41,
					0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x0a, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a,
					0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x4d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e,
					0x64, 0x73, 0x2f, 0x73, 0x70, 0x72, 0x69, 0x67, 0x0a, 0x0a, 0x0a, 0x42,
					0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c, 0x69, 0x2f, 0x67,
					0x6f, 0x75, 0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30,
					0x2c, 0x20, 0x4a, 0x61, 0x6e, 0x75, 0x61, 0x72, 0x79, 0x20, 0x32, 0x30,
					0x30, 0x34, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77,
					0x77, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x6f, 0x72, 0x67,
					0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x2f, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x54, 0x45, 0x52, 0x4d, 0x53, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x52, 0x45, 0x50,
					0x52, 0x4f, 0x44, 0x55, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x2c, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x44, 0x49, 0x53, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54,
					0x49, 0x4f, 0x4e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x31, 0x2e, 0x20, 0x44,
					0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d,
					0x65, 0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d,
					0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65,
					0x2c, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x61, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x65,
					0x64, 0x20, 0x62, 0x79, 0x20, 0x53, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x31, 0x20, 0x74, 0x68, 0x72, 0x6f, 0x75, 0x67, 0x68, 0x20,
					0x39, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x64, 0x6f,
					0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72,
					0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x6f, 0x72, 0x20,
					0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f,
					0x72, 0x69, 0x7a, 0x65, 0x64, 0x20, 0x62, 0x79, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20, 0x74,
					0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x22, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74, 0x69, 0x74,
					0x79, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x6e, 0x69, 0x6f, 0x6e, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x63, 0x74, 0x69, 0x6e,
					0x67, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x69, 0x65,
					0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72,
					0x6f, 0x6c, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74,
					0x72, 0x6f, 0x6c, 0x6c, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f,
					0x72, 0x20, 0x61, 0x72, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20,
					0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74,
					0x79, 0x2e, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70,
					0x75, 0x72, 0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x22, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x73, 0x20, 0x28, 0x69, 0x29, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f,
					0x77, 0x65, 0x72, 0x2c, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x20,
					0x6f, 0x72, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68,
					0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x69, 0x72, 0x65,
					0x63, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6e,
					0x61, 0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2c, 0x20,
					0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x61, 0x63, 0x74, 0x20, 0x6f, 0x72, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69,
					0x73, 0x65, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x29, 0x20,
					0x6f, 0x77, 0x6e, 0x65, 0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66,
					0x20, 0x66, 0x69, 0x66, 0x74, 0x79, 0x20, 0x70, 0x65, 0x72, 0x63, 0x65,
					0x6e, 0x74, 0x20, 0x28, 0x35, 0x30, 0x25, 0x29, 0x20, 0x6f, 0x72, 0x20,
					0x6d, 0x6f, 0x72, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61,
					0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x68, 0x61, 0x72, 0x65, 0x73,
					0x2c, 0x20, 0x6f, 0x72, 0x20, 0x28, 0x69, 0x69, 0x69, 0x29, 0x20, 0x62,
					0x65, 0x6e, 0x65, 0x66, 0x69, 0x63, 0x69, 0x61, 0x6c, 0x20, 0x6f, 0x77,
					0x6e, 0x65, 0x72, 0x73, 0x68, 0x69, 0x70, 0x20, 0x6f, 0x66, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x22,
					0x20, 0x28, 0x6f, 0x72, 0x20, 0x22, 0x59, 0x6f, 0x75, 0x72, 0x22, 0x29,
					0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20,
					0x61, 0x6e, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61,
					0x6c, 0x20, 0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45,
					0x6e, 0x74, 0x69, 0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x65, 0x78, 0x65, 0x72, 0x63, 0x69, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x67,
					0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x53, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x22, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x73, 0x68, 0x61,
					0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x70, 0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6b, 0x69, 0x6e,
					0x67, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65,
					0x2c, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f,
					0x6e, 0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x22, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x22, 0x20, 0x66,
					0x6f, 0x72, 0x6d, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65,
					0x61, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20,
					0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x72,
					0x6f, 0x6d, 0x20, 0x6d, 0x65, 0x63, 0x68, 0x61, 0x6e, 0x69, 0x63, 0x61,
					0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x72, 0x61, 0x6e,
					0x73, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f,
					0x72, 0x20, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69,
					0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x69,
					0x6c, 0x65, 0x64, 0x20, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x63,
					0x6f, 0x64, 0x65, 0x2c, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x74,
					0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x73, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x20, 0x74, 0x79, 0x70, 0x65, 0x73,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x57, 0x6f,
					0x72, 0x6b, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65,
					0x61, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20,
					0x6f, 0x66, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x68, 0x69,
					0x70, 0x2c, 0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69,
					0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x6f, 0x72, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74,
					0x20, 0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x6d, 0x61, 0x64, 0x65, 0x20,
					0x61, 0x76, 0x61, 0x69, 0x6c, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x75, 0x6e,
					0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x73, 0x20, 0x69, 0x6e, 0x64, 0x69,
					0x63, 0x61, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x74, 0x68,
					0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x74, 0x74,
					0x61, 0x63, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x77, 0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x28, 0x61, 0x6e, 0x20, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x20,
					0x69, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x70, 0x65, 0x6e,
					0x64, 0x69, 0x78, 0x20, 0x62, 0x65, 0x6c, 0x6f, 0x77, 0x29, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x22, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x77,
					0x68, 0x65, 0x74, 0x68, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65,
					0x63, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x2c, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x62,
					0x61, 0x73, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x28, 0x6f, 0x72, 0x20,
					0x64, 0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d,
					0x29, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x77, 0x68, 0x69, 0x63, 0x68,
					0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65,
					0x64, 0x69, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x72, 0x65, 0x76,
					0x69, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x61, 0x6e, 0x6e, 0x6f,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x65, 0x6c, 0x61,
					0x62, 0x6f, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6f,
					0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69,
					0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e,
					0x74, 0x2c, 0x20, 0x61, 0x73, 0x20, 0x61, 0x20, 0x77, 0x68, 0x6f, 0x6c,
					0x65, 0x2c, 0x20, 0x61, 0x6e, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e,
					0x61, 0x6c, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61,
					0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x68, 0x69, 0x70, 0x2e, 0x20, 0x46,
					0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f,
					0x73, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2c, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x65, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x72, 0x65, 0x6d, 0x61, 0x69, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x73, 0x65, 0x70, 0x61, 0x72, 0x61, 0x62, 0x6c, 0x65, 0x20,
					0x66, 0x72, 0x6f, 0x6d, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x65, 0x72,
					0x65, 0x6c, 0x79, 0x20, 0x6c, 0x69, 0x6e, 0x6b, 0x20, 0x28, 0x6f, 0x72,
					0x20, 0x62, 0x69, 0x6e, 0x64, 0x20, 0x62, 0x79, 0x20, 0x6e, 0x61, 0x6d,
					0x65, 0x29, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x69, 0x6e,
					0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x2c,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2e, 0x0a, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77,
					0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f,
					0x72, 0x73, 0x68, 0x69, 0x70, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x69, 0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20,
					0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x64, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68,
					0x65, 0x72, 0x65, 0x6f, 0x66, 0x2c, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x69, 0x73, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e,
					0x61, 0x6c, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73,
					0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72,
					0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x62,
					0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x61, 0x6e,
					0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c, 0x20,
					0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e, 0x74,
					0x69, 0x74, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a,
					0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74,
					0x20, 0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f,
					0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77,
					0x6e, 0x65, 0x72, 0x2e, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x64, 0x65, 0x66, 0x69, 0x6e, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x22, 0x73, 0x75, 0x62, 0x6d, 0x69,
					0x74, 0x74, 0x65, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x6d, 0x65, 0x61, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x20, 0x6f, 0x66, 0x20, 0x65, 0x6c, 0x65, 0x63, 0x74, 0x72,
					0x6f, 0x6e, 0x69, 0x63, 0x2c, 0x20, 0x76, 0x65, 0x72, 0x62, 0x61, 0x6c,
					0x2c, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e,
					0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x73, 0x65, 0x6e, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x74, 0x73,
					0x20, 0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x20,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69,
					0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x6e, 0x20, 0x65, 0x6c,
					0x65, 0x63, 0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63, 0x20, 0x6d, 0x61, 0x69,
					0x6c, 0x69, 0x6e, 0x67, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x73, 0x2c, 0x20,
					0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20,
					0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x73, 0x79, 0x73, 0x74,
					0x65, 0x6d, 0x73, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x69, 0x73, 0x73, 0x75, 0x65, 0x20, 0x74, 0x72, 0x61,
					0x63, 0x6b, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
					0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d,
					0x61, 0x6e, 0x61, 0x67, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f,
					0x72, 0x20, 0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20,
					0x6f, 0x66, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x66,
					0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f,
					0x73, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x64, 0x69, 0x73, 0x63, 0x75, 0x73,
					0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x69, 0x6d, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x62, 0x75, 0x74, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73, 0x20, 0x63,
					0x6f, 0x6e, 0x73, 0x70, 0x69, 0x63, 0x75, 0x6f, 0x75, 0x73, 0x6c, 0x79,
					0x20, 0x6d, 0x61, 0x72, 0x6b, 0x65, 0x64, 0x20, 0x6f, 0x72, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x64, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67,
					0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65, 0x72, 0x20,
					0x61, 0x73, 0x20, 0x22, 0x4e, 0x6f, 0x74, 0x20, 0x61, 0x20, 0x43, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x22,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x22, 0x20, 0x73, 0x68,
					0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x76, 0x69, 0x64, 0x75, 0x61, 0x6c,
					0x20, 0x6f, 0x72, 0x20, 0x4c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x45, 0x6e,
					0x74, 0x69, 0x74, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f,
					0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20, 0x6f, 0x66, 0x20,
					0x77, 0x68, 0x6f, 0x6d, 0x20, 0x61, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x68, 0x61, 0x73, 0x20,
					0x62, 0x65, 0x65, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76, 0x65,
					0x64, 0x20, 0x62, 0x79, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f,
					0x72, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x73, 0x75, 0x62, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x6c, 0x79,
					0x20, 0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65,
					0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x32,
					0x2e, 0x20, 0x47, 0x72, 0x61, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63,
					0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72,
					0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x2c, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43, 0x6f,
					0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68, 0x65,
					0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x73, 0x20,
					0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61, 0x20, 0x70, 0x65, 0x72,
					0x70, 0x65, 0x74, 0x75, 0x61, 0x6c, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x6f, 0x72, 0x6c, 0x64, 0x77, 0x69, 0x64, 0x65, 0x2c,
					0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x73, 0x69,
					0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x2d, 0x63, 0x68, 0x61, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x72, 0x6f, 0x79, 0x61, 0x6c, 0x74, 0x79, 0x2d, 0x66,
					0x72, 0x65, 0x65, 0x2c, 0x20, 0x69, 0x72, 0x72, 0x65, 0x76, 0x6f, 0x63,
					0x61, 0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x72, 0x65, 0x70, 0x72,
					0x6f, 0x64, 0x75, 0x63, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x65, 0x70, 0x61,
					0x72, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x6f, 0x66, 0x2c, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x63,
					0x6c, 0x79, 0x20, 0x64, 0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x2c, 0x20,
					0x70, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x6c, 0x79, 0x20, 0x70, 0x65, 0x72,
					0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x20, 0x74, 0x68, 0x65,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x44, 0x65, 0x72,
					0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b,
					0x73, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x66, 0x6f,
					0x72, 0x6d, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x33, 0x2e, 0x20, 0x47,
					0x72, 0x61, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x50, 0x61, 0x74, 0x65,
					0x6e, 0x74, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x20,
					0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x65,
					0x61, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x6f, 0x72, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67,
					0x72, 0x61, 0x6e, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x61, 0x20, 0x70, 0x65, 0x72, 0x70, 0x65, 0x74, 0x75, 0x61, 0x6c,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x6f, 0x72, 0x6c,
					0x64, 0x77, 0x69, 0x64, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x65,
					0x78, 0x63, 0x6c, 0x75, 0x73, 0x69, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f,
					0x2d, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x72, 0x6f, 0x79,
					0x61, 0x6c, 0x74, 0x79, 0x2d, 0x66, 0x72, 0x65, 0x65, 0x2c, 0x20, 0x69,
					0x72, 0x72, 0x65, 0x76, 0x6f, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74,
					0x20, 0x61, 0x73, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x65, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x29, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61,
					0x6b, 0x65, 0x2c, 0x20, 0x68, 0x61, 0x76, 0x65, 0x20, 0x6d, 0x61, 0x64,
					0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x75, 0x73, 0x65,
					0x2c, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x20, 0x74, 0x6f, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x2c, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x2c, 0x20, 0x69,
					0x6d, 0x70, 0x6f, 0x72, 0x74, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f,
					0x74, 0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x20, 0x74, 0x72, 0x61,
					0x6e, 0x73, 0x66, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68,
					0x65, 0x72, 0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x6f,
					0x73, 0x65, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x73, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x61,
					0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x79,
					0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x61,
					0x72, 0x65, 0x20, 0x6e, 0x65, 0x63, 0x65, 0x73, 0x73, 0x61, 0x72, 0x69,
					0x6c, 0x79, 0x20, 0x69, 0x6e, 0x66, 0x72, 0x69, 0x6e, 0x67, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x61, 0x6c, 0x6f, 0x6e,
					0x65, 0x20, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x63, 0x6f, 0x6d, 0x62,
					0x69, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x69, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x69, 0x63,
					0x68, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x28, 0x73, 0x29, 0x20, 0x77,
					0x61, 0x73, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64,
					0x2e, 0x20, 0x49, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x69, 0x6e, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x65,
					0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x74, 0x69,
					0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e,
					0x73, 0x74, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x65, 0x6e, 0x74, 0x69, 0x74,
					0x79, 0x20, 0x28, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67,
					0x20, 0x61, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x72, 0x6f,
					0x73, 0x73, 0x2d, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x20, 0x6f, 0x72, 0x20,
					0x63, 0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x6c, 0x61, 0x69, 0x6d,
					0x20, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x6c, 0x61, 0x77, 0x73, 0x75, 0x69,
					0x74, 0x29, 0x20, 0x61, 0x6c, 0x6c, 0x65, 0x67, 0x69, 0x6e, 0x67, 0x20,
					0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x61,
					0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74,
					0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x74,
					0x69, 0x74, 0x75, 0x74, 0x65, 0x73, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63,
					0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x63,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x79, 0x20,
					0x70, 0x61, 0x74, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e, 0x66, 0x72, 0x69,
					0x6e, 0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x2c, 0x20, 0x74, 0x68, 0x65,
					0x6e, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x74, 0x65, 0x6e, 0x74,
					0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x74, 0x65,
					0x72, 0x6d, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x61, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x64, 0x61, 0x74, 0x65, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x6c, 0x69,
					0x74, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x34,
					0x2e, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61,
					0x79, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76,
					0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20,
					0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x20, 0x69, 0x6e, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x6d, 0x65, 0x64, 0x69, 0x75, 0x6d, 0x2c, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x6f, 0x64,
					0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x20, 0x6f, 0x72, 0x20, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x2c, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x65, 0x65, 0x74, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x61, 0x29, 0x20,
					0x59, 0x6f, 0x75, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x67, 0x69, 0x76,
					0x65, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20,
					0x72, 0x65, 0x63, 0x69, 0x70, 0x69, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f,
					0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x73, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x28, 0x62, 0x29, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x65, 0x64, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x61, 0x72,
					0x72, 0x79, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x69, 0x6e, 0x65, 0x6e, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x74, 0x61, 0x74, 0x69,
					0x6e, 0x67, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75, 0x20,
					0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x73, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x63, 0x29, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69,
					0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74,
					0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x2c, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2c, 0x20, 0x70, 0x61, 0x74, 0x65,
					0x6e, 0x74, 0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d, 0x61, 0x72,
					0x6b, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73,
					0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x75, 0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x78, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61,
					0x74, 0x20, 0x64, 0x6f, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x70, 0x65, 0x72,
					0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65,
					0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72,
					0x6b, 0x73, 0x3b, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x28, 0x64, 0x29, 0x20, 0x49, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75,
					0x64, 0x65, 0x73, 0x20, 0x61, 0x20, 0x22, 0x4e, 0x4f, 0x54, 0x49, 0x43,
					0x45, 0x22, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x69, 0x6c, 0x65,
					0x20, 0x61, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x69, 0x74, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x6e, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59,
					0x6f, 0x75, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x20, 0x61, 0x20, 0x72, 0x65, 0x61, 0x64, 0x61, 0x62, 0x6c, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x63, 0x6f, 0x6e, 0x74,
					0x61, 0x69, 0x6e, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x2c, 0x20, 0x65, 0x78, 0x63, 0x6c, 0x75, 0x64, 0x69,
					0x6e, 0x67, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f,
					0x20, 0x6e, 0x6f, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x70, 0x65, 0x72, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74,
					0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x2c, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x73, 0x74, 0x20,
					0x6f, 0x6e, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x6c, 0x61, 0x63, 0x65,
					0x73, 0x3a, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20,
					0x4e, 0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x61, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6f,
					0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x69, 0x66, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x64, 0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61,
					0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x3b, 0x20,
					0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x61, 0x20, 0x64,
					0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x69, 0x66, 0x20, 0x61, 0x6e, 0x64,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77,
					0x68, 0x65, 0x72, 0x65, 0x76, 0x65, 0x72, 0x20, 0x73, 0x75, 0x63, 0x68,
					0x20, 0x74, 0x68, 0x69, 0x72, 0x64, 0x2d, 0x70, 0x61, 0x72, 0x74, 0x79,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73, 0x20, 0x6e, 0x6f, 0x72,
					0x6d, 0x61, 0x6c, 0x6c, 0x79, 0x20, 0x61, 0x70, 0x70, 0x65, 0x61, 0x72,
					0x2e, 0x20, 0x54, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e,
					0x74, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f, 0x54, 0x49,
					0x43, 0x45, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x61, 0x72, 0x65, 0x20,
					0x66, 0x6f, 0x72, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x70, 0x75, 0x72, 0x70, 0x6f, 0x73,
					0x65, 0x73, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x61, 0x6e, 0x64, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x6f,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x61, 0x64, 0x64,
					0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x61, 0x74,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x73, 0x20, 0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x44,
					0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c,
					0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67, 0x73, 0x69, 0x64, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20,
					0x61, 0x73, 0x20, 0x61, 0x6e, 0x20, 0x61, 0x64, 0x64, 0x65, 0x6e, 0x64,
					0x75, 0x6d, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e, 0x4f,
					0x54, 0x49, 0x43, 0x45, 0x20, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x72,
					0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x2c,
					0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x73,
					0x20, 0x63, 0x61, 0x6e, 0x6e, 0x6f, 0x74, 0x20, 0x62, 0x65, 0x20, 0x63,
					0x6f, 0x6e, 0x73, 0x74, 0x72, 0x75, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x73, 0x20, 0x6d, 0x6f,
					0x64, 0x69, 0x66, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20,
					0x61, 0x64, 0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x6f, 0x77, 0x6e,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x73,
					0x74, 0x61, 0x74, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x6f, 0x20,
					0x59, 0x6f, 0x75, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x61, 0x6c, 0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x66, 0x66, 0x65,
					0x72, 0x65, 0x6e, 0x74, 0x20, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x75, 0x73, 0x65, 0x2c,
					0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x59, 0x6f,
					0x75, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73,
					0x75, 0x63, 0x68, 0x20, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69,
					0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x20,
					0x61, 0x20, 0x77, 0x68, 0x6f, 0x6c, 0x65, 0x2c, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20, 0x75,
					0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65,
					0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77, 0x69,
					0x73, 0x65, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x65, 0x73, 0x20,
					0x77, 0x69, 0x74, 0x68, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x35, 0x2e, 0x20, 0x53, 0x75, 0x62,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x43,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x2e, 0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x59, 0x6f, 0x75,
					0x20, 0x65, 0x78, 0x70, 0x6c, 0x69, 0x63, 0x69, 0x74, 0x6c, 0x79, 0x20,
					0x73, 0x74, 0x61, 0x74, 0x65, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77,
					0x69, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f,
					0x6e, 0x61, 0x6c, 0x6c, 0x79, 0x20, 0x73, 0x75, 0x62, 0x6d, 0x69, 0x74,
					0x74, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x69, 0x6e, 0x63, 0x6c,
					0x75, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x57, 0x6f, 0x72, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x62, 0x79, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x73,
					0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x61, 0x6e,
					0x79, 0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c,
					0x20, 0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x77, 0x69, 0x74, 0x68, 0x73,
					0x74, 0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x2c, 0x20, 0x6e, 0x6f, 0x74, 0x68, 0x69,
					0x6e, 0x67, 0x20, 0x68, 0x65, 0x72, 0x65, 0x69, 0x6e, 0x20, 0x73, 0x68,
					0x61, 0x6c, 0x6c, 0x20, 0x73, 0x75, 0x70, 0x65, 0x72, 0x73, 0x65, 0x64,
					0x65, 0x20, 0x6f, 0x72, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x65,
					0x72, 0x6d, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73,
					0x65, 0x70, 0x61, 0x72, 0x61, 0x74, 0x65, 0x20, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x6d, 0x65, 0x6e,
					0x74, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x68, 0x61,
					0x76, 0x65, 0x20, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x65, 0x64, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x72, 0x65, 0x67, 0x61,
					0x72, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x36, 0x2e, 0x20, 0x54, 0x72, 0x61,
					0x64, 0x65, 0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2e, 0x20, 0x54, 0x68, 0x69,
					0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x64, 0x6f,
					0x65, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74,
					0x72, 0x61, 0x64, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e,
					0x61, 0x6d, 0x65, 0x73, 0x2c, 0x20, 0x74, 0x72, 0x61, 0x64, 0x65, 0x6d,
					0x61, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63,
					0x65, 0x20, 0x6d, 0x61, 0x72, 0x6b, 0x73, 0x2c, 0x20, 0x6f, 0x72, 0x20,
					0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x20, 0x6e, 0x61, 0x6d, 0x65,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x61, 0x73, 0x20, 0x72,
					0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x72, 0x65, 0x61, 0x73, 0x6f, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x63, 0x75, 0x73, 0x74, 0x6f, 0x6d, 0x61, 0x72, 0x79,
					0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x73, 0x63,
					0x72, 0x69, 0x62, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63,
					0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x74,
					0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4e,
					0x4f, 0x54, 0x49, 0x43, 0x45, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x37, 0x2e, 0x20, 0x44, 0x69, 0x73, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x57, 0x61, 0x72,
					0x72, 0x61, 0x6e, 0x74, 0x79, 0x2e, 0x20, 0x55, 0x6e, 0x6c, 0x65, 0x73,
					0x73, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62,
					0x79, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65,
					0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20,
					0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57, 0x6f,
					0x72, 0x6b, 0x20, 0x28, 0x61, 0x6e, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69,
					0x64, 0x65, 0x73, 0x20, 0x69, 0x74, 0x73, 0x20, 0x43, 0x6f, 0x6e, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x29, 0x20, 0x6f,
					0x6e, 0x20, 0x61, 0x6e, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x20, 0x42, 0x41, 0x53, 0x49, 0x53, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72, 0x65,
					0x73, 0x73, 0x20, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x69, 0x6d, 0x70, 0x6c, 0x69, 0x65, 0x64, 0x2c, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x77, 0x61, 0x72, 0x72,
					0x61, 0x6e, 0x74, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x54, 0x49, 0x54, 0x4c, 0x45, 0x2c,
					0x20, 0x4e, 0x4f, 0x4e, 0x2d, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47,
					0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2c, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x6f, 0x72, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x41, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50,
					0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55,
					0x52, 0x50, 0x4f, 0x53, 0x45, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61,
					0x72, 0x65, 0x20, 0x73, 0x6f, 0x6c, 0x65, 0x6c, 0x79, 0x20, 0x72, 0x65,
					0x73, 0x70, 0x6f, 0x6e, 0x73, 0x69, 0x62, 0x6c, 0x65, 0x20, 0x66, 0x6f,
					0x72, 0x20, 0x64, 0x65, 0x74, 0x65, 0x72, 0x6d, 0x69, 0x6e, 0x69, 0x6e,
					0x67, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x61, 0x70, 0x70, 0x72, 0x6f, 0x70, 0x72, 0x69, 0x61, 0x74, 0x65, 0x6e,
					0x65, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x75, 0x73, 0x69, 0x6e, 0x67,
					0x20, 0x6f, 0x72, 0x20, 0x72, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x74, 0x68, 0x65, 0x20, 0x57,
					0x6f, 0x72, 0x6b, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x75,
					0x6d, 0x65, 0x20, 0x61, 0x6e, 0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x72, 0x69, 0x73, 0x6b, 0x73, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63,
					0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x59,
					0x6f, 0x75, 0x72, 0x20, 0x65, 0x78, 0x65, 0x72, 0x63, 0x69, 0x73, 0x65,
					0x20, 0x6f, 0x66, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a,
					0x0a, 0x20, 0x20, 0x20, 0x38, 0x2e, 0x20, 0x4c, 0x69, 0x6d, 0x69, 0x74,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x4c, 0x69, 0x61,
					0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2e, 0x20, 0x49, 0x6e, 0x20, 0x6e,
					0x6f, 0x20, 0x65, 0x76, 0x65, 0x6e, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x6e, 0x6f, 0x20, 0x6c, 0x65, 0x67,
					0x61, 0x6c, 0x20, 0x74, 0x68, 0x65, 0x6f, 0x72, 0x79, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x68, 0x65, 0x74, 0x68, 0x65, 0x72,
					0x20, 0x69, 0x6e, 0x20, 0x74, 0x6f, 0x72, 0x74, 0x20, 0x28, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x65, 0x67, 0x6c,
					0x69, 0x67, 0x65, 0x6e, 0x63, 0x65, 0x29, 0x2c, 0x20, 0x63, 0x6f, 0x6e,
					0x74, 0x72, 0x61, 0x63, 0x74, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74,
					0x68, 0x65, 0x72, 0x77, 0x69, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x75, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65,
					0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70,
					0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77,
					0x20, 0x28, 0x73, 0x75, 0x63, 0x68, 0x20, 0x61, 0x73, 0x20, 0x64, 0x65,
					0x6c, 0x69, 0x62, 0x65, 0x72, 0x61, 0x74, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x67, 0x72, 0x6f, 0x73, 0x73, 0x6c, 0x79, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6e, 0x65, 0x67, 0x6c, 0x69, 0x67, 0x65, 0x6e, 0x74,
					0x20, 0x61, 0x63, 0x74, 0x73, 0x29, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67,
					0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77,
					0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x62, 0x65, 0x0a, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x74, 0x6f,
					0x20, 0x59, 0x6f, 0x75, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x6d,
					0x61, 0x67, 0x65, 0x73, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x64, 0x69, 0x72, 0x65,
					0x63, 0x74, 0x2c, 0x20, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74,
					0x2c, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x61, 0x6c, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x69, 0x64, 0x65, 0x6e,
					0x74, 0x61, 0x6c, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x73,
					0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x64, 0x61,
					0x6d, 0x61, 0x67, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79,
					0x20, 0x63, 0x68, 0x61, 0x72, 0x61, 0x63, 0x74, 0x65, 0x72, 0x20, 0x61,
					0x72, 0x69, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x73, 0x20, 0x61, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x75, 0x74, 0x20,
					0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6f,
					0x72, 0x20, 0x69, 0x6e, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20,
					0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x20, 0x28, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x62, 0x75, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x20,
					0x66, 0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x67, 0x6f, 0x6f, 0x64, 0x77, 0x69, 0x6c, 0x6c, 0x2c, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x20, 0x73, 0x74, 0x6f,
					0x70, 0x70, 0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x75,
					0x74, 0x65, 0x72, 0x20, 0x66, 0x61, 0x69, 0x6c, 0x75, 0x72, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x6d, 0x61, 0x6c, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x65,
					0x72, 0x63, 0x69, 0x61, 0x6c, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x73, 0x73, 0x65, 0x73, 0x29,
					0x2c, 0x20, 0x65, 0x76, 0x65, 0x6e, 0x20, 0x69, 0x66, 0x20, 0x73, 0x75,
					0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
					0x6f, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x61, 0x73,
					0x20, 0x62, 0x65, 0x65, 0x6e, 0x20, 0x61, 0x64, 0x76, 0x69, 0x73, 0x65,
					0x64, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x6f, 0x73,
					0x73, 0x69, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x6f, 0x66, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x64, 0x61, 0x6d, 0x61, 0x67, 0x65, 0x73,
					0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x39, 0x2e, 0x20, 0x41, 0x63, 0x63,
					0x65, 0x70, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x57, 0x61, 0x72, 0x72, 0x61,
					0x6e, 0x74, 0x79, 0x20, 0x6f, 0x72, 0x20, 0x41, 0x64, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x4c, 0x69, 0x61, 0x62, 0x69, 0x6c,
					0x69, 0x74, 0x79, 0x2e, 0x20, 0x57, 0x68, 0x69, 0x6c, 0x65, 0x20, 0x72,
					0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6e,
					0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x57, 0x6f, 0x72, 0x6b, 0x20, 0x6f, 0x72, 0x20, 0x44, 0x65, 0x72, 0x69,
					0x76, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x57, 0x6f, 0x72, 0x6b, 0x73,
					0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x6f, 0x66, 0x2c, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x63, 0x68, 0x6f, 0x6f, 0x73, 0x65,
					0x20, 0x74, 0x6f, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x63, 0x68, 0x61,
					0x72, 0x67, 0x65, 0x20, 0x61, 0x20, 0x66, 0x65, 0x65, 0x20, 0x66, 0x6f,
					0x72, 0x2c, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x61, 0x6e, 0x63,
					0x65, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74,
					0x2c, 0x20, 0x77, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x2c, 0x20,
					0x69, 0x6e, 0x64, 0x65, 0x6d, 0x6e, 0x69, 0x74, 0x79, 0x2c, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65,
					0x72, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20,
					0x6f, 0x62, 0x6c, 0x69, 0x67, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x69, 0x73, 0x74, 0x65, 0x6e, 0x74,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x69, 0x73, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x2e, 0x20, 0x48, 0x6f, 0x77, 0x65, 0x76, 0x65, 0x72, 0x2c, 0x20, 0x69,
					0x6e, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74, 0x69, 0x6e, 0x67, 0x20,
					0x73, 0x75, 0x63, 0x68, 0x20, 0x6f, 0x62, 0x6c, 0x69, 0x67, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x6d, 0x61,
					0x79, 0x20, 0x61, 0x63, 0x74, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72,
					0x20, 0x6f, 0x77, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x59, 0x6f, 0x75, 0x72, 0x20,
					0x73, 0x6f, 0x6c, 0x65, 0x20, 0x72, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x73,
					0x69, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2c, 0x20, 0x6e, 0x6f, 0x74,
					0x20, 0x6f, 0x6e, 0x20, 0x62, 0x65, 0x68, 0x61, 0x6c, 0x66, 0x0a, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x20,
					0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6f,
					0x6e, 0x6c, 0x79, 0x20, 0x69, 0x66, 0x20, 0x59, 0x6f, 0x75, 0x20, 0x61,
					0x67, 0x72, 0x65, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x64, 0x65,
					0x6d, 0x6e, 0x69, 0x66, 0x79, 0x2c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x64, 0x65, 0x66, 0x65, 0x6e, 0x64, 0x2c, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x68, 0x6f, 0x6c, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x43,
					0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x68,
					0x61, 0x72, 0x6d, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x66, 0x6f, 0x72, 0x20,
					0x61, 0x6e, 0x79, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74,
					0x79, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x63, 0x75,
					0x72, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x2c, 0x20, 0x6f, 0x72, 0x20,
					0x63, 0x6c, 0x61, 0x69, 0x6d, 0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72,
					0x74, 0x65, 0x64, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e, 0x73, 0x74, 0x2c,
					0x20, 0x73, 0x75, 0x63, 0x68, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x72, 0x65, 0x61,
					0x73, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x66,
					0x20, 0x79, 0x6f, 0x75, 0x72, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74,
					0x69, 0x6e, 0x67, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x73, 0x75, 0x63, 0x68,
					0x20, 0x77, 0x61, 0x72, 0x72, 0x61, 0x6e, 0x74, 0x79, 0x20, 0x6f, 0x72,
					0x20, 0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20,
					0x6c, 0x69, 0x61, 0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x2e, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x45, 0x4e, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x45,
					0x52, 0x4d, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x44,
					0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x41,
					0x50, 0x50, 0x45, 0x4e, 0x44, 0x49, 0x58, 0x3a, 0x20, 0x48, 0x6f, 0x77,
					0x20, 0x74, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75, 0x72,
					0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x20, 0x54, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x79, 0x6f, 0x75,
					0x72, 0x20, 0x77, 0x6f, 0x72, 0x6b, 0x2c, 0x20, 0x61, 0x74, 0x74, 0x61,
					0x63, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62,
					0x6f, 0x69, 0x6c, 0x65, 0x72, 0x70, 0x6c, 0x61, 0x74, 0x65, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x73, 0x20, 0x65,
					0x6e, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x62,
					0x72, 0x61, 0x63, 0x6b, 0x65, 0x74, 0x73, 0x20, 0x22, 0x5b, 0x5d, 0x22,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x61,
					0x63, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x79, 0x6f, 0x75,
					0x72, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69,
					0x66, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x72, 0x6d,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x20, 0x28, 0x44, 0x6f, 0x6e, 0x27,
					0x74, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x65, 0x20, 0x62, 0x72, 0x61, 0x63,
					0x6b, 0x65, 0x74, 0x73, 0x21, 0x29, 0x20, 0x20, 0x54, 0x68, 0x65, 0x20,
					0x74, 0x65, 0x78, 0x74, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20,
					0x62, 0x65, 0x20, 0x65, 0x6e, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x70, 0x70, 0x72, 0x6f,
					0x70, 0x72, 0x69, 0x61, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x73, 0x79, 0x6e,
					0x74, 0x61, 0x78, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x69, 0x6c, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x2e,
					0x20, 0x57, 0x65, 0x20, 0x61, 0x6c, 0x73, 0x6f, 0x20, 0x72, 0x65, 0x63,
					0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
					0x61, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x65,
					0x20, 0x6f, 0x72, 0x20, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x6e, 0x61,
					0x6d, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x65, 0x73, 0x63, 0x72,
					0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x75,
					0x72, 0x70, 0x6f, 0x73, 0x65, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x61, 0x6d, 0x65, 0x20,
					0x22, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x70, 0x61, 0x67,
					0x65, 0x22, 0x20, 0x61, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x65, 0x61, 0x73, 0x69, 0x65,
					0x72, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x64, 0x65, 0x6e,
					0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x69, 0x72, 0x64, 0x2d,
					0x70, 0x61, 0x72, 0x74, 0x79, 0x20, 0x61, 0x72, 0x63, 0x68, 0x69, 0x76,
					0x65, 0x73, 0x2e, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x43, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x5b, 0x79, 0x79, 0x79, 0x79, 0x5d,
					0x20, 0x5b, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6f, 0x77, 0x6e, 0x65,
					0x72, 0x5d, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f,
					0x6e, 0x20, 0x32, 0x2e, 0x30, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x22, 0x29, 0x3b, 0x0a, 0x20,
					0x20, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6e, 0x6f,
					0x74, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x66,
					0x69, 0x6c, 0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x20, 0x69,
					0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x59, 0x6f,
					0x75, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e,
					0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61,
					0x74, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68, 0x74,
					0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x70, 0x61,
					0x63, 0x68, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x73, 0x2f, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x2d, 0x32, 0x2e, 0x30, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x55, 0x6e, 0x6c,
					0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64,
					0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62,
					0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20, 0x6f, 0x72, 0x20, 0x61, 0x67,
					0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x77,
					0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x73, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x20, 0x22,
					0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53, 0x49, 0x53,
					0x2c, 0x0a, 0x20, 0x20, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e,
					0x44, 0x2c, 0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78,
					0x70, 0x72, 0x65, 0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70,
					0x6c, 0x69, 0x65, 0x64, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x53, 0x65, 0x65,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65,
					0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61,
					0x67, 0x65, 0x20, 0x67, 0x6f, 0x76, 0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x61, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x6c, 0x69, 0x6d, 0x69,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x75, 0x6e, 0x64, 0x65,
					0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73,
					0x65, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x6f, 0x6b, 0x6f, 0x6c,
					0x69, 0x2f, 0x67, 0x6f, 0x75, 0x74, 0x69, 0x6c, 0x73, 0x0a, 0x0a, 0x0a,
					0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61, 0x70, 0x65, 0x78, 0x2f, 0x6c, 0x6f,
					0x67, 0x0a, 0x0a, 0x0a, 0x28, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x29, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29,
					0x20, 0x32, 0x30, 0x31, 0x35, 0x20, 0x54, 0x4a, 0x20, 0x48, 0x6f, 0x6c,
					0x6f, 0x77, 0x61, 0x79, 0x63, 0x68, 0x75, 0x6b, 0x20, 0x26, 0x6c, 0x74,
					0x3b, 0x74, 0x6a, 0x40, 0x74, 0x6a, 0x68, 0x6f, 0x6c, 0x6f, 0x77, 0x61,
					0x79, 0x63, 0x68, 0x75, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x61, 0x26, 0x67,
					0x74, 0x3b, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79,
					0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72,
					0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65,
					0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72,
					0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e,
					0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69,
					0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73,
					0x20, 0x28, 0x74, 0x68, 0x65, 0x0a, 0x27, 0x53, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x27, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65,
					0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
					0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e,
					0x67, 0x0a, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69,
					0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x75,
					0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f,
					0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c,
					0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f,
					0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20,
					0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77,
					0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e,
					0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20,
					0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20,
					0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20,
					0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63,
					0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c,
					0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73,
					0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70,
					0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e,
					0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44,
					0x45, 0x44, 0x20, 0x27, 0x41, 0x53, 0x20, 0x49, 0x53, 0x27, 0x2c, 0x20,
					0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20,
					0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49,
					0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57,
					0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46,
					0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53,
					0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54,
					0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f,
					0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e,
					0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x0a,
					0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20,
					0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55,
					0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50,
					0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45,
					0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x43, 0x4c, 0x41,
					0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x41, 0x43, 0x54, 0x2c, 0x0a, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f,
					0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c,
					0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f,
					0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49,
					0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x0a,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47,
					0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46,
					0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20,
					0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x61,
					0x70, 0x65, 0x78, 0x2f, 0x6c, 0x6f, 0x67, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x67, 0x68, 0x6f, 0x64, 0x73, 0x73, 0x2f, 0x79, 0x61,
					0x6d, 0x6c, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49,
					0x54, 0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x34, 0x20, 0x53,
					0x61, 0x6d, 0x20, 0x47, 0x68, 0x6f, 0x64, 0x73, 0x0a, 0x0a, 0x50, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62,
					0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x0a, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20,
					0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73,
					0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x0a, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20,
					0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69,
					0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69,
					0x73, 0x0a, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e,
					0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55,
					0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f,
					0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f,
					0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54,
					0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44,
					0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54,
					0x48, 0x45, 0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e,
					0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x32, 0x20, 0x54, 0x68,
					0x65, 0x20, 0x47, 0x6f, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x73,
					0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73,
					0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a,
					0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69,
					0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d,
					0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66,
					0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70,
					0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69,
					0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a,
					0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20,
					0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74, 0x61, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73,
					0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69,
					0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20,
					0x20, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69,
					0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75,
					0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x0a, 0x63,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74,
					0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69,
					0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64,
					0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a, 0x69, 0x6e,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
					0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f,
					0x72, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65,
					0x72, 0x69, 0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x0a,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69, 0x74, 0x68,
					0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20,
					0x6f, 0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e,
					0x63, 0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e,
					0x61, 0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74, 0x73, 0x0a,
					0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x6f, 0x72, 0x73,
					0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65, 0x20, 0x75, 0x73, 0x65, 0x64,
					0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x73, 0x65, 0x20,
					0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d, 0x6f, 0x74, 0x65, 0x20, 0x70,
					0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20, 0x64, 0x65, 0x72, 0x69,
					0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x0a, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69,
					0x66, 0x69, 0x63, 0x20, 0x70, 0x72, 0x69, 0x6f, 0x72, 0x20, 0x77, 0x72,
					0x69, 0x74, 0x74, 0x65, 0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f,
					0x52, 0x53, 0x0a, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45,
					0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45,
					0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c,
					0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d,
					0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52,
					0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d,
					0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45,
					0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50, 0x41, 0x52,
					0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50,
					0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x20, 0x44, 0x49, 0x53, 0x43,
					0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e,
					0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c,
					0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49,
					0x47, 0x48, 0x54, 0x0a, 0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52,
					0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45,
					0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54,
					0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c,
					0x2c, 0x0a, 0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45,
					0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52,
					0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49,
					0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28,
					0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42,
					0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54,
					0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55,
					0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55,
					0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f,
					0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43,
					0x45, 0x53, 0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20,
					0x55, 0x53, 0x45, 0x2c, 0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f,
					0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f,
					0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49,
					0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29,
					0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x20, 0x43, 0x41, 0x55,
					0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41,
					0x4e, 0x59, 0x0a, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52,
					0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28,
					0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45,
					0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41,
					0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20,
					0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41,
					0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54,
					0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c,
					0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67,
					0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x68,
					0x6f, 0x64, 0x73, 0x73, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x0a, 0x0a, 0x0a,
					0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x65, 0x73, 0x73, 0x65, 0x76, 0x64,
					0x6b, 0x2f, 0x67, 0x6f, 0x2d, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x0a, 0x0a,
					0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28,
					0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x32, 0x20, 0x4a, 0x65, 0x73, 0x73,
					0x65, 0x20, 0x76, 0x61, 0x6e, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x4b, 0x69,
					0x65, 0x62, 0x6f, 0x6f, 0x6d, 0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76,
					0x65, 0x64, 0x2e, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x75,
					0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20,
					0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d,
					0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c,
					0x20, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x74,
					0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20,
					0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x0a, 0x6d, 0x65,
					0x74, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63,
					0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74,
					0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76,
					0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20,
					0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72,
					0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20,
					0x72, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64,
					0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68,
					0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73,
					0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x4e, 0x65, 0x69, 0x74, 0x68,
					0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20,
					0x6f, 0x66, 0x20, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e,
					0x63, 0x2e, 0x20, 0x6e, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6e,
					0x61, 0x6d, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x74, 0x73, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69, 0x62,
					0x75, 0x74, 0x6f, 0x72, 0x73, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65,
					0x20, 0x75, 0x73, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e, 0x64,
					0x6f, 0x72, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x6d,
					0x6f, 0x74, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73,
					0x20, 0x64, 0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f,
					0x6d, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69,
					0x63, 0x20, 0x70, 0x72, 0x69, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74,
					0x74, 0x65, 0x6e, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69,
					0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52,
					0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20,
					0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20,
					0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53,
					0x0a, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20,
					0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42,
					0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54,
					0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52,
					0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x20, 0x41, 0x4e, 0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53,
					0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49,
					0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53,
					0x45, 0x20, 0x41, 0x52, 0x45, 0x20, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41,
					0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48,
					0x54, 0x0a, 0x4f, 0x57, 0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x4e, 0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20,
					0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54,
					0x2c, 0x20, 0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20,
					0x49, 0x4e, 0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x0a,
					0x53, 0x50, 0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45,
					0x4d, 0x50, 0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x4e, 0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c,
					0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54,
					0x20, 0x4e, 0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44,
					0x20, 0x54, 0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53,
					0x54, 0x49, 0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53,
					0x3b, 0x20, 0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53,
					0x45, 0x2c, 0x0a, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20,
					0x50, 0x52, 0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20,
					0x42, 0x55, 0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54,
					0x45, 0x52, 0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48,
					0x4f, 0x57, 0x45, 0x56, 0x45, 0x52, 0x20, 0x43, 0x41, 0x55, 0x53, 0x45,
					0x44, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59,
					0x0a, 0x54, 0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48,
					0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43,
					0x54, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c,
					0x20, 0x4f, 0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c,
					0x49, 0x47, 0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49,
					0x53, 0x49, 0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20,
					0x57, 0x41, 0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48,
					0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c,
					0x20, 0x45, 0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56,
					0x49, 0x53, 0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x50, 0x4f, 0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20,
					0x4f, 0x46, 0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x65, 0x73, 0x73,
					0x65, 0x76, 0x64, 0x6b, 0x2f, 0x67, 0x6f, 0x2d, 0x66, 0x6c, 0x61, 0x67,
					0x73, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49,
					0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69,
					0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x6f, 0x68,
					0x6f, 0x2f, 0x67, 0x6f, 0x64, 0x6f, 0x74, 0x65, 0x6e, 0x76, 0x0a, 0x0a,
					0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28,
					0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x33, 0x20, 0x4a, 0x6f, 0x68, 0x6e,
					0x20, 0x42, 0x61, 0x72, 0x74, 0x6f, 0x6e, 0x0a, 0x0a, 0x4d, 0x49, 0x54,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a, 0x0a, 0x50, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62,
					0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x0a,
					0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73,
					0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20,
					0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69,
					0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x0a, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69,
					0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e,
					0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55,
					0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x0a,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44,
					0x0a, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x0a,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x4f,
					0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f,
					0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57, 0x49, 0x54,
					0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44,
					0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e,
					0x0a, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6a, 0x6f, 0x68, 0x6f, 0x2f, 0x67,
					0x6f, 0x64, 0x6f, 0x74, 0x65, 0x6e, 0x76, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x6b, 0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f, 0x73,
					0x2f, 0x6f, 0x73, 0x65, 0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32,
					0x30, 0x31, 0x32, 0x20, 0x54, 0x68, 0x65, 0x20, 0x47, 0x6f, 0x20, 0x41,
					0x75, 0x74, 0x68, 0x6f, 0x72, 0x73, 0x2e, 0x20, 0x41, 0x6c, 0x6c, 0x20,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65, 0x72,
					0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e, 0x64,
					0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x72,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72,
					0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74,
					0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
					0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65,
					0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66,
					0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e,
					0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65, 0x0a,
					0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52,
					0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
					0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
					0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72,
					0x65, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62,
					0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x0a, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68,
					0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77,
					0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d,
					0x65, 0x72, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a, 0x20, 0x52, 0x65, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66,
					0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70,
					0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61,
					0x62, 0x6f, 0x76, 0x65, 0x0a, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69,
					0x6d, 0x65, 0x72, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x6f, 0x74, 0x68, 0x65,
					0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x73, 0x20,
					0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x20, 0x74, 0x68, 0x65, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x20, 0x20, 0x20, 0x2a,
					0x20, 0x4e, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x47, 0x6f, 0x6f,
					0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x20, 0x6e, 0x6f, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x69, 0x74, 0x73, 0x0a, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x6f, 0x72, 0x73, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62,
					0x65, 0x20, 0x75, 0x73, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e,
					0x64, 0x6f, 0x72, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f,
					0x6d, 0x6f, 0x74, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74,
					0x73, 0x20, 0x64, 0x65, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72,
					0x6f, 0x6d, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74,
					0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x70, 0x72,
					0x69, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x74, 0x65, 0x6e, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x2e, 0x0a,
					0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44,
					0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f,
					0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44,
					0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54,
					0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x0a, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52,
					0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49,
					0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e,
					0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52,
					0x0a, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41,
					0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52,
					0x45, 0x20, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44,
					0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x0a, 0x4f, 0x57,
					0x4e, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c,
					0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e,
					0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49,
					0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x0a, 0x53, 0x50, 0x45, 0x43,
					0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41,
					0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45,
					0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d,
					0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x0a, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c,
					0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55,
					0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x20,
					0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x20, 0x4c, 0x4f,
					0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x0a, 0x44,
					0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46,
					0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49,
					0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55,
					0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56,
					0x45, 0x52, 0x20, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x0a, 0x54, 0x48, 0x45,
					0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49,
					0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45,
					0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43,
					0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e,
					0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57,
					0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47,
					0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20,
					0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55,
					0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45,
					0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44,
					0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53,
					0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53,
					0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a,
					0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e,
					0x63, 0x6f, 0x6d, 0x2f, 0x6b, 0x61, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x6f,
					0x73, 0x2f, 0x6f, 0x73, 0x65, 0x78, 0x74, 0x0a, 0x0a, 0x0a, 0x42, 0x45,
					0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x6f, 0x6c, 0x65, 0x6b, 0x75, 0x6b, 0x6f, 0x6e, 0x6b,
					0x6f, 0x2f, 0x74, 0x73, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72,
					0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x32, 0x30, 0x31,
					0x34, 0x20, 0x62, 0x79, 0x20, 0x4f, 0x6c, 0x65, 0x6b, 0x75, 0x20, 0x4b,
					0x6f, 0x6e, 0x6b, 0x6f, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73,
					0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65,
					0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20,
					0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72,
					0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70,
					0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e,
					0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77,
					0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f,
					0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d,
					0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c,
					0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20,
					0x64, 0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63,
					0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64,
					0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20,
					0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f,
					0x20, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20,
					0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20,
					0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a,
					0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20,
					0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f,
					0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f,
					0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75,
					0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64,
					0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63,
					0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
					0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63,
					0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69,
					0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x0a, 0x61,
					0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72,
					0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c,
					0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72,
					0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56,
					0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22,
					0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41,
					0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52,
					0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49,
					0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e,
					0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49,
					0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e,
					0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41,
					0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52,
					0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e,
					0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54,
					0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a,
					0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43,
					0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c,
					0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42,
					0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43,
					0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c,
					0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48,
					0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20,
					0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f,
					0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53,
					0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46,
					0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20,
					0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43,
					0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f,
					0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52,
					0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49,
					0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53,
					0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x45, 0x4e, 0x44,
					0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
					0x6f, 0x6c, 0x65, 0x6b, 0x75, 0x6b, 0x6f, 0x6e, 0x6b, 0x6f, 0x2f, 0x74,
					0x73, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49,
					0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69,
					0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x67,
					0x2f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x73, 0x0a, 0x0a, 0x0a, 0x43, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20,
					0x32, 0x30, 0x31, 0x35, 0x2c, 0x20, 0x44, 0x61, 0x76, 0x65, 0x20, 0x43,
					0x68, 0x65, 0x6e, 0x65, 0x79, 0x20, 0x3c, 0x64, 0x61, 0x76, 0x65, 0x40,
					0x63, 0x68, 0x65, 0x6e, 0x65, 0x79, 0x2e, 0x6e, 0x65, 0x74, 0x3e, 0x0a,
					0x41, 0x6c, 0x6c, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72,
					0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20,
					0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62,
					0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74,
					0x68, 0x6f, 0x75, 0x74, 0x0a, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63,
					0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x61, 0x72, 0x65, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f,
					0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
					0x61, 0x72, 0x65, 0x20, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x2a, 0x20,
					0x52, 0x65, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63,
					0x65, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20,
					0x72, 0x65, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61,
					0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74,
					0x68, 0x69, 0x73, 0x0a, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f,
					0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c,
					0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c,
					0x61, 0x69, 0x6d, 0x65, 0x72, 0x2e, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65,
					0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20,
					0x66, 0x6f, 0x72, 0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65,
					0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69,
					0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x0a,
					0x20, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20,
					0x6f, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f,
					0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63,
					0x6c, 0x61, 0x69, 0x6d, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74,
					0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72,
					0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72,
					0x69, 0x61, 0x6c, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65,
					0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e,
					0x0a, 0x0a, 0x54, 0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49,
					0x44, 0x45, 0x44, 0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43,
					0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c,
					0x44, 0x45, 0x52, 0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x22, 0x41,
					0x53, 0x20, 0x49, 0x53, 0x22, 0x0a, 0x41, 0x4e, 0x44, 0x20, 0x41, 0x4e,
					0x59, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52,
					0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43,
					0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20,
					0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20,
					0x54, 0x4f, 0x2c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x49, 0x4d, 0x50, 0x4c,
					0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49,
					0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
					0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e,
					0x44, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c,
					0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41,
					0x52, 0x45, 0x0a, 0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45,
					0x44, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45,
					0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48,
					0x4f, 0x4c, 0x44, 0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x49, 0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45,
					0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x0a, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x4e, 0x59, 0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20,
					0x49, 0x4e, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e,
					0x43, 0x49, 0x44, 0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x20, 0x53, 0x50,
					0x45, 0x43, 0x49, 0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50,
					0x4c, 0x41, 0x52, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e,
					0x53, 0x45, 0x51, 0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x0a, 0x44,
					0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e,
					0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54,
					0x4f, 0x2c, 0x20, 0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45,
					0x4e, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49,
					0x54, 0x55, 0x54, 0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f,
					0x52, 0x0a, 0x53, 0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x20,
					0x4c, 0x4f, 0x53, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c,
					0x20, 0x44, 0x41, 0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52,
					0x4f, 0x46, 0x49, 0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55,
					0x53, 0x49, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52,
					0x52, 0x55, 0x50, 0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57,
					0x45, 0x56, 0x45, 0x52, 0x0a, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20,
					0x41, 0x4e, 0x44, 0x20, 0x4f, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x54,
					0x48, 0x45, 0x4f, 0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54,
					0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52,
					0x41, 0x43, 0x54, 0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x4f,
					0x52, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x28, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47,
					0x45, 0x4e, 0x43, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41,
					0x59, 0x20, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x55, 0x53, 0x45, 0x0a, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45,
					0x56, 0x45, 0x4e, 0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53,
					0x45, 0x44, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f,
					0x53, 0x53, 0x49, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46,
					0x20, 0x53, 0x55, 0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45,
					0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e,
					0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75,
					0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x67, 0x2f, 0x65, 0x72,
					0x72, 0x6f, 0x72, 0x73, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e,
					0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
					0x72, 0x64, 0x73, 0x75, 0x62, 0x68, 0x61, 0x73, 0x2f, 0x67, 0x6f, 0x2d,
					0x65, 0x6c, 0x61, 0x73, 0x74, 0x69, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65,
					0x0a, 0x0a, 0x0a, 0x4d, 0x49, 0x54, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
					0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x53,
					0x75, 0x62, 0x68, 0x61, 0x73, 0x20, 0x44, 0x61, 0x6e, 0x64, 0x61, 0x70,
					0x61, 0x6e, 0x69, 0x0a, 0x0a, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73,
					0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62,
					0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66,
					0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67,
					0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65,
					0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69,
					0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x0a, 0x6f, 0x66,
					0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63,
					0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65,
					0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65,
					0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64,
					0x65, 0x61, 0x6c, 0x0a, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74,
					0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69,
					0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c,
					0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x0a, 0x74, 0x6f, 0x20,
					0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d,
					0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65,
					0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73,
					0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61,
					0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73, 0x65, 0x6c, 0x6c, 0x0a, 0x63,
					0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74,
					0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20,
					0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69, 0x73, 0x0a, 0x66, 0x75, 0x72,
					0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f,
					0x20, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74,
					0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c,
					0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74,
					0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61,
					0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
					0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65,
					0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c,
					0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20,
					0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20,
					0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57,
					0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49,
					0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c,
					0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52,
					0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59,
					0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45,
					0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45,
					0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47,
					0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d,
					0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f,
					0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42,
					0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a, 0x46, 0x49, 0x54, 0x4e, 0x45,
					0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52,
					0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50,
					0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49,
					0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e,
					0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54,
					0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x41,
					0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f,
					0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44,
					0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c,
					0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53,
					0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x4c, 0x49,
					0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41,
					0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e,
					0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20,
					0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45,
					0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52,
					0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f,
					0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54,
					0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45,
					0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20,
					0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e,
					0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54, 0x48, 0x45, 0x0a, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44,
					0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
					0x72, 0x64, 0x73, 0x75, 0x62, 0x68, 0x61, 0x73, 0x2f, 0x67, 0x6f, 0x2d,
					0x65, 0x6c, 0x61, 0x73, 0x74, 0x69, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65,
					0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43,
					0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74,
					0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x61, 0x74, 0x6f,
					0x72, 0x69, 0x2f, 0x67, 0x6f, 0x2e, 0x75, 0x75, 0x69, 0x64, 0x0a, 0x0a,
					0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28,
					0x43, 0x29, 0x20, 0x32, 0x30, 0x31, 0x33, 0x2d, 0x32, 0x30, 0x31, 0x36,
					0x20, 0x62, 0x79, 0x20, 0x4d, 0x61, 0x78, 0x69, 0x6d, 0x20, 0x42, 0x75,
					0x62, 0x6c, 0x69, 0x73, 0x20, 0x3c, 0x62, 0x40, 0x63, 0x6f, 0x64, 0x65,
					0x6d, 0x6f, 0x6e, 0x6b, 0x65, 0x79, 0x2e, 0x72, 0x75, 0x3e, 0x0a, 0x0a,
					0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69,
					0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61,
					0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f,
					0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20,
					0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x0a, 0x61, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64,
					0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68,
					0x65, 0x0a, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22,
					0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69,
					0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72,
					0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20,
					0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x0a, 0x77, 0x69,
					0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x75, 0x73, 0x65, 0x2c, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79,
					0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62,
					0x6c, 0x69, 0x73, 0x68, 0x2c, 0x0a, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72,
					0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f,
					0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73,
					0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x20,
					0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x0a, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a,
					0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x62, 0x65, 0x0a, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70,
					0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74,
					0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49,
					0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22,
					0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59,
					0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44,
					0x2c, 0x0a, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52,
					0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20,
					0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20,
					0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x0a, 0x4d, 0x45, 0x52,
					0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c,
					0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41,
					0x4e, 0x44, 0x0a, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e,
					0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e,
					0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c,
					0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52,
					0x53, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47,
					0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42,
					0x45, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20,
					0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20,
					0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e,
					0x0a, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54,
					0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49,
					0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f,
					0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20,
					0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x0a, 0x57,
					0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x61, 0x74, 0x6f, 0x72,
					0x69, 0x2f, 0x67, 0x6f, 0x2e, 0x75, 0x75, 0x69, 0x64, 0x0a, 0x0a, 0x0a,
					0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x67, 0x75, 0x6c, 0x61, 0x63, 0x73,
					0x69, 0x2f, 0x77, 0x72, 0x61, 0x70, 0x0a, 0x0a, 0x0a, 0x43, 0x6f, 0x70,
					0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32,
					0x30, 0x31, 0x34, 0x2c, 0x20, 0x42, 0x72, 0x61, 0x6e, 0x64, 0x6f, 0x6e,
					0x20, 0x54, 0x68, 0x6f, 0x6d, 0x73, 0x6f, 0x6e, 0x0a, 0x41, 0x6c, 0x6c,
					0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x73, 0x20, 0x72, 0x65, 0x73, 0x65,
					0x72, 0x76, 0x65, 0x64, 0x2e, 0x0a, 0x0a, 0x52, 0x65, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x6f, 0x75,
					0x72, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x69, 0x6e, 0x61,
					0x72, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x73, 0x2c, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x20, 0x6f, 0x72, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
					0x74, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x2c, 0x0a, 0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x72, 0x6d,
					0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64,
					0x65, 0x64, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f,
					0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x72, 0x65,
					0x20, 0x6d, 0x65, 0x74, 0x3a, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64,
					0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x6f, 0x66, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63,
					0x6f, 0x64, 0x65, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x74,
					0x61, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76,
					0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20,
					0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x0a, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77,
					0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69, 0x6d,
					0x65, 0x72, 0x2e, 0x0a, 0x0a, 0x2a, 0x20, 0x52, 0x65, 0x64, 0x69, 0x73,
					0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x69,
					0x6e, 0x20, 0x62, 0x69, 0x6e, 0x61, 0x72, 0x79, 0x20, 0x66, 0x6f, 0x72,
					0x6d, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x6f,
					0x64, 0x75, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f,
					0x76, 0x65, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x0a, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f,
					0x77, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x69, 0x73, 0x63, 0x6c, 0x61, 0x69,
					0x6d, 0x65, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x0a, 0x20, 0x20, 0x6f, 0x74,
					0x68, 0x65, 0x72, 0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61, 0x6c,
					0x73, 0x20, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x64, 0x65, 0x64, 0x20, 0x77,
					0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64, 0x69, 0x73, 0x74,
					0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x0a, 0x54,
					0x48, 0x49, 0x53, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45,
					0x20, 0x49, 0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44,
					0x20, 0x42, 0x59, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f, 0x50, 0x59,
					0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52,
					0x53, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49,
					0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x22, 0x41, 0x53, 0x20, 0x49,
					0x53, 0x22, 0x20, 0x41, 0x4e, 0x44, 0x0a, 0x41, 0x4e, 0x59, 0x20, 0x45,
					0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4d,
					0x50, 0x4c, 0x49, 0x45, 0x44, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e,
					0x54, 0x49, 0x45, 0x53, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44,
					0x49, 0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54,
					0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44,
					0x0a, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20,
					0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41, 0x4e, 0x54, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x41, 0x4e, 0x44, 0x20, 0x46,
					0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52, 0x20,
					0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x52, 0x45, 0x0a,
					0x44, 0x49, 0x53, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x45, 0x44, 0x2e, 0x20,
					0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20,
					0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x43, 0x4f,
					0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44,
					0x45, 0x52, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x49,
					0x42, 0x55, 0x54, 0x4f, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20, 0x4c, 0x49,
					0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x0a, 0x41, 0x4e, 0x59,
					0x20, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x44,
					0x49, 0x52, 0x45, 0x43, 0x54, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x49, 0x44,
					0x45, 0x4e, 0x54, 0x41, 0x4c, 0x2c, 0x20, 0x53, 0x50, 0x45, 0x43, 0x49,
					0x41, 0x4c, 0x2c, 0x20, 0x45, 0x58, 0x45, 0x4d, 0x50, 0x4c, 0x41, 0x52,
					0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x43, 0x4f, 0x4e, 0x53, 0x45, 0x51,
					0x55, 0x45, 0x4e, 0x54, 0x49, 0x41, 0x4c, 0x20, 0x44, 0x41, 0x4d, 0x41,
					0x47, 0x45, 0x53, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49,
					0x4e, 0x47, 0x2c, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f, 0x54, 0x20,
					0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f, 0x2c, 0x20,
					0x50, 0x52, 0x4f, 0x43, 0x55, 0x52, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x20,
					0x4f, 0x46, 0x20, 0x53, 0x55, 0x42, 0x53, 0x54, 0x49, 0x54, 0x55, 0x54,
					0x45, 0x20, 0x47, 0x4f, 0x4f, 0x44, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x53,
					0x45, 0x52, 0x56, 0x49, 0x43, 0x45, 0x53, 0x3b, 0x0a, 0x4c, 0x4f, 0x53,
					0x53, 0x20, 0x4f, 0x46, 0x20, 0x55, 0x53, 0x45, 0x2c, 0x20, 0x44, 0x41,
					0x54, 0x41, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x50, 0x52, 0x4f, 0x46, 0x49,
					0x54, 0x53, 0x3b, 0x20, 0x4f, 0x52, 0x20, 0x42, 0x55, 0x53, 0x49, 0x4e,
					0x45, 0x53, 0x53, 0x20, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x52, 0x55, 0x50,
					0x54, 0x49, 0x4f, 0x4e, 0x29, 0x20, 0x48, 0x4f, 0x57, 0x45, 0x56, 0x45,
					0x52, 0x20, 0x43, 0x41, 0x55, 0x53, 0x45, 0x44, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x4f, 0x4e, 0x0a, 0x41, 0x4e, 0x59, 0x20, 0x54, 0x48, 0x45, 0x4f,
					0x52, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c,
					0x49, 0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54,
					0x2c, 0x20, 0x53, 0x54, 0x52, 0x49, 0x43, 0x54, 0x20, 0x4c, 0x49, 0x41,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20, 0x4f, 0x52, 0x20, 0x54,
					0x4f, 0x52, 0x54, 0x0a, 0x28, 0x49, 0x4e, 0x43, 0x4c, 0x55, 0x44, 0x49,
					0x4e, 0x47, 0x20, 0x4e, 0x45, 0x47, 0x4c, 0x49, 0x47, 0x45, 0x4e, 0x43,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x57, 0x49,
					0x53, 0x45, 0x29, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49, 0x4e, 0x47, 0x20,
					0x49, 0x4e, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x57, 0x41, 0x59, 0x20, 0x4f,
					0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x46, 0x20, 0x54, 0x48, 0x49, 0x53, 0x0a, 0x53, 0x4f,
					0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2c, 0x20, 0x45, 0x56, 0x45, 0x4e,
					0x20, 0x49, 0x46, 0x20, 0x41, 0x44, 0x56, 0x49, 0x53, 0x45, 0x44, 0x20,
					0x4f, 0x46, 0x20, 0x54, 0x48, 0x45, 0x20, 0x50, 0x4f, 0x53, 0x53, 0x49,
					0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x20, 0x4f, 0x46, 0x20, 0x53, 0x55,
					0x43, 0x48, 0x20, 0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x2e, 0x0a, 0x0a,
					0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20,
					0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63,
					0x6f, 0x6d, 0x2f, 0x74, 0x67, 0x75, 0x6c, 0x61, 0x63, 0x73, 0x69, 0x2f,
					0x77, 0x72, 0x61, 0x70, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47, 0x49, 0x4e,
					0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
					0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x67, 0x6a, 0x73, 0x6f,
					0x6e, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49, 0x54, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d, 0x49, 0x54,
					0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20, 0x4a, 0x6f,
					0x73, 0x68, 0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a, 0x50, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62,
					0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20,
					0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73,
					0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20,
					0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69,
					0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69,
					0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a, 0x73, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e,
					0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55,
					0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x20,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20,
					0x4f, 0x52, 0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f,
					0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a, 0x43, 0x4f,
					0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54,
					0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44,
					0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54,
					0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e,
					0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62,
					0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c,
					0x2f, 0x67, 0x6a, 0x73, 0x6f, 0x6e, 0x0a, 0x0a, 0x0a, 0x42, 0x45, 0x47,
					0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x20, 0x46,
					0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68, 0x75, 0x62, 0x2e, 0x63, 0x6f,
					0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61, 0x6c, 0x6c, 0x2f, 0x6d, 0x61,
					0x74, 0x63, 0x68, 0x0a, 0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x49,
					0x54, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x28, 0x4d,
					0x49, 0x54, 0x29, 0x0a, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x20, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x31, 0x36, 0x20,
					0x4a, 0x6f, 0x73, 0x68, 0x20, 0x42, 0x61, 0x6b, 0x65, 0x72, 0x0a, 0x0a,
					0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69,
					0x73, 0x20, 0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61,
					0x6e, 0x74, 0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f,
					0x66, 0x20, 0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f,
					0x20, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20,
					0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e,
					0x64, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64,
					0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68,
					0x65, 0x20, 0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22,
					0x29, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69,
					0x6e, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72,
					0x65, 0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20,
					0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69,
					0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61,
					0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67,
					0x68, 0x74, 0x73, 0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79,
					0x2c, 0x20, 0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62,
					0x6c, 0x69, 0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69,
					0x62, 0x75, 0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63,
					0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72,
					0x20, 0x73, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73,
					0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74,
					0x77, 0x61, 0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73,
					0x6f, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x69, 0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65,
					0x64, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x6f, 0x2c, 0x0a,
					0x73, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74,
					0x68, 0x65, 0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67,
					0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a,
					0x0a, 0x0a, 0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20,
					0x63, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f,
					0x74, 0x69, 0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69,
					0x73, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
					0x20, 0x6e, 0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c,
					0x6c, 0x20, 0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65,
					0x64, 0x20, 0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70,
					0x69, 0x65, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74,
					0x61, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69,
					0x6f, 0x6e, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48,
					0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49,
					0x53, 0x20, 0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22,
					0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48,
					0x4f, 0x55, 0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59,
					0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44,
					0x2c, 0x20, 0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52,
					0x0a, 0x49, 0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e,
					0x43, 0x4c, 0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20,
					0x4e, 0x4f, 0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20,
					0x54, 0x4f, 0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41,
					0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52,
					0x43, 0x48, 0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x0a, 0x46, 0x4f,
					0x52, 0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c,
					0x41, 0x52, 0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41,
					0x4e, 0x44, 0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e,
					0x47, 0x45, 0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e,
					0x4f, 0x20, 0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c,
					0x4c, 0x20, 0x54, 0x48, 0x45, 0x20, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52,
					0x53, 0x20, 0x4f, 0x52, 0x0a, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47,
					0x48, 0x54, 0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42,
					0x45, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52,
					0x20, 0x41, 0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20,
					0x44, 0x41, 0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f,
					0x54, 0x48, 0x45, 0x52, 0x20, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49,
					0x54, 0x59, 0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x0a,
					0x49, 0x4e, 0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e,
					0x20, 0x4f, 0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54,
					0x2c, 0x20, 0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54,
					0x48, 0x45, 0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49,
					0x53, 0x49, 0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x20, 0x4f,
					0x55, 0x54, 0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x0a,
					0x43, 0x4f, 0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57,
					0x49, 0x54, 0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54,
					0x57, 0x41, 0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20,
					0x55, 0x53, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52,
					0x20, 0x44, 0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52,
					0x45, 0x2e, 0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45,
					0x4e, 0x53, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x69, 0x74, 0x68,
					0x75, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x74, 0x69, 0x64, 0x77, 0x61,
					0x6c, 0x6c, 0x2f, 0x6d, 0x61, 0x74, 0x63, 0x68, 0x0a, 0x0a, 0x0a, 0x42,
					0x45, 0x47, 0x49, 0x4e, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45,
					0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e, 0x69,
					0x6e, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a, 0x0a,
					0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x32, 0x30,
					0x31, 0x31, 0x2d, 0x32, 0x30, 0x31, 0x36, 0x20, 0x43, 0x61, 0x6e, 0x6f,
					0x6e, 0x69, 0x63, 0x61, 0x6c, 0x20, 0x4c, 0x74, 0x64, 0x2e, 0x0a, 0x0a,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x64, 0x20, 0x75, 0x6e, 0x64,
					0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x41, 0x70, 0x61, 0x63, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c, 0x20, 0x56,
					0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x32, 0x2e, 0x30, 0x20, 0x28,
					0x74, 0x68, 0x65, 0x20, 0x22, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65,
					0x22, 0x29, 0x3b, 0x0a, 0x79, 0x6f, 0x75, 0x20, 0x6d, 0x61, 0x79, 0x20,
					0x6e, 0x6f, 0x74, 0x20, 0x75, 0x73, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74,
					0x20, 0x69, 0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x61, 0x6e,
					0x63, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20,
					0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x59, 0x6f, 0x75,
					0x20, 0x6d, 0x61, 0x79, 0x20, 0x6f, 0x62, 0x74, 0x61, 0x69, 0x6e, 0x20,
					0x61, 0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x61, 0x74,
					0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
					0x2f, 0x77, 0x77, 0x77, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e,
					0x6f, 0x72, 0x67, 0x2f, 0x6c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x73,
					0x2f, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53, 0x45, 0x2d, 0x32, 0x2e, 0x30,
					0x0a, 0x0a, 0x55, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x72, 0x65, 0x71,
					0x75, 0x69, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70,
					0x6c, 0x69, 0x63, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x6c, 0x61, 0x77, 0x20,
					0x6f, 0x72, 0x20, 0x61, 0x67, 0x72, 0x65, 0x65, 0x64, 0x20, 0x74, 0x6f,
					0x20, 0x69, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2c,
					0x20, 0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x0a, 0x64, 0x69,
					0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x75, 0x6e,
					0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c, 0x69, 0x63, 0x65,
					0x6e, 0x73, 0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72,
					0x69, 0x62, 0x75, 0x74, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20, 0x61, 0x6e,
					0x20, 0x22, 0x41, 0x53, 0x20, 0x49, 0x53, 0x22, 0x20, 0x42, 0x41, 0x53,
					0x49, 0x53, 0x2c, 0x0a, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55, 0x54, 0x20,
					0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x49, 0x45, 0x53, 0x20, 0x4f,
					0x52, 0x20, 0x43, 0x4f, 0x4e, 0x44, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x53,
					0x20, 0x4f, 0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44,
					0x2c, 0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x65, 0x78, 0x70,
					0x72, 0x65, 0x73, 0x73, 0x20, 0x6f, 0x72, 0x20, 0x69, 0x6d, 0x70, 0x6c,
					0x69, 0x65, 0x64, 0x2e, 0x0a, 0x53, 0x65, 0x65, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x66, 0x6f, 0x72,
					0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x66, 0x69,
					0x63, 0x20, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x20, 0x67,
					0x6f, 0x76, 0x65, 0x72, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x65, 0x72,
					0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x61, 0x6e, 0x64,
					0x0a, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73,
					0x20, 0x75, 0x6e, 0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4c,
					0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2e, 0x0a, 0x54, 0x68, 0x65, 0x20,
					0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x69,
					0x6c, 0x65, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x70, 0x6f, 0x72,
					0x74, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x47, 0x6f, 0x20, 0x66, 0x72,
					0x6f, 0x6d, 0x20, 0x43, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x6f,
					0x66, 0x20, 0x6c, 0x69, 0x62, 0x79, 0x61, 0x6d, 0x6c, 0x2c, 0x20, 0x61,
					0x6e, 0x64, 0x20, 0x74, 0x68, 0x75, 0x73, 0x0a, 0x61, 0x72, 0x65, 0x20,
					0x73, 0x74, 0x69, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x76, 0x65, 0x72, 0x65,
					0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x69, 0x72, 0x20, 0x6f,
					0x72, 0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x79,
					0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6c, 0x69,
					0x63, 0x65, 0x6e, 0x73, 0x65, 0x3a, 0x0a, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x61, 0x70, 0x69, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20,
					0x65, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a,
					0x20, 0x20, 0x20, 0x20, 0x70, 0x61, 0x72, 0x73, 0x65, 0x72, 0x63, 0x2e,
					0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x61, 0x64, 0x65,
					0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x73, 0x63,
					0x61, 0x6e, 0x6e, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f, 0x0a, 0x20, 0x20,
					0x20, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x72, 0x63, 0x2e, 0x67, 0x6f,
					0x0a, 0x20, 0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x68, 0x2e, 0x67,
					0x6f, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x79, 0x61, 0x6d, 0x6c, 0x70, 0x72,
					0x69, 0x76, 0x61, 0x74, 0x65, 0x68, 0x2e, 0x67, 0x6f, 0x0a, 0x0a, 0x43,
					0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x28, 0x63, 0x29,
					0x20, 0x32, 0x30, 0x30, 0x36, 0x20, 0x4b, 0x69, 0x72, 0x69, 0x6c, 0x6c,
					0x20, 0x53, 0x69, 0x6d, 0x6f, 0x6e, 0x6f, 0x76, 0x0a, 0x0a, 0x50, 0x65,
					0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
					0x68, 0x65, 0x72, 0x65, 0x62, 0x79, 0x20, 0x67, 0x72, 0x61, 0x6e, 0x74,
					0x65, 0x64, 0x2c, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20, 0x6f, 0x66, 0x20,
					0x63, 0x68, 0x61, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x74, 0x6f, 0x20, 0x61,
					0x6e, 0x79, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x20, 0x6f, 0x62,
					0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x74, 0x68, 0x69, 0x73, 0x20, 0x73,
					0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20,
					0x61, 0x73, 0x73, 0x6f, 0x63, 0x69, 0x61, 0x74, 0x65, 0x64, 0x20, 0x64,
					0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
					0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20, 0x28, 0x74, 0x68, 0x65, 0x20,
					0x22, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x22, 0x29, 0x2c,
					0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x61, 0x6c, 0x20, 0x69, 0x6e, 0x0a,
					0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
					0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x72, 0x65, 0x73,
					0x74, 0x72, 0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x6e,
					0x63, 0x6c, 0x75, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x69, 0x74, 0x68,
					0x6f, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69,
					0x6f, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x69, 0x67, 0x68, 0x74,
					0x73, 0x20, 0x74, 0x6f, 0x0a, 0x75, 0x73, 0x65, 0x2c, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x2c, 0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x2c, 0x20,
					0x6d, 0x65, 0x72, 0x67, 0x65, 0x2c, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69,
					0x73, 0x68, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
					0x74, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x65, 0x6e,
					0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72, 0x20, 0x73,
					0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x0a, 0x6f,
					0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61,
					0x72, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x70,
					0x65, 0x72, 0x6d, 0x69, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
					0x73, 0x20, 0x74, 0x6f, 0x20, 0x77, 0x68, 0x6f, 0x6d, 0x20, 0x74, 0x68,
					0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x69,
					0x73, 0x20, 0x66, 0x75, 0x72, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x64, 0x20,
					0x74, 0x6f, 0x20, 0x64, 0x6f, 0x0a, 0x73, 0x6f, 0x2c, 0x20, 0x73, 0x75,
					0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
					0x20, 0x66, 0x6f, 0x6c, 0x6c, 0x6f, 0x77, 0x69, 0x6e, 0x67, 0x20, 0x63,
					0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0x0a, 0x0a,
					0x54, 0x68, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x76, 0x65, 0x20, 0x63, 0x6f,
					0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69,
					0x63, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20,
					0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x6e,
					0x6f, 0x74, 0x69, 0x63, 0x65, 0x20, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20,
					0x62, 0x65, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x64, 0x20,
					0x69, 0x6e, 0x20, 0x61, 0x6c, 0x6c, 0x0a, 0x63, 0x6f, 0x70, 0x69, 0x65,
					0x73, 0x20, 0x6f, 0x72, 0x20, 0x73, 0x75, 0x62, 0x73, 0x74, 0x61, 0x6e,
					0x74, 0x69, 0x61, 0x6c, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x69, 0x6f, 0x6e,
					0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66,
					0x74, 0x77, 0x61, 0x72, 0x65, 0x2e, 0x0a, 0x0a, 0x54, 0x48, 0x45, 0x20,
					0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x20, 0x49, 0x53, 0x20,
					0x50, 0x52, 0x4f, 0x56, 0x49, 0x44, 0x45, 0x44, 0x20, 0x22, 0x41, 0x53,
					0x20, 0x49, 0x53, 0x22, 0x2c, 0x20, 0x57, 0x49, 0x54, 0x48, 0x4f, 0x55,
					0x54, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59, 0x20, 0x4f,
					0x46, 0x20, 0x41, 0x4e, 0x59, 0x20, 0x4b, 0x49, 0x4e, 0x44, 0x2c, 0x20,
					0x45, 0x58, 0x50, 0x52, 0x45, 0x53, 0x53, 0x20, 0x4f, 0x52, 0x0a, 0x49,
					0x4d, 0x50, 0x4c, 0x49, 0x45, 0x44, 0x2c, 0x20, 0x49, 0x4e, 0x43, 0x4c,
					0x55, 0x44, 0x49, 0x4e, 0x47, 0x20, 0x42, 0x55, 0x54, 0x20, 0x4e, 0x4f,
					0x54, 0x20, 0x4c, 0x49, 0x4d, 0x49, 0x54, 0x45, 0x44, 0x20, 0x54, 0x4f,
					0x20, 0x54, 0x48, 0x45, 0x20, 0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54,
					0x49, 0x45, 0x53, 0x20, 0x4f, 0x46, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48,
					0x41, 0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59, 0x2c, 0x0a,
					0x46, 0x49, 0x54, 0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52, 0x20,
					0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49, 0x43, 0x55, 0x4c, 0x41, 0x52,
					0x20, 0x50, 0x55, 0x52, 0x50, 0x4f, 0x53, 0x45, 0x20, 0x41, 0x4e, 0x44,
					0x20, 0x4e, 0x4f, 0x4e, 0x49, 0x4e, 0x46, 0x52, 0x49, 0x4e, 0x47, 0x45,
					0x4d, 0x45, 0x4e, 0x54, 0x2e, 0x20, 0x49, 0x4e, 0x20, 0x4e, 0x4f, 0x20,
					0x45, 0x56, 0x45, 0x4e, 0x54, 0x20, 0x53, 0x48, 0x41, 0x4c, 0x4c, 0x20,
					0x54, 0x48, 0x45, 0x0a, 0x41, 0x55, 0x54, 0x48, 0x4f, 0x52, 0x53, 0x20,
					0x4f, 0x52, 0x20, 0x43, 0x4f, 0x50, 0x59, 0x52, 0x49, 0x47, 0x48, 0x54,
					0x20, 0x48, 0x4f, 0x4c, 0x44, 0x45, 0x52, 0x53, 0x20, 0x42, 0x45, 0x20,
					0x4c, 0x49, 0x41, 0x42, 0x4c, 0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x41,
					0x4e, 0x59, 0x20, 0x43, 0x4c, 0x41, 0x49, 0x4d, 0x2c, 0x20, 0x44, 0x41,
					0x4d, 0x41, 0x47, 0x45, 0x53, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48,
					0x45, 0x52, 0x0a, 0x4c, 0x49, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54, 0x59,
					0x2c, 0x20, 0x57, 0x48, 0x45, 0x54, 0x48, 0x45, 0x52, 0x20, 0x49, 0x4e,
					0x20, 0x41, 0x4e, 0x20, 0x41, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x4f,
					0x46, 0x20, 0x43, 0x4f, 0x4e, 0x54, 0x52, 0x41, 0x43, 0x54, 0x2c, 0x20,
					0x54, 0x4f, 0x52, 0x54, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45,
					0x52, 0x57, 0x49, 0x53, 0x45, 0x2c, 0x20, 0x41, 0x52, 0x49, 0x53, 0x49,
					0x4e, 0x47, 0x20, 0x46, 0x52, 0x4f, 0x4d, 0x2c, 0x0a, 0x4f, 0x55, 0x54,
					0x20, 0x4f, 0x46, 0x20, 0x4f, 0x52, 0x20, 0x49, 0x4e, 0x20, 0x43, 0x4f,
					0x4e, 0x4e, 0x45, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x20, 0x57, 0x49, 0x54,
					0x48, 0x20, 0x54, 0x48, 0x45, 0x20, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41,
					0x52, 0x45, 0x20, 0x4f, 0x52, 0x20, 0x54, 0x48, 0x45, 0x20, 0x55, 0x53,
					0x45, 0x20, 0x4f, 0x52, 0x20, 0x4f, 0x54, 0x48, 0x45, 0x52, 0x20, 0x44,
					0x45, 0x41, 0x4c, 0x49, 0x4e, 0x47, 0x53, 0x20, 0x49, 0x4e, 0x20, 0x54,
					0x48, 0x45, 0x0a, 0x53, 0x4f, 0x46, 0x54, 0x57, 0x41, 0x52, 0x45, 0x2e,
					0x0a, 0x0a, 0x45, 0x4e, 0x44, 0x20, 0x4c, 0x49, 0x43, 0x45, 0x4e, 0x53,
					0x45, 0x20, 0x46, 0x4f, 0x52, 0x20, 0x67, 0x6f, 0x70, 0x6b, 0x67, 0x2e,
					0x69, 0x6e, 0x2f, 0x79, 0x61, 0x6d, 0x6c, 0x2e, 0x76, 0x32, 0x0a, 0x0a,
					0x0a,
				},
				fi: FileInfo{
					name:    "LICENSES",
					size:    124825,
					modTime: time.Unix(0, 1548683678287010268),
					isDir:   false,
				},
			}, "/tmp/version": {
				data: []byte{
					0x6d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x0a,
				},
				fi: FileInfo{
					name:    "version",
					size:    7,
					modTime: time.Unix(0, 1548683676553929667),
					isDir:   false,
				},
			},
		},
	}
}
