## Schemas and Rust Modules

A `pgx`-based extension is created in the schema determined by the `CREATE EXTENSION` statement.
If unspecified, that schema is whatever the first schema in the user's `search_path` is, otherwise
it is the schema argument to `CREATE EXTENSION`.

In general, any `pgx` object (a function, operator, type, etc), regardless of the Rust source
file it is defined in, is created in that schema unless that object appears in a 
`mod modname { ... }` block.  In this case, `pgx` generates a top-level schema named the
same as the module, and creates the contained objects within that schema.    

Unlike Rust, which supports nested modules, Postgres only supports one-level of schemas,
although a Postgres session can have many schemas in its `search_path`.  As such, any
`mod modname { ... }` block containing `pgx` objects is hoisted to a top-level schema.

### `#[pg_extern]/#[pg_operator]` Functions and their Postgres `search_path`

When `pgx` generates the DDL for a function (`CREATE FUNCTION ...`), it uses uses the schema
it understands the function to belong in two different ways.

First off, if there's a `schema = foo` attribute in your extension `.control` file, the
function is created in that schema.  If there is no `schema = foo` attribute, then the
function is *not* schema-qualified, which indicates it'll be created in the schema
determined by the `CREATE EXTENSION` function.

Secondly, `pgx` can apply `search_path` to that function that limits that function's
search_path to whatever you specify.  This is done via the `#[search_path(...)]` attribute macro
applied to the function with `#[pg_extern]` or `#[pg_operator]` or `#[pg_test]`.

For example:
 
```rust
#[derive(PostgresType, Serialize, Deserialize, Debug, Eq, PartialEq)]
pub struct SomeStruct {}

#[pg_extern]
#[search_path(@extschema@)]
fn return_vec_of_customtype() -> Vec<SomeStruct> {
    vec![SomeStruct {}]
}
```

`@extschema@` is likely all you'd ever want.  It's a token that Postgres itself will substitute during `CREATE EXTENSION`
to be that of the schema in which the extension is being installed.

You can, however instead use whatever search_path you'd like:

```rust
#[derive(PostgresType, Serialize, Deserialize, Debug, Eq, PartialEq)]
pub struct SomeStruct {}

#[pg_extern]
#[search_path(schema_a, schema_b, public, $user)]
fn return_vec_of_customtype() -> Vec<SomeStruct> {
    vec![SomeStruct {}]
}
```

In general this is only necessary when returning a `Vec<T: PostgresType>`.  In this situation, pgx needs to know that type's
`oid` (from the `pg_catalog.pg_type` system catalog) and as such, the schema in which that type lives must be on that 
function's `search_path`.

