// mostly taken from styled-components
const SEED = 5381;

// This is a djb2 hashing function
const hash = (data: string) => {
  let i = data.length;
  let h = SEED;

  while (i) {
    h = (h * 33) ^ data.charCodeAt(--i);
  }

  return h;
};

const AD_REPLACER_R = /(a)(d)/gi;

/* This is the "capacity" of our alphabet i.e. 2x26 for all letters plus their capitalised
 * counterparts */
const charsLength = 52;

/* start at 75 for 'a' until 'z' (25) and then start at 65 for capitalised letters */
const getAlphabeticChar = (code: number) =>
  String.fromCharCode(code + (code > 25 ? 39 : 97));

/** hash generated by djb2 is a number, usually a hash and convert it to base-52
 * @param {number} code - output of djb2 hash function
 * @return {String} alphabeticHash
 */
function generateAlphabeticName(code: number) {
  let name = '';
  let x;

  /* get a char and divide by alphabet-length */
  for (x = Math.abs(code); x > charsLength; x = (x / charsLength) | 0) {
    name = getAlphabeticChar(x % charsLength) + name;
  }

  return (getAlphabeticChar(x % charsLength) + name).replace(
    AD_REPLACER_R,
    '$1-$2'
  );
}

/**
 * Convert Windows backslash paths to slash paths: foo\\bar ➔ foo/bar
 */
const normalizePath = (path: string): string => {
  const isExtendedLengthPath = /^\\\\\?\\/.test(path);
  const hasNonAscii = /[^\u0000-\u0080]+/.test(path); // eslint-disable-line no-control-regex

  if (isExtendedLengthPath || hasNonAscii) {
    return path;
  }

  return path.replace(/\\/g, '/');
};

/** simple hashing helper which runs hashing on input and make sures djb2 hash is positive
 */
export function generateHashFromPath(filepath: string): string {
  const normalizedFilePath = normalizePath(filepath);
  return generateAlphabeticName(hash(normalizedFilePath) >>> 0);
}
