!!****m* ABINIT/m_dfpt_mkvxc
!! NAME
!!  m_dfpt_mkvxc
!!
!! FUNCTION
!!
!! COPYRIGHT
!!  Copyright (C) 2001-2025 ABINIT group (XG, DRH, FR, EB, SPr)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

module m_dfpt_mkvxc

 use defs_basis
 use m_errors
 use m_abicore
 use m_xc_noncoll

 use defs_abitypes,     only : MPI_type
 use m_time,     only : timab
 use m_matrix,   only : matr3inv
 use m_xctk,     only : xcden, xcpot, xcpotdq

 implicit none

 private
!!***

 public :: dfpt_mkvxc
 public :: dfpt_mkvxc_noncoll
 public :: dfpt_mkvxcggadq
 public :: dfpt_mkvxcgga_n0met
 public :: dfpt_mkvxcccdq
!!***

contains
!!***

!!****f* ABINIT/dfpt_mkvxc
!! NAME
!! dfpt_mkvxc
!!
!! FUNCTION
!! Compute the first-order change of exchange-correlation potential
!! due to atomic displacement: assemble the first-order
!! density change with the frozen-core density change, then use
!! the exchange-correlation kernel.
!!
!! INPUTS
!!  cplex= if 1, real space 1-order functions on FFT grid are REAL,
!!         if 2, COMPLEX
!!  ixc= choice of exchange-correlation scheme
!!  kxc(nfft,nkxc)=exchange and correlation kernel (see below)
!!  mpi_enreg=information about MPI parallelization
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(18)=contain all needed information about 3D FFT,
!!     see ~abinit/doc/variables/vargs.htm#ngfft
!!  nhat1(cplex*nfft,2nspden*nhat1dim)= -PAW only- 1st-order compensation density
!!  nhat1dim= -PAW only- 1 if nhat1 array is used ; 0 otherwise
!!  nhat1gr(cplex*nfft,nspden,3*nhat1grdim)= -PAW only- gradients of 1st-order compensation density
!!  nhat1grdim= -PAW only- 1 if nhat1gr array is used ; 0 otherwise
!!  nkxc=second dimension of the kxc array
!!  non_magnetic_xc= if true, handle density/potential as non-magnetic (even if it is)
!!  nspden=number of spin-density components
!!  n3xccc=dimension of xccc3d1 ; 0 if no XC core correction is used, otherwise, nfft
!!  option=if 0, work only with the XC core-correction,
!!         if 1, treat both density change and XC core correction
!!         if 2, treat only density change
!!  qphon(3)=reduced coordinates for the phonon wavelength (needed if cplex==2).
!!  rhor1(cplex*nfft,nspden)=array for electron density in electrons/bohr**3.
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!  usexcnhat= -PAW only- 1 if nhat density has to be taken into account in Vxc
!!  xccc3d1(cplex*n3xccc)=3D change in core charge density, see n3xccc
!!
!! OUTPUT
!!  vxc1(cplex*nfft,nspden)=change in exchange-correlation potential (including
!!   core-correction, if applicable)
!!
!! NOTES
!!  Content of Kxc array:
!!   ===== if LDA
!!    if nspden==1: kxc(:,1)= d2Exc/drho2
!!                 (kxc(:,2)= d2Exc/drho_up drho_dn)
!!    if nspden>=2: kxc(:,1)= d2Exc/drho_up drho_up
!!                  kxc(:,2)= d2Exc/drho_up drho_dn
!!                  kxc(:,3)= d2Exc/drho_dn drho_dn
!!   ===== if GGA (or mGGA)
!!    if nspden==1:
!!       kxc(:,1)= d2Exc/drho2
!!       kxc(:,2)= 1/|grad(rho)| dExc/d|grad(rho)|
!!       kxc(:,3)= 1/|grad(rho)| d2Exc/d|grad(rho)| drho
!!       kxc(:,4)= 1/|grad(rho)| * d/d|grad(rho)| ( 1/|grad(rho)| dExc/d|grad(rho)| )
!!       kxc(:,5)= gradx(rho)
!!       kxc(:,6)= grady(rho)
!!       kxc(:,7)= gradz(rho)
!!    if nspden>=2:
!!       kxc(:,1)= d2Exc/drho_up drho_up
!!       kxc(:,2)= d2Exc/drho_up drho_dn
!!       kxc(:,3)= d2Exc/drho_dn drho_dn
!!       kxc(:,4)= 1/|grad(rho_up)| dEx/d|grad(rho_up)|
!!       kxc(:,5)= 1/|grad(rho_dn)| dEx/d|grad(rho_dn)|
!!       kxc(:,6)= 1/|grad(rho_up)| d2Ex/d|grad(rho_up)| drho_up
!!       kxc(:,7)= 1/|grad(rho_dn)| d2Ex/d|grad(rho_dn)| drho_dn
!!       kxc(:,8)= 1/|grad(rho_up)| * d/d|grad(rho_up)| ( 1/|grad(rho_up)| dEx/d|grad(rho_up)| )
!!       kxc(:,9)= 1/|grad(rho_dn)| * d/d|grad(rho_dn)| ( 1/|grad(rho_dn)| dEx/d|grad(rho_dn)| )
!!       kxc(:,10)=1/|grad(rho)| dEc/d|grad(rho)|
!!       kxc(:,11)=1/|grad(rho)| d2Ec/d|grad(rho)| drho_up
!!       kxc(:,12)=1/|grad(rho)| d2Ec/d|grad(rho)| drho_dn
!!       kxc(:,13)=1/|grad(rho)| * d/d|grad(rho)| ( 1/|grad(rho)| dEc/d|grad(rho)| )
!!       kxc(:,14)=gradx(rho_up)
!!       kxc(:,15)=gradx(rho_dn)
!!       kxc(:,16)=grady(rho_up)
!!       kxc(:,17)=grady(rho_dn)
!!       kxc(:,18)=gradz(rho_up)
!!       kxc(:,19)=gradz(rho_dn)
!!    Note about mGGA: 2nd derivatives involving Tau or Laplacian are not taken into account (yet)
!!
!! SOURCE

subroutine dfpt_mkvxc(cplex,ixc,kxc,mpi_enreg,nfft,ngfft,nhat1,nhat1dim,nhat1gr,nhat1grdim,&
&          nkxc,non_magnetic_xc,nspden,n3xccc,option,qphon,rhor1,rprimd,usexcnhat,vxc1,xccc3d1)

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,ixc,n3xccc,nfft,nhat1dim,nhat1grdim
 integer,intent(in) :: nkxc,nspden,option,usexcnhat
 logical,intent(in) :: non_magnetic_xc
 type(MPI_type),intent(in) :: mpi_enreg
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in),target :: nhat1(cplex*nfft,nspden*nhat1dim)
 real(dp),intent(in),target :: nhat1gr(cplex*nfft,nspden,3*nhat1grdim)
 real(dp),intent(in) :: kxc(nfft,nkxc),qphon(3)
 real(dp),intent(in),target :: rhor1(cplex*nfft,nspden)
 real(dp),intent(in) :: rprimd(3,3),xccc3d1(cplex*n3xccc)
 real(dp),intent(out) :: vxc1(cplex*nfft,nspden)

!Local variables-------------------------------
!scalars
 integer :: ii,ir,ispden,nhat1dim_,nhat1rgdim_
 real(dp) :: rho1_dn,rho1_up,rho1im_dn,rho1im_up,rho1re_dn,rho1re_up
 real(dp) :: spin_scale
!arrays
 real(dp) :: gprimd(3,3),tsec(2)
 real(dp), ABI_CONTIGUOUS pointer :: nhat1_(:,:),nhat1gr_(:,:,:),rhor1_(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 call timab(181,1,tsec)

 if(nspden/=1 .and. nspden/=2) then
   ABI_BUG('For nspden==4 please use dfpt_mkvxc_noncoll!')
 end if

!Special case: no XC applied
 if (ixc==0.or.nkxc==0) then
   ABI_WARNING('Note that no xc is applied (ixc=0)')
   vxc1=zero
   return
 end if

!Treat first LDA
 if(nkxc==1.or.nkxc==3)then

!  PAW: eventually substract compensation density
   if (option/=0) then
     if ((usexcnhat==0.and.nhat1dim==1).or.(non_magnetic_xc)) then
       ABI_MALLOC(rhor1_,(cplex*nfft,nspden))
       if (usexcnhat==0.and.nhat1dim==1) then
         rhor1_(:,:)=rhor1(:,:)-nhat1(:,:)
       else
         rhor1_(:,:)=rhor1(:,:)
       end if
       if (non_magnetic_xc) then
         if(nspden==2) rhor1_(:,2)=rhor1_(:,1)*half
         if(nspden==4) rhor1_(:,2:4)=zero
       end if
     else
       rhor1_ => rhor1
     end if
   end if

!  Case without non-linear core correction
   if(n3xccc==0 .or. option==2)then

     if(option==0)then  ! No straight XC to compute

       vxc1(:,:)=zero

     else               ! XC, without non-linear XC correction

!      Non-spin-polarized
       if(nspden==1)then
         if(cplex==1)then
           do ir=1,nfft
             vxc1(ir,1)=kxc(ir,1)*rhor1_(ir,1)
           end do
         else
           do ir=1,nfft
             vxc1(2*ir-1,1)=kxc(ir,1)*rhor1_(2*ir-1,1)
             vxc1(2*ir  ,1)=kxc(ir,1)*rhor1_(2*ir  ,1)
           end do
         end if ! cplex==1

!        Spin-polarized
       else
         if(cplex==1)then
           do ir=1,nfft
             rho1_dn=rhor1_(ir,1)-rhor1_(ir,2)
             vxc1(ir,1)=kxc(ir,1)*rhor1_(ir,2)+kxc(ir,2)*rho1_dn
             vxc1(ir,2)=kxc(ir,2)*rhor1_(ir,2)+kxc(ir,3)*rho1_dn
           end do
         else
           do ir=1,nfft
             rho1re_dn=rhor1_(2*ir-1,1)-rhor1_(2*ir-1,2)
             rho1im_dn=rhor1_(2*ir  ,1)-rhor1_(2*ir  ,2)
             vxc1(2*ir-1,1)=kxc(ir,1)*rhor1_(2*ir-1,2)+kxc(ir,2)*rho1re_dn
             vxc1(2*ir  ,1)=kxc(ir,1)*rhor1_(2*ir  ,2)+kxc(ir,2)*rho1im_dn
             vxc1(2*ir-1,2)=kxc(ir,2)*rhor1_(2*ir-1,2)+kxc(ir,3)*rho1re_dn
             vxc1(2*ir  ,2)=kxc(ir,2)*rhor1_(2*ir  ,2)+kxc(ir,3)*rho1im_dn
           end do
         end if ! cplex==1
       end if ! nspden==1

     end if ! option==0

!    Treat case with non-linear core correction
   else

     if(option==0)then

       if(nspden==1)then
         if(cplex==1)then
           do ir=1,nfft
             vxc1(ir,1)=kxc(ir,1)*xccc3d1(ir)
           end do
         else
           do ir=1,nfft
             vxc1(2*ir-1,1)=kxc(ir,1)*xccc3d1(2*ir-1)
             vxc1(2*ir  ,1)=kxc(ir,1)*xccc3d1(2*ir  )
           end do
         end if ! cplex==1
       else
         if(cplex==1)then
           do ir=1,nfft
             vxc1(ir,1)=(kxc(ir,1)+kxc(ir,2))*xccc3d1(ir)*half
             vxc1(ir,2)=(kxc(ir,2)+kxc(ir,3))*xccc3d1(ir)*half
           end do
         else
           do ir=1,nfft
             vxc1(2*ir-1,1)=(kxc(ir,1)+kxc(ir,2))*xccc3d1(2*ir-1)*half
             vxc1(2*ir  ,1)=(kxc(ir,1)+kxc(ir,2))*xccc3d1(2*ir  )*half
             vxc1(2*ir-1,2)=(kxc(ir,2)+kxc(ir,3))*xccc3d1(2*ir-1)*half
             vxc1(2*ir  ,2)=(kxc(ir,2)+kxc(ir,3))*xccc3d1(2*ir  )*half
           end do
         end if ! cplex==1
       end if ! nspden==1

     else ! option/=0

       if(nspden==1)then
         if(cplex==1)then
           do ir=1,nfft
             vxc1(ir,1)=kxc(ir,1)*(rhor1_(ir,1)+xccc3d1(ir))
           end do
         else
           do ir=1,nfft
             vxc1(2*ir-1,1)=kxc(ir,1)*(rhor1_(2*ir-1,1)+xccc3d1(2*ir-1))
             vxc1(2*ir  ,1)=kxc(ir,1)*(rhor1_(2*ir  ,1)+xccc3d1(2*ir  ))
           end do
         end if ! cplex==1
       else
         if(cplex==1)then
           do ir=1,nfft
             rho1_dn=rhor1_(ir,1)-rhor1_(ir,2) + xccc3d1(ir)*half
             rho1_up=rhor1_(ir,2)             + xccc3d1(ir)*half
             vxc1(ir,1)=kxc(ir,1)*rho1_up+kxc(ir,2)*rho1_dn
             vxc1(ir,2)=kxc(ir,2)*rho1_up+kxc(ir,3)*rho1_dn
           end do
         else
           do ir=1,nfft
             rho1re_dn=rhor1_(2*ir-1,1)-rhor1_(2*ir-1,2) + xccc3d1(2*ir-1)*half
             rho1im_dn=rhor1_(2*ir  ,1)-rhor1_(2*ir  ,2) + xccc3d1(2*ir  )*half
             rho1re_up=rhor1_(2*ir-1,2)                 + xccc3d1(2*ir-1)*half
             rho1im_up=rhor1_(2*ir  ,2)                 + xccc3d1(2*ir  )*half
             vxc1(2*ir-1,1)=kxc(ir,1)*rho1re_up+kxc(ir,2)*rho1re_dn
             vxc1(2*ir  ,1)=kxc(ir,1)*rho1im_up+kxc(ir,2)*rho1im_dn
             vxc1(2*ir-1,2)=kxc(ir,2)*rho1re_up+kxc(ir,3)*rho1re_dn
             vxc1(2*ir  ,2)=kxc(ir,2)*rho1im_up+kxc(ir,3)*rho1im_dn
           end do
         end if ! cplex==1
       end if ! nspden==1

     end if ! option==0

   end if ! n3xccc==0

   if (option/=0.and.((usexcnhat==0.and.nhat1dim==1).or.(non_magnetic_xc))) then
     ABI_FREE(rhor1_)
   end if

!  Treat GGA
 else if (nkxc==7.or.nkxc==19) then

!  Transfer the data to spin-polarized storage

!  Treat the density change
   ABI_MALLOC(rhor1_,(cplex*nfft,nspden))
   if (option==1 .or. option==2) then
     if (nspden==1) then
       do ir=1,cplex*nfft
         rhor1_(ir,1)=rhor1(ir,1)
       end do
     else
       if(non_magnetic_xc) then
         do ir=1,cplex*nfft
           rho1_dn=rhor1(ir,1)*half
           rhor1_(ir,1)=rho1_dn
           rhor1_(ir,2)=rho1_dn
         end do
       else
         do ir=1,cplex*nfft
           rho1_dn=rhor1(ir,1)-rhor1(ir,2)
           rhor1_(ir,1)=rhor1(ir,2)
           rhor1_(ir,2)=rho1_dn
         end do
       end if
     end if
   else
     do ispden=1,nspden
       do ir=1,cplex*nfft
         rhor1_(ir,ispden)=zero
       end do
     end do
   end if

   if( (option==0 .or. option==1) .and. n3xccc/=0)then
     spin_scale=one;if (nspden==2) spin_scale=half
     do ispden=1,nspden
       do ir=1,cplex*nfft
         rhor1_(ir,ispden)=rhor1_(ir,ispden)+xccc3d1(ir)*spin_scale
       end do
     end do
   end if

!  PAW: treat also compensation density (and gradients)
   nhat1dim_=nhat1dim ; nhat1rgdim_=nhat1grdim
   if (option/=0.and.nhat1dim==1.and.nspden==2) then
     ABI_MALLOC(nhat1_,(cplex*nfft,nspden))
     if (non_magnetic_xc) then
       do ir=1,cplex*nfft
         rho1_dn=nhat1(ir,1)*half
         nhat1_(ir,1:2)=rho1_dn
       end do
     else
       do ir=1,cplex*nfft
         rho1_dn=nhat1(ir,1)-nhat1(ir,2)
         nhat1_(ir,1)=nhat1(ir,2)
         nhat1_(ir,2)=rho1_dn
       end do
     end if
   else if (option==0) then
     ABI_MALLOC(nhat1_,(0,0))
     nhat1dim_=0
   else
     nhat1_ => nhat1
   end if
   if (option/=0.and.nhat1grdim==1.and.nspden==2) then
     ABI_MALLOC(nhat1gr_,(cplex*nfft,nspden,3))
     if (non_magnetic_xc) then
       do ii=1,3
         do ir=1,cplex*nfft
           rho1_dn=nhat1(ir,1)*half
           nhat1gr_(ir,1:2,ii)=rho1_dn
         end do
       end do
     else
       do ii=1,3
         do ir=1,cplex*nfft
           rho1_dn=nhat1gr(ir,1,ii)-nhat1gr(ir,2,ii)
           nhat1gr_(ir,1,ii)=nhat1gr(ir,2,ii)
           nhat1gr_(ir,2,ii)=rho1_dn
         end do
       end do
     end if
   else if (option==0) then
     ABI_MALLOC(nhat1gr_,(0,0,0))
     nhat1rgdim_=0
   else
     nhat1gr_ => nhat1gr
   end if

   call matr3inv(rprimd,gprimd)

   call dfpt_mkvxcgga(cplex,gprimd,kxc,mpi_enreg,nfft,ngfft,nhat1_,nhat1dim_,&
&   nhat1gr_,nhat1rgdim_,nkxc,nspden,qphon,rhor1_,usexcnhat,vxc1)

   ABI_FREE(rhor1_)
   if ((option==0).or.(nhat1dim==1.and.nspden==2)) then
     ABI_FREE(nhat1_)
   end if
   if ((option==0).or.(nhat1grdim==1.and.nspden==2)) then
     ABI_FREE(nhat1gr_)
   end if

 else
   ABI_BUG('Invalid nkxc!')

 end if ! LDA or GGA

 call timab(181,2,tsec)

 DBG_EXIT("COLL")

end subroutine dfpt_mkvxc
!!***

!!****f* ABINIT/dfpt_mkvxcgga
!! NAME
!! dfpt_mkvxcgga
!!
!! FUNCTION
!! Compute the first-order change of exchange-correlation potential
!! in case of GGA functionals
!! Use the exchange-correlation kernel.
!!
!! INPUTS
!!  cplex= if 1, real space 1-order functions on FFT grid are REAL,
!!    if 2, COMPLEX
!!  gmet(3,3)=metrix tensor in G space in Bohr**-2.
!!  gprimd(3,3)=dimensional primitive translations in reciprocal space (bohr^-1)
!!  gsqcut=cutoff value on G**2 for sphere inside fft box.
!!  kxc(nfft,nkxc)=exchange and correlation kernel (see below)
!!  mpi_enreg=information about MPI parallelization
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(18)=contain all needed information about 3D FFT
!!  nhat1(cplex*nfft,2*nhat1dim)= -PAW only- 1st-order compensation density
!!  nhat1dim= -PAW only- 1 if nhat1 array is used ; 0 otherwise
!!  nhat1gr(cplex*nfft,2,3*nhat1grdim)= -PAW only- gradients of 1st-order compensation density
!!  nhat1grdim= -PAW only- 1 if nhat1gr array is used ; 0 otherwise
!!  nkxc=second dimension of the kxc array
!!  nspden=number of spin-density components
!!  qphon(3)=reduced coordinates for the phonon wavelength (needed if cplex==2).
!!  rhor1tmp(cplex*nfft,2)=array for first-order electron spin-density
!!   in electrons/bohr**3 (second index corresponds to spin-up and spin-down)
!!  usexcnhat= -PAW only- 1 if nhat density has to be taken into account in Vxc
!!
!! OUTPUT
!!  vxc1(cplex*nfft,nspden)=change in exchange-correlation potential
!!
!! NOTES
!!  For the time being, a rather crude coding, to be optimized ...
!!  Content of Kxc array:
!!   ===== if GGA
!!    if nspden==1:
!!       kxc(:,1)= d2Exc/drho2
!!       kxc(:,2)= 1/|grad(rho)| dExc/d|grad(rho)|
!!       kxc(:,3)= 1/|grad(rho)| d2Exc/d|grad(rho)| drho
!!       kxc(:,4)= 1/|grad(rho)| * d/d|grad(rho)| ( 1/|grad(rho)| dExc/d|grad(rho)| )
!!       kxc(:,5)= gradx(rho)
!!       kxc(:,6)= grady(rho)
!!       kxc(:,7)= gradz(rho)
!!    if nspden>=2:
!!       kxc(:,1)= d2Exc/drho_up drho_up
!!       kxc(:,2)= d2Exc/drho_up drho_dn
!!       kxc(:,3)= d2Exc/drho_dn drho_dn
!!       kxc(:,4)= 1/|grad(rho_up)| dEx/d|grad(rho_up)|
!!       kxc(:,5)= 1/|grad(rho_dn)| dEx/d|grad(rho_dn)|
!!       kxc(:,6)= 1/|grad(rho_up)| d2Ex/d|grad(rho_up)| drho_up
!!       kxc(:,7)= 1/|grad(rho_dn)| d2Ex/d|grad(rho_dn)| drho_dn
!!       kxc(:,8)= 1/|grad(rho_up)| * d/d|grad(rho_up)| ( 1/|grad(rho_up)| dEx/d|grad(rho_up)| )
!!       kxc(:,9)= 1/|grad(rho_dn)| * d/d|grad(rho_dn)| ( 1/|grad(rho_dn)| dEx/d|grad(rho_dn)| )
!!       kxc(:,10)=1/|grad(rho)| dEc/d|grad(rho)|
!!       kxc(:,11)=1/|grad(rho)| d2Ec/d|grad(rho)| drho_up
!!       kxc(:,12)=1/|grad(rho)| d2Ec/d|grad(rho)| drho_dn
!!       kxc(:,13)=1/|grad(rho)| * d/d|grad(rho)| ( 1/|grad(rho)| dEc/d|grad(rho)| )
!!       kxc(:,14)=gradx(rho_up)
!!       kxc(:,15)=gradx(rho_dn)
!!       kxc(:,16)=grady(rho_up)
!!       kxc(:,17)=grady(rho_dn)
!!       kxc(:,18)=gradz(rho_up)
!!       kxc(:,19)=gradz(rho_dn)
!!
!! SOURCE

subroutine dfpt_mkvxcgga(cplex,gprimd,kxc,mpi_enreg,nfft,ngfft,&
&                    nhat1,nhat1dim,nhat1gr,nhat1grdim,nkxc,&
&                    nspden,qphon,rhor1,usexcnhat,vxc1)

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,nfft,nhat1dim,nhat1grdim,nkxc,nspden,usexcnhat
 type(MPI_type),intent(in) :: mpi_enreg
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: gprimd(3,3)
 real(dp),intent(in) :: kxc(nfft,nkxc)
 real(dp),intent(in) :: nhat1(cplex*nfft,nspden*nhat1dim)
 real(dp),intent(in) :: nhat1gr(cplex*nfft,nspden,3*nhat1grdim)
 real(dp),intent(in) :: qphon(3)
 real(dp),intent(in),target :: rhor1(cplex*nfft,nspden)
 real(dp),intent(out) :: vxc1(cplex*nfft,nspden)

!Local variables-------------------------------
!scalars
 integer :: ii,ir,ishift,ngrad,nspgrad,use_laplacian
 logical :: test_nhat
 real(dp) :: coeff_grho,coeff_grho_corr,coeff_grho_dn,coeff_grho_up
 real(dp) :: coeffim_grho,coeffim_grho_corr,coeffim_grho_dn,coeffim_grho_up
 real(dp) :: gradrho_gradrho1,gradrho_gradrho1_dn,gradrho_gradrho1_up
 real(dp) :: gradrho_gradrho1im,gradrho_gradrho1im_dn,gradrho_gradrho1im_up
 character(len=500) :: msg
!arrays
 real(dp) :: r0(3),r0_dn(3),r0_up(3),r1(3),r1_dn(3),r1_up(3)
 real(dp) :: r1im(3),r1im_dn(3),r1im_up(3)
 real(dp),allocatable :: dnexcdn(:,:),rho1now(:,:,:)
 real(dp),ABI_CONTIGUOUS pointer :: rhor1_ptr(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 if (nkxc/=12*min(nspden,2)-5) then
   msg='Wrong nkxc value for GGA!'
   ABI_BUG(msg)
 end if

!metaGGA contributions are not taken into account here
 use_laplacian=0

!PAW: substract 1st-order compensation density from 1st-order density
 test_nhat=((nhat1dim==1).and.(usexcnhat==0.or.nhat1grdim==1))
 if (test_nhat) then
   ABI_MALLOC(rhor1_ptr,(cplex*nfft,nspden))
   rhor1_ptr(:,:)=rhor1(:,:)-nhat1(:,:)
 else
   rhor1_ptr => rhor1
 end if

!call filterpot(paral_kgb,cplex,gmet,gsqcut,nfft,ngfft,2,qphon,rhor1_ptr)

!Compute the gradients of the first-order density
!rho1now(:,:,1) contains the first-order density, and
!rho1now(:,:,2:4) contains the gradients of the first-order density
 ishift=0 ; ngrad=2
 ABI_MALLOC(rho1now,(cplex*nfft,nspden,ngrad*ngrad))
 call xcden(cplex,gprimd,ishift,mpi_enreg,nfft,ngfft,ngrad,nspden,qphon,rhor1_ptr,rho1now)

!PAW: add "exact" gradients of compensation density
 if (test_nhat.and.usexcnhat==1) then
   rho1now(:,1:nspden,1)=rho1now(:,1:nspden,1)+nhat1(:,1:nspden)
 end if
 if (nhat1grdim==1) then
   do ii=1,3
     rho1now(:,1:nspden,ii+1)=rho1now(:,1:nspden,ii+1)+nhat1gr(:,1:nspden,ii)
   end do
 end if
 if (test_nhat) then
   ABI_FREE(rhor1_ptr)
 end if

!Apply the XC kernel
 nspgrad=2; if (nspden==2) nspgrad=5
 ABI_MALLOC(dnexcdn,(cplex*nfft,nspgrad))

 if (cplex==1) then  ! Treat real case first
   if (nspden==1) then
     do ir=1,nfft
       r0(:)=kxc(ir,5:7) ; r1(:)=rho1now(ir,1,2:4)
       gradrho_gradrho1=dot_product(r0,r1)
       dnexcdn(ir,1)=kxc(ir,1)*rho1now(ir,1,1) + kxc(ir,3)*gradrho_gradrho1
       coeff_grho=kxc(ir,3)*rho1now(ir,1,1) + kxc(ir,4)*gradrho_gradrho1
  !    Reuse the storage in rho1now
       rho1now(ir,1,2:4)=r1(:)*kxc(ir,2)+r0(:)*coeff_grho
     end do
   else
     do ir=1,nfft
       do ii=1,3  ! grad of spin-up ans spin_dn GS rho
         r0_up(ii)=kxc(ir,13+2*ii);r0_dn(ii)=kxc(ir,12+2*ii)-kxc(ir,13+2*ii)
       end do
       r0(:)=r0_up(:)+r0_dn(:)      ! grad of GS rho
       r1_up(:)=rho1now(ir,1,2:4)   ! grad of spin-up rho1
       r1_dn(:)=rho1now(ir,2,2:4)   ! grad of spin-down rho1
       r1(:)=r1_up(:)+r1_dn(:)      ! grad of GS rho1
       gradrho_gradrho1_up=dot_product(r0_up,r1_up)
       gradrho_gradrho1_dn=dot_product(r0_dn,r1_dn)
       gradrho_gradrho1   =dot_product(r0,r1)
       dnexcdn(ir,1)=kxc(ir, 1)*rho1now(ir,1,1)     &
&       +kxc(ir, 2)*rho1now(ir,2,1)     &
&       +kxc(ir, 6)*gradrho_gradrho1_up &
&       +kxc(ir,11)*gradrho_gradrho1
       dnexcdn(ir,2)=kxc(ir, 3)*rho1now(ir,2,1)     &
&       +kxc(ir, 2)*rho1now(ir,1,1)     &
&       +kxc(ir, 7)*gradrho_gradrho1_dn &
&       +kxc(ir,12)*gradrho_gradrho1
       coeff_grho_corr=kxc(ir,11)*rho1now(ir,1,1) &
&       +kxc(ir,12)*rho1now(ir,2,1) &
&       +kxc(ir,13)*gradrho_gradrho1
       coeff_grho_up=kxc(ir,6)*rho1now(ir,1,1)+kxc(ir,8)*gradrho_gradrho1_up
       coeff_grho_dn=kxc(ir,7)*rho1now(ir,2,1)+kxc(ir,9)*gradrho_gradrho1_dn
  !    Reuse the storage in rho1now
       rho1now(ir,1,2:4)=(kxc(ir,4)+kxc(ir,10))*r1_up(:) &
&       +kxc(ir,10)            *r1_dn(:) &
&       +coeff_grho_up         *r0_up(:) &
&       +coeff_grho_corr       *r0(:)
       rho1now(ir,2,2:4)=(kxc(ir,5)+kxc(ir,10))*r1_dn(:) &
&       +kxc(ir,10)            *r1_up(:) &
&       +coeff_grho_dn         *r0_dn(:) &
&       +coeff_grho_corr       *r0(:)
     end do
   end if ! nspden

 else ! if cplex==2
   if (nspden==1) then
     do ir=1,nfft
       r0(:)=kxc(ir,5:7)
       r1(:)  =rho1now(2*ir-1,1,2:4)
       r1im(:)=rho1now(2*ir  ,1,2:4)
       gradrho_gradrho1  =dot_product(r0,r1)
       gradrho_gradrho1im=dot_product(r0,r1im)
       dnexcdn(2*ir-1,1)=kxc(ir,1)*rho1now(2*ir-1,1,1) + kxc(ir,3)*gradrho_gradrho1
       dnexcdn(2*ir  ,1)=kxc(ir,1)*rho1now(2*ir  ,1,1) + kxc(ir,3)*gradrho_gradrho1im
       coeff_grho  =kxc(ir,3)*rho1now(2*ir-1,1,1) + kxc(ir,4)*gradrho_gradrho1
       coeffim_grho=kxc(ir,3)*rho1now(2*ir  ,1,1) + kxc(ir,4)*gradrho_gradrho1im
  !    Reuse the storage in rho1now
       rho1now(2*ir-1,1,2:4)=r1(:)  *kxc(ir,2)+r0(:)*coeff_grho
       rho1now(2*ir  ,1,2:4)=r1im(:)*kxc(ir,2)+r0(:)*coeffim_grho
     end do
   else
     do ir=1,nfft
       do ii=1,3  ! grad of spin-up ans spin_dn GS rho
         r0_up(ii)=kxc(ir,13+2*ii);r0_dn(ii)=kxc(ir,12+2*ii)-kxc(ir,13+2*ii)
       end do
       r0(:)=r0_up(:)+r0_dn(:)          ! grad of GS rho
       r1_up(:)=rho1now(2*ir-1,1,2:4)   ! grad of spin-up rho1
       r1im_up(:)=rho1now(2*ir,1,2:4)   ! grad of spin-up rho1 , im part
       r1_dn(:)=rho1now(2*ir-1,2,2:4)   ! grad of spin-down rho1
       r1im_dn(:)=rho1now(2*ir,2,2:4)   ! grad of spin-down rho1 , im part
       r1(:)=r1_up(:)+r1_dn(:)      ! grad of GS rho1
       r1im(:)=r1im_up(:)+r1im_dn(:)      ! grad of GS rho1, im part
       gradrho_gradrho1_up  =dot_product(r0_up,r1_up)
       gradrho_gradrho1_dn  =dot_product(r0_dn,r1_dn)
       gradrho_gradrho1     =dot_product(r0,r1)
       gradrho_gradrho1im_up=dot_product(r0_up,r1im_up)
       gradrho_gradrho1im_dn=dot_product(r0_dn,r1im_dn)
       gradrho_gradrho1im   =dot_product(r0,r1im)
       dnexcdn(2*ir-1,1)=kxc(ir, 1)*rho1now(2*ir-1,1,1) &
&       +kxc(ir, 2)*rho1now(2*ir-1,2,1) &
&       +kxc(ir, 6)*gradrho_gradrho1_up &
&       +kxc(ir,11)*gradrho_gradrho1
       dnexcdn(2*ir-1,2)=kxc(ir, 3)*rho1now(2*ir-1,2,1) &
&       +kxc(ir, 2)*rho1now(2*ir-1,1,1) &
&       +kxc(ir, 7)*gradrho_gradrho1_dn &
&       +kxc(ir,12)*gradrho_gradrho1
       dnexcdn(2*ir  ,1)=kxc(ir, 1)*rho1now(2*ir  ,1,1) &
&       +kxc(ir, 2)*rho1now(2*ir  ,2,1) &
&       +kxc(ir, 6)*gradrho_gradrho1im_up &
&       +kxc(ir,11)*gradrho_gradrho1im
       dnexcdn(2*ir  ,2)=kxc(ir, 3)*rho1now(2*ir  ,2,1) &
&       +kxc(ir, 2)*rho1now(2*ir  ,1,1) &
&       +kxc(ir, 7)*gradrho_gradrho1im_dn &
&       +kxc(ir,12)*gradrho_gradrho1im
       coeff_grho_corr  =kxc(ir,11)*rho1now(2*ir-1,1,1) &
&       +kxc(ir,12)*rho1now(2*ir-1,2,1) &
&       +kxc(ir,13)*gradrho_gradrho1
       coeffim_grho_corr=kxc(ir,11)*rho1now(2*ir  ,1,1) &
&       +kxc(ir,12)*rho1now(2*ir  ,2,1) &
&       +kxc(ir,13)*gradrho_gradrho1im
       coeff_grho_up  =kxc(ir,6)*rho1now(2*ir-1,1,1)+kxc(ir,8)*gradrho_gradrho1_up
       coeff_grho_dn  =kxc(ir,7)*rho1now(2*ir-1,2,1)+kxc(ir,9)*gradrho_gradrho1_dn
       coeffim_grho_up=kxc(ir,6)*rho1now(2*ir  ,1,1)+kxc(ir,8)*gradrho_gradrho1im_up
       coeffim_grho_dn=kxc(ir,7)*rho1now(2*ir  ,2,1)+kxc(ir,9)*gradrho_gradrho1im_dn
!      Reuse the storage in rho1now
       rho1now(2*ir-1,1,2:4)=(kxc(ir,4)+kxc(ir,10))*r1_up(:) &
&       +kxc(ir,10)            *r1_dn(:) &
&       +coeff_grho_up         *r0_up(:) &
&       +coeff_grho_corr*r0(:)
       rho1now(2*ir-1,2,2:4)=(kxc(ir,5)+kxc(ir,10))*r1_dn(:) &
&       +kxc(ir,10)            *r1_up(:) &
&       +coeff_grho_dn         *r0_dn(:) &
&       +coeff_grho_corr*r0(:)
       rho1now(2*ir  ,1,2:4)=(kxc(ir,4)+kxc(ir,10))*r1im_up(:) &
&       +kxc(ir,10)            *r1im_dn(:) &
&       +coeffim_grho_up       *r0_up(:)   &
&       +coeffim_grho_corr     *r0(:)
       rho1now(2*ir  ,2,2:4)=(kxc(ir,5)+kxc(ir,10))*r1im_dn(:) &
&       +kxc(ir,10)            *r1im_up(:) &
&       +coeffim_grho_dn       *r0_dn(:)   &
&       +coeffim_grho_corr     *r0(:)
     end do
   end if ! nspden

 end if

 vxc1(:,:)=zero
 call xcpot(cplex,gprimd,ishift,use_laplacian,mpi_enreg,nfft,ngfft,ngrad,nspden,&
& nspgrad,qphon,depsxc=dnexcdn,rhonow=rho1now,vxc=vxc1)

!call filterpot(paral_kgb,cplex,gmet,gsqcut,nfft,ngfft,nspden,qphon,vxc1)

 ABI_FREE(dnexcdn)
 ABI_FREE(rho1now)

 DBG_EXIT("COLL")

end subroutine dfpt_mkvxcgga
!!***

!!****f* ABINIT/dfpt_mkvxc_noncoll
!! NAME
!! dfpt_mkvxc_noncoll
!!
!! FUNCTION
!! Compute the first-order change of exchange-correlation potential
!! due to atomic displacement for non-collinear spins: assemble the first-order
!! density change with the frozen-core density change, then use
!! the exchange-correlation kernel.
!!
!! INPUTS
!!  cplex= if 1, real space 1-order functions on FFT grid are REAL,
!!         if 2, COMPLEX
!!  ixc= choice of exchange-correlation scheme
!!  ixcrot= option for rotation of collinear spin potential to non collinear full matrix
!!  kxc(nfft,nkxc)=exchange and correlation kernel (see rhotoxc.F90)
!!  mpi_enreg=information about MPI parallelization
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(18)=contain all needed information about 3D FFT,
!!     see ~abinit/doc/variables/vargs.htm#ngfft
!!  nhat(nfft,nspden*nhatdim)= -PAW only- GS compensation density
!!  nhatdim= -PAW only- 1 if nhat array is used ; 0 otherwise
!!  nhat1(cplex*nfft,nspden*nhat1dim)= -PAW only- 1st-order compensation density
!!  nhat1dim= -PAW only- 1 if nhat1 array is used ; 0 otherwise
!!  nhat1gr(cplex*nfft,nspden,3*nhat1grdim)= -PAW only- gradients of 1st-order compensation density
!!  nhat1grdim= -PAW only- 1 if nhat1gr array is used ; 0 otherwise
!!  nkxc=second dimension of the kxc array
!!  non_magnetic_xc= if true, handle density/potential as non-magnetic (even if it is)
!!  nspden=number of spin-density components
!!  n3xccc=dimension of xccc3d1 ; 0 if no XC core correction is used, otherwise, nfft
!!  optnc=option for non-collinear magnetism (nspden=4):
!!       1: the whole 2x2 Vres matrix is computed
!!       2: only Vres^{11} and Vres^{22} are computed
!!  option=if 0, work only with the XC core-correction,
!!         if 1, treat both density change and XC core correction
!!         if 2, treat only density change
!!  qphon(3)=reduced coordinates for the phonon wavelength (needed if cplex==2).
!!  rhor(nfft,nspden)=GS electron density in real space
!!  rhor1(cplex*nfft,nspden)=1st-order electron density in real space
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!  usexcnhat= -PAW only- 1 if nhat density has to be taken into account in Vxc
!!  vxc(nfft,nspden)=GS XC potential
!!
!!
!! OUTPUT
!!  vxc1(cplex*nfft,nspden)=change in exchange-correlation potential (including
!!   core-correction, if applicable)
!!
!! SOURCE

subroutine dfpt_mkvxc_noncoll(cplex,ixc,kxc,mpi_enreg,nfft,ngfft,nhat,nhatdim,nhat1,nhat1dim,&
&          nhat1gr,nhat1grdim,nkxc,non_magnetic_xc,nspden,n3xccc,optnc,option,qphon,&
&          rhor,rhor1,rprimd,usexcnhat,vxc,vxc1,xccc3d1,ixcrot)

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,ixc,n3xccc,nfft,nhatdim,nhat1dim,nhat1grdim,optnc
 integer,intent(in) :: nkxc,nspden,option,usexcnhat
 logical,intent(in) :: non_magnetic_xc
 type(MPI_type),intent(in) :: mpi_enreg
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: nhat1gr(cplex*nfft,nspden,3*nhat1grdim)
 real(dp),intent(in) :: kxc(nfft,nkxc)
 real(dp),intent(in) :: vxc(nfft,nspden)
 real(dp),intent(in) :: nhat(nfft,nspden*nhatdim),nhat1(cplex*nfft,nspden*nhat1dim)
 real(dp),intent(in),target :: rhor(nfft,nspden),rhor1(cplex*nfft,nspden)
 real(dp),intent(in) :: qphon(3),rprimd(3,3),xccc3d1(cplex*n3xccc)
 real(dp),intent(out) :: vxc1(cplex*nfft,nspden)
 integer,optional,intent(in) :: ixcrot
!Local variables-------------------------------
!scalars
!arrays
 real(dp) :: nhat1_zero(0,0),nhat1gr_zero(0,0,0),tsec(2)
 real(dp),allocatable :: m_norm(:),rhor1_diag(:,:),vxc1_diag(:,:)
 real(dp), ABI_CONTIGUOUS pointer :: mag(:,:),rhor_(:,:),rhor1_(:,:)
! *************************************************************************

!  Non-collinear magnetism
!  Has to locally "rotate" rho(r)^(1) (according to magnetization),
!  Compute Vxc(r)^(1) in the spin frame aligned with \vec{m} and rotate it back

 DBG_ENTER("COLL")
 ABI_UNUSED(nhat1gr)

 call timab(181,1,tsec)

 if(nspden/=4) then
   ABI_BUG('only for nspden=4!')
 end if

 if(nkxc/=2*min(nspden,2)-1) then
   ABI_BUG('nspden=4 works only with LSDA.')
 end if

!Special case: no XC applied
 if (ixc==0.or.nkxc==0) then
   ABI_WARNING('Note that no xc is applied (ixc=0)')
   vxc1(:,:)=zero
   return
 end if



!Treat first LDA
 if(nkxc==1.or.nkxc==3)then

   vxc1(:,:)=zero

!  PAW: possibly substract compensation density
   if ((usexcnhat==0.and.nhatdim==1).or.(non_magnetic_xc)) then
     ABI_MALLOC(rhor_,(nfft,nspden))
     if (usexcnhat==0.and.nhatdim==1) then
       rhor_(:,:) =rhor(:,:)-nhat(:,:)
     else
       rhor_(:,:) =rhor(:,:)
     end if
     if (non_magnetic_xc) then
       if(nspden==2) rhor_(:,2)=rhor_(:,1)*half
       if(nspden==4) rhor_(:,2:4)=zero
     end if
   else
     rhor_ => rhor
   end if
   if ((usexcnhat==0.and.nhat1dim==1).or.(non_magnetic_xc)) then
     ABI_MALLOC(rhor1_,(cplex*nfft,nspden))
     if (usexcnhat==0.and.nhatdim==1) then
       rhor1_(:,:)=rhor1(:,:)-nhat1(:,:)
     else
       rhor1_(:,:)=rhor1(:,:)
     end if
     if (non_magnetic_xc) then
       if(nspden==2) rhor1_(:,2)=rhor1_(:,1)*half
       if(nspden==4) rhor1_(:,2:4)=zero
     end if
   else
     rhor1_ => rhor1
   end if

!  Magnetization
   mag => rhor_(:,2:4)
   ABI_MALLOC(rhor1_diag,(cplex*nfft,2))
   ABI_MALLOC(vxc1_diag,(cplex*nfft,2))
   ABI_MALLOC(m_norm,(nfft))

!  -- Rotate rho(r)^(1)
!  SPr: for option=0 the rhor is not used, only core density xccc3d1
!       rotate_mag is only to compute the m_norm
   call rotate_mag(rhor1_,rhor1_diag,mag,nfft,cplex,mag_norm_out=m_norm,&
&   rho_out_format=2)

!  -- Compute Vxc(r)^(1)=Kxc(r).rho(r)^(1)_rotated
!  Note for PAW: nhat has already been substracted; don't use it in dfpt_mkvxc
!                 (put all nhat options to zero).
!  The collinear routine dfpt_mkvxc wants a general density built as (tr[rho],rho_upup)
   call dfpt_mkvxc(cplex,ixc,kxc,mpi_enreg,nfft,ngfft,nhat1_zero,0,nhat1gr_zero,0,&
&   nkxc,non_magnetic_xc,2,n3xccc,option,qphon,rhor1_diag,rprimd,0,vxc1_diag,xccc3d1)

   !call test_rotations(0,1)

!  -- Rotate back Vxc(r)^(1)
   if (optnc==1) then
     if(present(ixcrot)) then
       call rotate_back_mag_dfpt(option,vxc1_diag,vxc1,vxc,kxc,rhor1_,mag,nfft,cplex,&
&       mag_norm_in=m_norm,rot_method=ixcrot)
     else
       call rotate_back_mag_dfpt(option,vxc1_diag,vxc1,vxc,kxc,rhor1_,mag,nfft,cplex,&
&       mag_norm_in=m_norm)
     end if
   else
     call rotate_back_mag(vxc1_diag,vxc1,mag,nfft,mag_norm_in=m_norm)
     vxc1(:,3:4)=zero
   end if

   ABI_FREE(rhor1_diag)
   ABI_FREE(vxc1_diag)
   ABI_FREE(m_norm)
   if ((usexcnhat==0.and.nhatdim==1).or.(non_magnetic_xc)) then
     ABI_FREE(rhor_)
   end if
   if ((usexcnhat==0.and.nhat1dim==1).or.(non_magnetic_xc)) then
     ABI_FREE(rhor1_)
   end if

 end if ! nkxc=1 or nkxc=3

 call timab(181,2,tsec)

 DBG_EXIT("COLL")

end subroutine dfpt_mkvxc_noncoll
!!***

!!****f* ABINIT/dfpt_mkvxcggadq
!! NAME
!! dfpt_mkvxcggadq
!!
!! FUNCTION
!! Compute the first-order change of exchange-correlation potential
!! in case of GGA functionals
!! Use the q-gradient (Cartesian) of the exchange-correlation kernel.
!!
!! INPUTS
!!  cplex= if 1, real space 1-order functions on FFT grid are REAL,
!!    if 2, COMPLEX
!!  gmet(3,3)=metrix tensor in G space in Bohr**-2.
!!  gprimd(3,3)=dimensional primitive translations in reciprocal space (bohr^-1)
!!  gsqcut=cutoff value on G**2 for sphere inside fft box.
!!  kxc(nfft,nkxc)=exchange and correlation kernel (see below)
!!  mpi_enreg=information about MPI parallelization
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(18)=contain all needed information about 3D FFT
!!  nkxc=second dimension of the kxc array
!!  nspden=number of spin-density components
!!  qdirc= indicates the Cartesian direction of the q-gradient (1,2 or 3)
!!  rhor1tmp(cplex*nfft,2)=array for first-order electron spin-density
!!   in electrons/bohr**3 (second index corresponds to spin-up and spin-down)
!!
!! OUTPUT
!!  vxc1(2*nfft,nspden)=change in exchange-correlation potential
!!
!! NOTES
!!  For the time being, a rather crude coding, to be optimized ...
!!  Content of Kxc array:
!!  Only works with nspden=1
!!   ===== if GGA
!!    if nspden==1:
!!       kxc(:,1)= d2Exc/drho2
!!       kxc(:,2)= 1/|grad(rho)| dExc/d|grad(rho)|
!!       kxc(:,3)= 1/|grad(rho)| d2Exc/d|grad(rho)| drho
!!       kxc(:,4)= 1/|grad(rho)| * d/d|grad(rho)| ( 1/|grad(rho)| dExc/d|grad(rho)| )
!!       kxc(:,5)= gradx(rho)
!!       kxc(:,6)= grady(rho)
!!       kxc(:,7)= gradz(rho)
!!
!! SOURCE

subroutine dfpt_mkvxcggadq(cplex,gprimd,kxc,mpi_enreg,nfft,ngfft,&
&                    nkxc,nspden,qdirc,rhor1,vxc1)

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,nfft,nkxc,nspden,qdirc
 type(MPI_type),intent(in) :: mpi_enreg
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: gprimd(3,3)
 real(dp),intent(in) :: kxc(nfft,nkxc)
 real(dp),intent(in),target :: rhor1(cplex*nfft,nspden)
 real(dp),intent(out) :: vxc1(2*nfft,nspden)

!Local variables-------------------------------
!scalars
 integer :: ii,ir,ishift,ngrad,nspgrad
 real(dp) :: gradrho_gradrho1
 character(len=500) :: msg
!arrays
 real(dp) :: qphon(3)
 real(dp) :: r0(3),r1(3)
 real(dp),allocatable :: ar1(:,:)
 real(dp),allocatable :: a_gradi_r1(:,:)
 real(dp),allocatable :: dadgradn_t1(:,:,:),dadgradn_t2(:,:)
 real(dp),allocatable :: rho1now(:,:,:)
 real(dp),ABI_CONTIGUOUS pointer :: rhor1_ptr(:,:)

! *************************************************************************

 DBG_EXIT("COLL")

 if (nkxc/=7) then
   msg='Wrong nkxc value for GGA in the longwave driver (optdriver=10)!'
   ABI_BUG(msg)
 end if

!Compute the gradients of the first-order density
!rho1now(:,:,1) contains the first-order density, and
!rho1now(:,:,2:4) contains the gradients of the first-order density
 ishift=0 ; ngrad=2
 qphon(:)=zero
 rhor1_ptr => rhor1
 ABI_MALLOC(rho1now,(cplex*nfft,nspden,ngrad*ngrad))
 call xcden(cplex,gprimd,ishift,mpi_enreg,nfft,ngfft,ngrad,nspden,qphon,rhor1_ptr,rho1now)

!Apply the XC kernel
 nspgrad=1
 ABI_MALLOC(ar1,(cplex*nfft,nspgrad))
 ABI_MALLOC(a_gradi_r1,(cplex*nfft,nspgrad))
 ABI_MALLOC(dadgradn_t1,(cplex*nfft,nspgrad,3))
 ABI_MALLOC(dadgradn_t2,(cplex*nfft,nspgrad))
 do ir=1,nfft
   r0(:)=kxc(ir,5:7); r1(:)=rho1now(ir,1,2:4)
   gradrho_gradrho1=dot_product(r0,r1)
   ar1(ir,1)=kxc(ir,2)*rho1now(ir,1,1)
   a_gradi_r1(ir,1)=kxc(ir,2)*r1(qdirc)
   dadgradn_t2(ir,1)=kxc(ir,4)*gradrho_gradrho1*r0(qdirc)
   dadgradn_t1(ir,1,:)=kxc(ir,4)*r0(:)*r0(qdirc)*rho1now(ir,1,1)
 end do
 do ii=1,3
   if (ii==qdirc) dadgradn_t1(:,1,ii)=dadgradn_t1(:,1,ii)+ar1(:,1)
 end do

!Incorporate the terms that do not need further treatment
!(a -i factor is applied here)
 do ir=1,nfft
   ii=2*ir
   vxc1(ii-1,1)=zero
   vxc1(ii,1)= -a_gradi_r1(ir,1) -dadgradn_t2(ir,1)
 end do
 ABI_FREE(rho1now)
 ABI_FREE(a_gradi_r1)
 ABI_FREE(dadgradn_t2)
 ABI_FREE(ar1)

!Now the term whose sum over real-space derivatives has to be computed
 call xcpotdq(dadgradn_t1,cplex,gprimd,ishift,mpi_enreg,nfft, &
& ngfft,ngrad,nspden,nspgrad,vxc1)

 ABI_FREE(dadgradn_t1)

end subroutine dfpt_mkvxcggadq
!!***

!!****f* ABINIT/dfpt_mkvxcgga_n0met
!! NAME
!! dfpt_mkvxcgga_n0met
!!
!! FUNCTION
!! Compute the contribution to the second q-gradient of the metric
!! perturbation that comes from gga XC potentials and depends only
!! on ground state rho
!!
!! INPUTS
!!  beta= indicates the Cartesian direction of the metric perturbation
!!  cplex= if 1, real space 1-order functions on FFT grid are REAL,
!!    if 2, COMPLEX
!!  delta= indicates the Cartesian direction of the first q-gradient
!!  gamma= indicates the Cartesian direction of the second q-gradient
!!  gmet(3,3)=metrix tensor in G space in Bohr**-2.
!!  gprimd(3,3)=dimensional primitive translations in reciprocal space (bohr^-1)
!!  gsqcut=cutoff value on G**2 for sphere inside fft box.
!!  kxc(nfft,nkxc)=exchange and correlation kernel (see below)
!!  mpi_enreg=information about MPI parallelization
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(18)=contain all needed information about 3D FFT
!!  nkxc=second dimension of the kxc array
!!  nspden=number of spin-density components
!!  rho(cplex*nfft,2)=array for ground-state electron spin-density
!!   in electrons/bohr**3 (second index corresponds to spin-up and spin-down)
!!
!! OUTPUT
!!  vxc1(2*nfft,nspden)=change in exchange-correlation potential
!!
!! NOTES
!!  For the time being, a rather crude coding, to be optimized ...
!!  Content of Kxc array:
!!  Only works with nspden=1
!!   ===== if GGA
!!    if nspden==1:
!!       kxc(:,1)= d2Exc/drho2
!!       kxc(:,2)= 1/|grad(rho)| dExc/d|grad(rho)|
!!       kxc(:,3)= 1/|grad(rho)| d2Exc/d|grad(rho)| drho
!!       kxc(:,4)= 1/|grad(rho)| * d/d|grad(rho)| ( 1/|grad(rho)| dExc/d|grad(rho)| )
!!       kxc(:,5)= gradx(rho)
!!       kxc(:,6)= grady(rho)
!!       kxc(:,7)= gradz(rho)
!!
!! SOURCE

subroutine dfpt_mkvxcgga_n0met(beta,cplex,delta,gamma,gprimd,kxc,mpi_enreg,nfft,ngfft,&
&                    nkxc,nspden,rhor,vxc1)

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: beta,cplex,delta,gamma,nfft,nkxc,nspden
 type(MPI_type),intent(in) :: mpi_enreg
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: gprimd(3,3)
 real(dp),intent(in) :: kxc(nfft,nkxc)
 real(dp),intent(in),target :: rhor(cplex*nfft,nspden)
 real(dp),intent(out) :: vxc1(2*nfft,nspden)

!Local variables-------------------------------
!scalars
 integer :: alpha,ii,ir,ishift,ngrad,nspgrad
 real(dp) :: delag,delad,delbd,delbg,deldg
 real(dp) :: gmodsq
 character(len=500) :: msg
!arrays
 real(dp) :: r0(3)
 real(dp),allocatable :: dadgg(:,:),dadgtgn(:,:),gna(:,:),dadgngn_1(:,:),dadgngn_2(:,:)
 real(dp),allocatable :: dadgngn(:,:,:),kro_an(:,:,:),sumgrad(:,:,:)

! *************************************************************************

 DBG_EXIT("COLL")

 if (nkxc/=7) then
   msg='Wrong nkxc value for GGA in the longwave driver (optdriver=10)!'
   ABI_BUG(msg)
 end if

!Kronecker deltas
 delbd=0.0_dp; delbg=0.0_dp; deldg=0.0_dp
 if (beta==delta) delbd=1.0_dp
 if (beta==gamma) delbg=1.0_dp
 if (delta==gamma) deldg=1.0_dp

!Apply the XC kernel
 nspgrad=1
 ABI_MALLOC(dadgg,(cplex*nfft,nspgrad))
 ABI_MALLOC(dadgtgn,(cplex*nfft,nspgrad))
 ABI_MALLOC(gna,(cplex*nfft,nspgrad))
 ABI_MALLOC(dadgngn_1,(cplex*nfft,nspgrad))
 ABI_MALLOC(dadgngn_2,(cplex*nfft,nspgrad))
 do ir=1,nfft
   r0(:)=kxc(ir,5:7)
   gmodsq=r0(1)**2+r0(2)**2+r0(3)**2
   dadgg(ir,1)=kxc(ir,4)*gmodsq*(delbd*r0(gamma)+delbg*r0(delta))
   dadgtgn(ir,1)=two*kxc(ir,4)*r0(beta)*r0(delta)*r0(gamma)
   gna(ir,1)=(delbg*r0(delta)+delbd*r0(gamma)+two*deldg*r0(beta))*kxc(ir,2)
   dadgngn_1(ir,1)=delbd*kxc(ir,4)*rhor(ir,1)*r0(gamma)
   dadgngn_2(ir,1)=delbg*kxc(ir,4)*rhor(ir,1)*r0(delta)
 end do

!Incorporate the terms that do not need further treatment
 do ir=1,nfft
   ii=2*ir
   vxc1(ii-1,1)= -dadgg(ir,1)-dadgtgn(ir,1)-gna(ir,1)
   vxc1(ii,1)= zero
 end do
 ABI_FREE(dadgg)
 ABI_FREE(dadgtgn)
 ABI_FREE(gna)

!Build the last term whose gradient needs to be computed
 ABI_MALLOC(dadgngn,(cplex*nfft,nspgrad,3))
 ABI_MALLOC(kro_an,(cplex*nfft,nspgrad,3))
 ABI_MALLOC(sumgrad,(cplex*nfft,nspgrad,3))
 do alpha=1,3
   delad=0.0_dp; delag=0.0_dp
   if (alpha==delta) delad=1.0_dp
   if (alpha==gamma) delag=1.0_dp
   do ir=1,nfft
     r0(:)=kxc(ir,5:7)
     dadgngn(ir,1,alpha)=(dadgngn_1(ir,1)+dadgngn_2(ir,1))*r0(alpha)
     kro_an(ir,1,alpha)=(delbd*delag+delbg*delad)*rhor(ir,1)*kxc(ir,2)
     sumgrad(ir,1,alpha)=dadgngn(ir,1,alpha)+kro_an(ir,1,alpha)
   end do
 end do

 ABI_FREE(dadgngn_1)
 ABI_FREE(dadgngn_2)
 ABI_FREE(dadgngn)
 ABI_FREE(kro_an)

!Now the term whose sum over real-space derivatives has to be computed.
!(Use the same routine as in the q-gradient of the XC kernel. It saves
! the gradient sum in the imaginary part of vxc1 and includes an additional
! two_pi factor. Need to fix this after the call.)
 ishift=0 ; ngrad=2
 call xcpotdq(sumgrad,cplex,gprimd,ishift,mpi_enreg,nfft, &
& ngfft,ngrad,nspden,nspgrad,vxc1)

 do ir=1,nfft
   ii=2*ir
   vxc1(ii-1,1)=vxc1(ii-1,1)+vxc1(ii,1)/two_pi
   vxc1(ii,1)=zero
 end do


 ABI_FREE(sumgrad)

end subroutine dfpt_mkvxcgga_n0met
!!***

!!****f* ABINIT/dfpt_mkvxcccdq
!! NAME
!!  dfpt_mkvxcccdq
!!
!! FUNCTION
!!  Computes the first q-gradient of the first-order exchange-correlation potential
!!  due to the  pseudocore density.
!!
!! INPUTS
!!  cplex= if 1, real space 1-order functions on FFT grid are REAL,
!!         if 2, COMPLEX
!!  i3dir= reduced direction of the q-gradient
!!  ixc= choice of exchange-correlation scheme
!!  gmet(3,3)=reciprocal space metric tensor in bohr**-2
!!  gprimd(3,3)=reciprocal space dimensional primitive translations
!!  kxc(nfft,nkxc)=exchange and correlation kernel
!!  mpi_enreg=information about MPI parallelization
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(1:18)=integer array with FFT box dimensions and other
!!  nspden=number of spin-density components
!!  nkxc=second dimension of the kxc array. If /=0, the XC kernel must be computed.
!!  qphon(3)=reduced coordinates for the phonon wavelength (needed if cplex==2).
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!  xccc3d1(cplex*nfft)=3D change in core charge density
!!  xccc3d1dq(2*nfft)=q_i3dir-gradient of 3D change in core charge density
!!
!! OUTPUT
!!  vxccc1dq(2*nfft,nspden)= q-gradient of first-order XC potential due to pseudocore charge
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine dfpt_mkvxcccdq(cplex,i3dir,ixc,gprimd,kxc,mpi_enreg,nfft, &
& ngfft,nkxc,nspden,qphon,rprimd,vxccc1dq,xccc3d1,xccc3d2dq)

!Arguments ------------------------------------
 !scalars
 integer , intent(in)  :: cplex,i3dir,ixc,nfft,nkxc,nspden
 type(MPI_type),intent(inout) :: mpi_enreg

 !arrays
 integer,intent(in) :: ngfft(18)
 real(dp), intent(in)  :: gprimd(3,3)
 real(dp), intent(in)  :: kxc(nfft,nkxc)
 real(dp), intent(in)  :: qphon(3),rprimd(3,3)
 real(dp), intent(in)  :: xccc3d1(cplex*nfft)
 real(dp), intent(in)  :: xccc3d2dq(2*nfft)
 real(dp), intent(out) :: vxccc1dq(2*nfft,nspden)

!Local variables-------------------------------
 !scalars
 integer :: ii,ispden,ir,jj,nhat1grdim,option,qcar,usexcnhat,usepaw
 real(dp) :: spin_scale
 logical :: non_magnetic_xc
 !arrays
 real(dp),allocatable :: nhat1(:,:),nhat1gr(:,:,:)
 real(dp),allocatable :: rhor1(:,:), rhor1_cplx(:,:)
 real(dp),allocatable :: vxc1dq_a(:,:),vxc1dq_b(:,:),vxc1dq_car(:,:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 vxccc1dq= zero

!If GGA xc first calculate the contribution from the q gradient of the xc potential
 if (nkxc == 7) then

   !Adapt the format of xccc3d1
   ABI_MALLOC(rhor1,(cplex*nfft,nspden))
   spin_scale=one;if (nspden==2) spin_scale=half
   do ispden=1,nspden
     do ir=1,cplex*nfft
       rhor1(ir,ispden)=xccc3d1(ir)*spin_scale
     end do
   end do

   !The gradient of the potential is calculated in Cartesian coordinates
   ABI_MALLOC(vxc1dq_a,(2*nfft,nspden))
   ABI_MALLOC(vxc1dq_car,(2*nfft,nspden,3))
   do qcar=1,3
     call dfpt_mkvxcggadq(cplex,gprimd,kxc,mpi_enreg,nfft,ngfft,nkxc,nspden,qcar,rhor1,vxc1dq_a)

     !Here we apply an i factor, to compensate the lake of the -i factor in
     !vxc1dq_b (see notes in dfpt_vlocaldq).
     do ir=1,nfft
       ii=2*ir-1
       jj=2*ir
       vxc1dq_car(ii,:,qcar)=-vxc1dq_a(jj,:)
       vxc1dq_car(jj,:,qcar)= vxc1dq_a(ii,:)
     end do
   end do
   ABI_FREE(rhor1)

   !Convert to reduced coordinate i3dir
   vxc1dq_a=zero
   do qcar=1,3
     vxc1dq_a(:,:)=vxc1dq_a(:,:) + gprimd(qcar,i3dir) * vxc1dq_car(:,:,qcar)
   end do
   ABI_FREE(vxc1dq_car)

   !Accumulate this term
   vxccc1dq= vxc1dq_a

   ABI_FREE(vxc1dq_a)
 end if

!Calculate the term with the gradient of the first-order pseudocore density
!Dummy arguments for mkvxc
 ABI_MALLOC(rhor1_cplx,(2*nfft,nspden))
 usexcnhat= 0
 nhat1grdim= 0
 ABI_MALLOC(nhat1gr,(0,0,0))
 nhat1gr(:,:,:)= zero
 usepaw= 0
 ABI_MALLOC(nhat1,(2*nfft,nspden*usepaw))
 nhat1= zero
 non_magnetic_xc= .true.
 option= 0
 ABI_MALLOC(vxc1dq_b,(2*nfft,nspden))
 call dfpt_mkvxc(2,ixc,kxc,mpi_enreg,nfft,ngfft,nhat1,usepaw,nhat1gr,nhat1grdim,nkxc,&
& non_magnetic_xc,nspden,nfft,option,qphon,rhor1_cplx,rprimd,usexcnhat,vxc1dq_b,xccc3d2dq)

!Accumulate this term
 vxccc1dq= vxccc1dq + vxc1dq_b

!Deallocations
 ABI_FREE(vxc1dq_b)
 ABI_FREE(rhor1_cplx)
 ABI_FREE(nhat1)
 ABI_FREE(nhat1gr)

 DBG_EXIT("COLL")

end subroutine dfpt_mkvxcccdq
!!***


end module m_dfpt_mkvxc
!!***
