---
id: infer-types
title: Inferring Types
sidebar_label: Inferring Types
slug: /infer-types
---

It is often useful to wrap functionality of your `@trpc/client` or `@trpc/react` api within other functions. For this purpose, it's necessary to be able to infer input types and output types generated by your `@trpc/server` router.

## Inference Helpers

`@trpc/server` exports the following helper types to assist with inferring these types from the `AppRouter` exported by your `@trpc/server` router:

- `inferProcedureOutput<TProcedure>`
- `inferProcedureInput<TProcedure>`

Let's assume we have this example router:

```ts title='routers/_app.ts'
export const appRouter = t.router({
  user: t.router({
    me: t.procedure.query(() => {
      // ...
      return { id: 1, name: 'John' };
    }),
  }),
  post: t.procedure.input(z.string()).mutation(() => {
    // ...
    return post.id;
    }),
});

export type AppRouter = typeof appRouter;
```

By traversing the router object, you can infer the types of the procedures. The following example shows how to infer the types of the procedures of the example `appRouter`:

```ts title='types.ts'
import type { 
  inferProcedureOutput, 
  inferProcedureInput, 
} from '@trpc/server';
import type { AppRouter } from 'routers/_app';

export type User = inferProcedureOutput<AppRouter['user']['me']>;
//           ^? { id: number, name: string }}

export type PostInput = inferProcedureInput<AppRouter['post']>;
//           ^? string
```

## Infer `TRPClientError`s based on your router

```ts title='src/client.ts'
import type {
  AppRouter
} from '~/server/routers/_app';
import { TRPCClientError } from '@trpc/client';
import { trpc } from './trpc';

export function isTRPCClientError(
  cause: unknown,
): cause is TRPCClientError<AppRouter> {
  return cause instanceof TRPCClientError;
}

async function main() {
  try {
    await trpc.postById.query({ id: 1 })
  } catch (cause) {
    if (isTRPCClientError(cause)) {
      // `cause` is now typed as your router's `TRPCClientError`
      console.log('data', cause.data);
      //                   ^?
    } else {
      // [...]
    }
  }
}


main();
```
