#!/usr/bin/perl -w
# -*- mode: perl; -*-
# $Id: busyboxcmds.pl,v 1.6 2025/06/03 19:01:15 cvsuser Exp $
# busybox command line generation
#
# Copyright Adam Young 2023 - 2025
# All rights reserved.
#
# The applications are free software: you can redistribute it
# and/or modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation, version 3.
#
# Redistributions of source code must retain the above copyright
# notice, and must be distributed with the license document above.
#
# Redistributions in binary form must reproduce the above copyright
# notice, and must include the license document above in
# the documentation and/or other materials provided with the
# distribution.
#
# The applications are distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ==end==
#

use strict;
use warnings 'all';
use Getopt::Long;
use POSIX qw(strftime asctime);
use Capture::Tiny qw(capture capture_merged);

my $output  = "busyboxcmds.h";
my $busybox = "busybox";
my $notcmd  = 0;

my $help    = 0;

Usage() if (0 == GetOptions(
		'output|o=s'    => \$output,
		'busybox|x=s'   => \$busybox,
		'notcmd'        => \$notcmd,
		'help|h'        => \$help)
			|| $help);

Generate();

sub
Generate	#()
{
	# cmd, list
	my %cmdhash;

	if ($notcmd) {
		my ($cmdhelp) = capture_merged {
			system("cmd", "/c help");
		};

		foreach (split(/[\r\n]+/, $cmdhelp)) {
			$cmdhash{lc($1)} = 1
				if (/^([A-Za-z]+)/);
		}
	}

	$cmdhash{'busybox'} = 2;    # remove
	$cmdhash{'install'} = 1;    # rename
	$cmdhash{'link'} = 1;
	$cmdhash{'make'} = 1;

	# busybox, version
	my ($bxout, $bxerr, $bxexit) = capture {
		system($busybox, "--help");
	};
	(0 == $bxexit) || die "cannot exec <${busybox}> : ${bxexit}\n";
	($bxerr eq '') || die "cannot exec <${busybox}> : ${bxerr}\n";
	my @bxhelp = split(/[\r\n]+/, $bxout);

	# busybox, cmd-list
	($bxout, $bxerr, $bxexit) = capture {
		system($busybox, "--list");
	};

	(0 == $bxexit) || die "cannot exec <${busybox}> : ${bxexit}\n";
	($bxerr eq '') || die "cannot exec <${busybox}> : ${bxerr}\n";
	my @bxcmds = split(/[\r\n]+/, $bxout);

	open(FILE, ">${output}") or
		die "cannot create <${output}> : $!\n";
	my $timestamp = asctime(localtime);
	$timestamp = chomp($timestamp);

	# busyboxcmds.h
	print FILE <<"EOT";
// Auto-generated by Makefile ${timestamp}
//
// $bxhelp[0]
// $bxhelp[1]

static const char *busyboxcmds[] = {
EOT

	foreach (@bxcmds) {
		(!/^\[/) || next;
		if (exists $cmdhash{$_}) { # filter cmd's
			print FILE "\t// \"$_\",\n"
				if ($cmdhash{$_} == 1);
		} else {
			print FILE "\t\"$_\",\n";
		}
	}

	print FILE <<"EOT";
	};

//end
EOT
	close(FILE);
}


sub
Usage		# ([message])
{
	print "\nbusyboxcmds.pl @_\n\n" if (@_);
	print <<EOU;

Usage: perl busyboxcmds.pl [options]

Options:
    --output <file>         Output file.
    --busybox <path>        busybox executable fullname.
    --notcmd                dont generate shims for duplicate cmd applets.
    --help                  Help.

EOU
	exit 3;
}

1;

#end
