import React from 'react';
import classNames from 'classnames';
import cuid from 'cuid';
import {
    string,
    shape,
    bool,
    func,
    arrayOf,
    number,
} from 'prop-types';

import CardFooter from './CardFooter/CardFooter';
import prettyFormatDate from '../Helpers/prettyFormat';
import { INFOBIT_TYPE } from '../Helpers/constants';
import { hasTag } from '../Helpers/Helpers';
import {
    getEventBanner,
    getLinkTarget,
    isDateBeforeInterval,
    isDateWithinInterval,
    isDateAfterInterval,
    getCurrentDate,
    getSearchParam,
    removeMarkDown,
} from '../Helpers/general';
import { useConfig, useRegistered } from '../Helpers/hooks';
import {
    stylesType,
    contentAreaType,
    overlaysType,
    footerType,
    tagsType,
    bannerMapType,
} from '../types/card';
import LinkBlocker from './LinkBlocker/LinkBlocker';
import VideoButton from '../Modal/videoButton';

const CardType = {
    cardStyle: string,
    isBookmarked: bool,
    dateFormat: string,
    id: string.isRequired,
    lh: string,
    styles: shape(stylesType),
    disableBookmarkIco: bool,
    onClick: func.isRequired,
    overlays: shape(overlaysType),
    footer: arrayOf(shape(footerType)),
    contentArea: shape(contentAreaType),
    renderBorder: bool,
    renderDivider: bool,
    renderOverlay: bool,
    overlayLink: string,
    hideCTA: bool,
    startDate: string,
    endDate: string,
    cardDate: string,
    modifiedDate: string,
    bannerMap: shape(bannerMapType),
    tags: arrayOf(shape(tagsType)),
    onFocus: func.isRequired,
    origin: string,
    ariaHidden: bool,
    tabIndex: number,
};

const defaultProps = {
    cardStyle: '',
    footer: [],
    styles: {},
    overlays: {},
    dateFormat: '',
    contentArea: {},
    lh: '',
    isBookmarked: false,
    disableBookmarkIco: false,
    renderBorder: true,
    renderDivider: false,
    renderOverlay: false,
    overlayLink: '',
    hideCTA: false,
    startDate: '',
    endDate: '',
    cardDate: '',
    modifiedDate: '',
    tags: [],
    bannerMap: {},
    origin: '',
    ariaHidden: false,
    tabIndex: 0,
};

/**
 * 1/2 image aspect ratio card
 *
 * @component
 * @example
 * const props= {
    id: String,
    styles: Object,
    contentArea: Object,
    overlays: Object,
    renderBorder: Boolean,
    renderOverlay: Boolean,
    overlayLink: String,
 * }
 * return (
 *   <Card {...props}/>
 * )
 */
const Card = (props) => {
    const {
        id,
        footer,
        lh,
        tags,
        cardStyle,
        disableBookmarkIco,
        isBookmarked,
        onClick,
        dateFormat,
        cardDate,
        modifiedDate,
        styles: {
            backgroundImage: image,
            backgroundAltText: altText,
            mnemonic,
            icon: cardIcon,
            iconAlt,
        },
        contentArea: {
            title,
            highlightedTitle,
            detailText: label,
            description,
            highlightedDescription,
            dateDetailText: {
                startTime = '',
                endTime = '',
            } = {},
        } = {},
        overlays: {
            banner: {
                description: bannerDescription,
                fontColor: bannerFontColor,
                backgroundColor: bannerBackgroundColor,
                icon: bannerIcon,
            } = {},
            videoButton: {
                url: videoURL,
            } = {},
            logo: {
                src: logoSrc,
                alt: logoAlt,
                backgroundColor: logoBg,
                borderColor: logoBorderBg,
            } = {},
            label: {
                description: badgeText,
            } = {},
        } = {},
        renderBorder,
        renderDivider,
        renderOverlay,
        overlayLink,
        hideCTA,
        startDate,
        endDate,
        bannerMap,
        onFocus,
        origin,
        ariaHidden,
        tabIndex,
    } = props;

    let bannerBackgroundColorToUse = bannerBackgroundColor;
    let bannerIconToUse = bannerIcon;
    let bannerFontColorToUse = bannerFontColor;
    let bannerDescriptionToUse = bannerDescription;
    let videoURLToUse = videoURL;
    let gateVideo = false;

    const getConfig = useConfig();

    /**
     **** Authored Configs ****
     */
    const i18nFormat = getConfig('collection', 'i18n.prettyDateIntervalFormat');
    const locale = getConfig('language', '');
    const disableBanners = getConfig('collection', 'disableBanners');
    const cardButtonStyle = getConfig('collection', 'button.style');
    const headingLevel = getConfig('collection.i18n', 'cardTitleAccessibilityLevel') || 3;
    const additionalParams = getConfig('collection', 'additionalRequestParams');
    const detailsTextOption = getConfig('collection', 'detailsTextOption');
    const lastModified = getConfig('collection', 'i18n.lastModified');
    const registrationUrl = getConfig('collection', 'banner.register.url');
    const hideDateInterval = getConfig('collection', 'hideDateInterval');
    const showCardBadges = getConfig('collection', 'showCardBadges');
    const altCtaUsed = getConfig('collection', 'dynamicCTAForLiveEvents');
    const ctaAction = getConfig('collection', 'ctaAction');
    const bladeCard = getConfig('collection', 'bladeCard');
    const useCenterVideoPlay = getConfig('collection', 'useCenterVideoPlay');
    const searchEnabled = getConfig('search', 'enabled');

    /**
     * Class name for the card:
     * whether card border should be rendered or no;
     * @type {String}
     */
    const cardClassName = classNames({
        'consonant-Card': true,
        'consonant-u-noBorders': !renderBorder,
        'consonant-hide-cta': hideCTA,
    });

    /**
     * Formatted date string
     * @type {String}
     */
    const prettyDate = startTime ? prettyFormatDate(startTime, endTime, locale, i18nFormat) : '';

    /**
     * Detail text
     * @type {String}
     */
    let detailText = prettyDate || label;
    if (detailsTextOption === 'modifiedDate' && modifiedDate) {
        const localModifiedDate = new Date(modifiedDate);
        detailText = lastModified
            && lastModified.replace('{date}', localModifiedDate.toLocaleDateString())
            || localModifiedDate.toLocaleDateString();
    } else if (detailsTextOption === 'createdDate' && cardDate) {
        const localCreatedDate = new Date(cardDate);
        detailText = localCreatedDate.toLocaleDateString();
    } else if (detailsTextOption === 'staticDate' && cardDate) {
        const staticDate = new Date(cardDate.replace(/Z$/, ''));
        detailText = staticDate.toLocaleDateString();
    }

    /**
     * isGated
     * @type {Boolean}
     */
    const isGated = hasTag(/caas:gated/, tags)
        || hasTag(/caas:card-style\/half-height-featured/, tags)
        || hasTag(/7ed3/, tags)
        || hasTag(/1j6zgcx\/3bhv/, tags);

    /**
     * isRegistered
     * @type {Boolean}
     */
    const isRegistered = useRegistered(false);

    /**
     * isInPerson
     * @type {Boolean}
     */
    const isInPerson = hasTag(/events\/session-format\/in-person/, tags)
        || hasTag(/e505\/3ssk/, tags);

    /**
     * Extends infobits with the configuration data
     * @param {Array} data - Array of the infobits
     * @return {Array} - Array of the infobits with the configuration data added
     */
    function extendFooterData(data) {
        if (!data) return [];

        return data.map((infobit) => {
            // MWPW-129085: Compiler wrongly compiles this object to private, read-only,
            // Created copy so object instance has public methods and properties.
            const copy = { ...infobit };
            if (copy.type === INFOBIT_TYPE.BOOKMARK) {
                if (isGated) {
                    copy.type = INFOBIT_TYPE.GATED;
                }
                return {
                    ...copy,
                    cardId: id,
                    disableBookmarkIco,
                    isBookmarked,
                    onClick,
                };
            } else if (copy.type === INFOBIT_TYPE.DATE) {
                return {
                    ...copy,
                    dateFormat,
                    locale,
                };
            } else if (cardButtonStyle === 'link') {
                copy.type = INFOBIT_TYPE.LINK;
            }
            return {
                ...copy,
                isCta: true,
            };
        });
    }

    /**
     * Extends footer data and extracts an alt CTA if it exists
     * this is important for when overlay links are being used for live events
     * @param {Array} footerData
     * @return {String}
     */
    function getAltCtaLink(footerData) {
        if (!footerData) return '';
        if (footerData.length === 1) {
            const {
                altCta = [],
            } = footerData[0];
            if (altCta.length === 1) {
                return altCta[0].href;
            }
        }
        // default value is an unauthored alt cta
        return '';
    }

    /**
     * Get CTA text from footer data for analytics on overlay cards
     * @param {Array} footerData
     * @return {String}
     */
    function getCtaText(footerData, ctaUsed) {
        if (!footerData) return '';
        if (footerData.length === 1) {
            const {
                altCta = [],
                right = [],
            } = footerData[0];
            if (ctaUsed === 'right' && right.length === 1) {
                return right[0].text;
            } else if (ctaUsed === 'alt' && altCta.length === 1) {
                return altCta[0].text;
            }
            return '';
        }
        return '';
    }

    // Card styles
    const isOneHalf = cardStyle === 'one-half';
    const isThreeFourths = cardStyle === 'three-fourths';
    const isDoubleWide = cardStyle === 'double-wide';
    const isHalfHeight = cardStyle === 'half-height';
    const isProduct = cardStyle === 'product';
    const isText = cardStyle === 'text-card';
    const isFull = cardStyle === 'full-card';
    const isIcon = cardStyle === 'icon-card';
    const isNews = cardStyle === 'news-card';

    const isBlade = cardStyle === 'blade-card';
    const bladeVariant = isBlade
        ? [
            bladeCard.reverse ? 'reverse' : '',
            bladeCard.lightText ? 'light-text' : '',
            bladeCard.transparent ? 'transparent' : '',
        ].filter(Boolean).join(' ')
        : '';
    const isHorizontal = cardStyle === 'horizontal-card';

    // Card elements to show
    const isTitleOnly = isHalfHeight || isThreeFourths || isFull || isIcon
        || isNews || isHorizontal;
    const showHeader = !isProduct;
    const fromDexter = origin === 'Dexter';
    const showBadge = (isOneHalf || isThreeFourths || isFull) && (fromDexter || showCardBadges);
    const showLogo = isOneHalf || isThreeFourths || isFull || isText
        || (isHalfHeight && showCardBadges);
    const showLabel = !isProduct && !isText;
    const showVideoButton = !isProduct && !isText && !isIcon;
    const videoButtonStyle = useCenterVideoPlay && !isHalfHeight ? "center" : "";
    const showText = !isHalfHeight && !isFull && !isNews && !isHorizontal;
    const showFooter = isOneHalf || isProduct || isText || isNews || isBlade;
    const showFooterLeft = !isProduct;
    const showFooterCenter = !isProduct && !altCtaUsed;
    let hideBanner = false;
    let eventBanner = '';
    const hideOnDemandDates = hideDateInterval && isDateAfterInterval(getCurrentDate(), endDate);
    const isEventsCard = getSearchParam(getConfig('collection', 'endpoint'), 'originSelection') === 'events';

    if (isHalfHeight && isGated && !isRegistered) {
        bannerDescriptionToUse = bannerMap.register.description;
        bannerIconToUse = '';
        bannerBackgroundColorToUse = bannerMap.register.backgroundColor;
        bannerFontColorToUse = bannerMap.register.fontColor;
        videoURLToUse = registrationUrl;
        gateVideo = true;
    } else if (startDate && endDate) {
        eventBanner = getEventBanner(startDate, endDate, bannerMap);
        bannerBackgroundColorToUse = eventBanner.backgroundColor;
        bannerDescriptionToUse = eventBanner.description;
        bannerFontColorToUse = eventBanner.fontColor;
        bannerIconToUse = eventBanner.icon;
        if (isHalfHeight) {
            const now = getCurrentDate();
            if (isDateBeforeInterval(now, startDate)) {
                detailText = prettyFormatDate(startDate, endDate, locale, i18nFormat);
            }
        }
    }

    // Events card custom banners
    /* istanbul ignore if */
    if (isEventsCard) {
        hideBanner = isInPerson && eventBanner === bannerMap.onDemand;
        bannerDescriptionToUse = isInPerson && eventBanner === bannerMap.live
            ? 'Live Today'
            : bannerDescriptionToUse;
    }

    const hasBanner = bannerDescriptionToUse
        && bannerFontColorToUse
        && bannerBackgroundColorToUse
        && !hideBanner;

    const headingAria = (videoURL ||
        label || detailText || description || logoSrc || badgeText || (hasBanner && !disableBanners) || !isIcon) ? '' : title;

    const linkBlockerTarget = getLinkTarget(overlayLink, ctaAction);
    const addParams = new URLSearchParams(additionalParams);
    const overlayParams = (additionalParams && addParams.keys().next().value) ? `${overlayLink}?${addParams.toString()}` : overlayLink;
    const isLive = isDateWithinInterval(getCurrentDate(), startDate, endDate);
    const isUpcoming = isDateBeforeInterval(getCurrentDate(), startDate);
    const altCtaLink = getAltCtaLink(footer);
    const ctaText = (altCtaUsed && isUpcoming && altCtaLink !== '') ? getCtaText(footer, 'alt') : getCtaText(footer, 'right');
    const overlay = (altCtaUsed && isLive && altCtaLink !== '') ? altCtaLink : overlayParams;
    const getsFocus = (isHalfHeight && !videoURLToUse)
        || isThreeFourths
        || isFull
        || isDoubleWide
        || isIcon
        || hideCTA;

    // Sanitize markdown before dangerouslySetInnerHTML
    const parseMarkDown = (md = '') => {
        if (searchEnabled) {
            return removeMarkDown(md.replace(/<[^>]*>/g, ''));
        }
        let markup = '';
        if (isProduct && mnemonic) {
            markup += `<img src=${mnemonic} alt="mnemonic" loading="lazy" />`;
        }
        markup += md && md.toString()
            .replace(/<[^>]*>/g, '') // remove any markup <>
            .replaceAll('{**', '<b>')
            .replaceAll('**}', '</b>')
            .replaceAll('{*', '<i>')
            .replaceAll('*}', '</i>');
        return markup;
    };

    return (
        <li
            daa-lh={lh}
            className={`${cardStyle} ${cardClassName} ${bladeVariant}`}
            data-testid="consonant-Card"
            id={id}>
            {showHeader &&
            <div
                data-testid="consonant-Card-header"
                className="consonant-Card-header"
                style={{ backgroundImage: `url("${image}")` }}
                role={(!isIcon && altText) ? 'img' : ''}
                aria-label={!isIcon ? altText : ''}>
                {hasBanner && !disableBanners && !isIcon && !isNews &&
                <span
                    data-testid="consonant-Card-banner"
                    className="consonant-Card-banner"
                    style={({
                        backgroundColor: bannerBackgroundColorToUse,
                        color: bannerFontColorToUse,
                    })}>
                    {bannerIconToUse &&
                        <div
                            className="consonant-Card-bannerIconWrapper">
                            <img
                                alt=""
                                loading="lazy"
                                src={bannerIconToUse}
                                data-testid="consonant-Card-bannerImg" />
                        </div>
                    }
                    <span>{bannerDescriptionToUse}</span>
                </span>
                }
                {showBadge &&
                badgeText &&
                <span
                    className="consonant-Card-badge">
                    {badgeText}
                </span>
                }
                {showVideoButton &&
                videoURL &&
                !isHalfHeight &&
                <VideoButton
                    title={title}
                    videoURL={videoURLToUse}
                    gateVideo={gateVideo}
                    onFocus={onFocus}
                    tabIndex={tabIndex}
                    className={`consonant-Card-videoIco ${videoButtonStyle}`} />
                }
                {showLogo &&
                (logoSrc || (isText && image)) &&
                <div
                    style={({
                        backgroundColor: logoBg,
                        borderColor: logoBorderBg,
                    })}
                    data-testid="consonant-Card-logo"
                    className="consonant-Card-logo">
                    <img
                        // the text card uses the image as logo
                        src={isText ? image : logoSrc}
                        alt={isText ? '' : (logoAlt || '')}
                        loading="lazy"
                        width="32" />
                </div>
                }
                {isIcon &&
                <div
                    data-testid="consonant-Card-logo"
                    className="consonant-Card-logo">
                    <img
                        src={cardIcon}
                        alt=""
                        loading="lazy"
                        width="32"
                        data-testid="consonant-Card-logoImg" />
                </div>
                }
            </div>
            }
            <div
                className="consonant-Card-content">
                {showVideoButton &&
                videoURL &&
                isHalfHeight &&
                <VideoButton
                    title={title}
                    videoURL={videoURLToUse}
                    gateVideo={gateVideo}
                    onFocus={onFocus}
                    tabIndex={tabIndex}
                    className={`consonant-Card-videoIco ${videoButtonStyle}`} />
                }

                {showLabel &&
                detailText &&
                <span
                    data-testid="consonant-Card-label"
                    className="consonant-Card-label">
                    {detailText}
                </span>
                }
                {isIcon &&
                (detailText === '') &&
                <span
                    data-testid="consonant-Card-label"
                    className="consonant-Card-label">
                    {iconAlt}
                </span>
                }
                { (isTitleOnly && highlightedTitle) &&
                    <p
                        data-testid="consonant-Card-title"
                        className="consonant-Card-title">
                        {highlightedTitle}
                    </p>
                }
                { (isTitleOnly && !highlightedTitle) &&
                    <p
                        data-testid="consonant-Card-title"
                        className="consonant-Card-title"
                        title={removeMarkDown(title)}
                        dangerouslySetInnerHTML={{ __html: parseMarkDown(title) }} />
                }
                { (!isTitleOnly && highlightedTitle) &&
                    <p
                        role="heading"
                        {...(headingAria && { 'aria-label': headingAria })}
                        aria-level={headingLevel}
                        data-testid="consonant-Card-title"
                        className="consonant-Card-title"
                        title={removeMarkDown(title)}>
                        {highlightedTitle}
                    </p>
                }
                { (!isTitleOnly && !highlightedTitle) &&
                    <p
                        role="heading"
                        {...(headingAria && { 'aria-label': headingAria })}
                        aria-level={headingLevel}
                        data-testid="consonant-Card-title"
                        className="consonant-Card-title"
                        title={removeMarkDown(title)}
                        dangerouslySetInnerHTML={{ __html: parseMarkDown(title) }} />
                }
                { showText && !isIcon && (
                    highlightedDescription ? (
                        <p
                            data-testid="consonant-Card-text"
                            className="consonant-Card-text">
                            {highlightedDescription}
                        </p>
                    ) : (
                        description && (
                            <p
                                data-testid="consonant-Card-text"
                                className="consonant-Card-text"
                                dangerouslySetInnerHTML={{ __html: parseMarkDown(description) }} />
                        )
                    )
                ) }
                {showFooter &&
                !hideCTA &&
                footer.map(footerItem => (
                    <CardFooter
                        divider={renderDivider || footerItem.divider}
                        isFluid={footerItem.isFluid}
                        key={cuid()}
                        left={(showFooterLeft && !hideOnDemandDates) ?
                            extendFooterData(footerItem.left) : []}
                        center={showFooterCenter ? extendFooterData(footerItem.center) : []}
                        right={extendFooterData(footerItem.right)}
                        altRight={altCtaUsed ? extendFooterData(footerItem.altCta) : []}
                        startDate={startDate}
                        endDate={endDate}
                        cardStyle={cardStyle}
                        onFocus={onFocus}
                        title={title}
                        tabIndex={tabIndex}
                        renderOverlay={renderOverlay} />
                ))}
                {(isThreeFourths || isDoubleWide || isFull)
                    && !renderOverlay
                    && <LinkBlocker
                        target={linkBlockerTarget}
                        link={overlay}
                        title={title}
                        getsFocus={getsFocus || true}
                        daa={ctaText} />}
            </div>
            {(renderOverlay || hideCTA || isHalfHeight || isIcon || isHorizontal)
            && <LinkBlocker
                target={linkBlockerTarget}
                link={overlay}
                title={title}
                getsFocus={getsFocus || true}
                ariaHidden={ariaHidden}
                tabIndex={ariaHidden ? -1 : 0}
                daa={ctaText} />}
        </li>
    );
};

Card.propTypes = CardType;
Card.defaultProps = defaultProps;

export default Card;
