/++
Aedi, a dependency injection framework.

Aedi is a dependency injection framework. It does provide a set of containers that do
IoC, and an interface to configure application components (structs, objects, etc.) managed by framework.

Aim:
The aim of library is to provide a dependency injection solution that is
feature rich, easy to use, easy to learn, and easy to extend up to your needs.

Usage:
The process of configuring components using Aedi consists of following steps:

$(UL
    $(LI Create a container )
    $(LI Register an application component. )
    $(LI Bind dependencies to it. )
    $(LI Repeat process for other components. )
    $(LI Boot container, and use components from it )
    $(LI Shutdown container )
)

Following code example shows how can a IoC container be configured and used:

-------------------
module app;

import aermicioi.aedi;
import std.stdio;

/**
A struct that should be managed by container.
**/
@component
struct Color {
    @setter(cast(ubyte) 255)
    ubyte r;

    @setter(cast(ubyte) 10)
    ubyte g;

    @setter(cast(ubyte) 10)
    ubyte b;
}

/**
Size of a car.
**/
@component // Register component using annotations
struct Size {

    @setter(200UL) // Set property to specific value
    ulong width;

    @setter(150UL)
    ulong height;

    @setter(500UL)
    ulong length;
}

/**
A class representing a car.
**/
@component
class Car {

    private {
        Color color_; // Car color
        Size size_; // Car size
    }

    public {

        @constructor(lref!Size) // Construct component using Size component from IoC container.
        this(Size size) {
            this.size_ = size;
        }

        @property {

            @autowired // Autowire property with a component from IoC container
            Car color(Color color) @safe nothrow {
            	this.color_ = color;

            	return this;
            }

            inout(Color) color() @safe nothrow pure inout {
                return this.color_;
            }

            inout(Size) size() @safe nothrow pure inout {
                return this.size_;
            }
        }
    }
}

class Mercedes : Car {
    this(Size s) {
        super(s);
    }
}

class Volkswagen : Car {
    this(Size s) {
        super(s);
    }
}

@component // Configuration component that creates components managed by container.
class Manufacturer {

    public {
        @component // Add component to container that is constructed by Manufacturer.
        Mercedes makeMercedes() {
            return new Mercedes(Size(201, 150, 501));
        }
    }
}

void print(Car car) {
    writeln("You bought a new ", car.classinfo.name, " with following specs:");
    writeln("Size:\t", car.size());
    writeln("Color:\t", car.color());
}

void main() {
    auto container = singleton(); // 1. Create container that will manage a color
    scope(exit) container.terminate(); // 6. Shutdown the container

    with (container.configure) {

        register!Volkswagen // 2. Register color into container.
            .construct(Size(100, 200, 500))
            .set!"color"("blue".lref); // 3. Bind blue color from container

        register!Color("blue")
            .set!"r"(cast(ubyte) 0)
            .set!"g"(cast(ubyte) 0)
            .set!"b"(cast(ubyte) 255);
    }

    container.scan!app; // 4. Scan app module, and register all annotated components.

    container.instantiate(); // 5. Start the IoC container.

    container.locate!Car.print; // 5. Use component from IoC container.
    container.locate!Mercedes.print;
    container.locate!Volkswagen.print;
}
-------------------

License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:

	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
++/

module minimal;

import aermicioi.aedi;
import std.stdio;

/**
A struct that should be managed by container.
**/
@component
struct Color {
    @setter(cast(ubyte) 255)
    ubyte r;

    @setter(cast(ubyte) 10)
    ubyte g;

    @setter(cast(ubyte) 10)
    ubyte b;
}

/**
Size of a car.
**/
@component // Register component using annotations
struct Size {

    @setter(200UL) // Set property to specific value
    ulong width;

    @setter(150UL)
    ulong height;

    @setter(500UL)
    ulong length;
}

/**
A class representing a car.
**/
@component
class Car {

    private {
        Color color_; // Car color
        Size size_; // Car size
    }

    public {

        @constructor(lref!Size) // Construct component using Size component from IoC container.
        this(Size size) {
            this.size_ = size;
        }

        @property {

            @autowired // Autowire property with a component from IoC container
            Car color(Color color) @safe nothrow {
            	this.color_ = color;

            	return this;
            }

            inout(Color) color() @safe nothrow pure inout {
                return this.color_;
            }

            inout(Size) size() @safe nothrow pure inout {
                return this.size_;
            }
        }
    }
}

class Mercedes : Car {
    this(Size s) {
        super(s);
    }
}

class Volkswagen : Car {
    this(Size s) {
        super(s);
    }
}

@component // Configuration component that creates components managed by container.
class Manufacturer {

    public {
        @component // Add component to container that is constructed by Manufacturer.
        Mercedes makeMercedes() {
            return new Mercedes(Size(201, 150, 501));
        }
    }
}

void print(Car car) {
    writeln("You bought a new ", car.classinfo.name, " with following specs:");
    writeln("Size:\t", car.size());
    writeln("Color:\t", car.color());
}

void main() {
    auto container = singleton(); // 1. Create container that will manage a color
    scope(exit) container.terminate(); // 6. Shutdown the container

    with (container.configure) {

        register!Volkswagen // 2. Register color into container.
            .construct(Size(100, 200, 500))
            .set!"color"("blue".lref); // 3. Bind blue color from container

        register!Color("blue")
            .set!"r"(cast(ubyte) 0)
            .set!"g"(cast(ubyte) 0)
            .set!"b"(cast(ubyte) 255);
    }

    container.scan!minimal; // 4. Scan minimal module, and register all annotated components.

    container.instantiate(); // 5. Start the IoC container.

    container.locate!Car.print; // 5. Use component from IoC container.
    container.locate!Mercedes.print;
    container.locate!Volkswagen.print;
}