/**
Provides annotation based configuration and registration for components.

License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:

	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/
module aermicioi.aedi.configurer.annotation.component_scan;

import aermicioi.aedi.configurer.annotation.annotation;
import aermicioi.aedi.storage.locator;
import aermicioi.aedi.storage.storage;
import aermicioi.aedi.storage.wrapper;
import aermicioi.aedi.container.container;
import aermicioi.aedi.factory.factory;
import aermicioi.aedi.factory.reference;
import aermicioi.aedi.factory.generic_factory;
import aermicioi.aedi.factory.proxy_factory;
import aermicioi.aedi.exception;
import aermicioi.util.traits;
import aermicioi.aedi.factory.wrapping_factory : WrappingFactory;

import std.traits;
import std.meta;
import std.typecons;
import std.conv : to;
import std.algorithm;

/**
Check if a type T is a factory policy.

Check if a type T is a factory policy.
A factory policy is a policy that based on type X, can
create a factory for type X.
**/
enum bool isFactoryPolicy(T, X = Object) = is(T == struct) && is(typeof(&T.createFactory!X) : Z function(Locator!()), Z : Factory!X);

/**
Check if a type T is a configurator policy.

Check if a type T is a configurator policy.
A configurator policy is a policy that given a type
X can create InstanceFactory, InstanceDestructor, or PropertyConfigurer
instances and add them to a factory that implements GenericFactory!X interface, as well
as modify other properties exposed by GenericFactory!X interface.
**/
enum bool isConfiguratorPolicy(T, X : GenericFactory!Z = GenericFactory!Object, Z) =
    is(T == struct) &&
    is(typeof(&T.configure!(X)) : void function(M, Locator!()), M : X);

/**
Create a GenericFactory!T if T is annotated with @component annotation.
**/
struct GenericFactoryPolicy {

    private alias getComponents(alias T) = Filter!(
            isComponentAnnotation,
            allUDAs!T
        );

    /**
    Create a GenericFactory!T if T is annotated with @component annotation.

    Params:
        locator = locator used by factory to fetch it's dependencies

    Returns:
        GenericFactory!T
    **/
    static GenericFactory!T createFactory(T)(Locator!() locator) {
        alias Component = getComponents!T;

        static if (Component.length > 0) {
            debug(annotationScanDebug) pragma(msg, "Creating factory for component ", T);
            return new GenericFactoryImpl!T(locator);
        } else {

            return null;
        }
    };
}

/**
A factory policy that uses annotations implementing factory policy interface on component to instantiate the component.
**/
struct GenericFactoryAnnotationPolicy {

    private alias getGenericFactoryPolicies(T, X) = Filter!(
        chain!(
            ApplyRight!(isFactoryPolicy, X),
            toType
        ),
        allUDAs!T
    );

    /**
    Create a component factory for T using annotations on it that implement factory policy interface.

    Params:
        locator = locator of components used by component factory

    Returns:
        A component factory
    **/
    static GenericFactory!T createFactory(T)(Locator!() locator) {

        alias FactoryPolicies = getGenericFactoryPolicies!(T, T);

        foreach (FactoryPolicy; tuple(FactoryPolicies)) {
            auto factory = FactoryPolicy.createFactory!T(locator);

            if (factory !is null) {
                return factory;
            }
        }

        return null;
    }
}

/**
A factory policy that applies in order a set of factory policies to create component factory.
**/
struct FallbackFactoryPolicy(FactoryPolicies...)
    if (allSatisfy!(isFactoryPolicy, FactoryPolicies)) {

    /**
    Create component factory using one of supplied factory policies

    Params:
        locator = locator of components used by component factory

    Returns:
        A component factory, from the first factory policy that returned an instance
    **/
    static GenericFactory!T createFactory(T)(Locator!() locator) {

        foreach (FactoryPolicy; FactoryPolicies) {
            auto factory = FactoryPolicy.createFactory!T(locator);

            if (factory !is null) {
                return factory;
            }
        }

        return null;
    }
}

/**
Chain a set of configurator policies.
**/
struct ChainedConfiguratorPolicy(ConfiguratorPolicies...)
    if (allSatisfy!(isConfiguratorPolicy, ConfiguratorPolicies)) {

    /**
    Run ConfiguratorPolicies on instantiator in sequential manner

    Params:
        instantiator = factory which is configured by ConfiguratorPolicies
        locator = locator used by factory

    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        foreach (policy; ConfiguratorPolicies) {
            policy.configure(instantiator, locator);
        }
    }
}

/**
Set allocator used by factory to instantiate component T.
**/
struct AllocatorConfiguratorPolicy {

    private alias getAllocators(alias T) = Filter!(
            isAllocatorAnnotation,
            allUDAs!T
        );

    /**
    Set allocator from @allocator annotation into GenericFactory!Z.

    Params:
        instantiator = factory which is set allocator from @allocator annotation
        locator = locator used by factory
    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        static foreach (Allocator; getAllocators!Z) {
            debug(annotationScanDebug) pragma(msg, "Found custom allocator for ", Z, " provisioning with ", typeof(Allocator.allocator));
            instantiator.allocator = Allocator.iallocator;
        }
    }
}

/**
Set callback instance factory from @callbac annotation into GenericFactory!Z
**/
struct CallbackFactoryConfiguratorPolicy {
    private alias getCallbackFactories(alias T) = Filter!(
            isCallbackFactoryAnnotation,
            allUDAs!T
        );

    /**
    Set callback instance factory from @callback annotation into GenericFactory!Z.

    Params:
        instantiator = factory which is set callback instance factory from @callback annotation
        locator = locator used by factory
    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {
        import std.experimental.allocator;

        foreach (CallbackFactory; tuple(getCallbackFactories!Z).expand) {
            debug(annotationScanDebug) pragma(msg,
                "Found callback factory for ",
                Z,
                " provisioning with ",
                typeof(CallbackFactory.dg),
                "(",
                typeof(CallbackFactory.args.expand),
                ")"
            );
            instantiator.setInstanceFactory(callbackFactory(CallbackFactory.dg, CallbackFactory.args.expand));
        }
    }
}

/**
Set value factory that takes component from @value annotation and provides it as a new component.
**/
struct ValueFactoryConfiguratorPolicy {
    private alias getValueFactories(alias T) = Filter!(
            isValueAnnotation,
            allUDAs!T
        );

    /**
    Set value factory that takes component from @value annotation and provides it as a new component.

    Params:
        instantiator = factory which is set value instance factory from @value annotation
        locator = locator used by factory
    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {
        import std.experimental.allocator;

        foreach (ValueAnnotation; tuple(getValueFactories!Z).expand) {
            debug(annotationScanDebug) pragma(msg,
                "Found value factory for ",
                Z,
                " provisioning with ",
                typeof(ValueAnnotation.value)
            );
            instantiator.setInstanceFactory(new ValueInstanceFactory!Z(ValueAnnotation.value));
        }
    }
}

/**
A policy that uses annotations that implement isConfigurerPolicy interface to configure the component factory
**/
struct GenericConfigurerConfiguratorPolicy {
    private alias getGenerics(alias T, X) = Filter!(
                chain!(
                    ApplyRight!(isConfiguratorPolicy, X),
                    toType
                ),
                allUDAs!T
            );

    /**
    Scans for annotations implementing isConfigurerPolicy interface and uses them to configure component factory.

    Params:
        instantiator = factory upon which isConfigurerPolicy annotations are applied
        locator = locator used by factory
    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {
        foreach (Generic; tuple(getGenerics!(Z, T)).expand) {
            debug(annotationScanDebug) pragma(msg, "Found annotation implementing configurer annotation contract ", typeof(Generic), " applying to ", Z, " factory");
            Generic.configure!(T)(instantiator, locator);
        }
    }
}

/**
A policy that configures factory to use callback to destroy created components.
**/
struct CallbackDestructorConfigurerPolicy {
    private alias getCallbackDestructors(alias T, X) = Filter!(
        chain!(
            ApplyRight!(isCallbackDestructor, X),
            toType
        ),
        allUDAs!T
    );

    /**
    Configure instantiator to use callback for destruction of components

    Params:
        instantiator = component factory
        locator = container for component's dependencies
        Z = type of component created
    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        foreach (CallbackDestructor; tuple!(getCallbackDestructors!(Z, T)).expand) {
            debug(annotationsScanDebug) pragma(msg, "Found callback destructor for " ~ fullyQualifiedName!Z);
            instantiator.setInstanceDestructor(callbackDestructor(CallbackDestructor.dg, CallbackDestructor.args));
        }
    }
}

/**
A dummy structure that is providing a simple field for isFieldConfiguratorPolicy interface for testing purposes of configurator templates.
**/
struct ConfiguredFieldTester {

    /**
    Dummy field
    **/
    int field;
}

/**
A dummy structure that is providing a simple method for isMethodConfiguratorPolicy interface for testing purposes of configurator templates.
**/
struct ConfiguredMethodTester {

    /**
    Dummy method
    **/
    void method(int x) {}
}

/**
Check if T is a field configurator policy that operates upon fields of a component.

Params:
    T = the type that is tested for field configurator policy interface
    member = member field of constructed component by GenericFactory!Z
    X = a factory for a component, on which policy can operate
Returns:
    true if it is compliant, false otherwise
**/
enum bool isFieldConfiguratorPolicy(T, string member = "field", X : GenericFactory!Z = GenericFactory!ConfiguredFieldTester, Z) =
    is(T == struct) &&
    is(typeof(&T.configureField!(member, X)) : void function(X, Locator!()));

/**
Check if T is a field configurator policy that operates upon methods of a component.

Params:
    T = the type that is tested for field configurator policy interface
    member = member field of constructed component by GenericFactory!Z
    X = a factory for a component, on which policy can operate

Returns:
    true if it is compliant, false otherwise
**/
enum bool isMethodConfiguratorPolicy(T, string member = "method", X : GenericFactory!Z = GenericFactory!ConfiguredMethodTester, Z) =
    is(T == struct) &&
    is(typeof(&T.configureMethod!(member, X)) : void function(X, Locator!()));

/**
Configurator policy that applies method configurator policies on all public methods of a component
**/
struct MethodScanningConfiguratorPolicy(MethodConfiguratorPolicies...)
    if (allSatisfy!(isMethodConfiguratorPolicy, MethodConfiguratorPolicies)) {

    /**
    Apply a set of method configurator policies on all public methods of component Z

    Params:
        Z = component which will have it's methods scanned, including overloads.
        instantiator = Z component factory
        locator = locator used by method configurator policies
    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {
        foreach (member; __traits(allMembers, Z)) {
            static if ((getProtection!(Z, member) == "public") && isSomeFunction!(__traits(getMember, Z, member))) {
                foreach (methodConfigurer; MethodConfiguratorPolicies) {
                    methodConfigurer.configureMethod!member(instantiator, locator);
                }
            }
        }
    }
}

/**
Configurator policy that applies field configurator policies on all public methods of a component
**/
struct FieldScanningConfiguratorPolicy(FieldConfiguratorPolicies...)
    if (allSatisfy!(isFieldConfiguratorPolicy, FieldConfiguratorPolicies)) {

    /**
    Apply a set of field configurator policies on all public fields of component Z

    Params:
        Z = component which will have it's methods scanned, including overloads.
        instantiator = Z component factory
        locator = locator used by method configurator policies
    **/
    static void configure(T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {
        foreach (member; __traits(allMembers, Z)) {
            static if (isField!(Z, member) && (getProtection!(Z, member) == "public")) {
                foreach (fieldConfigurer; FieldConfiguratorPolicies) {
                    fieldConfigurer.configureField!member(instantiator, locator);
                }
            }
        }
    }
}

/**
Method configurator policy that scans only constructors for @constructor annotation, for using them to instantiate component Z
**/
struct ConstructorMethodConfiguratorPolicy {

    /**
    Checks if scanned method is a constructor and has @constructor annotation, and sets it to be used to construct component if so.

    Params:
        Z = component type
        member = member that should be scanned, only constructors are taken into account
        instantiator = Z component factory, which policy will set constructor to be used to construct component if possible
        locator = locator used to supply dependencies into constructor of component
    **/
    static void configureMethod(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        static if (member == "__ctor") {
            foreach (overload; __traits(getOverloads, Z, member)) {


                alias Configurers = Filter!(
                    isConstructorAnnotation,
                    allUDAs!(overload)
                );

                foreach (Configurer; tuple(staticMap!(toValue, Configurers))) {
                    debug(annotationScanDebug) pragma(msg, "Found elaborate constructor for ", Z, " provisioning with ", toType!Configurer);
                    instantiator.setInstanceFactory(
                        constructorBasedFactory!Z(Configurer.args.expand)
                    );
                }
            }
        }
    }
}

/**
Method policy that scans constructors for @autowired annotation to use them to construct component with dependencies identified by their type.
**/
struct AutowiredConstructorMethodConfiguratorPolicy {

    /**
    Configures instantiator to use @autowired constructor method, to construct the component.

    Params:
        Z = component type
        member = method that is scanned. Only constructors are taken into account
        instantiator = component constructor
        locator = locator for component dependencies
    **/
    static void configureMethod(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        static if (member == "__ctor") {
            foreach (overload; __traits(getOverloads, Z, member)) {

                alias Configurers = Filter!(
                    isAutowiredAnnotation,
                    allUDAs!(overload)
                );

                foreach (Configurer; tuple(staticMap!(toValue, Configurers))) {
                    debug(annotationScanDebug) pragma(msg, "Autowiring constructor of ", Z, " with arguments ", Parameters!(overload));
                    instantiator.setInstanceFactory(
                        constructorBasedFactory!Z(staticMap!(toLref, Parameters!(overload)))
                    );
                }
            }
        }
    }
}

/**
Field configurator policy that will set a field annotated @setter annotation to value contained in it.
**/
struct SetterFieldConfiguratorPolicy {

    /**
    Configures instantiator to inject into field a predefined value or a dependency from @setter annotation

    Params:
        Z = component type
        member = field member of component
        instantiator = component factory
        locator = locator for component dependencies
    **/
    static void configureField(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        alias Configurers = Filter!(
            isSetterAnnotation,
            allUDAs!(__traits(getMember, Z, member))
        );

        foreach (Configurer; tuple(staticMap!(toValue, Configurers))) {
            debug(annotationScanDebug) pragma(msg, "Setting field ", member, " of ", Z, " to ", staticMap!(toType, Configurer.args.expand));
            instantiator.addPropertyConfigurer(fieldConfigurer!(member, Z)(Configurer.args.expand));
        }
    }
}

/**
Field configurator policy that will set a field to value returned by a callback in @callback annotation
**/
struct CallbackFieldConfiguratorPolicy {

    /**
    Configure instantiator to inject return value of callback from @callback annotation into field

    Params:
        member = field that will be injected
        Z = component type
        instantiator = component factory
        locator = locator for component dependencies
    **/
    static void configureField(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        alias Callbacks = Filter!(
            isCallbackConfigurerAnnotation,
            allUDAs!(__traits(getMember, Z, member))
        );

        foreach (Callback; tuple(staticMap!(toValue, Callbacks))) {
            debug(annotationScanDebug) pragma(msg, "Calling callback on field ", member, " of ", Z, " to ", staticMap!(toType, Callback.args.expand));
            instantiator.addPropertyConfigurer(callbackConfigurer!Z(Callback.dg, Callback.args.expand));
        }
    }
}

/**
Field configurator policy that will try to inject a dependency that matches fields type for fields that are annotated with @autowired annotation
**/
struct AutowiredFieldConfiguratorPolicy {

    /**
    Configure instantiator to inject into field a component identified by field's type if field has @autowired annotation.

    Params:
        member = field that will be injected
        Z = component type
        instantiator = component factory
        locator = locator for component dependencies
    **/
    static void configureField(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        alias Callbacks = Filter!(
            isAutowiredAnnotation,
            allUDAs!(__traits(getMember, Z, member))
        );

        foreach (Callback; tuple(staticMap!(toValue, Callbacks))) {
            debug(annotationScanDebug) pragma(msg, "Autowiring field ", member, " of ", Z, " to ", typeof(getMember!(Z, member)));
            instantiator.addPropertyConfigurer(fieldConfigurer!(member, Z)(toLref!(typeof(getMember!(Z, member)))));
        }
    }
}

/**
Method configurator policy that will call a method with resolved arguments from @setter annotation.
**/
struct SetterMethodConfiguratorPolicy {

    /**
    Configure instantiator to call @setter annotated method with arguments from annotation

    Params:
        Z = component type
        member = method that would be called by factory
        instantiator = component factory
        locator = locator for component dependencies
    **/
    static void configureMethod(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        foreach (overload; __traits(getOverloads, Z, member)) {

            alias Configurers = Filter!(
                isSetterAnnotation,
                allUDAs!(overload)
            );

            foreach (Configurer; tuple(staticMap!(toValue, Configurers))) {
                debug(annotationScanDebug) pragma(msg, "Calling method ", member, " of ", Z, " with ", staticMap!(toType, Configurer.args.expand));
                instantiator.addPropertyConfigurer(methodConfigurer!(member, Z)(Configurer.args.expand));
            }
        }
    }
}

/**
Method configurator policy that will call callback from @callback annotated methods.
**/
struct CallbackMethodConfiguratorPolicy {

    /**
    Configre instantiator to call callback from @callback annotation with arguments that are stored in annotation

    Params:
        Z = component type
        member = method which is annotated with @callback. It is supposed that callback will use it somehow in logic.
        instantiator = component factory
        locator = locator for component dependencies
    **/
    static void configureMethod(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {

        foreach (overload; __traits(getOverloads, Z, member)) {

            alias Configurers = Filter!(
                isCallbackConfigurerAnnotation,
                allUDAs!(overload)
            );

            foreach (Configurer; tuple(staticMap!(toValue, Configurers))) {
                debug(annotationScanDebug) pragma(msg, "Calling callback on method ", member, " of ", Z, " with ", staticMap!(toType, Configurer.args.expand));
                instantiator.addPropertyConfigurer(callbackConfigurer!Z(Configurer.dg, Configurer.args.expand));
            }
        }
    }
}

/**
Method configurator policy that will call method annotated with @autowire with arguments extracted from locator by their type.
**/
struct AutowiredMethodConfiguratorPolicy {

    /**
    Configure instantiator to call a method annotated with @autowired with arguments extracted from locator

    Params:
        Z = component type
        member = method that is annotated with @autowired annotation
        instantiator = component factory
        locator = locator for component dependencies
    **/
    static void configureMethod(string member, T : GenericFactory!Z, Z)(T instantiator, Locator!() locator) {
        static if (member != "__ctor") {
            foreach (overload; __traits(getOverloads, Z, member)) {

                alias Configurers = Filter!(
                    isAutowiredAnnotation,
                    allUDAs!(overload)
                );

                foreach (Configurer; tuple(staticMap!(toValue, Configurers))) {

                    debug(annotationScanDebug) pragma(msg, "Calling method ", member, " of ", Z, " with autowired ", Parameters!overload);
                    instantiator.addPropertyConfigurer(methodConfigurer!(member, Z)(staticMap!(toLref, Parameters!overload)));
                }
            }
        }
    }
}

/**
Check if a policy implements transformer interface.

A transformer is a policy that takes a type, and optionally a member of it, and transform it
into a component factory using annotations from component, or member.

Params:
    T = type to be tested for interface implementation
    X = a test object that is used to test templated methods of policy
Returns:
    true if it implements Transformer interface, false otherwise
**/
enum bool isTransformer(T, X = Object) = is(T == struct) && is(typeof(&T.transform!X) : void function (Locator!(), Storage!(ObjectFactory, string)));

/**
A transformer that creates out of a type a GenericFactory for passed type
**/
struct TypeTransformer(FactoryPolicy, ConfigurerPolicy)
    if (isFactoryPolicy!FactoryPolicy && isConfiguratorPolicy!ConfigurerPolicy) {

    /**
    Transform type T into a GenericFactory!T

    Params:
        T = type which is transformed into it's factory
        locator = locator for T's dependencies
    Returns:
        instance of GenericFactory!T, or null if T is not transformable into a factory
    **/
    static auto transform(T)(Locator!() locator)
        if (is(T)) {
        auto instantiator = FactoryPolicy.createFactory!T(locator);

        if (instantiator is null) {
            return instantiator;
        }

        ConfigurerPolicy.configure(instantiator, locator);

        return instantiator;
    }
}

/**
Transformer that wraps results of another transformer in WrappingFactory
**/
struct ObjectFactoryTransformer(TransformerPolicy) {

    /**
    Wrap up results of another transformer into WrappingFactory

    Params:
        T = type of component that is transformed

    Returns:
        null if T is not transformable, or instance of WrappingFactory that wraps result of TransformerPolicy
    **/
    static auto transform(T)(Locator!() locator)
        if (is(T)) {

        auto instantiator = TransformerPolicy.transform!T(locator);

        if (instantiator is null) {
            return cast(WrappingFactory!(typeof(instantiator))) null;
        }

        return new WrappingFactory!(typeof(instantiator))(instantiator);
    }
}

/**
Check if T implements ContainerAdder interface.

A ContainerAdder is component that is responsible to scan
a symbol for it's members, transform them using passed Transformers,
and add them to storage.

Params:
    T = component that is tested for ContainerAdder interface.
    X = a dummy symbol used to test templated methods of component

Note:
    Due to possibility to store any kind of symbol in X (not only types),
    the use of this checker is limited in use for generic testing.
    Ex. ContainerAdder for modules and types are technically different interfaces
    and will fail the checker when are tested both with same X argument
    (module is not a type), though they are conceptually the same.
Returns:
    true if it implements ContainerAdder interface, false otherwise
**/
enum bool isContainerAdder(T, alias X = Object) =
    is(T == struct) &&
    is(typeof(&T.scan!X) : V function (X, Y), X : Locator!(), Y : Storage!(Factory!Object, string), V);

/**
Check if T implements component storing interface.

The responsibility of component storing policy is to
store component's factory into appropiate storage,
by apropiate identity, using information about type T.

Params:
    T = type that is tested for interface compliance
    X = component type against which T's templates are tested to comply to interface
Returns:
    true if it implements the interface, false otherwise
**/
enum bool isComponentStoringPolicy(T, X = Object) =
    is(T == struct) &&
    is(
        typeof(&T.store!X) : void function (F, Locator!(), S),
        F : Factory!Z,
        Z,
        S : Storage!(Factory!Z, string)
    );


/**
A default implementation of component storing policy that looks for @qualifier and @contained annotations to store component factory.
**/
struct ComponentStoringPolicyImpl {

    /**
    Store component factory into storage.

    Store component factory into storage. The storage where component is stored
    on presence of @contained annotation on T, will be extracted from locator, otherwise
    storage passed as argument is used. The identity of component by default is
    it's type FQN, but is overriden by @qualifier annotation.

    Params:
        factory = component factory to be stored
        locator = locator that is used to fetch storage in case of @contained annotation
        storage = storage were component is stored when no @contained annotation is provided
    **/
    static void store(T)(Factory!Object factory, Locator!() locator, Storage!(Factory!Object, string) storage) {

        string identity = fullyQualifiedName!T;
        alias Qualifiers = Filter!(
            isQualifierAnnotation,
            allUDAs!T
        );

        foreach (Qualifier; Qualifiers) {
            debug(annotationScanDebug) pragma(msg, "Found custom identity for ", fullyQualifiedName!T, " ", Qualifier.id);
            identity = Qualifier.id;
        }

        alias ContainedAnnotations = Filter!(
            isContainedAnnotation,
            allUDAs!T
        );

        foreach (ContainedAnnotation; ContainedAnnotations) {
            debug(annotationScanDebug) pragma(msg, "Found custom storage to be used to store component ", fullyQualifiedName!T, " ", ContainedAnnotation.id);
            storage = locator.locate!(Storage!(Factory!Object, string))(ContainedAnnotation.id);
        }

        storage.set(factory, identity);
    }
}

/**
ContainerAdder that chains a set of ContainerAdders on a symbol.
**/
struct ChainedContainerAdder(ContainerAdderPolicies...) {
    import aermicioi.util.traits;

    /**
    Check if at least one ContainerAdder supports passed symbol

    Params:
        T = symbol that at least one ContainerAdder, must know how to scan
    Returns:
        true if at least one ContainerAdder knows how to scan symbol, false otherwise
    **/
    enum bool isSupported(alias T) = anySatisfied!(T, staticMap!(ApplyRight!(getMember, "isSupported"), ContainerAdderPolicies));

    /**
    Apply all ContainerAdders that know how to scan T symbol

    Params:
        T = symbol to be scanned
        locator = locator of components, used by transformer that is applied on members of scanned T symbol
        storage = the storage that will store component factories from transformed members of T symbol
    **/
    static void scan(alias T)(Locator!() locator, Storage!(ObjectFactory, string) storage)
        if (isSupported!T) {
        foreach (ContainerAdderPolicy; ContainerAdderPolicies) {

            static if (ContainerAdderPolicy.isSupported!T) {
                ContainerAdderPolicy.scan!T(locator, storage);
            }
        }
    }
}

/**
Applies a transformer on passed symbol if it is a type.
**/
struct TypeContainerAdder(TypeTransformerPolicy, ComponentStoringPolicy = ComponentStoringPolicyImpl)
    if (isComponentStoringPolicy!ComponentStoringPolicy) {

    /**
    Check if symbol T is a type definition.

    Params:
        T = symbol to be tested
    Returns:
        true if it is a type, false otherwise
    **/
    enum bool isSupported(alias T) = is(T);

    /**
    Transform T component into a factory using TypeTransformerPolicy

    Params:
        T = component type
        locator = component locator used by transformed component factory
        storage = storage wich will contain component factory transformed out of T
    **/
    static void scan(T)(Locator!() locator, Storage!(ObjectFactory, string) storage)
        if (isSupported!T) {

        auto transformed = TypeTransformerPolicy.transform!T(locator);

        if (transformed is null) {
            return;
        }

        ComponentStoringPolicy.store!T(transformed, locator, storage);
    }
}

/**
ContainerAdder that scans a type for inner static types, to transform and store into a storage.
**/
struct InnerTypeContainerAdder(ContainerAdderPolicy) {

    /**
    Check if T symbol is a type

    Params:
        T = symbol to be tested

    Returns:
        true if it is a type, false otherwise
    **/
    enum bool isSupported(alias T) = is(T);

    /**
    Scan type T for inner static types, to transform them and store in storage

    Params:
        T = type that will be scanned for inner static types
        locator = locator of components used by transformed component factories
        storage = storage which will contain transformed component factories
    **/
    static void scan(T)(Locator!() locator, Storage!(ObjectFactory, string) storage)
        if(isSupported!T) {

        foreach (member; __traits(allMembers, T)) {
            static if (isPublic!(T, member) && is(Alias!(__traits(getMember, T, member)))) {

                ContainerAdderPolicy.scan!(getMember!(T, member))(locator, storage);
                scan!(getMember!(T, member))(locator, storage);
            }
        }
    }
}

/**
ContainerAdder that will scan a module for it's members, to transform into component factories and add them into a storage
**/
struct ModuleContainerAdder(ContainerAdderPolicy) {

    /**
    Check if T symbol is a module.

    Params:
        T = symbol to be tested

    Returns:
        true if it is a module, false otherwise
    **/
    enum bool isSupported(alias T) = is(typeof({mixin("import " ~ fullyQualifiedName!T ~ ";");}));

    /**
    Scan module T, transform it's members into component factories, and store them into storage.

    Params:
        T = module that is scanned
        locator = locator of components used by transformed component factories
        storage = storage which will contain component factories
    **/
    static void scan(alias T)(Locator!() locator, Storage!(ObjectFactory, string) storage)
        if (isSupported!T) {

        foreach (member; __traits(allMembers, T)) {
            static if (isPublic!(T, member) && is(Alias!(__traits(getMember, T, member)))) {

                ContainerAdderPolicy.scan!(getMember!(T, member))(locator, storage);
            }
        }
    }
}

/**
ContainerAdder that will scan a type for it's methods, and use them to create component factories out of their return type
**/
struct FactoryMethodContainerAdder(ComponentStoringPolicy = ComponentStoringPolicyImpl)
    if (isComponentStoringPolicy!ComponentStoringPolicy) {

    /**
    Check if symbol T is a type

    Params:
        T = symbol to be tested

    Returns:
        true if it is a type, false otherwise
    **/
    enum bool isSupported(alias T) = is(T);

    /**
    Scan T's methods, for methods annotated with @component annotation, transform them into component factories
    that will use them to create components of returned type.

    Params:
        T = type to be scanned
        locator = locator of components used by component factories
        storage = storage which will contain component factories
    **/
    static void scan(alias T)(Locator!() locator, Storage!(ObjectFactory, string) storage)
        if (isSupported!T) {

        foreach (member; __traits(allMembers, T)) {
            static if (isPublic!(T, member) && isSomeFunction!(__traits(getMember, T, member))) {

                foreach (overload; __traits(getOverloads, T, member)) {

                    alias FactoryMethods = Filter!(
                        isComponentAnnotation,
                        allUDAs!overload
                    );

                    foreach (FactoryMethod; FactoryMethods) {
                        debug(annotationScanDebug) pragma(msg, "Found factory method ", member, " on component ", T);
                        auto factory = new WrappingFactory!(GenericFactoryImpl!(ReturnType!overload))(
                            new GenericFactoryImpl!(ReturnType!overload)(locator)
                        );

                        if (factory !is null) {
                            static if (__traits(isStaticFunction, overload)) {
                                auto instanceFactory = factoryMethodBasedFactory!(T, member)(staticMap!(toLref, Parameters!overload));
                            } else {
                                auto instanceFactory = factoryMethodBasedFactory!(T, member)(staticMap!(toLref, AliasSeq!(T, Parameters!overload)));
                            }

                            import aermicioi.aedi.storage.decorator : Decorator;

                            static if (is(typeof(factory) : Decorator!X, X : GenericFactory!Z, Z)) {
                                factory.decorated.setInstanceFactory = instanceFactory;
                            } else static if (is(typeof(factory) : GenericFactory!Z, Z) && !is(Z == Object)) {
                                factory.setInstanceFactory = instanceFactory;
                            }

                            ComponentStoringPolicy.store!(ReturnType!overload)(factory, locator, storage);
                        }
                    }
                }
            }
        }
    }
}

/**
Register an object into storage using annotations provided in it.

An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    T = type of object to be registered
    storage = the storage where to register the object
    locator = the locator used to find object dependencies
    id = identity by which object will be stored in storage
**/
deprecated
auto componentScan(T)(Storage!(ObjectFactory, string) storage, Locator!() locator, string id) {
    auto factory = componentScanImpl!T(locator);

    alias SubComponents = staticMap!(
        partialPrefixed!(
            getMember,
            T
        ),
        Filter!(
            templateAnd!(
                partialPrefixed!(
                    partialSuffixed!(
                        isProtection,
                        "public"
                    ),
                    T
                ),
                chain!(
                    isType,
                    partialPrefixed!(
                        getMember,
                        T
                    )
                ),
                chain!(
                    isClass,
                    partialPrefixed!(
                        getMember,
                        T
                    )
                )
            ),
            __traits(allMembers, T)
        )
    );

    static if (SubComponents.length > 0) {
        storage.componentScan!SubComponents(locator);
    }

    if (factory !is null) {
        storage.set(new WrappingFactory!(Factory!T)(factory), id);
    }

    return storage;
}

/**
Register an object into storage by it's type FQN using annotations provided in it.

An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    T = type of object to be registered
    storage = the storage where to register the object
**/
deprecated
auto componentScan(T)(Storage!(ObjectFactory, string) storage, Locator!() locator) {

    alias qualifiers = Filter!(isQualifier, allUDAs!T);

    static if (qualifiers.length > 0) {
        return storage.componentScan!T(locator, qualifiers[0].id);
    } else {
        return storage.componentScan!T(locator, name!T);
    }
}

/**
ditto
**/
deprecated
auto componentScan(T)(ConfigurableContainer storage) {

    return storage.componentScan!T(storage);
}

/**
Register an object into storage by I's interface FQN that it implements using annotations provided in it.

An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    I = the inteface that object implements.
    T = type of object to be registered
    storage = the storage where to register the object
**/
deprecated
auto componentScan(I, T)(Storage!(ObjectFactory, string) storage, Locator!() locator)
    if (is(I == interface) && is(T == class) && is(T : I)) {

    alias qualifiers = Filter!(
        isQualifier,
        allUDAs!I
    );

    static if (qualifiers.length > 0) {
        return storage.componentScan!T(locator, qualifier[0].id);
    } else {
        return storage.componentScan!T(locator, name!I);
    }
}

/**
ditto
**/
deprecated
auto componentScan(I, T)(ConfigurableContainer storage)
    if (is(I == interface) && is(T == class) && is(T : I)) {

    return storage.componentScan!(I, T)(storage);
}

/**
Register a set of objects by it's type, or implemented interface into a storage.

When registering an object by it's interface, next to interface it is required to specify the original type of object.
Note: An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    I = the inteface that object implements.
    T = type of object to be registered
    storage = the storage where to register the object
**/
deprecated
auto componentScan(T, V...)(Storage!(ObjectFactory, string) storage, Locator!() locator) {
    storage.componentScan!T(locator);
    return storage.componentScan!V(locator);
}

/**
ditto
**/
deprecated
auto componentScan(T, V...)(ConfigurableContainer storage) {

    return storage.componentScan!(T, V)(storage);
}

/**
ditto
**/
deprecated
auto componentScan(I, T, V...)(Storage!(ObjectFactory, string) storage, Locator!() locator)
    if (is(I == interface) && is(T == class) && is(T : I)) {
    storage.componentScan!(I, T)(locator);

    return storage.componentScan!(V)(locator);
}

/**
ditto
**/
deprecated
auto componentScan(I, T, V...)(ConfigurableContainer storage)
    if (is(I == interface) && is(T == class) && is(T : I)) {

    return storage.componentScan!(I, T, V)(storage);
}

/**
Scan a module and register all public objects that are annotated with @component annotation.

Note: An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    Module = module to scan for components.
    storage = the storage where to register the object
    locator = the locator used to fetch registered object's dependencies.
**/
deprecated
auto componentScan(alias Module)(Storage!(ObjectFactory, string) storage, Locator!() locator)
    if (startsWith(Module.stringof, "module ")) {

    alias components = staticMap!(
        partialPrefixed!(
            getMember,
            Module
        ),
        Filter!(
            templateAnd!(
                partialSuffixed!(
                    partialPrefixed!(
                        isProtection,
                        Module
                    ),
                    "public"
                ),
                chain!(
                    isType,
                    partialPrefixed!(
                        getMember,
                        Module
                    )
                ),
                chain!(
                    isClass,
                    partialPrefixed!(
                        getMember,
                        Module
                    )
                )
            ),
            __traits(allMembers, Module)
        )
    );

    storage.componentScan!components(locator);

    return storage;
}

/**
ditto
**/
deprecated
auto componentScan(alias M)(ConfigurableContainer storage)
    if (startsWith(M.stringof, "module")) {

    return storage.componentScan!(M)(storage);
}

/**
Scan a set of modules and register all public objects that are annotated with @component annotation.

Due to limitations of D language currently it is impossible to recursively scan all public imports of a module to register all
depencies of a package. Each particular module should be specified in order to register dependencies.
Note: An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    M = current module to scan.
    V = rest set of modules waiting for scan.
    storage = the storage where to register the object
    locator = the locator used to fetch registered object's dependencies.
**/
deprecated
auto componentScan(alias M, V...)(Storage!(ObjectFactory, string) storage, Locator!() locator)
    if (startsWith(M.stringof, "module")) {
    storage.componentScan!M(locator);
    return storage.componentScan!V(locator);
}

/**
ditto
**/
deprecated
auto componentScan(alias M, V...)(ConfigurableContainer storage)
    if (startsWith(M.stringof, "module")) {

    return storage.componentScan!(M, V)(storage);
}

/**
Register an object into a storage contained in storageLocator and identified by @container annotation using annotations provided in it.

An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    T = type of object to be registered
    storageLocator = the locator from which to fetch storage for object
    locator = locator used to find dependencies for object
    id = identity by which object will be stored in storage
**/
deprecated
auto componentScan(T, R : Locator!())(R storageLocator, Locator!() locator, string id)
    if (!is(R : Storage!(ObjectFactory, string))) {

    alias containers = Filter!(
        isContained,
        allUDAs!T
    );

    static if (containers.length > 0) {
        string storageId = containers[0].id;
    } else {
        string storageId = "singleton";
    }

    auto storage = storageLocator.locate!(Storage!(ObjectFactory, string))(storageId);

    if (storage !is null) {

        storage.componentScan!T(locator, id);
    } else {

        throw new NotFoundException("Could not find storage to save factory for object of identity " ~ id);
    }

    return storageLocator;
}

/**
ditto
**/
deprecated
auto componentScan(T, R : Locator!())(R locator, string id)
    if (!is(R : Storage!(ObjectFactory, string))) {
    return locator.componentScan!T(locator, id);
}

/**
ditto
**/
deprecated
auto componentScan(T, R : Locator!())(R storageLocator, Locator!() locator)
    if (!is(R : Storage!(ObjectFactory, string))) {

    alias qualifiers = Filter!(
        isQualifier,
        allUDAs!T
    );

    static if (qualifiers.length > 0) {
        return storageLocator.componentScan!T(locator, qualifiers[0].id);
    } else {
        return storageLocator.componentScan!T(locator, name!T);
    }
}

/**
ditto
**/
deprecated
auto componentScan(T, R : Locator!())(R locator)
    if (!is(R : Storage!(ObjectFactory, string))) {
    return locator.componentScan!T(locator);
}

/**
ditto
**/
template componentScan(T, V...)
    if((V.length > 0)) {

    /**
    ditto
    **/
    deprecated
auto componentScan(R : Locator!())(R storageLocator, Locator!() locator)
        if (!is(R : Storage!(ObjectFactory, string))) {
        .componentScan!T(storageLocator, locator);

        return .componentScan!V(storageLocator, locator);
    }

    /**
    ditto
    **/
    deprecated
auto componentScan(R : Locator!())(R locator)
        if (!is(R : Storage!(ObjectFactory, string))) {
        .componentScan!T(locator, locator);

        return .componentScan!V(locator, locator);
    }
}

/**
Register an object into a storage contained in storageLocator and identified by @container annotation using annotations provided in it.

An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    I = interface implemented by object, by which to register it.
    T = type of object to be registered
    storageLocator = locator used to find storage for object
    locator = locator used to find dependencies for object
**/
deprecated
auto componentScan(I, T, R : Locator!())(R storageLocator, Locator!() locator)
    if (is (I == interface) && is (T == class) && is (T : I) && !is(R : Storage!(ObjectFactory, string))) {
    alias qualifiers = Filter!(
        isQualifier,
        allUDAs!I
    );

    static if (qualifiers.length > 0) {
        return storageLocator.componentScan!T(locator, qualifiers[0].id);
    } else {
        return storageLocator.componentScan!T(locator, name!I);
    }
}

/**
ditto
**/
deprecated
auto componentScan(I, T, R : Locator!())(R locator)
    if (is (I == interface) && is (T == class) && is (T : I) && !is(R : Storage!(ObjectFactory, string))) {

    return locator.componentScan!(I, T)(locator);
}

/**
ditto
**/
template componentScan(I, T, V...)
    if (is (I == interface) && is (T == class) && is (T : I) && (V.length > 0)) {

    /**
    ditto
    **/
    deprecated
auto componentScan(R : Locator!())(R storageLocator, Locator!() locator)
        if (!is(R : Storage!(ObjectFactory, string))) {

        .componentScan!(I, T)(storageLocator, locator);

        return .componentScan!(V)(storageLocator, locator);
    }

    /**
    ditto
    **/
    deprecated
auto componentScan(R : Locator!())(R locator)
        if (!is(R : Storage!(ObjectFactory, string))) {

        .componentScan!(I, T)(locator);

        return .componentScan!(V)(locator);
    }
}

/**
Register module's objects into a storage contained in storageLocator and identified by @container annotation using annotations provided in it.

An object will be registered in storage only in case when it is annotated with @component annotation. In case when no @component
annotation is found, object is not registered in storage.

Params:
    M = module to scan for instantiable objects.
    storageLocator = locator used to find storage for objects
    locator = locator used to find object dependencies
**/
deprecated
auto componentScan(alias M, R : Locator!())(R storageLocator, Locator!() locator)
    if (M.stringof.startsWith("module ") && !is(R : Storage!(ObjectFactory, string)))  {

    alias components = getPossibleComponents!M;

    storageLocator.componentScan!components(locator);

    return storageLocator;
}

/**
ditto
**/
deprecated
auto componentScan(alias M, R : Locator!())(R locator)
    if (M.stringof.startsWith("module ") && !is(R : Storage!(ObjectFactory, string)))  {

    locator.componentScan!M(locator);

    return locator;
}

/**
ditto
**/
template componentScan(alias M, V...)
    if (M.stringof.startsWith("module ") && (V.length > 0)) {

    /**
    ditto
    **/
    deprecated
auto componentScan(R : Locator!())(R locatorStorage, Locator!() locator)
        if (!is(R : Storage!(ObjectFactory, string))) {

        .componentScan!M(locatorStorage, locator);
        return .componentScan!V(locatorStorage, locator);
    }

    /**
    ditto
    **/
    deprecated
auto componentScan(R : Locator!())(R locator)
        if (!is(R : Storage!(ObjectFactory, string))) {

        .componentScan!M(locator);
        return .componentScan!V(locator);
    }
}

/**
Implementation of field scanning configurator policy, with built in field configurators.
**/
alias FieldScanningConfiguratorPolicyImpl = FieldScanningConfiguratorPolicy!(
        SetterFieldConfiguratorPolicy,
        CallbackFieldConfiguratorPolicy,
        AutowiredFieldConfiguratorPolicy
    );

/**
Implementation of method scanning configurator policy, with built in method configurators.
**/
alias MethodScanningConfiguratorPolicyImpl = MethodScanningConfiguratorPolicy!(
        ConstructorMethodConfiguratorPolicy,
        AutowiredConstructorMethodConfiguratorPolicy,
        SetterMethodConfiguratorPolicy,
        CallbackMethodConfiguratorPolicy,
        AutowiredMethodConfiguratorPolicy
    );

/**
Implementation of configurator policy, with built in configurators
**/
alias ConfiguratorPolicyImpl = ChainedConfiguratorPolicy!(
            FieldScanningConfiguratorPolicyImpl,
            MethodScanningConfiguratorPolicyImpl,
            AllocatorConfiguratorPolicy,
            CallbackFactoryConfiguratorPolicy,
            ValueFactoryConfiguratorPolicy,
            GenericConfigurerConfiguratorPolicy
        );

/**
Implementation of factory policy, with built in factory creators
**/
alias FactoryPolicyImpl = FallbackFactoryPolicy!(
    GenericFactoryPolicy,
    GenericFactoryAnnotationPolicy
);

/**
Implementation of type to type factory transformer, with built in funcionality
**/
alias TypeTransformerImpl = TypeTransformer!(
        FactoryPolicyImpl,
        ConfiguratorPolicyImpl
    );

/**
Implementation of object wrapping factory wrapping TypeTransformerImpl
**/
alias ObjectFactoryTransformerImpl =
    ObjectFactoryTransformer!(
        TypeTransformerImpl
    );

/**
Implementation of module container adder, featuring built in scanners
**/
alias ModuleContainerAdderImpl(TransformerPolicy = ObjectFactoryTransformerImpl) = ModuleContainerAdder!(
        ChainedContainerAdder!(
            TypeContainerAdder!TransformerPolicy,
            InnerTypeContainerAdder!(TypeContainerAdder!TransformerPolicy),
            FactoryMethodContainerAdder!(),
        )
    );

/**
Customizable implementation of container adder, with built in functionality
**/
alias ContainerAdderImpl(TransformerPolicy = ObjectFactoryTransformerImpl) = ChainedContainerAdder!(
        TypeContainerAdder!TransformerPolicy,
        InnerTypeContainerAdder!(TypeContainerAdder!TransformerPolicy),
        FactoryMethodContainerAdder!(),
        ModuleContainerAdderImpl!TransformerPolicy
    );

/**
Template for defining scanning functions instantiated with particular container adder policy.

The functions defined in this template mixin are the entry point for running scans over symbols that
are desired to be added into a container. The template will instantiate $(D_INLINECODE scan) family of functions
that will use passed container adder policy to scan symbols passed to them. It is advised to define your own
set of scanning methods in case when additional scanning and transformation logic is expected.
**/
mixin template Scanner(ContainerAdderPolicy) {

    /**
    Scan symbol T for possible components using ContainerAdderPolicy

    Params:
        storage = storage that will contain component factories
        locator = locator of components used to by component factories
    **/
    void scan(alias T)(Storage!(ObjectFactory, string) storage, Locator!() locator) {
        debug(annotationScanDebug) pragma(msg, "Scanning ", fullyQualifiedName!T, " for possible components");

        ContainerAdderPolicy.scan!T(locator, storage);
    }

    /**
    ditto
    **/
    void scan(alias T, X...)(Storage!(ObjectFactory, string) storage, Locator!() locator) {

        scan!T(storage, locator);

        static if (X.length > 0) {

            scan!X(storage, locator);
        }
    }

    /**
    Scan symbol T for possible components using ContainerAdderPolicy

    Params:
        storage = identity by which to search storage in locator, that will be used to store components
        locator = locator of components used to by component factories
    **/
    void scan(alias T)(string storage, Locator!() locator) {

        scan!T(locator.locate!(Storage!(ObjectFactory, string))(storage), locator);
    }

    /**
    ditto
    **/
    void scan(alias T, X...)(string storage, Locator!() locator) {

        scan!(T, X)(locator.locate!(Storage!(ObjectFactory, string))(storage), locator);
    }

    /**
    Scan symbol T for possible components using ContainerAdderPolicy

    Params:
        container = container where to store and from which to locate dependencies for components
    **/
    void scan(alias T)(ConfigurableContainer container) {

        scan!T(container, container);
    }


    void scan(alias T, St : Storage!(ObjectFactory, string) = Storage!(ObjectFactory, string))(Locator!() locator) {

        scan!T(locator.locate!St, locator);
    }

    /**
    ditto
    **/
    void scan(alias T, St : Storage!(ObjectFactory, string), X...)(Locator!() locator) {

        scan!(T, St)(locator);

        static if (X.length > 0) {

            scan!X(locator);
        }
    }

    /**
    ditto
    **/
    void scan(alias T, X...)(Locator!() locator) {
        scan!T(locator);

        static if (X.length > 0) {

            scan!X(locator);
        }
    }
}

/**
Default implementation of $(D_INLINECODE scan) family of functions featuring all scanning features provided by library.
**/
mixin Scanner!(ContainerAdderImpl!());

deprecated
auto componentScanImpl(T)(Locator!() locator) {

    return TypeTransformerImpl.transform!T(locator);
}

private template isQualifier(alias T) {
    alias isQualifier = isQualifier!(typeof(T));
}

private template isQualifier(T) {
    enum bool isQualifier = is(T == QualifierAnnotation);
}

private template isContained(alias T) {
    alias isContained = isContained!(typeof(T));
}

private template isContained(T) {
    enum bool isContained = is(T == ContainedAnnotation);
}

private template isValue(T) {
    enum bool isValue = is (typeof(T));
}

private template isReturnTypeEq(alias symbol, Type)
    if (isSomeFunction!symbol) {
    enum bool isReturnTypeEq = is(ReturnType!symbol : Type);
}

private template isNamedTemplate(alias T, string name) {
    enum bool isNamedTemplate = isTemplate!T && (identifier!T == name);
}

private template isEqByFQDN(alias first, alias second) {
    enum bool isEqByFQDN = fullyQualifiedName!first == fullyQualifiedName!second;
}

private template instantiatonToTemplate(alias T, alias Template = T) {
    static if (isTemplateInstantiationOf!(T, Template)) {
        alias instantiatonToTemplate = Template;
    } else static if (isTemplate!T) {
        alias instantiatonToTemplate = T;
    }
}

private enum bool isTemplateInstantiationOf(T, alias Template) = is(T : Template!(Z), Z...);
private enum bool isTemplateInstantiationOf(alias T, alias Template) = is(typeof(T) : Template!(Z), Z...);

private template identifierEq(alias T, string identity) {
    enum bool identifierEq = identifier!T == identity;
}

private template allUDAs(alias symbol) {
    alias allUDAs = AliasSeq!(__traits(getAttributes, symbol));
}

private template toValue(T) {
    enum auto toValue = T();
}

private template toValue(alias T) {
    alias toValue = T;
}

private template isType(alias T) {
    static if (__traits(compiles, () { T z = T.init; })) {

        enum bool isType = true;
    } else {

        enum bool isType = false;
    }
}

private template isClass(alias T) {
    enum bool isClass = is(typeof(T) == class);
}

private template isClass(T) {
    enum bool isClass = is(T == class);
}

private template isStruct(alias T) {
    enum bool isStruct = is(typeof(T) == struct);
}

private template isStruct(T) {
    enum bool isStruct = is(T == struct);
}

private template getPublicAggregateMembers(alias Symbol) {
    alias getPublicAggregateMembers = Filter!(
        templateAnd!(
            partialSuffixed!(
                partialPrefixed!(
                    isProtection,
                    Symbol
                ),
                "public"
            ),
            chain!(
                hasMembers,
                partialPrefixed!(
                    getMember,
                    Symbol
                )
            )
        ),
        __traits(allMembers, Symbol)
    );
}

private template getPossibleComponents(alias Symbol) {
    alias getPossibleComponents = staticMap!(
        partialPrefixed!(
            getMember,
            Symbol
        ),
        Filter!(
            templateAnd!(
                partialSuffixed!(
                    partialPrefixed!(
                        isProtection,
                        Symbol
                    ),
                    "public"
                ),
                chain!(
                    isType,
                    partialPrefixed!(
                        getMember,
                        Symbol
                    )
                ),
                chain!(
                    templateOr!(
                        isClass,
                        isStruct
                    ),
                    partialPrefixed!(
                        getMember,
                        Symbol
                    )
                )
            ),
            __traits(allMembers, Symbol)
        )
    );
}