/**
License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:

	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/
module aermicioi.aedi.configurer.register.container;

import aermicioi.aedi.container;
import std.traits;
import std.meta;
import aermicioi.util.traits;

/**
Create a singleton container

Returns:
	SingletonContainer
**/
auto singleton() {
    return new SingletonContainer();
}

/**
Create a prototype container

Returns:
	PrototypeContainer
**/
auto prototype() {
    return new PrototypeContainer();
}

/**
Create a container for values

Returns:
	ValueContainer
**/
auto values() {
    return new ValueContainer();
}

/**
Wrap up a container into switchable container.

Wraps up a container to provide switching capabilites to it.

Params:
	container = container to wrap up in switchable container

Returns:
	SwitchableContainer!T
**/
auto switchable(T : Container)(auto ref T container) {
    return (new SwitchableContainer!T()).decorated(container);
}

/**
Wrap up a container into subscribable container.

Wraps up a container to provide events to subscribe to.

Params:
	container = container to wrap up in subscribable container

Returns:
	SubscribableContainer!T
**/
auto subscribable(T : Container)(auto ref T container) {
    return (new SubscribableContainer!T()).decorated(container);
}

/**
Wrap up a container into a type based container.

Wraps up container into a container that adds capability to
search for a component based on it's type, or implemented
hierarchy of classes and interfaces.

Params:
	container = container to wrap up in typed container

Returns:
	TypeBasedContainer!T
**/
auto typed(T : Container)(auto ref T container) {
    return (new TypeBasedContainer!T()).decorated(container);
}

/**
Wrap up a container into aliasing container.

Wraps up container into aliasing container which provides
capabilities to alias identity of components in original container.

Params:
	container = container to wrap up in aliasing container

Returns:
	AliasingContainer!T
**/
auto aliasing(T)(auto ref T container) {
    return (new AliasingContainer!T()).decorated(container);
}

/**
Wrap up a container into a defferring container.

Wraps up container into a defferring container which executes defferred actions
when a component from it is requested from exterior and not interior of container.
Therefore with help of it, is possible to solve circular dependency errors by defferring
setting a dependency at a later time when dependents are fully constructed.

Params:
	container = container to wrap up in defferred container
	defferedExecutionerIdentity = identity of container for defferred actions that will be used by contained factories if needed.

Returns:
	DefferedContainer!T
**/
auto deffered(T)(auto ref T container, string defferedExecutionerIdentity) {
	return (new DefferedContainer!T(container, defferedExecutionerIdentity));
}

/**
ditto
**/
auto deffered(T)(auto ref T container) {
	return (new DefferedContainer!T(container));
}

/**
Wrap up container into gc registering container.

Wrap up container into gc registering container, that will automatically
register all created components by it into garbage collector for proper
scanning.

Params:
	container = container to decorate with gc component registration.

Returns:
	GcRegisteringContainer!T
**/
auto gcRegistered(T)(auto ref T container) {
	return (new GcRegisteringContainer!T).decorated(container);
}

/**
Wraps up several containers into one.

Params:
	containers = a list of containers to be used together

Returns:
	TupleContainer!T
**/
auto container(T...)(T containers)
    if (allSatisfy!(partialSuffixed!(isDerived, Container), T)) {
    return new TupleContainer!T(containers);
}

/**
Wraps up several containers into one.

Params:
	managed = first container managed by aggregate one
	identity = identity of container by which it is possible to identify it
	manageds = a set of containers in pairs of (managed, identity)

Returns:
	TupleContainer!T
**/
auto aggregate(T...)(Container managed, string identity, T manageds) {
	AggregateContainer container = new AggregateContainer;

	return container.aggregate(managed, identity, manageds);
}

private {

	auto aggregate(T...)(AggregateContainer container, Container managed, string identity, T manageds) {
		container.set(managed, identity);

		static if (T.length > 1) {
			container.aggregate(manageds);
		}
		return container;
	}
}