/**
License:
    Boost Software License - Version 1.0 - August 17th, 2003

    Permission is hereby granted, free of charge, to any person or organization
    obtaining a copy of the software and accompanying documentation covered by
    this license (the "Software") to use, reproduce, display, distribute,
    execute, and transmit the Software, and to prepare derivative works of the
    Software, and to permit third-parties to whom the Software is furnished to
    do so, all subject to the following:

    The copyright notices in the Software and this entire statement, including
    the above license grant, this restriction and the following disclaimer,
    must be included in all copies of the Software, in whole or in part, and
    all derivative works of the Software, unless such copies or derivative
    works are solely in the form of machine-executable object code generated by
    a source language processor.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
    SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
    FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
    ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.

Authors:
    Alexandru Ermicioi
**/
module aermicioi.aedi.container.decorating_mixin;

import aermicioi.aedi.container.container;
import aermicioi.aedi.storage.alias_aware;
import aermicioi.aedi.storage.decorator;
import aermicioi.aedi.storage.storage;
import aermicioi.aedi.storage.locator;

import aermicioi.aedi.container.container : FactoryLocator;

/**
Mix in container interface implementation that delegates
it to decorated container.
**/
mixin template ContainerMixin(T : Container) {

    /**
    Sets up the internal state of container.

    Sets up the internal state of container (Ex, for singleton container it will spawn all objects that locator contains).
    **/
    typeof(this) instantiate()
    {
        this.decorated.instantiate();

        return this;
    }


    /**
    Destruct all managed components.

    Destruct all managed components. The method denotes the end of container lifetime, and therefore destruction of all managed components
    by it.
    **/
    typeof(this) terminate() {
        this.decorated.terminate();

        return this;
    }
}

/**
Mix in alias aware interface implementation that delegates
the logic to decorated container.
**/
mixin template AliasAwareMixin(T : AliasAware!W, W) {
    /**
    Alias a identity to an alias_.

    Params:
        identity = the originial identity which is to be aliased.
        alias_ = the alias of identity.

    Returns:
        this
    **/
    AliasAware!W link(W identity, W alias_)
    {
        decorated.link(identity, alias_);

        return this;
    }

    /**
    Removes alias.

    Params:
        alias_ = alias to remove.

    Returns:
        this

    **/
    AliasAware!W unlink(W alias_)
    {
        decorated.unlink(alias_);

        return this;
    }

    /**
    Resolve an alias to original identity, if possible.

    Params:
        alias_ = alias of original identity

    Returns:
        Type the last identity in alias chain.

    **/
    const(W) resolve(in W alias_) const
    {
        return decorated_.resolve(alias_);
    }
}

/**
ditto
**/
mixin template AliasAwareMixin(T) {

}

/**
Mix in storage interface implementation that delegates
the logic to decorated container.
**/
mixin template StorageMixin(T : Storage!(W, X), W, X) {
    /**
    Set component in decorated by identity.

    Params:
        identity = identity of factory.
        element = factory that is to be saved in decorated.

    Return:
        SwitchableContainer!T decorating decorated.
    **/
    T set(W element, X identity) {
        this.decorated.set(element, identity);

        return this;
    }

    /**
    Remove factory from decorated with identity.

    Remove factory from decorated with identity.

    Params:
        identity = the identity of factory to be removed.

    Return:
        SwitchableContainer!T decorating decorated
    **/
    T remove(X identity) {
        decorated.remove(identity);

        return this;
    }
}

/**
Mix in locator interface implementation that delegates
the logic to decorated container.
**/
mixin template LocatorMixin(T : Locator!(W, X), W, X) {
    mixin LocatorMixin!(W, X);
}

/**
ditto
**/
mixin template LocatorMixin(W, X) {
    /**
    Get object created by a factory identified by key

    Params:
        key = identity of factory
    Returns:
    Object
    **/
    W get(X key)
    {
        return this.decorated.get(key);
    }

    /**
    Check if an object factory for it exists in container.

    Params:
        key = identity of factory
    Returns:
        bool
    **/
    bool has(in X key) inout
    {
        return this.decorated_.has(key);
    }
}

/**
Mix in factory locator interface implementation that delegates
the logic to decorated container.
**/
mixin template FactoryLocatorMixin(T : FactoryLocator!W, W) {
    import std.range.interfaces : InputRange;
    import std.typecons : Tuple;
    /**
    Get factory for constructed component identified by identity.

    Get factory for constructed component identified by identity.
    Params:
        identity = the identity of component that factory constructs.

    Throws:
        NotFoundException when factory for it is not found.

    Returns:
        ObjectFactory the factory for constructed component.
    **/
    W getFactory(string identity)
    {
        return this.decorated.getFactory(identity);
    }

    /**
    Get all factories available in container.

    Get all factories available in container.

    Returns:
        InputRange!(Tuple!(ObjectFactory, string)) a tuple of factory => identity.
    **/
    InputRange!(Tuple!(W, string)) getFactories()
    {
        return this.decorated.getFactories();
    }
}