/**
License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:

	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/

module aermicioi.aedi.container.factory;

import aermicioi.aedi.factory.factory;
import aermicioi.aedi.factory.decorating_factory;
import aermicioi.aedi.exception;
import aermicioi.aedi.storage.locator;
import aermicioi.aedi.storage.allocator_aware;
import aermicioi.aedi.storage.decorator : MutableDecoratorMixin;

/**
A decorating factory that detects circular reference insantiations.
**/
class InProcessObjectFactoryDecorator : ObjectFactory, ObjectFactoryDecorator {

    mixin AllocatorAwareMixin!(typeof(this));
    mixin DestructDecoratorMixin!(typeof(this));

    private {
        bool inProcess;
    }

    public {
        mixin MutableDecoratorMixin!(ObjectFactory);

        /**
         * Default constructor for InProcessObjectFactoryDecorator
        **/
        this() {

        }

        /**
         * Constructor for InProcessObjectFactoryDecorator
         *
         * Params:
         * 	decorated = factory that is decorated with InProcessObjectFactoryDecorator
        **/
        this(ObjectFactory decorated) {
            this.decorated = decorated;
        }

        @property {
            /**
             * Get type of created object
             *
             * Returns:
             * 	TypeInfo
            **/
            TypeInfo type() const {
            	return this.decorated.type();
            }

            /**
             * Set locator
             *
             * Params:
             * 	locator = ${param-description}
             * Throws:
             * Returns:
             * 	typeof(this)
            **/
            InProcessObjectFactoryDecorator locator(Locator!() locator) {
            	this.decorated.locator = locator;

            	return this;
            }
        }

        /**
		Factory an object.

		Returns:
			Object created object.
		**/
        Object factory() {
            if (inProcess) {
                throw new InProgressException("ObjectFactory is already instantiating, type: " ~ this.decorated.type.toString());
            }

            inProcess = true;
            scope(exit) inProcess = false;

            Object obj = this.decorated.factory();

            return obj;
        }
    }
}

/**
A decorating factory that catches any thrown exceptions by decorated factory,
and rethrows them with additional information.

A decorating factory that catches any thrown exceptions by decorated factory,
and chains it in a new exception that adds additonal debugging information
such as registered identity, and cause of exception. It is useful for chaining
component instantiation pipeline and printing it for debugging purposes.
**/
class ExceptionChainingObjectFactory : ObjectFactory, ObjectFactoryDecorator {

    mixin AllocatorAwareMixin!(typeof(this));
    mixin DestructDecoratorMixin!(typeof(this));

    private {
        string id_;
    }

    public {
        mixin MutableDecoratorMixin!(ObjectFactory);

        /**
         * Default constructor for ExceptionChainingObjectFactory
        **/
        this() {

        }

        /**
         * Constructor for ExceptionChainingObjectFactory
         *
         * Params:
         * 	decorated = factory to be decorated
         *  id = identity of created object, used for exception message.
        **/
        this(ObjectFactory decorated, string id) {
            this.decorated = decorated;
            this.id = id;
        }

        @property {
            /**
             * Set id
             *
             * Params:
             * 	id = identity of created object, used for exception message.
             * Returns:
             * 	typeof(this)
            **/
            ExceptionChainingObjectFactory id(string id) @safe nothrow {
            	this.id_ = id;

            	return this;
            }

            /**
             * Get id
             *
             * Returns:
             * 	string
            **/
            string id() @safe nothrow {
            	return this.id_;
            }

            /**
             * Get type of created object
             *
             * Returns:
             * 	TypeInfo
            **/
            TypeInfo type() const {
            	return this.decorated.type();
            }

            /**
             * Set locator
             *
             * Params:
             * 	locator = the locator used to locate dependencies for created object.
             * Returns:
             * 	typeof(this)
            **/
            ExceptionChainingObjectFactory locator(Locator!() locator) {
            	this.decorated.locator = locator;

            	return this;
            }
        }

        /**
		Factory an object, and catch any exception wrapping it in a library exception to be rethrown further.

		Returns:
			Object created object.
		**/
        Object factory() {

            try {
                Object obj = this.decorated.factory();

                return obj;
            } catch (NotFoundException e) {

                throw new NotFoundException("A dependency for " ~ this.id ~ " could not be found", e);
            } catch (InProgressException e) {

                throw new CircularReferenceException("Circular reference detected during construction of " ~ this.id, e);
            } catch(CircularReferenceException e) {

                throw new CircularReferenceException("Circular reference detected during construction of " ~ this.id, e);
            } catch (AediException e) {

                throw new AediException("A library exception occurred during construction of " ~ this.id, e);
            } catch (Exception e) {

                throw new Exception("A general exception occurred during construction of " ~ this.id, e);
            }
        }
    }
}