/**

License:
	Boost Software License - Version 1.0 - August 17th, 2003

    Permission is hereby granted, free of charge, to any person or organization
    obtaining a copy of the software and accompanying documentation covered by
    this license (the "Software") to use, reproduce, display, distribute,
    execute, and transmit the Software, and to prepare derivative works of the
    Software, and to permit third-parties to whom the Software is furnished to
    do so, all subject to the following:

    The copyright notices in the Software and this entire statement, including
    the above license grant, this restriction and the following disclaimer,
    must be included in all copies of the Software, in whole or in part, and
    all derivative works of the Software, unless such copies or derivative
    works are solely in the form of machine-executable object code generated by
    a source language processor.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
    SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
    FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
    ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.

Authors:
	Alexandru Ermicioi
**/
module aermicioi.aedi.container.prototype_container;

import aermicioi.aedi.container.container;
import aermicioi.aedi.storage.object_storage;
import aermicioi.aedi.factory.factory;
import aermicioi.aedi.exception;
import aermicioi.aedi.container.factory;

import std.range.interfaces;
import std.typecons;

/**
 Prototype container.

 Instantiates a new object using passed ObjectFactory implementation, on each request of it by some part of an application.
**/
class PrototypeContainer : ConfigurableContainer {

    private {

        ObjectStorage!(ObjectFactory, string) factories;
        ObjectStorage!(Object[], string) proxies;
    }

    public {

        /**
         * Default constructor for PrototypeContainer
        **/
        this() {
            this.factories = new ObjectStorage!(ObjectFactory, string);
            this.proxies = new ObjectStorage!(Object[], string);
        }

        /**
         * Set object factory
         *
         * Params:
         * 	object = factory for a object that is to be managed by prototype container.
         *  key = identity of factory
         * Returns:
         * 	typeof(this)
        **/
        PrototypeContainer set(ObjectFactory object, string key) {
            this.factories.set(new ExceptionChainingObjectFactory(new InProcessObjectFactoryDecorator(object), key), key);

            return this;
        }

        /**
         * Remove an object factory from container.
         *
         * Params:
         * 	key = identity of factory to be removed
         * Returns:
         * 	typeof(this)
        **/
        PrototypeContainer remove(string key) {
            if (this.proxies.has(key)) {
                import std.algorithm : each;

                this.proxies.get(key).each!((proxy) => this.factories.get(key).destruct(proxy));
            }

            this.factories.remove(key);
            this.proxies.remove(key);

            return this;
        }

        /**
         * Get object created by a factory identified by key
         *
         * Params:
         *  key = identity of factory
         * Returns:
         * 	Object
        **/
        Object get(string key) {
            if (this.factories.has(key)) {

                Object obj = this.factories.get(key).factory();

                Object[] proxies;

                if (this.proxies.has(key)) {
                    proxies = this.proxies.get(key);
                }

                this.proxies.set(proxies ~ obj, key);

                return obj;
            }

            throw new NotFoundException("Component by id " ~ key ~ " not found");
        }

        /**
         * Check if an object factory for it exists in container.
         *
         * Params:
         * 	key = identity of factory
         * Returns:
         * 	bool
        **/
        bool has(in string key) inout {
            return this.factories.has(key);
        }

        /**
        Sets up the internal state of container.

        Sets up the internal state of container (Ex, for singleton container it will spawn all objects that locator contains).
        **/
        PrototypeContainer instantiate() {

            return this;
        }

        /**
        Destruct all managed components.

        Destruct all managed components. The method denotes the end of container lifetime, and therefore destruction of all managed components
        by it.
        **/
        PrototypeContainer terminate() {
            import std.algorithm : each;

            foreach (pair; this.proxies.contents.byKeyValue) {
                pair.value.each!((proxy) => this.factories.get(pair.key).destruct(proxy));
            }

            this.proxies.contents.clear;

            return this;
        }
        /**
        Alias a key to an alias_.

        Params:
        	key = the originial identity which is to be aliased.
        	alias_ = the alias of identity.

		Returns:
			this
        **/
        PrototypeContainer link(string key, string alias_) {
            this.factories.link(key, alias_);

            return this;
        }

        /**
        Removes alias.

        Params:
        	alias_ = alias to remove.

        Returns:
            this

        **/
        PrototypeContainer unlink(string alias_) {
            this.factories.unlink(alias_);

            return this;
        }

        /**
        Resolve an alias to original identity, if possible.

        Params:
        	key = alias of original identity

        Returns:
        	Type the last identity in alias chain.

        **/
        const(string) resolve(in string key) const {
            return this.factories.resolve(key);
        }

        /**
        Get factory for constructed component identified by identity.

        Get factory for constructed component identified by identity.
        Params:
        	identity = the identity of component that factory constructs.

        Throws:
        	NotFoundException when factory for it is not found.

        Returns:
        	ObjectFactory the factory for constructed component.
        **/
        ObjectFactory getFactory(string identity) {
            return this.factories.get(identity);
        }

        /**
        Get all factories available in container.

        Get all factories available in container.

        Returns:
        	InputRange!(Tuple!(ObjectFactory, string)) a tuple of factory => identity.
        **/
        InputRange!(Tuple!(ObjectFactory, string)) getFactories() {
            import std.algorithm : map;

            return this.factories.contents.byKeyValue.map!(
                a => tuple(a.value, a.key)
            ).inputRangeObject;
        }
    }
}