/**

License:
	Boost Software License - Version 1.0 - August 17th, 2003

    Permission is hereby granted, free of charge, to any person or organization
    obtaining a copy of the software and accompanying documentation covered by
    this license (the "Software") to use, reproduce, display, distribute,
    execute, and transmit the Software, and to prepare derivative works of the
    Software, and to permit third-parties to whom the Software is furnished to
    do so, all subject to the following:

    The copyright notices in the Software and this entire statement, including
    the above license grant, this restriction and the following disclaimer,
    must be included in all copies of the Software, in whole or in part, and
    all derivative works of the Software, unless such copies or derivative
    works are solely in the form of machine-executable object code generated by
    a source language processor.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
    SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
    FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
    ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.

Authors:
	Alexandru Ermicioi
**/
module aermicioi.aedi.container.singleton_container;

import aermicioi.aedi.container.container;
import aermicioi.aedi.storage.object_storage;
import aermicioi.aedi.storage.locator_aware;
import aermicioi.aedi.storage.locator;
import aermicioi.aedi.factory.factory;
import aermicioi.aedi.exception;
import aermicioi.aedi.container.factory;

import std.range.interfaces;
import std.typecons;

/**
 Singleton container.

 It creates objects from ObjectFactory implementations and sets them as long as it lives in application.
**/
class SingletonContainer : ConfigurableContainer {

    private {

        ObjectStorage!() singletons;
        ObjectStorage!(ObjectFactory, string) factories;
    }

    public {

        /**
         * Default constructor for SingletonContainer
        **/
        this() {
            this.singletons = new ObjectStorage!();
            this.factories = new ObjectStorage!(ObjectFactory, string);
        }

        /**
         * Set object factory
         *
         * Params:
         * 	object = factory for a object that is to be managed by prototype container.
         *  key = identity of factory
         * Returns:
         * 	typeof(this)
        **/
        SingletonContainer set(ObjectFactory object, string key) {
            this.factories.set(new ExceptionChainingObjectFactory(new InProcessObjectFactoryDecorator(object), key), key);

            return this;
        }

        /**
         * Remove an object factory from container.
         *
         * Params:
         * 	key = identity of factory to be removed
         * Returns:
         * 	typeof(this)
        **/
        SingletonContainer remove(string key) {

            if (this.singletons.has(key)) {

                auto temporary = this.singletons.get(key);
                this.factories.get(key).destruct(temporary);
            }

            this.factories.remove(key);
            this.singletons.remove(key);

            return this;
        }

        /**
         * Get object created by a factory identified by key
         *
         * Params:
         *  key = identity of factory
         * Returns:
         * 	Object
        **/
        Object get(string key) {

            if (!this.singletons.has(key)) {
                if (!this.factories.has(key)) {
                    throw new NotFoundException("Component with id " ~ key ~ " not found.");
                }

                this.singletons.set(
                    this.factories.get(key).factory(),
                    this.resolve(key),
                );
            }

            return this.singletons.get(key);
        }

        /**
         * Check if an object factory for it exists in container.
         *
         * Params:
         * 	key = identity of factory
         * Returns:
         * 	bool
        **/
        bool has(in string key) inout {
            return this.factories.has(key);
        }

        /**
        Sets up the internal state of container.

        Sets up the internal state of container (Ex, for singleton container it will spawn all objects that locator contains).
        **/
        SingletonContainer instantiate() {
            import std.algorithm : filter;
            foreach (pair; this.factories.contents.byKeyValue.filter!((pair) => pair.key !in this.singletons.contents)) {
                this.singletons.set(
                    pair.value.factory,
                    pair.key,
                );
            }

            return this;
        }

        /**
        Destruct all managed components.

        Destruct all managed components. The method denotes the end of container lifetime, and therefore destruction of all managed components
        by it.
        **/
        Container terminate() {
            foreach (pair; this.singletons.contents.byKeyValue) {
                this.factories.get(pair.key).destruct(pair.value);
            }

            this.singletons.contents.clear;

            return this;
        }

        /**
        Alias a key to an alias_.

        Params:
        	key = the originial identity which is to be aliased.
        	alias_ = the alias of identity.

		Returns:
			this
        **/
        SingletonContainer link(string key, string alias_) {
            this.singletons.link(key, alias_);
            this.factories.link(key, alias_);

            return this;
        }

        /**
        Removes alias.

        Params:
        	alias_ = alias to remove.

        Returns:
            this

        **/
        SingletonContainer unlink(string alias_) {
            this.singletons.unlink(alias_);
            this.factories.unlink(alias_);

            return this;
        }

        /**
        Resolve an alias to original identity, if possible.

        Params:
        	key = alias of original identity

        Returns:
        	Type the last identity in alias chain.

        **/
        const(string) resolve(in string key) const {
            return this.factories.resolve(key);
        }

        /**
        Get factory for constructed component identified by identity.

        Get factory for constructed component identified by identity.
        Params:
        	identity = the identity of component that factory constructs.

        Throws:
        	NotFoundException when factory for it is not found.

        Returns:
        	ObjectFactory the factory for constructed component.
        **/
        ObjectFactory getFactory(string identity) {
            return this.factories.get(identity);
        }

        /**
        Get all factories available in container.

        Get all factories available in container.

        Returns:
        	InputRange!(Tuple!(ObjectFactory, string)) a tuple of factory => identity.
        **/
        InputRange!(Tuple!(ObjectFactory, string)) getFactories() {
            import std.algorithm : map;

            return this.factories.contents.byKeyValue.map!(
                a => tuple(a.value, a.key)
            ).inputRangeObject;
        }
    }
}