/**
License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:

	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/
module aermicioi.aedi.container.subscribable_container;

import aermicioi.aedi.container.container;
import aermicioi.aedi.container.decorating_mixin;
import aermicioi.aedi.storage.object_storage;
import aermicioi.aedi.storage.decorator;
import aermicioi.aedi.storage.alias_aware;
import aermicioi.aedi.storage.storage;
import aermicioi.aedi.factory.factory;
import aermicioi.aedi.exception.not_found_exception;
import aermicioi.util.traits;
import std.meta;
import std.traits;

import std.range.interfaces;
import std.typecons;

/**
Interface for objects that can be subscribed to specific events emmited by them.
**/
interface Subscribable(T)
{

    public
    {

        /**
        Subscriber a delegate to a particular event emmited by object

        Params:
        	type = type of event emmited by object
         subscriber = the callback to be called on event emmited
        Returns:
        	typeof(this)
        **/
        Subscribable subscribe(T type, void delegate() subscriber);
    }
}

/**
Enumeration of events supported by SubscribableContainer for instantiation functionality.
**/
enum ContainerInstantiationEventType
{

    /**
    Event run before beggining of instantiation process
    **/
    pre,

    /**
    Event run after end of instantiation process
    **/
    post,
}

/**
Decorating container that adds a set of events at different actions during lifetime of a container
to which subscribers can subscribe to. This decorated will
inherit following interfaces only and only if the
T also implements them:
  $(OL
      $(LI Storage!(ObjectFactory, string))
      $(LI Container)
      $(LI AliasAware!string)
  )
Decorated container must implement following interfaces:
    $(OL
        $(LI Container)
        $(LI MutableDecorator!T)
        $(LI Subscribable!ContainerInstantiationEventType)
        $(LI Decorator!Container)
    )

Params:
    T = The decorated that switchable decorated will decorate.
**/
template SubscribableContainer(T)
{

    /**
    Set which the switchable container will decorate for T. By default
    Locator!() and Subscribable!ContainerInstantiationEventType is included.
    **/
    alias InheritanceSet = NoDuplicates!(Filter!(
        templateOr!(
            partialSuffixed!(
                isDerived,
                Storage!(ObjectFactory, string)
            ),
            partialSuffixed!(
                isDerived,
                AliasAware!string
            ),
            partialSuffixed!(
                isDerived,
                FactoryLocator!ObjectFactory
            )
        ),
        InterfacesTuple!T),
        Container,
        Subscribable!ContainerInstantiationEventType,
        MutableDecorator!T,
        Decorator!Container
    );

    class SubscribableContainer : InheritanceSet
    {
        private
        {
            void delegate()[][ContainerInstantiationEventType] subscribers;
            T decorated_;
        }

        public
        {

            /**
            Default constructor for SubscribableContainer
            **/
            this()
            {
                subscribers[ContainerInstantiationEventType.pre] = null;
                subscribers[ContainerInstantiationEventType.post] = null;
            }

            mixin MutableDecoratorMixin!T;
            mixin LocatorMixin!(typeof(this));

            /**
            Subscriber a delegate to a particular event emmited by object

            Params:
                event = type of event emmited by object
                subscriber = the callback to be called on event emmited
            Returns:
           	    typeof(this)
            **/
            SubscribableContainer subscribe(ContainerInstantiationEventType event,
                    void delegate() subscriber)
            {
                this.subscribers[event] ~= subscriber;

                return this;
            }

            /**
            Sets up the internal state of container.

            Sets up the internal state of container (Ex, for singleton container it will spawn all objects that locator contains).
            **/
            SubscribableContainer instantiate()
            {
                foreach (preProcessor; this.subscribers[ContainerInstantiationEventType.pre])
                {
                    preProcessor();
                }

                this.decorated.instantiate();

                foreach (postProcessor; this.subscribers[ContainerInstantiationEventType.post])
                {
                    postProcessor();
                }

                return this;
            }

            /**
            Destruct all managed components.

            Destruct all managed components. The method denotes the end of container lifetime, and therefore destruction of all managed components
            by it.
            **/
            Container terminate() {
                this.decorated.terminate();

                return this;
            }

            static if (is(T : Storage!(ObjectFactory, string)))
            {
                mixin StorageMixin!(typeof(this));
            }

            static if (is(T : AliasAware!string))
            {
                mixin AliasAwareMixin!(typeof(this));
            }

            static if (is(T : FactoryLocator!ObjectFactory))
            {
                mixin FactoryLocatorMixin!(typeof(this));
            }
        }
    }
}
