/**

License:
	Boost Software License - Version 1.0 - August 17th, 2003

    Permission is hereby granted, free of charge, to any person or organization
    obtaining a copy of the software and accompanying documentation covered by
    this license (the "Software") to use, reproduce, display, distribute,
    execute, and transmit the Software, and to prepare derivative works of the
    Software, and to permit third-parties to whom the Software is furnished to
    do so, all subject to the following:

    The copyright notices in the Software and this entire statement, including
    the above license grant, this restriction and the following disclaimer,
    must be included in all copies of the Software, in whole or in part, and
    all derivative works of the Software, unless such copies or derivative
    works are solely in the form of machine-executable object code generated by
    a source language processor.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
    SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
    FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
    ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.

Authors:
	Alexandru Ermicioi
**/
module aermicioi.aedi.factory.generic_factory;


import aermicioi.aedi.exception;
import aermicioi.aedi.factory.factory;
import aermicioi.aedi.factory.reference;
import aermicioi.aedi.storage.decorator;
import aermicioi.aedi.storage.locator;
import aermicioi.aedi.storage.allocator_aware;
import aermicioi.aedi.storage.locator_aware;
import aermicioi.aedi.storage.wrapper;
import aermicioi.util.traits;


import std.conv : to;
import std.meta;
import std.traits;
import std.typecons;

/**
Interface for objects that are executing delayed/defferred actions.
**/
interface DefferredExecutioner {

    public {

        /**
        Add a new defferred action to be executed

        Params:
            dg = the deffered action to execute

        Returns:
            typeof(this)
        **/
        DefferredExecutioner add(void delegate() dg);

        /**
        Add a set of defferred actions to be executed

        Params:
            dgs = list of defferred actions

        Returns:
            typeof(this)
        **/
        DefferredExecutioner add(void delegate()[] dgs);

        /**
        Execute defferred actions

        Returns:
            typeof(this)
        **/
        DefferredExecutioner execute();
    }
}

/**
Interface for objects that are aware of deffered executioner, and are able to store their defferred actions in it
for later execution, by themselves or other third party.
**/
interface DefferredExecutionerAware {
    public {
        @property {
            /**
            Set executioner

            Params:
                executioner = the executioner used to deffer actions for a later time

            Returns:
                typeof(this)
            **/
            typeof(this) executioner(DefferredExecutioner executioner) @safe nothrow pure;

            /**
            Get executioner

            Returns:
                DefferredExecutioner
            **/
            DefferredExecutioner executioner() @safe nothrow pure;
        }
    }
}

/**
A property configurer, has the purpose to modify component of type T according to some logic encapsulated in it.

**/
interface PropertyConfigurer(T) : LocatorAware!() {

    public {

        /**
        Accepts a reference to an object that is to be configured by the configurer.

        Params:
        	object = An object of type T, that will be configured
        **/
        void configure(ref T object);
    }
}

/**
An instance factory, instantiates component of type T.
**/
interface InstanceFactory(T) : LocatorAware!(), AllocatorAware!() {

    public {

        /**
        Create a new instance of object of type T.

        Returns:
            T instantiated component
        **/
        T factory();
    }
}

/**
An interface for components that can destruct components of type T and deallocate them using provided allocator.
**/
interface InstanceDestructor(T) : LocatorAware!(), AllocatorAware!() {

    public {

        /**
        Destruct a component of type T and deallocate it using stored allocator.

        Params:
            destructable = element to be destructed and deallocated using stored allocator
        **/
        void destruct(ref T destructable);
    }
}

/**
Interface for objects that are aware of an instance factory
and can use it to instantiate a component.
**/
interface InstanceFactoryAware(T) {
    public {

        @property {
        	/**
            Sets the constructor of new object.

            Params:
            	factory = a factory of objects of type T.

        	Returns:
    			The InstanceFactoryAware
            **/
            InstanceFactoryAware!T setInstanceFactory(InstanceFactory!T factory);
        }
    }
}

/**
Interface for objects that are aware of an instance factory
and can use it to instantiate a component.
**/
interface InstanceDestructorAware(T) {
    public {

        @property {
        	/**
            Sets the destructor of component.

            Params:
            	destructor = destructor for components of type T.

        	Returns:
    			The InstanceDestructorAware!T
            **/
            InstanceDestructorAware!T setInstanceDestructor(InstanceDestructor!T destructor);
        }
    }
}

/**
Interface for objects that are aware of a set of property configurers
and can use them to configure some component.
**/
interface PropertyConfigurersAware(T) {
    public {
        /**
        Adds an configurer to the PropertyConfigurersAware.

        Params:
        	configurer = a configurer that will be invoked after factory of an object.

    	Returns:
    		The PropertyConfigurersAware instance
        **/
        PropertyConfigurersAware!T addPropertyConfigurer(PropertyConfigurer!T configurer);
    }
}
/**
A generic factory, is a factory that instantiates component of type T using InstanceFactory and a list of PropertyConfigurers.

A generic factory, is a factory that instantiates component of type T using InstanceFactory and a list of PropertyConfigurers.
It can optionally provide a Locator!() object to InstanceFactory and PropertyConfigurer to be used as a source of component.
**/
interface GenericFactory(T) : Factory!T, InstanceFactoryAware!T, PropertyConfigurersAware!T, InstanceDestructorAware!T {

    public {

        @property {

            alias locator = Factory!T.locator;

            /**
            Get the GenericFactory locator.

            Returns:
            	Locator!() the locator that should be used by underlying constructor or property configurer.
            **/
            Locator!() locator();
        }
    }
}

/**
A concrete implementation of GenericFactory interface.
**/
class GenericFactoryImpl(T) : GenericFactory!T, LocatorAware!(), DefferredExecutionerAware {
    import aermicioi.aedi.storage.allocator_aware : AllocatorAwareMixin;

    private {
        Locator!() locator_;
        RCIAllocator allocator_;

        InstanceFactory!T factory_;
        InstanceDestructor!T destructor_;
        PropertyConfigurer!T[] configurers;
        DefferredExecutioner executioner_;
    }

    public {

        /**
            Constructor for GenericFactoryImpl

            Params:
                locator = the locator used by constructor to fetch dependencies of the created object
        **/
        this(Locator!() locator) {
            import std.experimental.allocator : theAllocator;

            static if (hasDefaultCtor!T) {
                this.setInstanceFactory(new DefaultInstanceFactory!T);
            } else {
                this.setInstanceFactory(new DefaultFailingInstanceFactory!T);
            }

            this.setInstanceDestructor(new DefaultInstanceDestructor!T);
            this.locator = locator;
            this.allocator = theAllocator;
        }

        /**
        Create a new instance of object of type T using provided instance factory and property configurers.

        Returns:
            T instantiated component
        **/
        T factory() {
            T instance = this.factory_.factory();

            foreach (key, configurer; this.configurers) {

                try {

                    configurer.configure(instance);
                } catch (AediException exception) {

                    if (this.executioner !is null) {

                        Throwable current = exception;

                        while (current !is null) {
                            CircularReferenceException e = cast(CircularReferenceException) current;

                            if (e !is null) {
                                this.executioner.add(() {
                                    configurer.configure(instance);
                                });

                                break;
                            }

                            current = current.next;
                        }

                        if (current is null) {
                            throw exception;
                        }
                    } else {

                        throw exception;
                    }
                }
            }

            return instance;
        }

        /**
        Destructs a component of type T.

        Params:
            component = component that is to ve destroyed.

        Returns:

        **/
        void destruct(ref T component) {
            this.destructor_.destruct(component);
        }

        @property {

            /**
            Set executioner

            Params:
                executioner = executioner used to deffer configurations at later time

            Returns:
                typeof(this)
            **/
            typeof(this) executioner(DefferredExecutioner executioner) @safe nothrow pure {
                this.executioner_ = executioner;

                return this;
            }

            /**
            Get executioner

            Returns:
                DefferredExecutioner
            **/
            DefferredExecutioner executioner() @safe nothrow pure {
                return this.executioner_;
            }

            /**
            Set destructor

            Params:
                destructor = the destructor used to destruct components created by this factory.

            Returns:
                typeof(this)
            **/
            typeof(this) setInstanceDestructor(InstanceDestructor!T destructor)
            in {
                assert(destructor !is null);
            }
            body {
                this.destructor_ = destructor;

                this.destructor_.allocator = this.allocator;
                this.destructor_.locator = this.locator;

                return this;
            }

            /**
            Sets the constructor of new object.

            Params:
            	factory = a factory of objects of type T.

        	Returns:
    			The InstanceFactoryAware
            **/
            GenericFactory!T setInstanceFactory(InstanceFactory!T factory)
            in {
                assert(factory !is null);
            }
            body {
                this.factory_ = factory;

                this.factory_.allocator = this.allocator;
                this.factory_.locator = this.locator;

                return this;
            }

            /**
                Set locator

                Params:
                    locator = the locator used to fetch created's object dependencies
                Returns:
                    typeof(this)
            **/
            GenericFactory!T locator(Locator!() locator) @safe nothrow {
                this.locator_ = locator;

                this.factory_.locator = this.locator;
                this.destructor_.locator = this.locator;

                foreach (configurer; this.configurers) {
                    configurer.locator = locator;
                }

                return this;
            }

            /**
                Get locator

                Returns:
                    Locator!()
            **/
            Locator!() locator() @safe nothrow {
                return this.locator_;
            }

            /**
            Set allocator

            Params:
                allocator = the allocator used to allocate place for component.

            Returns:
                typeof(this)
            **/
            typeof(this) allocator(RCIAllocator allocator) @safe nothrow {
                this.allocator_ = allocator;

                this.factory_.allocator = this.allocator;
                this.destructor_.allocator = this.allocator;

                return this;
            }

            /**
            Get allocator

            Returns:
                RCIAllocator
            **/
            RCIAllocator allocator() @safe nothrow {
                return this.allocator_;
            }

            /**
    		Get the type info of object that is created.

    		Returns:
    			TypeInfo object of created object.
    		**/
    		TypeInfo type() @safe nothrow const {
    		    return typeid(T);
    		}
        }

        /**
        Adds an configurer to the PropertyConfigurersAware.

        Params:
        	configurer = a configurer that will be invoked after factory of an object.

    	Returns:
    		The PropertyConfigurersAware instance
        **/
        GenericFactory!T addPropertyConfigurer(PropertyConfigurer!T configurer) {

            this.configurers ~= configurer;

            configurer.locator = this.locator;

            return this;
        }
    }

}

/**
ditto
**/
GenericFactory!T genericFactory(T)(Locator!() locator) {
    return new GenericFactoryImpl!T(locator);
}


/**
Standard implementation of DefferredExecutioner interface.
**/
class DefferredExecutionerImpl : DefferredExecutioner {

    private {
        void delegate()[] deffered;
    }

    public {

        /**
        Add a new defferred action to be executed

        Params:
            dg = the deffered action to execute

        Returns:
            typeof(this)
        **/
        DefferredExecutioner add(void delegate() dg) {
            this.deffered ~= dg;

            return this;
        }

        /**
        Add a set of defferred actions to be executed

        Params:
            dgs = list of defferred actions

        Returns:
            typeof(this)
        **/
        DefferredExecutioner add(void delegate()[] dgs) {
            this.deffered ~= dgs;

            return this;
        }

        /**
        Execute defferred actions

        Returns:
            typeof(this)
        **/
        DefferredExecutioner execute() {
            foreach (dg; this.deffered) {
                dg();
            }

            this.deffered = null;

            return this;
        }
    }
}

/**
ParameterHolder Stores a set of Args for futher usage in it's subclasses.

Params:
    Args = a type tuple of args that ParameterHolder can hold.
**/
mixin template ParameterHolder(Args...) {

    protected {
        Tuple!Args args_;
    }

    public {

        static if (Args.length > 0) {

            /**
            Set args

            Params:
                args = arguments that parameter holder should hold.
            Returns:
                typeof(this)
            **/
            typeof(this) args(ref Args args) @safe nothrow {
            	this.args_ = tuple(args);

            	return this;
            }
        }

        /**
        Get args

        Returns:
            Tuple!Args arguments stored by argument holder
        **/
        Tuple!Args args() @safe nothrow {
        	return this.args_;
        }

    }
}

/**
Calls component's method with a set of arguments.

Encapsulates a call to component's method, with a set of arguments.
The algorithm that calls component's method, will automatically replace
RuntimeReferences from args list with components extracted from container, casted to
type that is extracted from method's signature.

Params:
    T = the component type
    property = method that will be called
    Args = type tuple of args that method can be called with.
**/
class MethodConfigurer(T, string property, Args...) : PropertyConfigurer!T
	if (
	    isMethodCompatible!(T, property, Args)
    ) {

    mixin ParameterHolder!Args;
    mixin LocatorAwareMixin!(typeof(this));

    public {

        /**
            Constructor for MethodConfigurer!(T, property, Args)

            Params:
                args = list of arguments passed to T's method
        **/
        this(ref Args args) {
            this.args(args);
        }

        /**
        See PropertyConfigurer interface

        Throws:
            InvalidCastException when extracted component by reference, is not of type expected by argument
            of component's method
        **/
        void configure(ref T obj) {

            try {

                alias ArgTuple = Parameters!(
                    Filter!(
                        partialSuffixed!(
                            isArgumentListCompatible,
                            Args
                        ),
                        getOverloads!(
                            T,
                            property
                        )
                    )[0]
                );

                mixin(q{__traits(getMember, obj, property)(} ~ compileArgumentsTuple!ArgTuple(q{ArgTuple}, q{this.args}, q{this.locator}) ~ q{);});

            } catch (Exception e) {
                throw new PropertyConfigurerException("Error occurred during call of " ~ name!T ~ "." ~ property, e);
            }
        }
    }
}

/**
ditto
**/
auto methodConfigurer(string property, T, Args...)(auto ref Args args)
    if (!isField!(T, property)) {
    mixin assertObjectMethodCompatible!(T, property, Args);

    auto propertySetter = new MethodConfigurer!(T, property, Args)(args);

    return propertySetter;
}

/**
Sets component's field to a value.

Encapsulates logic that sets component's field to a certain value.
If argument that is contained by configurer is a RuntimeReference, it will be automatically
replaced with value extracted from locator, and set to component's field.
**/
class FieldConfigurer(T, string property, Arg) : PropertyConfigurer!T
	if (
	    isFieldCompatible!(T, property, Arg)
    ) {

    mixin ParameterHolder!Arg;
    mixin LocatorAwareMixin!(typeof(this));

    public {
        /**
            Constructor for MethodConfigurer!(T, property, Args)

            Params:
                arg = list of arguments passed to T's method
        **/
        this(ref Arg arg) {
            this.args(arg);
        }

        /**
        See PropertyConfigurer interface

        Throws:
            InvalidCastException when extracted component by reference, is not of type expected by argument
            of component's field
        **/
        void configure(ref T obj) {

            try {

                __traits(getMember, obj, property) = args[0].resolve!(
                    typeof(__traits(getMember, obj, property))
                )(
                    this.locator
                );
            } catch (Exception e) {

                throw new PropertyConfigurerException("Error occurred during set of " ~ name!T ~ "." ~ property, e);
            }
        }
    }
}

/**
ditto
**/
auto fieldConfigurer(string property, T, Arg)(auto ref Arg arg)
    if (isField!(T, property)) {
    mixin assertFieldCompatible!(T, property, Arg);

    auto propertySetter = new FieldConfigurer!(T, property, Arg)(arg);

    return propertySetter;
}

/**
Instantiates a component using it's constructor with no arguments.
**/
class DefaultInstanceFactory(T) : InstanceFactory!T
    if (
        hasDefaultCtor!T
    ) {

    mixin LocatorAwareMixin!(typeof(this));
    mixin AllocatorAwareMixin!(typeof(this));

    public {
        /**
        Create a new instance of object of type T.

        Returns:
            T instantiated component
        **/
        T factory() {

            try {
                static if (is(T == class) && !isAbstractClass!T) {
                    return this.allocator.make!T();
                } else {
                    return T.init;
                }
            } catch (Exception e) {
                throw new InstanceFactoryException("Error occurred during instantiation of " ~ name!T, e);
            }
        }
    }
}

/**
An instance constructor aimed to throw exception.

An instance constructor aimed to throw exception,
when a component has elaborate constructor, yet
no constructor was configured for generic factory to
use those elaborate constructors.
**/
class DefaultFailingInstanceFactory(T) : InstanceFactory!T {

    mixin LocatorAwareMixin!(typeof(this));
    mixin AllocatorAwareMixin!(typeof(this));

    public {
        /**
        Create a new instance of object of type T.

        Returns:
            T instantiated component
        **/
        T factory() {

            throw new InstanceFactoryException("Component " ~ name!T ~ " has elaborate constructor, yet no instance constructor was provided.");
        }
    }
}

/**
Instantiates component using it's constructor with args.

Encapsulates construction of component using a constructor, with args.
Arguments from argument list that are RuntimeReferences, are automatically
replaced with component extracted from locator.

Params:
    T = component type
    Args = type tuple of args that are passed to T's constructor
**/
class ConstructorBasedFactory(T, Args...) : InstanceFactory!T
	if (
	    isObjectConstructorCompatible!(T, Args)
	) {

    mixin ParameterHolder!Args;
    mixin AllocatorAwareMixin!(typeof(this));
    mixin LocatorAwareMixin!(typeof(this));

    public {
        /**
            Constructor for ConstructorBasedFactory!(T, Args)

            Params:
                args = arguments used for constructor
        **/
        this(ref Args args) {
            this.args(args);
        }

        /**
        See InstanceFactory interface

        Throws:
            InvalidCastException when extracted component by reference, is not of type expected by argument
            of component's constructor
        **/
        T factory() {

            try {

                alias ConstructorArgs = Parameters!(
                    Filter!(
                        partialSuffixed!(
                            isArgumentListCompatible,
                            Args
                        ),
                        __traits(getOverloads, T, "__ctor")
                    )[0]
                );

                static if (is(T : Object)) {
                    mixin(q{return this.allocator.make!T(} ~ compileArgumentsTuple!ConstructorArgs(q{ConstructorArgs}, q{this.args}, q{this.locator}) ~ q{);});
                } else {
                    mixin(q{return T(} ~ compileArgumentsTuple!ConstructorArgs(q{ConstructorArgs}, q{this.args}, q{this.locator}) ~ q{);});
                }
            } catch (Exception e) {

                throw new InstanceFactoryException("Error occurred during instantiation of " ~ name!T, e);
            }
        }
    }
}

/**
ditto
**/
auto constructorBasedFactory(T, Args...)(auto ref Args args) {
    mixin assertObjectConstructorCompatible!(T, Args);
    auto constructor = new ConstructorBasedFactory!(T, Args)(args);

    return constructor;
}

/**
Instantiates a component using a method from other component (factory method pattern).

Encapsulates construction of component using factory method.
Arguments that are RuntimeReferences, will be replaced with components extracted
from locator, and passed to factory's method.
In case when method is not static member, the algorithm will use
an instantiaton of factory passed to it, or extracted from locator
if a RuntimeReference is passed.

Params:
    T = factory that is used to instantiate component using it's method
    method = the name of method used to instantiate component
    W = the factory T, or a RuntimeReference to the factory.
    Args = type tuple of arguments passed to factory.
**/

template FactoryMethodBasedFactory(T, string method, W, Args...) {
    alias FirstTemplate = FactoryMethodInstanceFactory!(method, T, Args);

    alias FactoryMethodBasedFactory = FirstTemplate!W;
}

/**
ditto
**/
template FactoryMethodInstanceFactory(string method, T, Args...)
    if (
        isMethodCompatible!(T, method, Args) &&
        isAggregateType!(ReturnType!(getCompatibleOverload!(T, method, Args)))
    ) {

    template FactoryMethodInstanceFactory(W = T)
        if (is(W : RuntimeReference) || is(W : T)) {

        alias Compatible = getCompatibleOverload!(T, method, Args);
        alias Z = ReturnType!Compatible;
        alias Params = Parameters!Compatible;

        class FactoryMethodInstanceFactory : InstanceFactory!Z {

            mixin AllocatorAwareMixin!(typeof(this));
            mixin LocatorAwareMixin!(typeof(this));
            mixin ParameterHolder!Args;

            static if (!isStaticFunction!(Compatible)) {

                private {
                    W factoryMethodComponent_;
                }

                // invariant {

                // }

                @property {

                    this(ref W factoryMethodComponent) {
                        this.factoryMethodComponent = factoryMethodComponent;
                    }

                    /**
                    Set factoryMethodComponent

                    Params:
                        factoryMethodComponent = the component used to instantiate components of type Z.

                    Returns:
                        typeof(this)
                    **/
                    typeof(this) factoryMethodComponent(W factoryMethodComponent) @safe nothrow pure
                    in {
                         static if (is(W == class) || is(W == interface)) {
                            assert(factoryMethodComponent !is null);
                        }
                    }
                    body {
                        this.factoryMethodComponent_ = factoryMethodComponent;

                        return this;
                    }

                    /**
                    Get factoryMethodComponent

                    Returns:
                        W
                    **/
                    W factoryMethodComponent() @safe nothrow pure {
                        return this.factoryMethodComponent_;
                    }
                }
            }

            /**
            Create a new instance of object of type T.

            Returns:
                T instantiated component
            **/
            Z factory() {
                try {
                    static if (!__traits(isStaticFunction, Compatible)) {

                        mixin(
                            q{return __traits(getMember, this.factoryMethodComponent.resolve!(T)(this.locator), method)(} ~
                            compileArgumentsTuple!Params(q{Params}, q{this.args}, q{this.locator}) ~
                            q{);}
                            );
                    } else {

                        mixin(
                            q{return __traits(getMember, T, method)(} ~
                            compileArgumentsTuple!Params(q{Params}, q{this.args}, q{this.locator}) ~
                            q{);}
                            );
                    }
                } catch (Exception e) {

                    throw new InstanceFactoryException(
                        "Error occurred during instantiation of " ~
                        name!T ~
                        " using factory method of "
                        ~ name!T ~
                        "." ~
                        method,
                        e
                    );
                }
            }
        }
    }
}

/**
ditto
**/
auto factoryMethodBasedFactory
        (T, string method, Args...)
        (Args args)
{

    static if (
            (Args.length > 0) &&
            (
                is(Args[0] : RuntimeReference) ||
                is(Args[0] : T)
            ) &&
            isNonStaticMethodCompatible!(T, method, Args[1 .. $])
        ) {

        auto f()(Args[0] fact, Args[1 .. $] args)
        {

            alias FactoryMethodInstanceFactoryInitial = FactoryMethodInstanceFactory!(method, T, Args[1 .. $]);
            auto factory = new FactoryMethodInstanceFactoryInitial!(Args[0])(fact);

            static if (Args[1 .. $].length > 0) {
                factory.args = args;
            }

            return factory;
        }
    }

    static if (
            isStaticMethodCompatible!(T, method, Args)
        ) {

        auto f()(Args args)
            {

            alias FactoryMethodInstanceFactoryInitial = FactoryMethodInstanceFactory!(method, T, Args);
            auto factory = new FactoryMethodInstanceFactoryInitial!T;

            static if (Args.length > 0) {
                factory.args = args;
            }

            return factory;
        }
    }

    return f(args);
}


/**
Instantiates component of type T using a delegate or function.

Encapsulates component's construction logic using a delegate.
The algorithm uses a delegate to create required component,
with a set of Args that are passed to delegate, and a locator
for dependency fetching.

Params:
    T = the constructed component
    Args = type tuple of arguments passed to delegate for component's construction.
**/
class CallbackFactory(T, Dg, Args...) : InstanceFactory!T
    if ((is(Dg == T delegate (RCIAllocator, Locator!(), Args)) || is(Dg == T function (RCIAllocator, Locator!(), Args)))) {

    mixin AllocatorAwareMixin!(typeof(this));
    mixin LocatorAwareMixin!(typeof(this));
    mixin ParameterHolder!Args;

    private {
        Dg dg;
    }

    public {
        /**
            Constructor for CallbackFactory!(T, Dg, Args)

            Params:
                dg = delegate used to create object
                args = arguments passed to delegate
        **/
        this(Dg dg, ref Args args) {
            this.dg = dg;
            this.args(args);
        }

        /**
        See InstanceFactory interface
        **/
        T factory() {
            try {

                return this.dg(this.allocator, this.locator, args.expand);
            } catch (Exception e) {

                throw new InstanceFactoryException(
                    "Error occurred during instantiation of " ~
                    name!T ~
                    " using callback factory",
                    e
                );
            }
        }
    }
}

/**
ditto
**/
auto callbackFactory(T, Args...)(T delegate(RCIAllocator, Locator!(), Args) dg, auto ref Args args) {
    auto constr = new CallbackFactory!(T, T delegate(RCIAllocator, Locator!(), Args), Args)(dg, args);
    return constr;
}

/**
ditto
**/
auto callbackFactory(T, Args...)(T function(RCIAllocator, Locator!(), Args) dg, auto ref Args args) {
    auto constr = new CallbackFactory!(T, T function(RCIAllocator, Locator!(), Args), Args)(dg, args);
    return constr;
}

/**
Configures/modifies component of type T with help of a delegate or function.

Encapsulates component configuration logic using a delegate.
The algorithm calls delegate, with a locator, a set of Args,
and configured component, in order to modify the component.

Note:
    If component is not a reference type it is recommended to pass it by ref
    in order to avoid receiving of a copy and not original one in delegate.
    It is expected that the callback will use somehow method on which it was annotated with.
Params:
    T = the component
    Args = type tuple of arguments used by delegate for customization.
**/
class CallbackConfigurer(T, X, Dg, Args...) : PropertyConfigurer!T
    if (
        is(T : X) && (
            is(Dg : void delegate (Locator!(), X, Args)) ||
            is(Dg : void function (Locator!(), X, Args)) ||
            is(Dg : void delegate (Locator!(), ref X, Args)) ||
            is(Dg : void function (Locator!(), ref X, Args))
        )
    ) {

    mixin ParameterHolder!Args;
    mixin LocatorAwareMixin!(typeof(this));

    private {
        Dg dg;
    }

    public {
        /**
            Constructor for CallbackConfigurer!(T, Dg, Args)

            Params:
                dg = delegate used to configure the created object
                args = arguments passed to delegate
        **/
        this(Dg dg, ref Args args) {
            this.dg = dg;
            this.args(args);
        }

        /**
        Accepts a reference to an object that is to be configured by the configurer.

        Params:
        	object = An object of type T, that will be configured
        **/
        void configure(ref T object) {

            try {

                return this.dg(this.locator_, object, args.expand);
            } catch (Exception e) {
            	throw new PropertyConfigurerException("Error occurred during callback configuration of " ~ name!T, e);
            }
        }
    }
}

/**
ditto
**/
auto callbackConfigurer(T, X, Args...)(void delegate(Locator!(), X, Args) dg, auto ref Args args) if (is(T : X)) {
    auto constr = new CallbackConfigurer!(T, X, void delegate(Locator!(), X, Args), Args)(dg, args);
    return constr;
}

/**
ditto
**/
auto callbackConfigurer(T, X, Args...)(void function(Locator!(), X, Args) dg, auto ref Args args) if (is(T : X))  {
    auto constr = new CallbackConfigurer!(T, X, void function(Locator!(), X, Args), Args)(dg, args);
    return constr;
}

/**
ditto
**/
auto callbackConfigurer(T, X, Args...)(void delegate(Locator!(), ref X, Args) dg, auto ref Args args) if (is(T : X))  {
    auto constr = new CallbackConfigurer!(T, X, void delegate(Locator!(), ref X, Args), Args)(dg, args);
    return constr;
}

/**
ditto
**/
auto callbackConfigurer(T, X, Args...)(void function(Locator!(), ref X, Args) dg, auto ref Args args) if (is(T : X))  {
    auto constr = new CallbackConfigurer!(T, X, void function(Locator!(), ref X, Args), Args)(dg, args);
    return constr;
}


/**
Instantiates a component using a value as basis.

Instantiates a component using a value as basis.
As a consequence, any reference based type will
point to same content when it is instantiated
multiple times.
**/
class ValueInstanceFactory(T) : InstanceFactory!T {
    private {
        T initial_;
    }

    mixin AllocatorAwareMixin!(typeof(this));
    mixin LocatorAwareMixin!(typeof(this));

    public {
        /**
            Constructor for ValueInstanceFactory!T

            Params:
                initial = argument that is to be passed as created object
        **/
        this(T initial) {
            this.initial = initial;
        }

        @property {

            /**
                Set initial

                Params:
                    initial = value which will be passed as created component
                Returns:
                    typeof(this)
            **/
        	ValueInstanceFactory!T initial(T initial) @safe nothrow {
        		this.initial_ = initial;

        		return this;
        	}

            /**
                Get initial

                Returns:
                    T
            **/
        	T initial() @safe nothrow {
        		return this.initial_;
        	}
        }

        /**
        Create a new instance of object of type T.

        Returns:
            T instantiated component
        **/
        T factory() {
            return this.initial();
        }
    }
}

/**
InstanceFactory that delegates the task of instantiating a component
to some third party factory.
**/
class DelegatingInstanceFactory(T, X : T) : InstanceFactory!T, MutableDecorator!(Factory!X) {

    mixin AllocatorAwareMixin!(typeof(this));
    mixin LocatorAwareMixin!(typeof(this));

    private {
        Factory!X decorated_;
    }

    public {

        /**
            Default constructor for DelegatingInstanceFactory!(T, X)
        **/
        this() {

        }

        /**
            Constructor for DelegatingInstanceFactory!(T, X)

            Params:
                factory = the factory to which this instance will delegate the task of creating a component
        **/
        this(Factory!X factory) {
            this.decorated = factory;
        }

        mixin MutableDecoratorMixin!(Factory!X);

        /**
        Create a new instance of object of type T.

        Returns:
            T instantiated component
        **/
        T factory() {
            return this.decorated.factory();
        }
    }
}

/**
Default implementation of destructor that calls dispose upon classes only.
**/
class DefaultInstanceDestructor(T) : InstanceDestructor!T {

    mixin AllocatorAwareMixin!(typeof(this));
    mixin LocatorAwareMixin!(typeof(this));

    /**
    Destruct a component of type T and deallocate it using stored allocator.

    Params:
        destructable = element to be destructed and deallocated using stored allocator
    **/
    void destruct(ref T component) {
        import std.experimental.allocator : dispose;

        static if (is(T == class)) {

            this.allocator.dispose(component);
        }

        // Do nothing here.
    }
}

/**
Instance destructor that uses a callback to destroy and deallocate components of type T.
**/
class CallbackInstaceDestructor(T, Dg : void delegate(RCIAllocator, ref T destructable, Args), Args...) : InstanceDestructor!(T) {
    mixin AllocatorAwareMixin!(typeof(this));
    mixin LocatorAwareMixin!(typeof(this));
    mixin ParameterHolder!(Args);

    private {
        Dg dg_;
    }

    public {
        @property {
            /**
            Set dg

            Params:
                dg = the delegate used to destroy component
            Returns:
                typeof(this)
            **/
            typeof(this) dg(Dg dg) @safe nothrow pure {
                this.dg_ = dg;

                return this;
            }

            /**
            Get dg

            Returns:
                Dg
            **/
            Dg dg() @safe nothrow pure {
                return this.dg_;
            }
        }

        /**
        Destruct a component of type T and deallocate it using stored allocator.

        Params:
            destructable = element to be destructed and deallocated using stored allocator
        **/
        void destruct(ref T destructable) {
            this.dg()(this.allocator, destructable, this.args.expand);
        }
    }
}

/**
ditto
**/
CallbackInstaceDestructor!(T, Dg, Args) callbackInstanceDestructor
    (T, Dg : void delegate(RCIAllocator, ref T destructable, Args), Args...)
    (Dg dg, Args args) {

    auto callbackInstanceDestructor =
        new CallbackInstaceDestructor!(T, Dg, Args)()
        .dg(dg);

    static if (Args.length > 0) {

        callbackInstanceDestructor.args(args);
    }

    return callbackInstanceDestructor;
}

/**
Instance destructor using an third party component to do destruction of components.
**/
template FactoryMethodInstanceDestructor(string method, T, Z, Args...)
    if (
        isSomeFunction!(getMember!(T, method)) &&
        isMethodCompatible!(T, method, Z, Args)
    ) {

    alias Compatible = getCompatibleOverload!(T, method, Z, Args);

    class FactoryMethodInstanceDestructor : InstanceDestructor!Z {

        mixin LocatorAwareMixin!(typeof(this));
        mixin AllocatorAwareMixin!(typeof(this));
        mixin ParameterHolder!(Args);

        public {

            static if (!isStaticFunction!(Compatible)) {

                T destructor_;

                @property {
                    /**
                    Set destructor

                    Params:
                        destructor = the component destructor used to destroy component

                    Returns:
                        typeof(this)
                    **/
                    typeof(this) destructor(T destructor) @safe nothrow pure {
                        this.destructor_ = destructor;

                        return this;
                    }

                    /**
                    Get destructor

                    Returns:
                        T
                    **/
                    T destructor() @safe nothrow pure {
                        return this.destructor_;
                    }
                }
            }

            /**
            Destruct a component of type T and deallocate it using stored allocator.

            Params:
                destructable = element to be destructed and deallocated using stored allocator
            **/
            void destruct(ref Z destructable) {
                static if (!isStaticFunction!(Compatible)) {
                    __traits(getMember, this.destructor, method)(destructable, this.args.expand);
                } else {
                    __traits(getMember, T, method)(destructable, this.args.expand);
                }
            }
        }
    }
}

/**
Create an instance destructor that uses third party component's method to do destruction of a component.

Create an instance destructor that uses third party component's method to do destruction of a component.

Params:
    Z = the type of destructed object
    method = method used from component T to destroy component Z
    destructor = type of destructor component
    arguments = list of arguments passed to destructor component's method

Returns:
    FactoryMethodInstanceDestructor!(method, T, Z, Args)
**/
auto factoryMethodInstanceDestructor(
    Z,
    string method,
    T,
    Args...
)(
    T destructor,
    Args arguments
) if (isNonStaticMethodCompatible!(T, method, Z, Args)) {

    auto instanceDestructor = new FactoryMethodInstanceDestructor!(method, T, Z, Args)();
    instanceDestructor.destructor = destructor;

    static if (Args.length > 0) {

        instanceDestructor.args = arguments;
    }

    return instanceDestructor;
}

/**
ditto
**/
auto factoryMethodInstanceDestructor(
    Z,
    string method,
    T,
    Args...
)(
    Args arguments
)
if (isStaticMethodCompatible!(T, method, Z, Args)) {

    auto destructor = new FactoryMethodInstanceDestructor!(method, T, Z, Args)();

    static if (Args.length > 0) {
        destructor.args = arguments;
    }

    return destructor;
}

/**
An check if the argument list passed to ConstructorBasedFactory or MethodConfigurer is compatible with signature of underlying
method or constructor.

Note:
	For now it checks if the lengths are equal. For future it should also check if types are compatible.
**/
template isArgumentListCompatible(alias func, ArgTuple...)
	if (isSomeFunction!func) {
    bool isArgumentListCompatible() {
        alias FuncParams = Parameters!func;
        alias Required = Filter!(partialSuffixed!(isValueOfType, void), ParameterDefaults!func);

        static if ((ArgTuple.length < Required.length) || (ArgTuple.length > FuncParams.length)) {

            return false;
        } else {

            bool result = true;
            foreach (index, Argument; ArgTuple) {

                static if (!is(Argument : RuntimeReference) && !isImplicitlyConvertible!(Argument, FuncParams[index])) {

                    result = false;
                    break;
                }
            }

            return result;
        }
    }
}

mixin template assertFieldCompatible(T, string field, Arg) {
    import aermicioi.util.traits;
    import std.traits;
    import std.meta;

    static assert(isField!(T, field), name!T ~ "'s " ~ field ~ " member is not a field");
    static assert(isProtection!(T, field, "public"), name!T ~ "'s " ~ field ~ " is not public and therefore cannot be accessed.");
    static assert(is(Arg : RuntimeReference) ? true : isImplicitlyConvertible!(Arg, typeof(getMember!(T, field))), name!T ~"'s " ~ field ~ " type " ~ name!(typeof(getMember!(T, field))) ~ " doesn't match with passed arguments type " ~ name!Arg);
}

enum bool isFieldCompatible(T, string field, Arg) =
    isField!(T, field) &&
    isProtection!(T, field, "public") &&
    is(Arg : RuntimeReference) ? true : isImplicitlyConvertible!(Arg, typeof(getMember!(T, field)));

mixin template assertObjectConstructorCompatible(T, Args...) {
    import aermicioi.util.traits;
    import std.traits;
    import std.meta;

    static assert(hasMember!(T, "__ctor"), name!T ~ " doesn't have any constructor to call.");
    static assert(isProtection!(T, "__ctor", "public"), name!T ~ "'s constructor is not public.");
    static assert(isSomeFunction!(__traits(getMember, T, "__ctor")), name!T ~ "'s constructor is not a function, probably a template.");
    static assert(variadicFunctionStyle!(__traits(getMember, T, "__ctor")) == Variadic.no, name!T ~ "'s constructor is a variadic function. Only non-variadic constructors are supported.");
    static assert(Filter!(partialSuffixed!(isArgumentListCompatible, Args), __traits(getOverloads, T, "__ctor")).length == 1, "None, or multiple overloads found for " ~ name!T ~ "'s constructor with passed arguments.");
}

enum bool isObjectConstructorCompatible(T, Args...) = isMethodCompatible!(T, "__ctor", Args);

mixin template assertObjectMethodCompatible(T, string method, Args...) {
    import std.range : only;
    import std.array : array;
    import aermicioi.util.traits;
    import std.traits;
    import std.meta;

    static assert(hasMember!(T, method), name!T ~ "'s method " ~ method ~ " not found.");
    static assert(isProtection!(T, method, "public"), name!T ~ "'s method " ~ method ~ " is not public");
    static assert(isSomeFunction!(__traits(getMember, T, method)), name!T ~ "'s member " ~ method ~ " is not a function, probably a field, or a template.");
    static assert(variadicFunctionStyle!(__traits(getMember, T, method)) == Variadic.no, name!T ~ "'s method " ~ method ~ "is variadic function. Only non-variadic methods are supported.");
    static assert(Filter!(partialSuffixed!(isArgumentListCompatible, Args), getOverloads!(T, method)).length == 1, name!T ~ "'s " ~ method ~ " doesn't have overload matching passed arguments (" ~ only(staticMap!(name, Args)).joiner(", ").array ~ "), or has several overloads that match.");
}

enum bool isObjectMethodCompatible(T, string method, Args...) = isMethodCompatible!(T, method, Args);


template isMethodCompatible(T, string method, Args...) {
    enum bool isMethodCompatible =
            hasMember!(T, method) &&
            isProtection!(T, method, "public") &&
            isSomeFunction!(__traits(getMember, T, method)) &&
            (variadicFunctionStyle!(__traits(getMember, T, method)) == Variadic.no) &&
            (Filter!(partialSuffixed!(isArgumentListCompatible, Args), getOverloads!(T, method)).length == 1);
}

template getCompatibleOverload(T, string method, Args...)
    if (isObjectMethodCompatible!(T, method, Args)) {

    alias getCompatibleOverload = Filter!(partialSuffixed!(isArgumentListCompatible, Args), getOverloads!(T, method))[0];
}

alias isStaticMethodCompatible = templateAnd!(
    isMethodCompatible,
    chain!(
        isStaticFunction,
        getCompatibleOverload
    )
);

alias isNonStaticMethodCompatible = templateAnd!(
    isMethodCompatible,
    chain!(
        templateNot!isStaticFunction,
        getCompatibleOverload
    )
);

alias isNonStaticMethodCompatibleAndReturnTypeOf(X) = templateAnd!(
    isMethodCompatible,
    chain!(
        templateNot!isStaticFunction,
        getCompatibleOverload
    ),
    partialSuffixed!(
        chain!(
            partialPrefixed!(
                get,
                0
            ),
            getCompatibleOverload
        ),
        X
    )
);

private {
    template isValueOfType(alias value, Type) {
        enum bool isValueOfType = is(typeof(value) == Type);
    }

    template isValueOfType(Value, Type) {
        enum bool isValueOfType = is(Value == Type);
    }

    enum bool isStruct(T) = is(T == struct);

    alias hasDefaultCtor =
        partialSuffixed!(
            templateOr!(
                templateNot!hasMember,
                chain!(
                    isStruct,
                    get!0
                ),
                templateAnd!(
                    partialSuffixed!(
                        isProtection,
                        "public"
                    ),
                    chain!(
                        partialPrefixed!(
                            anySatisfy,
                            eq!0
                        ),
                        partialPrefixed!(
                            staticMap,
                            arity
                        ),
                        chain!(
                            partialPrefixed!(
                                Filter,
                                partialSuffixed!(
                                    isProtection,
                                    "public"
                                )
                            ),
                            getOverloads
                        )
                    )
                )
            ),
            "__ctor"
        );
}

private {
    string compileArgumentsTuple(Tuple...)(string types, string array, string locator) {
        import std.conv : to;
        import std.array : join;
        string[] stmt;

        foreach (index, Type; Tuple) {
            stmt ~= array ~ "[" ~ index.to!string ~ "].resolve!(" ~ types ~ "[" ~ index.to!string ~ "])(" ~ locator ~ ")";
        }

        return stmt.join(", ");
    }
}