/**
License:
	Boost Software License - Version 1.0 - August 17th, 2003

    Permission is hereby granted, free of charge, to any person or organization
    obtaining a copy of the software and accompanying documentation covered by
    this license (the "Software") to use, reproduce, display, distribute,
    execute, and transmit the Software, and to prepare derivative works of the
    Software, and to permit third-parties to whom the Software is furnished to
    do so, all subject to the following:

    The copyright notices in the Software and this entire statement, including
    the above license grant, this restriction and the following disclaimer,
    must be included in all copies of the Software, in whole or in part, and
    all derivative works of the Software, unless such copies or derivative
    works are solely in the form of machine-executable object code generated by
    a source language processor.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
    SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
    FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
    ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/
module aermicioi.aedi.storage.wrapper;

import std.traits;
import std.meta : staticMap;

/**
Interface for components that wrap a component of type T. Provides boxing and automatic unboxing of wrapped values.
**/
interface Wrapper(T) {
    public {

        /**
        Get wrapped value out of wrapper.

        Returns:
            T the wrapped value
        **/
        @property ref inout(T) value() inout;

        /**
        Alias wrapper to T for automatic unboxing of values.
        **/
        alias value this;
    }
}

/**
Interface for components that denotes tha ability to be casted to a particular type T.
**/
interface Castable(T) {
    public {

        /**
        Get a representation of current component in type T.

        Returns:
            T component transformed into T component.
        **/
        @property inout(T) casted() inout;

        /**
        Alias casted type for automatic casting of component.
        **/
        alias casted this;
    }
}

/**
Wrapper over some component of type T.

Wraps up a value of type T, and aliases itself to it. This object is used
wrap any component that is not of reference type (class or interface),
in order to be saveable into an object storage.
Also thanks to alias value this semantics, in D is possible to do automatic
unboxing of values, just like Java does with simple values :P.
**/
class WrapperImpl(T) : Wrapper!T {
    mixin WrapperMixin!T;
}

/**
ditto
**/
class CastableWrapperImpl(T, Castables...) : Wrapper!T, staticMap!(toCastable, Castables) {
    import std.meta;
    import aermicioi.util.traits;
    mixin WrapperMixin!T;

    mixin CastableMixin!(Castables);
}

private {

    mixin template WrapperMixin(T) {
        private {
            T value_;
        }

        public {

            this() @disable;

            this(ref T value) {
                this.value_ = value;
            }

            this(T value) {
                this.value_ = value;
            }

            @property ref inout(T) value() inout {
                return this.value_;
            }
        }
    }

    mixin template CastableMixin() {

    }

    mixin template CastableMixin(Type) {
        @property {
            inout(Type) casted() inout {
                return cast(inout(Type)) this.value;
            }
        }
    }

    mixin template CastableMixin(Type, Second, More...) {

        mixin CastableMixin!(Type);

        mixin CastableMixin!(Second);

        mixin CastableMixin!(More);
    }

    template toCastable(T) {
        alias toCastable = Castable!T;
    }
}
