/**
License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:

	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/

module aermicioi.aedi.test.factory_configurer;

import aermicioi.aedi.storage;
import aermicioi.aedi.container.aggregate_container;
import aermicioi.aedi.container.value_container;
import aermicioi.aedi.configurer.register.context;
import aermicioi.aedi.configurer.register.factory_configurer;
import aermicioi.aedi.container.singleton_container;
import aermicioi.aedi.exception;
import aermicioi.aedi.test.fixture;
import aermicioi.aedi.storage.allocator_aware;
import std.exception;
import std.traits;

unittest {
    auto container = new SingletonContainer;
    auto parameters = new ValueContainer;
    parameters.configure.register!long(21L);
    assert(parameters.locate!long == 21, "well not here");

    with (container.configure) {
        register!(Employee)("employee.custom").locator(parameters);
        register!(Person)("person.custom");
        register!(Company)
            .factoryMethod!(StructFixtureFactory, "company")
            .locator(parameters);
        register!(Employee)();
        register!(Person, Employee)
            .callback(
                delegate void(Locator!() loc, Employee emp) {
                    emp.name = "Trustindelegates";
                }
            )
            .locator(parameters);

        register!(Identifiable!ulong, Employee)();
        register!(StructFixtureFactory)
            .callback(
                function StructFixtureFactory(RCIAllocator alloc, Locator!() loc)
                {
                    return StructFixtureFactory(new Job("Aligator", Currency(20)));
                }
            )
            .callback(
                delegate void(Locator!() loc, ref StructFixtureFactory factory) {
                    factory.person = new Person("Ande muhardad", 20);
                }
            )
            .callback(
                function void(Locator!() loc, ref StructFixtureFactory factory) {
                    factory.employee = new Employee("Amhad muhamhad", 22);
                }
            );

        register!Currency
            .autowire
            .locator(parameters);
    }

    assert(container.locate!(StructFixtureFactory).employee.name == "Amhad muhamhad");
    assert(container.locate!(StructFixtureFactory).person.name == "Ande muhardad");
    assert(container.locate!(StructFixtureFactory).job.name == "Aligator");
    assert(container.locate!(Currency)().amount == 21);
    assert(container.locate!(Company)().id == 20);
}

unittest {
    AggregateContainer cont = new AggregateContainer;
    auto singleton = new SingletonContainer();
    auto parameters = new ValueContainer;

    cont.set(singleton, "singleton");
    cont.set(parameters, "parameters");

    auto fact = cont.configure("singleton").register!(Employee)("employee.custom")
        .tag("dev-ops")
        .tag(10L)
        .tag(["127.0.0.1", "192.168.0.1"])
        .tag("team-lead");

    import aermicioi.aedi.factory.decorating_factory;
    auto tagOne = cast(TaggableFactoryDecorator!(Object, string[])) fact.wrapper;
    auto tagTwo = cast(TaggableFactoryDecorator!(Object, long)) tagOne.decorated;
    auto tagThree = cast(TaggableFactoryDecorator!(Object, string)) tagTwo.decorated;

    assert(tagOne !is null);
    assert(tagOne.tags == [ ["127.0.0.1", "192.168.0.1"] ]);

    assert(tagTwo !is null);
    assert(tagTwo.tags == [ 10L ]);

    assert(tagThree !is null);
    assert(tagThree.tags == ["dev-ops", "team-lead"]);
}

unittest {

    SingletonContainer container = new SingletonContainer();
    auto obj = new MockObject;

    container.configure.register!MockObject()
        .value(obj);

    assert(container.locate!MockObject() is obj);
}

unittest {

    SingletonContainer container = new SingletonContainer();

    MockObject obj = new MockObject;
    MockValueFactory!MockObject fact = new MockValueFactory!MockObject;

    with (container.configure) {

        register!MockObject()
            .value(obj);

        register!MockInterface
            .parent(fact);
    }

    container.instantiate();

    assert(container.locate!MockObject is obj);
    assert(container.locate!MockInterface !is null);
}

unittest {
   import aermicioi.aedi.container.proxy_container;
   import aermicioi.aedi.factory.proxy_factory;

   AggregateContainer cont = new AggregateContainer;
   auto singleton = new SingletonContainer();
   auto proxy = new ProxyContainerImpl!SingletonContainer();
   proxy.decorated = singleton;
   auto parameters = new ValueContainer;

   cont.set(proxy, "singleton");
   cont.set(parameters, "parameters");

   cont.configure("singleton").register!(ProxyablePerson)("employee")
       .proxy;

   assert(proxy.has("employee"));

   assert((cast(Proxy!ProxyablePerson) cont.locate!ProxyablePerson("employee")) !is null);
}

unittest {
    import std.experimental.allocator;

    SingletonContainer container = new SingletonContainer();

    MockObject obj = new MockObject;
    bool destroyed = false;
    with (container.configure) {
        register!MockObject
            .destructor((RCIAllocator allocator, ref MockObject object, int i) {
                destroyed = true;
            }, 1);
    }

    container.instantiate;
    container.terminate;

    assert(destroyed);
}

class DummyDestructor {
    static staticDestroy(MockObject obj, bool* destroyed) {
        destroy(obj);
        *destroyed = true;
    }

    auto destruct(MockObject obj, bool* destroyed) {
        destroy(obj);
        *destroyed = true;
    }
}

unittest {
    import std.experimental.allocator;

    SingletonContainer container = new SingletonContainer();

    MockObject obj = new MockObject;
    bool destroyed = false;
    with (container.configure) {
        register!MockObject
            .destructor!("staticDestroy", DummyDestructor)(&destroyed);
    }

    container.instantiate;
    container.terminate;

    assert(destroyed);
}

unittest {
    import std.experimental.allocator;

    SingletonContainer container = new SingletonContainer();

    MockObject obj = new MockObject;
    bool destroyed = false;
    with (container.configure) {
        register!MockObject
            .destructor!("destruct")(new DummyDestructor, &destroyed);
    }

    container.instantiate;
    container.terminate;

    assert(destroyed);
}