/**
License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:

	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/
module aermicioi.aedi.test.factory.factory;

import aermicioi.aedi.factory;
import aermicioi.aedi.container;
import aermicioi.aedi.storage;
import aermicioi.aedi.exception;
import aermicioi.aedi.test.fixture;

unittest {
    SingletonContainer container = new SingletonContainer;
    GenericFactory!Employee employee = new GenericFactoryImpl!Employee(container);
    GenericFactory!Company company = new GenericFactoryImpl!Company(container);
    GenericFactory!Job job = new GenericFactoryImpl!Job(container);

    container.set(new WrappingFactory!(Factory!Employee)(employee), "employee");
    container.set(new WrappingFactory!(Factory!Company)(company), "company");
    container.set(new WrappingFactory!(Factory!Job)(job), "job");

    container.instantiate();
}

unittest {
    ObjectStorage!() parameters = new ObjectStorage!();
    parameters.set(new WrapperImpl!string("scrapper"), name!string);
    parameters.set(new WrapperImpl!ubyte(20), name!ubyte);
    parameters.set(new Job("scrapper", Currency(20000)), name!Job);
    parameters.link(name!Job, "job");
    parameters.link(name!string, "name");
    parameters.link(name!ubyte, "age");
    GenericFactory!Employee employeeFactory = new GenericFactoryImpl!Employee(parameters);

    auto employee = employeeFactory.factory();
    assert(employee !is null);

    employeeFactory.setInstanceFactory(constructorBasedFactory!Employee("scrapper", lref!ubyte, lref!Job));
    employee = employeeFactory.factory();

    assert(employee !is null);
    assert(employee.name == "scrapper");
    assert(employee.age == 20);
    assert(employee.job == parameters.get("job"));
    assert(employee.job.name == "scrapper");
    assert(employee.job.payment == 20000);

    employeeFactory.setInstanceFactory(callbackFactory(delegate (RCIAllocator alloc, Locator!() loc, string name, ubyte age) {
        return new Employee(name, age, new Job("scrapped", Currency(0)));
    }, "test", cast(ubyte) 13));
    employeeFactory.addPropertyConfigurer(callbackConfigurer!Employee(delegate (Locator!() loc, Employee e, Company comp) {
        e.company = comp;
    },
    new Company(30)));

    employee = employeeFactory.factory();

    assert(employee !is null);
    assert(employee.name == "test");
    assert(employee.age == 13);
    assert(employee.job != parameters.get("job"));
    assert(employee.job.name == "scrapped");
    assert(employee.job.payment == 0);
    assert(employee.company.id == 30);
}

unittest {
    ObjectStorage!() parameters = new ObjectStorage!();
    parameters.set(new WrapperImpl!string("scrapper"), name!string);
    parameters.set(new WrapperImpl!ubyte(20), name!ubyte);
    parameters.set(new Job("scrapper", Currency(20000)), name!Job);
    parameters.set(new FixtureFactory(new Job("salaryman", Currency(2000))), "factory");
    parameters.set(new WrapperImpl!StructFixtureFactory(StructFixtureFactory(new Job("salaryman", Currency(2000)))), "structFactory");
    parameters.link(name!Job, "job");
    parameters.link(name!string, "name");
    parameters.link(name!ubyte, "age");

    {
        GenericFactory!(Union) unionFactory = new GenericFactoryImpl!(Union)(parameters);

        assert(unionFactory.factory == Union());
        unionFactory.addPropertyConfigurer(fieldConfigurer!("c", Union)(10));
        assert(unionFactory.factory.a == 10);
        auto structure = StructFixtureFactory(new Job("salaryman", Currency(2000)));
        unionFactory.addPropertyConfigurer(fieldConfigurer!("f", Union)(structure));
        assert(unionFactory.factory.f == structure);
    }

    {
        GenericFactory!(Currency) currencyFactory = new GenericFactoryImpl!(Currency)(parameters);

        auto currency = currencyFactory.factory;
        assert(currency == Currency());
        currencyFactory.setInstanceFactory(constructorBasedFactory!Currency(10));
        assert(currencyFactory.factory == Currency(10));
        currencyFactory.addPropertyConfigurer(methodConfigurer!("amount", Currency)(20));
        assert(currencyFactory.factory == Currency(20));
        currencyFactory.addPropertyConfigurer(fieldConfigurer!("amount_", Currency)(30));
        assert(currencyFactory.factory == Currency(30));
    }

    {
        auto companyFactory = new GenericFactoryImpl!(Company)(parameters);
        companyFactory.setInstanceFactory(factoryMethodBasedFactory!(FixtureFactory, "company"));
        assert(companyFactory.factory !is null);
        assert(companyFactory.factory.id == 20);

        auto jobFactory = new GenericFactoryImpl!(Job)(parameters);
        jobFactory.setInstanceFactory(factoryMethodBasedFactory!(FixtureFactory, "job")(new FixtureFactory(new Job("billionaire", Currency(2 ^^ 32)))));
        assert(jobFactory.factory !is null);
        assert(jobFactory.factory.name == "billionaire");
        assert(jobFactory.factory.payment == Currency(2 ^^ 32));

        jobFactory = new GenericFactoryImpl!(Job)(parameters);
        jobFactory.setInstanceFactory(factoryMethodBasedFactory!(FixtureFactory, "job")("factory".lref));
        assert(jobFactory.factory !is null);
        assert(jobFactory.factory.name == "salaryman");
        assert(jobFactory.factory.payment == Currency(2000));
    }

    {
        auto companyFactory = new GenericFactoryImpl!(Company)(parameters);
        companyFactory.setInstanceFactory(factoryMethodBasedFactory!(StructFixtureFactory, "company"));
        assert(companyFactory.factory !is null);
        assert(companyFactory.factory.id == 20);

        auto jobFactory = new GenericFactoryImpl!(Job)(parameters);
        jobFactory.setInstanceFactory(factoryMethodBasedFactory!(StructFixtureFactory, "job")(StructFixtureFactory(new Job("billionaire", Currency(2 ^^ 32)))));
        assert(jobFactory.factory !is null);
        assert(jobFactory.factory.name == "billionaire");
        assert(jobFactory.factory.payment == Currency(2 ^^ 32));

        jobFactory = new GenericFactoryImpl!(Job)(parameters);
        jobFactory.setInstanceFactory(factoryMethodBasedFactory!(StructFixtureFactory, "job")("structFactory".lref));
        jobFactory.addPropertyConfigurer(fieldConfigurer!("averagePayment", Job)(Currency(150)));
        assert(jobFactory.factory !is null);
        assert(jobFactory.factory.name == "salaryman");
        assert(jobFactory.factory.payment == Currency(2000));
        assert(jobFactory.factory.averagePayment == Currency(150));
    }

    {
        auto currencyFactory = new GenericFactoryImpl!Currency(parameters);
        currencyFactory.setInstanceFactory(factoryMethodBasedFactory!(StructFixtureFactory, "currency")(StructFixtureFactory(new Job())));
        assert(currencyFactory.factory.amount == 0);

        currencyFactory.setInstanceFactory(factoryMethodBasedFactory!(StructFixtureFactory, "basicPayment")(cast(ptrdiff_t) 20));
        assert(currencyFactory.factory.amount == 20);
    }

    {
        auto currencyFactory = new GenericFactoryImpl!(Currency)(parameters);
        currencyFactory.setInstanceFactory(callbackFactory(
            delegate Currency(RCIAllocator alloc, Locator!() loc) {
                return Currency(20);
            }
        ));
        assert(currencyFactory.factory.amount == 20);

        currencyFactory.addPropertyConfigurer(callbackConfigurer!Currency(
            delegate (Locator!() loc, ref Currency c, int amount) {
                c.amount = amount;
            },
            39
        ));

        assert(currencyFactory.factory.amount == 39);
    }
}

unittest {
    ObjectStorage!() parameters = new ObjectStorage!();
    parameters.set(new WrapperImpl!string("scrapper"), "name");
    parameters.set(new WrapperImpl!ubyte(20), "age");
    parameters.set(new Job("scrapper", Currency(20000)), name!Job);
    parameters.link(name!Job, "job");
    GenericFactory!Employee employeeFactory = new GenericFactoryImpl!Employee(parameters);

    employeeFactory.addPropertyConfigurer(methodConfigurer!("name", Employee)("name".lref));
    employeeFactory.addPropertyConfigurer(methodConfigurer!("age", Employee)("age".lref));
    employeeFactory.addPropertyConfigurer(methodConfigurer!("job", Employee)("job".lref));
    auto employee = employeeFactory.factory();

    assert(employee !is null);
    assert(employee.name == "scrapper");
    assert(employee.age == 20);
    assert(employee.job == parameters.get("job"));
    assert(employee.job.name == "scrapper");
    assert(employee.job.payment == 20000);
}