/**
License:
	Boost Software License - Version 1.0 - August 17th, 2003

	Permission is hereby granted, free of charge, to any person or organization
	obtaining a copy of the software and accompanying documentation covered by
	this license (the "Software") to use, reproduce, display, distribute,
	execute, and transmit the Software, and to prepare derivative works of the
	Software, and to permit third-parties to whom the Software is furnished to
	do so, all subject to the following:
	
	The copyright notices in the Software and this entire statement, including
	the above license grant, this restriction and the following disclaimer,
	must be included in all copies of the Software, in whole or in part, and
	all derivative works of the Software, unless such copies or derivative
	works are solely in the form of machine-executable object code generated by
	a source language processor.
	
	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
	SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
	FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
	ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

Authors:
	aermicioi
**/
module aermicioi.aedi.test.factory.reference;

import aermicioi.aedi.storage.object_storage;
import aermicioi.aedi.storage.wrapper;
import aermicioi.aedi.test.fixture;
import aermicioi.aedi.factory.reference;
import aermicioi.aedi.exception.invalid_cast_exception;
import std.exception;

unittest {
    ObjectStorage!() storage = new ObjectStorage!();
    
    MockObject object = new MockObject(20);
    MockExternObject eobject = new MockExternObject(30);
    MockStruct structt = MockStruct(10);
    
    storage.set(object, "obj");
    storage.set(new WrapperImpl!MockExternObject(eobject), "eobj");
    storage.set(new WrapperImpl!MockExternInterface(eobject), "eiface");
    storage.set(new WrapperImpl!MockStruct(structt), "str");
    
    auto objReference = new LocatorReference("obj");
    auto eobjReference = new LocatorReference("eobj");
    auto eifaceReference = new LocatorReference("eiface");
    auto strReference = new LocatorReference("str");
    
    assert(objReference.get(storage) is object);
    assert(strReference.get(storage) is storage.get("str"));
    
    assert(objReference.resolve!MockObject(storage) is object);
    assert(objReference.resolve!MockInterface(storage) is object);
    
    assert(eobjReference.resolve!MockExternObject(storage) is eobject);
    assert(eifaceReference.resolve!MockExternInterface(storage) is eobject);
    
    assert(strReference.resolve!MockStruct(storage) == structt);
    
    assertThrown!InvalidCastException(objReference.resolve!MockObjectFactoryMethod(storage));
    assertThrown!InvalidCastException(objReference.resolve!MockTotallyNotInheritedInterface(storage));
    
    assertThrown!InvalidCastException(eobjReference.resolve!MockExternInterface(storage));
    
    assertThrown!InvalidCastException(strReference.resolve!int(storage));
}

unittest {
    ObjectStorage!() storage = new ObjectStorage!();
    storage.set(new CastableWrapperImpl!(int, long)(10), "cobj");
    
    auto objReference = new LocatorReference("cobj");
    
    assert(objReference.resolve!int(storage) == 10);
    assert(objReference.resolve!long(storage) == 10L);
    
    assertThrown!InvalidCastException(objReference.resolve!ubyte(storage));
}

unittest {
    ObjectStorage!() storage = new ObjectStorage!();
    auto object = new MockExternObject;
    storage.set(new CastableWrapperImpl!(MockExternObject, MockExternInterface)(object), "cobj");
    
    auto objReference = new LocatorReference("cobj");
    
    assert(objReference.resolve!MockExternObject(storage) is object);
    assert(objReference.resolve!MockExternInterface(storage) is object);
    
    assertThrown!InvalidCastException(objReference.resolve!ubyte(storage));
}

unittest {
    ObjectStorage!() storage = new ObjectStorage!();
    
    MockFactory!MockObject cfactory = new MockFactory!MockObject;
    MockValueFactory!int sfactory = new MockValueFactory!int();
    
    assert(anonymous(cfactory).resolve!MockObject(storage) !is null);
    assert(anonymous(cfactory).resolve!MockObject(storage).classinfo is typeid(MockObject));
    
    assert(anonymous(sfactory).resolve!int(storage) == 0);
    assert(anonymous(sfactory).resolve!long(storage) == 0);
    assert(anonymous(sfactory).resolve!double(storage) == 0.0);
}

unittest {
    ObjectStorage!() storage = new ObjectStorage!();
    
    storage.set(new MockObject, "reliable");
    storage.set(new MockObject, "reliable.the.second");

    auto reliable = "reliable".lref.alternate("reliable.the.second".lref);
    auto unreliable = "unreliable".lref.alternate("reliable.the.second".lref);

    assert(reliable.resolve!MockObject(storage) is storage.get("reliable"));
    assert(unreliable.resolve!MockObject(storage) is storage.get("reliable.the.second"));
}

unittest {
    ObjectStorage!() storage = new ObjectStorage!();
    
    storage.set(new MockFactory!MockObject(), name!(MockFactory!MockObject));

    auto typed = lref!(MockFactory!MockObject);

    assert(typed.resolve!(MockFactory!MockObject)(storage) is storage.get(name!(MockFactory!MockObject)));
}