/**
 *  @file
 *  @copyright defined in aergo/LICENSE.txt
 */

package key

import (
	"crypto/aes"
	"crypto/cipher"
	"errors"

	sha256 "github.com/minio/sha256-simd"
)

func hashBytes(b1 []byte, b2 []byte) []byte {
	h := sha256.New()
	h.Write(b1)
	h.Write(b2)
	return h.Sum(nil)
}

func encrypt(base, key, data []byte) ([]byte, error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}
	// Never use more than 2^32 random nonces with a given key because of the risk of a repeat.
	if len(base) < 16 {
		return nil, errors.New("too short address length")
	}
	nonce := base[4:16]

	aesgcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	cipherbytes := aesgcm.Seal(nil, nonce, data, nil)
	return cipherbytes, nil
}

func decrypt(base, key, data []byte) ([]byte, error) {
	if len(base) < 16 {
		return nil, errors.New("too short address length")
	}
	nonce := base[4:16]

	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	aesgcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	plainbytes, err := aesgcm.Open(nil, nonce, data, nil)

	if err != nil {
		return nil, err
	}
	return plainbytes, nil
}
