---
author: Lucas Duailibe (@duailibe)
authorURL: https://twitter.com/duailibe
title: "Prettier 1.9: JSX Fragments, EditorConfig and Arrow Parens"
---

This release adds an option for arrow function parens in arguments, support for the [new JSX fragment syntax (`<>`)](https://reactjs.org/blog/2017/11/28/react-v16.2.0-fragment-support.html), support for `.editorconfig` files, and nice additions to our GraphQL and Markdown support.

<!--truncate-->

## Highlights

### JavaScript

#### Option to add parens in arrow function arguments ([#3324](https://github.com/prettier/prettier/pull/3324)) by [@rattrayalex](https://github.com/rattrayalex) and [@suchipi](https://github.com/suchipi)

When printing arrow functions, Prettier omitted parens around the arguments if they weren’t strictly necessary, like so:

<!-- prettier-ignore -->
```js
// no parens
foo => {};

// parens
(foo: Number) => {};

// parens
({ foo }) => {}

// parens
(foo = 5) => {}
```

This lead to the [most commented thread](https://github.com/prettier/prettier/issues/812) in our issue tracker. Prettier now has the `--arrow-parens` option (`arrowParens` in the config) that can assume two values today:

- `"avoid"` - (default) preserve the behavior that omits parens when possible
- `"always"` - always includes parens

#### JSX fragment syntax ([#3237](https://github.com/prettier/prettier/pull/3237)) by [@duailibe](https://github.com/duailibe)

Prettier will now recognize and format JSX with the [new fragment syntax](https://github.com/facebook/jsx/issues/84), like the code below:

<!-- prettier-ignore -->
```jsx
function MyComponent() {
  return (
    <>
      <Children1 />
      <Children2 />
      <Children3 />
    </>
  );
}
```

#### Fix slow formatting long texts in JSX ([#3268](https://github.com/prettier/prettier/pull/3268), [#3273](https://github.com/prettier/prettier/pull/3273)) by [@duailibe](https://github.com/duailibe)

We received feedback that formatting a JSX file with a really long text (~1000 lines) was really slow and noticed there was two performance bottlenecks in our `fill` primitive, which prints text until it reaches the print width and then insert a line break.

### Markdown

#### Add an option to preserve text line breaks ([#3340](https://github.com/prettier/prettier/pull/3340)) by [@ikatyang](https://github.com/ikatyang)

After the release of our Markdown support, we received feedback that breaking text to respect the print width could affect some renderers that could be sensitive to line breaks. In _1.8.2_ we released a new option `proseWrap: false` that would print a paragraph in a single line, and users would rely on the "soft wrapping" feature of editors.

In _1.9_ we are releasing a new option `proseWrap: "preserve"` which will respect the original line breaks of text, and lets the users decide where the text should break.

[WARNING] `proseWrap` with boolean value is deprecated, use `"always"`, `"never"` or `"preserve"` instead.

**[BREAKING CHANGE]** `proseWrap` option now defaults to `"preserve"` as some renderers are linebreak-sensitive.

### GraphQL

#### Support top-level interpolations ([#3370](https://github.com/prettier/prettier/pull/3370)) by [@lydell](https://github.com/lydell)

When GraphQL support was released, Prettier did not support interpolation so it would skip formatting if any interpolations were present, because interpolations make formatting very difficult. While that works well for the most part, users of the [Apollo Client](https://www.apollographql.com/) were missing out on Prettier’s GraphQL support sometimes, because Apollo Client uses interpolation to share fragments between queries. The good news is that only _top-level_ interpolations are allowed, and that was way easier to add support for in Prettier.

In _1.9_ we format GraphQL queries with top-level interpolation:

<!-- prettier-ignore -->
```js
gql`
  query User {
    user(id: "Bob") {
      ...UserDetails
    }
  }

  ${UserDetailsFragment}
`
```

(Prettier will continue to skip formatting if the interpolation is inside a query or mutation or so.)

#### Preserve intentional new lines in GraphQL ([#3252](https://github.com/prettier/prettier/pull/3252)) by [@duailibe](https://github.com/duailibe)

Prettier will now respect intentional line breaks inside GraphQL queries (but limit to 1), where before it would remove them.

<!-- prettier-ignore -->
```graphql
query User {
  name

  age
}
```

### CSS

#### Don't lowercase element names and attribute names in selectors ([#3317](https://github.com/prettier/prettier/pull/3317)) by [@lydell](https://github.com/lydell)

CSS is mostly case insensitive, so Prettier has been lowercasing stuff for a while to keep things consistent. Turns out we overlooked a detail in the CSS spec. Element and attribute names in selectors depend on the markup language: In HTML they are case insensitive, but in SVG (XML) they are not. Previously Prettier would incorrectly lowercase element and attribute names, but now we don’t anymore.

### Configuration

#### Add EditorConfig support ([#3255](https://github.com/prettier/prettier/pull/3255)) by [@josephfrazier](https://github.com/josephfrazier)

It's taken a while, but Prettier will finally respect your `.editorconfig` file. This includes:

- `indent_style`
- `indent_size`/`tab_width`
- `max_line_length`

The `prettier` CLI respects `.editorconfig` by default, but you can opt out with `--no-editorconfig`.
However, the API _doesn't_ respect `.editorconfig` by default, in order to avoid potential editor integration issues (see [here](https://github.com/prettier/prettier/pull/3255#discussion_r152259674) for details). To opt in, add `editorconfig: true` to the `prettier.resolveConfig` options.

## Other changes

### JavaScript

#### Don't break simple elements in JSX ([#3250](https://github.com/prettier/prettier/pull/3250)) by [@duailibe](https://github.com/duailibe)

Prettier won't break an element with no attributes anymore, keeping elements like `<br />` as an unit.

#### Don't break identifiers inside template literals expressions ([#3299](https://github.com/prettier/prettier/pull/3299)) by [@duailibe](https://github.com/duailibe)

In the previous release we tried a new strategy of breaking template literals with expressions inside to respect the print width. We've received feedback that for some cases it was actually preferred that it would exceed print width than breaking in multiple lines.

From now on, template literals expressions that contain a single identifier won't break anymore:

<!-- prettier-ignore -->
```js
const foo = `Hello ${username}. Today is ${month} ${day}. You have ${newMessages} new messages`.
```

#### Fix formatting of comment inside arrow function ([#3334](https://github.com/prettier/prettier/pull/3334)) by [@jackyho112](https://github.com/jackyho112)

Fixes an edge case where Prettier was moving comments around breaking tools like Webpack:

<!-- prettier-ignore -->
```js
const API = {
  loader: () => import('./test' /* webpackChunkName: "test" */),
};

// would get formatted to

const API = {
  loader: (/* webpackChunkName: "test" */) => import("./test")
};
```

#### Fix printing of comments around decorators and class properties ([#3382](https://github.com/prettier/prettier/pull/3382)) by [@azz](https://github.com/azz)

There was a case where comments between a decorator and a class property were moved to an invalid position.

<!-- prettier-ignore -->
```js
// Before
class Something {
  @decorator
  static // comment
  property = 1;
}

// After
class Something {
  @decorator
  // comment
  static property = 1;
}
```

### Flow

#### Do not break on empty type parameters ([#3281](https://github.com/prettier/prettier/pull/3281)) by [@Vjeux](https://github.com/Vjeux)

It won't break empty type parameters (`foo: Type<>`) anymore.

#### Add support for flow mixins when using babylon ([#3391](https://github.com/prettier/prettier/pull/3391)) by [@bakkot](https://github.com/bakkot)

We were accidentally dropping flow mixins, this has been fixed, but only for the `babylon` parser.

<!-- prettier-ignore -->
```js
// Before
class Foo extends Bar {}

// After
class Foo extends Bar mixins Baz {}
```

### TypeScript

#### Don't print a trailing comma after object rest spread ([#3313](https://github.com/prettier/prettier/pull/3313)) by [@duailibe](https://github.com/duailibe)

This was inconsistent with JavaScript and Flow, Prettier won't print a trailing comma in the following cases, when using the TypeScript parser:

<!-- prettier-ignore -->
```js
const {
  bar,
  baz,
  ...rest
} = foo;
```

#### Print parens around type assertions for decorators ([#3329](https://github.com/prettier/prettier/pull/3329)) by [@azz](https://github.com/azz)

We were omitting parens around type assertions inside decorators:

<!-- prettier-ignore -->
```typescript
@(bind as ClassDecorator)
class Decorated {}
```

### Markdown

#### Don't break `inlineCode` ([#3230](https://github.com/prettier/prettier/pull/3230)) by [@ikatyang](https://github.com/ikatyang)

Prettier won't break text inside `inlineCode` meaning it will only break of after it.

#### No extra whitespace between non-CJK and CJK-punctuation and vice-versa ([#3244](https://github.com/prettier/prettier/pull/3244), [#3249](https://github.com/prettier/prettier/pull/3249)) by [@ikatyang](https://github.com/ikatyang)

Fixes cases where Prettier would insert extra whitespace like in the following examples:

<!-- prettier-ignore -->
```markdown
扩展运算符（spread ）是三个点（`...`）。

This is an english paragraph with a CJK quote " 中文 ".
```

#### Escape all emphasis-like text ([#3246](https://github.com/prettier/prettier/pull/3246)) by [@ikatyang](https://github.com/ikatyang)

Fixes the case where `\_\_text\_\_` would be formatted as `__text__`.

#### Handle punctuation variants ([#3254](https://github.com/prettier/prettier/pull/3254)) by [@ikatyang](https://github.com/ikatyang)

Prettier now considers not only ASCII punctuation characters but Unicode as well.

#### Support TOML front matter ([#3290](https://github.com/prettier/prettier/pull/3290)) by [@ikatyang](https://github.com/ikatyang)

We already supported YAML in the front matter of Markdown files and we added the TOML format as well, since some static site generators support it.

<!-- prettier-ignore -->
```markdown
+++
date: '2017-10-10T22:49:47.369Z'
title: 'My Post Title'
categories: ['foo', 'bar']
+++

This is the markdown body of my post.
```

#### Only indent the first non-list node in checkbox list item ([#3297](https://github.com/prettier/prettier/pull/3297)) by [@ikatyang](https://github.com/ikatyang)

Fixes a bug where it would indent lines after a list when it shouldn't:

<!-- prettier-ignore -->
```markdown
* parent list item

     * child list item

* [x] parent task list item

     * [x] child task list item
```

would become:

<!-- prettier-ignore -->
```markdown
* parent list item

  * child list item

* [x] parent task list item

      * [x] child task list item
```

#### Preserve non-breaking whitespaces ([#3327](https://github.com/prettier/prettier/pull/3327)) by [@ikatyang](https://github.com/ikatyang)

Non-breaking whitespaces are useful to keep words separated by spaces together in the same line (i.e. number and units or multi-word product names). Prettier was wrongfully converting them to regular whitespaces.

#### Do not break before special prefix ([#3347](https://github.com/prettier/prettier/pull/3347)) by [@ikatyang](https://github.com/ikatyang)

Fixes a bug where Prettier could break text if it went over the print width right before a number followed by `.` which would be parsed as a numbered list:

<!-- prettier-ignore -->
```markdown
She grew up in an isolated village in the 19th century and met her father aged
29. Oh no, why are we in a numbered list now?
```

#### Omit semicolon in simple JSX expressions ([#3330](https://github.com/prettier/prettier/pull/3330)) by [@sapegin](https://github.com/sapegin)

Prettier will omit the semicolon (before and after) inside code samples if it's a simple JSX expression:

````markdown
No semi:

<!-- prettier-ignore -->
```jsx
<div>Example</div>
```
````
