---
author: "Georgii Dolzhykov (@thorn0)"
authorURL: "https://github.com/thorn0"
title: "Prettier 2.3. In which assignments are consistent, short keys non-breaking, and Handlebars official"
---

This release focuses on fixing long-standing issues in the JavaScript printer. Be warned that, unfortunately, reformatting a project with the new version might result in quite a big diff. If you don’t use [`ignoreRevsFile`](https://git-scm.com/docs/git-blame#Documentation/git-blame.txt---ignore-revs-fileltfilegt) to hide such wholesale changes from `git blame`, it might be about time.

A remarkable milestone is the long-awaited release of the Ember / Handlebars formatter. It’s supposed to be the last formatter included directly in the core library. In the future, for sustainability, languages should be added only by plugins.

We are grateful to our financial contributors: [Salesforce](https://engineering.salesforce.com/foss-fund-brings-more-than-just-financial-contributions-7b0664067b1e), [Indeed](https://engineering.indeedblog.com/blog/2019/07/foss-fund-six-months-in/), [Frontend Masters](https://blog.opencollective.com/frontend-masters/), Airbnb, Shogun Labs, Skyscanner, Konstantin Pschera, and [many others](https://opencollective.com/prettier#section-contributors) who help us keep going. If you enjoy Prettier and would like to support our work, head to our [OpenCollective](https://opencollective.com/prettier). Please consider also supporting the projects Prettier depends on, such as [typescript-eslint](https://opencollective.com/typescript-eslint), [remark](https://opencollective.com/unified), and [Babel](https://opencollective.com/babel).

Most of the changes in this release are thanks to the hard work of [Fisker Cheung](https://github.com/fisker), [Georgii Dolzhykov](https://github.com/thorn0), and [Sosuke Suzuki](https://github.com/sosukesuzuki), along with many other contributors.

And just a reminder, when Prettier is installed or updated, it’s [strongly recommended](https://prettier.io/docs/en/install.html#summary) to specify the exact version in `package.json`: `"2.3.0"`, not `"^2.3.0"`.

<!--truncate-->

## Highlights

### JavaScript

#### Format assignments more consistently ([#10222](https://github.com/prettier/prettier/pull/10222), [#10643](https://github.com/prettier/prettier/pull/10643), [#10672](https://github.com/prettier/prettier/pull/10672) by [@thorn0](https://github.com/thorn0); [#10158](https://github.com/prettier/prettier/pull/10158) by [@sosukesuzuki](https://github.com/sosukesuzuki))

Previously, Prettier had a lot of trouble with figuring out how to break lines in assignments. E.g., long right-hand sides often stayed unbroken. Not anymore.

<!-- prettier-ignore -->
```ts
// Prettier 2.2
aParticularlyLongAndObnoxiousNameForIllustrativePurposes = anotherVeryLongNameForIllustrativePurposes;

aParticularlyLongAndObnoxiousNameForIllustrativePurposes = "a very long string for illustrative purposes"
  .length;

someReallyLongThingStoredInAMapWithAReallyBigName[
  pageletID
] = _someVariableThatWeAreCheckingForFalsiness
  ? Date.now() - _someVariableThatWeAreCheckingForFalsiness
  : 0;

class x {
  private readonly rawConfigFromFile$: BehaviorSubject<any> = new BehaviorSubject(
    notRead
  );
}

// Prettier 2.3
aParticularlyLongAndObnoxiousNameForIllustrativePurposes =
  anotherVeryLongNameForIllustrativePurposes;

aParticularlyLongAndObnoxiousNameForIllustrativePurposes =
  "a very long string for illustrative purposes".length;

someReallyLongThingStoredInAMapWithAReallyBigName[pageletID] =
  _someVariableThatWeAreCheckingForFalsiness
    ? Date.now() - _someVariableThatWeAreCheckingForFalsiness
    : 0;

class x {
  private readonly rawConfigFromFile$: BehaviorSubject<any> =
    new BehaviorSubject(notRead);
}
```

#### Prevent wrapping object properties with short keys ([#10335](https://github.com/prettier/prettier/pull/10335) by [@thorn0](https://github.com/thorn0))

Line breaks after short property names in object literals often look unnatural. Even when such a line break yields a line length benefit of 1 or 2 characters, it rarely looks justified. Prettier 2.3 avoids line breaks after property names shorter than `tabWidth + 3` – for example, 5 characters in the default configuration, or 7 characters with `tabWidth: 4`. This heuristic may be revised in future versions.

<!-- prettier-ignore -->
```js
// Prettier 2.2
const importantLink = {
  url:
    "https://prettier.io/docs/en/rationale.html#what-prettier-is-concerned-about",
  gitHubUrl:
    "https://github.com/prettier/prettier/blob/main/docs/rationale.md#what-prettier-is-concerned-about",
};

// Prettier 2.3
const importantLink = {
  url: "https://prettier.io/docs/en/rationale.html#what-prettier-is-concerned-about",
  gitHubUrl:
    "https://github.com/prettier/prettier/blob/main/docs/rationale.md#what-prettier-is-concerned-about",
};
```

### Ember / Handlebars

#### Move Handlebars support out of alpha to release ([#10290](https://github.com/prettier/prettier/pull/10290) by [@dcyriller](https://github.com/dcyriller) & [@thorn0](https://github.com/thorn0))

This started in 2017. Handlebars support has been in Prettier for a while, but it wasn’t released officially as it wasn’t really ready. Its status went from “alpha” to “experimental” to “beta” and then, if you checked older release notes, you can see that after “beta” somehow it was “alpha” again…

Well, anyway, it finally is happening: Prettier can now officially format HTML templates with Handlebars! 🎉

It uses [Glimmer], [Ember]’s Handlebars parser, so it should be compliant with the HTML spec thanks to the Ember team.

The [`--html-whitespace-sensitivity`][hws-option] option is supported and defaults to `strict`, which means that Prettier will always respect the presence or absence of whitespace around tags and consider it unsafe to add whitespace where there were none and vice versa as this can affect how the document is rendered in the browser. The `css` value is not yet supported (treated as `strict` for now).

The feature is called “Ember / Handlebars” and not just “Handlebars” because Glimmer doesn’t support some syntax and use cases of Handlebars. This is mostly due to the fact that [Handlebars], being a template engine (a preprocessor), doesn’t care about the underlying syntax of the content it processes whereas Glimmer parses two syntaxes – HTML and Handlebars – at the same time and combines the result into a single tree, which Prettier can print. This means Prettier won’t format Handlebars files that can’t be parsed into such a tree, either because the underlying syntax isn’t HTML or because template directives and tags overlap in a way that can’t be represented in a tree (e.g., `{{#if foo}}<div>{{/if}`). Even with these restrictions, the formatter still seems to be useful enough to non-Ember Handlebars users. As for the syntax unsupported by Ember, there is a good chance to see support for it in future versions of Prettier as Glimmer uses a full-fledged Handlebars parser under the hood.

Files with the extensions `.hbs` and `.handlebars` are recognized as Handlebars by default. For other extensions, the [`--parser`](parser-option) option with the value `glimmer` has to be specified – for example, using command line or, better yet, [configuration overrides].

See the formatter in action on the [playground][playground-glimmer]!

[glimmer]: https://www.npmjs.com/package/@glimmer/syntax
[ember]: https://emberjs.com
[hws-option]: https://prettier.io/docs/en/options.html#html-whitespace-sensitivity
[handlebars]: https://handlebarsjs.com/
[parser-option]: https://prettier.io/docs/en/options.html#parser
[configuration overrides]: https://prettier.io/docs/en/configuration.html#configuration-overrides
[playground-glimmer]: https://prettier.io/playground/#N4Igxg9gdgLgprEAuEACVAeAJgSwG7qFgA2AhgM7kC8AOiAjAE4CedhqAfDVOhgBYBGDsGDoYOGMTjsAvjIwB6QVx6ZcBdCQrU6AIwhZWIFe1OERhfYbOE53Reo6YFjkABoQEAA7jo5ZKCkjIwQAO4ACkEI-iikxKGkzP4euoykYADWcDAAyl7pOFAA5shMAK5wHnAAtrpwWFj1ADKkxWWkRXAAYhCM1aQw4sXIIKRlMBDuIHww1cQA6nwScOT5YHA50RL4Eswj5Ew4YDBTheRwjDDhaUX9yABmceceAFbkAB4AQmmZ2Tmk1TgTUKcAeT0qIDe7xyhSKUgAimUIPAwcRniB8oxzowRnCcNVATiPF5GIUYPMcFgYHxkAAOAAMxJC53maS8IxJKwueFBHgAjkj4NdvDFRuQALRQOD1epTRhwAU4eXXDp3JCPNEQ87VHClRgVDzkWEIwWg9XgjwwUi6ClUmlIABMlrSOGIsIAwhACaQRisAKxTMrnAAq1piGvReAqAEkoI1YDkwKSfABBOM5GDMKSo85yIA

## Formatting Improvements

### JavaScript

#### Refine formatting of curried arrow functions ([#9992](https://github.com/prettier/prettier/pull/9992), [#10543](https://github.com/prettier/prettier/pull/10543) by [@sosukesuzuki](https://github.com/sosukesuzuki) & [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```js
// Prettier 2.2
const currying = (argument1) => (argument2) => (argument3) => (argument4) => (
  argument5
) => (argument6) => foo;

// Prettier 2.3
const currying =
  (argument1) =>
  (argument2) =>
  (argument3) =>
  (argument4) =>
  (argument5) =>
  (argument6) =>
    foo;
```

#### Improve formatting for React Hooks calls ([#10238](https://github.com/prettier/prettier/pull/10238) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```js
// Prettier 2.2
const { firstName, lastName } = useMemo(() => parseFullName(fullName), [
  fullName,
]);

// Prettier 2.3
const { firstName, lastName } = useMemo(
  () => parseFullName(fullName),
  [fullName]
);
```

#### Improve visual separation between header and body in classes with multiline headers ([#10085](https://github.com/prettier/prettier/pull/10085) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```ts
// Prettier 2.2
class loooooooooooooooooooong
  extends looooooooooooooooooong
  implements loooooooooooooooooooong {
  property: string;
}

// Prettier 2.3
class loooooooooooooooooooong
  extends looooooooooooooooooong
  implements loooooooooooooooooooong
{
  property: string;
}
```

#### Concise formatting of number-only arrays ([#10106](https://github.com/prettier/prettier/pull/10106), [#10160](https://github.com/prettier/prettier/pull/10160) by [@thorn0](https://github.com/thorn0))

While in general, Prettier avoids this kind of formatting because it's not diff-friendly, in this special case we decided that the benefits outweigh the risks.

If at least one element has a trailing single-line (`// ...`) comment on the same line, the concise formatting isn't applied. On the other hand, single-line comments placed on separate lines don't have such an effect and – as well as empty lines – can be used for logical grouping.

<!-- prettier-ignore -->
```jsx
// Input
const lazyCatererNumbers  =  [1, 2, 4, 7, 11, 16, 22, 29, 37, 46,

// n > 10
56, 67, 79, 92, 106, 121, 137, 154, 172, 191, 211, 232, 254, 277, 301, 326, 352, 379, 407, 436, 466,
497, 529, 562, 596, 631, 667, 704, 742, 781,
// n > 40
821, 862, 904, 947, 991, 1036, 1082, 1129, 1177, 1226, 1276, 1327, 1379];

// Prettier 2.2
const lazyCatererNumbers = [
  1,
  2,
  4,
  7,
  11,
  16,
  22,
  29,
  37,
  // ... ✂ 46 lines ✂ ...
  1379,
];

// Prettier 2.3
const lazyCatererNumbers = [
  1, 2, 4, 7, 11, 16, 22, 29, 37, 46,

  // n > 10
  56, 67, 79, 92, 106, 121, 137, 154, 172, 191, 211, 232, 254, 277, 301, 326,
  352, 379, 407, 436, 466, 497, 529, 562, 596, 631, 667, 704, 742, 781,
  // n > 40
  821, 862, 904, 947, 991, 1036, 1082, 1129, 1177, 1226, 1276, 1327, 1379,
];
```

#### Improve formatting for nested await expressions in heads of member and call expressions ([#10342](https://github.com/prettier/prettier/pull/10342) by [@thorn0](https://github.com/thorn0))

Even though Prettier tries to be helpful here, please don't write code like this. Have mercy upon your teammates and use intermediate variables.

<!-- prettier-ignore -->
```jsx
// Input
const getAccountCount = async () =>
  (await
    (await (
      await focusOnSection(BOOKMARKED_PROJECTS_SECTION_NAME)
    ).findItem("My bookmarks")).getChildren()
  ).length

// Prettier 2.2
const getAccountCount = async () =>
  (
    await (
      await (await focusOnSection(BOOKMARKED_PROJECTS_SECTION_NAME)).findItem(
        "My bookmarks"
      )
    ).getChildren()
  ).length;

// Prettier 2.3
const getAccountCount = async () =>
  (
    await (
      await (
        await focusOnSection(BOOKMARKED_PROJECTS_SECTION_NAME)
      ).findItem("My bookmarks")
    ).getChildren()
  ).length;
```

#### Improve formatting for `do` expressions in function calls ([#10693](https://github.com/prettier/prettier/pull/10693) by [@sosukesuzuki](https://github.com/sosukesuzuki))

“`do` expressions” are a [stage 1 ECMAScript proposal](https://github.com/tc39/proposal-do-expressions).

<!-- prettier-ignore -->
```js
// Prettier 2.2
expect(
  do {
    var bar = "foo";
    bar;
  }
).toBe("foo");

// Prettier 2.3
expect(do {
  var bar = "foo";
  bar;
}).toBe("foo");
```

#### Consistent indentation for conditional operators ([#10187](https://github.com/prettier/prettier/pull/10187), [#10266](https://github.com/prettier/prettier/pull/10266) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```ts
// Prettier 2.2
const dotNotationMemberExpression = (callNode.parent?.type ===
AST_NODE_TYPES.ChainExpression
  ? callNode.parent.parent
  : callNode.parent
).TSESTree.BinaryExpression;

const computedMemberExpression = (callNode.parent?.type ===
AST_NODE_TYPES.ChainExpression
  ? callNode.parent.parent
  : callNode.parent)[TSESTree.BinaryExpression];

const callExpressionCallee = (callNode.parent?.type ===
  AST_NODE_TYPES.ChainExpression
  ? callNode.parent.parent
  : callNode.parent)(TSESTree.BinaryExpression);

const typeScriptAsExpression = (callNode.parent?.type ===
AST_NODE_TYPES.ChainExpression
  ? callNode.parent.parent
  : callNode.parent) as TSESTree.BinaryExpression;

// Prettier 2.3
const dotNotationMemberExpression = (
  callNode.parent?.type === AST_NODE_TYPES.ChainExpression
    ? callNode.parent.parent
    : callNode.parent
).TSESTree.BinaryExpression;

const computedMemberExpression = (
  callNode.parent?.type === AST_NODE_TYPES.ChainExpression
    ? callNode.parent.parent
    : callNode.parent
)[TSESTree.BinaryExpression];

const callExpressionCallee = (
  callNode.parent?.type === AST_NODE_TYPES.ChainExpression
    ? callNode.parent.parent
    : callNode.parent
)(TSESTree.BinaryExpression);

const typeScriptAsExpression = (
  callNode.parent?.type === AST_NODE_TYPES.ChainExpression
    ? callNode.parent.parent
    : callNode.parent
) as TSESTree.BinaryExpression;
```

### HTML

#### Prefix-based multiline formatting for the `class` attribute ([#7865](https://github.com/prettier/prettier/pull/7865) by [@thorn0](https://github.com/thorn0))

Formatting of HTML class names will now keep classes on one line until the line length limit is reached; at that point, consecutive classes with the same prefix will be grouped together on each line. For layout frameworks such as Bootstrap and Tailwind CSS, which add many classes to an element, this is important for readability and maintainability vs. the previous behavior (keeping all classes on one line) or e.g. putting each class on its own line.

<!-- prettier-ignore -->
```html
<!-- Prettier 2.2 -->
<div
  class="SomeComponent__heading-row d-flex flex-column flex-lg-row justify-content-start justify-content-lg-between align-items-start align-items-lg-center"
></div>

<!-- Prettier 2.3 -->
<div
  class="
    SomeComponent__heading-row
    d-flex
    flex-column flex-lg-row
    justify-content-start justify-content-lg-between
    align-items-start align-items-lg-center
  "
></div>
```

## Other Changes

### JavaScript

#### Fix unstable multiple comments on the same line ([#9672](https://github.com/prettier/prettier/pull/9672) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```js
// Input
a;
/*1*//*2*/
/*3*/
b;

// Prettier 2.2
a; /*2*/
/*1*/ /*3*/
b;

// Prettier 2.2 (second format)
a; /*2*/ /*3*/
/*1*/ b;

// Prettier 2.3
a;
/*1*/ /*2*/
/*3*/
b;
```

#### Don’t format nodes that end just before `rangeStart` ([#9704](https://github.com/prettier/prettier/pull/9704) by [@fisker](https://github.com/fisker))

Previously, when [range formatting](https://prettier.io/docs/en/options.html#range) was performed, such nodes were considered part of the range, now they're excluded. This affects other languages that the range formatting feature supports, not only JavaScript.

<!-- prettier-ignore -->
```js
// Input
foo = 1.0000;bar = 1.0000;baz=1.0000;
             ^^^^^^^^^^^^^ Range

// Prettier 2.2
foo = 1.0;
bar = 1.0;baz=1.0000;

// Prettier 2.3
foo = 1.0000;bar = 1.0;baz=1.0000;
```

#### Fix comments inside JSX end tag ([#9711](https://github.com/prettier/prettier/pull/9711) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```jsx
// Input
<a><// comment
/a>;

// Prettier 2.2
<a></// comment
a>;

// Prettier 2.3
<a></
  // comment
  a
>;
```

#### Fix inconsistent language comment detection ([#9743](https://github.com/prettier/prettier/pull/9743) by [@fisker](https://github.com/fisker))

An `/* HTML */` comment should directly precede a template literal for the latter to be recognized as HTML-in-JS. Previously, the comment was erroneously recognized is some other locations.

<!-- prettier-ignore -->
```jsx
// Input
foo /* HTML */ = `<DIV>
</DIV>`;

// Prettier 2.2 (--parser=babel)
foo /* HTML */ = `<div></div>`;

// Prettier 2.2 (--parser=meriyah)
foo /* HTML */ = `<DIV>
</DIV>`;

// Prettier 2.3 (All JavaScript parsers)
foo /* HTML */ = `<DIV>
</DIV>`;
```

#### Fix extra semicolon added to ignored directives ([#9850](https://github.com/prettier/prettier/pull/9850) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```js
// Input
// prettier-ignore
'use strict';

function foo() {
// prettier-ignore
"use strict";;
}

// Prettier 2.2
// prettier-ignore
'use strict';;

function foo() {
  // prettier-ignore
  "use strict";;
}

// Prettier 2.3
// prettier-ignore
'use strict';

function foo() {
  // prettier-ignore
  "use strict";
}
```

#### Fix unstable JSX formatting with U+3000 ([#9866](https://github.com/prettier/prettier/pull/9866) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```js
// Input
<p>
  <span />　{this.props.data.title}　<span />
  //----- ^ U+3000 --------------- ^ U+3000
</p>

// Prettier 2.2
<p>
  <span />
  　{this.props.data.title}　<span />
  //----- ^ U+3000 --------------- ^ U+3000
</p>;

// Prettier 2.2 (second format)
<p>
  <span />　{this.props.data.title}　<span />
  //----- ^ U+3000 --------------- ^ U+3000
</p>;

// Prettier 2.3
<p>
  <span />　{this.props.data.title}　<span />
  //----- ^ U+3000 --------------- ^ U+3000
</p>;
```

#### Fix error thrown on expressions like `a(b => c => function (){})` ([#10278](https://github.com/prettier/prettier/pull/10278) by [@thorn0](https://github.com/thorn0))

Regression from v2.2.0.

<!-- prettier-ignore -->
```jsx
// Input
a(b => c => function (){})

// Prettier 2.2
TypeError: Cannot read property 'length' of undefined

// Prettier 2.3
a((b) => (c) => function () {});
```

#### Improve formatting for inline decorators ([#10296](https://github.com/prettier/prettier/pull/10296) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```jsx
// Input
class Foo {
  @decorator([]) bar() {}
  @decorator(
    []
  ) baz() {}
}

// Prettier 2.2
class Foo {
  @decorator([]) bar() {}
  @decorator([])
  baz() {}
}

// Prettier 2.3
class Foo {
  @decorator([]) bar() {}
  @decorator([]) baz() {}
}
```

#### Fix ASI protection for private fields ([#10334](https://github.com/prettier/prettier/pull/10334) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```jsx
// Input
class C {
  #field = 'value';
  ["method"]() {}
}

// Prettier 2.2 (with --no-semi)
class C {
  #field = "value"
  ["method"]() {}
}

// Prettier 2.3 (with --no-semi)
class C {
  #field = "value";
  ["method"]() {}
}
```

#### Support Module Blocks proposal ([#10417](https://github.com/prettier/prettier/pull/10417) by [@sosukesuzuki](https://github.com/sosukesuzuki), [@thorn0](https://github.com/thorn0))

Support formatting for [Module Blocks Stage 2 proposal](https://github.com/tc39/proposal-js-module-blocks).

<!-- prettier-ignore -->
```js
// Input
module { export let foo = "foo"; };

// Prettier 2.2
SyntaxError: Unexpected token, expected ";"

// Prettier 2.3
module {
  export let foo = "foo";
};
```

#### Fix missing parentheses for `yield` in a pipeline ([#10446](https://github.com/prettier/prettier/pull/10446) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```jsx
// Input
function* f() {
  return x |> (yield #);
}

// Prettier 2.2
function* f() {
  return x |> yield #;
}

// Prettier 2.3
function* f() {
  return x |> (yield #);
}
```

#### Make Babel’s error recovery more selective ([#10495](https://github.com/prettier/prettier/pull/10495) by [@fisker](https://github.com/fisker), [#9787](https://github.com/prettier/prettier/pull/9787) by [@sosukesuzuki](https://github.com/sosukesuzuki), [#10065](https://github.com/prettier/prettier/pull/10065), [#10322](https://github.com/prettier/prettier/pull/10322) by [@thorn0](https://github.com/thorn0))

Previously, because of the [error recovery](https://babeljs.io/blog/2019/11/05/7.7.0#parser-error-recovery-10363httpsgithubcombabelbabelpull10363), the Babel parser was too permissive, which lead to all kinds of AST shapes that Prettier couldn’t print. Prettier 2.3 lets Babel recover only from a few harmless types of errors – for example, multiple `const` declarations with the same name. Anything else is reported as syntax errors.

<!-- prettier-ignore -->
```js
// Input
foo("a", , "b");

// Prettier 2.2
TypeError: Cannot read property 'type' of null

// Prettier 2.3
[error] stdin: SyntaxError: Argument expression expected. (1:10)
[error] > 1 | foo("a", , "b");
[error]     |          ^
```

<!-- prettier-ignore -->
```js
// Input
const \u{20} = 1

// Prettier 2.2
const   = 1;

// Prettier 2.3
[error] stdin: SyntaxError: Invalid Unicode escape (1:7)
[error] > 1 | const \u{20} = 1
[error]     |       ^         ^
```

#### Avoid last-argument hugging for number-only arrays ([#10517](https://github.com/prettier/prettier/pull/10517) by [@thorn0](https://github.com/thorn0))

Another special case for number-only arrays.

<!-- prettier-ignore -->
```jsx
// Input
instantiate(game, [
  transform([-0.7, 0.5, 0]),
  render_colored_diffuse(game.MaterialDiffuse, game.Meshes["monkey_flat"], [1, 1, 0.3, 1]),
]);

// Prettier 2.2
instantiate(game, [
  transform([-0.7, 0.5, 0]),
  render_colored_diffuse(game.MaterialDiffuse, game.Meshes["monkey_flat"], [
    1,
    1,
    0.3,
    1,
  ]),
]);

// Prettier 2.3
instantiate(game, [
  transform([-0.7, 0.5, 0]),
  render_colored_diffuse(
    game.MaterialDiffuse,
    game.Meshes["monkey_flat"],
    [1, 1, 0.3, 1]
  ),
]);
```

#### Improve detection of AMD `define` ([#10528](https://github.com/prettier/prettier/pull/10528) by [@thorn0](https://github.com/thorn0))

Prettier special-cases AMD `define` calls to avoid unexpected line breaks in them. We now only format `define` calls if they are at the top level of a function or program and are passed arguments in the way AMD expects.

<!-- prettier-ignore -->
```jsx
// Prettier 2.2
const someVariable = define("some string literal", anotherVariable, yetAnotherVariable);

// Prettier 2.3
const someVariable = define(
  "some string literal",
  anotherVariable,
  yetAnotherVariable
);
```

#### Fix duplicated `prettier-ignore` comments ([#10666](https://github.com/prettier/prettier/pull/10666) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```jsx
// Input
foo = a.
  // prettier-ignore
  b;

// Prettier 2.2
foo =
  // prettier-ignore
  a.
  // prettier-ignore
  b;

// Prettier 2.3
foo = a.
  // prettier-ignore
  b;
```

#### Process conditional groups in `mapDoc` ([#10695](https://github.com/prettier/prettier/pull/10695) by [@thorn0](https://github.com/thorn0))

In particular, this fixes broken substitutions in HTML-in-JS.

<!-- prettier-ignore -->
```jsx
// Input
export default function include_photoswipe(
	gallery_selector = ".my-gallery"
): string {
	return /* HTML */ `
		<script>
			window.addEventListener("load", () =>
				initPhotoSwipeFromDOM("${gallery_selector}")
			);
		</script>`;
}

// Prettier 2.2
export default function include_photoswipe(
  gallery_selector = ".my-gallery"
): string {
  return /* HTML */ ` <script>
    window.addEventListener("load", () =>
      initPhotoSwipeFromDOM("PRETTIER_HTML_PLACEHOLDER_0_13_IN_JS")
    );
  </script>`;
}

// Prettier 2.3
export default function include_photoswipe(
  gallery_selector = ".my-gallery"
): string {
  return /* HTML */ ` <script>
    window.addEventListener("load", () =>
      initPhotoSwipeFromDOM("${gallery_selector}")
    );
  </script>`;
}
```

#### Avoid arg expansion that leads to broken code ([#10712](https://github.com/prettier/prettier/pull/10712) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```jsx
// Input
glimseGlyphsHazardNoopsTieTie(({ a, b = () => {
  console.log();
}}) => averredBathersBoxroomBuggyNurl());

// Prettier 2.2
glimseGlyphsHazardNoopsTieTie(({ a, b = () => {
    console.log();
  } }) => averredBathersBoxroomBuggyNurl());

// Prettier 2.3
glimseGlyphsHazardNoopsTieTie(
  ({
    a,
    b = () => {
      console.log();
    },
  }) => averredBathersBoxroomBuggyNurl()
);
```

#### Fix missing parentheses around `async` in `for-of` ([#10781](https://github.com/prettier/prettier/pull/10781) by [@fisker](https://github.com/fisker))

See https://github.com/tc39/ecma262/issues/2034

<!-- prettier-ignore -->
```jsx
// Input
for ((async) of []);

// Prettier 2.2
for (async of []);

// Prettier 2.2 (second format)
SyntaxError: Unexpected token, expected "=>" (1:15)
> 1 | for (async of []);

// Prettier 2.3
for ((async) of []);
```

#### Support `async do` expressions proposal ([#10813](https://github.com/prettier/prettier/pull/10813) by [@sosukesuzuki](https://github.com/sosukesuzuki))

See https://github.com/tc39/proposal-async-do-expressions

<!-- prettier-ignore -->
```js
// Input
const x = async do {
  await requestAPI().json();
};

// Prettier 2.2
SyntaxError: Unexpected token, expected ";" (1:17)

// Prettier 2.3
const x = async do {
  await requestAPI().json();
};
```

### TypeScript

#### Fix missing comments in `MethodDefinition` ([#9872](https://github.com/prettier/prettier/pull/9872) by [@fisker](https://github.com/fisker))

`typescript` parser only, `babel-ts` doesn't have this issue.

<!-- prettier-ignore -->
```typescript
// Input
class Foo {
  bar() /* bat */;
}

// Prettier 2.2
Error: Comment "bat" was not printed. Please report this error!

// Prettier 2.3
class Foo {
  bar /* bat */();
}
```

#### Fix unnecessary line breaks in method type declaration parameters ([#10024](https://github.com/prettier/prettier/pull/10024) by [@sosukesuzuki](https://github.com/sosukesuzuki), [#10357](https://github.com/prettier/prettier/pull/10357) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```js
// Input
type Foo = {
  method(foo: "foo"): `
  `
};

// Prettier 2.2
type Foo = {
  method(
    foo: "foo"
  ): `
  `;
};

// Prettier 2.3
type Foo = {
  method(foo: "foo"): `
  `;
};

```

#### Print trailing commas in type parameters ([#10109](https://github.com/prettier/prettier/pull/10109) by [@sosukesuzuki](https://github.com/sosukesuzuki), [#10353](https://github.com/prettier/prettier/pull/10353) by [@thorn0](https://github.com/thorn0))

TypeScript has been supporting trailing commas in type parameters since TypeScript 2.7 released in January 2018. Prettier 2.3 prints them if the `trailingComma` option is set to `all`. Keep this option at the more conservative default value `es5` if compatibility with TypeScript 2.7 or older is needed. Note that TypeScript [still doesn't support][ts-issue-21984] trailing commas in type arguments (type parameter instantiations).

[ts-issue-21984]: https://github.com/microsoft/TypeScript/issues/21984

<!-- prettier-ignore -->
```ts
// Input
export class BaseSingleLevelProfileTargeting<
	T extends ValidSingleLevelProfileNode,
> {
  // ...
}

// Prettier 2.2
export class BaseSingleLevelProfileTargeting<
  T extends ValidSingleLevelProfileNode
> {
  // ...
}

// Prettier 2.3 with --trailling-comma=all
export class BaseSingleLevelProfileTargeting<
  T extends ValidSingleLevelProfileNode,
> {
  // ...
}
```

#### Allow hugging arguments that are non-concise arrow functions with return type annotations ([#10316](https://github.com/prettier/prettier/pull/10316) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```ts
// Prettier 2.2
users.map(
  (user: User): User => {
    return user;
  }
);

// Prettier 2.3
users.map((user: User): User => {
  return user;
})
```

#### Correct parentheses for non-null assertions ([#10337](https://github.com/prettier/prettier/pull/10337) by [@thorn0](https://github.com/thorn0))

Necessary parentheses sometimes weren't printed in expressions containing non-null assertions. This has been fixed.

<!-- prettier-ignore -->
```ts
// Input
const myFunction2 = (key: string): number =>
  ({
    a: 42,
    b: 42,
  }[key]!)

// Prettier 2.2 (invalid syntax)
const myFunction2 = (key: string): number =>
  {
    a: 42,
    b: 42,
  }[key]!;

// Prettier 2.3
const myFunction2 = (key: string): number =>
  ({
    a: 42,
    b: 42,
  }[key]!);
```

#### Indent type assertions in heads of member and call expressions ([#10341](https://github.com/prettier/prettier/pull/10341) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```ts
// Input
const accountCount = (findItemInSection(BOOKMARKED_PROJECTS_SECTION_NAME,
  "My bookmarks") as TreeItem).getChildren().length;

// Prettier 2.2
const accountCount = (findItemInSection(
  BOOKMARKED_PROJECTS_SECTION_NAME,
  "My bookmarks"
) as TreeItem).getChildren().length;

// Prettier 2.3
const accountCount = (
  findItemInSection(
    BOOKMARKED_PROJECTS_SECTION_NAME,
    "My bookmarks"
  ) as TreeItem
).getChildren().length;
```

#### Support intrinsic keyword ([#10390](https://github.com/prettier/prettier/pull/10390) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```ts
// Input
type Uppercase<S extends string> = intrinsic;

// Prettier 2.2
Error: unknown type: "TSIntrinsicKeyword"

// Prettier 2.3
type Uppercase<S extends string> = intrinsic;

```

#### Support TypeScript 4.2 ([#10418](https://github.com/prettier/prettier/pull/10418), [#10466](https://github.com/prettier/prettier/pull/10466), [#10546](https://github.com/prettier/prettier/pull/10546), [#10589](https://github.com/prettier/prettier/pull/10589) by [@sosukesuzuki](https://github.com/sosukesuzuki))

##### [`abstract` Construct Signatures](https://devblogs.microsoft.com/typescript/announcing-typescript-4-2/#abstract-construct-signatures)

<!-- prettier-ignore -->
```ts
// Input
type T = abstract new () => void;

// Prettier 2.2
SyntaxError: Unexpected token, expected ";" (1:19)

// Prettier 2.3
type T = abstract new () => void;

```

##### [Type imports in import require declaration](https://github.com/microsoft/TypeScript/pull/41573)

<!-- prettier-ignore -->
```ts
// Input
import type A = require("A");

// Prettier 2.2
SyntaxError: Only ECMAScript imports may use 'import type'.

// Prettier 2.3
import type A = require("A");

```

#### Fix misplaced comments in unions and intersections ([#10457](https://github.com/prettier/prettier/pull/10457) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```ts
// Input
type Foo = "1" | "2" /* two */ | "3";

// Prettier 2.2
type Foo = "1" | "2" | /* two */ "3";

// Prettier 2.3
type Foo = "1" | "2" /* two */ | "3";
```

#### Don’t print parens around nested type assertions ([#10702](https://github.com/prettier/prettier/pull/10702) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```ts
// Input
foo as unknown as Bar

// Prettier 2.2
(foo as unknown) as Bar;

// Prettier 2.3
foo as unknown as Bar;
```

#### Support TypeScript 4.3 via `babel-ts` ([#10811](https://github.com/prettier/prettier/pull/10811) by [@sosukesuzuki](https://github.com/sosukesuzuki))

##### [`override` modifiers in class elements](https://devblogs.microsoft.com/typescript/announcing-typescript-4-3-beta/#override-and-the-noimplicitoverride-flag)

```ts
class Foo extends  {
  override method() {}
}
```

##### [static index signatures (`[key: KeyType]: ValueType`) in classes](https://devblogs.microsoft.com/typescript/announcing-typescript-4-3-beta/#static-index-signatures)

```ts
class Foo {
  static [key: string]: Bar;
}
```

##### [`get` / `set` in type declarations](https://devblogs.microsoft.com/typescript/announcing-typescript-4-3-beta/#separate-write-types-on-properties)

```ts
interface Foo {
  set foo(value);
  get foo(): string;
}
```

### Flow

#### Fix missing semicolon in `declare export * from …` ([#9767](https://github.com/prettier/prettier/pull/9767) by [@fisker](https://github.com/fisker))

<!-- Optional description if it makes sense. -->

<!-- prettier-ignore -->
```jsx
// Input
declare export * from "ES6_ExportAllFrom_Source2";

// Prettier 2.2
declare export * from "ES6_ExportAllFrom_Source2"

// Prettier 2.3
declare export * from "ES6_ExportAllFrom_Source2";
```

#### Support `this` type annotation in functions via `babel-flow` ([#10397](https://github.com/prettier/prettier/pull/10397) by [@sosukesuzuki](https://github.com/sosukesuzuki))

`this` type annotation is supported since [Babel 7.13](https://babeljs.io/blog/2021/02/22/7.13.0#new-flow-features-12193httpsgithubcombabelbabelpull12193-12234httpsgithubcombabelbabelpull12234).

```js
// Input
var foo: (this: boolean) => void;

// Prettier 2.2
SyntaxError: Unexpected token, expected ")" (1:15)

// Prettier 2.3
var foo: (this: boolean) => void;

```

#### Fix range formatting issues ([#10505](https://github.com/prettier/prettier/pull/10505) by [@thorn0](https://github.com/thorn0))

Prettier had trouble formatting some ranges in function declarations. `SyntaxError` was thrown. Prettier 2.3 formats these cases without errors. Examples of problematic ranges are shown below:

<!-- prettier-ignore -->
```jsx
declare export function graphql<Props, Variables, Component: React$ComponentType<Props>>
  //                            ^^^^^ range 1
  (query: GQLDocument, config?: Config<Props, QueryConfigOptions<Variables>>):
  (Component: Component) => React$ComponentType<$Diff<React$ElementConfig<Component>, {
    data: Object|void,
    //           ^^^^ range 2
    mutate: Function|void
  }>>
```

#### Support for Flow's indexed access type ([#10594](https://github.com/prettier/prettier/pull/10594) by [@gkz](https://github.com/gkz))

<!-- prettier-ignore -->
```js
// Input
const x: Obj['foo'] = 1;

// Prettier 2.2
// Error: unsupported node type "IndexedAccessType"

// Prettier 2.3
const x: Obj["foo"] = 1;
```

#### Support for Flow's optional indexed access type ([#10788](https://github.com/prettier/prettier/pull/10788) by [@gkz](https://github.com/gkz))

<!-- prettier-ignore -->
```js
// Input
type T = Obj?.['foo'];

// Prettier 2.2
// Error: unsupported node type "OptionalIndexedAccessType"

// Prettier 2.3
type T = Obj?.['foo'];
```

### JSON

#### Don't use smart quotes for JSON5 with `--quote-props=preserve` ([#10323](https://github.com/prettier/prettier/pull/10323) by [@thorn0](https://github.com/thorn0))

With the `quoteProps` option set to `preserve` and `singleQuotes` to `false` (default), double quotes are always used for printing strings, including situations like `"bla\"bla"`. This effectively allows using `--parser json5` for "JSON with comments and trailing commas".

<!-- prettier-ignore -->
```js
// Input
{
  "char": "\"",
}

// Prettier 2.2
{
  "char": '"',
}

// Prettier 2.3
{
  "char": "\"",
}
```

#### Stricter JSON parser ([#10346](https://github.com/prettier/prettier/pull/10346), [#10443](https://github.com/prettier/prettier/pull/10443), [#10456](https://github.com/prettier/prettier/pull/10456), [#10434](https://github.com/prettier/prettier/pull/10434) by [@fisker](https://github.com/fisker))

Prettier internally uses a JavaScript expression parser to parse JSON. That's why the `json` and `json5` parsers used to be very forgiving and allowed all kinds of JavaScript expressions. Now they are much stricter, although some simple non-standard syntax is still allowed (e.g., [JSON6](https://github.com/d3x0r/JSON6) is supported, except for multiple minus signs: `----123`).

<!-- prettier-ignore -->
```js
// Input
[1, 2, 1 + 2]

// Prettier 2.2
[1, 2, 1 + 2]

// Prettier 2.3
SyntaxError: BinaryExpression is not allowed in JSON. (1:8)
> 1 | [1, 2, 1 + 2]
    |        ^
```

#### Improve error message ([#10433](https://github.com/prettier/prettier/pull/10433) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```js
// Input
{key: "foo" + "bar"}

// Prettier 2.2 (--parser=json-stringify)
SyntaxError: BinaryExpression is not allowed in JSON. (1:7)
> 1 | {key: "foo" + "bar"}
    |       ^

// Prettier 2.3
SyntaxError: BinaryExpression is not allowed in JSON. (1:7)
> 1 | {key: "foo" + "bar"}
    |       ^^^^^^^^^^^^^
```

#### Fix syntax error on JSON range formatting ([#10497](https://github.com/prettier/prettier/pull/10497) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```js
// Input
[{ a: 1.0000}, {"b":   2.0000 }]
//                 ^^^^^^^^^^^ range

// Prettier 2.2
SyntaxError: Unexpected token (1:4)
> 1 | "b":   2.0000
    |    ^

// Prettier 2.3
[{ a: 1.0000}, { "b": 2.0 }]
```

### CSS

#### Fix absolute path in custom CSS `-custom-url()` calls ([#9966](https://github.com/prettier/prettier/pull/9966) by [@vjeux](https://github.com/vjeux))

The CSS parser is parsing this as `["division", "absolute/path"]` instead of a single `"/absolute/path"` token unless you are in a `url()` call. Because we put space after division, it results in an incorrect path. The fix was to avoid printing a space if a division is the first token of a call, which hopefully should be safe.

<!-- prettier-ignore -->
```css
/* Input */
-custom-url(/absolute/path)

/* Prettier 2.2 */
-custom-url(/ absolute/path)

/* Prettier 2.3 */
-custom-url(/absolute/path)
```

#### Exclude `@keyframes` params from being parsed as a Less variable ([#10773](https://github.com/prettier/prettier/pull/10773) by [@tmazeika](https://github.com/tmazeika))

<!-- prettier-ignore -->
```css
/* Input */
@keyframes :global(spin) {
  from {
    transform: rotate(0deg);
  }
  to {
    transform: rotate(360deg);
  }
}

/* Prettier 2.2 */
@keyframes: global(spin){
  from {
    transform: rotate(0deg);
  }
  to {
    transform: rotate(360deg);
  }
};

/* Prettier 2.3 */
@keyframes :global(spin) {
  from {
    transform: rotate(0deg);
  }
  to {
    transform: rotate(360deg);
  }
}
```

### SCSS

#### Fix broken comment inside parens ([#9710](https://github.com/prettier/prettier/pull/9710) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```scss
// Input
.simplification {
  foo: (
    calc() // not a comment anymore
  );
}

// Prettier 2.2
.simplification {
  foo: (calc() // not a comment anymore);
}

// Prettier 2.3
.simplification {
  foo: (
    calc() // not a comment anymore
  );
}
```

#### Fix maps with keys that are lists or maps ([#10005](https://github.com/prettier/prettier/pull/10005) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```scss
// Input
$map: (
  ('my list'): 'hello world',
);

// Prettier 2.2
TypeError: Cannot read property 'length' of undefined

// Prettier 2.3
$map: (
  ("my list"): "hello world",
);
```

### Ember / Handlebars

#### Preserve tilde comments ([#9082](https://github.com/prettier/prettier/pull/9082) by [@kangax](https://github.com/kangax), [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
{{~! Comment }}
{{! Comment ~}}
{{~! Comment ~}}

{{!-- Prettier 2.2 --}}
{{! Comment }}
{{! Comment }}
{{! Comment }}

{{!-- Prettier 2.3 --}}
{{~! Comment }}
{{! Comment ~}}
{{~! Comment ~}}
```

#### Add a whitespace sensitive mode ([#9885](https://github.com/prettier/prettier/pull/9885) by [@dcyriller](https://github.com/dcyriller))

```sh
--html-whitespace-sensitivity strict
```

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<span>123 {{mustache}}</span>
<span>
123 {{mustache}}</span>
<span>123 {{mustache}}
</span>
<span>
123 {{mustache}}
</span>

{{!-- Prettier 2.2 --}}
<span>
  123 {{mustache}}
</span>
<span>
  123 {{mustache}}
</span>
<span>
  123 {{mustache}}
</span>
<span>
  123 {{mustache}}
</span>

{{!-- Prettier 2.3 --}}
<span>123 {{mustache}}</span>
<span>
  123
  {{mustache}}</span>
<span>123
  {{mustache}}
</span>
<span>
  123
  {{mustache}}
</span>
```

#### Print final `blockParam` on its own line ([#9978](https://github.com/prettier/prettier/pull/9978) by [@dcyriller](https://github.com/dcyriller))

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<MyComponent @prop={{true}} @prop2={{true}} @prop3={{true}} @prop4={{true}} as |thing|></MyComponent>
{{#block param hashKey=hashValue hashKey=hashValue hashKey=hashValue as |blockParam|}}
  Hello
{{/block}}

{{!-- Prettier 2.2 --}}
<MyComponent
  @prop={{true}}
  @prop2={{true}}
  @prop3={{true}}
  @prop4={{true}} as |thing|
/>
{{#block
  param
  hashKey=hashValue
  hashKey=hashValue
  hashKey=hashValue as |blockParam|
}}
  Hello
{{/block}}

{{!-- Prettier 2.3 --}}
<MyComponent
  @prop={{true}}
  @prop2={{true}}
  @prop3={{true}}
  @prop4={{true}}
  as |thing|
/>
{{#block
  param
  hashKey=hashValue
  hashKey=hashValue
  hashKey=hashValue
  as |blockParam|
}}
  Hello
{{/block}}
```

#### Fix formatting of attributes ([#10145](https://github.com/prettier/prettier/pull/10145) by [@thorn0](https://github.com/thorn0))

- fix escaping of `{{` in attributes and text
- fix the choice between `'` and `"` for attributes with interpolations
- fix the bug with `[object Object]` printed in the `class` attribute
- implement simple formatting for the `class` attribute, like Prettier formatted it in HTML before v2.3.0

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<div class="
foo"></div>
<div bar='"{{expr}}"'></div>
<div baz="\{{ non-expression }}"></div>

{{!-- Prettier 2.2 --}}
<div class="[object Object],foo"></div>
<div bar=""{{expr}}""></div>
<div baz="{{ non-expression }}"></div>

{{!-- Prettier 2.3 --}}
<div class="foo"></div>
<div bar='"{{expr}}"'></div>
<div baz="\{{ non-expression }}"></div>
```

#### Split text content on multiple lines ([#10179](https://github.com/prettier/prettier/pull/10179) by [@dcyriller](https://github.com/dcyriller))

<!-- prettier-ignore -->
```handlebars
{{!-- Input --}}
<div>
  A long enough string to trigger a line break that would prevent wrapping more and more.
</div>

{{!-- Prettier 2.2 --}}
<div>
  A long enough string to trigger a line break that would prevent wrapping more and more.
</div>

{{!-- Prettier 2.3 --}}
<div>
  A long enough string to trigger a line break that would prevent wrapping more
  and more.
</div>
```

#### Preserve numeric character references ([#10550](https://github.com/prettier/prettier/pull/10550) by [@rwjblue](https://github.com/rwjblue) and [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```hbs
{{! Input }}
<span class="stampFont" style="font-family: 'stampfont'">&#xf000;</span>

{{! Prettier 2.2 }}
<span class="stampFont" style="font-family: 'stampfont'"></span>

{{! Prettier 2.3 }}
<span class="stampFont" style="font-family: 'stampfont'">&#xf000;</span>
```

#### Do not break opening mustache and path ([#10586](https://github.com/prettier/prettier/pull/10586) by [@dcyriller](https://github.com/dcyriller))

<!-- prettier-ignore -->
```handlebars
{{!-- Input --}}
<GlimmerComponent
  @errors={{or this.aVeryLongProperty (and this.aProperty (v-get bike "number" "message"))}}
  data-test-beneficiary-account-number
/>
<GlimmerComponent
  @progress={{aPropertyEngdingAfterEightiethColumnToHighlightAWeirdClosingParenIssue}}
/>

{{!-- Prettier 2.2 --}}
<GlimmerComponent
  @errors={{
    or
    this.aVeryLongProperty
    (and this.aProperty (v-get bike "number" "message"))
  }}
  data-test-beneficiary-account-number
/>
<GlimmerComponent
  @progress={{
    aPropertyEngdingAfterEightiethColumnToHighlightAWeirdClosingParenIssue
  }}
/>

{{!-- Prettier 2.3 --}}
<GlimmerComponent
  @errors={{or
    this.aLongProperty
    (and this.aProperty (v-get bike "number" "message"))
  }}
  data-test-beneficiary-account-number
/>
<GlimmerComponent
  @progress={{aPropertyEngdingAfterEightiethColumnToHighlightAWeirdClosingParenIssue}}
/>
```

### GraphQL

#### Fix missing space after keyword in anonymous operations ([#10689](https://github.com/prettier/prettier/pull/10689) by [@patriscus](https://github.com/patriscus))

<!-- prettier-ignore -->
```graphql
# Input
query ($unnamed: String) {
  id
}

# Prettier 2.2
query($unnamed: String) {
  id
}

# Prettier 2.3
query ($unnamed: String) {
  id
}
```

### Markdown

#### Fix extra newline at the end of JavaScript fenced code block with string literal ([#9736](https://github.com/prettier/prettier/pull/9736) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
````markdown
<!-- Input -->
Markdown

```js
"· "
```

<!-- Prettier 2.2 -->
Markdown

```js
"· ";

```

<!-- Prettier 2.3 -->
Markdown

```js
"· ";
```
````

#### Fix empty front matter formatting ([#9791](https://github.com/prettier/prettier/pull/9791) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```markdown
<!-- Input -->
---
---

# Title

a|b|c|
|:--|:-:|--:|
|d|e|f|

---

text

<!-- Prettier 2.2 -->
---
---
# Title

a|b|c|
|:--|:-:|--:|
|d|e|f|
---

text

<!-- Prettier 2.3 -->
---
---

# Title

| a   |  b  |   c |
| :-- | :-: | --: |
| d   |  e  |   f |

---

text
```

#### Support YAML document end marker in front matter ([#9878](https://github.com/prettier/prettier/pull/9878) by [@michaelbeaumont](https://github.com/michaelbeaumont))

Add the ability to delimit the end of front matter with `...`.

<!-- prettier-ignore -->
```markdown
<!-- Input -->
---
title: Hello
slug: home
...

Markdown

<!-- Prettier 2.2 -->
---

title: Hello
slug: home
...

Markdown

<!-- Prettier 2.3 -->
---
title: Hello
slug: home
...

Markdown
```

### YAML

#### Fix SyntaxError incorrectly thrown on anchors followed by empty lines ([#10516](https://github.com/prettier/prettier/pull/10516) by [@eemeli](https://github.com/eemeli) & [@thorn0](https://github.com/thorn0))

Prettier couldn't parse this valid YAML. Thanks to Eemeli Aro for fixing this bug in the underlying parser.

<!-- prettier-ignore -->
```yaml
# Input
key1: &default

  subkey1: value1

key2:
  <<: *default

# Prettier 2.2
SyntaxError: Nested mappings are not allowed in compact mappings (1:7)

# Prettier 2.3
key1: &default
  subkey1: value1

key2:
  <<: *default
```

### API

#### Treat `.prettierrc` as YAML when formatting it ([#8105](https://github.com/prettier/prettier/pull/8105) by [@fisker](https://github.com/fisker))

The `.prettierrc` file can be written in either JSON or YAML. Previously, when Prettier formatted it, the parser was inferred to be `json`, which lead to a `SyntaxError` thrown if the content was YAML. Now it’s treated as a YAML file. However, if it's JSON, it will be formatted as JSON (not as JSON-like YAML).

#### Use arrays instead of `concat` ([#9733](https://github.com/prettier/prettier/pull/9733) by [@fisker](https://github.com/fisker), [@thorn0](https://github.com/thorn0))

To simplify the code of AST printers, the data structure for the concatenation command has been changed from `{ type: 'concat', parts: Doc[] }` to `Doc[]`. The old format is deprecated, but for compatibility, the doc printer still supports it, and `doc.builders.concat` (as well as some other builder functions) will keep using it until the next major version of Prettier.

If you're a plugin author, this change should only concern you if your plugin introspects or modifies composed docs. If it happens to be the case, please make your plugin compatible with future versions of Prettier by tweaking the introspecting code to support the new format. There also is an off-chance where this change can break things, namely if a plugin calls another plugin to [print an embedded language](https://prettier.io/docs/en/plugins.html#optional-embed) and then introspects the returned doc. There seems to be no reason for plugins to do that though.

To replace `concat(…)` calls in your plugins, you can try auto-fix by this ESLint rule [`prettier-doc/no-concat`](https://github.com/fisker/eslint-plugin-prettier-doc#no-concat).

```jsx
// Prettier 2.2
myDoc = group(concat(["foo", line, "bar"]));

// Prettier 2.3
myDoc = group(["foo", line, "bar"]);
```

#### Fix `lineSuffixBoundary` IR command ([#10122](https://github.com/prettier/prettier/pull/10122) by [@thorn0](https://github.com/thorn0))

There was a bug in the implementation of the [`lineSuffixBoundary`][lsb] command that significantly limited its usefulness: the printer algorithm didn't correctly consider it a potential line break. Now that the bug has been fixed, we urge plugin authors to give this command another try and see if it can help them simplify printing of trailing comments.

[lsb]: https://github.com/prettier/prettier/blob/main/commands.md#linesuffixboundary

<!-- prettier-ignore -->
```jsx
// Input
group([
  "let foo = [",
  indent([
    softline,
    [lineSuffixBoundary, "item1,"],
    line,
    [lineSuffixBoundary, "item2,", lineSuffix(" // comment")],
    line,
    [lineSuffixBoundary, "item3"],
  ]),
  softline,
  "];",
])

// Prettier 2.2
let foo = [item1, item2,  // comment
  item3];

// Prettier 2.3
let foo = [
  item1,
  item2, // comment
  item3
];
```

#### Add `indentIfBreak` IR command ([#10221](https://github.com/prettier/prettier/pull/10221) by [@thorn](https://github.com/thorn))

`indentIfBreak(doc, { groupId })` is an optimized version of `ifBreak(indent(doc), doc, { groupId })`.

#### Simplified `print` callback ([#10557](https://github.com/prettier/prettier/pull/10557) by [@fisker](https://github.com/fisker))

The third argument of the `print` method of plugin printers (the `print` callback) has been updated. Its first argument can be a string or an array of strings now.

To print the current node, call `print` without arguments:

```diff
 function print(path, options, print) {
   const parts = [];
   path.each((childPath) => {
-    parts.push(print(childPath));
+    parts.push(print());
   }, "children");
   return parts;
 }
```

To print a property of the current node, use `"property"` or `["property"]`:

```diff
 function print(path, options, print) {
-  return path.call(print, "property");
+  return print("property");
 }
```

To print a sub-property of the current node, use `["property1", "property2"]`:

```diff
 function print(path, options, print) {
   // print `node.child.child`
-  return path.call(print, "child", "child");
+  return print(["child", "child"]);
 }
```

See also an example in the [docs](https://prettier.io/docs/en/plugins.html#print).

### CLI

#### Add CLI option to prevent errors on unmatched pattern ([#10058](https://github.com/prettier/prettier/pull/10058) by [@daronmcintosh](https://github.com/daronmcintosh))

The Prettier CLI will no longer display an error when no files match the glob pattern passed as input.

```sh
# Prettier 2.2
$ npx prettier --check "prettier/docs/*.yaml"
Checking formatting...
[error] No files matching the pattern were found: "prettier/docs/*.yaml".
All matched files use Prettier code style!

# Prettier 2.3
$ npx prettier --check --no-error-on-unmatched-pattern "prettier/docs/*.yaml"
Checking formatting...
All matched files use Prettier code style!
```

#### Add CLI flag for debugging comment attachment issues ([#10124](https://github.com/prettier/prettier/pull/10124) by [@thorn0](https://github.com/thorn0))

A new `--debug-print-comments` CLI flag and corresponding functionality for the Playground.

#### Round-trippable `--debug-print-doc` ([#10169](https://github.com/prettier/prettier/pull/10169), [#10177](https://github.com/prettier/prettier/pull/10177) by [@thorn0](https://github.com/thorn0))

The idea is to make the output of `--debug-print-doc` closer to actual code for generating docs (Prettier's intermediate representation). Ideally, it should be possible for it to work without modification after copy-pasting into a JS file. That ideal hasn't probably been reached by this PR, but it's pretty close. This is going to make `--debug-print-doc` and the corresponding part of the Playground a bit more useful.

#### Print an error message when `--find-config-path` can't find config file ([#10208](https://github.com/prettier/prettier/pull/10208) by [@fisker](https://github.com/fisker))

```console
# Prettier 2.2
$ prettier --find-config-path /prettier.js
# Silently failed

# Prettier 2.3
$ prettier --find-config-path /prettier.js
[error] Can not find configure file for "/prettier.js"
```

#### Clear printed long filenames of formatting file ([#10217](https://github.com/prettier/prettier/pull/10217) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```console
# Prettier 2.2
$ prettier tests/flow-repo/config_module_system_node_resolve_dirname --check
Checking formatting...
tests\flow-repo\config_module_system_node_resolve_dirname\custom_resolve_dir\tes
tests\flow-repo\config_module_system_node_resolve_dirname\custom_resolve_dir\tes
tests\flow-repo\config_module_system_node_resolve_dirname\subdir\custom_resolve_
All matched files use Prettier code style!

# Prettier 2.3
$ prettier tests/flow-repo/config_module_system_node_resolve_dirname --check
Checking formatting...
All matched files use Prettier code style!
```

#### Don't skip file paths containing special words like `constructor` ([#10256](https://github.com/prettier/prettier/pull/10256) by [@ashlkv](https://github.com/ashlkv))

Directories whose names happened to coincide with the properties of `Object.prototype` were ignored by Prettier CLI because of a classic bug (introduced in Prettier 2.0.0) with object properties not being checked for being own.

<!-- prettier-ignore -->
```console
# Prettier 2.2
$ prettier "js/constructor/*.js" --write
[error] No matching files. Patterns: js/constructor/*.js

# Prettier 2.3
$ prettier "js/constructor/*.js" --write
js/constructor/test.js 42ms
```
