import Session       from '@/shared/services/session'
import Records       from '@/shared/services/records'
import Flash         from '@/shared/services/flash'
import EventBus       from '@/shared/services/event_bus'
import AbilityService from '@/shared/services/ability_service'
import LmoUrlService  from '@/shared/services/lmo_url_service'
import openModal      from '@/shared/helpers/open_modal'
import AppConfig      from '@/shared/services/app_config'
import i18n from '@/i18n.coffee'

export default new class GroupService
  actions: (group) ->
    membership = group.membershipFor(Session.user())

    translate_group:
      name: 'common.action.translate'
      icon: 'mdi-translate'
      dock: 2
      canPerform: ->
        group.description && AbilityService.canTranslate(group)
      perform: ->
        Session.user() && group.translate(Session.user().locale)

    change_volume:
      name: 'group_page.options.email_settings'
      icon: 'mdi-email'
      dock: 1
      canPerform: ->
        AbilityService.canChangeGroupVolume(group)
      perform: ->
        openModal
          component: 'ChangeVolumeForm'
          props:
            model: membership

    edit_group:
      name: 'group_page.options.edit_group'
      icon: 'mdi-cog'
      dock: 1
      canPerform: ->
        AbilityService.canEditGroup(group)
      perform: ->
        openModal
          component: 'GroupForm'
          props:
            group: group.clone()

    become_coordinator:
      name: 'group_page.options.become_coordinator'
      icon: 'mdi-shield-star'
      menu: true
      canPerform: ->
        membership && membership.admin == false &&
          (group.adminMembershipsCount == 0 or group.parentOrSelf().adminsInclude(Session.user()))
      perform: ->
        Records.memberships.makeAdmin(membership).then ->
          Flash.success "memberships_page.messages.make_admin_success", name: Session.user().name

    chatbots:
      name: 'chatbot.chatbot'
      icon: 'mdi-robot'
      menu: true
      canPerform: ->
        AppConfig.features.app.chatbots && group.adminsInclude(Session.user())
      perform: ->
        openModal
          component: 'ChatbotForm'
          props:
            group: group

    webhooks:
      name: 'webhook.integrations'
      icon: 'mdi-webhook'
      menu: true
      canPerform: ->
        group.adminsInclude(Session.user())
      perform: ->
        openModal
          component: 'WebhookList'
          props:
            group: group

    group_stats:
      name: 'group_page.stats'
      icon: 'mdi-chart-bar'
      menu: true
      canPerform: ->
        AbilityService.canAdminister(group)
      perform: ->
        window.open("#{AppConfig.baseUrl}g/#{group.key}/stats?export=1", "_blank")

    export_data:
      name: 'group_page.options.export_data'
      icon: 'mdi-database-export'
      menu: true
      canPerform: ->
        membership && group.adminsInclude(Session.user())
      perform: ->
        openModal
          component: 'ExportDataModal'
          props:
            group: group

    manage_subscription:
      name: 'subscription_status.manage_subscription'
      icon: 'mdi-credit-card-outline'
      menu: true
      perform: ->
        window.location = "/upgrade/#{group.id}"
      canPerform: ->
        AppConfig.features.app.subscriptions &&
        group.isParent() &&
        group.adminsInclude(Session.user())

    leave_group:
      name: 'group_page.options.leave_group'
      icon: 'mdi-exit-to-app'
      menu: true
      canPerform: ->
        AbilityService.canRemoveMembership(membership)
      perform: ->
        openModal
          component: 'ConfirmModal'
          props:
            confirm:
              submit: ->
                membership.destroy().then ->
                  Records.discussions.find(groupId: membership.groupId).forEach (d) -> d.update(discussionReaderUserId: null)

              text:
                title:    'leave_group_form.title'
                helptext: 'leave_group_form.question'
                confirm:  'leave_group_form.submit'
                flash:    'group_page.messages.leave_group_success'
              redirect: '/dashboard'

    destroy_group:
      name: 'delete_group_modal.title'
      icon: 'mdi-delete'
      menu: true
      canPerform: ->
        AbilityService.canArchiveGroup(group)
      perform: ->
        confirmText = group.handle || group.name.trim()
        openModal
          component: 'ConfirmModal'
          props:
            confirm:
              submit: group.destroy
              text:
                title:    'delete_group_modal.title'
                helptext: 'delete_group_modal.body'
                raw_confirm_text_placeholder: i18n.t('delete_group_modal.confirm', name: confirmText)
                confirm_text: confirmText
                flash:    'delete_group_modal.success'
                submit:   'delete_group_modal.title'
              redirect:   '/dashboard'
