/*globals describe it */
var should = require("should");
var path = require("path");
var fs = require("fs");

var webpack = require("../lib/webpack");

var base = path.join(__dirname, "statsCases");
var tests = fs.readdirSync(base);
var Stats = require("../lib/Stats");

describe("Stats", function() {
	tests.forEach(function(testName) {
		it("should print correct stats for " + testName, function(done) {
			var options = {
				entry: "./index"
			};
			if(fs.existsSync(path.join(base, testName, "webpack.config.js"))) {
				options = require(path.join(base, testName, "webpack.config.js"));
			}
			(Array.isArray(options) ? options : [options]).forEach(function(options) {
				options.context = path.join(base, testName);
			});
			var c = webpack(options);
			var files = {};
			var compilers = c.compilers ? c.compilers : [c];
			compilers.forEach(function(c) {
				c.outputFileSystem = {
					join: path.join.bind(path),
					mkdirp: function(path, callback) {
						callback();
					},
					writeFile: function(name, content, callback) {
						files[name] = content.toString("utf-8");
						callback();
					}
				};

				var ifs = c.inputFileSystem;
				c.inputFileSystem = Object.create(ifs);
				c.inputFileSystem.readFile = function() {
					var args = Array.prototype.slice.call(arguments);
					var callback = args.pop();
					ifs.readFile.apply(ifs, args.concat([function(err, result) {
						if(err) return callback(err);
						callback(null, result.toString("utf-8").replace(/\r/g, ""));
					}]));
				};
				c.apply(new webpack.optimize.OccurrenceOrderPlugin());
			});
			c.run(function(err, stats) {
				if(err) return done(err);

				if(/error$/.test(testName)) {
					stats.hasErrors().should.be.equal(true);
				} else if(stats.hasErrors()) {
					done(new Error(stats.toJson().errors.join("\n\n")));
				}

				var toStringOptions = {
					colors: false
				};
				if(typeof options.stats !== "undefined") {
					toStringOptions = options.stats;
				}

				var actual = stats.toString(toStringOptions);
				(typeof actual).should.be.eql("string");
				actual =
					actual.replace(/\u001b\[[0-9;]*m/g, "")
					.replace(/Version:.+\n/, "")
					.replace(/[0-9]+(\s?ms)/g, "X$1")
					.replace(/\r/g, "")
					.replace(path.join(base, testName), "Xdir/" + testName);
				var expected = fs.readFileSync(path.join(base, testName, "expected.txt"), "utf-8").replace(/\r/g, "");
				if(actual !== expected) {
					fs.writeFileSync(path.join(base, testName, "actual.txt"), actual, "utf-8");
				} else if(fs.existsSync(path.join(base, testName, "actual.txt"))) {
					fs.unlinkSync(path.join(base, testName, "actual.txt"));
				}
				actual.should.be.eql(expected);
				done();
			});
		});
	});
	describe("Error Handling", function() {
		describe("does have", function() {
			it("hasErrors", function() {
				var mockStats = new Stats({
					errors: ['firstError'],
					hash: '1234'
				});
				mockStats.hasErrors().should.be.ok;
			});
			it("hasWarnings", function() {
				var mockStats = new Stats({
					warnings: ['firstError'],
					hash: '1234'
				});
				mockStats.hasWarnings().should.be.ok;
			});
		});
		describe("does not have", function() {
			it("hasErrors", function() {
				var mockStats = new Stats({
					errors: [],
					hash: '1234'
				});
				mockStats.hasErrors().should.not.be.ok;
			});
			it("hasWarnings", function() {
				var mockStats = new Stats({
					warnings: [],
					hash: '1234'
				});
				mockStats.hasWarnings().should.not.be.ok;
			});
		});
		it("formatError handles string errors", function() {
			var mockStats = new Stats({
				errors: ['firstError'],
				warnings: [],
				assets: [],
				chunks: [],
				modules: [],
				children: [],
				hash: '1234',
				mainTemplate: {
					getPublicPath: function() {
						return 'path';
					}
				}
			});
			var obj = mockStats.toJson();
			obj.errors[0].should.be.equal('firstError');
		});
	});
	describe("Presets", function() {
		describe("presetToOptions", function() {
			it("returns correct object with 'Normal'", function() {
				Stats.presetToOptions("Normal").should.eql({
					assets: false,
					version: false,
					timings: true,
					hash: true,
					chunks: true,
					chunkModules: false,
					errorDetails: true,
					reasons: false,
					colors: true
				});
			});
			it("truthy values behave as 'normal'", function() {
				var normalOpts = Stats.presetToOptions('normal');
				Stats.presetToOptions("pizza").should.eql(normalOpts);
				Stats.presetToOptions(true).should.eql(normalOpts);
				Stats.presetToOptions(1).should.eql(normalOpts);

				Stats.presetToOptions("verbose").should.not.eql(normalOpts);
				Stats.presetToOptions(false).should.not.eql(normalOpts);
			});
			it("returns correct object with 'none'", function() {
				Stats.presetToOptions("none").should.eql({
					hash: false,
					version: false,
					timings: false,
					assets: false,
					chunks: false,
					modules: false,
					reasons: false,
					children: false,
					source: false,
					errors: false,
					errorDetails: false,
					warnings: false,
					publicPath: false
				});
			});
			it("falsy values behave as 'none'", function() {
				var noneOpts = Stats.presetToOptions('none');
				Stats.presetToOptions("").should.eql(noneOpts);
				Stats.presetToOptions(null).should.eql(noneOpts);
				Stats.presetToOptions().should.eql(noneOpts);
				Stats.presetToOptions(0).should.eql(noneOpts);
				Stats.presetToOptions(false).should.eql(noneOpts);
			});
		});
	});
});
