﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.Editor.Shared.Utilities;
using Microsoft.CodeAnalysis.Navigation;
using Microsoft.CodeAnalysis.Shared.Extensions;
using Microsoft.VisualStudio.GraphModel;
using Microsoft.VisualStudio.GraphModel.CodeSchema;
using Microsoft.VisualStudio.GraphModel.Schemas;
using Roslyn.Utilities;

namespace Microsoft.VisualStudio.LanguageServices.Implementation.Progression
{
    internal sealed class GraphNavigatorExtension : ForegroundThreadAffinitizedObject, IGraphNavigateToItem
    {
        private readonly Workspace _workspace;

        public GraphNavigatorExtension(Workspace workspace)
        {
            _workspace = workspace;
        }

        public void NavigateTo(GraphObject graphObject)
        {
            var graphNode = graphObject as GraphNode;

            if (graphNode != null)
            {
                var sourceLocation = graphNode.GetValue<SourceLocation>(CodeNodeProperties.SourceLocation);
                if (sourceLocation.FileName == null)
                {
                    return;
                }

                var projectId = graphNode.GetValue<ProjectId>(RoslynGraphProperties.ContextProjectId);
                var symbolId = graphNode.GetValue<SymbolKey>(RoslynGraphProperties.SymbolId);

                if (projectId != null)
                {
                    var solution = _workspace.CurrentSolution;
                    var project = solution.GetProject(projectId);

                    if (project == null)
                    {
                        return;
                    }

                    var document = project.Documents.FirstOrDefault(
                        d => string.Equals(
                            d.FilePath,
                            sourceLocation.FileName.LocalPath,
                            StringComparison.OrdinalIgnoreCase));

                    if (document == null)
                    {
                        return;
                    }

                    Task.Factory.SafeStartNew(
                        () => NavigateOnForegroundThread(sourceLocation, symbolId, project, document),
                        CancellationToken.None,
                        ForegroundTaskScheduler);
                }
            }
        }

        private void NavigateOnForegroundThread(SourceLocation sourceLocation, SymbolKey symbolId, Project project, Document document)
        {
            AssertIsForeground();

            // Notify of navigation so third parties can intercept the navigation
            if (symbolId != null)
            {
                var symbolNavigationService = _workspace.Services.GetService<ISymbolNavigationService>();
                var symbol = symbolId.Resolve(project.GetCompilationAsync(CancellationToken.None).WaitAndGetResult(CancellationToken.None)).Symbol;

                // Do not allow third party navigation to types or constructors
                if (symbol != null &&
                    !(symbol is ITypeSymbol) &&
                    !symbol.IsConstructor() &&
                    symbolNavigationService.TrySymbolNavigationNotify(symbol, project.Solution))
                {
                    return;
                }
            }

            if (sourceLocation.IsValid)
            {
                // We must find the right document in this project. This may not be the
                // ContextDocumentId if you have a partial member that is shown under one
                // document, but only exists in the other

                if (document != null)
                {
                    var editorWorkspace = document.Project.Solution.Workspace;
                    var navigationService = editorWorkspace.Services.GetService<IDocumentNavigationService>();
                    navigationService.TryNavigateToLineAndOffset(
                        editorWorkspace,
                        document.Id,
                        sourceLocation.StartPosition.Line,
                        sourceLocation.StartPosition.Character);
                }
            }
        }

        public int GetRank(GraphObject graphObject)
        {
            var graphNode = graphObject as GraphNode;

            if (graphNode != null)
            {
                var sourceLocation = graphNode.GetValue<SourceLocation>(CodeNodeProperties.SourceLocation);
                var projectId = graphNode.GetValue<ProjectId>(RoslynGraphProperties.ContextProjectId);

                if (sourceLocation.IsValid && projectId != null)
                {
                    return GraphNavigateToItemRanks.OwnItem;
                }
            }

            return GraphNavigateToItemRanks.CanNavigateToItem;
        }
    }
}
