#region 文件描述
/******************************************************************************
* 创建: Daoting
* 摘要: 
* 日志: 2014-07-02 创建
******************************************************************************/
#endregion

#region 引用命名
using Dt.CalcEngine;
using Dt.CalcEngine.Expressions;
using Dt.CalcEngine.Functions;
using Dt.Xls;
using Dt.Xls.Chart;

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.CompilerServices;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.WindowsRuntime;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using Windows.Foundation;
using Microsoft.UI;
using Microsoft.UI.Xaml;
using Microsoft.UI.Xaml.Media;
using Windows.UI;
#endregion

namespace Dt.Cells.Data
{
    /// <summary>
    /// Represents a sheet.
    /// </summary>
    public partial class Worksheet : IXmlSerializable, ICellsSupport, ICalcSource, IEqualityComparer<ICalcSource>, ICustomNameSupport, ISubtotalSupport, IMultiSourceProvider, IBindableSheet, ISparklineSheet, ITableSheet, IUIActionExecuter, IPrintSupportInternal, IFloatingObjectSheet, ICalcEvaluator, IThemeSupport
    {
        #region 成员变量
        CalcParser _cachedParser;
        List<IExcelChart> _charts;
        IDisposable _currentUIActionWrapper;
        FloatingObjects _floatingObjects;
        SpreadPictures _pictures;
        Brush _selectionBackground;
        Color _selectionBorderColor;
        string _selectionBorderThemeColor;
        SpreadCharts _spreadCharts;
        Color _touchSelectionGripperBackgroundColor;
        string _touchSelectionGripperBackgroundThemeColor;
        List<IExcelChart> _unsupportedChartList;
        List<IUnsupportRecord> _unsupportedRecords;
        IList<object> _unsupportImages;
        bool _autoGeneratedColumns;
        bool _autoRecalculation;
        SheetBindingManager _bindingManager;
        StyleInfo _buff;
        DataMatrix<object> _cachedDeserializedBindingValues;
        List<List<object>> _cachedDeserializedCustomNames;
        object _cachedDeserializedDatasource;
        List<List<object>> _cachedDeserializedFormulaStrings;
        SpreadCalcAxialManager _calcAxial;
        short _calcEngineSuspended;
        CalcCalculationManager _calcManager;
        CalcService _calcService;
        SpreadCalcStorage _calcStorage;
        Cells _cells;
        StorageBlock _cellsBlock;
        bool _columnFooterVisible;
        ColumnHeader _columnHeader;
        HeaderAutoText _columnHeaderAutoText;
        int _columnHeaderAutoTextIndex;
        SparseArray<object> _columnHeaderColumnsStyles;
        StyleInfo _columnHeaderDefaultStyle;
        SparseArray<AxisInfo> _columnHeaderRows;
        SparseArray<object> _columnHeaderRowsStyles;
        StorageBlock _columnHeadersBlock;
        SheetSpanModel _columnHeaderSpanModel;
        bool _columnHeaderVisible;
        RangeGroup _columnRangeGroup;
        Columns _columns;
        ConditionalFormat _conditionalFormats;
        bool _dataAutoHeadings;
        object _datasource;
        const int _DEFAULT_COLUMN_FOOTER_ROWHEIGHT = 30;
        const int _DEFAULT_COLUMN_HEADER_ROWHEIGHT = 30;
        const int _DEFAULT_COLUMN_WIDTH = 80;
        const int _DEFAULT_ROW_HEADER_COLUMNWIDTH = 40;
        const int _DEFAULT_ROW_HEIGHT = 30;
        const UnitType _DEFAULT_UNIT = UnitType.Pixel;
        internal static readonly string DefaultCellsStyleName = "CellsDefault";
        double _defaultColumnFooterRowHeight;
        internal static readonly string DefaultColumnFooterStyleName = "ColumnFooterDefault";
        double _defaultColumnHeaderRowHeight;
        internal static readonly string DefaultColumnHeaderStyleName = "ColumnHeaderDefault";
        double _defaultColumnWidth;
        BorderLine _defaultGridLine;
        internal static readonly Color DefaultGridLineColor = Color.FromArgb(0xff, 0xe0, 0xe0, 0xe0);
        double _defaultRowHeaderColumnWidth;
        internal static readonly string DefaultRowHeaderStyleName = "RowHeaderDefault";
        double _defaultRowHeight;
        internal static readonly Color DefaultSelectionBorderColor = Color.FromArgb(220, 0, 0, 0);
        internal static readonly Color DefaultSheetTabColor = Colors.Transparent;
        const int _DefaultStartingColumnNumber = 1;
        const int _DefaultStartingRowNumber = 1;
        StyleInfo _defaultStyle;
        int _frozenColumnCount;
        int _frozenRowCount;
        int _frozenTrailingColumnCount;
        int _frozenTrailingRowCount;
        IDictionary<string, CalcFunction> _functions;
        BorderLine _gridLine;
        Color _gridLineColor;
        const double _HORIZONTALSPLITBOX_WIDTH = 6.0;
        bool _isSettingFormula;
        bool _isValidatingCell;
        const int _MaxColumnHeaderRowCount = 0x100;
        const int _MaxRowHeaderColumnCount = 0x100;
        string _name;
        StyleInfoCollection _namedStyles;
        NameInfoCollection _names;
        PrintInfo _printInfo;
        bool _protect;
        ReferenceStyle _referenceStyle;
        RowFilterBase _rowFilter;
        RowHeader _rowHeader;
        HeaderAutoText _rowHeaderAutoText;
        int _rowHeaderAutoTextIndex;
        SparseArray<AxisInfo> _rowHeaderColumns;
        SparseArray<object> _rowHeaderColumnsStyles;
        StyleInfo _rowHeaderDefaultStyle;
        SparseArray<object> _rowHeaderRowsStyles;
        StorageBlock _rowHeadersBlock;
        SheetSpanModel _rowHeaderSpanModel;
        bool _rowHeaderVisible;
        RangeGroup _rowRangeGroup;
        Rows _rows;
        SheetSelectionModel _selectionModel;
        Color _sheetTabColor;
        string _sheetTabThemeColor;
        bool _showGridLine;
        SortImp _sortImp;
        SheetSpanModel _spanModel;
        WorksheetSparklineGroupManager _sparklineGroupManager;
        internal SpreadCalcReference _standaloneCalcRef;
        int _startingColumnNumber;
        int _startingRowNumber;
        EricTables _tables;
        object _tag;
        int _validatingColumn;
        int _validatingRow;
        object _validatingValue;
        const double _VERTICALSPLITBOX_HEIGHT = 6.0;
        SparseArray<AxisInfo> _viewportColumns;
        SparseArray<object> _viewportColumnsStyles;
        ViewportInfo _viewportInfo;
        SparseArray<AxisInfo> _viewportRows;
        SparseArray<object> _viewportRowsStyles;
        bool _visible;
        Workbook _workbook;
        float _zoomFactor;
        #endregion

        #region 事件
        /// <summary>
        /// Occurs when a change is made to a cell or range of cells in this sheet
        /// that may require the cell or range of cells to be repainted.
        /// </summary>
        public event EventHandler<CellChangedEventArgs> CellChanged;

        /// <summary>
        /// Occurs when the chart has changed.
        /// </summary>
        public event EventHandler<ChartChangedEventArgs> ChartChanged;

        /// <summary>
        /// Occurs when the chart selection has changed.
        /// </summary>
        public event EventHandler<ChartSelectionChangedEventArgs> ChartSelectionChanged;

        /// <summary>
        /// Occurs when a change is made to a column or range of columns in this sheet 
        /// that may require the column or range of columns to be repainted.
        /// </summary>
        public event EventHandler<SheetChangedEventArgs> ColumnChanged;

        /// <summary>
        /// Occurs when two columns are swapped during sorting.
        /// </summary>
        public event EventHandler<ColumnsSwappedEventArgs> ColumnsSwapped;

        /// <summary>
        /// Occurs when the floating object has changed.
        /// </summary>
        public event EventHandler<FloatingObjectChangedEventArgs> FloatingObjectChanged;

        /// <summary>
        /// Occurs when the floating object selection has changed.
        /// </summary>
        public event EventHandler<FloatingObjectSelectionChangedEventArgs> FloatingObjectSelectionChanged;

        event EventHandler<CellChangedEventArgs> ISparklineSheet.CellChanged
        {
            add
            {
                CellChanged += value;
            }
            remove
            {
                CellChanged -= value;
            }
        }

        /// <summary>
        /// Occurs when the picture has changed.
        /// </summary>
        public event EventHandler<PictureChangedEventArgs> PictureChanged;

        /// <summary>
        /// Occurs when the picture selection has changed.
        /// </summary>
        public event EventHandler<PictureSelectionChangedEventArgs> PictureSelectionChanged;

        /// <summary>
        /// Occurs when the user changes a sheet property.
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// Occurs when a change is made to a row or range of rows in this sheet
        /// that may require the row or range of rows to be repainted.
        /// </summary>
        public event EventHandler<SheetChangedEventArgs> RowChanged;

        /// <summary>
        /// Occurs when two rows are swapped during sorting.
        /// </summary>
        public event EventHandler<RowsSwappedEventArgs> RowsSwapped;

        /// <summary>
        /// Occurs when selections are changed 
        /// that may require the selection area to be repainted.
        /// </summary>
        public event EventHandler<SheetSelectionChangedEventArgs> SelectionChanged;
        #endregion

        #region 构造方法
        /// <summary>
        /// Creates a new sheet.
        /// </summary>
        public Worksheet()
            : this(500, 100)
        {
        }

        /// <summary>
        /// Creates a new sheet with the specified named style.
        /// </summary>
        /// <param name="namedStyles">The named style.</param>
        public Worksheet(StyleInfoCollection namedStyles)
            : this(500, 100)
        {
            NamedStyles = new StyleInfoCollection();
            NamedStyles.AddRange(namedStyles);
        }

        /// <summary>
        /// Creates a new sheet with a specified name.
        /// </summary>
        /// <param name="name">The name of the sheet.</param>
        public Worksheet(string name)
            : this(500, 100)
        {
            _name = name;
        }

        /// <summary>
        /// Creates a new sheet with the specified row count and column count.
        /// </summary>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        public Worksheet(int rowCount, int columnCount)
            : this(rowCount, columnCount, 1, 1)
        {
        }

        /// <summary>
        /// Creates a new sheet with the specified row count, column count, row header column count, and column header row count.
        /// </summary>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <param name="rowHeaderColumnCount">The row header column count.</param>
        /// <param name="columnHeaderRowCount">The column header row count.</param>
        public Worksheet(int rowCount, int columnCount, int rowHeaderColumnCount, int columnHeaderRowCount)
        {
            _cachedParser = new CalcParser();
            _charts = new List<IExcelChart>();
            Init(rowCount, columnCount, rowHeaderColumnCount, columnHeaderRowCount, false);
        }
        #endregion

        /// <summary>
        /// Adds the chart.
        /// </summary>
        /// <param name="name">The name of the chart.</param>
        /// <param name="chartType">The type of the chart.</param>
        /// <param name="x">The x value or left position.</param>
        /// <param name="y">The y value or top position.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns></returns>
        /// <remarks>
        /// The row parameter represents the start row index and column represents the start column index.  This method will auto detect a cell range based on these two indexes. The seriesType parameter indicates which type of chart is added. The chartWidth and chartHeight parameters specify the size (width and height) of the chart. The absolute position is specified by the x and y parameters. The ChartViewType parameter is the view type of the chart (2D or 3D). The showLegend parameter indicates whether to show a default legend area in the chart.
        /// </remarks>
        public SpreadChart AddChart(string name, SpreadChartType chartType, double x, double y, double width, double height)
        {
            SpreadChart item = new SpreadChart(name, chartType, x, y, width, height);
            Charts.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the chart.
        /// </summary>
        /// <param name="name">The name of the chart.</param>
        /// <param name="chartType">The type of the chart.</param>
        /// <param name="formula">The formula.</param>
        /// <param name="x">The x value or left position.</param>
        /// <param name="y">The y value or top position.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns></returns>
        /// <remarks>
        /// The chartType parameter indicates which type of chart will be added. The width and height parameters specify the size of the chart. The absolute position is specified by the x and y parameters.
        /// </remarks>
        public SpreadChart AddChart(string name, SpreadChartType chartType, string formula, double x, double y, double width, double height)
        {
            SpreadChart item = new SpreadChart(name, chartType, formula, x, y, width, height);
            Charts.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the chart.
        /// </summary>
        /// <param name="name">The name of the chart.</param>
        /// <param name="chartType">Type of the chart.</param>
        /// <param name="formula">The formula.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns></returns>
        /// <remarks>
        /// The row parameter represents the start row index and column represents the start column index.  This method will auto detect a cell range based on these two indexes. The chartType parameter indicates which type of chart will be added. The width and height parameters specify the size of the chart. 
        /// </remarks>
        public SpreadChart AddChart(string name, SpreadChartType chartType, string formula, int row, double rowOffset, int column, double columnOffset, double width, double height)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            SpreadChart item = new SpreadChart(name, chartType, formula, point.X, point.Y, width, height);
            Charts.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the chart in the sheet.
        /// </summary>
        /// <param name="name">The name of the chart.</param>
        /// <param name="chartType">The type of the chart.</param>
        /// <param name="nameFormula">The series name formula.</param>
        /// <param name="itemsFormula">The item label formula.</param>
        /// <param name="dataFormula">The data formula.</param>
        /// <param name="x">The x value or left position.</param>
        /// <param name="y">The y value or top position.</param>
        /// <param name="width">The width of chart.</param>
        /// <param name="height">The height of chart.</param>
        /// <returns></returns>
        public SpreadChart AddChart(string name, SpreadChartType chartType, string itemsFormula, string nameFormula, string dataFormula, double x, double y, double width, double height)
        {
            SpreadChart item = new SpreadChart(name, chartType, itemsFormula, nameFormula, dataFormula, x, y, width, height);
            Charts.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the chart.
        /// </summary>
        /// <param name="name">The name of the chart.</param>
        /// <param name="chartType">Type of the chart.</param>
        /// <param name="formula">The formula.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="endRow">The end row.</param>
        /// <param name="endRowOffset">The end row offset.</param>
        /// <param name="endColumn">The end column.</param>
        /// <param name="endColumnOffset">The end column offset.</param>
        /// <returns></returns>
        public SpreadChart AddChart(string name, SpreadChartType chartType, string formula, int row, double rowOffset, int column, double columnOffset, int endRow, double endRowOffset, int endColumn, double endColumnOffset)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            Point point2 = CalcLocation(endRow, endRowOffset, endColumn, endColumnOffset);
            double x = (point.X < point2.X) ? point.X : point2.X;
            double y = (point.Y < point2.Y) ? point.Y : point2.Y;
            double width = Math.Abs((double)(point.X - point2.X));
            double height = Math.Abs((double)(point.Y - point2.Y));
            return AddChart(name, chartType, formula, x, y, width, height);
        }

        /// <summary>
        /// Adds the chart.
        /// </summary>
        /// <param name="name">The name of the chart.</param>
        /// <param name="chartType">Type of the chart.</param>
        /// <param name="itemsFormula">The items label formula.</param>
        /// <param name="nameFormula">The name formula.</param>
        /// <param name="dataFormula">The data formula.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns></returns>
        /// <remarks>
        /// The row parameter represents the start row index and column represents the start column index.  This method will auto detect a cell range based on these two indexes. The chartType parameter indicates which type of chart will be added. The width and height parameters specify the size of the chart. 
        /// </remarks>
        public SpreadChart AddChart(string name, SpreadChartType chartType, string itemsFormula, string nameFormula, string dataFormula, int row, double rowOffset, int column, double columnOffset, double width, double height)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            SpreadChart item = new SpreadChart(name, chartType, itemsFormula, nameFormula, dataFormula, point.X, point.Y, width, height);
            Charts.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the column or columns to the data model at the specified index.
        /// </summary>
        /// <param name="column">Column index at which to add the new columns.</param>
        /// <param name="count">The number of columns to add.</param>
        public void AddColumns(int column, int count)
        {
            AddColumns(column, count, SheetArea.Cells);
        }

        /// <summary>
        /// Adds the column or columns to the data model at the specified index in the specified sheet area.
        /// </summary>
        /// <param name="sheetArea">The sheet area to which to add columns.</param>
        /// <param name="column">Column index at which to add the new columns.</param>
        /// <param name="count">The number of columns to add.</param>
        public void AddColumns(int column, int count, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == SheetArea.ColumnHeader))
            {
                CalcExpressionHelper.ValidateIntersectedArrayFormula(CalcManager, -1, column, -1, 1, true);
                _cellsBlock.AddColumns(column, count);
                _viewportColumns.InsertRange(column, count);
                _viewportColumnsStyles.InsertRange(column, count);
                _columnHeadersBlock.AddColumns(column, count);
                _columnHeaderColumnsStyles.InsertRange(column, count);
            }
            else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                _rowHeadersBlock.AddColumns(column, count);
                _rowHeaderColumns.InsertRange(column, count);
                _rowHeaderColumnsStyles.InsertRange(column, count);
            }
            OnColumnsAdded(column, count, sheetArea);
            RaiseColumnChanged(string.Empty, column, count, sheetArea, SheetChangedEventAction.ColumnsAdded);
        }

        /// <summary>
        /// Adds the column viewport.
        /// </summary>
        /// <param name="columnViewportIndex">The index of the column viewport.</param>
        /// <param name="viewportWidth">The width of the viewport.</param>
        public void AddColumnViewport(int columnViewportIndex, double viewportWidth)
        {
            if (((columnViewportIndex > -1) && (columnViewportIndex < GetViewportInfo().ColumnViewportCount)) && (viewportWidth != 0.0))
            {
                ViewportInfo viewportInfo = GetViewportInfo();
                ViewportInfo info2 = new ViewportInfo(this, viewportInfo.RowViewportCount, viewportInfo.ColumnViewportCount + 1);
                if (columnViewportIndex <= viewportInfo.ActiveColumnViewport)
                {
                    info2.ActiveColumnViewport = viewportInfo.ActiveColumnViewport + 1;
                }
                else
                {
                    info2.ActiveColumnViewport = viewportInfo.ActiveColumnViewport;
                }
                info2.ActiveRowViewport = viewportInfo.ActiveRowViewport;
                viewportInfo.TopRows.CopyTo(info2.TopRows, 0);
                viewportInfo.ViewportHeight.CopyTo(info2.ViewportHeight, 0);
                for (int i = 0; i < columnViewportIndex; i++)
                {
                    info2.LeftColumns[i] = viewportInfo.LeftColumns[i];
                    info2.ViewportWidth[i] = viewportInfo.ViewportWidth[i];
                }
                for (int j = columnViewportIndex + 1; j < viewportInfo.ColumnViewportCount; j++)
                {
                    info2.LeftColumns[j + 1] = viewportInfo.LeftColumns[j];
                    info2.ViewportWidth[j + 1] = viewportInfo.ViewportWidth[j];
                }
                info2.LeftColumns[columnViewportIndex] = FrozenColumnCount;
                info2.LeftColumns[columnViewportIndex + 1] = viewportInfo.LeftColumns[columnViewportIndex];
                if (viewportWidth == -1.0)
                {
                    double num3 = viewportInfo.ViewportWidth[columnViewportIndex];
                    double num4 = Math.Max((double)0.0, (double)(num3 - 6.0));
                    if (num3 == -1.0)
                    {
                        info2.ViewportWidth[columnViewportIndex] = -1.0;
                        info2.ViewportWidth[columnViewportIndex + 1] = -1.0;
                    }
                    else
                    {
                        info2.ViewportWidth[columnViewportIndex] = num4 / 2.0;
                        info2.ViewportWidth[columnViewportIndex + 1] = (info2.ViewportWidth[columnViewportIndex] > 0.0) ? (num4 - info2.ViewportWidth[columnViewportIndex]) : num3;
                    }
                }
                else
                {
                    info2.ViewportWidth[columnViewportIndex] = viewportWidth;
                    double num5 = viewportInfo.ViewportWidth[columnViewportIndex];
                    if (num5 != -1.0)
                    {
                        num5 = Math.Max((double)0.0, (double)((num5 - viewportWidth) - 6.0));
                    }
                    info2.ViewportWidth[columnViewportIndex + 1] = num5;
                }
                SetViewportInfo(info2);
                if (viewportWidth != -1.0)
                {
                    if (info2.ViewportWidth[columnViewportIndex + 1] == 0.0)
                    {
                        MergeColumnViewports(columnViewportIndex + 1, columnViewportIndex);
                    }
                    else if (info2.ViewportWidth[columnViewportIndex] == 0.0)
                    {
                        MergeColumnViewports(columnViewportIndex, columnViewportIndex + 1);
                    }
                }
                info2 = GetViewportInfo();
                info2.ViewportWidth[info2.ColumnViewportCount - 1] = -1.0;
            }
        }

        /// <summary>
        /// Adds the specified user-defined custom function to the collection. 
        /// </summary>
        /// <param name="function">The function to add.</param>
        public void AddCustomFunction(CalcFunction function)
        {
            string name = function.Name.ToUpper(CultureInfo.CurrentCulture);
            if (Functions.ContainsKey(name))
            {
                Functions[name] = function;
            }
            else
            {
                Functions.Add(name, function);
            }
            OnCustomFunctionChanged(name);
            if (AutoRecalculation && (_calcEngineSuspended == 0))
            {
                Recalculate();
            }
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="imageStream">The image stream.</param>
        /// <returns>The added picture</returns>
        public Picture AddPicture(string name, Stream imageStream)
        {
            Picture item = new Picture(name, imageStream);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, Uri source)
        {
            Picture item = new Picture(name, source);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The image source.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, ImageSource source)
        {
            Picture item = new Picture(name, source);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="imageStream">The image source.</param>
        /// <param name="x">The x value.</param>
        /// <param name="y">The y value.</param>
        /// <returns>The added picture</returns>
        public Picture AddPicture(string name, Stream imageStream, double x, double y)
        {
            Picture item = new Picture(name, imageStream, x, y);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The uri source.</param>
        /// <param name="x">The x value.</param>
        /// <param name="y">The y value.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, Uri source, double x, double y)
        {
            Picture item = new Picture(name, source, x, y);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="x">The x value.</param>
        /// <param name="y">The y value.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, ImageSource source, double x, double y)
        {
            Picture item = new Picture(name, source, x, y);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="imageStream">The image source.</param>
        /// <param name="x">The x value.</param>
        /// <param name="y">The y value.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns>The added picture</returns>
        public Picture AddPicture(string name, Stream imageStream, double x, double y, double width, double height)
        {
            Picture item = new Picture(name, imageStream, x, y, width, height);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="imageStream">The image source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <returns>The added picture</returns>
        public Picture AddPicture(string name, Stream imageStream, int row, double rowOffset, int column, double columnOffset)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            return AddPicture(name, imageStream, point.X, point.Y);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The uri source.</param>
        /// <param name="x">The x value.</param>
        /// <param name="y">The y value.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, Uri source, double x, double y, double width, double height)
        {
            Picture item = new Picture(name, source, x, y, width, height);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, Uri source, int row, double rowOffset, int column, double columnOffset)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            return AddPicture(name, source, point.X, point.Y);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="x">The x value.</param>
        /// <param name="y">The y value.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, ImageSource source, double x, double y, double width, double height)
        {
            Picture item = new Picture(name, source, x, y, width, height);
            Pictures.Add(item);
            return item;
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, ImageSource source, int row, double rowOffset, int column, double columnOffset)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            return AddPicture(name, source, point.X, point.Y);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="imageStream">The image source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns>The added picture</returns>
        public Picture AddPicture(string name, Stream imageStream, int row, double rowOffset, int column, double columnOffset, double width, double height)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            return AddPicture(name, imageStream, point.X, point.Y, width, height);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="width">The picture width.</param>
        /// <param name="height">The picture height.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, Uri source, int row, double rowOffset, int column, double columnOffset, double width, double height)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            return AddPicture(name, source, point.X, point.Y, width, height);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, ImageSource source, int row, double rowOffset, int column, double columnOffset, double width, double height)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            return AddPicture(name, source, point.X, point.Y, width, height);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="imageStream">The image source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="endRow">The end row.</param>
        /// <param name="endRowOffset">The end row offset.</param>
        /// <param name="endColumn">The end column.</param>
        /// <param name="endColumnOffset">The end column offset.</param>
        /// <returns>The added picture</returns>
        public Picture AddPicture(string name, Stream imageStream, int row, double rowOffset, int column, double columnOffset, int endRow, double endRowOffset, int endColumn, double endColumnOffset)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            Point point2 = CalcLocation(endRow, endRowOffset, endColumn, endColumnOffset);
            double x = (point.X < point2.X) ? point.X : point2.X;
            double y = (point.Y < point2.Y) ? point.Y : point2.Y;
            double width = Math.Abs((double)(point.X - point2.X));
            double height = Math.Abs((double)(point.Y - point2.Y));
            return AddPicture(name, imageStream, x, y, width, height);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="endRow">The end row.</param>
        /// <param name="endRowOffset">The end row offset.</param>
        /// <param name="endColumn">The end column.</param>
        /// <param name="endColumnOffset">The end column offset.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, Uri source, int row, double rowOffset, int column, double columnOffset, int endRow, double endRowOffset, int endColumn, double endColumnOffset)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            Point point2 = CalcLocation(endRow, endRowOffset, endColumn, endColumnOffset);
            double x = (point.X < point2.X) ? point.X : point2.X;
            double y = (point.Y < point2.Y) ? point.Y : point2.Y;
            double width = Math.Abs((double)(point.X - point2.X));
            double height = Math.Abs((double)(point.Y - point2.Y));
            return AddPicture(name, source, x, y, width, height);
        }

        /// <summary>
        /// Adds the picture.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="source">The source.</param>
        /// <param name="row">The row.</param>
        /// <param name="rowOffset">The row offset.</param>
        /// <param name="column">The column.</param>
        /// <param name="columnOffset">The column offset.</param>
        /// <param name="endRow">The end row.</param>
        /// <param name="endRowOffset">The end row offset.</param>
        /// <param name="endColumn">The end column.</param>
        /// <param name="endColumnOffset">The end column offset.</param>
        /// <returns></returns>
        public Picture AddPicture(string name, ImageSource source, int row, double rowOffset, int column, double columnOffset, int endRow, double endRowOffset, int endColumn, double endColumnOffset)
        {
            Point point = CalcLocation(row, rowOffset, column, columnOffset);
            Point point2 = CalcLocation(endRow, endRowOffset, endColumn, endColumnOffset);
            double x = (point.X < point2.X) ? point.X : point2.X;
            double y = (point.Y < point2.Y) ? point.Y : point2.Y;
            double width = Math.Abs((double)(point.X - point2.X));
            double height = Math.Abs((double)(point.Y - point2.Y));
            return AddPicture(name, source, x, y, width, height);
        }

        /// <summary>
        /// Adds rows in this sheet.
        /// </summary>
        /// <param name="row">The row index at which to add rows.</param>
        /// <param name="count">The number of rows to add.</param>
        public void AddRows(int row, int count)
        {
            AddRows(row, count, SheetArea.Cells);
        }

        /// <summary>
        /// Adds rows in a specified sheet area.
        /// </summary>
        /// <param name="sheetArea">The sheet area in which to add rows.</param>
        /// <param name="row">The row index at which to add rows.</param>
        /// <param name="count">The number of rows to add.</param>
        public void AddRows(int row, int count, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)))
            {
                CalcExpressionHelper.ValidateIntersectedArrayFormula(CalcManager, row, -1, 1, -1, true);
                _cellsBlock.AddRows(row, count);
                _viewportRows.InsertRange(row, count);
                _viewportRowsStyles.InsertRange(row, count);
                _rowHeadersBlock.AddRows(row, count);
                _rowHeaderRowsStyles.InsertRange(row, count);
            }
            else if (sheetArea == SheetArea.ColumnHeader)
            {
                _columnHeadersBlock.AddRows(row, count);
                _columnHeaderRows.InsertRange(row, count);
                _columnHeaderRowsStyles.InsertRange(row, count);
            }
            OnRowsAdded(row, count, sheetArea);
            RaiseRowChanged(string.Empty, row, count, sheetArea, SheetChangedEventAction.RowsAdded);
        }

        /// <summary>
        /// Adds the row viewport.
        /// </summary>
        /// <param name="rowViewportIndex">The index of the row viewport.</param>
        /// <param name="viewportHeight">The height of the viewport.</param>
        public void AddRowViewport(int rowViewportIndex, double viewportHeight)
        {
            if (((rowViewportIndex > -1) && (rowViewportIndex < GetViewportInfo().RowViewportCount)) && (viewportHeight != 0.0))
            {
                ViewportInfo viewportInfo = GetViewportInfo();
                ViewportInfo info2 = new ViewportInfo(this, viewportInfo.RowViewportCount + 1, viewportInfo.ColumnViewportCount)
                {
                    ActiveColumnViewport = viewportInfo.ActiveColumnViewport
                };
                if (rowViewportIndex <= viewportInfo.ActiveRowViewport)
                {
                    info2.ActiveRowViewport = viewportInfo.ActiveRowViewport + 1;
                }
                else
                {
                    info2.ActiveRowViewport = viewportInfo.ActiveRowViewport;
                }
                viewportInfo.LeftColumns.CopyTo(info2.LeftColumns, 0);
                viewportInfo.ViewportWidth.CopyTo(info2.ViewportWidth, 0);
                for (int i = 0; i < rowViewportIndex; i++)
                {
                    info2.TopRows[i] = viewportInfo.TopRows[i];
                    info2.ViewportHeight[i] = viewportInfo.ViewportHeight[i];
                }
                for (int j = rowViewportIndex + 1; j < viewportInfo.RowViewportCount; j++)
                {
                    info2.TopRows[j + 1] = viewportInfo.TopRows[j];
                    info2.ViewportHeight[j + 1] = viewportInfo.ViewportHeight[j];
                }
                info2.TopRows[rowViewportIndex] = FrozenRowCount;
                info2.TopRows[rowViewportIndex + 1] = viewportInfo.TopRows[rowViewportIndex];
                if (viewportHeight == -1.0)
                {
                    double num3 = viewportInfo.ViewportHeight[rowViewportIndex];
                    double num4 = Math.Max((double)0.0, (double)(num3 - 6.0));
                    if (num3 == -1.0)
                    {
                        info2.ViewportHeight[rowViewportIndex] = -1.0;
                        info2.ViewportHeight[rowViewportIndex + 1] = -1.0;
                    }
                    else
                    {
                        info2.ViewportHeight[rowViewportIndex] = num4 / 2.0;
                        info2.ViewportHeight[rowViewportIndex + 1] = (info2.ViewportHeight[rowViewportIndex] > 0.0) ? (num4 - info2.ViewportHeight[rowViewportIndex]) : num3;
                    }
                }
                else
                {
                    info2.ViewportHeight[rowViewportIndex] = viewportHeight;
                    double num5 = viewportInfo.ViewportHeight[rowViewportIndex];
                    if (num5 != -1.0)
                    {
                        num5 = Math.Max((double)0.0, (double)((num5 - viewportHeight) - 6.0));
                    }
                    info2.ViewportHeight[rowViewportIndex + 1] = num5;
                }
                SetViewportInfo(info2);
                if (viewportHeight != -1.0)
                {
                    if (info2.ViewportHeight[rowViewportIndex + 1] == 0.0)
                    {
                        MergeRowViewports(rowViewportIndex + 1, rowViewportIndex);
                    }
                    else if (info2.ViewportHeight[rowViewportIndex] == 0.0)
                    {
                        MergeRowViewports(rowViewportIndex, rowViewportIndex + 1);
                    }
                }
                info2 = GetViewportInfo();
                info2.ViewportHeight[info2.RowViewportCount - 1] = -1.0;
            }
        }

        /// <summary>
        /// Adds a cell range to the selection.
        /// </summary>
        /// <param name="range">The selected range.</param>
        public void AddSelection(CellRange range)
        {
            AddSelection(range.Row, range.Column, range.RowCount, range.ColumnCount, true);
        }

        /// <summary>
        /// Adds a cell range to the selection.
        /// </summary>
        /// <param name="range">The selected range.</param>
        /// <param name="moveAnchorCell">Whether to reset the anchor cell to the top-left corner of the selection area.</param>
        public void AddSelection(CellRange range, bool moveAnchorCell)
        {
            AddSelection(range.Row, range.Column, range.RowCount, range.ColumnCount, moveAnchorCell);
        }

        /// <summary>
        /// Adds a cell or cells to the selection.
        /// </summary>
        /// <param name="row">The row index of the first cell to add.</param>
        /// <param name="column">The column index of the first cell to add.</param>
        /// <param name="rowCount">The number of rows to add.</param>
        /// <param name="columnCount">The number of columns to add.</param>
        public void AddSelection(int row, int column, int rowCount, int columnCount)
        {
            AddSelection(row, column, rowCount, columnCount, true);
        }

        /// <summary>
        /// Adds a cell or cells to the selection.
        /// </summary>
        /// <param name="row">The row index of the first cell to add.</param>
        /// <param name="column">The column index of the first cell to add.</param>
        /// <param name="rowCount">The number of rows to add.</param>
        /// <param name="columnCount">The number of columns to add.</param>
        /// <param name="moveAnchorCell">Whether to reset the anchor cell to the top-left corner of the selection area.</param>
        public void AddSelection(int row, int column, int rowCount, int columnCount, bool moveAnchorCell)
        {
            int num = row;
            int num2 = column;
            int num3 = rowCount;
            int num4 = columnCount;
            if ((num != -1) && (num2 != -1))
            {
                List<CellRange> spans = GetSpans();
                if ((spans != null) && (spans.Count > 0))
                {
                    CellRange range = CellRangeInflate(spans, new CellRange(row, column, rowCount, columnCount));
                    num = range.Row;
                    num2 = range.Column;
                    num3 = range.RowCount;
                    num4 = range.ColumnCount;
                }
            }
            Selection.AddSelection(num, num2, num3, num4, moveAnchorCell);
            if (moveAnchorCell)
            {
                RaisePropertyChanged("ActiveCell");
            }
        }

        /// <summary>
        /// Adds a span of cells in this sheet in the specified sheet area.
        /// </summary>        
        /// <param name="range">The cell range to span.</param> 
        /// <param name="sheetArea">The sheet area.</param>
        public void AddSpanCell(CellRange range, SheetArea sheetArea)
        {
            AddSpanCell(range.Row, range.Column, range.RowCount, range.ColumnCount, sheetArea);
        }

        /// <summary>
        /// Adds a span of cells in this sheet in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index of the cell at which to start the span.</param>
        /// <param name="column">The column index of the cell at which to start the span.</param>
        /// <param name="rowCount">The number of rows to span.</param>
        /// <param name="columnCount">The number of columns to span.</param>
        public void AddSpanCell(int row, int column, int rowCount, int columnCount)
        {
            AddSpanCell(row, column, rowCount, columnCount, SheetArea.Cells);
        }

        /// <summary>
        /// Adds a span of cells in this sheet in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index of the cell at which to start the span.</param>
        /// <param name="column">The column index of the cell at which to start the span.</param>
        /// <param name="rowCount">The number of rows to span.</param>
        /// <param name="columnCount">The number of columns to span.</param>
        /// <param name="sheetArea">The sheet area.</param>
        public void AddSpanCell(int row, int column, int rowCount, int columnCount, SheetArea sheetArea)
        {
            SheetSpanModelBase spanModel = GetSpanModel(sheetArea);
            if (spanModel != null)
            {
                int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
                int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
                spanModel.Add(modelRowFromViewRow, modelColumnFromViewColumn, rowCount, columnCount);
                if ((((sheetArea == SheetArea.Cells) && (ActiveRowIndex >= modelRowFromViewRow)) && ((ActiveRowIndex < (modelRowFromViewRow + rowCount)) && (ActiveColumnIndex >= modelColumnFromViewColumn))) && (ActiveColumnIndex < (modelColumnFromViewColumn + columnCount)))
                {
                    SetActiveCell(modelRowFromViewRow, modelColumnFromViewColumn, false);
                }
            }
        }

        /// <summary>
        /// Adds a range table with a specified data source to the sheet.
        /// </summary>
        /// <param name="tableName">The table name.</param>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="datasource">The data source for the table.</param>
        /// <returns>The new table instance.</returns>
        public SheetTable AddTable(string tableName, int row, int column, object datasource)
        {
            return AddTable(tableName, row, column, datasource, TableStyles.Medium2);
        }

        /// <summary>
        /// Adds a range table with a specified size to the sheet.
        /// </summary>
        /// <param name="tableName">The table name.</param>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="rowCount">The row count of the table.</param>
        /// <param name="columnCount">The column count of the table.</param>
        /// <returns>The new table instance.</returns>
        public SheetTable AddTable(string tableName, int row, int column, int rowCount, int columnCount)
        {
            return AddTable(tableName, row, column, rowCount, columnCount, (Workbook != null) ? Workbook.DefaultTableStyle : TableStyles.Medium2);
        }

        /// <summary>
        /// Adds a range table with a specified data source to the sheet.
        /// </summary>
        /// <param name="tableName">The table name.</param>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="datasource">The data source for the table.</param>
        /// <param name="tableStyle">The style of the table.</param>
        /// <returns>The new table instance.</returns>
        public SheetTable AddTable(string tableName, int row, int column, object datasource, TableStyle tableStyle)
        {
            SheetTable table2;
            if (string.IsNullOrEmpty(tableName))
            {
                throw new NotSupportedException(ResourceStrings.EmptyTableNameError);
            }
            if (_workbook != null)
            {
                Worksheet sheet = new Worksheet();
                if (_workbook.FindTable(tableName, out sheet) != null)
                {
                    throw new NotSupportedException(string.Format(ResourceStrings.TableAlreayExistInOtherWorksheet, (object[])new object[] { _name, sheet.Name }));
                }
            }
            else if (FindTable(tableName) != null)
            {
                throw new NotSupportedException(ResourceStrings.CurrentWorksheetHasTheSameTableError);
            }
            if ((row < 0) || (row >= RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidTableLocationRowIndex, (object[])new object[] { ((int)row), ((int)(RowCount - 1)) }));
            }
            if ((column < 0) || (column >= ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidTableLocationColumnIndex, (object[])new object[] { ((int)column), ((int)(ColumnCount - 1)) }));
            }
            if (datasource == null)
            {
                throw new ArgumentNullException(ResourceStrings.TableDataSourceCannotBeNull);
            }
            ConnectionBase dataConnection = null;
            try
            {
                dataConnection = ConnectionBuilder.Build(datasource);
                if (dataConnection == null)
                {
                    throw new NotSupportedException(ResourceStrings.TableNotSupportDataSouceError);
                }
                if (!dataConnection.CanOpen())
                {
                    throw new NotSupportedException(ResourceStrings.TableNotSupportDataSouceError);
                }
                dataConnection.Open();
                int rowCount = dataConnection.GetRecordCount() + 1;
                int length = dataConnection.DataFields.Length;
                if (RowCount < (row + rowCount))
                {
                    RowCount = row + rowCount;
                }
                if (ColumnCount < (column + length))
                {
                    ColumnCount = column + length;
                }
                if ((_tables != null) && Tables.Intersects(row, column, rowCount, length))
                {
                    throw new NotSupportedException(ResourceStrings.IntersectTableError);
                }
                SheetTable table = Tables.Add(new SheetTable(tableName, row, column, rowCount, length, dataConnection, tableStyle));
                table.LoadData();
                table.UpdateHeader(2);
                table.PropertyChanged += new PropertyChangedEventHandler(OnTablePropertyChanged);
                RaiseCellChanged("[Table]", row, column, rowCount, length, SheetArea.Cells);
                table2 = table;
            }
            catch
            {
                if (dataConnection != null)
                {
                    dataConnection.Close();
                }
                dataConnection = null;
                throw;
            }
            return table2;
        }

        /// <summary>
        /// Adds a range table with a specified size to the sheet. 
        /// </summary>
        /// <param name="tableName">The table name.</param>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="rowCount">The row count of the table.</param>
        /// <param name="columnCount">The column count of the table.</param>
        /// <param name="tableStyle">The style of the table.</param>
        /// <returns>The new table instance.</returns>
        public SheetTable AddTable(string tableName, int row, int column, int rowCount, int columnCount, TableStyle tableStyle)
        {
            if ((_tables != null) && Tables.Intersects(row, column, rowCount, columnCount))
            {
                throw new NotSupportedException(ResourceStrings.IntersectTableError);
            }
            SheetTable table = AddTable(tableName, row, column, new object[0], tableStyle);
            table.Resize(rowCount, columnCount);
            RaiseCellChanged("[Table]", row, column, rowCount, columnCount, SheetArea.Cells);
            return table;
        }

        /// <summary>
        /// Binds the column to the data source column for a sheet.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="dataField">The name of the data source column.</param>
        public void BindDataColumn(int column, string dataField)
        {
            if (_bindingManager == null)
            {
                _bindingManager = new SheetBindingManager(this);
            }
            _bindingManager.SetBoundField(column, dataField);
        }

        Point CalcLocation(int row, double rowOffset, int column, double columnOffset)
        {
            double actualColumnWidth = GetActualColumnWidth(column, SheetArea.Cells);
            if (columnOffset > actualColumnWidth)
            {
                columnOffset = actualColumnWidth;
            }
            double num2 = 0.0;
            for (int i = 0; i < column; i++)
            {
                actualColumnWidth = GetActualColumnWidth(i, SheetArea.Cells);
                num2 += actualColumnWidth;
            }
            double actualRowHeight = GetActualRowHeight(row, SheetArea.Cells);
            if (rowOffset > actualRowHeight)
            {
                rowOffset = actualRowHeight;
            }
            double num5 = 0.0;
            for (int j = 0; j < row; j++)
            {
                actualRowHeight = GetActualRowHeight(j, SheetArea.Cells);
                num5 += actualRowHeight;
            }
            return new Point(num2 + columnOffset, num5 + rowOffset);
        }

        /// <summary>
        /// Inflates cell range with the specified spans.
        /// </summary>
        /// <param name="spans">The cell spans in work sheet.</param>
        /// <param name="cellRange">The cell range which will be inflated.</param>
        /// <returns>Inflated cell range</returns>
        internal CellRange CellRangeInflate(List<CellRange> spans, CellRange cellRange)
        {
            if ((spans != null) && (spans.Count > 0))
            {
                for (int i = 0; i < spans.Count; i++)
                {
                    CellRange range = spans[i];
                    if (cellRange.Intersects(range.Row, range.Column, range.RowCount, range.ColumnCount))
                    {
                        spans.RemoveAt(i--);
                        return CellRangeInflate(spans, UnionCellRange(cellRange, range));
                    }
                }
            }
            return cellRange;
        }

        static void CheckArguments(Worksheet src, int fromRow, int fromColumn, Worksheet dest, int toRow, int toColumn, int rowCount, int columnCount)
        {
            if (src == null)
            {
                throw new ArgumentNullException("src");
            }
            if (dest == null)
            {
                throw new ArgumentNullException("dest");
            }
            if ((fromRow < -1) || (fromRow >= src.RowCount))
            {
                object[] args = new object[4];
                args[0] = "from";
                args[1] = "row";
                args[2] = ((int)fromRow).ToString();
                int num5 = src.RowCount - 1;
                args[3] = ((int)num5).ToString();
                throw new ArgumentOutOfRangeException(string.Format(ResourceStrings.CheckArgumentsInvalidRowColumn, args));
            }
            if ((fromColumn < -1) || (fromColumn >= src.ColumnCount))
            {
                object[] objArray2 = new object[4];
                objArray2[0] = "from";
                objArray2[1] = "column";
                objArray2[2] = ((int)fromColumn).ToString();
                int num6 = src.ColumnCount - 1;
                objArray2[3] = ((int)num6).ToString();
                throw new ArgumentOutOfRangeException(string.Format(ResourceStrings.CheckArgumentsInvalidRowColumn, objArray2));
            }
            if ((toRow < -1) || (toRow >= dest.RowCount))
            {
                object[] objArray3 = new object[4];
                objArray3[0] = "to";
                objArray3[1] = "row";
                objArray3[2] = ((int)toRow).ToString();
                int num7 = dest.RowCount - 1;
                objArray3[3] = ((int)num7).ToString();
                throw new ArgumentOutOfRangeException(string.Format(ResourceStrings.CheckArgumentsInvalidRowColumn, objArray3));
            }
            if ((toColumn < -1) || (toColumn >= dest.ColumnCount))
            {
                object[] objArray4 = new object[4];
                objArray4[0] = "to";
                objArray4[1] = "column";
                objArray4[2] = ((int)toColumn).ToString();
                int num8 = dest.ColumnCount - 1;
                objArray4[3] = ((int)num8).ToString();
                throw new ArgumentOutOfRangeException(string.Format(ResourceStrings.CheckArgumentsInvalidRowColumn, objArray4));
            }
            int num = fromColumn;
            if (fromColumn < 0)
            {
                num = 0;
                columnCount = src.ColumnCount;
            }
            int num2 = (toColumn < 0) ? 0 : toColumn;
            if (((columnCount < 1) || ((num + columnCount) > src.ColumnCount)) || ((num2 + columnCount) > dest.ColumnCount))
            {
                throw new ArgumentException(string.Format(ResourceStrings.InvalidRowColumntCount2, (object[])new object[] { "column", ((int)columnCount), ((int)Math.Min((int)(src.ColumnCount - num), (int)(dest.ColumnCount - num2))).ToString() }));
            }
            int num3 = fromRow;
            if (fromRow < 0)
            {
                num3 = 0;
                rowCount = src.RowCount;
            }
            int num4 = (toRow < 0) ? 0 : toRow;
            if (((rowCount < 1) || ((num3 + rowCount) > src.RowCount)) || ((num4 + rowCount) > dest.RowCount))
            {
                throw new ArgumentException(string.Format(ResourceStrings.InvalidRowColumntCount2, (object[])new object[] { "row", ((int)rowCount), ((int)Math.Min((int)(src.RowCount - num3), (int)(dest.RowCount - num4))).ToString() }));
            }
        }

        /// <summary>
        /// Clears the specified area.
        /// </summary>
        /// <param name="row">The start index.</param>
        /// <param name="column">The start column.</param>
        /// <param name="rowCount">The number of rows to clear.</param>
        /// <param name="columnCount">The number of columns to clear.</param>
        public void Clear(int row, int column, int rowCount, int columnCount)
        {
            Clear(row, column, rowCount, columnCount, SheetArea.Cells);
        }

        /// <summary>
        /// Clear the specified area.
        /// </summary>
        /// <param name="row">The start index.</param>
        /// <param name="column">The start column.</param>
        /// <param name="rowCount">The number or rows to clear.</param>
        /// <param name="columnCount">The number of columns to clear.</param>
        /// <param name="area">The area to clear.</param>
        public void Clear(int row, int column, int rowCount, int columnCount, SheetArea area)
        {
            Clear(row, column, rowCount, columnCount, area, StorageType.Sparkline | StorageType.Tag | StorageType.Data);
        }

        /// <summary>
        /// Clears the specified area.
        /// </summary>
        /// <param name="row">The start index.</param>
        /// <param name="column">The start column.</param>
        /// <param name="rowCount">The number or rows to clear.</param>
        /// <param name="columnCount">The number of columns to clear.</param>
        /// <param name="area">The area to clear.</param>
        /// <param name="type">The clear type.</param>
        public void Clear(int row, int column, int rowCount, int columnCount, SheetArea area, StorageType type)
        {
            SuspendCalcService();
            try
            {
                if (area == SheetArea.Cells)
                {
                    if ((type & StorageType.Data) == StorageType.Data)
                    {
                        if (_bindingManager != null)
                        {
                            ((IRangeSupport)_bindingManager).Clear(row, column, rowCount, columnCount);
                        }
                        CalcAxial.Clear(row, column, rowCount, columnCount);
                        if (_conditionalFormats != null)
                        {
                            _conditionalFormats.ClearCache();
                        }
                        if (_rowFilter != null)
                        {
                            ((IRangeSupport)_rowFilter).Clear(row, column, rowCount, columnCount);
                        }
                    }
                    if (((type & StorageType.Sparkline) == StorageType.Sparkline) && (_sparklineGroupManager != null))
                    {
                        _sparklineGroupManager.Clear(row, column, rowCount, columnCount);
                    }
                    if ((((type & StorageType.Data) == StorageType.Data) && (_tables != null)) && _tables.Intersects(row, column, rowCount, columnCount))
                    {
                        ((IRangeSupport)_tables).Clear(row, column, rowCount, columnCount);
                    }
                    if ((type & StorageType.Axis) == StorageType.Axis)
                    {
                        if (((row == -1) && (rowCount == -1)) && ((column == -1) && (columnCount == -1)))
                        {
                            _viewportRows.Clear();
                            _viewportColumns.Clear();
                        }
                        else if ((row == -1) && (rowCount == -1))
                        {
                            _viewportColumns.Clear(column, columnCount);
                        }
                        else if ((column == -1) && (columnCount == -1))
                        {
                            _viewportRows.Clear(row, rowCount);
                        }
                    }
                    ClearArrayFormula(this, row, column, rowCount, columnCount);
                }
                else if (area == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    if ((type & StorageType.Axis) == StorageType.Axis)
                    {
                        if (((row == -1) && (rowCount == -1)) && ((column == -1) && (columnCount == -1)))
                        {
                            _viewportRows.Clear();
                            _rowHeaderColumns.Clear();
                        }
                        else if ((row == -1) && (rowCount == -1))
                        {
                            _rowHeaderColumns.Clear(column, columnCount);
                        }
                        else if ((column == -1) && (columnCount == -1))
                        {
                            _viewportRows.Clear(row, rowCount);
                        }
                    }
                }
                else if ((area == SheetArea.ColumnHeader) && ((type & StorageType.Axis) == StorageType.Axis))
                {
                    if (((row == -1) && (rowCount == -1)) && ((column == -1) && (columnCount == -1)))
                    {
                        _viewportColumns.Clear();
                        _columnHeaderRows.Clear();
                    }
                    else if ((row == -1) && (rowCount == -1))
                    {
                        _viewportColumns.Clear(column, columnCount);
                    }
                    else if ((column == -1) && (columnCount == -1))
                    {
                        _columnHeaderRows.Clear(row, rowCount);
                    }
                }
                StorageBlock storage = GetStorage(area);
                if (storage != null)
                {
                    int num = (row == -1) ? 0 : row;
                    int num2 = (row == -1) ? storage.RowCount : rowCount;
                    int num3 = (column == -1) ? 0 : column;
                    int num4 = (column == -1) ? storage.ColumnCount : columnCount;
                    if (storage != null)
                    {
                        if ((row == -1) && (columnCount == -1))
                        {
                            storage.Clear(type);
                        }
                        else
                        {
                            storage.Clear(num, num3, num2, num4, type);
                        }
                    }
                    if (((type & StorageType.Style) == StorageType.Style) && ((row < 0) || (column < 0)))
                    {
                        if (column >= 0)
                        {
                            for (int j = 0; j < num4; j++)
                            {
                                SetStyleObject(-1, num3 + j, area, null);
                            }
                        }
                        else if (row >= 0)
                        {
                            for (int k = 0; k < num2; k++)
                            {
                                SetStyleObject(num + k, -1, area, null);
                            }
                        }
                        else
                        {
                            SetStyleObject(-1, -1, area, null);
                        }
                    }
                    for (int i = 0; i < num2; i++)
                    {
                        for (int m = 0; m < num4; m++)
                        {
                            num = row + i;
                            num3 = column + m;
                            if ((type & StorageType.Data) == StorageType.Data)
                            {
                                RaiseCellChanged("Value", num, num3, area);
                            }
                            if ((type & StorageType.Style) == StorageType.Style)
                            {
                                RaiseCellChanged("Style", num, num3, area);
                            }
                            if ((type & StorageType.Tag) == StorageType.Tag)
                            {
                                RaiseCellChanged("Tag", num, num3, area);
                            }
                            if ((type & StorageType.Sparkline) == StorageType.Sparkline)
                            {
                                RaiseCellChanged("Sparkline", num, num3, area);
                            }
                        }
                    }
                    if ((type & StorageType.Axis) == StorageType.Axis)
                    {
                        if ((row == -1) && (rowCount == -1))
                        {
                            RaiseColumnChanged("Axis", column, columnCount, area, SheetChangedEventAction.Updated);
                        }
                        if ((column == -1) && (columnCount == -1))
                        {
                            RaiseRowChanged("Axis", row, rowCount, area, SheetChangedEventAction.Updated);
                        }
                    }
                }
            }
            finally
            {
                ResumeCalcService();
            }
        }

        static void ClearArrayFormula(Worksheet sheet, int row, int column, int rowCount, int columnCount)
        {
            if (row < 0)
            {
                row = 0;
                rowCount = sheet.RowCount;
            }
            if (column < 0)
            {
                column = 0;
                columnCount = sheet.ColumnCount;
            }
            CalcRangeIdentity id = CalcExpressionHelper.CreateRangeExpressionByCount(row, column, rowCount, columnCount, false, false, false, false).GetId(0, 0) as CalcRangeIdentity;
            if (sheet.CalcManager.IsIsIntersectantWithArrayFormula(id))
            {
                Dictionary<CalcRangeIdentity, CalcRangeIdentity> dictionary = new Dictionary<CalcRangeIdentity, CalcRangeIdentity>();
                for (int i = row; i < rowCount; i++)
                {
                    for (int j = column; j < columnCount; j++)
                    {
                        CalcRangeIdentity identity3;
                        CalcCellIdentity identity2 = new CalcCellIdentity(row, column);
                        sheet._calcManager.GetArrayFormulaByCell(identity2, out identity3);
                        if ((identity3 != null) && !dictionary.ContainsKey(identity3))
                        {
                            if (!IsContains(id, identity3))
                            {
                                throw new Exception();
                            }
                            dictionary.Add(identity3, identity3);
                        }
                    }
                }
                foreach (CalcRangeIdentity identity4 in dictionary.Keys)
                {
                    int num3;
                    int num4;
                    int num5;
                    int num6;
                    sheet.LocalIdentity2RangeIndexes(identity4, out num3, out num4, out num5, out num6);
                    sheet.SetFormula(num3, num4, num5, num6, SheetArea.Cells, null, true);
                }
            }
        }

        /// <summary>
        /// Removes all user-defined custom functions (FunctionInfo object) on this sheet. 
        /// </summary>
        public void ClearCustomFunctions()
        {
            string[] strArray = Enumerable.ToArray<string>((IEnumerable<string>)Functions.Keys);
            Functions.Clear();
            foreach (string str in strArray)
            {
                OnCustomFunctionChanged(str);
            }
            if (AutoRecalculation && (_calcEngineSuspended == 0))
            {
                Recalculate();
            }
        }

        /// <summary>
        /// Clears the specified name in the name collection.
        /// </summary>
        /// <param name="name"></param>
        public void ClearCustomName(string name)
        {
            Names.Remove(name);
            OnCustomNameChanged(name, true);
            if (AutoRecalculation && (_calcEngineSuspended == 0))
            {
                Recalculate();
            }
        }

        /// <summary>
        /// Removes all custom names on this sheet. 
        /// </summary>
        public void ClearCustomNames()
        {
            string[] names = Names.GetNames();
            Names.Clear();
            foreach (string str in names)
            {
                OnCustomNameChanged(str, true);
            }
            if (AutoRecalculation && (_calcEngineSuspended == 0))
            {
                Recalculate();
            }
        }

        static void ClearFormula(Worksheet sheet, int row, int column, int rowCount, int columnCount)
        {
            if (row < 0)
            {
                row = 0;
                rowCount = sheet.RowCount;
            }
            if (column < 0)
            {
                column = 0;
                columnCount = sheet.ColumnCount;
            }
            for (int i = row; i < rowCount; i++)
            {
                for (int j = column; j < columnCount; j++)
                {
                    sheet.SetFormula(i, j, null);
                }
            }
        }

        /// <summary>
        /// Clears the specified selection range from the selection list, if one exists. 
        /// </summary>
        /// <param name="range">The cell range to be removed from the selection.</param>
        public void ClearSelection(CellRange range)
        {
            ClearSelection(range.Row, range.Column, range.RowCount, range.ColumnCount);
        }

        /// <summary>
        /// Clears the specified selection range from the selection list, if one exists. 
        /// </summary>
        /// <param name="row">The row index of the cell at which to start.</param>
        /// <param name="column">The column index of the cell at which to start.</param>
        /// <param name="rowCount">The number of rows to deselect.</param>
        /// <param name="columnCount">The number of columns to deselect.</param>
        public void ClearSelection(int row, int column, int rowCount, int columnCount)
        {
            Selection.RemoveSelection(row, column, rowCount, columnCount);
        }

        /// <summary>
        /// Clears the selections.
        /// </summary>
        public void ClearSelections()
        {
            Selection.ClearSelection();
        }

        /// <summary>
        /// Clears the span that contains a specified anchor cell from a range of cells in the specified sheet area.
        /// </summary> 
        /// <param name="sheetArea">The sheet area.</param>
        public void ClearSpanCells(SheetArea sheetArea)
        {
            SheetSpanModelBase spanModel = GetSpanModel(sheetArea);
            if (spanModel != null)
            {
                spanModel.Clear();
            }
        }

        static List<AxisInfo> CloneAxisInfo(SparseArray<AxisInfo> srcInfo, int fromIndex, int count)
        {
            List<AxisInfo> list = new List<AxisInfo>();
            for (int i = 0; i < count; i++)
            {
                AxisInfo info = srcInfo[fromIndex + i];
                if (info == null)
                {
                    list.Add(null);
                }
                else
                {
                    AxisInfo info2 = new AxisInfo();
                    info2.CopyFrom(info);
                    list.Add(info2);
                }
            }
            return list;
        }

        static List<object> CloneAxisStyle(SparseArray<object> srcStyle, int fromIndex, int count)
        {
            List<object> list = new List<object>();
            for (int i = 0; i < count; i++)
            {
                list.Add(CloneObject(srcStyle[fromIndex + i]));
            }
            return list;
        }

        static void CloneFloatingObjectToSheet(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount, out FloatingObject[] sourceRangeFloatingObjects)
        {
            int row = srcRow;
            int column = srcColumn;
            int rowCount = moveRowCount;
            int columnCount = moveColumnCount;
            if (srcRow < 0)
            {
                row = 0;
                rowCount = src.RowCount;
            }
            if (column < 0)
            {
                column = 0;
                columnCount = src.ColumnCount;
            }
            CellRange range = new CellRange(row, column, rowCount, columnCount);
            sourceRangeFloatingObjects = GetFloatingObjectsInRange(range, src);
            if ((sourceRangeFloatingObjects != null) && (sourceRangeFloatingObjects.Length != 0))
            {
                double num5 = 0.0;
                double num6 = 0.0;
                for (int i = 0; i < srcRow; i++)
                {
                    num6 += src.GetActualRowHeight(i, SheetArea.Cells);
                }
                for (int j = 0; j < srcColumn; j++)
                {
                    num5 += src.GetActualColumnWidth(j, SheetArea.Cells);
                }
                Point[] pointArray = new Point[sourceRangeFloatingObjects.Length];
                for (int k = 0; k < sourceRangeFloatingObjects.Length; k++)
                {
                    pointArray[k] = new Point(sourceRangeFloatingObjects[k].Location.X - num5, sourceRangeFloatingObjects[k].Location.Y - num6);
                }
                double num10 = 0.0;
                double num11 = 0.0;
                for (int m = 0; m < destRow; m++)
                {
                    num11 += dest.GetActualRowHeight(m, SheetArea.Cells);
                }
                for (int n = 0; n < destColumn; n++)
                {
                    num10 += dest.GetActualColumnWidth(n, SheetArea.Cells);
                }
                Point[] pointArray2 = new Point[sourceRangeFloatingObjects.Length];
                for (int num14 = 0; num14 < sourceRangeFloatingObjects.Length; num14++)
                {
                    pointArray2[num14] = new Point(num10 + pointArray[num14].X, num11 + pointArray[num14].Y);
                }
                int count = dest.Charts.Count;
                int num16 = dest.Pictures.Count;
                int num17 = dest.FloatingObjects.Count;
                List<SpreadChart> list = new List<SpreadChart>();
                List<FloatingObject> list2 = new List<FloatingObject>();
                List<Picture> list3 = new List<Picture>();
                for (int num18 = 0; num18 < sourceRangeFloatingObjects.Length; num18++)
                {
                    FloatingObject item = sourceRangeFloatingObjects[num18].Clone() as FloatingObject;
                    item.Location = pointArray2[num18];
                    if (item is SpreadChart)
                    {
                        string str = "Chart";
                        while (dest.FindChart(str + ((int)count)) != null)
                        {
                            count++;
                        }
                        item.Name = str + ((int)count);
                        dest.Charts.Add(item as SpreadChart);
                        list.Add(item as SpreadChart);
                    }
                    else if (item is Picture)
                    {
                        string str2 = "Picture";
                        while (dest.FindPicture(str2 + ((int)num16)) != null)
                        {
                            num16++;
                        }
                        item.Name = str2 + ((int)num16);
                        dest.Pictures.Add(item as Picture);
                        list3.Add(item as Picture);
                    }
                    else
                    {
                        string str3 = "FloatingObject";
                        while (dest.FindFloatingObject(str3 + ((int)num17)) != null)
                        {
                            num17++;
                        }
                        item.Name = str3 + ((int)num17);
                        dest.FloatingObjects.Add(item);
                        list2.Add(item);
                    }
                    num17++;
                }
                if (list.Count > 0)
                {
                    foreach (SpreadChart chart in list)
                    {
                        dest.RaiseChartChanged(chart, ChartArea.All, "");
                    }
                }
                if (list2.Count > 0)
                {
                    foreach (FloatingObject obj3 in list2)
                    {
                        dest.RaiseFloatingObjectChanged(obj3, "");
                    }
                }
                if (list3.Count > 0)
                {
                    foreach (Picture picture in list3)
                    {
                        dest.RaisePictureChanged(picture, "");
                    }
                }
            }
        }

        internal static object CloneObject(object obj)
        {
            if (obj is StyleInfo)
            {
                StyleInfo info = new StyleInfo();
                info.CopyFromInternal(obj, true, true);
                return info;
            }
            if (obj is AxisInfo)
            {
                AxisInfo info2 = new AxisInfo();
                info2.CopyFrom((AxisInfo)obj);
                return info2;
            }
            if (obj is ICloneable)
            {
                return (obj as ICloneable).Clone();
            }
            if (obj is Sparkline)
            {
                return (obj as Sparkline).Clone();
            }
            return obj;
        }

        void ComposeStyle(object source, StyleInfo dest)
        {
            StyleInfo info = null;
            if (_buff == null)
            {
                _buff = new StyleInfo();
                _buff.suspendEvents = true;
            }
            else
            {
                _buff.Init();
            }
            StyleInfo info2 = source as StyleInfo;
            if (info2 != null)
            {
                _buff.Compose(info2);
                info = Name2Style(info2.Parent);
            }
            else
            {
                info2 = Name2Style((string)(source as string));
                if (info2 != null)
                {
                    _buff.Compose(info2);
                    info = Name2Style(info2.Parent);
                }
            }
            while (info != null)
            {
                _buff.Merge(info);
                info = Name2Style(info.Parent);
            }
            try
            {
                dest.suspendEvents = true;
                dest.Compose(_buff);
            }
            finally
            {
                dest.suspendEvents = false;
            }
        }

        static void CopyColumnAxis(Worksheet src, int srcColumn, Worksheet dest, int destColumn, int copyColumnCount, CopyToOption option)
        {
            int fromIndex = srcColumn;
            int toIndex = destColumn;
            int count = copyColumnCount;
            if (srcColumn < 0)
            {
                fromIndex = 0;
                count = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destColumn < 0)
            {
                toIndex = 0;
            }
            SparseArray<AxisInfo> columnsStorage = src.GetColumnsStorage(SheetArea.Cells);
            if (columnsStorage != null)
            {
                SetAxisInfo(CloneAxisInfo(columnsStorage, fromIndex, count), dest.GetColumnsStorage(SheetArea.Cells), toIndex);
            }
            if ((option & CopyToOption.Style) > ((CopyToOption)0))
            {
                SparseArray<object> columnAxisStyles = src.GetColumnAxisStyles(SheetArea.Cells);
                if (columnAxisStyles != null)
                {
                    SetAxisStyle(CloneAxisStyle(columnAxisStyles, fromIndex, count), dest.GetColumnAxisStyles(SheetArea.Cells), toIndex);
                }
                SparseArray<object> srcStyle = src.GetColumnAxisStyles(SheetArea.ColumnHeader);
                if (srcStyle != null)
                {
                    SetAxisStyle(CloneAxisStyle(srcStyle, fromIndex, count), dest.GetColumnAxisStyles(SheetArea.ColumnHeader), toIndex);
                }
            }
            dest.RaiseColumnChanged("Axis", destColumn, count, SheetArea.Cells, SheetChangedEventAction.Updated);
        }

        static void CopyColumnRangeGroup(Worksheet src, int srcColumn, Worksheet dest, int destColumn, int copyColumnCount)
        {
            int from = srcColumn;
            int to = destColumn;
            int count = copyColumnCount;
            if (from < 0)
            {
                from = 0;
                count = src.ColumnCount;
            }
            if (destColumn < 0)
            {
                to = 0;
            }
            if (!object.ReferenceEquals(src, dest))
            {
                CrossSheetCopyRangeGroup(src.ColumnRangeGroup, from, dest.ColumnRangeGroup, to, count);
            }
            else
            {
                IMultipleSupport columnRangeGroup = src.ColumnRangeGroup;
                if (columnRangeGroup != null)
                {
                    columnRangeGroup.Copy(from, to, count);
                }
            }
        }

        static void CopyFloatingObjects(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            FloatingObject[] sourceRangeFloatingObjects = null;
            CloneFloatingObjectToSheet(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount, out sourceRangeFloatingObjects);
        }

        static void CopyFormula(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int copyRowCount, int copyColumnCount)
        {
            int fromRow = srcRow;
            int fromColumn = srcColumn;
            int toRow = destRow;
            int toColumn = destColumn;
            int rowCount = copyRowCount;
            int columnCount = copyColumnCount;
            if (srcRow < 0)
            {
                fromRow = 0;
                rowCount = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                fromColumn = 0;
                columnCount = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                toRow = 0;
            }
            if (destColumn < 0)
            {
                toColumn = 0;
            }
            if (!object.ReferenceEquals(src, dest))
            {
                ICrossSheetRangeSupport calcAxial = dest.CalcAxial;
                if (calcAxial != null)
                {
                    calcAxial.Copy(src, fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
            else
            {
                IRangeSupport support2 = src.CalcAxial;
                if ((support2 != null) && src.HasFormula)
                {
                    support2.Copy(fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
        }

        static void CopyRowAxis(Worksheet src, int srcRow, Worksheet dest, int destRow, int copyRowCount, CopyToOption option)
        {
            int fromIndex = srcRow;
            int toIndex = destRow;
            int count = copyRowCount;
            if (srcRow < 0)
            {
                fromIndex = 0;
                count = Math.Min(src.RowCount, dest.RowCount);
            }
            if (destRow < 0)
            {
                toIndex = 0;
            }
            SparseArray<AxisInfo> rowsStorage = src.GetRowsStorage(SheetArea.Cells);
            if (rowsStorage != null)
            {
                SetAxisInfo(CloneAxisInfo(rowsStorage, fromIndex, count), dest.GetRowsStorage(SheetArea.Cells), toIndex);
            }
            if ((option & CopyToOption.Style) > ((CopyToOption)0))
            {
                SparseArray<object> rowAxisStyles = src.GetRowAxisStyles(SheetArea.Cells);
                if (rowAxisStyles != null)
                {
                    SetAxisStyle(CloneAxisStyle(rowAxisStyles, fromIndex, count), dest.GetRowAxisStyles(SheetArea.Cells), toIndex);
                }
                SparseArray<object> srcStyle = src.GetRowAxisStyles(SheetArea.CornerHeader | SheetArea.RowHeader);
                if (srcStyle != null)
                {
                    SetAxisStyle(CloneAxisStyle(srcStyle, fromIndex, count), dest.GetRowAxisStyles(SheetArea.CornerHeader | SheetArea.RowHeader), toIndex);
                }
            }
            dest.RaiseRowChanged("Axis", destRow, count, SheetArea.Cells, SheetChangedEventAction.Updated);
        }

        static void CopyRowRangeGroup(Worksheet src, int srcRow, Worksheet dest, int destRow, int copyRowCount)
        {
            int from = srcRow;
            int to = destRow;
            int count = copyRowCount;
            if (srcRow < 0)
            {
                from = 0;
                count = src.RowCount;
            }
            if (destRow < 0)
            {
                to = 0;
            }
            if (!object.ReferenceEquals(src, dest))
            {
                CrossSheetCopyRangeGroup(src.RowRangeGroup, from, dest.RowRangeGroup, to, count);
            }
            else
            {
                IMultipleSupport rowRangeGroup = src.RowRangeGroup;
                if (rowRangeGroup != null)
                {
                    rowRangeGroup.Copy(from, to, count);
                }
            }
        }

        static void CopySheetInfo(Worksheet src, Worksheet dest, CopyToOption option)
        {
            if (!object.ReferenceEquals(src, dest))
            {
                if ((option & CopyToOption.Style) > ((CopyToOption)0))
                {
                    dest.DefaultStyle = CloneObject(src.DefaultStyle) as StyleInfo;
                    dest.ColumnHeader.DefaultStyle = CloneObject(src.ColumnHeader.DefaultStyle) as StyleInfo;
                    dest.RowHeader.DefaultStyle = CloneObject(src.RowHeader.DefaultStyle) as StyleInfo;
                }
                dest.DefaultColumnWidth = src.DefaultColumnWidth;
                dest.DefaultRowHeight = src.DefaultRowHeight;
                dest.ColumnHeader.DefaultRowHeight = src.ColumnHeader.DefaultRowHeight;
                dest.RowHeader.DefaultColumnWidth = src.RowHeader.DefaultColumnWidth;
            }
        }

        static void CopySpan(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int copyRowCount, int copyColumnCount)
        {
            int fromRow = srcRow;
            int fromColumn = srcColumn;
            int toRow = destRow;
            int toColumn = destColumn;
            int rowCount = copyRowCount;
            int columnCount = copyColumnCount;
            if (srcRow < 0)
            {
                fromRow = 0;
                rowCount = src.RowCount;
            }
            if (fromColumn < 0)
            {
                fromColumn = 0;
                columnCount = src.ColumnCount;
            }
            if (destRow < 0)
            {
                toRow = 0;
            }
            if (destColumn < 0)
            {
                toColumn = 0;
            }
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                if (flag)
                {
                    CrossSheetCopySpans(src.ColumnHeaderSpanModel, -1, fromColumn, dest.ColumnHeaderSpanModel, -1, toColumn, -1, columnCount);
                }
                else
                {
                    IRangeSupport spanModel = src.GetSpanModel(SheetArea.ColumnHeader);
                    if (spanModel != null)
                    {
                        spanModel.Copy(-1, fromColumn, -1, toColumn, -1, columnCount);
                    }
                }
            }
            if (srcColumn < 0)
            {
                if (flag)
                {
                    CrossSheetCopySpans(src.RowHeaderSpanModel, fromRow, -1, dest.RowHeaderSpanModel, toRow, -1, rowCount, -1);
                }
                else
                {
                    IRangeSupport support2 = src.GetSpanModel(SheetArea.CornerHeader | SheetArea.RowHeader);
                    if (support2 != null)
                    {
                        support2.Copy(fromRow, -1, toRow, -1, rowCount, -1);
                    }
                }
            }
            if (flag)
            {
                CrossSheetCopySpans(src.SpanModel, fromRow, fromColumn, dest.SpanModel, toRow, toColumn, rowCount, columnCount);
            }
            else
            {
                IRangeSupport support3 = src.GetSpanModel(SheetArea.Cells);
                if (support3 != null)
                {
                    support3.Copy(fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
        }

        static void CopySparkline(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int copyRowCount, int copyColumnCount)
        {
            int fromRow = srcRow;
            int fromColumn = srcColumn;
            int toRow = destRow;
            int toColumn = destColumn;
            int rowCount = copyRowCount;
            int columnCount = copyColumnCount;
            if (srcRow < 0)
            {
                fromRow = 0;
                rowCount = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                fromColumn = 0;
                columnCount = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                toRow = 0;
            }
            if (destColumn < 0)
            {
                toColumn = 0;
            }
            if (!object.ReferenceEquals(src, dest))
            {
                ICrossSheetRangeSupport sheetSparklineGroupManager = dest.SheetSparklineGroupManager;
                if (sheetSparklineGroupManager != null)
                {
                    sheetSparklineGroupManager.Copy(src, fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
            else
            {
                IRangeSupport support2 = src.SheetSparklineGroupManager;
                if (support2 != null)
                {
                    support2.Copy(fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
        }

        static void CopyStyle(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int copyRowCount, int copyColumnCount)
        {
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                int num = srcColumn;
                int num2 = destColumn;
                int rowCount = src.ColumnHeader.RowCount;
                int columnCount = copyColumnCount;
                if (srcColumn < 0)
                {
                    num = 0;
                    columnCount = src.ColumnCount;
                }
                if (destColumn < 0)
                {
                    num2 = 0;
                }
                StorageBlock block = src.GetStorage(SheetArea.ColumnHeader);
                if (flag)
                {
                    for (int i = 0; i < rowCount; i++)
                    {
                        for (int j = 0; j < columnCount; j++)
                        {
                            StyleInfo info = src.GetCompositeStyle(i, num + j, SheetArea.ColumnHeader, block);
                            if (info == null)
                            {
                                dest.SetStyleInfo(i, num2 + j, SheetArea.ColumnHeader, null);
                            }
                            else
                            {
                                dest.SetStyleInfo(i, num2 + j, SheetArea.ColumnHeader, (StyleInfo)CloneObject(info));
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<StyleInfo> matrix = new DataMatrix<StyleInfo>(rowCount, columnCount);
                    for (int k = 0; k < rowCount; k++)
                    {
                        for (int n = 0; n < columnCount; n++)
                        {
                            StyleInfo info2 = src.GetCompositeStyle(k, num + n, SheetArea.ColumnHeader, block);
                            if (info2 != null)
                            {
                                matrix.SetValue(k, n, (StyleInfo)CloneObject(info2));
                            }
                        }
                    }
                    for (int m = 0; m < rowCount; m++)
                    {
                        for (int num10 = 0; num10 < columnCount; num10++)
                        {
                            dest.SetStyleInfo(m, num2 + num10, SheetArea.ColumnHeader, matrix.GetValue(m, num10));
                        }
                    }
                }
            }
            if (srcColumn < 0)
            {
                int num11 = srcRow;
                int num12 = destRow;
                int num13 = copyRowCount;
                int num14 = src.RowHeader.ColumnCount;
                if (srcRow < 0)
                {
                    num11 = 0;
                    num13 = src.RowCount;
                }
                if (destRow < 0)
                {
                    num12 = 0;
                }
                StorageBlock block2 = src.GetStorage(SheetArea.CornerHeader | SheetArea.RowHeader);
                if (flag)
                {
                    for (int num15 = 0; num15 < num13; num15++)
                    {
                        for (int num16 = 0; num16 < num14; num16++)
                        {
                            StyleInfo info3 = src.GetCompositeStyle(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, block2);
                            if (info3 == null)
                            {
                                dest.SetStyleInfo(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                            }
                            else
                            {
                                dest.SetStyleInfo(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, (StyleInfo)CloneObject(info3));
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<StyleInfo> matrix2 = new DataMatrix<StyleInfo>(num13, num14);
                    for (int num17 = 0; num17 < num13; num17++)
                    {
                        for (int num18 = 0; num18 < num14; num18++)
                        {
                            StyleInfo info4 = src.GetCompositeStyle(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader, block2);
                            if (info4 != null)
                            {
                                matrix2.SetValue(num17, num18, (StyleInfo)CloneObject(info4));
                            }
                        }
                    }
                    for (int num19 = 0; num19 < num13; num19++)
                    {
                        for (int num20 = 0; num20 < num14; num20++)
                        {
                            dest.SetStyleInfo(num12 + num19, num20, SheetArea.CornerHeader | SheetArea.RowHeader, matrix2.GetValue(num19, num20));
                        }
                    }
                }
            }
            int num21 = srcRow;
            int num22 = srcColumn;
            int num23 = destRow;
            int num24 = destColumn;
            int row = copyRowCount;
            int column = copyColumnCount;
            if (srcRow < 0)
            {
                num21 = 0;
                row = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                num22 = 0;
                column = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                num23 = 0;
            }
            if (destColumn < 0)
            {
                num24 = 0;
            }
            StorageBlock storage = src.GetStorage(SheetArea.Cells);
            if (flag)
            {
                for (int num27 = 0; num27 < row; num27++)
                {
                    for (int num28 = 0; num28 < column; num28++)
                    {
                        StyleInfo info5 = src.GetCompositeStyle(num21 + num27, num22 + num28, SheetArea.Cells, storage);
                        if (info5 == null)
                        {
                            dest.SetStyleInfo(num23 + num27, num24 + num28, SheetArea.Cells, null);
                        }
                        else
                        {
                            dest.SetStyleInfo(num23 + num27, num24 + num28, SheetArea.Cells, (StyleInfo)CloneObject(info5));
                        }
                    }
                }
            }
            else
            {
                DataMatrix<StyleInfo> matrix3 = new DataMatrix<StyleInfo>(row, column);
                for (int num29 = 0; num29 < row; num29++)
                {
                    for (int num30 = 0; num30 < column; num30++)
                    {
                        StyleInfo info6 = src.GetCompositeStyle(num21 + num29, num22 + num30, SheetArea.Cells, storage);
                        if (info6 != null)
                        {
                            matrix3.SetValue(num29, num30, (StyleInfo)CloneObject(info6));
                        }
                    }
                }
                for (int num31 = 0; num31 < row; num31++)
                {
                    for (int num32 = 0; num32 < column; num32++)
                    {
                        dest.SetStyleInfo(num23 + num31, num24 + num32, SheetArea.Cells, matrix3.GetValue(num31, num32));
                    }
                }
            }
        }

        static void CopyTag(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int copyRowCount, int copyColumnCount)
        {
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                int num = srcColumn;
                int num2 = destColumn;
                int rowCount = src.ColumnHeader.RowCount;
                int columnCount = copyColumnCount;
                if (srcColumn < 0)
                {
                    num = 0;
                    columnCount = src.ColumnCount;
                }
                if (destColumn < 0)
                {
                    num2 = 0;
                }
                if (flag)
                {
                    for (int i = 0; i < rowCount; i++)
                    {
                        for (int j = 0; j < columnCount; j++)
                        {
                            object obj2 = src.GetTag(i, num + j, SheetArea.ColumnHeader);
                            if (obj2 == null)
                            {
                                dest.SetTag(i, num2 + j, SheetArea.ColumnHeader, null);
                            }
                            else
                            {
                                dest.SetTag(i, num2 + j, SheetArea.ColumnHeader, CloneObject(obj2));
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix = new DataMatrix<object>(rowCount, columnCount);
                    for (int k = 0; k < rowCount; k++)
                    {
                        for (int n = 0; n < columnCount; n++)
                        {
                            object obj3 = src.GetTag(k, num + n, SheetArea.ColumnHeader);
                            if (obj3 != null)
                            {
                                matrix.SetValue(k, n, CloneObject(obj3));
                            }
                        }
                    }
                    for (int m = 0; m < rowCount; m++)
                    {
                        for (int num10 = 0; num10 < columnCount; num10++)
                        {
                            dest.SetTag(m, num2 + num10, SheetArea.ColumnHeader, matrix.GetValue(m, num10));
                        }
                    }
                }
            }
            if (srcColumn < 0)
            {
                int num11 = srcRow;
                int num12 = destRow;
                int num13 = copyRowCount;
                int num14 = src.RowHeader.ColumnCount;
                if (srcRow < 0)
                {
                    num11 = 0;
                    num13 = src.RowCount;
                }
                if (destRow < 0)
                {
                    num12 = 0;
                }
                if (flag)
                {
                    for (int num15 = 0; num15 < num13; num15++)
                    {
                        for (int num16 = 0; num16 < num14; num16++)
                        {
                            object obj4 = src.GetTag(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj4 == null)
                            {
                                dest.SetTag(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                            }
                            else
                            {
                                dest.SetTag(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, CloneObject(obj4));
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix2 = new DataMatrix<object>(num13, num14);
                    for (int num17 = 0; num17 < num13; num17++)
                    {
                        for (int num18 = 0; num18 < num14; num18++)
                        {
                            object obj5 = src.GetTag(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj5 != null)
                            {
                                matrix2.SetValue(num17, num18, CloneObject(obj5));
                            }
                        }
                    }
                    for (int num19 = 0; num19 < num13; num19++)
                    {
                        for (int num20 = 0; num20 < num14; num20++)
                        {
                            dest.SetTag(num12 + num19, num20, SheetArea.CornerHeader | SheetArea.RowHeader, matrix2.GetValue(num19, num20));
                        }
                    }
                }
            }
            int num21 = srcRow;
            int num22 = srcColumn;
            int num23 = destRow;
            int num24 = destColumn;
            int row = copyRowCount;
            int column = copyColumnCount;
            if (srcRow < 0)
            {
                num21 = 0;
                row = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                num22 = 0;
                column = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                num23 = 0;
            }
            if (destColumn < 0)
            {
                num24 = 0;
            }
            if (flag)
            {
                for (int num27 = 0; num27 < row; num27++)
                {
                    for (int num28 = 0; num28 < column; num28++)
                    {
                        object obj6 = src.GetTag(num21 + num27, num22 + num28, SheetArea.Cells);
                        if (obj6 == null)
                        {
                            dest.SetTag(num23 + num27, num24 + num28, SheetArea.Cells, null);
                        }
                        else
                        {
                            dest.SetTag(num23 + num27, num24 + num28, SheetArea.Cells, CloneObject(obj6));
                        }
                    }
                }
            }
            else
            {
                DataMatrix<object> matrix3 = new DataMatrix<object>(row, column);
                for (int num29 = 0; num29 < row; num29++)
                {
                    for (int num30 = 0; num30 < column; num30++)
                    {
                        object obj7 = src.GetTag(num21 + num29, num22 + num30, SheetArea.Cells);
                        if (obj7 != null)
                        {
                            matrix3.SetValue(num29, num30, CloneObject(obj7));
                        }
                    }
                }
                for (int num31 = 0; num31 < row; num31++)
                {
                    for (int num32 = 0; num32 < column; num32++)
                    {
                        dest.SetTag(num23 + num31, num24 + num32, SheetArea.Cells, matrix3.GetValue(num31, num32));
                    }
                }
            }
        }

        /// <summary>
        /// Copies data from one range to another.
        /// </summary>
        /// <param name="fromRow">The source row.</param>
        /// <param name="fromColumn">The source column.</param>
        /// <param name="toRow">The target row.</param>
        /// <param name="toColumn">The target column.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <param name="option">The copy option.</param>
        public void CopyTo(int fromRow, int fromColumn, int toRow, int toColumn, int rowCount, int columnCount, CopyToOption option)
        {
            CopyTo(this, fromRow, fromColumn, this, toRow, toColumn, rowCount, columnCount, option);
        }

        internal static void CopyTo(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int copyRowCount, int copyColumnCount, CopyToOption option)
        {
            CheckArguments(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
            src.SuspendCalcService();
            dest.SuspendCalcService();
            try
            {
                if (src._conditionalFormats != null)
                {
                    src._conditionalFormats.ClearCache();
                }
                if ((option & CopyToOption.Value) > ((CopyToOption)0))
                {
                    CopyValue(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
                    if ((option & CopyToOption.Formula) == ((CopyToOption)0))
                    {
                        ClearFormula(dest, destRow, destColumn, copyRowCount, copyColumnCount);
                    }
                }
                if ((option & CopyToOption.Formula) > ((CopyToOption)0))
                {
                    CopyFormula(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
                }
            }
            finally
            {
                src.ResumeCalcService();
                dest.ResumeCalcService();
            }
            if ((option & CopyToOption.Style) > ((CopyToOption)0))
            {
                CopyStyle(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
            }
            if ((option & CopyToOption.Tag) > ((CopyToOption)0))
            {
                CopyTag(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
            }
            if ((option & CopyToOption.Sparkline) > ((CopyToOption)0))
            {
                CopySparkline(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
            }
            if ((option & CopyToOption.RangeGroup) > ((CopyToOption)0))
            {
                if (srcRow < 0)
                {
                    CopyColumnRangeGroup(src, srcColumn, dest, destColumn, copyColumnCount);
                }
                if (srcColumn < 0)
                {
                    CopyRowRangeGroup(src, srcRow, dest, destRow, copyRowCount);
                }
            }
            if ((option & CopyToOption.Span) > ((CopyToOption)0))
            {
                if (src.IsEventSuspend())
                {
                    src.SuspendSpanModelEvent();
                }
                if (dest.IsEventSuspend())
                {
                    dest.SuspendSpanModelEvent();
                }
                try
                {
                    CopySpan(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
                }
                finally
                {
                    if (src.IsEventSuspend())
                    {
                        src.ResumeSpanModelEvent();
                    }
                    if (dest.IsEventSuspend())
                    {
                        dest.ResumeSpanModelEvent();
                    }
                }
            }
            if ((option & CopyToOption.FloatingObject) > ((CopyToOption)0))
            {
                src.SuspendChartEvents();
                dest.SuspendChartEvents();
                src.SuspendFloatingObjectEvents();
                dest.SuspendFloatingObjectEvents();
                src.SuspendPicturesEvents();
                dest.SuspendPicturesEvents();
                try
                {
                    CopyFloatingObjects(src, srcRow, srcColumn, dest, destRow, destColumn, copyRowCount, copyColumnCount);
                }
                finally
                {
                    src.ResumeChartEvents();
                    dest.ResumeChartEvents();
                    src.ResumeFloatingObjectEvents();
                    dest.ResumeFloatingObjectEvents();
                    src.ResumePicturesEvents();
                    dest.ResumePicturesEvents();
                }
            }
            if (srcRow < 0)
            {
                CopyColumnAxis(src, srcColumn, dest, destColumn, copyColumnCount, option);
            }
            if (srcColumn < 0)
            {
                CopyRowAxis(src, srcRow, dest, destRow, copyRowCount, option);
            }
            if (((srcRow < 0) && (destRow < 0)) && ((srcColumn < 0) && (destColumn < 0)))
            {
                CopySheetInfo(src, dest, option);
            }
            dest.RaisePropertyChanged("[CopyTo]");
        }

        static void CopyValue(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int copyRowCount, int copyColumnCount)
        {
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                int num = srcColumn;
                int num2 = destColumn;
                int rowCount = src.ColumnHeader.RowCount;
                int columnCount = copyColumnCount;
                if (srcColumn < 0)
                {
                    num = 0;
                    columnCount = src.ColumnCount;
                }
                if (destColumn < 0)
                {
                    num2 = 0;
                }
                if (flag)
                {
                    for (int i = 0; i < rowCount; i++)
                    {
                        for (int j = 0; j < columnCount; j++)
                        {
                            object dataColumnName;
                            if (src.IsColumnBound(num + j))
                            {
                                dataColumnName = src.GetDataColumnName(num + j);
                            }
                            else
                            {
                                dataColumnName = src.GetValue(i, num + j, SheetArea.ColumnHeader);
                            }
                            if (dataColumnName == null)
                            {
                                dest.SetValueInternal(i, num2 + j, SheetArea.ColumnHeader, null, true, false);
                            }
                            else
                            {
                                dest.SetValueInternal(i, num2 + j, SheetArea.ColumnHeader, CloneObject(dataColumnName), true, false);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix = new DataMatrix<object>(rowCount, columnCount);
                    for (int k = 0; k < rowCount; k++)
                    {
                        for (int n = 0; n < columnCount; n++)
                        {
                            object obj3 = null;
                            if (src.IsColumnBound(num + n))
                            {
                                obj3 = src.GetDataColumnName(num + n);
                            }
                            else
                            {
                                obj3 = src.GetValue(k, num + n, SheetArea.ColumnHeader);
                            }
                            if (obj3 != null)
                            {
                                matrix.SetValue(k, n, CloneObject(obj3));
                            }
                        }
                    }
                    for (int m = 0; m < rowCount; m++)
                    {
                        for (int num10 = 0; num10 < columnCount; num10++)
                        {
                            dest.SetValueInternal(m, num2 + num10, SheetArea.ColumnHeader, matrix.GetValue(m, num10), true, false);
                        }
                    }
                }
            }
            if (srcColumn < 0)
            {
                int num11 = srcRow;
                int num12 = destRow;
                int num13 = copyRowCount;
                int num14 = src.RowHeader.ColumnCount;
                if (srcRow < 0)
                {
                    num11 = 0;
                    num13 = src.RowCount;
                }
                if (destRow < 0)
                {
                    num12 = 0;
                }
                if (flag)
                {
                    for (int num15 = 0; num15 < num13; num15++)
                    {
                        for (int num16 = 0; num16 < num14; num16++)
                        {
                            object obj4 = src.GetValue(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj4 == null)
                            {
                                dest.SetValueInternal(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null, true, false);
                            }
                            else
                            {
                                dest.SetValueInternal(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, CloneObject(obj4), true, false);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix2 = new DataMatrix<object>(num13, num14);
                    for (int num17 = 0; num17 < num13; num17++)
                    {
                        for (int num18 = 0; num18 < num14; num18++)
                        {
                            object obj5 = src.GetValue(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj5 != null)
                            {
                                matrix2.SetValue(num17, num18, CloneObject(obj5));
                            }
                        }
                    }
                    for (int num19 = 0; num19 < num13; num19++)
                    {
                        for (int num20 = 0; num20 < num14; num20++)
                        {
                            dest.SetValueInternal(num12 + num19, num20, SheetArea.CornerHeader | SheetArea.RowHeader, matrix2.GetValue(num19, num20), true, false);
                        }
                    }
                }
            }
            int num21 = srcRow;
            int num22 = srcColumn;
            int num23 = destRow;
            int num24 = destColumn;
            int row = copyRowCount;
            int column = copyColumnCount;
            if (srcRow < 0)
            {
                num21 = 0;
                row = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                num22 = 0;
                column = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                num23 = 0;
            }
            if (destColumn < 0)
            {
                num24 = 0;
            }
            if (flag)
            {
                for (int num27 = 0; num27 < row; num27++)
                {
                    for (int num28 = 0; num28 < column; num28++)
                    {
                        object obj6 = src.GetValue(num21 + num27, num22 + num28, SheetArea.Cells);
                        if (obj6 == null)
                        {
                            dest.SetValueInternal(num23 + num27, num24 + num28, SheetArea.Cells, null, true, false);
                        }
                        else
                        {
                            dest.SetValueInternal(num23 + num27, num24 + num28, SheetArea.Cells, CloneObject(obj6), true, false);
                        }
                    }
                }
            }
            else
            {
                DataMatrix<object> matrix3 = new DataMatrix<object>(row, column);
                for (int num29 = 0; num29 < row; num29++)
                {
                    for (int num30 = 0; num30 < column; num30++)
                    {
                        object obj7 = src.GetValue(num21 + num29, num22 + num30, SheetArea.Cells);
                        if (obj7 != null)
                        {
                            matrix3.SetValue(num29, num30, CloneObject(obj7));
                        }
                    }
                }
                IRangeSupport sheetBindingManager = src.SheetBindingManager;
                if (sheetBindingManager != null)
                {
                    sheetBindingManager.Copy(srcRow, srcColumn, destRow, destColumn, copyRowCount, copyColumnCount);
                }
                for (int num31 = 0; num31 < row; num31++)
                {
                    for (int num32 = 0; num32 < column; num32++)
                    {
                        dest.SetValueInternal(num23 + num31, num24 + num32, SheetArea.Cells, matrix3.GetValue(num31, num32), true, false);
                    }
                }
            }
        }

        void CreateViewportInfo()
        {
            _viewportInfo = new ViewportInfo(this, 1, 1);
        }

        static void CrossSheetCopyRangeGroup(RangeGroup srcGroup, int from, RangeGroup destGroup, int to, int count)
        {
            if (from < 0)
            {
                from = 0;
            }
            if (to < 0)
            {
                to = 0;
            }
            List<KeyValuePair<int, RangeGroup.RangeGroupItemInfo>> list = new List<KeyValuePair<int, RangeGroup.RangeGroupItemInfo>>();
            if (srcGroup != null)
            {
                for (int i = srcGroup.Items.NextNonEmptyIndex(from - 1); (i >= 0) && (i < (from + count)); i = srcGroup.Items.NextNonEmptyIndex(i))
                {
                    RangeGroup.RangeGroupItemInfo info = new RangeGroup.RangeGroupItemInfo(srcGroup.Items[i]);
                    list.Add(new KeyValuePair<int, RangeGroup.RangeGroupItemInfo>(i - from, info));
                }
            }
            if (destGroup != null)
            {
                destGroup.Items.Clear(to, count);
                if (list.Count > 0)
                {
                    foreach (KeyValuePair<int, RangeGroup.RangeGroupItemInfo> pair in list)
                    {
                        destGroup.Items[to + pair.Key] = pair.Value;
                    }
                }
            }
        }

        static void CrossSheetCopySpans(SheetSpanModel srcSpanModel, int fromRow, int fromColumn, SheetSpanModel destSpanModel, int toRow, int toColumn, int rowCount, int columnCount)
        {
            List<CellRange> spans = GetSpans(srcSpanModel);
            List<CellRange> list2 = GetSpans(destSpanModel);
            List<CellRange> list3 = new List<CellRange>();
            if (fromRow == -1)
            {
                if ((spans != null) && (spans.Count > 0))
                {
                    foreach (CellRange range in spans)
                    {
                        if ((range.Column >= fromColumn) && ((range.Column + range.ColumnCount) <= (fromColumn + columnCount)))
                        {
                            list3.Add(new CellRange(range.Row, (toColumn + range.Column) - fromColumn, range.RowCount, range.ColumnCount));
                        }
                    }
                }
                if ((list2 != null) && (list2.Count > 0))
                {
                    for (int i = 0; i < list2.Count; i++)
                    {
                        CellRange range2 = list2[i];
                        if ((range2.Column >= toColumn) && (range2.Column < (toColumn + columnCount)))
                        {
                            destSpanModel.Remove(range2.Row, range2.Column);
                        }
                    }
                }
            }
            else if (fromColumn == -1)
            {
                if ((spans != null) && (spans.Count > 0))
                {
                    foreach (CellRange range3 in spans)
                    {
                        if ((fromRow <= range3.Row) && ((range3.Row + range3.RowCount) <= (fromRow + rowCount)))
                        {
                            list3.Add(new CellRange((toRow + range3.Row) - fromRow, range3.Column, range3.RowCount, range3.ColumnCount));
                        }
                    }
                }
                if ((list2 != null) && (list2.Count > 0))
                {
                    for (int j = 0; j < list2.Count; j++)
                    {
                        CellRange range4 = list2[j];
                        if ((range4.Row >= toRow) && (range4.Row < (toRow + rowCount)))
                        {
                            destSpanModel.Remove(range4.Row, range4.Column);
                        }
                    }
                }
            }
            else
            {
                if ((spans != null) && (spans.Count > 0))
                {
                    foreach (CellRange range5 in spans)
                    {
                        if (((fromRow <= range5.Row) && (range5.Row < (fromRow + rowCount))) && ((fromColumn <= range5.Column) && (range5.Column < (fromColumn + columnCount))))
                        {
                            list3.Add(new CellRange((toRow + range5.Row) - fromRow, (toColumn + range5.Column) - fromColumn, range5.RowCount, range5.ColumnCount));
                        }
                    }
                }
                if ((list2 != null) && (list2.Count > 0))
                {
                    for (int k = 0; k < list2.Count; k++)
                    {
                        CellRange range6 = list2[k];
                        if (((range6.Row >= toRow) && (range6.Row < (toRow + rowCount))) && ((range6.Column >= toColumn) && (range6.Column < (toColumn + columnCount))))
                        {
                            destSpanModel.Remove(range6.Row, range6.Column);
                        }
                    }
                }
            }
            if ((list3.Count > 0) && (destSpanModel != null))
            {
                foreach (CellRange range7 in list3)
                {
                    destSpanModel.Add(range7.Row, range7.Column, range7.RowCount, range7.ColumnCount);
                }
            }
        }

        internal void DisposeCalcManger()
        {
            if (_calcManager != null)
            {
                _calcManager.Dispose();
                _calcManager = null;
            }
        }

        /// <summary>
        /// Specifies the last cell in the selection of cells. 
        /// </summary>  
        /// <param name="row">The row index of the extended selection.</param>
        /// <param name="column">The column index of the extended selection.</param>
        public void ExtendSelection(int row, int column)
        {
            if ((row == -1) && (column == -1))
            {
                Selection.ExtendSelection(-1, -1, -1, -1);
            }
            else if (row == -1)
            {
                Selection.ExtendSelection(-1, column, -1, 1);
            }
            else if (column == -1)
            {
                Selection.ExtendSelection(row, -1, 1, -1);
            }
            else
            {
                int anchorRow = Selection.AnchorRow;
                int anchorColumn = Selection.AnchorColumn;
                CellRange range = new CellRange(anchorRow, anchorColumn, 1, 1);
                CellRange range2 = SpanModel.Find(anchorRow, anchorColumn);
                if (range2 != null)
                {
                    range = range2;
                }
                CellRange range3 = new CellRange(row, column, 1, 1);
                CellRange range4 = SpanModel.Find(row, column);
                if (range4 != null)
                {
                    range3 = range4;
                }
                CellRange cellRange = UnionCellRange(range, range3);
                List<CellRange> spans = GetSpans();
                if ((spans != null) && (spans.Count > 0))
                {
                    cellRange = CellRangeInflate(spans, cellRange);
                }
                Selection.ExtendSelection(cellRange.Row, cellRange.Column, cellRange.RowCount, cellRange.ColumnCount);
            }
        }

        static object ExtractValueFromReference(CalcLocalIdentity localId, object value)
        {
            for (CalcReference reference = value as CalcReference; (reference != null) && !(localId is CalcRangeIdentity); reference = value as CalcReference)
            {
                CalcCellIdentity objA = localId as CalcCellIdentity;
                int rowCount = reference.GetRowCount(0);
                int columnCount = reference.GetColumnCount(0);
                if (((reference.RangeCount <= 0) || (rowCount <= 0)) || (columnCount <= 0))
                {
                    value = CalcErrors.Reference;
                }
                else
                {
                    try
                    {
                        if (!object.ReferenceEquals(objA, null))
                        {
                            if ((reference.RangeCount != 1) || ((rowCount > 1) && (columnCount > 1)))
                            {
                                value = CalcErrors.Value;
                            }
                            else
                            {
                                int rowOffset = objA.RowIndex - reference.GetRow(0);
                                int columnOffset = objA.ColumnIndex - reference.GetColumn(0);
                                if ((rowCount == 1) && (columnCount == 1))
                                {
                                    value = reference.GetValue(0, 0, 0);
                                }
                                else if (((rowCount == 1) && (columnCount > 1)) && ((columnOffset >= 0) && (columnOffset < columnCount)))
                                {
                                    value = reference.GetValue(0, 0, columnOffset);
                                }
                                else if (((rowCount > 1) && (columnCount == 1)) && ((rowOffset >= 0) && (rowOffset < rowCount)))
                                {
                                    value = reference.GetValue(0, rowOffset, 0);
                                }
                                else
                                {
                                    value = CalcErrors.Value;
                                }
                            }
                        }
                        else
                        {
                            value = reference.GetValue(0, 0, 0);
                        }
                    }
                    catch (InvalidCastException)
                    {
                        value = CalcErrors.Value;
                    }
                }
            }
            return value;
        }

        /// <summary>
        /// Fills the specified range in the specified direction.
        /// </summary>
        /// <param name="range">The cell range.</param>
        /// <param name="direction">The fill direction.</param>
        /// <remarks>For instance : CellRange cellrange=new (0,0,10,10); sheet.FillRange(cellrange,FillDirection.Down);
        /// This example fills the Cell[0,1] to Cell[0,9] using Cell[0,0]. </remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null. 
        /// </exception>
        public void FillAuto(CellRange range, FillDirection direction)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillAuto(range, direction);
        }

        /// <summary>
        /// Fills the specified range automatically. 
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <remarks>When the value is a string, the value is copied to other cells. When the value is a number, the new value is generated
        /// by the TREND formula.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null.  
        /// </exception>
        public void FillAuto(CellRange range, FillSeries series)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillAuto(range, series);
        }

        /// <summary>
        /// Fills the specified range when the source value type is date.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <param name="unit">The fill date unit.</param>
        /// <param name="step">Fills the date using the step value. The step value is the adjacent cells' offset.</param>
        /// <remarks>The next value is generated by adding the step value to the current value. The step value is affected by the fill date unit.
        /// The default stop value is positive infinity.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null.  
        /// </exception>
        public void FillDate(CellRange range, FillSeries series, FillDateUnit unit, double step)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillDate(range, series, unit, step);
        }

        /// <summary>
        /// Fills the specified range when the source value type is date.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <param name="unit">The fill date unit.</param>
        /// <param name="step">The fill date step value.</param>
        /// <param name="stop">Stops when the fill value exceeds the stop value.</param>
        /// <remarks>The next value is generated by adding the step value to the current value. The step value is affected by the fill date unit.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null.  
        /// </exception>
        public void FillDate(CellRange range, FillSeries series, FillDateUnit unit, double step, DateTime stop)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillDate(range, series, unit, step, new double?(stop.ToOADate()));
        }

        /// <summary>
        /// Fills the specified range growth trend when the source value type is number.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <remarks>The next value is generated by the GROWTH formula.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null. 
        /// </exception>
        public void FillGrowth(CellRange range, FillSeries series)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillGrowth(range, series);
        }

        /// <summary>
        /// Fills the specified range when the source value type is number.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <param name="step">The fill step value.</param>
        /// <remarks>The next value is generated by the step and stop values. The next value is computed by multiplying the current cell with the step value.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null. 
        /// </exception>
        public void FillGrowth(CellRange range, FillSeries series, double step)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillGrowth(range, series, step);
        }

        /// <summary>
        /// Fills the specified range when the source value type is number.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <param name="step">The fill step value.</param>
        /// <param name="stop">The fill stop value.</param>
        /// <remarks>The next value is generated by the step and stop values. The next value is computed by multiplying the step value with the current cell.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null.  
        /// </exception>
        public void FillGrowth(CellRange range, FillSeries series, double step, double stop)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillGrowth(range, series, step, new double?(stop));
        }

        /// <summary>
        /// Fills the specified range linear trend when the source value type is number.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <remarks>The next value is generated by the TREND formula.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null.  
        /// </exception>
        public void FillLinear(CellRange range, FillSeries series)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillLinear(range, series);
        }

        /// <summary>
        /// Fills the specified range when the source value type is number.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <param name="step">The fill step value.</param>
        /// <remarks>The next value is generated by the step and stop values. The next value is computed by adding the step value to the current cell value.
        /// The default stop value is positive infinity.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null.  
        /// </exception>
        public void FillLinear(CellRange range, FillSeries series, double step)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillLinear(range, series, step);
        }

        /// <summary>
        /// Fills the specified range when the source value type is number.
        /// </summary>
        /// <param name="range">The fill range.</param>
        /// <param name="series">The fill direction.</param>
        /// <param name="step">The fill step value.</param>
        /// <param name="stop">The fill stop value.</param>
        /// <remarks>The next value is generated by the step and stop values. The next value is computed by adding the step value to the current cell value.</remarks>
        /// <exception cref="T:System.ArgumentNullException"> 
        /// The range cannot be null.  
        /// </exception>
        public void FillLinear(CellRange range, FillSeries series, double step, double stop)
        {
            if (range == null)
            {
                throw new ArgumentNullException("range");
            }
            new FillImp(this).FillLinear(range, series, step, new double?(stop));
        }

        /// <summary>
        /// Finds the cells area.
        /// </summary>
        /// <param name="sheetArea">The sheet area</param>
        /// <returns>Returns the accessor for the specified area of the sheet.</returns>
        internal ICellsSupport FindCellsArea(SheetArea sheetArea)
        {
            switch (sheetArea)
            {
                case SheetArea.Cells:
                    return this;

                case (SheetArea.CornerHeader | SheetArea.RowHeader):
                    return RowHeader;

                case SheetArea.ColumnHeader:
                    return ColumnHeader;
            }
            return null;
        }

        /// <summary>
        /// Finds the chart with specified name.
        /// </summary>
        /// <param name="name">The name of chart to find.</param>
        /// <returns></returns>
        public SpreadChart FindChart(string name)
        {
            if (_spreadCharts != null)
            {
                return _spreadCharts.Find(name);
            }
            return null;
        }

        CalcFunction FindCustomFunction(string name)
        {
            CalcFunction customFunction = GetCustomFunction(name);
            if ((customFunction == null) && (Workbook != null))
            {
                customFunction = Workbook.GetCustomFunction(name);
            }
            return customFunction;
        }

        NameInfo FindCustomName(string name)
        {
            NameInfo info = Names.Find(name);
            if ((info == null) && (Workbook != null))
            {
                info = Workbook.Names.Find(name);
            }
            return info;
        }

        /// <summary>
        /// Finds the floating object with specified name.
        /// </summary>
        /// <param name="name">The name of the floating object to find.</param>
        /// <returns></returns>
        public FloatingObject FindFloatingObject(string name)
        {
            if (_floatingObjects != null)
            {
                return _floatingObjects.Find(name);
            }
            return null;
        }

        /// <summary>
        /// Gets formulas in the specified area.
        /// </summary>
        /// <param name="row">The start row.</param>
        /// <param name="column">The start column.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <returns>Two dimensional array. The first dimension saves the cell range and the second dimension saves the formula related to the cell range.</returns>
        public object[,] FindFormulas(int row, int column, int rowCount, int columnCount)
        {
            ICalcEvaluator evaluator = this;
            List<string> list = new List<string>();
            List<CellRange> list2 = new List<CellRange>();
            if ((_calcStorage != null) && HasFormula)
            {
                CellRange range = new CellRange(row, column, rowCount, columnCount);
                IEnumerator<KeyValuePair<CalcLocalIdentity, CalcExpression>> enumerator = _calcStorage.GetEnumerator();
                while (enumerator.MoveNext())
                {
                    CalcLocalIdentity identity = enumerator.Current.Key;
                    if (identity is CalcCellIdentity)
                    {
                        CalcCellIdentity identity2 = identity as CalcCellIdentity;
                        if (range.Contains(identity2.RowIndex, identity2.ColumnIndex))
                        {
                            string str = evaluator.Expression2Formula(enumerator.Current.Value, identity2.RowIndex, identity2.ColumnIndex);
                            if (enumerator.Current.Value is CalcArrayExpression)
                            {
                                str = "{" + str + "}";
                            }
                            list.Add(str);
                            list2.Add(new CellRange(identity2.RowIndex, identity2.ColumnIndex, 1, 1));
                        }
                    }
                    else if (identity is CalcRangeIdentity)
                    {
                        CalcRangeIdentity identity3 = identity as CalcRangeIdentity;
                        if (range.Intersects(identity3.RowIndex, identity3.ColumnIndex, identity3.RowCount, identity3.ColumnCount))
                        {
                            list.Add("{" + evaluator.Expression2Formula(enumerator.Current.Value, identity3.RowIndex, identity3.ColumnIndex) + "}");
                            list2.Add(new CellRange(identity3.RowIndex, identity3.ColumnIndex, identity3.RowCount, identity3.ColumnCount));
                        }
                    }
                }
            }
            object[,] objArray = new object[list.Count, 2];
            for (int i = 0; i < list.Count; i++)
            {
                objArray[i, 0] = list2[i];
                objArray[i, 1] = list[i];
            }
            return objArray;
        }

        /// <summary>
        /// Finds the picture with specified name.
        /// </summary>
        /// <param name="name">The picture name to find.</param>
        /// <returns></returns>
        public Picture FindPicture(string name)
        {
            if (_pictures != null)
            {
                return _pictures.Find(name);
            }
            return null;
        }

        internal SheetSpanModel FindSpanModel(SheetArea area)
        {
            if (area == SheetArea.Cells)
            {
                return _spanModel;
            }
            if (area == SheetArea.ColumnHeader)
            {
                return _columnHeaderSpanModel;
            }
            if (area == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return _rowHeaderSpanModel;
            }
            return null;
        }

        /// <summary>
        /// Gets the table with a specified name.
        /// </summary>
        /// <param name="tableName">The table name.</param>
        /// <returns><c>The table instance</c> if the cell belongs to a table; otherwise, <c>null</c>.</returns>
        public SheetTable FindTable(string tableName)
        {
            if (_tables != null)
            {
                return _tables.FindTable(tableName);
            }
            return null;
        }

        /// <summary>
        /// Gets the table of the specified cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns><c>The table instance</c> if the cell belongs to a table; otherwise, <c>null</c>.</returns>
        public SheetTable FindTable(int row, int column)
        {
            if (_tables == null)
            {
                return null;
            }
            return _tables.FindTable(row, column);
        }

        /// <summary>
        /// Gets the index of the active column viewport.
        /// </summary>
        /// <returns>The index of the active column viewport.</returns>
        public int GetActiveColumnViewportIndex()
        {
            return GetViewportInfo().ActiveColumnViewport;
        }

        /// <summary>
        /// Gets the index of the active row viewport.
        /// </summary>
        /// <returns>The index of the active row viewport.</returns>
        public int GetActiveRowViewportIndex()
        {
            return GetViewportInfo().ActiveRowViewport;
        }

        /// <summary>
        /// Gets whether a column in the specified sheet area is displayed.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>
        /// Returns <c>true</c> if the column is visible in the sheet area; otherwise, <c>false</c>.
        /// </returns>
        public bool GetActualColumnVisible(int column, SheetArea sheetArea)
        {
            if (((sheetArea == SheetArea.ColumnHeader) || (sheetArea == SheetArea.Cells)) && (((_columnRangeGroup != null) && !_columnRangeGroup.IsEmpty()) && _columnRangeGroup.IsCollapsed(column)))
            {
                return false;
            }
            AxisInfo columnAxisStyle = GetColumnAxisStyle(column, sheetArea);
            if (columnAxisStyle != null)
            {
                if (!columnAxisStyle.Visible)
                {
                    return false;
                }
                if (columnAxisStyle.IsSizeSet())
                {
                    return (columnAxisStyle.Size > 0.0);
                }
            }
            return true;
        }

        /// <summary>
        /// Gets the width in pixels for the specified column in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the width of the column range in pixels.</returns>
        public double GetActualColumnWidth(int column, SheetArea sheetArea)
        {
            if (column <= -1)
            {
                return 0.0;
            }
            if (!GetColumnVisible(column, sheetArea))
            {
                return 0.0;
            }
            if (((sheetArea == SheetArea.ColumnHeader) || (sheetArea == SheetArea.Cells)) && (((_columnRangeGroup != null) && !_columnRangeGroup.IsEmpty()) && _columnRangeGroup.IsCollapsed(column)))
            {
                return 0.0;
            }
            return GetColumnWidth(column, sheetArea);
        }

        /// <summary>
        /// Gets the width in pixels for the specified range of columns in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="count">The number of columns for which to get the width.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the width of the column range in pixels.</returns>
        public double GetActualColumnWidth(int column, int count, SheetArea sheetArea)
        {
            double num = 0.0;
            for (int i = 0; i < count; i++)
            {
                if (GetColumnVisible(column + i, sheetArea))
                {
                    num += GetActualColumnWidth(column + i, sheetArea);
                }
            }
            return num;
        }

        internal IFormatter GetActualFormatter(int row, int column, SheetArea sheetArea)
        {
            IFormatter formatter = null;
            int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            StyleInfo style = new StyleInfo();
            GetDirectInfo(modelRowFromViewRow, modelColumnFromViewColumn, style, sheetArea);
            formatter = style.Formatter;
            if (!style.IsFormatterSet())
            {
                GetDirectInfo(modelRowFromViewRow, -1, style, sheetArea);
                formatter = style.Formatter;
                if (!style.IsFormatterSet())
                {
                    GetDirectInfo(-1, modelColumnFromViewColumn, style, sheetArea);
                    formatter = style.Formatter;
                    if (!style.IsFormatterSet())
                    {
                        GetDirectInfo(-1, -1, style, sheetArea);
                        formatter = style.Formatter;
                    }
                }
            }
            if ((_conditionalFormats != null) && (_conditionalFormats.RuleCount > 0))
            {
                FormattingRuleBase[] baseArray = ConditionalFormats[row, column];
                for (int i = 0; i < baseArray.Length; i++)
                {
                    FormattingRuleBase base2 = baseArray[(baseArray.Length - 1) - i];
                    if ((base2 != null) && base2.CanFormattingStyle)
                    {
                        StyleInfo info2 = base2.Evaluate(this, row, column, new ActualValue(this, row, column)) as StyleInfo;
                        if ((info2 != null) && info2.IsFormatterSet())
                        {
                            formatter = info2.Formatter;
                        }
                    }
                }
            }
            return formatter;
        }

        /// <summary>
        /// Gets the height in pixels for the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the row height in pixels.</returns>
        public double GetActualRowHeight(int row, SheetArea sheetArea)
        {
            if (row <= -1)
            {
                return 0.0;
            }
            if (!GetRowVisible(row, sheetArea))
            {
                return 0.0;
            }
            if (((sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)) || (sheetArea == SheetArea.Cells)) && (((_rowRangeGroup != null) && !_rowRangeGroup.IsEmpty()) && _rowRangeGroup.IsCollapsed(row)))
            {
                return 0.0;
            }
            return GetRowHeight(row, sheetArea);
        }

        /// <summary>
        /// Gets the height in pixels for the specified range of rows in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="count">The number of rows for which to return the height.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the height of the range of rows in pixels.</returns>
        public double GetActualRowHeight(int row, int count, SheetArea sheetArea)
        {
            double num = 0.0;
            for (int i = 0; i < count; i++)
            {
                if (GetRowVisible(row + i, sheetArea))
                {
                    num += GetActualRowHeight(row + i, sheetArea);
                }
            }
            return num;
        }

        /// <summary>
        /// Gets whether the control displays the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>
        /// Returns <c>true</c> if the row is visible in the sheet area; otherwise, <c>false</c>.
        /// </returns>
        public bool GetActualRowVisible(int row, SheetArea sheetArea)
        {
            if (((sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)) || (sheetArea == SheetArea.Cells)) && (((_rowRangeGroup != null) && !_rowRangeGroup.IsEmpty()) && _rowRangeGroup.IsCollapsed(row)))
            {
                return false;
            }
            AxisInfo rowAxisStyle = GetRowAxisStyle(row, sheetArea);
            if (rowAxisStyle != null)
            {
                if (!rowAxisStyle.Visible)
                {
                    return false;
                }
                if (rowAxisStyle.IsSizeSet())
                {
                    return (rowAxisStyle.Size > 0.0);
                }
            }
            return true;
        }

        /// <summary>
        /// Gets the actual style information for a specified cell in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the cell style of the specified cell.</returns>
        public virtual StyleInfo GetActualStyleInfo(int row, int column, SheetArea sheetArea)
        {
            return GetActualStyleInfo(row, column, sheetArea, false);
        }

        /// <summary>
        /// Gets the actual style information for a specified cell in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="sheetStyleOnly">
        /// If <c>true</c>, the row filter and the conditional format style are not applied to the return style; 
        /// otherwise, the return style only contains cell's inherited style.
        /// </param>
        /// <returns>Returns the cell style of the specified cell.</returns>
        public virtual StyleInfo GetActualStyleInfo(int row, int column, SheetArea sheetArea, bool sheetStyleOnly)
        {
            StorageBlock storage = GetStorage(sheetArea);
            StyleInfo info = GetCompositeStyle(row, column, sheetArea, storage);
            if ((sheetArea == SheetArea.Cells) && !sheetStyleOnly)
            {
                IColorFormatter formatter = info.Formatter as IColorFormatter;
                if ((formatter != null) && formatter.HasFormatedColor)
                {
                    Color? color;
                    formatter.Format(GetValue(row, column), out color);
                    if (color.HasValue)
                    {
                        info.Foreground = new SolidColorBrush(color.Value);
                    }
                }
                if ((_conditionalFormats != null) && (_conditionalFormats.RuleCount > 0))
                {
                    FormattingRuleBase[] baseArray = ConditionalFormats[row, column];
                    if (baseArray.Length > 0)
                    {
                        baseArray = Enumerable.ToArray<FormattingRuleBase>((IEnumerable<FormattingRuleBase>)Enumerable.OrderBy<FormattingRuleBase, int>(baseArray, delegate(FormattingRuleBase r)
                        {
                            return r.Priority;
                        }));
                        int index = 0;
                        bool flag = false;
                        while (index < baseArray.Length)
                        {
                            if (baseArray[index].StopIfTrue && baseArray[index].IsConditionEvaluateToTrue(this, row, column, new ActualValue(this, row, column)))
                            {
                                flag = true;
                                break;
                            }
                            index++;
                        }
                        List<FormattingRuleBase> list = new List<FormattingRuleBase>(baseArray);
                        if (flag)
                        {
                            list.RemoveRange(index + 1, (baseArray.Length - index) - 1);
                        }
                        for (int i = 0; i < list.Count; i++)
                        {
                            FormattingRuleBase base2 = list[(list.Count - 1) - i];
                            if (base2 != null)
                            {
                                if (base2 is ScaleRule)
                                {
                                    Color? color = base2.Evaluate(this, row, column, new ActualValue(this, row, column)) as Color?;
                                    if (color.HasValue)
                                    {
                                        info.Background = new SolidColorBrush(color.Value);
                                    }
                                }
                                else
                                {
                                    StyleInfo source = base2.Evaluate(this, row, column, new ActualValue(this, row, column)) as StyleInfo;
                                    if (source != null)
                                    {
                                        info.Compose(source);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            if (((_workbook != null) && !info.IsFontFamilySet()) && !info.IsFontThemeSet())
            {
                SpreadTheme currentTheme = _workbook.CurrentTheme;
                if ((currentTheme != null) && (currentTheme.BodyFontFamily != null))
                {
                    info.FontFamily = currentTheme.BodyFontFamily;
                }
            }
            if (!info.IsFontSizeSet() || (info.FontSize < 0.0))
            {
                info.FontSize = DefaultStyleCollection.DefaultFontSize;
            }
            return info;
        }

        internal object GetActualValue(int modelRow, int modelColumn, SheetArea sheetArea)
        {
            if ((_tables != null) && (sheetArea == SheetArea.Cells))
            {
                SheetTable table = _tables.FindTable(modelRow, modelColumn);
                if (table != null)
                {
                    int num = modelColumn - table.Range.Column;
                    SheetTable.TableColumn column = table.GetColumn(num);
                    if ((column != null) && (modelRow == table.HeaderIndex))
                    {
                        return column.Name;
                    }
                }
            }
            object obj2 = null;
            if (((_bindingManager != null) && (sheetArea == SheetArea.Cells)) && (_bindingManager.IsBoundRow(modelRow) && _bindingManager.IsBoundColumn(modelColumn)))
            {
                return _bindingManager.GetValue(modelRow, modelColumn);
            }
            StorageBlock storage = GetStorage(sheetArea);
            if (storage != null)
            {
                obj2 = storage.GetValue(modelRow, modelColumn);
            }
            return obj2;
        }

        /// <summary>
        /// Gets an object array from a specified range of cells.
        /// </summary>
        /// <param name="row">Row index of the cell from which to read into the array</param>
        /// <param name="column">Column index of the cell from which to read into the array</param>
        /// <param name="rowCount">Number of rows in the array</param>
        /// <param name="columnCount">Number of columns in the array</param>
        public object[,] GetArray(int row, int column, int rowCount, int columnCount)
        {
            return GetArray(row, column, rowCount, columnCount, false);
        }

        /// <summary>
        /// Gets an object array from a specified range of cells.
        /// </summary>
        /// <param name="row">Row index of the cell from which to read into the array</param>
        /// <param name="column">Column index of the cell from which to read into the array</param>
        /// <param name="rowCount">Number of rows in the array</param>
        /// <param name="columnCount">Number of columns in the array</param>
        /// <param name="formula">If true, return formulas; otherwise, return values.</param>
        public object[,] GetArray(int row, int column, int rowCount, int columnCount, bool formula)
        {
            object[,] objArray = null;
            if (((0 <= row) && (row < RowCount)) && ((0 <= column) && (column < ColumnCount)))
            {
                if ((row + rowCount) > RowCount)
                {
                    rowCount = RowCount - row;
                }
                if ((column + columnCount) > ColumnCount)
                {
                    columnCount = ColumnCount - column;
                }
                objArray = new object[rowCount, columnCount];
                for (int i = row; i < (row + rowCount); i++)
                {
                    for (int j = column; j < (column + columnCount); j++)
                    {
                        if (formula)
                        {
                            objArray[i - row, j - column] = GetFormula(i, j);
                        }
                        else
                        {
                            objArray[i - row, j - column] = GetValue(i, j);
                        }
                    }
                }
            }
            return objArray;
        }

        /// <summary>
        /// Gets the formula in the specified area in this sheet.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <returns>Returns the formula string.</returns>
        public string GetArrayFormula(int row, int column, int rowCount, int columnCount)
        {
            return GetFormula(row, column, rowCount, columnCount, SheetArea.Cells, true);
        }

        static List<AxisInfo> GetAxisInfo(SparseArray<AxisInfo> srcInfo, int fromIndex, int count)
        {
            List<AxisInfo> list = new List<AxisInfo>();
            for (int i = 0; i < count; i++)
            {
                list.Add(srcInfo[fromIndex + i]);
            }
            return list;
        }

        StyleInfo GetAxisStyle(SparseArray<object> axis, int index)
        {
            object obj2 = axis[index];
            if (obj2 is StyleInfo)
            {
                return (obj2 as StyleInfo);
            }
            return Name2Style((string)(obj2 as string));
        }

        static List<object> GetAxisStyle(SparseArray<object> srcStyle, int fromIndex, int count)
        {
            List<object> list = new List<object>();
            for (int i = 0; i < count; i++)
            {
                list.Add(srcStyle[fromIndex + i]);
            }
            return list;
        }

        StyleInfo GetBlockStyle(StorageBlock block, int row, int column)
        {
            object style = block.GetStyle(row, column);
            if (style is StyleInfo)
            {
                return (style as StyleInfo);
            }
            return Name2Style((string)(style as string));
        }

        /// <summary>
        /// Gets the automatic text displayed in the column header for the specified column.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the automatically generated column text.</returns>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified column index is out of range; must be at least zero.
        /// </exception>
        public string GetColumnAutoText(int column)
        {
            if ((column < 0) || (column >= ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidColumnIndexWithAllowedRangeBehind, (object[])new object[] { ((int)column), ((int)(ColumnCount - 1)) }));
            }
            if (_columnHeaderAutoText == HeaderAutoText.Numbers)
            {
                int num = column + StartingColumnNumber;
                return ((int)num).ToString();
            }
            if (_columnHeaderAutoText == HeaderAutoText.Letters)
            {
                return IndexToLetters(column + StartingColumnNumber);
            }
            return string.Empty;
        }

        internal AxisInfo GetColumnAxisStyle(int column)
        {
            return GetColumnAxisStyle(column, SheetArea.Cells);
        }

        internal AxisInfo GetColumnAxisStyle(int column, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == SheetArea.ColumnHeader))
            {
                return _viewportColumns[column];
            }
            if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return _rowHeaderColumns[column];
            }
            return null;
        }

        SparseArray<object> GetColumnAxisStyles(SheetArea area)
        {
            if (area == SheetArea.Cells)
            {
                return _viewportColumnsStyles;
            }
            if (area == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return _rowHeaderColumnsStyles;
            }
            if (area == SheetArea.ColumnHeader)
            {
                return _columnHeaderColumnsStyles;
            }
            return null;
        }

        internal int GetColumnCount(SheetArea area)
        {
            return GetStorage(area).ColumnCount;
        }

        /// <summary>
        /// Gets the column data type at the specified position.
        /// </summary>
        /// <param name="column">The column index for which the type is requested.</param>
        /// <returns>Returns the column type for data binding.</returns>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified index is out of range; must be between 0 and the total number of columns.
        /// </exception>
        public Type GetColumnDataType(int column)
        {
            if (_bindingManager != null)
            {
                return _bindingManager.GetColumnDataType(column);
            }
            return typeof(object);
        }

        /// <summary>
        /// Gets the text in the specified column header cell.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the column label string.</returns>
        public string GetColumnLabel(int column)
        {
            int autoTextIndex = ColumnHeader.AutoTextIndex;
            if ((autoTextIndex < 0) || (autoTextIndex >= ColumnHeaderRowCount))
            {
                autoTextIndex = ColumnHeaderRowCount - 1;
            }
            return GetColumnLabel(autoTextIndex, column);
        }

        /// <summary>
        /// Gets the text in the specified column header cell in the specified column header row.
        /// </summary>
        /// <param name="row">The column header row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the column label string.</returns>
        public string GetColumnLabel(int row, int column)
        {
            if (row < 0)
            {
                row = (ColumnHeaderAutoTextIndex > -1) ? ColumnHeaderAutoTextIndex : (ColumnHeader.RowCount - 1);
            }
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, SheetArea.ColumnHeader);
            int num2 = GetModelColumnFromViewColumn(row, SheetArea.ColumnHeader);
            if (modelColumnFromViewColumn < 0)
            {
                return string.Empty;
            }
            object obj2 = GetStorage(SheetArea.ColumnHeader).GetValue(num2, modelColumnFromViewColumn);
            if (obj2 != null)
            {
                return obj2.ToString();
            }
            if ((ColumnHeader.AutoTextIndex != row) && (((ColumnHeader.AutoTextIndex >= 0) && (ColumnHeader.AutoTextIndex < ColumnHeader.RowCount)) || (row != (ColumnHeader.RowCount - 1))))
            {
                return string.Empty;
            }
            if (DataAutoHeadings && (_bindingManager != null))
            {
                string dataSourceColumnFromModelColumn = _bindingManager.GetDataSourceColumnFromModelColumn(column);
                if (!string.IsNullOrEmpty(dataSourceColumnFromModelColumn))
                {
                    return dataSourceColumnFromModelColumn;
                }
            }
            return GetColumnAutoText(column);
        }

        /// <summary>
        /// Gets whether a forced page break is inserted before the specified column when printing.
        /// </summary>
        /// <param name="column">The column index (use -1 for all columns).</param>
        /// <returns><c>true</c> if the column has a forced page break when printing; otherwise, <c>false</c>.</returns>
        internal bool GetColumnPageBreak(int column)
        {
            return false;
        }

        /// <summary>
        /// Gets a value that indicates whether the user can resize a specified column in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <returns>
        /// Returns <c>true</c> if the user can resize the specified column; otherwise, <c>false</c>.
        /// </returns>
        public bool GetColumnResizable(int column)
        {
            return GetColumnResizable(column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets a value that indicates whether the user can resize a specified column in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>
        /// Returns <c>true</c> if the user can resize the specified column; otherwise, <c>false</c>.
        /// </returns>
        public bool GetColumnResizable(int column, SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            AxisInfo info = GetColumnsStorage(sheetArea)[modelColumnFromViewColumn];
            if (info != null)
            {
                return info.Resizable;
            }
            return true;
        }

        SparseArray<AxisInfo> GetColumnsStorage(SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.ColumnHeader) || (sheetArea == SheetArea.Cells))
            {
                return _viewportColumns;
            }
            if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return _rowHeaderColumns;
            }
            return null;
        }

        /// <summary>
        /// Gets the column viewport count.
        /// </summary>
        /// <returns>The column viewport count.</returns>
        public int GetColumnViewportCount()
        {
            return GetViewportInfo().ColumnViewportCount;
        }

        /// <summary>
        /// Gets whether a column in this sheet is displayed.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <returns>Returns <c>true</c> if the column is visible; otherwise, <c>false</c>.</returns>
        public bool GetColumnVisible(int column)
        {
            return GetColumnVisible(column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets whether a column in the specified sheet area is displayed.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>
        /// Returns <c>true</c> if the column is visible in the sheet area; otherwise, <c>false</c>.
        /// </returns>
        public bool GetColumnVisible(int column, SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            AxisInfo info = GetColumnsStorage(sheetArea)[modelColumnFromViewColumn];
            if (info != null)
            {
                return info.Visible;
            }
            return true;
        }

        /// <summary>
        /// Gets the width in pixels for the specified column in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the column width in pixels.</returns>
        public double GetColumnWidth(int column)
        {
            return GetColumnWidth(column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets the width in pixels for the specified column in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the column width in pixels.</returns>
        public double GetColumnWidth(int column, SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if (column > -1)
            {
                int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
                AxisInfo info = GetColumnsStorage(sheetArea)[modelColumnFromViewColumn];
                if ((info != null) && info.IsSizeSet())
                {
                    double size = info.Size;
                    if (size >= 0.0)
                    {
                        return (double)((int)size);
                    }
                }
            }
            if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return DefaultRowHeaderColumnWidth;
            }
            return DefaultColumnWidth;
        }

        internal double GetColumnWidth(int column, SheetArea sheetArea, UnitType unit)
        {
            double columnWidth = GetColumnWidth(column, sheetArea);
            return UnitManager.ConvertTo(columnWidth, UnitType.Pixel, unit);
        }

        /// <summary>
        /// The method never returns null.
        /// </summary>
        internal StyleInfo GetCompositeStyle(int row, int column, SheetArea sheetArea, StorageBlock block)
        {
            StyleInfo dest = new StyleInfo();
            StyleInfo defaultStyle = GetDefaultStyle(sheetArea);
            int rowCount = block.RowCount;
            int columnCount = block.ColumnCount;
            if (defaultStyle != null)
            {
                ComposeStyle(defaultStyle, dest);
            }
            if ((0 <= column) && (column < columnCount))
            {
                object source = null;
                if (sheetArea == SheetArea.Cells)
                {
                    source = _viewportColumnsStyles[column];
                }
                else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    source = _rowHeaderColumnsStyles[column];
                }
                else if (sheetArea == SheetArea.ColumnHeader)
                {
                    source = _columnHeaderColumnsStyles[column];
                }
                if (source != null)
                {
                    ComposeStyle(source, dest);
                }
            }
            if ((0 <= row) && (row < rowCount))
            {
                object obj3 = null;
                if (sheetArea == SheetArea.Cells)
                {
                    obj3 = _viewportRowsStyles[row];
                }
                else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    obj3 = _rowHeaderRowsStyles[row];
                }
                else if (sheetArea == SheetArea.ColumnHeader)
                {
                    obj3 = _columnHeaderRowsStyles[row];
                }
                if (obj3 != null)
                {
                    ComposeStyle(obj3, dest);
                }
            }
            if (((_tables != null) && (_tables.Count > 0)) && (sheetArea == SheetArea.Cells))
            {
                SheetTable table = _tables.FindTable(row, column);
                if (table != null)
                {
                    table.Render(row, column, dest);
                }
            }
            if (((0 <= row) && (row < rowCount)) && ((0 <= column) && (column < columnCount)))
            {
                object style = block.GetStyle(row, column);
                if (style != null)
                {
                    ComposeStyle(style, dest);
                }
            }
            return dest;
        }

        /// <summary>
        /// Gets the CSV text from a range.
        /// </summary>
        /// <param name="row">The start row.</param>
        /// <param name="column">The start column.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <param name="rowDelimiter">The row delimiter that is appended to the end of the row.</param>
        /// <param name="columnDelimiter">The column delimiter that is appended to the end of the column.</param>
        /// <param name="cellDelimiter">The cell delimiter that is appended to the beginning and end of the cell.</param>
        public string GetCsv(int row, int column, int rowCount, int columnCount, string rowDelimiter, string columnDelimiter, string cellDelimiter)
        {
            return GetCsv(row, column, rowCount, columnCount, rowDelimiter, columnDelimiter, cellDelimiter, true);
        }

        /// <summary>
        /// Gets the CSV text from a range.
        /// </summary>
        /// <param name="row">The start row.</param>
        /// <param name="column">The start column.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <param name="rowDelimiter">The row delimiter that is appended to the end of the row.</param>
        /// <param name="columnDelimiter">The column delimiter that is appended to the end of the column.</param>
        /// <param name="cellDelimiter">The cell delimiter that is appended to the beginning and end of the cell.</param>
        /// <param name="forceCellDelimiter">If <c>true</c> forces a cell delimiter to be appended to a cell.</param>
        public string GetCsv(int row, int column, int rowCount, int columnCount, string rowDelimiter, string columnDelimiter, string cellDelimiter, bool forceCellDelimiter)
        {
            return GetCsv(row, column, rowCount, columnCount, rowDelimiter, columnDelimiter, cellDelimiter, forceCellDelimiter, TextFileSaveFlags.None);
        }

        /// <summary>
        /// Gets the CSV text from a range.
        /// </summary>
        /// <param name="row">The start row.</param>
        /// <param name="column">The start column.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">the column count.</param>
        /// <param name="rowDelimiter">The row delimiter that is appended to the end of the row.</param>
        /// <param name="columnDelimiter">The column delimiter that is appended to the end of the column.</param>
        /// <param name="cellDelimiter">The cell delimiter that is appended to the beginning and end of the cell.</param>
        /// <param name="forceCellDelimiter">If <c>true</c> forces the cell delimiter to be appended to a cell.</param>
        /// <param name="flags">The import flags.</param>
        public string GetCsv(int row, int column, int rowCount, int columnCount, string rowDelimiter, string columnDelimiter, string cellDelimiter, bool forceCellDelimiter, TextFileSaveFlags flags)
        {
            return CsvImp.GetRangeText(this, row, rowCount, column, columnCount, rowDelimiter, columnDelimiter, cellDelimiter, forceCellDelimiter, flags);
        }

        /// <summary>
        /// Gets the specified user-defined custom function.
        /// </summary>
        /// <param name="name">The name of the user-defined custom function.</param>
        /// <returns>The function.</returns>
        public CalcFunction GetCustomFunction(string name)
        {
            name = name.ToUpper(CultureInfo.CurrentCulture);
            if (Functions.ContainsKey(name))
            {
                return Functions[name];
            }
            return null;
        }

        /// <summary>
        /// Finds the specified custom name.
        /// </summary>
        /// <param name="name"></param>
        /// <returns>The NameInfo object of the specified custom name.</returns>
        public NameInfo GetCustomName(string name)
        {
            return Names.Find(name);
        }

        /// <summary>
        /// Gets the column name at the specified position.
        /// </summary>
        /// <param name="column">The column index for which the name is requested.</param>
        /// <returns>Returns the column name for data binding.</returns>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified index is out of range; must be between 0 and the total number of columns.
        /// </exception>
        public string GetDataColumnName(int column)
        {
            if (_bindingManager != null)
            {
                return _bindingManager.GetDataSourceColumnFromModelColumn(column);
            }
            return string.Empty;
        }

        StyleInfo GetDefaultStyle(SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.Cells)
            {
                return DefaultStyle;
            }
            if (sheetArea == SheetArea.ColumnHeader)
            {
                return ColumnHeaderDefaultStyle;
            }
            if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return RowHeaderDefaultStyle;
            }
            return null;
        }

        internal StyleInfo GetDirectInfo(int row, int column, StyleInfo style, SheetArea sheetArea)
        {
            StyleInfo o = GetStyleInfo(row, column, sheetArea);
            if (style != null)
            {
                style.CopyFrom(o);
            }
            return o;
        }

        /// <summary>
        /// Gets the data bar object or icon object of the current sheet in the specified range.
        /// </summary>
        /// <param name="row">The row.</param>
        /// <param name="column">The column.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <returns>The collection of drawing objects.</returns>
        public virtual DrawingObject[] GetDrawingObject(int row, int column, int rowCount, int columnCount)
        {
            List<DrawingObject> list = null;
            if (_conditionalFormats != null)
            {
                FormattingRuleBase[] baseArray = ConditionalFormats[row, column];
                baseArray = Enumerable.ToArray<FormattingRuleBase>((IEnumerable<FormattingRuleBase>)Enumerable.OrderBy<FormattingRuleBase, int>(baseArray, delegate(FormattingRuleBase r)
                {
                    return r.Priority;
                }));
                int index = 0;
                bool flag = false;
                while (index < baseArray.Length)
                {
                    if (baseArray[index].StopIfTrue && baseArray[index].IsConditionEvaluateToTrue(this, row, column, new ActualValue(this, row, column)))
                    {
                        flag = true;
                        break;
                    }
                    index++;
                }
                List<FormattingRuleBase> list2 = new List<FormattingRuleBase>(baseArray);
                if (flag)
                {
                    list2.RemoveRange(index + 1, (baseArray.Length - index) - 1);
                }
                CellRange range = new CellRange(row, column, rowCount, columnCount);
                for (int i = 0; i < list2.Count; i++)
                {
                    FormattingRuleBase base2 = list2[i];
                    if (((base2 != null) && (base2.Ranges != null)) && ((base2 is IconSetRule) || (base2 is DataBarRule)))
                    {
                        for (int j = 0; j < base2.Ranges.Length; j++)
                        {
                            CellRange range2 = base2.Ranges[j];
                            if (range2 != null)
                            {
                                CellRange range3 = CellRange.GetIntersect(range, range2, RowCount, ColumnCount);
                                if (range3 != null)
                                {
                                    for (int k = 0; k < range3.RowCount; k++)
                                    {
                                        int baseRow = k + range3.Row;
                                        for (int m = 0; m < range3.ColumnCount; m++)
                                        {
                                            int baseColumn = m + range3.Column;
                                            DrawingObject obj2 = base2.Evaluate(this, baseRow, baseColumn, new ActualValue(this, baseRow, baseColumn)) as DrawingObject;
                                            if (obj2 != null)
                                            {
                                                if (list == null)
                                                {
                                                    list = new List<DrawingObject>();
                                                }
                                                list.Add(obj2);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                if (list != null)
                {
                    return list.ToArray();
                }
            }
            return null;
        }

        internal List<IExcelChart> GetExcelCharts()
        {
            return _charts;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="row"></param>
        /// <param name="column"></param>
        /// <param name="fillType"></param>
        /// <param name="fillDirection"></param>
        /// <returns></returns>
        public string GetFillText(int row, int column, AutoFillType fillType, FillDirection fillDirection)
        {
            FillImp imp = new FillImp(this);
            return imp.GetFillText(row, column, fillType, fillDirection);
        }

        static FloatingObject[] GetFloatingObjectsInRange(CellRange range, Worksheet sheet)
        {
            List<FloatingObject> list = new List<FloatingObject>();
            foreach (SpreadChart chart in sheet.Charts)
            {
                list.Add(chart);
            }
            foreach (Picture picture in sheet.Pictures)
            {
                list.Add(picture);
            }
            foreach (FloatingObject obj2 in sheet.FloatingObjects)
            {
                list.Add(obj2);
            }
            List<FloatingObject> list2 = new List<FloatingObject>();
            foreach (FloatingObject obj3 in list)
            {
                CellRange range2 = new CellRange(obj3.StartRow, obj3.StartColumn, (obj3.EndRow - obj3.StartRow) + 1, (obj3.EndColumn - obj3.StartColumn) + 1);
                if (range.Contains(range2))
                {
                    list2.Add(obj3);
                }
            }
            return list2.ToArray();
        }

        /// <summary>
        /// Gets the formula in the specified cell in this sheet.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the formula string.</returns>
        public string GetFormula(int row, int column)
        {
            return GetFormula(row, column, 1, 1, SheetArea.Cells, false);
        }

        internal string GetFormula(int row, int column, SheetArea sheetArea)
        {
            string formula = GetFormula(row, column);
            if (!string.IsNullOrEmpty(formula))
            {
                return formula;
            }
            return GetArrayFormula(row, column, 1, 1);
        }

        internal string GetFormula(int row, int column, int rowCount, int columnCount, SheetArea sheetArea, bool arrayFormula)
        {
            if ((sheetArea != SheetArea.Cells) || !HasFormula)
            {
                return string.Empty;
            }
            int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            if (arrayFormula)
            {
                CalcRangeIdentity id = CalcExpressionHelper.CreateRangeExpressionByCount(modelRowFromViewRow, modelColumnFromViewColumn, rowCount, columnCount, false, false, false, false).GetId(0, 0) as CalcRangeIdentity;
                return CalcManager.GetFormula(id);
            }
            return CalcManager.GetFormula(new CalcCellIdentity(modelRowFromViewRow, modelColumnFromViewColumn));
        }

        /// <summary>
        /// Gets the control gridline; the gridline color is based on the ShowGridLine property.
        /// </summary>
        public BorderLine GetGridLine(SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.Cells)
            {
                if (_showGridLine)
                {
                    if (_gridLine == null)
                    {
                        _gridLine = new BorderLine(_gridLineColor, BorderLineStyleDatas.GridLine);
                    }
                    return _gridLine;
                }

                if (_gridLine == null)
                {
                    // 无需绘制
                    _gridLine = BorderLine.NoBorder;
                    //_gridLine = new BorderLine(Colors.Transparent, BorderLineStyleDatas.GridLine);
                }
                return _gridLine;
            }

            // 默认网格线
            if (_defaultGridLine == null)
            {
                _defaultGridLine = new BorderLine(DefaultGridLineColor, BorderLineStyleDatas.GridLine);
            }
            return _defaultGridLine;
        }

        /// <summary>
        /// Gets the last dirty column index.
        /// </summary>
        /// <returns>The index of the last dirty column.</returns>
        public int GetLastDirtyColumn(StorageType type)
        {
            int num = (_cellsBlock == null) ? -1 : _cellsBlock.GetLastDirtyColumn(type);
            if ((((type & StorageType.Data) == StorageType.Data) && (_bindingManager != null)) && _bindingManager.IsBound)
            {
                List<int> boundColumns = _bindingManager.GetBoundColumns();
                if ((boundColumns.Count > 0) && (boundColumns[boundColumns.Count - 1] > num))
                {
                    num = boundColumns[boundColumns.Count - 1];
                }
            }
            if (((type & (StorageType.Style | StorageType.Data)) > ((StorageType)0)) && (_tables != null))
            {
                int lastTableColumn = GetLastTableColumn();
                if (num < lastTableColumn)
                {
                    num = lastTableColumn;
                }
            }
            if ((type & StorageType.Axis) == StorageType.Axis)
            {
                for (int i = ColumnCount - 1; i >= 0; i--)
                {
                    AxisInfo info = _viewportColumns[i];
                    if ((info != null) && ((info.IsResizableSet() || info.IsSizeSet()) || (info.IsVisibleSet() || info.IsTagSet())))
                    {
                        return Math.Max(num, i);
                    }
                }
            }
            return num;
        }

        /// <summary>
        /// Gets the last dirty row index.
        /// </summary>
        /// <returns>The index of the last dirty row.</returns>
        public int GetLastDirtyRow(StorageType type)
        {
            int num = (_cellsBlock == null) ? -1 : _cellsBlock.GetLastDirtyRow(type);
            if (((type & StorageType.Axis) == StorageType.Axis) && (_viewportRows != null))
            {
                int lastDirtyIndex = _viewportRows.LastDirtyIndex;
                if (num < lastDirtyIndex)
                {
                    num = lastDirtyIndex;
                }
            }
            if ((((type & StorageType.Data) == StorageType.Data) && (_bindingManager != null)) && _bindingManager.IsBound)
            {
                int lastBoundRow = _bindingManager.GetLastBoundRow();
                if (num < lastBoundRow)
                {
                    num = lastBoundRow;
                }
            }
            if (((type & (StorageType.Style | StorageType.Data)) > ((StorageType)0)) && (_tables != null))
            {
                int lastTableRow = GetLastTableRow();
                if (num < lastTableRow)
                {
                    num = lastTableRow;
                }
            }
            return num;
        }

        int GetLastTableColumn()
        {
            int num = -1;
            for (int i = 0; (_tables != null) && (i < _tables.Count); i++)
            {
                CellRange range = _tables[0].Range;
                int num3 = (range.Column + range.ColumnCount) - 1;
                if (num < num3)
                {
                    num = num3;
                }
            }
            return num;
        }

        int GetLastTableRow()
        {
            int num = -1;
            for (int i = 0; (_tables != null) && (i < _tables.Count); i++)
            {
                CellRange range = _tables[i].Range;
                int num3 = (range.Row + range.RowCount) - 1;
                if (num < num3)
                {
                    num = num3;
                }
            }
            return num;
        }

        internal int GetModelColumnFromViewColumn(int column)
        {
            return column;
        }

        internal int GetModelColumnFromViewColumn(int column, SheetArea sheetArea)
        {
            return column;
        }

        internal int GetModelRowFromViewRow(int row)
        {
            return row;
        }

        internal int GetModelRowFromViewRow(int row, SheetArea sheetArea)
        {
            return row;
        }

        /// <summary>
        /// Returns a list of nonempty column indexes.
        /// </summary>
        /// <returns>A list of indexes.</returns>
        public List<int> GetNonEmptyColumns()
        {
            List<int> nonEmptyColumns = _cellsBlock.GetNonEmptyColumns();
            List<int> nonEmptyIndexes = _viewportColumns.GetNonEmptyIndexes();
            List<int> list3 = _viewportColumnsStyles.GetNonEmptyIndexes();
            List<int> list4 = new List<int>(Enumerable.Union<int>(Enumerable.Union<int>((IEnumerable<int>)nonEmptyColumns, (IEnumerable<int>)nonEmptyIndexes), (IEnumerable<int>)list3));
            if ((_bindingManager != null) && _bindingManager.IsBound)
            {
                List<int> boundColumns = _bindingManager.GetBoundColumns();
                list4 = new List<int>(Enumerable.Union<int>((IEnumerable<int>)list4, (IEnumerable<int>)boundColumns));
            }
            List<int> list6 = new List<int>();
            for (int i = 0; (_tables != null) && (i < _tables.Count); i++)
            {
                SheetTable table = _tables[i];
                CellRange range = table.Range;
                for (int j = 0; j < range.ColumnCount; j++)
                {
                    int num3 = j + range.Column;
                    list6.Add(num3);
                }
            }
            list4 = new List<int>(Enumerable.Union<int>((IEnumerable<int>)list4, (IEnumerable<int>)list6));
            list4.Sort();
            return list4;
        }

        /// <summary>
        /// Returns a list of nonempty row indexes.
        /// </summary>
        /// <returns>A list of indexes.</returns>
        public List<int> GetNonEmptyRows()
        {
            List<int> nonEmptyRows = _cellsBlock.GetNonEmptyRows();
            List<int> nonEmptyIndexes = _viewportRows.GetNonEmptyIndexes();
            List<int> list3 = _viewportRowsStyles.GetNonEmptyIndexes();
            List<int> list4 = new List<int>(Enumerable.Union<int>(Enumerable.Union<int>((IEnumerable<int>)nonEmptyRows, (IEnumerable<int>)nonEmptyIndexes), (IEnumerable<int>)list3));
            if ((_bindingManager != null) && _bindingManager.IsBound)
            {
                List<int> boundRows = _bindingManager.GetBoundRows();
                list4 = new List<int>(Enumerable.Union<int>((IEnumerable<int>)list4, (IEnumerable<int>)boundRows));
            }
            List<int> list6 = new List<int>();
            for (int i = 0; (_tables != null) && (i < _tables.Count); i++)
            {
                SheetTable table = _tables[i];
                CellRange range = table.Range;
                for (int j = 0; j < range.RowCount; j++)
                {
                    int num3 = j + range.Row;
                    list6.Add(num3);
                }
            }
            list4 = new List<int>(Enumerable.Union<int>((IEnumerable<int>)list4, (IEnumerable<int>)list6));
            list4.Sort();
            return list4;
        }

        internal GcReport GetPrintableReport()
        {
            GcReport report = new GcReport();
            if (PrintInfo != null)
            {
                string workbookName = (Workbook != null) ? Workbook.Name : string.Empty;
                PrintInfo.ApplyToGcReport(report, workbookName, Name);
            }
            GcSheetSection section = new GcSheetSection(this);
            report.Sections.Add(section);
            return report;
        }

        /// <summary>
        /// Gets the automatic text displayed in the row header for the specified row.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <returns>Returns the automatically generated row text.</returns>
        /// <excpetion cref="T:System.ArgumentOutOfRangeException">
        /// Specified row index is out of range; must be at least zero.
        /// </excpetion>
        public string GetRowAutoText(int row)
        {
            if ((row < 0) || (row >= RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvaildRowIndexWithAllowedRangeBehind, (object[])new object[] { ((int)row), ((int)(RowCount - 1)) }));
            }
            if (_rowHeaderAutoText == HeaderAutoText.Numbers)
            {
                int num = row + StartingRowNumber;
                return ((int)num).ToString();
            }
            if (_rowHeaderAutoText == HeaderAutoText.Letters)
            {
                return IndexToLetters(row + StartingRowNumber);
            }
            return string.Empty;
        }

        internal AxisInfo GetRowAxisStyle(int row)
        {
            return GetRowAxisStyle(row, SheetArea.Cells);
        }

        internal AxisInfo GetRowAxisStyle(int row, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)))
            {
                return _viewportRows[row];
            }
            if (sheetArea == SheetArea.ColumnHeader)
            {
                return _columnHeaderRows[row];
            }
            return null;
        }

        SparseArray<object> GetRowAxisStyles(SheetArea area)
        {
            if (area == SheetArea.Cells)
            {
                return _viewportRowsStyles;
            }
            if (area == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return _rowHeaderRowsStyles;
            }
            if (area == SheetArea.ColumnHeader)
            {
                return _columnHeaderRowsStyles;
            }
            return null;
        }

        internal int GetRowCount(SheetArea sheetArea)
        {
            return GetStorage(sheetArea).RowCount;
        }

        /// <summary>
        /// Gets the height using the specified unit type for the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <returns>Returns the row height in the specified units.</returns>
        public double GetRowHeight(int row)
        {
            return GetRowHeight(row, SheetArea.Cells);
        }

        /// <summary>
        /// Gets the height using the specified unit type for the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the row height in the specified units.</returns>
        public double GetRowHeight(int row, SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if (row > -1)
            {
                AxisInfo info = GetRowsStorage(sheetArea)[row];
                if ((info != null) && info.IsSizeSet())
                {
                    double size = info.Size;
                    if (size >= 0.0)
                    {
                        return size;
                    }
                }
            }
            switch (sheetArea)
            {
                case SheetArea.Cells:
                case (SheetArea.CornerHeader | SheetArea.RowHeader):
                    return DefaultRowHeight;

                case SheetArea.ColumnHeader:
                    return DefaultColumnHeaderRowHeight;
            }
            return 0.0;
        }

        internal double GetRowHeight(int row, SheetArea sheetArea, UnitType unit)
        {
            double rowHeight = GetRowHeight(row, sheetArea);
            return UnitManager.ConvertTo(rowHeight, UnitType.Pixel, unit);
        }

        /// <summary>
        /// Gets the text in the specified row header cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <returns>Returns the row label string.</returns>
        public string GetRowLabel(int row)
        {
            return GetRowLabel(row, RowHeaderAutoTextIndex);
        }

        /// <summary>
        /// Gets the text in the specified row header cell in the specified row header column.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The row header column index.</param>
        /// <returns>Returns the row label string.</returns>
        public string GetRowLabel(int row, int column)
        {
            if (column < 0)
            {
                column = (RowHeaderAutoTextIndex > -1) ? RowHeaderAutoTextIndex : (RowHeader.ColumnCount - 1);
            }
            int modelRowFromViewRow = GetModelRowFromViewRow(row, SheetArea.CornerHeader | SheetArea.RowHeader);
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column);
            if (modelRowFromViewRow < 0)
            {
                return string.Empty;
            }
            object obj2 = GetStorage(SheetArea.CornerHeader | SheetArea.RowHeader).GetValue(modelRowFromViewRow, modelColumnFromViewColumn);
            if (obj2 != null)
            {
                return obj2.ToString();
            }
            if ((RowHeader.AutoTextIndex != column) && (((RowHeader.AutoTextIndex >= 0) && (RowHeader.AutoTextIndex < RowHeader.ColumnCount)) || (column != (RowHeader.ColumnCount - 1))))
            {
                return string.Empty;
            }
            return GetRowAutoText(row);
        }

        /// <summary>
        /// Gets whether a forced page break is inserted before the specified row when printing.
        /// </summary>
        /// <param name="row">The row index (use -1 for all rows).</param>
        /// <returns><c>true</c> if the row has a forced page break when printing; otherwise, <c>false</c>.</returns>
        internal bool GetRowPageBreak(int row)
        {
            return false;
        }

        /// <summary>
        /// Gets a value that indicates whether users can resize the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <returns>Returns <c>true</c> if the users can resize the specified row; otherwise, <c>false</c>.</returns>
        public bool GetRowResizable(int row)
        {
            return GetRowResizable(row, SheetArea.Cells);
        }

        /// <summary>
        /// Gets a value that indicates whether users can resize the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns <c>true</c> if the users can resize the specified row; otherwise, <c>false</c>.</returns>
        public bool GetRowResizable(int row, SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if (row > -1)
            {
                GetModelRowFromViewRow(row, sheetArea);
                AxisInfo info = GetRowsStorage(sheetArea)[row];
                if ((info != null) && info.IsResizableSet())
                {
                    return info.Resizable;
                }
            }
            return true;
        }

        SparseArray<AxisInfo> GetRowsStorage(SheetArea sheetArea)
        {
            if ((sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)) || (sheetArea == SheetArea.Cells))
            {
                return _viewportRows;
            }
            if (sheetArea == SheetArea.ColumnHeader)
            {
                return _columnHeaderRows;
            }
            return null;
        }

        /// <summary>
        /// Gets the row viewport count.
        /// </summary>
        /// <returns>The row viewport count.</returns>
        public int GetRowViewportCount()
        {
            return GetViewportInfo().RowViewportCount;
        }

        /// <summary>
        /// Gets whether the control displays the specified row.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <returns>
        /// Returns <c>true</c> if the row is visible in this sheet; otherwise, <c>false</c>.
        /// </returns>
        public bool GetRowVisible(int row)
        {
            return GetRowVisible(row, SheetArea.Cells);
        }

        /// <summary>
        /// Gets whether the control displays the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>
        /// Returns <c>true</c> if the row is visible in the sheet area; otherwise, <c>false</c>.
        /// </returns>
        public bool GetRowVisible(int row, SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            GetModelRowFromViewRow(row, sheetArea);
            AxisInfo info = GetRowsStorage(sheetArea)[row];
            if (info != null)
            {
                return info.Visible;
            }
            return true;
        }

        /// <summary>
        /// Gets the range of cells that are spanned at a specified cell in this sheet.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>
        /// Returns a <see cref="T:CellRange" /> object that contains the span information, or null if no span exists.
        /// </returns>
        public CellRange GetSpanCell(int row, int column)
        {
            return GetSpanCell(row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets the range of cells that are spanned at a specified cell on this sheet in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>
        /// Returns a <see cref="T:CellRange" /> object that contains the span information, or null if no span exists.
        /// </returns>
        public CellRange GetSpanCell(int row, int column, SheetArea sheetArea)
        {
            SheetSpanModelBase spanModel = GetSpanModel(sheetArea);
            if ((spanModel == null) || spanModel.IsEmpty())
            {
                return null;
            }
            CellRange range = spanModel.Find(row, column);
            if (range != null)
            {
                range = new CellRange(range.Row, range.Column, range.RowCount, range.ColumnCount);
            }
            return range;
        }

        SheetSpanModel GetSpanModel(SheetArea area)
        {
            if (area == SheetArea.Cells)
            {
                return SpanModel;
            }
            if (area == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return RowHeaderSpanModel;
            }
            if (area != SheetArea.ColumnHeader)
            {
                throw new Exception(string.Format(ResourceStrings.SetSpanToNotSupportAreaError, (object[])new object[] { area.ToString() }));
            }
            return ColumnHeaderSpanModel;
        }

        /// <summary>
        /// Gets all the spans in the worksheet.
        /// </summary>
        internal List<CellRange> GetSpans()
        {
            return GetSpans(SheetArea.Cells);
        }

        internal List<CellRange> GetSpans(SheetArea area)
        {
            return GetSpans(GetSpanModel(area));
        }

        static List<CellRange> GetSpans(SheetSpanModel spanModel)
        {
            if ((spanModel == null) || spanModel.IsEmpty())
            {
                return null;
            }
            List<CellRange> list = new List<CellRange>();
            foreach (CellRange range in spanModel)
            {
                list.Add(range);
            }
            return list;
        }

        /// <summary>
        /// Gets the sparkline for the specified cell.
        /// </summary>
        /// <param name="row">The row.</param>
        /// <param name="column">The column.</param>
        /// <returns>
        /// The sparkline of the cell.
        /// </returns>
        public Sparkline GetSparkline(int row, int column)
        {
            return GetStorage(SheetArea.Cells).GetSparkline(row, column);
        }

        internal StorageBlock GetStorage(SheetArea area)
        {
            if (area == SheetArea.Cells)
            {
                return _cellsBlock;
            }
            if (area == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                return _rowHeadersBlock;
            }
            if (area == SheetArea.ColumnHeader)
            {
                return _columnHeadersBlock;
            }
            return null;
        }

        /// <summary>
        /// Gets the style information of the specified cell, row, or column.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>The style information of the specified cell, row, or column.</returns>
        public virtual StyleInfo GetStyleInfo(int row, int column)
        {
            return GetStyleInfo(row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets the style information of the specified cell, row, or column.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>The style information of the specified cell, row, or column.</returns>
        /// <exception cref="T:System.ArgumentException">The sheet area cannot be found.</exception>
        /// <exception cref="T:System.IndexOutOfRangeException"> The row index must be between 0 and the row count of the block.</exception> 
        /// <exception cref="T:System.IndexOutOfRangeException"> The column index must be between 0 and the column count of the block.</exception>
        public virtual StyleInfo GetStyleInfo(int row, int column, SheetArea sheetArea)
        {
            StorageBlock storage = GetStorage(sheetArea);
            if (storage == null)
            {
                throw new ArgumentException(ResourceStrings.CouldnotFindSpecifiedSheetArea);
            }
            if ((row < -1) || (row >= storage.RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvaildRowIndexWithAllowedRangeBehind, (object[])new object[] { ((int)row), ((int)(storage.RowCount - 1)) }));
            }
            if ((column < -1) || (column >= storage.ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidColumnCountWithAllowedColumnCountBehind, (object[])new object[] { ((int)column), ((int)(storage.ColumnCount - 1)) }));
            }
            object obj2 = GetStyleObject(row, column, sheetArea);
            if (obj2 is StyleInfo)
            {
                return (obj2 as StyleInfo);
            }
            return Name2Style((string)(obj2 as string));
        }

        /// <summary>
        /// Gets the name of the style assigned to a cell, column,
        /// row, or default model in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index of the cell in the model, or -1 to specify an entire column or the default model.</param>
        /// <param name="column">The column index of the cell in the model, or -1 to specify an entire row or the default model.</param>
        /// <returns>Returns the name string for the style.</returns>
        public string GetStyleName(int row, int column)
        {
            return GetStyleName(row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets the name of the style assigned to a cell, column,
        /// row, or default model in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index of the cell in the model, or -1 to specify an entire column or the default model.</param>
        /// <param name="column">The column index of the cell in the model, or -1 to specify an entire row or the default model.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the name string for the style.</returns>
        public string GetStyleName(int row, int column, SheetArea sheetArea)
        {
            StorageBlock storage = GetStorage(sheetArea);
            if (storage == null)
            {
                throw new ArgumentException(ResourceStrings.CouldnotFindSpecifiedSheetArea);
            }
            if ((row < -1) || (row >= storage.RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvaildRowIndexWithAllowedRangeBehind, (object[])new object[] { ((int)row), ((int)(storage.RowCount - 1)) }));
            }
            if ((column < -1) || (column >= storage.ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidColumnCountWithAllowedColumnCountBehind, (object[])new object[] { ((int)column), ((int)(storage.ColumnCount - 1)) }));
            }
            object obj2 = GetStyleObject(row, column, sheetArea);
            if (obj2 is StyleInfo)
            {
                return ((StyleInfo)obj2).Name;
            }
            return (string)(obj2 as string);
        }

        internal object GetStyleObject(int row, int column, SheetArea sheetArea)
        {
            if ((row >= 0) && (column >= 0))
            {
                StorageBlock storage = GetStorage(sheetArea);
                if (storage != null)
                {
                    return storage.GetStyle(row, column);
                }
            }
            else if (column >= 0)
            {
                if (sheetArea == SheetArea.Cells)
                {
                    return _viewportColumnsStyles[column];
                }
                if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    return _rowHeaderColumnsStyles[column];
                }
                if (sheetArea == SheetArea.ColumnHeader)
                {
                    return _columnHeaderColumnsStyles[column];
                }
            }
            else if (row >= 0)
            {
                if (sheetArea == SheetArea.Cells)
                {
                    return _viewportRowsStyles[row];
                }
                if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    return _rowHeaderRowsStyles[row];
                }
                if (sheetArea == SheetArea.ColumnHeader)
                {
                    return _columnHeaderRowsStyles[row];
                }
            }
            else
            {
                if (sheetArea == SheetArea.Cells)
                {
                    return _defaultStyle;
                }
                if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    return _rowHeaderDefaultStyle;
                }
                if (sheetArea == SheetArea.ColumnHeader)
                {
                    return _columnHeaderDefaultStyle;
                }
            }
            return null;
        }

        /// <summary>
        /// Gets the index of the swap.
        /// </summary>
        /// <param name="array">The index array</param>
        /// <param name="start">The start index</param>
        /// <param name="index">The index of row/column</param>
        /// <param name="count">The item count</param>
        /// <param name="row">if set to <c>true</c> [row]</param>
        /// <returns>Returns swap index.</returns>
        int GetSwapIndex(int[] array, int start, int index, int count, bool row)
        {
            if (row)
            {
                int viewRowFromModelRow = GetViewRowFromModelRow(array[index - start]);
                while (viewRowFromModelRow < index)
                {
                    viewRowFromModelRow = GetViewRowFromModelRow(array[viewRowFromModelRow - start]);
                }
                return viewRowFromModelRow;
            }
            int viewColumnFromModelColumn = GetViewColumnFromModelColumn(array[index - start]);
            while (viewColumnFromModelColumn < index)
            {
                viewColumnFromModelColumn = GetViewColumnFromModelColumn(array[viewColumnFromModelColumn - start]);
            }
            return viewColumnFromModelColumn;
        }

        /// <summary>
        /// Gets the sheet tables.
        /// </summary>
        /// <returns>The array of table instances. The array is never null.</returns>
        public SheetTable[] GetTables()
        {
            if (_tables != null)
            {
                return _tables.ToArray();
            }
            return new SheetTable[0];
        }

        /// <summary>
        /// Gets the application-defined tag value for the specified cell, column, row, or entire sheet in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the tag for the specified cell.</returns>
        public object GetTag(int row, int column)
        {
            return GetTag(row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets the application-defined tag value for the specified cell, column, row, or entire sheet in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the tag for the specified cell.</returns>
        public object GetTag(int row, int column, SheetArea sheetArea)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if ((row >= 0) && (column >= 0))
            {
                StorageBlock storage = GetStorage(sheetArea);
                if (storage != null)
                {
                    return storage.GetTag(row, column);
                }
            }
            else if (column >= 0)
            {
                AxisInfo info = GetColumnsStorage(sheetArea)[column];
                if (info != null)
                {
                    return info.Tag;
                }
            }
            else if (row >= 0)
            {
                AxisInfo info2 = GetRowsStorage(sheetArea)[row];
                if (info2 != null)
                {
                    return info2.Tag;
                }
            }
            return null;
        }

        /// <summary>
        /// Gets the formatted text in the cell 
        /// using the <see cref="T:IFormatter" /> object for the cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the formatted text of the cell.</returns>
        public string GetText(int row, int column)
        {
            return GetText(row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets the formatted text in the cell in the specified sheet area 
        /// using the <see cref="T:IFormatter" /> object for the cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the formatted text of the cell.</returns>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified row index is out of range; must be between 0 and the total number of rows.
        /// </exception>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified column index is out of range; must be between 0 and the total number of columns.
        /// </exception>
        public string GetText(int row, int column, SheetArea sheetArea)
        {
            object obj2 = GetValue(row, column, sheetArea);
            if (((obj2 == null) && (sheetArea == SheetArea.ColumnHeader)) && ((row == ColumnHeaderAutoTextIndex) || ((row == (GetRowCount(sheetArea) - 1)) && (ColumnHeaderAutoTextIndex == -1))))
            {
                return GetColumnLabel(ColumnHeaderAutoTextIndex, column);
            }
            if (((obj2 == null) && (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))) && ((column == RowHeaderAutoTextIndex) || ((column == (GetColumnCount(sheetArea) - 1)) && (RowHeaderAutoTextIndex == -1))))
            {
                return GetRowLabel(row, RowHeaderAutoTextIndex);
            }
            int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            IFormatter formatter = GetActualFormatter(modelRowFromViewRow, modelColumnFromViewColumn, sheetArea);
            return Value2Text(obj2, formatter);
        }

        /// <summary>
        /// Gets unformatted data from the specified cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>Returns the value of the cell.</returns>
        public object GetValue(int row, int column)
        {
            return GetValue(row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Gets unformatted data from the specified cell in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>Returns the value of the cell.</returns>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified row index is out of range; must be between 0 and the total number of rows.
        /// </exception>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified column index is out of range; must be between 0 and the total number of columns.
        /// </exception>
        public object GetValue(int row, int column, SheetArea sheetArea)
        {
            int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            return GetActualValue(modelRowFromViewRow, modelColumnFromViewColumn, sheetArea);
        }

        internal int GetViewColumnFromModelColumn(int column)
        {
            return column;
        }

        /// <summary>
        /// Gets the height of the viewport.
        /// </summary>
        /// <param name="rowViewportIndex">The index of the row viewport.</param>
        /// <returns>The height of the viewport.</returns>
        public double GetViewportHeight(int rowViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if (rowViewportIndex == -1)
            {
                return FrozenHeight;
            }
            if ((rowViewportIndex >= 0) && (rowViewportIndex < viewportInfo.ViewportHeight.Length))
            {
                return viewportInfo.ViewportHeight[rowViewportIndex];
            }
            if (rowViewportIndex == viewportInfo.ViewportHeight.Length)
            {
                return FrozenTrailingHeight;
            }
            return 0.0;
        }

        /// <summary>
        /// Gets the viewport information of the current sheet.
        /// </summary>
        /// <returns>The viewport information.</returns>
        public ViewportInfo GetViewportInfo()
        {
            if (_viewportInfo == null)
            {
                CreateViewportInfo();
            }
            return _viewportInfo;
        }

        /// <summary>
        /// Gets the viewport left column.
        /// </summary>
        /// <param name="columnViewportIndex">The index of the column viewport.</param>
        /// <returns>The viewport left column.</returns>
        public int GetViewportLeftColumn(int columnViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if ((columnViewportIndex < -1) || (columnViewportIndex > viewportInfo.ColumnViewportCount))
            {
                return 0;
            }
            if (columnViewportIndex == -1)
            {
                return 0;
            }
            if (columnViewportIndex < viewportInfo.ColumnViewportCount)
            {
                return viewportInfo.LeftColumns[columnViewportIndex];
            }
            return Math.Max(FrozenColumnCount, ColumnCount - FrozenTrailingColumnCount);
        }

        /// <summary>
        /// Gets the viewport top row.
        /// </summary>
        /// <param name="rowViewportIndex">The index of the row viewport.</param>
        /// <returns>The viewport top row.</returns>
        public int GetViewportTopRow(int rowViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if ((viewportInfo.RowViewportCount > 0) && (viewportInfo.ColumnViewportCount > 0))
            {
                if (rowViewportIndex == -1)
                {
                    return 0;
                }
                if ((rowViewportIndex >= 0) && (rowViewportIndex < viewportInfo.RowViewportCount))
                {
                    return viewportInfo.TopRows[rowViewportIndex];
                }
                if (rowViewportIndex == viewportInfo.RowViewportCount)
                {
                    return Math.Max(FrozenRowCount, RowCount - FrozenTrailingRowCount);
                }
            }
            return -1;
        }

        /// <summary>
        /// Gets the width of the viewport.
        /// </summary>
        /// <param name="columnViewportIndex">The index of the column viewport.</param>
        /// <returns>The width of the viewport.</returns>
        public double GetViewportWidth(int columnViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if (columnViewportIndex == -1)
            {
                return FrozenWidth;
            }
            if ((columnViewportIndex >= 0) && (columnViewportIndex < viewportInfo.ViewportWidth.Length))
            {
                return viewportInfo.ViewportWidth[columnViewportIndex];
            }
            if (columnViewportIndex == viewportInfo.ViewportWidth.Length)
            {
                return FrozenTrailingWidth;
            }
            return 0.0;
        }

        internal int GetViewRowFromModelRow(int row)
        {
            return row;
        }

        int ICalcSource.GetColumnCount()
        {
            return ColumnCount;
        }

        CalcExpression ICalcSource.GetDefinedName(string name, int row, int column)
        {
            NameInfo info = FindCustomName(name);
            if ((info != null) && (info.Expression != null))
            {
                return info.Expression;
            }
            return null;
        }

        CalcEvaluatorContext ICalcSource.GetEvaluatorContext(CalcLocalIdentity baseAddress)
        {
            int row = 0;
            int column = 0;
            int rowCount = 1;
            int columnCount = 1;
            LocalIdentity2RangeIndexes(baseAddress, out row, out column, out rowCount, out columnCount);
            SheetTable table = FindTable(row, column);
            if (table != null)
            {
                return new TableCalcEvaluatorContext(table, this, baseAddress is CalcRangeIdentity, row, column, rowCount, columnCount);
            }
            return new CalcEvaluatorContext(this, baseAddress is CalcRangeIdentity, row, column, rowCount, columnCount);
        }

        CalcFunction ICalcSource.GetFunction(string functionName)
        {
            return FindCustomFunction(functionName);
        }

        CalcParserContext ICalcSource.GetParserContext(CalcLocalIdentity baseAddress)
        {
            int row = 0;
            int column = 0;
            LocalIdentity2Indexes(baseAddress, out row, out column);
            return new SpreadCalcParserContext(this, _referenceStyle == ReferenceStyle.R1C1, row, column, (_currentUIActionWrapper != null) ? CultureInfo.CurrentCulture : null);
        }

        object ICalcSource.GetReference(CalcLocalIdentity id)
        {
            StorageBlock cellsBlock = _cellsBlock;
            int row = 0;
            int column = 0;
            int rowCount = 1;
            int columnCount = 1;
            if (id is CalcCellIdentity)
            {
                if (!LocalIdentity2Indexes(id, out row, out column))
                {
                    return CalcErrors.Reference;
                }
            }
            else if (id is CalcRangeIdentity)
            {
                CalcRangeIdentity identity = id as CalcRangeIdentity;
                if (identity.IsFullColumn)
                {
                    row = 0;
                    rowCount = cellsBlock.RowCount;
                }
                else
                {
                    row = identity.RowIndex;
                    rowCount = identity.RowCount;
                }
                if (identity.IsFullRow)
                {
                    column = 0;
                    columnCount = cellsBlock.ColumnCount;
                }
                else
                {
                    column = identity.ColumnIndex;
                    columnCount = identity.ColumnCount;
                }
            }
            return new SpreadCalcReference(_standaloneCalcRef, this, row, column, rowCount, columnCount, RowCount, ColumnCount);
        }

        int ICalcSource.GetRowCount()
        {
            return RowCount;
        }

        object ICalcSource.GetValue(CalcLocalIdentity id)
        {
            int row = 0;
            int column = 0;
            if (!LocalIdentity2Indexes(id, out row, out column))
            {
                return null;
            }
            if ((_isValidatingCell && (_validatingRow == row)) && (_validatingColumn == column))
            {
                return _validatingValue;
            }
            return GetActualValue(row, column, SheetArea.Cells);
        }

        void ICalcSource.SetValue(CalcLocalIdentity id, object value)
        {
            if (id is CalcCellIdentity)
            {
                CalcCellIdentity identity = id as CalcCellIdentity;
                SetValueInternal(identity.RowIndex, identity.ColumnIndex, SheetArea.Cells, value, true, true);
            }
            else if (id is CalcRangeIdentity)
            {
                CalcRangeIdentity identity2 = id as CalcRangeIdentity;
                int rowIndex = identity2.RowIndex;
                int columnIndex = identity2.ColumnIndex;
                int rowCount = identity2.RowCount;
                int columnCount = identity2.ColumnCount;
                if (identity2.IsFullRow)
                {
                    columnIndex = 0;
                    columnCount = ColumnCount;
                }
                if (identity2.IsFullColumn)
                {
                    rowIndex = 0;
                    rowCount = RowCount;
                }
                CalcReference reference = value as CalcReference;
                if (reference != null)
                {
                    int num5 = reference.GetRowCount(0);
                    int num6 = reference.GetColumnCount(0);
                    for (int i = 0; i < rowCount; i++)
                    {
                        for (int j = 0; j < columnCount; j++)
                        {
                            object notAvailable;
                            if ((i < num5) && (j < num6))
                            {
                                notAvailable = reference.GetValue(0, i, j);
                            }
                            else if ((num5 == 1) && (j < num6))
                            {
                                notAvailable = reference.GetValue(0, 0, j);
                            }
                            else if ((num6 == 1) && (i < num5))
                            {
                                notAvailable = reference.GetValue(0, i, 0);
                            }
                            else if ((num5 == 1) && (num6 == 1))
                            {
                                notAvailable = reference.GetValue(0, 0, 0);
                            }
                            else
                            {
                                notAvailable = CalcErrors.NotAvailable;
                            }
                            CalcReference objA = notAvailable as CalcReference;
                            if (!object.ReferenceEquals(objA, null))
                            {
                                int num9 = objA.GetRowCount(0);
                                int num10 = objA.GetColumnCount(0);
                                if (((objA.RangeCount != 1) || ((num9 > 1) && (num10 > 1))) || (((i >= num9) || (j >= num10)) || ((rowCount > 1) || (columnCount > 1))))
                                {
                                    notAvailable = CalcErrors.Value;
                                }
                                else if (num10 == 1)
                                {
                                    notAvailable = objA.GetValue(0, i, 0);
                                }
                                else if (num9 == 1)
                                {
                                    notAvailable = objA.GetValue(0, 0, j);
                                }
                                else
                                {
                                    notAvailable = CalcErrors.Value;
                                }
                            }
                            SetValueInternal(i + rowIndex, j + columnIndex, SheetArea.Cells, notAvailable, true, true);
                        }
                    }
                }
                else
                {
                    CalcArray array = value as CalcArray;
                    if (array != null)
                    {
                        int num11 = array.RowCount;
                        int num12 = array.ColumnCount;
                        for (int k = 0; k < rowCount; k++)
                        {
                            for (int m = 0; m < columnCount; m++)
                            {
                                object obj3;
                                if ((k < num11) && (m < num12))
                                {
                                    obj3 = array.GetValue(k, m);
                                }
                                else if ((num11 == 1) && (m < num12))
                                {
                                    obj3 = array.GetValue(0, m);
                                }
                                else if ((num12 == 1) && (k < num11))
                                {
                                    obj3 = array.GetValue(k, 0);
                                }
                                else if ((num11 == 1) && (num12 == 1))
                                {
                                    obj3 = array.GetValue(0, 0);
                                }
                                else
                                {
                                    obj3 = CalcErrors.NotAvailable;
                                }
                                SetValueInternal(k + rowIndex, m + columnIndex, SheetArea.Cells, obj3, true, true);
                            }
                        }
                    }
                    else
                    {
                        for (int n = 0; n < rowCount; n++)
                        {
                            for (int num16 = 0; num16 < columnCount; num16++)
                            {
                                SetValueInternal(n + rowIndex, num16 + columnIndex, SheetArea.Cells, value, true, true);
                            }
                        }
                    }
                }
            }
        }

        IEnumerable<ICalcSource> IMultiSourceProvider.GetCalcSources(ICalcSource startSource, ICalcSource endSource)
        {
            Worksheet item = startSource as Worksheet;
            Worksheet worksheet2 = endSource as Worksheet;
            List<ICalcSource> list = new List<ICalcSource>();
            if (((startSource != null) && (endSource != null)) && (_workbook != null))
            {
                int index = _workbook.Sheets.IndexOf(item);
                int num2 = _workbook.Sheets.IndexOf(worksheet2);
                if ((index != -1) && (num2 != -1))
                {
                    for (int i = index; i <= num2; i++)
                    {
                        list.Add(_workbook.Sheets[i]);
                    }
                }
            }
            return (IEnumerable<ICalcSource>)list;
        }

        void IBindableSheet.GenerateColumns(string[] fields)
        {
            if (fields != null)
            {
                ColumnCount = fields.Length;
                for (int i = 0; i < ColumnCount; i++)
                {
                    _bindingManager.SetBoundField(i, fields[i]);
                }
            }
        }

        object ICalcEvaluator.CreateExpression(string functionName, params object[] parameters)
        {
            Type type = IntrospectionExtensions.GetTypeInfo((Type)typeof(CalcFunctionExpression)).Assembly.GetType(string.Format("{0}.{1}", (object[])new object[] { typeof(CalcFunction).Namespace, functionName }));
            if (type != null)
            {
                CalcFunction calcFunction = null;
                try
                {
                    calcFunction = Activator.CreateInstance(type) as CalcFunction;
                }
                catch
                {
                }
                if (calcFunction != null)
                {
                    CalcExpression[] args = new CalcExpression[(parameters == null) ? 0 : parameters.Length];
                    for (int i = 0; (parameters != null) && (i < parameters.Length); i++)
                    {
                        object obj2 = parameters[i];
                        if (obj2 is CalcExpression)
                        {
                            args[i] = obj2 as CalcExpression;
                        }
                        else if (obj2 is CellRange)
                        {
                            CellRange range = obj2 as CellRange;
                            args[i] = CalcExpressionHelper.CreateRangeExpressionByCount(range.Row, range.Column, range.RowCount, range.ColumnCount, false, false, false, false);
                        }
                        else
                        {
                            if (!(obj2 is double))
                            {
                                throw new NotSupportedException(ResourceStrings.WorksheetCreateExpressionError);
                            }
                            args[i] = new CalcDoubleExpression((double)((double)obj2));
                        }
                    }
                    return new CalcFunctionExpression(calcFunction, args);
                }
            }
            return null;
        }

        object ICalcEvaluator.EvaluateExpression(object expression, int rowOffset, int columnOffset, int baseRow, int baseColumn, bool isArrayFormula)
        {
            if (!(expression is CalcExpression))
            {
                return null;
            }
            CalcEvaluatorContext context = new CalcEvaluatorContext(this, isArrayFormula, baseRow, baseColumn, 1, 1);
            object obj2 = CalcManager.Evaluator.Evaluate(expression as CalcExpression, context);
            if (!isArrayFormula)
            {
                while ((obj2 is CalcReference) || (obj2 is CalcArray))
                {
                    if (obj2 is CalcReference)
                    {
                        CalcCellIdentity localId = new CalcCellIdentity(baseRow, baseColumn);
                        obj2 = ExtractValueFromReference(localId, obj2);
                    }
                    if (obj2 is CalcArray)
                    {
                        obj2 = (obj2 as CalcArray).GetValue(0);
                    }
                }
                return obj2;
            }
            if (obj2 is CalcReference)
            {
                CalcReference reference = obj2 as CalcReference;
                int num = reference.GetRowCount(0);
                int num2 = reference.GetColumnCount(0);
                object[,] objArray = new object[num2, num];
                for (int j = 0; j < num; j++)
                {
                    for (int k = 0; k < num2; k++)
                    {
                        objArray[k, j] = reference.GetValue(0, j, k);
                    }
                }
                return objArray;
            }
            if (!(obj2 is CalcArray))
            {
                return obj2;
            }
            CalcArray array = obj2 as CalcArray;
            int rowCount = array.RowCount;
            int columnCount = array.ColumnCount;
            object[,] objArray2 = new object[columnCount, rowCount];
            for (int i = 0; i < rowCount; i++)
            {
                for (int m = 0; m < columnCount; m++)
                {
                    objArray2[m, i] = array.GetValue(i, m);
                }
            }
            return objArray2;
        }

        object ICalcEvaluator.EvaluateFunction(object function, params object[] parameters)
        {
            if (!(function is CalcFunction))
            {
                return null;
            }
            object[] args = new object[parameters.Length];
            for (int i = 0; i < parameters.Length; i++)
            {
                object obj2 = parameters[i];
                if (obj2 is CellRange)
                {
                    CellRange range = obj2 as CellRange;
                    args[i] = new SpreadCalcReference(_standaloneCalcRef, this, range.Row, range.Column, range.RowCount, range.ColumnCount, RowCount, ColumnCount);
                }
                else if (obj2 is object[,])
                {
                    args[i] = new ObjectCalcArray(obj2 as object[,]);
                }
                else
                {
                    args[i] = obj2;
                }
            }
            return ((CalcFunction)function).Evaluate(args);
        }

        string ICalcEvaluator.Expression2Formula(object expression, int baseRow, int baseColumn)
        {
            CalcParserContext context;
            if (!(expression is CalcExpression))
            {
                return string.Empty;
            }
            CalcLocalIdentity id = CalcExpressionHelper.CreateRangeExpressionByCount(baseRow, baseColumn, 1, 1, false, false, false, false).GetId(0, 0) as CalcLocalIdentity;
            context = context = ((ICalcSource)this).GetParserContext(id);
            if (_calcManager != null)
            {
                return _calcManager.Parser.Unparse(expression as CalcExpression, context);
            }
            return _cachedParser.Unparse(expression as CalcExpression, context);
        }

        object ICalcEvaluator.Formula2Expression(string formula, int baseRow, int baseColumn)
        {
            CalcParserContext context;
            CalcLocalIdentity id = CalcExpressionHelper.CreateRangeExpressionByCount(baseRow, baseColumn, 1, 1, false, false, false, false).GetId(0, 0) as CalcLocalIdentity;
            context = context = ((ICalcSource)this).GetParserContext(id);
            if (_calcManager != null)
            {
                return _calcManager.Parser.Parse(formula, context);
            }
            return _cachedParser.Parse(formula, context);
        }

        bool ICalcEvaluator.IsCalcError(object value)
        {
            return (value is CalcError);
        }

        void ICustomNameSupport.EnumerateAllNames(out Dictionary<string, CalcExpression> workbookNames, out Dictionary<ICustomNameSupport, Dictionary<string, CalcExpression>> worksheetNames)
        {
            workbookNames = new Dictionary<string, CalcExpression>();
            worksheetNames = new Dictionary<ICustomNameSupport, Dictionary<string, CalcExpression>>();
            if (Workbook != null)
            {
                foreach (NameInfo info in (IEnumerable<NameInfo>)Workbook.Names)
                {
                    workbookNames.Add(info.Name, info.Expression);
                }
                if (Workbook.Sheets != null)
                {
                    foreach (Worksheet worksheet in Workbook.Sheets)
                    {
                        Dictionary<string, CalcExpression> dictionary = new Dictionary<string, CalcExpression>();
                        worksheetNames.Add(worksheet, dictionary);
                        foreach (NameInfo info2 in (IEnumerable<NameInfo>)worksheet.Names)
                        {
                            dictionary.Add(info2.Name, info2.Expression);
                        }
                    }
                }
            }
        }

        void ICustomNameSupport.SetDefinedName(string name, CalcExpression expression, bool isWorkbookName)
        {
            NameInfoCollection infos = (isWorkbookName && (_workbook != null)) ? _workbook.Names : _names;
            NameInfo info = (infos == null) ? null : infos.Find(name);
            if (info != null)
            {
                info.Expression = expression;
            }
        }

        void IPrintSupportInternal.WriteXml(Stream stream)
        {
            SaveXmlForPrint(stream);
        }

        Sparkline ISparklineSheet.GetSparkline(int row, int column)
        {
            return GetSparkline(row, column);
        }

        void ISparklineSheet.SetSparkline(int row, int column, Sparkline sparkline)
        {
            GetStorage(SheetArea.Cells).SetSparkline(row, column, sparkline);
        }

        bool ISubtotalSupport.IsSubtotal(SpreadCalcReference reference, int range, int rowOffset, int columnOffset)
        {
            if (_calcManager == null)
            {
                return false;
            }
            CalcFunctionExpression expression = _calcManager.GetExpression(new CalcCellIdentity(rowOffset, columnOffset)) as CalcFunctionExpression;
            return ((expression != null) && (expression.Function is CalcSubtotalFunction));
        }

        double ITableSheet.GetActualColumnWidth(int column)
        {
            return GetActualColumnWidth(column, SheetArea.Cells);
        }

        double ITableSheet.GetActualRowHeight(int row)
        {
            return GetActualRowHeight(row, SheetArea.Cells);
        }

        string ITableSheet.GetCellFormula(int row, int column)
        {
            return GetFormula(row, column);
        }

        /// <summary>
        /// Gets the value from the specified cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>The value of the specified cell in the sheet.</returns>
        string ITableSheet.GetCellText(int row, int column)
        {
            object obj2 = null;
            if (((_bindingManager != null) && _bindingManager.IsBoundRow(row)) && _bindingManager.IsBoundColumn(column))
            {
                obj2 = _bindingManager.GetValue(row, column);
            }
            else
            {
                StorageBlock storage = GetStorage(SheetArea.Cells);
                if (storage != null)
                {
                    obj2 = storage.GetValue(row, column);
                }
            }
            if (obj2 != null)
            {
                return obj2.ToString();
            }
            return null;
        }

        void ITableSheet.Move(int fromRow, int fromColumn, int toRow, int toColumn, int rowCount, int columnCount)
        {
            MoveTo(fromRow, fromColumn, toRow, toColumn, rowCount, columnCount, CopyToOption.Value);
        }

        void ITableSheet.SetCellFormula(int row, int column, string formula)
        {
            try
            {
                _isSettingFormula = true;
                int modelRowFromViewRow = GetModelRowFromViewRow(row, SheetArea.Cells);
                int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, SheetArea.Cells);
                CalcLocalIdentity id = new CalcCellIdentity(modelRowFromViewRow, modelColumnFromViewColumn);
                if (string.IsNullOrEmpty(formula))
                {
                    if (!string.IsNullOrEmpty(CalcManager.GetFormula(id)))
                    {
                        CalcManager.ClearFormula(id);
                        SetValue(row, column, SheetArea.Cells, null);
                    }
                }
                else
                {
                    CalcExpressionHelper.SetFormula(CalcManager, id, formula, false);
                    if (_calcEngineSuspended == 0)
                    {
                        CalcManager.Invalidate(id, AutoRecalculation);
                    }
                }
            }
            finally
            {
                _isSettingFormula = false;
            }
        }

        /// <summary>
        /// Sets a value to a specified cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="value">The value.</param>
        void ITableSheet.SetCellText(int row, int column, string value)
        {
            if (((_bindingManager != null) && _bindingManager.IsBoundRow(row)) && _bindingManager.IsBoundColumn(column))
            {
                _bindingManager.SetValue(row, column, value);
            }
            else
            {
                GetStorage(SheetArea.Cells).SetValue(row, column, value);
                RaiseCellChanged("[TableText]", row, column, SheetArea.Cells);
            }
        }

        Color IThemeSupport.GetThemeColor(string themeName)
        {
            if (_workbook != null)
            {
                return _workbook.GetThemeColor(themeName);
            }
            return Colors.Transparent;
        }

        FontFamily IThemeSupport.GetThemeFont(string themeName)
        {
            if (_workbook != null)
            {
                return _workbook.GetThemeFont(themeName);
            }
            return null;
        }

        /// <summary>
        /// Begins the UI action.
        /// </summary>
        /// <returns></returns>
        IDisposable IUIActionExecuter.BeginUIAction()
        {
            return (_currentUIActionWrapper = new UIActionWrapper(this));
        }

        /// <summary>
        /// Groups the sparklines.
        /// </summary>
        /// <param name="sparklines">The sparklines to group.</param>
        /// <returns>The group.</returns>
        public SparklineGroup GroupSparkline(Sparkline[] sparklines)
        {
            SparklineGroup group = null;
            foreach (Sparkline sparkline in sparklines)
            {
                if (sparkline != null)
                {
                    if (group == null)
                    {
                        group = sparkline.Group;
                    }
                    else
                    {
                        SparklineGroup group2 = sparkline.Group;
                        group2.Remove(sparkline);
                        group.Add(sparkline);
                        if (group2.Count <= 0)
                        {
                            SheetSparklineGroupManager.Remove(group2);
                        }
                    }
                }
            }
            return group;
        }

        /// <summary>
        /// Converts a row or column index to a text represent ion
        /// consisting of letters.
        /// </summary>
        /// <param name="index">The index</param>
        /// <returns>Returns letter by index.</returns>
        static string IndexToLetters(int index)
        {
            StringBuilder builder = new StringBuilder();
            while (index > 0)
            {
                builder.Append((char)(0x41 + ((index - 1) % 0x1a)));
                index = (index - 1) / 0x1a;
            }
            for (int i = 0; i < (builder.Length / 2); i++)
            {
                char ch = builder[i];
                builder[i] = builder[(builder.Length - i) - 1];
                builder[(builder.Length - i) - 1] = ch;
            }
            return builder.ToString();
        }

        void Init(int rowCount, int columnCount, int rowHeaderColumnCount, int columnHeaderRowCount, bool keepSetting)
        {
            if (!keepSetting)
            {
                _name = null;
                _visible = true;
                _columnHeaderVisible = true;
                _rowHeaderVisible = true;
                _columnFooterVisible = false;
                _startingColumnNumber = 1;
                _startingRowNumber = 1;
                _columnHeaderAutoText = HeaderAutoText.Letters;
                _rowHeaderAutoText = HeaderAutoText.Numbers;
                _columnHeaderAutoTextIndex = -1;
                _rowHeaderAutoTextIndex = -1;
                _defaultColumnFooterRowHeight = _DEFAULT_COLUMN_FOOTER_ROWHEIGHT;
                _defaultColumnHeaderRowHeight = _DEFAULT_COLUMN_HEADER_ROWHEIGHT;
                _defaultRowHeight = _DEFAULT_ROW_HEIGHT;
                _defaultRowHeaderColumnWidth = _DEFAULT_ROW_HEADER_COLUMNWIDTH;
                _defaultColumnWidth = _DEFAULT_COLUMN_WIDTH;
            }
            _frozenRowCount = 0;
            _frozenColumnCount = 0;
            _frozenTrailingColumnCount = 0;
            _frozenTrailingRowCount = 0;
            if (!keepSetting)
            {
                _referenceStyle = ReferenceStyle.A1;
                if (_names != null)
                {
                    _names.Changed -= new EventHandler<NameInfoCollectionChangedEventArgs>(OnNameInfoCollectionChanged);
                    _names = null;
                }
                _functions = null;
            }
            _autoRecalculation = true;
            _calcEngineSuspended = 0;
            if (!keepSetting)
            {
                _zoomFactor = 1f;
                _autoGeneratedColumns = true;
                _dataAutoHeadings = true;
            }
            if (!keepSetting)
            {
                if (_namedStyles != null)
                {
                    _namedStyles.Changed -= new EventHandler<StyleInfoCollectionChangedEventArgs>(OnNamedStyleChanged);
                    _namedStyles = null;
                }
                if (_defaultStyle != null)
                {
                    _defaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                    _defaultStyle = null;
                }
                if (_columnHeaderDefaultStyle != null)
                {
                    _columnHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                    _columnHeaderDefaultStyle = null;
                }
                if (_rowHeaderDefaultStyle != null)
                {
                    _rowHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                    _rowHeaderDefaultStyle = null;
                }
                _gridLineColor = DefaultGridLineColor;
                _gridLine = null;
                _defaultGridLine = null;
                _showGridLine = true;
                _sheetTabThemeColor = null;
                _sheetTabColor = DefaultSheetTabColor;
            }
            _cellsBlock = new StorageBlock(rowCount, columnCount);
            _rowHeadersBlock = new StorageBlock(rowCount, rowHeaderColumnCount);
            _columnHeadersBlock = new StorageBlock(columnHeaderRowCount, columnCount);
            _viewportRows = new SparseArray<AxisInfo>(rowCount);
            _viewportColumns = new SparseArray<AxisInfo>(ColumnCount);
            _rowHeaderColumns = new SparseArray<AxisInfo>(rowHeaderColumnCount);
            _columnHeaderRows = new SparseArray<AxisInfo>(columnHeaderRowCount);
            _viewportRowsStyles = new SparseArray<object>(rowCount);
            _viewportColumnsStyles = new SparseArray<object>(ColumnCount);
            _rowHeaderRowsStyles = new SparseArray<object>(rowCount);
            _rowHeaderColumnsStyles = new SparseArray<object>(rowHeaderColumnCount);
            _columnHeaderRowsStyles = new SparseArray<object>(columnHeaderRowCount);
            _columnHeaderColumnsStyles = new SparseArray<object>(ColumnCount);
            _spanModel = new SheetSpanModel();
            _rowHeaderSpanModel = new SheetSpanModel();
            _columnHeaderSpanModel = new SheetSpanModel();
            if (_selectionModel == null)
            {
                _selectionModel = new SheetSelectionModel();
                _selectionModel.Changed += new EventHandler<SheetSelectionChangedEventArgs>(selectionModel_Changed);
            }
            else
            {
                _selectionModel.ClearSelection();
            }
            _selectionModel.SetAnchor(0, 0);
            _calcStorage = new SpreadCalcStorage();
            _standaloneCalcRef = new SpreadCalcReference(this);
            _cachedDeserializedCustomNames = null;
            _cachedDeserializedFormulaStrings = null;
            _bindingManager = null;
            _datasource = null;
            _cachedDeserializedBindingValues = null;
            _cachedDeserializedDatasource = null;
            if (_rowFilter != null)
            {
                _rowFilter.Changed -= new EventHandler<FilterEventArgs>(OnFilterStatusChanged);
                _rowFilter = null;
            }
            if (_rowRangeGroup != null)
            {
                _rowRangeGroup.Changed -= new EventHandler(OnRowRangeGroupChanged);
                _rowRangeGroup = null;
            }
            if (_columnRangeGroup != null)
            {
                _columnRangeGroup.Changed -= new EventHandler(OnColumnRangeGroupChanged);
                _columnRangeGroup = null;
            }
            if (_conditionalFormats != null)
            {
                _conditionalFormats.RulesChanged -= new EventHandler<RulesChangedEventArgs>(OnConditionalFormatsRulesChanged);
                _conditionalFormats = null;
            }
            if (_sparklineGroupManager != null)
            {
                _sparklineGroupManager.Dispose();
            }
            _sparklineGroupManager = null;
            _printInfo = null;
            _tables = null;
            if (_calcManager != null)
            {
                _calcManager.Dispose();
            }
            _calcManager = null;
            _calcService = null;
            _calcAxial = null;
            _sortImp = new SortImp(this);
            _columnHeader = null;
            _rowHeader = null;
            _cells = null;
            _columns = null;
            _rows = null;
            _viewportInfo = null;
            if (!keepSetting)
            {
                _protect = false;
            }
            _selectionBorderColor = Color.FromArgb(220, 0, 0, 0);
            _touchSelectionGripperBackgroundColor = Color.FromArgb(0xff, 0xff, 0xff, 0xff);
            _selectionBorderThemeColor = null;
            _touchSelectionGripperBackgroundThemeColor = null;
            if (_spreadCharts != null)
            {
                _spreadCharts.Dispose();
            }
            _spreadCharts = null;
            if (_floatingObjects != null)
            {
                _floatingObjects.Dispose();
            }
            _floatingObjects = null;
            if (_pictures != null)
            {
                _pictures.Dispose();
            }
            _pictures = null;
            _unsupportedChartList = null;
            _unsupportedRecords = null;
            _tag = null;
        }

        /// <summary>
        /// Removes the sparkline of the specified cell.
        /// </summary>
        /// <param name="row">The row.</param>
        /// <param name="column">The column.</param>
        internal void InternalRemoveSparkline(int row, int column)
        {
            StorageBlock storage = GetStorage(SheetArea.Cells);
            Sparkline item = storage.GetSparkline(row, column);
            if (item != null)
            {
                SparklineGroup group = item.Group;
                group.Remove(item);
                if (group.Count <= 0)
                {
                    SheetSparklineGroupManager.Remove(group);
                }
            }
            storage.SetSparkline(row, column, null);
        }

        internal void InvalidateAfterRecalculate()
        {
            if (_rowRangeGroup != null)
            {
                _rowRangeGroup.Refresh();
            }
            if (_columnRangeGroup != null)
            {
                _columnRangeGroup.Refresh();
            }
            if (_sparklineGroupManager != null)
            {
                _sparklineGroupManager.Sheet = this;
                _sparklineGroupManager.CalcEvaluator = this;
                StorageBlock storage = GetStorage(SheetArea.Cells);
                foreach (SparklineGroup group in _sparklineGroupManager.Groups)
                {
                    for (int i = 0; i < group.Count; i++)
                    {
                        Sparkline sparkline = group[i];
                        storage.SetSparkline(sparkline.Row, sparkline.Column, sparkline);
                    }
                }
                _sparklineGroupManager.ResumeAfterDeserialization();
            }
            if ((_namedStyles == null) && (_workbook != null))
            {
                NamedStyles = _workbook.NamedStyles;
            }
            if (_tables != null)
            {
                for (int j = 0; j < _tables.Count; j++)
                {
                    _tables[j].UpdateFooter(1);
                    _tables[j].UpdateFilter();
                }
                _tables.SheetTableChanged += new EventHandler<TableChangedArgs>(OnSheetTableChanged);
            }
            if (_rowFilter != null)
            {
                _rowFilter.ReFilter();
            }
            if (_spreadCharts != null)
            {
                _spreadCharts.ResumeAfterDeserialization();
            }
            if (_floatingObjects != null)
            {
                _floatingObjects.ResumeAfterDeserialization();
            }
            if (_pictures != null)
            {
                _pictures.ResumeAfterDeserialization();
            }
        }

        internal void InvalidateBeforeRecalculate()
        {
            if (_bindingManager != null)
            {
                _bindingManager.Sheet = this;
            }
            if (_cachedDeserializedDatasource != null)
            {
                if (_bindingManager == null)
                {
                    DataSource = _cachedDeserializedDatasource;
                }
                else
                {
                    _datasource = _cachedDeserializedDatasource;
                    _bindingManager.ReOpenDataSource(_datasource);
                    _bindingManager.IsBound = true;
                }
            }
            _cachedDeserializedDatasource = null;
            if ((_bindingManager != null) && (_cachedDeserializedBindingValues != null))
            {
                List<int> nonEmptyRows = _cachedDeserializedBindingValues.GetNonEmptyRows();
                for (int i = 0; i < nonEmptyRows.Count; i++)
                {
                    int modelRowIndex = nonEmptyRows[i];
                    if (_bindingManager.GetDataSourceRowFromModelRow(modelRowIndex) >= 0)
                    {
                        for (int j = 0; j < _cachedDeserializedBindingValues.ColumnCount; j++)
                        {
                            if (!string.IsNullOrEmpty(_bindingManager.GetDataSourceColumnFromModelColumn(j)))
                            {
                                object obj2 = _cachedDeserializedBindingValues.GetValue(modelRowIndex, j);
                                SetValue(modelRowIndex, j, obj2);
                            }
                        }
                    }
                }
            }
            if (_names == null)
            {
                _names = new NameInfoCollection();
                _names.Changed += new EventHandler<NameInfoCollectionChangedEventArgs>(OnNameInfoCollectionChanged);
            }
            Serializer.UpdateCutomNameString(_names, this, _cachedDeserializedCustomNames);
            _cachedDeserializedCustomNames = null;
            UpadteFormulaString();
            if (_calcStorage != null)
            {
                SuspendCalcService();
                try
                {
                    _calcManager = FormulaService.GetCalculationManager(this, _calcStorage, true);
                    if ((_calcAxial != null) && (_calcAxial.Manager != _calcManager))
                    {
                        _calcAxial = null;
                    }
                }
                catch
                {
                    ResumeCalcService();
                    throw;
                }
            }
        }

        /// <summary>
        /// Invalidates all formulas that are intersected with a specified range.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        public void InvalidateFormulas(int row, int column, int rowCount, int columnCount)
        {
            if (_calcManager != null)
            {
                CalcManager.Invalidate(new CalcRangeIdentity(row, column, rowCount, columnCount), true);
            }
        }

        internal void InvalidateLoadedData()
        {
            InvalidateBeforeRecalculate();
            ResumeCalcService();
            InvalidateAfterRecalculate();
        }

        /// <summary>
        /// Determines whether any cell in the specified column is in the selection.
        /// </summary>
        /// <param name="column">The column index of the column to check.</param>
        /// <returns>
        /// <c>true</c> if any cell in a specified column is in the selection; otherwise, <c>false</c>.
        /// </returns>
        public bool IsAnyCellInColumnSelected(int column)
        {
            return Selection.IsAnyCellInColumnSelected(column);
        }

        /// <summary>
        /// Determines whether any cell in the specified row is in the selection.
        /// </summary>
        /// <param name="row">The row index of the row to check.</param>
        /// <returns>
        /// <c>true</c> if any cell in the specified row is in the selection; otherwise, <c>false</c>.
        /// </returns>
        public bool IsAnyCellInRowSelected(int row)
        {
            return Selection.IsAnyCellInRowSelected(row);
        }

        /// <summary>
        /// Gets whether the specified column is bound to a data source.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <returns>
        /// <c>true</c> if the column is bound; otherwise, <c>false</c>.
        /// </returns>
        public bool IsColumnBound(int column)
        {
            if ((column < 0) || (column >= ColumnCount))
            {
                return false;
            }
            return ((_bindingManager != null) && _bindingManager.IsBoundColumn(column));
        }

        static bool IsContains(CalcRangeIdentity outRange, CalcRangeIdentity inRange)
        {
            if ((!outRange.IsFullRow && ((inRange.IsFullRow || (outRange.ColumnIndex > inRange.ColumnIndex)) || ((outRange.ColumnIndex + outRange.ColumnCount) < (inRange.ColumnIndex + inRange.ColumnCount)))) && !outRange.IsFullColumn)
            {
                return ((!inRange.IsFullColumn && (outRange.RowIndex <= inRange.RowIndex)) && ((outRange.RowIndex + outRange.RowCount) >= (inRange.RowIndex + inRange.RowCount)));
            }
            return true;
        }

        /// <summary>
        /// Determines whether [is event suspend].
        /// </summary>
        /// <returns>
        /// <c>true</c> if [is event suspend]; otherwise, <c>false</c>
        /// </returns>
        internal bool IsEventSuspend()
        {
            if (Workbook != null)
            {
                return Workbook.IsEventSuspend();
            }
            return true;
        }

        /// <summary>
        /// Gets whether the specified row is bound to a data source.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <returns>
        /// <c>true</c> if the row is bound; otherwise, <c>false</c>.
        /// </returns>
        public bool IsRowBound(int row)
        {
            if ((row < 0) || (row >= RowCount))
            {
                return false;
            }
            return ((_bindingManager != null) && _bindingManager.IsBoundRow(row));
        }

        internal bool IsRowFilteredOut(int row)
        {
            if (((_rowFilter != null) && _rowFilter.IsFiltered) && _rowFilter.IsRowFilteredOut(row))
            {
                return true;
            }
            foreach (SheetTable table in GetTables())
            {
                if (((table != null) && (table.RowFilter != null)) && table.RowFilter.IsRowFilteredOut(row))
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// Determines whether the specified cell is in the selection.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <returns>
        /// <c>true</c> if the specified row is selected; otherwise, <c>false</c>.
        /// </returns>
        public bool IsSelected(int row, int column)
        {
            return Selection.IsSelected(row, column);
        }

        /// <summary>
        /// Determines whether the specified cell value is valid.
        /// </summary>
        /// <param name="row">The row.</param>
        /// <param name="column">The column.</param>
        /// <param name="value">The value.</param>
        /// <returns>
        /// <c>true</c> if the specified cell value is valid; otherwise, <c>false</c>.
        /// </returns>
        public bool IsValid(int row, int column, object value)
        {
            int modelRowFromViewRow = GetModelRowFromViewRow(row);
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column);
            _validatingRow = modelRowFromViewRow;
            _validatingColumn = modelColumnFromViewColumn;
            _validatingValue = value;
            _isValidatingCell = true;
            try
            {
                StyleInfo info = GetActualStyleInfo(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells);
                if ((info != null) && (info.DataValidator != null))
                {
                    return info.DataValidator.IsValid(this, row, column, value);
                }
            }
            finally
            {
                _validatingRow = -1;
                _validatingColumn = -1;
                _validatingValue = null;
                _isValidatingCell = false;
            }
            return true;
        }

        /// <summary>
        /// Converts a column A1 notation string to an index.
        /// </summary>
        /// <param name="letterNotation">The letter notation</param>
        /// <returns>Returns index by letter.</returns>
        static int LettersToIndex(string letterNotation)
        {
            int num = 0;
            string str = (letterNotation != null) ? letterNotation.ToUpper(CultureInfo.InvariantCulture) : "";
            string str2 = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
            int num2 = 0;
            for (num = str.Length - 1; num >= 0; num--)
            {
                num2 += (str2.IndexOf(str[num]) + 1) * ((int)Math.Pow(26.0, (double)((str.Length - 1) - num)));
            }
            num2--;
            return num2;
        }

        bool LocalIdentity2Indexes(CalcLocalIdentity baseAddress, out int row, out int column)
        {
            row = 0;
            column = 0;
            if (baseAddress is CalcCellIdentity)
            {
                CalcCellIdentity identity = baseAddress as CalcCellIdentity;
                row = identity.RowIndex;
                column = identity.ColumnIndex;
                if (column < 0)
                {
                    column = (column % ColumnCount) + ColumnCount;
                }
                else if (column >= ColumnCount)
                {
                    column = column % ColumnCount;
                }
                if (row < 0)
                {
                    row = (row % RowCount) + RowCount;
                }
                else if (row >= RowCount)
                {
                    row = row % RowCount;
                }
                return true;
            }
            if (baseAddress is CalcRangeIdentity)
            {
                CalcRangeIdentity identity2 = baseAddress as CalcRangeIdentity;
                row = identity2.IsFullColumn ? 0 : identity2.RowIndex;
                column = identity2.IsFullRow ? 0 : identity2.ColumnIndex;
                return true;
            }
            return false;
        }

        bool LocalIdentity2RangeIndexes(CalcLocalIdentity baseAddress, out int row, out int column, out int rowCount, out int columnCount)
        {
            if (baseAddress is CalcCellIdentity)
            {
                rowCount = columnCount = 1;
                return LocalIdentity2Indexes(baseAddress, out row, out column);
            }
            if (baseAddress is CalcRangeIdentity)
            {
                CalcRangeIdentity identity = baseAddress as CalcRangeIdentity;
                row = identity.IsFullColumn ? 0 : identity.RowIndex;
                column = identity.IsFullRow ? 0 : identity.ColumnIndex;
                rowCount = identity.IsFullColumn ? RowCount : identity.RowCount;
                columnCount = identity.IsFullRow ? ColumnCount : identity.ColumnCount;
                return true;
            }
            row = column = 0;
            rowCount = columnCount = 1;
            return false;
        }

        internal void MergeColumnViewports(int columnViewportIndex, int mergeInColumnViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if (((columnViewportIndex > -1) && (columnViewportIndex < viewportInfo.ColumnViewportCount)) && (viewportInfo.ColumnViewportCount != 1))
            {
                ViewportInfo info2 = new ViewportInfo(this, viewportInfo.RowViewportCount, viewportInfo.ColumnViewportCount - 1);
                if (mergeInColumnViewportIndex == viewportInfo.ActiveColumnViewport)
                {
                    info2.ActiveColumnViewport = columnViewportIndex;
                }
                else if (mergeInColumnViewportIndex > viewportInfo.ActiveColumnViewport)
                {
                    info2.ActiveColumnViewport = viewportInfo.ActiveColumnViewport;
                }
                else
                {
                    info2.ActiveColumnViewport = viewportInfo.ActiveColumnViewport - 1;
                }
                if (info2.ActiveColumnViewport >= info2.ColumnViewportCount)
                {
                    info2.ActiveColumnViewport = info2.ColumnViewportCount - 1;
                }
                else if (info2.ActiveColumnViewport < 0)
                {
                    info2.ActiveColumnViewport = 0;
                }
                info2.ActiveRowViewport = viewportInfo.ActiveRowViewport;
                viewportInfo.TopRows.CopyTo(info2.TopRows, 0);
                viewportInfo.ViewportHeight.CopyTo(info2.ViewportHeight, 0);
                for (int i = 0; i < columnViewportIndex; i++)
                {
                    info2.LeftColumns[i] = viewportInfo.LeftColumns[i];
                    info2.ViewportWidth[i] = viewportInfo.ViewportWidth[i];
                }
                for (int j = columnViewportIndex + 1; j < viewportInfo.ColumnViewportCount; j++)
                {
                    info2.LeftColumns[j - 1] = viewportInfo.LeftColumns[j];
                    info2.ViewportWidth[j - 1] = viewportInfo.ViewportWidth[j];
                }
                int index = (mergeInColumnViewportIndex > columnViewportIndex) ? (mergeInColumnViewportIndex - 1) : mergeInColumnViewportIndex;
                if ((viewportInfo.ViewportWidth[columnViewportIndex] != 0.0) && ((viewportInfo.ViewportWidth[columnViewportIndex] != -1.0) || (viewportInfo.ViewportWidth[mergeInColumnViewportIndex] != -1.0)))
                {
                    double viewportWidth = GetViewportWidth(mergeInColumnViewportIndex);
                    double num5 = GetViewportWidth(columnViewportIndex);
                    info2.ViewportWidth[index] = viewportWidth + num5;
                }
                info2.ViewportWidth[info2.ColumnViewportCount - 1] = -1.0;
                SetViewportInfo(info2);
            }
        }

        internal void MergeRowViewports(int rowViewportIndex, int mergeInRowViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if (((rowViewportIndex > -1) && (rowViewportIndex < viewportInfo.RowViewportCount)) && (viewportInfo.RowViewportCount != 1))
            {
                ViewportInfo info2 = new ViewportInfo(this, viewportInfo.RowViewportCount - 1, viewportInfo.ColumnViewportCount)
                {
                    ActiveColumnViewport = viewportInfo.ActiveColumnViewport
                };
                if (mergeInRowViewportIndex == viewportInfo.ActiveRowViewport)
                {
                    info2.ActiveRowViewport = rowViewportIndex;
                }
                else if (mergeInRowViewportIndex > viewportInfo.ActiveRowViewport)
                {
                    info2.ActiveRowViewport = viewportInfo.ActiveRowViewport;
                }
                else
                {
                    info2.ActiveRowViewport = viewportInfo.ActiveRowViewport - 1;
                }
                if (info2.ActiveRowViewport >= info2.RowViewportCount)
                {
                    info2.ActiveRowViewport = info2.RowViewportCount - 1;
                }
                else if (info2.ActiveRowViewport < 0)
                {
                    info2.ActiveRowViewport = 0;
                }
                viewportInfo.LeftColumns.CopyTo(info2.LeftColumns, 0);
                viewportInfo.ViewportWidth.CopyTo(info2.ViewportWidth, 0);
                for (int i = 0; i < rowViewportIndex; i++)
                {
                    info2.TopRows[i] = viewportInfo.TopRows[i];
                    info2.ViewportHeight[i] = viewportInfo.ViewportHeight[i];
                }
                for (int j = rowViewportIndex + 1; j < viewportInfo.RowViewportCount; j++)
                {
                    info2.TopRows[j - 1] = viewportInfo.TopRows[j];
                    info2.ViewportHeight[j - 1] = viewportInfo.ViewportHeight[j];
                }
                int index = (mergeInRowViewportIndex > rowViewportIndex) ? (mergeInRowViewportIndex - 1) : mergeInRowViewportIndex;
                if ((viewportInfo.ViewportHeight[rowViewportIndex] != 0.0) && ((viewportInfo.ViewportHeight[rowViewportIndex] != -1.0) || (viewportInfo.ViewportHeight[mergeInRowViewportIndex] != -1.0)))
                {
                    double viewportHeight = GetViewportHeight(mergeInRowViewportIndex);
                    double num5 = GetViewportHeight(rowViewportIndex);
                    info2.ViewportHeight[index] = viewportHeight + num5;
                }
                info2.ViewportHeight[info2.RowViewportCount - 1] = -1.0;
                SetViewportInfo(info2);
            }
        }

        static void MoveColumnAxis(Worksheet src, int srcColumn, Worksheet dest, int destColumn, int moveColumnCount, CopyToOption option)
        {
            int fromIndex = srcColumn;
            int toIndex = destColumn;
            int count = moveColumnCount;
            if (srcColumn < 0)
            {
                fromIndex = 0;
                count = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destColumn < 0)
            {
                toIndex = 0;
            }
            SparseArray<AxisInfo> columnsStorage = src.GetColumnsStorage(SheetArea.Cells);
            if (columnsStorage != null)
            {
                List<AxisInfo> infos = GetAxisInfo(columnsStorage, fromIndex, count);
                columnsStorage.Clear(fromIndex, count);
                SetAxisInfo(infos, dest.GetColumnsStorage(SheetArea.Cells), toIndex);
            }
            if ((option & CopyToOption.Style) > ((CopyToOption)0))
            {
                SparseArray<object> columnAxisStyles = src.GetColumnAxisStyles(SheetArea.Cells);
                if (columnAxisStyles != null)
                {
                    List<object> styles = GetAxisStyle(columnAxisStyles, fromIndex, count);
                    columnAxisStyles.Clear(fromIndex, count);
                    SetAxisStyle(styles, dest.GetColumnAxisStyles(SheetArea.Cells), toIndex);
                }
                SparseArray<object> srcStyle = src.GetColumnAxisStyles(SheetArea.ColumnHeader);
                if (srcStyle != null)
                {
                    List<object> list3 = GetAxisStyle(srcStyle, fromIndex, count);
                    srcStyle.Clear(fromIndex, count);
                    SetAxisStyle(list3, dest.GetColumnAxisStyles(SheetArea.ColumnHeader), toIndex);
                }
            }
            src.RaiseColumnChanged("Axis", srcColumn, moveColumnCount, SheetArea.Cells, SheetChangedEventAction.Updated);
            dest.RaiseColumnChanged("Axis", destColumn, moveColumnCount, SheetArea.Cells, SheetChangedEventAction.Updated);
        }

        static void MoveColumnRangeGroup(Worksheet src, int srcColumn, Worksheet dest, int destColumn, int moveColumnCount)
        {
            int from = srcColumn;
            int to = destColumn;
            int count = moveColumnCount;
            if (from < 0)
            {
                from = 0;
                count = src.ColumnCount;
            }
            if (destColumn < 0)
            {
                to = 0;
            }
            if (!object.ReferenceEquals(src, dest))
            {
                CrossSheetCopyRangeGroup(src.ColumnRangeGroup, from, dest.ColumnRangeGroup, to, count);
                if (src.ColumnRangeGroup != null)
                {
                    src.ColumnRangeGroup.Items.Clear(from, count);
                }
            }
            else
            {
                IMultipleSupport columnRangeGroup = src.ColumnRangeGroup;
                if (columnRangeGroup != null)
                {
                    columnRangeGroup.Move(from, to, count);
                }
            }
        }

        static void MoveFloatingObjects(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            FloatingObject[] sourceRangeFloatingObjects = null;
            CloneFloatingObjectToSheet(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount, out sourceRangeFloatingObjects);
            if ((sourceRangeFloatingObjects != null) && (sourceRangeFloatingObjects.Length > 0))
            {
                List<SpreadChart> list = new List<SpreadChart>();
                List<Picture> list2 = new List<Picture>();
                List<FloatingObject> list3 = new List<FloatingObject>();
                foreach (FloatingObject obj2 in sourceRangeFloatingObjects)
                {
                    if (obj2 is SpreadChart)
                    {
                        src.Charts.Remove(obj2 as SpreadChart);
                        list.Add(obj2 as SpreadChart);
                    }
                    else if (obj2 is Picture)
                    {
                        src.Pictures.Remove(obj2 as Picture);
                        list2.Add(obj2 as Picture);
                    }
                    else
                    {
                        src.FloatingObjects.Remove(obj2);
                        list3.Add(obj2);
                    }
                }
                if (object.ReferenceEquals(src, dest))
                {
                    if (list.Count > 0)
                    {
                        foreach (SpreadChart chart in list)
                        {
                            src.RaiseChartChanged(chart, ChartArea.All, "");
                        }
                    }
                    if (list2.Count > 0)
                    {
                        foreach (Picture picture in list2)
                        {
                            src.RaisePictureChanged(picture, "");
                        }
                    }
                    if (list3.Count > 0)
                    {
                        foreach (FloatingObject obj3 in list3)
                        {
                            src.RaiseFloatingObjectChanged(obj3, "");
                        }
                    }
                }
            }
        }

        static void MoveFormula(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            int fromRow = srcRow;
            int fromColumn = srcColumn;
            int toRow = destRow;
            int toColumn = destColumn;
            int rowCount = moveRowCount;
            int columnCount = moveColumnCount;
            if (srcRow < 0)
            {
                rowCount = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                columnCount = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (!object.ReferenceEquals(src, dest))
            {
                ICrossSheetRangeSupport calcAxial = dest.CalcAxial;
                if ((calcAxial != null) && src.HasFormula)
                {
                    calcAxial.Move(src, fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
            else
            {
                IRangeSupport support2 = src.CalcAxial;
                if ((support2 != null) && src.HasFormula)
                {
                    support2.Move(fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
        }

        static void MoveRowAxis(Worksheet src, int srcRow, Worksheet dest, int destRow, int moveRowCount, CopyToOption option)
        {
            int fromIndex = srcRow;
            int toIndex = destRow;
            int count = moveRowCount;
            if (srcRow < 0)
            {
                fromIndex = 0;
                count = Math.Min(src.RowCount, dest.RowCount);
            }
            if (destRow < 0)
            {
                toIndex = 0;
            }
            SparseArray<AxisInfo> rowsStorage = src.GetRowsStorage(SheetArea.Cells);
            if (rowsStorage != null)
            {
                List<AxisInfo> infos = GetAxisInfo(rowsStorage, fromIndex, count);
                rowsStorage.Clear(fromIndex, count);
                SetAxisInfo(infos, dest.GetRowsStorage(SheetArea.Cells), toIndex);
            }
            if ((option & CopyToOption.Style) > ((CopyToOption)0))
            {
                SparseArray<object> rowAxisStyles = src.GetRowAxisStyles(SheetArea.Cells);
                if (rowAxisStyles != null)
                {
                    List<object> styles = GetAxisStyle(rowAxisStyles, fromIndex, count);
                    rowAxisStyles.Clear(fromIndex, count);
                    SetAxisStyle(styles, dest.GetRowAxisStyles(SheetArea.Cells), toIndex);
                }
                SparseArray<object> srcStyle = src.GetRowAxisStyles(SheetArea.CornerHeader | SheetArea.RowHeader);
                if (srcStyle != null)
                {
                    List<object> list3 = GetAxisStyle(srcStyle, fromIndex, count);
                    srcStyle.Clear(fromIndex, count);
                    SetAxisStyle(list3, dest.GetRowAxisStyles(SheetArea.CornerHeader | SheetArea.RowHeader), toIndex);
                }
            }
            src.RaiseRowChanged("Axis", srcRow, moveRowCount, SheetArea.Cells, SheetChangedEventAction.Updated);
            dest.RaiseRowChanged("Axis", destRow, moveRowCount, SheetArea.Cells, SheetChangedEventAction.Updated);
        }

        static void MoveRowRangeGroup(Worksheet src, int srcRow, Worksheet dest, int destRow, int moveRowCount)
        {
            int from = srcRow;
            int to = destRow;
            int count = moveRowCount;
            if (srcRow < 0)
            {
                from = 0;
                count = src.RowCount;
            }
            if (destRow < 0)
            {
                to = 0;
            }
            if (!object.ReferenceEquals(src, dest))
            {
                CrossSheetCopyRangeGroup(src.RowRangeGroup, from, dest.RowRangeGroup, to, count);
                if (src.RowRangeGroup != null)
                {
                    src.RowRangeGroup.Items.Clear(from, count);
                }
            }
            else
            {
                IMultipleSupport rowRangeGroup = src.RowRangeGroup;
                if (rowRangeGroup != null)
                {
                    rowRangeGroup.Move(from, to, count);
                }
            }
        }

        static void MoveSheetInfo(Worksheet src, Worksheet dest, CopyToOption option)
        {
            if (!object.ReferenceEquals(src, dest))
            {
                if ((option & CopyToOption.Style) > ((CopyToOption)0))
                {
                    dest.DefaultStyle = src.DefaultStyle;
                    dest.ColumnHeader.DefaultStyle = src.ColumnHeader.DefaultStyle;
                    dest.RowHeader.DefaultStyle = src.RowHeader.DefaultStyle;
                    src.DefaultStyle = null;
                    src.ColumnHeader.DefaultStyle = null;
                    src.RowHeader.DefaultStyle = null;
                }
                dest.DefaultColumnWidth = src.DefaultColumnWidth;
                dest.DefaultRowHeight = src.DefaultRowHeight;
                dest.ColumnHeader.DefaultRowHeight = src.ColumnHeader.DefaultRowHeight;
                dest.RowHeader.DefaultColumnWidth = src.RowHeader.DefaultColumnWidth;
                src.DefaultColumnWidth = _DEFAULT_COLUMN_WIDTH;
                src.DefaultRowHeight = _DEFAULT_ROW_HEIGHT;
                src.ColumnHeader.DefaultRowHeight = _DEFAULT_COLUMN_HEADER_ROWHEIGHT;
                src.RowHeader.DefaultColumnWidth = _DEFAULT_ROW_HEADER_COLUMNWIDTH;
            }
        }

        static void MoveSpan(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            int fromRow = srcRow;
            int fromColumn = srcColumn;
            int toRow = destRow;
            int toColumn = destColumn;
            int rowCount = moveRowCount;
            int columnCount = moveColumnCount;
            if (srcRow < 0)
            {
                fromRow = 0;
                rowCount = src.RowCount;
            }
            if (fromColumn < 0)
            {
                fromColumn = 0;
                columnCount = src.ColumnCount;
            }
            if (destRow < 0)
            {
                toRow = 0;
            }
            if (destColumn < 0)
            {
                toColumn = 0;
            }
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                if (flag)
                {
                    CrossSheetCopySpans(src.ColumnHeaderSpanModel, -1, fromColumn, dest.ColumnHeaderSpanModel, -1, toColumn, -1, columnCount);
                    if (src.ColumnHeaderSpanModel != null)
                    {
                        src.ColumnHeaderSpanModel.Clear(-1, fromColumn, -1, columnCount);
                    }
                }
                else
                {
                    IRangeSupport spanModel = src.GetSpanModel(SheetArea.ColumnHeader);
                    if (spanModel != null)
                    {
                        spanModel.Move(-1, fromColumn, -1, toColumn, -1, columnCount);
                    }
                }
            }
            if (srcColumn < 0)
            {
                if (flag)
                {
                    CrossSheetCopySpans(src.RowHeaderSpanModel, fromRow, -1, dest.RowHeaderSpanModel, toRow, -1, rowCount, -1);
                    if (src.RowHeaderSpanModel != null)
                    {
                        src.RowHeaderSpanModel.Clear(fromRow, -1, rowCount, -1);
                    }
                }
                else
                {
                    IRangeSupport support2 = src.GetSpanModel(SheetArea.CornerHeader | SheetArea.RowHeader);
                    if (support2 != null)
                    {
                        support2.Move(fromRow, -1, toRow, -1, rowCount, -1);
                    }
                }
            }
            if (flag)
            {
                CrossSheetCopySpans(src.SpanModel, fromRow, fromColumn, dest.SpanModel, toRow, toColumn, rowCount, columnCount);
                if (src.SpanModel != null)
                {
                    src.SpanModel.Clear(fromRow, fromColumn, rowCount, columnCount);
                }
            }
            else
            {
                IRangeSupport support3 = src.GetSpanModel(SheetArea.Cells);
                if (support3 != null)
                {
                    support3.Move(fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
        }

        static void MoveSparkline(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            int fromRow = srcRow;
            int fromColumn = srcColumn;
            int toRow = destRow;
            int toColumn = destColumn;
            int rowCount = moveRowCount;
            int columnCount = moveColumnCount;
            if (srcRow < 0)
            {
                fromRow = 0;
                rowCount = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                fromColumn = 0;
                columnCount = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                toRow = 0;
            }
            if (destColumn < 0)
            {
                toColumn = 0;
            }
            if (!object.ReferenceEquals(src, dest))
            {
                ICrossSheetRangeSupport sheetSparklineGroupManager = dest.SheetSparklineGroupManager;
                if (sheetSparklineGroupManager != null)
                {
                    sheetSparklineGroupManager.Move(src, fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
            else
            {
                IRangeSupport support2 = src.SheetSparklineGroupManager;
                if (support2 != null)
                {
                    support2.Move(fromRow, fromColumn, toRow, toColumn, rowCount, columnCount);
                }
            }
        }

        static void MoveStyle(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                int num = srcColumn;
                int num2 = destColumn;
                int rowCount = src.ColumnHeader.RowCount;
                int columnCount = moveColumnCount;
                if (srcColumn < 0)
                {
                    num = 0;
                    columnCount = src.ColumnCount;
                }
                if (destColumn < 0)
                {
                    num2 = 0;
                }
                StorageBlock block = src.GetStorage(SheetArea.ColumnHeader);
                if (flag)
                {
                    for (int i = 0; i < rowCount; i++)
                    {
                        for (int j = 0; j < columnCount; j++)
                        {
                            StyleInfo style = src.GetCompositeStyle(i, num + j, SheetArea.ColumnHeader, block);
                            if (style == null)
                            {
                                dest.SetStyleInfo(i, num2 + j, SheetArea.ColumnHeader, null);
                            }
                            else
                            {
                                dest.SetStyleInfo(i, num2 + j, SheetArea.ColumnHeader, style);
                                src.SetStyleInfo(i, num + j, SheetArea.ColumnHeader, null);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<StyleInfo> matrix = new DataMatrix<StyleInfo>(rowCount, columnCount);
                    for (int k = 0; k < rowCount; k++)
                    {
                        for (int n = 0; n < columnCount; n++)
                        {
                            StyleInfo info2 = src.GetCompositeStyle(k, num + n, SheetArea.ColumnHeader, block);
                            if (info2 != null)
                            {
                                matrix.SetValue(k, n, info2);
                            }
                            src.SetStyleInfo(k, num + n, SheetArea.ColumnHeader, null);
                        }
                    }
                    for (int m = 0; m < rowCount; m++)
                    {
                        for (int num10 = 0; num10 < columnCount; num10++)
                        {
                            dest.SetStyleInfo(m, num2 + num10, SheetArea.ColumnHeader, matrix.GetValue(m, num10));
                        }
                    }
                }
            }
            if (srcColumn < 0)
            {
                int num11 = srcRow;
                int num12 = destRow;
                int num13 = moveRowCount;
                int num14 = src.RowHeader.ColumnCount;
                if (srcRow < 0)
                {
                    num11 = 0;
                    num13 = src.RowCount;
                }
                if (destRow < 0)
                {
                    num12 = 0;
                }
                StorageBlock block2 = src.GetStorage(SheetArea.CornerHeader | SheetArea.RowHeader);
                if (flag)
                {
                    for (int num15 = 0; num15 < num13; num15++)
                    {
                        for (int num16 = 0; num16 < num14; num16++)
                        {
                            StyleInfo info3 = src.GetCompositeStyle(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, block2);
                            if (info3 == null)
                            {
                                dest.SetStyleInfo(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                            }
                            else
                            {
                                dest.SetStyleInfo(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, info3);
                                src.SetStyleInfo(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<StyleInfo> matrix2 = new DataMatrix<StyleInfo>(num13, num14);
                    for (int num17 = 0; num17 < num13; num17++)
                    {
                        for (int num18 = 0; num18 < num14; num18++)
                        {
                            StyleInfo info4 = src.GetCompositeStyle(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader, block2);
                            if (info4 != null)
                            {
                                matrix2.SetValue(num17, num18, info4);
                            }
                            src.SetStyleInfo(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                        }
                    }
                    for (int num19 = 0; num19 < num13; num19++)
                    {
                        for (int num20 = 0; num20 < num14; num20++)
                        {
                            dest.SetStyleInfo(num12 + num19, num20, SheetArea.CornerHeader | SheetArea.RowHeader, matrix2.GetValue(num19, num20));
                        }
                    }
                }
            }
            int num21 = srcRow;
            int num22 = srcColumn;
            int num23 = destRow;
            int num24 = destColumn;
            int row = moveRowCount;
            int column = moveColumnCount;
            if (srcRow < 0)
            {
                num21 = 0;
                row = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                num22 = 0;
                column = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                num23 = 0;
            }
            if (destColumn < 0)
            {
                num24 = 0;
            }
            StorageBlock storage = src.GetStorage(SheetArea.Cells);
            if (flag)
            {
                for (int num27 = 0; num27 < row; num27++)
                {
                    for (int num28 = 0; num28 < column; num28++)
                    {
                        StyleInfo info5 = src.GetCompositeStyle(num21 + num27, num22 + num28, SheetArea.Cells, storage);
                        if (info5 == null)
                        {
                            dest.SetStyleInfo(num23 + num27, num24 + num28, SheetArea.Cells, null);
                        }
                        else
                        {
                            dest.SetStyleInfo(num23 + num27, num24 + num28, SheetArea.Cells, info5);
                            src.SetStyleInfo(num21 + num27, num22 + num28, SheetArea.Cells, null);
                        }
                    }
                }
            }
            else
            {
                DataMatrix<StyleInfo> matrix3 = new DataMatrix<StyleInfo>(row, column);
                for (int num29 = 0; num29 < row; num29++)
                {
                    for (int num30 = 0; num30 < column; num30++)
                    {
                        StyleInfo info6 = src.GetCompositeStyle(num21 + num29, num22 + num30, SheetArea.Cells, storage);
                        if (info6 != null)
                        {
                            matrix3.SetValue(num29, num30, info6);
                        }
                        src.SetStyleInfo(num21 + num29, num22 + num30, SheetArea.Cells, null);
                    }
                }
                for (int num31 = 0; num31 < row; num31++)
                {
                    for (int num32 = 0; num32 < column; num32++)
                    {
                        dest.SetStyleInfo(num23 + num31, num24 + num32, SheetArea.Cells, matrix3.GetValue(num31, num32));
                    }
                }
            }
        }

        /// <summary>
        /// Changes the table location.
        /// </summary>
        /// <param name="table">The table instance to move.</param>
        /// <param name="row">The new row index.</param>
        /// <param name="column">The new column index.</param>
        /// <remarks>
        /// The row and column index should be the top, left table cell.
        /// </remarks>
        public void MoveTable(SheetTable table, int row, int column)
        {
            if (table == null)
            {
                throw new ArgumentNullException("table");
            }
            table.IsCopying = true;
            try
            {
                CellRange range = table.Range;
                table.MoveTo(row, column);
                CellRange range2 = table.Range;
                RaiseCellChanged("[Table]", range.Row, range.Column, range.RowCount, range.ColumnCount, SheetArea.Cells);
                RaiseCellChanged("[Table]", range2.Row, range2.Column, range2.RowCount, range2.ColumnCount, SheetArea.Cells);
            }
            finally
            {
                table.IsCopying = false;
            }
        }

        /// <summary>
        /// Changes the table location.
        /// </summary>
        /// <param name="tableName">The table name to move.</param>
        /// <param name="row">The new row index.</param>
        /// <param name="column">The new column index.</param>
        /// <remarks>
        /// The row and column index should be the top, left table cell.
        /// </remarks>
        public void MoveTable(string tableName, int row, int column)
        {
            SheetTable table = FindTable(tableName);
            if (table == null)
            {
                throw new NotSupportedException(ResourceStrings.TableNotFoundError);
            }
            MoveTable(table, row, column);
        }

        static void MoveTag(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                int num = srcColumn;
                int num2 = destColumn;
                int rowCount = src.ColumnHeader.RowCount;
                int columnCount = moveColumnCount;
                if (srcColumn < 0)
                {
                    num = 0;
                    columnCount = src.ColumnCount;
                }
                if (destColumn < 0)
                {
                    num2 = 0;
                }
                if (flag)
                {
                    for (int i = 0; i < rowCount; i++)
                    {
                        for (int j = 0; j < columnCount; j++)
                        {
                            object obj2 = src.GetTag(i, num + j, SheetArea.ColumnHeader);
                            if (obj2 == null)
                            {
                                dest.SetTag(i, num2 + j, SheetArea.ColumnHeader, null);
                            }
                            else
                            {
                                dest.SetTag(i, num2 + j, SheetArea.ColumnHeader, obj2);
                                src.SetTag(i, num + j, SheetArea.ColumnHeader, null);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix = new DataMatrix<object>(rowCount, columnCount);
                    for (int k = 0; k < rowCount; k++)
                    {
                        for (int n = 0; n < columnCount; n++)
                        {
                            object obj3 = src.GetTag(k, num + n, SheetArea.ColumnHeader);
                            if (obj3 != null)
                            {
                                matrix.SetValue(k, n, obj3);
                            }
                            src.SetTag(k, num + n, SheetArea.ColumnHeader, null);
                        }
                    }
                    for (int m = 0; m < rowCount; m++)
                    {
                        for (int num10 = 0; num10 < columnCount; num10++)
                        {
                            dest.SetTag(m, num2 + num10, SheetArea.ColumnHeader, matrix.GetValue(m, num10));
                        }
                    }
                }
            }
            if (srcColumn < 0)
            {
                int num11 = srcRow;
                int num12 = destRow;
                int num13 = moveRowCount;
                int num14 = src.RowHeader.ColumnCount;
                if (srcRow < 0)
                {
                    num11 = 0;
                    num13 = src.RowCount;
                }
                if (destRow < 0)
                {
                    num12 = 0;
                }
                if (flag)
                {
                    for (int num15 = 0; num15 < num13; num15++)
                    {
                        for (int num16 = 0; num16 < num14; num16++)
                        {
                            object obj4 = src.GetTag(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj4 == null)
                            {
                                dest.SetTag(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                            }
                            else
                            {
                                dest.SetTag(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, obj4);
                                src.SetTag(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix2 = new DataMatrix<object>(num13, num14);
                    for (int num17 = 0; num17 < num13; num17++)
                    {
                        for (int num18 = 0; num18 < num14; num18++)
                        {
                            object obj5 = src.GetTag(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj5 != null)
                            {
                                matrix2.SetValue(num17, num18, obj5);
                            }
                            src.SetTag(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader, null);
                        }
                    }
                    for (int num19 = 0; num19 < num13; num19++)
                    {
                        for (int num20 = 0; num20 < num14; num20++)
                        {
                            dest.SetTag(num12 + num19, num20, SheetArea.CornerHeader | SheetArea.RowHeader, matrix2.GetValue(num19, num20));
                        }
                    }
                }
            }
            int num21 = srcRow;
            int num22 = srcColumn;
            int num23 = destRow;
            int num24 = destColumn;
            int row = moveRowCount;
            int column = moveColumnCount;
            if (srcRow < 0)
            {
                num21 = 0;
                row = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                num22 = 0;
                column = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                num23 = 0;
            }
            if (destColumn < 0)
            {
                num24 = 0;
            }
            if (flag)
            {
                for (int num27 = 0; num27 < row; num27++)
                {
                    for (int num28 = 0; num28 < column; num28++)
                    {
                        object obj6 = src.GetTag(num21 + num27, num22 + num28, SheetArea.Cells);
                        if (obj6 == null)
                        {
                            dest.SetTag(num23 + num27, num24 + num28, SheetArea.Cells, null);
                        }
                        else
                        {
                            dest.SetTag(num23 + num27, num24 + num28, SheetArea.Cells, obj6);
                            src.SetTag(num21 + num27, num22 + num28, SheetArea.Cells, null);
                        }
                    }
                }
            }
            else
            {
                DataMatrix<object> matrix3 = new DataMatrix<object>(row, column);
                for (int num29 = 0; num29 < row; num29++)
                {
                    for (int num30 = 0; num30 < column; num30++)
                    {
                        object obj7 = src.GetTag(num21 + num29, num22 + num30, SheetArea.Cells);
                        if (obj7 != null)
                        {
                            matrix3.SetValue(num29, num30, obj7);
                        }
                        src.SetTag(num21 + num29, num22 + num30, SheetArea.Cells, null);
                    }
                }
                for (int num31 = 0; num31 < row; num31++)
                {
                    for (int num32 = 0; num32 < column; num32++)
                    {
                        dest.SetTag(num23 + num31, num24 + num32, SheetArea.Cells, matrix3.GetValue(num31, num32));
                    }
                }
            }
        }

        /// <summary>
        /// Moves data from one range to another.
        /// </summary>
        /// <param name="fromRow">The source row.</param>
        /// <param name="fromColumn">The source column.</param>
        /// <param name="toRow">The target row.</param>
        /// <param name="toColumn">The target column.</param>
        /// <param name="rowCount">The row count.</param>
        /// <param name="columnCount">The column count.</param>
        /// <param name="option">The move option.</param>
        public void MoveTo(int fromRow, int fromColumn, int toRow, int toColumn, int rowCount, int columnCount, CopyToOption option)
        {
            MoveTo(this, fromRow, fromColumn, this, toRow, toColumn, rowCount, columnCount, option);
        }

        internal static void MoveTo(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount, CopyToOption option)
        {
            CheckArguments(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
            src.SuspendCalcService();
            dest.SuspendCalcService();
            try
            {
                if (src._conditionalFormats != null)
                {
                    src._conditionalFormats.ClearCache();
                }
                if ((option & CopyToOption.Value) > ((CopyToOption)0))
                {
                    MoveValue(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
                    if ((option & CopyToOption.Formula) == ((CopyToOption)0))
                    {
                        ClearFormula(dest, destRow, destColumn, moveRowCount, moveColumnCount);
                    }
                }
                if ((option & CopyToOption.Formula) > ((CopyToOption)0))
                {
                    MoveFormula(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
                }
            }
            finally
            {
                src.ResumeCalcService();
                dest.ResumeCalcService();
            }
            if ((option & CopyToOption.Style) > ((CopyToOption)0))
            {
                MoveStyle(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
            }
            if ((option & CopyToOption.Tag) > ((CopyToOption)0))
            {
                MoveTag(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
            }
            if ((option & CopyToOption.Sparkline) > ((CopyToOption)0))
            {
                MoveSparkline(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
            }
            if ((option & CopyToOption.RangeGroup) > ((CopyToOption)0))
            {
                if (srcRow < 0)
                {
                    MoveColumnRangeGroup(src, srcColumn, dest, destColumn, moveColumnCount);
                }
                if (srcColumn < 0)
                {
                    MoveRowRangeGroup(src, srcRow, dest, destRow, moveRowCount);
                }
            }
            if ((option & CopyToOption.Span) > ((CopyToOption)0))
            {
                if (src.IsEventSuspend())
                {
                    src.SuspendSpanModelEvent();
                }
                if (dest.IsEventSuspend())
                {
                    dest.SuspendSpanModelEvent();
                }
                try
                {
                    MoveSpan(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
                }
                finally
                {
                    if (src.IsEventSuspend())
                    {
                        src.ResumeSpanModelEvent();
                    }
                    if (dest.IsEventSuspend())
                    {
                        dest.ResumeSpanModelEvent();
                    }
                }
            }
            if ((option & CopyToOption.FloatingObject) > ((CopyToOption)0))
            {
                src.Workbook.SuspendEvent();
                dest.Workbook.SuspendEvent();
                try
                {
                    MoveFloatingObjects(src, srcRow, srcColumn, dest, destRow, destColumn, moveRowCount, moveColumnCount);
                }
                finally
                {
                    src.Workbook.ResumeEvent();
                    dest.Workbook.ResumeEvent();
                }
            }
            if (srcRow < 0)
            {
                MoveColumnAxis(src, srcColumn, dest, destColumn, moveColumnCount, option);
            }
            if (srcColumn < 0)
            {
                MoveRowAxis(src, srcRow, dest, destRow, moveRowCount, option);
            }
            if (((srcRow < 0) && (destRow < 0)) && ((srcColumn < 0) && (destColumn < 0)))
            {
                MoveSheetInfo(src, dest, option);
            }
            src.RaisePropertyChanged("[MoveTo]");
            if (src != dest)
            {
                dest.RaisePropertyChanged("[MoveTo]");
            }
        }

        static void MoveValue(Worksheet src, int srcRow, int srcColumn, Worksheet dest, int destRow, int destColumn, int moveRowCount, int moveColumnCount)
        {
            bool flag = !object.ReferenceEquals(src, dest);
            if (srcRow < 0)
            {
                int num = srcColumn;
                int num2 = destColumn;
                int rowCount = src.ColumnHeader.RowCount;
                int columnCount = moveColumnCount;
                if (srcColumn < 0)
                {
                    num = 0;
                    columnCount = src.ColumnCount;
                }
                if (destColumn < 0)
                {
                    num2 = 0;
                }
                if (flag)
                {
                    for (int i = 0; i < rowCount; i++)
                    {
                        for (int j = 0; j < columnCount; j++)
                        {
                            object dataColumnName;
                            if (src.IsColumnBound(num + j))
                            {
                                dataColumnName = src.GetDataColumnName(num + j);
                            }
                            else
                            {
                                dataColumnName = src.GetValue(i, num + j, SheetArea.ColumnHeader);
                            }
                            if (dataColumnName == null)
                            {
                                dest.SetValueInternal(i, num2 + j, SheetArea.ColumnHeader, null, true, false);
                            }
                            else
                            {
                                dest.SetValueInternal(i, num2 + j, SheetArea.ColumnHeader, dataColumnName, true, false);
                                src.SetValueInternal(i, num + j, SheetArea.ColumnHeader, null, true, false);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix = new DataMatrix<object>(rowCount, columnCount);
                    for (int k = 0; k < rowCount; k++)
                    {
                        for (int n = 0; n < columnCount; n++)
                        {
                            object obj3 = null;
                            if (src.IsColumnBound(num + n))
                            {
                                obj3 = src.GetDataColumnName(num + n);
                            }
                            else
                            {
                                obj3 = src.GetValue(k, num + n, SheetArea.ColumnHeader);
                            }
                            if (obj3 != null)
                            {
                                matrix.SetValue(k, n, obj3);
                            }
                            src.SetValueInternal(k, num + n, SheetArea.ColumnHeader, null, true, false);
                        }
                    }
                    for (int m = 0; m < rowCount; m++)
                    {
                        for (int num10 = 0; num10 < columnCount; num10++)
                        {
                            dest.SetValueInternal(m, num2 + num10, SheetArea.ColumnHeader, matrix.GetValue(m, num10), true, false);
                        }
                    }
                }
            }
            if (srcColumn < 0)
            {
                int num11 = srcRow;
                int num12 = destRow;
                int num13 = moveRowCount;
                int num14 = src.RowHeader.ColumnCount;
                if (srcRow < 0)
                {
                    num11 = 0;
                    num13 = src.RowCount;
                }
                if (destRow < 0)
                {
                    num12 = 0;
                }
                if (flag)
                {
                    for (int num15 = 0; num15 < num13; num15++)
                    {
                        for (int num16 = 0; num16 < num14; num16++)
                        {
                            object obj4 = src.GetValue(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj4 == null)
                            {
                                dest.SetValueInternal(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null, true, false);
                            }
                            else
                            {
                                dest.SetValueInternal(num12 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, obj4, true, false);
                                src.SetValueInternal(num11 + num15, num16, SheetArea.CornerHeader | SheetArea.RowHeader, null, true, false);
                            }
                        }
                    }
                }
                else
                {
                    DataMatrix<object> matrix2 = new DataMatrix<object>(num13, num14);
                    for (int num17 = 0; num17 < num13; num17++)
                    {
                        for (int num18 = 0; num18 < num14; num18++)
                        {
                            object obj5 = src.GetValue(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader);
                            if (obj5 != null)
                            {
                                matrix2.SetValue(num17, num18, obj5);
                            }
                            src.SetValueInternal(num11 + num17, num18, SheetArea.CornerHeader | SheetArea.RowHeader, null, true, false);
                        }
                    }
                    for (int num19 = 0; num19 < num13; num19++)
                    {
                        for (int num20 = 0; num20 < num14; num20++)
                        {
                            dest.SetValueInternal(num12 + num19, num20, SheetArea.CornerHeader | SheetArea.RowHeader, matrix2.GetValue(num19, num20), true, false);
                        }
                    }
                }
            }
            int num21 = srcRow;
            int num22 = srcColumn;
            int num23 = destRow;
            int num24 = destColumn;
            int row = moveRowCount;
            int column = moveColumnCount;
            if (srcRow < 0)
            {
                num21 = 0;
                row = Math.Min(src.RowCount, dest.RowCount);
            }
            if (srcColumn < 0)
            {
                num22 = 0;
                column = Math.Min(src.ColumnCount, dest.ColumnCount);
            }
            if (destRow < 0)
            {
                num23 = 0;
            }
            if (destColumn < 0)
            {
                num24 = 0;
            }
            if (flag)
            {
                for (int num27 = 0; num27 < row; num27++)
                {
                    for (int num28 = 0; num28 < column; num28++)
                    {
                        object obj6 = src.GetValue(num21 + num27, num22 + num28, SheetArea.Cells);
                        if (obj6 == null)
                        {
                            dest.SetValueInternal(num23 + num27, num24 + num28, SheetArea.Cells, null, true, false);
                        }
                        else
                        {
                            dest.SetValueInternal(num23 + num27, num24 + num28, SheetArea.Cells, obj6, true, false);
                            src.SetValueInternal(num21 + num27, num22 + num28, SheetArea.Cells, null, true, false);
                            if ((srcRow == -1) && (destRow == -1))
                            {
                                for (int num29 = num22; num29 < (num22 + column); num29++)
                                {
                                    if (src.IsColumnBound(num29))
                                    {
                                        src.BindDataColumn(num29, null);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else
            {
                DataMatrix<object> matrix3 = new DataMatrix<object>(row, column);
                for (int num30 = 0; num30 < row; num30++)
                {
                    for (int num31 = 0; num31 < column; num31++)
                    {
                        object obj7 = src.GetValue(num21 + num30, num22 + num31, SheetArea.Cells);
                        if (obj7 != null)
                        {
                            matrix3.SetValue(num30, num31, obj7);
                        }
                    }
                }
                IRangeSupport sheetBindingManager = src.SheetBindingManager;
                if (sheetBindingManager != null)
                {
                    sheetBindingManager.Move(srcRow, srcColumn, destRow, destColumn, moveRowCount, moveColumnCount);
                }
                for (int num32 = 0; num32 < row; num32++)
                {
                    for (int num33 = 0; num33 < column; num33++)
                    {
                        src.SetValueInternal(num21 + num32, num22 + num33, SheetArea.Cells, null, true, false);
                    }
                }
                if ((srcRow == -1) && (destRow == -1))
                {
                    for (int num34 = num22; num34 < (num22 + column); num34++)
                    {
                        if (src.IsColumnBound(num34))
                        {
                            src.BindDataColumn(num34, null);
                        }
                    }
                }
                for (int num35 = 0; num35 < row; num35++)
                {
                    for (int num36 = 0; num36 < column; num36++)
                    {
                        dest.SetValueInternal(num23 + num35, num24 + num36, SheetArea.Cells, matrix3.GetValue(num35, num36), true, false);
                    }
                }
            }
        }

        StyleInfo Name2Style(string name)
        {
            if (_namedStyles != null)
            {
                StyleInfo info = _namedStyles.Find(name);
                if (info != null)
                {
                    return info;
                }
                if ((_workbook != null) && (_workbook.NamedStyles != null))
                {
                    return _workbook.NamedStyles.Find(name);
                }
            }
            return null;
        }

        /// <summary>
        /// Gets the index of the next visible column from the current column.
        /// </summary>
        /// <param name="column">The index of the current column.</param>
        /// <returns>The index of the next visible column.</returns>
        public int NextActualVisibleColumn(int column)
        {
            for (int i = column + 1; i < ColumnCount; i++)
            {
                if (GetActualColumnVisible(i, SheetArea.Cells))
                {
                    return i;
                }
            }
            return -1;
        }

        /// <summary>
        /// Gets the index of the next visible row from the current row.
        /// </summary>
        /// <param name="row">The index of the current row.</param>
        /// <returns>The index of the next visible row.</returns>
        public int NextActualVisibleRow(int row)
        {
            for (int i = row + 1; i < RowCount; i++)
            {
                if (GetActualRowVisible(i, SheetArea.Cells))
                {
                    return i;
                }
            }
            return -1;
        }

        /// <summary>
        /// Returns the next nonempty column index for the specified column.
        /// </summary>
        /// <param name="column">The column from which the next nonempty column is to be returned.</param>
        /// <returns>The next nonempty column index.</returns>
        public int NextNonEmptyColumn(int column)
        {
            return NextNonEmptyColumn(column, SheetArea.Cells);
        }

        /// <summary>
        /// Returns the next nonempty column index for the specified column.
        /// </summary>
        /// <param name="column">The column from which the next nonempty column is to be returned.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <returns>
        /// The next nonempty column index.
        /// </returns>
        public int NextNonEmptyColumn(int column, SheetArea sheetArea)
        {
            int num = -1;
            if ((sheetArea != SheetArea.Cells) && (sheetArea != SheetArea.ColumnHeader))
            {
                if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    num = _rowHeaderColumns.NextNonEmptyIndex(column);
                }
                return num;
            }
            num = _viewportColumns.NextNonEmptyIndex(column);
            if ((_bindingManager == null) || !_bindingManager.IsBound)
            {
                return num;
            }
            List<int> boundColumns = _bindingManager.GetBoundColumns();
            if ((boundColumns == null) || (boundColumns.Count <= 0))
            {
                return num;
            }
            int num2 = -1;
            for (int i = 0; i < boundColumns.Count; i++)
            {
                if (boundColumns[i] > column)
                {
                    num2 = boundColumns[i];
                    break;
                }
            }
            if (num2 == -1)
            {
                return num;
            }
            if (num > -1)
            {
                return ((num2 < num) ? num2 : num);
            }
            return num2;
        }

        /// <summary>
        /// Returns the next nonempty row index for the specified column.
        /// </summary>
        /// <param name="row">The row whose next nonempty row is to be returned.</param>
        /// <returns>The next nonempty row index.</returns>
        public int NextNonEmptyRow(int row)
        {
            return NextNonEmptyRow(row, StorageType.Axis | StorageType.Sparkline | StorageType.Tag | StorageType.Style | StorageType.Data);
        }

        /// <summary>
        /// Gets the index of the next nonempty row of the specified type in the specified area.
        /// </summary>
        /// <param name="row">The row whose next nonempty row is to be returned.</param>
        /// <param name="type">Search the next nonempty row based on this type. </param>
        /// <returns>The next nonempty row index.</returns>
        public int NextNonEmptyRow(int row, StorageType type)
        {
            return NextNonEmptyRow(row, SheetArea.Cells, type);
        }

        /// <summary>
        /// Gets the index of the next nonempty row of the specified type in the specified area.
        /// </summary>
        /// <param name="row">The row from which to start searching for the next nonempty row.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="type">Search the next nonempty row based on this type.</param>
        /// <returns>The next nonempty row index.</returns>
        public int NextNonEmptyRow(int row, SheetArea sheetArea, StorageType type)
        {
            StorageBlock cellsBlock = null;
            SparseArray<object> viewportRowsStyles = null;
            if (sheetArea == SheetArea.Cells)
            {
                cellsBlock = _cellsBlock;
                viewportRowsStyles = _viewportRowsStyles;
            }
            else if (sheetArea == SheetArea.ColumnHeader)
            {
                cellsBlock = _columnHeadersBlock;
                viewportRowsStyles = _columnHeaderRowsStyles;
            }
            else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                cellsBlock = _rowHeadersBlock;
                viewportRowsStyles = _rowHeaderRowsStyles;
            }
            if ((cellsBlock == null) || (viewportRowsStyles == null))
            {
                return -1;
            }
            int num = cellsBlock.NextNonEmptyRow(row, type);
            int num2 = ((type & StorageType.Style) > ((StorageType)0)) ? viewportRowsStyles.NextNonEmptyIndex(row) : -1;
            if ((num < 0) || ((num2 >= 0) && (num2 < num)))
            {
                num = num2;
            }
            if (sheetArea == SheetArea.Cells)
            {
                if ((((type & StorageType.Data) == StorageType.Data) && (_bindingManager != null)) && _bindingManager.IsBound)
                {
                    int firstBoundRow = _bindingManager.GetFirstBoundRow();
                    int lastBoundRow = _bindingManager.GetLastBoundRow();
                    if (((num <= -1) || (num > firstBoundRow)) && (row < lastBoundRow))
                    {
                        return (row + 1);
                    }
                }
                if ((type & (StorageType.Style | StorageType.Data)) <= ((StorageType)0))
                {
                    return num;
                }
                for (int i = 0; (_tables != null) && (i < _tables.Count); i++)
                {
                    CellRange range = _tables[i].Range;
                    if ((row >= (range.Row - 1)) && (row < ((range.Row + range.RowCount) - 1)))
                    {
                        return (row + 1);
                    }
                }
            }
            return num;
        }

        void NotifyChartShapeOnSheetPropertyChanged(string sheetProperty)
        {
            IList<SpreadChartBase> chartShapeAffectedByRowFilter = null;
            if (sheetProperty == "RowFilter")
            {
                chartShapeAffectedByRowFilter = SpreadChartUtility.GetChartShapeAffectedByRowFilter(this);
            }
            else if (sheetProperty == "RowRangeGroup")
            {
                chartShapeAffectedByRowFilter = SpreadChartUtility.GetChartShapeAffectedByRowRangeGroup(this);
            }
            else if (sheetProperty == "ColumnRangeGroup")
            {
                chartShapeAffectedByRowFilter = SpreadChartUtility.GetChartShapeAffectedByColumnRangeGroup(this);
            }
            if ((chartShapeAffectedByRowFilter != null) && (chartShapeAffectedByRowFilter.Count > 0))
            {
                foreach (SpreadChartBase base2 in chartShapeAffectedByRowFilter)
                {
                    if (base2 is SpreadChart)
                    {
                        RaiseChartChanged(base2 as SpreadChart, ChartArea.All, sheetProperty);
                    }
                }
            }
        }

        void NotifyChartShapeOnSheetTableChanged(TableChangedArgs e)
        {
            if (e.PropertyName == "TableFilter")
            {
                IList<SpreadChartBase> chartShapeAffectedBySheetTable = SpreadChartUtility.GetChartShapeAffectedBySheetTable(this, e.Table);
                if ((chartShapeAffectedBySheetTable != null) && (chartShapeAffectedBySheetTable.Count > 0))
                {
                    foreach (SpreadChartBase base2 in chartShapeAffectedBySheetTable)
                    {
                        if (base2 is SpreadChart)
                        {
                            RaiseChartChanged(base2 as SpreadChart, ChartArea.All, "TableFilter");
                        }
                    }
                }
            }
        }

        void NotifyFloatingObjectOnColumnChanged(string columnProperty)
        {
            if ((columnProperty == "Width") || (columnProperty == "Axis"))
            {
                RaiseFloatingObjectChangedOnSheetPropertyChanged("SheetColumnChanged");
            }
        }

        void NotifyFloatingObjectOnRowChanged(string rowProperty)
        {
            if ((rowProperty == "Height") || (rowProperty == "Axis"))
            {
                RaiseFloatingObjectChangedOnSheetPropertyChanged("SheetRowChanged");
            }
        }

        void OnColumnHeaderDefaultStylePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            RaisePropertyChanged("ColumnHeaderDefaultStyle");
        }

        void OnColumnRangeGroupChanged(object sender, EventArgs e)
        {
            RaisePropertyChanged("ColumnRangeGroup");
        }


        void OnColumnsAdded(int column, int count, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.ColumnHeader) || (sheetArea == SheetArea.Cells))
            {
                IRangeSupport spanModel = GetSpanModel(SheetArea.ColumnHeader);
                if (spanModel != null)
                {
                    spanModel.AddColumns(column, count);
                }
                spanModel = GetSpanModel(SheetArea.Cells);
                if (spanModel != null)
                {
                    spanModel.AddColumns(column, count);
                }
            }
            else
            {
                IRangeSupport support2 = GetSpanModel(sheetArea);
                if (support2 != null)
                {
                    support2.AddColumns(column, count);
                }
            }
            if ((sheetArea == SheetArea.ColumnHeader) || (sheetArea == SheetArea.Cells))
            {
                if (_rowFilter != null)
                {
                    ((IRangeSupport)_rowFilter).AddColumns(column, count);
                }
                if (_conditionalFormats != null)
                {
                    ((IRangeSupport)_conditionalFormats).AddColumns(column, count);
                }
                IMultipleSupport columnRangeGroup = ColumnRangeGroup;
                if (columnRangeGroup != null)
                {
                    columnRangeGroup.Add(column, count);
                }
                if (_bindingManager != null)
                {
                    ((IRangeSupport)_bindingManager).AddColumns(column, count);
                }
                if (_sparklineGroupManager != null)
                {
                    _sparklineGroupManager.AddColumns(column, count);
                }
                if (_tables != null)
                {
                    ((IRangeSupport)_tables).AddColumns(column, count);
                }
                if (_spreadCharts != null)
                {
                    ((IRangeSupport)_spreadCharts).AddColumns(column, count);
                }
                if (_floatingObjects != null)
                {
                    ((IRangeSupport)_floatingObjects).AddColumns(column, count);
                }
                if (_pictures != null)
                {
                    ((IRangeSupport)_pictures).AddColumns(column, count);
                }
            }
            if ((sheetArea == SheetArea.Cells) && HasFormula)
            {
                CalcAxial.AddColumns(column, count);
                Recalculate(false);
            }
        }

        void OnColumnsRemoved(int column, int count, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.ColumnHeader) || (sheetArea == SheetArea.Cells))
            {
                IRangeSupport spanModel = GetSpanModel(SheetArea.ColumnHeader);
                if (spanModel != null)
                {
                    spanModel.RemoveColumns(column, count);
                }
                spanModel = GetSpanModel(SheetArea.Cells);
                if (spanModel != null)
                {
                    spanModel.RemoveColumns(column, count);
                }
            }
            else
            {
                IRangeSupport support2 = GetSpanModel(sheetArea);
                if (support2 != null)
                {
                    support2.RemoveColumns(column, count);
                }
            }
            if ((sheetArea == SheetArea.ColumnHeader) || (sheetArea == SheetArea.Cells))
            {
                if (_rowFilter != null)
                {
                    ((IRangeSupport)_rowFilter).RemoveColumns(column, count);
                }
                if (_conditionalFormats != null)
                {
                    ((IRangeSupport)_conditionalFormats).RemoveColumns(column, count);
                }
                IMultipleSupport columnRangeGroup = ColumnRangeGroup;
                if (columnRangeGroup != null)
                {
                    columnRangeGroup.Remove(column, count);
                }
                if (_bindingManager != null)
                {
                    ((IRangeSupport)_bindingManager).RemoveColumns(column, count);
                }
                if (_sparklineGroupManager != null)
                {
                    _sparklineGroupManager.RemoveColumns(column, count);
                }
                if (_tables != null)
                {
                    ((IRangeSupport)_tables).RemoveColumns(column, count);
                }
                if (_spreadCharts != null)
                {
                    ((IRangeSupport)(_spreadCharts)).RemoveColumns(column, count);
                }
                if (_floatingObjects != null)
                {
                    ((IRangeSupport)(_floatingObjects)).RemoveColumns(column, count);
                }
                if (_pictures != null)
                {
                    ((IRangeSupport)(_pictures)).RemoveColumns(column, count);
                }
                ViewportInfo viewportInfo = GetViewportInfo();
                for (int i = 0; i < viewportInfo.LeftColumns.Length; i++)
                {
                    int num2 = viewportInfo.LeftColumns[i];
                    if ((num2 >= 0) && (num2 >= (ColumnCount - FrozenTrailingColumnCount)))
                    {
                        int num3 = -1;
                        for (int j = (ColumnCount - FrozenTrailingColumnCount) - 1; j >= FrozenColumnCount; j--)
                        {
                            if (GetActualColumnWidth(j, sheetArea) > 0.0)
                            {
                                num3 = j;
                                break;
                            }
                        }
                        if (num3 == -1)
                        {
                            viewportInfo.LeftColumns[i] = 0;
                        }
                        else
                        {
                            viewportInfo.LeftColumns[i] = num3;
                        }
                    }
                }
                SetViewportInfo(viewportInfo);
            }
            if ((sheetArea == SheetArea.Cells) && HasFormula)
            {
                CalcAxial.RemoveColumns(column, count);
                Recalculate(false);
            }
        }

        void OnConditionalFormatsRulesChanged(object sender, RulesChangedEventArgs e)
        {
            switch (e.Action)
            {
                case RulesChangedAction.Add:
                    e.Rule.PropertyChanged += new PropertyChangedEventHandler(OnRulePropertyChanged);
                    break;

                case RulesChangedAction.Remove:
                    e.Rule.PropertyChanged -= new PropertyChangedEventHandler(OnRulePropertyChanged);
                    break;
            }
            RaiseCellChanged("[CondtionalFormat]", -1, -1, -1, -1, SheetArea.Cells);
        }

        internal void OnCustomFunctionChanged(string name)
        {
            if ((_calcManager != null) && (_calcManager.FormulaCount > 0))
            {
                CalcFunctionIdentity id = new CalcFunctionIdentity(name);
                _calcManager.Graph.RefreshNode(id, false);
            }
        }

        internal void OnCustomNameChanged(string name, bool refreshNode)
        {
            if ((_calcManager != null) && (_calcManager.FormulaCount > 0))
            {
                CalcNameIdentity id = new CalcNameIdentity(name);
                if (refreshNode)
                {
                    _calcManager.Graph.RefreshNode(id, false);
                }
            }
        }

        void OnDefaultStyleInfoPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            RaisePropertyChanged("DefaultStyle");
        }

        void OnFilterStatusChanged(object sender, FilterEventArgs e)
        {
            RaisePropertyChanged("RowFilter");
        }

        void OnFrozenColumnCountChanged()
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            for (int i = 0; i < viewportInfo.LeftColumns.Length; i++)
            {
                viewportInfo.LeftColumns[i] = Math.Max(viewportInfo.LeftColumns[i], FrozenColumnCount);
            }
        }

        void OnFrozenRowCountChanged()
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            for (int i = 0; i < viewportInfo.TopRows.Length; i++)
            {
                viewportInfo.TopRows[i] = Math.Max(viewportInfo.TopRows[i], FrozenRowCount);
            }
        }

        void OnNamedStyleChanged(object sender, StyleInfoCollectionChangedEventArgs e)
        {
            RaisePropertyChanged("NamedStyles");
        }

        void OnNameInfoCollectionChanged(object sender, NameInfoCollectionChangedEventArgs e)
        {
            Recalculate(false);
            RaisePropertyChanged("Names");
        }

        void OnRowHeaderDefaultStylePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            RaisePropertyChanged("RowHeaderDefaultStyle");
        }

        void OnRowRangeGroupChanged(object sender, EventArgs e)
        {
            RaisePropertyChanged("RowRangeGroup");
        }

        void OnRowsAdded(int row, int count, SheetArea sheetArea)
        {
            if ((sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)) || (sheetArea == SheetArea.Cells))
            {
                IRangeSupport spanModel = GetSpanModel(SheetArea.CornerHeader | SheetArea.RowHeader);
                if (spanModel != null)
                {
                    spanModel.AddRows(row, count);
                }
                spanModel = GetSpanModel(SheetArea.Cells);
                if (spanModel != null)
                {
                    spanModel.AddRows(row, count);
                }
            }
            else
            {
                IRangeSupport support2 = GetSpanModel(sheetArea);
                if (support2 != null)
                {
                    support2.AddRows(row, count);
                }
            }
            if ((sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)) || (sheetArea == SheetArea.Cells))
            {
                if (_rowFilter != null)
                {
                    ((IRangeSupport)_rowFilter).AddRows(row, count);
                }
                if (_conditionalFormats != null)
                {
                    ((IRangeSupport)_conditionalFormats).AddRows(row, count);
                }
                IMultipleSupport rowRangeGroup = RowRangeGroup;
                if (rowRangeGroup != null)
                {
                    rowRangeGroup.Add(row, count);
                }
                if (_bindingManager != null)
                {
                    ((IRangeSupport)_bindingManager).AddRows(row, count);
                }
                if (_sparklineGroupManager != null)
                {
                    _sparklineGroupManager.AddRows(row, count);
                }
                if (_tables != null)
                {
                    ((IRangeSupport)_tables).AddRows(row, count);
                }
                if (_spreadCharts != null)
                {
                    ((IRangeSupport)_spreadCharts).AddRows(row, count);
                }
                if (_floatingObjects != null)
                {
                    ((IRangeSupport)_floatingObjects).AddRows(row, count);
                }
                if (_pictures != null)
                {
                    ((IRangeSupport)_pictures).AddRows(row, count);
                }
            }
            if ((sheetArea == SheetArea.Cells) && HasFormula)
            {
                CalcAxial.AddRows(row, count);
                Recalculate(false);
            }
        }

        void OnRowsRemoved(int row, int count, SheetArea sheetArea)
        {
            if ((sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)) || (sheetArea == SheetArea.Cells))
            {
                IRangeSupport spanModel = GetSpanModel(SheetArea.CornerHeader | SheetArea.RowHeader);
                if (spanModel != null)
                {
                    spanModel.RemoveRows(row, count);
                }
                spanModel = GetSpanModel(SheetArea.Cells);
                if (spanModel != null)
                {
                    spanModel.RemoveRows(row, count);
                }
            }
            else
            {
                IRangeSupport support2 = GetSpanModel(sheetArea);
                if (support2 != null)
                {
                    support2.RemoveRows(row, count);
                }
            }
            if ((sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)) || (sheetArea == SheetArea.Cells))
            {
                if (_rowFilter != null)
                {
                    ((IRangeSupport)_rowFilter).RemoveRows(row, count);
                }
                if (_conditionalFormats != null)
                {
                    ((IRangeSupport)_conditionalFormats).RemoveRows(row, count);
                }
                IMultipleSupport rowRangeGroup = RowRangeGroup;
                if (rowRangeGroup != null)
                {
                    rowRangeGroup.Remove(row, count);
                }
                if (_bindingManager != null)
                {
                    ((IRangeSupport)_bindingManager).RemoveRows(row, count);
                }
                if (_sparklineGroupManager != null)
                {
                    _sparklineGroupManager.RemoveRows(row, count);
                }
                if (_tables != null)
                {
                    ((IRangeSupport)_tables).RemoveRows(row, count);
                }
                if (_spreadCharts != null)
                {
                    ((IRangeSupport)(_spreadCharts)).RemoveRows(row, count);
                }
                if (_floatingObjects != null)
                {
                    ((IRangeSupport)(_floatingObjects)).RemoveRows(row, count);
                }
                if (_pictures != null)
                {
                    ((IRangeSupport)(_pictures)).RemoveRows(row, count);
                }
                ViewportInfo viewportInfo = GetViewportInfo();
                for (int i = 0; i < viewportInfo.TopRows.Length; i++)
                {
                    int num2 = viewportInfo.TopRows[i];
                    if ((num2 >= 0) && (num2 >= (RowCount - FrozenTrailingRowCount)))
                    {
                        int num3 = -1;
                        for (int j = (RowCount - FrozenTrailingRowCount) - 1; j >= FrozenRowCount; j--)
                        {
                            if (GetActualRowHeight(j, sheetArea) > 0.0)
                            {
                                num3 = j;
                                break;
                            }
                        }
                        if (num3 == -1)
                        {
                            viewportInfo.TopRows[i] = 0;
                        }
                        else
                        {
                            viewportInfo.TopRows[i] = num3;
                        }
                    }
                }
                SetViewportInfo(viewportInfo);
            }
            if ((sheetArea == SheetArea.Cells) && HasFormula)
            {
                CalcAxial.RemoveRows(row, count);
                Recalculate(false);
            }
        }

        void OnRulePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (sender is FormattingRuleBase)
            {
                RaiseCellChanged("[CondtionalFormat]", -1, -1, -1, -1, SheetArea.Cells);
            }
        }

        void OnSheetTableChanged(object sender, TableChangedArgs e)
        {
            if (e.PropertyName == "TableFilter")
            {
                RaisePropertyChanged("TableFilter");
            }
            else
            {
                RaisePropertyChanged("SheetTable");
            }
            NotifyChartShapeOnSheetTableChanged(e);
        }

        void OnTablePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (sender is SheetTable)
            {
                CellRange range = (sender as SheetTable).Range;
                RaiseCellChanged("[Table]", range.Row, range.Column, range.RowCount, range.ColumnCount, SheetArea.Cells);
            }
        }

        /// <summary>
        /// Loads the CSV (comma-separated values) file asynchronously.
        /// </summary>
        /// <param name="stream">The stream from which to load the file.</param>
        /// <param name="flags">The import flags.</param>
        public void OpenCsv(Stream stream, TextFileOpenFlags flags)
        {
            OpenTextFileBackGround(stream, flags, "\r\n", ",", "\"", Encoding.UTF8);
        }

        /// <summary>
        /// Loads the CSV (comma-separated values) file asynchronously.
        /// </summary>
        /// <param name="stream">The stream from which to load to file.</param>
        /// <param name="flags">The import flags.</param>
        /// <param name="encoding">The encoding.</param>
        public IAsyncAction OpenCsvAsync(Stream stream, TextFileOpenFlags flags, Encoding encoding)
        {
            return OpenTextFileAsync(stream, flags, "\r\n", ",", "\"", encoding);
        }

        public void OpenTextFile(Stream stream, TextFileOpenFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter)
        {
            OpenTextFileBackGround(stream, flags, rowDelimiter, columnDelimiter, cellDelimiter, Encoding.UTF8);
        }

        /// <summary>
        /// Loads the CSV file with the specified separator asynchronously.
        /// </summary>
        /// <param name="stream">The stream from which to load.</param>
        /// <param name="flags">The import flags.</param>
        /// <param name="rowDelimiter">The row delimiter string.</param>
        /// <param name="columnDelimiter">The column delimiter string.</param>
        /// <param name="cellDelimiter">The cell delimiter string.</param>
        public IAsyncAction OpenTextFileAsync(Stream stream, TextFileOpenFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter)
        {
            return OpenTextFileAsync(stream, flags, rowDelimiter, columnDelimiter, cellDelimiter, Encoding.UTF8);
        }

        /// <summary>
        /// Loads delimited text from a specified stream into the sheet, with or without headers, with the specified encoding asynchronously. 
        /// </summary>
        /// <param name="stream">The stream from which to load data.</param>
        /// <param name="flags">The import flags.</param>
        /// <param name="rowDelimiter">The row delimiter string.</param>
        /// <param name="columnDelimiter">The column delimiter string.</param>
        /// <param name="cellDelimiter">The cell delimiter string.</param>
        /// <param name="encoding">The encoding.</param>
        public IAsyncAction OpenTextFileAsync(Stream stream, TextFileOpenFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter, Encoding encoding)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    OpenTextFileBackGround(stream, flags, rowDelimiter, columnDelimiter, cellDelimiter, encoding);
                });
            });
        }

        void OpenTextFileBackGround(Stream stream, TextFileOpenFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter, Encoding encoding)
        {
            if (encoding == null)
            {
                throw new ArgumentNullException("encoding");
            }
            if (stream == null)
            {
                throw new ArgumentNullException("stream");
            }
            if (!stream.CanRead)
            {
                throw new ArgumentNullException("stream");
            }
            if (_workbook != null)
            {
                _workbook.SuspendEvent();
                _workbook.SuspendCalcService();
            }
            try
            {
                int length = (int)stream.Length;
                int position = (int)stream.Position;
                byte[] buffer = new byte[length - position];
                while ((position < length) && (stream.Position < stream.Length))
                {
                    position += stream.Read(buffer, position, length - position);
                }
                StreamReader @this = new StreamReader((Stream)new MemoryStream(buffer), encoding);
                string data = @this.ReadToEnd();
                CsvImp.SetRangeText(this, 0, 0, data, rowDelimiter, columnDelimiter, cellDelimiter, flags);
                @this.Close();
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
            finally
            {
                if (_workbook != null)
                {
                    _workbook.ResumeEvent();
                    _workbook.ResumeCalcService();
                }
                RaisePropertyChanged("[ImportFile]");
            }
        }

        /// <summary>
        /// Loads the XML stream to build a new worksheet.
        /// </summary>
        /// <param name="xmlStream">The XML stream.</param>
        public void OpenXml(Stream xmlStream)
        {
            OpenXmlOnBackground(xmlStream);
        }

        /// <summary>
        /// Loads the XML stream to build a new worksheet asynchronously.
        /// </summary>
        /// <param name="xmlStream">The XML stream.</param>
        public IAsyncAction OpenXmlAsync(Stream xmlStream)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    OpenXmlOnBackground(xmlStream);
                });
            });
        }

        /// <summary>
        /// Reads the XML stream to build a new worksheet asynchronously. 
        /// </summary>
        /// <param name="reader">The XML reader.</param>
        public IAsyncAction OpenXmlAsync(XmlReader reader)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    OpenXmlOnBackground(reader);
                });
            });
        }

        internal void OpenXmlOnBackground(Stream xmlStream)
        {
            try
            {
                using (XmlReader reader = XmlReader.Create(xmlStream))
                {
                    OpenXmlOnBackground(reader);
                    RaisePropertyChanged("[OpenXml]");
                }
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
        }

        internal void OpenXmlOnBackground(XmlReader reader)
        {
            try
            {
                Serializer.InitReader(reader);
                ((IXmlSerializable)this).ReadXml(reader);
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
        }

        /// <summary>
        /// Parses the string to a two dimensional array.
        /// </summary>
        /// <param name="text">The text to parse.</param>
        /// <param name="rowDelimiter">The row delimiter.</param>
        /// <param name="columnDelimiter">The column delimiter.</param>
        /// <param name="cellDelimiter">The cell delimiter.</param>
        public static string[,] ParseCsv(string text, string rowDelimiter, string columnDelimiter, string cellDelimiter)
        {
            string[,] strArray = null;
            List<List<string>> data = CsvImp.ParseText(text, rowDelimiter, columnDelimiter, cellDelimiter);
            if (data == null)
            {
                return strArray;
            }
            int num = data.Count;
            int maxLength = CsvImp.GetMaxLength(data);
            string[,] strArray2 = new string[num, maxLength];
            for (int i = 0; i < num; i++)
            {
                for (int j = 0; j < maxLength; j++)
                {
                    if (j < data[i].Count)
                    {
                        strArray2[i, j] = data[i][j];
                    }
                    else
                    {
                        strArray2[i, j] = null;
                    }
                }
            }
            return strArray2;
        }

        /// <summary>
        /// Gets the index of the previous visible column from the current column.
        /// </summary>
        /// <param name="column">The index of the current column.</param>
        /// <returns>The index of the previous visible column.</returns>
        public int PreviousActualVisibleColumn(int column)
        {
            for (int i = column - 1; i >= 0; i--)
            {
                if (GetActualColumnVisible(i, SheetArea.Cells))
                {
                    return i;
                }
            }
            return -1;
        }

        /// <summary>
        /// Gets the index of the previous visible row from the current row.
        /// </summary>
        /// <param name="row">The index of the current row.</param>
        /// <returns>The index of the previous visible row.</returns>
        public int PreviousActualVisibleRow(int row)
        {
            for (int i = row - 1; i >= 0; i--)
            {
                if (GetActualRowVisible(i, SheetArea.Cells))
                {
                    return i;
                }
            }
            return -1;
        }

        internal void RaiseCellChanged(string propertyName, int row, int column, SheetArea sheetArea)
        {
            RaiseCellChanged(propertyName, row, column, 1, 1, sheetArea);
        }

        internal void RaiseCellChanged(string propertyName, int row, int column, int rowCount, int columnCount, SheetArea sheetArea)
        {
            if (!IsEventSuspend() && (CellChanged != null))
            {
                CellChanged(this, new CellChangedEventArgs(propertyName, row, column, rowCount, columnCount, sheetArea));
            }
        }

        internal void RaiseChartChanged(SpreadChart chart, ChartArea chartArea, string property)
        {
            if (!IsEventSuspend() && (ChartChanged != null))
            {
                ChartChanged(this, new ChartChangedEventArgs(chart, chartArea, property));
            }
        }

        internal void RaiseChartCollectionChanged()
        {
            RaisePropertyChanged("Charts");
        }

        internal void RaiseChartSelectionChangedEvent(SpreadChart chart)
        {
            if (!IsEventSuspend() && (ChartSelectionChanged != null))
            {
                ChartSelectionChanged(this, new ChartSelectionChangedEventArgs(chart));
            }
        }

        internal void RaiseColumnChanged(string propertyName, int column, int count, SheetArea sheetArea, SheetChangedEventAction type)
        {
            if (!IsEventSuspend() && (ColumnChanged != null))
            {
                ColumnChanged(this, new SheetChangedEventArgs(propertyName, -1, column, 0, count, sheetArea, type));
                NotifyFloatingObjectOnColumnChanged(propertyName);
            }
        }

        internal void RaiseColumnsSwapped(int column1, int column2)
        {
            if (ColumnsSwapped != null)
            {
                ColumnsSwapped(this, new ColumnsSwappedEventArgs(column1, column2));
            }
        }

        internal void RaiseFloatingObjectChanged(FloatingObject floatingObject, string property)
        {
            if (!IsEventSuspend() && (FloatingObjectChanged != null))
            {
                FloatingObjectChanged(this, new FloatingObjectChangedEventArgs(floatingObject, property));
            }
        }

        internal void RaiseFloatingObjectChangedOnSheetPropertyChanged(string sheetPropertyName)
        {
            RaiseFloatingObjectChanged(null, sheetPropertyName);
        }

        internal void RaiseFloatingObjectCollectionChanged()
        {
            RaisePropertyChanged("FloatingObjects");
        }

        internal void RaiseFloatingObjectSelectionChangedEvent(FloatingObject floatingObject)
        {
            if (!IsEventSuspend() && (FloatingObjectSelectionChanged != null))
            {
                FloatingObjectSelectionChanged(this, new FloatingObjectSelectionChangedEventArgs(floatingObject));
            }
        }

        internal void RaisePictureChanged(Picture picture, string property)
        {
            if (!IsEventSuspend() && (PictureChanged != null))
            {
                PictureChanged(this, new PictureChangedEventArgs(picture, property));
            }
        }

        internal void RaisePictureCollectionChanged()
        {
            RaisePropertyChanged("Pictures");
        }

        internal void RaisePictureSelectionChangedEvent(Picture picture)
        {
            if (!IsEventSuspend() && (PictureSelectionChanged != null))
            {
                PictureSelectionChanged(this, new PictureSelectionChangedEventArgs(picture));
            }
        }

        void RaisePropertyChanged(string propertyName)
        {
            if (!IsEventSuspend() && (PropertyChanged != null))
            {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
                NotifyChartShapeOnSheetPropertyChanged(propertyName);
            }
        }

        internal void RaiseRowChanged(string propertyName, int row, int count, SheetArea sheetArea, SheetChangedEventAction type)
        {
            if (!IsEventSuspend() && (RowChanged != null))
            {
                RowChanged(this, new SheetChangedEventArgs(propertyName, row, -1, count, 0, sheetArea, type));
                NotifyFloatingObjectOnRowChanged(propertyName);
            }
        }

        internal void RaiseRowsSwapped(int row1, int row2)
        {
            if (RowsSwapped != null)
            {
                RowsSwapped(this, new RowsSwappedEventArgs(row1, row2));
            }
        }

        internal void RaiseSurfaceChartCollectionChanged()
        {
            RaisePropertyChanged("SurfaceCharts");
        }

        internal bool ReadXmlInternal(XmlReader reader, bool invalidateLoadedData)
        {
            Serializer.InitReader(reader);
            Init(0, 0, 0, 0, false);
            _name = Serializer.ReadAttribute("name", reader);
            while (reader.Read())
            {
                if (reader.NodeType == ((XmlNodeType)((int)XmlNodeType.Element)))
                {
                    switch (reader.Name)
                    {
                        case "IsVisible":
                            _visible = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "ColumnHeaderVisible":
                            _columnHeaderVisible = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "RowHeaderVisible":
                            _rowHeaderVisible = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "ColumnFooterVisible":
                            _columnFooterVisible = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "StartingColumnNumber":
                            _startingColumnNumber = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "StartingRowNumber":
                            _startingRowNumber = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "ColumnHeaderAutoText":
                            _columnHeaderAutoText = (HeaderAutoText)Serializer.DeserializeObj(typeof(HeaderAutoText), reader);
                            break;

                        case "RowHeaderAutoText":
                            _rowHeaderAutoText = (HeaderAutoText)Serializer.DeserializeObj(typeof(HeaderAutoText), reader);
                            break;

                        case "ColumnHeaderAutoTextIndex":
                            _columnHeaderAutoTextIndex = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "RowHeaderAutoTextIndex":
                            _rowHeaderAutoTextIndex = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "DefaultColumnFooterRowHeight":
                            _defaultColumnFooterRowHeight = (double)((double)Serializer.DeserializeObj(typeof(double), reader));
                            break;

                        case "DefaultColumnHeaderRowHeight":
                            _defaultColumnHeaderRowHeight = (double)((double)Serializer.DeserializeObj(typeof(double), reader));
                            break;

                        case "DefaultRowHeight":
                            _defaultRowHeight = (double)((double)Serializer.DeserializeObj(typeof(double), reader));
                            break;

                        case "DefaultRowHeaderColumnWidth":
                            _defaultRowHeaderColumnWidth = (double)((double)Serializer.DeserializeObj(typeof(double), reader));
                            break;

                        case "DefaultColumnWidth":
                            _defaultColumnWidth = (double)((double)Serializer.DeserializeObj(typeof(double), reader));
                            break;

                        case "ZoomFactor":
                            _zoomFactor = (float)((float)Serializer.DeserializeObj(typeof(float), reader));
                            break;

                        case "AutoGenerateColumns":
                            _autoGeneratedColumns = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "DataAutoHeadings":
                            _dataAutoHeadings = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "PrintInfo":
                            _printInfo = Serializer.DeserializeObj(typeof(PrintInfo), reader) as PrintInfo;
                            break;

                        case "Protect":
                            _protect = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "NamedStyles":
                            _namedStyles = Serializer.DeserializeObj(typeof(StyleInfoCollection), reader) as StyleInfoCollection;
                            break;

                        case "DefaultStyle":
                            goto Label_0899;

                        case "DefaultColumnHeaderStyle":
                            goto Label_0900;

                        case "DefaultRowHeaderStyle":
                            goto Label_0997;

                        case "GridLineColor":
                            _gridLineColor = (Color)Serializer.DeserializeObj(typeof(Color), reader);
                            break;

                        case "ShowGridLine":
                            _showGridLine = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "SheetTabColor":
                            _sheetTabColor = (Color)Serializer.DeserializeObj(typeof(Color), reader);
                            break;

                        case "SheetTabThemeColor":
                            _sheetTabThemeColor = (string)((string)Serializer.DeserializeObj(typeof(string), reader));
                            break;

                        case "CellBlock":
                            _cellsBlock = Serializer.DeserializeObj(typeof(StorageBlock), reader) as StorageBlock;
                            break;

                        case "RowHeaderBlock":
                            _rowHeadersBlock = Serializer.DeserializeObj(typeof(StorageBlock), reader) as StorageBlock;
                            break;

                        case "ColumnHeaderBlock":
                            _columnHeadersBlock = Serializer.DeserializeObj(typeof(StorageBlock), reader) as StorageBlock;
                            break;

                        case "ViewportRowsAxis":
                            _viewportRows = Serializer.DeserializeObj(typeof(SparseArray<AxisInfo>), reader) as SparseArray<AxisInfo>;
                            break;

                        case "ViewportColumnsAxis":
                            _viewportColumns = Serializer.DeserializeObj(typeof(SparseArray<AxisInfo>), reader) as SparseArray<AxisInfo>;
                            break;

                        case "RowHeaderColumnsAxis":
                            _rowHeaderColumns = Serializer.DeserializeObj(typeof(SparseArray<AxisInfo>), reader) as SparseArray<AxisInfo>;
                            break;

                        case "ColumnHeaderRowsAxis":
                            _columnHeaderRows = Serializer.DeserializeObj(typeof(SparseArray<AxisInfo>), reader) as SparseArray<AxisInfo>;
                            break;

                        case "ViewportRowsAxisStyle":
                            _viewportRowsStyles = Serializer.DeserializeObj(typeof(SparseArray<object>), reader) as SparseArray<object>;
                            break;

                        case "ViewportColumnsAxisStyle":
                            _viewportColumnsStyles = Serializer.DeserializeObj(typeof(SparseArray<object>), reader) as SparseArray<object>;
                            break;

                        case "RowHeaderRowsAxisStyle":
                            _rowHeaderRowsStyles = Serializer.DeserializeObj(typeof(SparseArray<object>), reader) as SparseArray<object>;
                            break;

                        case "RowHeaderColumnsAxisStyle":
                            _rowHeaderColumnsStyles = Serializer.DeserializeObj(typeof(SparseArray<object>), reader) as SparseArray<object>;
                            break;

                        case "ColumnHeaderRowsAxisStyle":
                            _columnHeaderRowsStyles = Serializer.DeserializeObj(typeof(SparseArray<object>), reader) as SparseArray<object>;
                            break;

                        case "ColumnHeaderColumnsAxisStyle":
                            _columnHeaderColumnsStyles = Serializer.DeserializeObj(typeof(SparseArray<object>), reader) as SparseArray<object>;
                            break;

                        case "CellsSpanModel":
                            _spanModel = Serializer.DeserializeObj(typeof(SheetSpanModel), reader) as SheetSpanModel;
                            break;

                        case "RowHeaderSpanModel":
                            _rowHeaderSpanModel = Serializer.DeserializeObj(typeof(SheetSpanModel), reader) as SheetSpanModel;
                            break;

                        case "ColumnHeaderSpanModel":
                            _columnHeaderSpanModel = Serializer.DeserializeObj(typeof(SheetSpanModel), reader) as SheetSpanModel;
                            break;

                        case "SelectionModel":
                            {
                                SheetSelectionModel model = Serializer.DeserializeObj(typeof(SheetSelectionModel), reader) as SheetSelectionModel;
                                _selectionModel.CopyFrom(model);
                                break;
                            }
                        case "RowFilter":
                            goto Label_0D56;

                        case "Conditions":
                            goto Label_0DBD;

                        case "SelectionBorderColor":
                            goto Label_0E21;

                        case "SelectionBorderThemeColor":
                            _selectionBorderThemeColor = (string)((string)Serializer.DeserializeObj(typeof(string), reader));
                            break;

                        case "TouchSelectionGripperBackgroundColor":
                            goto Label_0E66;

                        case "TouchSelectionGripperBackgroundThemeColor":
                            _touchSelectionGripperBackgroundThemeColor = (string)((string)Serializer.DeserializeObj(typeof(string), reader));
                            break;

                        case "SelectionBackground":
                            goto Label_0EAB;

                        case "FrozenRowCount":
                            FrozenRowCount = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "FrozenColumnCount":
                            FrozenColumnCount = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "FrozenTrailingRowCount":
                            FrozenTrailingRowCount = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "FrozenTrailingColumnCount":
                            FrozenTrailingColumnCount = (int)((int)Serializer.DeserializeObj(typeof(int), reader));
                            break;

                        case "ReferenceStyle":
                            _referenceStyle = (ReferenceStyle)Serializer.DeserializeObj(typeof(ReferenceStyle), reader);
                            break;

                        case "AutoCalculation":
                            _autoRecalculation = (bool)((bool)Serializer.DeserializeObj(typeof(bool), reader));
                            break;

                        case "Names":
                            goto Label_0FAE;

                        case "Formula":
                            goto Label_0FD8;

                        case "BindingCellValues":
                            {
                                int row = int.Parse(Serializer.ReadAttribute("rc", reader));
                                int column = int.Parse(Serializer.ReadAttribute("cc", reader));
                                _cachedDeserializedBindingValues = new DataMatrix<object>(row, column);
                                Serializer.DeserializeMatrix<object>(_cachedDeserializedBindingValues, null, reader);
                                break;
                            }
                        case "BindingManager":
                            _bindingManager = Serializer.DeserializeObj(typeof(SheetBindingManager), reader) as SheetBindingManager;
                            break;

                        case "DataSource":
                            _cachedDeserializedDatasource = Serializer.DeserializeObj(null, reader);
                            break;

                        case "RowRangeGroup":
                            goto Label_1094;

                        case "ColumnRangeGroup":
                            goto Label_10FB;

                        case "ViewportInfo":
                            _viewportInfo = Serializer.DeserializeObj(typeof(ViewportInfo), reader) as ViewportInfo;
                            break;

                        case "SparklineGroupManager":
                            _sparklineGroupManager = Serializer.DeserializeObj(typeof(WorksheetSparklineGroupManager), reader) as WorksheetSparklineGroupManager;
                            break;

                        case "Tables":
                            _tables = new EricTables(this);
                            Serializer.DeserializeTables(_tables, reader);
                            break;

                        case "Charts":
                            _spreadCharts = new SpreadCharts(this);
                            _spreadCharts.ReadXmlInternal(reader);
                            break;

                        case "FloatingObjects":
                            _floatingObjects = new FloatingObjects(this);
                            _floatingObjects.ReadXmlInternal(reader);
                            break;

                        case "Pictures":
                            _pictures = new SpreadPictures(this);
                            _pictures.ReadXmlInternal(reader);
                            break;

                        case "Tag":
                            _tag = Serializer.DeserializeTag(reader);
                            break;
                    }
                }
                goto Label_1240;
            Label_0899:
                if (_defaultStyle != null)
                    _defaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                _defaultStyle = Serializer.DeserializeObj(typeof(StyleInfo), reader) as StyleInfo;
                if (_defaultStyle != null)
                    _defaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                goto Label_1240;
            Label_0900:
                if (_columnHeaderDefaultStyle != null)
                    _columnHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                _columnHeaderDefaultStyle = Serializer.DeserializeObj(typeof(StyleInfo), reader) as StyleInfo;
                if (_columnHeaderDefaultStyle.Foreground == null)
                    _columnHeaderDefaultStyle.Foreground = new SolidColorBrush(Colors.Black);
                if (_columnHeaderDefaultStyle != null)
                    _columnHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                goto Label_1240;
            Label_0997:
                if (_rowHeaderDefaultStyle != null)
                    _rowHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                _rowHeaderDefaultStyle = Serializer.DeserializeObj(typeof(StyleInfo), reader) as StyleInfo;
                if (_rowHeaderDefaultStyle.Foreground == null)
                    _rowHeaderDefaultStyle.Foreground = new SolidColorBrush(Colors.Black);
                if (_rowHeaderDefaultStyle != null)
                    _rowHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                goto Label_1240;
            Label_0D56:
                if (_rowFilter != null)
                    _rowFilter.Changed -= new EventHandler<FilterEventArgs>(OnFilterStatusChanged);
                _rowFilter = Serializer.DeserializeObj(null, reader) as RowFilterBase;
                if (_rowFilter != null)
                    _rowFilter.Changed += new EventHandler<FilterEventArgs>(OnFilterStatusChanged);
                _rowFilter.Sheet = this;
                goto Label_1240;
            Label_0DBD:
                if (_conditionalFormats != null)
                    _conditionalFormats.RulesChanged -= new EventHandler<RulesChangedEventArgs>(OnConditionalFormatsRulesChanged);
                _conditionalFormats = new ConditionalFormat(this);
                Serializer.DeserializeConditionalFormats(_conditionalFormats, reader);
                if (_conditionalFormats != null)
                    _conditionalFormats.RulesChanged += new EventHandler<RulesChangedEventArgs>(OnConditionalFormatsRulesChanged);
                goto Label_1240;
            Label_0E21:
                _selectionBorderColor = (Color)Serializer.DeserializeObj(typeof(Color), reader);
                goto Label_1240;
            Label_0E66:
                _touchSelectionGripperBackgroundColor = (Color)Serializer.DeserializeObj(typeof(Color), reader);
                goto Label_1240;
            Label_0EAB:
                _selectionBackground = (Brush)Serializer.DeserializeObj(typeof(Brush), reader);
                goto Label_1240;
            Label_0FAE:
                if (_cachedDeserializedCustomNames == null)
                    _cachedDeserializedCustomNames = new List<List<object>>();
                Serializer.DeserializeNameInfos(reader, _cachedDeserializedCustomNames);
                goto Label_1240;
            Label_0FD8:
                if (_cachedDeserializedFormulaStrings == null)
                    _cachedDeserializedFormulaStrings = new List<List<object>>();
                Serializer.DeserializeCalcStorage(_cachedDeserializedFormulaStrings, this, reader);
                goto Label_1240;
            Label_1094:
                if (_rowRangeGroup != null)
                    _rowRangeGroup.Changed -= new EventHandler(OnRowRangeGroupChanged);
                _rowRangeGroup = Serializer.DeserializeObj(typeof(RangeGroup), reader) as RangeGroup;
                if (_rowRangeGroup != null)
                    _rowRangeGroup.Changed += new EventHandler(OnRowRangeGroupChanged);
                goto Label_1240;
            Label_10FB:
                if (_columnRangeGroup != null)
                    _columnRangeGroup.Changed -= new EventHandler(OnColumnRangeGroupChanged);
                _columnRangeGroup = Serializer.DeserializeObj(typeof(RangeGroup), reader) as RangeGroup;
                if (_columnRangeGroup != null)
                    _columnRangeGroup.Changed += new EventHandler(OnColumnRangeGroupChanged);
            Label_1240:
                if ((reader.NodeType == ((XmlNodeType)((int)XmlNodeType.EndElement))) && (reader.Name == "Sheet"))
                    break;
            }
            if (invalidateLoadedData)
                InvalidateLoadedData();
            return true;
        }

        /// <summary>
        /// Evaluates all the cell formulas on the sheet that have changed since the last calculation cycle.
        /// </summary>
        public void Recalculate()
        {
            Recalculate(true);
        }

        /// <summary>
        /// Evaluates all the cell formulas on the sheet that have changed since the last calculation cycle.
        /// </summary>
        /// <param name="forceRecalculateAll">Whether to force recalculation of all formulas in the current manager.</param>
        public void Recalculate(bool forceRecalculateAll)
        {
            if ((_calcManager != null) && (forceRecalculateAll || (_calcEngineSuspended == 0)))
            {
                _calcManager.Recalculate(0xc350, true);
            }
        }

        /// <summary>
        /// Reloads the binding data.
        /// </summary>
        public void ReloadBindingData()
        {
            if (_bindingManager != null)
            {
                _bindingManager.ReloadBindingData();
                GetStorage(SheetArea.Cells).Clear(StorageType.Data);
            }
        }

        /// <summary>
        /// Removes the chart with specified name.
        /// </summary>
        /// <param name="name">The name.</param>
        public void RemoveChart(string name)
        {
            if (_spreadCharts != null)
            {
                SpreadChart item = _spreadCharts.Find(name);
                if (item != null)
                {
                    _spreadCharts.Remove(item);
                }
            }
        }

        /// <summary>
        /// Removes the column or columns on this sheet at the specified index.
        /// </summary>
        /// <param name="column">The index of the first column to remove.</param>
        /// <param name="count">The number of columns to remove.</param>
        public void RemoveColumns(int column, int count)
        {
            RemoveColumns(column, count, SheetArea.Cells);
        }

        /// <summary>
        /// Removes the column or columns on this sheet at the specified index in the specified sheet area.
        /// </summary>
        /// <param name="sheetArea">The sheet area from which to remove columns.</param>
        /// <param name="column">The index of the first column to remove.</param>
        /// <param name="count">The number of columns to remove.</param>
        public void RemoveColumns(int column, int count, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == SheetArea.ColumnHeader))
            {
                CalcExpressionHelper.ValidateIntersectedArrayFormula(CalcManager, -1, column, -1, count, false);
                _cellsBlock.RemoveColumns(column, count);
                _viewportColumns.RemoveRange(column, count);
                _viewportColumnsStyles.RemoveRange(column, count);
                _columnHeadersBlock.RemoveColumns(column, count);
                _columnHeaderColumnsStyles.RemoveRange(column, count);
            }
            else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                _rowHeadersBlock.RemoveColumns(column, count);
                _rowHeaderColumns.RemoveRange(column, count);
                _rowHeaderColumnsStyles.RemoveRange(column, count);
            }
            OnColumnsRemoved(column, count, sheetArea);
            RaiseColumnChanged(string.Empty, column, count, sheetArea, SheetChangedEventAction.ColumnsRemoved);
        }

        /// <summary>
        /// Removes the column viewport.
        /// </summary>
        /// <param name="columnViewportIndex">The index of the column viewport.</param>
        public void RemoveColumnViewport(int columnViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if ((columnViewportIndex < 0) && (columnViewportIndex >= viewportInfo.ColumnViewportCount))
            {
                throw new ArgumentException("columnViewportIndex");
            }
            if (viewportInfo.ColumnViewportCount <= 1)
            {
                throw new ArgumentException(string.Format(ResourceStrings.CouldnotRemoveTheLastViewport, (object[])new object[] { "column" }));
            }
            int num = columnViewportIndex + 1;
            if (num >= viewportInfo.ColumnViewportCount)
            {
                columnViewportIndex--;
                num--;
            }
            MergeColumnViewports(num, columnViewportIndex);
        }

        /// <summary>
        /// Removes the specified user-defined custom function.
        /// </summary>
        /// <param name="name">The name of the custom function to remove.</param>
        public void RemoveCustomFunctions(string name)
        {
            name = name.ToUpper(CultureInfo.CurrentCulture);
            if (Functions.ContainsKey(name))
            {
                Functions.Remove(name);
                OnCustomFunctionChanged(name);
                if (AutoRecalculation && (_calcEngineSuspended == 0))
                {
                    Recalculate();
                }
            }
        }

        /// <summary>
        /// Removes the floating object with the specified name.
        /// </summary>
        /// <param name="name">The name of the floating object to remove.</param>
        public void RemoveFloatingObject(string name)
        {
            if (_floatingObjects != null)
            {
                FloatingObject item = _floatingObjects.Find(name);
                if (item != null)
                {
                    _floatingObjects.Remove(item);
                }
            }
        }

        /// <summary>
        /// Removes the picture with the specified name.
        /// </summary>
        /// <param name="name">The name of the picture to remove.</param>
        public void RemovePicture(string name)
        {
            if (_pictures != null)
            {
                Picture item = _pictures.Find(name);
                if (item != null)
                {
                    _pictures.Remove(item);
                }
            }
        }

        /// <summary>
        /// Removes the row or rows on this sheet at the specified index.
        /// </summary>
        /// <param name="row">The index of the first row to remove.</param>
        /// <param name="count">The number of rows to remove.</param>
        public void RemoveRows(int row, int count)
        {
            RemoveRows(row, count, SheetArea.Cells);
        }

        /// <summary>
        /// Removes the row or rows on this sheet at the specified index in the specified sheet area.
        /// </summary>
        /// <param name="sheetArea">The sheet area in which to remove rows.</param>
        /// <param name="row">The index of the first row to remove.</param>
        /// <param name="count">The number of rows to remove.</param>
        public void RemoveRows(int row, int count, SheetArea sheetArea)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)))
            {
                CalcExpressionHelper.ValidateIntersectedArrayFormula(CalcManager, row, -1, count, -1, false);
                _cellsBlock.RemoveRows(row, count);
                _viewportRows.RemoveRange(row, count);
                _viewportRowsStyles.RemoveRange(row, count);
                _rowHeadersBlock.RemoveRows(row, count);
                _rowHeaderRowsStyles.RemoveRange(row, count);
            }
            else if (sheetArea == SheetArea.ColumnHeader)
            {
                _columnHeadersBlock.RemoveRows(row, count);
                _columnHeaderRows.RemoveRange(row, count);
                _columnHeaderRowsStyles.RemoveRange(row, count);
            }
            OnRowsRemoved(row, count, sheetArea);
            RaiseRowChanged(string.Empty, row, count, sheetArea, SheetChangedEventAction.RowsRemoved);
        }

        /// <summary>
        /// Removes the row viewport.
        /// </summary>
        /// <param name="rowViewportIndex">The index of the row viewport.</param>
        public void RemoveRowViewport(int rowViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if ((rowViewportIndex < 0) && (rowViewportIndex >= viewportInfo.RowViewportCount))
            {
                throw new ArgumentException("rowViewportIndex");
            }
            if (viewportInfo.RowViewportCount <= 1)
            {
                throw new ArgumentException(string.Format(ResourceStrings.CouldnotRemoveTheLastViewport, (object[])new object[] { "row" }));
            }
            int num = rowViewportIndex + 1;
            if (num >= viewportInfo.RowViewportCount)
            {
                rowViewportIndex--;
                num--;
            }
            MergeRowViewports(num, rowViewportIndex);
        }

        /// <summary>
        /// Removes the span that contains a specified anchor cell from a range of cells in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index of the anchor cell for the span (at which spanned cells start).</param>
        /// <param name="column">The column index of the anchor cell for the span (at which spanned cells start).</param>
        public void RemoveSpanCell(int row, int column)
        {
            RemoveSpanCell(row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Removes the span that contains a specified anchor cell from a range of cells in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index of the anchor cell for the span (at which spanned cells start).</param>
        /// <param name="column">The column index of the anchor cell for the span (at which spanned cells start).</param>
        /// <param name="sheetArea">The sheet area.</param>
        public void RemoveSpanCell(int row, int column, SheetArea sheetArea)
        {
            SheetSpanModelBase spanModel = GetSpanModel(sheetArea);
            if (spanModel != null)
            {
                spanModel.Remove(row, column);
            }
        }

        /// <summary>
        /// Removes the sparkline of the specified cell.
        /// </summary>
        /// <param name="row">The row.</param>
        /// <param name="column">The column.</param>
        public void RemoveSparkline(int row, int column)
        {
            InternalRemoveSparkline(row, column);
            RaiseCellChanged("Sparkline", row, column, SheetArea.Cells);
        }

        /// <summary>
        /// Removes a specified table.
        /// </summary>
        /// <param name="table">The table instance to remove.</param>
        public void RemoveTable(SheetTable table)
        {
            if (table != null)
            {
                table.PropertyChanged -= new PropertyChangedEventHandler(OnTablePropertyChanged);
            }
            if (_tables != null)
            {
                CellRange range = table.Range;
                RaiseCellChanged("[Table]", range.Row, range.Column, range.RowCount, range.ColumnCount, SheetArea.Cells);
                _tables.RemoveTable(table);
            }
        }

        /// <summary>
        /// Removes the table with the specified name.
        /// </summary>
        /// <param name="tableName">The table name to remove.</param>
        public void RemoveTable(string tableName)
        {
            SheetTable table = FindTable(tableName);
            if (table == null)
            {
                throw new NotSupportedException(ResourceStrings.TableNotFoundError);
            }
            table.PropertyChanged -= new PropertyChangedEventHandler(OnTablePropertyChanged);
            RemoveTable(table);
        }

        internal void ResetCalcEngine()
        {
            if (_calcManager != null)
            {
                _calcManager.Dispose();
            }
            _calcManager = null;
            _calcService = null;
            _calcAxial = null;
            _calcStorage = null;
        }

        /// <summary>
        /// Resets the color of the touch selection gripper background.
        /// </summary>
        public void ResetTouchSelectionGripperBackgroundColor()
        {
            _touchSelectionGripperBackgroundColor = Color.FromArgb(0xff, 0xff, 0xff, 0xff);
        }

        /// <summary>
        /// Resets the theme color of the touch selection gripper background.
        /// </summary>
        public void ResetTouchSelectionGripperBackgroundThemeColor()
        {
            _touchSelectionGripperBackgroundThemeColor = null;
        }

        /// <summary>
        /// Changes the table size.
        /// </summary>
        /// <param name="table">The table instance to resize.</param>
        /// <param name="rowCount">The new table row count.</param>
        /// <param name="ColumnCount">The new table column count.</param>
        public void ResizeTable(SheetTable table, int rowCount, int ColumnCount)
        {
            if (table == null)
            {
                throw new ArgumentNullException("table");
            }
            CellRange range = table.Range;
            table.Resize(rowCount, ColumnCount);
            CellRange range2 = table.Range;
            RaiseCellChanged("[Table]", range.Row, range.Column, range.RowCount, range.ColumnCount, SheetArea.Cells);
            RaiseCellChanged("[Table]", range2.Row, range2.Column, range2.RowCount, range2.ColumnCount, SheetArea.Cells);
        }

        /// <summary>
        /// Changes the table size.
        /// </summary>
        /// <param name="tableName">The table name to resize.</param>
        /// <param name="rowCount">The new table row count.</param>
        /// <param name="ColumnCount">The new table column count.</param>
        public void ResizeTable(string tableName, int rowCount, int ColumnCount)
        {
            SheetTable table = FindTable(tableName);
            if (table == null)
            {
                throw new NotSupportedException(ResourceStrings.TableNotFoundError);
            }
            ResizeTable(table, rowCount, ColumnCount);
        }

        /// <summary>
        /// Resumes the calculation service.
        /// </summary>
        public void ResumeCalcService()
        {
            ResumeCalcService(true);
        }

        internal void ResumeCalcService(bool recalculateAll)
        {
            _calcEngineSuspended--;
            _calcEngineSuspended = Math.Max((short)0, _calcEngineSuspended);
            if ((_calcEngineSuspended == 0) && recalculateAll)
            {
                FormulaService.Recalculate(0xc350, true);
            }
        }

        /// <summary>
        /// Resumes the chart events.
        /// </summary>
        internal void ResumeChartEvents()
        {
            if (_spreadCharts != null)
            {
                _spreadCharts.ResumeItemsEvent();
                _spreadCharts.ResumeEvent();
            }
        }

        /// <summary>
        /// Resumes the floating object events.
        /// </summary>
        internal void ResumeFloatingObjectEvents()
        {
            if (_floatingObjects != null)
            {
                _floatingObjects.ResumeItemsEvent();
                _floatingObjects.ResumeEvent();
            }
        }

        /// <summary>
        /// Resumes the pictures events.
        /// </summary>
        internal void ResumePicturesEvents()
        {
            if (_pictures != null)
            {
                _pictures.ResumeEvent();
                _pictures.ResumeItemsEvent();
            }
        }

        /// <summary>
        /// Resumes the SpanModel, RowHeaderSpanModel, and ColumnHeaderSpanModel events.
        /// </summary>
        public void ResumeSpanModelEvent()
        {
            if (SpanModel != null)
            {
                SpanModel.ResumeEvent();
            }
            if (RowHeaderSpanModel != null)
            {
                RowHeaderSpanModel.ResumeEvent();
            }
            if (ColumnHeaderSpanModel != null)
            {
                ColumnHeaderSpanModel.ResumeEvent();
            }
        }

        public void SaveCsv(Stream stream, TextFileSaveFlags flags, Encoding encoding)
        {
            SaveTextFileRangeBackGround(-1, -1, -1, -1, stream, flags, "\r\n", ",", "\"", encoding);
        }

        /// <summary>
        /// Saves the CSV (comma-separated values) file asynchronously.
        /// </summary>
        /// <param name="stream">The stream to which to save the content.</param>
        /// <param name="flags">The export flags.</param>
        /// <returns></returns>
        public IAsyncAction SaveCsvAsync(Stream stream, TextFileSaveFlags flags)
        {
            return SaveCsvAsync(stream, flags, Encoding.UTF8);
        }

        /// <summary>
        /// Saves the CSV (comma-separated values) file asynchronously.
        /// </summary>
        /// <param name="stream">The stream to which to save the content.</param>
        /// <param name="flags">The export flags.</param>
        /// <param name="encoding">The encoding.</param>
        /// <returns></returns>
        public IAsyncAction SaveCsvAsync(Stream stream, TextFileSaveFlags flags, Encoding encoding)
        {
            return SaveCsvAsync(-1, -1, -1, -1, stream, flags, encoding);
        }

        /// <summary>
        /// Saves the CSV (comma-separated values) file asynchronously.
        /// </summary>
        /// <param name="row">The starting row index.</param>
        /// <param name="column">The starting column index.</param>
        /// <param name="rowCount">The number of rows.</param>
        /// <param name="columnCount">The number of columns.</param>
        /// <param name="stream">The stream to which to save the content.</param>
        /// <param name="flags">The export flags.</param>
        /// <returns></returns>
        public IAsyncAction SaveCsvAsync(int row, int column, int rowCount, int columnCount, Stream stream, TextFileSaveFlags flags)
        {
            return SaveCsvAsync(row, column, rowCount, columnCount, stream, flags, Encoding.UTF8);
        }

        /// <summary>
        /// Saves the CSV (comma-separated values) file asynchronously.
        /// </summary>
        /// <param name="row">The starting row index.</param>
        /// <param name="column">The starting column index.</param>
        /// <param name="rowCount">The number of rows.</param>
        /// <param name="columnCount">The number of columns.</param>
        /// <param name="stream">The stream to which to save the content.</param>
        /// <param name="flags">The export flags.</param>
        /// <param name="encoding">The encoding.</param>
        /// <returns></returns>
        public IAsyncAction SaveCsvAsync(int row, int column, int rowCount, int columnCount, Stream stream, TextFileSaveFlags flags, Encoding encoding)
        {
            return SaveTextFileRangeAsync(row, column, rowCount, columnCount, stream, flags, "\r\n", ",", "\"", encoding);
        }

        /// <summary>
        /// Saves the data on the sheet to the specified stream asynchronously.
        /// </summary>
        /// <param name="stream">The HTML stream.</param>
        public IAsyncAction SaveHtmlAsync(Stream stream)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    SaveHtmlBackGround(stream);
                });
            });
        }

        void SaveHtmlBackGround(Stream stream)
        {
            if (stream == null)
            {
                throw new ArgumentNullException("stream");
            }
            StreamWriter writer = null;
            try
            {
                writer = new StreamWriter(stream)
                {
                    AutoFlush = true
                };
                string str = HtmlReport.SaveHtmlRange(this, -1, -1, -1, -1, RowHeader.IsVisible, ColumnHeader.IsVisible);
                writer.Write(str);
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
        }

        /// <summary>
        /// Saves the sheet to the specified stream asynchronously.
        /// </summary>
        /// <param name="stream">The stream to which to save the data.</param>
        public IAsyncAction SavePdfAsync(Stream stream)
        {
            return SavePdfAsync(stream, null);
        }

        /// <summary>
        /// Saves the sheet to the specified stream using the specified settings asynchronously.
        /// </summary>
        /// <param name="stream">The stream to which to save the data.</param>
        /// <param name="settings">The export settings.</param>
        public IAsyncAction SavePdfAsync(Stream stream, PdfExportSettings settings)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    SavePdfBackGround(stream, settings);
                });
            });
        }

        void SavePdfBackGround(Stream stream, PdfExportSettings settings)
        {
            if (stream == null)
            {
                throw new ArgumentNullException("stream");
            }
            if (!stream.CanWrite)
            {
                throw new ArgumentException();
            }
            new PdfExporter(GetPrintableReport(), settings).Export(stream);
        }

        public void SaveTextFileRange(int row, int column, int rowCount, int columnCount, Stream stream, TextFileSaveFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter)
        {
            SaveTextFileRangeBackGround(row, column, rowCount, columnCount, stream, flags, rowDelimiter, columnDelimiter, cellDelimiter, Encoding.UTF8);
        }

        /// <summary>
        /// Saves the range of cells in the sheet as delimited text and saves 
        /// to a stream with the specified delimiters asynchronously. 
        /// </summary>
        /// <param name="row">The starting row index.</param>
        /// <param name="column">The starting column index.</param>
        /// <param name="rowCount">The number of rows.</param>
        /// <param name="columnCount">The number of columns.</param>
        /// <param name="stream">The stream to which to save the text of the range.</param>
        /// <param name="flags">The export flags.</param>
        /// <param name="rowDelimiter">The row delimiter string.</param>
        /// <param name="columnDelimiter">The column delimiter string.</param>
        /// <param name="cellDelimiter">The cell delimiter string.</param>
        public IAsyncAction SaveTextFileRangeAsync(int row, int column, int rowCount, int columnCount, Stream stream, TextFileSaveFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter)
        {
            return SaveTextFileRangeAsync(row, column, rowCount, columnCount, stream, flags, rowDelimiter, columnDelimiter, cellDelimiter, Encoding.UTF8);
        }

        /// <summary>
        /// Saves the range of cells in the sheet as delimited text and saves 
        /// to a stream with the specified delimiters and encoding asynchronously.
        /// </summary>
        /// <param name="row">The starting row index.</param>
        /// <param name="column">The starting column index.</param>
        /// <param name="rowCount">The number of rows.</param>
        /// <param name="columnCount">The number of columns.</param>
        /// <param name="stream">The stream to which to save the range text.</param>
        /// <param name="flags">The export flags.</param>
        /// <param name="rowDelimiter">The row delimiter string.</param>
        /// <param name="columnDelimiter">The column delimiter string.</param>
        /// <param name="cellDelimiter">The cell delimiter string.</param>
        /// <param name="encoding">The encoding.</param>
        /// <returns>
        /// <c>true</c> if the data is saved to the stream successfully; otherwise, <c>false</c>.
        /// </returns>
        /// <remarks>
        /// Note that the cell delimiter cannot be contained in a row or column delimiter.
        /// </remarks>
        /// <exception cref="T:System.ArgumentException">
        /// Cell delimiter cannot be contained in a row or column delimiter.
        /// </exception>
        public IAsyncAction SaveTextFileRangeAsync(int row, int column, int rowCount, int columnCount, Stream stream, TextFileSaveFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter, Encoding encoding)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    SaveTextFileRangeBackGround(row, column, rowCount, columnCount, stream, flags, rowDelimiter, columnDelimiter, cellDelimiter, encoding);
                });
            });
        }

        void SaveTextFileRangeBackGround(int row, int column, int rowCount, int columnCount, Stream stream, TextFileSaveFlags flags, string rowDelimiter, string columnDelimiter, string cellDelimiter, Encoding encoding)
        {
            if (encoding == null)
            {
                throw new ArgumentNullException("encoding");
            }
            if (stream == null)
            {
                throw new ArgumentNullException("stream");
            }
            if (!stream.CanWrite)
            {
                throw new ArgumentNullException("stream");
            }
            StreamWriter writer = null;
            try
            {
                writer = new StreamWriter(stream, encoding)
                {
                    AutoFlush = true
                };
                string str = CsvImp.GetRangeText(this, row, rowCount, column, columnCount, rowDelimiter, columnDelimiter, cellDelimiter, true, flags);
                writer.Write(str);
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
        }

        /// <summary>
        /// Saves the worksheet to an XML stream asynchronously.
        /// </summary>
        /// <param name="xmlStream">The XML stream.</param>
        public IAsyncAction SaveXmlAsync(Stream xmlStream)
        {
            return SaveXmlAsync(xmlStream, false);
        }

        /// <summary>
        /// Saves the worksheet to an XML writer asynchronously.
        /// </summary>
        /// <param name="writer">The XML writer.</param>
        public IAsyncAction SaveXmlAsync(XmlWriter writer)
        {
            return SaveXmlAsync(writer, false);
        }

        /// <summary>
        /// Saves the worksheet to an XML stream asynchronously.
        /// </summary>
        /// <param name="xmlStream">The memory stream.</param>
        /// <param name="dataOnly">Specifies whether to only save data.</param>
        public IAsyncAction SaveXmlAsync(Stream xmlStream, bool dataOnly)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    SaveXmlBackGround(xmlStream, dataOnly);
                });
            });
        }

        /// <summary>
        /// Saves the worksheet to an XML writer asynchronously.
        /// </summary>
        /// <param name="writer">The XML writer.</param>
        /// <param name="dataOnly">Specifies whether to only save data.</param>
        public IAsyncAction SaveXmlAsync(XmlWriter writer, bool dataOnly)
        {
            return AsyncInfo.Run(delegate(CancellationToken token)
            {
                return Task.Factory.StartNew(delegate
                {
                    SaveXmlBackGround(writer, dataOnly);
                });
            });
        }

        void SaveXmlBackGround(Stream xmlStream, bool dataOnly)
        {
            try
            {
                XmlWriter writer = XmlWriter.Create(xmlStream);
                if (writer != null)
                {
                    SaveXmlBackGround(writer, dataOnly);
                    writer.Close();
                }
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
        }

        void SaveXmlBackGround(XmlWriter writer, bool dataOnly)
        {
            try
            {
                writer.WriteStartElement("Sheet");
                WriteXmlInternal(writer, dataOnly, ExcludePrintContent.None);
                writer.WriteEndElement();
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
        }

        /// <summary>
        /// Save Xml For printing.
        /// </summary>
        /// <param name="xmlStream"></param>
        internal void SaveXmlForPrint(Stream xmlStream)
        {
            try
            {
                XmlWriter writer = XmlWriter.Create(xmlStream);
                if (writer != null)
                {
                    writer.WriteStartElement("Sheet");
                    WriteXmlInternal(writer, false, ExcludePrintContent.ConditionalFormat | ExcludePrintContent.Formula);
                    writer.WriteEndElement();
                    writer.Close();
                }
            }
            catch (Exception exception)
            {
                while ((exception is TargetInvocationException) && (exception.InnerException != null))
                {
                    exception = exception.InnerException;
                }
                throw exception;
            }
        }

        /// <summary>
        /// Searches the text in the cells.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="foundRowIndex">The row index at which the match is found.</param>
        /// <param name="foundColumnIndex">The column index at which the match is found.</param>
        public void Search(string searchString, out int foundRowIndex, out int foundColumnIndex)
        {
            Search(searchString, SearchFlags.UseWildCards | SearchFlags.ExactMatch, out foundRowIndex, out foundColumnIndex);
        }

        /// <summary>
        /// Searches the text in the cells for the specified string with the specified criteria.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="searchFlags">The <see cref="T:SearchFlags" /> enumeration that specifies the search options.</param>
        /// <param name="foundRowIndex">The row index at which the match is found.</param>
        /// <param name="foundColumnIndex">The column index at which the match is found.</param>
        public void Search(string searchString, SearchFlags searchFlags, out int foundRowIndex, out int foundColumnIndex)
        {
            Search(searchString, searchFlags, SearchOrder.ZOrder, out foundRowIndex, out foundColumnIndex);
        }

        /// <summary>
        /// Searches the text in the cells for the specified string with the specified criteria.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="searchFlags">The <see cref="T:SearchFlags" /> enumeration that specifies the search options.</param>
        /// <param name="searchOrder">The <see cref="T:SearchOrder" /> enumeration that specifies whether the search goes by column, row coordinates or row, column coordinates.</param>
        /// <param name="foundRowIndex">The row index at which the match is found.</param>
        /// <param name="foundColumnIndex">The column index at which the match is found.</param>
        public void Search(string searchString, SearchFlags searchFlags, SearchOrder searchOrder, out int foundRowIndex, out int foundColumnIndex)
        {
            Search(searchString, searchFlags, searchOrder, SearchFoundFlags.CellText, out foundRowIndex, out foundColumnIndex);
        }

        /// <summary>
        /// Searches the text in the cells for the specified string with the specified criteria
        /// and whether to search notes and tags as well.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="searchFlags">The <see cref="T:SearchFlags" /> enumeration that specifies the search options.</param>
        /// <param name="searchOrder">The <see cref="T:SearchOrder" /> enumeration that specifies whether the search goes by column, row coordinates or row, column coordinates.</param>
        /// <param name="searchTarget">The <see cref="T:SearchFoundFlags" /> enumeration that indicates whether the search includes the content in the cell notes, tags, or text.</param>
        /// <param name="foundRowIndex">The row index at which the match is found.</param>
        /// <param name="foundColumnIndex">The column index at which the match is found.</param>
        /// <returns>A <see cref="T:SearchFoundFlags" /> enumeration that specifies what is matched.</returns>
        public SearchFoundFlags Search(string searchString, SearchFlags searchFlags, SearchOrder searchOrder, SearchFoundFlags searchTarget, out int foundRowIndex, out int foundColumnIndex)
        {
            return Search(searchString, searchFlags, searchOrder, searchTarget, SheetArea.Cells, out foundRowIndex, out foundColumnIndex);
        }

        /// <summary>
        /// Searches the text in the cells in the specified sheet area for the specified string with the specified criteria 
        /// and whether to search notes and tags as well.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="searchFlags">The <see cref="T:SearchFlags" /> enumeration that specifies the search options.</param>
        /// <param name="searchOrder">The <see cref="T:SearchOrder" /> enumeration that specifies whether the search goes by column, row coordinates or row, column coordinates.</param>
        /// <param name="searchTarget">The <see cref="T:SearchFoundFlags" /> enumeration that indicates whether the search includes the content in the cell notes, tags, or text.</param>
        /// <param name="sheetArea">The sheet area to search.</param>
        /// <param name="foundRowIndex">The row index at which the match is found.</param>
        /// <param name="foundColumnIndex">The column index at which the match is found.</param>
        /// <returns>A <see cref="T:SearchFoundFlags" /> enumeration that specifies what is matched.</returns>
        public SearchFoundFlags Search(string searchString, SearchFlags searchFlags, SearchOrder searchOrder, SearchFoundFlags searchTarget, SheetArea sheetArea, out int foundRowIndex, out int foundColumnIndex)
        {
            return Search(searchString, searchFlags, searchOrder, searchTarget, sheetArea, 0, 0, out foundRowIndex, out foundColumnIndex);
        }

        /// <summary>
        /// Searches the text in the cells in the specified sheet area for the specified string with the specified criteria 
        /// and start location, and whether to search notes and tags as well.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="searchFlags">The <see cref="T:SearchFlags" /> enumeration that specifies the search options.</param>
        /// <param name="searchOrder">The <see cref="T:SearchOrder" /> enumeration that specifies whether the search uses column, row  coordinates or row, column coordinates.</param>
        /// <param name="searchTarget">The <see cref="T:SearchFoundFlags" /> enumeration that indicates whether the search includes the content in the cell notes, tags, or text.</param>
        /// <param name="sheetArea">The sheet area to search.</param>
        /// <param name="rowStart">The row index at which to start.</param>
        /// <param name="columnStart">The column index at which to start.</param>
        /// <param name="foundRowIndex">The row index at which the match is found.</param>
        /// <param name="foundColumnIndex">The column index at which the match is found.</param>
        /// <returns>A <see cref="T:SearchFoundFlags" /> enumeration that specifies what is matched.</returns>
        public SearchFoundFlags Search(string searchString, SearchFlags searchFlags, SearchOrder searchOrder, SearchFoundFlags searchTarget, SheetArea sheetArea, int rowStart, int columnStart, out int foundRowIndex, out int foundColumnIndex)
        {
            return Search(searchString, searchFlags, searchOrder, searchTarget, sheetArea, rowStart, columnStart, GetRowCount(sheetArea) - 1, GetColumnCount(sheetArea) - 1, out foundRowIndex, out foundColumnIndex);
        }

        /// <summary>
        /// Searches the cell text in the specified sheet area for the specified string with the specified criteria 
        /// and start and end location, and whether to search notes and tags as well.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="searchFlags">The <see cref="T:SearchFlags" /> enumeration that specifies the options of the search.</param>
        /// <param name="searchOrder">The <see cref="T:SearchOrder" /> enumeration that specifies whether the search goes by column then row coordinates or row then column coordinates.</param>
        /// <param name="searchTarget">The <see cref="T:SearchFoundFlags" /> enumeration that indicates whether the search includes the content in the cell notes, tags, or text.</param>
        /// <param name="sheetArea">The sheet area to search.</param>
        /// <param name="rowStart">The row index at which to start.</param>
        /// <param name="columnStart">The column index at which to start.</param>
        /// <param name="rowEnd">The row index at which to end.</param>
        /// <param name="columnEnd">The column index at which to end.</param>
        /// <param name="foundRowIndex">The row index at which a match is found.</param>
        /// <param name="foundColumnIndex">The column index at which a match is found.</param>
        /// <returns>A <see cref="T:SearchFoundFlags" /> enumeration that specifies what is matched.</returns>
        public SearchFoundFlags Search(string searchString, SearchFlags searchFlags, SearchOrder searchOrder, SearchFoundFlags searchTarget, SheetArea sheetArea, int rowStart, int columnStart, int rowEnd, int columnEnd, out int foundRowIndex, out int foundColumnIndex)
        {
            foundRowIndex = -1;
            foundColumnIndex = -1;
            if ((searchString != null) && (searchTarget != SearchFoundFlags.None))
            {
                SearchFoundFlags none = SearchFoundFlags.None;
                CellsEnumerator enumerator = new CellsEnumerator(this, sheetArea, searchOrder, rowStart, columnStart, rowEnd, columnEnd)
                {
                    Options = EnumeratorOption.HasValue
                };
                if ((searchFlags & SearchFlags.BlockRange) > ((SearchFlags)0))
                {
                    enumerator.IsBlockRange = true;
                }
                while (enumerator.MoveNext())
                {
                    int currentRowIndex = enumerator.CurrentRowIndex;
                    int currentColumnIndex = enumerator.CurrentColumnIndex;
                    Cell current = enumerator.Current;
                    if ((searchTarget & SearchFoundFlags.CellText) > SearchFoundFlags.None)
                    {
                        string text = current.Text;
                        if (TrySearch(text, searchString, searchFlags) && (text != string.Empty))
                        {
                            none |= SearchFoundFlags.CellText;
                        }
                    }
                    if ((searchTarget & SearchFoundFlags.CellTag) > SearchFoundFlags.None)
                    {
                        string source = (current.Tag is string) ? current.Tag.ToString() : null;
                        if (TrySearch(source, searchString, searchFlags))
                        {
                            none |= SearchFoundFlags.CellTag;
                        }
                    }
                    if ((searchTarget & SearchFoundFlags.CellFormula) > SearchFoundFlags.None)
                    {
                        string str3 = (current.Formula != null) ? current.Formula.ToString() : null;
                        if (TrySearch(str3, searchString, searchFlags) && (str3 != string.Empty))
                        {
                            none |= SearchFoundFlags.CellFormula;
                        }
                    }
                    if (none != SearchFoundFlags.None)
                    {
                        foundRowIndex = currentRowIndex;
                        foundColumnIndex = currentColumnIndex;
                        return none;
                    }
                }
            }
            return SearchFoundFlags.None;
        }

        /// <summary>
        /// Searches the text in the cells in the specified sheet for the specified string with the specified criteria
        /// and start and end location, and whether to search notes and tags as well.
        /// </summary>
        /// <param name="searchString">The string for which to search.</param>
        /// <param name="searchFlags">The <see cref="T:SearchFlags" /> enumeration that specifies the options of the search.</param>
        /// <param name="searchOrder">The <see cref="T:SearchFlags" /> enumeration that specifies whether the search goes by column, row coordinates or row, column coordinates.</param>
        /// <param name="searchTarget">The <see cref="T:SearchFoundFlags" /> enumeration that indicates whether the search includes the content in the cell notes, tag, or text.</param>
        /// <param name="sheetArea">The area of the sheet to search.</param>
        /// <param name="rowStart">The starting index of the row at which to start.</param>
        /// <param name="columnStart">The starting index of the column at which to start.</param>
        /// <param name="rowEnd">The ending index of the row at which to end.</param>
        /// <param name="columnEnd">The ending index of the column at which to end.</param>
        /// <param name="foundRowIndex">The row index at which a match is found.</param>
        /// <param name="foundColumnIndex">The column index at which a match is found.</param>
        /// <param name="foundString">The found string.</param>
        /// <returns>
        /// A <see cref="T:SearchFoundFlags" /> enumeration that specifies what is matched.
        /// </returns>
        public SearchFoundFlags Search(string searchString, SearchFlags searchFlags, SearchOrder searchOrder, SearchFoundFlags searchTarget, SheetArea sheetArea, int rowStart, int columnStart, int rowEnd, int columnEnd, out int foundRowIndex, out int foundColumnIndex, out string foundString)
        {
            foundString = null;
            SearchFoundFlags flags = Search(searchString, searchFlags, searchOrder, searchTarget, sheetArea, rowStart, columnStart, rowEnd, columnEnd, out foundRowIndex, out foundColumnIndex);
            object text = null;
            if ((flags & SearchFoundFlags.CellText) == SearchFoundFlags.CellText)
            {
                text = FindCellsArea(sheetArea).Cells[foundRowIndex, foundColumnIndex].Text;
            }
            else if ((flags & SearchFoundFlags.CellTag) == SearchFoundFlags.CellTag)
            {
                text = FindCellsArea(sheetArea).Cells[foundRowIndex, foundColumnIndex].Tag;
            }
            else if (((flags & SearchFoundFlags.CellFormula) == SearchFoundFlags.CellFormula) && (sheetArea == SheetArea.Cells))
            {
                text = FindCellsArea(sheetArea).Cells[foundRowIndex, foundColumnIndex].Formula;
            }
            if (text is string)
            {
                foundString = text.ToString();
            }
            return flags;
        }

        void selectionModel_Changed(object sender, SheetSelectionChangedEventArgs e)
        {
            if (SelectionChanged != null)
            {
                SelectionChanged(this, e);
            }
        }

        /// <summary>
        /// Sets the active cell on this sheet.
        /// </summary>
        /// <param name="row">The row index of the cell.</param>
        /// <param name="column">The column index of the cell.</param>
        public void SetActiveCell(int row, int column)
        {
            SetActiveCell(row, column, true);
        }

        /// <summary>
        /// Sets the active cell on this sheet.
        /// </summary>
        /// <param name="row">The row index of the cell.</param>
        /// <param name="column">The column index of the cell.</param>
        /// <param name="clearSelection">Whether to clear selections.</param>
        public void SetActiveCell(int row, int column, bool clearSelection)
        {
            if ((ColumnCount != 0) && (RowCount != 0))
            {
                row = Math.Max(0, Math.Min(row, RowCount - 1));
                column = Math.Max(0, Math.Min(column, ColumnCount - 1));
                if ((row != ActiveRowIndex) || (column != ActiveColumnIndex))
                {
                    if (clearSelection)
                    {
                        ClearSelections();
                    }
                    _selectionModel.SetAnchor(row, column);
                    UpdateActiveViewportByAnchor();
                    RaisePropertyChanged("ActiveCell");
                }
            }
        }

        /// <summary>
        /// Sets the active viewport.
        /// </summary>
        /// <param name="rowViewportIndex">The index of the row viewport.</param>
        /// <param name="columnViewportIndex">The index of the column viewport.</param>
        public void SetActiveViewport(int rowViewportIndex, int columnViewportIndex)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            viewportInfo.ActiveColumnViewport = columnViewportIndex;
            viewportInfo.ActiveRowViewport = rowViewportIndex;
            SetViewportInfo(viewportInfo);
        }

        /// <summary>
        /// Sets the values in the specified two-dimensional array of objects 
        /// into the specified range of cells on this sheet.
        /// </summary>
        /// <param name="row">Row index</param>
        /// <param name="column">Column index</param>
        /// <param name="value">Array from which to set values</param>
        public void SetArray(int row, int column, object[,] value)
        {
            SetArray(row, column, value, false);
        }

        /// <summary>
        /// Sets the values in the specified two-dimensional array of objects 
        /// into the specified range of cells on this sheet.
        /// </summary>
        /// <param name="row">Row index</param>
        /// <param name="column">Column index</param>
        /// <param name="value">Array from which to set values</param>
        /// <param name="formula">If true, set formulas; otherwise, set values.</param>
        public void SetArray(int row, int column, object[,] value, bool formula)
        {
            if ((((value != null) && (0 <= row)) && ((row < RowCount) && (0 <= column))) && (column < ColumnCount))
            {
                int lowerBound = value.GetLowerBound(0);
                int upperBound = value.GetUpperBound(0);
                int num3 = value.GetLowerBound(1);
                int num4 = value.GetUpperBound(1);
                SuspendCalcService();
                int num5 = row;
                while ((num5 < RowCount) && (lowerBound <= upperBound))
                {
                    num3 = value.GetLowerBound(1);
                    int num6 = column;
                    while ((num6 < ColumnCount) && (num3 <= num4))
                    {
                        if (formula)
                        {
                            string str = (string)(value[lowerBound, num3] as string);
                            SetFormula(num5, num6, str);
                        }
                        else
                        {
                            SetValueInternal(num5, num6, SheetArea.Cells, value[lowerBound, num3], false, false);
                        }
                        num6++;
                        num3++;
                    }
                    num5++;
                    lowerBound++;
                }
                ResumeCalcService();
                int rowCount = Math.Min(RowCount - row, value.GetLength(0));
                int columnCount = Math.Min(ColumnCount - column, value.GetLength(1));
                RaiseCellChanged("Value", row, column, rowCount, columnCount, SheetArea.Cells);
            }
        }

        /// <summary>
        /// Sets an array formula in a specified cell area in the specified sheet area.
        /// </summary>
        /// <param name="row">The start row index.</param>
        /// <param name="column">The start column index.</param>
        /// <param name="rowCount">The number of rows in all areas.</param>
        /// <param name="columnCount">The number of columns in all areas.</param>
        /// <param name="formula">The formula to place in the specified cell.</param>
        public void SetArrayFormula(int row, int column, int rowCount, int columnCount, string formula)
        {
            SetFormula(row, column, rowCount, columnCount, SheetArea.Cells, formula, true);
        }

        static void SetAxisInfo(List<AxisInfo> infos, SparseArray<AxisInfo> destAxis, int toIndex)
        {
            if (destAxis != null)
            {
                for (int i = 0; i < infos.Count; i++)
                {
                    destAxis[toIndex + i] = infos[i];
                }
            }
        }

        static void SetAxisStyle(List<object> styles, SparseArray<object> destAxis, int toIndex)
        {
            if (destAxis != null)
            {
                for (int i = 0; i < styles.Count; i++)
                {
                    destAxis[toIndex + i] = styles[i];
                }
            }
        }

        /// <summary>
        /// Sets the border for the specified area.
        /// </summary>
        /// <param name="cellRange">The cell area.</param>
        /// <param name="borderLine">The border line.</param>
        /// <param name="option">Determines which part of the cell range to set.</param>
        public void SetBorder(CellRange cellRange, BorderLine borderLine, SetBorderOptions option)
        {
            SetBorder(cellRange, SheetArea.Cells, borderLine, option);
        }

        /// <summary>
        /// Sets the border for the specified area in the specified sheet area.
        /// </summary>
        /// <param name="cellRange">The cell area.</param>
        /// <param name="borderLine">The border line.</param>
        /// <param name="sheetArea"> The sheet area.</param>
        /// <param name="option">Determines which part of the cell range to set.</param>
        public void SetBorder(CellRange cellRange, SheetArea sheetArea, BorderLine borderLine, SetBorderOptions option)
        {
            if (cellRange == null)
            {
                throw new ArgumentNullException("cellRange");
            }
            if (borderLine == null)
            {
                throw new ArgumentNullException("borderLine");
            }
            if (!Enum.IsDefined((Type)typeof(SheetArea), sheetArea))
            {
                throw new ArgumentException(string.Format(ResourceStrings.InvalidSheetArea, (object[])new object[] { sheetArea }));
            }
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            StorageBlock storage = GetStorage(sheetArea);
            if ((cellRange.Row < -1) || (cellRange.Row >= storage.RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvaildRowIndexWithAllowedRangeBehind, (object[])new object[] { ((int)cellRange.Row), ((int)(storage.RowCount - 1)) }));
            }
            if ((cellRange.Column < -1) || (cellRange.Column >= storage.ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidColumnCountWithAllowedColumnCountBehind, (object[])new object[] { ((int)cellRange.Column), ((int)(storage.ColumnCount - 1)) }));
            }
            if ((cellRange.Row > -1) && (cellRange.Column > -1))
            {
                BorderSettingImp.SetCellsBorder(cellRange, sheetArea, this, option, borderLine);
            }
            else if (cellRange.Row > -1)
            {
                BorderSettingImp.SetRowsBorder(cellRange, sheetArea, this, option, borderLine);
            }
            else if (cellRange.Column > -1)
            {
                BorderSettingImp.SetColumnsBorder(cellRange, sheetArea, this, option, borderLine);
            }
            else
            {
                StyleInfo defaultStyle = DefaultStyle;
                if (sheetArea == SheetArea.ColumnHeader)
                {
                    defaultStyle = ColumnHeader.DefaultStyle;
                }
                if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    defaultStyle = RowHeader.DefaultStyle;
                }
                BorderSettingImp.SetSheetBorder(sheetArea, this, option, borderLine, sheetArea, defaultStyle);
            }
        }

        internal void SetColumnAxisStyleInternal(int column, SheetArea sheetArea, AxisInfo axisInfo)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == SheetArea.ColumnHeader))
            {
                _viewportColumns[column] = axisInfo;
            }
            else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                _rowHeaderColumns[column] = axisInfo;
            }
        }

        internal void SetColumnCount(SheetArea sheetArea, int value)
        {
            if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
            {
                RowHeaderColumnCount = value;
            }
            else
            {
                ColumnCount = value;
            }
        }

        /// <summary>
        /// Sets the value for a specified cell in a column header on this sheet.
        /// </summary>
        /// <param name="row">The column header row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="value">Text for the label.</param>
        public void SetColumnLabel(int row, int column, string value)
        {
            StorageBlock storage = GetStorage(SheetArea.ColumnHeader);
            if (storage != null)
            {
                int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, SheetArea.ColumnHeader);
                if (row == -1)
                {
                    if (ColumnHeaderAutoTextIndex > -1)
                    {
                        row = ColumnHeaderAutoTextIndex;
                    }
                    else
                    {
                        row = GetRowCount(SheetArea.ColumnHeader) - 1;
                    }
                }
                storage.SetValue(row, modelColumnFromViewColumn, ((value != null) && (value.Length > 0)) ? value : null);
                RaiseCellChanged("Label", row, column, SheetArea.CornerHeader | SheetArea.RowHeader);
            }
        }

        /// <summary>
        /// Sets whether a forced page break is inserted before the specified column when printing.
        /// </summary>
        /// <param name="column">The column index (use -1 for all columns).</param>
        /// <param name="value">Set to <c>true</c> for the column to have a forced page break when printing.</param>
        internal void SetColumnPageBreak(int column, bool value)
        {
        }

        /// <summary>
        /// Sets whether users can resize individual columns in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="value">Set to <c>true</c> to allow users to resize the column.</param>
        public void SetColumnResizable(int column, bool value)
        {
            SetColumnResizable(column, SheetArea.Cells, value);
        }

        /// <summary>
        /// Sets whether users can resize individual columns in the specified sheet area.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="value">Set to <c>true</c> to allow users to resize the column.</param>
        public void SetColumnResizable(int column, SheetArea sheetArea, bool value)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            AxisInfo info = GetColumnsStorage(sheetArea)[modelColumnFromViewColumn];
            if (info != null)
            {
                info.Resizable = value;
            }
            else
            {
                AxisInfo info2 = new AxisInfo
                {
                    Resizable = value
                };
                GetColumnsStorage(sheetArea)[modelColumnFromViewColumn] = info2;
            }
        }

        /// <summary>
        /// Sets whether a column in the specified sheet area is displayed.
        /// </summary>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="value">Set to <c>true</c> to display the column.</param>
        public void SetColumnVisible(int column, SheetArea sheetArea, bool value)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            SparseArray<AxisInfo> columnsStorage = GetColumnsStorage(sheetArea);
            AxisInfo info = columnsStorage[column];
            if ((info == null) || (info.Visible != value))
            {
                if (info == null)
                {
                    info = columnsStorage[column] = new AxisInfo();
                }
                info.Visible = value;
                RaiseColumnChanged("IsVisible", column, 1, sheetArea, SheetChangedEventAction.Updated);
            }
        }

        /// <summary>
        /// Sets the width using the specified unit type for the specified columns in this sheet.
        /// </summary>
        /// <param name="column">The column index of the first column.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="width">The width.</param>
        public void SetColumnWidth(int column, SheetArea sheetArea, double width)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if (width == 0.0)
            {
                SetColumnVisible(column, sheetArea, false);
            }
            else if (column < 0)
            {
                if (width < 0.0)
                {
                    throw new ArgumentException(ResourceStrings.CouldnotSetNegetiveColumnWidth);
                }
                if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    _defaultRowHeaderColumnWidth = width;
                }
                else
                {
                    _defaultColumnWidth = width;
                }
                SetColumnVisible(column, sheetArea, true);
            }
            else
            {
                double num = width;
                int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
                AxisInfo info = GetColumnsStorage(sheetArea)[modelColumnFromViewColumn];
                if (num < 0.0)
                {
                    num = -1.0;
                }
                if (info == null)
                {
                    info = new AxisInfo();
                    GetColumnsStorage(sheetArea)[modelColumnFromViewColumn] = info;
                }
                if (num == -1.0)
                {
                    info.ResetSize();
                }
                else
                {
                    info.Size = num;
                }
                SetColumnVisible(column, sheetArea, true);
            }
            RaiseColumnChanged("Width", column, 1, sheetArea, SheetChangedEventAction.Updated);
        }

        /// <summary>
        /// Sets CSV text in the sheet.
        /// </summary>
        /// <param name="row">The start row.</param>
        /// <param name="column">The start column.</param>
        /// <param name="text">The CSV text.</param>
        /// <param name="rowDelimiter">The row delimiter.</param>
        /// <param name="columnDelimiter">The column delimiter.</param>
        /// <param name="cellDelimiter">The cell delimiter.</param>
        public void SetCsv(int row, int column, string text, string rowDelimiter, string columnDelimiter, string cellDelimiter)
        {
            SetCsv(row, column, text, rowDelimiter, columnDelimiter, cellDelimiter, TextFileOpenFlags.None);
        }

        /// <summary>
        /// Sets CSV text in the sheet.
        /// </summary>
        /// <param name="row">The start row.</param>
        /// <param name="column">The start column.</param>
        /// <param name="text">The CSV text.</param>
        /// <param name="rowDelimiter">The row delimiter.</param>
        /// <param name="columnDelimiter">The column delimiter.</param>
        /// <param name="cellDelimiter">The cell delimiter.</param>
        /// <param name="flags">The import flags.</param>
        public void SetCsv(int row, int column, string text, string rowDelimiter, string columnDelimiter, string cellDelimiter, TextFileOpenFlags flags)
        {
            CsvImp.SetRangeText(this, row, column, text, rowDelimiter, columnDelimiter, cellDelimiter, flags);
        }

        /// <summary>
        /// Sets a custom name using the specified expression. 
        /// </summary>
        /// <param name="name">The custom name.</param>
        /// <param name="baseRow">The base row.</param>
        /// <param name="baseColumn">The base column.</param>
        /// <param name="expression">The expression to set. </param>
        public void SetCustomName(string name, int baseRow, int baseColumn, CalcExpression expression)
        {
            SetCustomNameInternal(name, baseRow, baseColumn, expression, true);
        }

        /// <summary>
        /// Sets a custom name using the specified formula. 
        /// </summary>
        /// <param name="name">The custom name.</param>
        /// <param name="baseRow">The base row.</param>
        /// <param name="baseColumn">The base column.</param>
        /// <param name="formula">The formula to set.</param>
        /// <remarks>For example, SetCustomName("GAMMA",2,2,"A6*A7")
        /// means to use formula "A6*A7" at 2,2 and call it "GAMMA". When using SetFormula(7,7,"GAMMA"),
        /// the offset is (7-2),(7-2), move A6,A7 by (5,5) then set value to A11 A12. The result is A11*A12.</remarks>
        public void SetCustomName(string name, int baseRow, int baseColumn, string formula)
        {
            CalcExpression expression = ((ICalcEvaluator)this).Formula2Expression(formula, baseRow, baseColumn) as CalcExpression;
            SetCustomName(name, baseRow, baseColumn, expression);
        }

        /// <summary>
        /// Sets a custom name for the specified area.
        /// </summary>
        /// <param name="name">The set name.</param>
        /// <param name="row">The start row index.</param>
        /// <param name="column">The start column index.</param>
        /// <param name="rowCount">The number of rows.</param>
        /// <param name="columnCount">The number of columns.</param>
        public void SetCustomName(string name, int row, int column, int rowCount, int columnCount)
        {
            SetCustomNameInternal(name, row, column, rowCount, columnCount, true);
        }

        internal void SetCustomNameInternal(string name, int baseRow, int baseColumn, CalcExpression expression, bool refreshNode)
        {
            Names.Add(new NameInfo(name, baseRow, baseColumn, expression));
            OnCustomNameChanged(name, refreshNode);
            if (AutoRecalculation && (_calcEngineSuspended == 0))
            {
                Recalculate();
            }
        }

        internal void SetCustomNameInternal(string name, int row, int column, int rowCount, int columnCount, bool refreshNode)
        {
            CalcExpression expression = CalcExpressionHelper.CreateExternalRangeExpressionByCount(this, row, column, rowCount, columnCount, false, false, false, false);
            Names.Add(new NameInfo(name, 0, 0, expression));
            OnCustomNameChanged(name, refreshNode);
            if (AutoRecalculation && (_calcEngineSuspended == 0))
            {
                Recalculate();
            }
        }

        /// <summary>
        /// Sets a formula in a specified cell in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="formula">The formula to place in the specified cell.</param>
        public void SetFormula(int row, int column, string formula)
        {
            if ((row < 0) || (row >= RowCount))
            {
                throw new ArgumentOutOfRangeException("row");
            }
            if ((column < 0) || (column >= ColumnCount))
            {
                throw new ArgumentOutOfRangeException("column");
            }
            SetFormula(row, column, 1, 1, SheetArea.Cells, formula, false);
        }

        internal void SetFormula(int row, int column, SheetArea sheetArea, string formula)
        {
            SetFormula(row, column, 1, 1, sheetArea, formula, false);
        }

        /// <summary>
        /// Sets a formula in a specified cell area in the specified sheet area.
        /// </summary>
        /// <param name="row">The start row index.</param>
        /// <param name="column">The start column index.</param>
        /// <param name="rowCount">The number of rows in all areas.</param>
        /// <param name="columnCount">The number of columns in all areas.</param>
        /// <param name="formula">The formula to place in the specified cell.</param>
        public void SetFormula(int row, int column, int rowCount, int columnCount, string formula)
        {
            if ((row < 0) && (column < 0))
            {
                if (row < 0)
                {
                    throw new ArgumentOutOfRangeException("row");
                }
                if (column < 0)
                {
                    throw new ArgumentOutOfRangeException("column");
                }
            }
            if (row >= RowCount)
            {
                throw new ArgumentOutOfRangeException("row");
            }
            if (column >= ColumnCount)
            {
                throw new ArgumentOutOfRangeException("column");
            }
            if ((rowCount < 0) && (columnCount < 0))
            {
                if (rowCount < 0)
                {
                    throw new ArgumentOutOfRangeException("rowCount");
                }
                if (columnCount < 0)
                {
                    throw new ArgumentOutOfRangeException("columnCount");
                }
            }
            SetFormula(row, column, rowCount, columnCount, SheetArea.Cells, formula, false);
        }

        internal void SetFormula(int row, int column, int rowCount, int columnCount, SheetArea sheetArea, string formula, bool arrayFormula)
        {
            try
            {
                _isSettingFormula = true;
                if (_conditionalFormats != null)
                {
                    _conditionalFormats.ClearCache();
                }
                if (sheetArea == SheetArea.Cells)
                {
                    CalcLocalIdentity id;
                    bool flag = false;
                    if (((_tables != null) && (rowCount == 1)) && (columnCount == 1))
                    {
                        SheetTable table = FindTable(row, column);
                        if ((table != null) && !table.IsCopying)
                        {
                            table.SetFooter(row, column, formula, null);
                            flag = table.HeaderIndex == row;
                        }
                    }
                    int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
                    int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
                    if (arrayFormula)
                    {
                        id = CalcExpressionHelper.CreateRangeExpressionByCount(row, column, rowCount, columnCount, false, false, false, false).GetId(0, 0) as CalcLocalIdentity;
                    }
                    else
                    {
                        id = new CalcCellIdentity(modelRowFromViewRow, modelColumnFromViewColumn);
                    }
                    if (string.IsNullOrEmpty(formula))
                    {
                        if (!string.IsNullOrEmpty(CalcManager.GetFormula(id)))
                        {
                            CalcManager.ClearFormula(id);
                            Clear(row, column, rowCount, columnCount, sheetArea, StorageType.Data);
                        }
                    }
                    else if (!flag)
                    {
                        CalcExpressionHelper.SetFormula(CalcManager, id, formula, arrayFormula);
                        if (!arrayFormula)
                        {
                            CalcExpression expression = CalcManager.GetExpression(id);
                            for (int i = 0; i < rowCount; i++)
                            {
                                for (int j = 0; j < columnCount; j++)
                                {
                                    if ((i != 0) || (j != 0))
                                    {
                                        int rowIndex = modelRowFromViewRow + i;
                                        int columnIndex = modelColumnFromViewColumn + j;
                                        CalcExpression expr = expression.Offset(0, 0, false, true);
                                        CalcCellIdentity identity2 = new CalcCellIdentity(rowIndex, columnIndex);
                                        CalcManager.SetExpression(identity2, expr, false);
                                        CalcManager.Invalidate(identity2, false);
                                    }
                                }
                            }
                        }
                        if (_calcEngineSuspended == 0)
                        {
                            CalcManager.Invalidate(id, AutoRecalculation);
                        }
                    }
                }
                else
                {
                    SetValue(row, column, sheetArea, CalcErrors.Reference);
                }
            }
            finally
            {
                _isSettingFormula = false;
            }
        }

        internal void SetRowAxisStyleInternal(int row, SheetArea sheetArea, AxisInfo axisInfo)
        {
            if ((sheetArea == SheetArea.Cells) || (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)))
            {
                _viewportRows[row] = axisInfo;
            }
            if (sheetArea == SheetArea.ColumnHeader)
            {
                _columnHeaderRows[row] = axisInfo;
            }
        }

        internal void SetRowCount(SheetArea sheetArea, int value)
        {
            if (sheetArea == SheetArea.ColumnHeader)
            {
                ColumnHeaderRowCount = value;
            }
            else if ((sheetArea == SheetArea.Cells) || (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader)))
            {
                RowCount = value;
            }
        }

        /// <summary>
        /// Sets the height using the specified unit type for the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="height">The height.</param>
        public void SetRowHeight(int row, SheetArea sheetArea, double height)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if (height == 0.0)
            {
                SetRowVisible(row, sheetArea, false);
            }
            else if (row >= 0)
            {
                double num = height;
                int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
                AxisInfo info = GetRowsStorage(sheetArea)[modelRowFromViewRow];
                if (num < 0.0)
                {
                    num = -1.0;
                }
                if (info == null)
                {
                    info = new AxisInfo();
                    GetRowsStorage(sheetArea)[modelRowFromViewRow] = info;
                }
                if (num == -1.0)
                {
                    info.ResetSize();
                }
                else
                {
                    info.Size = num;
                }
                SetRowVisible(row, sheetArea, true);
            }
            else
            {
                if (height < 0.0)
                {
                    throw new ArgumentException(ResourceStrings.CouldnotSetNegetiveRowHeight);
                }
                switch (sheetArea)
                {
                    case SheetArea.Cells:
                    case (SheetArea.CornerHeader | SheetArea.RowHeader):
                        _defaultRowHeight = height;
                        break;

                    case SheetArea.ColumnHeader:
                        _defaultColumnHeaderRowHeight = height;
                        break;
                }
                SetRowVisible(row, sheetArea, true);
            }
            RaiseRowChanged("Height", row, 1, sheetArea, SheetChangedEventAction.Updated);
        }

        /// <summary>
        /// Sets the value for a specified cell in a row header on this sheet.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The row header column index.</param>
        /// <param name="value">The label text.</param>
        public void SetRowLabel(int row, int column, string value)
        {
            StorageBlock storage = GetStorage(SheetArea.CornerHeader | SheetArea.RowHeader);
            if (storage != null)
            {
                int modelRowFromViewRow = GetModelRowFromViewRow(row, SheetArea.CornerHeader | SheetArea.RowHeader);
                if (column == -1)
                {
                    if (ColumnHeaderAutoTextIndex > -1)
                    {
                        column = ColumnHeaderAutoTextIndex;
                    }
                    else
                    {
                        column = GetRowCount(SheetArea.ColumnHeader) - 1;
                    }
                }
                storage.SetValue(modelRowFromViewRow, column, ((value != null) && (value.Length > 0)) ? value : null);
                RaiseCellChanged("Label", row, column, SheetArea.CornerHeader | SheetArea.RowHeader);
            }
        }

        /// <summary>
        /// Sets whether a forced page break is inserted before the specified row when printing.
        /// </summary>
        /// <param name="row">The row index (use -1 for all rows).</param>
        /// <param name="value">Set to <c>true</c> for the row to have a forced page break when printing.</param>
        internal void SetRowPageBreak(int row, bool value)
        {
        }

        /// <summary>
        /// Sets whether users can resize the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="value">Set to <c>true</c> to let the users resize the specified row.</param>
        public void SetRowResizable(int row, bool value)
        {
            SetRowResizable(row, SheetArea.Cells, value);
        }

        /// <summary>
        /// Sets whether users can resize the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="value">Set to <c>true</c> to let the users resize the specified row.</param>
        public void SetRowResizable(int row, SheetArea sheetArea, bool value)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if (row > -1)
            {
                GetModelRowFromViewRow(row, sheetArea);
                AxisInfo info = GetRowsStorage(sheetArea)[row];
                if (info != null)
                {
                    info.Resizable = value;
                }
                else
                {
                    AxisInfo info2 = new AxisInfo
                    {
                        Resizable = value
                    };
                    GetRowsStorage(sheetArea)[row] = info2;
                }
            }
        }

        /// <summary>
        /// Sets whether the control displays the specified row in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="value">Set to <c>true</c> to display the specified row.</param>
        public void SetRowVisible(int row, SheetArea sheetArea, bool value)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            SparseArray<AxisInfo> rowsStorage = GetRowsStorage(sheetArea);
            AxisInfo info = rowsStorage[row];
            if ((info == null) || (info.Visible != value))
            {
                if (info == null)
                {
                    info = rowsStorage[row] = new AxisInfo();
                }
                info.Visible = value;
                RaiseRowChanged("IsVisible", row, 1, sheetArea, SheetChangedEventAction.Updated);
            }
        }

        /// <summary>
        /// Selects the specified cells.
        /// </summary>
        /// <param name="range">The selected range.</param>
        public void SetSelection(CellRange range)
        {
            SetSelection(range.Row, range.Column, range.RowCount, range.ColumnCount);
        }

        /// <summary>
        /// Selects the specified cells.
        /// </summary>
        /// <param name="row">The row index of the first cell.</param>
        /// <param name="column">The column index of the first cell.</param>
        /// <param name="rowCount">The number of rows in the selection.</param>
        /// <param name="columnCount">The number of columns in the selection.</param>
        public void SetSelection(int row, int column, int rowCount, int columnCount)
        {
            int num = row;
            int num2 = column;
            int num3 = rowCount;
            int num4 = columnCount;
            if ((num != -1) && (num2 != -1))
            {
                List<CellRange> spans = GetSpans();
                if ((spans != null) && (spans.Count > 0))
                {
                    CellRange range = CellRangeInflate(spans, new CellRange(row, column, rowCount, columnCount));
                    num = range.Row;
                    num2 = range.Column;
                    num3 = range.RowCount;
                    num4 = range.ColumnCount;
                }
            }
            Selection.SetSelection(num, num2, num3, num4);
        }

        /// <summary>
        /// Sets the sparkline to a cell.
        /// </summary>
        /// <param name="row">The row.</param>
        /// <param name="column">The column.</param>
        /// <param name="dataRange">The data range.</param>
        /// <param name="dataOrientation">The data orientation.</param>
        /// <param name="type">The sparkline type.</param>
        /// <param name="setting">The sparkline setting.</param>
        /// <returns>The sparkline.</returns>
        public Sparkline SetSparkline(int row, int column, CellRange dataRange, DataOrientation dataOrientation, SparklineType type, SparklineSetting setting)
        {
            CalcRangeExpression dataReference = CalcExpressionHelper.CreateRangeExpressionByCount(dataRange.Row, dataRange.Column, dataRange.RowCount, dataRange.ColumnCount, true, true, true, true);
            Sparkline sparkline = new Sparkline(row, column, dataReference, dataOrientation, type, setting);
            InternalRemoveSparkline(row, column);
            GetStorage(SheetArea.Cells).SetSparkline(row, column, sparkline);
            SheetSparklineGroupManager.Add(sparkline.Group);
            RaiseCellChanged("Sparkline", row, column, SheetArea.Cells);
            return sparkline;
        }

        /// <summary>
        /// Sets the sparkline.
        /// </summary>
        /// <param name="row">The row.</param>
        /// <param name="column">The column.</param>
        /// <param name="dataRange">The data range.</param>
        /// <param name="dataOrientation">The data orientation.</param>
        /// <param name="type">The sparkline type.</param>
        /// <param name="dateAxisRange">The date axis range.</param>
        /// <param name="dateAxisRangeOrientation">The date axis range orientation.</param>
        /// <param name="setting">The sparkline setting.</param>
        /// <returns>The sparkline.</returns>
        public Sparkline SetSparkline(int row, int column, CellRange dataRange, DataOrientation dataOrientation, SparklineType type, CellRange dateAxisRange, DataOrientation dateAxisRangeOrientation, SparklineSetting setting)
        {
            CalcRangeExpression dataReference = CalcExpressionHelper.CreateRangeExpressionByCount(dataRange.Row, dataRange.Column, dataRange.RowCount, dataRange.ColumnCount, true, true, true, true);
            CalcRangeExpression dateAxisReference = CalcExpressionHelper.CreateRangeExpressionByCount(dateAxisRange.Row, dateAxisRange.Column, dateAxisRange.RowCount, dateAxisRange.ColumnCount, true, true, true, true);
            Sparkline sparkline = new Sparkline(row, column, dataReference, dataOrientation, type, dateAxisReference, dateAxisRangeOrientation, setting);
            InternalRemoveSparkline(row, column);
            GetStorage(SheetArea.Cells).SetSparkline(row, column, sparkline);
            SheetSparklineGroupManager.Add(sparkline.Group);
            RaiseCellChanged("Sparkline", row, column, SheetArea.Cells);
            return sparkline;
        }

        /// <summary>
        /// Sets the style for the specified cell, row, or column.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="style">The style.</param>
        public void SetStyleInfo(int row, int column, StyleInfo style)
        {
            SetStyleInfo(row, column, SheetArea.Cells, style);
        }

        /// <summary>
        /// Sets the style for the specified cell, row, or column of the specified area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="style">The style.</param>
        /// <exception cref="T:System.ArgumentException">The sheet area cannot be found. </exception>
        /// <exception cref="T:System.IndexOutOfRangeException"> The row index must be between 0 and the block row count.</exception>
        /// <exception cref="T:System.IndexOutOfRangeException"> The column index must be between 0 and the block column count . </exception>
        public void SetStyleInfo(int row, int column, SheetArea sheetArea, StyleInfo style)
        {
            StorageBlock storage = GetStorage(sheetArea);
            if (storage == null)
            {
                throw new ArgumentException(ResourceStrings.CouldnotFindSpecifiedSheetArea);
            }
            if ((row < -1) || (row >= storage.RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvaildRowIndexWithAllowedRangeBehind, (object[])new object[] { ((int)row), ((int)(storage.RowCount - 1)) }));
            }
            if ((column < -1) || (column >= storage.ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidColumnCountWithAllowedColumnCountBehind, (object[])new object[] { ((int)column), ((int)(storage.ColumnCount - 1)) }));
            }
            SetStyleObject(row, column, sheetArea, style);
        }

        /// <summary>
        /// Sets the style for the specified cell, column, row,
        /// or default model to the specified style name.
        /// </summary>
        /// <param name="row">The row index of the cell in the model, or -1 to specify a column or default model.</param>
        /// <param name="column">The column index of the cell in the model, or -1 to specify a row or default model.</param>
        /// <param name="value">The name of the style to set.</param>
        public void SetStyleName(int row, int column, string value)
        {
            SetStyleName(row, column, SheetArea.Cells, value);
        }

        /// <summary>
        /// Sets the style for the specified cell, column, row,
        /// or default model to the specified style name.
        /// </summary>
        /// <param name="row">The row index of the cell in the model, or -1 to specify a column or default model.</param>
        /// <param name="column">The column index of the cell in the model, or -1 to specify a row or default model.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="value">The name of the style to set.</param>
        public void SetStyleName(int row, int column, SheetArea sheetArea, string value)
        {
            StorageBlock storage = GetStorage(sheetArea);
            if (storage == null)
            {
                throw new ArgumentException(ResourceStrings.CouldnotFindSpecifiedSheetArea);
            }
            if ((row < -1) || (row >= storage.RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvaildRowIndexWithAllowedRangeBehind, (object[])new object[] { ((int)row), ((int)(storage.RowCount - 1)) }));
            }
            if ((column < -1) || (column >= storage.ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidColumnCountWithAllowedColumnCountBehind, (object[])new object[] { ((int)column), ((int)(storage.ColumnCount - 1)) }));
            }
            SetStyleObject(row, column, sheetArea, value);
        }

        internal void SetStyleObject(int row, int column, SheetArea sheetArea, object style)
        {
            if ((row >= 0) && (column >= 0))
            {
                StorageBlock storage = GetStorage(sheetArea);
                if (storage != null)
                {
                    storage.SetStyle(row, column, (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : style);
                    RaiseCellChanged("[StyleInfo]", row, column, sheetArea);
                }
            }
            else if (column >= 0)
            {
                if (sheetArea == SheetArea.Cells)
                {
                    _viewportColumnsStyles[column] = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : style;
                }
                else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    _rowHeaderColumnsStyles[column] = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : style;
                }
                else if (sheetArea == SheetArea.ColumnHeader)
                {
                    _columnHeaderColumnsStyles[column] = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : style;
                }
                RaiseCellChanged("[StyleInfo]", row, column, sheetArea);
            }
            else if (row >= 0)
            {
                if (sheetArea == SheetArea.Cells)
                {
                    _viewportRowsStyles[row] = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : style;
                }
                else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    _rowHeaderRowsStyles[row] = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : style;
                }
                else if (sheetArea == SheetArea.ColumnHeader)
                {
                    _columnHeaderRowsStyles[row] = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : style;
                }
                RaiseCellChanged("[StyleInfo]", row, column, sheetArea);
            }
            else
            {
                if ((style != null) && !(style is StyleInfo))
                {
                    throw new NotSupportedException(ResourceStrings.DonotAllowSetTheNameOfTheDefaultStyle);
                }
                if (sheetArea == SheetArea.Cells)
                {
                    if (_defaultStyle != null)
                    {
                        _defaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                    }
                    _defaultStyle = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : null;
                    if (_defaultStyle != null)
                    {
                        _defaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                    }
                }
                else if (sheetArea == (SheetArea.CornerHeader | SheetArea.RowHeader))
                {
                    if (_rowHeaderDefaultStyle != null)
                    {
                        _rowHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                    }
                    _rowHeaderDefaultStyle = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : null;
                    if (_rowHeaderDefaultStyle != null)
                    {
                        _rowHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                    }
                }
                else if (sheetArea == SheetArea.ColumnHeader)
                {
                    if (_columnHeaderDefaultStyle != null)
                    {
                        _columnHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                    }
                    _columnHeaderDefaultStyle = (style is StyleInfo) ? new StyleInfo(style as StyleInfo) : null;
                    if (_columnHeaderDefaultStyle != null)
                    {
                        _columnHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                    }
                }
                RaiseCellChanged("[StyleInfo]", row, column, sheetArea);
            }
        }

        /// <summary>
        /// Sets the application-defined tag value for the specified cell, column, row, or entire sheet for the specified sheet area.
        /// </summary>
        /// <param name="row">The row index (set to -1 for all rows).</param>
        /// <param name="column">The column index (set to -1 for all columns).</param>
        /// <param name="value">The tag value for the specified cell, column, row, or sheet.</param>
        public void SetTag(int row, int column, object value)
        {
            SetTag(row, column, SheetArea.Cells, value);
        }

        /// <summary>
        /// Sets the application-defined tag value for the specified cell, column, row, or entire sheet for the specified sheet area.
        /// </summary>
        /// <param name="row">The row index (set to -1 for all rows).</param>
        /// <param name="column">The column index (set to -1 for all columns).</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="value">The tag value for the specified cell, column, row, or sheet.</param>
        public void SetTag(int row, int column, SheetArea sheetArea, object value)
        {
            if (sheetArea == SheetArea.CornerHeader)
            {
                throw new NotSupportedException(ResourceStrings.CouldnotChangeConnerHeaderCell);
            }
            if ((row >= 0) && (column >= 0))
            {
                StorageBlock storage = GetStorage(sheetArea);
                if (storage != null)
                {
                    storage.SetTag(row, column, value);
                }
            }
            else if (column >= 0)
            {
                AxisInfo info = GetColumnsStorage(sheetArea)[column];
                if (info == null)
                {
                    AxisInfo info2 = new AxisInfo
                    {
                        Tag = value
                    };
                    GetColumnsStorage(sheetArea)[column] = info2;
                }
                else
                {
                    info.Tag = value;
                }
            }
            else
            {
                if (row < 0)
                {
                    throw new NotSupportedException(ResourceStrings.SetTagToNonExistingCell);
                }
                AxisInfo info3 = GetRowsStorage(sheetArea)[row];
                if (info3 == null)
                {
                    AxisInfo info4 = new AxisInfo
                    {
                        Tag = value
                    };
                    GetRowsStorage(sheetArea)[row] = info4;
                }
                else
                {
                    info3.Tag = value;
                }
            }
        }

        /// <summary>
        /// Sets the formatted text in the cell in the specified sheet area
        /// using the <see cref="T:IFormatter" /> object for the cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="value">The text for the specified cell.</param>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified row index is out of range; must be between 0 and the total number of rows.
        /// </exception>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified column index is out of range; must be between 0 and the total number of columns.
        /// </exception>
        public void SetText(int row, int column, string value)
        {
            SetText(row, column, SheetArea.Cells, value);
        }

        /// <summary>
        /// Sets the formatted text in the cell in the specified sheet area
        /// using the <see cref="T:IFormatter" /> object for the cell.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="sheetArea">The sheet area.</param>
        /// <param name="text">The text for the specified cell.</param>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified row index is out of range; must be between 0 and the total number of rows.
        /// </exception>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified column index is out of range; must be between 0 and the total number of columns.
        /// </exception>
        public void SetText(int row, int column, SheetArea sheetArea, string text)
        {
            StorageBlock storage = GetStorage(sheetArea);
            if ((row < -1) || (row >= storage.RowCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidRowIndex0MustBeBetween1And1, (object[])new object[] { ((int)row), ((int)(storage.RowCount - 1)) }));
            }
            if ((column < -1) || (column >= storage.ColumnCount))
            {
                throw new IndexOutOfRangeException(string.Format(ResourceStrings.InvalidColumnIndexWithAllowedRangeBehind, (object[])new object[] { ((int)column), ((int)(storage.ColumnCount - 1)) }));
            }
            int modelRowFromViewRow = GetModelRowFromViewRow(row, sheetArea);
            int modelColumnFromViewColumn = GetModelColumnFromViewColumn(column, sheetArea);
            StyleInfo info = GetCompositeStyle(modelRowFromViewRow, modelColumnFromViewColumn, sheetArea, storage);
            object obj2 = Text2Value(text, info.Formatter);
            SetValue(row, column, sheetArea, obj2);
        }

        /// <summary>
        /// Sets the value for the specified cell in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="value">The value to set for the specified cell.</param>
        public void SetValue(int row, int column, object value)
        {
            SetValue(row, column, SheetArea.Cells, value);
        }

        /// <summary>
        /// Sets the value for the specified cell in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="area">The sheet area.</param>
        /// <param name="value">The value to set for the specified cell.</param>
        public void SetValue(int row, int column, SheetArea area, object value)
        {
            SetValue(row, column, area, value, true);
        }

        /// <summary>
        /// Sets the value for the specified cell in the specified sheet area.
        /// </summary>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <param name="area">The sheet area.</param>        
        /// <param name="value">The value to set for the specified cell.</param>
        /// <param name="raiseCellPropertyChanged">A CellPropertyChanged event occurs if this value is true.</param>
        public void SetValue(int row, int column, SheetArea area, object value, bool raiseCellPropertyChanged)
        {
            if (_conditionalFormats != null)
            {
                _conditionalFormats.ClearCache();
            }
            SetValueInternal(row, column, area, value, raiseCellPropertyChanged, false);
            if (_calcEngineSuspended == 0)
            {
                CalcCalculationManager calcManager = CalcManager;
                if (!calcManager.Graph.IsEmpty)
                {
                    calcManager.Invalidate(new CalcCellIdentity(row, column), AutoRecalculation);
                }
            }
        }

        internal void SetValueInternal(int row, int column, SheetArea area, object value, bool raiseCellPropertyChanged, bool fromCalSource = false)
        {
            bool flag = false;
            if ((!_isSettingFormula && (_tables != null)) && (area == SheetArea.Cells))
            {
                SheetTable table = FindTable(row, column);
                if ((table != null) && !table.IsCopying)
                {
                    flag |= table.SetHeader(row, column, value);
                    if (!fromCalSource)
                    {
                        flag |= table.SetFooter(row, column, null, value);
                    }
                }
            }
            if (((_bindingManager != null) && (area == SheetArea.Cells)) && (_bindingManager.IsBoundRow(row) && _bindingManager.IsBoundColumn(column)))
            {
                _bindingManager.SetValue(row, column, value);
            }
            else if (!flag)
            {
                GetStorage(area).SetValue(row, column, value);
            }
            if ((raiseCellPropertyChanged && !IsEventSuspend()) && ((_workbook == null) || !_workbook.IsCalculating))
            {
                RaiseCellChanged("Value", row, column, area);
            }
        }

        /// <summary>
        /// Sets the height of the viewport.
        /// </summary>
        /// <param name="rowViewportIndex">The index of the row viewport.</param>
        /// <param name="value">The height value.</param>
        public void SetViewportHeight(int rowViewportIndex, int value)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if ((rowViewportIndex < 0) && (rowViewportIndex >= viewportInfo.ViewportHeight.Length))
            {
                throw new ArgumentException("rowViewportIndex");
            }
            viewportInfo.ViewportHeight[rowViewportIndex] = value;
            SetViewportInfo(viewportInfo);
        }

        /// <summary>
        /// Sets the viewport information to the current sheet.
        /// </summary>
        /// <param name="value">The value.</param>
        public void SetViewportInfo(ViewportInfo value)
        {
            if (value != null)
            {
                if (_viewportInfo != value)
                {
                    _viewportInfo = value;
                }
                RaisePropertyChanged("[ViewportInfo]");
            }
        }

        /// <summary>
        /// Sets the width of the viewport.
        /// </summary>
        /// <param name="columnViewportIndex">The index of the column viewport.</param>
        /// <param name="value">The width value.</param>
        public void SetViewportWidth(int columnViewportIndex, int value)
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            if ((columnViewportIndex < 0) || (columnViewportIndex >= viewportInfo.ViewportWidth.Length))
            {
                throw new ArgumentException("column");
            }
            viewportInfo.ViewportWidth[columnViewportIndex] = value;
            SetViewportInfo(viewportInfo);
        }

        /// <summary>
        /// Sorts a range of cells in this sheet in the data model.
        /// </summary>
        /// <param name="row">The index of the starting row of the block of cells to sort.</param>
        /// <param name="column">The index of the starting column of the block of cells to sort.</param>
        /// <param name="rowCount">The number of rows in the block of cells.</param>
        /// <param name="columnCount">The number of columns in the block of cells.</param>
        /// <param name="byRows">Set to <c>true</c> to sort by rows; false to sort by columns.</param>
        /// <param name="sortInfo">The SortInfo object with sort criteria and information about how to perform the sort.</param>
        /// <returns><c>true</c> if the data is successfully sorted; otherwise, <c>false</c>.</returns>
        public bool SortRange(int row, int column, int rowCount, int columnCount, bool byRows, SortInfo[] sortInfo)
        {
            bool flag;
            if (_cellsBlock == null)
            {
                return false;
            }
            try
            {
                CalcExpressionHelper.ValidateIntersectedArrayFormula(CalcManager, row, column, rowCount, columnCount, false);
                if (_workbook != null)
                {
                    _workbook.SuspendEvent();
                }
                SuspendCalcService();
                SuspendSpanModelEvent();
                int num = RowCount;
                int num2 = ColumnCount;
                if (row == -1)
                {
                    row = 0;
                }
                if (rowCount == -1)
                {
                    rowCount = _cellsBlock.RowCount;
                }
                if (column == -1)
                {
                    column = 0;
                }
                if (columnCount == -1)
                {
                    columnCount = _cellsBlock.ColumnCount;
                }
                if ((((row < 0) || (row >= num)) || ((column < 0) || (column >= num2))) || (((rowCount < 0) || ((row + rowCount) > num)) || (((columnCount < 0) || ((column + columnCount) > num2)) || (sortInfo == null))))
                {
                    return false;
                }
                if (_sortImp.RangeContainsSpans(row, column, rowCount, columnCount))
                {
                    return false;
                }
                int[] array = _sortImp.QuickSort(row, column, rowCount, columnCount, byRows, sortInfo, SortingAlgorithmType.QuickSort);
                if (array != null)
                {
                    int num3;
                    int num4;
                    int modelRowFromViewRow;
                    int modelColumnFromViewColumn;
                    object obj2;
                    object obj3;
                    object obj4;
                    if (byRows)
                    {
                        num4 = row;
                        while (num4 < (row + rowCount))
                        {
                            int num9 = GetSwapIndex(array, row, num4, rowCount, true);
                            if (num4 != num9)
                            {
                                modelRowFromViewRow = GetModelRowFromViewRow(num4);
                                int num7 = GetModelRowFromViewRow(num9);
                                for (num3 = column; num3 < (column + columnCount); num3++)
                                {
                                    modelColumnFromViewColumn = GetModelColumnFromViewColumn(num3);
                                    obj3 = GetValue(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells);
                                    obj4 = GetValue(num7, modelColumnFromViewColumn, SheetArea.Cells);
                                    CalcAxial.Swap(modelRowFromViewRow, modelColumnFromViewColumn, num7, modelColumnFromViewColumn, 1, 1);
                                    SetValue(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells, obj4);
                                    SetValue(num7, modelColumnFromViewColumn, SheetArea.Cells, obj3);
                                    obj2 = GetTag(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells);
                                    SetTag(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells, GetTag(num7, modelColumnFromViewColumn, SheetArea.Cells));
                                    SetTag(num7, modelColumnFromViewColumn, SheetArea.Cells, obj2);
                                    obj2 = GetStyleObject(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells);
                                    SetStyleObject(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells, GetStyleObject(num7, modelColumnFromViewColumn, SheetArea.Cells));
                                    SetStyleObject(num7, modelColumnFromViewColumn, SheetArea.Cells, obj2);
                                }
                                RaiseRowsSwapped(modelRowFromViewRow, num7);
                            }
                            num4++;
                        }
                    }
                    else
                    {
                        for (num3 = column; num3 < (column + columnCount); num3++)
                        {
                            int num10 = GetSwapIndex(array, column, num3, columnCount, false);
                            if (num3 != num10)
                            {
                                modelColumnFromViewColumn = GetModelColumnFromViewColumn(num3);
                                int num8 = GetModelColumnFromViewColumn(num10);
                                for (num4 = row; num4 < (row + rowCount); num4++)
                                {
                                    modelRowFromViewRow = GetModelRowFromViewRow(num4);
                                    obj3 = GetValue(modelRowFromViewRow, modelColumnFromViewColumn);
                                    obj4 = GetValue(modelRowFromViewRow, num8);
                                    CalcAxial.Swap(modelRowFromViewRow, modelColumnFromViewColumn, modelRowFromViewRow, num8, 1, 1);
                                    SetValue(modelRowFromViewRow, modelColumnFromViewColumn, obj4);
                                    SetValue(modelRowFromViewRow, num8, obj3);
                                    obj2 = GetTag(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells);
                                    SetTag(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells, GetTag(modelRowFromViewRow, num8, SheetArea.Cells));
                                    SetTag(modelRowFromViewRow, num8, SheetArea.Cells, obj2);
                                    obj2 = GetStyleObject(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells);
                                    SetStyleObject(modelRowFromViewRow, modelColumnFromViewColumn, SheetArea.Cells, GetStyleObject(modelRowFromViewRow, num8, SheetArea.Cells));
                                    SetStyleObject(modelRowFromViewRow, num8, SheetArea.Cells, obj2);
                                }
                                RaiseColumnsSwapped(modelColumnFromViewColumn, num8);
                            }
                        }
                    }
                    return true;
                }
                flag = false;
            }
            finally
            {
                ResumeCalcService();
                ResumeSpanModelEvent();
                if (_workbook != null)
                {
                    _workbook.ResumeEvent();
                }
                for (int i = 0; i < rowCount; i++)
                {
                    for (int j = 0; j < columnCount; j++)
                    {
                        RaiseCellChanged("Value", row + i, column + j, SheetArea.Cells);
                    }
                }
                RaisePropertyChanged("[Sort]");
            }
            return flag;
        }

        /// <summary>
        /// Suspends the calculation service.
        /// </summary>
        public void SuspendCalcService()
        {
            _calcEngineSuspended++;
        }

        /// <summary>
        /// Suspends the chart events.
        /// </summary>
        internal void SuspendChartEvents()
        {
            if (_spreadCharts != null)
            {
                _spreadCharts.SuspendEvent();
                _spreadCharts.SuspendItemsEvent();
            }
        }

        /// <summary>
        /// Suspends the floating object events.
        /// </summary>
        internal void SuspendFloatingObjectEvents()
        {
            if (_floatingObjects != null)
            {
                _floatingObjects.SuspendEvent();
                _floatingObjects.SuspendItemsEvent();
            }
        }

        /// <summary>
        /// Suspends the pictures events.
        /// </summary>
        internal void SuspendPicturesEvents()
        {
            if (_pictures != null)
            {
                _pictures.SuspendEvent();
                _pictures.SuspendItemsEvent();
            }
        }

        /// <summary>
        /// Suspends the SpanModel, RowHeaderSpanModel, and ColumnHeaderSpanModel events.
        /// </summary>
        public void SuspendSpanModelEvent()
        {
            if (SpanModel != null)
            {
                SpanModel.SuspendEvent();
            }
            if (RowHeaderSpanModel != null)
            {
                RowHeaderSpanModel.SuspendEvent();
            }
            if (ColumnHeaderSpanModel != null)
            {
                ColumnHeaderSpanModel.SuspendEvent();
            }
        }

        bool IEqualityComparer<ICalcSource>.Equals(ICalcSource x, ICalcSource y)
        {
            return object.Equals(x, y);
        }

        int IEqualityComparer<ICalcSource>.GetHashCode(ICalcSource obj)
        {
            if (obj == null)
            {
                return 0;
            }
            return obj.GetHashCode();
        }

        XmlSchema IXmlSerializable.GetSchema()
        {
            throw new NotImplementedException();
        }

        void IXmlSerializable.ReadXml(XmlReader reader)
        {
            ReadXmlInternal(reader, true);
        }

        void IXmlSerializable.WriteXml(XmlWriter writer)
        {
            WriteXmlInternal(writer, false, ExcludePrintContent.None);
        }

        internal object Text2Value(string text, IFormatter formatter)
        {
            if ((text != null) && (formatter != null))
            {
                return formatter.Parse(text);
            }
            return text;
        }

        /// <summary>
        /// Tries the search.
        /// </summary>
        /// <param name="source">The source</param>
        /// <param name="searchString">The search string</param>
        /// <param name="searchFlags">The search flags</param>
        /// <returns></returns>
        bool TrySearch(string source, string searchString, SearchFlags searchFlags)
        {
            if (source == null)
            {
                return false;
            }
            bool flag = (searchFlags & SearchFlags.ExactMatch) > ((SearchFlags)0);
            if (flag && ((searchFlags & SearchFlags.UseWildCards) == ((SearchFlags)0)))
            {
                if ((searchFlags & SearchFlags.IgnoreCase) > ((SearchFlags)0))
                {
                    return string.Equals(source, searchString, (StringComparison)StringComparison.CurrentCultureIgnoreCase);
                }
                return string.Equals(source, searchString);
            }
            TextCondition condition = new TextCondition(flag ? TextCompareType.EqualsTo : TextCompareType.Contains, searchString, null)
            {
                IgnoreCase = (searchFlags & SearchFlags.IgnoreCase) > ((SearchFlags)0),
                UseWildCards = (searchFlags & SearchFlags.UseWildCards) > ((SearchFlags)0)
            };
            return condition.Evaluate(this, 0, 0, new ActualValue(source));
        }

        /// <summary>
        /// Ungroups the sparklines in the group.
        /// </summary>
        /// <param name="group">The group.</param>
        public void UngroupSparkline(SparklineGroup group)
        {
            Sparkline[] array = new Sparkline[group.Count];
            group.CopyTo(array, 0);
            foreach (Sparkline sparkline in array)
            {
                if (sparkline != null)
                {
                    group.Remove(sparkline);
                    SparklineGroup group2 = new SparklineGroup(group.SparklineType, group.Setting.Clone())
                    {
                        DateAxisReference = group.DateAxisReference,
                        DateAxisOrientation = group.DateAxisOrientation
                    };
                    group2.Add(sparkline);
                    SheetSparklineGroupManager.Add(group2);
                }
            }
            SheetSparklineGroupManager.Remove(group);
        }

        /// <summary>
        /// Unions two cell range to one cell range.
        /// </summary>
        /// <param name="range1">The first cell range to union.</param>
        /// <param name="range2">The second cell range to union.</param>
        /// <returns>The union cell range</returns>
        internal static CellRange UnionCellRange(CellRange range1, CellRange range2)
        {
            int row = Math.Min(range1.Row, range2.Row);
            int column = Math.Min(range1.Column, range2.Column);
            int num3 = Math.Max((int)((range1.Row + range1.RowCount) - 1), (int)((range2.Row + range2.RowCount) - 1));
            int num4 = Math.Max((int)((range1.Column + range1.ColumnCount) - 1), (int)((range2.Column + range2.ColumnCount) - 1));
            if ((row >= 0) && (column >= 0))
            {
                return new CellRange(row, column, (num3 - row) + 1, (num4 - column) + 1);
            }
            if (row >= 0)
            {
                return new CellRange(row, -1, (num3 - row) + 1, -1);
            }
            if (column >= 0)
            {
                return new CellRange(-1, column, -1, (num4 - column) + 1);
            }
            return new CellRange(-1, -1, -1, -1);
        }

        void UpadteFormulaString()
        {
            ICalcEvaluator evaluator = this;
            if (_cachedDeserializedFormulaStrings != null)
            {
                foreach (List<object> list in _cachedDeserializedFormulaStrings)
                {
                    int rowIndex = (int)((int)list[0]);
                    int columnIndex = (int)((int)list[1]);
                    int bandCount = (int)((int)list[2]);
                    int num4 = (int)((int)list[3]);
                    string str = (string)((string)list[4]);
                    if (!string.IsNullOrEmpty(str))
                    {
                        CalcLocalIdentity identity;
                        if ((bandCount == -1) && (num4 == -1))
                        {
                            identity = new CalcCellIdentity(rowIndex, columnIndex);
                        }
                        else if ((rowIndex == -1) && (bandCount == -1))
                        {
                            identity = new CalcRangeIdentity(columnIndex, num4, false);
                        }
                        else if ((columnIndex == -1) && (num4 == -1))
                        {
                            identity = new CalcRangeIdentity(rowIndex, bandCount, true);
                        }
                        else
                        {
                            identity = new CalcRangeIdentity(rowIndex, columnIndex, bandCount, num4);
                        }
                        CalcExpression exp = evaluator.Formula2Expression(str, rowIndex, columnIndex) as CalcExpression;
                        if (exp != null)
                        {
                            _calcStorage.Add(identity, exp);
                        }
                    }
                }
                _cachedDeserializedFormulaStrings = null;
            }
        }

        internal void UpdateActiveViewportByAnchor()
        {
            ViewportInfo viewportInfo = GetViewportInfo();
            int anchorRow = Selection.AnchorRow;
            int anchorColumn = Selection.AnchorColumn;
            if (viewportInfo.ActiveRowViewport == -1)
            {
                if (anchorRow >= FrozenRowCount)
                {
                    if (((viewportInfo.RowViewportCount == 1) && (FrozenTrailingRowCount > 0)) && (anchorRow >= (RowCount - FrozenTrailingRowCount)))
                    {
                        viewportInfo.ActiveRowViewport = 1;
                    }
                    else
                    {
                        viewportInfo.ActiveRowViewport = 0;
                    }
                }
            }
            else if (viewportInfo.ActiveRowViewport == viewportInfo.RowViewportCount)
            {
                if (anchorRow < (RowCount - FrozenTrailingRowCount))
                {
                    if (((viewportInfo.RowViewportCount == 1) && (FrozenRowCount > 0)) && (anchorRow < FrozenRowCount))
                    {
                        viewportInfo.ActiveRowViewport = -1;
                    }
                    else
                    {
                        viewportInfo.ActiveRowViewport--;
                    }
                }
            }
            else
            {
                if (anchorRow < FrozenRowCount)
                {
                    viewportInfo.ActiveRowViewport = -1;
                }
                if (anchorRow >= (RowCount - FrozenTrailingRowCount))
                {
                    viewportInfo.ActiveRowViewport = viewportInfo.RowViewportCount;
                }
            }
            if (viewportInfo.ActiveColumnViewport == -1)
            {
                if (anchorColumn >= FrozenColumnCount)
                {
                    if (((viewportInfo.ColumnViewportCount == 1) && (FrozenTrailingColumnCount > 0)) && (anchorColumn >= (ColumnCount - FrozenTrailingColumnCount)))
                    {
                        viewportInfo.ActiveColumnViewport = 1;
                    }
                    else
                    {
                        viewportInfo.ActiveColumnViewport = 0;
                    }
                }
            }
            else if (viewportInfo.ActiveColumnViewport == viewportInfo.ColumnViewportCount)
            {
                if (anchorColumn < (ColumnCount - FrozenTrailingColumnCount))
                {
                    if (((viewportInfo.ColumnViewportCount == 1) && (FrozenColumnCount > 0)) && (anchorColumn < FrozenColumnCount))
                    {
                        viewportInfo.ActiveColumnViewport = -1;
                    }
                    else
                    {
                        viewportInfo.ActiveColumnViewport--;
                    }
                }
            }
            else
            {
                if (anchorColumn < FrozenColumnCount)
                {
                    viewportInfo.ActiveColumnViewport = -1;
                }
                if (anchorColumn >= (ColumnCount - FrozenTrailingColumnCount))
                {
                    viewportInfo.ActiveColumnViewport = viewportInfo.ColumnViewportCount;
                }
            }
        }

        internal string Value2Text(object value, IFormatter formatter)
        {
            if (value == null)
            {
                return string.Empty;
            }
            GeneralFormatter formatter2 = formatter as GeneralFormatter;
            if (formatter2 != null)
            {
                if ((value is string) && formatter2.IsDefaultFormat)
                {
                    return (string)(value as string);
                }
                return formatter2.Format(value);
            }
            if (formatter != null)
            {
                return formatter.Format(value);
            }
            if (value is bool)
            {
                return value.ToString().ToUpper();
            }
            return value.ToString();
        }

        internal bool WriteXmlInternal(XmlWriter writer, bool dataOnly, ExcludePrintContent excludeContent)
        {
            Serializer.InitWriter(writer);
            if (!dataOnly && !string.IsNullOrEmpty(_name))
            {
                Serializer.WriteAttr("name", _name, writer);
            }
            if (!dataOnly && ((excludeContent & ExcludePrintContent.Formula) != ExcludePrintContent.Formula))
            {
                if (_referenceStyle != ReferenceStyle.A1)
                {
                    Serializer.SerializeObj(_referenceStyle, "ReferenceStyle", writer);
                }
                if (!_autoRecalculation)
                {
                    Serializer.SerializeObj((bool)_autoRecalculation, "AutoCalculation", writer);
                }
                if ((_names != null) && (_names.Count > 0))
                {
                    Serializer.SerializeNameInfos(_names, "Names", this, writer);
                }
                if (((_calcManager != null) && (_calcManager.FormulaCount > 0)) && (_calcStorage != null))
                {
                    Serializer.SerializeCalcStorage(_calcStorage, "Formula", this, writer);
                }
            }
            if (!dataOnly)
            {
                if (!_visible)
                {
                    Serializer.SerializeObj((bool)_visible, "IsVisible", writer);
                }
                if (!_columnHeaderVisible)
                {
                    Serializer.SerializeObj((bool)_columnHeaderVisible, "ColumnHeaderVisible", writer);
                }
                if (!_rowHeaderVisible)
                {
                    Serializer.SerializeObj((bool)_rowHeaderVisible, "RowHeaderVisible", writer);
                }
                if (_columnFooterVisible)
                {
                    Serializer.SerializeObj((bool)_columnFooterVisible, "ColumnFooterVisible", writer);
                }
                if (_startingColumnNumber != 1)
                {
                    Serializer.SerializeObj((int)_startingColumnNumber, "StartingColumnNumber", writer);
                }
                if (_startingRowNumber != 1)
                {
                    Serializer.SerializeObj((int)_startingRowNumber, "StartingRowNumber", writer);
                }
                if (_columnHeaderAutoText != HeaderAutoText.Letters)
                {
                    Serializer.SerializeObj(_columnHeaderAutoText, "ColumnHeaderAutoText", writer);
                }
                if (_rowHeaderAutoText != HeaderAutoText.Numbers)
                {
                    Serializer.SerializeObj(_rowHeaderAutoText, "RowHeaderAutoText", writer);
                }
                if (_columnHeaderAutoTextIndex != -1)
                {
                    Serializer.SerializeObj((int)_columnHeaderAutoTextIndex, "ColumnHeaderAutoTextIndex", writer);
                }
                if (_rowHeaderAutoTextIndex != -1)
                {
                    Serializer.SerializeObj((int)_rowHeaderAutoTextIndex, "RowHeaderAutoTextIndex", writer);
                }
                if (_defaultColumnFooterRowHeight != _DEFAULT_COLUMN_FOOTER_ROWHEIGHT)
                {
                    Serializer.SerializeObj((double)_defaultColumnFooterRowHeight, "DefaultColumnFooterRowHeight", writer);
                }
                if (_defaultColumnHeaderRowHeight != _DEFAULT_COLUMN_HEADER_ROWHEIGHT)
                {
                    Serializer.SerializeObj((double)_defaultColumnHeaderRowHeight, "DefaultColumnHeaderRowHeight", writer);
                }
                if (_defaultRowHeight != _DEFAULT_ROW_HEIGHT)
                {
                    Serializer.SerializeObj((double)_defaultRowHeight, "DefaultRowHeight", writer);
                }
                if (_defaultRowHeaderColumnWidth != _DEFAULT_ROW_HEADER_COLUMNWIDTH)
                {
                    Serializer.SerializeObj((double)_defaultRowHeaderColumnWidth, "DefaultRowHeaderColumnWidth", writer);
                }
                if (_defaultColumnWidth != _DEFAULT_COLUMN_WIDTH)
                {
                    Serializer.SerializeObj((double)_defaultColumnWidth, "DefaultColumnWidth", writer);
                }
                if (_zoomFactor != 1f)
                {
                    Serializer.SerializeObj((float)_zoomFactor, "ZoomFactor", writer);
                }
                if (!_autoGeneratedColumns)
                {
                    Serializer.SerializeObj((bool)_autoGeneratedColumns, "AutoGenerateColumns", writer);
                }
                if (!_dataAutoHeadings)
                {
                    Serializer.SerializeObj((bool)_dataAutoHeadings, "DataAutoHeadings", writer);
                }
                if (_printInfo != null)
                {
                    Serializer.SerializeObj(_printInfo, "PrintInfo", writer);
                }
                if (_protect)
                {
                    Serializer.SerializeObj((bool)_protect, "Protect", writer);
                }
                if (((_namedStyles != null) && (_namedStyles.Count > 0)) && ((_namedStyles != null) && (_namedStyles.Count > 0)))
                {
                    Serializer.SerializeObj(_namedStyles, "NamedStyles", writer);
                }
                if ((_defaultStyle != null) && !_defaultStyle.IsEmpty)
                {
                    Serializer.SerializeObj(_defaultStyle, "DefaultStyle", writer);
                }
                if ((_columnHeaderDefaultStyle != null) && !_columnHeaderDefaultStyle.IsEmpty)
                {
                    Serializer.SerializeObj(_columnHeaderDefaultStyle, "DefaultColumnHeaderStyle", writer);
                }
                if ((_rowHeaderDefaultStyle != null) && !_rowHeaderDefaultStyle.IsEmpty)
                {
                    Serializer.SerializeObj(_rowHeaderDefaultStyle, "DefaultRowHeaderStyle", writer);
                }
                if (!DefaultGridLineColor.Equals(_gridLineColor))
                {
                    Serializer.SerializeObj(_gridLineColor, "GridLineColor", writer);
                }
                if (!string.IsNullOrWhiteSpace(_sheetTabThemeColor))
                {
                    Serializer.SerializeObj(_sheetTabThemeColor, "SheetTabThemeColor", writer);
                }
                else if (!DefaultSheetTabColor.Equals(_sheetTabColor))
                {
                    Serializer.SerializeObj(_sheetTabColor, "SheetTabColor", writer);
                }
                if (!string.IsNullOrWhiteSpace(_selectionBorderThemeColor))
                {
                    Serializer.SerializeObj(_selectionBorderThemeColor, "SelectionBorderThemeColor", writer);
                }
                else if (!DefaultSheetTabColor.Equals(_selectionBorderColor))
                {
                    Serializer.SerializeObj(_selectionBorderColor, "SelectionBorderColor", writer);
                }
                if (!string.IsNullOrWhiteSpace(_touchSelectionGripperBackgroundThemeColor))
                {
                    Serializer.SerializeObj(_touchSelectionGripperBackgroundThemeColor, "TouchSelectionGripperBackgroundThemeColor", writer);
                }
                else if (!DefaultSheetTabColor.Equals(_touchSelectionGripperBackgroundColor))
                {
                    Serializer.SerializeObj(_touchSelectionGripperBackgroundColor, "TouchSelectionGripperBackgroundColor", writer);
                }
                if (_selectionBackground != null)
                {
                    Serializer.SerializeObj(_selectionBackground, "SelectionBackground", writer);
                }
                if (_gridLineColor != DefaultGridLineColor)
                {
                    Serializer.SerializeObj(_gridLineColor, "GridLineColor", writer);
                }
                if (!_showGridLine)
                {
                    Serializer.SerializeObj((bool)_showGridLine, "ShowGridLine", writer);
                }
            }
            Serializer.SerializeStorageBlock(_cellsBlock, "CellBlock", writer, dataOnly);
            Serializer.SerializeStorageBlock(_rowHeadersBlock, "RowHeaderBlock", writer, dataOnly);
            Serializer.SerializeStorageBlock(_columnHeadersBlock, "ColumnHeaderBlock", writer, dataOnly);
            bool flag = false;
            if ((_datasource != null) && (_datasource is IXmlSerializable))
            {
                Serializer.WriteStartObj("DataSource", writer);
                Serializer.WriteTypeAttr(_datasource, writer);
                Serializer.SerializeObj(_datasource, null, writer);
                Serializer.WriteEndObj(writer);
                flag = true;
            }
            if (_bindingManager != null)
            {
                Serializer.SerializeObj(_bindingManager, "BindingManager", writer);
                DataMatrix<object> allCachedData = null;
                if (flag)
                {
                    allCachedData = _bindingManager.GetAllCachedData();
                }
                else
                {
                    allCachedData = _bindingManager.GetAllBoundData();
                }
                if (allCachedData != null)
                {
                    writer.WriteStartElement("BindingCellValues");
                    Serializer.WriteAttribute("rc", allCachedData.RowCount, writer);
                    Serializer.WriteAttribute("cc", allCachedData.ColumnCount, writer);
                    Serializer.SerializeMatrix<object>(allCachedData, true, writer, true);
                    writer.WriteEndElement();
                }
            }
            if (!dataOnly)
            {
                Serializer.SerializeObj(_viewportRows, "ViewportRowsAxis", writer);
                Serializer.SerializeObj(_viewportColumns, "ViewportColumnsAxis", writer);
                Serializer.SerializeObj(_rowHeaderColumns, "RowHeaderColumnsAxis", writer);
                Serializer.SerializeObj(_columnHeaderRows, "ColumnHeaderRowsAxis", writer);
                Serializer.SerializeObj(_viewportRowsStyles, "ViewportRowsAxisStyle", true, writer);
                Serializer.SerializeObj(_viewportColumnsStyles, "ViewportColumnsAxisStyle", true, writer);
                Serializer.SerializeObj(_rowHeaderRowsStyles, "RowHeaderRowsAxisStyle", true, writer);
                Serializer.SerializeObj(_rowHeaderColumnsStyles, "RowHeaderColumnsAxisStyle", true, writer);
                Serializer.SerializeObj(_columnHeaderRowsStyles, "ColumnHeaderRowsAxisStyle", true, writer);
                Serializer.SerializeObj(_columnHeaderColumnsStyles, "ColumnHeaderColumnsAxisStyle", true, writer);
                Serializer.SerializeObj(_spanModel, "CellsSpanModel", writer);
                Serializer.SerializeObj(_rowHeaderSpanModel, "RowHeaderSpanModel", writer);
                Serializer.SerializeObj(_columnHeaderSpanModel, "ColumnHeaderSpanModel", writer);
                Serializer.SerializeObj(_selectionModel, "SelectionModel", writer);
                if ((_rowFilter != null) && (_rowFilter is HideRowFilter))
                {
                    Serializer.SerializeObj(_rowFilter, "RowFilter", writer);
                }
                if (FrozenRowCount > 0)
                {
                    Serializer.SerializeObj((int)FrozenRowCount, "FrozenRowCount", writer);
                }
                if (FrozenColumnCount > 0)
                {
                    Serializer.SerializeObj((int)FrozenColumnCount, "FrozenColumnCount", writer);
                }
                if (FrozenTrailingRowCount > 0)
                {
                    Serializer.SerializeObj((int)FrozenTrailingRowCount, "FrozenTrailingRowCount", writer);
                }
                if (FrozenTrailingColumnCount > 0)
                {
                    Serializer.SerializeObj((int)FrozenTrailingColumnCount, "FrozenTrailingColumnCount", writer);
                }
                if ((_rowRangeGroup != null) && (!_rowRangeGroup.IsEmpty() || (_rowRangeGroup.Direction != RangeGroupDirection.Forward)))
                {
                    Serializer.SerializeObj(_rowRangeGroup, "RowRangeGroup", writer);
                }
                if ((_columnRangeGroup != null) && (!_columnRangeGroup.IsEmpty() || (_columnRangeGroup.Direction != RangeGroupDirection.Forward)))
                {
                    Serializer.SerializeObj(_columnRangeGroup, "ColumnRangeGroup", writer);
                }
                if ((((excludeContent & ExcludePrintContent.ConditionalFormat) != ExcludePrintContent.ConditionalFormat) && (_conditionalFormats != null)) && (_conditionalFormats.RuleCount > 0))
                {
                    Serializer.SerializeConditionalFormats(_conditionalFormats, "Conditions", writer);
                }
                if (_viewportInfo != null)
                {
                    Serializer.SerializeObj(_viewportInfo, "ViewportInfo", writer);
                }
                if (_tables != null)
                {
                    Serializer.SerializeTables(_tables, "Tables", writer);
                }
                if ((_sparklineGroupManager != null) && (_sparklineGroupManager.Count > 0))
                {
                    Serializer.SerializeObj(_sparklineGroupManager, "SparklineGroupManager", writer);
                }
                if ((_spreadCharts != null) && (_spreadCharts.Count > 0))
                {
                    Serializer.SerializeList(_spreadCharts, "Charts", writer);
                }
                if ((_floatingObjects != null) && (_floatingObjects.Count > 0))
                {
                    Serializer.SerializeList(_floatingObjects, "FloatingObjects", writer);
                }
                if ((_pictures != null) && (_pictures.Count > 0))
                {
                    Serializer.SerializeList(_pictures, "Pictures", writer);
                }
                if (_tag != null)
                {
                    Serializer.SerializeTag(writer, _tag);
                }
            }
            return true;
        }

        /// <summary>
        /// Gets the active cell in the sheet.
        /// </summary>
        /// <value>The active cell in the sheet.</value>
        public Cell ActiveCell
        {
            get
            {
                if ((ActiveRowIndex > -1) && (ActiveColumnIndex > -1))
                {
                    return Cells[ActiveRowIndex, ActiveColumnIndex];
                }
                return null;
            }
        }

        /// <summary>
        /// Gets the active column in the sheet.
        /// </summary>
        /// <value>The active column in the sheet.</value>
        public Column ActiveColumn
        {
            get
            {
                if (ActiveColumnIndex > -1)
                {
                    return Columns[ActiveColumnIndex];
                }
                return null;
            }
        }

        /// <summary>
        /// Gets or sets the column index of the active cell in the sheet.
        /// </summary>
        /// <value>The column index of the active cell in the sheet.</value>
        public int ActiveColumnIndex
        {
            get
            {
                if (((Selection != null) && (Selection.AnchorColumn > -1)) && (Selection.AnchorColumn < ColumnCount))
                {
                    return Selection.AnchorColumn;
                }
                return -1;
            }
            set
            {
                if ((value < 0) || (value >= ColumnCount))
                {
                    throw new ArgumentOutOfRangeException("value");
                }
                if (Selection != null)
                {
                    int anchorRow = Selection.AnchorRow;
                    Selection.ClearSelection();
                    Selection.SetAnchor(anchorRow, value);
                    UpdateActiveViewportByAnchor();
                    RaisePropertyChanged("ActiveColumnIndex");
                }
            }
        }

        /// <summary>
        /// Gets the active row in the sheet.
        /// </summary>
        /// <value>The active row in the sheet.</value>
        public Row ActiveRow
        {
            get
            {
                if (ActiveRowIndex > -1)
                {
                    return Rows[ActiveRowIndex];
                }
                return null;
            }
        }

        /// <summary>
        /// Gets or sets the row index of the active cell in the sheet.
        /// </summary>
        /// <value>The row index of the active cell in the sheet.</value>
        public int ActiveRowIndex
        {
            get
            {
                if (((Selection != null) && (Selection.AnchorRow > -1)) && (Selection.AnchorRow < RowCount))
                {
                    return Selection.AnchorRow;
                }
                return -1;
            }
            set
            {
                if ((value < 0) || (value >= RowCount))
                {
                    throw new ArgumentOutOfRangeException("value");
                }
                if (Selection == null)
                {
                    throw new InvalidOperationException(ResourceStrings.WorksheetEmptySelection);
                }
                int anchorColumn = Selection.AnchorColumn;
                Selection.ClearSelection();
                Selection.SetAnchor(value, anchorColumn);
                UpdateActiveViewportByAnchor();
                RaisePropertyChanged("ActiveRowIndex");
            }
        }

        /// <summary>
        /// Gets or sets whether to automatically generate the columns for a sheet based on the data source. 
        /// </summary>
        /// <value>Boolean: true to automatically generate columns; otherwise, false.</value>
        [DefaultValue(true)]
        public bool AutoGenerateColumns
        {
            get { return _autoGeneratedColumns; }
            set
            {
                _autoGeneratedColumns = value;
                RaisePropertyChanged("AutoGenerateColumns");
            }
        }

        /// <summary>
        /// Gets or sets whether the worksheet needs to automatically recalculate.
        /// </summary>
        internal bool AutoRecalculation
        {
            get
            {
                if (_workbook != null)
                {
                    return _workbook.AutoRecalculation;
                }
                return _autoRecalculation;
            }
            set
            {
                if (_autoRecalculation != value)
                {
                    _autoRecalculation = value;
                    if (AutoRecalculation && (_calcEngineSuspended == 0))
                    {
                        CalcManager.Recalculate(0xc350, false);
                    }
                    RaisePropertyChanged("AutoRecalculation");
                }
            }
        }

        internal BorderCollapse BorderCollapse
        {
            get { return BorderCollapse.Collapse; }
        }

        internal SpreadCalcAxialManager CalcAxial
        {
            get
            {
                if (_calcAxial == null)
                {
                    _calcAxial = new SpreadCalcAxialManager(this, CalcManager, CalcStorage);
                }
                return _calcAxial;
            }
        }

        CalcCalculationManager CalcManager
        {
            get
            {
                if (_calcManager == null)
                {
                    _calcManager = FormulaService.GetCalculationManager(this, CalcStorage, true);
                }
                return _calcManager;
            }
        }

        SpreadCalcStorage CalcStorage
        {
            get { return _calcStorage; }
        }

        /// <summary>
        /// Gets the two-dimensional collection of Cell objects.
        /// </summary>
        /// <value>The two-dimensional collection of Cell objects.</value>
        public Cells Cells
        {
            get
            {
                if (_cells == null)
                {
                    _cells = new Cells(this, SheetArea.Cells);
                }
                return _cells;
            }
        }

        /// <summary>
        /// Gets the charts.
        /// </summary>
        public SpreadCharts Charts
        {
            get
            {
                if (_spreadCharts == null)
                {
                    _spreadCharts = new SpreadCharts(this);
                }
                return _spreadCharts;
            }
        }

        internal string CodeName { get; set; }

        /// <summary>
        /// Gets or sets the number of columns in the sheet.
        /// </summary>
        /// <value>The number of columns in the sheet.</value>
        [DefaultValue(100)]
        public int ColumnCount
        {
            get { return _cellsBlock.ColumnCount; }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException(ResourceStrings.WorksheetColumnCountMsg);
                }
                int columnCount = _cellsBlock.ColumnCount;
                _columnHeadersBlock.ColumnCount = _cellsBlock.ColumnCount = value;
                _columnHeaderColumnsStyles.Length = _viewportColumnsStyles.Length = _viewportColumns.Length = value;
                if (columnCount > value)
                {
                    OnColumnsRemoved(value, columnCount - value, SheetArea.Cells);
                    RaiseColumnChanged(string.Empty, value, columnCount - value, SheetArea.Cells, SheetChangedEventAction.ColumnsRemoved);
                }
                else if (columnCount < value)
                {
                    OnColumnsAdded(columnCount, value - columnCount, SheetArea.Cells);
                    RaiseColumnChanged(string.Empty, columnCount, value - columnCount, SheetArea.Cells, SheetChangedEventAction.ColumnsAdded);
                }
                RaisePropertyChanged("ColumnCount");
            }
        }

        /// <summary>
        /// Gets or sets whether the column footers in this sheet are visible.
        /// </summary>
        /// <value><c>true</c> if the column footers are visible; otherwise, <c>false</c>.</value>
        [DefaultValue(false)]
        internal bool ColumnFooterVisible
        {
            get { return _columnFooterVisible; }
            set
            {
                if (_columnFooterVisible != value)
                {
                    _columnFooterVisible = value;
                    RaisePropertyChanged("ColumnFooterVisible");
                }
            }
        }

        /// <summary>
        /// Gets the column header area of the sheet.
        /// </summary>
        /// <value>The <see cref="P:Worksheet.ColumnHeader" /> object for the column header area of the sheet.</value>
        public ColumnHeader ColumnHeader
        {
            get
            {
                if (_columnHeader == null)
                {
                    _columnHeader = new ColumnHeader(this);
                }
                return _columnHeader;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether column headers for the sheet display letters or numbers or are blank.
        /// </summary>
        /// <value>The <see cref="T:HeaderAutoText" /> enumeration that indicates whether column headers for the sheet display letters or numbers or are blank.</value>
        /// <exception cref="T:System.Exception">
        /// Specified header setting is not valid; must be one of the HeaderAutoText enumeration settings
        /// </exception>
        [DefaultValue(2)]
        internal HeaderAutoText ColumnHeaderAutoText
        {
            get { return _columnHeaderAutoText; }
            set
            {
                if (_columnHeaderAutoText != value)
                {
                    _columnHeaderAutoText = value;
                    RaisePropertyChanged("ColumnHeaderAutoText");
                }
            }
        }

        /// <summary>
        /// Gets or sets which column header row for the sheet displays the automatic text
        /// when there are multiple column header rows.
        /// </summary>
        /// <value>The column header row in this sheet that displays the automatic text.</value>
        [DefaultValue(-1)]
        internal int ColumnHeaderAutoTextIndex
        {
            get { return _columnHeaderAutoTextIndex; }
            set
            {
                if ((-1 <= value) && (value < ColumnHeaderRowCount))
                {
                    _columnHeaderAutoTextIndex = value;
                    RaisePropertyChanged("ColumnHeaderAutoTextIndex");
                }
            }
        }

        /// <summary>
        /// Gets or sets a cell style object for the column header default style of the sheet.
        /// </summary>
        /// <value>A <see cref="T:StyleInfo" /> object that specifies the column header default style of the sheet.</value>
        internal StyleInfo ColumnHeaderDefaultStyle
        {
            get
            {
                if (_columnHeaderDefaultStyle == null)
                {
                    _columnHeaderDefaultStyle = new StyleInfo();
                    _columnHeaderDefaultStyle.HorizontalAlignment = CellHorizontalAlignment.Center;
                    _columnHeaderDefaultStyle.VerticalAlignment = CellVerticalAlignment.Center;
                    _columnHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                    if (Application.Current.RequestedTheme == ApplicationTheme.Dark)
                    {
                        _columnHeaderDefaultStyle.Foreground = new SolidColorBrush(Colors.Black);
                    }
                }
                return _columnHeaderDefaultStyle;
            }
            set
            {
                if (_columnHeaderDefaultStyle != null)
                {
                    _columnHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                }
                if (value != null)
                    _columnHeaderDefaultStyle = new StyleInfo(value.Name, DefaultColumnHeaderStyleName, value);
                else
                    _columnHeaderDefaultStyle = new StyleInfo(null, DefaultColumnHeaderStyleName);

                _columnHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnColumnHeaderDefaultStylePropertyChanged);
                RaisePropertyChanged("ColumnHeaderDefaultStyle");
            }
        }

        /// <summary>
        /// Gets or sets the number of rows in the column header in the sheet.
        /// </summary>
        /// <value>The number of rows in the column header in the sheet.</value>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified row count is out of range; must be between 0 and 256.
        /// </exception>
        [DefaultValue(1)]
        internal int ColumnHeaderRowCount
        {
            get
            {
                StorageBlock storage = GetStorage(SheetArea.ColumnHeader);
                if (storage != null)
                {
                    return storage.RowCount;
                }
                return 0;
            }
            set
            {
                if ((0 > value) || (value > 0x100))
                {
                    throw new ArgumentOutOfRangeException("value", string.Format(ResourceStrings.WorksheetInvalidRowHeaderColumnCount, (object[])new object[] { ((int)0x100) }));
                }
                int rowCount = _columnHeadersBlock.RowCount;
                _columnHeadersBlock.RowCount = value;
                _columnHeaderRowsStyles.Length = _columnHeaderRows.Length = value;
                if (rowCount > value)
                {
                    OnRowsRemoved(value, rowCount - value, SheetArea.ColumnHeader);
                }
                else if (rowCount < value)
                {
                    OnRowsAdded(rowCount, value - rowCount, SheetArea.ColumnHeader);
                }
                RaisePropertyChanged("ColumnHeaderRowCount");
            }
        }

        /// <summary>
        /// Gets the column header span model.
        /// </summary>
        public SheetSpanModel ColumnHeaderSpanModel
        {
            get { return _columnHeaderSpanModel; }
        }

        /// <summary>
        /// Gets or sets whether the column headers in this sheet are visible.
        /// </summary>
        /// <value><c>true</c> if the column headers are visible; otherwise, <c>false</c>.</value>
        [DefaultValue(true)]
        internal bool ColumnHeaderVisible
        {
            get { return _columnHeaderVisible; }
            set
            {
                if (_columnHeaderVisible != value)
                {
                    _columnHeaderVisible = value;
                    RaisePropertyChanged("ColumnHeaderVisible");
                }
            }
        }

        /// <summary>
        /// Gets  the column range group.
        /// </summary>
        /// <value>The column range group.</value>
        public RangeGroup ColumnRangeGroup
        {
            get
            {
                if (_columnRangeGroup == null)
                {
                    _columnRangeGroup = new RangeGroup(ColumnCount, new SortedIndexAdapter.SortedColumnIndexAdapter(this));
                    _columnRangeGroup.Changed += new EventHandler(OnColumnRangeGroupChanged);
                }
                return _columnRangeGroup;
            }
        }

        /// <summary>
        /// Gets the one-dimensional collection of Column objects.
        /// </summary>
        /// <value>The one-dimensional collection of Column objects.</value>
        public Columns Columns
        {
            get
            {
                if (_columns == null)
                {
                    _columns = new Columns(this, SheetArea.Cells);
                }
                return _columns;
            }
        }

        /// <summary>
        /// Gets the conditional format for the sheet.
        /// </summary>
        /// <value>The conditional format for the sheet.</value>
        public ConditionalFormat ConditionalFormats
        {
            get
            {
                if (_conditionalFormats == null)
                {
                    _conditionalFormats = new ConditionalFormat(this);
                    _conditionalFormats.RulesChanged += new EventHandler<RulesChangedEventArgs>(OnConditionalFormatsRulesChanged);
                }
                return _conditionalFormats;
            }
        }

        /// <summary>
        /// Gets the name of all custom functions.
        /// </summary>
        public string[] CustomFunctions
        {
            get { return Enumerable.ToArray<string>((IEnumerable<string>)Functions.Keys); }
        }

        /// <summary>
        /// Gets all custom names.
        /// </summary>
        public string[] CustomNames
        {
            get { return Names.GetNames(); }
        }

        /// <summary>
        /// Gets or sets whether to automatically generate column names for a sheet based on the data source. 
        /// </summary>
        /// <value>Boolean: true to automatically generate column names; otherwise, false.</value>
        [DefaultValue(true)]
        public bool DataAutoHeadings
        {
            get { return _dataAutoHeadings; }
            set
            {
                _dataAutoHeadings = value;
                RaisePropertyChanged("DataAutoHeadings");
            }
        }

        /// <summary>
        /// Gets or sets the data source that populates the sheet.
        /// </summary>
        /// <value>The data source that populates the sheet.</value>
        [DefaultValue((string)null)]
        public object DataSource
        {
            get { return _datasource; }
            set
            {
                if (_bindingManager != null)
                {
                    _bindingManager.Unbind();
                }
                string name = _name;
                SheetBindingManager bindingManager = _bindingManager;
                Init(0, ColumnCount, RowHeaderColumnCount, ColumnHeaderRowCount, true);
                _bindingManager = bindingManager;
                _name = name;
                if (_workbook != null)
                {
                    _workbook.SuspendEvent();
                }
                _datasource = value;
                try
                {
                    if (value != null)
                    {
                        if (_bindingManager == null)
                        {
                            _bindingManager = new SheetBindingManager(this);
                        }
                        _bindingManager.Bind(value, AutoGenerateColumns);
                    }
                }
                finally
                {
                    if (_workbook != null)
                    {
                        _workbook.ResumeEvent();
                    }
                }
                RaisePropertyChanged("DataSource");
            }
        }

        /// <summary>
        /// Gets or sets the default height of the column footer row.
        /// </summary>
        /// <value>The default height of the column footer row.</value>
        [DefaultValue(30)]
        internal double DefaultColumnFooterRowHeight
        {
            get { return _defaultColumnFooterRowHeight; }
            set
            {
                _defaultColumnFooterRowHeight = value;
                RaisePropertyChanged("DefaultColumnFooterRowHeight");
            }
        }

        /// <summary>
        /// Gets or sets the default height of the column header row.
        /// </summary>
        /// <value>The default height of the column header row.</value>
        [DefaultValue(30)]
        internal double DefaultColumnHeaderRowHeight
        {
            get { return _defaultColumnHeaderRowHeight; }
            set
            {
                _defaultColumnHeaderRowHeight = value;
                RaisePropertyChanged("DefaultColumnHeaderRowHeight");
            }
        }

        /// <summary>
        /// Gets or sets the default column width.
        /// </summary>
        /// <value>The default width of the column.</value>
        [DefaultValue(80)]
        public double DefaultColumnWidth
        {
            get { return _defaultColumnWidth; }
            set
            {
                _defaultColumnWidth = value;
                RaisePropertyChanged("DefaultColumnWidth");
            }
        }

        // hdt 此处在导出pdf时异常，调整为固定Light模式
        //internal Color DefaultHeaderGridLineColor
        //{
        //    get
        //    {
        //        if (_defaultHeaderGridLineColor == Color.FromArgb(0, 0, 0, 0))
        //        {
        //            if (Application.Current.RequestedTheme == ApplicationTheme.Light)
        //            {
        //                _defaultHeaderGridLineColor = Color.FromArgb(0xff, 0x86, 140, 0x99);
        //            }
        //            else
        //            {
        //                _defaultHeaderGridLineColor = Color.FromArgb(0xff, 224, 224, 224);
        //            }
        //        }
        //        return _defaultHeaderGridLineColor;
        //    }
        //}

        /// <summary>
        /// Gets or sets the default width of the row header columns.
        /// </summary>
        /// <value>The default width of the row header columns.</value>
        [DefaultValue(40)]
        internal double DefaultRowHeaderColumnWidth
        {
            get { return _defaultRowHeaderColumnWidth; }
            set
            {
                _defaultRowHeaderColumnWidth = value;
                RaisePropertyChanged("DefaultRowHeaderColumnWidth");
            }
        }

        /// <summary>
        /// Gets or sets the default row height.
        /// </summary>
        /// <value>The default height of the row.</value>
        [DefaultValue(0x19)]
        public double DefaultRowHeight
        {
            get { return _defaultRowHeight; }
            set
            {
                _defaultRowHeight = value;
                RaisePropertyChanged("DefaultRowHeight");
            }
        }

        /// <summary>
        /// Gets or sets default style information for the workbook.
        /// </summary>
        public StyleInfo DefaultStyle
        {
            get
            {
                if (_defaultStyle == null)
                {
                    _defaultStyle = new StyleInfo();
                    _defaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                }
                return _defaultStyle;
            }
            set
            {
                if (_defaultStyle != value)
                {
                    if (_defaultStyle != null)
                    {
                        _defaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                    }
                    _defaultStyle = value;
                    if (_defaultStyle != null)
                    {
                        _defaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnDefaultStyleInfoPropertyChanged);
                    }
                    StyleInfoCollection.IncreaseStyleInfoVersion();
                }
                RaisePropertyChanged("DefaultStyle");
            }
        }

        internal Dt.Xls.ExcelSheetType ExcelSheetType { get; set; }

        /// <summary>
        /// Gets the floating objects.
        /// </summary>
        /// <value>
        /// The floating objects.
        /// </value>
        public FloatingObjects FloatingObjects
        {
            get
            {
                if (_floatingObjects == null)
                {
                    _floatingObjects = new FloatingObjects(this);
                }
                return _floatingObjects;
            }
        }

        internal CalcService FormulaService
        {
            get
            {
                if (_workbook != null)
                {
                    return _workbook.FormulaService;
                }
                if (_calcService == null)
                {
                    _calcService = new CalcService();
                    CalcEvaluator.EvaluatorCulture = CultureInfo.InvariantCulture;
                }
                return _calcService;
            }
        }

        /// <summary>
        /// Gets or sets the number of nonscrolling columns on the leading edge of this sheet.
        /// </summary>
        /// <value>The number of nonscrolling columns on the leading edge of this sheet.</value>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified value is out of range; must be greater than or equal to zero.
        /// </exception>
        [DefaultValue(0)]
        public int FrozenColumnCount
        {
            get { return _frozenColumnCount; }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException();
                }
                if (_frozenColumnCount != value)
                {
                    _frozenColumnCount = value;
                    OnFrozenColumnCountChanged();
                    RaisePropertyChanged("FrozenColumnCount");
                }
            }
        }

        internal double FrozenHeight
        {
            get
            {
                double num = 0.0;
                for (int i = 0; i < FrozenRowCount; i++)
                {
                    num += Rows[i].ActualHeight * ZoomFactor;
                }
                return num;
            }
        }

        /// <summary>
        /// Gets or sets the number of nonscrolling rows on the leading edge of this sheet.
        /// </summary>
        /// <value>The number of nonscrolling rows on the leading edge of this sheet.</value>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified value is out of range; must be greater than or equal to zero.
        /// </exception>
        [DefaultValue(0)]
        public int FrozenRowCount
        {
            get { return _frozenRowCount; }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException();
                }
                if (_frozenRowCount != value)
                {
                    _frozenRowCount = value;
                    OnFrozenRowCountChanged();
                    RaisePropertyChanged("FrozenRowCount");
                }
            }
        }

        /// <summary>
        /// Gets or sets the number of nonscrolling columns on the trailing edge of this sheet.
        /// </summary>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified value is out of range; must be greater than or equal to zero.
        /// </exception>
        [DefaultValue(0)]
        public int FrozenTrailingColumnCount
        {
            get { return _frozenTrailingColumnCount; }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException();
                }
                if (_frozenTrailingColumnCount != value)
                {
                    _frozenTrailingColumnCount = value;
                    RaisePropertyChanged("FrozenTrailingColumnCount");
                }
            }
        }

        internal double FrozenTrailingHeight
        {
            get
            {
                double num = 0.0;
                for (int i = Math.Max(FrozenRowCount, RowCount - FrozenTrailingRowCount); i < RowCount; i++)
                {
                    num += Rows[i].ActualHeight * ZoomFactor;
                }
                return num;
            }
        }

        /// <summary>
        /// Gets or sets the number of nonscrolling rows on the trailing edge of this sheet.
        /// </summary>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified value is out of range; must be greater than or equal to zero.
        /// </exception>
        [DefaultValue(0)]
        public int FrozenTrailingRowCount
        {
            get { return _frozenTrailingRowCount; }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException();
                }
                if (_frozenTrailingRowCount != value)
                {
                    _frozenTrailingRowCount = value;
                    RaisePropertyChanged("FrozenTrailingRowCount");
                }
            }
        }

        internal double FrozenTrailingWidth
        {
            get
            {
                double num = 0.0;
                for (int i = Math.Max(FrozenColumnCount, ColumnCount - FrozenTrailingColumnCount); i < ColumnCount; i++)
                {
                    num += Columns[i].ActualWidth * ZoomFactor;
                }
                return num;
            }
        }

        internal double FrozenWidth
        {
            get
            {
                double num = 0.0;
                for (int i = 0; i < FrozenColumnCount; i++)
                {
                    num += Columns[i].ActualWidth * ZoomFactor;
                }
                return num;
            }
        }

        internal IDictionary<string, CalcFunction> Functions
        {
            get
            {
                if (_functions == null)
                {
                    _functions = (IDictionary<string, CalcFunction>)new Dictionary<string, CalcFunction>();
                }
                return _functions;
            }
        }

        /// <summary>
        /// Gets or sets the gridline color.
        /// </summary>
        public Color GridLineColor
        {
            get { return _gridLineColor; }
            set
            {
                _gridLineColor = value;
                _gridLine = null;
                RaisePropertyChanged("GridLineColor");
            }
        }

        internal bool HasFormula
        {
            get { return ((_calcManager != null) && !_calcManager.Graph.IsEmpty); }
        }

        internal bool IsColumnSorted
        {
            get { return false; }
        }

        internal bool IsFormulaSuspended
        {
            get { return (_calcEngineSuspended != 0); }
        }

        internal bool IsRowSorted
        {
            get { return false; }
        }

        /// <summary>
        /// Gets a new cell with a specified tag, or returns null if there is no cell with the specified tag.
        /// </summary>
        /// <param name="tag">A specified tag that can be used to find a cell.</param>
        /// <value>A <see cref="T:Cell" /> object.</value>
        public Cell this[string tag]
        {
            get { return Cells[tag]; }
        }

        /// <summary>
        /// Gets a new cell for the specified row and column.
        /// </summary>
        /// <value>A <see cref="T:Cell" /> object.</value>
        /// <param name="row">The row index.</param>
        /// <param name="column">The column index.</param>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified row index is not valid; must be between zero and the total number of rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified column index is not valid; must be between zero and the total number of columns.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified column header row index is not valid; must be between zero and the total number of column header rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified row header column index is not valid; must be between zero and the total number of row header columns.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified sheet corner row index is not valid; must be between zero and the total number of sheet corner rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified sheet corner column index is not valid; must be between zero and the total number of sheet corner columns.
        /// </exception>
        public Cell this[int row, int column]
        {
            get { return Cells[row, column]; }
        }

        /// <summary>
        /// Gets a new cell from the range of cells with the specified rows and columns.
        /// </summary>
        /// <param name="row">The starting row index.</param>
        /// <param name="column">The starting column index.</param>
        /// <param name="row2">The ending row index.</param>
        /// <param name="column2">The ending column index.</param>
        /// <value>A <see cref="T:Cell" /> object.</value>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified starting row index is not valid; must be between zero and the total number of rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified starting column index is not valid; must be between zero and the total number of columns.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified starting column header row index is not valid; must be between zero and the total number of column header rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified starting row header column index is not valid; must be between zero and the total number of row header columns.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified starting sheet corner row index is not valid; must be between zero and the total number of sheet corner rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified starting sheet corner column index is not valid; must be between zero and the total number of sheet corner columns.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified ending row index is not valid; must be between zero and the total number of rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified ending column index is not valid; must be between zero and the total number of columns.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified ending column header row index is not valid; must be between zero and the total number of column header rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified ending row header column index is not valid; must be between zero and the total number of row header columns.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified ending sheet corner row index is not valid; must be between zero and the total number of sheet corner rows.
        /// </exception>
        /// <exception cref="T:System.IndexOutOfRangeException">
        /// Specified ending sheet corner column index is not valid; must be between zero and the total number of sheet corner columns.
        /// </exception>
        public Cell this[int row, int column, int row2, int column2]
        {
            get { return Cells[row, column, row2, column2]; }
        }

        /// <summary>
        /// Gets the last nonempty bound row index on the sheet.
        /// </summary>
        /// <value>
        /// The last index of the nonempty row.
        /// </value>
        [DefaultValue(0)]
        public int LastNonEmptyRowIndex
        {
            get
            {
                int num = -1;
                foreach (int num2 in _cellsBlock.GetNonEmptyRows())
                {
                    if (num2 > num)
                    {
                        num = num2;
                    }
                }
                if ((_bindingManager != null) && _bindingManager.IsBound)
                {
                    int lastBoundRow = _bindingManager.GetLastBoundRow();
                    if (lastBoundRow > num)
                    {
                        num = lastBoundRow;
                    }
                }
                if (_tables != null)
                {
                    int lastTableRow = GetLastTableRow();
                    if (num < lastTableRow)
                    {
                        num = lastTableRow;
                    }
                }
                return num;
            }
        }

        /// <summary>
        /// Gets or sets the name of the sheet.
        /// </summary>
        /// <value>The name of the sheet.</value>
        [DefaultValue("")]
        public string Name
        {
            get
            {
                if (_name == null)
                {
                    return string.Empty;
                }
                return _name;
            }
            set
            {
                if (_name != value)
                {
                    if (_workbook != null)
                    {
                        if (string.IsNullOrEmpty(value))
                        {
                            throw new NotSupportedException(ResourceStrings.SheetNameCannotBeNullOrEmpty);
                        }
                        if (_workbook.Sheets != null)
                        {
                            foreach (Worksheet worksheet in _workbook.Sheets)
                            {
                                if ((worksheet != this) && (worksheet.Name == value))
                                {
                                    throw new NotSupportedException(ResourceStrings.AnotherWorksheetWithTheSameNameError);
                                }
                            }
                        }
                    }
                    _name = value;
                    RaisePropertyChanged("Name");
                }
            }
        }

        /// <summary>
        /// Gets or sets a collection of StyleInfo objects for this sheet.
        /// </summary>
        /// <value>The collection of StyleInfo objects for this sheet.</value>
        [DefaultValue((string)null)]
        public StyleInfoCollection NamedStyles
        {
            get { return _namedStyles; }
            set
            {
                if (_namedStyles != null)
                {
                    _namedStyles.Changed -= new EventHandler<StyleInfoCollectionChangedEventArgs>(OnNamedStyleChanged);
                }
                if (value == null)
                {
                    _namedStyles = new StyleInfoCollection();
                }
                else
                {
                    _namedStyles = value;
                }
                if ((value != null) && (value.Owner != Workbook))
                {
                    value.Owner = this;
                }
                _namedStyles.Changed += new EventHandler<StyleInfoCollectionChangedEventArgs>(OnNamedStyleChanged);
                RaisePropertyChanged("NamedStyles");
            }
        }

        internal NameInfoCollection Names
        {
            get
            {
                if (_names == null)
                {
                    _names = new NameInfoCollection();
                    _names.Changed += new EventHandler<NameInfoCollectionChangedEventArgs>(OnNameInfoCollectionChanged);
                }
                return _names;
            }
        }

        /// <summary>
        /// Gets the number of rows in cell blocks that contain data, styles, tags, or sparklines.
        /// </summary>
        [DefaultValue(0)]
        public int NonEmptyRowCount
        {
            get { return GetNonEmptyRows().Count; }
        }

        /// <summary>
        /// Gets the pictures.
        /// </summary>
        /// <value>
        /// The pictures.
        /// </value>
        public SpreadPictures Pictures
        {
            get
            {
                if (_pictures == null)
                {
                    _pictures = new SpreadPictures(this);
                }
                return _pictures;
            }
        }

        /// <summary>
        /// Gets or sets the print information for printing to PDF.
        /// </summary>
        public PrintInfo PrintInfo
        {
            get
            {
                if (_printInfo == null)
                {
                    _printInfo = new PrintInfo();
                }
                return _printInfo;
            }
            set
            {
                if (_printInfo != value)
                {
                    _printInfo = value;
                    RaisePropertyChanged("PrintInfo");
                }
            }
        }

        /// <summary>
        /// Gets or sets whether cells on this sheet that are marked as protected are uneditable. 
        /// </summary>
        [DefaultValue(false)]
        public bool Protect
        {
            get { return _protect; }
            set
            {
                if (value != _protect)
                {
                    _protect = value;
                    RaisePropertyChanged("Protect");
                }
            }
        }

        /// <summary>
        /// Gets or sets the style for cell and range references in cell formulas on this sheet.
        /// </summary>
        [DefaultValue(0)]
        public ReferenceStyle ReferenceStyle
        {
            get { return _referenceStyle; }
            set
            {
                if (_referenceStyle != value)
                {
                    _referenceStyle = value;
                    if ((ConditionalFormats != null) && (ConditionalFormats.ConditionalGraph != null))
                    {
                        ConditionalFormats.ConditionalGraph.UpdataFormulas();
                    }
                    RaisePropertyChanged("ReferenceStyle");
                }
            }
        }

        /// <summary>
        /// Gets or sets the number of rows in the sheet.
        /// </summary>
        /// <value>The number of rows in the sheet.</value>
        [DefaultValue(500)]
        public int RowCount
        {
            get { return _cellsBlock.RowCount; }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException(ResourceStrings.WorksheetRowCountMsg);
                }
                int rowCount = _cellsBlock.RowCount;
                _rowHeadersBlock.RowCount = _cellsBlock.RowCount = value;
                _rowHeaderRowsStyles.Length = _viewportRowsStyles.Length = _viewportRows.Length = value;
                if (rowCount > value)
                {
                    OnRowsRemoved(value, rowCount - value, SheetArea.Cells);
                    RaiseRowChanged(string.Empty, value, rowCount - value, SheetArea.Cells, SheetChangedEventAction.RowsRemoved);
                }
                else if (rowCount < value)
                {
                    OnRowsAdded(rowCount, value - rowCount, SheetArea.Cells);
                    RaiseRowChanged(string.Empty, rowCount, value - rowCount, SheetArea.Cells, SheetChangedEventAction.RowsAdded);
                }
                RaisePropertyChanged("RowCount");
            }
        }

        /// <summary>
        /// Gets or sets the row filter for the sheet.
        /// </summary>
        /// <value>The row filter for the sheet.</value>
        [DefaultValue((string)null)]
        public RowFilterBase RowFilter
        {
            get { return _rowFilter; }
            set
            {
                if (_rowFilter != value)
                {
                    if (_rowFilter != null)
                    {
                        _rowFilter.Reset();
                        _rowFilter.Changed -= new EventHandler<FilterEventArgs>(OnFilterStatusChanged);
                    }
                    _rowFilter = value;
                    if (_rowFilter != null)
                    {
                        _rowFilter.Changed += new EventHandler<FilterEventArgs>(OnFilterStatusChanged);
                        _rowFilter.Sheet = this;
                    }
                    RaisePropertyChanged("RowFilter");
                }
            }
        }

        /// <summary>
        /// Gets the row header area of the sheet.
        /// </summary>
        /// <value>The <see cref="P:Worksheet.RowHeader" /> object for the row header area of the sheet.</value>
        public RowHeader RowHeader
        {
            get
            {
                if (_rowHeader == null)
                {
                    _rowHeader = new RowHeader(this);
                }
                return _rowHeader;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether row headers on this sheet display letters or numbers or are blank.
        /// </summary>
        /// <value>The <see cref="T:HeaderAutoText" /> enumeration that indicates whether row headers on this sheet display letters or numbers or are blank.</value>
        /// <exception cref="T:System.Exception">
        /// Specified header setting is not valid; must be one of the HeaderAutoText enumeration settings
        /// </exception>
        [DefaultValue(1)]
        internal HeaderAutoText RowHeaderAutoText
        {
            get { return _rowHeaderAutoText; }
            set
            {
                if (_rowHeaderAutoText != value)
                {
                    _rowHeaderAutoText = value;
                    RaisePropertyChanged("RowHeaderAutoText");
                }
            }
        }

        /// <summary>
        /// Gets or sets which row header column on this sheet displays the automatic text
        /// when there are multiple row header columns.
        /// </summary>
        /// <value>The row header column in this sheet that displays the automatic text.</value>
        [DefaultValue(-1)]
        internal int RowHeaderAutoTextIndex
        {
            get { return _rowHeaderAutoTextIndex; }
            set
            {
                if ((-1 <= value) && (value < RowHeaderColumnCount))
                {
                    _rowHeaderAutoTextIndex = value;
                    RaisePropertyChanged("RowHeaderAutoTextIndex");
                }
            }
        }

        /// <summary>
        /// Gets or sets the number of columns in the row header in this sheet.
        /// </summary>
        /// <value>The number of columns in the row header in this sheet.</value>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified column count is out of range; must be between 0 and 256.
        /// </exception>
        [DefaultValue(1)]
        internal int RowHeaderColumnCount
        {
            get
            {
                StorageBlock storage = GetStorage(SheetArea.CornerHeader | SheetArea.RowHeader);
                if (storage != null)
                {
                    return storage.ColumnCount;
                }
                return 0;
            }
            set
            {
                if ((0 > value) || (value > 0x100))
                {
                    throw new ArgumentOutOfRangeException("value", string.Format(ResourceStrings.WorksheetInvalidRowHeaderColumnCount, (object[])new object[] { ((int)0x100) }));
                }
                int columnCount = _rowHeadersBlock.ColumnCount;
                _rowHeadersBlock.ColumnCount = value;
                _rowHeaderColumnsStyles.Length = _rowHeaderColumns.Length = value;
                if (columnCount > value)
                {
                    OnColumnsRemoved(value, columnCount - value, SheetArea.CornerHeader | SheetArea.RowHeader);
                }
                else if (columnCount < value)
                {
                    OnColumnsAdded(columnCount, value - columnCount, SheetArea.CornerHeader | SheetArea.RowHeader);
                }
                RaisePropertyChanged("RowHeaderColumnCount");
            }
        }

        /// <summary>
        /// Gets or sets a cell style object for the row header default style of the sheet.
        /// </summary>
        /// <value>A <see cref="T:StyleInfo" /> object that specifies the row header default style of the sheet.</value>
        internal StyleInfo RowHeaderDefaultStyle
        {
            get
            {
                if (_rowHeaderDefaultStyle == null)
                {
                    _rowHeaderDefaultStyle = new StyleInfo();
                    _rowHeaderDefaultStyle.HorizontalAlignment = CellHorizontalAlignment.Center;
                    _rowHeaderDefaultStyle.VerticalAlignment = CellVerticalAlignment.Center;
                    _rowHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                    if (Application.Current.RequestedTheme == ApplicationTheme.Dark)
                    {
                        _rowHeaderDefaultStyle.Foreground = new SolidColorBrush(Colors.Black);
                    }
                }
                return _rowHeaderDefaultStyle;
            }
            set
            {
                if (_rowHeaderDefaultStyle != null)
                {
                    _rowHeaderDefaultStyle.PropertyChanged -= new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                }

                if (value != null)
                {
                    _rowHeaderDefaultStyle = new StyleInfo(value.Name, DefaultRowHeaderStyleName, value);
                }
                else
                {
                    _rowHeaderDefaultStyle = new StyleInfo(null, DefaultRowHeaderStyleName);
                    if ((_rowHeaderDefaultStyle != null) && (Application.Current.RequestedTheme == ApplicationTheme.Dark))
                    {
                        _rowHeaderDefaultStyle.Foreground = new SolidColorBrush(Colors.White);
                    }
                }
                _rowHeaderDefaultStyle.PropertyChanged += new PropertyChangedEventHandler(OnRowHeaderDefaultStylePropertyChanged);
                RaisePropertyChanged("RowHeaderDefaultStyle");
            }
        }

        /// <summary>
        /// Gets the row header span model.
        /// </summary>
        public SheetSpanModel RowHeaderSpanModel
        {
            get { return _rowHeaderSpanModel; }
        }

        /// <summary>
        /// Gets or sets whether the row headers in this sheet are visible.
        /// </summary>
        /// <value><c>true</c> if the row headers are visible; otherwise, <c>false</c>.</value>
        [DefaultValue(true)]
        internal bool RowHeaderVisible
        {
            get { return _rowHeaderVisible; }
            set
            {
                if (_rowHeaderVisible != value)
                {
                    _rowHeaderVisible = value;
                    RaisePropertyChanged("RowHeaderVisible");
                }
            }
        }

        /// <summary>
        /// Gets  the row range group.
        /// </summary>
        /// <value>The row range group.</value>
        public RangeGroup RowRangeGroup
        {
            get
            {
                if (_rowRangeGroup == null)
                {
                    _rowRangeGroup = new RangeGroup(RowCount, new SortedIndexAdapter.SortedRowIndexAdapter(this));
                    _rowRangeGroup.Changed += new EventHandler(OnRowRangeGroupChanged);
                }
                return _rowRangeGroup;
            }
        }

        /// <summary>
        /// Gets the one-dimensional collection of Row objects.
        /// </summary>
        /// <value>The one-dimensional collection of Row objects.</value>
        public Rows Rows
        {
            get
            {
                if (_rows == null)
                {
                    _rows = new Rows(this, SheetArea.Cells);
                }
                return _rows;
            }
        }

        /// <summary>
        /// Gets the selection model for this sheet.
        /// </summary>
        /// <value>The selection model for this sheet.</value>
        internal SheetSelectionModelBase Selection
        {
            get { return _selectionModel; }
        }

        /// <summary>
        /// Gets or sets a Brush object for the selection style.
        /// </summary>
        /// <value>The Brush object for this sheet.</value>
        public Brush SelectionBackground
        {
            get
            {
                if (_selectionBackground == null)
                {
                    _selectionBackground = new SolidColorBrush(Color.FromArgb(0x1F, 0x1B, 0xA1, 0xE2));
                }
                return _selectionBackground;
            }
            set
            {
                if (_selectionBackground != value)
                {
                    _selectionBackground = value;
                    RaisePropertyChanged("SelectionBackground");
                }
            }
        }

        /// <summary>
        /// Gets or sets a Color object for the selection style.
        /// </summary>
        /// <value>The Color object for this sheet.</value>
        public Color SelectionBorderColor
        {
            get { return _selectionBorderColor; }
            set
            {
                _selectionBorderThemeColor = null;
                if (_selectionBorderColor != value)
                {
                    _selectionBorderColor = value;
                    RaisePropertyChanged("SelectionBorderColor");
                }
            }
        }

        /// <summary>
        /// Gets or sets a string object for the selection style.
        /// </summary>
        /// <value>The string object for this sheet.</value>
        [DefaultValue((string)null)]
        public string SelectionBorderThemeColor
        {
            get { return _selectionBorderThemeColor; }
            set
            {
                _selectionBorderColor = Color.FromArgb(220, 0, 0, 0);
                if (_selectionBorderThemeColor != value)
                {
                    _selectionBorderThemeColor = value;
                    RaisePropertyChanged("SelectionBorderThemeColor");
                }
            }
        }

        /// <summary>
        /// Gets or sets whether users can select ranges of items.
        /// </summary>
        /// <value>The <see cref="P:Worksheet.SelectionPolicy" /> enumeration that indicates how users can select ranges of items.</value>
        public SelectionPolicy SelectionPolicy
        {
            get { return Selection.SelectionPolicy; }
            set
            {
                if (Selection.SelectionPolicy != value)
                {
                    Selection.SelectionPolicy = value;
                    RaisePropertyChanged("SelectionPolicy");
                }
            }
        }

        /// <summary>
        /// Returns a collection of selected cell ranges.
        /// </summary>
        public ReadOnlyCollection<CellRange> Selections
        {
            get { return Selection.Items; }
        }

        /// <summary>
        /// Gets or sets whether users can select cells, rows, or columns.
        /// </summary>
        /// <value>The <see cref="P:Worksheet.SelectionUnit" /> enumeration that specifies the selection type.</value>
        public SelectionUnit SelectionUnit
        {
            get { return Selection.SelectionUnit; }
            set
            {
                if (Selection.SelectionUnit != value)
                {
                    Selection.SelectionUnit = value;
                    RaisePropertyChanged("SelectionUnit");
                }
            }
        }

        internal SheetBindingManager SheetBindingManager
        {
            get { return _bindingManager; }
        }

        internal WorksheetSparklineGroupManager SheetSparklineGroupManager
        {
            get
            {
                if (_sparklineGroupManager == null)
                {
                    _sparklineGroupManager = new WorksheetSparklineGroupManager(this, this);
                }
                return _sparklineGroupManager;
            }
        }

        /// <summary>
        /// A color object used to represent the sheet tab color.
        /// </summary>
        /// <remarks>If the SheetTabColor and SheetTabThemeColor properties are both set for the same sheet, then the last setting takes effect.</remarks>
        public Color SheetTabColor
        {
            get { return _sheetTabColor; }
            set
            {
                _sheetTabColor = value;
                _sheetTabThemeColor = null;
                RaisePropertyChanged("SheetTabColor");
            }
        }

        /// <summary>
        /// A theme color used to represent the sheet tab color.
        /// </summary>
        /// <remarks>
        /// If the SheetTabColor and SheetTabThemeColor properties are both set for the same sheet, then the last setting takes effect. This property has a high priority when the color is not null.
        /// </remarks>
        public string SheetTabThemeColor
        {
            get { return _sheetTabThemeColor; }
            set
            {
                _sheetTabThemeColor = value;
                _sheetTabColor = DefaultSheetTabColor;
                RaisePropertyChanged("SheetTabThemeColor");
            }
        }

        /// <summary>
        /// Determines whether to show gridlines.
        /// </summary>
        [DefaultValue(true)]
        public bool ShowGridLine
        {
            get { return _showGridLine; }
            set
            {
                _showGridLine = value;
                _gridLine = null;
                RaisePropertyChanged("ShowGridLine");
            }
        }

        /// <summary>
        /// Gets the sheet span model.
        /// </summary>
        public SheetSpanModel SpanModel
        {
            get { return _spanModel; }
        }

        /// <summary>
        /// Gets or sets the number or letter displayed in the first column header in this sheet.
        /// </summary>
        /// <value>The number or letter displayed in the first column header in this sheet.</value>
        [DefaultValue(1)]
        public int StartingColumnNumber
        {
            get { return _startingColumnNumber; }
            set
            {
                if (_startingColumnNumber != value)
                {
                    _startingColumnNumber = value;
                    RaisePropertyChanged("StartingColumnNumber");
                }
            }
        }

        /// <summary>
        /// Gets or sets the number or letter displayed in the first row header in this sheet.
        /// </summary>
        /// <value>The number or letter displayed in the first row header in this sheet.</value>
        [DefaultValue(1)]
        public int StartingRowNumber
        {
            get { return _startingRowNumber; }
            set
            {
                if (_startingRowNumber != value)
                {
                    _startingRowNumber = value;
                    RaisePropertyChanged("StartingRowNumber");
                }
            }
        }

        EricTables Tables
        {
            get
            {
                if (_tables == null)
                {
                    _tables = new EricTables(this);
                    _tables.SheetTableChanged += new EventHandler<TableChangedArgs>(OnSheetTableChanged);
                }
                return _tables;
            }
        }

        /// <summary>
        /// Gets or sets the tag.
        /// </summary>
        /// <value>
        /// The tag.
        /// </value>
        public object Tag
        {
            get { return _tag; }
            set
            {
                if (value != Tag)
                {
                    _tag = value;
                }
            }
        }

        /// <summary>
        /// Gets or sets a Color object for the selection style.
        /// </summary>
        /// <value>The Color object for this sheet.</value>
        public Color TouchSelectionGripperBackgroundColor
        {
            get { return _touchSelectionGripperBackgroundColor; }
            set
            {
                _touchSelectionGripperBackgroundThemeColor = null;
                if (_touchSelectionGripperBackgroundColor != value)
                {
                    _touchSelectionGripperBackgroundColor = value;
                    RaisePropertyChanged("TouchSelectionGripperBackgroundColor");
                }
            }
        }

        /// <summary>
        /// Gets or sets a string object for the selection style.
        /// </summary>
        /// <value>The string object for this sheet.</value>
        [DefaultValue((string)null)]
        public string TouchSelectionGripperBackgroundThemeColor
        {
            get { return _touchSelectionGripperBackgroundThemeColor; }
            set
            {
                _touchSelectionGripperBackgroundColor = Color.FromArgb(0xff, 0xff, 0xff, 0xff);
                if (_touchSelectionGripperBackgroundThemeColor != value)
                {
                    _touchSelectionGripperBackgroundThemeColor = value;
                    RaisePropertyChanged("TouchSelectionGripperBackgroundThemeColor");
                }
            }
        }

        internal List<IExcelChart> UnSupportedCharts
        {
            get
            {
                if (_unsupportedChartList == null)
                {
                    _unsupportedChartList = new List<IExcelChart>();
                }
                return _unsupportedChartList;
            }
        }


        internal List<IUnsupportRecord> UnSupportExcelRecrods
        {
            get
            {
                if (_unsupportedRecords == null)
                {
                    _unsupportedRecords = new List<IUnsupportRecord>();
                }
                return _unsupportedRecords;
            }
        }


        internal IList<object> UnsupportImages
        {
            get
            {
                if (_unsupportImages == null)
                {
                    _unsupportImages = (IList<object>)new List<object>();
                }
                return _unsupportImages;
            }
        }

        /// <summary>
        /// Gets or sets a value that indicates whether to display the sheet.
        /// </summary>
        /// <value><c>true</c> if the sheet is visible; otherwise, <c>false</c>.</value>
        [DefaultValue(true)]
        public bool Visible
        {
            get { return _visible; }
            set
            {
                if (_visible != value)
                {
                    _visible = value;
                    RaisePropertyChanged("Visible");
                }
            }
        }

        /// <summary>
        /// Gets the parent workbook for this sheet.
        /// </summary>
        /// <value>The parent workbook for this sheet.</value>
        [DefaultValue((string)null)]
        public Workbook Workbook
        {
            get { return _workbook; }
            internal set
            {
                if (_workbook != value)
                {
                    if (((_workbook != null) || (value == null)) && (_calcManager != null))
                    {
                        _calcManager.Dispose();
                    }
                    _workbook = value;
                    _calcManager = FormulaService.GetCalculationManager(this, _calcStorage, true);
                    CalcEvaluator.EvaluatorCulture = CultureInfo.InvariantCulture;
                    if ((_calcAxial != null) && (_calcAxial.Manager != _calcManager))
                    {
                        _calcAxial = null;
                        CalcAxial.ToString();
                    }
                    RaisePropertyChanged("Workbook");
                }
            }
        }

        /// <summary>
        /// Gets or sets the scaling factor for displaying this sheet.
        /// </summary>
        /// <value>The scaling factor for displaying this sheet.</value>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// Specified scaling amount is out of range; must be between 0.5 (50%) and 4.0 (400%).
        /// </exception>
        [DefaultValue((float)1f)]
        public float ZoomFactor
        {
            get { return _zoomFactor; }
            set
            {
                if ((value < 0.1f) || (4f < value))
                {
                    throw new ArgumentOutOfRangeException();
                }
                _zoomFactor = value;
                RaisePropertyChanged("ZoomFactor");
            }
        }

        class UIActionWrapper : IDisposable
        {
            public UIActionWrapper(Worksheet sheet)
            {
                Sheet = sheet;
            }

            public void Dispose()
            {
                if (Sheet != null)
                {
                    Sheet._currentUIActionWrapper = null;
                }
            }

            public Worksheet Sheet { get; private set; }
        }
    }
}

