---
id: jsx-in-depth
title: JSXの深層
permalink: docs/jsx-in-depth-ja-JP.html
prev: displaying-data-ja-JP.html
next: jsx-spread-ja_JP.html
---

[JSX](https://facebook.github.io/jsx/)はXMLに似たJavaScriptのシンタックスの拡張です。Reactでは、単純なJSXのシンタックスの変換を使うことができます。

## なぜJSXを使うのでしょうか？

ReactでJSXの使用を強制されるわけではありません。生のJSを使うこともできます。しかし、JSXは簡潔で、木構造とReactの特性を定義しやすいシンタックスであるため、JSXを使うことをお勧めします。

デザイナーのようなカジュアルな開発者にとってはさらに馴染みやすいでしょう。

XMLにはバランスの取れた開始タグと終了タグという利益があります。このことで、関数がオブジェクトリテラルを呼んでいるのを読むよりも簡単に大きな木構造を作ることができます。

これはJavaScriptのセマンティックスを代替するものではありません。

## HTMLタグ対Reactコンポーネント

ReactはHTMLタグ（文字列）とReactコンポーネント（クラス）の両方をレンダリングすることができます。

以下のようにJSXで小文字のタグ名を使用するだけで、HTMLタグをレンダリングできます。

```javascript
var myDivElement = <div className="foo" />;
ReactDOM.render(myDivElement, document.getElementById('example'));
```

以下のように大文字から始まるローカル変数を作成するだけで、Reactのコンポーネントをレンダリングできます。

```javascript
var MyComponent = React.createClass({/*...*/});
var myElement = <MyComponent someProperty={true} />;
ReactDOM.render(myElement, document.getElementById('example'));
```

ReactのJSXは大文字と小文字を使うことで、ローカルのコンポーネントクラスとHTMLタグを識別する習慣があります。

> 注意:
>
> JSXはJavaScriptなので、 `class` や `for` といった識別子はXMLの属性名としては使用しません。代わりに、 ReactのDOMコンポーネントはDOMのプロパティ名がそれぞれ `className` や `htmlFor` といったものであることを期待します。

## The Transform

ReactのJSXはXMLに似たシンタックスをネイティブなJavaScriptに変換します。XML要素や属性や子要素は `React.createElement` で渡される引数に変換されます。

```javascript
var Nav;
// 入力 (JSX):
var app = <Nav color="blue" />;
// 出力 (JS):
var app = React.createElement(Nav, {color:"blue"});
```

`<Nav />` を使うためには、 `Nav` 変数がスコープの中にないといけないことに注意してください。

以下のように、JSXはXMLシンタックスを使うことで、細かな子要素の使用も許可します。

```javascript
var Nav, Profile;
// 入力 (JSX):
var app = <Nav color="blue"><Profile>click</Profile></Nav>;
// 出力 (JS):
var app = React.createElement(
  Nav,
  {color:"blue"},
  React.createElement(Profile, null, "click")
);
```

以下のように、displayNameがundefinedの時には、JSXはクラスの[displayName](/react/docs/component-specs.html#displayname)を変数の割り当てから予測します。

```javascript
// 入力 (JSX):
var Nav = React.createClass({ });
// 出力 (JS):
var Nav = React.createClass({displayName: "Nav", });
```

JSXを試し、どのようにネイティブなJavaScriptに変換されるか見るには、[JSX Compiler](/react/jsx-compiler.html)を、すでに存在するHTMLをJSXに変換するには[HTMLからJSXへのコンバーター](http://magic.reactjs.net/htmltojsx.htm)を使ってください。

JSXを使いたい場合は、[始めてみましょう](/react/docs/getting-started-ja-JP.html)というガイドがどのようにコンパイルを設定するか示してくれます。

> 注意:
>
> JSXという表現は常にReactElementを評価します。実際に実行する際の詳細はおそらく異なっているでしょう。最適化されたモードでは `React.createElement` のコードのバリデーションを避けるためにReactElementをオブジェクトリテラルとして配置するでしょう。


## ネームスペース化されたコンポーネント

formのように、たくさんの子要素を持つコンポーネントを構築する際には、以下のように多くの変数を宣言しなければいけないでしょう。

```javascript
// 変数宣言のあまりよくない部分
var Form = MyFormComponent;
var FormRow = Form.Row;
var FormLabel = Form.Label;
var FormInput = Form.Input;

var App = (
  <Form>
    <FormRow>
      <FormLabel />
      <FormInput />
    </FormRow>
  </Form>
);
```

これを単純で簡単にするために、 *ネームスペース化されたコンポーネント* では、以下のように他のコンポーネントを付属物として持つ1つのコンポーネントを使うことができます。

```javascript
var Form = MyFormComponent;

var App = (
  <Form>
    <Form.Row>
      <Form.Label />
      <Form.Input />
    </Form.Row>
  </Form>
);
```

これを行うためには、以下のようにメインコンポーネントの付属物として、「サブコンポーネント」を作るだけで大丈夫です。

```javascript
var MyFormComponent = React.createClass({ ... });

MyFormComponent.Row = React.createClass({ ... });
MyFormComponent.Label = React.createClass({ ... });
MyFormComponent.Input = React.createClass({ ... });
```

JSXはコードをコンパイルする際にこのプロパティをハンドルします。

```javascript
var App = (
  React.createElement(Form, null,
    React.createElement(Form.Row, null,
      React.createElement(Form.Label, null),
      React.createElement(Form.Input, null)
    )
  )
);
```

> 注意:
> この特徴は [v0.11](/react/blog/2014/07/17/react-v0.11.html#jsx) 以上で使用できます。

## JavaScriptの表現

### アトリビュートの表現

JavaScriptで書いたものをアトリビュートの値として使うためには、その表現を引用(`""`)ではなく波括弧(`{}`)で囲ってください。

```javascript
// 入力 (JSX):
var person = <Person name={window.isLoggedIn ? window.name : ''} />;
// 出力 (JS):
var person = React.createElement(
  Person,
  {name: window.isLoggedIn ? window.name : ''}
);
```

### Booleanのアトリビュート

アトリビュートの値を記述しないと、JSXはそれを `true` として扱ってしまいます。`false` を渡すためには、アトリビュートが使われる必要があります。これらはHTMLのform要素の `disabled` 、 `required` 、 `checked` 、 `readOnly` といったアトリビュートを使う際によく見かけられます。


```javascript
// 以下の2つはボタンを使用不能にするという意味でJSXでは同義です。
<input type="button" disabled />;
<input type="button" disabled={true} />;

// 以下の2つはボタンを使用不能にしないという意味でJSXでは同義です。
<input type="button" />;
<input type="button" disabled={false} />;
```

### 子要素の表現

同様に、JavaScriptは子要素を表現するのに使われることもあります。

```javascript
// 入力 (JSX):
var content = <Container>{window.isLoggedIn ? <Nav /> : <Login />}</Container>;
// 出力 (JS):
var content = React.createElement(
  Container,
  null,
  window.isLoggedIn ? React.createElement(Nav) : React.createElement(Login)
);
```

### コメント

JSXにコメントを加えるのは簡単です。ただのJSの書き方です。タグの内側にコメントを書く時には、 `{}` で囲うことに注意してください。

```javascript
var content = (
  <Nav>
    {/* 子要素にコメントを書く時には、 {} で囲う */}
    <Person
      /* 複数
         行
         コメント */
      name={window.isLoggedIn ? window.name : ''} // 行末コメント
    />
  </Nav>
);
```

> 注意:
> JSXはHTMLに似ていますが、全く同じではありません。いくつかのキーの違いについては[JSXの理解](/react/docs/jsx-gotchas.html) をご覧ください。
