---
id: jsx-gotchas
title: JSXの理解
permalink: docs/jsx-gotchas-ja-JP.html
prev: jsx-spread-ja-JP.html
next: interactivity-and-dynamic-uis-ja-JP.html

---

JSXはHTMLに似ていますが、重要な違いがいくつかあります。

> 注意:
> インラインの `style` 属性のようなDOMの違いについては、[こちら](/react/docs/dom-differences.html)を確認してください。

## HTMLのエンティティ

以下のように、JSXの文字のテキストの中にHTMLのエンティティを挿入することができます。

```javascript
<div>First &middot; Second</div>
```

動的なコンテンツとしてHTMLのエンティティを表示する際に、ダブルエスケープを書いてしまうことがあるかと思いますが、Reactはデフォルトで広い範囲のXSS攻撃を防ぐために表示しようとしている全ての文字列をエスケープします。

```javascript
// 悪い例: 以下は "First &middot; Second" を表示します。
<div>{'First &middot; Second'}</div>
```

この問題を防止するためにはいくつかの方法があります。最も簡単な方法は、unicodeの文字を直でJavaScriptに記述する方法です。その際には、ファイルがUTF-8で保存されていることと、ブラウザが正しく表示できるように、UTF-8で表示する命令が正しくセットされていることを確認してください。

```javascript
<div>{'First · Second'}</div>
```

さらに安全な代替の方法は、[エンティティに対応するunicodeの数値](http://www.fileformat.info/info/unicode/char/b7/index.htm)を見つけて、JavaScriptの文字列の中でそれを使うことです。

```javascript
<div>{'First \u00b7 Second'}</div>
<div>{'First ' + String.fromCharCode(183) + ' Second'}</div>
```

配列と文字列とJSXの要素を混ぜて使うこともできます。

```javascript
<div>{['First ', <span>&middot;</span>, ' Second']}</div>
```

最後の手段として、常に[生のHTMLを挿入](/react/tips/dangerously-set-inner-html.html)することもできます。

```javascript
<div dangerouslySetInnerHTML={{'{{'}}__html: 'First &middot; Second'}} />
```


## カスタムされたHTMLの属性

HTMLの仕様書に存在しない要素をネイティブなHTML要素に渡した場合は、Reactはそれらをレンダリングしません。カスタムした属性を使う場合は、 `data-` を頭につけてください。

```javascript
<div data-custom-attribute="foo" />
```

`aria-` から始まる[Webアクセシビリティ](http://www.w3.org/WAI/intro/aria)属性はプロパティをレンダリングします。

```javascript
<div aria-hidden={true} />
```
