---
id: animation-it-IT
title: Animazioni
permalink: docs/animation-it-IT.html
prev: addons-it-IT.html
next: two-way-binding-helpers-it-IT.html
---

React offre un componente addon `ReactTransitionGroup` come una API di basso livello per le animazioni, e un `ReactCSSTransitionGroup` per implementare facilmente animazioni e transizioni CSS di base.

## API di Alto Livello: `ReactCSSTransitionGroup`

`ReactCSSTransitionGroup` è basato su `ReactTransitionGroup` ed è una maniera semplice di effettuare transizioni e animazioni CSS quando un componente React viene aggiunto o rimosso dal DOM. È ispirato all'eccellente libreria [ng-animate](http://www.nganimate.org/).

### Per Cominciare

`ReactCSSTransitionGroup` è l'interfaccia a `ReactTransitions`. Questo è un semplice elemento che racchiude tutti i componenti che desideri animare. Ecco un esempio in cui facciamo apparire e scomparire gli elementi di una lista.

```javascript{28-30}
var ReactCSSTransitionGroup = React.addons.CSSTransitionGroup;

var TodoList = React.createClass({
  getInitialState: function() {
    return {items: ['ciao', 'mondo', 'clicca', 'qui']};
  },
  handleAdd: function() {
    var newItems =
      this.state.items.concat([prompt('Scrivi del testo')]);
    this.setState({items: newItems});
  },
  handleRemove: function(i) {
    var newItems = this.state.items;
    newItems.splice(i, 1);
    this.setState({items: newItems});
  },
  render: function() {
    var items = this.state.items.map(function(item, i) {
      return (
        <div key={item} onClick={this.handleRemove.bind(this, i)}>
          {item}
        </div>
      );
    }.bind(this));
    return (
      <div>
        <button onClick={this.handleAdd}>Aggiungi Elemento</button>
        <ReactCSSTransitionGroup transitionName="example" transitionEnterTimeout={500} transitionLeaveTimeout={300}>
          {items}
        </ReactCSSTransitionGroup>
      </div>
    );
  }
});
```
> Nota:
>
> Devi fornire [l'attributo `key`](/react/docs/multiple-components.html#dynamic-children) per tutti i figli di `ReactCSSTransitionGroup`, anche quando stai visualizzando un singolo elemento. Questo è il modo in cui React determina quali figli sono stati aggiunti, rimossi, o sono rimasti.

In questo componente, quando un nuovo elemento viene aggiunto a `ReactCSSTransitionGroup` riceverà la classe CSS `example-enter` e la classe CSS `example-enter-active` allo scatto successivo. Questa è una convenzione basata sul valore della proprietà `transitionName`.

Puoi usare queste classi per scatenare una animazione o transizione CSS. Ad esempio, prova ad aggiungere questo CSS e aggiungere un nuovo elemento alla lista:

```css
.example-enter {
  opacity: 0.01;
}

.example-enter.example-enter-active {
  opacity: 1;
  transition: opacity 500ms ease-in;
}

.example-leave {
  opacity: 1;
}

.example-leave.example-leave-active {
  opacity: 0.01;
  transition: opacity 300ms ease-in;
}
```

Ti accorgerai che la durata delle animazioni devono essere specificate sia nel CSS che nel metodo render; questo suggerisce a React quando rimuovere le classi di animazione dall'elemento e -- se sta venendo rimosso -- quando rimuovere l'elemento dal DOM.

### Animare il Montaggio Iniziale

`ReactCSSTransitionGroup` fornisce la proprietà opzionale `transitionAppear`, per aggiungere una fase aggiuntiva di transizione al montaggio iniziale del componente. In genere non c'è alcuna fase di transizione al montaggio iniziale in quanto il valore predefinito di `transitionAppear` è `false`. L'esempio seguente passa la proprietà `transitionAppear` con il valore `true`.

```javascript{3-5}
  render: function() {
    return (
      <ReactCSSTransitionGroup transitionName="example" transitionAppear={true} transitionAppearTimeout={500}>
        <h1>Dissolvenza al Montaggio Iniziale</h1>
      </ReactCSSTransitionGroup>
    );
  }
```

Durante il montaggio iniziale `ReactCSSTransitionGroup` otterrà la classe CSS `example-appear` e la classe CSS `example-appear-active` allo scatto successivo.

```css
.example-appear {
  opacity: 0.01;
}

.example-appear.example-appear-active {
  opacity: 1;
  transition: opacity .5s ease-in;
}
```

Al montaggio iniziale, tutti i figli di `ReactCSSTransitionGroup` saranno marcati come `appear` ma non `enter`. Tuttavia, tutti i figli aggiunti in seguito ad un `ReactCSSTransitionGroup` esistente saranno marcati come `enter` ma non `appear`.

> Nota:
>
> La proprietà `transitionAppear` è stata aggiunta a `ReactCSSTransitionGroup` nella versione `0.13`. Per mantenere la compatibilità con le versioni precedenti, il valore predefinito è impostato a `false`.

### Classi Personalizzate

È anche possibile usare nomi di classi personalizzate per ciascuna delle fasi delle tue transizioni. Anziché passare una stringa come valore di `transitionName` puoi assegnare un oggetto che contiene i nomi delle classi `enter` o `leave`, oppure un oggetto contenente i nomi delle classi per `enter`, `enter-active`, `leave-active`e  `leave`. Se vengono forniti soltanto i nomi delle classi enter e leave, le classi per enter-active e leave-active saranno determinate aggiungendo il suffisso '-active' ai rispettivi nomi delle classi. Ecco due esempi che usano le classi personalizzate:

```javascript
  ...
  <ReactCSSTransitionGroup
    transitionName={ {
      enter: 'enter',
      enterActive: 'enterActive',
      leave: 'leave',
      leaveActive: 'leaveActive',
      appear: 'appear',
      appearActive: 'appearActive'
    } }>
    {item}
  </ReactCSSTransitionGroup>

  <ReactCSSTransitionGroup
    transitionName={ {
      enter: 'enter',
      leave: 'leave',
      appear: 'appear'
    } }>
    {item2}
  </ReactCSSTransitionGroup>
  ...
```

### I Gruppi di Animazioni Devono Essere Montati per Funzionare

Per applicare le transizioni ai suoi figli, il `ReactCSSTransitionGroup` deve già essere montato nel DOM oppure la proprietà `transitionAppear` deve essere impostata a `true`. L'esempio seguente non funziona, poiché `ReactCSSTransitionGroup` sta venendo montato assieme al nuovo elemento, anziché il nuovo elemento venire montato dentro di esso. Confronta questo esempio con la sezione precedente [Per Cominciare](#getting-started) per notare la differenza.

```javascript{12-15}
  render: function() {
    var items = this.state.items.map(function(item, i) {
      return (
        <div key={item} onClick={this.handleRemove.bind(this, i)}>
          <ReactCSSTransitionGroup transitionName="example">
            {item}
          </ReactCSSTransitionGroup>
        </div>
      );
    }, this);
    return (
      <div>
        <button onClick={this.handleAdd}>Aggiungi Elemento</button>
        {items}
      </div>
    );
  }
```

### Animare Uno o Nessun Elemento

Nell'esempio precedente, abbiamo visualizzato una lista di elementi in `ReactCSSTransitionGroup`. Tuttavia, i figli di `ReactCSSTransitionGroup` possono anche essere un solo o nessun elemento. Questo rende possibile animare un singolo elemento che viene aggiunto o rimosso. Similarmente, puoi animare un nuovo elemento che sistituisce l'elemento corrente. Ad esempio, possiamo implementare un semplice carosello di immagini come segue:

```javascript{10-12}
var ReactCSSTransitionGroup = React.addons.CSSTransitionGroup;

var ImageCarousel = React.createClass({
  propTypes: {
    imageSrc: React.PropTypes.string.isRequired
  },
  render: function() {
    return (
      <div>
        <ReactCSSTransitionGroup transitionName="carousel" transitionEnterTimeout={300} transitionLeaveTimeout={300}>
          <img src={this.props.imageSrc} key={this.props.imageSrc} />
        </ReactCSSTransitionGroup>
      </div>
    );
  }
});
```

### Disattivare le Animazioni

Se lo desideri, puoi disattivare le animazioni `enter` o `leave`. Ad esempio, a volte potresti volere un'animazione per `enter` ma non una per `leave`, ma `ReactCSSTransitionGroup` attende che l'animazione sia completata prima di rimuovere il tuo nodo DOM. Puoi aggiungere le proprietà `transitionEnter={false}` o `transitionLeave={false}` a `ReactCSSTransitionGroup` per disattivare le rispettive animazioni.

> Nota:
>
> Quando si usa `ReactCSSTransitionGroup`, non c'è alcun modo per cui i tuoi componenti vengano avvisati quando la transizione è terminata o per effettuare una logica complessa durante l'animazione. Se vuoi un controllo più fine, puoi usare la API di basso livello `ReactTransitionGroup` che fornisce degli hook che puoi utilizzare per effettuare transizioni personalizzate.

## API di Basso Livello: `ReactTransitionGroup`

`ReactTransitionGroup` è la base per le animazioni. È accessibile come `React.addons.TransitionGroup`. Quando vi sono aggiunti o rimossi dichiarativamente dei figli (come negli esempi precedenti) degli speciali hook del ciclo di vita sono invocati su di essi.

### `componentWillAppear(callback)`

Viene chiamato allo stesso momento di `componentDidMount()` per i componenti inizialmente montati in un `TransitionGroup`. Bloccherà l'esecuzione di altre animazioni finché `callback` non viene chiamata. Viene chiamata solo durante il rendering iniziale di un `TransitionGroup`.

### `componentDidAppear()`

Viene chiamato dopo che la funzione `callback` passata a `componentWillAppear` è stata chiamata.

### `componentWillEnter(callback)`

Viene chiamato allo stesso momento di `componentDidMount()` per i componenti aggiunti ad un `TransitionGroup` esistente. Bloccherà l'esecuzione di altre animazioni finché `callback` non viene chiamata. Non viene chiamata durante il rendering iniziale di un `TransitionGroup`.

### `componentDidEnter()`

Viene chiamato dopo che la funzione `callback` passata a `componentWillEnter` è stata chiamata.

### `componentWillLeave(callback)`

Viene chiamato quando il figlio è stato rimosso dal `ReactTransitionGroup`. Nonostante il figlio sia stato rimosso, `ReactTransitionGroup` lo manterrà nel DOM finché `callback` non viene chiamata.

### `componentDidLeave()`

Viene chiamato quando la `callback` `willLeave` viene chiamata (contemporaneamente a `componentWillUnmount`).

### Rendering di un Componente Diverso

In maniera predefinita, `ReactTransitionGroup` viene visualizzato come uno `span`. Puoi cambiare questo comportamento fornento una proprietà `component`. Ad esempio, ecco come puoi visualizzare un `<ul>`:

```javascript{1}
<ReactTransitionGroup component="ul">
  ...
</ReactTransitionGroup>
```

È possibile utilizzare ciascun componente DOM che React può visualizzare. Tuttavia, `component` non deve necessariamente essere un componente DOM. Può infatti essere qualunque componente React; anche componenti scritti da te!

> Nota:
>
> Prima della v0.12, quando venivano usati componenti DOM, la proprietà `component` doveva essere un riferimento a `React.DOM.*`. Dal momento che il componente è semplicemente passato a `React.createElement`, deve ora essere una stringa. Per componenti compositi si deve passare il metodo factory.

Ciascuna proprietà aggiuntiva definita dall'utente diverrà una proprietà del componente visualizzato. Ad esempio, ecco come visualizzeresti un `<ul>` con una classe CSS:

```javascript{1}
<ReactTransitionGroup component="ul" className="animated-list">
  ...
</ReactTransitionGroup>
```
