---
id: two-way-binding-helpers-it-IT
title: Helper Per Binding Bidirezionali
permalink: docs/two-way-binding-helpers-it-IT.html
prev: animation-it-IT.html
next: test-utils-it-IT.html
---

`ReactLink` è una maniera semplice di esprimere binding bidirezionali con React.

> Nota:
>
> Se hai poca esperienza del framework, nota che `ReactLink` non è necessario per molte applicazioni e dovrebbe essere utilizzato con cautela.

In React, i dati fluiscono in una direzione: dal proprietario ai figli. Questo poiché i dati fluiscono in una sola direzione nel [modello di computazione di Von Neumann](https://en.wikipedia.org/wiki/Von_Neumann_architecture). Puoi pensare ad esso come "binding unidirezionale dei dati."

Tuttavia, esistono parecchie applicazioni che richiedono di leggere dati e farli fluire nuovamente nel tuo programma. Ad esempio, quando sviluppi dei moduli, vorrai spesso aggiornare uno `state` di React quando ricevi un input dall'utente. O forse vuoi effettuare il layout in JavaScript e reagire ai cambiamenti nelle dimensioni di alcuni elementi DOM.

In React, questo verrebbe implementato ascoltando un evento "change", leggendo la tua fonte di dati (solitamente il DOM) e chiamando `setState()` su uno dei tuoi componenti. "Chiudere il ciclo del flusso dei dati" esplicitamente conduce a programmi più comprensibili e mantenibili. Consulta [la nostra documentazione sui moduli](/react/docs/forms.html) per maggiori informazioni.

Il binding bidirezionale -- assicurarsi implicitamente che alcuni valori nel DOM siano sempre consistenti con degli `state` in  React -- è più conciso e supporta un'ampia varietà di applicazioni. Abbiamo fornito `ReactLink`: zucchero sintattico per impostare il pattern del ciclo del flusso di dati descritto in predecenza, ovvero "collegare" una fonte di dati con lo `state` di React.

> Nota:
>
> `ReactLink` è soltanto uno strato di astrazione e convenzioni attorno al pattern `onChange`/`setState()`. Non cambia fondamentalmente la maniera in cui i dati fluiscono all'interno della tua applicazione React.

## ReactLink: Prima e Dopo

Ecco un semplice esempio di modulo che non utilizza `ReactLink`:

```javascript
var NoLink = React.createClass({
  getInitialState: function() {
    return {message: 'Ciao!'};
  },
  handleChange: function(event) {
    this.setState({message: event.target.value});
  },
  render: function() {
    var message = this.state.message;
    return <input type="text" value={message} onChange={this.handleChange} />;
  }
});
```

Ciò funziona molto bene e il flusso di dati è molto chiaro. Tuttavia, in presenza di un gran numero di campi del modulo può risultare assai prolisso. Utilizziamo `ReactLink` per risparmiarci la scrittura di un po' di codice:

```javascript{2,7}
var WithLink = React.createClass({
  mixins: [React.addons.LinkedStateMixin],
  getInitialState: function() {
    return {message: 'Ciao!'};
  },
  render: function() {
    return <input type="text" valueLink={this.linkState('message')} />;
  }
});
```

`LinkedStateMixin` aggiunge un metodo chiamato `linkState()` al tuo componente React. `linkState()` restituisce un oggetto `ReactLink` contenente il valore attuae dello stato React e una callback per cambiarlo.

Gli oggetti `ReactLink` possono essere passati su e giù nell'albero come proprietà, quindi è facile (ed esplicito) impostare un binding bidirezionale tra un componente in profondità nella gerarchia e dello stato che si trova più in alto nella gerarchia.

Nota che i checkbox hanno un comportamento speciale riguardo il loro attributo `value`, che è il valore che sarà inviato all'inoltro del modulo se il checkbox è spuntato (il valore predefinito è `on`). L'attributo `value` non è aggiornato quando il checkbox viene spuntato o deselezionato. Per i checkbox occorre usare `checkedLink` anziché `valueLink`:
```
<input type="checkbox" checkedLink={this.linkState('booleanValue')} />
```


## Dietro le Quinte

Ci sono due ambiti in `ReactLink`: il posto in cui crei l'istanza di `ReactLink` e il posto in cui la utilizzi. Per dimostrare quanto sia semplice usare `ReactLink`, riscriviamo ciascun ambito separatamente perché sia più esplicito.

### ReactLink Senza LinkedStateMixin

```javascript{5-7,9-12}
var WithoutMixin = React.createClass({
  getInitialState: function() {
    return {message: 'Ciao!'};
  },
  handleChange: function(newValue) {
    this.setState({message: newValue});
  },
  render: function() {
    var valueLink = {
      value: this.state.message,
      requestChange: this.handleChange
    };
    return <input type="text" valueLink={valueLink} />;
  }
});
```

Come puoi vedere, gli oggetti `ReactLink` sono semplici oggetti che hanno due proprietà: `value` e `requestChange`. `LinkedStateMixin` è altrettanto semplice: popola semplicemente questi campi con un valore da `this.state` e una callback che invoca `this.setState()`.

### ReactLink Senza valueLink

```javascript
var WithoutLink = React.createClass({
  mixins: [React.addons.LinkedStateMixin],
  getInitialState: function() {
    return {message: 'Ciao!'};
  },
  render: function() {
    var valueLink = this.linkState('message');
    var handleChange = function(e) {
      valueLink.requestChange(e.target.value);
    };
    return <input type="text" value={valueLink.value} onChange={handleChange} />;
  }
});
```

La proprietà `valueLink` è anche abbastanza semplice. Gestisce semplicemente l'evento `onChange` e invoca `this.props.valueLink.requestChange()`, e inoltre utilizza `this.props.valueLink.value` anziché `this.props.value`. Tutto qua!
