---
id: two-way-binding-helpers
title: 2ウェイバインディングのヘルパ
permalink: docs/two-way-binding-helpers-ja-JP.html
prev: animation-ja-JP.html
next: test-utils-ja-JP.html
---

`ReactLink` はReactで2ウェイバインディングを表現する簡単な方法です。

> 注意:
> もしあなたがフレームワークについてあまりよく知らないのであれば、 `ReactLink` は多くのアプリケーションには必要なく、慎重に使うべきであることに注意してください。

Reactでは所有者から子要素へと、データの流れは一方向です。これはデータが[Von Neumannのコンピューティングモデル](https://en.wikipedia.org/wiki/Von_Neumann_architecture)によって一方向にのみ流れるからです。これを「1ウェイデータバインディング」だと考えることができます。

しかし、データやプログラムに戻る流れを読む必要が有るアプリケーションもたくさんあります。例えば、フォームを作る際に、ユーザのインプットを受け取った時には、Reactの `state` を更新したいと思うことはよくあるでしょう。または、JavaScriptでレイアウトを形作ったり、DOM要素のサイズの変化に反応したいと思うでしょう。

Reactでは、「変更」のイベントを検知し、データソース（普通はDOMです）を読み、コンポーネントのうちの1つの上で `setState()` を呼ぶことでこの要求を満たすことができます。「データフローのループを止めること」は理解しやすく、維持しやすいプログラムを明確に導きます。詳細な情報については、[フォームのドキュメント](/react/docs/forms-ja-JP.html)をご覧ください。

2ウェイバインディングはDOMの値が常にReactの `state` と一致していることを暗黙に強制しますが、簡潔で、多くの種類のアプリケーションをサポートします。上で説明されているような共通のデータフローループパターンのセットアップや、データソースをReactの `state` に「接続する」ための糖衣構文である `ReactLink` が提供されています。

> 注意:
> `ReactLink` は薄いラッパーであり、 `onChange`/`setState()` パターンにおける習慣です。Reactアプリケーションのデータフローの方法を根本から変えるようなものではありません。

## ReactLink: ビフォーアフター

以下が、 `ReactLink` を使用しない単純なフォームの例です。

```javascript
var NoLink = React.createClass({
  getInitialState: function() {
    return {message: 'Hello!'};
  },
  handleChange: function(event) {
    this.setState({message: event.target.value});
  },
  render: function() {
    var message = this.state.message;
    return <input type="text" value={message} onChange={this.handleChange} />;
  }
});
```

これはとてもよく動き、データがどう流れているかとても明確です。しかし、たくさんのフォームのフィールドがあった場合、少し冗長になります。以下のように、 `ReactLink` を使うことでタイピング量が少なくて済みます。

```javascript{2,7}
var WithLink = React.createClass({
  mixins: [React.addons.LinkedStateMixin],
  getInitialState: function() {
    return {message: 'Hello!'};
  },
  render: function() {
    return <input type="text" valueLink={this.linkState('message')} />;
  }
});
```

`LinkedStateMixin` は `linkState()` と呼ばれるReactコンポーネントにメソッドを追加します。
`linkState()` はReactのステータスの現在の値と、それを変更するコールバックを持った `ReactLink` オブジェクトを返します。

`ReactLink` オブジェクトはプロパティとして、木構造の上や下に渡される可能性があります。だから、階層の深い位置にいるコンポーネントと階層の高い位置にいるステータスの間の2ウェイバインディングをセットアップすることは簡単（で、明確）です。

チェックボックスはその `value` 属性に対しての特別な態度を持っていることに注意してください。それは、チェックボックスがチェックされている（デフォルトで `on` ）場合にフォームのサブミットで送信される値です。 `value` 属性はチェックボックスがチェックされていても、チェックされていなくても、更新されません。チェックボックスについては、`valueLink` の代わりに、 `checkedLink` を使うべきです。以下のように。

```
<input type="checkbox" checkedLink={this.linkState('booleanValue')} />
```


## フードの下

`ReactLink` には2つの側面があります。 `ReactLink` のインスタンスを作成する場所と、それを使う場所です。 `ReactLink` がどれだけ単純か証明するために、それぞれの側面を分けて、明確に再度記述してみましょう。

### LinkedStateMixinを使わないReactLink

```javascript{5-7,9-12}
var WithoutMixin = React.createClass({
  getInitialState: function() {
    return {message: 'Hello!'};
  },
  handleChange: function(newValue) {
    this.setState({message: newValue});
  },
  render: function() {
    var valueLink = {
      value: this.state.message,
      requestChange: this.handleChange
    };
    return <input type="text" valueLink={valueLink} />;
  }
});
```

今まで見てきたように `ReactLink` オブジェクトは `value` と `requestChange` プロパティだけを持ったとても単純なオブジェクトです。そして、 `LinkedStateMixin` も同様に単純です。それらのフィールドを `this.state` の値と、 `this.setState()` を呼ぶコールバックで満たします。

### valueLinkを使わないReactLink

```javascript
var WithoutLink = React.createClass({
  mixins: [React.addons.LinkedStateMixin],
  getInitialState: function() {
    return {message: 'Hello!'};
  },
  render: function() {
    var valueLink = this.linkState('message');
    var handleChange = function(e) {
      valueLink.requestChange(e.target.value);
    };
    return <input type="text" value={valueLink.value} onChange={handleChange} />;
  }
});
```

`valueLink` プロパティも同様にとても単純です。単純に `onChange` イベントをハンドルし、 `this.props.valueLink.requestChange()` を呼び、 `this.props.value` の代わりに `this.props.valueLink.value` を使用します。それだけです！
