---
id: test-utils
title: Test Utilities
permalink: docs/test-utils.html
prev: two-way-binding-helpers.html
next: clone-with-props.html
---

`ReactTestUtils` makes it easy to test React components in the testing framework of your choice (we use [Jest](https://facebook.github.io/jest/)).

```
var ReactTestUtils = require('react-addons-test-utils');
```

> Note:
> 
> Airbnb has released a testing utility called Enzyme, which makes it easy to assert, manipulate, and traverse your React Components' output. If you're deciding on a unit testing library, it's worth checking out: [http://airbnb.io/enzyme/](http://airbnb.io/enzyme/)

### Simulate

```javascript
Simulate.{eventName}(
  DOMElement element,
  [object eventData]
)
```

Simulate an event dispatch on a DOM node with optional `eventData` event data. **This is possibly the single most useful utility in `ReactTestUtils`.**

**Clicking an element**

```javascript
// <button ref="button">...</button>
var node = this.refs.button;
ReactTestUtils.Simulate.click(node);
```

**Changing the value of an input field and then pressing ENTER.**

```javascript
// <input ref="input" />
var node = this.refs.input;
node.value = 'giraffe';
ReactTestUtils.Simulate.change(node);
ReactTestUtils.Simulate.keyDown(node, {key: "Enter", keyCode: 13, which: 13});
```

*Note that you will have to provide any event property that you're using in your component (e.g. keyCode, which, etc...) as React is not creating any of these for you.*

`Simulate` has a method for [every event that React understands](/react/docs/events.html#supported-events).

### renderIntoDocument

```javascript
ReactComponent renderIntoDocument(
  ReactElement instance
)
```

Render a component into a detached DOM node in the document. **This function requires a DOM.**

> Note:
>
> You will need to have `window`, `window.document` and `window.document.createElement`
 globally available **before** you import React. Otherwise React will think it can't access the DOM and methods like `setState` won't work.

### mockComponent

```javascript
object mockComponent(
  function componentClass,
  [string mockTagName]
)
```

Pass a mocked component module to this method to augment it with useful methods that allow it to be used as a dummy React component. Instead of rendering as usual, the component will become a simple `<div>` (or other tag if `mockTagName` is provided) containing any provided children.

### isElement

```javascript
boolean isElement(
  ReactElement element
)
```

Returns `true` if `element` is any ReactElement.

### isElementOfType

```javascript
boolean isElementOfType(
  ReactElement element,
  function componentClass
)
```

Returns `true` if `element` is a ReactElement whose type is of a React `componentClass`.

### isDOMComponent

```javascript
boolean isDOMComponent(
  ReactComponent instance
)
```

Returns `true` if `instance` is a DOM component (such as a `<div>` or `<span>`).

### isCompositeComponent

```javascript
boolean isCompositeComponent(
  ReactComponent instance
)
```

Returns `true` if `instance` is a composite component (created with `React.createClass()`).

### isCompositeComponentWithType

```javascript
boolean isCompositeComponentWithType(
  ReactComponent instance,
  function componentClass
)
```

Returns `true` if `instance` is a composite component (created with `React.createClass()`) whose type is of a React `componentClass`.

### findAllInRenderedTree

```javascript
array findAllInRenderedTree(
  ReactComponent tree,
  function test
)
```

Traverse all components in `tree` and accumulate all components where `test(component)` is `true`. This is not that useful on its own, but it's used as a primitive for other test utils.

### scryRenderedDOMComponentsWithClass

```javascript
array scryRenderedDOMComponentsWithClass(
  ReactComponent tree, 
  string className
)
```

Finds all DOM elements of components in the rendered tree that are DOM components with the class name matching `className`.

### findRenderedDOMComponentWithClass

```javascript
DOMElement findRenderedDOMComponentWithClass(
  ReactComponent tree,
  string className
)
```

Like `scryRenderedDOMComponentsWithClass()` but expects there to be one result, and returns that one result, or throws exception if there is any other number of matches besides one.

### scryRenderedDOMComponentsWithTag

```javascript
array scryRenderedDOMComponentsWithTag(
  ReactComponent tree,
  string tagName
)
```

Finds all DOM elements of components in the rendered tree that are DOM components with the tag name matching `tagName`.

### findRenderedDOMComponentWithTag

```javascript
DOMElement findRenderedDOMComponentWithTag(
  ReactComponent tree,
  string tagName
)
```

Like `scryRenderedDOMComponentsWithTag()` but expects there to be one result, and returns that one result, or throws exception if there is any other number of matches besides one.

### scryRenderedComponentsWithType

```javascript
array scryRenderedComponentsWithType(
  ReactComponent tree,
  function componentClass
)
```

Finds all instances of components with type equal to `componentClass`.

### findRenderedComponentWithType

```javascript
ReactComponent findRenderedComponentWithType(
  ReactComponent tree, 
  function componentClass
)
```

Same as `scryRenderedComponentsWithType()` but expects there to be one result and returns that one result, or throws exception if there is any other number of matches besides one.

## Shallow rendering

Shallow rendering is an experimental feature that lets you render a component "one level deep" and assert facts about what its render method returns, without worrying about the behavior of child components, which are not instantiated or rendered. This does not require a DOM.

```javascript
ReactShallowRenderer createRenderer()
```

Call this in your tests to create a shallow renderer. You can think of this as a "place" to render the component you're testing, where it can respond to events and update itself.

```javascript
shallowRenderer.render(
  ReactElement element
)
```

Similar to `ReactDOM.render`.

```javascript
ReactElement shallowRenderer.getRenderOutput()
```

After `render` has been called, returns shallowly rendered output. You can then begin to assert facts about the output. For example, if your component's render method returns:

```javascript
<div>
  <span className="heading">Title</span>
  <Subcomponent foo="bar" />
</div>
```

Then you can assert:

```javascript
var renderer = ReactTestUtils.createRenderer();
result = renderer.getRenderOutput();
expect(result.type).toBe('div');
expect(result.props.children).toEqual([
  <span className="heading">Title</span>,
  <Subcomponent foo="bar" />
]);
```

Shallow testing currently has some limitations, namely not supporting refs. We're releasing this feature early and would appreciate the React community's feedback on how it should evolve.
