---
id: update-it-IT
title: Helper per l'Immutabilità
permalink: docs/update-it-IT.html
prev: create-fragment-it-IT.html
next: pure-render-mixin-it-IT.html
---

React ti permette di usare qualunque stile per la gestione dei dati che desideri, incluso la mutazione. Tuttavia, se puoi usare dati immutabili in parti critiche per le prestazioni della tua applicazione è facile implementare rapidamente un metodo `shouldComponentUpdate()` che aumenta significativamente la velocità della tua applicazione.

Avere a che fare con dati immutabili in JavaScript è più difficile che in linguaggi progettati a tale scopo, come [Clojure](http://clojure.org/). Tuttavia, abbiamo fornito un semplice helper per l'immutabilità, `update()`, che rende avere a che fare con questo tipo di dati molto più semplice, *senza* cambiare fondamentalmente la rappresentazione dei tuoi dati. Se puoi anche dare un'occhiata alla libreria [Immutable-js](https://facebook.github.io/immutable-js/docs/) di Facebook e la sezione [Prestazioni Avanzate](/react/docs/advanced-performance.html) per maggiori dettagli su Immutable-js.

## L'idea fondamentale

Se muti i tuoi dati nella seguente maniera:

```js
myData.x.y.z = 7;
// oppure...
myData.a.b.push(9);
```

non hai modo di determinare quali dati siano cambiati dal momento che la copia precedente è stata sovrascritta. Invece, devi creare una nuova copia di `myData` e cambiare solo le parti che vanno cambiate. Allora puoi confrontare la vecchia copia di `myData` con la nuova in `shouldComponentUpdate()` usando l'operatore di uguaglianza stretta `===`:

```js
var newData = deepCopy(myData);
newData.x.y.z = 7;
newData.a.b.push(9);
```

Sfortunatamente, le copie profonde sono costose, e a volte impossibili. Puoi alleviare questa limitazione copiando soltanto gli oggetti che devono essere cambiati e riutilizzando gli oggetti che nonsono cambiati. Sfortunatamente, nel JavaScript odierno questa può essere un'operazione difficoltosa:

```js
var newData = extend(myData, {
  x: extend(myData.x, {
    y: extend(myData.x.y, {z: 7}),
  }),
  a: extend(myData.a, {b: myData.a.b.concat(9)})
});
```

Mentre questo codice ha prestazioni accettabili (dal momento che effettua soltanto una copia superficiale di `log n` oggetti e riutilizza i rimanenti), è una gran scocciatura da scrivere. Guarda quanta ripetizione! Questo non è soltanto fastidioso, ma offre una grande superficie di attacco per i bachi.

`update()` fornisce un semplice zucchero sintattico attorno a questo pattern per rendere più semplice la scrittura di questo codice. Questo codice diventa:

```js
var newData = React.addons.update(myData, {
  x: {y: {z: {$set: 7}}},
  a: {b: {$push: [9]}}
});
```

Mentre la sintassi richiede qualche tempo per abituarsi (anche se è ispirata dal [linguaggio di query di MongoDB](http://docs.mongodb.org/manual/core/crud-introduction/#query)) non c'è ridondanza, può essere analizzato staticamente e non richiede la scrittura di più codice della versione mutativa.

Le chiavi con il prefisso `$` sono chiamate *comandi*. La struttura dati che stanno "mutando" viene chiamata *bersaglio*.

## Comandi disponibili

  * `{$push: array}` invoca `push()` sul bersagio passando ciascun elemento di `array`.
  * `{$unshift: array}` invoca `unshift()` sul bersagio passando ciascun elemento di `array`.
  * `{$splice: array of arrays}` per ogni elemento di `arrays` invoca `splice()` sul bersaglio con i parametri forniti dall'elemento.
  * `{$set: any}` sostituisce l'intero bersaglio.
  * `{$merge: object}` unisce le chiavi di `object` con il bersaglio.
  * `{$apply: function}` passa il valore attuale alla funzione e lo aggiorna con il nuovo valore da essa restituito.

## Esempi

### Semplice inserimento in coda

```js
var initialArray = [1, 2, 3];
var newArray = update(initialArray, {$push: [4]}); // => [1, 2, 3, 4]
```
`initialArray` is still `[1, 2, 3]`.

### Collezioni annidate

```js
var collection = [1, 2, {a: [12, 17, 15]}];
var newCollection = update(collection, {2: {a: {$splice: [[1, 1, 13, 14]]}}});
// => [1, 2, {a: [12, 13, 14, 15]}]
```
Questo accede all'indice `2` di `collection`, alla chiave `a`, ed effettua lo splice di un elemento a partire dall'indice `1` (per rimuovere `17`) e al contempo inserisce `13` e `14`.

### Aggiornare un valore basandosi sul suo valore attuale

```js
var obj = {a: 5, b: 3};
var newObj = update(obj, {b: {$apply: function(x) {return x * 2;}}});
// => {a: 5, b: 6}
// Questa è una forma equivalente, ma diventa prolissa per profonde collezioni annidate:
var newObj2 = update(obj, {b: {$set: obj.b * 2}});
```

### Unione (superficiale)

```js
var obj = {a: 5, b: 3};
var newObj = update(obj, {$merge: {b: 6, c: 7}}); // => {a: 5, b: 6, c: 7}
```
