# typed: false

# DO NOT EDIT MANUALLY
# This is an autogenerated file for types exported from the `net-imap` gem.
# Please instead update this file by running `bin/tapioca gem net-imap`.


# Net::IMAP implements Internet Message Access Protocol (\IMAP) client
# functionality.  The protocol is described
# in {IMAP4rev1 [RFC3501]}[https://tools.ietf.org/html/rfc3501]
# and {IMAP4rev2 [RFC9051]}[https://tools.ietf.org/html/rfc9051].
#
# == \IMAP Overview
#
# An \IMAP client connects to a server, and then authenticates
# itself using either #authenticate or #login.  Having
# authenticated itself, there is a range of commands
# available to it.  Most work with mailboxes, which may be
# arranged in an hierarchical namespace, and each of which
# contains zero or more messages.  How this is implemented on
# the server is implementation-dependent; on a UNIX server, it
# will frequently be implemented as files in mailbox format
# within a hierarchy of directories.
#
# To work on the messages within a mailbox, the client must
# first select that mailbox, using either #select or #examine
# (for read-only access).  Once the client has successfully
# selected a mailbox, they enter the "_selected_" state, and that
# mailbox becomes the _current_ mailbox, on which mail-item
# related commands implicitly operate.
#
# === Sequence numbers and UIDs
#
# Messages have two sorts of identifiers: message sequence
# numbers and UIDs.
#
# Message sequence numbers number messages within a mailbox
# from 1 up to the number of items in the mailbox.  If a new
# message arrives during a session, it receives a sequence
# number equal to the new size of the mailbox.  If messages
# are expunged from the mailbox, remaining messages have their
# sequence numbers "shuffled down" to fill the gaps.
#
# To avoid sequence number race conditions, servers must not expunge messages
# when no command is in progress, nor when responding to #fetch, #store, or
# #search.  Expunges _may_ be sent during any other command, including
# #uid_fetch, #uid_store, and #uid_search.  The #noop and #idle commands are
# both useful for this side-effect: they allow the server to send all mailbox
# updates, including expunges.
#
# UIDs, on the other hand, are permanently guaranteed not to
# identify another message within the same mailbox, even if
# the existing message is deleted.  UIDs are required to
# be assigned in ascending (but not necessarily sequential)
# order within a mailbox; this means that if a non-IMAP client
# rearranges the order of mail items within a mailbox, the
# UIDs have to be reassigned.  An \IMAP client thus cannot
# rearrange message orders.
#
# === Examples of Usage
#
# ==== List sender and subject of all recent messages in the default mailbox
#
#   imap = Net::IMAP.new('mail.example.com')
#   imap.authenticate('PLAIN', 'joe_user', 'joes_password')
#   imap.examine('INBOX')
#   imap.search(["RECENT"]).each do |message_id|
#     envelope = imap.fetch(message_id, "ENVELOPE")[0].attr["ENVELOPE"]
#     puts "#{envelope.from[0].name}: \t#{envelope.subject}"
#   end
#
# ==== Move all messages from April 2003 from "Mail/sent-mail" to "Mail/sent-apr03"
#
#   imap = Net::IMAP.new('mail.example.com')
#   imap.authenticate('PLAIN', 'joe_user', 'joes_password')
#   imap.select('Mail/sent-mail')
#   if not imap.list('Mail/', 'sent-apr03')
#     imap.create('Mail/sent-apr03')
#   end
#   imap.search(["BEFORE", "30-Apr-2003", "SINCE", "1-Apr-2003"]).each do |message_id|
#     imap.copy(message_id, "Mail/sent-apr03")
#     imap.store(message_id, "+FLAGS", [:Deleted])
#   end
#   imap.expunge
#
# == Capabilities
#
# Most Net::IMAP methods do not _currently_ modify their behaviour according
# to the server's advertised #capabilities.  Users of this class must check
# that the server is capable of extension commands or command arguments before
# sending them.  Special care should be taken to follow the #capabilities
# requirements for #starttls, #login, and #authenticate.
#
# See #capable?, #auth_capable?, #capabilities, #auth_mechanisms to discover
# server capabilities.  For relevant capability requirements, see the
# documentation on each \IMAP command.
#
#   imap = Net::IMAP.new("mail.example.com")
#   imap.capable?(:IMAP4rev1) or raise "Not an IMAP4rev1 server"
#   imap.capable?(:starttls)  or raise "Cannot start TLS"
#   imap.starttls
#
#   if imap.auth_capable?("PLAIN")
#     imap.authenticate "PLAIN", username, password
#   elsif !imap.capability?("LOGINDISABLED")
#     imap.login username, password
#   else
#     raise "No acceptable authentication mechanisms"
#   end
#
#   # Support for "UTF8=ACCEPT" implies support for "ENABLE"
#   imap.enable :utf8 if imap.capable?("UTF8=ACCEPT")
#
#   namespaces  = imap.namespace if imap.capable?(:namespace)
#   mbox_prefix = namespaces&.personal&.first&.prefix || ""
#   mbox_delim  = namespaces&.personal&.first&.delim  || "/"
#   mbox_path   = prefix + %w[path to my mailbox].join(delim)
#   imap.create mbox_path
#
# === Basic IMAP4rev1 capabilities
#
# IMAP4rev1 servers must advertise +IMAP4rev1+ in their capabilities list.
# IMAP4rev1 servers must _implement_ the +STARTTLS+, <tt>AUTH=PLAIN</tt>,
# and +LOGINDISABLED+ capabilities.  See #starttls, #login, and #authenticate
# for the implications of these capabilities.
#
# === Caching +CAPABILITY+ responses
#
# Net::IMAP automatically stores and discards capability data according to the
# the requirements and recommendations in
# {IMAP4rev2 §6.1.1}[https://www.rfc-editor.org/rfc/rfc9051#section-6.1.1],
# {§6.2}[https://www.rfc-editor.org/rfc/rfc9051#section-6.2], and
# {§7.1}[https://www.rfc-editor.org/rfc/rfc9051#section-7.1].
# Use #capable?, #auth_capable?, or #capabilities to use this cache and avoid
# sending the #capability command unnecessarily.
#
# The server may advertise its initial capabilities using the +CAPABILITY+
# ResponseCode in a +PREAUTH+ or +OK+ #greeting.  When TLS has started
# (#starttls) and after authentication (#login or #authenticate), the server's
# capabilities may change and cached capabilities are discarded.  The server
# may send updated capabilities with an +OK+ TaggedResponse to #login or
# #authenticate, and these will be cached by Net::IMAP.  But the
# TaggedResponse to #starttls MUST be ignored--it is sent before TLS starts
# and is unprotected.
#
# When storing capability values to variables, be careful that they are
# discarded or reset appropriately, especially following #starttls.
#
# === Using IMAP4rev1 extensions
#
# See the {IANA IMAP4 capabilities
# registry}[http://www.iana.org/assignments/imap4-capabilities] for a list of
# all standard capabilities, and their reference RFCs.
#
# IMAP4rev1 servers must not activate behavior that is incompatible with the
# base specification until an explicit client action invokes a capability,
# e.g. sending a command or command argument specific to that capability.
# Servers may send data with backward compatible behavior, such as response
# codes or mailbox attributes, at any time without client action.
#
# Invoking capabilities which are unknown to Net::IMAP may cause unexpected
# behavior and errors.  For example, ResponseParseError is raised when
# unknown response syntax is received.  Invoking commands or command
# parameters that are unsupported by the server may raise NoResponseError,
# BadResponseError, or cause other unexpected behavior.
#
# Some capabilities must be explicitly activated using the #enable command.
# See #enable for details.
#
# == Thread Safety
#
# Net::IMAP supports concurrent threads. For example,
#
#   imap = Net::IMAP.new("imap.foo.net", "imap2")
#   imap.authenticate("scram-md5", "bar", "password")
#   imap.select("inbox")
#   fetch_thread = Thread.start { imap.fetch(1..-1, "UID") }
#   search_result = imap.search(["BODY", "hello"])
#   fetch_result = fetch_thread.value
#   imap.disconnect
#
# This script invokes the FETCH command and the SEARCH command concurrently.
#
# == Errors
#
# An \IMAP server can send three different types of responses to indicate
# failure:
#
# NO:: the attempted command could not be successfully completed.  For
#      instance, the username/password used for logging in are incorrect;
#      the selected mailbox does not exist; etc.
#
# BAD:: the request from the client does not follow the server's
#       understanding of the \IMAP protocol.  This includes attempting
#       commands from the wrong client state; for instance, attempting
#       to perform a SEARCH command without having SELECTed a current
#       mailbox.  It can also signal an internal server
#       failure (such as a disk crash) has occurred.
#
# BYE:: the server is saying goodbye.  This can be part of a normal
#       logout sequence, and can be used as part of a login sequence
#       to indicate that the server is (for some reason) unwilling
#       to accept your connection.  As a response to any other command,
#       it indicates either that the server is shutting down, or that
#       the server is timing out the client connection due to inactivity.
#
# These three error response are represented by the errors
# Net::IMAP::NoResponseError, Net::IMAP::BadResponseError, and
# Net::IMAP::ByeResponseError, all of which are subclasses of
# Net::IMAP::ResponseError.  Essentially, all methods that involve
# sending a request to the server can generate one of these errors.
# Only the most pertinent instances have been documented below.
#
# Because the IMAP class uses Sockets for communication, its methods
# are also susceptible to the various errors that can occur when
# working with sockets.  These are generally represented as
# Errno errors.  For instance, any method that involves sending a
# request to the server and/or receiving a response from it could
# raise an Errno::EPIPE error if the network connection unexpectedly
# goes down.  See the socket(7), ip(7), tcp(7), socket(2), connect(2),
# and associated man pages.
#
# Finally, a Net::IMAP::DataFormatError is thrown if low-level data
# is found to be in an incorrect format (for instance, when converting
# between UTF-8 and UTF-16), and Net::IMAP::ResponseParseError is
# thrown if a server response is non-parseable.
#
# == What's here?
#
# * {Connection control}[rdoc-ref:Net::IMAP@Connection+control+methods]
# * {Server capabilities}[rdoc-ref:Net::IMAP@Server+capabilities]
# * {Handling server responses}[rdoc-ref:Net::IMAP@Handling+server+responses]
# * {Core IMAP commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands]
#   * {for any state}[rdoc-ref:Net::IMAP@Any+state]
#   * {for the "not authenticated" state}[rdoc-ref:Net::IMAP@Not+Authenticated+state]
#   * {for the "authenticated" state}[rdoc-ref:Net::IMAP@Authenticated+state]
#   * {for the "selected" state}[rdoc-ref:Net::IMAP@Selected+state]
#   * {for the "logout" state}[rdoc-ref:Net::IMAP@Logout+state]
# * {IMAP extension support}[rdoc-ref:Net::IMAP@IMAP+extension+support]
#
# === Connection control methods
#
# - Net::IMAP.new: Creates a new \IMAP client which connects immediately and
#   waits for a successful server greeting before the method returns.
# - #starttls: Asks the server to upgrade a clear-text connection to use TLS.
# - #logout: Tells the server to end the session. Enters the "_logout_" state.
# - #disconnect: Disconnects the connection (without sending #logout first).
# - #disconnected?: True if the connection has been closed.
#
# === Server capabilities
#
# - #capable?: Returns whether the server supports a given capability.
# - #capabilities: Returns the server's capabilities as an array of strings.
# - #auth_capable?: Returns whether the server advertises support for a given
#   SASL mechanism, for use with #authenticate.
# - #auth_mechanisms: Returns the #authenticate SASL mechanisms which
#   the server claims to support as an array of strings.
# - #clear_cached_capabilities: Clears cached capabilities.
#
#   <em>The capabilities cache is automatically cleared after completing
#   #starttls, #login, or #authenticate.</em>
# - #capability: Sends the +CAPABILITY+ command and returns the #capabilities.
#
#   <em>In general, #capable? should be used rather than explicitly sending a
#   +CAPABILITY+ command to the server.</em>
#
# === Handling server responses
#
# - #greeting: The server's initial untagged response, which can indicate a
#   pre-authenticated connection.
# - #responses: Yields unhandled UntaggedResponse#data and <em>non-+nil+</em>
#   ResponseCode#data.
# - #clear_responses: Deletes unhandled data from #responses and returns it.
# - #add_response_handler: Add a block to be called inside the receiver thread
#   with every server response.
# - #response_handlers: Returns the list of response handlers.
# - #remove_response_handler: Remove a previously added response handler.
#
# === Core \IMAP commands
#
# The following commands are defined either by
# the [IMAP4rev1[https://tools.ietf.org/html/rfc3501]] base specification, or
# by one of the following extensions:
# [IDLE[https://tools.ietf.org/html/rfc2177]],
# [NAMESPACE[https://tools.ietf.org/html/rfc2342]],
# [UNSELECT[https://tools.ietf.org/html/rfc3691]],
# [ENABLE[https://tools.ietf.org/html/rfc5161]],
# [MOVE[https://tools.ietf.org/html/rfc6851]].
# These extensions are widely supported by modern IMAP4rev1 servers and have
# all been integrated into [IMAP4rev2[https://tools.ietf.org/html/rfc9051]].
# <em>*NOTE:* Net::IMAP doesn't support IMAP4rev2 yet.</em>
#
# ==== Any state
#
# - #capability: Returns the server's capabilities as an array of strings.
#
#   <em>In general, #capable? should be used rather than explicitly sending a
#   +CAPABILITY+ command to the server.</em>
# - #noop: Allows the server to send unsolicited untagged #responses.
# - #logout: Tells the server to end the session. Enters the "_logout_" state.
#
# ==== Not Authenticated state
#
# In addition to the commands for any state, the following commands are valid
# in the "<em>not authenticated</em>" state:
#
# - #starttls: Upgrades a clear-text connection to use TLS.
#
#   <em>Requires the +STARTTLS+ capability.</em>
# - #authenticate: Identifies the client to the server using the given
#   {SASL mechanism}[https://www.iana.org/assignments/sasl-mechanisms/sasl-mechanisms.xhtml]
#   and credentials.  Enters the "_authenticated_" state.
#
#   <em>The server should list <tt>"AUTH=#{mechanism}"</tt> capabilities for
#   supported mechanisms.</em>
# - #login: Identifies the client to the server using a plain text password.
#   Using #authenticate is generally preferred.  Enters the "_authenticated_"
#   state.
#
#   <em>The +LOGINDISABLED+ capability</em> <b>must NOT</b> <em>be listed.</em>
#
# ==== Authenticated state
#
# In addition to the commands for any state, the following commands are valid
# in the "_authenticated_" state:
#
# - #enable: Enables backwards incompatible server extensions.
#   <em>Requires the +ENABLE+ or +IMAP4rev2+ capability.</em>
# - #select:  Open a mailbox and enter the "_selected_" state.
# - #examine: Open a mailbox read-only, and enter the "_selected_" state.
# - #create: Creates a new mailbox.
# - #delete: Permanently remove a mailbox.
# - #rename: Change the name of a mailbox.
# - #subscribe: Adds a mailbox to the "subscribed" set.
# - #unsubscribe: Removes a mailbox from the "subscribed" set.
# - #list: Returns names and attributes of mailboxes matching a given pattern.
# - #namespace: Returns mailbox namespaces, with path prefixes and delimiters.
#   <em>Requires the +NAMESPACE+ or +IMAP4rev2+ capability.</em>
# - #status: Returns mailbox information, e.g. message count, unseen message
#   count, +UIDVALIDITY+ and +UIDNEXT+.
# - #append: Appends a message to the end of a mailbox.
# - #idle: Allows the server to send updates to the client, without the client
#   needing to poll using #noop.
#   <em>Requires the +IDLE+ or +IMAP4rev2+ capability.</em>
# - *Obsolete* #lsub: <em>Replaced by <tt>LIST-EXTENDED</tt> and removed from
#   +IMAP4rev2+.</em>  Lists mailboxes in the "subscribed" set.
#
#   <em>*Note:* Net::IMAP hasn't implemented <tt>LIST-EXTENDED</tt> yet.</em>
#
# ==== Selected state
#
# In addition to the commands for any state and the "_authenticated_"
# commands, the following commands are valid in the "_selected_" state:
#
# - #close: Closes the mailbox and returns to the "_authenticated_" state,
#   expunging deleted messages, unless the mailbox was opened as read-only.
# - #unselect: Closes the mailbox and returns to the "_authenticated_" state,
#   without expunging any messages.
#   <em>Requires the +UNSELECT+ or +IMAP4rev2+ capability.</em>
# - #expunge: Permanently removes messages which have the Deleted flag set.
# - #uid_expunge: Restricts expunge to only remove the specified UIDs.
#   <em>Requires the +UIDPLUS+ or +IMAP4rev2+ capability.</em>
# - #search, #uid_search: Returns sequence numbers or UIDs of messages that
#   match the given searching criteria.
# - #fetch, #uid_fetch: Returns data associated with a set of messages,
#   specified by sequence number or UID.
# - #store, #uid_store: Alters a message's flags.
# - #copy, #uid_copy: Copies the specified messages to the end of the
#   specified destination mailbox.
# - #move, #uid_move: Moves the specified messages to the end of the
#   specified destination mailbox, expunging them from the current mailbox.
#   <em>Requires the +MOVE+ or +IMAP4rev2+ capability.</em>
# - #check: <em>*Obsolete:* removed from +IMAP4rev2+.</em>
#   Can be replaced with #noop or #idle.
#
# ==== Logout state
#
# No \IMAP commands are valid in the "_logout_" state.  If the socket is still
# open, Net::IMAP will close it after receiving server confirmation.
# Exceptions will be raised by \IMAP commands that have already started and
# are waiting for a response, as well as any that are called after logout.
#
# === \IMAP extension support
#
# ==== RFC9051: +IMAP4rev2+
#
# Although IMAP4rev2[https://tools.ietf.org/html/rfc9051] is not supported
# yet, Net::IMAP supports several extensions that have been folded into it:
# +ENABLE+, +IDLE+, +MOVE+, +NAMESPACE+, +SASL-IR+, +UIDPLUS+, +UNSELECT+,
# <tt>STATUS=SIZE</tt>, and the fetch side of +BINARY+.
# Commands for these extensions are listed with the {Core IMAP
# commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands], above.
#
# >>>
#   <em>The following are folded into +IMAP4rev2+ but are currently
#   unsupported or incompletely supported by</em> Net::IMAP<em>: RFC4466
#   extensions, +ESEARCH+, +SEARCHRES+, +LIST-EXTENDED+, +LIST-STATUS+,
#   +LITERAL-+, and +SPECIAL-USE+.</em>
#
# ==== RFC2087: +QUOTA+
# - #getquota: returns the resource usage and limits for a quota root
# - #getquotaroot: returns the list of quota roots for a mailbox, as well as
#   their resource usage and limits.
# - #setquota: sets the resource limits for a given quota root.
#
# ==== RFC2177: +IDLE+
# Folded into IMAP4rev2[https://tools.ietf.org/html/rfc9051] and also included
# above with {Core IMAP commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands].
# - #idle: Allows the server to send updates to the client, without the client
#   needing to poll using #noop.
#
# ==== RFC2342: +NAMESPACE+
# Folded into IMAP4rev2[https://tools.ietf.org/html/rfc9051] and also included
# above with {Core IMAP commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands].
# - #namespace: Returns mailbox namespaces, with path prefixes and delimiters.
#
# ==== RFC2971: +ID+
# - #id: exchanges client and server implementation information.
#
# ==== RFC3516: +BINARY+
# The fetch side of +BINARY+ has been folded into
# IMAP4rev2[https://tools.ietf.org/html/rfc9051].
# - Updates #fetch and #uid_fetch with the +BINARY+, +BINARY.PEEK+, and
#   +BINARY.SIZE+ items.  See FetchData#binary and FetchData#binary_size.
#
# >>>
#   *NOTE:* The binary extension the #append command is _not_ supported yet.
#
# ==== RFC3691: +UNSELECT+
# Folded into IMAP4rev2[https://tools.ietf.org/html/rfc9051] and also included
# above with {Core IMAP commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands].
# - #unselect: Closes the mailbox and returns to the "_authenticated_" state,
#   without expunging any messages.
#
# ==== RFC4314: +ACL+
# - #getacl: lists the authenticated user's access rights to a mailbox.
# - #setacl: sets the access rights for a user on a mailbox
# >>>
#   *NOTE:* +DELETEACL+, +LISTRIGHTS+, and +MYRIGHTS+ are not supported yet.
#
# ==== RFC4315: +UIDPLUS+
# Folded into IMAP4rev2[https://tools.ietf.org/html/rfc9051] and also included
# above with {Core IMAP commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands].
# - #uid_expunge: Restricts #expunge to only remove the specified UIDs.
# - Updates #select, #examine with the +UIDNOTSTICKY+ ResponseCode
# - Updates #append with the +APPENDUID+ ResponseCode
# - Updates #copy, #move with the +COPYUID+ ResponseCode
#
# ==== RFC4959: +SASL-IR+
# Folded into IMAP4rev2[https://tools.ietf.org/html/rfc9051].
# - Updates #authenticate with the option to send an initial response.
#
# ==== RFC5161: +ENABLE+
# Folded into IMAP4rev2[https://tools.ietf.org/html/rfc9051] and also included
# above with {Core IMAP commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands].
# - #enable: Enables backwards incompatible server extensions.
#
# ==== RFC5256: +SORT+
# - #sort, #uid_sort: An alternate version of #search or #uid_search which
#   sorts the results by specified keys.
# ==== RFC5256: +THREAD+
# - #thread, #uid_thread: An alternate version of #search or #uid_search,
#   which arranges the results into ordered groups or threads according to a
#   chosen algorithm.
#
# ==== +X-GM-EXT-1+
# +X-GM-EXT-1+ is a non-standard Gmail extension.  See {Google's
# documentation}[https://developers.google.com/gmail/imap/imap-extensions].
# - Updates #fetch and #uid_fetch with support for +X-GM-MSGID+ (unique
#   message ID), +X-GM-THRID+ (thread ID), and +X-GM-LABELS+ (Gmail labels).
# - Updates #search with the +X-GM-RAW+ search attribute.
# - #xlist: replaced by +SPECIAL-USE+ attributes in #list responses.
#
# *NOTE:* The +OBJECTID+ extension should replace +X-GM-MSGID+ and
# +X-GM-THRID+, but Gmail does not support it (as of 2023-11-10).
#
# ==== RFC6851: +MOVE+
# Folded into IMAP4rev2[https://tools.ietf.org/html/rfc9051] and also included
# above with {Core IMAP commands}[rdoc-ref:Net::IMAP@Core+IMAP+commands].
# - #move, #uid_move: Moves the specified messages to the end of the
#   specified destination mailbox, expunging them from the current mailbox.
#
# ==== RFC6855: <tt>UTF8=ACCEPT</tt>, <tt>UTF8=ONLY</tt>
#
# - See #enable for information about support for UTF-8 string encoding.
#
# ==== RFC7162: +CONDSTORE+
#
# - Updates #enable with +CONDSTORE+ parameter.  +CONDSTORE+ will also be
#   enabled by using any of the extension's command parameters, listed below.
# - Updates #status with the +HIGHESTMODSEQ+ status attribute.
# - Updates #select and #examine with the +condstore+ modifier, and adds
#   either a +HIGHESTMODSEQ+ or +NOMODSEQ+ ResponseCode to the responses.
# - Updates #search, #uid_search, #sort, and #uid_sort with the +MODSEQ+
#   search criterion, and adds SearchResult#modseq to the search response.
# - Updates #thread and #uid_thread with the +MODSEQ+ search criterion
#   <em>(but thread responses are unchanged)</em>.
# - Updates #fetch and #uid_fetch with the +changedsince+ modifier and
#   +MODSEQ+ FetchData attribute.
# - Updates #store and #uid_store with the +unchangedsince+ modifier and adds
#   the +MODIFIED+ ResponseCode to the tagged response.
#
# ==== RFC8438: <tt>STATUS=SIZE</tt>
# - Updates #status with the +SIZE+ status attribute.
#
# ==== RFC8474: +OBJECTID+
# - Adds +MAILBOXID+ ResponseCode to #create tagged response.
# - Adds +MAILBOXID+ ResponseCode to #select and #examine untagged response.
# - Updates #fetch and #uid_fetch with the +EMAILID+ and +THREADID+ items.
#   See FetchData#emailid and FetchData#emailid.
# - Updates #status with support for the +MAILBOXID+ status attribute.
#
# == References
#
# [{IMAP4rev1}[https://www.rfc-editor.org/rfc/rfc3501.html]]::
#   Crispin, M., "INTERNET MESSAGE ACCESS PROTOCOL - \VERSION 4rev1",
#   RFC 3501, DOI 10.17487/RFC3501, March 2003,
#   <https://www.rfc-editor.org/info/rfc3501>.
#
# [IMAP-ABNF-EXT[https://www.rfc-editor.org/rfc/rfc4466.html]]::
#   Melnikov, A. and C. Daboo, "Collected Extensions to IMAP4 ABNF",
#   RFC 4466, DOI 10.17487/RFC4466, April 2006,
#   <https://www.rfc-editor.org/info/rfc4466>.
#
#   <em>Note: Net::IMAP cannot parse the entire RFC4466 grammar yet.</em>
#
# [{IMAP4rev2}[https://www.rfc-editor.org/rfc/rfc9051.html]]::
#   Melnikov, A., Ed., and B. Leiba, Ed., "Internet Message Access Protocol
#   (\IMAP) - Version 4rev2", RFC 9051, DOI 10.17487/RFC9051, August 2021,
#   <https://www.rfc-editor.org/info/rfc9051>.
#
#   <em>Note: Net::IMAP is not fully compatible with IMAP4rev2 yet.</em>
#
# [IMAP-IMPLEMENTATION[https://www.rfc-editor.org/info/rfc2683]]::
#   Leiba, B., "IMAP4 Implementation Recommendations",
#   RFC 2683, DOI 10.17487/RFC2683, September 1999,
#   <https://www.rfc-editor.org/info/rfc2683>.
#
# [IMAP-MULTIACCESS[https://www.rfc-editor.org/info/rfc2180]]::
#   Gahrns, M., "IMAP4 Multi-Accessed Mailbox Practice", RFC 2180, DOI
#   10.17487/RFC2180, July 1997, <https://www.rfc-editor.org/info/rfc2180>.
#
# [UTF7[https://tools.ietf.org/html/rfc2152]]::
#   Goldsmith, D. and M. Davis, "UTF-7 A Mail-Safe Transformation Format of
#   Unicode", RFC 2152, DOI 10.17487/RFC2152, May 1997,
#   <https://www.rfc-editor.org/info/rfc2152>.
#
# === Message envelope and body structure
#
# [RFC5322[https://tools.ietf.org/html/rfc5322]]::
#   Resnick, P., Ed., "Internet Message Format",
#   RFC 5322, DOI 10.17487/RFC5322, October 2008,
#   <https://www.rfc-editor.org/info/rfc5322>.
#
#   <em>Note: obsoletes</em>
#   RFC-2822[https://tools.ietf.org/html/rfc2822]<em> (April 2001) and</em>
#   RFC-822[https://tools.ietf.org/html/rfc822]<em> (August 1982).</em>
#
# [CHARSET[https://tools.ietf.org/html/rfc2978]]::
#   Freed, N. and J. Postel, "IANA Charset Registration Procedures", BCP 19,
#   RFC 2978, DOI 10.17487/RFC2978, October 2000,
#   <https://www.rfc-editor.org/info/rfc2978>.
#
# [DISPOSITION[https://tools.ietf.org/html/rfc2183]]::
#    Troost, R., Dorner, S., and K. Moore, Ed., "Communicating Presentation
#    Information in Internet Messages: The Content-Disposition Header
#    Field", RFC 2183, DOI 10.17487/RFC2183, August 1997,
#    <https://www.rfc-editor.org/info/rfc2183>.
#
# [MIME-IMB[https://tools.ietf.org/html/rfc2045]]::
#    Freed, N. and N. Borenstein, "Multipurpose Internet Mail Extensions
#    (MIME) Part One: Format of Internet Message Bodies",
#    RFC 2045, DOI 10.17487/RFC2045, November 1996,
#    <https://www.rfc-editor.org/info/rfc2045>.
#
# [MIME-IMT[https://tools.ietf.org/html/rfc2046]]::
#    Freed, N. and N. Borenstein, "Multipurpose Internet Mail Extensions
#    (MIME) Part Two: Media Types", RFC 2046, DOI 10.17487/RFC2046,
#    November 1996, <https://www.rfc-editor.org/info/rfc2046>.
#
# [MIME-HDRS[https://tools.ietf.org/html/rfc2047]]::
#    Moore, K., "MIME (Multipurpose Internet Mail Extensions) Part Three:
#    Message Header Extensions for Non-ASCII Text",
#    RFC 2047, DOI 10.17487/RFC2047, November 1996,
#    <https://www.rfc-editor.org/info/rfc2047>.
#
# [RFC2231[https://tools.ietf.org/html/rfc2231]]::
#    Freed, N. and K. Moore, "MIME Parameter Value and Encoded Word
#    Extensions: Character Sets, Languages, and Continuations",
#    RFC 2231, DOI 10.17487/RFC2231, November 1997,
#    <https://www.rfc-editor.org/info/rfc2231>.
#
# [I18n-HDRS[https://tools.ietf.org/html/rfc6532]]::
#    Yang, A., Steele, S., and N. Freed, "Internationalized Email Headers",
#    RFC 6532, DOI 10.17487/RFC6532, February 2012,
#    <https://www.rfc-editor.org/info/rfc6532>.
#
# [LANGUAGE-TAGS[https://www.rfc-editor.org/info/rfc3282]]::
#    Alvestrand, H., "Content Language Headers",
#    RFC 3282, DOI 10.17487/RFC3282, May 2002,
#    <https://www.rfc-editor.org/info/rfc3282>.
#
# [LOCATION[https://www.rfc-editor.org/info/rfc2557]]::
#    Palme, J., Hopmann, A., and N. Shelness, "MIME Encapsulation of
#    Aggregate Documents, such as HTML (MHTML)",
#    RFC 2557, DOI 10.17487/RFC2557, March 1999,
#    <https://www.rfc-editor.org/info/rfc2557>.
#
# [MD5[https://tools.ietf.org/html/rfc1864]]::
#    Myers, J. and M. Rose, "The Content-MD5 Header Field",
#    RFC 1864, DOI 10.17487/RFC1864, October 1995,
#    <https://www.rfc-editor.org/info/rfc1864>.
#
# [RFC3503[https://tools.ietf.org/html/rfc3503]]::
#    Melnikov, A., "Message Disposition Notification (MDN)
#    profile for Internet Message Access Protocol (IMAP)",
#    RFC 3503, DOI 10.17487/RFC3503, March 2003,
#    <https://www.rfc-editor.org/info/rfc3503>.
#
# === \IMAP Extensions
#
# [QUOTA[https://tools.ietf.org/html/rfc9208]]::
#   Melnikov, A., "IMAP QUOTA Extension", RFC 9208, DOI 10.17487/RFC9208,
#   March 2022, <https://www.rfc-editor.org/info/rfc9208>.
#
#   <em>Note: obsoletes</em>
#   RFC-2087[https://tools.ietf.org/html/rfc2087]<em> (January 1997)</em>.
#   <em>Net::IMAP does not fully support the RFC9208 updates yet.</em>
# [IDLE[https://tools.ietf.org/html/rfc2177]]::
#   Leiba, B., "IMAP4 IDLE command", RFC 2177, DOI 10.17487/RFC2177,
#   June 1997, <https://www.rfc-editor.org/info/rfc2177>.
# [NAMESPACE[https://tools.ietf.org/html/rfc2342]]::
#   Gahrns, M. and C. Newman, "IMAP4 Namespace", RFC 2342,
#   DOI 10.17487/RFC2342, May 1998, <https://www.rfc-editor.org/info/rfc2342>.
# [ID[https://tools.ietf.org/html/rfc2971]]::
#   Showalter, T., "IMAP4 ID extension", RFC 2971, DOI 10.17487/RFC2971,
#   October 2000, <https://www.rfc-editor.org/info/rfc2971>.
# [BINARY[https://tools.ietf.org/html/rfc3516]]::
#   Nerenberg, L., "IMAP4 Binary Content Extension", RFC 3516,
#   DOI 10.17487/RFC3516, April 2003,
#   <https://www.rfc-editor.org/info/rfc3516>.
# [ACL[https://tools.ietf.org/html/rfc4314]]::
#   Melnikov, A., "IMAP4 Access Control List (ACL) Extension", RFC 4314,
#   DOI 10.17487/RFC4314, December 2005,
#   <https://www.rfc-editor.org/info/rfc4314>.
# [UIDPLUS[https://www.rfc-editor.org/rfc/rfc4315.html]]::
#   Crispin, M., "Internet Message Access Protocol (\IMAP) - UIDPLUS
#   extension", RFC 4315, DOI 10.17487/RFC4315, December 2005,
#   <https://www.rfc-editor.org/info/rfc4315>.
# [SORT[https://tools.ietf.org/html/rfc5256]]::
#   Crispin, M. and K. Murchison, "Internet Message Access Protocol - SORT and
#   THREAD Extensions", RFC 5256, DOI 10.17487/RFC5256, June 2008,
#   <https://www.rfc-editor.org/info/rfc5256>.
# [THREAD[https://tools.ietf.org/html/rfc5256]]::
#   Crispin, M. and K. Murchison, "Internet Message Access Protocol - SORT and
#   THREAD Extensions", RFC 5256, DOI 10.17487/RFC5256, June 2008,
#   <https://www.rfc-editor.org/info/rfc5256>.
# [RFC5530[https://www.rfc-editor.org/rfc/rfc5530.html]]::
#   Gulbrandsen, A., "IMAP Response Codes", RFC 5530, DOI 10.17487/RFC5530,
#   May 2009, <https://www.rfc-editor.org/info/rfc5530>.
# [MOVE[https://tools.ietf.org/html/rfc6851]]::
#   Gulbrandsen, A. and N. Freed, Ed., "Internet Message Access Protocol
#   (\IMAP) - MOVE Extension", RFC 6851, DOI 10.17487/RFC6851, January 2013,
#   <https://www.rfc-editor.org/info/rfc6851>.
# [UTF8=ACCEPT[https://tools.ietf.org/html/rfc6855]]::
# [UTF8=ONLY[https://tools.ietf.org/html/rfc6855]]::
#   Resnick, P., Ed., Newman, C., Ed., and S. Shen, Ed.,
#   "IMAP Support for UTF-8", RFC 6855, DOI 10.17487/RFC6855, March 2013,
#   <https://www.rfc-editor.org/info/rfc6855>.
# [CONDSTORE[https://tools.ietf.org/html/rfc7162]]::
# [QRESYNC[https://tools.ietf.org/html/rfc7162]]::
#   Melnikov, A. and D. Cridland, "IMAP Extensions: Quick Flag Changes
#   Resynchronization (CONDSTORE) and Quick Mailbox Resynchronization
#   (QRESYNC)", RFC 7162, DOI 10.17487/RFC7162, May 2014,
#   <https://www.rfc-editor.org/info/rfc7162>.
# [OBJECTID[https://tools.ietf.org/html/rfc8474]]::
#   Gondwana, B., Ed., "IMAP Extension for Object Identifiers",
#   RFC 8474, DOI 10.17487/RFC8474, September 2018,
#   <https://www.rfc-editor.org/info/rfc8474>.
#
# === IANA registries
# * {IMAP Capabilities}[http://www.iana.org/assignments/imap4-capabilities]
# * {IMAP Response Codes}[https://www.iana.org/assignments/imap-response-codes/imap-response-codes.xhtml]
# * {IMAP Mailbox Name Attributes}[https://www.iana.org/assignments/imap-mailbox-name-attributes/imap-mailbox-name-attributes.xhtml]
# * {IMAP and JMAP Keywords}[https://www.iana.org/assignments/imap-jmap-keywords/imap-jmap-keywords.xhtml]
# * {IMAP Threading Algorithms}[https://www.iana.org/assignments/imap-threading-algorithms/imap-threading-algorithms.xhtml]
# * {SASL Mechanisms and SASL SCRAM Family Mechanisms}[https://www.iana.org/assignments/sasl-mechanisms/sasl-mechanisms.xhtml]
# * {Service Name and Transport Protocol Port Number Registry}[https://www.iana.org/assignments/service-names-port-numbers/service-names-port-numbers.xml]:
#   +imap+: tcp/143, +imaps+: tcp/993
# * {GSSAPI/Kerberos/SASL Service Names}[https://www.iana.org/assignments/gssapi-service-names/gssapi-service-names.xhtml]:
#   +imap+
# * {Character sets}[https://www.iana.org/assignments/character-sets/character-sets.xhtml]
# ===== For currently unsupported features:
# * {IMAP Quota Resource Types}[http://www.iana.org/assignments/imap4-capabilities#imap-capabilities-2]
# * {LIST-EXTENDED options and responses}[https://www.iana.org/assignments/imap-list-extended/imap-list-extended.xhtml]
# * {IMAP METADATA Server Entry and Mailbox Entry Registries}[https://www.iana.org/assignments/imap-metadata/imap-metadata.xhtml]
# * {IMAP ANNOTATE Extension Entries and Attributes}[https://www.iana.org/assignments/imap-annotate-extension/imap-annotate-extension.xhtml]
# * {IMAP URLAUTH Access Identifiers and Prefixes}[https://www.iana.org/assignments/urlauth-access-ids/urlauth-access-ids.xhtml]
# * {IMAP URLAUTH Authorization Mechanism Registry}[https://www.iana.org/assignments/urlauth-authorization-mechanism-registry/urlauth-authorization-mechanism-registry.xhtml]
#
# source://net-imap//lib/net/imap.rb#719
class Net::IMAP < ::Net::Protocol
  include ::Net::IMAP::DeprecatedClientOptions
  include ::MonitorMixin
  include ::OpenSSL
  include ::OpenSSL::SSL
  extend ::Net::IMAP::Authenticators

  # Creates a new Net::IMAP object and connects it to the specified
  # +host+.
  #
  # ==== Options
  #
  # Accepts the following options:
  #
  # [port]
  #   Port number.  Defaults to 993 when +ssl+ is truthy, and 143 otherwise.
  #
  # [ssl]
  #   If +true+, the connection will use TLS with the default params set by
  #   {OpenSSL::SSL::SSLContext#set_params}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#method-i-set_params].
  #   If +ssl+ is a hash, it's passed to
  #   {OpenSSL::SSL::SSLContext#set_params}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#method-i-set_params];
  #   the keys are names of attribute assignment methods on
  #   SSLContext[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html].  For example:
  #
  #   [{ca_file}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#attribute-i-ca_file]]
  #     The path to a file containing a PEM-format CA certificate.
  #   [{ca_path}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#attribute-i-ca_path]]
  #     The path to a directory containing CA certificates in PEM format.
  #   [{min_version}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#method-i-min_version-3D]]
  #     Sets the lower bound on the supported SSL/TLS protocol version. Set to
  #     an +OpenSSL+ constant such as +OpenSSL::SSL::TLS1_2_VERSION+,
  #   [{verify_mode}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#attribute-i-verify_mode]]
  #     SSL session verification mode.  Valid modes include
  #     +OpenSSL::SSL::VERIFY_PEER+ and +OpenSSL::SSL::VERIFY_NONE+.
  #
  #   See {OpenSSL::SSL::SSLContext}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html] for other valid SSL context params.
  #
  #   See DeprecatedClientOptions.new for deprecated SSL arguments.
  #
  # [config]
  #   A Net::IMAP::Config object to use as the basis for #config.  By default,
  #   the global Net::IMAP.config is used.
  #
  #   >>>
  #     *NOTE:* +config+ does not set #config directly---it sets the _parent_
  #     config for inheritance.  Every client creates its own unique #config.
  #
  #   All other keyword arguments are forwarded to Net::IMAP::Config.new, to
  #   initialize the client's #config. For example:
  #
  #   [{open_timeout}[rdoc-ref:Config#open_timeout]]
  #     Seconds to wait until a connection is opened
  #   [{idle_response_timeout}[rdoc-ref:Config#idle_response_timeout]]
  #     Seconds to wait until an IDLE response is received
  #
  #   See Net::IMAP::Config for other valid options.
  #
  # ==== Examples
  #
  # Connect to cleartext port 143 at mail.example.com and receive the server greeting:
  #   imap = Net::IMAP.new('mail.example.com', ssl: false) # => #<Net::IMAP:0x00007f79b0872bd0>
  #   imap.port          => 143
  #   imap.tls_verified? => false
  #   imap.greeting      => name: ("OK" | "PREAUTH") => status
  #   status # => "OK"
  #   # The client is connected in the "Not Authenticated" state.
  #
  # Connect with TLS to port 993
  #   imap = Net::IMAP.new('mail.example.com', ssl: true) # => #<Net::IMAP:0x00007f79b0872bd0>
  #   imap.port          => 993
  #   imap.tls_verified? => true
  #   imap.greeting      => name: (/OK/i | /PREAUTH/i) => status
  #   case status
  #   in /OK/i
  #     # The client is connected in the "Not Authenticated" state.
  #     imap.authenticate("PLAIN", "joe_user", "joes_password")
  #   in /PREAUTH/i
  #     # The client is connected in the "Authenticated" state.
  #   end
  #
  # Connect with prior authentication, for example using an SSL certificate:
  #   ssl_ctx_params = {
  #     cert: OpenSSL::X509::Certificate.new(File.read("client.crt")),
  #     key:  OpenSSL::PKey::EC.new(File.read('client.key')),
  #     extra_chain_cert: [
  #       OpenSSL::X509::Certificate.new(File.read("intermediate.crt")),
  #     ],
  #   }
  #   imap = Net::IMAP.new('mail.example.com', ssl: ssl_ctx_params)
  #   imap.port          => 993
  #   imap.tls_verified? => true
  #   imap.greeting      => name: "PREAUTH"
  #   # The client is connected in the "Authenticated" state.
  #
  # ==== Exceptions
  #
  # The most common errors are:
  #
  # [Errno::ECONNREFUSED]
  #   Connection refused by +host+ or an intervening firewall.
  # [Errno::ETIMEDOUT]
  #   Connection timed out (possibly due to packets being dropped by an
  #   intervening firewall).
  # [Errno::ENETUNREACH]
  #   There is no route to that network.
  # [SocketError]
  #   Hostname not known or other socket error.
  # [Net::IMAP::ByeResponseError]
  #   Connected to the host successfully, but it immediately said goodbye.
  #
  # @return [IMAP] a new instance of IMAP
  #
  # source://net-imap//lib/net/imap/deprecated_client_options.rb#72
  def initialize(host, port_or_options = T.unsafe(nil), *deprecated, **options); end

  # Adds a response handler. For example, to detect when
  # the server sends a new EXISTS response (which normally
  # indicates new messages being added to the mailbox),
  # add the following handler after selecting the
  # mailbox:
  #
  #   imap.add_response_handler { |resp|
  #     if resp.kind_of?(Net::IMAP::UntaggedResponse) and resp.name == "EXISTS"
  #       puts "Mailbox now has #{resp.data} messages"
  #     end
  #   }
  #
  # Related: #remove_response_handler, #response_handlers
  #
  # @raise [ArgumentError]
  #
  # source://net-imap//lib/net/imap.rb#2614
  def add_response_handler(handler = T.unsafe(nil), &block); end

  # Sends an {APPEND command [IMAP4rev1 §6.3.11]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.11]
  # to append the +message+ to the end of the +mailbox+. The optional +flags+
  # argument is an array of flags initially passed to the new message.  The
  # optional +date_time+ argument specifies the creation time to assign to the
  # new message; it defaults to the current time.
  #
  # For example:
  #
  #   imap.append("inbox", <<EOF.gsub(/\n/, "\r\n"), [:Seen], Time.now)
  #   Subject: hello
  #   From: shugo@ruby-lang.org
  #   To: shugo@ruby-lang.org
  #
  #   hello world
  #   EOF
  #
  # A Net::IMAP::NoResponseError is raised if the mailbox does
  # not exist (it is not created automatically), or if the flags,
  # date_time, or message arguments contain errors.
  #
  # ===== Capabilities
  #
  # If +UIDPLUS+ [RFC4315[https://www.rfc-editor.org/rfc/rfc4315.html]] is
  # supported and the destination supports persistent UIDs, the server's
  # response should include an +APPENDUID+ response code with UIDPlusData.
  # This will report the UIDVALIDITY of the destination mailbox and the
  # assigned UID of the appended message.
  #
  # --
  # TODO: add MULTIAPPEND support
  # ++
  #
  # source://net-imap//lib/net/imap.rb#1866
  def append(mailbox, message, flags = T.unsafe(nil), date_time = T.unsafe(nil)); end

  # Returns whether the server supports a given SASL +mechanism+ for use with
  # the #authenticate command.  The +mechanism+ is supported when
  # #capabilities includes <tt>"AUTH=#{mechanism.to_s.upcase}"</tt>.  When
  # available, cached capabilities are used without sending a new #capability
  # command to the server.
  #
  #   imap.capable?      "AUTH=PLAIN"  # => true
  #   imap.auth_capable? "PLAIN"       # => true
  #   imap.auth_capable? "blurdybloop" # => false
  #
  # Related: #authenticate, #auth_mechanisms, #capable?, #capabilities
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap.rb#1059
  def auth_capable?(mechanism); end

  # Returns the #authenticate mechanisms that the server claims to support.
  # These are derived from the #capabilities with an <tt>AUTH=</tt> prefix.
  #
  # This may be different when the connection is cleartext or using TLS.  Most
  # servers will drop all <tt>AUTH=</tt> mechanisms from #capabilities after
  # the connection has authenticated.
  #
  #    imap = Net::IMAP.new(hostname, ssl: false)
  #    imap.capabilities    # => ["IMAP4REV1", "LOGINDISABLED"]
  #    imap.auth_mechanisms # => []
  #
  #    imap.starttls
  #    imap.capabilities    # => ["IMAP4REV1", "AUTH=PLAIN", "AUTH=XOAUTH2",
  #                         #     "AUTH=OAUTHBEARER"]
  #    imap.auth_mechanisms # => ["PLAIN", "XOAUTH2", "OAUTHBEARER"]
  #
  #    imap.authenticate("XOAUTH2", username, oauth2_access_token)
  #    imap.auth_mechanisms # => []
  #
  # Related: #authenticate, #auth_capable?, #capabilities
  #
  # source://net-imap//lib/net/imap.rb#1042
  def auth_mechanisms; end

  # :call-seq:
  #   authenticate(mechanism, *, sasl_ir: config.sasl_ir, registry: Net::IMAP::SASL.authenticators, **, &) -> ok_resp
  #
  # Sends an {AUTHENTICATE command [IMAP4rev1 §6.2.2]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.2.2]
  # to authenticate the client.  If successful, the connection enters the
  # "_authenticated_" state.
  #
  # +mechanism+ is the name of the \SASL authentication mechanism to be used.
  #
  # +sasl_ir+ allows or disallows sending an "initial response" (see the
  # +SASL-IR+ capability, below).  Defaults to the #config value for
  # {sasl_ir}[rdoc-ref:Config#sasl_ir], which defaults to +true+.
  #
  # All other arguments are forwarded to the registered SASL authenticator for
  # the requested mechanism.  <em>The documentation for each individual
  # mechanism must be consulted for its specific parameters.</em>
  #
  # Related: #login, #starttls, #auth_capable?, #auth_mechanisms
  #
  # ==== Mechanisms
  #
  # Each mechanism has different properties and requirements.  Please consult
  # the documentation for the specific mechanisms you are using:
  #
  # +ANONYMOUS+::
  #     See AnonymousAuthenticator[rdoc-ref:Net::IMAP::SASL::AnonymousAuthenticator].
  #
  #     Allows the user to gain access to public services or resources without
  #     authenticating or disclosing an identity.
  #
  # +EXTERNAL+::
  #     See ExternalAuthenticator[rdoc-ref:Net::IMAP::SASL::ExternalAuthenticator].
  #
  #     Authenticates using already established credentials, such as a TLS
  #     certificate or IPsec.
  #
  # +OAUTHBEARER+::
  #     See OAuthBearerAuthenticator[rdoc-ref:Net::IMAP::SASL::OAuthBearerAuthenticator].
  #
  #     Login using an OAuth2 Bearer token.  This is the standard mechanism
  #     for using OAuth2 with \SASL, but it is not yet deployed as widely as
  #     +XOAUTH2+.
  #
  # +PLAIN+::
  #     See PlainAuthenticator[rdoc-ref:Net::IMAP::SASL::PlainAuthenticator].
  #
  #     Login using clear-text username and password.
  #
  # +SCRAM-SHA-1+::
  # +SCRAM-SHA-256+::
  #     See ScramAuthenticator[rdoc-ref:Net::IMAP::SASL::ScramAuthenticator].
  #
  #     Login by username and password.  The password is not sent to the
  #     server but is used in a salted challenge/response exchange.
  #     +SCRAM-SHA-1+ and +SCRAM-SHA-256+ are directly supported by
  #     Net::IMAP::SASL.  New authenticators can easily be added for any other
  #     <tt>SCRAM-*</tt> mechanism if the digest algorithm is supported by
  #     OpenSSL::Digest.
  #
  # +XOAUTH2+::
  #     See XOAuth2Authenticator[rdoc-ref:Net::IMAP::SASL::XOAuth2Authenticator].
  #
  #     Login using a username and an OAuth2 access token.  Non-standard and
  #     obsoleted by +OAUTHBEARER+, but widely supported.
  #
  # See the {SASL mechanism
  # registry}[https://www.iana.org/assignments/sasl-mechanisms/sasl-mechanisms.xhtml]
  # for a list of all SASL mechanisms and their specifications.  To register
  # new authenticators, see Authenticators.
  #
  # ===== Deprecated mechanisms
  #
  # <em>Obsolete mechanisms should be avoided, but are still available for
  # backwards compatibility.  See</em> Net::IMAP::SASL@Deprecated+mechanisms.
  # <em>Using a deprecated mechanism will print a warning.</em>
  #
  # ==== Capabilities
  #
  # <tt>"AUTH=#{mechanism}"</tt> capabilities indicate server support for
  # mechanisms.  Use #auth_capable? or #auth_mechanisms to check for support
  # before using a particular mechanism.
  #
  #    if imap.auth_capable? "XOAUTH2"
  #      imap.authenticate "XOAUTH2", username, oauth2_access_token
  #    elsif imap.auth_capable? "PLAIN"
  #      imap.authenticate "PLAIN", username, password
  #    elsif !imap.capability? "LOGINDISABLED"
  #      imap.login username, password
  #    else
  #      raise "No acceptable authentication mechanism is available"
  #    end
  #
  # Although servers should list all supported \SASL mechanisms, they may
  # allow authentication with an unlisted +mechanism+.
  #
  # If [SASL-IR[https://www.rfc-editor.org/rfc/rfc4959.html]] is supported
  # and the appropriate <tt>"AUTH=#{mechanism}"</tt> capability is present,
  # an "initial response" may be sent as an argument to the +AUTHENTICATE+
  # command, saving a round-trip.  The SASL exchange allows for server
  # challenges and client responses, but many mechanisms expect the client to
  # "respond" first.  The initial response will only be sent for
  # "client-first" mechanisms.
  #
  # Server capabilities may change after #starttls, #login, and #authenticate.
  # Previously cached #capabilities will be cleared when this method
  # completes.  If the TaggedResponse to #authenticate includes updated
  # capabilities, they will be cached.
  #
  # source://net-imap//lib/net/imap.rb#1339
  def authenticate(mechanism, *creds, sasl_ir: T.unsafe(nil), **props, &callback); end

  # Returns the server capabilities.  When available, cached capabilities are
  # used without sending a new #capability command to the server.
  #
  # To ensure a case-insensitive comparison, #capable? can be used instead.
  #
  # <em>*NOTE:* Most Net::IMAP methods do not _currently_ modify their
  # behaviour according to the server's advertised #capabilities.</em>
  #
  # See Net::IMAP@Capabilities for more about \IMAP capabilities.
  #
  # Related: #capable?, #auth_capable?, #auth_mechanisms, #capability, #enable
  #
  # source://net-imap//lib/net/imap.rb#1018
  def capabilities; end

  # Returns whether capabilities have been cached.  When true, #capable? and
  # #capabilities don't require sending a #capability command to the server.
  #
  # See Net::IMAP@Capabilities for more about \IMAP capabilities.
  #
  # Related: #capable?, #capability, #clear_cached_capabilities
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap.rb#1069
  def capabilities_cached?; end

  # Sends a {CAPABILITY command [IMAP4rev1 §6.1.1]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.1.1]
  # and returns an array of capabilities that are supported by the server.
  # The result is stored for use by #capable? and #capabilities.
  #
  # <em>*NOTE:* Most Net::IMAP methods do not _currently_ modify their
  # behaviour according to the server's advertised #capabilities.</em>
  #
  # Net::IMAP automatically stores and discards capability data according to
  # the requirements and recommendations in
  # {IMAP4rev2 §6.1.1}[https://www.rfc-editor.org/rfc/rfc9051#section-6.1.1],
  # {§6.2}[https://www.rfc-editor.org/rfc/rfc9051#section-6.2], and
  # {§7.1}[https://www.rfc-editor.org/rfc/rfc9051#section-7.1].
  # Use #capable?, #auth_capable?, or #capabilities to this cache and avoid
  # sending the #capability command unnecessarily.
  #
  # See Net::IMAP@Capabilities for more about \IMAP capabilities.
  #
  # Related: #capable?, #auth_capable?, #capability, #enable
  #
  # source://net-imap//lib/net/imap.rb#1107
  def capability; end

  # Returns whether the server supports a given +capability+.  When available,
  # cached #capabilities are used without sending a new #capability command to
  # the server.
  #
  # <em>*NOTE:* Most Net::IMAP methods do not _currently_ modify their
  # behaviour according to the server's advertised #capabilities.</em>
  #
  # See Net::IMAP@Capabilities for more about \IMAP capabilities.
  #
  # Related: #auth_capable?, #capabilities, #capability, #enable
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap.rb#1004
  def capability?(capability); end

  # Returns whether the server supports a given +capability+.  When available,
  # cached #capabilities are used without sending a new #capability command to
  # the server.
  #
  # <em>*NOTE:* Most Net::IMAP methods do not _currently_ modify their
  # behaviour according to the server's advertised #capabilities.</em>
  #
  # See Net::IMAP@Capabilities for more about \IMAP capabilities.
  #
  # Related: #auth_capable?, #capabilities, #capability, #enable
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap.rb#1004
  def capable?(capability); end

  # Sends a {CHECK command [IMAP4rev1 §6.4.1]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.1]
  # to request a checkpoint of the currently selected mailbox.  This performs
  # implementation-specific housekeeping; for instance, reconciling the
  # mailbox's in-memory and on-disk state.
  #
  # Related: #idle, #noop
  #
  # source://net-imap//lib/net/imap.rb#1882
  def check; end

  # Clears capabilities that have been remembered by the Net::IMAP client.
  # This forces a #capability command to be sent the next time a #capabilities
  # query method is called.
  #
  # Net::IMAP automatically discards its cached capabilities when they can
  # change.  Explicitly calling this _should_ be unnecessary for well-behaved
  # servers.
  #
  # Related: #capable?, #capability, #capabilities_cached?
  #
  # source://net-imap//lib/net/imap.rb#1082
  def clear_cached_capabilities; end

  # :call-seq:
  #   clear_responses       -> hash
  #   clear_responses(type) -> array
  #
  # Clears and returns the unhandled #responses hash or the unhandled
  # responses array for a single response +type+.
  #
  # Clearing responses is synchronized with other threads.  The lock is
  # released before returning.
  #
  # Related: #responses, #response_handlers
  #
  # source://net-imap//lib/net/imap.rb#2571
  def clear_responses(type = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap.rb#957
  def client_thread; end

  # Sends a {CLOSE command [IMAP4rev1 §6.4.2]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.2]
  # to close the currently selected mailbox.  The CLOSE command permanently
  # removes from the mailbox all messages that have the <tt>\\Deleted</tt>
  # flag set.
  #
  # Related: #unselect
  #
  # source://net-imap//lib/net/imap.rb#1892
  def close; end

  # The client configuration.  See Net::IMAP::Config.
  #
  # By default, the client's local configuration inherits from the global
  # Net::IMAP.config.
  #
  # source://net-imap//lib/net/imap.rb#772
  def config; end

  # Sends a {COPY command [IMAP4rev1 §6.4.7]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.7]
  # to copy the specified message(s) to the end of the specified destination
  # +mailbox+. The +set+ parameter is a number, an array of numbers, or a
  # Range object.  The number is a message sequence number.
  #
  # Related: #uid_copy
  #
  # ===== Capabilities
  #
  # If +UIDPLUS+ [RFC4315[https://www.rfc-editor.org/rfc/rfc4315.html]] is
  # supported, the server's response should include a +COPYUID+ response code
  # with UIDPlusData.  This will report the UIDVALIDITY of the destination
  # mailbox, the UID set of the source messages, and the assigned UID set of
  # the moved messages.
  #
  # source://net-imap//lib/net/imap.rb#2197
  def copy(set, mailbox); end

  # Sends a {CREATE command [IMAP4rev1 §6.3.3]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.3]
  # to create a new +mailbox+.
  #
  # A Net::IMAP::NoResponseError is raised if a mailbox with that name
  # cannot be created.
  #
  # Related: #rename, #delete
  #
  # source://net-imap//lib/net/imap.rb#1466
  def create(mailbox); end

  # Sends a {DELETE command [IMAP4rev1 §6.3.4]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.4]
  # to remove the +mailbox+.
  #
  # A Net::IMAP::NoResponseError is raised if a mailbox with that name
  # cannot be deleted, either because it does not exist or because the
  # client does not have permission to delete it.
  #
  # Related: #create, #rename
  #
  # source://net-imap//lib/net/imap.rb#1478
  def delete(mailbox); end

  # Disconnects from the server.
  #
  # Related: #logout, #logout!
  #
  # source://net-imap//lib/net/imap.rb#965
  def disconnect; end

  # Returns true if disconnected from the server.
  #
  # Related: #logout, #disconnect
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap.rb#990
  def disconnected?; end

  # Sends an {ENABLE command [RFC5161 §3.2]}[https://www.rfc-editor.org/rfc/rfc5161#section-3.1]
  # {[IMAP4rev2 §6.3.1]}[https://www.rfc-editor.org/rfc/rfc9051#section-6.3.1]
  # to enable the specified server +capabilities+.  Each capability may be an
  # array, string, or symbol.  Returns a list of the capabilities that were
  # enabled.
  #
  # The +ENABLE+ command is only valid in the _authenticated_ state, before
  # any mailbox is selected.
  #
  # Related: #capable?, #capabilities, #capability
  #
  # ===== Capabilities
  #
  # The server's capabilities must include
  # +ENABLE+ [RFC5161[https://tools.ietf.org/html/rfc5161]]
  # or +IMAP4REV2+ [RFC9051[https://tools.ietf.org/html/rfc9051]].
  #
  # Additionally, the server capabilities must include a capability matching
  # each enabled extension (usually the same name as the enabled extension).
  # The following capabilities may be enabled:
  #
  # [+CONDSTORE+ {[RFC7162]}[https://www.rfc-editor.org/rfc/rfc7162.html]]
  #
  #   Updates various commands to return +CONDSTORE+ extension responses.  It
  #   is not necessary to explicitly enable +CONDSTORE+—using any of the
  #   command parameters defined by the extension will implicitly enable it.
  #   See {[RFC7162 §3.1]}[https://www.rfc-editor.org/rfc/rfc7162.html#section-3.1].
  #
  # [+:utf8+ --- an alias for <tt>"UTF8=ACCEPT"</tt>]
  #
  #   In a future release, <tt>enable(:utf8)</tt> will enable either
  #   <tt>"UTF8=ACCEPT"</tt> or <tt>"IMAP4rev2"</tt>, depending on server
  #   capabilities.
  #
  # [<tt>"UTF8=ACCEPT"</tt> [RFC6855[https://tools.ietf.org/html/rfc6855]]]
  #
  #   The server's capabilities must include <tt>UTF8=ACCEPT</tt> _or_
  #   <tt>UTF8=ONLY</tt>.
  #
  #   This allows the server to send strings encoded as UTF-8 which might
  #   otherwise need to use a 7-bit encoding, such as {modified
  #   UTF-7}[::decode_utf7] for mailbox names, or RFC2047 encoded-words for
  #   message headers.
  #
  #   *Note:* <em>A future update may set string encodings slightly
  #   differently</em>, e.g: "US-ASCII" when UTF-8 is not enabled, and "UTF-8"
  #   when it is.  Currently, the encoding of strings sent as "quoted" or
  #   "text" will _always_ be "UTF-8", even when only ASCII characters are
  #   used (e.g. "Subject: Agenda") And currently, string "literals" sent
  #   by the server will always have an "ASCII-8BIT" (binary)
  #   encoding, even if they generally contain UTF-8 data, if they are
  #   text at all.
  #
  # [<tt>"UTF8=ONLY"</tt> [RFC6855[https://tools.ietf.org/html/rfc6855]]]
  #
  #   A server that reports the <tt>UTF8=ONLY</tt> capability _requires_ that
  #   the client <tt>enable("UTF8=ACCEPT")</tt> before any mailboxes may be
  #   selected.  For convenience, <tt>enable("UTF8=ONLY")</tt> is aliased to
  #   <tt>enable("UTF8=ACCEPT")</tt>.
  #
  # ===== Unsupported capabilities
  #
  # *Note:* Some extensions that use ENABLE permit the server to send syntax
  # that Net::IMAP cannot parse, which may raise an exception and disconnect.
  # Some extensions may work, but the support may be incomplete, untested, or
  # experimental.
  #
  # Until a capability is documented here as supported, enabling it may result
  # in undocumented behavior and a future release may update with incompatible
  # behavior <em>without warning or deprecation</em>.
  #
  # <em>Caution is advised.</em>
  #
  # source://net-imap//lib/net/imap.rb#2408
  def enable(*capabilities); end

  # Sends a {EXAMINE command [IMAP4rev1 §6.3.2]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.2]
  # to select a +mailbox+ so that messages in the +mailbox+ can be accessed.
  # Behaves the same as #select, except that the selected +mailbox+ is
  # identified as read-only.
  #
  # A Net::IMAP::NoResponseError is raised if the mailbox does not
  # exist or is for some reason non-examinable.
  #
  # Related: #select
  #
  # source://net-imap//lib/net/imap.rb#1450
  def examine(mailbox, condstore: T.unsafe(nil)); end

  # Sends an {EXPUNGE command [IMAP4rev1 §6.4.3]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.3]
  # Sends a EXPUNGE command to permanently remove from the currently
  # selected mailbox all messages that have the \Deleted flag set.
  #
  # Related: #uid_expunge
  #
  # source://net-imap//lib/net/imap.rb#1917
  def expunge; end

  # :call-seq:
  #   fetch(set, attr, changedsince: nil) -> array of FetchData
  #
  # Sends a {FETCH command [IMAP4rev1 §6.4.5]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.5]
  # to retrieve data associated with a message in the mailbox.
  #
  # The +set+ parameter is a number or a range between two numbers,
  # or an array of those.  The number is a message sequence number,
  # where -1 represents a '*' for use in range notation like 100..-1
  # being interpreted as '100:*'.  Beware that the +exclude_end?+
  # property of a Range object is ignored, and the contents of a
  # range are independent of the order of the range endpoints as per
  # the protocol specification, so 1...5, 5..1 and 5...1 are all
  # equivalent to 1..5.
  #
  # +attr+ is a list of attributes to fetch; see the documentation
  # for FetchData for a list of valid attributes.
  #
  # +changedsince+ is an optional integer mod-sequence.  It limits results to
  # messages with a mod-sequence greater than +changedsince+.
  #
  # The return value is an array of FetchData.
  #
  # Related: #uid_search, FetchData
  #
  # ===== For example:
  #
  #   p imap.fetch(6..8, "UID")
  #   #=> [#<Net::IMAP::FetchData seqno=6, attr={"UID"=>98}>, \\
  #        #<Net::IMAP::FetchData seqno=7, attr={"UID"=>99}>, \\
  #        #<Net::IMAP::FetchData seqno=8, attr={"UID"=>100}>]
  #   p imap.fetch(6, "BODY[HEADER.FIELDS (SUBJECT)]")
  #   #=> [#<Net::IMAP::FetchData seqno=6, attr={"BODY[HEADER.FIELDS (SUBJECT)]"=>"Subject: test\r\n\r\n"}>]
  #   data = imap.uid_fetch(98, ["RFC822.SIZE", "INTERNALDATE"])[0]
  #   p data.seqno
  #   #=> 6
  #   p data.attr["RFC822.SIZE"]
  #   #=> 611
  #   p data.attr["INTERNALDATE"]
  #   #=> "12-Oct-2000 22:40:59 +0900"
  #   p data.attr["UID"]
  #   #=> 98
  #
  # ===== Capabilities
  #
  # Many extensions define new message +attr+ names.  See FetchData for a list
  # of supported extension fields.
  #
  # The server's capabilities must include +CONDSTORE+
  # {[RFC7162]}[https://tools.ietf.org/html/rfc7162] in order to use the
  # +changedsince+ argument.  Using +changedsince+ implicitly enables the
  # +CONDSTORE+ extension.
  #
  # source://net-imap//lib/net/imap.rb#2092
  def fetch(set, attr, mod = T.unsafe(nil), changedsince: T.unsafe(nil)); end

  # Sends a {GETACL command [RFC4314 §3.3]}[https://www.rfc-editor.org/rfc/rfc4314#section-3.3]
  # along with a specified +mailbox+.  If this mailbox exists, an array
  # containing objects of MailboxACLItem will be returned.
  #
  # Related: #setacl, MailboxACLItem
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +ACL+
  # [RFC4314[https://tools.ietf.org/html/rfc4314]].
  #
  # source://net-imap//lib/net/imap.rb#1746
  def getacl(mailbox); end

  # Sends a {GETQUOTA command [RFC2087 §4.2]}[https://www.rfc-editor.org/rfc/rfc2087#section-4.2]
  # along with specified +mailbox+.  If this mailbox exists, then an array
  # containing a MailboxQuota object is returned.  This command is generally
  # only available to server admin.
  #
  # Related: #getquotaroot, #setquota, MailboxQuota
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +QUOTA+
  # [RFC2087[https://tools.ietf.org/html/rfc2087]].
  #
  # source://net-imap//lib/net/imap.rb#1690
  def getquota(mailbox); end

  # Sends a {GETQUOTAROOT command [RFC2087 §4.3]}[https://www.rfc-editor.org/rfc/rfc2087#section-4.3]
  # along with the specified +mailbox+.  This command is generally available
  # to both admin and user.  If this mailbox exists, it returns an array
  # containing objects of type MailboxQuotaRoot and MailboxQuota.
  #
  # Related: #getquota, #setquota, MailboxQuotaRoot, MailboxQuota
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +QUOTA+
  # [RFC2087[https://tools.ietf.org/html/rfc2087]].
  #
  # source://net-imap//lib/net/imap.rb#1669
  def getquotaroot(mailbox); end

  # Returns the initial greeting the server, an UntaggedResponse.
  #
  # source://net-imap//lib/net/imap.rb#766
  def greeting; end

  # The hostname this client connected to
  #
  # source://net-imap//lib/net/imap.rb#783
  def host; end

  # Sends an {ID command [RFC2971 §3.1]}[https://www.rfc-editor.org/rfc/rfc2971#section-3.1]
  # and returns a hash of the server's response, or nil if the server does not
  # identify itself.
  #
  # Note that the user should first check if the server supports the ID
  # capability. For example:
  #
  #    if capable?(:ID)
  #      id = imap.id(
  #        name: "my IMAP client (ruby)",
  #        version: MyIMAP::VERSION,
  #        "support-url": "mailto:bugs@example.com",
  #        os: RbConfig::CONFIG["host_os"],
  #      )
  #    end
  #
  # See [ID[https://tools.ietf.org/html/rfc2971]] for field definitions.
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +ID+
  # [RFC2971[https://tools.ietf.org/html/rfc2971]].
  #
  # source://net-imap//lib/net/imap.rb#1136
  def id(client_id = T.unsafe(nil)); end

  # Sends an {IDLE command [RFC2177 §3]}[https://www.rfc-editor.org/rfc/rfc6851#section-3]
  # {[IMAP4rev2 §6.3.13]}[https://www.rfc-editor.org/rfc/rfc9051#section-6.3.13]
  # that waits for notifications of new or expunged messages.  Yields
  # responses from the server during the IDLE.
  #
  # Use #idle_done to leave IDLE.
  #
  # If +timeout+ is given, this method returns after +timeout+ seconds passed.
  # +timeout+ can be used for keep-alive.  For example, the following code
  # checks the connection for each 60 seconds.
  #
  #   loop do
  #     imap.idle(60) do |response|
  #       do_something_with(response)
  #       imap.idle_done if some_condition?(response)
  #     end
  #   end
  #
  # Returns the server's response to indicate the IDLE state has ended.
  # Returns +nil+ if the server does not respond to #idle_done within
  # {config.idle_response_timeout}[rdoc-ref:Config#idle_response_timeout]
  # seconds.
  #
  # Related: #idle_done, #noop, #check
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +IDLE+
  # [RFC2177[https://tools.ietf.org/html/rfc2177]].
  #
  # @raise [LocalJumpError]
  #
  # source://net-imap//lib/net/imap.rb#2452
  def idle(timeout = T.unsafe(nil), &response_handler); end

  # Leaves IDLE, allowing #idle to return.
  #
  # If the server does not respond within
  # {config.idle_response_timeout}[rdoc-ref:Config#idle_response_timeout]
  # seconds, #idle will return +nil+.
  #
  # Related: #idle
  #
  # source://net-imap//lib/net/imap.rb#2488
  def idle_done; end

  # Seconds to wait until an IDLE response is received.
  #
  # source://net-imap//lib/net/imap.rb#780
  def idle_response_timeout; end

  # Sends a {LIST command [IMAP4rev1 §6.3.8]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.8]
  # and returns a subset of names from the complete set of all names available
  # to the client.  +refname+ provides a context (for instance, a base
  # directory in a directory-based mailbox hierarchy).  +mailbox+ specifies a
  # mailbox or (via wildcards) mailboxes under that context.  Two wildcards
  # may be used in +mailbox+: <tt>"*"</tt>, which matches all characters
  # *including* the hierarchy delimiter (for instance, "/" on a UNIX-hosted
  # directory-based mailbox hierarchy); and <tt>"%"</tt>, which matches all
  # characters *except* the hierarchy delimiter.
  #
  # If +refname+ is empty, +mailbox+ is used directly to determine
  # which mailboxes to match.  If +mailbox+ is empty, the root
  # name of +refname+ and the hierarchy delimiter are returned.
  #
  # The return value is an array of MailboxList.
  #
  # Related: #lsub, MailboxList
  #
  # ===== For example:
  #
  #   imap.create("foo/bar")
  #   imap.create("foo/baz")
  #   p imap.list("", "foo/%")
  #   #=> [#<Net::IMAP::MailboxList attr=[:Noselect], delim="/", name="foo/">, \\
  #        #<Net::IMAP::MailboxList attr=[:Noinferiors, :Marked], delim="/", name="foo/bar">, \\
  #        #<Net::IMAP::MailboxList attr=[:Noinferiors], delim="/", name="foo/baz">]
  #
  # --
  # TODO: support LIST-EXTENDED extension [RFC5258].  Needed for IMAP4rev2.
  # ++
  #
  # source://net-imap//lib/net/imap.rb#1550
  def list(refname, mailbox); end

  # Sends a {LOGIN command [IMAP4rev1 §6.2.3]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.2.3]
  # to identify the client and carries the plaintext +password+ authenticating
  # this +user+.  If successful, the connection enters the "_authenticated_"
  # state.
  #
  # Using #authenticate {should be
  # preferred}[https://www.rfc-editor.org/rfc/rfc9051.html#name-login-command]
  # over #login.  The LOGIN command is not the same as #authenticate with the
  # "LOGIN" +mechanism+.
  #
  # A Net::IMAP::NoResponseError is raised if authentication fails.
  #
  # Related: #authenticate, #starttls
  #
  # ===== Capabilities
  #
  # An IMAP client MUST NOT call #login when the server advertises the
  # +LOGINDISABLED+ capability.
  #
  #    if imap.capability? "LOGINDISABLED"
  #      raise "Remote server has disabled the login command"
  #    else
  #      imap.login username, password
  #    end
  #
  # Server capabilities may change after #starttls, #login, and #authenticate.
  # Cached capabilities _must_ be invalidated after this method completes.
  # The TaggedResponse to #login may include updated capabilities in its
  # ResponseCode.
  #
  # source://net-imap//lib/net/imap.rb#1394
  def login(user, password); end

  # Sends a {LOGOUT command [IMAP4rev1 §6.1.3]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.1.3]
  # to inform the command to inform the server that the client is done with
  # the connection.
  #
  # Related: #disconnect, #logout!
  #
  # source://net-imap//lib/net/imap.rb#1164
  def logout; end

  # Calls #logout then, after receiving the TaggedResponse for the +LOGOUT+,
  # calls #disconnect.  Returns the TaggedResponse from +LOGOUT+.  Returns
  # +nil+ when the client is already disconnected, in contrast to #logout
  # which raises an exception.
  #
  # If #logout raises a StandardError, a warning will be printed but the
  # exception will not be re-raised.
  #
  # This is useful in situations where the connection must be dropped, for
  # example for security or after tests.  If logout errors need to be handled,
  # use #logout and #disconnect instead.
  #
  # Related: #logout, #disconnect
  #
  # source://net-imap//lib/net/imap.rb#1181
  def logout!; end

  # Sends a {LSUB command [IMAP4rev1 §6.3.9]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.9]
  # and returns a subset of names from the set of names that the user has
  # declared as being "active" or "subscribed."  +refname+ and +mailbox+ are
  # interpreted as for #list.
  #
  # The return value is an array of MailboxList objects.
  #
  # Related: #subscribe, #unsubscribe, #list, MailboxList
  #
  # source://net-imap//lib/net/imap.rb#1761
  def lsub(refname, mailbox); end

  # Sends a {MOVE command [RFC6851 §3.1]}[https://www.rfc-editor.org/rfc/rfc6851#section-3.1]
  # {[IMAP4rev2 §6.4.8]}[https://www.rfc-editor.org/rfc/rfc9051#section-6.4.8]
  # to move the specified message(s) to the end of the specified destination
  # +mailbox+. The +set+ parameter is a number, an array of numbers, or a
  # Range object. The number is a message sequence number.
  #
  # Related: #uid_move
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +MOVE+
  # [RFC6851[https://tools.ietf.org/html/rfc6851]].
  #
  # If +UIDPLUS+ [RFC4315[https://www.rfc-editor.org/rfc/rfc4315.html]] is
  # supported, the server's response should include a +COPYUID+ response code
  # with UIDPlusData.  This will report the UIDVALIDITY of the destination
  # mailbox, the UID set of the source messages, and the assigned UID set of
  # the moved messages.
  #
  # source://net-imap//lib/net/imap.rb#2233
  def move(set, mailbox); end

  # Sends a {NAMESPACE command [RFC2342 §5]}[https://www.rfc-editor.org/rfc/rfc2342#section-5]
  # and returns the namespaces that are available.  The NAMESPACE command
  # allows a client to discover the prefixes of namespaces used by a server
  # for personal mailboxes, other users' mailboxes, and shared mailboxes.
  #
  # The return value is a Namespaces object which has +personal+, +other+, and
  # +shared+ fields, each an array of Namespace objects.  These arrays will be
  # empty when the server responds with +nil+.
  #
  # Many \IMAP servers are configured with the default personal namespaces as
  # <tt>("" "/")</tt>: no prefix and the "+/+" hierarchy delimiter. In that
  # common case, the naive client may not have any trouble naming mailboxes.
  # But many servers are configured with the default personal namespace as
  # e.g.  <tt>("INBOX." ".")</tt>, placing all personal folders under INBOX,
  # with "+.+" as the hierarchy delimiter. If the client does not check for
  # this, but naively assumes it can use the same folder names for all
  # servers, then folder creation (and listing, moving, etc) can lead to
  # errors.
  #
  # From RFC2342[https://tools.ietf.org/html/rfc2342]:
  # >>>
  #    <em>Although typically a server will support only a single Personal
  #    Namespace, and a single Other User's Namespace, circumstances exist
  #    where there MAY be multiples of these, and a client MUST be prepared
  #    for them.  If a client is configured such that it is required to create
  #    a certain mailbox, there can be circumstances where it is unclear which
  #    Personal Namespaces it should create the mailbox in.  In these
  #    situations a client SHOULD let the user select which namespaces to
  #    create the mailbox in.</em>
  #
  # Related: #list, Namespaces, Namespace
  #
  # ===== For example:
  #
  #    if capable?("NAMESPACE")
  #      namespaces = imap.namespace
  #      if namespace = namespaces.personal.first
  #        prefix = namespace.prefix  # e.g. "" or "INBOX."
  #        delim  = namespace.delim   # e.g. "/" or "."
  #        # personal folders should use the prefix and delimiter
  #        imap.create(prefix + "foo")
  #        imap.create(prefix + "bar")
  #        imap.create(prefix + %w[path to my folder].join(delim))
  #      end
  #    end
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +NAMESPACE+
  # [RFC2342[https://tools.ietf.org/html/rfc2342]].
  #
  # source://net-imap//lib/net/imap.rb#1607
  def namespace; end

  # Sends a {NOOP command [IMAP4rev1 §6.1.2]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.1.2]
  # to the server.
  #
  # This allows the server to send unsolicited untagged EXPUNGE #responses,
  # but does not execute any client request.  \IMAP servers are permitted to
  # send unsolicited untagged responses at any time, except for +EXPUNGE+:
  #
  # * +EXPUNGE+ can only be sent while a command is in progress.
  # * +EXPUNGE+ must _not_ be sent during #fetch, #store, or #search.
  # * +EXPUNGE+ may be sent during #uid_fetch, #uid_store, or #uid_search.
  #
  # Related: #idle, #check
  #
  # source://net-imap//lib/net/imap.rb#1155
  def noop; end

  # Seconds to wait until a connection is opened.
  # If the IMAP object cannot open a connection within this time,
  # it raises a Net::OpenTimeout exception. The default value is 30 seconds.
  #
  # source://net-imap//lib/net/imap.rb#777
  def open_timeout; end

  # The port this client connected to
  #
  # source://net-imap//lib/net/imap.rb#786
  def port; end

  # Removes the response handler.
  #
  # Related: #add_response_handler, #response_handlers
  #
  # source://net-imap//lib/net/imap.rb#2624
  def remove_response_handler(handler); end

  # Sends a {RENAME command [IMAP4rev1 §6.3.5]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.5]
  # to change the name of the +mailbox+ to +newname+.
  #
  # A Net::IMAP::NoResponseError is raised if a mailbox with the
  # name +mailbox+ cannot be renamed to +newname+ for whatever
  # reason; for instance, because +mailbox+ does not exist, or
  # because there is already a mailbox with the name +newname+.
  #
  # Related: #create, #delete
  #
  # source://net-imap//lib/net/imap.rb#1491
  def rename(mailbox, newname); end

  # Returns all response handlers, including those that are added internally
  # by commands.  Each response handler will be called with every new
  # UntaggedResponse, TaggedResponse, and ContinuationRequest.
  #
  # Response handlers are called with a mutex inside the receiver thread.  New
  # responses cannot be processed and commands from other threads must wait
  # until all response_handlers return.  An exception will shut-down the
  # receiver thread and close the connection.
  #
  # For thread-safety, the returned array is a frozen copy of the internal
  # array.
  #
  # Related: #add_response_handler, #remove_response_handler
  #
  # source://net-imap//lib/net/imap.rb#2597
  def response_handlers; end

  # :call-seq:
  #   responses       {|hash|  ...} -> block result
  #   responses(type) {|array| ...} -> block result
  #
  # Yields unhandled responses and returns the result of the block.
  #
  # Unhandled responses are stored in a hash, with arrays of
  # <em>non-+nil+</em> UntaggedResponse#data keyed by UntaggedResponse#name
  # and ResponseCode#data keyed by ResponseCode#name.  Call without +type+ to
  # yield the entire responses hash.  Call with +type+ to yield only the array
  # of responses for that type.
  #
  # For example:
  #
  #   imap.select("inbox")
  #   p imap.responses("EXISTS", &:last)
  #   #=> 2
  #   p imap.responses("UIDVALIDITY", &:last)
  #   #=> 968263756
  #
  # >>>
  #   *Note:* Access to the responses hash is synchronized for thread-safety.
  #   The receiver thread and response_handlers cannot process new responses
  #   until the block completes.  Accessing either the response hash or its
  #   response type arrays outside of the block is unsafe.
  #
  #   Calling without a block is unsafe and deprecated.  Future releases will
  #   raise ArgumentError unless a block is given.
  #   See Config#responses_without_block.
  #
  # Previously unhandled responses are automatically cleared before entering a
  # mailbox with #select or #examine.  Long-lived connections can receive many
  # unhandled server responses, which must be pruned or they will continually
  # consume more memory.  Update or clear the responses hash or arrays inside
  # the block, or use #clear_responses.
  #
  # Only non-+nil+ data is stored.  Many important response codes have no data
  # of their own, but are used as "tags" on the ResponseText object they are
  # attached to.  ResponseText will be accessible by its response types:
  # "+OK+", "+NO+", "+BAD+", "+BYE+", or "+PREAUTH+".
  #
  # TaggedResponse#data is not saved to #responses, nor is any
  # ResponseCode#data on tagged responses.  Although some command methods do
  # return the TaggedResponse directly, #add_response_handler must be used to
  # handle all response codes.
  #
  # Related: #clear_responses, #response_handlers, #greeting
  #
  # source://net-imap//lib/net/imap.rb#2544
  def responses(type = T.unsafe(nil)); end

  # Sends a {SEARCH command [IMAP4rev1 §6.4.4]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.4]
  # to search the mailbox for messages that match the given searching
  # criteria, and returns message sequence numbers.  +keys+ can either be a
  # string holding the entire search string, or a single-dimension array of
  # search keywords and arguments.
  #
  # Returns a SearchResult object.  SearchResult inherits from Array (for
  # backward compatibility) but adds SearchResult#modseq when the +CONDSTORE+
  # capability has been enabled.
  #
  # Related: #uid_search
  #
  # ===== Search criteria
  #
  # For a full list of search criteria,
  # see [{IMAP4rev1 §6.4.4}[https://www.rfc-editor.org/rfc/rfc3501.html#section-6.4.4]],
  # or  [{IMAP4rev2 §6.4.4}[https://www.rfc-editor.org/rfc/rfc9051.html#section-6.4.4]],
  # in addition to documentation for
  # any [CAPABILITIES[https://www.iana.org/assignments/imap-capabilities/imap-capabilities.xhtml]]
  # reported by #capabilities which may define additional search filters, e.g:
  # +CONDSTORE+, +WITHIN+, +FILTERS+, <tt>SEARCH=FUZZY</tt>, +OBJECTID+, or
  # +SAVEDATE+.  The following are some common search criteria:
  #
  # <message set>:: a set of message sequence numbers.  "<tt>,</tt>" indicates
  #                 an interval, "+:+" indicates a range.  For instance,
  #                 "<tt>2,10:12,15</tt>" means "<tt>2,10,11,12,15</tt>".
  #
  # BEFORE <date>:: messages with an internal date strictly before
  #                 <b><date></b>.  The date argument has a format similar
  #                 to <tt>8-Aug-2002</tt>, and can be formatted using
  #                 Net::IMAP.format_date.
  #
  # BODY <string>:: messages that contain <string> within their body.
  #
  # CC <string>:: messages containing <string> in their CC field.
  #
  # FROM <string>:: messages that contain <string> in their FROM field.
  #
  # NEW:: messages with the \Recent, but not the \Seen, flag set.
  #
  # NOT <search-key>:: negate the following search key.
  #
  # OR <search-key> <search-key>:: "or" two search keys together.
  #
  # ON <date>:: messages with an internal date exactly equal to <date>,
  #             which has a format similar to 8-Aug-2002.
  #
  # SINCE <date>:: messages with an internal date on or after <date>.
  #
  # SUBJECT <string>:: messages with <string> in their subject.
  #
  # TO <string>:: messages with <string> in their TO field.
  #
  # ===== For example:
  #
  #   p imap.search(["SUBJECT", "hello", "NOT", "NEW"])
  #   #=> [1, 6, 7, 8]
  #
  # ===== Capabilities
  #
  # If [CONDSTORE[https://www.rfc-editor.org/rfc/rfc7162.html]] is supported
  # and enabled for the selected mailbox, a non-empty SearchResult will
  # include a +MODSEQ+ value.
  #   imap.select("mbox", condstore: true)
  #   result = imap.search(["SUBJECT", "hi there", "not", "new")
  #   #=> Net::IMAP::SearchResult[1, 6, 7, 8, modseq: 5594]
  #   result.modseq # => 5594
  #
  # source://net-imap//lib/net/imap.rb#2023
  def search(keys, charset = T.unsafe(nil)); end

  # Sends a {SELECT command [IMAP4rev1 §6.3.1]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.1]
  # to select a +mailbox+ so that messages in the +mailbox+ can be accessed.
  #
  # After you have selected a mailbox, you may retrieve the number of items in
  # that mailbox from <tt>imap.responses("EXISTS", &:last)</tt>, and the
  # number of recent messages from <tt>imap.responses("RECENT", &:last)</tt>.
  # Note that these values can change if new messages arrive during a session
  # or when existing messages are expunged; see #add_response_handler for a
  # way to detect these events.
  #
  # When the +condstore+ keyword argument is true, the server is told to
  # enable the extension.  If +mailbox+ supports persistence of mod-sequences,
  # the +HIGHESTMODSEQ+ ResponseCode will be sent as an untagged response to
  # #select and all `FETCH` responses will include FetchData#modseq.
  # Otherwise, the +NOMODSEQ+ ResponseCode will be sent.
  #
  # A Net::IMAP::NoResponseError is raised if the mailbox does not
  # exist or is for some reason non-selectable.
  #
  # Related: #examine
  #
  # ===== Capabilities
  #
  # If [UIDPLUS[https://www.rfc-editor.org/rfc/rfc4315.html]] is supported,
  # the server may return an untagged "NO" response with a "UIDNOTSTICKY"
  # response code indicating that the mailstore does not support persistent
  # UIDs:
  #   imap.responses("NO", &:last)&.code&.name == "UIDNOTSTICKY"
  #
  # If [CONDSTORE[https://www.rfc-editor.org/rfc/rfc7162.html]] is supported,
  # the +condstore+ keyword parameter may be used.
  #   imap.select("mbox", condstore: true)
  #   modseq = imap.responses("HIGHESTMODSEQ", &:last)
  #
  # source://net-imap//lib/net/imap.rb#1432
  def select(mailbox, condstore: T.unsafe(nil)); end

  # Sends a {SETACL command [RFC4314 §3.1]}[https://www.rfc-editor.org/rfc/rfc4314#section-3.1]
  # along with +mailbox+, +user+ and the +rights+ that user is to have on that
  # mailbox.  If +rights+ is nil, then that user will be stripped of any
  # rights to that mailbox.
  #
  # Related: #getacl
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +ACL+
  # [RFC4314[https://tools.ietf.org/html/rfc4314]].
  #
  # source://net-imap//lib/net/imap.rb#1728
  def setacl(mailbox, user, rights); end

  # Sends a {SETQUOTA command [RFC2087 §4.1]}[https://www.rfc-editor.org/rfc/rfc2087#section-4.1]
  # along with the specified +mailbox+ and +quota+.  If +quota+ is nil, then
  # +quota+ will be unset for that mailbox.  Typically one needs to be logged
  # in as a server admin for this to work.
  #
  # Related: #getquota, #getquotaroot
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +QUOTA+
  # [RFC2087[https://tools.ietf.org/html/rfc2087]].
  #
  # source://net-imap//lib/net/imap.rb#1708
  def setquota(mailbox, quota); end

  # Sends a {SORT command [RFC5256 §3]}[https://www.rfc-editor.org/rfc/rfc5256#section-3]
  # to search a mailbox for messages that match +search_keys+ and return an
  # array of message sequence numbers, sorted by +sort_keys+.  +search_keys+
  # are interpreted the same as for #search.
  #
  # --
  # TODO: describe +sort_keys+
  # ++
  #
  # Related: #uid_sort, #search, #uid_search, #thread, #uid_thread
  #
  # ===== For example:
  #
  #   p imap.sort(["FROM"], ["ALL"], "US-ASCII")
  #   #=> [1, 2, 3, 5, 6, 7, 8, 4, 9]
  #   p imap.sort(["DATE"], ["SUBJECT", "hello"], "US-ASCII")
  #   #=> [6, 7, 8, 1]
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +SORT+
  # [RFC5256[https://tools.ietf.org/html/rfc5256]].
  #
  # source://net-imap//lib/net/imap.rb#2277
  def sort(sort_keys, search_keys, charset); end

  # Returns the
  # {SSLContext}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html]
  # used by the SSLSocket when TLS is attempted, even when the TLS handshake
  # is unsuccessful.  The context object will be frozen.
  #
  # Returns +nil+ for a plaintext connection.
  #
  # source://net-imap//lib/net/imap.rb#794
  def ssl_ctx; end

  # Returns the parameters that were sent to #ssl_ctx
  # {set_params}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#method-i-set_params]
  # when the connection tries to use TLS (even when unsuccessful).
  #
  # Returns +false+ for a plaintext connection.
  #
  # source://net-imap//lib/net/imap.rb#801
  def ssl_ctx_params; end

  # Sends a {STARTTLS command [IMAP4rev1 §6.2.1]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.2.1]
  # to start a TLS session.
  #
  # Any +options+ are forwarded directly to
  # {OpenSSL::SSL::SSLContext#set_params}[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html#method-i-set_params];
  # the keys are names of attribute assignment methods on
  # SSLContext[https://docs.ruby-lang.org/en/master/OpenSSL/SSL/SSLContext.html].
  #
  # See DeprecatedClientOptions#starttls for deprecated arguments.
  #
  # This method returns after TLS negotiation and hostname verification are
  # both successful.  Any error indicates that the connection has not been
  # secured.
  #
  # *Note:*
  # >>>
  #   Any #response_handlers added before STARTTLS should be aware that the
  #   TaggedResponse to STARTTLS is sent clear-text, _before_ TLS negotiation.
  #   TLS starts immediately _after_ that response.  Any response code sent
  #   with the response (e.g. CAPABILITY) is insecure and cannot be trusted.
  #
  # Related: Net::IMAP.new, #login, #authenticate
  #
  # ===== Capability
  # Clients should not call #starttls unless the server advertises the
  # +STARTTLS+ capability.
  #
  # Server capabilities may change after #starttls, #login, and #authenticate.
  # Cached #capabilities will be cleared when this method completes.
  #
  # source://net-imap//lib/net/imap/deprecated_client_options.rb#104
  def starttls(*deprecated, **options); end

  # Sends a {STATUS command [IMAP4rev1 §6.3.10]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.10]
  # and returns the status of the indicated +mailbox+. +attr+ is a list of one
  # or more attributes whose statuses are to be requested.
  #
  # The return value is a hash of attributes.  Most status attributes return
  # integer values, but some return other value types (documented below).
  #
  # A Net::IMAP::NoResponseError is raised if status values
  # for +mailbox+ cannot be returned; for instance, because it
  # does not exist.
  #
  # ===== Supported attributes
  #
  # +MESSAGES+::    The number of messages in the mailbox.
  #
  # +UIDNEXT+::     The next unique identifier value of the mailbox.
  #
  # +UIDVALIDITY+:: The unique identifier validity value of the mailbox.
  #
  # +UNSEEN+::      The number of messages without the <tt>\Seen</tt> flag.
  #
  # +DELETED+::     The number of messages with the <tt>\Deleted</tt> flag.
  #
  # +SIZE+::
  #     The approximate size of the mailbox---must be greater than or equal to
  #     the sum of all messages' +RFC822.SIZE+ fetch item values.
  #
  # +HIGHESTMODSEQ+::
  #    The highest mod-sequence value of all messages in the mailbox.  See
  #    +CONDSTORE+ {[RFC7162]}[https://www.rfc-editor.org/rfc/rfc7162.html].
  #
  # +MAILBOXID+::
  #     A server-allocated unique _string_ identifier for the mailbox.  See
  #     +OBJECTID+ {[RFC8474]}[https://www.rfc-editor.org/rfc/rfc8474.html].
  #
  # +RECENT+::
  #     The number of messages with the <tt>\Recent</tt> flag.
  #     _NOTE:_ +RECENT+ was removed from IMAP4rev2.
  #
  # Unsupported attributes may be requested.  The attribute value will be
  # either an Integer or an ExtensionData object.
  #
  # ===== For example:
  #
  #   p imap.status("inbox", ["MESSAGES", "RECENT"])
  #   #=> {"RECENT"=>0, "MESSAGES"=>44}
  #
  # ===== Capabilities
  #
  # +SIZE+ requires the server's capabilities to include either +IMAP4rev2+ or
  # <tt>STATUS=SIZE</tt>
  # {[RFC8483]}[https://www.rfc-editor.org/rfc/rfc8483.html].
  #
  # +DELETED+ requires the server's capabilities to include +IMAP4rev2+.
  #
  # +HIGHESTMODSEQ+ requires the server's capabilities to include +CONDSTORE+
  # {[RFC7162]}[https://www.rfc-editor.org/rfc/rfc7162.html].
  #
  # +MAILBOXID+ requires the server's capabilities to include +OBJECTID+
  # {[RFC8474]}[https://www.rfc-editor.org/rfc/rfc8474.html].
  #
  # source://net-imap//lib/net/imap.rb#1828
  def status(mailbox, attr); end

  # :call-seq:
  #   store(set, attr, value, unchangedsince: nil) -> array of FetchData
  #
  # Sends a {STORE command [IMAP4rev1 §6.4.6]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.6]
  # to alter data associated with messages in the mailbox, in particular their
  # flags.
  #
  # +set+ is a number, an array of numbers, or a Range object.  Each number is
  # a message sequence number.
  #
  # +attr+ is the name of a data item to store.  The semantics of +value+
  # varies based on +attr+:
  # * When +attr+ is <tt>"FLAGS"</tt>, the flags in +value+ replace the
  #   message's flag list.
  # * When +attr+ is <tt>"+FLAGS"</tt>, the flags in +value+ are added to
  #   the flags for the message.
  # * When +attr+ is <tt>"-FLAGS"</tt>, the flags in +value+ are removed
  #   from the message.
  #
  # +unchangedsince+ is an optional integer mod-sequence.  It prohibits any
  # changes to messages with +mod-sequence+ greater than the specified
  # +unchangedsince+ value.  A SequenceSet of any messages that fail this
  # check will be returned in a +MODIFIED+ ResponseCode.
  #
  # The return value is an array of FetchData.
  #
  # Related: #uid_store
  #
  # ===== For example:
  #
  #   p imap.store(6..8, "+FLAGS", [:Deleted])
  #   #=> [#<Net::IMAP::FetchData seqno=6, attr={"FLAGS"=>[:Seen, :Deleted]}>,
  #        #<Net::IMAP::FetchData seqno=7, attr={"FLAGS"=>[:Seen, :Deleted]}>,
  #        #<Net::IMAP::FetchData seqno=8, attr={"FLAGS"=>[:Seen, :Deleted]}>]
  #
  # ===== Capabilities
  #
  # Extensions may define new data items to be used with #store.
  #
  # The server's capabilities must include +CONDSTORE+
  # {[RFC7162]}[https://tools.ietf.org/html/rfc7162] in order to use the
  # +unchangedsince+ argument.  Using +unchangedsince+ implicitly enables the
  # +CONDSTORE+ extension.
  #
  # source://net-imap//lib/net/imap.rb#2161
  def store(set, attr, flags, unchangedsince: T.unsafe(nil)); end

  # Sends a {SUBSCRIBE command [IMAP4rev1 §6.3.6]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.6]
  # to add the specified +mailbox+ name to the server's set of "active" or
  # "subscribed" mailboxes as returned by #lsub.
  #
  # A Net::IMAP::NoResponseError is raised if +mailbox+ cannot be
  # subscribed to; for instance, because it does not exist.
  #
  # Related: #unsubscribe, #lsub, #list
  #
  # source://net-imap//lib/net/imap.rb#1503
  def subscribe(mailbox); end

  # Sends a {THREAD command [RFC5256 §3]}[https://www.rfc-editor.org/rfc/rfc5256#section-3]
  # to search a mailbox and return message sequence numbers in threaded
  # format, as a ThreadMember tree.  +search_keys+ are interpreted the same as
  # for #search.
  #
  # The supported algorithms are:
  #
  # ORDEREDSUBJECT:: split into single-level threads according to subject,
  #                  ordered by date.
  # REFERENCES:: split into threads by parent/child relationships determined
  #              by which message is a reply to which.
  #
  # Unlike #search, +charset+ is a required argument.  US-ASCII
  # and UTF-8 are sample values.
  #
  # Related: #uid_thread, #search, #uid_search, #sort, #uid_sort
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +THREAD+
  # [RFC5256[https://tools.ietf.org/html/rfc5256]].
  #
  # source://net-imap//lib/net/imap.rb#2317
  def thread(algorithm, search_keys, charset); end

  # Returns true after the TLS negotiation has completed and the remote
  # hostname has been verified.  Returns false when TLS has been established
  # but peer verification was disabled.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap.rb#955
  def tls_verified?; end

  # Sends a {UID COPY command [IMAP4rev1 §6.4.8]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.8]
  # to copy the specified message(s) to the end of the specified destination
  # +mailbox+.
  #
  # Similar to #copy, but +set+ contains unique identifiers.
  #
  # ===== Capabilities
  #
  # +UIDPLUS+ affects #uid_copy the same way it affects #copy.
  #
  # source://net-imap//lib/net/imap.rb#2210
  def uid_copy(set, mailbox); end

  # Sends a {UID EXPUNGE command [RFC4315 §2.1]}[https://www.rfc-editor.org/rfc/rfc4315#section-2.1]
  # {[IMAP4rev2 §6.4.9]}[https://www.rfc-editor.org/rfc/rfc9051#section-6.4.9]
  # to permanently remove all messages that have both the <tt>\\Deleted</tt>
  # flag set and a UID that is included in +uid_set+.
  #
  # By using #uid_expunge instead of #expunge when resynchronizing with
  # the server, the client can ensure that it does not inadvertantly
  # remove any messages that have been marked as <tt>\\Deleted</tt> by other
  # clients between the time that the client was last connected and
  # the time the client resynchronizes.
  #
  # *Note:*
  # >>>
  #        Although the command takes a set of UIDs for its argument, the
  #        server still returns regular EXPUNGE responses, which contain
  #        a <em>sequence number</em>. These will be deleted from
  #        #responses and this method returns them as an array of
  #        <em>sequence number</em> integers.
  #
  # Related: #expunge
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +UIDPLUS+
  # [RFC4315[https://www.rfc-editor.org/rfc/rfc4315.html]].
  #
  # source://net-imap//lib/net/imap.rb#1949
  def uid_expunge(uid_set); end

  # :call-seq:
  #   uid_fetch(set, attr, changedsince: nil) -> array of FetchData
  #
  # Sends a {UID FETCH command [IMAP4rev1 §6.4.8]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.8]
  # to retrieve data associated with a message in the mailbox.
  #
  # Similar to #fetch, but the +set+ parameter contains unique identifiers
  # instead of message sequence numbers.
  #
  # >>>
  #   *Note:* Servers _MUST_ implicitly include the +UID+ message data item as
  #   part of any +FETCH+ response caused by a +UID+ command, regardless of
  #   whether a +UID+ was specified as a message data item to the +FETCH+.
  #
  # Related: #fetch, FetchData
  #
  # ===== Capabilities
  # Same as #fetch.
  #
  # source://net-imap//lib/net/imap.rb#2114
  def uid_fetch(set, attr, mod = T.unsafe(nil), changedsince: T.unsafe(nil)); end

  # Sends a {UID MOVE command [RFC6851 §3.2]}[https://www.rfc-editor.org/rfc/rfc6851#section-3.2]
  # {[IMAP4rev2 §6.4.9]}[https://www.rfc-editor.org/rfc/rfc9051#section-6.4.9]
  # to move the specified message(s) to the end of the specified destination
  # +mailbox+.
  #
  # Similar to #move, but +set+ contains unique identifiers.
  #
  # Related: #move
  #
  # ===== Capabilities
  #
  # Same as #move: The server's capabilities must include +MOVE+
  # [RFC6851[https://tools.ietf.org/html/rfc6851]].  +UIDPLUS+ also affects
  # #uid_move the same way it affects #move.
  #
  # source://net-imap//lib/net/imap.rb#2251
  def uid_move(set, mailbox); end

  # Sends a {UID SEARCH command [IMAP4rev1 §6.4.8]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.8]
  # to search the mailbox for messages that match the given searching
  # criteria, and returns unique identifiers (<tt>UID</tt>s).
  #
  # Returns a SearchResult object.  SearchResult inherits from Array (for
  # backward compatibility) but adds SearchResult#modseq when the +CONDSTORE+
  # capability has been enabled.
  #
  # See #search for documentation of search criteria.
  #
  # source://net-imap//lib/net/imap.rb#2036
  def uid_search(keys, charset = T.unsafe(nil)); end

  # Sends a {UID SORT command [RFC5256 §3]}[https://www.rfc-editor.org/rfc/rfc5256#section-3]
  # to search a mailbox for messages that match +search_keys+ and return an
  # array of unique identifiers, sorted by +sort_keys+.  +search_keys+ are
  # interpreted the same as for #search.
  #
  # Related: #sort, #search, #uid_search, #thread, #uid_thread
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +SORT+
  # [RFC5256[https://tools.ietf.org/html/rfc5256]].
  #
  # source://net-imap//lib/net/imap.rb#2292
  def uid_sort(sort_keys, search_keys, charset); end

  # :call-seq:
  #   uid_store(set, attr, value, unchangedsince: nil) -> array of FetchData
  #
  # Sends a {UID STORE command [IMAP4rev1 §6.4.8]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.4.8]
  # to alter data associated with messages in the mailbox, in particular their
  # flags.
  #
  # Similar to #store, but +set+ contains unique identifiers instead of
  # message sequence numbers.
  #
  # Related: #store
  #
  # ===== Capabilities
  # Same as #store.
  #
  # source://net-imap//lib/net/imap.rb#2179
  def uid_store(set, attr, flags, unchangedsince: T.unsafe(nil)); end

  # Sends a {UID THREAD command [RFC5256 §3]}[https://www.rfc-editor.org/rfc/rfc5256#section-3]
  # Similar to #thread, but returns unique identifiers instead of
  # message sequence numbers.
  #
  # Related: #thread, #search, #uid_search, #sort, #uid_sort
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +THREAD+
  # [RFC5256[https://tools.ietf.org/html/rfc5256]].
  #
  # source://net-imap//lib/net/imap.rb#2331
  def uid_thread(algorithm, search_keys, charset); end

  # Sends an {UNSELECT command [RFC3691 §2]}[https://www.rfc-editor.org/rfc/rfc3691#section-3]
  # {[IMAP4rev2 §6.4.2]}[https://www.rfc-editor.org/rfc/rfc9051#section-6.4.2]
  # to free the session resources for a mailbox and return to the
  # "_authenticated_" state.  This is the same as #close, except that
  # <tt>\\Deleted</tt> messages are not removed from the mailbox.
  #
  # Related: #close
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +UNSELECT+
  # [RFC3691[https://tools.ietf.org/html/rfc3691]].
  #
  # source://net-imap//lib/net/imap.rb#1908
  def unselect; end

  # Sends an {UNSUBSCRIBE command [IMAP4rev1 §6.3.7]}[https://www.rfc-editor.org/rfc/rfc3501#section-6.3.7]
  # to remove the specified +mailbox+ name from the server's set of "active"
  # or "subscribed" mailboxes.
  #
  # A Net::IMAP::NoResponseError is raised if +mailbox+ cannot be
  # unsubscribed from; for instance, because the client is not currently
  # subscribed to it.
  #
  # Related: #subscribe, #lsub, #list
  #
  # source://net-imap//lib/net/imap.rb#1516
  def unsubscribe(mailbox); end

  # Sends a XLIST command, and returns a subset of names from
  # the complete set of all names available to the client.
  # +refname+ provides a context (for instance, a base directory
  # in a directory-based mailbox hierarchy).  +mailbox+ specifies
  # a mailbox or (via wildcards) mailboxes under that context.
  # Two wildcards may be used in +mailbox+: '*', which matches
  # all characters *including* the hierarchy delimiter (for instance,
  # '/' on a UNIX-hosted directory-based mailbox hierarchy); and '%',
  # which matches all characters *except* the hierarchy delimiter.
  #
  # If +refname+ is empty, +mailbox+ is used directly to determine
  # which mailboxes to match.  If +mailbox+ is empty, the root
  # name of +refname+ and the hierarchy delimiter are returned.
  #
  # The XLIST command is like the LIST command except that the flags
  # returned refer to the function of the folder/mailbox, e.g. :Sent
  #
  # The return value is an array of MailboxList objects. For example:
  #
  #   imap.create("foo/bar")
  #   imap.create("foo/baz")
  #   p imap.xlist("", "foo/%")
  #   #=> [#<Net::IMAP::MailboxList attr=[:Noselect], delim="/", name="foo/">, \\
  #        #<Net::IMAP::MailboxList attr=[:Noinferiors, :Marked], delim="/", name="foo/bar">, \\
  #        #<Net::IMAP::MailboxList attr=[:Noinferiors], delim="/", name="foo/baz">]
  #
  # Related: #list, MailboxList
  #
  # ===== Capabilities
  #
  # The server's capabilities must include +XLIST+,
  # a deprecated Gmail extension (replaced by +SPECIAL-USE+).
  # --
  # TODO: Net::IMAP doesn't yet have full SPECIAL-USE support.  Supporting
  # servers MAY return SPECIAL-USE attributes, but are not *required* to
  # unless the SPECIAL-USE return option is supplied.
  # ++
  #
  # source://net-imap//lib/net/imap.rb#1651
  def xlist(refname, mailbox); end

  private

  # source://net-imap//lib/net/imap.rb#2964
  def build_ssl_ctx(ssl); end

  # NOTE: only call this for greeting, login, and authenticate
  #
  # source://net-imap//lib/net/imap.rb#2803
  def capabilities_from_resp_code(resp); end

  # source://net-imap//lib/net/imap.rb#2925
  def copy_internal(cmd, set, mailbox); end

  # source://net-imap//lib/net/imap.rb#2888
  def fetch_internal(cmd, set, attr, mod = T.unsafe(nil), changedsince: T.unsafe(nil)); end

  # source://net-imap//lib/net/imap.rb#2852
  def generate_tag; end

  # source://net-imap//lib/net/imap.rb#2766
  def get_response; end

  # @raise [Error]
  #
  # source://net-imap//lib/net/imap.rb#2645
  def get_server_greeting; end

  # source://net-imap//lib/net/imap.rb#2738
  def get_tagged_response(tag, cmd, timeout = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap.rb#2953
  def normalize_searching_criteria(keys); end

  # source://net-imap//lib/net/imap.rb#2857
  def put_string(str); end

  # source://net-imap//lib/net/imap.rb#2671
  def receive_responses; end

  # store name => [..., data]
  #
  # source://net-imap//lib/net/imap.rb#2790
  def record_untagged_response(resp); end

  # store code.name => [..., code.data]
  #
  # source://net-imap//lib/net/imap.rb#2796
  def record_untagged_response_code(resp); end

  # source://net-imap//lib/net/imap.rb#2993
  def sasl_adapter; end

  # source://net-imap//lib/net/imap.rb#2872
  def search_internal(cmd, keys, charset); end

  # source://net-imap//lib/net/imap.rb#2824
  def send_command(cmd, *args, &block); end

  # Calls send_command, yielding the text of each ContinuationRequest and
  # responding with each block result.  Returns TaggedResponse.  Raises
  # NoResponseError or BadResponseError.
  #
  # source://net-imap//lib/net/imap.rb#2815
  def send_command_with_continuations(cmd, *args); end

  # source://net-imap//lib/net/imap/command_data.rb#33
  def send_data(data, tag = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/command_data.rb#115
  def send_date_data(date); end

  # source://net-imap//lib/net/imap/command_data.rb#101
  def send_list_data(list, tag = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/command_data.rb#80
  def send_literal(str, tag = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/command_data.rb#97
  def send_number_data(num); end

  # source://net-imap//lib/net/imap/command_data.rb#76
  def send_quoted_string(str); end

  # source://net-imap//lib/net/imap/command_data.rb#54
  def send_string_data(str, tag = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/command_data.rb#118
  def send_symbol_data(symbol); end

  # source://net-imap//lib/net/imap/command_data.rb#116
  def send_time_data(time); end

  # source://net-imap//lib/net/imap.rb#2929
  def sort_internal(cmd, sort_keys, search_keys, charset); end

  # source://net-imap//lib/net/imap.rb#2636
  def start_imap_connection; end

  # source://net-imap//lib/net/imap.rb#2653
  def start_receiver_thread; end

  # source://net-imap//lib/net/imap.rb#2979
  def start_tls_session; end

  # source://net-imap//lib/net/imap.rb#2913
  def store_internal(cmd, set, attr, flags, unchangedsince: T.unsafe(nil)); end

  # source://net-imap//lib/net/imap.rb#2662
  def tcp_socket(host, port); end

  # source://net-imap//lib/net/imap.rb#2941
  def thread_internal(cmd, algorithm, search_keys, charset); end

  # source://net-imap//lib/net/imap/command_data.rb#12
  def validate_data(data); end

  class << self
    # Returns the global Config object
    #
    # source://net-imap//lib/net/imap.rb#739
    def config; end

    # Returns the global debug mode.
    #
    # source://net-imap//lib/net/imap.rb#742
    def debug; end

    # Sets the global debug mode.
    #
    # source://net-imap//lib/net/imap.rb#745
    def debug=(val); end

    # :call-seq: decode_date(string) -> Date
    #
    # Decodes +string+ as an IMAP formatted "date".
    #
    # Double quotes are optional.  Day of month may be padded with zero or
    # space.  See STRFDATE.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#90
    def decode_date(string); end

    # :call-seq: decode_datetime(string) -> DateTime
    #
    # Decodes +string+ as an IMAP4 formatted "date-time".
    #
    # NOTE: Although double-quotes are not optional in the IMAP grammar,
    # Net::IMAP currently parses "date-time" values as "quoted" strings and this
    # removes the quotation marks.  To be useful for strings which have already
    # been parsed as a quoted string, this method makes double-quotes optional.
    #
    # See STRFTIME.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#112
    def decode_datetime(string); end

    # :call-seq: decode_time(string) -> Time
    #
    # Decodes +string+ as an IMAP4 formatted "date-time".
    #
    # Same as +decode_datetime+, but returning a Time instead.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#124
    def decode_time(string); end

    # Decode a string from modified UTF-7 format to UTF-8.
    #
    # UTF-7 is a 7-bit encoding of Unicode [UTF7].  IMAP uses a
    # slightly modified version of this to encode mailbox names
    # containing non-ASCII characters; see [IMAP] section 5.1.3.
    #
    # Net::IMAP does _not_ automatically encode and decode
    # mailbox names to and from UTF-7.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#57
    def decode_utf7(s); end

    # The default port for IMAP connections, port 143
    #
    # source://net-imap//lib/net/imap.rb#750
    def default_imap_port; end

    # The default port for IMAPS connections, port 993
    #
    # source://net-imap//lib/net/imap.rb#755
    def default_imaps_port; end

    # The default port for IMAP connections, port 143
    #
    # source://net-imap//lib/net/imap.rb#750
    def default_port; end

    # The default port for IMAPS connections, port 993
    #
    # source://net-imap//lib/net/imap.rb#755
    def default_ssl_port; end

    # The default port for IMAPS connections, port 993
    #
    # source://net-imap//lib/net/imap.rb#755
    def default_tls_port; end

    # Formats +time+ as an IMAP4 date.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#80
    def encode_date(date); end

    # :call-seq: encode_datetime(time) -> string
    #
    # Formats +time+ as an IMAP4 date-time.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#98
    def encode_datetime(time); end

    # :call-seq: encode_datetime(time) -> string
    #
    # Formats +time+ as an IMAP4 date-time.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#98
    def encode_time(time); end

    # Encode a string from UTF-8 format to modified UTF-7.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#68
    def encode_utf7(s); end

    # Formats +time+ as an IMAP4 date.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#80
    def format_date(date); end

    # DEPRECATED:: The original version returned incorrectly formatted strings.
    #              Strings returned by encode_datetime or format_time use the
    #              correct IMAP4rev1 syntax for "date-time".
    #
    # This invalid format has been temporarily retained for backward
    # compatibility.  A future release will change this method to return the
    # correct format.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#149
    def format_datetime(time); end

    # :call-seq: encode_datetime(time) -> string
    #
    # Formats +time+ as an IMAP4 date-time.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#98
    def format_time(time); end

    # :call-seq: decode_date(string) -> Date
    #
    # Decodes +string+ as an IMAP formatted "date".
    #
    # Double quotes are optional.  Day of month may be padded with zero or
    # space.  See STRFDATE.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#90
    def parse_date(string); end

    # :call-seq: decode_datetime(string) -> DateTime
    #
    # Decodes +string+ as an IMAP4 formatted "date-time".
    #
    # NOTE: Although double-quotes are not optional in the IMAP grammar,
    # Net::IMAP currently parses "date-time" values as "quoted" strings and this
    # removes the quotation marks.  To be useful for strings which have already
    # been parsed as a quoted string, this method makes double-quotes optional.
    #
    # See STRFTIME.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#112
    def parse_datetime(string); end

    # :call-seq: decode_time(string) -> Time
    #
    # Decodes +string+ as an IMAP4 formatted "date-time".
    #
    # Same as +decode_datetime+, but returning a Time instead.
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#124
    def parse_time(string); end

    # --
    # We could get the saslprep method by extending the SASLprep module
    # directly.  It's done indirectly, so SASLprep can be lazily autoloaded,
    # because most users won't need it.
    # ++
    # Delegates to Net::IMAP::StringPrep::SASLprep#saslprep.
    #
    # source://net-imap//lib/net/imap.rb#3003
    def saslprep(string, **opts); end
  end
end

# Mailbox attribute indicating that this mailbox presents all messages in
# the user's message store. Implementations MAY omit some messages, such as,
# perhaps, those in \Trash and \Junk. When this special use is supported, it
# is almost certain to represent a virtual mailbox
#
# source://net-imap//lib/net/imap/flags.rb#218
Net::IMAP::ALL = T.let(T.unsafe(nil), Symbol)

# Mailbox attribute indicating that this mailbox is used to archive
# messages. The meaning of an "archival" mailbox is server dependent;
# typically, it will be used to get messages out of the inbox, or otherwise
# keep them out of the user's way, while still making them accessible
#
# source://net-imap//lib/net/imap/flags.rb#224
Net::IMAP::ARCHIVE = T.let(T.unsafe(nil), Symbol)

# source://net-imap//lib/net/imap/command_data.rb#137
class Net::IMAP::Atom
  # @return [Atom] a new instance of Atom
  #
  # source://net-imap//lib/net/imap/command_data.rb#147
  def initialize(data); end

  # source://net-imap//lib/net/imap/command_data.rb#138
  def send_data(imap, tag); end

  # source://net-imap//lib/net/imap/command_data.rb#142
  def validate; end
end

# Backward compatible delegators from Net::IMAP to Net::IMAP::SASL.
#
# source://net-imap//lib/net/imap/authenticators.rb#4
module Net::IMAP::Authenticators
  # Deprecated.  Use Net::IMAP::SASL.add_authenticator instead.
  #
  # source://net-imap//lib/net/imap/authenticators.rb#7
  def add_authenticator(*_arg0, **_arg1, &_arg2); end

  # Deprecated.  Use Net::IMAP::SASL.authenticator instead.
  #
  # source://net-imap//lib/net/imap/authenticators.rb#18
  def authenticator(*_arg0, **_arg1, &_arg2); end
end

# Net::IMAP::BodyStructure is included by all of the structs that can be
# returned from a <tt>"BODYSTRUCTURE"</tt> or <tt>"BODY"</tt>
# FetchData#attr value.  Although these classes don't share a base class,
# this module can be used to pattern match all of them.
#
# See {[IMAP4rev1] §7.4.2}[https://www.rfc-editor.org/rfc/rfc3501.html#section-7.4.2]
# and {[IMAP4rev2] §7.5.2}[https://www.rfc-editor.org/rfc/rfc9051.html#section-7.5.2-4.9]
# for full description of all +BODYSTRUCTURE+ fields, and also
# Net::IMAP@Message+envelope+and+body+structure for other relevant RFCs.
#
# === Classes that include BodyStructure
# BodyTypeBasic:: Represents any message parts that are not handled by
#                 BodyTypeText, BodyTypeMessage, or BodyTypeMultipart.
# BodyTypeText:: Used by <tt>text/*</tt> parts.  Contains all of the
#                BodyTypeBasic fields.
# BodyTypeMessage:: Used by <tt>message/rfc822</tt> and
#                   <tt>message/global</tt> parts.  Contains all of the
#                   BodyTypeBasic fields.  Other <tt>message/*</tt> types
#                   should use BodyTypeBasic.
# BodyTypeMultipart:: for <tt>multipart/*</tt> parts
#
# source://net-imap//lib/net/imap/response_data.rb#813
module Net::IMAP::BodyStructure; end

# BodyTypeAttachment is not used and will be removed in an upcoming release.
#
# === Bug Analysis
#
# \IMAP body structures are parenthesized lists and assign their fields
# positionally, so missing fields change the interpretation of all
# following fields.  Additionally, different body types have a different
# number of required fields, followed by optional "extension" fields.
#
# BodyTypeAttachment was previously returned when a "message/rfc822" part,
# which should be sent as <tt>body-type-msg</tt> with ten required fields,
# was actually sent as a <tt>body-type-basic</tt> with _seven_ required
# fields.
#
#   basic => type, subtype, param, id, desc, enc, octets, md5=nil,  dsp=nil, lang=nil, loc=nil, *ext
#   msg   => type, subtype, param, id, desc, enc, octets, envelope, body,    lines,    md5=nil, ...
#
# Normally, +envelope+ and +md5+ are incompatible, but Net::IMAP leniently
# allowed buggy servers to send +NIL+ for +envelope+.  As a result, when a
# server sent a <tt>message/rfc822</tt> part with +NIL+ for +md5+ and a
# non-<tt>NIL</tt> +dsp+, Net::IMAP misinterpreted the
# <tt>Content-Disposition</tt> as if it were a strange body type.  In all
# reported cases, the <tt>Content-Disposition</tt> was "attachment", so
# BodyTypeAttachment was created as the workaround.
#
# === Current behavior
#
# When interpreted strictly, +envelope+ and +md5+ are incompatible.  So the
# current parsing algorithm peeks ahead after it has received the seventh
# body field.  If the next token is not the start of an +envelope+, we assume
# the server has incorrectly sent us a <tt>body-type-basic</tt> and return
# BodyTypeBasic.  As a result, what was previously BodyTypeMessage#body =>
# BodyTypeAttachment is now BodyTypeBasic#disposition => ContentDisposition.
#
# source://net-imap//lib/net/imap/response_data.rb#1077
class Net::IMAP::BodyTypeAttachment < ::Struct
  # *invalid for BodyTypeAttachment*
  #
  # source://net-imap//lib/net/imap/response_data.rb#1079
  def media_type; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_data.rb#1107
  def multipart?; end

  # *invalid for BodyTypeAttachment*
  #
  # source://net-imap//lib/net/imap/response_data.rb#1087
  def subtype; end
end

# Net::IMAP::BodyTypeBasic represents basic body structures of messages and
# message parts, unless they have a <tt>Content-Type</tt> that is handled by
# BodyTypeText, BodyTypeMessage, or BodyTypeMultipart.
#
# See {[IMAP4rev1] §7.4.2}[https://www.rfc-editor.org/rfc/rfc3501.html#section-7.4.2]
# and {[IMAP4rev2] §7.5.2}[https://www.rfc-editor.org/rfc/rfc9051.html#section-7.5.2-4.9]
# for full description of all +BODYSTRUCTURE+ fields, and also
# Net::IMAP@Message+envelope+and+body+structure for other relevant RFCs.
#
# source://net-imap//lib/net/imap/response_data.rb#830
class Net::IMAP::BodyTypeBasic < ::Struct
  include ::Net::IMAP::BodyStructure

  # :call-seq: media_subtype -> subtype
  #
  # >>>
  #   [Obsolete]
  #     Use +subtype+ instead.  Calling this will generate a warning message
  #     to +stderr+, then return the value of +subtype+.
  # --
  # TODO: why not just keep this as an alias?  Would "media_subtype" be used
  # for something else?
  # ++
  #
  # source://net-imap//lib/net/imap/response_data.rb#941
  def media_subtype; end

  # :call-seq: multipart? -> false
  #
  # BodyTypeBasic is not used for multipart MIME parts.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_data.rb#927
  def multipart?; end
end

# === Obsolete
# BodyTypeExtension is not used and will be removed in an upcoming release.
#
# >>>
#   BodyTypeExtension was (incorrectly) used for <tt>message/*</tt> parts
#   (besides <tt>message/rfc822</tt>, which correctly uses BodyTypeMessage).
#
#   Net::IMAP now (correctly) parses all message types (other than
#   <tt>message/rfc822</tt> or <tt>message/global</tt>) as BodyTypeBasic.
#
# source://net-imap//lib/net/imap/response_data.rb#1201
class Net::IMAP::BodyTypeExtension < ::Struct
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_data.rb#1202
  def multipart?; end
end

# Net::IMAP::BodyTypeMessage represents the body structures of messages and
# message parts, when <tt>Content-Type</tt> is <tt>message/rfc822</tt> or
# <tt>message/global</tt>.
#
# BodyTypeMessage contains all of the fields of BodyTypeBasic.  See
# BodyTypeBasic for documentation of the following fields:
# * {media_type}[rdoc-ref:BodyTypeBasic#media_type]
# * subtype[rdoc-ref:BodyTypeBasic#subtype]
# * param[rdoc-ref:BodyTypeBasic#param]
# * {content_id}[rdoc-ref:BodyTypeBasic#content_id]
# * description[rdoc-ref:BodyTypeBasic#description]
# * encoding[rdoc-ref:BodyTypeBasic#encoding]
# * size[rdoc-ref:BodyTypeBasic#size]
#
# source://net-imap//lib/net/imap/response_data.rb#1011
class Net::IMAP::BodyTypeMessage < ::Struct
  include ::Net::IMAP::BodyStructure

  # Obsolete: use +subtype+ instead.  Calling this will
  # generate a warning message to +stderr+, then return
  # the value of +subtype+.
  #
  # source://net-imap//lib/net/imap/response_data.rb#1037
  def media_subtype; end

  # :call-seq: multipart? -> false
  #
  # BodyTypeMessage is not used for multipart MIME parts.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_data.rb#1030
  def multipart?; end
end

# Net::IMAP::BodyTypeMultipart represents body structures of messages and
# message parts, when <tt>Content-Type</tt> is <tt>multipart/*</tt>.
#
# source://net-imap//lib/net/imap/response_data.rb#1120
class Net::IMAP::BodyTypeMultipart < ::Struct
  include ::Net::IMAP::BodyStructure

  # Obsolete: use +subtype+ instead.  Calling this will
  # generate a warning message to +stderr+, then return
  # the value of +subtype+.
  #
  # source://net-imap//lib/net/imap/response_data.rb#1184
  def media_subtype; end

  # :call-seq: multipart? -> true
  #
  # BodyTypeMultipart is used for multipart MIME parts.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_data.rb#1176
  def multipart?; end
end

# Net::IMAP::BodyTypeText represents the body structures of messages and
# message parts, when <tt>Content-Type</tt> is <tt>text/*</tt>.
#
# BodyTypeText contains all of the fields of BodyTypeBasic.  See
# BodyTypeBasic for documentation of the following:
# * {media_type}[rdoc-ref:BodyTypeBasic#media_type]
# * subtype[rdoc-ref:BodyTypeBasic#subtype]
# * param[rdoc-ref:BodyTypeBasic#param]
# * {content_id}[rdoc-ref:BodyTypeBasic#content_id]
# * description[rdoc-ref:BodyTypeBasic#description]
# * encoding[rdoc-ref:BodyTypeBasic#encoding]
# * size[rdoc-ref:BodyTypeBasic#size]
#
# source://net-imap//lib/net/imap/response_data.rb#966
class Net::IMAP::BodyTypeText < ::Struct
  include ::Net::IMAP::BodyStructure

  # Obsolete: use +subtype+ instead.  Calling this will
  # generate a warning message to +stderr+, then return
  # the value of +subtype+.
  #
  # source://net-imap//lib/net/imap/response_data.rb#986
  def media_subtype; end

  # :call-seq: multipart? -> false
  #
  # BodyTypeText is not used for multipart MIME parts.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_data.rb#979
  def multipart?; end
end

# source://net-imap//lib/net/imap/command_data.rb#238
class Net::IMAP::ClientID
  # @return [ClientID] a new instance of ClientID
  #
  # source://net-imap//lib/net/imap/command_data.rb#250
  def initialize(data); end

  # source://net-imap//lib/net/imap/command_data.rb#240
  def send_data(imap, tag); end

  # source://net-imap//lib/net/imap/command_data.rb#244
  def validate; end

  private

  # source://net-imap//lib/net/imap/command_data.rb#264
  def format_internal(client_id); end

  # source://net-imap//lib/net/imap/command_data.rb#254
  def validate_internal(client_id); end
end

# Net::IMAP::Config stores configuration options for Net::IMAP clients.
# The global configuration can be seen at either Net::IMAP.config or
# Net::IMAP::Config.global, and the client-specific configuration can be
# seen at Net::IMAP#config.
#
# When creating a new client, all unhandled keyword arguments to
# Net::IMAP.new are delegated to Config.new.  Every client has its own
# config.
#
#   debug_client = Net::IMAP.new(hostname, debug: true)
#   quiet_client = Net::IMAP.new(hostname, debug: false)
#   debug_client.config.debug?  # => true
#   quiet_client.config.debug?  # => false
#
# == Inheritance
#
# Configs have a parent[rdoc-ref:Config::AttrInheritance#parent] config, and
# any attributes which have not been set locally will inherit the parent's
# value.  Every client creates its own specific config.  By default, client
# configs inherit from Config.global.
#
#   plain_client = Net::IMAP.new(hostname)
#   debug_client = Net::IMAP.new(hostname, debug: true)
#   quiet_client = Net::IMAP.new(hostname, debug: false)
#
#   plain_client.config.inherited?(:debug)  # => true
#   debug_client.config.inherited?(:debug)  # => false
#   quiet_client.config.inherited?(:debug)  # => false
#
#   plain_client.config.debug?  # => false
#   debug_client.config.debug?  # => true
#   quiet_client.config.debug?  # => false
#
#   # Net::IMAP.debug is delegated to Net::IMAP::Config.global.debug
#   Net::IMAP.debug = true
#   plain_client.config.debug?  # => true
#   debug_client.config.debug?  # => true
#   quiet_client.config.debug?  # => false
#
#   Net::IMAP.debug = false
#   plain_client.config.debug = true
#   plain_client.config.inherited?(:debug)  # => false
#   plain_client.config.debug?  # => true
#   plain_client.config.reset(:debug)
#   plain_client.config.inherited?(:debug)  # => true
#   plain_client.config.debug?  # => false
#
# == Versioned defaults
#
# The effective default configuration for a specific +x.y+ version of
# +net-imap+ can be loaded with the +config+ keyword argument to
# Net::IMAP.new.  Requesting default configurations for previous versions
# enables extra backward compatibility with those versions:
#
#   client = Net::IMAP.new(hostname, config: 0.3)
#   client.config.sasl_ir                  # => false
#   client.config.responses_without_block  # => :silence_deprecation_warning
#
#   client = Net::IMAP.new(hostname, config: 0.4)
#   client.config.sasl_ir                  # => true
#   client.config.responses_without_block  # => :silence_deprecation_warning
#
#   client = Net::IMAP.new(hostname, config: 0.5)
#   client.config.sasl_ir                  # => true
#   client.config.responses_without_block  # => :warn
#
#   client = Net::IMAP.new(hostname, config: :future)
#   client.config.sasl_ir                  # => true
#   client.config.responses_without_block  # => :raise
#
# The versioned default configs inherit certain specific config options from
# Config.global, for example #debug:
#
#   client = Net::IMAP.new(hostname, config: 0.4)
#   Net::IMAP.debug = false
#   client.config.debug?  # => false
#
#   Net::IMAP.debug = true
#   client.config.debug?  # => true
#
# Use #load_defaults to globally behave like a specific version:
#   client = Net::IMAP.new(hostname)
#   client.config.sasl_ir              # => true
#   Net::IMAP.config.load_defaults 0.3
#   client.config.sasl_ir              # => false
#
# === Named defaults
# In addition to +x.y+ version numbers, the following aliases are supported:
#
# [+:default+]
#   An alias for +:current+.
#
#   >>>
#   *NOTE*: This is _not_ the same as Config.default.  It inherits some
#   attributes from Config.global, for example: #debug.
# [+:current+]
#   An alias for the current +x.y+ version's defaults.
# [+:next+]
#   The _planned_ config for the next +x.y+ version.
# [+:future+]
#   The _planned_ eventual config for some future +x.y+ version.
#
# For example, to raise exceptions for all current deprecations:
#   client = Net::IMAP.new(hostname, config: :future)
#   client.responses  # raises an ArgumentError
#
# == Thread Safety
#
# *NOTE:* Updates to config objects are not synchronized for thread-safety.
#
# source://net-imap//lib/net/imap/config/attr_accessors.rb#7
class Net::IMAP::Config
  include ::Net::IMAP::Config::AttrAccessors
  include ::Net::IMAP::Config::AttrInheritance
  include ::Net::IMAP::Config::AttrTypeCoercion
  extend ::Net::IMAP::Config::AttrAccessors::Macros
  extend ::Net::IMAP::Config::AttrInheritance::Macros
  extend ::Net::IMAP::Config::AttrTypeCoercion::Macros

  # Creates a new config object and initialize its attribute with +attrs+.
  #
  # If +parent+ is not given, the global config is used by default.
  #
  # If a block is given, the new config object is yielded to it.
  #
  # @return [Config] a new instance of Config
  # @yield [_self]
  # @yieldparam _self [Net::IMAP::Config] the object that the method was called on
  #
  # source://net-imap//lib/net/imap/config.rb#236
  def initialize(parent = T.unsafe(nil), **attrs); end

  # :call-seq: load_defaults(version) -> self
  #
  # Resets the current config to behave like the versioned default
  # configuration for +version+.  #parent will not be changed.
  #
  # Some config attributes default to inheriting from their #parent (which
  # is usually Config.global) and are left unchanged, for example: #debug.
  #
  # See Config@Versioned+defaults and Config@Named+defaults.
  #
  # source://net-imap//lib/net/imap/config.rb#287
  def load_defaults(version); end

  # :call-seq: to_h -> hash
  #
  # Returns all config attributes in a hash.
  #
  # source://net-imap//lib/net/imap/config.rb#296
  def to_h; end

  # :call-seq: update(**attrs) -> self
  #
  # Assigns all of the provided +attrs+ to this config, and returns +self+.
  #
  # An ArgumentError is raised unless every key in +attrs+ matches an
  # assignment method on Config.
  #
  # >>>
  #   *NOTE:*  #update is not atomic.  If an exception is raised due to an
  #   invalid attribute value, +attrs+ may be partially applied.
  #
  # source://net-imap//lib/net/imap/config.rb#252
  def update(**attrs); end

  # :call-seq:
  #   with(**attrs) -> config
  #   with(**attrs) {|config| } -> result
  #
  # Without a block, returns a new config which inherits from self.  With a
  # block, yields the new config and returns the block's result.
  #
  # If no keyword arguments are given, an ArgumentError will be raised.
  #
  # If +self+ is frozen, the copy will also be frozen.
  #
  # source://net-imap//lib/net/imap/config.rb#270
  def with(**attrs); end

  protected

  # source://net-imap//lib/net/imap/config.rb#300
  def defaults_hash; end

  class << self
    # :call-seq:
    #  Net::IMAP::Config[number] -> versioned config
    #  Net::IMAP::Config[symbol] -> named config
    #  Net::IMAP::Config[hash]   -> new frozen config
    #  Net::IMAP::Config[config] -> same config
    #
    # Given a version number, returns the default configuration for the target
    # version.  See Config@Versioned+defaults.
    #
    # Given a version name, returns the default configuration for the target
    # version.  See Config@Named+defaults.
    #
    # Given a Hash, creates a new _frozen_ config which inherits from
    # Config.global.  Use Config.new for an unfrozen config.
    #
    # Given a config, returns that same config.
    #
    # source://net-imap//lib/net/imap/config.rb#151
    def [](config); end

    # The default config, which is hardcoded and frozen.
    #
    # source://net-imap//lib/net/imap/config.rb#126
    def default; end

    # The global config object.  Also available from Net::IMAP.config.
    #
    # source://net-imap//lib/net/imap/config.rb#129
    def global; end

    # A hash of hard-coded configurations, indexed by version number.
    #
    # source://net-imap//lib/net/imap/config.rb#132
    def version_defaults; end
  end
end

# >>>
#   *NOTE:* This module is an internal implementation detail, with no
#   guarantee of backward compatibility.
#
# +attr_accessor+ values are stored in a struct rather than ivars, making
# it simpler to ensure that all config objects share a single object
# shape.  This also simplifies iteration over all defined attributes.
#
# source://net-imap//lib/net/imap/config/attr_accessors.rb#15
module Net::IMAP::Config::AttrAccessors
  extend ::Forwardable

  mixes_in_class_methods ::Net::IMAP::Config::AttrAccessors::Macros

  # :notnew:
  #
  # source://net-imap//lib/net/imap/config/attr_accessors.rb#45
  def initialize; end

  # source://forwardable/1.3.3/forwardable.rb#231
  def debug(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def debug=(*args, **_arg1, &block); end

  # Freezes the internal attributes struct, in addition to +self+.
  #
  # source://net-imap//lib/net/imap/config/attr_accessors.rb#51
  def freeze; end

  # source://forwardable/1.3.3/forwardable.rb#231
  def idle_response_timeout(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def idle_response_timeout=(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def open_timeout(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def open_timeout=(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def responses_without_block(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def responses_without_block=(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def sasl_ir(*args, **_arg1, &block); end

  # source://forwardable/1.3.3/forwardable.rb#231
  def sasl_ir=(*args, **_arg1, &block); end

  protected

  # source://net-imap//lib/net/imap/config/attr_accessors.rb#58
  def data; end

  private

  # source://net-imap//lib/net/imap/config/attr_accessors.rb#62
  def initialize_clone(other); end

  # source://net-imap//lib/net/imap/config/attr_accessors.rb#67
  def initialize_dup(other); end

  class << self
    # source://net-imap//lib/net/imap/config/attr_accessors.rb#28
    def attr_accessor(name); end

    # source://net-imap//lib/net/imap/config/attr_accessors.rb#38
    def struct; end

    private

    # source://net-imap//lib/net/imap/config/attr_accessors.rb#33
    def attributes; end

    # @private
    #
    # source://net-imap//lib/net/imap/config/attr_accessors.rb#21
    def included(mod); end
  end
end

# source://net-imap//lib/net/imap/config/attr_accessors.rb#16
module Net::IMAP::Config::AttrAccessors::Macros
  # source://net-imap//lib/net/imap/config/attr_accessors.rb#17
  def attr_accessor(name); end
end

# source://net-imap//lib/net/imap/config.rb#0
class Net::IMAP::Config::AttrAccessors::Struct < ::Struct
  def debug; end
  def debug=(_); end
  def idle_response_timeout; end
  def idle_response_timeout=(_); end
  def open_timeout; end
  def open_timeout=(_); end
  def responses_without_block; end
  def responses_without_block=(_); end
  def sasl_ir; end
  def sasl_ir=(_); end

  class << self
    def [](*_arg0); end
    def inspect; end
    def keyword_init?; end
    def members; end
    def new(*_arg0); end
  end
end

# >>>
#   *NOTE:* The public methods on this module are part of the stable
#   public API of Net::IMAP::Config.  But the module itself is an internal
#   implementation detail, with no guarantee of backward compatibility.
#
# +attr_accessor+ methods will delegate to their #parent when the local
# value does not contain an override.  Inheritance forms a singly linked
# list, so lookup will be <tt>O(n)</tt> on the number of ancestors.  In
# practice, the ancestor chain is not expected to be long.  Without
# customization, it is only three deep:
# >>>
#     IMAP#config → Config.global → Config.default
#
# When creating a client with the +config+ keyword, for example to use
# the appropriate defaults for an application or a library while still
# relying on global for configuration of +debug+ or +logger+, most likely
# the ancestor chain is still only four deep:
# >>>
#     IMAP#config → alternate defaults → Config.global → Config.default
#
# source://net-imap//lib/net/imap/config/attr_inheritance.rb#25
module Net::IMAP::Config::AttrInheritance
  mixes_in_class_methods ::Net::IMAP::Config::AttrInheritance::Macros

  # :notnew:
  #
  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#48
  def initialize(parent = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#41
  def debug; end

  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#41
  def idle_response_timeout; end

  # Returns +true+ if +attr+ is inherited from #parent and not overridden
  # by this config.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#59
  def inherited?(attr); end

  # Creates a new config, which inherits from +self+.
  #
  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#55
  def new(**attrs); end

  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#41
  def open_timeout; end

  # The parent Config object
  #
  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#46
  def parent; end

  # :call-seq:
  #   reset -> self
  #   reset(attr) -> attribute value
  #
  # Resets an +attr+ to inherit from the #parent config.
  #
  # When +attr+ is nil or not given, all attributes are reset.
  #
  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#68
  def reset(attr = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#41
  def responses_without_block; end

  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#41
  def sasl_ir; end

  private

  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#82
  def initialize_copy(other); end

  class << self
    # source://net-imap//lib/net/imap/config/attr_inheritance.rb#39
    def attr_accessor(name); end

    private

    # @private
    #
    # source://net-imap//lib/net/imap/config/attr_inheritance.rb#34
    def included(mod); end
  end
end

# source://net-imap//lib/net/imap/config/attr_inheritance.rb#26
module Net::IMAP::Config::AttrInheritance::INHERITED; end

# source://net-imap//lib/net/imap/config/attr_inheritance.rb#29
module Net::IMAP::Config::AttrInheritance::Macros
  # source://net-imap//lib/net/imap/config/attr_inheritance.rb#30
  def attr_accessor(name); end
end

# >>>
#   *NOTE:* This module is an internal implementation detail, with no
#   guarantee of backward compatibility.
#
# Adds a +type+ keyword parameter to +attr_accessor+, to enforce that
# config attributes have valid types, for example: boolean, numeric,
# enumeration, non-nullable, etc.
#
# source://net-imap//lib/net/imap/config/attr_type_coercion.rb#13
module Net::IMAP::Config::AttrTypeCoercion
  mixes_in_class_methods ::Net::IMAP::Config::AttrTypeCoercion::Macros

  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#39
  def debug=(val); end

  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#40
  def debug?; end

  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#44
  def idle_response_timeout=(val); end

  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#44
  def open_timeout=(val); end

  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#50
  def responses_without_block=(val); end

  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#39
  def sasl_ir=(val); end

  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#40
  def sasl_ir?; end

  class << self
    # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#29
    def attr_accessor(attr, type: T.unsafe(nil)); end

    # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#38
    def boolean(attr); end

    # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#47
    def enum(attr, enum); end

    # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#43
    def integer(attr); end

    private

    # @private
    #
    # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#24
    def included(mod); end
  end
end

# :stopdoc: internal APIs only
#
# source://net-imap//lib/net/imap/config/attr_type_coercion.rb#16
module Net::IMAP::Config::AttrTypeCoercion::Macros
  # source://net-imap//lib/net/imap/config/attr_type_coercion.rb#17
  def attr_accessor(attr, type: T.unsafe(nil)); end
end

# Array of attribute names that are _not_ loaded by #load_defaults.
#
# source://net-imap//lib/net/imap/config.rb#122
Net::IMAP::Config::DEFAULT_TO_INHERIT = T.let(T.unsafe(nil), Array)

# Mailbox attribute indicating that this mailbox is used to hold draft
# messages -- typically, messages that are being composed but have not yet
# been sent. In some server implementations, this might be a virtual
# mailbox, containing messages from other mailboxes that are marked with the
# "\Draft" message flag. Alternatively, this might just be advice that a
# client put drafts here
#
# source://net-imap//lib/net/imap/flags.rb#232
Net::IMAP::DRAFTS = T.let(T.unsafe(nil), Symbol)

# This module handles deprecated arguments to various Net::IMAP methods.
#
# source://net-imap//lib/net/imap/deprecated_client_options.rb#7
module Net::IMAP::DeprecatedClientOptions
  # :call-seq:
  #   Net::IMAP.new(host, **options) # standard keyword options
  #   Net::IMAP.new(host, options)   # obsolete hash options
  #   Net::IMAP.new(host, port)      # obsolete port argument
  #   Net::IMAP.new(host, port, usessl, certs = nil, verify = true) # deprecated SSL arguments
  #
  # Translates Net::IMAP.new arguments for backward compatibility.
  #
  # ==== Obsolete arguments
  #
  # Use of obsolete arguments does not print a warning.  Obsolete arguments
  # will be deprecated by a future release.
  #
  # If a second positional argument is given and it is a hash (or is
  # convertible via +#to_hash+), it is converted to keyword arguments.
  #
  #     # Obsolete:
  #     Net::IMAP.new("imap.example.com", options_hash)
  #     # Use instead:
  #     Net::IMAP.new("imap.example.com", **options_hash)
  #
  # If a second positional argument is given and it is not a hash, it is
  # converted to the +port+ keyword argument.
  #     # Obsolete:
  #     Net::IMAP.new("imap.example.com", 114433)
  #     # Use instead:
  #     Net::IMAP.new("imap.example.com", port: 114433)
  #
  # ==== Deprecated arguments
  #
  # Using deprecated arguments prints a warning.  Convert to keyword
  # arguments to avoid the warning.  Deprecated arguments will be removed in
  # a future release.
  #
  # If +usessl+ is false, +certs+, and +verify+ are ignored.  When it true,
  # all three arguments are converted to the +ssl+ keyword argument.
  # Without +certs+ or +verify+, it is converted to <tt>ssl: true</tt>.
  #     # DEPRECATED:
  #     Net::IMAP.new("imap.example.com", nil, true) # => prints a warning
  #     # Use instead:
  #     Net::IMAP.new("imap.example.com", ssl: true)
  #
  # When +certs+ is a path to a directory, it is converted to <tt>ca_path:
  # certs</tt>.
  #     # DEPRECATED:
  #     Net::IMAP.new("imap.example.com", nil, true, "/path/to/certs") # => prints a warning
  #     # Use instead:
  #     Net::IMAP.new("imap.example.com", ssl: {ca_path: "/path/to/certs"})
  #
  # When +certs+ is a path to a file, it is converted to <tt>ca_file:
  # certs</tt>.
  #     # DEPRECATED:
  #     Net::IMAP.new("imap.example.com", nil, true, "/path/to/cert.pem") # => prints a warning
  #     # Use instead:
  #     Net::IMAP.new("imap.example.com", ssl: {ca_file: "/path/to/cert.pem"})
  #
  # When +verify+ is +false+, it is converted to <tt>verify_mode:
  # OpenSSL::SSL::VERIFY_NONE</tt>.
  #     # DEPRECATED:
  #     Net::IMAP.new("imap.example.com", nil, true, nil, false) # => prints a warning
  #     # Use instead:
  #     Net::IMAP.new("imap.example.com", ssl: {verify_mode: OpenSSL::SSL::VERIFY_NONE})
  #
  # source://net-imap//lib/net/imap/deprecated_client_options.rb#72
  def initialize(host, port_or_options = T.unsafe(nil), *deprecated, **options); end

  # :call-seq:
  #   starttls(**options) # standard
  #   starttls(options = {}) # obsolete
  #   starttls(certs = nil, verify = true) # deprecated
  #
  # Translates Net::IMAP#starttls arguments for backward compatibility.
  #
  # Support for +certs+ and +verify+ will be dropped in a future release.
  #
  # See ::new for interpretation of +certs+ and +verify+.
  #
  # source://net-imap//lib/net/imap/deprecated_client_options.rb#104
  def starttls(*deprecated, **options); end

  private

  # source://net-imap//lib/net/imap/deprecated_client_options.rb#123
  def create_ssl_params(certs = T.unsafe(nil), verify = T.unsafe(nil)); end
end

# Aliases for supported capabilities, to be used with the #enable command.
#
# source://net-imap//lib/net/imap.rb#723
Net::IMAP::ENABLE_ALIASES = T.let(T.unsafe(nil), Hash)

# **Note:** This represents an intentionally _unstable_ API.  Where
# instances of this class are returned, future releases may return a
# different (incompatible) object <em>without deprecation or warning</em>.
#
# Net::IMAP::ExtensionData represents data that is parsable according to the
# forward-compatible extension syntax in RFC3501, RFC4466, or RFC9051, but
# isn't directly known or understood by Net::IMAP yet.
#
# See also: UnparsedData, UnparsedNumericResponseData, IgnoredResponse
#
# source://net-imap//lib/net/imap/response_data.rb#120
class Net::IMAP::ExtensionData < ::Struct; end

# Net::IMAP::FetchData represents the contents of a FETCH response.
# Net::IMAP#fetch and Net::IMAP#uid_fetch both return an array of
# FetchData objects.
#
# === Fetch attributes
#
# See {[IMAP4rev1 §7.4.2]}[https://www.rfc-editor.org/rfc/rfc3501.html#section-7.4.2]
# and {[IMAP4rev2 §7.5.2]}[https://www.rfc-editor.org/rfc/rfc9051.html#section-7.5.2]
# for a full description of the standard fetch response data items, and
# Net::IMAP@Message+envelope+and+body+structure for other relevant RFCs.
#
# ==== Static fetch data items
#
# Most message attributes are static, and must never change for a given
# <tt>(server, account, mailbox, UIDVALIDITY, UID)</tt> tuple.
#
# The static fetch data items defined by both
# IMAP4rev1[https://www.rfc-editor.org/rfc/rfc3501.html] and
# IMAP4rev2[https://www.rfc-editor.org/rfc/rfc9051.html] are:
#
# * <b><tt>"UID"</tt></b> --- See #uid.
# * <b><tt>"BODY"</tt></b> --- See #body.
# * <b><tt>"BODY[#{section_spec}]"</tt></b>,
#   <b><tt>"BODY[#{section_spec}]<#{offset}>"</tt></b> --- See #message,
#   #part, #header, #header_fields, #header_fields_not, #mime, and #text.
# * <b><tt>"BODYSTRUCTURE"</tt></b> --- See #bodystructure.
# * <b><tt>"ENVELOPE"</tt></b> --- See #envelope.
# * <b><tt>"INTERNALDATE"</tt></b> --- See #internaldate.
# * <b><tt>"RFC822.SIZE"</tt></b> --- See #rfc822_size.
#
# IMAP4rev2[https://www.rfc-editor.org/rfc/rfc9051.html] adds the
# additional fetch items from the +BINARY+ extension
# {[RFC3516]}[https://www.rfc-editor.org/rfc/rfc3516.html]:
#
# * <b><tt>"BINARY[#{part}]"</tt></b>,
#   <b><tt>"BINARY[#{part}]<#{offset}>"</tt></b> -- See #binary.
# * <b><tt>"BINARY.SIZE[#{part}]"</tt></b> -- See #binary_size.
#
# Several static message attributes in
# IMAP4rev1[https://www.rfc-editor.org/rfc/rfc3501.html] are obsolete and
# been removed from
# IMAP4rev2[https://www.rfc-editor.org/rfc/rfc9051.html]:
#
# * <b><tt>"RFC822"</tt></b> --- See #rfc822 or replace with
#   <tt>"BODY[]"</tt> and #message.
# * <b><tt>"RFC822.HEADER"</tt></b> --- See #rfc822_header or replace with
#   <tt>"BODY[HEADER]"</tt> and #header.
# * <b><tt>"RFC822.TEXT"</tt></b> --- See #rfc822_text or replace with
#   <tt>"BODY[TEXT]"</tt> and #text.
#
# Net::IMAP supports static attributes defined by the following extensions:
# * +OBJECTID+ {[RFC8474]}[https://www.rfc-editor.org/rfc/rfc8474.html]
#   * <b><tt>"EMAILID"</tt></b> --- See #emailid.
#   * <b><tt>"THREADID"</tt></b> --- See #threadid.
#
# * +X-GM-EXT-1+ {[non-standard Gmail
#   extension]}[https://developers.google.com/gmail/imap/imap-extensions]
#   * <b><tt>"X-GM-MSGID"</tt></b> --- unique message ID.  Access via #attr.
#   * <b><tt>"X-GM-THRID"</tt></b> --- Thread ID.  Access via #attr.
#
# [Note:]
#   >>>
#     Additional static fields are defined in other \IMAP extensions, but
#     Net::IMAP can't parse them yet.
#
# ==== Dynamic message attributes
#
# Some message attributes can be dynamically changed, for example using the
# {STORE command}[rdoc-ref:Net::IMAP#store].
#
# The only dynamic message attribute defined by
# IMAP4rev1[https://www.rfc-editor.org/rfc/rfc3501.html] and
# IMAP4rev2[https://www.rfc-editor.org/rfc/rfc9051.html] is:
#
# * <b><tt>"FLAGS"</tt></b> --- See #flags.
#
# Net::IMAP supports dynamic attributes defined by the following extensions:
#
# * +CONDSTORE+ {[RFC7162]}[https://www.rfc-editor.org/rfc/rfc7162.html]:
#   * <b><tt>"MODSEQ"</tt></b> --- See #modseq.
# * +X-GM-EXT-1+ {[non-standard Gmail
#   extension]}[https://developers.google.com/gmail/imap/imap-extensions]
#   * <b><tt>"X-GM-LABELS"</tt></b> --- Gmail labels.  Access via #attr.
#
# [Note:]
#   >>>
#     Additional dynamic fields are defined in other \IMAP extensions, but
#     Net::IMAP can't parse them yet.
#
# === Implicitly setting <tt>\Seen</tt> and using +PEEK+
#
# Unless the mailbox is has been opened as read-only, fetching
# <tt>BODY[#{section}]</tt> or <tt>BINARY[#{section}]</tt>
# will implicitly set the <tt>\Seen</tt> flag.  To avoid this, fetch using
# <tt>BODY.PEEK[#{section}]</tt> or <tt>BINARY.PEEK[#{section}]</tt>
# instead.
#
# Note that the data will always be _returned_ without <tt>".PEEK"</tt>, in
# <tt>BODY[#{specifier}]</tt> or <tt>BINARY[#{section}]</tt>.
#
# source://net-imap//lib/net/imap/fetch_data.rb#106
class Net::IMAP::FetchData < ::Struct
  # :call-seq: attr_upcase -> hash
  #
  # A transformation of #attr, with all the keys converted to upper case.
  #
  # Header field names are case-preserved but not case-sensitive, so this is
  # used by #header_fields and #header_fields_not.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#136
  def attr_upcase; end

  # :call-seq:
  #   binary(*part_nums, offset: nil) -> string or nil
  #
  # Returns the binary representation of a particular MIME part, which has
  # already been decoded according to its Content-Transfer-Encoding.
  #
  # See #part for a description of +part_nums+ and +offset+.
  #
  # This is the same as getting the value of
  # <tt>"BINARY[#{part_nums.join(".")}]"</tt> or
  # <tt>"BINARY[#{part_nums.join(".")}]<#{offset}>"</tt> from #attr.
  #
  # The server must support either
  # IMAP4rev2[https://www.rfc-editor.org/rfc/rfc9051.html]
  # or the +BINARY+ extension
  # {[RFC3516]}[https://www.rfc-editor.org/rfc/rfc3516.html].
  #
  # See also: #binary_size, #mime
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#430
  def binary(*part_nums, offset: T.unsafe(nil)); end

  # :call-seq:
  #   binary_size(*part_nums) -> integer or nil
  #
  # Returns the decoded size of a particular MIME part (the size to expect
  # in response to a <tt>BINARY</tt> fetch request).
  #
  # See #part for a description of +part_nums+.
  #
  # This is the same as getting the value of
  # <tt>"BINARY.SIZE[#{part_nums.join(".")}]"</tt> from #attr.
  #
  # The server must support either
  # IMAP4rev2[https://www.rfc-editor.org/rfc/rfc9051.html]
  # or the +BINARY+ extension
  # {[RFC3516]}[https://www.rfc-editor.org/rfc/rfc3516.html].
  #
  # See also: #binary, #mime
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#451
  def binary_size(*part_nums); end

  # :call-seq:
  #   body -> body structure or nil
  #
  # Returns an alternate form of #bodystructure, without any extension data.
  #
  # This is the same as getting the value for <tt>"BODY"</tt> from #attr.
  #
  # [Note]
  #   Use #message, #part, #header, #header_fields, #header_fields_not,
  #   #text, or #mime to retrieve <tt>BODY[#{section_spec}]</tt> attributes.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#148
  def body; end

  # :call-seq:
  #   bodystructure -> BodyStructure struct or nil
  #
  # A BodyStructure object that describes the message, if it was fetched.
  #
  # This is the same as getting the value for <tt>"BODYSTRUCTURE"</tt> from
  # #attr.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#310
  def body_structure; end

  # :call-seq:
  #   bodystructure -> BodyStructure struct or nil
  #
  # A BodyStructure object that describes the message, if it was fetched.
  #
  # This is the same as getting the value for <tt>"BODYSTRUCTURE"</tt> from
  # #attr.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#310
  def bodystructure; end

  # :call-seq: emailid -> string or nil
  #
  # An ObjectID that uniquely identifies the immutable content of a single
  # message.
  #
  # The server must return the same +EMAILID+ for both the source and
  # destination messages after a COPY or MOVE command.  However, it is
  # possible for different messages with the same EMAILID to have different
  # mutable attributes, such as flags.
  #
  # This is the same as getting the value for <tt>"EMAILID"</tt> from
  # #attr.
  #
  # The server must support the +OBJECTID+ extension
  # {[RFC8474]}[https://www.rfc-editor.org/rfc/rfc8474.html].
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#484
  def emailid; end

  # :call-seq: envelope -> Envelope or nil
  #
  # An Envelope object that describes the envelope structure of a message.
  # See the documentation for Envelope for a description of the envelope
  # structure attributes.
  #
  # This is the same as getting the value for <tt>"ENVELOPE"</tt> from
  # #attr.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#321
  def envelope; end

  # :call-seq: flags -> array of Symbols and Strings
  #
  # A array of flags that are set for this message.  System flags are
  # symbols that have been capitalized by String#capitalize.  Keyword flags
  # are strings and their case is not changed.
  #
  # This is the same as getting the value for <tt>"FLAGS"</tt> from #attr.
  #
  # [Note]
  #   The +FLAGS+ field is dynamic, and can change for a uniquely identified
  #   message.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#334
  def flags; end

  # :call-seq:
  #   header(*part_nums,                offset: nil) -> string or nil
  #   header(*part_nums, fields: names, offset: nil) -> string or nil
  #   header(*part_nums, except: names, offset: nil) -> string or nil
  #
  # The {[RFC5322]}[https://www.rfc-editor.org/rfc/rfc5322.html] header of a
  # message or of an encapsulated
  # {[MIME-IMT]}[https://www.rfc-editor.org/rfc/rfc2046.html]
  # MESSAGE/RFC822 or MESSAGE/GLOBAL message.
  #
  # <em>Headers can be parsed using the "mail" gem.</em>
  #
  # See #part for a description of +part_nums+ and +offset+.
  #
  # ==== Without +fields+ or +except+
  # This is the same as getting the value from #attr for one of:
  # * <tt>BODY[HEADER]</tt>
  # * <tt>BODY[HEADER]<#{offset}></tt>
  # * <tt>BODY[#{part_nums.join "."}.HEADER]"</tt>
  # * <tt>BODY[#{part_nums.join "."}.HEADER]<#{offset}>"</tt>
  #
  # ==== With +fields+
  # When +fields+ is sent, returns a subset of the header which contains
  # only the header fields that match one of the names in the list.
  #
  # This is the same as getting the value from #attr_upcase for one of:
  # * <tt>BODY[HEADER.FIELDS (#{names.join " "})]</tt>
  # * <tt>BODY[HEADER.FIELDS (#{names.join " "})]<#{offset}></tt>
  # * <tt>BODY[#{part_nums.join "."}.HEADER.FIELDS (#{names.join " "})]</tt>
  # * <tt>BODY[#{part_nums.join "."}.HEADER.FIELDS (#{names.join " "})]<#{offset}></tt>
  #
  # See also: #header_fields
  #
  # ==== With +except+
  # When +except+ is sent, returns a subset of the header which contains
  # only the header fields that do _not_ match one of the names in the list.
  #
  # This is the same as getting the value from #attr_upcase for one of:
  # * <tt>BODY[HEADER.FIELDS.NOT (#{names.join " "})]</tt>
  # * <tt>BODY[HEADER.FIELDS.NOT (#{names.join " "})]<#{offset}></tt>
  # * <tt>BODY[#{part_nums.join "."}.HEADER.FIELDS.NOT (#{names.join " "})]</tt>
  # * <tt>BODY[#{part_nums.join "."}.HEADER.FIELDS.NOT (#{names.join " "})]<#{offset}></tt>
  #
  # See also: #header_fields_not
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#234
  def header(*part_nums, fields: T.unsafe(nil), except: T.unsafe(nil), offset: T.unsafe(nil)); end

  # :call-seq:
  #   header_fields(*names, part: [], offset: nil) -> string or nil
  #
  # The result from #header when called with <tt>fields: names</tt>.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#252
  def header_fields(first, *rest, part: T.unsafe(nil), offset: T.unsafe(nil)); end

  # :call-seq:
  #   header_fields_not(*names, part: [], offset: nil) -> string or nil
  #
  # The result from #header when called with <tt>except: names</tt>.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#260
  def header_fields_not(first, *rest, part: T.unsafe(nil), offset: T.unsafe(nil)); end

  # :call-seq: internaldate -> Time or nil
  #
  # The internal date and time of the message on the server.  This is not
  # the date and time in the [RFC5322[https://tools.ietf.org/html/rfc5322]]
  # header, but rather a date and time which reflects when the message was
  # received.
  #
  # This is similar to getting the value for <tt>"INTERNALDATE"</tt> from
  # #attr.
  #
  # [Note]
  #   <tt>attr["INTERNALDATE"]</tt> returns a string, and this method
  #   returns a Time object.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#349
  def internal_date; end

  # :call-seq: internaldate -> Time or nil
  #
  # The internal date and time of the message on the server.  This is not
  # the date and time in the [RFC5322[https://tools.ietf.org/html/rfc5322]]
  # header, but rather a date and time which reflects when the message was
  # received.
  #
  # This is similar to getting the value for <tt>"INTERNALDATE"</tt> from
  # #attr.
  #
  # [Note]
  #   <tt>attr["INTERNALDATE"]</tt> returns a string, and this method
  #   returns a Time object.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#349
  def internaldate; end

  # :call-seq:
  #   message(offset: bytes) -> string or nil
  #
  # The RFC5322[https://www.rfc-editor.org/rfc/rfc5322.html]
  # expression of the entire message, as a string.
  #
  # See #part for a description of +offset+.
  #
  # <em>RFC5322 messages can be parsed using the "mail" gem.</em>
  #
  # This is the same as getting the value for <tt>"BODY[]"</tt> or
  # <tt>"BODY[]<#{offset}>"</tt> from #attr.
  #
  # See also: #header, #text, and #mime.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#164
  def message(offset: T.unsafe(nil)); end

  # :call-seq:
  #   mime(*part_nums)                -> string or nil
  #   mime(*part_nums, offset: bytes) -> string or nil
  #
  # The {[MIME-IMB]}[https://www.rfc-editor.org/rfc/rfc2045.html] header for
  # a message part, if it was fetched.
  #
  # See #part for a description of +part_nums+ and +offset+.
  #
  # This is the same as getting the value for
  # <tt>"BODY[#{part_nums}.MIME]"</tt> or
  # <tt>"BODY[#{part_nums}.MIME]<#{offset}>"</tt> from #attr.
  #
  # See also: #message, #header, and #text.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#278
  def mime(part, *subparts, offset: T.unsafe(nil)); end

  # :call-seq: modseq -> Integer
  #
  # The modification sequence number associated with this IMAP message.
  #
  # This is the same as getting the value for <tt>"MODSEQ"</tt> from #attr.
  #
  # The server must support the +CONDSTORE+ extension
  # {[RFC7162]}[https://www.rfc-editor.org/rfc/rfc7162.html].
  #
  # [Note]
  #   The +MODSEQ+ field is dynamic, and can change for a uniquely
  #   identified message.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#467
  def modseq; end

  # :call-seq:
  #   part(*part_nums, offset: bytes) -> string or nil
  #
  # The string representation of a particular MIME part.
  #
  # +part_nums+ forms a path of MIME part numbers, counting up from +1+,
  # which may specify an arbitrarily nested part, similarly to Array#dig.
  # Messages that don't use MIME, or MIME messages that are not multipart
  # and don't hold an encapsulated message, only have part +1+.
  #
  # If a zero-based +offset+ is given, the returned string is a substring of
  # the entire contents, starting at that origin octet.  This means that
  # <tt>BODY[]<0></tt> MAY be truncated, but <tt>BODY[]</tt> is never
  # truncated.
  #
  # This is the same as getting the value of
  # <tt>"BODY[#{part_nums.join(".")}]"</tt> or
  # <tt>"BODY[#{part_nums.join(".")}]<#{offset}>"</tt> from #attr.
  #
  # See also: #message, #header, #text, and #mime.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#186
  def part(index, *subparts, offset: T.unsafe(nil)); end

  # :call-seq: rfc822 -> String
  #
  # Semantically equivalent to #message with no arguments.
  #
  # This is the same as getting the value for <tt>"RFC822"</tt> from #attr.
  #
  # [Note]
  #   +IMAP4rev2+ deprecates <tt>RFC822</tt>.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#362
  def rfc822; end

  # :call-seq: rfc822_header -> String
  #
  # Semantically equivalent to #header, with no arguments.
  #
  # This is the same as getting the value for <tt>"RFC822.HEADER"</tt> from #attr.
  #
  # [Note]
  #   +IMAP4rev2+ deprecates <tt>RFC822.HEADER</tt>.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#392
  def rfc822_header; end

  # :call-seq: rfc822_size -> Integer
  #
  # A number expressing the [RFC5322[https://tools.ietf.org/html/rfc5322]]
  # size of the message.
  #
  # This is the same as getting the value for <tt>"RFC822.SIZE"</tt> from
  # #attr.
  #
  # [Note]
  #   \IMAP was originally developed for the older
  #   RFC822[https://www.rfc-editor.org/rfc/rfc822.html] standard, and as a
  #   consequence several fetch items in \IMAP incorporate "RFC822" in their
  #   name.  With the exception of +RFC822.SIZE+, there are more modern
  #   replacements; for example, the modern version of +RFC822.HEADER+ is
  #   <tt>BODY.PEEK[HEADER]</tt>.  In all cases, "RFC822" should be
  #   interpreted as a reference to the updated
  #   RFC5322[https://www.rfc-editor.org/rfc/rfc5322.html] standard.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#381
  def rfc822_size; end

  # :call-seq: rfc822_text -> String
  #
  # Semantically equivalent to #text, with no arguments.
  #
  # This is the same as getting the value for <tt>"RFC822.TEXT"</tt> from
  # #attr.
  #
  # [Note]
  #   +IMAP4rev2+ deprecates <tt>RFC822.TEXT</tt>.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#403
  def rfc822_text; end

  # :call-seq: rfc822_size -> Integer
  #
  # A number expressing the [RFC5322[https://tools.ietf.org/html/rfc5322]]
  # size of the message.
  #
  # This is the same as getting the value for <tt>"RFC822.SIZE"</tt> from
  # #attr.
  #
  # [Note]
  #   \IMAP was originally developed for the older
  #   RFC822[https://www.rfc-editor.org/rfc/rfc822.html] standard, and as a
  #   consequence several fetch items in \IMAP incorporate "RFC822" in their
  #   name.  With the exception of +RFC822.SIZE+, there are more modern
  #   replacements; for example, the modern version of +RFC822.HEADER+ is
  #   <tt>BODY.PEEK[HEADER]</tt>.  In all cases, "RFC822" should be
  #   interpreted as a reference to the updated
  #   RFC5322[https://www.rfc-editor.org/rfc/rfc5322.html] standard.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#381
  def size; end

  # :call-seq:
  #   text(*part_nums)                -> string or nil
  #   text(*part_nums, offset: bytes) -> string or nil
  #
  # The text body of a message or a message part, if it was fetched,
  # omitting the {[RFC5322]}[https://www.rfc-editor.org/rfc/rfc5322.html]
  # header.
  #
  # See #part for a description of +part_nums+ and +offset+.
  #
  # This is the same as getting the value from #attr for one of:
  # * <tt>"BODY[TEXT]"</tt>,
  # * <tt>"BODY[TEXT]<#{offset}>"</tt>,
  # * <tt>"BODY[#{section}.TEXT]"</tt>, or
  # * <tt>"BODY[#{section}.TEXT]<#{offset}>"</tt>.
  #
  # See also: #message, #header, and #mime.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#299
  def text(*part, offset: T.unsafe(nil)); end

  # :call-seq: threadid -> string or nil
  #
  # An ObjectID that uniquely identifies a set of messages that the server
  # believes should be grouped together.
  #
  # It is generally based on some combination of References, In-Reply-To,
  # and Subject, but the exact implementation is left up to the server
  # implementation.  The server should return the same thread identifier for
  # related messages, even if they are in different mailboxes.
  #
  # This is the same as getting the value for <tt>"THREADID"</tt> from
  # #attr.
  #
  # The server must support the +OBJECTID+ extension
  # {[RFC8474]}[https://www.rfc-editor.org/rfc/rfc8474.html].
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#501
  def threadid; end

  # :call-seq: uid -> Integer
  #
  # A number expressing the unique identifier of the message.
  #
  # This is the same as getting the value for <tt>"UID"</tt> from #attr.
  #
  # source://net-imap//lib/net/imap/fetch_data.rb#410
  def uid; end

  private

  # source://net-imap//lib/net/imap/fetch_data.rb#505
  def body_section_attr(*_arg0, **_arg1, &_arg2); end

  # source://net-imap//lib/net/imap/fetch_data.rb#507
  def section_attr(attr, part = T.unsafe(nil), text = T.unsafe(nil), offset: T.unsafe(nil)); end
end

# Alias for HAS_CHILDREN, to match the \IMAP spelling.
#
# source://net-imap//lib/net/imap/flags.rb#183
Net::IMAP::HASCHILDREN = T.let(T.unsafe(nil), Symbol)

# Alias for HAS_NO_CHILDREN, to match the \IMAP spelling.
#
# source://net-imap//lib/net/imap/flags.rb#185
Net::IMAP::HASNOCHILDREN = T.let(T.unsafe(nil), Symbol)

# The presence of this attribute indicates that the mailbox has child
# mailboxes. A server SHOULD NOT set this attribute if there are child
# mailboxes and the user does not have permission to access any of them.  In
# this case, +\HasNoChildren+ SHOULD be used. In many cases, however, a
# server may not be able to efficiently compute whether a user has access to
# any child mailboxes. Note that even though the +\HasChildren+ attribute
# for a mailbox must be correct at the time of processing the mailbox, a
# client must be prepared to deal with a situation when a mailbox is marked
# with the +\HasChildren+ attribute, but no child mailbox appears in the
# response to the #list command. This might happen, for example, due to child
# mailboxes being deleted or made inaccessible to the user (using access
# control) by another client before the server is able to list them.
#
# It is an error for the server to return both a +\HasChildren+ and a
# +\HasNoChildren+ attribute in the same #list response. A client that
# encounters a #list response with both +\HasChildren+ and +\HasNoChildren+
# attributes present should act as if both are absent in the #list response.
#
# source://net-imap//lib/net/imap/flags.rb#136
Net::IMAP::HAS_CHILDREN = T.let(T.unsafe(nil), Symbol)

# The presence of this attribute indicates that the mailbox has NO child
# mailboxes that are accessible to the currently authenticated user.
#
# It is an error for the server to return both a +\HasChildren+ and a
# +\HasNoChildren+ attribute in the same #list response. A client that
# encounters a #list response with both +\HasChildren+ and +\HasNoChildren+
# attributes present should act as if both are absent in the #list response.
#
# Note: the +\HasNoChildren+ attribute should not be confused with the
# +\NoInferiors+ attribute, which indicates that no child mailboxes exist
# now and none can be created in the future.
#
# source://net-imap//lib/net/imap/flags.rb#149
Net::IMAP::HAS_NO_CHILDREN = T.let(T.unsafe(nil), Symbol)

# Net::IMAP::IgnoredResponse represents intentionally ignored responses.
#
# This includes untagged response "NOOP" sent by e.g. Zimbra to avoid
# some clients to close the connection.
#
# It matches no IMAP standard.
#
# source://net-imap//lib/net/imap/response_data.rb#65
class Net::IMAP::IgnoredResponse < ::Net::IMAP::UntaggedResponse; end

# Error raised when the server sends an invalid response.
#
# This is different from UnknownResponseError: the response has been
# rejected.  Although it may be parsable, the server is forbidden from
# sending it in the current context.  The client should automatically
# disconnect, abruptly (without logout).
#
# Note that InvalidResponseError does not inherit from ResponseError: it
# can be raised before the response is fully parsed.  A related
# ResponseParseError or ResponseError may be the #cause.
#
# source://net-imap//lib/net/imap/errors.rb#60
class Net::IMAP::InvalidResponseError < ::Net::IMAP::Error; end

# Mailbox attribute indicating that this mailbox is where messages deemed to
# be junk mail are held. Some server implementations might put messages here
# automatically.  Alternatively, this might just be advice to a client-side
# spam filter.
#
# source://net-imap//lib/net/imap/flags.rb#242
Net::IMAP::JUNK = T.let(T.unsafe(nil), Symbol)

# source://net-imap//lib/net/imap/command_data.rb#167
class Net::IMAP::Literal
  # @return [Literal] a new instance of Literal
  #
  # source://net-imap//lib/net/imap/command_data.rb#177
  def initialize(data); end

  # source://net-imap//lib/net/imap/command_data.rb#168
  def send_data(imap, tag); end

  # source://net-imap//lib/net/imap/command_data.rb#172
  def validate; end
end

# source://net-imap//lib/net/imap/command_data.rb#182
class Net::IMAP::MessageSet
  # @return [MessageSet] a new instance of MessageSet
  #
  # source://net-imap//lib/net/imap/command_data.rb#193
  def initialize(data); end

  # source://net-imap//lib/net/imap/command_data.rb#183
  def send_data(imap, tag); end

  # source://net-imap//lib/net/imap/command_data.rb#187
  def validate; end

  private

  # source://net-imap//lib/net/imap/command_data.rb#197
  def format_internal(data); end

  # source://net-imap//lib/net/imap/command_data.rb#218
  def validate_internal(data); end
end

# The +\NonExistent+ attribute indicates that a mailbox name does not refer
# to an existing mailbox. Note that this attribute is not meaningful by
# itself, as mailbox names that match the canonical #list pattern but don't
# exist must not be returned unless one of the two conditions listed below
# is also satisfied:
#
# 1. The mailbox name also satisfies the selection criteria (for example,
#    it is subscribed and the "SUBSCRIBED" selection option has been
#    specified).
#
# 2. "RECURSIVEMATCH" has been specified, and the mailbox name has at least
#    one descendant mailbox name that does not match the #list pattern and
#    does match the selection criteria.
#
# In practice, this means that the +\NonExistent+ attribute is usually
# returned with one or more of +\Subscribed+, +\Remote+, +\HasChildren+, or
# the CHILDINFO extended data item.
#
# The client must treat the presence of the +\NonExistent+ attribute as if the
# +\NoSelect+ attribute was also sent by the server
#
# source://net-imap//lib/net/imap/flags.rb#105
Net::IMAP::NONEXISTENT = T.let(T.unsafe(nil), Symbol)

# Mailbox attribute indicating it is not possible for any child levels of
# hierarchy to exist under this name; no child levels exist now and none can
# be created in the future children.
#
# The client must treat the presence of the +\NoInferiors+ attribute as if the
# +\HasNoChildren+ attribute was also sent by the server
#
# source://net-imap//lib/net/imap/flags.rb#113
Net::IMAP::NO_INFERIORS = T.let(T.unsafe(nil), Symbol)

# Mailbox attribute indicating it is not possible to use this name as a
# selectable mailbox.
#
# source://net-imap//lib/net/imap/flags.rb#117
Net::IMAP::NO_SELECT = T.let(T.unsafe(nil), Symbol)

# Net::IMAP::Namespace represents a single namespace contained inside a
# NAMESPACE response.
#
# Returned by Net::IMAP#namespace, contained inside a Namespaces object.
#
# source://net-imap//lib/net/imap/response_data.rb#496
class Net::IMAP::Namespace < ::Struct; end

# Net::IMAP::Namespaces represents a +NAMESPACE+ server response, which
# contains lists of #personal, #shared, and #other namespaces.
#
# Net::IMAP#namespace returns a Namespaces object.
#
# source://net-imap//lib/net/imap/response_data.rb#522
class Net::IMAP::Namespaces < ::Struct; end

# Common validators of number and nz_number types
#
# source://net-imap//lib/net/imap/data_encoding.rb#157
module Net::IMAP::NumValidator
  private

  # Ensure argument is 'mod_sequence_value' or raise DataFormatError
  #
  # source://net-imap//lib/net/imap/data_encoding.rb#204
  def ensure_mod_sequence_value(num); end

  # Ensure argument is 'number' or raise DataFormatError
  #
  # source://net-imap//lib/net/imap/data_encoding.rb#188
  def ensure_number(num); end

  # Ensure argument is 'nz_number' or raise DataFormatError
  #
  # source://net-imap//lib/net/imap/data_encoding.rb#196
  def ensure_nz_number(num); end

  # Check is passed argument valid 'mod_sequence_value' in RFC 4551 terminology
  #
  # source://net-imap//lib/net/imap/data_encoding.rb#179
  def valid_mod_sequence_value?(num); end

  # Check is passed argument valid 'number' in RFC 3501 terminology
  #
  # source://net-imap//lib/net/imap/data_encoding.rb#161
  def valid_number?(num); end

  # Check is passed argument valid 'nz_number' in RFC 3501 terminology
  #
  # source://net-imap//lib/net/imap/data_encoding.rb#170
  def valid_nz_number?(num); end

  class << self
    # Ensure argument is 'mod_sequence_value' or raise DataFormatError
    #
    # @raise [DataFormatError]
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#204
    def ensure_mod_sequence_value(num); end

    # Ensure argument is 'number' or raise DataFormatError
    #
    # @raise [DataFormatError]
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#188
    def ensure_number(num); end

    # Ensure argument is 'nz_number' or raise DataFormatError
    #
    # @raise [DataFormatError]
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#196
    def ensure_nz_number(num); end

    # Check is passed argument valid 'mod_sequence_value' in RFC 4551 terminology
    #
    # @return [Boolean]
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#179
    def valid_mod_sequence_value?(num); end

    # Check is passed argument valid 'number' in RFC 3501 terminology
    #
    # @return [Boolean]
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#161
    def valid_number?(num); end

    # Check is passed argument valid 'nz_number' in RFC 3501 terminology
    #
    # @return [Boolean]
    #
    # source://net-imap//lib/net/imap/data_encoding.rb#170
    def valid_nz_number?(num); end
  end
end

# source://net-imap//lib/net/imap/command_data.rb#152
class Net::IMAP::QuotedString
  # @return [QuotedString] a new instance of QuotedString
  #
  # source://net-imap//lib/net/imap/command_data.rb#162
  def initialize(data); end

  # source://net-imap//lib/net/imap/command_data.rb#153
  def send_data(imap, tag); end

  # source://net-imap//lib/net/imap/command_data.rb#157
  def validate; end
end

# The mailbox is a remote mailbox.
#
# source://net-imap//lib/net/imap/flags.rb#176
Net::IMAP::REMOTE = T.let(T.unsafe(nil), Symbol)

# source://net-imap//lib/net/imap/errors.rb#74
Net::IMAP::RESPONSE_ERRORS = T.let(T.unsafe(nil), Hash)

# source://net-imap//lib/net/imap/command_data.rb#122
class Net::IMAP::RawData
  # @return [RawData] a new instance of RawData
  #
  # source://net-imap//lib/net/imap/command_data.rb#132
  def initialize(data); end

  # source://net-imap//lib/net/imap/command_data.rb#123
  def send_data(imap, tag); end

  # source://net-imap//lib/net/imap/command_data.rb#127
  def validate; end
end

# Superclass of all errors used to encapsulate "fail" responses
# from the server.
#
# source://net-imap//lib/net/imap/errors.rb#20
class Net::IMAP::ResponseError < ::Net::IMAP::Error
  # @return [ResponseError] a new instance of ResponseError
  #
  # source://net-imap//lib/net/imap/errors.rb#25
  def initialize(response); end

  # The response that caused this error
  #
  # source://net-imap//lib/net/imap/errors.rb#23
  def response; end

  # The response that caused this error
  #
  # source://net-imap//lib/net/imap/errors.rb#23
  def response=(_arg0); end
end

# Parses an \IMAP server response.
#
# source://net-imap//lib/net/imap/response_parser/parser_utils.rb#5
class Net::IMAP::ResponseParser
  include ::Net::IMAP::ResponseParser::ParserUtils
  include ::Net::IMAP::ResponseParser::ResponseConditions
  extend ::Net::IMAP::ResponseParser::ParserUtils::Generator

  # :call-seq: Net::IMAP::ResponseParser.new -> Net::IMAP::ResponseParser
  #
  # @return [ResponseParser] a new instance of ResponseParser
  #
  # source://net-imap//lib/net/imap/response_parser.rb#17
  def initialize(config: T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def CRLF!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def CRLF?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def EOF!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def EOF?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def NIL!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def NIL?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#46
  def PLUS!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def PLUS?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#46
  def SP!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def SP?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#46
  def STAR!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def STAR?; end

  # RFC-3501 & RFC-9051:
  #   body-fld-enc    = (DQUOTE ("7BIT" / "8BIT" / "BINARY" / "BASE64"/
  #                     "QUOTED-PRINTABLE") DQUOTE) / string
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def body_fld_enc; end

  # valid number ranges are not enforced by parser
  #   number64        = 1*DIGIT
  #                       ; Unsigned 63-bit integer
  #                       ; (0 <= n <= 9,223,372,036,854,775,807)
  # number in 3501, number64 in 9051
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def body_fld_lines; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def body_fld_octets; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def case_insensitive__string; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def case_insensitive__string?; end

  # Returns the value of attribute config.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#14
  def config; end

  # date-time       = DQUOTE date-day-fixed "-" date-month "-" date-year
  #                     SP time SP zone DQUOTE
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def date_time; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#46
  def lbra; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def lbra?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_CRLF!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_EOF!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_NIL!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_PLUS?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_SP?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_STAR?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_body?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_case_insensitive__string!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_lbra?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_lpar?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_number!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_quoted!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_rbra?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_rpar?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_string!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_string8!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#86
  def lookahead_tagged_ext_label!; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_thread_list?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#25
  def lookahead_thread_nested?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#46
  def lpar; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def lpar?; end

  # text/*
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def media_subtype; end

  # valid number ranges are not enforced by parser
  #   nz-number       = digit-nz *DIGIT
  #                       ; Non-zero unsigned 32-bit integer
  #                       ; (0 < n < 4,294,967,296)
  # valid number ranges are not enforced by parser
  #   nz-number64     = digit-nz *DIGIT
  #                       ; Unsigned 63-bit integer
  #                       ; (0 < n <= 9,223,372,036,854,775,807)
  # RFC7162:
  # mod-sequence-value  = 1*DIGIT
  #                        ;; Positive unsigned 63-bit integer
  #                        ;; (mod-sequence)
  #                        ;; (1 <= n <= 9,223,372,036,854,775,807).
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def mod_sequence_value; end

  # valid number ranges are not enforced by parser
  #   number64        = 1*DIGIT
  #                       ; Unsigned 63-bit integer
  #                       ; (0 <= n <= 9,223,372,036,854,775,807)
  # RFC7162:
  # mod-sequence-valzer = "0" / mod-sequence-value
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def mod_sequence_valzer; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def number; end

  # valid number ranges are not enforced by parser
  #   number64        = 1*DIGIT
  #                       ; Unsigned 63-bit integer
  #                       ; (0 <= n <= 9,223,372,036,854,775,807)
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def number64; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def number64?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def number?; end

  # valid number ranges are not enforced by parser
  #   nz-number       = digit-nz *DIGIT
  #                       ; Non-zero unsigned 32-bit integer
  #                       ; (0 < n < 4,294,967,296)
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def nz_number; end

  # valid number ranges are not enforced by parser
  #   nz-number       = digit-nz *DIGIT
  #                       ; Non-zero unsigned 32-bit integer
  #                       ; (0 < n < 4,294,967,296)
  # valid number ranges are not enforced by parser
  #   nz-number64     = digit-nz *DIGIT
  #                       ; Unsigned 63-bit integer
  #                       ; (0 < n <= 9,223,372,036,854,775,807)
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def nz_number64; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def nz_number?; end

  # :call-seq:
  #   parse(str) -> ContinuationRequest
  #   parse(str) -> UntaggedResponse
  #   parse(str) -> TaggedResponse
  #
  # Raises ResponseParseError for unparsable strings.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#31
  def parse(str); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#30
  def peek_PLUS?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#30
  def peek_SP?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#30
  def peek_STAR?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#30
  def peek_lbra?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#30
  def peek_lpar?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#30
  def peek_rbra?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#30
  def peek_rpar?; end

  # valid number ranges are not enforced by parser
  #   nz-number       = digit-nz *DIGIT
  #                       ; Non-zero unsigned 32-bit integer
  #                       ; (0 < n < 4,294,967,296)
  # valid number ranges are not enforced by parser
  #   nz-number64     = digit-nz *DIGIT
  #                       ; Unsigned 63-bit integer
  #                       ; (0 < n <= 9,223,372,036,854,775,807)
  # RFC7162:
  # mod-sequence-value  = 1*DIGIT
  #                        ;; Positive unsigned 63-bit integer
  #                        ;; (mod-sequence)
  #                        ;; (1 <= n <= 9,223,372,036,854,775,807).
  # RFC7162:
  # permsg-modsequence  = mod-sequence-value
  #                        ;; Per-message mod-sequence.
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def permsg_modsequence; end

  # Used when servers erroneously send an extra SP.
  #
  # As of 2023-11-28, Outlook.com (still) sends SP
  #   between +address+ in <tt>env-*</tt> lists.
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def quirky_SP?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def quoted; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def quoted?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#46
  def rbra; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def rbra?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#46
  def rpar; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#35
  def rpar?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def string; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def string8; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def string8?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def string?; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def tagged_ext_label; end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#96
  def tagged_ext_label?; end

  # valid number ranges are not enforced by parser
  #   nz-number       = digit-nz *DIGIT
  #                       ; Non-zero unsigned 32-bit integer
  #                       ; (0 < n < 4,294,967,296)
  # valid number ranges are not enforced by parser
  #      uniqueid        = nz-number
  #                          ; Strictly ascending
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def uniqueid; end

  # valid number ranges are not enforced by parser
  #
  # a 64-bit unsigned integer and is the decimal equivalent for the ID hex
  # string used in the web interface and the Gmail API.
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#104
  def x_gm_id; end

  private

  # The RFC is very strict about this and usually we should be too.
  # But skipping spaces is usually a safe workaround for buggy servers.
  #
  # This advances @pos directly so it's safe before changing @lex_state.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#2028
  def accept_spaces; end

  # acl-data        = "ACL" SP mailbox *(SP identifier SP rights)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1429
  def acl_data; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def addr_adl; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def addr_host; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def addr_mailbox; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def addr_name; end

  # address         = "(" addr-name SP addr-adl SP addr-mailbox SP
  #                     addr-host ")"
  #   addr-adl        = nstring
  #   addr-host       = nstring
  #   addr-mailbox    = nstring
  #   addr-name       = nstring
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1897
  def address; end

  # astring         = 1*ASTRING-CHAR / string
  #
  # source://net-imap//lib/net/imap/response_parser.rb#506
  def astring; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#510
  def astring?; end

  # source://net-imap//lib/net/imap/response_parser.rb#491
  def astring_chars; end

  # TODO: handle atom, astring_chars, and tag entirely inside the lexer
  #
  # source://net-imap//lib/net/imap/response_parser.rb#490
  def atom; end

  # the #accept version of #atom
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#495
  def atom?; end

  # RFC-3501 & RFC-9051:
  #   body            = "(" (body-type-1part / body-type-mpart) ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1012
  def body; end

  # RFC2060
  #   body_ext_1part  ::= body_fld_md5 [SPACE body_fld_dsp
  #                       [SPACE body_fld_lang
  #                       [SPACE 1#body_extension]]]
  #                       ;; MUST NOT be returned on non-extensible
  #                       ;; "BODY" fetch
  # RFC3501 & RFC9051
  #   body-ext-1part  = body-fld-md5 [SP body-fld-dsp [SP body-fld-lang
  #                     [SP body-fld-loc *(SP body-extension)]]]
  #                       ; MUST NOT be returned on non-extensible
  #                       ; "BODY" fetch
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1184
  def body_ext_1part; end

  # RFC-2060:
  #   body_ext_mpart  = body_fld_param [SP body_fld_dsp SP body_fld_lang
  #                     [SP 1#body_extension]]
  #                       ;; MUST NOT be returned on non-extensible
  #                       ;; "BODY" fetch
  # RFC-3501 & RFC-9051:
  #   body-ext-mpart  = body-fld-param [SP body-fld-dsp [SP body-fld-lang
  #                     [SP body-fld-loc *(SP body-extension)]]]
  #                       ; MUST NOT be returned on non-extensible
  #                       ; "BODY" fetch
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1203
  def body_ext_mpart; end

  # body-extension  = nstring / number / number64 /
  #                      "(" body-extension *(SP body-extension) ")"
  #                       ; Future expansion.  Client implementations
  #                       ; MUST accept body-extension fields.  Server
  #                       ; implementations MUST NOT generate
  #                       ; body-extension fields except as defined by
  #                       ; future Standard or Standards Track
  #                       ; revisions of this specification.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1260
  def body_extension; end

  # body-extension *(SP body-extension)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1246
  def body_extensions; end

  # RFC-3501 & RFC-9051:
  #   body-fields     = body-fld-param SP body-fld-id SP body-fld-desc SP
  #                     body-fld-enc SP body-fld-octets
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1148
  def body_fields; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def body_fld_desc; end

  # body-fld-dsp    = "(" string SP body-fld-param ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1225
  def body_fld_dsp; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def body_fld_id; end

  # body-fld-lang   = nstring / "(" string *(SP string) ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1234
  def body_fld_lang; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def body_fld_loc; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def body_fld_md5; end

  # RFC3501, RFC9051:
  # body-fld-param  = "(" string SP string *(SP string SP string) ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1160
  def body_fld_param; end

  # RFC-3501 & RFC9051:
  #   body-type-1part = (body-type-basic / body-type-msg / body-type-text)
  #                     [SP body-ext-1part]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1024
  def body_type_1part; end

  # RFC-3501 & RFC9051:
  #   body-type-basic = media-basic SP body-fields
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1048
  def body_type_basic; end

  # This is a malformed body-type-mpart with no subparts.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1097
  def body_type_mixed; end

  # RFC-3501 & RFC-9051:
  #   body-type-mpart = 1*body SP media-subtype
  #                     [SP body-ext-mpart]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1107
  def body_type_mpart; end

  # RFC-3501 & RFC-9051:
  #   body-type-msg   = media-message SP body-fields SP envelope
  #                     SP body SP body-fld-lines
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1069
  def body_type_msg; end

  # RFC-3501 & RFC-9051:
  #   body-type-text  = media-text SP body-fields SP body-fld-lines
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1058
  def body_type_text; end

  # Returns <tt>atom.upcase</tt>
  # capability      = ("AUTH=" auth-type) / atom
  #                     ; New capabilities MUST begin with "X" or be
  #                     ; registered with IANA as standard or
  #                     ; standards-track
  #
  # source://net-imap//lib/net/imap/response_parser.rb#498
  def capability; end

  # Returns <tt>atom?&.upcase</tt>
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#501
  def capability?; end

  # As a workaround for buggy servers, allow a trailing SP:
  #     *(SP capability) [SP]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1632
  def capability__list; end

  # The presence of "IMAP4rev1" or "IMAP4rev2" is unenforced here.
  # The grammar rule is used by both response-data and resp-text-code.
  # But this method only returns UntaggedResponse (response-data).
  #
  # RFC3501:
  #   capability-data  = "CAPABILITY" *(SP capability) SP "IMAP4rev1"
  #                      *(SP capability)
  # RFC9051:
  #   capability-data  = "CAPABILITY" *(SP capability) SP "IMAP4rev2"
  #                      *(SP capability)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1621
  def capability_data__untagged; end

  # Returns <tt>atom.upcase</tt>
  #
  # source://net-imap//lib/net/imap/response_parser.rb#498
  def case_insensitive__atom; end

  # Returns <tt>atom?&.upcase</tt>
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#501
  def case_insensitive__atom?; end

  # use where nstring represents "LABEL" values
  #
  # source://net-imap//lib/net/imap/response_parser.rb#559
  def case_insensitive__nstring; end

  # See https://www.rfc-editor.org/errata/rfc3501
  #
  # charset = atom / quoted
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1968
  def charset; end

  # "(" charset *(SP charset) ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1851
  def charset__list; end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def comparator_data(klass = T.unsafe(nil)); end

  # RFC3501 & RFC9051:
  #   continue-req    = "+" SP (resp-text / base64) CRLF
  #
  # n.b: base64 is valid resp-text.  And in the spirit of RFC9051 Appx E 23
  # (and to workaround existing servers), we use the following grammar:
  #
  #   continue-req    = "+" (SP (resp-text)) CRLF
  #
  # source://net-imap//lib/net/imap/response_parser.rb#676
  def continue_req; end

  # enable-data   = "ENABLED" *(SP capability)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1626
  def enable_data; end

  # env-from        = "(" 1*address ")" / nil
  #   env-sender      = "(" 1*address ")" / nil
  #   env-reply-to    = "(" 1*address ")" / nil
  #   env-to          = "(" 1*address ")" / nil
  #   env-cc          = "(" 1*address ")" / nil
  #   env-bcc         = "(" 1*address ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#986
  def env_bcc; end

  # env-from        = "(" 1*address ")" / nil
  #   env-sender      = "(" 1*address ")" / nil
  #   env-reply-to    = "(" 1*address ")" / nil
  #   env-to          = "(" 1*address ")" / nil
  #   env-cc          = "(" 1*address ")" / nil
  #   env-bcc         = "(" 1*address ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#986
  def env_cc; end

  # nstring         = string / nil
  #   env-date        = nstring
  #   env-subject     = nstring
  #   env-in-reply-to = nstring
  #   env-message-id  = nstring
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def env_date; end

  # env-from        = "(" 1*address ")" / nil
  #   env-sender      = "(" 1*address ")" / nil
  #   env-reply-to    = "(" 1*address ")" / nil
  #   env-to          = "(" 1*address ")" / nil
  #   env-cc          = "(" 1*address ")" / nil
  #   env-bcc         = "(" 1*address ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#986
  def env_from; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def env_in_reply_to; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def env_message_id; end

  # env-from        = "(" 1*address ")" / nil
  #   env-sender      = "(" 1*address ")" / nil
  #   env-reply-to    = "(" 1*address ")" / nil
  #   env-to          = "(" 1*address ")" / nil
  #   env-cc          = "(" 1*address ")" / nil
  #   env-bcc         = "(" 1*address ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#986
  def env_reply_to; end

  # env-from        = "(" 1*address ")" / nil
  #   env-sender      = "(" 1*address ")" / nil
  #   env-reply-to    = "(" 1*address ")" / nil
  #   env-to          = "(" 1*address ")" / nil
  #   env-cc          = "(" 1*address ")" / nil
  #   env-bcc         = "(" 1*address ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#986
  def env_sender; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def env_subject; end

  # env-from        = "(" 1*address ")" / nil
  #   env-sender      = "(" 1*address ")" / nil
  #   env-reply-to    = "(" 1*address ")" / nil
  #   env-to          = "(" 1*address ")" / nil
  #   env-cc          = "(" 1*address ")" / nil
  #   env-bcc         = "(" 1*address ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#986
  def env_to; end

  # RFC3501 & RFC9051:
  #   envelope        = "(" env-date SP env-subject SP env-from SP
  #                     env-sender SP env-reply-to SP env-to SP env-cc SP
  #                     env-bcc SP env-in-reply-to SP env-message-id ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#952
  def envelope; end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def esearch_response(klass = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def expunged_resp(klass = T.unsafe(nil)); end

  # flag-list       = "(" [flag *(SP flag)] ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1918
  def flag_list; end

  # "(" [flag-perm *(SP flag-perm)] ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1928
  def flag_perm__list; end

  # TODO: handle atom, astring_chars, and tag entirely inside the lexer
  # this represents the partial size for BODY or BINARY
  #
  # source://net-imap//lib/net/imap/response_parser.rb#490
  def gt__number__lt; end

  # RFC3501 & RFC9051:
  #   header-fld-name = astring
  #
  # NOTE: Previously, Net::IMAP recreated the raw original source string.
  # Now, it grabs the raw encoded value using @str and @pos.  A future
  # version may simply return the decoded astring value.  Although that is
  # technically incompatible, it should almost never make a difference: all
  # standard header field names are valid atoms:
  #
  # https://www.iana.org/assignments/message-headers/message-headers.xhtml
  #
  # Although RFC3501 allows any astring, RFC5322-valid header names are one
  # or more of the printable US-ASCII characters, except SP and colon.  So
  # empty string isn't valid, and literals aren't needed and should not be
  # used.  This is explicitly unchanged by [I18N-HDRS] (RFC6532).
  #
  # RFC5233:
  #     optional-field  =   field-name ":" unstructured CRLF
  #     field-name      =   1*ftext
  #     ftext           =   %d33-57 /          ; Printable US-ASCII
  #                         %d59-126           ;  characters not including
  #                                            ;  ":".
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1338
  def header_fld_name; end

  # header-list     = "(" header-fld-name *(SP header-fld-name) ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1304
  def header_list; end

  # source://net-imap//lib/net/imap/response_parser.rb#1645
  def id_response; end

  # Use #label or #label_in to assert specific known labels
  # (+tagged-ext-label+ only, not +atom+).
  #
  # source://net-imap//lib/net/imap/response_parser.rb#516
  def label(word); end

  # Use #label or #label_in to assert specific known labels
  # (+tagged-ext-label+ only, not +atom+).
  #
  # source://net-imap//lib/net/imap/response_parser.rb#523
  def label_in(*labels); end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def language_data(klass = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def listrights_data(klass = T.unsafe(nil)); end

  # astring         = 1*ASTRING-CHAR / string
  # mailbox         = "INBOX" / astring
  #                     ; INBOX is case-insensitive.  All case variants of
  #                     ; INBOX (e.g., "iNbOx") MUST be interpreted as INBOX
  #                     ; not as an astring.  An astring which consists of
  #                     ; the case-insensitive sequence "I" "N" "B" "O" "X"
  #                     ; is considered to be INBOX and not an astring.
  #                     ;  Refer to section 5.1 for further
  #                     ; semantic details of mailbox names.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#506
  def mailbox; end

  # source://net-imap//lib/net/imap/response_parser.rb#835
  def mailbox_data__exists; end

  # mailbox-data    =  "FLAGS" SP flag-list / "LIST" SP mailbox-list /
  #                    "LSUB" SP mailbox-list / "SEARCH" *(SP nz-number) /
  #                    "STATUS" SP mailbox SP "(" [status-att-list] ")" /
  #                    number SP "EXISTS" / number SP "RECENT"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1351
  def mailbox_data__flags; end

  # source://net-imap//lib/net/imap/response_parser.rb#1357
  def mailbox_data__list; end

  # source://net-imap//lib/net/imap/response_parser.rb#1357
  def mailbox_data__lsub; end

  # source://net-imap//lib/net/imap/response_parser.rb#835
  def mailbox_data__recent; end

  # RFC3501:
  #   mailbox-data        = "SEARCH" *(SP nz-number) / ...
  # RFC5256: SORT
  #   sort-data           = "SORT" *(SP nz-number)
  # RFC7162: CONDSTORE, QRESYNC
  #   mailbox-data        =/ "SEARCH" [1*(SP nz-number) SP
  #                          search-sort-mod-seq]
  #   sort-data           = "SORT" [1*(SP nz-number) SP
  #                           search-sort-mod-seq]
  #                           ; Updates the SORT response from RFC 5256.
  #   search-sort-mod-seq = "(" "MODSEQ" SP mod-sequence-value ")"
  # RFC9051:
  #   mailbox-data        = obsolete-search-response / ...
  #   obsolete-search-response = "SEARCH" *(SP nz-number)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1469
  def mailbox_data__search; end

  # mailbox-data    =/ "STATUS" SP mailbox SP "(" [status-att-list] ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1535
  def mailbox_data__status; end

  # source://net-imap//lib/net/imap/response_parser.rb#1357
  def mailbox_data__xlist; end

  # mailbox-list    = "(" [mbx-list-flags] ")" SP
  #                    (DQUOTE QUOTED-CHAR DQUOTE / nil) SP mailbox
  #                    [SP mbox-list-extended]
  #             ; This is the list information pointed to by the ABNF
  #             ; item "mailbox-data", which is defined above
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1370
  def mailbox_list; end

  # See Patterns::MBX_LIST_FLAGS
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1946
  def mbx_list_flags; end

  # n.b. this handles both type and subtype
  #
  # RFC-3501 vs RFC-9051:
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "MESSAGE" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "FONT" / "MESSAGE" / "MODEL" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE "RFC822" DQUOTE
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE ("RFC822" / "GLOBAL") DQUOTE
  #
  # RFC-3501 & RFC-9051:
  #   media-text      = DQUOTE "TEXT" DQUOTE SP media-subtype
  #   media-subtype   = string
  # TODO: check types
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1131
  def media_basic; end

  # n.b. this handles both type and subtype
  #
  # RFC-3501 vs RFC-9051:
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "MESSAGE" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "FONT" / "MESSAGE" / "MODEL" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE "RFC822" DQUOTE
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE ("RFC822" / "GLOBAL") DQUOTE
  #
  # RFC-3501 & RFC-9051:
  #   media-text      = DQUOTE "TEXT" DQUOTE SP media-subtype
  #   media-subtype   = string
  # */* --- catchall
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1131
  def media_message; end

  # n.b. this handles both type and subtype
  #
  # RFC-3501 vs RFC-9051:
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "MESSAGE" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "FONT" / "MESSAGE" / "MODEL" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE "RFC822" DQUOTE
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE ("RFC822" / "GLOBAL") DQUOTE
  #
  # RFC-3501 & RFC-9051:
  #   media-text      = DQUOTE "TEXT" DQUOTE SP media-subtype
  #   media-subtype   = string
  # message/rfc822, message/global
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1131
  def media_text; end

  # n.b. this handles both type and subtype
  #
  # RFC-3501 vs RFC-9051:
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "MESSAGE" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" /
  #                     "FONT" / "MESSAGE" / "MODEL" /
  #                     "VIDEO") DQUOTE) / string) SP media-subtype
  #
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE "RFC822" DQUOTE
  #   media-message   = DQUOTE "MESSAGE" DQUOTE SP
  #                     DQUOTE ("RFC822" / "GLOBAL") DQUOTE
  #
  # RFC-3501 & RFC-9051:
  #   media-text      = DQUOTE "TEXT" DQUOTE SP media-subtype
  #   media-subtype   = string
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1131
  def media_type; end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def message_data__converted(klass = T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/response_parser.rb#835
  def message_data__expunge; end

  # message-data    = nz-number SP ("EXPUNGE" / ("FETCH" SP msg-att))
  #
  # source://net-imap//lib/net/imap/response_parser.rb#828
  def message_data__fetch; end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def metadata_resp(klass = T.unsafe(nil)); end

  # RFC3501 & RFC9051:
  #   msg-att         = "(" (msg-att-dynamic / msg-att-static)
  #                      *(SP (msg-att-dynamic / msg-att-static)) ")"
  #
  #   msg-att-dynamic = "FLAGS" SP "(" [flag-fetch *(SP flag-fetch)] ")"
  # RFC5257 (ANNOTATE extension):
  #   msg-att-dynamic =/ "ANNOTATION" SP
  #                        ( "(" entry-att *(SP entry-att) ")" /
  #                          "(" entry *(SP entry) ")" )
  # RFC7162 (CONDSTORE extension):
  #   msg-att-dynamic =/ fetch-mod-resp
  #   fetch-mod-resp  = "MODSEQ" SP "(" permsg-modsequence ")"
  # RFC8970 (PREVIEW extension):
  #   msg-att-dynamic =/ "PREVIEW" SP nstring
  #
  # RFC3501:
  #   msg-att-static  = "ENVELOPE" SP envelope /
  #                     "INTERNALDATE" SP date-time /
  #                     "RFC822" [".HEADER" / ".TEXT"] SP nstring /
  #                     "RFC822.SIZE" SP number /
  #                     "BODY" ["STRUCTURE"] SP body /
  #                     "BODY" section ["<" number ">"] SP nstring /
  #                     "UID" SP uniqueid
  # RFC3516 (BINARY extension):
  #   msg-att-static  =/ "BINARY" section-binary SP (nstring / literal8)
  #                    / "BINARY.SIZE" section-binary SP number
  # RFC8514 (SAVEDATE extension):
  #   msg-att-static  =/ "SAVEDATE" SP (date-time / nil)
  # RFC8474 (OBJECTID extension):
  #   msg-att-static =/ fetch-emailid-resp / fetch-threadid-resp
  #   fetch-emailid-resp  = "EMAILID" SP "(" objectid ")"
  #   fetch-threadid-resp = "THREADID" SP ( "(" objectid ")" / nil )
  # RFC9051:
  #   msg-att-static  = "ENVELOPE" SP envelope /
  #                     "INTERNALDATE" SP date-time /
  #                     "RFC822.SIZE" SP number64 /
  #                     "BODY" ["STRUCTURE"] SP body /
  #                     "BODY" section ["<" number ">"] SP nstring /
  #                     "BINARY" section-binary SP (nstring / literal8) /
  #                     "BINARY.SIZE" section-binary SP number /
  #                     "UID" SP uniqueid
  #
  # Re https://www.rfc-editor.org/errata/eid7246, I'm adding "offset" to the
  # official "BINARY" ABNF, like so:
  #
  #   msg-att-static   =/ "BINARY" section-binary ["<" number ">"] SP
  #                       (nstring / literal8)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#892
  def msg_att(n); end

  # appends "[section]" and "<partial>" to the base label
  #
  # source://net-imap//lib/net/imap/response_parser.rb#929
  def msg_att__label; end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def myrights_data(klass = T.unsafe(nil)); end

  # namespace         = nil / "(" 1*namespace-descr ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1692
  def namespace; end

  # namespace-descr   = "(" string SP
  #                        (DQUOTE QUOTED-CHAR DQUOTE / nil)
  #                         [namespace-response-extensions] ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1703
  def namespace_descr; end

  # namespace-response = "NAMESPACE" SP namespace
  #                       SP namespace SP namespace
  #                  ; The first Namespace is the Personal Namespace(s).
  #                  ; The second Namespace is the Other Users'
  #                  ; Namespace(s).
  #                  ; The third Namespace is the Shared Namespace(s).
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1680
  def namespace_response; end

  # namespace-response-extensions = *namespace-response-extension
  # namespace-response-extension = SP string SP
  #                   "(" string *(SP string) ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1715
  def namespace_response_extensions; end

  # source://net-imap//lib/net/imap/response_parser.rb#554
  def ndatetime; end

  # source://net-imap//lib/net/imap/response_parser.rb#2035
  def next_token; end

  # source://net-imap//lib/net/imap/response_parser.rb#2017
  def nil_atom; end

  # env-from        = "(" 1*address ")" / nil
  #   env-sender      = "(" 1*address ")" / nil
  #   env-reply-to    = "(" 1*address ")" / nil
  #   env-to          = "(" 1*address ")" / nil
  #   env-cc          = "(" 1*address ")" / nil
  #   env-bcc         = "(" 1*address ")" / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#986
  def nlist__address; end

  # source://net-imap//lib/net/imap/response_parser.rb#1995
  def nparens__objectid; end

  # source://net-imap//lib/net/imap/response_parser.rb#554
  def nquoted; end

  # nstring         = string / nil
  #
  # source://net-imap//lib/net/imap/response_parser.rb#546
  def nstring; end

  # source://net-imap//lib/net/imap/response_parser.rb#550
  def nstring8; end

  # TODO: handle atom, astring_chars, and tag entirely inside the lexer
  # RFC8474:
  # objectid = 1*255(ALPHA / DIGIT / "_" / "-")
  #         ; characters in object identifiers are case
  #         ; significant
  #
  # source://net-imap//lib/net/imap/response_parser.rb#490
  def objectid; end

  # source://net-imap//lib/net/imap/response_parser.rb#1986
  def parens__modseq; end

  # source://net-imap//lib/net/imap/response_parser.rb#1994
  def parens__objectid; end

  # This allows illegal "]" in flag names (Gmail),
  # or "\*" in a FLAGS response (greenmail).
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1939
  def quirky__flag_list(name); end

  # source://net-imap//lib/net/imap/response_parser.rb#1378
  def quota_response; end

  # source://net-imap//lib/net/imap/response_parser.rb#1411
  def quotaroot_response; end

  # reads all the way up until CRLF
  #
  # source://net-imap//lib/net/imap/response_parser.rb#764
  def remaining_unparsed; end

  # As a workaround for buggy servers, allow a trailing SP:
  #     *(SP capability) [SP]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1632
  def resp_code__capability; end

  # already matched:  "APPENDUID"
  #
  # +UIDPLUS+ ABNF:: https://www.rfc-editor.org/rfc/rfc4315.html#section-4
  #   resp-code-apnd  = "APPENDUID" SP nz-number SP append-uid
  #   append-uid      = uniqueid
  #   append-uid      =/ uid-set
  #                     ; only permitted if client uses [MULTIAPPEND]
  #                     ; to append multiple messages.
  #
  # n.b, uniqueid ⊂ uid-set.  To avoid inconsistent return types, we always
  # match uid_set even if that returns a single-member array.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1867
  def resp_code_apnd__data; end

  # already matched:  "COPYUID"
  #
  # resp-code-copy  = "COPYUID" SP nz-number SP uid-set SP uid-set
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1876
  def resp_code_copy__data; end

  # resp-cond-auth   = ("OK" / "PREAUTH") SP resp-text
  #
  # NOTE: In the spirit of RFC9051 Appx E 23 (and to workaround existing
  # servers), we don't require a final SP and instead parse this as:
  #
  #   resp-cond-auth   = ("OK" / "PREAUTH") [SP resp-text]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#809
  def resp_cond_auth; end

  # expects "OK" or "PREAUTH" and raises InvalidResponseError on failure
  #
  # @raise [InvalidResponseError]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#530
  def resp_cond_auth__name; end

  # resp-cond-bye    = "BYE" SP resp-text
  #
  # NOTE: In the spirit of RFC9051 Appx E 23 (and to workaround existing
  # servers), we don't require a final SP and instead parse this as:
  #
  #   resp-cond-bye    = "BYE" [SP resp-text]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#821
  def resp_cond_bye; end

  # RFC3501 & RFC9051:
  #   resp-cond-state  = ("OK" / "NO" / "BAD") SP resp-text
  #
  # NOTE: In the spirit of RFC9051 Appx E 23 (and to workaround existing
  # servers), we don't require a final SP and instead parse this as:
  #
  #   resp-cond-state = ("OK" / "NO" / "BAD") [SP resp-text]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#795
  def resp_cond_state; end

  # expects "OK" or "NO" or "BAD" and raises InvalidResponseError on failure
  #
  # @raise [InvalidResponseError]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#538
  def resp_cond_state__name; end

  # source://net-imap//lib/net/imap/response_parser.rb#799
  def resp_cond_state__untagged; end

  # RFC3501:
  #   resp-text       = ["[" resp-text-code "]" SP] text
  # RFC9051:
  #   resp-text       = ["[" resp-text-code "]" SP] [text]
  #
  # We leniently re-interpret this as
  #   resp-text       = ["[" resp-text-code "]" [SP [text]] / [text]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1751
  def resp_text; end

  # RFC3501 (See https://www.rfc-editor.org/errata/rfc3501):
  #   resp-text-code   = "ALERT" /
  #                      "BADCHARSET" [SP "(" charset *(SP charset) ")" ] /
  #                      capability-data / "PARSE" /
  #                      "PERMANENTFLAGS" SP "(" [flag-perm *(SP flag-perm)] ")" /
  #                      "READ-ONLY" / "READ-WRITE" / "TRYCREATE" /
  #                      "UIDNEXT" SP nz-number / "UIDVALIDITY" SP nz-number /
  #                      "UNSEEN" SP nz-number /
  #                      atom [SP 1*<any TEXT-CHAR except "]">]
  #   capability-data  = "CAPABILITY" *(SP capability) SP "IMAP4rev1"
  #                      *(SP capability)
  #
  # RFC5530:
  #   resp-text-code  =/ "UNAVAILABLE" / "AUTHENTICATIONFAILED" /
  #                     "AUTHORIZATIONFAILED" / "EXPIRED" /
  #                     "PRIVACYREQUIRED" / "CONTACTADMIN" / "NOPERM" /
  #                     "INUSE" / "EXPUNGEISSUED" / "CORRUPTION" /
  #                     "SERVERBUG" / "CLIENTBUG" / "CANNOT" /
  #                     "LIMIT" / "OVERQUOTA" / "ALREADYEXISTS" /
  #                     "NONEXISTENT"
  # RFC9051:
  #   resp-text-code   = "ALERT" /
  #                      "BADCHARSET" [SP "(" charset *(SP charset) ")" ] /
  #                      capability-data / "PARSE" /
  #                      "PERMANENTFLAGS" SP "(" [flag-perm *(SP flag-perm)] ")" /
  #                      "READ-ONLY" / "READ-WRITE" / "TRYCREATE" /
  #                      "UIDNEXT" SP nz-number / "UIDVALIDITY" SP nz-number /
  #                      resp-code-apnd / resp-code-copy / "UIDNOTSTICKY" /
  #                      "UNAVAILABLE" / "AUTHENTICATIONFAILED" /
  #                      "AUTHORIZATIONFAILED" / "EXPIRED" /
  #                      "PRIVACYREQUIRED" / "CONTACTADMIN" / "NOPERM" /
  #                      "INUSE" / "EXPUNGEISSUED" / "CORRUPTION" /
  #                      "SERVERBUG" / "CLIENTBUG" / "CANNOT" /
  #                      "LIMIT" / "OVERQUOTA" / "ALREADYEXISTS" /
  #                      "NONEXISTENT" / "NOTSAVED" / "HASCHILDREN" /
  #                      "CLOSED" /
  #                      "UNKNOWN-CTE" /
  #                      atom [SP 1*<any TEXT-CHAR except "]">]
  #   capability-data  = "CAPABILITY" *(SP capability) SP "IMAP4rev2"
  #                      *(SP capability)
  #
  # RFC4315 (UIDPLUS), RFC9051 (IMAP4rev2):
  #   resp-code-apnd   = "APPENDUID" SP nz-number SP append-uid
  #   resp-code-copy   = "COPYUID" SP nz-number SP uid-set SP uid-set
  #   resp-text-code   =/ resp-code-apnd / resp-code-copy / "UIDNOTSTICKY"
  #
  # RFC7162 (CONDSTORE):
  #   resp-text-code   =/ "HIGHESTMODSEQ" SP mod-sequence-value /
  #                       "NOMODSEQ" /
  #                       "MODIFIED" SP sequence-set
  # RFC7162 (QRESYNC):
  #   resp-text-code   =/ "CLOSED"
  #
  # RFC8474: OBJECTID
  #   resp-text-code   =/ "MAILBOXID" SP "(" objectid ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1815
  def resp_text_code; end

  # Returns <tt>atom.upcase</tt>
  #
  # source://net-imap//lib/net/imap/response_parser.rb#498
  def resp_text_code__name; end

  # [RFC3501 & RFC9051:]
  #   response        = *(continue-req / response-data) response-done
  #
  # For simplicity, response isn't interpreted as the combination of the
  # three response types, but instead represents any individual server
  # response.  Our simplified interpretation is defined as:
  #   response        = continue-req | response_data | response-tagged
  #
  # n.b: our "response-tagged" definition parses "greeting" too.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#657
  def response; end

  # [RFC3501:]
  #   response-data    = "*" SP (resp-cond-state / resp-cond-bye /
  #                      mailbox-data / message-data / capability-data) CRLF
  # [RFC4466:]
  #   response-data    = "*" SP response-payload CRLF
  #   response-payload = resp-cond-state / resp-cond-bye /
  #                       mailbox-data / message-data / capability-data
  # RFC5161 (ENABLE capability):
  #   response-data    =/ "*" SP enable-data CRLF
  # RFC5255 (LANGUAGE capability)
  #   response-payload =/ language-data
  # RFC5255 (I18NLEVEL=1 and I18NLEVEL=2 capabilities)
  #   response-payload =/ comparator-data
  # [RFC9051:]
  #   response-data    = "*" SP (resp-cond-state / resp-cond-bye /
  #                      mailbox-data / message-data / capability-data /
  #                      enable-data) CRLF
  #
  # [merging in greeting and response-fatal:]
  #   greeting         =  "*" SP (resp-cond-auth / resp-cond-bye) CRLF
  #   response-fatal   =  "*" SP resp-cond-bye CRLF
  #   response-data    =/ "*" SP (resp-cond-auth / resp-cond-bye) CRLF
  # [removing duplicates, this is simply]
  #   response-payload =/ resp-cond-auth
  #
  # TODO: remove resp-cond-auth and handle greeting separately
  #
  # source://net-imap//lib/net/imap/response_parser.rb#709
  def response_data; end

  # source://net-imap//lib/net/imap/response_parser.rb#769
  def response_data__ignored; end

  # source://net-imap//lib/net/imap/response_parser.rb#769
  def response_data__noop; end

  # source://net-imap//lib/net/imap/response_parser.rb#835
  def response_data__simple_numeric; end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def response_data__unhandled(klass = T.unsafe(nil)); end

  # RFC3501 & RFC9051:
  #   response-tagged = tag SP resp-cond-state CRLF
  #
  # source://net-imap//lib/net/imap/response_parser.rb#784
  def response_tagged; end

  # section         = "[" [section-spec] "]"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1268
  def section; end

  # section-binary  = "[" [section-part] "]"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1275
  def section_binary; end

  # TODO: handle atom, astring_chars, and tag entirely inside the lexer
  # section-part    = nz-number *("." nz-number)
  #                     ; body part reference.
  #                     ; Allows for accessing nested body parts.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#490
  def section_part; end

  # section-spec    = section-msgtext / (section-part ["." section-text])
  # section-msgtext = "HEADER" /
  #                   "HEADER.FIELDS" [".NOT"] SP header-list /
  #                   "TEXT"
  #                     ; top-level or MESSAGE/RFC822 or
  #                     ; MESSAGE/GLOBAL part
  # section-part    = nz-number *("." nz-number)
  #                     ; body part reference.
  #                     ; Allows for accessing nested body parts.
  # section-text    = section-msgtext / "MIME"
  #                     ; text other than actual body part (headers,
  #                     ; etc.)
  #
  # n.b: we could "cheat" here and just grab all text inside the brackets,
  # but literals would need special treatment.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1296
  def section_spec; end

  # sequence-set    = (seq-number / seq-range) ["," sequence-set]
  #   sequence-set    =/ seq-last-command
  #                       ; Allow for "result of the last command"
  #                       ; indicator.
  #   seq-last-command   = "$"
  #
  # *note*: doesn't match seq-last-command
  #
  # source://net-imap//lib/net/imap/response_parser.rb#471
  def sequence_set; end

  # RFC3501:
  #   mailbox-data        = "SEARCH" *(SP nz-number) / ...
  # RFC5256: SORT
  #   sort-data           = "SORT" *(SP nz-number)
  # RFC7162: CONDSTORE, QRESYNC
  #   mailbox-data        =/ "SEARCH" [1*(SP nz-number) SP
  #                          search-sort-mod-seq]
  #   sort-data           = "SORT" [1*(SP nz-number) SP
  #                           search-sort-mod-seq]
  #                           ; Updates the SORT response from RFC 5256.
  #   search-sort-mod-seq = "(" "MODSEQ" SP mod-sequence-value ")"
  # RFC9051:
  #   mailbox-data        = obsolete-search-response / ...
  #   obsolete-search-response = "SEARCH" *(SP nz-number)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1469
  def sort_data; end

  # RFC3501
  #   status-att-list = status-att SP number *(SP status-att SP number)
  # RFC4466, RFC9051, and RFC3501 Errata
  #   status-att-list = status-att-val *(SP status-att-val)
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1546
  def status_att_list; end

  # RFC3501 Errata:
  # status-att-val  = ("MESSAGES" SP number) / ("RECENT" SP number) /
  #                   ("UIDNEXT" SP nz-number) / ("UIDVALIDITY" SP nz-number) /
  #                   ("UNSEEN" SP number)
  # RFC4466:
  # status-att-val  = ("MESSAGES" SP number) /
  #                   ("RECENT" SP number) /
  #                   ("UIDNEXT" SP nz-number) /
  #                   ("UIDVALIDITY" SP nz-number) /
  #                   ("UNSEEN" SP number)
  #                   ;; Extensions to the STATUS responses
  #                   ;; should extend this production.
  #                   ;; Extensions should use the generic
  #                   ;; syntax defined by tagged-ext.
  # RFC9051:
  # status-att-val  = ("MESSAGES" SP number) /
  #                   ("UIDNEXT" SP nz-number) /
  #                   ("UIDVALIDITY" SP nz-number) /
  #                   ("UNSEEN" SP number) /
  #                   ("DELETED" SP number) /
  #                   ("SIZE" SP number64)
  #                     ; Extensions to the STATUS responses
  #                     ; should extend this production.
  #                     ; Extensions should use the generic
  #                     ; syntax defined by tagged-ext.
  # RFC7162:
  # status-att-val      =/ "HIGHESTMODSEQ" SP mod-sequence-valzer
  #                        ;; Extends non-terminal defined in [RFC4466].
  #                        ;; Value 0 denotes that the mailbox doesn't
  #                        ;; support persistent mod-sequences
  #                        ;; as described in Section 3.1.2.2.
  # RFC7889:
  # status-att-val =/ "APPENDLIMIT" SP (number / nil)
  #                 ;; status-att-val is defined in RFC 4466
  # RFC8438:
  # status-att-val =/ "SIZE" SP number64
  # RFC8474:
  # status-att-val =/ "MAILBOXID" SP "(" objectid ")"
  #         ; follows tagged-ext production from [RFC4466]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1591
  def status_att_val; end

  # source://net-imap//lib/net/imap/response_parser.rb#492
  def tag; end

  # tagged-ext-comp     = astring /
  #                       tagged-ext-comp *(SP tagged-ext-comp) /
  #                       "(" tagged-ext-comp ")"
  #                       ; Extensions that follow this general
  #                       ; syntax should use nstring instead of
  #                       ; astring when appropriate in the context
  #                       ; of the extension.
  #                       ; Note that a message set or a "number"
  #                       ; can always be represented as an "atom".
  #                       ; A URL should be represented as
  #                       ; a "quoted" string.
  #
  # source://net-imap//lib/net/imap/response_parser.rb#574
  def tagged_ext_comp; end

  # tagged-ext-simple is a subset of atom
  # TODO: recognize sequence-set in the lexer
  #
  # tagged-ext-simple   = sequence-set / number / number64
  #
  # source://net-imap//lib/net/imap/response_parser.rb#591
  def tagged_ext_simple; end

  # tagged-ext-val      = tagged-ext-simple /
  #                       "(" [tagged-ext-comp] ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#597
  def tagged_ext_val; end

  # TEXT-CHAR       = <any CHAR except CR and LF>
  # RFC3501:
  #   text            = 1*TEXT-CHAR
  # RFC9051:
  #   text            = 1*(TEXT-CHAR / UTF8-2 / UTF8-3 / UTF8-4)
  #                     ; Non-ASCII text can only be returned
  #                     ; after ENABLE IMAP4rev2 command
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1735
  def text; end

  # an "accept" versiun of #text
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1740
  def text?; end

  # 1*<any TEXT-CHAR except "]">
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1846
  def text_chars_except_rbra; end

  # RFC5256: THREAD
  #   thread-data     = "THREAD" [SP 1*thread-list]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1485
  def thread_data; end

  # RFC5256: THREAD
  #   thread-list     = "(" (thread-members / thread-nested) ")"
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1499
  def thread_list; end

  # RFC5256: THREAD
  #   thread-members  = nz-number *(SP nz-number) [SP thread-nested]
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1512
  def thread_members; end

  # RFC5256: THREAD
  #   thread-nested   = 2*thread-list
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1528
  def thread_nested; end

  # RFC-4315 (UIDPLUS) or RFC9051 (IMAP4rev2):
  #      uid-set         = (uniqueid / uid-range) *("," uid-set)
  #      uid-range       = (uniqueid ":" uniqueid)
  #                          ; two uniqueid values and all values
  #                          ; between these two regardless of order.
  #                          ; Example: 2:4 and 4:2 are equivalent.
  #      uniqueid        = nz-number
  #                          ; Strictly ascending
  #
  # source://net-imap//lib/net/imap/response_parser.rb#2005
  def uid_set; end

  # source://net-imap//lib/net/imap/response_parser.rb#751
  def uidfetch_resp(klass = T.unsafe(nil)); end

  # See https://developers.google.com/gmail/imap/imap-extensions
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1953
  def x_gm_label; end

  # See https://developers.google.com/gmail/imap/imap-extensions
  #
  # source://net-imap//lib/net/imap/response_parser.rb#1956
  def x_gm_labels; end
end

# ASTRING-CHAR    = ATOM-CHAR / resp-specials
# resp-specials   = "]"
#
# source://net-imap//lib/net/imap/response_parser.rb#482
Net::IMAP::ResponseParser::ASTRING_CHARS_TOKENS = T.let(T.unsafe(nil), Array)

# source://net-imap//lib/net/imap/response_parser.rb#484
Net::IMAP::ResponseParser::ASTRING_TOKENS = T.let(T.unsafe(nil), Array)

# basic utility methods for parsing.
#
# (internal API, subject to change)
#
# source://net-imap//lib/net/imap/response_parser/parser_utils.rb#9
module Net::IMAP::ResponseParser::ParserUtils
  private

  # like match, but does not raise error on failure.
  #
  # returns and shifts token on successful match
  # returns nil and leaves @token unshifted on no match
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#148
  def accept(*args); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#193
  def accept_re(re); end

  # To be used conditionally:
  #   assert_no_lookahead if config.debug?
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#158
  def assert_no_lookahead; end

  # TODO: after checking the lookahead, use a regexp for remaining chars.
  # That way a loop isn't needed.
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#121
  def combine_adjacent(*tokens); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#169
  def lookahead; end

  # like match, without consuming the token
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#174
  def lookahead!(*args); end

  # like accept, without consuming the token
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#165
  def lookahead?(*symbols); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#133
  def match(*args); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#199
  def match_re(re, name); end

  # @raise [ResponseParseError]
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#213
  def parse_error(fmt, *args); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#188
  def peek_re(re); end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#183
  def peek_str?(str); end

  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#209
  def shift_token; end
end

# source://net-imap//lib/net/imap/response_parser/parser_utils.rb#11
module Net::IMAP::ResponseParser::ParserUtils::Generator
  # we can skip lexer for single character matches, as a shortcut
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#17
  def def_char_matchers(name, char, token); end

  # TODO: move coersion to the token.value method?
  #
  # source://net-imap//lib/net/imap/response_parser/parser_utils.rb#62
  def def_token_matchers(name, *token_symbols, coerce: T.unsafe(nil), send: T.unsafe(nil)); end
end

# source://net-imap//lib/net/imap/response_parser/parser_utils.rb#13
Net::IMAP::ResponseParser::ParserUtils::Generator::LOOKAHEAD = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/response_parser/parser_utils.rb#14
Net::IMAP::ResponseParser::ParserUtils::Generator::SHIFT_TOKEN = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/response_parser.rb#80
module Net::IMAP::ResponseParser::Patterns
  include ::Net::IMAP::ResponseParser::Patterns::RFC5234
  include ::Net::IMAP::ResponseParser::Patterns::RFC3629

  private

  # source://net-imap//lib/net/imap/response_parser.rb#355
  def unescape_quoted(quoted); end

  # source://net-imap//lib/net/imap/response_parser.rb#349
  def unescape_quoted!(quoted); end

  class << self
    # source://net-imap//lib/net/imap/response_parser.rb#355
    def unescape_quoted(quoted); end

    # source://net-imap//lib/net/imap/response_parser.rb#349
    def unescape_quoted!(quoted); end
  end
end

# source://net-imap//lib/net/imap/response_parser.rb#177
Net::IMAP::ResponseParser::Patterns::ASTRING_CHAR = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#181
Net::IMAP::ResponseParser::Patterns::ASTRING_CHARS = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#175
Net::IMAP::ResponseParser::Patterns::ASTRING_SPECIALS = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#180
Net::IMAP::ResponseParser::Patterns::ATOM = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#182
Net::IMAP::ResponseParser::Patterns::ATOMISH = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#178
Net::IMAP::ResponseParser::Patterns::ATOM_CHAR = T.let(T.unsafe(nil), Regexp)

# atomish         = 1*<any ATOM-CHAR except "[">
#                 ; We use "atomish" for msg-att and section, in order
#                 ; to simplify "BODY[HEADER.FIELDS (foo bar)]".
#
# atom-specials   = "(" / ")" / "{" / SP / CTL / list-wildcards /
#                   quoted-specials / resp-specials
# ATOM-CHAR       = <any CHAR except atom-specials>
# atom            = 1*ATOM-CHAR
# ASTRING-CHAR    = ATOM-CHAR / resp-specials
# tag             = 1*<any ASTRING-CHAR except "+">
#
# source://net-imap//lib/net/imap/response_parser.rb#174
Net::IMAP::ResponseParser::Patterns::ATOM_SPECIALS = T.let(T.unsafe(nil), Regexp)

# CHAR8           = %x01-ff
#                     ; any OCTET except NUL, %x00
#
# source://net-imap//lib/net/imap/response_parser.rb#154
Net::IMAP::ResponseParser::Patterns::CHAR8 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#190
Net::IMAP::ResponseParser::Patterns::CODE_TEXT = T.let(T.unsafe(nil), Regexp)

# resp-text-code  = ... / atom [SP 1*<any TEXT-CHAR except "]">]
#
# source://net-imap//lib/net/imap/response_parser.rb#189
Net::IMAP::ResponseParser::Patterns::CODE_TEXT_CHAR = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#82
module Net::IMAP::ResponseParser::Patterns::CharClassSubtraction; end

# flag            = "\Answered" / "\Flagged" / "\Deleted" /
#                   "\Seen" / "\Draft" / flag-keyword / flag-extension
#                     ; Does not include "\Recent"
# flag-extension  = "\" atom
#                     ; Future expansion.  Client implementations
#                     ; MUST accept flag-extension flags.  Server
#                     ; implementations MUST NOT generate
#                     ; flag-extension flags except as defined by
#                     ; a future Standard or Standards Track
#                     ; revisions of this specification.
# flag-keyword    = "$MDNSent" / "$Forwarded" / "$Junk" /
#                   "$NotJunk" / "$Phishing" / atom
#
# flag-perm       = flag / "\*"
#
# Not checking for max one mbx-list-sflag in the parser.
# >>>
# mbx-list-oflag  = "\Noinferiors" / child-mbox-flag /
#                   "\Subscribed" / "\Remote" / flag-extension
#                    ; Other flags; multiple from this list are
#                    ; possible per LIST response, but each flag
#                    ; can only appear once per LIST response
# mbx-list-sflag  = "\NonExistent" / "\Noselect" / "\Marked" /
#                   "\Unmarked"
#                    ; Selectability flags; only one per LIST response
# child-mbox-flag =  "\HasChildren" / "\HasNoChildren"
#                    ; attributes for the CHILDREN return option, at most
#                    ; one possible per LIST response
#
# source://net-imap//lib/net/imap/response_parser.rb#220
Net::IMAP::ResponseParser::Patterns::FLAG = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#221
Net::IMAP::ResponseParser::Patterns::FLAG_EXTENSION = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#222
Net::IMAP::ResponseParser::Patterns::FLAG_KEYWORD = T.let(T.unsafe(nil), Regexp)

# flag-list       = "(" [flag *(SP flag)] ")"
# resp-text-code =/ "PERMANENTFLAGS" SP
#                   "(" [flag-perm *(SP flag-perm)] ")"
# mbx-list-flags  = *(mbx-list-oflag SP) mbx-list-sflag
#                   *(SP mbx-list-oflag) /
#                   mbx-list-oflag *(SP mbx-list-oflag)
# (Not checking for max one mbx-list-sflag in the parser.)
#
# source://net-imap//lib/net/imap/response_parser.rb#233
Net::IMAP::ResponseParser::Patterns::FLAG_LIST = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#223
Net::IMAP::ResponseParser::Patterns::FLAG_PERM = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#234
Net::IMAP::ResponseParser::Patterns::FLAG_PERM_LIST = T.let(T.unsafe(nil), Regexp)

# list-wildcards  = "%" / "*"
#
# source://net-imap//lib/net/imap/response_parser.rb#157
Net::IMAP::ResponseParser::Patterns::LIST_WILDCARDS = T.let(T.unsafe(nil), Regexp)

# RFC3501:
#   literal          = "{" number "}" CRLF *CHAR8
#                        ; Number represents the number of CHAR8s
# RFC9051:
#   literal          = "{" number64 ["+"] "}" CRLF *CHAR8
#                        ; <number64> represents the number of CHAR8s.
#                        ; A non-synchronizing literal is distinguished
#                        ; from a synchronizing literal by the presence of
#                        ; "+" before the closing "}".
#                        ; Non-synchronizing literals are not allowed when
#                        ; sent from server to the client.
#
# source://net-imap//lib/net/imap/response_parser.rb#335
Net::IMAP::ResponseParser::Patterns::LITERAL = T.let(T.unsafe(nil), Regexp)

# RFC3516 (BINARY):
#   literal8         =   "~{" number "}" CRLF *OCTET
#                        ; <number> represents the number of OCTETs
#                        ; in the response string.
# RFC9051:
#   literal8         =  "~{" number64 "}" CRLF *OCTET
#                        ; <number64> represents the number of OCTETs
#                        ; in the response string.
#
# source://net-imap//lib/net/imap/response_parser.rb#345
Net::IMAP::ResponseParser::Patterns::LITERAL8 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#224
Net::IMAP::ResponseParser::Patterns::MBX_FLAG = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#235
Net::IMAP::ResponseParser::Patterns::MBX_LIST_FLAGS = T.let(T.unsafe(nil), Regexp)

# nz-number       = digit-nz *DIGIT
#                     ; Non-zero unsigned 32-bit integer
#                     ; (0 < n < 4,294,967,296)
#
# source://net-imap//lib/net/imap/response_parser.rb#277
Net::IMAP::ResponseParser::Patterns::NZ_NUMBER = T.let(T.unsafe(nil), Regexp)

# Gmail allows SP and "]" in flags.......
#
# source://net-imap//lib/net/imap/response_parser.rb#238
Net::IMAP::ResponseParser::Patterns::QUIRKY_FLAG = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#239
Net::IMAP::ResponseParser::Patterns::QUIRKY_FLAGS_LIST = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#250
Net::IMAP::ResponseParser::Patterns::QUOTED_CHAR_esc = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#251
Net::IMAP::ResponseParser::Patterns::QUOTED_CHAR_rev1 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#252
Net::IMAP::ResponseParser::Patterns::QUOTED_CHAR_rev2 = T.let(T.unsafe(nil), Regexp)

# RFC3501:
#   QUOTED-CHAR   = <any TEXT-CHAR except quoted-specials> /
#                   "\" quoted-specials
# RFC9051:
#   QUOTED-CHAR   = <any TEXT-CHAR except quoted-specials> /
#                   "\" quoted-specials / UTF8-2 / UTF8-3 / UTF8-4
# RFC3501 & RFC9051:
#   quoted          = DQUOTE *QUOTED-CHAR DQUOTE
#
# source://net-imap//lib/net/imap/response_parser.rb#249
Net::IMAP::ResponseParser::Patterns::QUOTED_CHAR_safe = T.let(T.unsafe(nil), Regexp)

# quoted-specials = DQUOTE / "\"
#
# source://net-imap//lib/net/imap/response_parser.rb#159
Net::IMAP::ResponseParser::Patterns::QUOTED_SPECIALS = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#254
Net::IMAP::ResponseParser::Patterns::QUOTED_rev1 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#255
Net::IMAP::ResponseParser::Patterns::QUOTED_rev2 = T.let(T.unsafe(nil), Regexp)

# resp-specials   = "]"
#
# source://net-imap//lib/net/imap/response_parser.rb#161
Net::IMAP::ResponseParser::Patterns::RESP_SPECIALS = T.let(T.unsafe(nil), Regexp)

# UTF-8, a transformation format of ISO 10646
# >>>
#   UTF8-1      = %x00-7F
#   UTF8-tail   = %x80-BF
#   UTF8-2      = %xC2-DF UTF8-tail
#   UTF8-3      = %xE0 %xA0-BF UTF8-tail / %xE1-EC 2( UTF8-tail ) /
#                 %xED %x80-9F UTF8-tail / %xEE-EF 2( UTF8-tail )
#   UTF8-4      = %xF0 %x90-BF 2( UTF8-tail ) / %xF1-F3 3( UTF8-tail ) /
#                 %xF4 %x80-8F 2( UTF8-tail )
#   UTF8-char   = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4
#   UTF8-octets = *( UTF8-char )
#
# n.b. String * Integer is used for repetition, rather than /x{3}/,
# because ruby 3.2's linear-time cache-based optimization doesn't work
# with "bounded or fixed times repetition nesting in another repetition
# (e.g. /(a{2,3})*/). It is an implementation issue entirely, but we
# believe it is hard to support this case correctly."
# See https://bugs.ruby-lang.org/issues/19104
#
# source://net-imap//lib/net/imap/response_parser.rb#134
module Net::IMAP::ResponseParser::Patterns::RFC3629; end

# aka ASCII 7bit
#
# source://net-imap//lib/net/imap/response_parser.rb#135
Net::IMAP::ResponseParser::Patterns::RFC3629::UTF8_1 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#137
Net::IMAP::ResponseParser::Patterns::RFC3629::UTF8_2 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#138
Net::IMAP::ResponseParser::Patterns::RFC3629::UTF8_3 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#142
Net::IMAP::ResponseParser::Patterns::RFC3629::UTF8_4 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#145
Net::IMAP::ResponseParser::Patterns::RFC3629::UTF8_CHAR = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#146
Net::IMAP::ResponseParser::Patterns::RFC3629::UTF8_OCTETS = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#136
Net::IMAP::ResponseParser::Patterns::RFC3629::UTF8_TAIL = T.let(T.unsafe(nil), Regexp)

# From RFC5234, "Augmented BNF for Syntax Specifications: ABNF"
# >>>
#   ALPHA   =  %x41-5A / %x61-7A   ; A-Z / a-z
#   CHAR    = %x01-7F
#   CRLF    =  CR LF
#                   ; Internet standard newline
#   CTL     = %x00-1F / %x7F
#                ; controls
#   DIGIT   =  %x30-39
#                   ; 0-9
#   DQUOTE  =  %x22
#                   ; " (Double Quote)
#   HEXDIG  =  DIGIT / "A" / "B" / "C" / "D" / "E" / "F"
#   OCTET   = %x00-FF
#   SP      =  %x20
#
# source://net-imap//lib/net/imap/response_parser.rb#104
module Net::IMAP::ResponseParser::Patterns::RFC5234; end

# source://net-imap//lib/net/imap/response_parser.rb#105
Net::IMAP::ResponseParser::Patterns::RFC5234::ALPHA = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#106
Net::IMAP::ResponseParser::Patterns::RFC5234::CHAR = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#107
Net::IMAP::ResponseParser::Patterns::RFC5234::CRLF = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#108
Net::IMAP::ResponseParser::Patterns::RFC5234::CTL = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#109
Net::IMAP::ResponseParser::Patterns::RFC5234::DIGIT = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#110
Net::IMAP::ResponseParser::Patterns::RFC5234::DQUOTE = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#111
Net::IMAP::ResponseParser::Patterns::RFC5234::HEXDIG = T.let(T.unsafe(nil), Regexp)

# not using /./m for embedding purposes
#
# source://net-imap//lib/net/imap/response_parser.rb#112
Net::IMAP::ResponseParser::Patterns::RFC5234::OCTET = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#113
Net::IMAP::ResponseParser::Patterns::RFC5234::SP = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#321
Net::IMAP::ResponseParser::Patterns::SEQUENCE_SET = T.let(T.unsafe(nil), Regexp)

# sequence-set    = (seq-number / seq-range) ["," sequence-set]
#                     ; set of seq-number values, regardless of order.
#                     ; Servers MAY coalesce overlaps and/or execute
#                     ; the sequence in any order.
#                     ; Example: a message sequence number set of
#                     ; 2,4:7,9,12:* for a mailbox with 15 messages is
#                     ; equivalent to 2,4,5,6,7,9,12,13,14,15
#                     ; Example: a message sequence number set of
#                     ; *:4,5:7 for a mailbox with 10 messages is
#                     ; equivalent to 10,9,8,7,6,5,4,5,6,7 and MAY
#                     ; be reordered and overlap coalesced to be
#                     ; 4,5,6,7,8,9,10.
#
# source://net-imap//lib/net/imap/response_parser.rb#320
Net::IMAP::ResponseParser::Patterns::SEQUENCE_SET_ITEM = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#322
Net::IMAP::ResponseParser::Patterns::SEQUENCE_SET_STR = T.let(T.unsafe(nil), Regexp)

# seq-number      = nz-number / "*"
#                     ; message sequence number (COPY, FETCH, STORE
#                     ; commands) or unique identifier (UID COPY,
#                     ; UID FETCH, UID STORE commands).
#                     ; * represents the largest number in use.  In
#                     ; the case of message sequence numbers, it is
#                     ; the number of messages in a non-empty mailbox.
#                     ; In the case of unique identifiers, it is the
#                     ; unique identifier of the last message in the
#                     ; mailbox or, if the mailbox is empty, the
#                     ; mailbox's current UIDNEXT value.
#                     ; The server should respond with a tagged BAD
#                     ; response to a command that uses a message
#                     ; sequence number greater than the number of
#                     ; messages in the selected mailbox.  This
#                     ; includes "*" if the selected mailbox is empty.
#
# source://net-imap//lib/net/imap/response_parser.rb#295
Net::IMAP::ResponseParser::Patterns::SEQ_NUMBER = T.let(T.unsafe(nil), Regexp)

# seq-range       = seq-number ":" seq-number
#                     ; two seq-number values and all values between
#                     ; these two regardless of order.
#                     ; Example: 2:4 and 4:2 are equivalent and
#                     ; indicate values 2, 3, and 4.
#                     ; Example: a unique identifier sequence range of
#                     ; 3291:* includes the UID of the last message in
#                     ; the mailbox, even if that value is less than
#                     ; 3291.
#
# source://net-imap//lib/net/imap/response_parser.rb#306
Net::IMAP::ResponseParser::Patterns::SEQ_RANGE = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#183
Net::IMAP::ResponseParser::Patterns::TAG = T.let(T.unsafe(nil), Regexp)

# tagged-ext-label   = tagged-label-fchar *tagged-label-char
#                      ; Is a valid RFC 3501 "atom".
#
# source://net-imap//lib/net/imap/response_parser.rb#272
Net::IMAP::ResponseParser::Patterns::TAGGED_EXT_LABEL = T.let(T.unsafe(nil), Regexp)

# tagged-label-char  = tagged-label-fchar / DIGIT / ":"
#
# source://net-imap//lib/net/imap/response_parser.rb#269
Net::IMAP::ResponseParser::Patterns::TAGGED_LABEL_CHAR = T.let(T.unsafe(nil), Regexp)

# tagged-label-fchar = ALPHA / "-" / "_" / "."
#
# source://net-imap//lib/net/imap/response_parser.rb#267
Net::IMAP::ResponseParser::Patterns::TAGGED_LABEL_FCHAR = T.let(T.unsafe(nil), Regexp)

# TEXT-CHAR       = <any CHAR except CR and LF>
#
# source://net-imap//lib/net/imap/response_parser.rb#186
Net::IMAP::ResponseParser::Patterns::TEXT_CHAR = T.let(T.unsafe(nil), Regexp)

# RFC3501:
#   text          = 1*TEXT-CHAR
# RFC9051:
#   text          = 1*(TEXT-CHAR / UTF8-2 / UTF8-3 / UTF8-4)
#                     ; Non-ASCII text can only be returned
#                     ; after ENABLE IMAP4rev2 command
#
# source://net-imap//lib/net/imap/response_parser.rb#263
Net::IMAP::ResponseParser::Patterns::TEXT_rev1 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#264
Net::IMAP::ResponseParser::Patterns::TEXT_rev2 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/response_parser.rb#681
Net::IMAP::ResponseParser::RE_RESPONSE_TYPE = T.let(T.unsafe(nil), Regexp)

# end of response string
#
# source://net-imap//lib/net/imap/response_parser.rb#65
module Net::IMAP::ResponseParser::ResponseConditions; end

# source://net-imap//lib/net/imap/response_parser.rb#74
Net::IMAP::ResponseParser::ResponseConditions::AUTH_CONDS = T.let(T.unsafe(nil), Array)

# source://net-imap//lib/net/imap/response_parser.rb#68
Net::IMAP::ResponseParser::ResponseConditions::BAD = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/response_parser.rb#69
Net::IMAP::ResponseParser::ResponseConditions::BYE = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/response_parser.rb#75
Net::IMAP::ResponseParser::ResponseConditions::GREETING_CONDS = T.let(T.unsafe(nil), Array)

# source://net-imap//lib/net/imap/response_parser.rb#67
Net::IMAP::ResponseParser::ResponseConditions::NO = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/response_parser.rb#66
Net::IMAP::ResponseParser::ResponseConditions::OK = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/response_parser.rb#70
Net::IMAP::ResponseParser::ResponseConditions::PREAUTH = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/response_parser.rb#76
Net::IMAP::ResponseParser::ResponseConditions::RESP_CONDS = T.let(T.unsafe(nil), Array)

# source://net-imap//lib/net/imap/response_parser.rb#72
Net::IMAP::ResponseParser::ResponseConditions::RESP_COND_STATES = T.let(T.unsafe(nil), Array)

# source://net-imap//lib/net/imap/response_parser.rb#73
Net::IMAP::ResponseParser::ResponseConditions::RESP_DATA_CONDS = T.let(T.unsafe(nil), Array)

# source://net-imap//lib/net/imap/response_parser.rb#462
Net::IMAP::ResponseParser::SEQUENCE_SET_TOKENS = T.let(T.unsafe(nil), Array)

# source://net-imap//lib/net/imap/response_parser.rb#2022
Net::IMAP::ResponseParser::SPACES_REGEXP = T.let(T.unsafe(nil), Regexp)

# tag             = 1*<any ASTRING-CHAR except "+">
#
# source://net-imap//lib/net/imap/response_parser.rb#487
Net::IMAP::ResponseParser::TAG_TOKENS = T.let(T.unsafe(nil), Array)

# starts with atom special
#
# source://net-imap//lib/net/imap/response_parser.rb#60
Net::IMAP::ResponseParser::T_LITERAL8 = T.let(T.unsafe(nil), Symbol)

# Used to avoid an allocation when ResponseText is empty
#
# source://net-imap//lib/net/imap/response_data.rb#169
Net::IMAP::ResponseText::EMPTY = T.let(T.unsafe(nil), Net::IMAP::ResponseText)

# Pluggable authentication mechanisms for protocols which support SASL
# (Simple Authentication and Security Layer), such as IMAP4, SMTP, LDAP, and
# XMPP.  {RFC-4422}[https://tools.ietf.org/html/rfc4422] specifies the
# common \SASL framework:
# >>>
#   SASL is conceptually a framework that provides an abstraction layer
#   between protocols and mechanisms as illustrated in the following
#   diagram.
#
#               SMTP    LDAP    XMPP   Other protocols ...
#                  \       |    |      /
#                   \      |    |     /
#                  SASL abstraction layer
#                   /      |    |     \
#                  /       |    |      \
#           EXTERNAL   GSSAPI  PLAIN   Other mechanisms ...
#
# Net::IMAP uses SASL via the Net::IMAP#authenticate method.
#
# == Mechanisms
#
# Each mechanism has different properties and requirements.  Please consult
# the documentation for the specific mechanisms you are using:
#
# +ANONYMOUS+::
#     See AnonymousAuthenticator.
#
#     Allows the user to gain access to public services or resources without
#     authenticating or disclosing an identity.
#
# +EXTERNAL+::
#     See ExternalAuthenticator.
#
#     Authenticates using already established credentials, such as a TLS
#     certificate or IPSec.
#
# +OAUTHBEARER+::
#     See OAuthBearerAuthenticator.
#
#     Login using an OAuth2 Bearer token.  This is the standard mechanism
#     for using OAuth2 with \SASL, but it is not yet deployed as widely as
#     +XOAUTH2+.
#
# +PLAIN+::
#     See PlainAuthenticator.
#
#     Login using clear-text username and password.
#
# +SCRAM-SHA-1+::
# +SCRAM-SHA-256+::
#     See ScramAuthenticator.
#
#     Login by username and password.  The password is not sent to the
#     server but is used in a salted challenge/response exchange.
#     +SCRAM-SHA-1+ and +SCRAM-SHA-256+ are directly supported by
#     Net::IMAP::SASL.  New authenticators can easily be added for any other
#     <tt>SCRAM-*</tt> mechanism if the digest algorithm is supported by
#     OpenSSL::Digest.
#
# +XOAUTH2+::
#     See XOAuth2Authenticator.
#
#     Login using a username and an OAuth2 access token.  Non-standard and
#     obsoleted by +OAUTHBEARER+, but widely supported.
#
# See the {SASL mechanism
# registry}[https://www.iana.org/assignments/sasl-mechanisms/sasl-mechanisms.xhtml]
# for a list of all SASL mechanisms and their specifications.  To register
# new authenticators, see Authenticators.
#
# === Deprecated mechanisms
#
# <em>Obsolete mechanisms should be avoided, but are still available for
# backwards compatibility.</em>
#
# >>>
#   For +DIGEST-MD5+ see DigestMD5Authenticator.
#
#   For +LOGIN+, see LoginAuthenticator.
#
#   For +CRAM-MD5+, see CramMD5Authenticator.
#
# <em>Using a deprecated mechanism will print a warning.</em>
#
# source://net-imap//lib/net/imap/sasl.rb#90
module Net::IMAP::SASL
  private

  # See Net::IMAP::StringPrep::SASLprep#saslprep.
  #
  # source://net-imap//lib/net/imap/sasl.rb#173
  def saslprep(string, **opts); end

  class << self
    # Delegates to ::authenticators.  See Authenticators#add_authenticator.
    #
    # source://net-imap//lib/net/imap/sasl.rb#168
    def add_authenticator(*_arg0, **_arg1, &_arg2); end

    # Delegates to <tt>registry.new</tt>  See Authenticators#new.
    #
    # source://net-imap//lib/net/imap/sasl.rb#163
    def authenticator(*args, registry: T.unsafe(nil), **kwargs, &block); end

    # Returns the default global SASL::Authenticators instance.
    #
    # source://net-imap//lib/net/imap/sasl.rb#160
    def authenticators; end

    # See Net::IMAP::StringPrep::SASLprep#saslprep.
    #
    # source://net-imap//lib/net/imap/sasl.rb#173
    def saslprep(string, **opts); end
  end
end

# Authenticator for the "+ANONYMOUS+" SASL mechanism, as specified by
# RFC-4505[https://tools.ietf.org/html/rfc4505].  See
# Net::IMAP#authenticate.
#
# source://net-imap//lib/net/imap/sasl/anonymous_authenticator.rb#10
class Net::IMAP::SASL::AnonymousAuthenticator
  # :call-seq:
  #   new(anonymous_message = "", **) -> authenticator
  #   new(anonymous_message:  "", **) -> authenticator
  #
  # Creates an Authenticator for the "+ANONYMOUS+" SASL mechanism, as
  # specified in RFC-4505[https://tools.ietf.org/html/rfc4505].  To use
  # this, see Net::IMAP#authenticate or your client's authentication
  # method.
  #
  # ==== Parameters
  #
  # * _optional_ #anonymous_message — a message to send to the server.
  #
  # Any other keyword arguments are silently ignored.
  #
  # @return [AnonymousAuthenticator] a new instance of AnonymousAuthenticator
  #
  # source://net-imap//lib/net/imap/sasl/anonymous_authenticator.rb#37
  def initialize(anon_msg = T.unsafe(nil), anonymous_message: T.unsafe(nil), **_arg2); end

  # An optional token sent for the +ANONYMOUS+ mechanism., up to 255 UTF-8
  # characters in length.
  #
  # If it contains an "@" sign, the message must be a valid email address
  # (+addr-spec+ from RFC-2822[https://tools.ietf.org/html/rfc2822]).
  # Email syntax is _not_ validated by AnonymousAuthenticator.
  #
  # Otherwise, it can be any UTF8 string which is permitted by the
  # StringPrep::Trace profile.
  #
  # source://net-imap//lib/net/imap/sasl/anonymous_authenticator.rb#21
  def anonymous_message; end

  # Returns true when the initial client response was sent.
  #
  # The authentication should not succeed unless this returns true, but it
  # does *not* indicate success.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/anonymous_authenticator.rb#64
  def done?; end

  # :call-seq:
  #   initial_response? -> true
  #
  # +ANONYMOUS+ can send an initial client response.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/anonymous_authenticator.rb#51
  def initial_response?; end

  # Returns #anonymous_message.
  #
  # source://net-imap//lib/net/imap/sasl/anonymous_authenticator.rb#54
  def process(_server_challenge_string); end
end

# Indicates an authentication exchange that will be or has been canceled
# by the client, not due to any error or failure during processing.
#
# source://net-imap//lib/net/imap/sasl.rb#106
class Net::IMAP::SASL::AuthenticationCanceled < ::Net::IMAP::SASL::Error; end

# Indicates an error when processing a server challenge, e.g: an invalid
# or unparsable challenge.  An underlying exception may be available as
# the exception's #cause.
#
# source://net-imap//lib/net/imap/sasl.rb#111
class Net::IMAP::SASL::AuthenticationError < ::Net::IMAP::SASL::Error; end

# This API is *experimental*, and may change.
#
# TODO: catch exceptions in #process and send #cancel_response.
# TODO: raise an error if the command succeeds after being canceled.
# TODO: use with more clients, to verify the API can accommodate them.
#
# Create an AuthenticationExchange from a client adapter and a mechanism
# authenticator:
#     def authenticate(mechanism, ...)
#       authenticator = SASL.authenticator(mechanism, ...)
#       SASL::AuthenticationExchange.new(
#         sasl_adapter, mechanism, authenticator
#       ).authenticate
#     end
#
#     private
#
#     def sasl_adapter = MyClientAdapter.new(self, &method(:send_command))
#
# Or delegate creation of the authenticator to ::build:
#     def authenticate(...)
#       SASL::AuthenticationExchange.build(sasl_adapter, ...)
#         .authenticate
#     end
#
# As a convenience, ::authenticate combines ::build and #authenticate:
#     def authenticate(...)
#       SASL::AuthenticationExchange.authenticate(sasl_adapter, ...)
#     end
#
# Likewise, ClientAdapter#authenticate delegates to #authenticate:
#     def authenticate(...) = sasl_adapter.authenticate(...)
#
# source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#40
class Net::IMAP::SASL::AuthenticationExchange
  # @return [AuthenticationExchange] a new instance of AuthenticationExchange
  #
  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#52
  def initialize(client, mechanism, authenticator, sasl_ir: T.unsafe(nil)); end

  # Call #authenticate to execute an authentication exchange for #client
  # using #authenticator.  Authentication failures will raise an
  # exception.  Any exceptions other than those in RESPONSE_ERRORS will
  # drop the connection.
  #
  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#64
  def authenticate; end

  # Returns the value of attribute authenticator.
  #
  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#50
  def authenticator; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#85
  def done?; end

  # Returns the value of attribute mechanism.
  #
  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#50
  def mechanism; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#77
  def send_initial_response?; end

  private

  # Returns the value of attribute client.
  #
  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#91
  def client; end

  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#93
  def initial_response; end

  # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#98
  def process(challenge); end

  class << self
    # Convenience method for <tt>build(...).authenticate</tt>
    #
    # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#42
    def authenticate(*_arg0, **_arg1, &_arg2); end

    # Use +registry+ to override the global Authenticators registry.
    #
    # source://net-imap//lib/net/imap/sasl/authentication_exchange.rb#45
    def build(client, mechanism, *args, sasl_ir: T.unsafe(nil), **kwargs, &block); end
  end
end

# Indicates that authentication cannot proceed because one of the server's
# messages has not passed integrity checks.
#
# source://net-imap//lib/net/imap/sasl.rb#115
class Net::IMAP::SASL::AuthenticationFailed < ::Net::IMAP::SASL::Error; end

# Indicates that authentication cannot proceed because one of the server's
# ended authentication prematurely.
#
# source://net-imap//lib/net/imap/sasl.rb#119
class Net::IMAP::SASL::AuthenticationIncomplete < ::Net::IMAP::SASL::AuthenticationFailed
  # @return [AuthenticationIncomplete] a new instance of AuthenticationIncomplete
  #
  # source://net-imap//lib/net/imap/sasl.rb#123
  def initialize(response, message = T.unsafe(nil)); end

  # The success response from the server
  #
  # source://net-imap//lib/net/imap/sasl.rb#121
  def response; end
end

# Registry for SASL authenticators
#
# Registered authenticators must respond to +#new+ or +#call+ (e.g. a class or
# a proc), receiving any credentials and options and returning an
# authenticator instance. The returned object represents a single
# authentication exchange and <em>must not</em> be reused for multiple
# authentication attempts.
#
# An authenticator instance object must respond to +#process+, receiving the
# server's challenge and returning the client's response.  Optionally, it may
# also respond to +#initial_response?+ and +#done?+.  When
# +#initial_response?+ returns +true+, +#process+ may be called the first
# time with +nil+.  When +#done?+ returns +false+, the exchange is incomplete
# and an exception should be raised if the exchange terminates prematurely.
#
# See the source for PlainAuthenticator, XOAuth2Authenticator, and
# ScramSHA1Authenticator for examples.
#
# source://net-imap//lib/net/imap/sasl/authenticators.rb#22
class Net::IMAP::SASL::Authenticators
  # Create a new Authenticators registry.
  #
  # This class is usually not instantiated directly.  Use SASL.authenticators
  # to reuse the default global registry.
  #
  # When +use_defaults+ is +false+, the registry will start empty.  When
  # +use_deprecated+ is +false+, deprecated authenticators will not be
  # included with the defaults.
  #
  # @return [Authenticators] a new instance of Authenticators
  #
  # source://net-imap//lib/net/imap/sasl/authenticators.rb#32
  def initialize(use_defaults: T.unsafe(nil), use_deprecated: T.unsafe(nil)); end

  # :call-seq:
  #   add_authenticator(mechanism)
  #   add_authenticator(mechanism, authenticator_class)
  #   add_authenticator(mechanism, authenticator_proc)
  #
  # Registers an authenticator for #authenticator to use.  +mechanism+ is the
  # name of the
  # {SASL mechanism}[https://www.iana.org/assignments/sasl-mechanisms/sasl-mechanisms.xhtml]
  # implemented by +authenticator_class+ (for instance, <tt>"PLAIN"</tt>).
  #
  # If +mechanism+ refers to an existing authenticator,
  # the old authenticator will be replaced.
  #
  # When only a single argument is given, the authenticator class will be
  # lazily loaded from <tt>Net::IMAP::SASL::#{name}Authenticator</tt> (case is
  # preserved and non-alphanumeric characters are removed..
  #
  # source://net-imap//lib/net/imap/sasl/authenticators.rb#67
  def add_authenticator(name, authenticator = T.unsafe(nil)); end

  # :call-seq:
  #   authenticator(mechanism, ...) -> auth_session
  #
  # Builds an authenticator instance using the authenticator registered to
  # +mechanism+.  The returned object represents a single authentication
  # exchange and <em>must not</em> be reused for multiple authentication
  # attempts.
  #
  # All arguments (except +mechanism+) are forwarded to the registered
  # authenticator's +#new+ or +#call+ method.  Each authenticator must
  # document its own arguments.
  #
  # [Note]
  #   This method is intended for internal use by connection protocol code
  #   only.  Protocol client users should see refer to their client's
  #   documentation, e.g. Net::IMAP#authenticate.
  #
  # source://net-imap//lib/net/imap/sasl/authenticators.rb#107
  def authenticator(mechanism, *_arg1, **_arg2, &_arg3); end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/authenticators.rb#86
  def mechanism?(name); end

  # Returns the names of all registered SASL mechanisms.
  #
  # source://net-imap//lib/net/imap/sasl/authenticators.rb#49
  def names; end

  # :call-seq:
  #   authenticator(mechanism, ...) -> auth_session
  #
  # Builds an authenticator instance using the authenticator registered to
  # +mechanism+.  The returned object represents a single authentication
  # exchange and <em>must not</em> be reused for multiple authentication
  # attempts.
  #
  # All arguments (except +mechanism+) are forwarded to the registered
  # authenticator's +#new+ or +#call+ method.  Each authenticator must
  # document its own arguments.
  #
  # [Note]
  #   This method is intended for internal use by connection protocol code
  #   only.  Protocol client users should see refer to their client's
  #   documentation, e.g. Net::IMAP#authenticate.
  #
  # source://net-imap//lib/net/imap/sasl/authenticators.rb#107
  def new(mechanism, *_arg1, **_arg2, &_arg3); end

  # Removes the authenticator registered for +name+
  #
  # source://net-imap//lib/net/imap/sasl/authenticators.rb#81
  def remove_authenticator(name); end
end

# source://net-imap//lib/net/imap/sasl/stringprep.rb#8
Net::IMAP::SASL::BidiStringError = Net::IMAP::StringPrep::BidiStringError

# This API is *experimental*, and may change.
#
# TODO: use with more clients, to verify the API can accommodate them.
#
# An abstract base class for implementing a SASL authentication exchange.
# Different clients will each have their own adapter subclass, overridden
# to match their needs.
#
# Although the default implementations _may_ be sufficient, subclasses
# will probably need to override some methods.  Additionally, subclasses
# may need to include a protocol adapter mixin, if the default
# ProtocolAdapters::Generic isn't sufficient.
#
# source://net-imap//lib/net/imap/sasl/client_adapter.rb#19
class Net::IMAP::SASL::ClientAdapter
  include ::Net::IMAP::SASL::ProtocolAdapters::Generic

  # +command_proc+ can used to avoid exposing private methods on #client.
  # It should run a command with the arguments sent to it, yield each
  # continuation payload, respond to the server with the result of each
  # yield, and return the result.  Non-successful results *MUST* raise an
  # exception.  Exceptions in the block *MUST* cause the command to fail.
  #
  # Subclasses that override #run_command may use #command_proc for
  # other purposes.
  #
  # @return [ClientAdapter] a new instance of ClientAdapter
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#32
  def initialize(client, &command_proc); end

  # Does the server advertise support for the mechanism?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#43
  def auth_capable?(mechanism); end

  # Delegates to AuthenticationExchange.authenticate.
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#37
  def authenticate(*_arg0, **_arg1, &_arg2); end

  # Returns the value of attribute client.
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#22
  def client; end

  # Returns the value of attribute command_proc.
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#22
  def command_proc; end

  # Drop the connection gracefully.
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#65
  def drop_connection; end

  # Drop the connection abruptly.
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#68
  def drop_connection!; end

  # Returns an array of server responses errors raised by run_command.
  # Exceptions in this array won't drop the connection.
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#62
  def response_errors; end

  # Runs the authenticate command with +mechanism+ and +initial_response+.
  # When +initial_response+ is nil, an initial response must NOT be sent.
  #
  # Yields each continuation payload, responds to the server with the
  # result of each yield, and returns the result.  Non-successful results
  # *MUST* raise an exception.  Exceptions in the block *MUST* cause the
  # command to fail.
  #
  # Subclasses that override this may use #command_proc differently.
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#54
  def run_command(mechanism, initial_response = T.unsafe(nil), &block); end

  # Do the protocol and server both support an initial response?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/client_adapter.rb#40
  def sasl_ir_capable?; end
end

# Authenticator for the "+CRAM-MD5+" SASL mechanism, specified in
# RFC2195[https://tools.ietf.org/html/rfc2195].  See Net::IMAP#authenticate.
#
# == Deprecated
#
# +CRAM-MD5+ is obsolete and insecure.  It is included for compatibility with
# existing servers.
# {draft-ietf-sasl-crammd5-to-historic}[https://tools.ietf.org/html/draft-ietf-sasl-crammd5-to-historic-00.html]
# recommends using +SCRAM-*+ or +PLAIN+ protected by TLS instead.
#
# Additionally, RFC8314[https://tools.ietf.org/html/rfc8314] discourage the use
# of cleartext and recommends TLS version 1.2 or greater be used for all
# traffic.  With TLS +CRAM-MD5+ is okay, but so is +PLAIN+
#
# source://net-imap//lib/net/imap/sasl/cram_md5_authenticator.rb#16
class Net::IMAP::SASL::CramMD5Authenticator
  # @return [CramMD5Authenticator] a new instance of CramMD5Authenticator
  #
  # source://net-imap//lib/net/imap/sasl/cram_md5_authenticator.rb#17
  def initialize(user = T.unsafe(nil), pass = T.unsafe(nil), authcid: T.unsafe(nil), username: T.unsafe(nil), password: T.unsafe(nil), secret: T.unsafe(nil), warn_deprecation: T.unsafe(nil), **_arg7); end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/cram_md5_authenticator.rb#40
  def done?; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/cram_md5_authenticator.rb#31
  def initial_response?; end

  # source://net-imap//lib/net/imap/sasl/cram_md5_authenticator.rb#33
  def process(challenge); end

  private

  # source://net-imap//lib/net/imap/sasl/cram_md5_authenticator.rb#44
  def hmac_md5(text, key); end
end

# Net::IMAP authenticator for the "`DIGEST-MD5`" SASL mechanism type, specified
# in RFC-2831[https://tools.ietf.org/html/rfc2831].  See Net::IMAP#authenticate.
#
# == Deprecated
#
# "+DIGEST-MD5+" has been deprecated by
# RFC-6331[https://tools.ietf.org/html/rfc6331] and should not be relied on for
# security.  It is included for compatibility with existing servers.
#
# source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#11
class Net::IMAP::SASL::DigestMD5Authenticator
  # :call-seq:
  #   new(username,  password,  authzid = nil, **options) -> authenticator
  #   new(username:, password:, authzid:  nil, **options) -> authenticator
  #   new(authcid:,  password:, authzid:  nil, **options) -> authenticator
  #
  # Creates an Authenticator for the "+DIGEST-MD5+" SASL mechanism.
  #
  # Called by Net::IMAP#authenticate and similar methods on other clients.
  #
  # ==== Parameters
  #
  # * #authcid  ― Authentication identity that is associated with #password.
  #
  #   #username ― An alias for +authcid+.
  #
  # * #password ― A password or passphrase associated with this #authcid.
  #
  # * _optional_ #authzid  ― Authorization identity to act as or on behalf of.
  #
  #   When +authzid+ is not set, the server should derive the authorization
  #   identity from the authentication identity.
  #
  # * _optional_ +warn_deprecation+ — Set to +false+ to silence the warning.
  #
  # Any other keyword arguments are silently ignored.
  #
  # @return [DigestMD5Authenticator] a new instance of DigestMD5Authenticator
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#70
  def initialize(user = T.unsafe(nil), pass = T.unsafe(nil), authz = T.unsafe(nil), username: T.unsafe(nil), password: T.unsafe(nil), authzid: T.unsafe(nil), authcid: T.unsafe(nil), secret: T.unsafe(nil), warn_deprecation: T.unsafe(nil), **_arg9); end

  # Authentication identity: the identity that matches the #password.
  #
  # RFC-2831[https://tools.ietf.org/html/rfc2831] uses the term +username+.
  # "Authentication identity" is the generic term used by
  # RFC-4422[https://tools.ietf.org/html/rfc4422].
  # RFC-4616[https://tools.ietf.org/html/rfc4616] and many later RFCs abbreviate
  # this to +authcid+.
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#24
  def authcid; end

  # Authorization identity: an identity to act as or on behalf of.  The identity
  # form is application protocol specific.  If not provided or left blank, the
  # server derives an authorization identity from the authentication identity.
  # The server is responsible for verifying the client's credentials and
  # verifying that the identity it associates with the client's authentication
  # identity is allowed to act as (or on behalf of) the authorization identity.
  #
  # For example, an administrator or superuser might take on another role:
  #
  #     imap.authenticate "DIGEST-MD5", "root", ->{passwd}, authzid: "user"
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#43
  def authzid; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#156
  def done?; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#88
  def initial_response?; end

  # A password or passphrase that matches the #username.
  #
  # The +password+ will be used to create the response digest.
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#30
  def password; end

  # Responds to server challenge in two stages.
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#91
  def process(challenge); end

  # Authentication identity: the identity that matches the #password.
  #
  # RFC-2831[https://tools.ietf.org/html/rfc2831] uses the term +username+.
  # "Authentication identity" is the generic term used by
  # RFC-4422[https://tools.ietf.org/html/rfc4422].
  # RFC-4616[https://tools.ietf.org/html/rfc4616] and many later RFCs abbreviate
  # this to +authcid+.
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#24
  def username; end

  private

  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#160
  def nc(nonce); end

  # some responses need quoting
  #
  # source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#170
  def qdval(k, v); end
end

# source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#14
Net::IMAP::SASL::DigestMD5Authenticator::STAGE_DONE = T.let(T.unsafe(nil), Symbol)

# source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#12
Net::IMAP::SASL::DigestMD5Authenticator::STAGE_ONE = T.let(T.unsafe(nil), Symbol)

# source://net-imap//lib/net/imap/sasl/digest_md5_authenticator.rb#13
Net::IMAP::SASL::DigestMD5Authenticator::STAGE_TWO = T.let(T.unsafe(nil), Symbol)

# Exception class for any client error detected during the authentication
# exchange.
#
# When the _server_ reports an authentication failure, it will respond
# with a protocol specific error instead, e.g: +BAD+ or +NO+ in IMAP.
#
# When the client encounters any error, it *must* consider the
# authentication exchange to be unsuccessful and it might need to drop the
# connection.  For example, if the server reports that the authentication
# exchange was successful or the protocol does not allow additional
# authentication attempts.
#
# source://net-imap//lib/net/imap/sasl.rb#102
class Net::IMAP::SASL::Error < ::StandardError; end

# Authenticator for the "+EXTERNAL+" SASL mechanism, as specified by
# RFC-4422[https://tools.ietf.org/html/rfc4422].  See
# Net::IMAP#authenticate.
#
# The EXTERNAL mechanism requests that the server use client credentials
# established external to SASL, for example by TLS certificate or IPSec.
#
# source://net-imap//lib/net/imap/sasl/external_authenticator.rb#13
class Net::IMAP::SASL::ExternalAuthenticator
  # :call-seq:
  #   new(authzid: nil, **) -> authenticator
  #   new(username: nil, **) -> authenticator
  #   new(username = nil, **) -> authenticator
  #
  # Creates an Authenticator for the "+EXTERNAL+" SASL mechanism, as
  # specified in RFC-4422[https://tools.ietf.org/html/rfc4422].  To use
  # this, see Net::IMAP#authenticate or your client's authentication
  # method.
  #
  # ==== Parameters
  #
  # * _optional_ #authzid  ― Authorization identity to act as or on behalf of.
  #
  #   _optional_ #username ― An alias for #authzid.
  #
  #   Note that, unlike some other authenticators, +username+ sets the
  #   _authorization_ identity and not the _authentication_ identity.  The
  #   authentication identity is established for the client by the
  #   external credentials.
  #
  # Any other keyword parameters are quietly ignored.
  #
  # @return [ExternalAuthenticator] a new instance of ExternalAuthenticator
  #
  # source://net-imap//lib/net/imap/sasl/external_authenticator.rb#52
  def initialize(user = T.unsafe(nil), authzid: T.unsafe(nil), username: T.unsafe(nil), **_arg3); end

  # Authorization identity: an identity to act as or on behalf of.  The
  # identity form is application protocol specific.  If not provided or
  # left blank, the server derives an authorization identity from the
  # authentication identity.  The server is responsible for verifying the
  # client's credentials and verifying that the identity it associates
  # with the client's authentication identity is allowed to act as (or on
  # behalf of) the authorization identity.
  #
  # For example, an administrator or superuser might take on another role:
  #
  #     imap.authenticate "PLAIN", "root", passwd, authzid: "user"
  #
  # source://net-imap//lib/net/imap/sasl/external_authenticator.rb#27
  def authzid; end

  # Returns true when the initial client response was sent.
  #
  # The authentication should not succeed unless this returns true, but it
  # does *not* indicate success.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/external_authenticator.rb#78
  def done?; end

  # :call-seq:
  #   initial_response? -> true
  #
  # +EXTERNAL+ can send an initial client response.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/external_authenticator.rb#65
  def initial_response?; end

  # Returns #authzid, or an empty string if there is no authzid.
  #
  # source://net-imap//lib/net/imap/sasl/external_authenticator.rb#68
  def process(_); end

  # Authorization identity: an identity to act as or on behalf of.  The
  # identity form is application protocol specific.  If not provided or
  # left blank, the server derives an authorization identity from the
  # authentication identity.  The server is responsible for verifying the
  # client's credentials and verifying that the identity it associates
  # with the client's authentication identity is allowed to act as (or on
  # behalf of) the authorization identity.
  #
  # For example, an administrator or superuser might take on another role:
  #
  #     imap.authenticate "PLAIN", "root", passwd, authzid: "user"
  #
  # source://net-imap//lib/net/imap/sasl/external_authenticator.rb#27
  def username; end
end

# Originally defined for the GS2 mechanism family in
# RFC5801[https://tools.ietf.org/html/rfc5801],
# several different mechanisms start with a GS2 header:
# * +GS2-*+       --- RFC5801[https://tools.ietf.org/html/rfc5801]
# * +SCRAM-*+     --- RFC5802[https://tools.ietf.org/html/rfc5802]
#   (ScramAuthenticator)
# * +SAML20+      --- RFC6595[https://tools.ietf.org/html/rfc6595]
# * +OPENID20+    --- RFC6616[https://tools.ietf.org/html/rfc6616]
# * +OAUTH10A+    --- RFC7628[https://tools.ietf.org/html/rfc7628]
# * +OAUTHBEARER+ --- RFC7628[https://tools.ietf.org/html/rfc7628]
#   (OAuthBearerAuthenticator)
#
# Classes that include this module must implement +#authzid+.
#
# source://net-imap//lib/net/imap/sasl/gs2_header.rb#20
module Net::IMAP::SASL::GS2Header
  # The {RFC5801 §4}[https://www.rfc-editor.org/rfc/rfc5801#section-4]
  # +gs2-authzid+ header, when +#authzid+ is not empty.
  #
  # If +#authzid+ is empty or +nil+, an empty string is returned.
  #
  # source://net-imap//lib/net/imap/sasl/gs2_header.rb#59
  def gs2_authzid; end

  # The {RFC5801 §4}[https://www.rfc-editor.org/rfc/rfc5801#section-4]
  # +gs2-cb-flag+:
  #
  # "+n+":: The client doesn't support channel binding.
  # "+y+":: The client does support channel binding
  #         but thinks the server does not.
  # "+p+":: The client requires channel binding.
  #         The selected channel binding follows "+p=+".
  #
  # The default always returns "+n+".  A mechanism that supports channel
  # binding must override this method.
  #
  # source://net-imap//lib/net/imap/sasl/gs2_header.rb#53
  def gs2_cb_flag; end

  # The {RFC5801 §4}[https://www.rfc-editor.org/rfc/rfc5801#section-4]
  # +gs2-header+, which prefixes the #initial_client_response.
  #
  # >>>
  #   <em>Note: the actual GS2 header includes an optional flag to
  #   indicate that the GSS mechanism is not "standard", but since all of
  #   the SASL mechanisms using GS2 are "standard", we don't include that
  #   flag.  A class for a nonstandard GSSAPI mechanism should prefix with
  #   "+F,+".</em>
  #
  # source://net-imap//lib/net/imap/sasl/gs2_header.rb#37
  def gs2_header; end

  private

  # Encodes +str+ to match RFC5801_SASLNAME.
  #
  # source://net-imap//lib/net/imap/sasl/gs2_header.rb#67
  def gs2_saslname_encode(str); end

  class << self
    # Encodes +str+ to match RFC5801_SASLNAME.
    #
    # source://net-imap//lib/net/imap/sasl/gs2_header.rb#67
    def gs2_saslname_encode(str); end
  end
end

# source://net-imap//lib/net/imap/sasl/gs2_header.rb#21
Net::IMAP::SASL::GS2Header::NO_NULL_CHARS = T.let(T.unsafe(nil), Regexp)

# Matches {RFC5801 §4}[https://www.rfc-editor.org/rfc/rfc5801#section-4]
# +saslname+.  The output from gs2_saslname_encode matches this Regexp.
#
# source://net-imap//lib/net/imap/sasl/gs2_header.rb#26
Net::IMAP::SASL::GS2Header::RFC5801_SASLNAME = T.let(T.unsafe(nil), Regexp)

# Authenticator for the "+LOGIN+" SASL mechanism.  See Net::IMAP#authenticate.
#
# +LOGIN+ authentication sends the password in cleartext.
# RFC3501[https://tools.ietf.org/html/rfc3501] encourages servers to disable
# cleartext authentication until after TLS has been negotiated.
# RFC8314[https://tools.ietf.org/html/rfc8314] recommends TLS version 1.2 or
# greater be used for all traffic, and deprecate cleartext access ASAP.  +LOGIN+
# can be secured by TLS encryption.
#
# == Deprecated
#
# The {SASL mechanisms
# registry}[https://www.iana.org/assignments/sasl-mechanisms/sasl-mechanisms.xhtml]
# marks "LOGIN" as obsoleted in favor of "PLAIN".  It is included here for
# compatibility with existing servers.  See
# {draft-murchison-sasl-login}[https://www.iana.org/go/draft-murchison-sasl-login]
# for both specification and deprecation.
#
# source://net-imap//lib/net/imap/sasl/login_authenticator.rb#20
class Net::IMAP::SASL::LoginAuthenticator
  # @return [LoginAuthenticator] a new instance of LoginAuthenticator
  #
  # source://net-imap//lib/net/imap/sasl/login_authenticator.rb#26
  def initialize(user = T.unsafe(nil), pass = T.unsafe(nil), authcid: T.unsafe(nil), username: T.unsafe(nil), password: T.unsafe(nil), secret: T.unsafe(nil), warn_deprecation: T.unsafe(nil), **_arg7); end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/login_authenticator.rb#54
  def done?; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/login_authenticator.rb#39
  def initial_response?; end

  # source://net-imap//lib/net/imap/sasl/login_authenticator.rb#41
  def process(data); end
end

# source://net-imap//lib/net/imap/sasl/login_authenticator.rb#23
Net::IMAP::SASL::LoginAuthenticator::STATE_DONE = T.let(T.unsafe(nil), Symbol)

# source://net-imap//lib/net/imap/sasl/login_authenticator.rb#22
Net::IMAP::SASL::LoginAuthenticator::STATE_PASSWORD = T.let(T.unsafe(nil), Symbol)

# source://net-imap//lib/net/imap/sasl/login_authenticator.rb#21
Net::IMAP::SASL::LoginAuthenticator::STATE_USER = T.let(T.unsafe(nil), Symbol)

# Abstract base class for the SASL mechanisms defined in
# RFC7628[https://tools.ietf.org/html/rfc7628]:
# * OAUTHBEARER[rdoc-ref:OAuthBearerAuthenticator]
#   (OAuthBearerAuthenticator)
# * OAUTH10A
#
# source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#14
class Net::IMAP::SASL::OAuthAuthenticator
  include ::Net::IMAP::SASL::GS2Header

  # Creates an RFC7628[https://tools.ietf.org/html/rfc7628] OAuth
  # authenticator.
  #
  # ==== Parameters
  #
  # See child classes for required parameter(s).  The following parameters
  # are all optional, but it is worth noting that <b>application protocols
  # are allowed to require</b> #authzid (or other parameters, such as
  # #host or #port) <b>as are specific server implementations</b>.
  #
  # * _optional_ #authzid  ― Authorization identity to act as or on behalf of.
  #
  #   _optional_ #username — An alias for #authzid.
  #
  #   Note that, unlike some other authenticators, +username+ sets the
  #   _authorization_ identity and not the _authentication_ identity.  The
  #   authentication identity is established for the client by the OAuth
  #   token.
  #
  # * _optional_ #host — Hostname to which the client connected.
  # * _optional_ #port — Service port to which the client connected.
  # * _optional_ #mthd — HTTP method
  # * _optional_ #path — HTTP path data
  # * _optional_ #post — HTTP post data
  # * _optional_ #qs   — HTTP query string
  #
  #   _optional_ #query — An alias for #qs
  #
  # Any other keyword parameters are quietly ignored.
  #
  # @return [OAuthAuthenticator] a new instance of OAuthAuthenticator
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#84
  def initialize(authzid: T.unsafe(nil), host: T.unsafe(nil), port: T.unsafe(nil), username: T.unsafe(nil), query: T.unsafe(nil), mthd: T.unsafe(nil), path: T.unsafe(nil), post: T.unsafe(nil), qs: T.unsafe(nil), **_arg9); end

  # Value of the HTTP Authorization header
  #
  # <b>Implemented by subclasses.</b>
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#124
  def authorization; end

  # Authorization identity: an identity to act as or on behalf of.  The
  # identity form is application protocol specific.  If not provided or
  # left blank, the server derives an authorization identity from the
  # authentication identity.  The server is responsible for verifying the
  # client's credentials and verifying that the identity it associates
  # with the client's authentication identity is allowed to act as (or on
  # behalf of) the authorization identity.
  #
  # For example, an administrator or superuser might take on another role:
  #
  #     imap.authenticate "PLAIN", "root", passwd, authzid: "user"
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#29
  def authzid; end

  # Returns true when the initial client response was sent.
  #
  # The authentication should not succeed unless this returns true, but it
  # does *not* indicate success.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#119
  def done?; end

  # Hostname to which the client connected.  (optional)
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#33
  def host; end

  # The {RFC7628 §3.1}[https://www.rfc-editor.org/rfc/rfc7628#section-3.1]
  # formatted response.
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#99
  def initial_client_response; end

  # Stores the most recent server "challenge".  When authentication fails,
  # this may hold information about the failure reason, as JSON.
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#53
  def last_server_response; end

  # HTTP method.  (optional)
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#39
  def mthd; end

  # HTTP path data.  (optional)
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#42
  def path; end

  # Service port to which the client connected.  (optional)
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#36
  def port; end

  # HTTP post data.  (optional)
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#45
  def post; end

  # Returns initial_client_response the first time, then "<tt>^A</tt>".
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#108
  def process(data); end

  # The query string.  (optional)
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#48
  def qs; end

  # The query string.  (optional)
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#48
  def query; end

  # Authorization identity: an identity to act as or on behalf of.  The
  # identity form is application protocol specific.  If not provided or
  # left blank, the server derives an authorization identity from the
  # authentication identity.  The server is responsible for verifying the
  # client's credentials and verifying that the identity it associates
  # with the client's authentication identity is allowed to act as (or on
  # behalf of) the authorization identity.
  #
  # For example, an administrator or superuser might take on another role:
  #
  #     imap.authenticate "PLAIN", "root", passwd, authzid: "user"
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#29
  def username; end
end

# Authenticator for the "+OAUTHBEARER+" SASL mechanism, specified in
# RFC7628[https://tools.ietf.org/html/rfc7628].  Authenticates using OAuth
# 2.0 bearer tokens, as described in
# RFC6750[https://tools.ietf.org/html/rfc6750].  Use via
# Net::IMAP#authenticate.
#
# RFC6750[https://tools.ietf.org/html/rfc6750] requires Transport Layer
# Security (TLS) to secure the protocol interaction between the client and
# the resource server.  TLS _MUST_ be used for +OAUTHBEARER+ to protect
# the bearer token.
#
# source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#138
class Net::IMAP::SASL::OAuthBearerAuthenticator < ::Net::IMAP::SASL::OAuthAuthenticator
  # :call-seq:
  #   new(oauth2_token,          **options) -> authenticator
  #   new(authzid, oauth2_token, **options) -> authenticator
  #   new(oauth2_token:,         **options) -> authenticator
  #
  # Creates an Authenticator for the "+OAUTHBEARER+" SASL mechanism.
  #
  # Called by Net::IMAP#authenticate and similar methods on other clients.
  #
  # ==== Parameters
  #
  # * #oauth2_token — An OAuth2 bearer token
  #
  # All other keyword parameters are passed to
  # {super}[rdoc-ref:OAuthAuthenticator::new] (see OAuthAuthenticator).
  # The most common ones are:
  #
  # * _optional_ #authzid  ― Authorization identity to act as or on behalf of.
  #
  #   _optional_ #username — An alias for #authzid.
  #
  #   Note that, unlike some other authenticators, +username+ sets the
  #   _authorization_ identity and not the _authentication_ identity.  The
  #   authentication identity is established for the client by
  #   #oauth2_token.
  #
  # * _optional_ #host — Hostname to which the client connected.
  # * _optional_ #port — Service port to which the client connected.
  #
  # Although only oauth2_token is required by this mechanism, it is worth
  # noting that <b><em>application protocols are allowed to
  # require</em></b> #authzid (<em>or other parameters, such as</em> #host
  # _or_ #port) <b><em>as are specific server implementations</em></b>.
  #
  # @return [OAuthBearerAuthenticator] a new instance of OAuthBearerAuthenticator
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#177
  def initialize(arg1 = T.unsafe(nil), arg2 = T.unsafe(nil), oauth2_token: T.unsafe(nil), secret: T.unsafe(nil), **args, &blk); end

  # Value of the HTTP Authorization header
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#193
  def authorization; end

  # :call-seq:
  #   initial_response? -> true
  #
  # +OAUTHBEARER+ sends an initial client response.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#190
  def initial_response?; end

  # An OAuth 2.0 bearer token.  See {RFC-6750}[https://www.rfc-editor.org/rfc/rfc6750]
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#141
  def oauth2_token; end

  # An OAuth 2.0 bearer token.  See {RFC-6750}[https://www.rfc-editor.org/rfc/rfc6750]
  #
  # source://net-imap//lib/net/imap/sasl/oauthbearer_authenticator.rb#141
  def secret; end
end

# Authenticator for the "+PLAIN+" SASL mechanism, specified in
# RFC-4616[https://tools.ietf.org/html/rfc4616].  See Net::IMAP#authenticate.
#
# +PLAIN+ authentication sends the password in cleartext.
# RFC-3501[https://tools.ietf.org/html/rfc3501] encourages servers to disable
# cleartext authentication until after TLS has been negotiated.
# RFC-8314[https://tools.ietf.org/html/rfc8314] recommends TLS version 1.2 or
# greater be used for all traffic, and deprecate cleartext access ASAP.  +PLAIN+
# can be secured by TLS encryption.
#
# source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#12
class Net::IMAP::SASL::PlainAuthenticator
  # :call-seq:
  #   new(username,  password,  authzid: nil, **) -> authenticator
  #   new(username:, password:, authzid: nil, **) -> authenticator
  #   new(authcid:,  password:, authzid: nil, **) -> authenticator
  #
  # Creates an Authenticator for the "+PLAIN+" SASL mechanism.
  #
  # Called by Net::IMAP#authenticate and similar methods on other clients.
  #
  # ==== Parameters
  #
  # * #authcid ― Authentication identity that is associated with #password.
  #
  #   #username ― An alias for #authcid.
  #
  # * #password ― A password or passphrase associated with the #authcid.
  #
  # * _optional_ #authzid  ― Authorization identity to act as or on behalf of.
  #
  #   When +authzid+ is not set, the server should derive the authorization
  #   identity from the authentication identity.
  #
  # Any other keyword parameters are quietly ignored.
  #
  # @raise [ArgumentError]
  # @return [PlainAuthenticator] a new instance of PlainAuthenticator
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#67
  def initialize(user = T.unsafe(nil), pass = T.unsafe(nil), authcid: T.unsafe(nil), secret: T.unsafe(nil), username: T.unsafe(nil), password: T.unsafe(nil), authzid: T.unsafe(nil), **_arg7); end

  # Authentication identity: the identity that matches the #password.
  #
  # RFC-2831[https://tools.ietf.org/html/rfc2831] uses the term +username+.
  # "Authentication identity" is the generic term used by
  # RFC-4422[https://tools.ietf.org/html/rfc4422].
  # RFC-4616[https://tools.ietf.org/html/rfc4616] and many later RFCs abbreviate
  # this to +authcid+.
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#24
  def authcid; end

  # Authorization identity: an identity to act as or on behalf of.  The identity
  # form is application protocol specific.  If not provided or left blank, the
  # server derives an authorization identity from the authentication identity.
  # The server is responsible for verifying the client's credentials and
  # verifying that the identity it associates with the client's authentication
  # identity is allowed to act as (or on behalf of) the authorization identity.
  #
  # For example, an administrator or superuser might take on another role:
  #
  #     imap.authenticate "PLAIN", "root", passwd, authzid: "user"
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#42
  def authzid; end

  # Returns true when the initial client response was sent.
  #
  # The authentication should not succeed unless this returns true, but it
  # does *not* indicate success.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#99
  def done?; end

  # :call-seq:
  #   initial_response? -> true
  #
  # +PLAIN+ can send an initial client response.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#86
  def initial_response?; end

  # A password or passphrase that matches the #username.
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#28
  def password; end

  # Responds with the client's credentials.
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#89
  def process(data); end

  # A password or passphrase that matches the #username.
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#28
  def secret; end

  # Authentication identity: the identity that matches the #password.
  #
  # RFC-2831[https://tools.ietf.org/html/rfc2831] uses the term +username+.
  # "Authentication identity" is the generic term used by
  # RFC-4422[https://tools.ietf.org/html/rfc4422].
  # RFC-4616[https://tools.ietf.org/html/rfc4616] and many later RFCs abbreviate
  # this to +authcid+.
  #
  # source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#24
  def username; end
end

# source://net-imap//lib/net/imap/sasl/plain_authenticator.rb#14
Net::IMAP::SASL::PlainAuthenticator::NULL = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/sasl/stringprep.rb#9
Net::IMAP::SASL::ProhibitedCodepoint = Net::IMAP::StringPrep::ProhibitedCodepoint

# source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#7
module Net::IMAP::SASL::ProtocolAdapters; end

# This API is experimental, and may change.
#
# source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#9
module Net::IMAP::SASL::ProtocolAdapters::Generic
  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#17
  def cancel_response; end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#10
  def command_name; end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#16
  def decode(string); end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#15
  def encode(string); end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#14
  def encode_ir(string); end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#12
  def host; end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#13
  def port; end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#11
  def service; end
end

# See RFC-3501 (IMAP4rev1), RFC-4959 (SASL-IR capability),
# and RFC-9051 (IMAP4rev2).
#
# source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#22
module Net::IMAP::SASL::ProtocolAdapters::IMAP
  include ::Net::IMAP::SASL::ProtocolAdapters::Generic

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#24
  def service; end
end

# See RFC-5034 (SASL capability).
#
# source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#35
module Net::IMAP::SASL::ProtocolAdapters::POP
  include ::Net::IMAP::SASL::ProtocolAdapters::Generic

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#37
  def command_name; end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#38
  def service; end
end

# See RFC-4954 (AUTH capability).
#
# source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#28
module Net::IMAP::SASL::ProtocolAdapters::SMTP
  include ::Net::IMAP::SASL::ProtocolAdapters::Generic

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#30
  def command_name; end

  # source://net-imap//lib/net/imap/sasl/protocol_adapters.rb#31
  def service; end
end

# Alias for Net::IMAP::StringPrep::SASLprep.
#
# source://net-imap//lib/net/imap/sasl/stringprep.rb#6
Net::IMAP::SASL::SASLprep = Net::IMAP::StringPrep::SASLprep

# For method descriptions,
# see {RFC5802 §2.2}[https://www.rfc-editor.org/rfc/rfc5802#section-2.2]
# and {RFC5802 §3}[https://www.rfc-editor.org/rfc/rfc5802#section-3].
#
# source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#10
module Net::IMAP::SASL::ScramAlgorithm
  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#24
  def H(str); end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#26
  def HMAC(key, data); end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#13
  def Hi(str, salt, iterations); end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#11
  def Normalize(str); end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#28
  def XOR(str1, str2); end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#35
  def auth_message; end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#48
  def client_key; end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#53
  def client_proof; end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#51
  def client_signature; end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#44
  def salted_password; end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#49
  def server_key; end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#52
  def server_signature; end

  # source://net-imap//lib/net/imap/sasl/scram_algorithm.rb#50
  def stored_key; end
end

# Abstract base class for the "+SCRAM-*+" family of SASL mechanisms,
# defined in RFC5802[https://tools.ietf.org/html/rfc5802].  Use via
# Net::IMAP#authenticate.
#
# Directly supported:
# * +SCRAM-SHA-1+   --- ScramSHA1Authenticator
# * +SCRAM-SHA-256+ --- ScramSHA256Authenticator
#
# New +SCRAM-*+ mechanisms can easily be added for any hash algorithm
# supported by
# OpenSSL::Digest[https://ruby.github.io/openssl/OpenSSL/Digest.html].
# Subclasses need only set an appropriate +DIGEST_NAME+ constant.
#
# === SCRAM algorithm
#
# See the documentation and method definitions on ScramAlgorithm for an
# overview of the algorithm.  The different mechanisms differ only by
# which hash function that is used (or by support for channel binding with
# +-PLUS+).
#
# See also the methods on GS2Header.
#
# ==== Server messages
#
# As server messages are received, they are validated and loaded into
# the various attributes, e.g: #snonce, #salt, #iterations, #verifier,
# #server_error, etc.
#
# Unlike many other SASL mechanisms, the +SCRAM-*+ family supports mutual
# authentication and can return server error data in the server messages.
# If #process raises an Error for the server-final-message, then
# server_error may contain error details.
#
# === TLS Channel binding
#
# <em>The <tt>SCRAM-*-PLUS</tt> mechanisms and channel binding are not
# supported yet.</em>
#
# === Caching SCRAM secrets
#
# <em>Caching of salted_password, client_key, stored_key, and server_key
# is not supported yet.</em>
#
# source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#56
class Net::IMAP::SASL::ScramAuthenticator
  include ::Net::IMAP::SASL::GS2Header
  include ::Net::IMAP::SASL::ScramAlgorithm

  # :call-seq:
  #   new(username,  password,  **options) -> auth_ctx
  #   new(username:, password:, **options) -> auth_ctx
  #   new(authcid:,  password:, **options) -> auth_ctx
  #
  # Creates an authenticator for one of the "+SCRAM-*+" SASL mechanisms.
  # Each subclass defines #digest to match a specific mechanism.
  #
  # Called by Net::IMAP#authenticate and similar methods on other clients.
  #
  # === Parameters
  #
  # * #authcid  ― Identity whose #password is used.
  #
  #   #username - An alias for #authcid.
  # * #password ― Password or passphrase associated with this #username.
  # * _optional_ #authzid ― Alternate identity to act as or on behalf of.
  # * _optional_ #min_iterations - Overrides the default value (4096).
  #
  # Any other keyword parameters are quietly ignored.
  #
  # @return [ScramAuthenticator] a new instance of ScramAuthenticator
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#80
  def initialize(username_arg = T.unsafe(nil), password_arg = T.unsafe(nil), authcid: T.unsafe(nil), username: T.unsafe(nil), authzid: T.unsafe(nil), password: T.unsafe(nil), secret: T.unsafe(nil), min_iterations: T.unsafe(nil), cnonce: T.unsafe(nil), **options); end

  # Authentication identity: the identity that matches the #password.
  #
  # RFC-2831[https://tools.ietf.org/html/rfc2831] uses the term +username+.
  # "Authentication identity" is the generic term used by
  # RFC-4422[https://tools.ietf.org/html/rfc4422].
  # RFC-4616[https://tools.ietf.org/html/rfc4616] and many later RFCs abbreviate
  # this to +authcid+.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#107
  def authcid; end

  # Authorization identity: an identity to act as or on behalf of.  The
  # identity form is application protocol specific.  If not provided or
  # left blank, the server derives an authorization identity from the
  # authentication identity.  For example, an administrator or superuser
  # might take on another role:
  #
  #     imap.authenticate "SCRAM-SHA-256", "root", passwd, authzid: "user"
  #
  # The server is responsible for verifying the client's credentials and
  # verifying that the identity it associates with the client's
  # authentication identity is allowed to act as (or on behalf of) the
  # authorization identity.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#126
  def authzid; end

  # See {RFC5802 §7}[https://www.rfc-editor.org/rfc/rfc5802#section-7]
  # +cbind-input+.
  #
  # >>>
  #   *TODO:* implement channel binding, appending +cbind-data+ here.
  #
  # source://net-imap//lib/net/imap/sasl/gs2_header.rb#37
  def cbind_input; end

  # The client nonce, generated by SecureRandom
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#133
  def cnonce; end

  # Returns a new OpenSSL::Digest object, set to the appropriate hash
  # function for the chosen mechanism.
  #
  # <em>The class's +DIGEST_NAME+ constant must be set to the name of an
  # algorithm supported by OpenSSL::Digest.</em>
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#155
  def digest; end

  # Is the authentication exchange complete?
  #
  # If false, another server continuation is required.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#185
  def done?; end

  # See {RFC5802 §7}[https://www.rfc-editor.org/rfc/rfc5802#section-7]
  # +client-first-message+.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#159
  def initial_client_response; end

  # The iteration count for the selected hash function and user
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#142
  def iterations; end

  # The minimal allowed iteration count.  Lower #iterations will raise an
  # Error.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#130
  def min_iterations; end

  # A password or passphrase that matches the #username.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#111
  def password; end

  # responds to the server's challenges
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#164
  def process(challenge); end

  # The salt used by the server for this user
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#139
  def salt; end

  # A password or passphrase that matches the #username.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#111
  def secret; end

  # An error reported by the server during the \SASL exchange.
  #
  # Does not include errors reported by the protocol, e.g.
  # Net::IMAP::NoResponseError.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#148
  def server_error; end

  # The server nonce, which must start with #cnonce
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#136
  def snonce; end

  # Authentication identity: the identity that matches the #password.
  #
  # RFC-2831[https://tools.ietf.org/html/rfc2831] uses the term +username+.
  # "Authentication identity" is the generic term used by
  # RFC-4422[https://tools.ietf.org/html/rfc4422].
  # RFC-4616[https://tools.ietf.org/html/rfc4616] and many later RFCs abbreviate
  # this to +authcid+.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#107
  def username; end

  private

  # See {RFC5802 §7}[https://www.rfc-editor.org/rfc/rfc5802#section-7]
  # +client-final-message-without-proof+.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#240
  def client_final_message_without_proof; end

  # See {RFC5802 §7}[https://www.rfc-editor.org/rfc/rfc5802#section-7]
  # +client-first-message-bare+.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#225
  def client_first_message_bare; end

  # See {RFC5802 §7}[https://www.rfc-editor.org/rfc/rfc5802#section-7]
  # +client-final-message+.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#233
  def final_message_with_proof; end

  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#192
  def format_message(hash); end

  # RFC5802 specifies "that the order of attributes in client or server
  # messages is fixed, with the exception of extension attributes", but
  # this parses it simply as a hash, without respect to order.  Note that
  # repeated keys (violating the spec) will use the last value.
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#257
  def parse_challenge(challenge); end

  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#211
  def recv_server_final_message(server_final_message); end

  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#194
  def recv_server_first_message(server_first_message); end

  # Need to store this for auth_message
  #
  # source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#190
  def server_first_message; end
end

# Authenticator for the "+SCRAM-SHA-1+" SASL mechanism, defined in
# RFC5802[https://tools.ietf.org/html/rfc5802].
#
# Uses the "SHA-1" digest algorithm from OpenSSL::Digest.
#
# See ScramAuthenticator.
#
# source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#271
class Net::IMAP::SASL::ScramSHA1Authenticator < ::Net::IMAP::SASL::ScramAuthenticator; end

# source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#272
Net::IMAP::SASL::ScramSHA1Authenticator::DIGEST_NAME = T.let(T.unsafe(nil), String)

# Authenticator for the "+SCRAM-SHA-256+" SASL mechanism, defined in
# RFC7677[https://tools.ietf.org/html/rfc7677].
#
# Uses the "SHA-256" digest algorithm from OpenSSL::Digest.
#
# See ScramAuthenticator.
#
# source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#281
class Net::IMAP::SASL::ScramSHA256Authenticator < ::Net::IMAP::SASL::ScramAuthenticator; end

# source://net-imap//lib/net/imap/sasl/scram_authenticator.rb#282
Net::IMAP::SASL::ScramSHA256Authenticator::DIGEST_NAME = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/sasl/stringprep.rb#7
Net::IMAP::SASL::StringPrep = Net::IMAP::StringPrep

# source://net-imap//lib/net/imap/sasl/stringprep.rb#10
Net::IMAP::SASL::StringPrepError = Net::IMAP::StringPrep::StringPrepError

# Authenticator for the "+XOAUTH2+" SASL mechanism.  This mechanism was
# originally created for GMail and widely adopted by hosted email providers.
# +XOAUTH2+ has been documented by
# Google[https://developers.google.com/gmail/imap/xoauth2-protocol] and
# Microsoft[https://learn.microsoft.com/en-us/exchange/client-developer/legacy-protocols/how-to-authenticate-an-imap-pop-smtp-application-by-using-oauth].
#
# This mechanism requires an OAuth2 access token which has been authorized
# with the appropriate OAuth2 scopes to access the user's services.  Most of
# these scopes are not standardized---consult each service provider's
# documentation for their scopes.
#
# Although this mechanism was never standardized and has been obsoleted by
# "+OAUTHBEARER+", it is still very widely supported.
#
# See Net::IMAP::SASL::OAuthBearerAuthenticator.
#
# source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#18
class Net::IMAP::SASL::XOAuth2Authenticator
  # :call-seq:
  #   new(username,  oauth2_token,  **) -> authenticator
  #   new(username:, oauth2_token:, **) -> authenticator
  #   new(authzid:,  oauth2_token:, **) -> authenticator
  #
  # Creates an Authenticator for the "+XOAUTH2+" SASL mechanism, as specified by
  # Google[https://developers.google.com/gmail/imap/xoauth2-protocol],
  # Microsoft[https://learn.microsoft.com/en-us/exchange/client-developer/legacy-protocols/how-to-authenticate-an-imap-pop-smtp-application-by-using-oauth]
  # and Yahoo[https://senders.yahooinc.com/developer/documentation].
  #
  # === Properties
  #
  # * #username --- the username for the account being accessed.
  #
  #   #authzid  --- an alias for #username.
  #
  #   Note that, unlike some other authenticators, +username+ sets the
  #   _authorization_ identity and not the _authentication_ identity.  The
  #   authenticated identity is established for the client with the OAuth token.
  #
  # * #oauth2_token --- An OAuth2.0 access token which is authorized to access
  #   the service for #username.
  #
  # Any other keyword parameters are quietly ignored.
  #
  # @return [XOAuth2Authenticator] a new instance of XOAuth2Authenticator
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#71
  def initialize(user = T.unsafe(nil), token = T.unsafe(nil), username: T.unsafe(nil), oauth2_token: T.unsafe(nil), authzid: T.unsafe(nil), secret: T.unsafe(nil), **_arg6); end

  # It is unclear from {Google's original XOAUTH2
  # documentation}[https://developers.google.com/gmail/imap/xoauth2-protocol],
  # whether "User" refers to the authentication identity (+authcid+) or the
  # authorization identity (+authzid+).  The authentication identity is
  # established for the client by the OAuth token, so it seems that +username+
  # must be the authorization identity.
  #
  # {Microsoft's documentation for shared
  # mailboxes}[https://learn.microsoft.com/en-us/exchange/client-developer/legacy-protocols/how-to-authenticate-an-imap-pop-smtp-application-by-using-oauth#sasl-xoauth2-authentication-for-shared-mailboxes-in-office-365]
  # _clearly_ indicates that the Office 365 server interprets it as the
  # authorization identity.
  #
  # Although they _should_ validate that the token has been authorized to access
  # the service for +username+, _some_ servers appear to ignore this field,
  # relying only the identity and scope authorized by the token.
  # Note that, unlike most other authenticators, #username is an alias for the
  # authorization identity and not the authentication identity.  The
  # authenticated identity is established for the client by the #oauth2_token.
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#35
  def authzid; end

  # Returns true when the initial client response was sent.
  #
  # The authentication should not succeed unless this returns true, but it
  # does *not* indicate success.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#98
  def done?; end

  # :call-seq:
  #   initial_response? -> true
  #
  # +XOAUTH2+ can send an initial client response.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#84
  def initial_response?; end

  # An OAuth2 access token which has been authorized with the appropriate OAuth2
  # scopes to use the service for #username.
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#44
  def oauth2_token; end

  # Returns the XOAUTH2 formatted response, which combines the +username+
  # with the +oauth2_token+.
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#88
  def process(_data); end

  # An OAuth2 access token which has been authorized with the appropriate OAuth2
  # scopes to use the service for #username.
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#44
  def secret; end

  # It is unclear from {Google's original XOAUTH2
  # documentation}[https://developers.google.com/gmail/imap/xoauth2-protocol],
  # whether "User" refers to the authentication identity (+authcid+) or the
  # authorization identity (+authzid+).  The authentication identity is
  # established for the client by the OAuth token, so it seems that +username+
  # must be the authorization identity.
  #
  # {Microsoft's documentation for shared
  # mailboxes}[https://learn.microsoft.com/en-us/exchange/client-developer/legacy-protocols/how-to-authenticate-an-imap-pop-smtp-application-by-using-oauth#sasl-xoauth2-authentication-for-shared-mailboxes-in-office-365]
  # _clearly_ indicates that the Office 365 server interprets it as the
  # authorization identity.
  #
  # Although they _should_ validate that the token has been authorized to access
  # the service for +username+, _some_ servers appear to ignore this field,
  # relying only the identity and scope authorized by the token.
  #
  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#35
  def username; end

  private

  # source://net-imap//lib/net/imap/sasl/xoauth2_authenticator.rb#102
  def build_oauth2_string(username, oauth2_token); end
end

# Experimental
#
# source://net-imap//lib/net/imap/sasl_adapter.rb#7
class Net::IMAP::SASLAdapter < ::Net::IMAP::SASL::ClientAdapter
  include ::Net::IMAP::SASL::ProtocolAdapters::IMAP

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl_adapter.rb#15
  def auth_capable?(mechanism); end

  # source://net-imap//lib/net/imap/sasl_adapter.rb#16
  def drop_connection; end

  # source://net-imap//lib/net/imap/sasl_adapter.rb#17
  def drop_connection!; end

  # source://net-imap//lib/net/imap/sasl_adapter.rb#13
  def response_errors; end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sasl_adapter.rb#14
  def sasl_ir_capable?; end
end

# source://net-imap//lib/net/imap/sasl_adapter.rb#10
Net::IMAP::SASLAdapter::RESPONSE_ERRORS = T.let(T.unsafe(nil), Array)

# Mailbox attribute indicating that this mailbox is used to hold copies of
# messages that have been sent. Some server implementations might put
# messages here automatically. Alternatively, this might just be advice that
# a client save sent messages here.
#
# source://net-imap//lib/net/imap/flags.rb#248
Net::IMAP::SENT = T.let(T.unsafe(nil), Symbol)

# strftime/strptime format for an IMAP4 +date+, excluding optional dquotes.
# Use via the encode_date and decode_date methods.
#
#   date            = date-text / DQUOTE date-text DQUOTE
#   date-text       = date-day "-" date-month "-" date-year
#
#   date-day        = 1*2DIGIT
#                       ; Day of month
#   date-month      = "Jan" / "Feb" / "Mar" / "Apr" / "May" / "Jun" /
#                     "Jul" / "Aug" / "Sep" / "Oct" / "Nov" / "Dec"
#   date-year       = 4DIGIT
#
# source://net-imap//lib/net/imap/data_encoding.rb#22
Net::IMAP::STRFDATE = T.let(T.unsafe(nil), String)

# strftime/strptime format for an IMAP4 +date-time+, including dquotes.
# See the encode_datetime and decode_datetime methods.
#
#   date-time       = DQUOTE date-day-fixed "-" date-month "-" date-year
#                     SP time SP zone DQUOTE
#
#   date-day-fixed  = (SP DIGIT) / 2DIGIT
#                       ; Fixed-format version of date-day
#   date-month      = "Jan" / "Feb" / "Mar" / "Apr" / "May" / "Jun" /
#                     "Jul" / "Aug" / "Sep" / "Oct" / "Nov" / "Dec"
#   date-year       = 4DIGIT
#   time            = 2DIGIT ":" 2DIGIT ":" 2DIGIT
#                       ; Hours minutes seconds
#   zone            = ("+" / "-") 4DIGIT
#                       ; Signed four-digit value of hhmm representing
#                       ; hours and minutes east of Greenwich (that is,
#                       ; the amount that the given time differs from
#                       ; Universal Time).  Subtracting the timezone
#                       ; from the given time will give the UT form.
#                       ; The Universal Time zone is "+0000".
#
# Note that Time.strptime <tt>"%d"</tt> flexibly parses either space or zero
# padding.  However, the DQUOTEs are *not* optional.
#
# source://net-imap//lib/net/imap/data_encoding.rb#47
Net::IMAP::STRFTIME = T.let(T.unsafe(nil), String)

# The mailbox name was subscribed to using the #subscribe command.
#
# source://net-imap//lib/net/imap/flags.rb#173
Net::IMAP::SUBSCRIBED = T.let(T.unsafe(nil), Symbol)

# An array of sequence numbers returned by Net::IMAP#search, or unique
# identifiers returned by Net::IMAP#uid_search.
#
# For backward compatibility, SearchResult inherits from Array.
#
# source://net-imap//lib/net/imap/search_result.rb#10
class Net::IMAP::SearchResult < ::Array
  # Returns a SearchResult populated with the given +seq_nums+.
  #
  #     Net::IMAP::SearchResult.new([1, 3, 5], modseq: 9)
  #     # => Net::IMAP::SearchResult[1, 3, 5, modseq: 9]
  #
  # @return [SearchResult] a new instance of SearchResult
  #
  # source://net-imap//lib/net/imap/search_result.rb#29
  def initialize(seq_nums, modseq: T.unsafe(nil)); end

  # Returns whether +other+ is a SearchResult with the same values and the
  # same #modseq.  The order of numbers is irrelevant.
  #
  #     Net::IMAP::SearchResult[123, 456, modseq: 789] ==
  #       Net::IMAP::SearchResult[123, 456, modseq: 789]
  #     # => true
  #     Net::IMAP::SearchResult[123, 456, modseq: 789] ==
  #       Net::IMAP::SearchResult[456, 123, modseq: 789]
  #     # => true
  #
  #     Net::IMAP::SearchResult[123, 456, modseq: 789] ==
  #       Net::IMAP::SearchResult[987, 654, modseq: 789]
  #     # => false
  #     Net::IMAP::SearchResult[123, 456, modseq: 789] ==
  #       Net::IMAP::SearchResult[1, 2, 3, modseq: 9999]
  #     # => false
  #
  # SearchResult can be compared directly with Array, if #modseq is nil and
  # the array is sorted.
  #
  #     Net::IMAP::SearchResult[9, 8, 6, 4, 1] == [1, 4, 6, 8, 9] # => true
  #     Net::IMAP::SearchResult[3, 5, 7, modseq: 99] == [3, 5, 7] # => false
  #
  # Note that Array#== does require matching order and ignores #modseq.
  #
  #     [9, 8, 6, 4, 1] == Net::IMAP::SearchResult[1, 4, 6, 8, 9] # => false
  #     [3, 5, 7] == Net::IMAP::SearchResult[3, 5, 7, modseq: 99] # => true
  #
  # source://net-imap//lib/net/imap/search_result.rb#62
  def ==(other); end

  # Hash equality.  Unlike #==, order will be taken into account.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/search_result.rb#77
  def eql?(other); end

  # Hash equality.  Unlike #==, order will be taken into account.
  #
  # source://net-imap//lib/net/imap/search_result.rb#71
  def hash; end

  # Returns a string that represents the SearchResult.
  #
  #    Net::IMAP::SearchResult[123, 456, 789].inspect
  #    # => "[123, 456, 789]"
  #
  #    Net::IMAP::SearchResult[543, 210, 678, modseq: 2048].inspect
  #    # => "Net::IMAP::SearchResult[543, 210, 678, modseq: 2048]"
  #
  # source://net-imap//lib/net/imap/search_result.rb#90
  def inspect; end

  # A modification sequence number, as described by the +CONDSTORE+
  # extension in {[RFC7162
  # §3.1.6]}[https://www.rfc-editor.org/rfc/rfc7162.html#section-3.1.6].
  #
  # source://net-imap//lib/net/imap/search_result.rb#23
  def modseq; end

  # source://net-imap//lib/net/imap/search_result.rb#123
  def pretty_print(pp); end

  # Returns a string that follows the formal \IMAP syntax.
  #
  #    data = Net::IMAP::SearchResult[2, 8, 32, 128, 256, 512]
  #    data.to_s           # => "* SEARCH 2 8 32 128 256 512"
  #    data.to_s("SEARCH") # => "* SEARCH 2 8 32 128 256 512"
  #    data.to_s("SORT")   # => "* SORT 2 8 32 128 256 512"
  #    data.to_s(nil)      # => "2 8 32 128 256 512"
  #
  #    data = Net::IMAP::SearchResult[1, 3, 16, 1024, modseq: 2048].to_s
  #    data.to_s           # => "* SEARCH 1 3 16 1024 (MODSEQ 2048)"
  #    data.to_s("SORT")   # => "* SORT 1 3 16 1024 (MODSEQ 2048)"
  #    data.to_s           # => "1 3 16 1024 (MODSEQ 2048)"
  #
  # source://net-imap//lib/net/imap/search_result.rb#108
  def to_s(type = T.unsafe(nil)); end

  # Converts the SearchResult into a SequenceSet.
  #
  #     Net::IMAP::SearchResult[9, 1, 2, 4, 10, 12, 3, modseq: 123_456]
  #       .to_sequence_set
  #     # => Net::IMAP::SequenceSet["1:4,9:10,12"]
  #
  # source://net-imap//lib/net/imap/search_result.rb#121
  def to_sequence_set; end

  class << self
    # Returns a SearchResult populated with the given +seq_nums+.
    #
    #     Net::IMAP::SearchResult[1, 3, 5, modseq: 9]
    #     # => Net::IMAP::SearchResult[1, 3, 5, modseq: 9]
    #
    # source://net-imap//lib/net/imap/search_result.rb#16
    def [](*seq_nums, modseq: T.unsafe(nil)); end
  end
end

# An \IMAP sequence set is a set of message sequence numbers or unique
# identifier numbers ("UIDs").  It contains numbers and ranges of numbers.
# The numbers are all non-zero unsigned 32-bit integers and one special
# value (<tt>"*"</tt>) that represents the largest value in the mailbox.
#
# Certain types of \IMAP responses will contain a SequenceSet, for example
# the data for a <tt>"MODIFIED"</tt> ResponseCode.  Some \IMAP commands may
# receive a SequenceSet as an argument, for example IMAP#search, IMAP#fetch,
# and IMAP#store.
#
# == EXPERIMENTAL API
#
# SequenceSet is currently experimental.  Only two methods, ::[] and
# #valid_string, are considered stable.  Although the API isn't expected to
# change much, any other methods may be removed or changed without
# deprecation.
#
# == Creating sequence sets
#
# SequenceSet.new with no arguments creates an empty sequence set.  Note
# that an empty sequence set is invalid in the \IMAP grammar.
#
#     set = Net::IMAP::SequenceSet.new
#     set.empty?        #=> true
#     set.valid?        #=> false
#     set.valid_string  #!> raises DataFormatError
#     set << 1..10
#     set.empty?        #=> false
#     set.valid?        #=> true
#     set.valid_string  #=> "1:10"
#
# SequenceSet.new may receive a single optional argument: a non-zero 32 bit
# unsigned integer, a range, a <tt>sequence-set</tt> formatted string,
# another sequence set, or an enumerable containing any of these.
#
#     set = Net::IMAP::SequenceSet.new(1)
#     set.valid_string  #=> "1"
#     set = Net::IMAP::SequenceSet.new(1..100)
#     set.valid_string  #=> "1:100"
#     set = Net::IMAP::SequenceSet.new(1...100)
#     set.valid_string  #=> "1:99"
#     set = Net::IMAP::SequenceSet.new([1, 2, 5..])
#     set.valid_string  #=> "1:2,5:*"
#     set = Net::IMAP::SequenceSet.new("1,2,3:7,5,6:10,2048,1024")
#     set.valid_string  #=> "1,2,3:7,5,6:10,2048,1024"
#     set = Net::IMAP::SequenceSet.new(1, 2, 3..7, 5, 6..10, 2048, 1024)
#     set.valid_string  #=> "1:10,55,1024:2048"
#
# Use ::[] with one or more arguments to create a frozen SequenceSet.  An
# invalid (empty) set cannot be created with ::[].
#
#     set = Net::IMAP::SequenceSet["1,2,3:7,5,6:10,2048,1024"]
#     set.valid_string  #=> "1,2,3:7,5,6:10,2048,1024"
#     set = Net::IMAP::SequenceSet[1, 2, [3..7, 5], 6..10, 2048, 1024]
#     set.valid_string  #=> "1:10,55,1024:2048"
#
# == Normalized form
#
# When a sequence set is created with a single String value, that #string
# representation is preserved.  SequenceSet's internal representation
# implicitly sorts all entries, de-duplicates numbers, and coalesces
# adjacent or overlapping ranges.  Most enumeration methods and offset-based
# methods use this normalized representation.  Most modification methods
# will convert #string to its normalized form.
#
# In some cases the order of the string representation is significant, such
# as the +ESORT+, <tt>CONTEXT=SORT</tt>, and +UIDPLUS+ extensions.  Use
# #entries or #each_entry to enumerate the set in its original order.  To
# preserve #string order while modifying a set, use #append, #string=, or
# #replace.
#
# == Using <tt>*</tt>
#
# \IMAP sequence sets may contain a special value <tt>"*"</tt>, which
# represents the largest number in use.  From +seq-number+ in
# {RFC9051 §9}[https://www.rfc-editor.org/rfc/rfc9051.html#section-9-5]:
# >>>
#   In the case of message sequence numbers, it is the number of messages
#   in a non-empty mailbox.  In the case of unique identifiers, it is the
#   unique identifier of the last message in the mailbox or, if the
#   mailbox is empty, the mailbox's current UIDNEXT value.
#
# When creating a SequenceSet, <tt>*</tt> may be input as <tt>-1</tt>,
# <tt>"*"</tt>, <tt>:*</tt>, an endless range, or a range ending in
# <tt>-1</tt>.  When converting to #elements, #ranges, or #numbers, it will
# output as either <tt>:*</tt> or an endless range.  For example:
#
#   Net::IMAP::SequenceSet["1,3,*"].to_a      #=> [1, 3, :*]
#   Net::IMAP::SequenceSet["1,234:*"].to_a    #=> [1, 234..]
#   Net::IMAP::SequenceSet[1234..-1].to_a     #=> [1234..]
#   Net::IMAP::SequenceSet[1234..].to_a       #=> [1234..]
#
#   Net::IMAP::SequenceSet[1234..].to_s       #=> "1234:*"
#   Net::IMAP::SequenceSet[1234..-1].to_s     #=> "1234:*"
#
# Use #limit to convert <tt>"*"</tt> to a maximum value.  When a range
# includes <tt>"*"</tt>, the maximum value will always be matched:
#
#   Net::IMAP::SequenceSet["9999:*"].limit(max: 25)
#   #=> Net::IMAP::SequenceSet["25"]
#
# === Surprising <tt>*</tt> behavior
#
# When a set includes <tt>*</tt>, some methods may have surprising behavior.
#
# For example, #complement treats <tt>*</tt> as its own number.  This way,
# the #intersection of a set and its #complement will always be empty.
# This is not how an \IMAP server interprets the set: it will convert
# <tt>*</tt> to either the number of messages in the mailbox or +UIDNEXT+,
# as appropriate.  And there _will_ be overlap between a set and its
# complement after #limit is applied to each:
#
#   ~Net::IMAP::SequenceSet["*"]  == Net::IMAP::SequenceSet[1..(2**32-1)]
#   ~Net::IMAP::SequenceSet[1..5] == Net::IMAP::SequenceSet["6:*"]
#
#   set = Net::IMAP::SequenceSet[1..5]
#   (set & ~set).empty? => true
#
#   (set.limit(max: 4) & (~set).limit(max: 4)).to_a => [4]
#
# When counting the number of numbers in a set, <tt>*</tt> will be counted
# _except_ when UINT32_MAX is also in the set:
#   UINT32_MAX = 2**32 - 1
#   Net::IMAP::SequenceSet["*"].count                   => 1
#   Net::IMAP::SequenceSet[1..UINT32_MAX - 1, :*].count => UINT32_MAX
#
#   Net::IMAP::SequenceSet["1:*"].count                 => UINT32_MAX
#   Net::IMAP::SequenceSet[UINT32_MAX, :*].count        => 1
#   Net::IMAP::SequenceSet[UINT32_MAX..].count          => 1
#
# == What's here?
#
# SequenceSet provides methods for:
# * {Creating a SequenceSet}[rdoc-ref:SequenceSet@Methods+for+Creating+a+SequenceSet]
# * {Comparing}[rdoc-ref:SequenceSet@Methods+for+Comparing]
# * {Querying}[rdoc-ref:SequenceSet@Methods+for+Querying]
# * {Iterating}[rdoc-ref:SequenceSet@Methods+for+Iterating]
# * {Set Operations}[rdoc-ref:SequenceSet@Methods+for+Set+Operations]
# * {Assigning}[rdoc-ref:SequenceSet@Methods+for+Assigning]
# * {Deleting}[rdoc-ref:SequenceSet@Methods+for+Deleting]
# * {IMAP String Formatting}[rdoc-ref:SequenceSet@Methods+for+IMAP+String+Formatting]
#
# === Methods for Creating a \SequenceSet
# * ::[]: Creates a validated frozen sequence set from one or more inputs.
# * ::new: Creates a new mutable sequence set, which may be empty (invalid).
# * ::try_convert: Calls +to_sequence_set+ on an object and verifies that
#   the result is a SequenceSet.
# * ::empty: Returns a frozen empty (invalid) SequenceSet.
# * ::full: Returns a frozen SequenceSet containing every possible number.
#
# === Methods for Comparing
#
# <i>Comparison to another \SequenceSet:</i>
# - #==: Returns whether a given set contains the same numbers as +self+.
# - #eql?: Returns whether a given set uses the same #string as +self+.
#
# <i>Comparison to objects which are convertible to \SequenceSet:</i>
# - #===:
#   Returns whether a given object is fully contained within +self+, or
#   +nil+ if the object cannot be converted to a compatible type.
# - #cover? (aliased as #===):
#   Returns whether a given object is fully contained within +self+.
# - #intersect? (aliased as #overlap?):
#   Returns whether +self+ and a given object have any common elements.
# - #disjoint?:
#   Returns whether +self+ and a given object have no common elements.
#
# === Methods for Querying
# These methods do not modify +self+.
#
# <i>Set membership:</i>
# - #include? (aliased as #member?):
#   Returns whether a given object (nz-number, range, or <tt>*</tt>) is
#   contained by the set.
# - #include_star?: Returns whether the set contains <tt>*</tt>.
#
# <i>Minimum and maximum value elements:</i>
# - #min: Returns the minimum number in the set.
# - #max: Returns the maximum number in the set.
# - #minmax: Returns the minimum and maximum numbers in the set.
#
# <i>Accessing value by offset:</i>
# - #[] (aliased as #slice): Returns the number or consecutive subset at a
#   given offset or range of offsets.
# - #at: Returns the number at a given offset.
# - #find_index: Returns the given number's offset in the set
#
# <i>Set cardinality:</i>
# - #count (aliased as #size): Returns the count of numbers in the set.
# - #empty?: Returns whether the set has no members.  \IMAP syntax does not
#   allow empty sequence sets.
# - #valid?: Returns whether the set has any members.
# - #full?: Returns whether the set contains every possible value, including
#   <tt>*</tt>.
#
# === Methods for Iterating
#
# - #each_element: Yields each number and range in the set, sorted and
#   coalesced, and returns +self+.
# - #elements (aliased as #to_a): Returns an Array of every number and range
#   in the set, sorted and coalesced.
# - #each_entry: Yields each number and range in the set, unsorted and
#   without deduplicating numbers or coalescing ranges, and returns +self+.
# - #entries: Returns an Array of every number and range in the set,
#   unsorted and without deduplicating numbers or coalescing ranges.
# - #each_range:
#   Yields each element in the set as a Range and returns +self+.
# - #ranges: Returns an Array of every element in the set, converting
#   numbers into ranges of a single value.
# - #each_number: Yields each number in the set and returns +self+.
# - #numbers: Returns an Array with every number in the set, expanding
#   ranges into all of their contained numbers.
# - #to_set: Returns a Set containing all of the #numbers in the set.
#
# === Methods for \Set Operations
# These methods do not modify +self+.
#
# - #| (aliased as #union and #+): Returns a new set combining all members
#   from +self+ with all members from the other object.
# - #& (aliased as #intersection): Returns a new set containing all members
#   common to +self+ and the other object.
# - #- (aliased as #difference): Returns a copy of +self+ with all members
#   in the other object removed.
# - #^ (aliased as #xor): Returns a new set containing all members from
#   +self+ and the other object except those common to both.
# - #~ (aliased as #complement): Returns a new set containing all members
#   that are not in +self+
# - #limit: Returns a copy of +self+ which has replaced <tt>*</tt> with a
#   given maximum value and removed all members over that maximum.
#
# === Methods for Assigning
# These methods add or replace elements in +self+.
#
# - #add (aliased as #<<): Adds a given object to the set; returns +self+.
# - #add?: If the given object is not an element in the set, adds it and
#   returns +self+; otherwise, returns +nil+.
# - #merge: Merges multiple elements into the set; returns +self+.
# - #append: Adds a given object to the set, appending it to the existing
#   string, and returns +self+.
# - #string=: Assigns a new #string value and replaces #elements to match.
# - #replace: Replaces the contents of the set with the contents
#   of a given object.
# - #complement!: Replaces the contents of the set with its own #complement.
#
# === Methods for Deleting
# These methods remove elements from +self+.
#
# - #clear: Removes all elements in the set; returns +self+.
# - #delete: Removes a given object from the set; returns +self+.
# - #delete?: If the given object is an element in the set, removes it and
#   returns it; otherwise, returns +nil+.
# - #delete_at: Removes the number at a given offset.
# - #slice!: Removes the number or consecutive numbers at a given offset or
#   range of offsets.
# - #subtract: Removes each given object from the set; returns +self+.
# - #limit!: Replaces <tt>*</tt> with a given maximum value and removes all
#   members over that maximum; returns +self+.
#
# === Methods for \IMAP String Formatting
#
# - #to_s: Returns the +sequence-set+ string, or an empty string when the
#   set is empty.
# - #string: Returns the +sequence-set+ string, or nil when empty.
# - #valid_string: Returns the +sequence-set+ string, or raises
#   DataFormatError when the set is empty.
# - #normalized_string: Returns a <tt>sequence-set</tt> string with its
#   elements sorted and coalesced, or nil when the set is empty.
# - #normalize: Returns a new set with this set's normalized +sequence-set+
#   representation.
# - #normalize!: Updates #string to its normalized +sequence-set+
#   representation and returns +self+.
#
# source://net-imap//lib/net/imap/sequence_set.rb#279
class Net::IMAP::SequenceSet
  # Create a new SequenceSet object from +input+, which may be another
  # SequenceSet, an IMAP formatted +sequence-set+ string, a number, a
  # range, <tt>:*</tt>, or an enumerable of these.
  #
  # Use ::[] to create a frozen (non-empty) SequenceSet.
  #
  # @return [SequenceSet] a new instance of SequenceSet
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#348
  def initialize(input = T.unsafe(nil)); end

  # :call-seq:
  #   self & other        -> sequence set
  #   intersection(other) -> sequence set
  #
  # Returns a new sequence set containing only the numbers common to this
  # set and +other+.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet[1..5] & [2, 4, 6]
  #     #=> Net::IMAP::SequenceSet["2,4"]
  #
  # <tt>(seqset & other)</tt> is equivalent to <tt>(seqset - ~other)</tt>.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#623
  def &(other); end

  # :call-seq:
  #   self + other -> sequence set
  #   self | other -> sequence set
  #   union(other) -> sequence set
  #
  # Returns a new sequence set that has every number in the +other+ object
  # added.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet["1:5"] | 2 | [4..6, 99]
  #     #=> Net::IMAP::SequenceSet["1:6,99"]
  #
  # Related: #add, #merge
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#586
  def +(other); end

  # :call-seq:
  #   self - other      -> sequence set
  #   difference(other) -> sequence set
  #
  # Returns a new sequence set built by duplicating this set and removing
  # every number that appears in +other+.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet[1..5] - 2 - 4 - 6
  #     #=> Net::IMAP::SequenceSet["1,3,5"]
  #
  # Related: #subtract
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#605
  def -(other); end

  # :call-seq:
  #   add(object)   -> self
  #   self << other -> self
  #
  # Adds a range or number to the set and returns +self+.
  #
  # #string will be regenerated.  Use #merge to add many elements at once.
  #
  # Related: #add?, #merge, #union
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#674
  def <<(object); end

  # :call-seq: self == other -> true or false
  #
  # Returns true when the other SequenceSet represents the same message
  # identifiers.  Encoding difference—such as order, overlaps, or
  # duplicates—are ignored.
  #
  #   Net::IMAP::SequenceSet["1:3"]   == Net::IMAP::SequenceSet["1:3"]
  #   #=> true
  #   Net::IMAP::SequenceSet["1,2,3"] == Net::IMAP::SequenceSet["1:3"]
  #   #=> true
  #   Net::IMAP::SequenceSet["1,3"]   == Net::IMAP::SequenceSet["3,1"]
  #   #=> true
  #   Net::IMAP::SequenceSet["9,1:*"] == Net::IMAP::SequenceSet["1:*"]
  #   #=> true
  #
  # Related: #eql?, #normalize
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#441
  def ==(other); end

  # :call-seq: self === other -> true | false | nil
  #
  # Returns whether +other+ is contained within the set.  Returns +nil+ if a
  # StandardError is raised while converting +other+ to a comparable type.
  #
  # Related: #cover?, #include?, #include_star?
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#471
  def ===(other); end

  # :call-seq:
  #    seqset[index]         -> integer or :* or nil
  #    slice(index)          -> integer or :* or nil
  #    seqset[start, length] -> sequence set or nil
  #    slice(start, length)  -> sequence set or nil
  #    seqset[range]         -> sequence set or nil
  #    slice(range)          -> sequence set or nil
  #
  # Returns a number or a subset from +self+, without modifying the set.
  #
  # When an Integer argument +index+ is given, the number at offset +index+
  # is returned:
  #
  #     set = Net::IMAP::SequenceSet["10:15,20:23,26"]
  #     set[0]   #=> 10
  #     set[5]   #=> 15
  #     set[10]  #=> 26
  #
  # If +index+ is negative, it counts relative to the end of +self+:
  #     set = Net::IMAP::SequenceSet["10:15,20:23,26"]
  #     set[-1]  #=> 26
  #     set[-3]  #=> 22
  #     set[-6]  #=> 15
  #
  # If +index+ is out of range, +nil+ is returned.
  #
  #     set = Net::IMAP::SequenceSet["10:15,20:23,26"]
  #     set[11]  #=> nil
  #     set[-12] #=> nil
  #
  # The result is based on the normalized set—sorted and de-duplicated—not
  # on the assigned value of #string.
  #
  #     set = Net::IMAP::SequenceSet["12,20:23,11:16,21"]
  #     set[0]   #=> 11
  #     set[-1]  #=> 23
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1088
  def [](index, length = T.unsafe(nil)); end

  # :call-seq:
  #   self ^ other -> sequence set
  #   xor(other)   -> sequence set
  #
  # Returns a new sequence set containing numbers that are exclusive between
  # this set and +other+.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet[1..5] ^ [2, 4, 6]
  #     #=> Net::IMAP::SequenceSet["1,3,5:6"]
  #
  # <tt>(seqset ^ other)</tt> is equivalent to <tt>((seqset | other) -
  # (seqset & other))</tt>.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#644
  def ^(other); end

  # :call-seq:
  #   add(object)   -> self
  #   self << other -> self
  #
  # Adds a range or number to the set and returns +self+.
  #
  # #string will be regenerated.  Use #merge to add many elements at once.
  #
  # Related: #add?, #merge, #union
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#674
  def add(object); end

  # :call-seq: add?(object) -> self or nil
  #
  # Adds a range or number to the set and returns +self+.  Returns +nil+
  # when the object is already included in the set.
  #
  # #string will be regenerated.  Use #merge to add many elements at once.
  #
  # Related: #add, #merge, #union, #include?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#700
  def add?(object); end

  # Adds a range or number to the set and returns +self+.
  #
  # Unlike #add, #merge, or #union, the new value is appended to #string.
  # This may result in a #string which has duplicates or is out-of-order.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#684
  def append(object); end

  # :call-seq: at(index) -> integer or nil
  #
  # Returns a number from +self+, without modifying the set.  Behaves the
  # same as #[], except that #at only allows a single integer argument.
  #
  # Related: #[], #slice
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1037
  def at(index); end

  # Removes all elements and returns self.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#351
  def clear; end

  # :call-seq:
  #   ~ self     -> sequence set
  #   complement -> sequence set
  #
  # Returns the complement of self, a SequenceSet which contains all numbers
  # _except_ for those in this set.
  #
  #     ~Net::IMAP::SequenceSet.full  #=> Net::IMAP::SequenceSet.empty
  #     ~Net::IMAP::SequenceSet.empty #=> Net::IMAP::SequenceSet.full
  #     ~Net::IMAP::SequenceSet["1:5,100:222"]
  #     #=> Net::IMAP::SequenceSet["6:99,223:*"]
  #     ~Net::IMAP::SequenceSet["6:99,223:*"]
  #     #=> Net::IMAP::SequenceSet["1:5,100:222"]
  #
  # Related: #complement!
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#662
  def complement; end

  # :call-seq: complement! -> self
  #
  # Converts the SequenceSet to its own #complement.  It will contain all
  # possible values _except_ for those currently in the set.
  #
  # Related: #complement
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1168
  def complement!; end

  # Returns the count of #numbers in the set.
  #
  # If <tt>*</tt> and <tt>2**32 - 1</tt> (the maximum 32-bit unsigned
  # integer value) are both in the set, they will only be counted once.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#989
  def count; end

  # :call-seq: cover?(other) -> true | false | nil
  #
  # Returns whether +other+ is contained within the set.  +other+ may be any
  # object that would be accepted by ::new.
  #
  # Related: #===, #include?, #include_star?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#483
  def cover?(other); end

  # :call-seq: delete(object) -> self
  #
  # Deletes the given range or number from the set and returns +self+.
  #
  # #string will be regenerated after deletion.  Use #subtract to remove
  # many elements at once.
  #
  # Related: #delete?, #delete_at, #subtract, #difference
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#712
  def delete(object); end

  # :call-seq:
  #   delete?(number) -> integer or nil
  #   delete?(star)   -> :* or nil
  #   delete?(range)  -> sequence set or nil
  #
  # Removes a specified value from the set, and returns the removed value.
  # Returns +nil+ if nothing was removed.
  #
  # Returns an integer when the specified +number+ argument was removed:
  #     set = Net::IMAP::SequenceSet.new [5..10, 20]
  #     set.delete?(7)      #=> 7
  #     set                 #=> #<Net::IMAP::SequenceSet "5:6,8:10,20">
  #     set.delete?("20")   #=> 20
  #     set                 #=> #<Net::IMAP::SequenceSet "5:6,8:10">
  #     set.delete?(30)     #=> nil
  #
  # Returns <tt>:*</tt> when <tt>*</tt> or <tt>-1</tt> is specified and
  # removed:
  #     set = Net::IMAP::SequenceSet.new "5:9,20,35,*"
  #     set.delete?(-1)  #=> :*
  #     set              #=> #<Net::IMAP::SequenceSet "5:9,20,35">
  #
  # And returns a new SequenceSet when a range is specified:
  #
  #     set = Net::IMAP::SequenceSet.new [5..10, 20]
  #     set.delete?(9..)  #=> #<Net::IMAP::SequenceSet "9:10,20">
  #     set               #=> #<Net::IMAP::SequenceSet "5:8">
  #     set.delete?(21..) #=> nil
  #
  # #string will be regenerated after deletion.
  #
  # Related: #delete, #delete_at, #subtract, #difference, #disjoint?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#749
  def delete?(object); end

  # :call-seq: delete_at(index) -> number or :* or nil
  #
  # Deletes a number the set, indicated by the given +index+.  Returns the
  # number that was removed, or +nil+ if nothing was removed.
  #
  # #string will be regenerated after deletion.
  #
  # Related: #delete, #delete?, #slice!, #subtract, #difference
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#772
  def delete_at(index); end

  # :call-seq:
  #   self - other      -> sequence set
  #   difference(other) -> sequence set
  #
  # Returns a new sequence set built by duplicating this set and removing
  # every number that appears in +other+.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet[1..5] - 2 - 4 - 6
  #     #=> Net::IMAP::SequenceSet["1,3,5"]
  #
  # Related: #subtract
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#605
  def difference(other); end

  # Returns +true+ if the set and a given object have no common elements,
  # +false+ otherwise.
  #
  #     Net::IMAP::SequenceSet["5:10"].disjoint? "7,9,11" #=> false
  #     Net::IMAP::SequenceSet["5:10"].disjoint? "11:33"  #=> true
  #
  # Related: #intersection, #intersect?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#535
  def disjoint?(other); end

  # Yields each number or range (or <tt>:*</tt>) in #elements to the block
  # and returns self.  Returns an enumerator when called without a block.
  #
  # The returned numbers are sorted and de-duplicated, even when the input
  # #string is not.  See #normalize.
  #
  # Related: #elements, #each_entry
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#924
  def each_element; end

  # Yields each number or range in #string to the block and returns +self+.
  # Returns an enumerator when called without a block.
  #
  # The entries are yielded in the same order they appear in #string, with
  # no sorting, deduplication, or coalescing.  When #string is in its
  # normalized form, this will yield the same values as #each_element.
  #
  # Related: #entries, #each_element
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#910
  def each_entry(&block); end

  # Yields each number in #numbers to the block and returns self.
  # If the set contains a <tt>*</tt>, RangeError will be raised.
  #
  # Returns an enumerator when called without a block (even if the set
  # contains <tt>*</tt>).
  #
  # Related: #numbers
  #
  # @raise [RangeError]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#964
  def each_number(&block); end

  # Yields each range in #ranges to the block and returns self.
  # Returns an enumerator when called without a block.
  #
  # Related: #ranges
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#946
  def each_range; end

  # Returns an array of ranges and integers and <tt>:*</tt>.
  #
  # The returned elements are sorted and coalesced, even when the input
  # #string is not.  <tt>*</tt> will sort last.  See #normalize.
  #
  # By itself, <tt>*</tt> translates to <tt>:*</tt>.  A range containing
  # <tt>*</tt> translates to an endless range.  Use #limit to translate both
  # cases to a maximum value.
  #
  # If the original input was unordered or contains overlapping ranges, the
  # returned ranges will be ordered and coalesced.
  #
  #   Net::IMAP::SequenceSet["2,5:9,6,*,12:11"].elements
  #   #=> [2, 5..9, 11..12, :*]
  #
  # Related: #each_element, #ranges, #numbers
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#851
  def elements; end

  # Returns true if the set contains no elements
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#565
  def empty?; end

  # Returns an array of ranges and integers and <tt>:*</tt>.
  #
  # The entries are in the same order they appear in #string, with no
  # sorting, deduplication, or coalescing.  When #string is in its
  # normalized form, this will return the same result as #elements.
  # This is useful when the given order is significant, for example in a
  # ESEARCH response to IMAP#sort.
  #
  # Related: #each_entry, #elements
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#833
  def entries; end

  # :call-seq: eql?(other) -> true or false
  #
  # Hash equality requires the same encoded #string representation.
  #
  #   Net::IMAP::SequenceSet["1:3"]  .eql? Net::IMAP::SequenceSet["1:3"]
  #   #=> true
  #   Net::IMAP::SequenceSet["1,2,3"].eql? Net::IMAP::SequenceSet["1:3"]
  #   #=> false
  #   Net::IMAP::SequenceSet["1,3"]  .eql? Net::IMAP::SequenceSet["3,1"]
  #   #=> false
  #   Net::IMAP::SequenceSet["9,1:*"].eql? Net::IMAP::SequenceSet["1:*"]
  #   #=> false
  #
  # Related: #==, #normalize
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#460
  def eql?(other); end

  # Returns the index of +number+ in the set, or +nil+ if +number+ isn't in
  # the set.
  #
  # Related: #[]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1000
  def find_index(number); end

  # Freezes and returns the set.  A frozen SequenceSet is Ractor-safe.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#418
  def freeze; end

  # Returns true if the set contains every possible element.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#568
  def full?; end

  # See #eql?
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#463
  def hash; end

  # Returns +true+ when a given number or range is in +self+, and +false+
  # otherwise.  Returns +false+ unless +number+ is an Integer, Range, or
  # <tt>*</tt>.
  #
  #     set = Net::IMAP::SequenceSet["5:10,100,111:115"]
  #     set.include? 1      #=> false
  #     set.include? 5..10  #=> true
  #     set.include? 11..20 #=> false
  #     set.include? 100    #=> true
  #     set.include? 6      #=> true, covered by "5:10"
  #     set.include? 4..9   #=> true, covered by "5:10"
  #     set.include? "4:9"  #=> true, strings are parsed
  #     set.include? 4..9   #=> false, intersection is not sufficient
  #     set.include? "*"    #=> false, use #limit to re-interpret "*"
  #     set.include? -1     #=> false, -1 is interpreted as "*"
  #
  #     set = Net::IMAP::SequenceSet["5:10,100,111:*"]
  #     set.include? :*     #=> true
  #     set.include? "*"    #=> true
  #     set.include? -1     #=> true
  #     set.include? 200..  #=> true
  #     set.include? 100..  #=> false
  #
  # Related: #include_star?, #cover?, #===
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#509
  def include?(element); end

  # Returns +true+ when the set contains <tt>*</tt>.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#514
  def include_star?; end

  # source://net-imap//lib/net/imap/sequence_set.rb#1214
  def inspect; end

  # Returns +true+ if the set and a given object have any common elements,
  # +false+ otherwise.
  #
  #     Net::IMAP::SequenceSet["5:10"].intersect? "7,9,11" #=> true
  #     Net::IMAP::SequenceSet["5:10"].intersect? "11:33"  #=> false
  #
  # Related: #intersection, #disjoint?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#523
  def intersect?(other); end

  # :call-seq:
  #   self & other        -> sequence set
  #   intersection(other) -> sequence set
  #
  # Returns a new sequence set containing only the numbers common to this
  # set and +other+.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet[1..5] & [2, 4, 6]
  #     #=> Net::IMAP::SequenceSet["2,4"]
  #
  # <tt>(seqset & other)</tt> is equivalent to <tt>(seqset - ~other)</tt>.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#623
  def intersection(other); end

  # Returns a frozen SequenceSet with <tt>*</tt> converted to +max+, numbers
  # and ranges over +max+ removed, and ranges containing +max+ converted to
  # end at +max+.
  #
  #   Net::IMAP::SequenceSet["5,10:22,50"].limit(max: 20).to_s
  #   #=> "5,10:20"
  #
  # <tt>*</tt> is always interpreted as the maximum value.  When the set
  # contains <tt>*</tt>, it will be set equal to the limit.
  #
  #   Net::IMAP::SequenceSet["*"].limit(max: 37)
  #   #=> Net::IMAP::SequenceSet["37"]
  #   Net::IMAP::SequenceSet["5:*"].limit(max: 37)
  #   #=> Net::IMAP::SequenceSet["5:37"]
  #   Net::IMAP::SequenceSet["500:*"].limit(max: 37)
  #   #=> Net::IMAP::SequenceSet["37"]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1141
  def limit(max:); end

  # Removes all members over +max+ and returns self.  If <tt>*</tt> is a
  # member, it will be converted to +max+.
  #
  # Related: #limit
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1154
  def limit!(max:); end

  # :call-seq: max(star: :*) => integer or star or nil
  #
  # Returns the maximum value in +self+, +star+ when the set includes
  # <tt>*</tt>, or +nil+ when the set is empty.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#543
  def max(star: T.unsafe(nil)); end

  # Returns +true+ when a given number or range is in +self+, and +false+
  # otherwise.  Returns +false+ unless +number+ is an Integer, Range, or
  # <tt>*</tt>.
  #
  #     set = Net::IMAP::SequenceSet["5:10,100,111:115"]
  #     set.include? 1      #=> false
  #     set.include? 5..10  #=> true
  #     set.include? 11..20 #=> false
  #     set.include? 100    #=> true
  #     set.include? 6      #=> true, covered by "5:10"
  #     set.include? 4..9   #=> true, covered by "5:10"
  #     set.include? "4:9"  #=> true, strings are parsed
  #     set.include? 4..9   #=> false, intersection is not sufficient
  #     set.include? "*"    #=> false, use #limit to re-interpret "*"
  #     set.include? -1     #=> false, -1 is interpreted as "*"
  #
  #     set = Net::IMAP::SequenceSet["5:10,100,111:*"]
  #     set.include? :*     #=> true
  #     set.include? "*"    #=> true
  #     set.include? -1     #=> true
  #     set.include? 200..  #=> true
  #     set.include? 100..  #=> false
  #
  # Related: #include_star?, #cover?, #===
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#509
  def member?(element); end

  # Merges all of the elements that appear in any of the +inputs+ into the
  # set, and returns +self+.
  #
  # The +inputs+ may be any objects that would be accepted by ::new:
  # non-zero 32 bit unsigned integers, ranges, <tt>sequence-set</tt>
  # formatted strings, other sequence sets, or enumerables containing any of
  # these.
  #
  # #string will be regenerated after all inputs have been merged.
  #
  # Related: #add, #add?, #union
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#805
  def merge(*inputs); end

  # :call-seq: min(star: :*) => integer or star or nil
  #
  # Returns the minimum value in +self+, +star+ when the only value in the
  # set is <tt>*</tt>, or +nil+ when the set is empty.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#551
  def min(star: T.unsafe(nil)); end

  # :call-seq: minmax(star: :*) => nil or [integer, integer or star]
  #
  # Returns a 2-element array containing the minimum and maximum numbers in
  # +self+, or +nil+ when the set is empty.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#559
  def minmax(star: T.unsafe(nil)); end

  # Returns a new SequenceSet with a normalized string representation.
  #
  # The returned set's #string is sorted and deduplicated.  Adjacent or
  # overlapping elements will be merged into a single larger range.
  #
  #   Net::IMAP::SequenceSet["1:5,3:7,10:9,10:11"].normalize
  #   #=> Net::IMAP::SequenceSet["1:7,9:11"]
  #
  # Related: #normalize!, #normalized_string
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1187
  def normalize; end

  # Resets #string to be sorted, deduplicated, and coalesced.  Returns
  # +self+.
  #
  # Related: #normalize, #normalized_string
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1197
  def normalize!; end

  # Returns a normalized +sequence-set+ string representation, sorted
  # and deduplicated.  Adjacent or overlapping elements will be merged into
  # a single larger range.  Returns +nil+ when the set is empty.
  #
  #   Net::IMAP::SequenceSet["1:5,3:7,10:9,10:11"].normalized_string
  #   #=> "1:7,9:11"
  #
  # Related: #normalize!, #normalize
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1210
  def normalized_string; end

  # Returns a sorted array of all of the number values in the sequence set.
  #
  # The returned numbers are sorted and de-duplicated, even when the input
  # #string is not.  See #normalize.
  #
  #   Net::IMAP::SequenceSet["2,5:9,6,12:11"].numbers
  #   #=> [2, 5, 6, 7, 8, 9, 11, 12]
  #
  # If the set contains a <tt>*</tt>, RangeError is raised.  See #limit.
  #
  #   Net::IMAP::SequenceSet["10000:*"].numbers
  #   #!> RangeError
  #
  # *WARNING:* Even excluding sets with <tt>*</tt>, an enormous result can
  # easily be created.  An array with over 4 billion integers could be
  # returned, requiring up to 32GiB of memory on a 64-bit architecture.
  #
  #   Net::IMAP::SequenceSet[10000..2**32-1].numbers
  #   # ...probably freezes the process for a while...
  #   #!> NoMemoryError (probably)
  #
  # For safety, consider using #limit or #intersection to set an upper
  # bound.  Alternatively, use #each_element, #each_range, or even
  # #each_number to avoid allocation of a result array.
  #
  # Related: #elements, #ranges, #to_set
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#900
  def numbers; end

  # Returns +true+ if the set and a given object have any common elements,
  # +false+ otherwise.
  #
  #     Net::IMAP::SequenceSet["5:10"].intersect? "7,9,11" #=> true
  #     Net::IMAP::SequenceSet["5:10"].intersect? "11:33"  #=> false
  #
  # Related: #intersection, #disjoint?
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#523
  def overlap?(other); end

  # Returns an array of ranges
  #
  # The returned elements are sorted and coalesced, even when the input
  # #string is not.  <tt>*</tt> will sort last.  See #normalize.
  #
  # <tt>*</tt> translates to an endless range.  By itself, <tt>*</tt>
  # translates to <tt>:*..</tt>.  Use #limit to set <tt>*</tt> to a maximum
  # value.
  #
  # The returned ranges will be ordered and coalesced, even when the input
  # #string is not.  <tt>*</tt> will sort last.  See #normalize.
  #
  #   Net::IMAP::SequenceSet["2,5:9,6,*,12:11"].ranges
  #   #=> [2..2, 5..9, 11..12, :*..]
  #   Net::IMAP::SequenceSet["123,999:*,456:789"].ranges
  #   #=> [123..123, 456..789, 999..]
  #
  # Related: #each_range, #elements, #numbers, #to_set
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#872
  def ranges; end

  # Replace the contents of the set with the contents of +other+ and returns
  # +self+.
  #
  # +other+ may be another SequenceSet, or it may be an IMAP +sequence-set+
  # string, a number, a range, <tt>*</tt>, or an enumerable of these.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#358
  def replace(other); end

  # Unstable API: for internal use only (Net::IMAP#send_data)
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1234
  def send_data(imap, tag); end

  # Returns the count of #numbers in the set.
  #
  # If <tt>*</tt> and <tt>2**32 - 1</tt> (the maximum 32-bit unsigned
  # integer value) are both in the set, they will only be counted once.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#989
  def size; end

  # :call-seq:
  #    seqset[index]         -> integer or :* or nil
  #    slice(index)          -> integer or :* or nil
  #    seqset[start, length] -> sequence set or nil
  #    slice(start, length)  -> sequence set or nil
  #    seqset[range]         -> sequence set or nil
  #    slice(range)          -> sequence set or nil
  #
  # Returns a number or a subset from +self+, without modifying the set.
  #
  # When an Integer argument +index+ is given, the number at offset +index+
  # is returned:
  #
  #     set = Net::IMAP::SequenceSet["10:15,20:23,26"]
  #     set[0]   #=> 10
  #     set[5]   #=> 15
  #     set[10]  #=> 26
  #
  # If +index+ is negative, it counts relative to the end of +self+:
  #     set = Net::IMAP::SequenceSet["10:15,20:23,26"]
  #     set[-1]  #=> 26
  #     set[-3]  #=> 22
  #     set[-6]  #=> 15
  #
  # If +index+ is out of range, +nil+ is returned.
  #
  #     set = Net::IMAP::SequenceSet["10:15,20:23,26"]
  #     set[11]  #=> nil
  #     set[-12] #=> nil
  #
  # The result is based on the normalized set—sorted and de-duplicated—not
  # on the assigned value of #string.
  #
  #     set = Net::IMAP::SequenceSet["12,20:23,11:16,21"]
  #     set[0]   #=> 11
  #     set[-1]  #=> 23
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1088
  def slice(index, length = T.unsafe(nil)); end

  # :call-seq:
  #    slice!(index)          -> integer or :* or nil
  #    slice!(start, length)  -> sequence set or nil
  #    slice!(range)          -> sequence set or nil
  #
  # Deletes a number or consecutive numbers from the set, indicated by the
  # given +index+, +start+ and +length+, or +range+ of offsets.  Returns the
  # number or sequence set that was removed, or +nil+ if nothing was
  # removed.  Arguments are interpreted the same as for #slice or #[].
  #
  # #string will be regenerated after deletion.
  #
  # Related: #slice, #delete_at, #delete, #delete?, #subtract, #difference
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#789
  def slice!(index, length = T.unsafe(nil)); end

  # Returns the \IMAP +sequence-set+ string representation, or +nil+ when
  # the set is empty.  Note that an empty set is invalid in the \IMAP
  # syntax.
  #
  # Use #valid_string to raise an exception when the set is empty, or #to_s
  # to return an empty string.
  #
  # If the set was created from a single string, it is not normalized.  If
  # the set is updated the string will be normalized.
  #
  # Related: #valid_string, #normalized_string, #to_s
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#390
  def string; end

  # Assigns a new string to #string and resets #elements to match.  It
  # cannot be set to an empty string—assign +nil+ or use #clear instead.
  # The string is validated but not normalized.
  #
  # Use #add or #merge to add a string to an existing set.
  #
  # Related: #replace, #clear
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#399
  def string=(str); end

  # Removes all of the elements that appear in any of the given +objects+
  # from the set, and returns +self+.
  #
  # The +objects+ may be any objects that would be accepted by ::new:
  # non-zero 32 bit unsigned integers, ranges, <tt>sequence-set</tt>
  # formatted strings, other sequence sets, or enumerables containing any of
  # these.
  #
  # Related: #difference
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#819
  def subtract(*objects); end

  # Returns an array of ranges and integers and <tt>:*</tt>.
  #
  # The returned elements are sorted and coalesced, even when the input
  # #string is not.  <tt>*</tt> will sort last.  See #normalize.
  #
  # By itself, <tt>*</tt> translates to <tt>:*</tt>.  A range containing
  # <tt>*</tt> translates to an endless range.  Use #limit to translate both
  # cases to a maximum value.
  #
  # If the original input was unordered or contains overlapping ranges, the
  # returned ranges will be ordered and coalesced.
  #
  #   Net::IMAP::SequenceSet["2,5:9,6,*,12:11"].elements
  #   #=> [2, 5..9, 11..12, :*]
  #
  # Related: #each_element, #ranges, #numbers
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#851
  def to_a; end

  # Returns the \IMAP +sequence-set+ string representation, or an empty
  # string when the set is empty.  Note that an empty set is invalid in the
  # \IMAP syntax.
  #
  # Related: #valid_string, #normalized_string, #to_s
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#415
  def to_s; end

  # Returns self
  def to_sequence_set; end

  # Returns a Set with all of the #numbers in the sequence set.
  #
  # If the set contains a <tt>*</tt>, RangeError will be raised.
  #
  # See #numbers for the warning about very large sets.
  #
  # Related: #elements, #ranges, #numbers
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#983
  def to_set; end

  # :call-seq:
  #   self + other -> sequence set
  #   self | other -> sequence set
  #   union(other) -> sequence set
  #
  # Returns a new sequence set that has every number in the +other+ object
  # added.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet["1:5"] | 2 | [4..6, 99]
  #     #=> Net::IMAP::SequenceSet["1:6,99"]
  #
  # Related: #add, #merge
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#586
  def union(other); end

  # Returns false when the set is empty.
  #
  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#562
  def valid?; end

  # Returns the \IMAP +sequence-set+ string representation, or raises a
  # DataFormatError when the set is empty.
  #
  # Use #string to return +nil+ or #to_s to return an empty string without
  # error.
  #
  # Related: #string, #normalized_string, #to_s
  #
  # @raise [DataFormatError]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#374
  def valid_string; end

  # Unstable API: currently for internal use only (Net::IMAP#validate_data)
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1228
  def validate; end

  # :call-seq:
  #   self ^ other -> sequence set
  #   xor(other)   -> sequence set
  #
  # Returns a new sequence set containing numbers that are exclusive between
  # this set and +other+.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet[1..5] ^ [2, 4, 6]
  #     #=> Net::IMAP::SequenceSet["1,3,5:6"]
  #
  # <tt>(seqset ^ other)</tt> is equivalent to <tt>((seqset | other) -
  # (seqset & other))</tt>.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#644
  def xor(other); end

  # :call-seq:
  #   self + other -> sequence set
  #   self | other -> sequence set
  #   union(other) -> sequence set
  #
  # Returns a new sequence set that has every number in the +other+ object
  # added.
  #
  # +other+ may be any object that would be accepted by ::new: a non-zero 32
  # bit unsigned integer, range, <tt>sequence-set</tt> formatted string,
  # another sequence set, or an enumerable containing any of these.
  #
  #     Net::IMAP::SequenceSet["1:5"] | 2 | [4..6, 99]
  #     #=> Net::IMAP::SequenceSet["1:6,99"]
  #
  # Related: #add, #merge
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#586
  def |(other); end

  # :call-seq:
  #   ~ self     -> sequence set
  #   complement -> sequence set
  #
  # Returns the complement of self, a SequenceSet which contains all numbers
  # _except_ for those in this set.
  #
  #     ~Net::IMAP::SequenceSet.full  #=> Net::IMAP::SequenceSet.empty
  #     ~Net::IMAP::SequenceSet.empty #=> Net::IMAP::SequenceSet.full
  #     ~Net::IMAP::SequenceSet["1:5,100:222"]
  #     #=> Net::IMAP::SequenceSet["6:99,223:*"]
  #     ~Net::IMAP::SequenceSet["6:99,223:*"]
  #     #=> Net::IMAP::SequenceSet["1:5,100:222"]
  #
  # Related: #complement!
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#662
  def ~; end

  protected

  # source://net-imap//lib/net/imap/sequence_set.rb#1240
  def tuples; end

  private

  # source://net-imap//lib/net/imap/sequence_set.rb#1011
  def each_tuple_with_index; end

  # source://net-imap//lib/net/imap/sequence_set.rb#1304
  def from_tuple_int(num); end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1313
  def include_tuple?(_arg0); end

  # frozen clones are shallow copied
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1247
  def initialize_clone(other); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1251
  def initialize_dup(other); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1257
  def input_to_tuple(obj); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1268
  def input_to_tuples(obj); end

  # unlike SequenceSet#try_convert, this returns an Integer, Range,
  # String, Set, Array, or... any type of object.
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1285
  def input_try_convert(input); end

  # @return [Boolean]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1315
  def intersect_tuple?(_arg0); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1409
  def nz_number(num); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1404
  def range_gte_to(num); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1293
  def range_to_tuple(range); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1244
  def remain_frozen(set); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1020
  def reverse_each_tuple_with_index; end

  # @raise [ArgumentError]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1099
  def slice_length(start, length); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1107
  def slice_range(range); end

  # @raise [DataFormatError]
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1308
  def str_to_tuple(str); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1307
  def str_to_tuples(str); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1303
  def to_tuple_int(obj); end

  # --|=====| |=====new tuple=====|                 append
  #   ?????????-|=====new tuple=====|-|===lower===|-- insert
  #
  #             |=====new tuple=====|
  #   ---------??=======lower=======??--------------- noop
  #
  #   ---------??===lower==|--|==|                    join remaining
  #   ---------??===lower==|--|==|----|===upper===|-- join until upper
  #   ---------??===lower==|--|==|--|=====upper===|-- join to upper
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1333
  def tuple_add(tuple); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1342
  def tuple_coalesce(lower, lower_idx, min, max); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1400
  def tuple_gte_with_index(num); end

  # |====tuple================|
  # --|====|                               no more       1. noop
  # --|====|---------------------------|====lower====|-- 2. noop
  # -------|======lower================|---------------- 3. split
  # --------|=====lower================|---------------- 4. trim beginning
  #
  # -------|======lower====????????????----------------- trim lower
  # --------|=====lower====????????????----------------- delete lower
  #
  # -------??=====lower===============|----------------- 5. trim/delete one
  # -------??=====lower====|--|====|       no more       6. delete rest
  # -------??=====lower====|--|====|---|====upper====|-- 7. delete until
  # -------??=====lower====|--|====|--|=====upper====|-- 8. delete and trim
  #
  # source://net-imap//lib/net/imap/sequence_set.rb#1369
  def tuple_subtract(tuple); end

  # source://net-imap//lib/net/imap/sequence_set.rb#932
  def tuple_to_entry(_arg0); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1306
  def tuple_to_str(tuple); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1379
  def tuple_trim_or_split(lower, idx, tmin, tmax); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1320
  def tuples_add(tuples); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1321
  def tuples_subtract(tuples); end

  # source://net-imap//lib/net/imap/sequence_set.rb#1386
  def tuples_trim_or_delete(lower, lower_idx, tmin, tmax); end

  class << self
    # :call-seq:
    #   SequenceSet[*values] -> valid frozen sequence set
    #
    # Returns a frozen SequenceSet, constructed from +values+.
    #
    # An empty SequenceSet is invalid and will raise a DataFormatError.
    #
    # Use ::new to create a mutable or empty SequenceSet.
    #
    # source://net-imap//lib/net/imap/sequence_set.rb#305
    def [](first, *rest); end

    # Returns a frozen empty set singleton.  Note that valid \IMAP sequence
    # sets cannot be empty, so this set is _invalid_.
    #
    # source://net-imap//lib/net/imap/sequence_set.rb#336
    def empty; end

    # Returns a frozen full set singleton: <tt>"1:*"</tt>
    #
    # source://net-imap//lib/net/imap/sequence_set.rb#339
    def full; end

    # :call-seq:
    #   SequenceSet.try_convert(obj) -> sequence set or nil
    #
    # If +obj+ is a SequenceSet, returns +obj+.  If +obj+ responds_to
    # +to_sequence_set+, calls +obj.to_sequence_set+ and returns the result.
    # Otherwise returns +nil+.
    #
    # If +obj.to_sequence_set+ doesn't return a SequenceSet, an exception is
    # raised.
    #
    # @raise [DataFormatError]
    #
    # source://net-imap//lib/net/imap/sequence_set.rb#326
    def try_convert(obj); end
  end
end

# source://net-imap//lib/net/imap/sequence_set.rb#291
Net::IMAP::SequenceSet::COERCIBLE = T.let(T.unsafe(nil), Proc)

# intentionally defined after the class implementation
#
# source://net-imap//lib/net/imap/sequence_set.rb#1420
Net::IMAP::SequenceSet::EMPTY = T.let(T.unsafe(nil), Net::IMAP::SequenceSet)

# source://net-imap//lib/net/imap/sequence_set.rb#292
Net::IMAP::SequenceSet::ENUMABLE = T.let(T.unsafe(nil), Proc)

# source://net-imap//lib/net/imap/sequence_set.rb#1421
Net::IMAP::SequenceSet::FULL = T.let(T.unsafe(nil), Net::IMAP::SequenceSet)

# valid inputs for "*"
#
# source://net-imap//lib/net/imap/sequence_set.rb#288
Net::IMAP::SequenceSet::STARS = T.let(T.unsafe(nil), Array)

# represents "*" internally, to simplify sorting (etc)
#
# source://net-imap//lib/net/imap/sequence_set.rb#284
Net::IMAP::SequenceSet::STAR_INT = T.let(T.unsafe(nil), Integer)

# The largest possible non-zero unsigned 32-bit integer
#
# source://net-imap//lib/net/imap/sequence_set.rb#281
Net::IMAP::SequenceSet::UINT32_MAX = T.let(T.unsafe(nil), Integer)

# source://net-imap//lib/net/imap/command_data.rb#273
module Net::IMAP::StringFormatter
  private

  # coerces non-nil using +to_s+
  #
  # source://net-imap//lib/net/imap/command_data.rb#300
  def nstring(str); end

  # coerces using +to_s+
  #
  # source://net-imap//lib/net/imap/command_data.rb#290
  def string(str); end

  # Allows nil, symbols, and strings
  #
  # source://net-imap//lib/net/imap/command_data.rb#285
  def valid_nstring?(str); end

  # Allows symbols in addition to strings
  #
  # source://net-imap//lib/net/imap/command_data.rb#280
  def valid_string?(str); end

  class << self
    # coerces non-nil using +to_s+
    #
    # source://net-imap//lib/net/imap/command_data.rb#300
    def nstring(str); end

    # coerces using +to_s+
    #
    # source://net-imap//lib/net/imap/command_data.rb#290
    def string(str); end

    # Allows nil, symbols, and strings
    #
    # @return [Boolean]
    #
    # source://net-imap//lib/net/imap/command_data.rb#285
    def valid_nstring?(str); end

    # Allows symbols in addition to strings
    #
    # @return [Boolean]
    #
    # source://net-imap//lib/net/imap/command_data.rb#280
    def valid_string?(str); end
  end
end

# source://net-imap//lib/net/imap/command_data.rb#275
Net::IMAP::StringFormatter::LITERAL_REGEX = T.let(T.unsafe(nil), Regexp)

# Regexps and utility methods for implementing stringprep profiles.  The
# \StringPrep algorithm is defined by
# {RFC-3454}[https://www.rfc-editor.org/rfc/rfc3454.html].  Each
# codepoint table defined in the RFC-3454 appendices is matched by a Regexp
# defined in this module.
#
# source://net-imap//lib/net/imap/stringprep.rb#11
module Net::IMAP::StringPrep
  private

  # Checks that +string+ obeys all of the "Bidirectional Characters"
  # requirements in RFC-3454, §6:
  #
  # * The characters in \StringPrep\[\"C.8\"] MUST be prohibited
  # * If a string contains any RandALCat character, the string MUST NOT
  #   contain any LCat character.
  # * If a string contains any RandALCat character, a RandALCat
  #   character MUST be the first character of the string, and a
  #   RandALCat character MUST be the last character of the string.
  #
  # This is usually combined with #check_prohibited!, so table "C.8" is only
  # checked when <tt>c_8: true</tt>.
  #
  # Raises either ProhibitedCodepoint or BidiStringError unless all
  # requirements are met.  +profile+ is an optional string which will be
  # added to any exception that is raised (it does not affect behavior).
  #
  # source://net-imap//lib/net/imap/stringprep.rb#144
  def check_bidi!(string, c_8: T.unsafe(nil), profile: T.unsafe(nil)); end

  # Checks +string+ for any codepoint in +tables+. Raises a
  # ProhibitedCodepoint describing the first matching table.
  #
  # Also checks bidirectional characters, when <tt>bidi: true</tt>, which may
  # raise a BidiStringError.
  #
  # +profile+ is an optional string which will be added to any exception that
  # is raised (it does not affect behavior).
  #
  # source://net-imap//lib/net/imap/stringprep.rb#104
  def check_prohibited!(string, *tables, bidi: T.unsafe(nil), unassigned: T.unsafe(nil), stored: T.unsafe(nil), profile: T.unsafe(nil)); end

  # source://net-imap//lib/net/imap/stringprep.rb#88
  def map_tables!(string, *tables); end

  # >>>
  #   1. Map -- For each character in the input, check if it has a mapping
  #      and, if so, replace it with its mapping.  This is described in
  #      section 3.
  #
  #   2. Normalize -- Possibly normalize the result of step 1 using Unicode
  #      normalization.  This is described in section 4.
  #
  #   3. Prohibit -- Check for any characters that are not allowed in the
  #      output.  If any are found, return an error.  This is described in
  #      section 5.
  #
  #   4. Check bidi -- Possibly check for right-to-left characters, and if
  #      any are found, make sure that the whole string satisfies the
  #      requirements for bidirectional strings.  If the string does not
  #      satisfy the requirements for bidirectional strings, return an
  #      error.  This is described in section 6.
  #
  #   The above steps MUST be performed in the order given to comply with
  #   this specification.
  #
  # source://net-imap//lib/net/imap/stringprep.rb#76
  def stringprep(string, maps:, normalization:, prohibited:, **opts); end

  class << self
    # Returns a Regexp matching the given +table+ name.
    #
    # source://net-imap//lib/net/imap/stringprep.rb#49
    def [](table); end

    # Checks that +string+ obeys all of the "Bidirectional Characters"
    # requirements in RFC-3454, §6:
    #
    # * The characters in \StringPrep\[\"C.8\"] MUST be prohibited
    # * If a string contains any RandALCat character, the string MUST NOT
    #   contain any LCat character.
    # * If a string contains any RandALCat character, a RandALCat
    #   character MUST be the first character of the string, and a
    #   RandALCat character MUST be the last character of the string.
    #
    # This is usually combined with #check_prohibited!, so table "C.8" is only
    # checked when <tt>c_8: true</tt>.
    #
    # Raises either ProhibitedCodepoint or BidiStringError unless all
    # requirements are met.  +profile+ is an optional string which will be
    # added to any exception that is raised (it does not affect behavior).
    #
    # source://net-imap//lib/net/imap/stringprep.rb#144
    def check_bidi!(string, c_8: T.unsafe(nil), profile: T.unsafe(nil)); end

    # Checks +string+ for any codepoint in +tables+. Raises a
    # ProhibitedCodepoint describing the first matching table.
    #
    # Also checks bidirectional characters, when <tt>bidi: true</tt>, which may
    # raise a BidiStringError.
    #
    # +profile+ is an optional string which will be added to any exception that
    # is raised (it does not affect behavior).
    #
    # source://net-imap//lib/net/imap/stringprep.rb#104
    def check_prohibited!(string, *tables, bidi: T.unsafe(nil), unassigned: T.unsafe(nil), stored: T.unsafe(nil), profile: T.unsafe(nil)); end

    # source://net-imap//lib/net/imap/stringprep.rb#88
    def map_tables!(string, *tables); end

    # >>>
    #   1. Map -- For each character in the input, check if it has a mapping
    #      and, if so, replace it with its mapping.  This is described in
    #      section 3.
    #
    #   2. Normalize -- Possibly normalize the result of step 1 using Unicode
    #      normalization.  This is described in section 4.
    #
    #   3. Prohibit -- Check for any characters that are not allowed in the
    #      output.  If any are found, return an error.  This is described in
    #      section 5.
    #
    #   4. Check bidi -- Possibly check for right-to-left characters, and if
    #      any are found, make sure that the whole string satisfies the
    #      requirements for bidirectional strings.  If the string does not
    #      satisfy the requirements for bidirectional strings, return an
    #      error.  This is described in section 6.
    #
    #   The above steps MUST be performed in the order given to comply with
    #   this specification.
    #
    # source://net-imap//lib/net/imap/stringprep.rb#76
    def stringprep(string, maps:, normalization:, prohibited:, **opts); end
  end
end

# StringPrepError raised when +string+ contains bidirectional characters
# which violate the StringPrep requirements.
#
# source://net-imap//lib/net/imap/stringprep.rb#45
class Net::IMAP::StringPrep::BidiStringError < ::Net::IMAP::StringPrep::StringPrepError; end

# Defined in RFC3491[https://tools.ietf.org/html/rfc3491], the +nameprep+
# profile of "Stringprep" is:
# >>>
#   used by the IDNA protocol for preparing domain names; it is not
#   designed for any other purpose.  It is explicitly not designed for
#   processing arbitrary free text and SHOULD NOT be used for that
#   purpose.
#
#   ...
#
#   This profile specifies prohibiting using the following tables...:
#
#   - C.1.2 (Non-ASCII space characters)
#   - C.2.2 (Non-ASCII control characters)
#   - C.3 (Private use characters)
#   - C.4 (Non-character code points)
#   - C.5 (Surrogate codes)
#   - C.6 (Inappropriate for plain text)
#   - C.7 (Inappropriate for canonical representation)
#   - C.8 (Change display properties are deprecated)
#   - C.9 (Tagging characters)
#
#   IMPORTANT NOTE: This profile MUST be used with the IDNA protocol.
#   The IDNA protocol has additional prohibitions that are checked
#   outside of this profile.
#
# source://net-imap//lib/net/imap/stringprep/nameprep.rb#32
module Net::IMAP::StringPrep::NamePrep
  private

  # source://net-imap//lib/net/imap/stringprep/nameprep.rb#54
  def nameprep(string, **opts); end

  class << self
    # source://net-imap//lib/net/imap/stringprep/nameprep.rb#54
    def nameprep(string, **opts); end
  end
end

# From RFC3491[https://www.rfc-editor.org/rfc/rfc3491.html] §6
#
# source://net-imap//lib/net/imap/stringprep/nameprep.rb#50
Net::IMAP::StringPrep::NamePrep::CHECK_BIDI = T.let(T.unsafe(nil), TrueClass)

# From RFC3491[https://www.rfc-editor.org/rfc/rfc3491.html] §3
#
# source://net-imap//lib/net/imap/stringprep/nameprep.rb#41
Net::IMAP::StringPrep::NamePrep::MAPPING_TABLES = T.let(T.unsafe(nil), Array)

# From RFC3491[https://www.rfc-editor.org/rfc/rfc3491.html] §4
#
# source://net-imap//lib/net/imap/stringprep/nameprep.rb#44
Net::IMAP::StringPrep::NamePrep::NORMALIZATION = T.let(T.unsafe(nil), Symbol)

# From RFC3491[https://www.rfc-editor.org/rfc/rfc3491.html] §5
#
# source://net-imap//lib/net/imap/stringprep/nameprep.rb#47
Net::IMAP::StringPrep::NamePrep::PROHIBITED_TABLES = T.let(T.unsafe(nil), Array)

# From RFC3491[https://www.rfc-editor.org/rfc/rfc3491.html] §10
#
# source://net-imap//lib/net/imap/stringprep/nameprep.rb#35
Net::IMAP::StringPrep::NamePrep::STRINGPREP_PROFILE = T.let(T.unsafe(nil), String)

# From RFC3491[https://www.rfc-editor.org/rfc/rfc3491.html] §2
#
# source://net-imap//lib/net/imap/stringprep/nameprep.rb#38
Net::IMAP::StringPrep::NamePrep::UNASSIGNED_TABLE = T.let(T.unsafe(nil), String)

# StringPrepError raised when +string+ contains a codepoint prohibited by
# +table+.
#
# source://net-imap//lib/net/imap/stringprep.rb#31
class Net::IMAP::StringPrep::ProhibitedCodepoint < ::Net::IMAP::StringPrep::StringPrepError
  # @return [ProhibitedCodepoint] a new instance of ProhibitedCodepoint
  #
  # source://net-imap//lib/net/imap/stringprep.rb#34
  def initialize(table, *args, **kwargs); end

  # Returns the value of attribute table.
  #
  # source://net-imap//lib/net/imap/stringprep.rb#32
  def table; end
end

# SASLprep#saslprep can be used to prepare a string according to [RFC4013].
#
# \SASLprep maps characters three ways: to nothing, to space, and Unicode
# normalization form KC.  \SASLprep prohibits codepoints from nearly all
# standard StringPrep tables (RFC3454, Appendix "C"), and uses
# \StringPrep's standard bidirectional characters requirements (Appendix
# "D").  \SASLprep also uses \StringPrep's definition of "Unassigned"
# codepoints (Appendix "A").
#
# source://net-imap//lib/net/imap/stringprep/saslprep.rb#15
module Net::IMAP::StringPrep::SASLprep
  private

  # Prepares a UTF-8 +string+ for comparison, using the \SASLprep profile
  # RFC4013 of the StringPrep algorithm RFC3454.
  #
  # By default, prohibited strings will return +nil+.  When +exception+ is
  # +true+, a StringPrepError describing the violation will be raised.
  #
  # When +stored+ is +true+, "unassigned" codepoints will be prohibited.
  # For \StringPrep and the \SASLprep profile, "unassigned" refers to
  # Unicode 3.2, and not later versions.  See RFC3454 §7 for more
  # information.
  #
  # source://net-imap//lib/net/imap/stringprep/saslprep.rb#42
  def saslprep(str, stored: T.unsafe(nil), exception: T.unsafe(nil)); end

  class << self
    # Prepares a UTF-8 +string+ for comparison, using the \SASLprep profile
    # RFC4013 of the StringPrep algorithm RFC3454.
    #
    # By default, prohibited strings will return +nil+.  When +exception+ is
    # +true+, a StringPrepError describing the violation will be raised.
    #
    # When +stored+ is +true+, "unassigned" codepoints will be prohibited.
    # For \StringPrep and the \SASLprep profile, "unassigned" refers to
    # Unicode 3.2, and not later versions.  See RFC3454 §7 for more
    # information.
    #
    # source://net-imap//lib/net/imap/stringprep/saslprep.rb#42
    def saslprep(str, stored: T.unsafe(nil), exception: T.unsafe(nil)); end
  end
end

# Used to short-circuit strings that don't need preparation.
#
# source://net-imap//lib/net/imap/stringprep/saslprep.rb#18
Net::IMAP::StringPrep::SASLprep::ASCII_NO_CTRLS = T.let(T.unsafe(nil), Regexp)

# Bidirectional Characters [StringPrep, §6]
#
# A Regexp for strings that don't satisfy StringPrep's Bidirectional
# Characters rules.
#
# Equal to StringPrep::Tables::BIDI_FAILURE.
# Redefined here to avoid loading StringPrep::Tables unless necessary.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#79
Net::IMAP::StringPrep::SASLprep::BIDI_FAILURE = T.let(T.unsafe(nil), Regexp)

# RFC4013 §2.1 Mapping - mapped to nothing
# >>>
#   the "commonly mapped to nothing" characters
#   (\StringPrep\[\"B.1\"]) that can be mapped to nothing.
#
# Equal to \StringPrep\[\"B.1\"].
# Redefined here to avoid loading StringPrep::Tables unless necessary.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#27
Net::IMAP::StringPrep::SASLprep::MAP_TO_NOTHING = T.let(T.unsafe(nil), Regexp)

# RFC4013 §2.1 Mapping - mapped to space
# >>>
#   non-ASCII space characters (\StringPrep\[\"C.1.2\"]) that can
#   be mapped to SPACE (U+0020)
#
# Equal to \StringPrep\[\"C.1.2\"].
# Redefined here to avoid loading StringPrep::Tables unless necessary.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#18
Net::IMAP::StringPrep::SASLprep::MAP_TO_SPACE = T.let(T.unsafe(nil), Regexp)

# A Regexp matching strings prohibited by RFC4013 §2.3 and §2.4.
#
# This combines PROHIBITED_OUTPUT and BIDI_FAILURE.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#84
Net::IMAP::StringPrep::SASLprep::PROHIBITED = T.let(T.unsafe(nil), Regexp)

# A Regexp matching codepoints prohibited by RFC4013 §2.3.
#
# This combines all of the TABLES_PROHIBITED tables.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#54
Net::IMAP::StringPrep::SASLprep::PROHIBITED_OUTPUT = T.let(T.unsafe(nil), Regexp)

# A Regexp matching codepoints prohibited by RFC4013 §2.3 and §2.5.
#
# This combines PROHIBITED_OUTPUT and UNASSIGNED.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#68
Net::IMAP::StringPrep::SASLprep::PROHIBITED_OUTPUT_STORED = T.let(T.unsafe(nil), Regexp)

# A Regexp matching strings prohibited by RFC4013 §2.3, §2.4, and §2.5.
#
# This combines PROHIBITED_OUTPUT_STORED and BIDI_FAILURE.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#91
Net::IMAP::StringPrep::SASLprep::PROHIBITED_STORED = T.let(T.unsafe(nil), Regexp)

# RFC4013 §2.3 Prohibited Output
# >>>
# * Non-ASCII space characters — \StringPrep\[\"C.1.2\"]
# * ASCII control characters — \StringPrep\[\"C.2.1\"]
# * Non-ASCII control characters — \StringPrep\[\"C.2.2\"]
# * Private Use characters — \StringPrep\[\"C.3\"]
# * Non-character code points — \StringPrep\[\"C.4\"]
# * Surrogate code points — \StringPrep\[\"C.5\"]
# * Inappropriate for plain text characters — \StringPrep\[\"C.6\"]
# * Inappropriate for canonical representation characters — \StringPrep\[\"C.7\"]
# * Change display properties or deprecated characters — \StringPrep\[\"C.8\"]
# * Tagging characters — \StringPrep\[\"C.9\"]
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#41
Net::IMAP::StringPrep::SASLprep::TABLES_PROHIBITED = T.let(T.unsafe(nil), Array)

# Adds unassigned (by Unicode 3.2) codepoints to TABLES_PROHIBITED.
#
# RFC4013 §2.5 Unassigned Code Points
# >>>
#   This profile specifies the \StringPrep\[\"A.1\"] table as its
#   list of unassigned code points.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#49
Net::IMAP::StringPrep::SASLprep::TABLES_PROHIBITED_STORED = T.let(T.unsafe(nil), Array)

# RFC4013 §2.5 Unassigned Code Points
# >>>
#   This profile specifies the \StringPrep\[\"A.1\"] table as its
#   list of unassigned code points.
#
# Equal to \StringPrep\[\"A.1\"].
# Redefined here to avoid loading StringPrep::Tables unless necessary.
#
# source://net-imap//lib/net/imap/stringprep/saslprep_tables.rb#63
Net::IMAP::StringPrep::SASLprep::UNASSIGNED = T.let(T.unsafe(nil), Regexp)

# ArgumentError raised when +string+ is invalid for the stringprep
# +profile+.
#
# source://net-imap//lib/net/imap/stringprep.rb#19
class Net::IMAP::StringPrep::StringPrepError < ::ArgumentError
  # @return [StringPrepError] a new instance of StringPrepError
  #
  # source://net-imap//lib/net/imap/stringprep.rb#22
  def initialize(*args, string: T.unsafe(nil), profile: T.unsafe(nil)); end

  # Returns the value of attribute profile.
  #
  # source://net-imap//lib/net/imap/stringprep.rb#20
  def profile; end

  # Returns the value of attribute string.
  #
  # source://net-imap//lib/net/imap/stringprep.rb#20
  def string; end
end

# source://net-imap//lib/net/imap/stringprep/tables.rb#9
module Net::IMAP::StringPrep::Tables; end

# source://net-imap//lib/net/imap/stringprep/tables.rb#75
Net::IMAP::StringPrep::Tables::BIDI_DESC_REQ2 = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/stringprep/tables.rb#83
Net::IMAP::StringPrep::Tables::BIDI_DESC_REQ3 = T.let(T.unsafe(nil), String)

# Bidirectional Characters [StringPrep, §6], Requirement 2
# >>>
#   If a string contains any RandALCat character, the string MUST NOT
#   contain any LCat character.
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#81
Net::IMAP::StringPrep::Tables::BIDI_FAILS_REQ2 = T.let(T.unsafe(nil), Regexp)

# Bidirectional Characters [StringPrep, §6], Requirement 3
# >>>
#   If a string contains any RandALCat character, a RandALCat
#   character MUST be the first character of the string, and a
#   RandALCat character MUST be the last character of the string.
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#90
Net::IMAP::StringPrep::Tables::BIDI_FAILS_REQ3 = T.let(T.unsafe(nil), Regexp)

# Bidirectional Characters [StringPrep, §6]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#93
Net::IMAP::StringPrep::Tables::BIDI_FAILURE = T.let(T.unsafe(nil), Regexp)

# Unassigned code points in Unicode 3.2 \StringPrep\[\"A.1\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#12
Net::IMAP::StringPrep::Tables::IN_A_1 = T.let(T.unsafe(nil), Regexp)

# Commonly mapped to nothing \StringPrep\[\"B.1\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#15
Net::IMAP::StringPrep::Tables::IN_B_1 = T.let(T.unsafe(nil), Regexp)

# Mapping for case-folding used with NFKC \StringPrep\[\"B.2\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#18
Net::IMAP::StringPrep::Tables::IN_B_2 = T.let(T.unsafe(nil), Regexp)

# Mapping for case-folding used with no normalization \StringPrep\[\"B.3\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#21
Net::IMAP::StringPrep::Tables::IN_B_3 = T.let(T.unsafe(nil), Regexp)

# ASCII space characters \StringPrep\[\"C.1.1\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#33
Net::IMAP::StringPrep::Tables::IN_C_1_1 = T.let(T.unsafe(nil), Regexp)

# Non-ASCII space characters \StringPrep\[\"C.1.2\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#36
Net::IMAP::StringPrep::Tables::IN_C_1_2 = T.let(T.unsafe(nil), Regexp)

# ASCII control characters \StringPrep\[\"C.2.1\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#39
Net::IMAP::StringPrep::Tables::IN_C_2_1 = T.let(T.unsafe(nil), Regexp)

# Non-ASCII control characters \StringPrep\[\"C.2.2\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#42
Net::IMAP::StringPrep::Tables::IN_C_2_2 = T.let(T.unsafe(nil), Regexp)

# Private use \StringPrep\[\"C.3\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#45
Net::IMAP::StringPrep::Tables::IN_C_3 = T.let(T.unsafe(nil), Regexp)

# Non-character code points \StringPrep\[\"C.4\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#48
Net::IMAP::StringPrep::Tables::IN_C_4 = T.let(T.unsafe(nil), Regexp)

# Surrogate codes \StringPrep\[\"C.5\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#51
Net::IMAP::StringPrep::Tables::IN_C_5 = T.let(T.unsafe(nil), Regexp)

# Inappropriate for plain text \StringPrep\[\"C.6\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#54
Net::IMAP::StringPrep::Tables::IN_C_6 = T.let(T.unsafe(nil), Regexp)

# Inappropriate for canonical representation \StringPrep\[\"C.7\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#57
Net::IMAP::StringPrep::Tables::IN_C_7 = T.let(T.unsafe(nil), Regexp)

# Change display properties or are deprecated \StringPrep\[\"C.8\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#60
Net::IMAP::StringPrep::Tables::IN_C_8 = T.let(T.unsafe(nil), Regexp)

# Tagging characters \StringPrep\[\"C.9\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#63
Net::IMAP::StringPrep::Tables::IN_C_9 = T.let(T.unsafe(nil), Regexp)

# Characters with bidirectional property "R" or "AL" \StringPrep\[\"D.1\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#66
Net::IMAP::StringPrep::Tables::IN_D_1 = T.let(T.unsafe(nil), Regexp)

# Used to check req3 of bidirectional checks
# Matches the negation of the D.1 table
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#70
Net::IMAP::StringPrep::Tables::IN_D_1_NEGATED = T.let(T.unsafe(nil), Regexp)

# Characters with bidirectional property "L" \StringPrep\[\"D.2\"]
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#73
Net::IMAP::StringPrep::Tables::IN_D_2 = T.let(T.unsafe(nil), Regexp)

# source://net-imap//lib/net/imap/stringprep/tables.rb#139
Net::IMAP::StringPrep::Tables::MAPPINGS = T.let(T.unsafe(nil), Hash)

# Replacements for IN_B.1
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#24
Net::IMAP::StringPrep::Tables::MAP_B_1 = T.let(T.unsafe(nil), String)

# Replacements for IN_B.2
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#27
Net::IMAP::StringPrep::Tables::MAP_B_2 = T.let(T.unsafe(nil), Hash)

# Replacements for IN_B.3
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#30
Net::IMAP::StringPrep::Tables::MAP_B_3 = T.let(T.unsafe(nil), Hash)

# Regexps matching each codepoint table in the RFC-3454 appendices
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#119
Net::IMAP::StringPrep::Tables::REGEXPS = T.let(T.unsafe(nil), Hash)

# Names of each codepoint table in the RFC-3454 appendices
#
# source://net-imap//lib/net/imap/stringprep/tables.rb#96
Net::IMAP::StringPrep::Tables::TITLES = T.let(T.unsafe(nil), Hash)

# Defined in RFC-4505[https://tools.ietf.org/html/rfc4505] §3, The +trace+
# profile of \StringPrep is used by the +ANONYMOUS+ \SASL mechanism.
#
# source://net-imap//lib/net/imap/stringprep/trace.rb#9
module Net::IMAP::StringPrep::Trace
  private

  # From RFC-4505[https://tools.ietf.org/html/rfc4505] §3, The "trace"
  # Profile of "Stringprep":
  # >>>
  #   The character repertoire of this profile is Unicode 3.2 [Unicode].
  #
  #   No mapping is required by this profile.
  #
  #   No Unicode normalization is required by this profile.
  #
  #   The list of unassigned code points for this profile is that provided
  #   in Appendix A of [StringPrep].  Unassigned code points are not
  #   prohibited.
  #
  #   Characters from the following tables of [StringPrep] are prohibited:
  #   (documented on PROHIBITED_TABLES)
  #
  #   This profile requires bidirectional character checking per Section 6
  #   of [StringPrep].
  #
  # source://net-imap//lib/net/imap/stringprep/trace.rb#68
  def stringprep_trace(string, **opts); end

  class << self
    # From RFC-4505[https://tools.ietf.org/html/rfc4505] §3, The "trace"
    # Profile of "Stringprep":
    # >>>
    #   The character repertoire of this profile is Unicode 3.2 [Unicode].
    #
    #   No mapping is required by this profile.
    #
    #   No Unicode normalization is required by this profile.
    #
    #   The list of unassigned code points for this profile is that provided
    #   in Appendix A of [StringPrep].  Unassigned code points are not
    #   prohibited.
    #
    #   Characters from the following tables of [StringPrep] are prohibited:
    #   (documented on PROHIBITED_TABLES)
    #
    #   This profile requires bidirectional character checking per Section 6
    #   of [StringPrep].
    #
    # source://net-imap//lib/net/imap/stringprep/trace.rb#68
    def stringprep_trace(string, **opts); end
  end
end

# >>>
#   This profile requires bidirectional character checking per Section 6
#   of [StringPrep].
#
# source://net-imap//lib/net/imap/stringprep/trace.rb#46
Net::IMAP::StringPrep::Trace::CHECK_BIDI = T.let(T.unsafe(nil), TrueClass)

# >>>
#   No mapping is required by this profile.
#
# source://net-imap//lib/net/imap/stringprep/trace.rb#20
Net::IMAP::StringPrep::Trace::MAPPING_TABLES = T.let(T.unsafe(nil), T.untyped)

# >>>
#   No Unicode normalization is required by this profile.
#
# source://net-imap//lib/net/imap/stringprep/trace.rb#24
Net::IMAP::StringPrep::Trace::NORMALIZATION = T.let(T.unsafe(nil), T.untyped)

# From RFC-4505[https://tools.ietf.org/html/rfc4505] §3, The "trace"
# Profile of "Stringprep":
# >>>
#   Characters from the following tables of [StringPrep] are prohibited:
#
#   - C.2.1 (ASCII control characters)
#   - C.2.2 (Non-ASCII control characters)
#   - C.3 (Private use characters)
#   - C.4 (Non-character code points)
#   - C.5 (Surrogate codes)
#   - C.6 (Inappropriate for plain text)
#   - C.8 (Change display properties are deprecated)
#   - C.9 (Tagging characters)
#
#   No additional characters are prohibited.
#
# source://net-imap//lib/net/imap/stringprep/trace.rb#41
Net::IMAP::StringPrep::Trace::PROHIBITED_TABLES = T.let(T.unsafe(nil), Array)

# Defined in RFC-4505[https://tools.ietf.org/html/rfc4505] §3.
#
# source://net-imap//lib/net/imap/stringprep/trace.rb#12
Net::IMAP::StringPrep::Trace::STRINGPREP_PROFILE = T.let(T.unsafe(nil), String)

# >>>
#   The character repertoire of this profile is Unicode 3.2 [Unicode].
#
# source://net-imap//lib/net/imap/stringprep/trace.rb#16
Net::IMAP::StringPrep::Trace::UNASSIGNED_TABLE = T.let(T.unsafe(nil), String)

# Mailbox attribute indicating that this mailbox is used to hold messages
# that have been deleted or marked for deletion. In some server
# implementations, this might be a virtual mailbox, containing messages from
# other mailboxes that are marked with the +\Deleted+ message flag.
# Alternatively, this might just be advice that a client that chooses not to
# use the \IMAP +\Deleted+ model should use as its trash location. In server
# implementations that strictly expect the \IMAP +\Deleted+ model, this
# special use is likely not to be supported.
#
# source://net-imap//lib/net/imap/flags.rb#258
Net::IMAP::TRASH = T.let(T.unsafe(nil), Symbol)

# Net::IMAP::ThreadMember represents a thread-node returned
# by Net::IMAP#thread.
#
# source://net-imap//lib/net/imap/response_data.rb#764
class Net::IMAP::ThreadMember < ::Struct
  # Returns a SequenceSet containing #seqno and all #children's seqno,
  # recursively.
  #
  # source://net-imap//lib/net/imap/response_data.rb#780
  def to_sequence_set; end

  protected

  # source://net-imap//lib/net/imap/response_data.rb#786
  def all_seqnos(node = T.unsafe(nil)); end
end

# Net::IMAP::UIDPlusData represents the ResponseCode#data that accompanies
# the +APPENDUID+ and +COPYUID+ response codes.
#
# See [[UIDPLUS[https://www.rfc-editor.org/rfc/rfc4315.html]].
#
# ==== Capability requirement
#
# The +UIDPLUS+ capability[rdoc-ref:Net::IMAP#capability] must be supported.
# A server that supports +UIDPLUS+ should send a UIDPlusData object inside
# every TaggedResponse returned by the append[rdoc-ref:Net::IMAP#append],
# copy[rdoc-ref:Net::IMAP#copy], move[rdoc-ref:Net::IMAP#move], {uid
# copy}[rdoc-ref:Net::IMAP#uid_copy], and {uid
# move}[rdoc-ref:Net::IMAP#uid_move] commands---unless the destination
# mailbox reports +UIDNOTSTICKY+.
#
# --
# TODO: support MULTIAPPEND
# ++
#
# source://net-imap//lib/net/imap/response_data.rb#346
class Net::IMAP::UIDPlusData < ::Struct
  # :call-seq: uid_mapping -> nil or a hash
  #
  # Returns a hash mapping each source UID to the newly assigned destination
  # UID.
  #
  # Note:: Returns +nil+ for Net::IMAP#append.
  #
  # source://net-imap//lib/net/imap/response_data.rb#376
  def uid_mapping; end
end

# Error raised upon an unknown response from the server.
#
# This is different from InvalidResponseError: the response may be a
# valid extension response and the server may be allowed to send it in
# this context, but Net::IMAP either does not know how to parse it or
# how to handle it.  This could result from enabling unknown or
# unhandled extensions.  The connection may still be usable,
# but—depending on context—it may be prudent to disconnect.
#
# source://net-imap//lib/net/imap/errors.rb#71
class Net::IMAP::UnknownResponseError < ::Net::IMAP::ResponseError; end

# **Note:** This represents an intentionally _unstable_ API.  Where
# instances of this class are returned, future releases may return a
# different (incompatible) object <em>without deprecation or warning</em>.
#
# Net::IMAP::UnparsedData represents data for unknown response types or
# unknown extensions to response types without a well-defined extension
# grammar.
#
# See also: UnparsedNumericResponseData, ExtensionData, IgnoredResponse
#
# source://net-imap//lib/net/imap/response_data.rb#77
class Net::IMAP::UnparsedData < ::Struct; end

# **Note:** This represents an intentionally _unstable_ API.  Where
# instances of this class are returned, future releases may return a
# different (incompatible) object <em>without deprecation or warning</em>.
#
# Net::IMAP::UnparsedNumericResponseData represents data for unhandled
# response types with a numeric prefix.  See the documentation for #number.
#
# See also: UnparsedData, ExtensionData, IgnoredResponse
#
# source://net-imap//lib/net/imap/response_data.rb#93
class Net::IMAP::UnparsedNumericResponseData < ::Struct; end

# source://net-imap//lib/net/imap.rb#720
Net::IMAP::VERSION = T.let(T.unsafe(nil), String)

# source://net-imap//lib/net/imap/authenticators.rb#35
Net::IMAP::XOauth2Authenticator = Net::IMAP::SASL::XOAuth2Authenticator
