'use strict';

const express = require('express');
const router = new express.Router();

const util = require('util');
const pem = require('pem');
const Packer = require('zip-stream');
const punycode = require('punycode/');
const removeDiacritics = require('diacritics').remove;

router.get('/keys', serveKeys);
router.post('/keys', handleKeys);

function serveKeys(req, res) {
    res.render('index', {
        pageTitle: 'Võtmete genereerimine',
        page: '/keys'
    });
}

function handleKeys(req, res) {
    Object.keys(req.body).forEach(key => {
        req.body[key] = req.body[key].trim();

        if (key === 'commonName') {
            req.body[key] = punycode.toASCII(
                req.body[key]
                    .replace(/^https?:\/+/i, '')
                    .split('/')
                    .shift()
                    .toLowerCase()
                    .trim()
            );
        }

        if (key === 'hash') {
            if (['sha1', 'md5', 'sha256'].indexOf(req.body[key].toLowerCase()) < 0) {
                req.body[key] = 'sha1';
            }
        }

        if (key === 'keyBitsize') {
            req.body[key] = Number(req.body[key].trim()) || 1024;
            if ([1024, 2048, 4096].indexOf(req.body[key]) < 0) {
                req.body[key] = 1024;
            }
        }

        if (key === 'emailAddress') {
            req.body[key] = req.body[key].replace(/@(.*)$/, (o, domain) => '@' + punycode.toASCII(domain.split('/').shift().toLowerCase().trim()));
        }

        if (typeof req.body[key] === 'string') {
            req.body[key] = removeDiacritics(req.body[key]);
        }
    });

    pem.createCSR(req.body, (err, keys) => {
        if (err) {
            req.flash('error', (err && err.message) || err);
            res.render('index', {
                pageTitle: 'Võtmete genereerimine',
                page: '/keys'
            });
            return;
        }

        let archive = new Packer({
                comment: 'Generated by https://pangalink.net/'
            }),
            chunks = [];

        archive.on('error', err => {
            req.flash('error', (err && err.message) || err);
            res.render('index', {
                pageTitle: 'Võtmete genereerimine',
                page: '/keys'
            });
        });

        archive.on('data', chunk => {
            if (chunk && chunk.length) {
                chunks.push(chunk);
            }
            return true;
        });

        archive.on('end', chunk => {
            if (chunk && chunk.length) {
                chunks.push(chunk);
            }

            res.status(200);
            res.set('Content-Description', 'File Transfer');
            res.set('Content-Type', 'application/octet-stream');
            res.set('Content-Disposition', util.format('attachment; filename="%s"', 'banklink.zip'));

            res.send(Buffer.concat(chunks));
        });

        archive.entry(
            keys.clientKey,
            {
                name: 'private_key.pem'
            },
            err => {
                if (err) {
                    req.flash('error', (err && err.message) || err);
                    res.render('index', {
                        pageTitle: 'Võtmete genereerimine',
                        page: '/keys'
                    });
                    return;
                }

                archive.entry(
                    keys.csr,
                    {
                        name: 'csr.pem'
                    },
                    err => {
                        if (err) {
                            req.flash('error', (err && err.message) || err);
                            res.render('index', {
                                pageTitle: 'Võtmete genereerimine',
                                page: '/keys'
                            });
                            return;
                        }

                        archive.finish();
                    }
                );
            }
        );
    });
}

module.exports = router;
