"""
Unit tests for the malware analyzer
"""
import pytest

from anchore_engine.analyzers import malware

test_sig = "Unix.Trojan.MSShellcode-40"
scanned_file = "/analysis_scratch/a5346cac-802a-4cb9-a0c5-756b9b51858b/squashed.tar"

malware_matches = [
    ("/analysis_scratch/a5346cac-802a-4cb9-a0c5-756b9b51858b/squashed.tar!POSIX_TAR:bin/busybox!...!(7)POSIX_TAR:elf_payload1: Unix.Trojan.MSShellcode-40 FOUND", ("/elf_payload1", test_sig)),
    ("/analysis_scratch/a5346cac-802a-4cb9-a0c5-756b9b51858b/squashed.tar!POSIX_TAR:bin/busybox!...!(7)POSIX_TAR:somedir/somepath/elf_payload1: Unix.Trojan.MSShellcode-40 FOUND", ("/somedir/somepath/elf_payload1", test_sig)),
    ("/analysis_scratch/a5346cac-802a-4cb9-a0c5-756b9b51858b/squashed.tar: Unix.Trojan.MSShellcode-40 FOUND", ())
]


@pytest.mark.parametrize(["line", "expected"], malware_matches)
def test_clamav_lineparser(line, expected):
        parsed = malware.parse_clamscan_output_line(scanned_file, line)
        assert parsed == expected


def test_freshclam_parser():
    did_update = """ClamAV update process started at Fri Jul 28 07:44:12 2020
    WARNING: Your ClamAV installation is OUTDATED!
    WARNING: Local version: 0.102.3 Recommended version: 0.102.4
    DON'T PANIC! Read https://www.clamav.net/documents/upgrading-clamav
    daily database available for download (remote version: 25889)
    Time: 4.0s, ETA: 0.0s [=============================>] 92.69MiB/92.69MiB        
    Testing database: '/var/lib/clamav/tmp.9460b/clamav-0ec6238a545395818add884614c55bd6.tmp-daily.cvd' ...
    Database test passed.
    daily.cvd updated (version: 25889, sigs: 3716328, f-level: 63, builder: raynman)
    main database available for download (remote version: 59)
    Time: 3.9s, ETA: 0.0s [=============================>] 112.40MiB/112.40MiB       
    Testing database: '/var/lib/clamav/tmp.9460b/clamav-97cbae5075c8db063494a9b2e1c04a44.tmp-main.cvd' ...
    Database test passed.
    main.cvd updated (version: 59, sigs: 4564902, f-level: 60, builder: sigmgr)
    bytecode database available for download (remote version: 331)
    Time: 0.4s, ETA: 0.0s [=============================>] 289.44KiB/289.44KiB     
    Testing database: '/var/lib/clamav/tmp.9460b/clamav-f24193a72f209acc0a9c45b79612e8fb.tmp-bytecode.cvd' ...
    Database test passed.
    bytecode.cvd updated (version: 331, sigs: 94, f-level: 63, builder: anvilleg)"""

    result = malware.parse_refresh_output(did_update)
    assert result is not None
    assert result['daily'] == "25889"
    assert result['main'] == "59"
    assert result['bytecode'] == "331"

    no_update = """ClamAV update process started at Fri Jul 28 08:02:11 2020
    WARNING: Your ClamAV installation is OUTDATED!
    WARNING: Local version: 0.102.3 Recommended version: 0.102.4
    DON'T PANIC! Read https://www.clamav.net/documents/upgrading-clamav
    daily.cvd database is up to date (version: 25889, sigs: 3716328, f-level: 63, builder: raynman)
    main.cvd database is up to date (version: 59, sigs: 4564902, f-level: 60, builder: sigmgr)
    bytecode.cvd database is up to date (version: 331, sigs: 94, f-level: 63, builder: anvilleg)"""

    result = malware.parse_refresh_output(no_update)
    assert result is not None
    assert result['daily'] == "25889"
    assert result['main'] == "59"
    assert result['bytecode'] == "331"

    # Use '.cld' instead of '.cvd' as clamav can set both depending on how update is done
    no_update = """ClamAV update process started at Fri Jul 28 08:02:11 2020
    WARNING: Your ClamAV installation is OUTDATED!
    WARNING: Local version: 0.102.3 Recommended version: 0.102.4
    DON'T PANIC! Read https://www.clamav.net/documents/upgrading-clamav
    daily.cld database is up to date (version: 25889, sigs: 3716328, f-level: 63, builder: raynman)
    main.cvd database is up to date (version: 59, sigs: 4564902, f-level: 60, builder: sigmgr)
    bytecode.cvd database is up to date (version: 331, sigs: 94, f-level: 63, builder: anvilleg)"""

    result = malware.parse_refresh_output(no_update)
    assert result is not None
    assert result['daily'] == "25889"
    assert result['main'] == "59"
    assert result['bytecode'] == "331"


def mock_scan():
    return [{"path": "/root/foobar/badfile", "signature": "Some really bad signature"}]


def mock_update():
    return {
        "daily": "1",
        "main": "1",
        "bytecode": "1"
    }


def test_output_handler():
    test_config = {
        'clamav': {'enabled': True}
    }
    fake_path = '/analysis_scratch/123-123-123-123-123/squashed.tar'

    t = malware.ClamAVRunner(config=test_config, squashed_tar_path=fake_path, tempdir='/tmp')
    setattr(t, 'scan', mock_scan)
    setattr(t, 'refresh_db', mock_update)
    output = t.run()
    print(output)
    assert output.name == 'clamav'
    assert len(output.findings) > 0


@pytest.mark.parametrize('config', [None, {}, {'clamav': None}, {'clamav': {}}, {'clamav': {'enabled': False}}])
def test_disabled(config):
    t = malware.ClamAVRunner(config=config, squashed_tar_path='squashed.tar', tempdir='/tmp')
    out = t.run()
    assert out is not None
    assert out.findings is None
