% Rust Documentation

`rustdoc` is the built-in tool for generating documentation. It integrates
with the compiler to provide accurate hyperlinking between usage of types and
their documentation. Furthermore, by not using a separate parser, it will
never reject your valid Rust code.

# Creating Documentation

Documenting Rust APIs is quite simple. To document a given item, we have "doc
comments":

~~~
// the "link" crate attribute is currently required for rustdoc, but normally
// isn't needed.
#[crate_id = "universe"];
#[crate_type="lib"];

//! Tools for dealing with universes (this is a doc comment, and is shown on
//! the crate index page. The ! makes it apply to the parent of the comment,
//! rather than what follows).

/// Widgets are very common (this is a doc comment, and will show up on
/// Widget's documentation).
pub struct Widget {
	/// All widgets have a purpose (this is a doc comment, and will show up
	/// the field's documentation).
	purpose: ~str,
	/// Humans are not allowed to understand some widgets
	understandable: bool
}

pub fn recalibrate() {
	//! Recalibrate a pesky universe (this is also a doc comment, like above,
	//! the documentation will be applied to the *parent* item, so
	//! `recalibrate`).
	/* ... */
}
~~~

Doc comments are markdown, and are currently parsed with the
[sundown][sundown] library. rustdoc does not yet do any fanciness such as
referencing other items inline, like javadoc's `@see`. One exception to this
is that the first paragrah will be used as the "summary" of an item in the
generated documentation:

~~~
/// A whizbang. Does stuff. (this line is the summary)
///
/// Whizbangs are ...
struct Whizbang;
~~~

To generate the docs, run `rustdoc universe.rs`. By default, it generates a
directory called `doc`, with the documentation for `universe` being in
`doc/universe/index.html`. If you are using other crates with `extern mod`,
rustdoc will even link to them when you use their types, as long as their
documentation has already been generated by a previous run of rustdoc, or the
crate advertises that its documentation is hosted at a given URL.

The generated output can be controlled with the `doc` crate attribute, which
is how the above advertisement works. An example from the `libstd`
documentation:

~~~
#[doc(html_logo_url = "http://www.rust-lang.org/logos/rust-logo-128x128-blk.png",
      html_favicon_url = "http://www.rust-lang.org/favicon.ico",
      html_root_url = "http://static.rust-lang.org/doc/master")];
~~~

The `html_root_url` is the prefix that rustdoc will apply to any references to
that crate's types etc.

rustdoc can also generate JSON, for consumption by other tools, with
`rustdoc --output-format json`, and also consume already-generated JSON with
`rustdoc --input-format json`.

# Using the Documentation

The web pages generated by rustdoc present the same logical heirarchy that one
writes a library with. Every kind of item (function, struct, etc) has its own
color, and one can always click on a colored type to jump to its
documentation. There is a search bar at the top, which is powered by some
javascript and a statically-generated search index. No special web server is
required for the search.

[sundown]: https://github.com/vmg/sundown/

# Testing the Documentation

`rustdoc` has support for testing code examples which appear in the
documentation. This is helpful for keeping code examples up to date with the
source code.

To test documentation, the `--test` argument is passed to rustdoc:

~~~
rustdoc --test crate.rs
~~~

## Defining tests

Rust documentation currently uses the markdown format, and code blocks can refer
to any piece of code-related documentation, which isn't always rust. Because of
this, only code blocks with the language of "rust" will be considered for
testing.

~~~
```rust
// This is a testable code block
```

```
// This is not a testable code block
```

    // This is not a testable code block (4-space indent)
~~~

In addition to only testing "rust"-language code blocks, there are additional
specifiers that can be used to dictate how a code block is tested:

~~~
```rust,ignore
// This code block is ignored by rustdoc, but is passed through to the test
// harness
```

```rust,should_fail
// This code block is expected to generate a failure
```
~~~

Rustdoc also supplies some extra sugar for helping with some tedious
documentation examples. If a line is prefixed with `# `, then the line
will not show up in the HTML documentation, but it will be used when
testing the code block (NB. the space after the `#` is required, so
that one can still write things like `#[deriving(Eq)]`).

~~~
```rust
# // showing 'fib' in this documentation would just be tedious and detracts from
# // what's actualy being documented.
# fn fib(n: int) { n + 2 }

do spawn { fib(200); }
```
~~~

The documentation online would look like `do spawn { fib(200); }`, but when
testing this code, the `fib` function will be included (so it can compile).

## Running tests (advanced)

Running tests often requires some special configuration to filter tests, find
libraries, or try running ignored examples. The testing framework that rustdoc
uses is build on `extra::test`, which is also used when you compile crates with
rustc's `--test` flag. Extra arguments can be passed to rustdoc's test harness
with the `--test-args` flag.

~~~
// Only run tests containing 'foo' in their name
rustdoc --test lib.rs --test-args 'foo'

// See what's possible when running tests
rustdoc --test lib.rs --test-args '--help'

// Run all ignored tests
rustdoc --test lib.rs --test-args '--ignored'
~~~

When testing a library, code examples will often show how functions are used,
and this code often requires `use`-ing paths from the crate. To accomodate this,
rustdoc will implicitly add `extern mod <crate>;` where `<crate>` is the name of
the crate being tested to the top of each code example. This means that rustdoc
must be able to find a compiled version of the library crate being tested. Extra
search paths may be added via the `-L` flag to `rustdoc`.
