// Copyright 2012 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::os;
use std::run;
use std::str;
use std::io::process::ProcessExit;

#[cfg(target_os = "win32")]
fn target_env(lib_path: &str, prog: &str) -> ~[(~str,~str)] {

    let mut env = os::env();

    // Make sure we include the aux directory in the path
    assert!(prog.ends_with(".exe"));
    let aux_path = prog.slice(0u, prog.len() - 4u).to_owned() + ".libaux";

    env = env.map(|pair| {
        let (k,v) = (*pair).clone();
        if k == ~"PATH" { (~"PATH", v + ";" + lib_path + ";" + aux_path) }
        else { (k,v) }
    });
    if prog.ends_with("rustc.exe") {
        env.push((~"RUST_THREADS", ~"1"));
    }
    return env;
}

#[cfg(target_os = "linux")]
#[cfg(target_os = "macos")]
#[cfg(target_os = "freebsd")]
fn target_env(_lib_path: &str, _prog: &str) -> ~[(~str,~str)] {
    os::env()
}

pub struct Result {status: ProcessExit, out: ~str, err: ~str}

pub fn run(lib_path: &str,
           prog: &str,
           args: &[~str],
           env: ~[(~str, ~str)],
           input: Option<~str>) -> Option<Result> {

    let env = env + target_env(lib_path, prog);
    let mut opt_process = run::Process::new(prog, args, run::ProcessOptions {
        env: Some(env),
        dir: None,
        in_fd: None,
        out_fd: None,
        err_fd: None
    });

    match opt_process {
        Some(ref mut process) => {
            for input in input.iter() {
                process.input().write(input.as_bytes());
            }
            let run::ProcessOutput { status, output, error } = process.finish_with_output();

            Some(Result {
                status: status,
                out: str::from_utf8_owned(output),
                err: str::from_utf8_owned(error)
            })
        },
        None => None
    }
}

pub fn run_background(lib_path: &str,
           prog: &str,
           args: &[~str],
           env: ~[(~str, ~str)],
           input: Option<~str>) -> Option<run::Process> {

    let env = env + target_env(lib_path, prog);
    let opt_process = run::Process::new(prog, args, run::ProcessOptions {
        env: Some(env),
        dir: None,
        in_fd: None,
        out_fd: None,
        err_fd: None
    });

    match opt_process {
        Some(mut process) => {
            for input in input.iter() {
                process.input().write(input.as_bytes());
            }

            Some(process)
        },
        None => None
    }
}
