// Copyright 2012-2013 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.


use c = metadata::common;
use cstore = metadata::cstore;
use driver::session::Session;
use e = metadata::encoder;
use metadata::decoder;
use metadata::tydecode;
use metadata::tydecode::{DefIdSource, NominalType, TypeWithId, TypeParameter,
                         RegionParameter};
use metadata::tyencode;
use middle::freevars::freevar_entry;
use middle::typeck::{method_origin, method_map_entry};
use middle::{ty, typeck, moves};
use middle;
use util::ppaux::ty_to_str;

use syntax::{ast, ast_map, ast_util, codemap, fold};
use syntax::codemap::Span;
use syntax::diagnostic::SpanHandler;
use syntax::fold::ast_fold;
use syntax::parse::token;
use syntax;

use std::at_vec;
use std::libc;
use std::cast;
use std::io::Seek;

use extra::ebml::reader;
use extra::ebml;
use extra::serialize;
use extra::serialize::{Encoder, Encodable, EncoderHelpers, DecoderHelpers};
use extra::serialize::{Decoder, Decodable};
use writer = extra::ebml::writer;

#[cfg(test)] use syntax::parse;
#[cfg(test)] use syntax::print::pprust;

// Auxiliary maps of things to be encoded
pub struct Maps {
    root_map: middle::borrowck::root_map,
    method_map: middle::typeck::method_map,
    vtable_map: middle::typeck::vtable_map,
    capture_map: middle::moves::CaptureMap,
}

struct DecodeContext {
    cdata: @cstore::crate_metadata,
    tcx: ty::ctxt,
    maps: Maps
}

struct ExtendedDecodeContext {
    dcx: @DecodeContext,
    from_id_range: ast_util::id_range,
    to_id_range: ast_util::id_range
}

trait tr {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> Self;
}

trait tr_intern {
    fn tr_intern(&self, xcx: @ExtendedDecodeContext) -> ast::DefId;
}

// ______________________________________________________________________
// Top-level methods.

pub fn encode_inlined_item(ecx: &e::EncodeContext,
                           ebml_w: &mut writer::Encoder,
                           path: &[ast_map::path_elt],
                           ii: e::InlinedItemRef,
                           maps: Maps) {
    let ident = match ii {
        e::ii_item_ref(i) => i.ident,
        e::ii_foreign_ref(i) => i.ident,
        e::ii_method_ref(_, _, m) => m.ident,
    };
    debug!("> Encoding inlined item: {}::{} ({})",
           ast_map::path_to_str(path, token::get_ident_interner()),
           ecx.tcx.sess.str_of(ident),
           ebml_w.writer.tell());

    let ii = simplify_ast(ii);
    let id_range = ast_util::compute_id_range_for_inlined_item(&ii);

    ebml_w.start_tag(c::tag_ast as uint);
    id_range.encode(ebml_w);
    encode_ast(ebml_w, ii);
    encode_side_tables_for_ii(ecx, maps, ebml_w, &ii);
    ebml_w.end_tag();

    debug!("< Encoded inlined fn: {}::{} ({})",
           ast_map::path_to_str(path, token::get_ident_interner()),
           ecx.tcx.sess.str_of(ident),
           ebml_w.writer.tell());
}

pub fn decode_inlined_item(cdata: @cstore::crate_metadata,
                           tcx: ty::ctxt,
                           maps: Maps,
                           path: &[ast_map::path_elt],
                           par_doc: ebml::Doc)
                        -> Option<ast::inlined_item> {
    let dcx = @DecodeContext {
        cdata: cdata,
        tcx: tcx,
        maps: maps
    };
    match par_doc.opt_child(c::tag_ast) {
      None => None,
      Some(ast_doc) => {
        debug!("> Decoding inlined fn: {}::?",
               ast_map::path_to_str(path, token::get_ident_interner()));
        let mut ast_dsr = reader::Decoder(ast_doc);
        let from_id_range = Decodable::decode(&mut ast_dsr);
        let to_id_range = reserve_id_range(dcx.tcx.sess, from_id_range);
        let xcx = @ExtendedDecodeContext {
            dcx: dcx,
            from_id_range: from_id_range,
            to_id_range: to_id_range
        };
        let raw_ii = decode_ast(ast_doc);
        let ii = renumber_and_map_ast(xcx,
                                      tcx.sess.diagnostic(),
                                      dcx.tcx.items,
                                      path.to_owned(),
                                      raw_ii);
        let ident = match ii {
            ast::ii_item(i) => i.ident,
            ast::ii_foreign(i) => i.ident,
            ast::ii_method(_, _, m) => m.ident,
        };
        debug!("Fn named: {}", tcx.sess.str_of(ident));
        debug!("< Decoded inlined fn: {}::{}",
               ast_map::path_to_str(path, token::get_ident_interner()),
               tcx.sess.str_of(ident));
        decode_side_tables(xcx, ast_doc);
        match ii {
          ast::ii_item(i) => {
            debug!(">>> DECODED ITEM >>>\n{}\n<<< DECODED ITEM <<<",
                   syntax::print::pprust::item_to_str(i, tcx.sess.intr()));
          }
          _ => { }
        }
        Some(ii)
      }
    }
}

// ______________________________________________________________________
// Enumerating the IDs which appear in an AST

fn reserve_id_range(sess: Session,
                    from_id_range: ast_util::id_range) -> ast_util::id_range {
    // Handle the case of an empty range:
    if from_id_range.empty() { return from_id_range; }
    let cnt = from_id_range.max - from_id_range.min;
    let to_id_min = sess.reserve_node_ids(cnt);
    let to_id_max = to_id_min + cnt;
    ast_util::id_range { min: to_id_min, max: to_id_max }
}

impl ExtendedDecodeContext {
    pub fn tr_id(&self, id: ast::NodeId) -> ast::NodeId {
        /*!
         * Translates an internal id, meaning a node id that is known
         * to refer to some part of the item currently being inlined,
         * such as a local variable or argument.  All naked node-ids
         * that appear in types have this property, since if something
         * might refer to an external item we would use a def-id to
         * allow for the possibility that the item resides in another
         * crate.
         */

        // from_id_range should be non-empty
        assert!(!self.from_id_range.empty());
        (id - self.from_id_range.min + self.to_id_range.min)
    }
    pub fn tr_def_id(&self, did: ast::DefId) -> ast::DefId {
        /*!
         * Translates an EXTERNAL def-id, converting the crate number
         * from the one used in the encoded data to the current crate
         * numbers..  By external, I mean that it be translated to a
         * reference to the item in its original crate, as opposed to
         * being translated to a reference to the inlined version of
         * the item.  This is typically, but not always, what you
         * want, because most def-ids refer to external things like
         * types or other fns that may or may not be inlined.  Note
         * that even when the inlined function is referencing itself
         * recursively, we would want `tr_def_id` for that
         * reference--- conceptually the function calls the original,
         * non-inlined version, and trans deals with linking that
         * recursive call to the inlined copy.
         *
         * However, there are a *few* cases where def-ids are used but
         * we know that the thing being referenced is in fact *internal*
         * to the item being inlined.  In those cases, you should use
         * `tr_intern_def_id()` below.
         */

        decoder::translate_def_id(self.dcx.cdata, did)
    }
    pub fn tr_intern_def_id(&self, did: ast::DefId) -> ast::DefId {
        /*!
         * Translates an INTERNAL def-id, meaning a def-id that is
         * known to refer to some part of the item currently being
         * inlined.  In that case, we want to convert the def-id to
         * refer to the current crate and to the new, inlined node-id.
         */

        assert_eq!(did.crate, ast::LOCAL_CRATE);
        ast::DefId { crate: ast::LOCAL_CRATE, node: self.tr_id(did.node) }
    }
    pub fn tr_span(&self, _span: Span) -> Span {
        codemap::DUMMY_SP // FIXME (#1972): handle span properly
    }
}

impl tr_intern for ast::DefId {
    fn tr_intern(&self, xcx: @ExtendedDecodeContext) -> ast::DefId {
        xcx.tr_intern_def_id(*self)
    }
}

impl tr for ast::DefId {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> ast::DefId {
        xcx.tr_def_id(*self)
    }
}

impl tr for Option<ast::DefId> {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> Option<ast::DefId> {
        self.map(|d| xcx.tr_def_id(d))
    }
}

impl tr for Span {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> Span {
        xcx.tr_span(*self)
    }
}

trait def_id_encoder_helpers {
    fn emit_def_id(&mut self, did: ast::DefId);
}

impl<S:serialize::Encoder> def_id_encoder_helpers for S {
    fn emit_def_id(&mut self, did: ast::DefId) {
        did.encode(self)
    }
}

trait def_id_decoder_helpers {
    fn read_def_id(&mut self, xcx: @ExtendedDecodeContext) -> ast::DefId;
    fn read_def_id_noxcx(&mut self,
                         cdata: @cstore::crate_metadata) -> ast::DefId;
}

impl<D:serialize::Decoder> def_id_decoder_helpers for D {
    fn read_def_id(&mut self, xcx: @ExtendedDecodeContext) -> ast::DefId {
        let did: ast::DefId = Decodable::decode(self);
        did.tr(xcx)
    }

    fn read_def_id_noxcx(&mut self,
                         cdata: @cstore::crate_metadata) -> ast::DefId {
        let did: ast::DefId = Decodable::decode(self);
        decoder::translate_def_id(cdata, did)
    }
}

// ______________________________________________________________________
// Encoding and decoding the AST itself
//
// The hard work is done by an autogenerated module astencode_gen.  To
// regenerate astencode_gen, run src/etc/gen-astencode.  It will
// replace astencode_gen with a dummy file and regenerate its
// contents.  If you get compile errors, the dummy file
// remains---resolve the errors and then rerun astencode_gen.
// Annoying, I know, but hopefully only temporary.
//
// When decoding, we have to renumber the AST so that the node ids that
// appear within are disjoint from the node ids in our existing ASTs.
// We also have to adjust the spans: for now we just insert a dummy span,
// but eventually we should add entries to the local codemap as required.

fn encode_ast(ebml_w: &mut writer::Encoder, item: ast::inlined_item) {
    ebml_w.start_tag(c::tag_tree as uint);
    item.encode(ebml_w);
    ebml_w.end_tag();
}

struct NestedItemsDropper;

impl ast_fold for NestedItemsDropper {
    fn fold_block(&mut self, blk: ast::P<ast::Block>) -> ast::P<ast::Block> {
        let stmts_sans_items = blk.stmts.iter().filter_map(|stmt| {
            match stmt.node {
                ast::StmtExpr(_, _) | ast::StmtSemi(_, _) |
                ast::StmtDecl(@codemap::Spanned {
                    node: ast::DeclLocal(_),
                    span: _
                }, _) => Some(*stmt),
                ast::StmtDecl(@codemap::Spanned {
                    node: ast::DeclItem(_),
                    span: _
                }, _) => None,
                ast::StmtMac(..) => fail!("unexpanded macro in astencode")
            }
        }).collect();
        let blk_sans_items = ast::P(ast::Block {
            view_items: ~[], // I don't know if we need the view_items here,
                             // but it doesn't break tests!
            stmts: stmts_sans_items,
            expr: blk.expr,
            id: blk.id,
            rules: blk.rules,
            span: blk.span,
        });
        fold::noop_fold_block(blk_sans_items, self)
    }
}

// Produces a simplified copy of the AST which does not include things
// that we do not need to or do not want to export.  For example, we
// do not include any nested items: if these nested items are to be
// inlined, their AST will be exported separately (this only makes
// sense because, in Rust, nested items are independent except for
// their visibility).
//
// As it happens, trans relies on the fact that we do not export
// nested items, as otherwise it would get confused when translating
// inlined items.
fn simplify_ast(ii: e::InlinedItemRef) -> ast::inlined_item {
    let mut fld = NestedItemsDropper;

    match ii {
        // HACK we're not dropping items.
        e::ii_item_ref(i) => ast::ii_item(fold::noop_fold_item(i, &mut fld)
                                          .expect_one("expected one item")),
        e::ii_method_ref(d, p, m) => ast::ii_method(d, p, fold::noop_fold_method(m, &mut fld)),
        e::ii_foreign_ref(i) => ast::ii_foreign(fold::noop_fold_foreign_item(i, &mut fld))
    }
}

fn decode_ast(par_doc: ebml::Doc) -> ast::inlined_item {
    let chi_doc = par_doc.get(c::tag_tree as uint);
    let mut d = reader::Decoder(chi_doc);
    Decodable::decode(&mut d)
}

struct AstRenumberer {
    xcx: @ExtendedDecodeContext,
}

impl ast_map::FoldOps for AstRenumberer {
    fn new_id(&self, id: ast::NodeId) -> ast::NodeId {
        self.xcx.tr_id(id)
    }
    fn new_span(&self, span: Span) -> Span {
        self.xcx.tr_span(span)
    }
}

fn renumber_and_map_ast(xcx: @ExtendedDecodeContext,
                        diag: @SpanHandler,
                        map: ast_map::map,
                        path: ast_map::path,
                        ii: ast::inlined_item) -> ast::inlined_item {
    ast_map::map_decoded_item(diag, map, path, AstRenumberer { xcx: xcx }, |fld| {
        match ii {
            ast::ii_item(i) => {
                ast::ii_item(fld.fold_item(i).expect_one("expected one item"))
            }
            ast::ii_method(d, is_provided, m) => {
                ast::ii_method(xcx.tr_def_id(d), is_provided, fld.fold_method(m))
            }
            ast::ii_foreign(i) => ast::ii_foreign(fld.fold_foreign_item(i))
        }
    })
}

// ______________________________________________________________________
// Encoding and decoding of ast::def

fn decode_def(xcx: @ExtendedDecodeContext, doc: ebml::Doc) -> ast::Def {
    let mut dsr = reader::Decoder(doc);
    let def: ast::Def = Decodable::decode(&mut dsr);
    def.tr(xcx)
}

impl tr for ast::Def {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> ast::Def {
        match *self {
          ast::DefFn(did, p) => ast::DefFn(did.tr(xcx), p),
          ast::DefStaticMethod(did, wrapped_did2, p) => {
            ast::DefStaticMethod(did.tr(xcx),
                                   match wrapped_did2 {
                                    ast::FromTrait(did2) => {
                                        ast::FromTrait(did2.tr(xcx))
                                    }
                                    ast::FromImpl(did2) => {
                                        ast::FromImpl(did2.tr(xcx))
                                    }
                                   },
                                   p)
          }
          ast::DefMethod(did0, did1) => {
            ast::DefMethod(did0.tr(xcx), did1.map(|did1| did1.tr(xcx)))
          }
          ast::DefSelfTy(nid) => { ast::DefSelfTy(xcx.tr_id(nid)) }
          ast::DefSelf(nid, m) => { ast::DefSelf(xcx.tr_id(nid), m) }
          ast::DefMod(did) => { ast::DefMod(did.tr(xcx)) }
          ast::DefForeignMod(did) => { ast::DefForeignMod(did.tr(xcx)) }
          ast::DefStatic(did, m) => { ast::DefStatic(did.tr(xcx), m) }
          ast::DefArg(nid, b) => { ast::DefArg(xcx.tr_id(nid), b) }
          ast::DefLocal(nid, b) => { ast::DefLocal(xcx.tr_id(nid), b) }
          ast::DefVariant(e_did, v_did, is_s) => {
            ast::DefVariant(e_did.tr(xcx), v_did.tr(xcx), is_s)
          },
          ast::DefTrait(did) => ast::DefTrait(did.tr(xcx)),
          ast::DefTy(did) => ast::DefTy(did.tr(xcx)),
          ast::DefPrimTy(p) => ast::DefPrimTy(p),
          ast::DefTyParam(did, v) => ast::DefTyParam(did.tr(xcx), v),
          ast::DefBinding(nid, bm) => ast::DefBinding(xcx.tr_id(nid), bm),
          ast::DefUse(did) => ast::DefUse(did.tr(xcx)),
          ast::DefUpvar(nid1, def, nid2, nid3) => {
            ast::DefUpvar(xcx.tr_id(nid1),
                           @(*def).tr(xcx),
                           xcx.tr_id(nid2),
                           xcx.tr_id(nid3))
          }
          ast::DefStruct(did) => ast::DefStruct(did.tr(xcx)),
          ast::DefRegion(nid) => ast::DefRegion(xcx.tr_id(nid)),
          ast::DefTyParamBinder(nid) => {
            ast::DefTyParamBinder(xcx.tr_id(nid))
          }
          ast::DefLabel(nid) => ast::DefLabel(xcx.tr_id(nid))
        }
    }
}

// ______________________________________________________________________
// Encoding and decoding of adjustment information

impl tr for ty::AutoDerefRef {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> ty::AutoDerefRef {
        ty::AutoDerefRef {
            autoderefs: self.autoderefs,
            autoref: match self.autoref {
                Some(ref autoref) => Some(autoref.tr(xcx)),
                None => None
            }
        }
    }
}

impl tr for ty::AutoRef {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> ty::AutoRef {
        self.map_region(|r| r.tr(xcx))
    }
}

impl tr for ty::Region {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> ty::Region {
        match *self {
            ty::ReLateBound(id, br) => ty::ReLateBound(xcx.tr_id(id),
                                                       br.tr(xcx)),
            ty::ReEarlyBound(id, index, ident) => ty::ReEarlyBound(xcx.tr_id(id),
                                                                     index,
                                                                     ident),
            ty::ReScope(id) => ty::ReScope(xcx.tr_id(id)),
            ty::ReEmpty | ty::ReStatic | ty::ReInfer(..) => *self,
            ty::ReFree(ref fr) => {
                ty::ReFree(ty::FreeRegion {scope_id: xcx.tr_id(fr.scope_id),
                                            bound_region: fr.bound_region.tr(xcx)})
            }
        }
    }
}

impl tr for ty::BoundRegion {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> ty::BoundRegion {
        match *self {
            ty::BrAnon(_) |
            ty::BrFresh(_) => *self,
            ty::BrNamed(id, ident) => ty::BrNamed(xcx.tr_def_id(id),
                                                    ident),
        }
    }
}

// ______________________________________________________________________
// Encoding and decoding of freevar information

fn encode_freevar_entry(ebml_w: &mut writer::Encoder, fv: @freevar_entry) {
    (*fv).encode(ebml_w)
}

trait ebml_decoder_helper {
    fn read_freevar_entry(&mut self, xcx: @ExtendedDecodeContext)
                          -> freevar_entry;
}

impl<'a> ebml_decoder_helper for reader::Decoder<'a> {
    fn read_freevar_entry(&mut self, xcx: @ExtendedDecodeContext)
                          -> freevar_entry {
        let fv: freevar_entry = Decodable::decode(self);
        fv.tr(xcx)
    }
}

impl tr for freevar_entry {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> freevar_entry {
        freevar_entry {
            def: self.def.tr(xcx),
            span: self.span.tr(xcx),
        }
    }
}

// ______________________________________________________________________
// Encoding and decoding of CaptureVar information

trait capture_var_helper {
    fn read_capture_var(&mut self, xcx: @ExtendedDecodeContext)
                        -> moves::CaptureVar;
}

impl<'a> capture_var_helper for reader::Decoder<'a> {
    fn read_capture_var(&mut self, xcx: @ExtendedDecodeContext)
                        -> moves::CaptureVar {
        let cvar: moves::CaptureVar = Decodable::decode(self);
        cvar.tr(xcx)
    }
}

impl tr for moves::CaptureVar {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> moves::CaptureVar {
        moves::CaptureVar {
            def: self.def.tr(xcx),
            span: self.span.tr(xcx),
            mode: self.mode
        }
    }
}

// ______________________________________________________________________
// Encoding and decoding of method_map_entry

trait read_method_map_entry_helper {
    fn read_method_map_entry(&mut self, xcx: @ExtendedDecodeContext)
                             -> method_map_entry;
}

fn encode_method_map_entry(ecx: &e::EncodeContext,
                           ebml_w: &mut writer::Encoder,
                           mme: method_map_entry) {
    ebml_w.emit_struct("method_map_entry", 3, |ebml_w| {
        ebml_w.emit_struct_field("self_ty", 0u, |ebml_w| {
            ebml_w.emit_ty(ecx, mme.self_ty);
        });
        ebml_w.emit_struct_field("explicit_self", 2u, |ebml_w| {
            mme.explicit_self.encode(ebml_w);
        });
        ebml_w.emit_struct_field("origin", 1u, |ebml_w| {
            mme.origin.encode(ebml_w);
        });
        ebml_w.emit_struct_field("self_mode", 3, |ebml_w| {
            mme.self_mode.encode(ebml_w);
        });
    })
}

impl<'a> read_method_map_entry_helper for reader::Decoder<'a> {
    fn read_method_map_entry(&mut self, xcx: @ExtendedDecodeContext)
                             -> method_map_entry {
        self.read_struct("method_map_entry", 3, |this| {
            method_map_entry {
                self_ty: this.read_struct_field("self_ty", 0u, |this| {
                    this.read_ty(xcx)
                }),
                explicit_self: this.read_struct_field("explicit_self",
                                                      2,
                                                      |this| {
                    let explicit_self: ast::explicit_self_ = Decodable::decode(this);
                    explicit_self
                }),
                origin: this.read_struct_field("origin", 1, |this| {
                    let method_origin: method_origin =
                        Decodable::decode(this);
                    method_origin.tr(xcx)
                }),
                self_mode: this.read_struct_field("self_mode", 3, |this| {
                    let self_mode: ty::SelfMode = Decodable::decode(this);
                    self_mode
                }),
            }
        })
    }
}

impl tr for method_origin {
    fn tr(&self, xcx: @ExtendedDecodeContext) -> method_origin {
        match *self {
          typeck::method_static(did) => {
              typeck::method_static(did.tr(xcx))
          }
          typeck::method_param(ref mp) => {
            typeck::method_param(
                typeck::method_param {
                    trait_id: mp.trait_id.tr(xcx),
                    .. *mp
                }
            )
          }
          typeck::method_object(ref mo) => {
            typeck::method_object(
                typeck::method_object {
                    trait_id: mo.trait_id.tr(xcx),
                    .. *mo
                }
            )
          }
        }
    }
}

// ______________________________________________________________________
// Encoding and decoding vtable_res

pub fn encode_vtable_res(ecx: &e::EncodeContext,
                     ebml_w: &mut writer::Encoder,
                     dr: typeck::vtable_res) {
    // can't autogenerate this code because automatic code of
    // ty::t doesn't work, and there is no way (atm) to have
    // hand-written encoding routines combine with auto-generated
    // ones.  perhaps we should fix this.
    ebml_w.emit_from_vec(*dr, |ebml_w, param_tables| {
        encode_vtable_param_res(ecx, ebml_w, *param_tables);
    })
}

pub fn encode_vtable_param_res(ecx: &e::EncodeContext,
                     ebml_w: &mut writer::Encoder,
                     param_tables: typeck::vtable_param_res) {
    ebml_w.emit_from_vec(*param_tables, |ebml_w, vtable_origin| {
        encode_vtable_origin(ecx, ebml_w, vtable_origin)
    })
}


pub fn encode_vtable_origin(ecx: &e::EncodeContext,
                        ebml_w: &mut writer::Encoder,
                        vtable_origin: &typeck::vtable_origin) {
    ebml_w.emit_enum("vtable_origin", |ebml_w| {
        match *vtable_origin {
          typeck::vtable_static(def_id, ref tys, vtable_res) => {
            ebml_w.emit_enum_variant("vtable_static", 0u, 3u, |ebml_w| {
                ebml_w.emit_enum_variant_arg(0u, |ebml_w| {
                    ebml_w.emit_def_id(def_id)
                });
                ebml_w.emit_enum_variant_arg(1u, |ebml_w| {
                    ebml_w.emit_tys(ecx, *tys);
                });
                ebml_w.emit_enum_variant_arg(2u, |ebml_w| {
                    encode_vtable_res(ecx, ebml_w, vtable_res);
                })
            })
          }
          typeck::vtable_param(pn, bn) => {
            ebml_w.emit_enum_variant("vtable_param", 1u, 2u, |ebml_w| {
                ebml_w.emit_enum_variant_arg(0u, |ebml_w| {
                    pn.encode(ebml_w);
                });
                ebml_w.emit_enum_variant_arg(1u, |ebml_w| {
                    ebml_w.emit_uint(bn);
                })
            })
          }
        }
    })
}

pub trait vtable_decoder_helpers {
    fn read_vtable_res(&mut self,
                       tcx: ty::ctxt, cdata: @cstore::crate_metadata)
                      -> typeck::vtable_res;
    fn read_vtable_param_res(&mut self,
                       tcx: ty::ctxt, cdata: @cstore::crate_metadata)
                      -> typeck::vtable_param_res;
    fn read_vtable_origin(&mut self,
                          tcx: ty::ctxt, cdata: @cstore::crate_metadata)
                          -> typeck::vtable_origin;
}

impl<'a> vtable_decoder_helpers for reader::Decoder<'a> {
    fn read_vtable_res(&mut self,
                       tcx: ty::ctxt, cdata: @cstore::crate_metadata)
                      -> typeck::vtable_res {
        @self.read_to_vec(|this|
                          this.read_vtable_param_res(tcx, cdata))
    }

    fn read_vtable_param_res(&mut self,
                             tcx: ty::ctxt, cdata: @cstore::crate_metadata)
                      -> typeck::vtable_param_res {
        @self.read_to_vec(|this|
                          this.read_vtable_origin(tcx, cdata))
    }

    fn read_vtable_origin(&mut self,
                          tcx: ty::ctxt, cdata: @cstore::crate_metadata)
        -> typeck::vtable_origin {
        self.read_enum("vtable_origin", |this| {
            this.read_enum_variant(["vtable_static",
                                    "vtable_param",
                                    "vtable_self"],
                                   |this, i| {
                match i {
                  0 => {
                    typeck::vtable_static(
                        this.read_enum_variant_arg(0u, |this| {
                            this.read_def_id_noxcx(cdata)
                        }),
                        this.read_enum_variant_arg(1u, |this| {
                            this.read_tys_noxcx(tcx, cdata)
                        }),
                        this.read_enum_variant_arg(2u, |this| {
                            this.read_vtable_res(tcx, cdata)
                        })
                    )
                  }
                  1 => {
                    typeck::vtable_param(
                        this.read_enum_variant_arg(0u, |this| {
                            Decodable::decode(this)
                        }),
                        this.read_enum_variant_arg(1u, |this| {
                            this.read_uint()
                        })
                    )
                  }
                  // hard to avoid - user input
                  _ => fail!("bad enum variant")
                }
            })
        })
    }
}

// ______________________________________________________________________
// Encoding and decoding the side tables

trait get_ty_str_ctxt {
    fn ty_str_ctxt(&self) -> @tyencode::ctxt;
}

impl<'a> get_ty_str_ctxt for e::EncodeContext<'a> {
    fn ty_str_ctxt(&self) -> @tyencode::ctxt {
        @tyencode::ctxt {
            diag: self.tcx.sess.diagnostic(),
            ds: e::def_to_str,
            tcx: self.tcx,
            abbrevs: tyencode::ac_use_abbrevs(self.type_abbrevs)
        }
    }
}

trait ebml_writer_helpers {
    fn emit_ty(&mut self, ecx: &e::EncodeContext, ty: ty::t);
    fn emit_vstore(&mut self, ecx: &e::EncodeContext, vstore: ty::vstore);
    fn emit_tys(&mut self, ecx: &e::EncodeContext, tys: &[ty::t]);
    fn emit_type_param_def(&mut self,
                           ecx: &e::EncodeContext,
                           type_param_def: &ty::TypeParameterDef);
    fn emit_tpbt(&mut self,
                 ecx: &e::EncodeContext,
                 tpbt: ty::ty_param_bounds_and_ty);
    fn emit_substs(&mut self, ecx: &e::EncodeContext, substs: &ty::substs);
    fn emit_auto_adjustment(&mut self, ecx: &e::EncodeContext, adj: &ty::AutoAdjustment);
}

impl<'a> ebml_writer_helpers for writer::Encoder<'a> {
    fn emit_ty(&mut self, ecx: &e::EncodeContext, ty: ty::t) {
        self.emit_opaque(|this| e::write_type(ecx, this, ty))
    }

    fn emit_vstore(&mut self, ecx: &e::EncodeContext, vstore: ty::vstore) {
        self.emit_opaque(|this| e::write_vstore(ecx, this, vstore))
    }

    fn emit_tys(&mut self, ecx: &e::EncodeContext, tys: &[ty::t]) {
        self.emit_from_vec(tys, |this, ty| this.emit_ty(ecx, *ty))
    }

    fn emit_type_param_def(&mut self,
                           ecx: &e::EncodeContext,
                           type_param_def: &ty::TypeParameterDef) {
        self.emit_opaque(|this| {
            tyencode::enc_type_param_def(this.writer,
                                         ecx.ty_str_ctxt(),
                                         type_param_def)
        })
    }

    fn emit_tpbt(&mut self,
                 ecx: &e::EncodeContext,
                 tpbt: ty::ty_param_bounds_and_ty) {
        self.emit_struct("ty_param_bounds_and_ty", 2, |this| {
            this.emit_struct_field("generics", 0, |this| {
                this.emit_struct("Generics", 2, |this| {
                    this.emit_struct_field("type_param_defs", 0, |this| {
                        this.emit_from_vec(*tpbt.generics.type_param_defs,
                                           |this, type_param_def| {
                            this.emit_type_param_def(ecx, type_param_def);
                        })
                    });
                    this.emit_struct_field("region_param_defs", 1, |this| {
                        tpbt.generics.region_param_defs.encode(this);
                    })
                })
            });
            this.emit_struct_field("ty", 1, |this| {
                this.emit_ty(ecx, tpbt.ty);
            })
        })
    }

    fn emit_substs(&mut self, ecx: &e::EncodeContext, substs: &ty::substs) {
        self.emit_opaque(|this| tyencode::enc_substs(this.writer, ecx.ty_str_ctxt(), substs))
    }

    fn emit_auto_adjustment(&mut self, ecx: &e::EncodeContext, adj: &ty::AutoAdjustment) {
        self.emit_enum("AutoAdjustment", |this| {
            match *adj {
                ty::AutoAddEnv(region, sigil) => {
                    this.emit_enum_variant("AutoAddEnv", 0, 2, |this| {
                        this.emit_enum_variant_arg(0, |this| region.encode(this));
                        this.emit_enum_variant_arg(1, |this| sigil.encode(this));
                    });
                }

                ty::AutoDerefRef(ref auto_deref_ref) => {
                    this.emit_enum_variant("AutoDerefRef", 1, 1, |this| {
                        this.emit_enum_variant_arg(0, |this| auto_deref_ref.encode(this));
                    });
                }

                ty::AutoObject(sigil, region, m, b, def_id, ref substs) => {
                    this.emit_enum_variant("AutoObject", 2, 6, |this| {
                        this.emit_enum_variant_arg(0, |this| sigil.encode(this));
                        this.emit_enum_variant_arg(1, |this| region.encode(this));
                        this.emit_enum_variant_arg(2, |this| m.encode(this));
                        this.emit_enum_variant_arg(3, |this| b.encode(this));
                        this.emit_enum_variant_arg(4, |this| def_id.encode(this));
                        this.emit_enum_variant_arg(5, |this| this.emit_substs(ecx, substs));
                    });
                }
            }
        });
    }
}

trait write_tag_and_id {
    fn tag(&mut self, tag_id: c::astencode_tag, f: |&mut Self|);
    fn id(&mut self, id: ast::NodeId);
}

impl<'a> write_tag_and_id for writer::Encoder<'a> {
    fn tag(&mut self,
           tag_id: c::astencode_tag,
           f: |&mut writer::Encoder<'a>|) {
        self.start_tag(tag_id as uint);
        f(self);
        self.end_tag();
    }

    fn id(&mut self, id: ast::NodeId) {
        self.wr_tagged_u64(c::tag_table_id as uint, id as u64)
    }
}

struct SideTableEncodingIdVisitor<'a,'b> {
    ecx_ptr: *libc::c_void,
    new_ebml_w: &'a mut writer::Encoder<'b>,
    maps: Maps,
}

impl<'a,'b> ast_util::IdVisitingOperation for
        SideTableEncodingIdVisitor<'a,'b> {
    fn visit_id(&self, id: ast::NodeId) {
        // Note: this will cause a copy of ebml_w, which is bad as
        // it is mutable. But I believe it's harmless since we generate
        // balanced EBML.
        //
        // XXX(pcwalton): Don't copy this way.
        let mut new_ebml_w = unsafe {
            self.new_ebml_w.unsafe_clone()
        };
        // See above
        let ecx: &e::EncodeContext = unsafe {
            cast::transmute(self.ecx_ptr)
        };
        encode_side_tables_for_id(ecx, self.maps, &mut new_ebml_w, id)
    }
}

fn encode_side_tables_for_ii(ecx: &e::EncodeContext,
                             maps: Maps,
                             ebml_w: &mut writer::Encoder,
                             ii: &ast::inlined_item) {
    ebml_w.start_tag(c::tag_table as uint);
    let mut new_ebml_w = unsafe {
        ebml_w.unsafe_clone()
    };

    // Because the ast visitor uses @IdVisitingOperation, I can't pass in
    // ecx directly, but /I/ know that it'll be fine since the lifetime is
    // tied to the CrateContext that lives throughout this entire section.
    ast_util::visit_ids_for_inlined_item(ii, &SideTableEncodingIdVisitor {
        ecx_ptr: unsafe {
            cast::transmute(ecx)
        },
        new_ebml_w: &mut new_ebml_w,
        maps: maps,
    });
    ebml_w.end_tag();
}

fn encode_side_tables_for_id(ecx: &e::EncodeContext,
                             maps: Maps,
                             ebml_w: &mut writer::Encoder,
                             id: ast::NodeId) {
    let tcx = ecx.tcx;

    debug!("Encoding side tables for id {}", id);

    {
        let def_map = tcx.def_map.borrow();
        let r = def_map.get().find(&id);
        for def in r.iter() {
            ebml_w.tag(c::tag_table_def, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| (*def).encode(ebml_w));
            })
        }
    }

    {
        let node_types = tcx.node_types.borrow();
        let r = node_types.get().find(&(id as uint));
        for &ty in r.iter() {
            ebml_w.tag(c::tag_table_node_type, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    ebml_w.emit_ty(ecx, *ty);
                })
            })
        }
    }

    {
        let node_type_substs = tcx.node_type_substs.borrow();
        let r = node_type_substs.get().find(&id);
        for tys in r.iter() {
            ebml_w.tag(c::tag_table_node_type_subst, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    ebml_w.emit_tys(ecx, **tys)
                })
            })
        }
    }

    {
        let freevars = tcx.freevars.borrow();
        let r = freevars.get().find(&id);
        for &fv in r.iter() {
            ebml_w.tag(c::tag_table_freevars, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    ebml_w.emit_from_vec(**fv, |ebml_w, fv_entry| {
                        encode_freevar_entry(ebml_w, *fv_entry)
                    })
                })
            })
        }
    }

    let lid = ast::DefId { crate: ast::LOCAL_CRATE, node: id };
    {
        let tcache = tcx.tcache.borrow();
        let r = tcache.get().find(&lid);
        for &tpbt in r.iter() {
            ebml_w.tag(c::tag_table_tcache, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    ebml_w.emit_tpbt(ecx, *tpbt);
                })
            })
        }
    }

    {
        let r = {
            let ty_param_defs = tcx.ty_param_defs.borrow();
            ty_param_defs.get().find(&id).map(|def| *def)
        };
        for type_param_def in r.iter() {
            ebml_w.tag(c::tag_table_param_defs, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    ebml_w.emit_type_param_def(ecx, type_param_def)
                })
            })
        }
    }

    {
        let method_map = maps.method_map.borrow();
        let r = method_map.get().find(&id);
        for &mme in r.iter() {
            ebml_w.tag(c::tag_table_method_map, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    encode_method_map_entry(ecx, ebml_w, *mme)
                })
            })
        }
    }

    {
        let vtable_map = maps.vtable_map.borrow();
        let r = vtable_map.get().find(&id);
        for &dr in r.iter() {
            ebml_w.tag(c::tag_table_vtable_map, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    encode_vtable_res(ecx, ebml_w, *dr);
                })
            })
        }
    }

    {
        let adjustments = tcx.adjustments.borrow();
        let r = adjustments.get().find(&id);
        for adj in r.iter() {
            ebml_w.tag(c::tag_table_adjustments, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    ebml_w.emit_auto_adjustment(ecx, **adj);
                })
            })
        }
    }

    {
        let capture_map = maps.capture_map.borrow();
        let r = capture_map.get().find(&id);
        for &cap_vars in r.iter() {
            ebml_w.tag(c::tag_table_capture_map, |ebml_w| {
                ebml_w.id(id);
                ebml_w.tag(c::tag_table_val, |ebml_w| {
                    ebml_w.emit_from_vec(*cap_vars, |ebml_w, cap_var| {
                        cap_var.encode(ebml_w);
                    })
                })
            })
        }
    }
}

trait doc_decoder_helpers {
    fn as_int(&self) -> int;
    fn opt_child(&self, tag: c::astencode_tag) -> Option<Self>;
}

impl<'a> doc_decoder_helpers for ebml::Doc<'a> {
    fn as_int(&self) -> int { reader::doc_as_u64(*self) as int }
    fn opt_child(&self, tag: c::astencode_tag) -> Option<ebml::Doc<'a>> {
        reader::maybe_get_doc(*self, tag as uint)
    }
}

trait ebml_decoder_decoder_helpers {
    fn read_ty(&mut self, xcx: @ExtendedDecodeContext) -> ty::t;
    fn read_tys(&mut self, xcx: @ExtendedDecodeContext) -> ~[ty::t];
    fn read_type_param_def(&mut self, xcx: @ExtendedDecodeContext)
                           -> ty::TypeParameterDef;
    fn read_ty_param_bounds_and_ty(&mut self, xcx: @ExtendedDecodeContext)
                                -> ty::ty_param_bounds_and_ty;
    fn read_substs(&mut self, xcx: @ExtendedDecodeContext) -> ty::substs;
    fn read_auto_adjustment(&mut self, xcx: @ExtendedDecodeContext) -> ty::AutoAdjustment;
    fn convert_def_id(&mut self,
                      xcx: @ExtendedDecodeContext,
                      source: DefIdSource,
                      did: ast::DefId)
                      -> ast::DefId;

    // Versions of the type reading functions that don't need the full
    // ExtendedDecodeContext.
    fn read_ty_noxcx(&mut self,
                     tcx: ty::ctxt, cdata: @cstore::crate_metadata) -> ty::t;
    fn read_tys_noxcx(&mut self,
                      tcx: ty::ctxt,
                      cdata: @cstore::crate_metadata) -> ~[ty::t];
}

impl<'a> ebml_decoder_decoder_helpers for reader::Decoder<'a> {
    fn read_ty_noxcx(&mut self,
                     tcx: ty::ctxt, cdata: @cstore::crate_metadata) -> ty::t {
        self.read_opaque(|_, doc| {
            tydecode::parse_ty_data(
                doc.data,
                cdata.cnum,
                doc.start,
                tcx,
                |_, id| decoder::translate_def_id(cdata, id))
        })
    }

    fn read_tys_noxcx(&mut self,
                      tcx: ty::ctxt,
                      cdata: @cstore::crate_metadata) -> ~[ty::t] {
        self.read_to_vec(|this| this.read_ty_noxcx(tcx, cdata) )
    }

    fn read_ty(&mut self, xcx: @ExtendedDecodeContext) -> ty::t {
        // Note: regions types embed local node ids.  In principle, we
        // should translate these node ids into the new decode
        // context.  However, we do not bother, because region types
        // are not used during trans.

        return self.read_opaque(|this, doc| {
            debug!("read_ty({})", type_string(doc));

            let ty = tydecode::parse_ty_data(
                doc.data,
                xcx.dcx.cdata.cnum,
                doc.start,
                xcx.dcx.tcx,
                |s, a| this.convert_def_id(xcx, s, a));

            ty
        });

        fn type_string(doc: ebml::Doc) -> ~str {
            let mut str = ~"";
            for i in range(doc.start, doc.end) {
                str.push_char(doc.data[i] as char);
            }
            str
        }
    }

    fn read_tys(&mut self, xcx: @ExtendedDecodeContext) -> ~[ty::t] {
        self.read_to_vec(|this| this.read_ty(xcx) )
    }

    fn read_type_param_def(&mut self, xcx: @ExtendedDecodeContext)
                           -> ty::TypeParameterDef {
        self.read_opaque(|this, doc| {
            tydecode::parse_type_param_def_data(
                doc.data,
                doc.start,
                xcx.dcx.cdata.cnum,
                xcx.dcx.tcx,
                |s, a| this.convert_def_id(xcx, s, a))
        })
    }

    fn read_ty_param_bounds_and_ty(&mut self, xcx: @ExtendedDecodeContext)
                                   -> ty::ty_param_bounds_and_ty {
        self.read_struct("ty_param_bounds_and_ty", 2, |this| {
            ty::ty_param_bounds_and_ty {
                generics: this.read_struct_field("generics", 0, |this| {
                    this.read_struct("Generics", 2, |this| {
                        ty::Generics {
                            type_param_defs:
                                this.read_struct_field("type_param_defs",
                                                       0,
                                                       |this| {
                                    @this.read_to_vec(|this|
                                        this.read_type_param_def(xcx))
                            }),
                            region_param_defs:
                                this.read_struct_field("region_param_defs",
                                                       1,
                                                       |this| {
                                    Decodable::decode(this)
                                })
                        }
                    })
                }),
                ty: this.read_struct_field("ty", 1, |this| {
                    this.read_ty(xcx)
                })
            }
        })
    }

    fn read_substs(&mut self, xcx: @ExtendedDecodeContext) -> ty::substs {
        self.read_opaque(|this, doc| {
            tydecode::parse_substs_data(doc.data,
                                        xcx.dcx.cdata.cnum,
                                        doc.start,
                                        xcx.dcx.tcx,
                                        |s, a| this.convert_def_id(xcx, s, a))
        })
    }

    fn read_auto_adjustment(&mut self, xcx: @ExtendedDecodeContext) -> ty::AutoAdjustment {
        self.read_enum("AutoAdjustment", |this| {
            let variants = ["AutoAddEnv", "AutoDerefRef", "AutoObject"];
            this.read_enum_variant(variants, |this, i| {
                match i {
                    0 => {
                        let region: ty::Region =
                            this.read_enum_variant_arg(0, |this| Decodable::decode(this));
                        let sigil: ast::Sigil =
                            this.read_enum_variant_arg(1, |this| Decodable::decode(this));

                        ty:: AutoAddEnv(region.tr(xcx), sigil)
                    }
                    1 => {
                        let auto_deref_ref: ty::AutoDerefRef =
                            this.read_enum_variant_arg(0, |this| Decodable::decode(this));

                        ty::AutoDerefRef(auto_deref_ref.tr(xcx))
                    }
                    2 => {
                        let sigil: ast::Sigil =
                            this.read_enum_variant_arg(0, |this| Decodable::decode(this));
                        let region: Option<ty::Region> =
                            this.read_enum_variant_arg(1, |this| Decodable::decode(this));
                        let m: ast::Mutability =
                            this.read_enum_variant_arg(2, |this| Decodable::decode(this));
                        let b: ty::BuiltinBounds =
                            this.read_enum_variant_arg(3, |this| Decodable::decode(this));
                        let def_id: ast::DefId =
                            this.read_enum_variant_arg(4, |this| Decodable::decode(this));
                        let substs = this.read_enum_variant_arg(5, |this| this.read_substs(xcx));

                        let region = match region {
                            Some(r) => Some(r.tr(xcx)),
                            None => None
                        };

                        ty::AutoObject(sigil, region, m, b, def_id.tr(xcx), substs)
                    }
                    _ => fail!("bad enum variant for ty::AutoAdjustment")
                }
            })
        })
    }

    fn convert_def_id(&mut self,
                      xcx: @ExtendedDecodeContext,
                      source: tydecode::DefIdSource,
                      did: ast::DefId)
                      -> ast::DefId {
        /*!
         * Converts a def-id that appears in a type.  The correct
         * translation will depend on what kind of def-id this is.
         * This is a subtle point: type definitions are not
         * inlined into the current crate, so if the def-id names
         * a nominal type or type alias, then it should be
         * translated to refer to the source crate.
         *
         * However, *type parameters* are cloned along with the function
         * they are attached to.  So we should translate those def-ids
         * to refer to the new, cloned copy of the type parameter.
         * We only see references to free type parameters in the body of
         * an inlined function. In such cases, we need the def-id to
         * be a local id so that the TypeContents code is able to lookup
         * the relevant info in the ty_param_defs table.
         *
         * *Region parameters*, unfortunately, are another kettle of fish.
         * In such cases, def_id's can appear in types to distinguish
         * shadowed bound regions and so forth. It doesn't actually
         * matter so much what we do to these, since regions are erased
         * at trans time, but it's good to keep them consistent just in
         * case. We translate them with `tr_def_id()` which will map
         * the crate numbers back to the original source crate.
         *
         * It'd be really nice to refactor the type repr to not include
         * def-ids so that all these distinctions were unnecessary.
         */

        let r = match source {
            NominalType | TypeWithId | RegionParameter => xcx.tr_def_id(did),
            TypeParameter => xcx.tr_intern_def_id(did)
        };
        debug!("convert_def_id(source={:?}, did={:?})={:?}", source, did, r);
        return r;
    }
}

fn decode_side_tables(xcx: @ExtendedDecodeContext,
                      ast_doc: ebml::Doc) {
    let dcx = xcx.dcx;
    let tbl_doc = ast_doc.get(c::tag_table as uint);
    reader::docs(tbl_doc, |tag, entry_doc| {
        let id0 = entry_doc.get(c::tag_table_id as uint).as_int();
        let id = xcx.tr_id(id0 as ast::NodeId);

        debug!(">> Side table document with tag 0x{:x} \
                found for id {} (orig {})",
               tag, id, id0);

        match c::astencode_tag::from_uint(tag) {
            None => {
                xcx.dcx.tcx.sess.bug(
                    format!("unknown tag found in side tables: {:x}", tag));
            }
            Some(value) => {
                let val_doc = entry_doc.get(c::tag_table_val as uint);
                let mut val_dsr = reader::Decoder(val_doc);
                let val_dsr = &mut val_dsr;

                match value {
                    c::tag_table_def => {
                        let def = decode_def(xcx, val_doc);
                        let mut def_map = dcx.tcx.def_map.borrow_mut();
                        def_map.get().insert(id, def);
                    }
                    c::tag_table_node_type => {
                        let ty = val_dsr.read_ty(xcx);
                        debug!("inserting ty for node {:?}: {}",
                               id, ty_to_str(dcx.tcx, ty));
                        let mut node_types = dcx.tcx.node_types.borrow_mut();
                        node_types.get().insert(id as uint, ty);
                    }
                    c::tag_table_node_type_subst => {
                        let tys = val_dsr.read_tys(xcx);
                        let mut node_type_substs = dcx.tcx
                                                      .node_type_substs
                                                      .borrow_mut();
                        node_type_substs.get().insert(id, tys);
                    }
                    c::tag_table_freevars => {
                        let fv_info = @val_dsr.read_to_vec(|val_dsr| {
                            @val_dsr.read_freevar_entry(xcx)
                        });
                        let mut freevars = dcx.tcx.freevars.borrow_mut();
                        freevars.get().insert(id, fv_info);
                    }
                    c::tag_table_tcache => {
                        let tpbt = val_dsr.read_ty_param_bounds_and_ty(xcx);
                        let lid = ast::DefId { crate: ast::LOCAL_CRATE, node: id };
                        let mut tcache = dcx.tcx.tcache.borrow_mut();
                        tcache.get().insert(lid, tpbt);
                    }
                    c::tag_table_param_defs => {
                        let bounds = val_dsr.read_type_param_def(xcx);
                        let mut ty_param_defs = dcx.tcx
                                                   .ty_param_defs
                                                   .borrow_mut();
                        ty_param_defs.get().insert(id, bounds);
                    }
                    c::tag_table_method_map => {
                        let entry = val_dsr.read_method_map_entry(xcx);
                        let mut method_map = dcx.maps.method_map.borrow_mut();
                        method_map.get().insert(id, entry);
                    }
                    c::tag_table_vtable_map => {
                        let vtable_res =
                            val_dsr.read_vtable_res(xcx.dcx.tcx,
                                                    xcx.dcx.cdata);
                        let mut vtable_map = dcx.maps.vtable_map.borrow_mut();
                        vtable_map.get().insert(id, vtable_res);
                    }
                    c::tag_table_adjustments => {
                        let adj: @ty::AutoAdjustment = @val_dsr.read_auto_adjustment(xcx);
                        let mut adjustments = dcx.tcx
                                                 .adjustments
                                                 .borrow_mut();
                        adjustments.get().insert(id, adj);
                    }
                    c::tag_table_capture_map => {
                        let cvars =
                            at_vec::to_managed_move(
                                val_dsr.read_to_vec(
                                    |val_dsr| val_dsr.read_capture_var(xcx)));
                        let mut capture_map = dcx.maps
                                                 .capture_map
                                                 .borrow_mut();
                        capture_map.get().insert(id, cvars);
                    }
                    _ => {
                        xcx.dcx.tcx.sess.bug(
                            format!("unknown tag found in side tables: {:x}", tag));
                    }
                }
            }
        }

        debug!(">< Side table doc loaded");
        true
    });
}

// ______________________________________________________________________
// Testing of astencode_gen

#[cfg(test)]
fn encode_item_ast(ebml_w: &mut writer::Encoder, item: @ast::item) {
    ebml_w.start_tag(c::tag_tree as uint);
    (*item).encode(ebml_w);
    ebml_w.end_tag();
}

#[cfg(test)]
fn decode_item_ast(par_doc: ebml::Doc) -> @ast::item {
    let chi_doc = par_doc.get(c::tag_tree as uint);
    let mut d = reader::Decoder(chi_doc);
    @Decodable::decode(&mut d)
}

#[cfg(test)]
trait fake_ext_ctxt {
    fn cfg(&self) -> ast::CrateConfig;
    fn parse_sess(&self) -> @parse::ParseSess;
    fn call_site(&self) -> Span;
    fn ident_of(&self, st: &str) -> ast::Ident;
}

#[cfg(test)]
type fake_session = @parse::ParseSess;

#[cfg(test)]
impl fake_ext_ctxt for fake_session {
    fn cfg(&self) -> ast::CrateConfig { ~[] }
    fn parse_sess(&self) -> @parse::ParseSess { *self }
    fn call_site(&self) -> Span {
        codemap::Span {
            lo: codemap::BytePos(0),
            hi: codemap::BytePos(0),
            expn_info: None
        }
    }
    fn ident_of(&self, st: &str) -> ast::Ident {
        token::str_to_ident(st)
    }
}

#[cfg(test)]
fn mk_ctxt() -> @fake_ext_ctxt {
    @parse::new_parse_sess(None) as @fake_ext_ctxt
}

#[cfg(test)]
fn roundtrip(in_item: Option<@ast::item>) {
    use std::io::Decorator;
    use std::io::mem::MemWriter;

    let in_item = in_item.unwrap();
    let mut wr = MemWriter::new();
    {
        let mut ebml_w = writer::Encoder(&mut wr);
        encode_item_ast(&mut ebml_w, in_item);
    }
    let ebml_doc = reader::Doc(wr.inner_ref().as_slice());
    let out_item = decode_item_ast(ebml_doc);

    assert_eq!(in_item, out_item);
}

#[test]
fn test_basic() {
    let cx = mk_ctxt();
    roundtrip(quote_item!(cx,
        fn foo() {}
    ));
}

#[test]
fn test_smalltalk() {
    let cx = mk_ctxt();
    roundtrip(quote_item!(cx,
        fn foo() -> int { 3 + 4 } // first smalltalk program ever executed.
    ));
}

#[test]
fn test_more() {
    let cx = mk_ctxt();
    roundtrip(quote_item!(cx,
        fn foo(x: uint, y: uint) -> uint {
            let z = x + y;
            return z;
        }
    ));
}

#[test]
fn test_simplification() {
    let cx = mk_ctxt();
    let item = quote_item!(cx,
        fn new_int_alist<B>() -> alist<int, B> {
            fn eq_int(a: int, b: int) -> bool { a == b }
            return alist {eq_fn: eq_int, data: ~[]};
        }
    ).unwrap();
    let item_in = e::ii_item_ref(item);
    let item_out = simplify_ast(item_in);
    let item_exp = ast::ii_item(quote_item!(cx,
        fn new_int_alist<B>() -> alist<int, B> {
            return alist {eq_fn: eq_int, data: ~[]};
        }
    ).unwrap());
    match (item_out, item_exp) {
      (ast::ii_item(item_out), ast::ii_item(item_exp)) => {
        assert!(pprust::item_to_str(item_out,
                                    token::get_ident_interner())
                     == pprust::item_to_str(item_exp,
                                            token::get_ident_interner()));
      }
      _ => fail!()
    }
}
