package com.dotcms.api.system.event.dao;

import java.util.Collection;

import com.dotcms.api.system.event.dto.SystemEventDTO;
import com.dotmarketing.exception.DotDataException;

/**
 * This data access object provides access to information associated to System
 * Events. A System Event can be generated by any component or third-party code
 * in the application so that it can be processed and make dotCMS or any other
 * system react to it.
 * 
 * @author Jose Castro
 * @version 3.7
 * @since Jul 11, 2016
 *
 */
public interface SystemEventsDAO {

	/**
	 * Saves a {@link SystemEventDTO} in the database.
	 * 
	 * @param event
	 *            - The system event reported by a component running in the
	 *            application.
	 * @throws DotDataException
	 *             An error occurred when executing the SQL query. Please check
	 *             your database and/or query syntax.
	 */
	public void add(SystemEventDTO systemEvent) throws DotDataException;

	/**
	 * Returns a list of {@link SystemEventDTO} objects that were created from a
	 * specific date up to the present.
	 * 
	 * @param fromDate
	 *            - The date (in milliseconds) from which system events will be
	 *            selected.
	 * @return The collection of {@link SystemEventDTO} objects.
	 * @throws DotDataException
	 *             An error occurred when executing the SQL query. Please check
	 *             your database and/or query syntax.
	 */
	public Collection<SystemEventDTO> getEventsSince(long fromDate) throws DotDataException;

	/**
	 * Returns the list of all {@link SystemEventDTO} objects in the database.
	 * 
	 * @return The complete collection of {@link SystemEventDTO} objects.
	 * @throws DotDataException
	 *             An error occurred when executing the SQL query. Please check
	 *             your database and/or query syntax.
	 */
	public Collection<SystemEventDTO> getAll() throws DotDataException;

	/**
	 * Deletes all the {@link SystemEventDTO} objects up to the specified date.
	 * 
	 * @param toDate
	 *            - The date (in milliseconds) up to which all system events
	 *            will be deleted.
	 * @throws DotDataException
	 *             An error occurred when executing the SQL query. Please check
	 *             your database and/or query syntax.
	 */
	public void deleteEvents(long toDate) throws DotDataException;

	/**
	 * Deletes all the {@link SystemEventDTO} objects that fall into the
	 * specified date range.
	 * 
	 * @param fromDate
	 *            - The lower boundary of the date range (in milliseconds).
	 * @param toDate
	 *            - The upper boundary of the date range (in milliseconds).
	 * @throws DotDataException
	 *             An error occurred when executing the SQL query. Please check
	 *             your database and/or query syntax.
	 */
	public void deleteEvents(long fromDate, long toDate) throws DotDataException;

	/**
	 * Deletes all the {@link SystemEventDTO} objects from the database.
	 * 
	 * @throws DotDataException
	 *             An error occurred when executing the SQL query. Please check
	 *             your database and/or query syntax.
	 */
	public void deleteAll() throws DotDataException;

}
