package com.dotmarketing.portlets.languagesmanager.business;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.dotmarketing.exception.DotDataException;
import com.dotmarketing.portlets.languagesmanager.model.Language;
import com.dotmarketing.portlets.languagesmanager.model.LanguageKey;

/**
 * Provides access to information related to the different languages that can be added to the
 * system.
 * <p>
 * Multiple languages are fully supported in dotCMS. Back-end users can contribute the same content
 * in multiple languages and front end users may choose to view content in the supported languages.
 * After saving the new language users will be able to open, search for, or create a piece of
 * content of any Content Type in the new language.
 * 
 * @author root
 * @version N/A
 * @since Mar 22, 2012
 *
 */
public interface LanguageAPI {

    /**
     * 
     * @return
     */
	public Language createDefaultLanguage();

	/**
	 * 
	 * @param language
	 */
	public void deleteLanguage(Language language);

	/**
	 * Deletes a fallback Language, that you obtain with {@link #getFallbackLanguage(String)}
	 * @param fallbackPortugueseLanguage Language
	 */
	void deleteFallbackLanguage(Language fallbackPortugueseLanguage);

	/**
	 * 
	 * @param language
	 */
	public void saveLanguage(Language language);

	/**
	 * 
	 * @return
	 */
	public Language getDefaultLanguage();

	/**
	 * 
	 * @param languageCode
	 * @param countryCode
	 * @return
	 */
	public Language getLanguage(String languageCode, String countryCode);

	/**
	 * 
	 * @param id
	 * @return
	 */
	public Language getLanguage(String id);

	/**
	 * 
	 * @param id
	 * @return
	 */
	public Language getLanguage(long id);

	/**
	 * 
	 * @return
	 */
	public List<Language> getLanguages();

	/**
	 * 
	 * @param id
	 * @return
	 */
    public boolean hasLanguage (String id);

    /**
     * 
     * @param id
     * @return
     */
    public boolean hasLanguage (long id);

    /**
     * 
     * @param languageCode
     * @param countryCode
     * @return
     */
    public boolean hasLanguage (String languageCode, String countryCode);

    /**
     * 
     * @param id
     * @param langId
     * @return
     */
	public String getLanguageCodeAndCountry(long id, String langId);

	/**
	 * Retrieve the keys general for the language
	 * @param lang
	 * @return
	 */
	public List<LanguageKey> getLanguageKeys(Language lang);

	/**
	 * Retrieves the keys for the given language and specific for the given country
	 * @param langCode
	 * @param countryCode
	 * @return
	 */
	public List<LanguageKey> getLanguageKeys(String langCode);

	/**
	 * 
	 * @param langCode
	 * @param countryCode
	 * @return
	 */
	public List<LanguageKey> getLanguageKeys(String langCode, String countryCode);

	/**
	 * 
	 * @param lang
	 */
	public void createLanguageFiles(Language lang);

	/**
	 * 
	 * @param lang
	 * @param generalKeys
	 * @param specificKeys
	 * @param toDeleteKeys
	 * @throws DotDataException
	 */
	public void saveLanguageKeys(Language lang, Map<String, String> generalKeys, Map<String, String> specificKeys, Set<String> toDeleteKeys) throws DotDataException;

    /**
     * Returns a internationalized value for a given key and language
     *
     * @param lang
     * @param key
     * @return
     */
    public String getStringKey(Language lang, String key);

    /**
     * 
     * @param lang
     * @param key
     * @return
     */
    public int getIntKey(Language lang, String key);

    /**
     * 
     * @param lang
     * @param key
     * @param defaultVal
     * @return
     */
    public int getIntKey(Language lang, String key, int defaultVal);

    /**
     * 
     * @param lang
     * @param key
     * @return
     */
    public float getFloatKey(Language lang, String key);

    /**
     * 
     * @param lang
     * @param key
     * @param defaultVal
     * @return
     */
    public float getFloatKey(Language lang, String key, float defaultVal);

    /**
     * 
     * @param lang
     * @param key
     * @return
     */
    public boolean getBooleanKey(Language lang, String key);

    /**
     * 
     * @param lang
     * @param key
     * @param defaultVal
     * @return
     */
    public boolean getBooleanKey(Language lang, String key, boolean defaultVal);
    
    /**
     * Clear the language cache
     *
     */
    public void clearCache();

    /**
     * Checks if the parameter is an asset type language 
     * @param id
     * @return true if its a language type, otherwise returns false
     */
    public boolean isAssetTypeLanguage(String id);

    /**
     * Returns the fallback {@link Language} object for a specific language code.
     * <p>
     * When adding Language Variables, keys are usually looked up by language and country. However,
     * language-generic keys can be defined for a given language - without a country - so that keys
     * that are not found using a language-country combination can be looked up using the fallback
     * language instead.
     * 
     * @param languageCode - The ISO code of the fallback language.
     * @return The fallback {@link Language} object.
     */
    public Language getFallbackLanguage(final String languageCode);



}
