/*
 * Copyright 2007 Guy Van den Broeck
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.dotmarketing.util.diff.tag;

import com.dotmarketing.util.diff.output.TextDiffOutput;
import com.dotcms.repackage.org.xml.sax.ContentHandler;
import com.dotcms.repackage.org.xml.sax.SAXException;
import com.dotcms.repackage.org.xml.sax.helpers.AttributesImpl;

/**
 * Outputs the diff result as HTML elements to a SAX ContentHandler. The
 * startDocument and endDocument events are not generated by this class. This
 * version is used for tag based diff results.
 */
public class TagSaxDiffOutput implements TextDiffOutput{

    private ContentHandler consumer;

    public TagSaxDiffOutput(ContentHandler consumer) throws SAXException {
        this.consumer = consumer;
    }

    /**
     * {@inheritDoc}
     */
    public void addClearPart(String text) throws Exception {
        addBasicText(text);
    }

    private boolean insideTag = false;

    private void addBasicText(String text) throws SAXException {
        char[] c = text.toCharArray();

        AttributesImpl noattrs = new AttributesImpl();

        for (int i = 0; i < c.length; i++) {
            switch (c[i]) {
            case '\n':
                consumer.startElement("", "br", "br", noattrs);
                consumer.endElement("", "br", "br");
                consumer.characters("\n".toCharArray(), 0, "\n".length());
                break;
            case '<':
                if (insideTag == false) {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addAttribute("", "class", "class", "CDATA",
                            "diff-tag-html");
                    consumer.startElement("", "span", "span", attrs);
                    insideTag = true;
                } else {
                    consumer.endElement("", "span", "span");
                    insideTag = false;
                }
                consumer.characters("<".toCharArray(), 0, "<".length());
                break;
            case '>':
                consumer.characters(">".toCharArray(), 0, ">".length());

                if (insideTag == true) {
                    consumer.endElement("", "span", "span");
                    insideTag = false;
                }
                break;
            default:
                consumer.characters(c, i, 1);
            }
        }
    }

    private int removedID = 1;

    private int addedID = 1;

    /**
     * {@inheritDoc}
     */
    public void addRemovedPart(String text) throws Exception {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addAttribute("", "class", "class", "CDATA", "diff-tag-removed");
        attrs.addAttribute("", "id", "id", "CDATA", "removed" + removedID);
        attrs.addAttribute("", "title", "title", "CDATA", "#removed"
                + removedID);
        removedID++;
        consumer.startElement("", "span", "span", attrs);
        addBasicText(text);
        consumer.endElement("", "span", "span");
    }

    /**
     * {@inheritDoc}
     */
    public void addAddedPart(String text) throws Exception {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addAttribute("", "class", "class", "CDATA", "diff-tag-added");
        attrs.addAttribute("", "id", "id", "CDATA", "added" + addedID);
        attrs.addAttribute("", "title", "title", "CDATA", "#added" + addedID);
        addedID++;
        consumer.startElement("", "span", "span", attrs);
        addBasicText(text);
        consumer.endElement("", "span", "span");
    }
}
