//===----------------------------------------------------------------------===//
//
// Part of libcu++, the C++ Standard Library for your entire system,
// under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES.
//
//===----------------------------------------------------------------------===//

#include <cuda/std/__charconv_>
#include <cuda/std/array>
#include <cuda/std/cstddef>
#include <cuda/std/cstdint>
#include <cuda/std/cstring>
#include <cuda/std/type_traits>
#include <cuda/utility>

#include "test_macros.h"

constexpr int first_base = 2;
constexpr int last_base  = 36;

struct TestItem
{
  cuda::std::int64_t val;
  const char* str_signed;
  const char* str_unsigned;
};

template <int Base>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items();

// Source code for the generation of the test items
// #include <iostream>
// #include <charconv>
// #include <cstdint>
// #include <cstddef>
// #include <type_traits>
// #include <limits>
//
// constexpr std::int64_t list[] = {
//   0,
//   1,
//   87,
//   194,
//   6'427,
//   12'345'312,
//   33'312'234'991,
//   901'369'000'000,
//   1'579'542'123'461'234,
//   123'345'345'123'123'321,
//   9'223'372'036'854'775'806,
//   9'223'372'036'854'775'807,
//   -1,
//   -409,
//   -1'412,
//   -8'166,
//   -90'000'001,
//   -790'301'243,
//   -8'999'099'123'732,
//   -542'185'444'123'566'432,
//   -9'223'372'036'854'775'807,
//   -9'223'372'036'854'775'807 - 1,
// };
//
// int main()
// {
//   for (int base = 2; base <= 36; ++base)
//   {
//     std::cout <<
// "\n"
// "template <>\n"
// "__host__ __device__ constexpr cuda::std::array<TestItem, " << std::size(list) << "> get_test_items<" << base <<
// ">()\n" <<
// "{\n"
// "  return {{\n";
//     for (auto v : list)
//     {
//       constexpr std::size_t buff_size = 150;
//       char signed_buff[buff_size]{};
//       char unsigned_buff[buff_size]{};
//       std::to_chars(signed_buff, signed_buff + buff_size, v, base);
//       std::to_chars(unsigned_buff, unsigned_buff + buff_size, static_cast<std::make_unsigned_t<decltype(v)>>(v),
//       base);
//
//       std::cout <<
// "    TestItem{";
//       if (v == std::numeric_limits<decltype(v)>::min())
//       {
//         std::cout << (-std::numeric_limits<decltype(v)>::max()) << " - 1";
//       }
//       else
//       {
//         std::cout << v;
//       }
//       std::cout << ", \"" << signed_buff << "\", \"" << unsigned_buff << "\"},\n";
//     }
//     std::cout <<
// "  }};\n" <<
// "}\n";
//   }
// }

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<2>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "1010111", "1010111"},
    TestItem{194, "11000010", "11000010"},
    TestItem{6427, "1100100011011", "1100100011011"},
    TestItem{12345312, "101111000101111111100000", "101111000101111111100000"},
    TestItem{33312234991, "11111000001100100000101110111101111", "11111000001100100000101110111101111"},
    TestItem{901369000000, "1101000111011101110001110111000001000000", "1101000111011101110001110111000001000000"},
    TestItem{1579542123461234,
             "101100111001001010111001111001011010011101001110010",
             "101100111001001010111001111001011010011101001110010"},
    TestItem{123345345123123321,
             "110110110001101011111000000011000110000110000000001111001",
             "110110110001101011111000000011000110000110000000001111001"},
    TestItem{9223372036854775806,
             "111111111111111111111111111111111111111111111111111111111111110",
             "111111111111111111111111111111111111111111111111111111111111110"},
    TestItem{9223372036854775807,
             "111111111111111111111111111111111111111111111111111111111111111",
             "111111111111111111111111111111111111111111111111111111111111111"},
    TestItem{-1, "-1", "1111111111111111111111111111111111111111111111111111111111111111"},
    TestItem{-409, "-110011001", "1111111111111111111111111111111111111111111111111111111001100111"},
    TestItem{-1412, "-10110000100", "1111111111111111111111111111111111111111111111111111101001111100"},
    TestItem{-8166, "-1111111100110", "1111111111111111111111111111111111111111111111111110000000011010"},
    TestItem{
      -90000001, "-101010111010100101010000001", "1111111111111111111111111111111111111010101000101011010101111111"},
    TestItem{-790301243,
             "-101111000110110000101000111011",
             "1111111111111111111111111111111111010000111001001111010111000101"},
    TestItem{-8999099123732,
             "-10000010111101000100000110110100100000010100",
             "1111111111111111111101111101000010111011111001001011011111101100"},
    TestItem{-542185444123566432,
             "-11110000110001110101100100111101111010111110110000101100000",
             "1111100001111001110001010011011000010000101000001001111010100000"},
    TestItem{-9223372036854775807,
             "-111111111111111111111111111111111111111111111111111111111111111",
             "1000000000000000000000000000000000000000000000000000000000000001"},
    TestItem{-9223372036854775807 - 1,
             "-1000000000000000000000000000000000000000000000000000000000000000",
             "1000000000000000000000000000000000000000000000000000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<3>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "10020", "10020"},
    TestItem{194, "21012", "21012"},
    TestItem{6427, "22211001", "22211001"},
    TestItem{12345312, "212020012120210", "212020012120210"},
    TestItem{33312234991, "10011222120211212121011", "10011222120211212121011"},
    TestItem{901369000000, "10012011120221210111012001", "10012011120221210111012001"},
    TestItem{1579542123461234, "21200010200202022102002210100212", "21200010200202022102002210100212"},
    TestItem{123345345123123321, "211012002011121211111200120200210010", "211012002011121211111200120200210010"},
    TestItem{
      9223372036854775806, "2021110011022210012102010021220101220220", "2021110011022210012102010021220101220220"},
    TestItem{
      9223372036854775807, "2021110011022210012102010021220101220221", "2021110011022210012102010021220101220221"},
    TestItem{-1, "-1", "11112220022122120101211020120210210211220"},
    TestItem{-409, "-120011", "11112220022122120101211020120210210021210"},
    TestItem{-1412, "-1221022", "11112220022122120101211020120210201220122"},
    TestItem{-8166, "-102012110", "11112220022122120101211020120210101122111"},
    TestItem{-90000001, "-20021100110210101", "11112220022122120101211000022110100001120"},
    TestItem{-790301243, "-2001002002110111102", "11112220022122120101202012111201100100112"},
    TestItem{-8999099123732, "-1011212022020200122120102122", "11112220022121101112111222220011020102022"},
    TestItem{
      -542185444123566432, "-10121112100201100020210202021002222220", "11102021210021212001120102211112200212001"},
    TestItem{
      -9223372036854775807, "-2021110011022210012102010021220101220221", "2021110011022210012102010021220101221000"},
    TestItem{-9223372036854775807 - 1,
             "-2021110011022210012102010021220101220222",
             "2021110011022210012102010021220101220222"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<4>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "1113", "1113"},
    TestItem{194, "3002", "3002"},
    TestItem{6427, "1210123", "1210123"},
    TestItem{12345312, "233011333200", "233011333200"},
    TestItem{33312234991, "133001210011313233", "133001210011313233"},
    TestItem{901369000000, "31013131301313001000", "31013131301313001000"},
    TestItem{1579542123461234, "11213021113033023103221302", "11213021113033023103221302"},
    TestItem{123345345123123321, "12312031133000120300300001321", "12312031133000120300300001321"},
    TestItem{9223372036854775806, "13333333333333333333333333333332", "13333333333333333333333333333332"},
    TestItem{9223372036854775807, "13333333333333333333333333333333", "13333333333333333333333333333333"},
    TestItem{-1, "-1", "33333333333333333333333333333333"},
    TestItem{-409, "-12121", "33333333333333333333333333321213"},
    TestItem{-1412, "-112010", "33333333333333333333333333221330"},
    TestItem{-8166, "-1333212", "33333333333333333333333332000122"},
    TestItem{-90000001, "-11113110222001", "33333333333333333322220223111333"},
    TestItem{-790301243, "-233012300220323", "33333333333333333100321033113011"},
    TestItem{-8999099123732, "-2002331010012310200110", "33333333331331002323321023133230"},
    TestItem{-542185444123566432, "-132012032230213233113312011200", "33201321301103120100220021322200"},
    TestItem{-9223372036854775807, "-13333333333333333333333333333333", "20000000000000000000000000000001"},
    TestItem{-9223372036854775807 - 1, "-20000000000000000000000000000000", "20000000000000000000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<5>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "322", "322"},
    TestItem{194, "1234", "1234"},
    TestItem{6427, "201202", "201202"},
    TestItem{12345312, "11130022222", "11130022222"},
    TestItem{33312234991, "1021210413004431", "1021210413004431"},
    TestItem{901369000000, "104232000431000000", "104232000431000000"},
    TestItem{1579542123461234, "3124013204232101224414", "3124013204232101224414"},
    TestItem{123345345123123321, "2013314110113303004421241", "2013314110113303004421241"},
    TestItem{9223372036854775806, "1104332401304422434310311211", "1104332401304422434310311211"},
    TestItem{9223372036854775807, "1104332401304422434310311212", "1104332401304422434310311212"},
    TestItem{-1, "-1", "2214220303114400424121122430"},
    TestItem{-409, "-3114", "2214220303114400424121114312"},
    TestItem{-1412, "-21122", "2214220303114400424121101304"},
    TestItem{-8166, "-230131", "2214220303114400424120342300"},
    TestItem{-90000001, "-141020000001", "2214220303114400233101122430"},
    TestItem{-790301243, "-3104304114433", "2214220303114342314312002443"},
    TestItem{-8999099123732, "-2134420123333424412", "2214220300424430300232143014"},
    TestItem{-542185444123566432, "-14022010312304031113111212", "2200143242302041343003011214"},
    TestItem{-9223372036854775807, "-1104332401304422434310311212", "1104332401304422434310311214"},
    TestItem{-9223372036854775807 - 1, "-1104332401304422434310311213", "1104332401304422434310311213"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<6>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "223", "223"},
    TestItem{194, "522", "522"},
    TestItem{6427, "45431", "45431"},
    TestItem{12345312, "1120334120", "1120334120"},
    TestItem{33312234991, "23145312450051", "23145312450051"},
    TestItem{901369000000, "1530025545513344", "1530025545513344"},
    TestItem{1579542123461234, "23315223344542344122", "23315223344542344122"},
    TestItem{123345345123123321, "5342301353143151212133", "5342301353143151212133"},
    TestItem{9223372036854775806, "1540241003031030222122210", "1540241003031030222122210"},
    TestItem{9223372036854775807, "1540241003031030222122211", "1540241003031030222122211"},
    TestItem{-1, "-1", "3520522010102100444244423"},
    TestItem{-409, "-1521", "3520522010102100444242503"},
    TestItem{-1412, "-10312", "3520522010102100444234112"},
    TestItem{-8166, "-101450", "3520522010102100444142534"},
    TestItem{-90000001, "-12533002401", "3520522010102043511242023"},
    TestItem{-790301243, "-210230522015", "3520522010101450213322405"},
    TestItem{-8999099123732, "-31050043513355112", "3520521535012012530445312"},
    TestItem{-542185444123566432, "-40414324110253350231040", "3440103241551403054013344"},
    TestItem{-9223372036854775807, "-1540241003031030222122211", "1540241003031030222122213"},
    TestItem{-9223372036854775807 - 1, "-1540241003031030222122212", "1540241003031030222122212"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<7>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "153", "153"},
    TestItem{194, "365", "365"},
    TestItem{6427, "24511", "24511"},
    TestItem{12345312, "206635110", "206635110"},
    TestItem{33312234991, "2256336216430", "2256336216430"},
    TestItem{901369000000, "122056523041120", "122056523041120"},
    TestItem{1579542123461234, "653464103206202312", "653464103206202312"},
    TestItem{123345345123123321, "135513531051312220002", "135513531051312220002"},
    TestItem{9223372036854775806, "22341010611245052052266", "22341010611245052052266"},
    TestItem{9223372036854775807, "22341010611245052052300", "22341010611245052052300"},
    TestItem{-1, "-1", "45012021522523134134601"},
    TestItem{-409, "-1123", "45012021522523134133446"},
    TestItem{-1412, "-4055", "45012021522523134130514"},
    TestItem{-8166, "-32544", "45012021522523134102025"},
    TestItem{-90000001, "-2141662450", "45012021522520662142122"},
    TestItem{-790301243, "-25404310154", "45012021522464426524415"},
    TestItem{-8999099123732, "-1616110031306654", "45012016603413102524615"},
    TestItem{-542185444123566432, "-653644634443563304050", "44025043555046240530522"},
    TestItem{-9223372036854775807, "-22341010611245052052300", "22341010611245052052302"},
    TestItem{-9223372036854775807 - 1, "-22341010611245052052301", "22341010611245052052301"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<8>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "127", "127"},
    TestItem{194, "302", "302"},
    TestItem{6427, "14433", "14433"},
    TestItem{12345312, "57057740", "57057740"},
    TestItem{33312234991, "370144056757", "370144056757"},
    TestItem{901369000000, "15073561670100", "15073561670100"},
    TestItem{1579542123461234, "54711271713235162", "54711271713235162"},
    TestItem{123345345123123321, "6661537003060600171", "6661537003060600171"},
    TestItem{9223372036854775806, "777777777777777777776", "777777777777777777776"},
    TestItem{9223372036854775807, "777777777777777777777", "777777777777777777777"},
    TestItem{-1, "-1", "1777777777777777777777"},
    TestItem{-409, "-631", "1777777777777777777147"},
    TestItem{-1412, "-2604", "1777777777777777775174"},
    TestItem{-8166, "-17746", "1777777777777777760032"},
    TestItem{-90000001, "-527245201", "1777777777777250532577"},
    TestItem{-790301243, "-5706605073", "1777777777772071172705"},
    TestItem{-8999099123732, "-202750406644024", "1777777575027371133754"},
    TestItem{-542185444123566432, "-36061654475727660540", "1741716123302050117240"},
    TestItem{-9223372036854775807, "-777777777777777777777", "1000000000000000000001"},
    TestItem{-9223372036854775807 - 1, "-1000000000000000000000", "1000000000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<9>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "106", "106"},
    TestItem{194, "235", "235"},
    TestItem{6427, "8731", "8731"},
    TestItem{12345312, "25205523", "25205523"},
    TestItem{33312234991, "104876755534", "104876755534"},
    TestItem{901369000000, "3164527714161", "3164527714161"},
    TestItem{1579542123461234, "7603622272083325", "7603622272083325"},
    TestItem{123345345123123321, "735064554450520703", "735064554450520703"},
    TestItem{9223372036854775806, "67404283172107811826", "67404283172107811826"},
    TestItem{9223372036854775807, "67404283172107811827", "67404283172107811827"},
    TestItem{-1, "-1", "145808576354216723756"},
    TestItem{-409, "-504", "145808576354216723253"},
    TestItem{-1412, "-1838", "145808576354216721818"},
    TestItem{-8166, "-12173", "145808576354216711574"},
    TestItem{-90000001, "-207313711", "145808576354008410046"},
    TestItem{-790301243, "-2032073442", "145808576352174640315"},
    TestItem{-8999099123732, "-34768220576378", "145808541474886136368"},
    TestItem{-542185444123566432, "-3545321306722232886", "142253255046384480761"},
    TestItem{-9223372036854775807, "-67404283172107811827", "67404283172107811830"},
    TestItem{-9223372036854775807 - 1, "-67404283172107811828", "67404283172107811828"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<10>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "87", "87"},
    TestItem{194, "194", "194"},
    TestItem{6427, "6427", "6427"},
    TestItem{12345312, "12345312", "12345312"},
    TestItem{33312234991, "33312234991", "33312234991"},
    TestItem{901369000000, "901369000000", "901369000000"},
    TestItem{1579542123461234, "1579542123461234", "1579542123461234"},
    TestItem{123345345123123321, "123345345123123321", "123345345123123321"},
    TestItem{9223372036854775806, "9223372036854775806", "9223372036854775806"},
    TestItem{9223372036854775807, "9223372036854775807", "9223372036854775807"},
    TestItem{-1, "-1", "18446744073709551615"},
    TestItem{-409, "-409", "18446744073709551207"},
    TestItem{-1412, "-1412", "18446744073709550204"},
    TestItem{-8166, "-8166", "18446744073709543450"},
    TestItem{-90000001, "-90000001", "18446744073619551615"},
    TestItem{-790301243, "-790301243", "18446744072919250373"},
    TestItem{-8999099123732, "-8999099123732", "18446735074610427884"},
    TestItem{-542185444123566432, "-542185444123566432", "17904558629585985184"},
    TestItem{-9223372036854775807, "-9223372036854775807", "9223372036854775809"},
    TestItem{-9223372036854775807 - 1, "-9223372036854775808", "9223372036854775808"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<11>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "7a", "7a"},
    TestItem{194, "167", "167"},
    TestItem{6427, "4913", "4913"},
    TestItem{12345312, "6a72241", "6a72241"},
    TestItem{33312234991, "13144985222", "13144985222"},
    TestItem{901369000000, "3182a5335857", "3182a5335857"},
    TestItem{1579542123461234, "41832206412aa29", "41832206412aa29"},
    TestItem{123345345123123321, "275896a0116541880", "275896a0116541880"},
    TestItem{9223372036854775806, "1728002635214590696", "1728002635214590696"},
    TestItem{9223372036854775807, "1728002635214590697", "1728002635214590697"},
    TestItem{-1, "-1", "335500516a429071284"},
    TestItem{-409, "-342", "335500516a429070a43"},
    TestItem{-1412, "-1074", "335500516a429070211"},
    TestItem{-8166, "-6154", "335500516a429066131"},
    TestItem{-90000001, "-46891373", "335500516a39228aa12"},
    TestItem{-790301243, "-376117026", "335500516a062a5525a"},
    TestItem{-8999099123732, "-295a550938493", "335500230a9882338a2"},
    TestItem{-542185444123566432, "-108881a9a629422912", "324723317a8aa749473"},
    TestItem{-9223372036854775807, "-1728002635214590697", "1728002635214590699"},
    TestItem{-9223372036854775807 - 1, "-1728002635214590698", "1728002635214590698"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<12>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "73", "73"},
    TestItem{194, "142", "142"},
    TestItem{6427, "3877", "3877"},
    TestItem{12345312, "4174340", "4174340"},
    TestItem{33312234991, "6558249927", "6558249927"},
    TestItem{901369000000, "1268367a2854", "1268367a2854"},
    TestItem{1579542123461234, "1291a5a1928b042", "1291a5a1928b042"},
    TestItem{123345345123123321, "800a002296817049", "800a002296817049"},
    TestItem{9223372036854775806, "41a792678515120366", "41a792678515120366"},
    TestItem{9223372036854775807, "41a792678515120367", "41a792678515120367"},
    TestItem{-1, "-1", "839365134a2a240713"},
    TestItem{-409, "-2a1", "839365134a2a240433"},
    TestItem{-1412, "-998", "839365134a2a23b938"},
    TestItem{-8166, "-4886", "839365134a2a237a4a"},
    TestItem{-90000001, "-26183401", "839365134a04079313"},
    TestItem{-790301243, "-1a080630b", "839365134849636405"},
    TestItem{-8999099123732, "-1014104108a98", "839364120926133838"},
    TestItem{-542185444123566432, "-2b235857512655480", "80a12b4995177a7254"},
    TestItem{-9223372036854775807, "-41a792678515120367", "41a792678515120369"},
    TestItem{-9223372036854775807 - 1, "-41a792678515120368", "41a792678515120368"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<13>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "69", "69"},
    TestItem{194, "11c", "11c"},
    TestItem{6427, "2c05", "2c05"},
    TestItem{12345312, "2733225", "2733225"},
    TestItem{33312234991, "31ab66c369", "31ab66c369"},
    TestItem{901369000000, "66cca2a2691", "66cca2a2691"},
    TestItem{1579542123461234, "52a4924c3c889a", "52a4924c3c889a"},
    TestItem{123345345123123321, "25432c41a02226b3", "25432c41a02226b3"},
    TestItem{9223372036854775806, "10b269549075433c36", "10b269549075433c36"},
    TestItem{9223372036854775807, "10b269549075433c37", "10b269549075433c37"},
    TestItem{-1, "-1", "219505a9511a867b72"},
    TestItem{-409, "-256", "219505a9511a86791a"},
    TestItem{-1412, "-848", "219505a9511a867328"},
    TestItem{-8166, "-3942", "219505a9511a864231"},
    TestItem{-90000001, "-15851c50", "219505a95105015c23"},
    TestItem{-790301243, "-c7968494", "219505a95022bcc6ac"},
    TestItem{-8999099123732, "-5037c434a503", "21950559162651a670"},
    TestItem{-542185444123566432, "-a79189bcc79c7477", "20ba441c6122b706c9"},
    TestItem{-9223372036854775807, "-10b269549075433c37", "10b269549075433c39"},
    TestItem{-9223372036854775807 - 1, "-10b269549075433c38", "10b269549075433c38"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<14>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "63", "63"},
    TestItem{194, "dc", "dc"},
    TestItem{6427, "24b1", "24b1"},
    TestItem{12345312, "18d5040", "18d5040"},
    TestItem{33312234991, "18802cb487", "18802cb487"},
    TestItem{901369000000, "318ab148c80", "318ab148c80"},
    TestItem{1579542123461234, "1dc0a364ca7442", "1dc0a364ca7442"},
    TestItem{123345345123123321, "b158ca7b3c161a9", "b158ca7b3c161a9"},
    TestItem{9223372036854775806, "4340724c6c71dc7a6", "4340724c6c71dc7a6"},
    TestItem{9223372036854775807, "4340724c6c71dc7a7", "4340724c6c71dc7a7"},
    TestItem{-1, "-1", "8681049adb03db171"},
    TestItem{-409, "-213", "8681049adb03dad5d"},
    TestItem{-1412, "-72c", "8681049adb03da844"},
    TestItem{-8166, "-2d94", "8681049adb03d81bc"},
    TestItem{-90000001, "-bd4ab97", "8681049ada24903b9"},
    TestItem{-790301243, "-76d6292b", "8681049ad37478645"},
    TestItem{-8999099123732, "-2317b71bc364", "868102695d721cc0c"},
    TestItem{-542185444123566432, "-36b15477289a1560", "8313cd53685839a12"},
    TestItem{-9223372036854775807, "-4340724c6c71dc7a7", "4340724c6c71dc7a9"},
    TestItem{-9223372036854775807 - 1, "-4340724c6c71dc7a8", "4340724c6c71dc7a8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<15>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "5c", "5c"},
    TestItem{194, "ce", "ce"},
    TestItem{6427, "1d87", "1d87"},
    TestItem{12345312, "113cd0c", "113cd0c"},
    TestItem{33312234991, "cee7e6cb1", "cee7e6cb1"},
    TestItem{901369000000, "186a77eb46a", "186a77eb46a"},
    TestItem{1579542123461234, "c2927aeb3583e", "c2927aeb3583e"},
    TestItem{123345345123123321, "4359ea74e2b4d16", "4359ea74e2b4d16"},
    TestItem{9223372036854775806, "160e2ad3246366806", "160e2ad3246366806"},
    TestItem{9223372036854775807, "160e2ad3246366807", "160e2ad3246366807"},
    TestItem{-1, "-1", "2c1d56b648c6cd110"},
    TestItem{-409, "-1c4", "2c1d56b648c6cce3c"},
    TestItem{-1412, "-642", "2c1d56b648c6cc9be"},
    TestItem{-8166, "-2646", "2c1d56b648c6ca9ba"},
    TestItem{-90000001, "-7d7ba01", "2c1d56b6484851610"},
    TestItem{-790301243, "-495ad4e8", "2c1d56b644311eb18"},
    TestItem{-8999099123732, "-10914957a572", "2c1d55ac343152a8e"},
    TestItem{-542185444123566432, "-1388c279aea5038c", "2ad4b98d9ccb7cc74"},
    TestItem{-9223372036854775807, "-160e2ad3246366807", "160e2ad3246366809"},
    TestItem{-9223372036854775807 - 1, "-160e2ad3246366808", "160e2ad3246366808"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<16>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "57", "57"},
    TestItem{194, "c2", "c2"},
    TestItem{6427, "191b", "191b"},
    TestItem{12345312, "bc5fe0", "bc5fe0"},
    TestItem{33312234991, "7c1905def", "7c1905def"},
    TestItem{901369000000, "d1ddc77040", "d1ddc77040"},
    TestItem{1579542123461234, "59c95cf2d3a72", "59c95cf2d3a72"},
    TestItem{123345345123123321, "1b635f018c30079", "1b635f018c30079"},
    TestItem{9223372036854775806, "7ffffffffffffffe", "7ffffffffffffffe"},
    TestItem{9223372036854775807, "7fffffffffffffff", "7fffffffffffffff"},
    TestItem{-1, "-1", "ffffffffffffffff"},
    TestItem{-409, "-199", "fffffffffffffe67"},
    TestItem{-1412, "-584", "fffffffffffffa7c"},
    TestItem{-8166, "-1fe6", "ffffffffffffe01a"},
    TestItem{-90000001, "-55d4a81", "fffffffffaa2b57f"},
    TestItem{-790301243, "-2f1b0a3b", "ffffffffd0e4f5c5"},
    TestItem{-8999099123732, "-82f441b4814", "fffff7d0bbe4b7ec"},
    TestItem{-542185444123566432, "-7863ac9ef5f6160", "f879c53610a09ea0"},
    TestItem{-9223372036854775807, "-7fffffffffffffff", "8000000000000001"},
    TestItem{-9223372036854775807 - 1, "-8000000000000000", "8000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<17>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "52", "52"},
    TestItem{194, "b7", "b7"},
    TestItem{6427, "1541", "1541"},
    TestItem{12345312, "8bdd5e", "8bdd5e"},
    TestItem{33312234991, "4d31ba71d", "4d31ba71d"},
    TestItem{901369000000, "7a3age2576", "7a3age2576"},
    TestItem{1579542123461234, "2c18a11310db3", "2c18a11310db3"},
    TestItem{123345345123123321, "c7c06b4ggd0710", "c7c06b4ggd0710"},
    TestItem{9223372036854775806, "33d3d8307b214007", "33d3d8307b214007"},
    TestItem{9223372036854775807, "33d3d8307b214008", "33d3d8307b214008"},
    TestItem{-1, "-1", "67979g60f5428010"},
    TestItem{-409, "-171", "67979g60f5427fb0"},
    TestItem{-1412, "-4f1", "67979g60f5427c30"},
    TestItem{-8166, "-1b46", "67979g60f54265dc"},
    TestItem{-90000001, "-3c69cbc", "67979g60f18cf466"},
    TestItem{-790301243, "-1fca5367", "67979g60d6892dbb"},
    TestItem{-8999099123732, "-47f0f704934", "67979bf2e6e237ee"},
    TestItem{-542185444123566432, "-33ca201d9f3d309", "645bge5g1c5fbe09"},
    TestItem{-9223372036854775807, "-33d3d8307b214008", "33d3d8307b21400a"},
    TestItem{-9223372036854775807 - 1, "-33d3d8307b214009", "33d3d8307b214009"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<18>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "4f", "4f"},
    TestItem{194, "ae", "ae"},
    TestItem{6427, "11f1", "11f1"},
    TestItem{12345312, "69aeec", "69aeec"},
    TestItem{33312234991, "30779g81d", "30779g81d"},
    TestItem{901369000000, "49e55f507a", "49e55f507a"},
    TestItem{1579542123461234, "16a70baa7f0ee", "16a70baa7f0ee"},
    TestItem{123345345123123321, "5gb3hga6dbgh93", "5gb3hga6dbgh93"},
    TestItem{9223372036854775806, "16agh595df825fa6", "16agh595df825fa6"},
    TestItem{9223372036854775807, "16agh595df825fa7", "16agh595df825fa7"},
    TestItem{-1, "-1", "2d3fgb0b9cg4bd2f"},
    TestItem{-409, "-14d", "2d3fgb0b9cg4bbg3"},
    TestItem{-1412, "-468", "2d3fgb0b9cg4b8e8"},
    TestItem{-8166, "-173c", "2d3fgb0b9cg4a5h4"},
    TestItem{-90000001, "-2bb61e1", "2d3fgb0b9a4b5b6f"},
    TestItem{-790301243, "-1544736b", "2d3fgb0b87c049e5"},
    TestItem{-8999099123732, "-296b24c5588", "2d3fg894gaba67c8"},
    TestItem{-542185444123566432, "-180c4e7115796d6", "2bdf46448baf267a"},
    TestItem{-9223372036854775807, "-16agh595df825fa7", "16agh595df825fa9"},
    TestItem{-9223372036854775807 - 1, "-16agh595df825fa8", "16agh595df825fa8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<19>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "4b", "4b"},
    TestItem{194, "a4", "a4"},
    TestItem{6427, "hf5", "hf5"},
    TestItem{12345312, "4idga5", "4idga5"},
    TestItem{33312234991, "1i519eh4g", "1i519eh4g"},
    TestItem{901369000000, "2f176g3a04", "2f176g3a04"},
    TestItem{1579542123461234, "dabi3318i2ih", "dabi3318i2ih"},
    TestItem{123345345123123321, "2hdg1dbhh1aa9a", "2hdg1dbhh1aa9a"},
    TestItem{9223372036854775806, "ba643dci0ffeehg", "ba643dci0ffeehg"},
    TestItem{9223372036854775807, "ba643dci0ffeehh", "ba643dci0ffeehh"},
    TestItem{-1, "-1", "141c8786h1ccaagg"},
    TestItem{-409, "-12a", "141c8786h1cca9e7"},
    TestItem{-1412, "-3h6", "141c8786h1cca6ib"},
    TestItem{-8166, "-13bf", "141c8786h1cc9752"},
    TestItem{-90000001, "-1h6b893", "141c8786gie5i27e"},
    TestItem{-790301243, "-gf35125", "141c8786g3g959ec"},
    TestItem{-8999099123732, "-18gga8ff55h", "141c85i90a3fe5b0"},
    TestItem{-542185444123566432, "-cgi69542ei37a8", "137e90i1chgd7369"},
    TestItem{-9223372036854775807, "-ba643dci0ffeehh", "ba643dci0ffeei0"},
    TestItem{-9223372036854775807 - 1, "-ba643dci0ffeehi", "ba643dci0ffeehi"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<20>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "47", "47"},
    TestItem{194, "9e", "9e"},
    TestItem{6427, "g17", "g17"},
    TestItem{12345312, "3h335c", "3h335c"},
    TestItem{33312234991, "160a1979b", "160a1979b"},
    TestItem{901369000000, "1f43hg5000", "1f43hg5000"},
    TestItem{1579542123461234, "7e50h5dbcd1e", "7e50h5dbcd1e"},
    TestItem{123345345123123321, "1a258hahaja861", "1a258hahaja861"},
    TestItem{9223372036854775806, "5cbfjia3fh26ja6", "5cbfjia3fh26ja6"},
    TestItem{9223372036854775807, "5cbfjia3fh26ja7", "5cbfjia3fh26ja7"},
    TestItem{-1, "-1", "b53bjh07be4dj0f"},
    TestItem{-409, "-109", "b53bjh07be4di07"},
    TestItem{-1412, "-3ac", "b53bjh07be4dfa4"},
    TestItem{-8166, "-1086", "b53bjh07be4cica"},
    TestItem{-90000001, "-182a001", "b53bjh07a623j0f"},
    TestItem{-790301243, "-c6j7d23", "b53bjh06j7565id"},
    TestItem{-8999099123732, "-hbaai9a96c", "b53bij8h0ff39e4"},
    TestItem{-542185444123566432, "-6c77fii48c5g1c", "aib4c11975c82j4"},
    TestItem{-9223372036854775807, "-5cbfjia3fh26ja7", "5cbfjia3fh26ja9"},
    TestItem{-9223372036854775807 - 1, "-5cbfjia3fh26ja8", "5cbfjia3fh26ja8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<21>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "43", "43"},
    TestItem{194, "95", "95"},
    TestItem{6427, "ec1", "ec1"},
    TestItem{12345312, "30a0j0", "30a0j0"},
    TestItem{33312234991, "ia8bi2h7", "ia8bi2h7"},
    TestItem{901369000000, "12h9cje0j7", "12h9cje0j7"},
    TestItem{1579542123461234, "4aedc0jfbfj2", "4aedc0jfbfj2"},
    TestItem{123345345123123321, "gg2i04agah7g9", "gg2i04agah7g9"},
    TestItem{9223372036854775806, "2heiciiie82dh96", "2heiciiie82dh96"},
    TestItem{9223372036854775807, "2heiciiie82dh97", "2heiciiie82dh97"},
    TestItem{-1, "-1", "5e8g4ggg7g56dif"},
    TestItem{-409, "-ja", "5e8g4ggg7g56ck6"},
    TestItem{-1412, "-345", "5e8g4ggg7g56aeb"},
    TestItem{-8166, "-iai", "5e8g4ggg7g55g7j"},
    TestItem{-90000001, "-110g3d7", "5e8g4ggg6f4ba59"},
    TestItem{-790301243, "-94ada6b", "5e8g4ggfjbfe3c5"},
    TestItem{-8999099123732, "-b6ja0fkkkb", "5e8g459hifa6dj5"},
    TestItem{-542185444123566432, "-3aei796k9cejb0", "5aj1797986dcf7g"},
    TestItem{-9223372036854775807, "-2heiciiie82dh97", "2heiciiie82dh99"},
    TestItem{-9223372036854775807 - 1, "-2heiciiie82dh98", "2heiciiie82dh98"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<22>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3l", "3l"},
    TestItem{194, "8i", "8i"},
    TestItem{6427, "d63", "d63"},
    TestItem{12345312, "28f8ic", "28f8ic"},
    TestItem{33312234991, "d7hi8jhd", "d7hi8jhd"},
    TestItem{901369000000, "g97lgfidi", "g97lgfidi"},
    TestItem{1579542123461234, "2fa7jlfigahk", "2fa7jlfigahk"},
    TestItem{123345345123123321, "9d20j89c48a9b", "9d20j89c48a9b"},
    TestItem{9223372036854775806, "1adaibb21dckfa6", "1adaibb21dckfa6"},
    TestItem{9223372036854775807, "1adaibb21dckfa7", "1adaibb21dckfa7"},
    TestItem{-1, "-1", "2l4lf104353j8kf"},
    TestItem{-409, "-id", "2l4lf104353j823"},
    TestItem{-1412, "-2k4", "2l4lf104353j60c"},
    TestItem{-8166, "-gj4", "2l4lf104353ie1c"},
    TestItem{-90000001, "-ha4693", "2l4lf10429ff2bd"},
    TestItem{-790301243, "-6l7edhh", "2l4lf103i5i4h2l"},
    TestItem{-8999099123732, "-79lh493hfe", "2l4lefc480gfd52"},
    TestItem{-542185444123566432, "-1k3jef62fkgih2", "2j6hh86k0b52c3e"},
    TestItem{-9223372036854775807, "-1adaibb21dckfa7", "1adaibb21dckfa9"},
    TestItem{-9223372036854775807 - 1, "-1adaibb21dckfa8", "1adaibb21dckfa8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<23>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3i", "3i"},
    TestItem{194, "8a", "8a"},
    TestItem{6427, "c3a", "c3a"},
    TestItem{12345312, "1l2f1g", "1l2f1g"},
    TestItem{33312234991, "9i0ejkj2", "9i0ejkj2"},
    TestItem{901369000000, "bbgjf1e2h", "bbgjf1e2h"},
    TestItem{1579542123461234, "1f2m2ibih57c", "1f2m2ibih57c"},
    TestItem{123345345123123321, "5eaa7i1fi89ee", "5eaa7i1fi89ee"},
    TestItem{9223372036854775806, "i6k448cf4192c1", "i6k448cf4192c1"},
    TestItem{9223372036854775807, "i6k448cf4192c2", "i6k448cf4192c2"},
    TestItem{-1, "-1", "1ddh88h2782i515"},
    TestItem{-409, "-hi", "1ddh88h2782i46b"},
    TestItem{-1412, "-2f9", "1ddh88h2782i28k"},
    TestItem{-8166, "-fa1", "1ddh88h2782hce5"},
    TestItem{-90000001, "-dme17c", "1ddh88h26h343gh"},
    TestItem{-790301243, "-57i2b4e", "1ddh88h2207fgjf"},
    TestItem{-8999099123732, "-4ml0mdikag", "1ddh83h468bm7dd"},
    TestItem{-542185444123566432, "-11h0k8c9le9lm1", "1cc07b8ck9b8625"},
    TestItem{-9223372036854775807, "-i6k448cf4192c2", "i6k448cf4192c4"},
    TestItem{-9223372036854775807 - 1, "-i6k448cf4192c3", "i6k448cf4192c3"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<24>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3f", "3f"},
    TestItem{194, "82", "82"},
    TestItem{6427, "b3j", "b3j"},
    TestItem{12345312, "1d50k0", "1d50k0"},
    TestItem{33312234991, "767dj577", "767dj577"},
    TestItem{901369000000, "84cfnl82g", "84cfnl82g"},
    TestItem{1579542123461234, "10llfdjn1922", "10llfdjn1922"},
    TestItem{123345345123123321, "3919i2c702929", "3919i2c702929"},
    TestItem{9223372036854775806, "acd772jnc9l0l6", "acd772jnc9l0l6"},
    TestItem{9223372036854775807, "acd772jnc9l0l7", "acd772jnc9l0l7"},
    TestItem{-1, "-1", "l12ee5fn0ji1if"},
    TestItem{-409, "-h1", "l12ee5fn0ji11f"},
    TestItem{-1412, "-2ak", "l12ee5fn0jhn7k"},
    TestItem{-8166, "-e46", "l12ee5fn0jhbea"},
    TestItem{-90000001, "-b76a01", "l12ee5fmdcbfif"},
    TestItem{-790301243, "-4360iib", "l12ee5fildh705"},
    TestItem{-8999099123732, "-39i27cj2gk", "l12eajlkh6mn1k"},
    TestItem{-542185444123566432, "-ek78i3kbak440", "ka675bc2d8lleg"},
    TestItem{-9223372036854775807, "-acd772jnc9l0l7", "acd772jnc9l0l9"},
    TestItem{-9223372036854775807 - 1, "-acd772jnc9l0l8", "acd772jnc9l0l8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<25>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3c", "3c"},
    TestItem{194, "7j", "7j"},
    TestItem{6427, "a72", "a72"},
    TestItem{12345312, "16f2cc", "16f2cc"},
    TestItem{33312234991, "5bb480og", "5bb480og"},
    TestItem{901369000000, "5mh04g000", "5mh04g000"},
    TestItem{1579542123461234, "ge1h4db1co9", "ge1h4db1co9"},
    TestItem{123345345123123321, "21i9618ff4m7l", "21i9618ff4m7l"},
    TestItem{9223372036854775806, "64ie1focnn5g76", "64ie1focnn5g76"},
    TestItem{9223372036854775807, "64ie1focnn5g77", "64ie1focnn5g77"},
    TestItem{-1, "-1", "c9c336o0mlb7ef"},
    TestItem{-409, "-g9", "c9c336o0mlb6n7"},
    TestItem{-1412, "-26c", "c9c336o0mlb584"},
    TestItem{-8166, "-d1g", "c9c336o0mlajd0"},
    TestItem{-90000001, "-95a001", "c9c336o0dg17ef"},
    TestItem{-790301243, "-35n46oi", "c9c336nmgn70en"},
    TestItem{-8999099123732, "-28oa7iimo7", "c9c30moff2h9f9"},
    TestItem{-542185444123566432, "-92a5gd4368677", "c09hmfaljf3179"},
    TestItem{-9223372036854775807, "-64ie1focnn5g77", "64ie1focnn5g79"},
    TestItem{-9223372036854775807 - 1, "-64ie1focnn5g78", "64ie1focnn5g78"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<26>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "39", "39"},
    TestItem{194, "7c", "7c"},
    TestItem{6427, "9d5", "9d5"},
    TestItem{12345312, "110a7i", "110a7i"},
    TestItem{33312234991, "43lj3439", "43lj3439"},
    TestItem{901369000000, "485m0gehe", "485m0gehe"},
    TestItem{1579542123461234, "b4nmkd35ooa", "b4nmkd35ooa"},
    TestItem{123345345123123321, "17fjfm317bhp3", "17fjfm317bhp3"},
    TestItem{9223372036854775806, "3igoecjbmca686", "3igoecjbmca686"},
    TestItem{9223372036854775807, "3igoecjbmca687", "3igoecjbmca687"},
    TestItem{-1, "-1", "7b7n2pcniokcgf"},
    TestItem{-409, "-fj", "7b7n2pcniokc0n"},
    TestItem{-1412, "-228", "7b7n2pcniokae8"},
    TestItem{-8166, "-c22", "7b7n2pcniok0ee"},
    TestItem{-90000001, "-7eog2d", "7b7n2pcnb9lme3"},
    TestItem{-790301243, "-2edakhh", "7b7n2pcl4b9hop"},
    TestItem{-8999099123732, "-1h2b654e6g", "7b7n18accjfoa0"},
    TestItem{-542185444123566432, "-5hij315fjc7gk", "75g49mbi3584pm"},
    TestItem{-9223372036854775807, "-3igoecjbmca687", "3igoecjbmca689"},
    TestItem{-9223372036854775807 - 1, "-3igoecjbmca688", "3igoecjbmca688"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<27>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "36", "36"},
    TestItem{194, "75", "75"},
    TestItem{6427, "8m1", "8m1"},
    TestItem{12345312, "n65fl", "n65fl"},
    TestItem{33312234991, "34qfmng4", "34qfmng4"},
    TestItem{901369000000, "354fpld51", "354fpld51"},
    TestItem{1579542123461234, "7i3ik8b2l9n", "7i3ik8b2l9n"},
    TestItem{123345345123123321, "m524gmdifil3", "m524gmdifil3"},
    TestItem{9223372036854775806, "27c48l5b37oaoo", "27c48l5b37oaoo"},
    TestItem{9223372036854775807, "27c48l5b37oaop", "27c48l5b37oaop"},
    TestItem{-1, "-1", "4eo8hfam6fllmo"},
    TestItem{-409, "-f4", "4eo8hfam6fll7l"},
    TestItem{-1412, "-1p8", "4eo8hfam6fljoh"},
    TestItem{-8166, "-b5c", "4eo8hfam6flahd"},
    TestItem{-90000001, "-679cla", "4eo8hfam08c91f"},
    TestItem{-790301243, "-2122cdb", "4eo8hfak5dj99e"},
    TestItem{-8999099123732, "-14n86ihfbh", "4eo8gaedqo46b8"},
    TestItem{-542185444123566432, "-3ge9j96lk72qo", "4b7l7n1fbmein1"},
    TestItem{-9223372036854775807, "-27c48l5b37oaop", "27c48l5b37oap0"},
    TestItem{-9223372036854775807 - 1, "-27c48l5b37oaoq", "27c48l5b37oaoq"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<28>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "33", "33"},
    TestItem{194, "6q", "6q"},
    TestItem{6427, "85f", "85f"},
    TestItem{12345312, "k2ag0", "k2ag0"},
    TestItem{33312234991, "2d3gfbi7", "2d3gfbi7"},
    TestItem{901369000000, "2amdgph40", "2amdgph40"},
    TestItem{1579542123461234, "598oc944bg2", "598oc944bg2"},
    TestItem{123345345123123321, "eoc1cii8nec9", "eoc1cii8nec9"},
    TestItem{9223372036854775806, "1bk39f3ah3dmq6", "1bk39f3ah3dmq6"},
    TestItem{9223372036854775807, "1bk39f3ah3dmq7", "1bk39f3ah3dmq7"},
    TestItem{-1, "-1", "2nc6j26l66rhof"},
    TestItem{-409, "-eh", "2nc6j26l66rh9r"},
    TestItem{-1412, "-1mc", "2nc6j26l66rg24"},
    TestItem{-8166, "-abi", "2nc6j26l66r7cq"},
    TestItem{-90000001, "-56bnpl", "2nc6j26l10flqn"},
    TestItem{-790301243, "-1hpl98b", "2nc6j26jg968g5"},
    TestItem{-8999099123732, "-nmqg6o7o4", "2nc6i6bmi03a0c"},
    TestItem{-542185444123566432, "-29adkl3hme4o0", "2l2o59dhgcdd0g"},
    TestItem{-9223372036854775807, "-1bk39f3ah3dmq7", "1bk39f3ah3dmq9"},
    TestItem{-9223372036854775807 - 1, "-1bk39f3ah3dmq8", "1bk39f3ah3dmq8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<29>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "30", "30"},
    TestItem{194, "6k", "6k"},
    TestItem{6427, "7ii", "7ii"},
    TestItem{12345312, "hd59c", "hd59c"},
    TestItem{33312234991, "1r0308f9", "1r0308f9"},
    TestItem{901369000000, "1n7a97oq5", "1n7a97oq5"},
    TestItem{1579542123461234, "3lpf94jsra9", "3lpf94jsra9"},
    TestItem{123345345123123321, "a35b51ndo1dc", "a35b51ndo1dc"},
    TestItem{9223372036854775806, "q1se8f0m04isa", "q1se8f0m04isa"},
    TestItem{9223372036854775807, "q1se8f0m04isb", "q1se8f0m04isb"},
    TestItem{-1, "-1", "1n3rsh11f098rn"},
    TestItem{-409, "-e3", "1n3rsh11f098dl"},
    TestItem{-1412, "-1jk", "1n3rsh11f09784"},
    TestItem{-8166, "-9kh", "1n3rsh11f08s77"},
    TestItem{-90000001, "-4b75d9", "1n3rsh11ai23ef"},
    TestItem{-790301243, "-19fb030", "1n3rsh105dr8oo"},
    TestItem{-8999099123732, "-hsk0o5499", "1n3rrs1ae544if"},
    TestItem{-542185444123566432, "-1fcljlhpir0rl", "1lhf6q8ciab803"},
    TestItem{-9223372036854775807, "-q1se8f0m04isb", "q1se8f0m04isd"},
    TestItem{-9223372036854775807 - 1, "-q1se8f0m04isc", "q1se8f0m04isc"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<30>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2r", "2r"},
    TestItem{194, "6e", "6e"},
    TestItem{6427, "747", "747"},
    TestItem{12345312, "f770c", "f770c"},
    TestItem{33312234991, "1fkq6ed1", "1fkq6ed1"},
    TestItem{901369000000, "1b6db713a", "1b6db713a"},
    TestItem{1579542123461234, "2k7e4de85oe", "2k7e4de85oe"},
    TestItem{123345345123123321, "6sqhne8fl70l", "6sqhne8fl70l"},
    TestItem{9223372036854775806, "hajppbc1fc206", "hajppbc1fc206"},
    TestItem{9223372036854775807, "hajppbc1fc207", "hajppbc1fc207"},
    TestItem{-1, "-1", "14l9lkmo30o40f"},
    TestItem{-409, "-dj", "14l9lkmo30o3gr"},
    TestItem{-1412, "-1h2", "14l9lkmo30o2de"},
    TestItem{-8166, "-926", "14l9lkmo30nosa"},
    TestItem{-90000001, "-3l3a01", "14l9lkmnt9ko0f"},
    TestItem{-790301243, "-12fkcen", "14l9lkmn0f3lfn"},
    TestItem{-8999099123732, "-dled8rgb2", "14l9l719jlqhje"},
    TestItem{-542185444123566432, "-10i5q6qebgr4c", "13klfofrij76q4"},
    TestItem{-9223372036854775807, "-hajppbc1fc207", "hajppbc1fc209"},
    TestItem{-9223372036854775807 - 1, "-hajppbc1fc208", "hajppbc1fc208"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<31>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2p", "2p"},
    TestItem{194, "68", "68"},
    TestItem{6427, "6la", "6la"},
    TestItem{12345312, "dbc9r", "dbc9r"},
    TestItem{33312234991, "16ghrt9g", "16ghrt9g"},
    TestItem{901369000000, "11nj9gmdq", "11nj9gmdq"},
    TestItem{1579542123461234, "1smuhc3si6t", "1smuhc3si6t"},
    TestItem{123345345123123321, "4qf58q3slqli", "4qf58q3slqli"},
    TestItem{9223372036854775806, "bm03i95hia436", "bm03i95hia436"},
    TestItem{9223372036854775807, "bm03i95hia437", "bm03i95hia437"},
    TestItem{-1, "-1", "nd075ib45k86f"},
    TestItem{-409, "-d6", "nd075ib45k7oa"},
    TestItem{-1412, "-1eh", "nd075ib45k6mu"},
    TestItem{-8166, "-8fd", "nd075ib45jum3"},
    TestItem{-90000001, "-34e1dq", "nd075ib1166nl"},
    TestItem{-790301243, "-rin5pf", "nd075ia7hs2c1"},
    TestItem{-8999099123732, "-ah2od6ikc", "nd06q18andkh4"},
    TestItem{-542185444123566432, "-lafh1skf1sme", "mmkmjgdeliaf2"},
    TestItem{-9223372036854775807, "-bm03i95hia437", "bm03i95hia439"},
    TestItem{-9223372036854775807 - 1, "-bm03i95hia438", "bm03i95hia438"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<32>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2n", "2n"},
    TestItem{194, "62", "62"},
    TestItem{6427, "68r", "68r"},
    TestItem{12345312, "bonv0", "bonv0"},
    TestItem{33312234991, "v0p0nff", "v0p0nff"},
    TestItem{901369000000, "q7eses20", "q7eses20"},
    TestItem{1579542123461234, "1csin7iqeji", "1csin7iqeji"},
    TestItem{123345345123123321, "3dhlu0cc603p", "3dhlu0cc603p"},
    TestItem{9223372036854775806, "7vvvvvvvvvvvu", "7vvvvvvvvvvvu"},
    TestItem{9223372036854775807, "7vvvvvvvvvvvv", "7vvvvvvvvvvvv"},
    TestItem{-1, "-1", "fvvvvvvvvvvvv"},
    TestItem{-409, "-cp", "fvvvvvvvvvvj7"},
    TestItem{-1412, "-1c4", "fvvvvvvvvvujs"},
    TestItem{-8166, "-7v6", "fvvvvvvvvvo0q"},
    TestItem{-90000001, "-2lqik1", "fvvvvvvta5dbv"},
    TestItem{-790301243, "-nhm2hr", "fvvvvvv8e9te5"},
    TestItem{-8999099123732, "-85t21mi0k", "fvvvnq2tu9dvc"},
    TestItem{-542185444123566432, "-f1hqp7nluob0", "fgue56o8a17l0"},
    TestItem{-9223372036854775807, "-7vvvvvvvvvvvv", "8000000000001"},
    TestItem{-9223372036854775807 - 1, "-8000000000000", "8000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<33>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2l", "2l"},
    TestItem{194, "5t", "5t"},
    TestItem{6427, "5tp", "5tp"},
    TestItem{12345312, "adhcc", "adhcc"},
    TestItem{33312234991, "pq6p1fd", "pq6p1fd"},
    TestItem{901369000000, "l4v274k7", "l4v274k7"},
    TestItem{1579542123461234, "1113dfbj4qk", "1113dfbj4qk"},
    TestItem{123345345123123321, "2ehlblec90w0", "2ehlblec90w0"},
    TestItem{9223372036854775806, "5hg4ck9jd4u36", "5hg4ck9jd4u36"},
    TestItem{9223372036854775807, "5hg4ck9jd4u37", "5hg4ck9jd4u37"},
    TestItem{-1, "-1", "b1w8p7j5q9r6f"},
    TestItem{-409, "-cd", "b1w8p7j5q9qr3"},
    TestItem{-1412, "-19q", "b1w8p7j5q9ptn"},
    TestItem{-8166, "-7gf", "b1w8p7j5q9jn1"},
    TestItem{-90000001, "-29tckp", "b1w8p7j3gdeio"},
    TestItem{-790301243, "-k6d9qh", "b1w8p7iijthcw"},
    TestItem{-8999099123732, "-6d53roowe", "b1w8ire1vi272"},
    TestItem{-542185444123566432, "-ao04k44bw9io", "ao88kkf1eahkp"},
    TestItem{-9223372036854775807, "-5hg4ck9jd4u37", "5hg4ck9jd4u39"},
    TestItem{-9223372036854775807 - 1, "-5hg4ck9jd4u38", "5hg4ck9jd4u38"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<34>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2j", "2j"},
    TestItem{194, "5o", "5o"},
    TestItem{6427, "5j1", "5j1"},
    TestItem{12345312, "983be", "983be"},
    TestItem{33312234991, "lj61a9d", "lj61a9d"},
    TestItem{901369000000, "h5gfpqt6", "h5gfpqt6"},
    TestItem{1579542123461234, "q0h4r6ex5k", "q0h4r6ex5k"},
    TestItem{123345345123123321, "1ppgamjm260h", "1ppgamjm260h"},
    TestItem{9223372036854775806, "3tdtk1v8j6tpo", "3tdtk1v8j6tpo"},
    TestItem{9223372036854775807, "3tdtk1v8j6tpp", "3tdtk1v8j6tpp"},
    TestItem{-1, "-1", "7orp63sh4dphh"},
    TestItem{-409, "-c1", "7orp63sh4dp5h"},
    TestItem{-1412, "-17i", "7orp63sh4doa0"},
    TestItem{-8166, "-726", "7orp63sh4difc"},
    TestItem{-90000001, "-1xbsmt", "7orp63sf51usn"},
    TestItem{-790301243, "-hdddk7", "7orp63rxp0bvb"},
    TestItem{-8999099123732, "-51bdh0ja4", "7orp12h3ld67e"},
    TestItem{-542185444123566432, "-7oloakblghpq", "7h33fr85gv7pq"},
    TestItem{-9223372036854775807, "-3tdtk1v8j6tpp", "3tdtk1v8j6tpr"},
    TestItem{-9223372036854775807 - 1, "-3tdtk1v8j6tpq", "3tdtk1v8j6tpq"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<35>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2h", "2h"},
    TestItem{194, "5j", "5j"},
    TestItem{6427, "58m", "58m"},
    TestItem{12345312, "87ws7", "87ws7"},
    TestItem{33312234991, "i48vq7l", "i48vq7l"},
    TestItem{901369000000, "e0brdnk0", "e0brdnk0"},
    TestItem{1579542123461234, "k1f6psc609", "k1f6psc609"},
    TestItem{123345345123123321, "19oyhfllc98g", "19oyhfllc98g"},
    TestItem{9223372036854775806, "2pijmikexrxp6", "2pijmikexrxp6"},
    TestItem{9223372036854775807, "2pijmikexrxp7", "2pijmikexrxp7"},
    TestItem{-1, "-1", "5g24a25twkwff"},
    TestItem{-409, "-bo", "5g24a25twkw3r"},
    TestItem{-1412, "-15c", "5g24a25twkva4"},
    TestItem{-8166, "-6nb", "5g24a25twkpr5"},
    TestItem{-90000001, "-1oy4dl", "5g24a25s7ls1u"},
    TestItem{-790301243, "-f1mnui", "5g24a25eux8jx"},
    TestItem{-8999099123732, "-3yuf0oqaw", "5g2462aevv64j"},
    TestItem{-542185444123566432, "-5lj5lu0x1f57", "5afk4fasyjha9"},
    TestItem{-9223372036854775807, "-2pijmikexrxp7", "2pijmikexrxp9"},
    TestItem{-9223372036854775807 - 1, "-2pijmikexrxp8", "2pijmikexrxp8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<36>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2f", "2f"},
    TestItem{194, "5e", "5e"},
    TestItem{6427, "4yj", "4yj"},
    TestItem{12345312, "7clpc", "7clpc"},
    TestItem{33312234991, "fax8t0v", "fax8t0v"},
    TestItem{901369000000, "bi2ztvls", "bi2ztvls"},
    TestItem{1579542123461234, "fjwfmtqmpe", "fjwfmtqmpe"},
    TestItem{123345345123123321, "xqi9xajvddl", "xqi9xajvddl"},
    TestItem{9223372036854775806, "1y2p0ij32e8e6", "1y2p0ij32e8e6"},
    TestItem{9223372036854775807, "1y2p0ij32e8e7", "1y2p0ij32e8e7"},
    TestItem{-1, "-1", "3w5e11264sgsf"},
    TestItem{-409, "-bd", "3w5e11264sgh3"},
    TestItem{-1412, "-138", "3w5e11264sfp8"},
    TestItem{-8166, "-6au", "3w5e11264sahm"},
    TestItem{-90000001, "-1hl0g1", "3w5e1124n7gcf"},
    TestItem{-790301243, "-d2iwcb", "3w5e111t29kg5"},
    TestItem{-8999099123732, "-36u4n9nv8", "3w5dxu81hisx8"},
    TestItem{-542185444123566432, "-44akp6hluf6o", "3s13gbvoiy1ls"},
    TestItem{-9223372036854775807, "-1y2p0ij32e8e7", "1y2p0ij32e8e9"},
    TestItem{-9223372036854775807 - 1, "-1y2p0ij32e8e8", "1y2p0ij32e8e8"},
  }};
}

template <class T>
__host__ __device__ constexpr void test_from_chars(
  const char* data,
  cuda::std::ptrdiff_t size,
  int base,
  cuda::std::ptrdiff_t exp_ptr_offset,
  cuda::std::errc exp_errc,
  T exp_val = {})
{
  constexpr cuda::std::size_t buff_size = 150;
  constexpr auto init_val               = static_cast<T>(-23);

  char buff[buff_size]{};
  cuda::std::strncpy(buff, data, size);

  {
    T value(init_val);
    const auto result = cuda::std::from_chars(buff, buff + size, value, base);

    // Check if the result matches the expected result
    assert(result.ptr == buff + exp_ptr_offset);
    assert(result.ec == exp_errc);

    // Check if the value matches the expected value
    assert((exp_errc == cuda::std::errc{}) ? (value == exp_val) : (value == init_val));
  }

  // If the base is greater than 10, we need to handle case sensitivity for letters
  if (base > 10)
  {
    for (cuda::std::ptrdiff_t i = 0; i < size; ++i)
    {
      if (buff[i] >= 'a' && buff[i] <= 'z')
      {
        buff[i] -= 'a' - 'A'; // Convert lowercase to uppercase
      }
    }

    T value(init_val);
    const auto result = cuda::std::from_chars(buff, buff + size, value, base);

    // Check if the result matches the expected result
    assert(result.ptr == buff + exp_ptr_offset);
    assert(result.ec == exp_errc);

    // Check if the value matches the expected value
    assert((exp_errc == cuda::std::errc{}) ? (value == exp_val) : (value == init_val));
  }
}

template <class T>
__host__ __device__ constexpr void test_from_chars(const TestItem& item, int base, bool overflow = false)
{
  static_assert(
    cuda::std::is_same_v<
      cuda::std::from_chars_result,
      decltype(cuda::std::from_chars(
        cuda::std::declval<const char*>(), cuda::std::declval<const char*>(), cuda::std::declval<T&>(), int{}))>);
  static_assert(noexcept(cuda::std::from_chars(
    cuda::std::declval<const char*>(), cuda::std::declval<const char*>(), cuda::std::declval<T&>(), int{})));

  const auto ref_val = static_cast<T>(item.val);
  const char* str    = (cuda::std::is_signed_v<T>) ? item.str_signed : item.str_unsigned;
  const auto len     = cuda::std::strlen(str);

  constexpr cuda::std::size_t buff_size = 150;
  char buff[buff_size]{};

  const auto success_errc = (overflow) ? cuda::std::errc::result_out_of_range : cuda::std::errc{};

  // 1. Test the original string
  cuda::std::strncpy(buff, str, len);
  test_from_chars<T>(buff, len, base, len, success_errc, ref_val);

  // 2. Test a string that has invalid characters at the end
  cuda::std::strncpy(buff, str, len);
  buff[len] = '9' + 1;
  test_from_chars<T>(buff, len + 1, base, len, success_errc, ref_val);

  // 3. Test a string that has invalid characters at the beginning
  buff[0] = '9' + 1;
  cuda::std::strncpy(buff + 1, str, len);
  test_from_chars<T>(buff, len + 1, base, 0, cuda::std::errc::invalid_argument);

  // 3. Test a string that has white characters at the beginning
  buff[0] = ' ';
  cuda::std::strncpy(buff + 1, str, len);
  test_from_chars<T>(buff, len + 1, base, 0, cuda::std::errc::invalid_argument);

  // 5. Test a string that has '+' at the beginning
  buff[0] = '+';
  cuda::std::strncpy(buff + 1, str, len);
  test_from_chars<T>(buff, len + 1, base, 0, cuda::std::errc::invalid_argument);

  // 6. Test a string that has '-' at the beginning (unsigned only)
  if constexpr (cuda::std::is_unsigned_v<T>)
  {
    buff[0] = '-';
    cuda::std::strncpy(buff + 1, str, len);
    test_from_chars<T>(buff, len + 1, base, 0, cuda::std::errc::invalid_argument);
  }
}

template <class T>
__host__ __device__ constexpr void test_overflow()
{
  constexpr int base = 10;

  TestItem item{};

  // 1. Test `max + 1` overflow
  {
    item.str_signed   = "9223372036854775808";
    item.str_unsigned = "18446744073709551616";
    test_from_chars<T>(item, base, true);
  }

  // 2. Test positive huge value overflow
  {
    item.str_signed   = "12390981233333333333333333333";
    item.str_unsigned = "12390981233333333333333333333";
    test_from_chars<T>(item, base, true);
  }

  // 3. Test `min - 1` overflow
  if constexpr (cuda::std::is_signed_v<T>)
  {
    item.str_signed   = "-9223372036854775809";
    item.str_unsigned = "";
    test_from_chars<T>(item, base, true);
  }

  // 4. Test negative huge value overflow
  if constexpr (cuda::std::is_signed_v<T>)
  {
    item.str_signed   = "-12390981233333333333333333333";
    item.str_unsigned = "";
    test_from_chars<T>(item, base, true);
  }
}

template <int Base>
__host__ __device__ constexpr bool test_base()
{
  constexpr auto items = get_test_items<Base>();

  for (const auto& item : items)
  {
    test_from_chars<cuda::std::int64_t>(item, Base);
    test_from_chars<cuda::std::uint64_t>(item, Base);
  }

  // Test overflow cases (base 10 only)
  if constexpr (Base == 10)
  {
    test_overflow<cuda::std::int64_t>();
    test_overflow<cuda::std::uint64_t>();
  }

  return true;
}

struct TestBaseInvoker
{
  template <int Base>
  __host__ __device__ constexpr void operator()(cuda::std::integral_constant<int, Base>) const
  {
    test_base<Base>();
    static_assert(test_base<Base>());
  }
};

__host__ __device__ constexpr void test()
{
  cuda::static_for<int, first_base, last_base + 1>(TestBaseInvoker{});
}

int main(int, char**)
{
  test();
  return 0;
}
