//===----------------------------------------------------------------------===//
//
// Part of libcu++, the C++ Standard Library for your entire system,
// under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES.
//
//===----------------------------------------------------------------------===//

#include <cuda/std/__charconv_>
#include <cuda/std/array>
#include <cuda/std/cstddef>
#include <cuda/std/cstdint>
#include <cuda/std/cstring>
#include <cuda/std/type_traits>
#include <cuda/std/utility>

#include "test_macros.h"

#if _CCCL_HAS_INT128()

constexpr int first_base = 2;
constexpr int last_base  = 36;

struct TestItem
{
  __int128_t val;
  const char* str_signed;
  const char* str_unsigned;
};

template <int Base>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items();

// Source code for the generation of the test items
// #include <iostream>
// #include <charconv>
// #include <cstdint>
// #include <cstddef>
// #include <type_traits>
// #include <limits>

// constexpr std::int64_t list[] = {
//   0,
//   1,
//   87,
//   194,
//   6'427,
//   12'345'312,
//   33'312'234'991,
//   901'369'000'000,
//   1'579'542'123'461'234,
//   123'345'345'123'123'321,
//   9'223'372'036'854'775'806,
//   9'223'372'036'854'775'807,
//   -1,
//   -409,
//   -1'412,
//   -8'166,
//   -90'000'001,
//   -790'301'243,
//   -8'999'099'123'732,
//   -542'185'444'123'566'432,
//   -9'223'372'036'854'775'807,
//   -9'223'372'036'854'775'807 - 1,
// };

// int main()
// {
//   for (int base = 2; base <= 36; ++base)
//   {
//     std::cout <<
// "\n"
// "template <>\n"
// "__host__ __device__ constexpr cuda::std::array<TestItem, " << std::size(list) << "> get_test_items<" << base <<
// ">()\n" <<
// "{\n"
// "  return {{\n";
//     for (auto v : list)
//     {
//       constexpr std::size_t buff_size = 150;
//       char signed_buff[buff_size]{};
//       char unsigned_buff[buff_size]{};
//       std::to_chars(signed_buff, signed_buff + buff_size, __int128_t(v), base);
//       std::to_chars(unsigned_buff, unsigned_buff + buff_size, __uint128_t(v), base);

//       std::cout <<
// "    TestItem{";
//       if (v == std::numeric_limits<decltype(v)>::min())
//       {
//         std::cout << (-std::numeric_limits<decltype(v)>::max()) << " - 1";
//       }
//       else
//       {
//         std::cout << v;
//       }
//       std::cout << ", \"" << signed_buff << "\", \"" << unsigned_buff << "\"},\n";
//     }
//     std::cout <<
// "  }};\n" <<
// "}\n";
//   }
// }

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<2>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "1010111", "1010111"},
    TestItem{194, "11000010", "11000010"},
    TestItem{6427, "1100100011011", "1100100011011"},
    TestItem{12345312, "101111000101111111100000", "101111000101111111100000"},
    TestItem{33312234991, "11111000001100100000101110111101111", "11111000001100100000101110111101111"},
    TestItem{901369000000, "1101000111011101110001110111000001000000", "1101000111011101110001110111000001000000"},
    TestItem{1579542123461234,
             "101100111001001010111001111001011010011101001110010",
             "101100111001001010111001111001011010011101001110010"},
    TestItem{123345345123123321,
             "110110110001101011111000000011000110000110000000001111001",
             "110110110001101011111000000011000110000110000000001111001"},
    TestItem{9223372036854775806,
             "111111111111111111111111111111111111111111111111111111111111110",
             "111111111111111111111111111111111111111111111111111111111111110"},
    TestItem{9223372036854775807,
             "111111111111111111111111111111111111111111111111111111111111111",
             "111111111111111111111111111111111111111111111111111111111111111"},
    TestItem{-1,
             "-1",
             "111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
             "11111111111111111111111"},
    TestItem{-409,
             "-110011001",
             "111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
             "11111111111111001100111"},
    TestItem{-1412,
             "-10110000100",
             "111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
             "11111111111101001111100"},
    TestItem{-8166,
             "-1111111100110",
             "111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
             "11111111110000000011010"},
    TestItem{-90000001,
             "-101010111010100101010000001",
             "111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111110101"
             "01000101011010101111111"},
    TestItem{-790301243,
             "-101111000110110000101000111011",
             "111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111110100001"
             "11001001111010111000101"},
    TestItem{-8999099123732,
             "-10000010111101000100000110110100100000010100",
             "111111111111111111111111111111111111111111111111111111111111111111111111111111111111011111010000101110111"
             "11001001011011111101100"},
    TestItem{-542185444123566432,
             "-11110000110001110101100100111101111010111110110000101100000",
             "111111111111111111111111111111111111111111111111111111111111111111111000011110011100010100110110000100001"
             "01000001001111010100000"},
    TestItem{-9223372036854775807,
             "-111111111111111111111111111111111111111111111111111111111111111",
             "111111111111111111111111111111111111111111111111111111111111111110000000000000000000000000000000000000000"
             "00000000000000000000001"},
    TestItem{-9223372036854775807 - 1,
             "-1000000000000000000000000000000000000000000000000000000000000000",
             "111111111111111111111111111111111111111111111111111111111111111110000000000000000000000000000000000000000"
             "00000000000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<3>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "10020", "10020"},
    TestItem{194, "21012", "21012"},
    TestItem{6427, "22211001", "22211001"},
    TestItem{12345312, "212020012120210", "212020012120210"},
    TestItem{33312234991, "10011222120211212121011", "10011222120211212121011"},
    TestItem{901369000000, "10012011120221210111012001", "10012011120221210111012001"},
    TestItem{1579542123461234, "21200010200202022102002210100212", "21200010200202022102002210100212"},
    TestItem{123345345123123321, "211012002011121211111200120200210010", "211012002011121211111200120200210010"},
    TestItem{
      9223372036854775806, "2021110011022210012102010021220101220220", "2021110011022210012102010021220101220220"},
    TestItem{
      9223372036854775807, "2021110011022210012102010021220101220221", "2021110011022210012102010021220101220221"},
    TestItem{-1, "-1", "202201102121002021012000211012011021221022212021111001022110211020010021100121010"},
    TestItem{-409, "-120011", "202201102121002021012000211012011021221022212021111001022110211020010021100001000"},
    TestItem{-1412, "-1221022", "202201102121002021012000211012011021221022212021111001022110211020010021021122212"},
    TestItem{-8166, "-102012110", "202201102121002021012000211012011021221022212021111001022110211020010020221101201"},
    TestItem{-90000001,
             "-20021100110210101",
             "202201102121002021012000211012011021221022212021111001022110210222211220212210210"},
    TestItem{-790301243,
             "-2001002002110111102",
             "202201102121002021012000211012011021221022212021111001022110202012001011220002202"},
    TestItem{-8999099123732,
             "-1011212022020200122120102122",
             "202201102121002021012000211012011021221022212021111000010121111222102121210011112"},
    TestItem{-542185444123566432,
             "-10121112100201100020210202021002222220",
             "202201102121002021012000211012011021221022201122221200121010120102101000020121021"},
    TestItem{-9223372036854775807,
             "-2021110011022210012102010021220101220221",
             "202201102121002021012000211012011021221020120211022201112021102002211100221200020"},
    TestItem{-9223372036854775807 - 1,
             "-2021110011022210012102010021220101220222",
             "202201102121002021012000211012011021221020120211022201112021102002211100221200012"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<4>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "1113", "1113"},
    TestItem{194, "3002", "3002"},
    TestItem{6427, "1210123", "1210123"},
    TestItem{12345312, "233011333200", "233011333200"},
    TestItem{33312234991, "133001210011313233", "133001210011313233"},
    TestItem{901369000000, "31013131301313001000", "31013131301313001000"},
    TestItem{1579542123461234, "11213021113033023103221302", "11213021113033023103221302"},
    TestItem{123345345123123321, "12312031133000120300300001321", "12312031133000120300300001321"},
    TestItem{9223372036854775806, "13333333333333333333333333333332", "13333333333333333333333333333332"},
    TestItem{9223372036854775807, "13333333333333333333333333333333", "13333333333333333333333333333333"},
    TestItem{-1, "-1", "3333333333333333333333333333333333333333333333333333333333333333"},
    TestItem{-409, "-12121", "3333333333333333333333333333333333333333333333333333333333321213"},
    TestItem{-1412, "-112010", "3333333333333333333333333333333333333333333333333333333333221330"},
    TestItem{-8166, "-1333212", "3333333333333333333333333333333333333333333333333333333332000122"},
    TestItem{-90000001, "-11113110222001", "3333333333333333333333333333333333333333333333333322220223111333"},
    TestItem{-790301243, "-233012300220323", "3333333333333333333333333333333333333333333333333100321033113011"},
    TestItem{
      -8999099123732, "-2002331010012310200110", "3333333333333333333333333333333333333333331331002323321023133230"},
    TestItem{-542185444123566432,
             "-132012032230213233113312011200",
             "3333333333333333333333333333333333201321301103120100220021322200"},
    TestItem{-9223372036854775807,
             "-13333333333333333333333333333333",
             "3333333333333333333333333333333320000000000000000000000000000001"},
    TestItem{-9223372036854775807 - 1,
             "-20000000000000000000000000000000",
             "3333333333333333333333333333333320000000000000000000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<5>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "322", "322"},
    TestItem{194, "1234", "1234"},
    TestItem{6427, "201202", "201202"},
    TestItem{12345312, "11130022222", "11130022222"},
    TestItem{33312234991, "1021210413004431", "1021210413004431"},
    TestItem{901369000000, "104232000431000000", "104232000431000000"},
    TestItem{1579542123461234, "3124013204232101224414", "3124013204232101224414"},
    TestItem{123345345123123321, "2013314110113303004421241", "2013314110113303004421241"},
    TestItem{9223372036854775806, "1104332401304422434310311211", "1104332401304422434310311211"},
    TestItem{9223372036854775807, "1104332401304422434310311212", "1104332401304422434310311212"},
    TestItem{-1, "-1", "11031110441201303134210404233413032443021130230130231310"},
    TestItem{-409, "-3114", "11031110441201303134210404233413032443021130230130223142"},
    TestItem{-1412, "-21122", "11031110441201303134210404233413032443021130230130210134"},
    TestItem{-8166, "-230131", "11031110441201303134210404233413032443021130230130001130"},
    TestItem{-90000001, "-141020000001", "11031110441201303134210404233413032443021130034110231310"},
    TestItem{-790301243, "-3104304114433", "11031110441201303134210404233413032443021122120321111323"},
    TestItem{-8999099123732, "-2134420123333424412", "11031110441201303134210404233413032440331210101241301344"},
    TestItem{
      -542185444123566432, "-14022010312304031113111212", "11031110441201303134210404233344010432203321144012120044"},
    TestItem{-9223372036854775807,
             "-1104332401304422434310311212",
             "11031110441201303134210404232303200041211202240314420044"},
    TestItem{-9223372036854775807 - 1,
             "-1104332401304422434310311213",
             "11031110441201303134210404232303200041211202240314420043"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<6>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "223", "223"},
    TestItem{194, "522", "522"},
    TestItem{6427, "45431", "45431"},
    TestItem{12345312, "1120334120", "1120334120"},
    TestItem{33312234991, "23145312450051", "23145312450051"},
    TestItem{901369000000, "1530025545513344", "1530025545513344"},
    TestItem{1579542123461234, "23315223344542344122", "23315223344542344122"},
    TestItem{123345345123123321, "5342301353143151212133", "5342301353143151212133"},
    TestItem{9223372036854775806, "1540241003031030222122210", "1540241003031030222122210"},
    TestItem{9223372036854775807, "1540241003031030222122211", "1540241003031030222122211"},
    TestItem{-1, "-1", "23053353530155550541354043543542243325553444410303"},
    TestItem{-409, "-1521", "23053353530155550541354043543542243325553444404343"},
    TestItem{-1412, "-10312", "23053353530155550541354043543542243325553444355552"},
    TestItem{-8166, "-101450", "23053353530155550541354043543542243325553444304414"},
    TestItem{-90000001, "-12533002401", "23053353530155550541354043543542243325540511403503"},
    TestItem{-790301243, "-210230522015", "23053353530155550541354043543542243325343213444245"},
    TestItem{-8999099123732, "-31050043513355112", "23053353530155550541354043543542212235505531011152"},
    TestItem{-542185444123566432, "-40414324110253350231040", "23053353530155550541354043503123515215300054135224"},
    TestItem{-9223372036854775807, "-1540241003031030222122211", "23053353530155550541354042003301240254523222244053"},
    TestItem{
      -9223372036854775807 - 1, "-1540241003031030222122212", "23053353530155550541354042003301240254523222244052"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<7>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "153", "153"},
    TestItem{194, "365", "365"},
    TestItem{6427, "24511", "24511"},
    TestItem{12345312, "206635110", "206635110"},
    TestItem{33312234991, "2256336216430", "2256336216430"},
    TestItem{901369000000, "122056523041120", "122056523041120"},
    TestItem{1579542123461234, "653464103206202312", "653464103206202312"},
    TestItem{123345345123123321, "135513531051312220002", "135513531051312220002"},
    TestItem{9223372036854775806, "22341010611245052052266", "22341010611245052052266"},
    TestItem{9223372036854775807, "22341010611245052052300", "22341010611245052052300"},
    TestItem{-1, "-1", "3115512162124626343001006330151620356026315303"},
    TestItem{-409, "-1123", "3115512162124626343001006330151620356026314151"},
    TestItem{-1412, "-4055", "3115512162124626343001006330151620356026311216"},
    TestItem{-8166, "-32544", "3115512162124626343001006330151620356026252430"},
    TestItem{-90000001, "-2141662450", "3115512162124626343001006330151620353554322524"},
    TestItem{-790301243, "-25404310154", "3115512162124626343001006330151620330322005120"},
    TestItem{-8999099123732, "-1616110031306654", "3115512162124626343001006330150001245665005320"},
    TestItem{-542185444123566432, "-653644634443563304050", "3115512162124626343001005343203652612133011224"},
    TestItem{-9223372036854775807, "-22341010611245052052300", "3115512162124626343000653656141006110644233004"},
    TestItem{-9223372036854775807 - 1, "-22341010611245052052301", "3115512162124626343000653656141006110644233003"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<8>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "127", "127"},
    TestItem{194, "302", "302"},
    TestItem{6427, "14433", "14433"},
    TestItem{12345312, "57057740", "57057740"},
    TestItem{33312234991, "370144056757", "370144056757"},
    TestItem{901369000000, "15073561670100", "15073561670100"},
    TestItem{1579542123461234, "54711271713235162", "54711271713235162"},
    TestItem{123345345123123321, "6661537003060600171", "6661537003060600171"},
    TestItem{9223372036854775806, "777777777777777777776", "777777777777777777776"},
    TestItem{9223372036854775807, "777777777777777777777", "777777777777777777777"},
    TestItem{-1, "-1", "3777777777777777777777777777777777777777777"},
    TestItem{-409, "-631", "3777777777777777777777777777777777777777147"},
    TestItem{-1412, "-2604", "3777777777777777777777777777777777777775174"},
    TestItem{-8166, "-17746", "3777777777777777777777777777777777777760032"},
    TestItem{-90000001, "-527245201", "3777777777777777777777777777777777250532577"},
    TestItem{-790301243, "-5706605073", "3777777777777777777777777777777772071172705"},
    TestItem{-8999099123732, "-202750406644024", "3777777777777777777777777777575027371133754"},
    TestItem{-542185444123566432, "-36061654475727660540", "3777777777777777777777741716123302050117240"},
    TestItem{-9223372036854775807, "-777777777777777777777", "3777777777777777777777000000000000000000001"},
    TestItem{-9223372036854775807 - 1, "-1000000000000000000000", "3777777777777777777777000000000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<9>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "106", "106"},
    TestItem{194, "235", "235"},
    TestItem{6427, "8731", "8731"},
    TestItem{12345312, "25205523", "25205523"},
    TestItem{33312234991, "104876755534", "104876755534"},
    TestItem{901369000000, "3164527714161", "3164527714161"},
    TestItem{1579542123461234, "7603622272083325", "7603622272083325"},
    TestItem{123345345123123321, "735064554450520703", "735064554450520703"},
    TestItem{9223372036854775806, "67404283172107811826", "67404283172107811826"},
    TestItem{9223372036854775807, "67404283172107811827", "67404283172107811827"},
    TestItem{-1, "-1", "22642532235024164257285244038424203240533"},
    TestItem{-409, "-504", "22642532235024164257285244038424203240030"},
    TestItem{-1412, "-1838", "22642532235024164257285244038424203237585"},
    TestItem{-8166, "-12173", "22642532235024164257285244038424203227351"},
    TestItem{-90000001, "-207313711", "22642532235024164257285244038423884825723"},
    TestItem{-790301243, "-2032073442", "22642532235024164257285244038422161156082"},
    TestItem{-8999099123732, "-34768220576378", "22642532235024164257285244003544872553145"},
    TestItem{-542185444123566432, "-3545321306722232886", "22642532235024164257281587617116371006537"},
    TestItem{-9223372036854775807, "-67404283172107811827", "22642532235024164257216738645242084327606"},
    TestItem{-9223372036854775807 - 1, "-67404283172107811828", "22642532235024164257216738645242084327605"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<10>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "87", "87"},
    TestItem{194, "194", "194"},
    TestItem{6427, "6427", "6427"},
    TestItem{12345312, "12345312", "12345312"},
    TestItem{33312234991, "33312234991", "33312234991"},
    TestItem{901369000000, "901369000000", "901369000000"},
    TestItem{1579542123461234, "1579542123461234", "1579542123461234"},
    TestItem{123345345123123321, "123345345123123321", "123345345123123321"},
    TestItem{9223372036854775806, "9223372036854775806", "9223372036854775806"},
    TestItem{9223372036854775807, "9223372036854775807", "9223372036854775807"},
    TestItem{-1, "-1", "340282366920938463463374607431768211455"},
    TestItem{-409, "-409", "340282366920938463463374607431768211047"},
    TestItem{-1412, "-1412", "340282366920938463463374607431768210044"},
    TestItem{-8166, "-8166", "340282366920938463463374607431768203290"},
    TestItem{-90000001, "-90000001", "340282366920938463463374607431678211455"},
    TestItem{-790301243, "-790301243", "340282366920938463463374607430977910213"},
    TestItem{-8999099123732, "-8999099123732", "340282366920938463463374598432669087724"},
    TestItem{-542185444123566432, "-542185444123566432", "340282366920938463462832421987644645024"},
    TestItem{-9223372036854775807, "-9223372036854775807", "340282366920938463454151235394913435649"},
    TestItem{-9223372036854775807 - 1, "-9223372036854775808", "340282366920938463454151235394913435648"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<11>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "7a", "7a"},
    TestItem{194, "167", "167"},
    TestItem{6427, "4913", "4913"},
    TestItem{12345312, "6a72241", "6a72241"},
    TestItem{33312234991, "13144985222", "13144985222"},
    TestItem{901369000000, "3182a5335857", "3182a5335857"},
    TestItem{1579542123461234, "41832206412aa29", "41832206412aa29"},
    TestItem{123345345123123321, "275896a0116541880", "275896a0116541880"},
    TestItem{9223372036854775806, "1728002635214590696", "1728002635214590696"},
    TestItem{9223372036854775807, "1728002635214590697", "1728002635214590697"},
    TestItem{-1, "-1", "1000a504186892265432152aa27a7929366352"},
    TestItem{-409, "-342", "1000a504186892265432152aa27a7929366011"},
    TestItem{-1412, "-1074", "1000a504186892265432152aa27a792936528a"},
    TestItem{-8166, "-6154", "1000a504186892265432152aa27a79293601aa"},
    TestItem{-90000001, "-46891373", "1000a504186892265432152aa27a7892584a90"},
    TestItem{-790301243, "-376117026", "1000a504186892265432152aa27a756324a328"},
    TestItem{-8999099123732, "-295a550938493", "1000a504186892265432152a9a948388528970"},
    TestItem{-542185444123566432, "-108881a9a629422912", "1000a5041868922654320452208082aaa43541"},
    TestItem{-9223372036854775807, "-1728002635214590697", "1000a504186892265430525aa0172714885767"},
    TestItem{-9223372036854775807 - 1, "-1728002635214590698", "1000a504186892265430525aa0172714885766"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<12>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "73", "73"},
    TestItem{194, "142", "142"},
    TestItem{6427, "3877", "3877"},
    TestItem{12345312, "4174340", "4174340"},
    TestItem{33312234991, "6558249927", "6558249927"},
    TestItem{901369000000, "1268367a2854", "1268367a2854"},
    TestItem{1579542123461234, "1291a5a1928b042", "1291a5a1928b042"},
    TestItem{123345345123123321, "800a002296817049", "800a002296817049"},
    TestItem{9223372036854775806, "41a792678515120366", "41a792678515120366"},
    TestItem{9223372036854775807, "41a792678515120367", "41a792678515120367"},
    TestItem{-1, "-1", "5916b64b41143526a777873841863a6a6993"},
    TestItem{-409, "-2a1", "5916b64b41143526a777873841863a6a66b3"},
    TestItem{-1412, "-998", "5916b64b41143526a777873841863a6a5bb8"},
    TestItem{-8166, "-4886", "5916b64b41143526a777873841863a6a210a"},
    TestItem{-90000001, "-26183401", "5916b64b41143526a7778738418614523593"},
    TestItem{-790301243, "-1a080630b", "5916b64b41143526a7778738418459aa0685"},
    TestItem{-8999099123732, "-1014104108a98", "5916b64b41143526a7778737404536599ab8"},
    TestItem{-542185444123566432, "-2b235857512655480", "5916b64b41143526a7749502781128051514"},
    TestItem{-9223372036854775807, "-41a792678515120367", "5916b64b41143526a7359b65960125586629"},
    TestItem{-9223372036854775807 - 1, "-41a792678515120368", "5916b64b41143526a7359b65960125586628"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<13>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "69", "69"},
    TestItem{194, "11c", "11c"},
    TestItem{6427, "2c05", "2c05"},
    TestItem{12345312, "2733225", "2733225"},
    TestItem{33312234991, "31ab66c369", "31ab66c369"},
    TestItem{901369000000, "66cca2a2691", "66cca2a2691"},
    TestItem{1579542123461234, "52a4924c3c889a", "52a4924c3c889a"},
    TestItem{123345345123123321, "25432c41a02226b3", "25432c41a02226b3"},
    TestItem{9223372036854775806, "10b269549075433c36", "10b269549075433c36"},
    TestItem{9223372036854775807, "10b269549075433c37", "10b269549075433c37"},
    TestItem{-1, "-1", "47168c9c477c94ba75a2bc735955c7aa138"},
    TestItem{-409, "-256", "47168c9c477c94ba75a2bc735955c7a9bb3"},
    TestItem{-1412, "-848", "47168c9c477c94ba75a2bc735955c7a95c1"},
    TestItem{-8166, "-3942", "47168c9c477c94ba75a2bc735955c7a64c7"},
    TestItem{-90000001, "-15851c50", "47168c9c477c94ba75a2bc7359546c581b9"},
    TestItem{-790301243, "-c7968494", "47168c9c477c94ba75a2bc7359464b41975"},
    TestItem{-8999099123732, "-5037c434a503", "47168c9c477c94ba75a2bc6b55a6845c936"},
    TestItem{-542185444123566432, "-a79189bcc79c7477", "47168c9c477c94ba759543578a564ab2992"},
    TestItem{-9223372036854775807, "-10b269549075433c37", "47168c9c477c94ba749495ab104b7376202"},
    TestItem{-9223372036854775807 - 1, "-10b269549075433c38", "47168c9c477c94ba749495ab104b7376201"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<14>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "63", "63"},
    TestItem{194, "dc", "dc"},
    TestItem{6427, "24b1", "24b1"},
    TestItem{12345312, "18d5040", "18d5040"},
    TestItem{33312234991, "18802cb487", "18802cb487"},
    TestItem{901369000000, "318ab148c80", "318ab148c80"},
    TestItem{1579542123461234, "1dc0a364ca7442", "1dc0a364ca7442"},
    TestItem{123345345123123321, "b158ca7b3c161a9", "b158ca7b3c161a9"},
    TestItem{9223372036854775806, "4340724c6c71dc7a6", "4340724c6c71dc7a6"},
    TestItem{9223372036854775807, "4340724c6c71dc7a7", "4340724c6c71dc7a7"},
    TestItem{-1, "-1", "51a45cb9506962a31983dac25409715d03"},
    TestItem{-409, "-213", "51a45cb9506962a31983dac25409715ad1"},
    TestItem{-1412, "-72c", "51a45cb9506962a31983dac254097155b6"},
    TestItem{-8166, "-2d94", "51a45cb9506962a31983dac25409712d50"},
    TestItem{-90000001, "-bd4ab97", "51a45cb9506962a31983dac253db7a914b"},
    TestItem{-790301243, "-76d6292b", "51a45cb9506962a31983dac253727933b7"},
    TestItem{-8999099123732, "-2317b71bc364", "51a45cb9506962a31983da9d3a32537980"},
    TestItem{-542185444123566432, "-36b15477289a1560", "51a45cb9506962a3194b296bbac0b54784"},
    TestItem{-9223372036854775807, "-4340724c6c71dc7a7", "51a45cb9506962a3154dd39b6b2251753b"},
    TestItem{-9223372036854775807 - 1, "-4340724c6c71dc7a8", "51a45cb9506962a3154dd39b6b2251753a"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<15>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "5c", "5c"},
    TestItem{194, "ce", "ce"},
    TestItem{6427, "1d87", "1d87"},
    TestItem{12345312, "113cd0c", "113cd0c"},
    TestItem{33312234991, "cee7e6cb1", "cee7e6cb1"},
    TestItem{901369000000, "186a77eb46a", "186a77eb46a"},
    TestItem{1579542123461234, "c2927aeb3583e", "c2927aeb3583e"},
    TestItem{123345345123123321, "4359ea74e2b4d16", "4359ea74e2b4d16"},
    TestItem{9223372036854775806, "160e2ad3246366806", "160e2ad3246366806"},
    TestItem{9223372036854775807, "160e2ad3246366807", "160e2ad3246366807"},
    TestItem{-1, "-1", "7d491176809c28848a561186d4857d320"},
    TestItem{-409, "-1c4", "7d491176809c28848a561186d4857d14c"},
    TestItem{-1412, "-642", "7d491176809c28848a561186d4857cbce"},
    TestItem{-8166, "-2646", "7d491176809c28848a561186d4857abca"},
    TestItem{-90000001, "-7d7ba01", "7d491176809c28848a561186d40701820"},
    TestItem{-790301243, "-495ad4e8", "7d491176809c28848a561186cedebed28"},
    TestItem{-8999099123732, "-10914957a572", "7d491176809c28848a56107cbee002c9e"},
    TestItem{-542185444123566432, "-1388c279aea5038c", "7d491176809c2884891c745e388a2ce84"},
    TestItem{-9223372036854775807, "-160e2ad3246366807", "7d491176809c28847446d5a3b02216a19"},
    TestItem{-9223372036854775807 - 1, "-160e2ad3246366808", "7d491176809c28847446d5a3b02216a18"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<16>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "57", "57"},
    TestItem{194, "c2", "c2"},
    TestItem{6427, "191b", "191b"},
    TestItem{12345312, "bc5fe0", "bc5fe0"},
    TestItem{33312234991, "7c1905def", "7c1905def"},
    TestItem{901369000000, "d1ddc77040", "d1ddc77040"},
    TestItem{1579542123461234, "59c95cf2d3a72", "59c95cf2d3a72"},
    TestItem{123345345123123321, "1b635f018c30079", "1b635f018c30079"},
    TestItem{9223372036854775806, "7ffffffffffffffe", "7ffffffffffffffe"},
    TestItem{9223372036854775807, "7fffffffffffffff", "7fffffffffffffff"},
    TestItem{-1, "-1", "ffffffffffffffffffffffffffffffff"},
    TestItem{-409, "-199", "fffffffffffffffffffffffffffffe67"},
    TestItem{-1412, "-584", "fffffffffffffffffffffffffffffa7c"},
    TestItem{-8166, "-1fe6", "ffffffffffffffffffffffffffffe01a"},
    TestItem{-90000001, "-55d4a81", "fffffffffffffffffffffffffaa2b57f"},
    TestItem{-790301243, "-2f1b0a3b", "ffffffffffffffffffffffffd0e4f5c5"},
    TestItem{-8999099123732, "-82f441b4814", "fffffffffffffffffffff7d0bbe4b7ec"},
    TestItem{-542185444123566432, "-7863ac9ef5f6160", "fffffffffffffffff879c53610a09ea0"},
    TestItem{-9223372036854775807, "-7fffffffffffffff", "ffffffffffffffff8000000000000001"},
    TestItem{-9223372036854775807 - 1, "-8000000000000000", "ffffffffffffffff8000000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<17>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "52", "52"},
    TestItem{194, "b7", "b7"},
    TestItem{6427, "1541", "1541"},
    TestItem{12345312, "8bdd5e", "8bdd5e"},
    TestItem{33312234991, "4d31ba71d", "4d31ba71d"},
    TestItem{901369000000, "7a3age2576", "7a3age2576"},
    TestItem{1579542123461234, "2c18a11310db3", "2c18a11310db3"},
    TestItem{123345345123123321, "c7c06b4ggd0710", "c7c06b4ggd0710"},
    TestItem{9223372036854775806, "33d3d8307b214007", "33d3d8307b214007"},
    TestItem{9223372036854775807, "33d3d8307b214008", "33d3d8307b214008"},
    TestItem{-1, "-1", "279078gb8485d7b72e2ag3c08ed3g120"},
    TestItem{-409, "-171", "279078gb8485d7b72e2ag3c08ed3fgc0"},
    TestItem{-1412, "-4f1", "279078gb8485d7b72e2ag3c08ed3fd40"},
    TestItem{-8166, "-1b46", "279078gb8485d7b72e2ag3c08ed3e6ec"},
    TestItem{-90000001, "-3c69cbc", "279078gb8485d7b72e2ag3c08b0e6576"},
    TestItem{-790301243, "-1fca5367", "279078gb8485d7b72e2ag3c06g0aaecb"},
    TestItem{-8999099123732, "-47f0f704934", "279078gb8485d7b72e2afg427g63b8fe"},
    TestItem{-542185444123566432, "-33ca201d9f3d309", "279078gb8485d7b72aff61bfc4f02f19"},
    TestItem{-9223372036854775807, "-33d3d8307b214008", "279078gb8485d7b6ga672c9013b2c11a"},
    TestItem{-9223372036854775807 - 1, "-33d3d8307b214009", "279078gb8485d7b6ga672c9013b2c119"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<18>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "4f", "4f"},
    TestItem{194, "ae", "ae"},
    TestItem{6427, "11f1", "11f1"},
    TestItem{12345312, "69aeec", "69aeec"},
    TestItem{33312234991, "30779g81d", "30779g81d"},
    TestItem{901369000000, "49e55f507a", "49e55f507a"},
    TestItem{1579542123461234, "16a70baa7f0ee", "16a70baa7f0ee"},
    TestItem{123345345123123321, "5gb3hga6dbgh93", "5gb3hga6dbgh93"},
    TestItem{9223372036854775806, "16agh595df825fa6", "16agh595df825fa6"},
    TestItem{9223372036854775807, "16agh595df825fa7", "16agh595df825fa7"},
    TestItem{-1, "-1", "78a399ccdeb5bd6ha3184c0fh64da63"},
    TestItem{-409, "-14d", "78a399ccdeb5bd6ha3184c0fh64d919"},
    TestItem{-1412, "-468", "78a399ccdeb5bd6ha3184c0fh64d5he"},
    TestItem{-8166, "-173c", "78a399ccdeb5bd6ha3184c0fh64c32a"},
    TestItem{-90000001, "-2bb61e1", "78a399ccdeb5bd6ha3184c0fecb78a3"},
    TestItem{-790301243, "-1544736b", "78a399ccdeb5bd6ha3184c0ec2066hb"},
    TestItem{-8999099123732, "-296b24c5588", "78a399ccdeb5bd6ha31822c4f1a84fe"},
    TestItem{-542185444123566432, "-180c4e7115796d6", "78a399ccdeb5bd6h8d0dhfbeg0f43ag"},
    TestItem{-9223372036854775807, "-16agh595df825fa7", "78a399ccdeb5bd6g3a28h2d21g27cdf"},
    TestItem{-9223372036854775807 - 1, "-16agh595df825fa8", "78a399ccdeb5bd6g3a28h2d21g27cde"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<19>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "4b", "4b"},
    TestItem{194, "a4", "a4"},
    TestItem{6427, "hf5", "hf5"},
    TestItem{12345312, "4idga5", "4idga5"},
    TestItem{33312234991, "1i519eh4g", "1i519eh4g"},
    TestItem{901369000000, "2f176g3a04", "2f176g3a04"},
    TestItem{1579542123461234, "dabi3318i2ih", "dabi3318i2ih"},
    TestItem{123345345123123321, "2hdg1dbhh1aa9a", "2hdg1dbhh1aa9a"},
    TestItem{9223372036854775806, "ba643dci0ffeehg", "ba643dci0ffeehg"},
    TestItem{9223372036854775807, "ba643dci0ffeehh", "ba643dci0ffeehh"},
    TestItem{-1, "-1", "1910510fd19aig25hc6g5gebeb98h83"},
    TestItem{-409, "-12a", "1910510fd19aig25hc6g5gebeb98g5d"},
    TestItem{-1412, "-3h6", "1910510fd19aig25hc6g5gebeb98d9h"},
    TestItem{-8166, "-13bf", "1910510fd19aig25hc6g5gebeb97df8"},
    TestItem{-90000001, "-1h6b893", "1910510fd19aig25hc6g5gebcd2g8i1"},
    TestItem{-790301243, "-gf35125", "1910510fd19aig25hc6g5geagf63g5i"},
    TestItem{-8999099123732, "-18gga8ff55h", "1910510fd19aig25hc6g47ge42ccc26"},
    TestItem{-542185444123566432, "-cgi69542ei37a8", "1910510fd19aig25gi8gi797bfa59gf"},
    TestItem{-9223372036854775807, "-ba643dci0ffeehh", "1910510fd19aig25620c231cdecd296"},
    TestItem{-9223372036854775807 - 1, "-ba643dci0ffeehi", "1910510fd19aig25620c231cdecd295"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<20>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "47", "47"},
    TestItem{194, "9e", "9e"},
    TestItem{6427, "g17", "g17"},
    TestItem{12345312, "3h335c", "3h335c"},
    TestItem{33312234991, "160a1979b", "160a1979b"},
    TestItem{901369000000, "1f43hg5000", "1f43hg5000"},
    TestItem{1579542123461234, "7e50h5dbcd1e", "7e50h5dbcd1e"},
    TestItem{123345345123123321, "1a258hahaja861", "1a258hahaja861"},
    TestItem{9223372036854775806, "5cbfjia3fh26ja6", "5cbfjia3fh26ja6"},
    TestItem{9223372036854775807, "5cbfjia3fh26ja7", "5cbfjia3fh26ja7"},
    TestItem{-1, "-1", "66f609c19456i2h147iga2g17b68cf"},
    TestItem{-409, "-109", "66f609c19456i2h147iga2g17b67c7"},
    TestItem{-1412, "-3ac", "66f609c19456i2h147iga2g17b6524"},
    TestItem{-8166, "-1086", "66f609c19456i2h147iga2g17b584a"},
    TestItem{-90000001, "-182a001", "66f609c19456i2h147iga2fjj8g8cf"},
    TestItem{-790301243, "-c6j7d23", "66f609c19456i2h147iga2f90bifad"},
    TestItem{-8999099123732, "-hbaai9a96c", "66f609c19456i2h147ifcb5a91fj64"},
    TestItem{-542185444123566432, "-6c77fii48c5g1c", "66f609c19456i2h0hfb8e3hgij0cb4"},
    TestItem{-9223372036854775807, "-5cbfjia3fh26ja7", "66f609c19456i2gfbg2gbcc5a8j929"},
    TestItem{-9223372036854775807 - 1, "-5cbfjia3fh26ja8", "66f609c19456i2gfbg2gbcc5a8j928"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<21>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "43", "43"},
    TestItem{194, "95", "95"},
    TestItem{6427, "ec1", "ec1"},
    TestItem{12345312, "30a0j0", "30a0j0"},
    TestItem{33312234991, "ia8bi2h7", "ia8bi2h7"},
    TestItem{901369000000, "12h9cje0j7", "12h9cje0j7"},
    TestItem{1579542123461234, "4aedc0jfbfj2", "4aedc0jfbfj2"},
    TestItem{123345345123123321, "gg2i04agah7g9", "gg2i04agah7g9"},
    TestItem{9223372036854775806, "2heiciiie82dh96", "2heiciiie82dh96"},
    TestItem{9223372036854775807, "2heiciiie82dh97", "2heiciiie82dh97"},
    TestItem{-1, "-1", "1b71cc7c703ijd4g3k82ff6keb3c03"},
    TestItem{-409, "-ja", "1b71cc7c703ijd4g3k82ff6keb3b1f"},
    TestItem{-1412, "-345", "1b71cc7c703ijd4g3k82ff6keb38gk"},
    TestItem{-8166, "-iai", "1b71cc7c703ijd4g3k82ff6keb2ea7"},
    TestItem{-90000001, "-110g3d7", "1b71cc7c703ijd4g3k82ff6jda887i"},
    TestItem{-790301243, "-94ada6b", "1b71cc7c703ijd4g3k82ff6ba0b1ee"},
    TestItem{-8999099123732, "-b6ja0fkkkb", "1b71cc7c703ijd4g3k82488adg3c0e"},
    TestItem{-542185444123566432, "-3aei796k9cejb0", "1b71cc7c703ijd4g09e586004j9da4"},
    TestItem{-9223372036854775807, "-2heiciiie82dh97", "1b71cc7c703ijd4d75aahh9668afbi"},
    TestItem{-9223372036854775807 - 1, "-2heiciiie82dh98", "1b71cc7c703ijd4d75aahh9668afbh"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<22>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3l", "3l"},
    TestItem{194, "8i", "8i"},
    TestItem{6427, "d63", "d63"},
    TestItem{12345312, "28f8ic", "28f8ic"},
    TestItem{33312234991, "d7hi8jhd", "d7hi8jhd"},
    TestItem{901369000000, "g97lgfidi", "g97lgfidi"},
    TestItem{1579542123461234, "2fa7jlfigahk", "2fa7jlfigahk"},
    TestItem{123345345123123321, "9d20j89c48a9b", "9d20j89c48a9b"},
    TestItem{9223372036854775806, "1adaibb21dckfa6", "1adaibb21dckfa6"},
    TestItem{9223372036854775807, "1adaibb21dckfa7", "1adaibb21dckfa7"},
    TestItem{-1, "-1", "8h8b5gheh7legf73fb0had7bhd3dd"},
    TestItem{-409, "-id", "8h8b5gheh7legf73fb0had7bhd2h1"},
    TestItem{-1412, "-2k4", "8h8b5gheh7legf73fb0had7bhd0fa"},
    TestItem{-8166, "-gj4", "8h8b5gheh7legf73fb0had7bhc8ga"},
    TestItem{-90000001, "-ha4693", "8h8b5gheh7legf73fb0had6g78j4b"},
    TestItem{-790301243, "-6l7edhh", "8h8b5gheh7legf73fb0had0c9kbhj"},
    TestItem{-8999099123732, "-79lh493hfe", "8h8b5gheh7legf73fb0a0dc7897k0"},
    TestItem{-542185444123566432, "-1k3jef62fkgih2", "8h8b5gheh7legf71h732h74hii6ic"},
    TestItem{-9223372036854775807, "-1adaibb21dckfa7", "8h8b5gheh7legf5f2045lb5k4ea37"},
    TestItem{-9223372036854775807 - 1, "-1adaibb21dckfa8", "8h8b5gheh7legf5f2045lb5k4ea36"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<23>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3i", "3i"},
    TestItem{194, "8a", "8a"},
    TestItem{6427, "c3a", "c3a"},
    TestItem{12345312, "1l2f1g", "1l2f1g"},
    TestItem{33312234991, "9i0ejkj2", "9i0ejkj2"},
    TestItem{901369000000, "bbgjf1e2h", "bbgjf1e2h"},
    TestItem{1579542123461234, "1f2m2ibih57c", "1f2m2ibih57c"},
    TestItem{123345345123123321, "5eaa7i1fi89ee", "5eaa7i1fi89ee"},
    TestItem{9223372036854775806, "i6k448cf4192c1", "i6k448cf4192c1"},
    TestItem{9223372036854775807, "i6k448cf4192c2", "i6k448cf4192c2"},
    TestItem{-1, "-1", "2c59d9lld38jeh6fgh5m42j82lfdc"},
    TestItem{-409, "-hi", "2c59d9lld38jeh6fgh5m42j82leii"},
    TestItem{-1412, "-2f9", "2c59d9lld38jeh6fgh5m42j82lcl4"},
    TestItem{-8166, "-fa1", "2c59d9lld38jeh6fgh5m42j82l03c"},
    TestItem{-90000001, "-dme17c", "2c59d9lld38jeh6fgh5m42ih37e61"},
    TestItem{-790301243, "-57i2b4e", "2c59d9lld38jeh6fgh5m42e07j48m"},
    TestItem{-8999099123732, "-4ml0mdikag", "2c59d9lld38jeh6fgh5h44i8c2i2k"},
    TestItem{-542185444123566432, "-11h0k8c9le9lm1", "2c59d9lld38jeh6ef051id99bbgec"},
    TestItem{-9223372036854775807, "-i6k448cf4192c2", "2c59d9lld38jeh5k9k1hid441cd1b"},
    TestItem{-9223372036854775807 - 1, "-i6k448cf4192c3", "2c59d9lld38jeh5k9k1hid441cd1a"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<24>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3f", "3f"},
    TestItem{194, "82", "82"},
    TestItem{6427, "b3j", "b3j"},
    TestItem{12345312, "1d50k0", "1d50k0"},
    TestItem{33312234991, "767dj577", "767dj577"},
    TestItem{901369000000, "84cfnl82g", "84cfnl82g"},
    TestItem{1579542123461234, "10llfdjn1922", "10llfdjn1922"},
    TestItem{123345345123123321, "3919i2c702929", "3919i2c702929"},
    TestItem{9223372036854775806, "acd772jnc9l0l6", "acd772jnc9l0l6"},
    TestItem{9223372036854775807, "acd772jnc9l0l7", "acd772jnc9l0l7"},
    TestItem{-1, "-1", "iamei9lfd1i5k10n7fnfn25b3kaf"},
    TestItem{-409, "-h1", "iamei9lfd1i5k10n7fnfn25b3jhf"},
    TestItem{-1412, "-2ak", "iamei9lfd1i5k10n7fnfn25b3hnk"},
    TestItem{-8166, "-e46", "iamei9lfd1i5k10n7fnfn25b366a"},
    TestItem{-90000001, "-b76a01", "iamei9lfd1i5k10n7fnfn1i3laaf"},
    TestItem{-790301243, "-4360iib", "iamei9lfd1i5k10n7fnfmm2531g5"},
    TestItem{-8999099123732, "-39i27cj2gk", "iamei9lfd1i5k10n7fk64nlm8hhk"},
    TestItem{-542185444123566432, "-ek78i3kbak440", "iamei9lfd1i5k108b8elj5i07g6g"},
    TestItem{-9223372036854775807, "-acd772jnc9l0l7", "iamei9lfd1i5k0eai8gd32h16jd9"},
    TestItem{-9223372036854775807 - 1, "-acd772jnc9l0l8", "iamei9lfd1i5k0eai8gd32h16jd8"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<25>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "3c", "3c"},
    TestItem{194, "7j", "7j"},
    TestItem{6427, "a72", "a72"},
    TestItem{12345312, "16f2cc", "16f2cc"},
    TestItem{33312234991, "5bb480og", "5bb480og"},
    TestItem{901369000000, "5mh04g000", "5mh04g000"},
    TestItem{1579542123461234, "ge1h4db1co9", "ge1h4db1co9"},
    TestItem{123345345123123321, "21i9618ff4m7l", "21i9618ff4m7l"},
    TestItem{9223372036854775806, "64ie1focnn5g76", "64ie1focnn5g76"},
    TestItem{9223372036854775807, "64ie1focnn5g77", "64ie1focnn5g77"},
    TestItem{-1, "-1", "6365o71fgjb44dj83en26fd1fd85"},
    TestItem{-409, "-g9", "6365o71fgjb44dj83en26fd1fcgm"},
    TestItem{-1412, "-26c", "6365o71fgjb44dj83en26fd1fb1j"},
    TestItem{-8166, "-d1g", "6365o71fgjb44dj83en26fd1f06f"},
    TestItem{-90000001, "-95a001", "6365o71fgjb44dj83en26f3l5d85"},
    TestItem{-790301243, "-35n46oi", "6365o71fgjb44dj83en26c73b68d"},
    TestItem{-8999099123732, "-28oa7iimo7", "6365o71fgjb44dj83eki7557lf8o"},
    TestItem{-542185444123566432, "-92a5gd4368677", "6365o71fgjb44dio14haib9k770o"},
    TestItem{-9223372036854775807, "-64ie1focnn5g77", "6365o71fgjb44dd3a0lb72e39m0o"},
    TestItem{-9223372036854775807 - 1, "-64ie1focnn5g78", "6365o71fgjb44dd3a0lb72e39m0n"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<26>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "39", "39"},
    TestItem{194, "7c", "7c"},
    TestItem{6427, "9d5", "9d5"},
    TestItem{12345312, "110a7i", "110a7i"},
    TestItem{33312234991, "43lj3439", "43lj3439"},
    TestItem{901369000000, "485m0gehe", "485m0gehe"},
    TestItem{1579542123461234, "b4nmkd35ooa", "b4nmkd35ooa"},
    TestItem{123345345123123321, "17fjfm317bhp3", "17fjfm317bhp3"},
    TestItem{9223372036854775806, "3igoecjbmca686", "3igoecjbmca686"},
    TestItem{9223372036854775807, "3igoecjbmca687", "3igoecjbmca687"},
    TestItem{-1, "-1", "23745echihn4jcil7jec2kd69a1l"},
    TestItem{-409, "-fj", "23745echihn4jcil7jec2kd699c3"},
    TestItem{-1412, "-228", "23745echihn4jcil7jec2kd697pe"},
    TestItem{-8166, "-c22", "23745echihn4jcil7jec2kd68npk"},
    TestItem{-90000001, "-7eog2d", "23745echihn4jcil7jec2k5hajp9"},
    TestItem{-790301243, "-2edakhh", "23745echihn4jcil7jec2hoiofa5"},
    TestItem{-8999099123732, "-1h2b654e6g", "23745echihn4jcil7jcl09714ll6"},
    TestItem{-542185444123566432, "-5hij315fjc7gk", "23745echihn4jcifg0l91encn2b2"},
    TestItem{-9223372036854775807, "-3igoecjbmca687", "23745echihn4jcf2gkpp98gjp3jf"},
    TestItem{-9223372036854775807 - 1, "-3igoecjbmca688", "23745echihn4jcf2gkpp98gjp3je"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<27>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "36", "36"},
    TestItem{194, "75", "75"},
    TestItem{6427, "8m1", "8m1"},
    TestItem{12345312, "n65fl", "n65fl"},
    TestItem{33312234991, "34qfmng4", "34qfmng4"},
    TestItem{901369000000, "354fpld51", "354fpld51"},
    TestItem{1579542123461234, "7i3ik8b2l9n", "7i3ik8b2l9n"},
    TestItem{123345345123123321, "m524gmdifil3", "m524gmdifil3"},
    TestItem{9223372036854775806, "27c48l5b37oaoo", "27c48l5b37oaoo"},
    TestItem{9223372036854775807, "27c48l5b37oaop", "27c48l5b37oaop"},
    TestItem{-1, "-1", "kjbg2750m547p8n7d18cm6379g3"},
    TestItem{-409, "-f4", "kjbg2750m547p8n7d18cm637910"},
    TestItem{-1412, "-1p8", "kjbg2750m547p8n7d18cm6377hn"},
    TestItem{-8166, "-b5c", "kjbg2750m547p8n7d18cm636paj"},
    TestItem{-90000001, "-679cla", "kjbg2750m547p8n7d18clqmonll"},
    TestItem{-790301243, "-2122cdb", "kjbg2750m547p8n7d18ck514o2k"},
    TestItem{-8999099123732, "-14n86ihfbh", "kjbg2750m547p8n7d03gdqbgl4e"},
    TestItem{-542185444123566432, "-3ge9j96lk72qo", "kjbg2750m547p8jhpig3fba06g7"},
    TestItem{-9223372036854775807, "-27c48l5b37oaop", "kjbg2750m547p6fm8je7b2m9pi6"},
    TestItem{-9223372036854775807 - 1, "-27c48l5b37oaoq", "kjbg2750m547p6fm8je7b2m9pi5"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<28>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "33", "33"},
    TestItem{194, "6q", "6q"},
    TestItem{6427, "85f", "85f"},
    TestItem{12345312, "k2ag0", "k2ag0"},
    TestItem{33312234991, "2d3gfbi7", "2d3gfbi7"},
    TestItem{901369000000, "2amdgph40", "2amdgph40"},
    TestItem{1579542123461234, "598oc944bg2", "598oc944bg2"},
    TestItem{123345345123123321, "eoc1cii8nec9", "eoc1cii8nec9"},
    TestItem{9223372036854775806, "1bk39f3ah3dmq6", "1bk39f3ah3dmq6"},
    TestItem{9223372036854775807, "1bk39f3ah3dmq7", "1bk39f3ah3dmq7"},
    TestItem{-1, "-1", "81a71cgjeb6cjo7odb65d7icrl3"},
    TestItem{-409, "-eh", "81a71cgjeb6cjo7odb65d7icr6f"},
    TestItem{-1412, "-1mc", "81a71cgjeb6cjo7odb65d7icpqk"},
    TestItem{-8166, "-abi", "81a71cgjeb6cjo7odb65d7ich9e"},
    TestItem{-90000001, "-56bnpl", "81a71cgjeb6cjo7odb65d2c13nb"},
    TestItem{-790301243, "-1hpl98b", "81a71cgjeb6cjo7odb65bhkjicl"},
    TestItem{-8999099123732, "-nmqg6o7o4", "81a71cgjeb6cjo7odaaaejbgjp0"},
    TestItem{-542185444123566432, "-29adkl3hme4o0", "81a71cgjeb6cjo5f2pdc9hnqmp4"},
    TestItem{-9223372036854775807, "-1bk39f3ah3dmq7", "81a71cgjeb6cjmo4a1j22ier4mp"},
    TestItem{-9223372036854775807 - 1, "-1bk39f3ah3dmq8", "81a71cgjeb6cjmo4a1j22ier4mo"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<29>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "30", "30"},
    TestItem{194, "6k", "6k"},
    TestItem{6427, "7ii", "7ii"},
    TestItem{12345312, "hd59c", "hd59c"},
    TestItem{33312234991, "1r0308f9", "1r0308f9"},
    TestItem{901369000000, "1n7a97oq5", "1n7a97oq5"},
    TestItem{1579542123461234, "3lpf94jsra9", "3lpf94jsra9"},
    TestItem{123345345123123321, "a35b51ndo1dc", "a35b51ndo1dc"},
    TestItem{9223372036854775806, "q1se8f0m04isa", "q1se8f0m04isa"},
    TestItem{9223372036854775807, "q1se8f0m04isb", "q1se8f0m04isb"},
    TestItem{-1, "-1", "36l6q70ega3gd8s9ag8rce14rao"},
    TestItem{-409, "-e3", "36l6q70ega3gd8s9ag8rce14qpm"},
    TestItem{-1412, "-1jk", "36l6q70ega3gd8s9ag8rce14pk5"},
    TestItem{-8166, "-9kh", "36l6q70ega3gd8s9ag8rce14hj8"},
    TestItem{-90000001, "-4b75d9", "36l6q70ega3gd8s9ag8rc9iqlqg"},
    TestItem{-790301243, "-19fb030", "36l6q70ega3gd8s9ag8rb4emr7p"},
    TestItem{-8999099123732, "-hsk0o5499", "36l6q70ega3gd8s9afjrld5sn1g"},
    TestItem{-542185444123566432, "-1fcljlhpir0rl", "36l6q70ega3gd8qmqni5nhb6qc4"},
    TestItem{-9223372036854775807, "-q1se8f0m04isb", "36l6q70ega3gd827b20cbl108be"},
    TestItem{-9223372036854775807 - 1, "-q1se8f0m04isc", "36l6q70ega3gd827b20cbl108bd"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<30>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2r", "2r"},
    TestItem{194, "6e", "6e"},
    TestItem{6427, "747", "747"},
    TestItem{12345312, "f770c", "f770c"},
    TestItem{33312234991, "1fkq6ed1", "1fkq6ed1"},
    TestItem{901369000000, "1b6db713a", "1b6db713a"},
    TestItem{1579542123461234, "2k7e4de85oe", "2k7e4de85oe"},
    TestItem{123345345123123321, "6sqhne8fl70l", "6sqhne8fl70l"},
    TestItem{9223372036854775806, "hajppbc1fc206", "hajppbc1fc206"},
    TestItem{9223372036854775807, "hajppbc1fc207", "hajppbc1fc207"},
    TestItem{-1, "-1", "1a4p5qh8koob2e2gknbn3jbm88f"},
    TestItem{-409, "-dj", "1a4p5qh8koob2e2gknbn3jbm7or"},
    TestItem{-1412, "-1h2", "1a4p5qh8koob2e2gknbn3jbm6le"},
    TestItem{-8166, "-926", "1a4p5qh8koob2e2gknbn3jblt6a"},
    TestItem{-90000001, "-3l3a01", "1a4p5qh8koob2e2gknbn3fkis8f"},
    TestItem{-790301243, "-12fkcen", "1a4p5qh8koob2e2gknbn2gq1pnn"},
    TestItem{-8999099123732, "-dled8rgb2", "1a4p5qh8koob2e2gkms1j62olre"},
    TestItem{-542185444123566432, "-10i5q6qebgr4c", "1a4p5qh8koob2e1g2hfg7505b44"},
    TestItem{-9223372036854775807, "-hajppbc1fc207", "1a4p5qh8koob2df60rgblhqa689"},
    TestItem{-9223372036854775807 - 1, "-hajppbc1fc208", "1a4p5qh8koob2df60rgblhqa688"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<31>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2p", "2p"},
    TestItem{194, "68", "68"},
    TestItem{6427, "6la", "6la"},
    TestItem{12345312, "dbc9r", "dbc9r"},
    TestItem{33312234991, "16ghrt9g", "16ghrt9g"},
    TestItem{901369000000, "11nj9gmdq", "11nj9gmdq"},
    TestItem{1579542123461234, "1smuhc3si6t", "1smuhc3si6t"},
    TestItem{123345345123123321, "4qf58q3slqli", "4qf58q3slqli"},
    TestItem{9223372036854775806, "bm03i95hia436", "bm03i95hia436"},
    TestItem{9223372036854775807, "bm03i95hia437", "bm03i95hia437"},
    TestItem{-1, "-1", "hleoq9ui363gg16lmp8srn1je7"},
    TestItem{-409, "-d6", "hleoq9ui363gg16lmp8srn1j12"},
    TestItem{-1412, "-1eh", "hleoq9ui363gg16lmp8srn1hum"},
    TestItem{-8166, "-8fd", "hleoq9ui363gg16lmp8srn1atq"},
    TestItem{-90000001, "-34e1dq", "hleoq9ui363gg16lmp8soiii0d"},
    TestItem{-790301243, "-rin5pf", "hleoq9ui363gg16lmp8s049djo"},
    TestItem{-8999099123732, "-ah2od6ikc", "hleoq9ui363gg16lmemq39q0or"},
    TestItem{-542185444123566432, "-lafh1skf1sme", "hleoq9ui363gg0gb787077ulmp"},
    TestItem{-9223372036854775807, "-bm03i95hia437", "hleoq9ui363gfkflj6una4mfb1"},
    TestItem{-9223372036854775807 - 1, "-bm03i95hia438", "hleoq9ui363gfkflj6una4mfb0"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<32>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2n", "2n"},
    TestItem{194, "62", "62"},
    TestItem{6427, "68r", "68r"},
    TestItem{12345312, "bonv0", "bonv0"},
    TestItem{33312234991, "v0p0nff", "v0p0nff"},
    TestItem{901369000000, "q7eses20", "q7eses20"},
    TestItem{1579542123461234, "1csin7iqeji", "1csin7iqeji"},
    TestItem{123345345123123321, "3dhlu0cc603p", "3dhlu0cc603p"},
    TestItem{9223372036854775806, "7vvvvvvvvvvvu", "7vvvvvvvvvvvu"},
    TestItem{9223372036854775807, "7vvvvvvvvvvvv", "7vvvvvvvvvvvv"},
    TestItem{-1, "-1", "7vvvvvvvvvvvvvvvvvvvvvvvvv"},
    TestItem{-409, "-cp", "7vvvvvvvvvvvvvvvvvvvvvvvj7"},
    TestItem{-1412, "-1c4", "7vvvvvvvvvvvvvvvvvvvvvvujs"},
    TestItem{-8166, "-7v6", "7vvvvvvvvvvvvvvvvvvvvvvo0q"},
    TestItem{-90000001, "-2lqik1", "7vvvvvvvvvvvvvvvvvvvta5dbv"},
    TestItem{-790301243, "-nhm2hr", "7vvvvvvvvvvvvvvvvvvv8e9te5"},
    TestItem{-8999099123732, "-85t21mi0k", "7vvvvvvvvvvvvvvvvnq2tu9dvc"},
    TestItem{-542185444123566432, "-f1hqp7nluob0", "7vvvvvvvvvvvvvgue56o8a17l0"},
    TestItem{-9223372036854775807, "-7vvvvvvvvvvvv", "7vvvvvvvvvvvvo000000000001"},
    TestItem{-9223372036854775807 - 1, "-8000000000000", "7vvvvvvvvvvvvo000000000000"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<33>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2l", "2l"},
    TestItem{194, "5t", "5t"},
    TestItem{6427, "5tp", "5tp"},
    TestItem{12345312, "adhcc", "adhcc"},
    TestItem{33312234991, "pq6p1fd", "pq6p1fd"},
    TestItem{901369000000, "l4v274k7", "l4v274k7"},
    TestItem{1579542123461234, "1113dfbj4qk", "1113dfbj4qk"},
    TestItem{123345345123123321, "2ehlblec90w0", "2ehlblec90w0"},
    TestItem{9223372036854775806, "5hg4ck9jd4u36", "5hg4ck9jd4u36"},
    TestItem{9223372036854775807, "5hg4ck9jd4u37", "5hg4ck9jd4u37"},
    TestItem{-1, "-1", "3nakotlgi17b10fs1825j5cf1o"},
    TestItem{-409, "-cd", "3nakotlgi17b10fs1825j5cemc"},
    TestItem{-1412, "-19q", "3nakotlgi17b10fs1825j5cdow"},
    TestItem{-8166, "-7gf", "3nakotlgi17b10fs1825j5c7ia"},
    TestItem{-90000001, "-29tckp", "3nakotlgi17b10fs1825gsg2e0"},
    TestItem{-790301243, "-k6d9qh", "3nakotlgi17b10fs1824vvw588"},
    TestItem{-8999099123732, "-6d53roowe", "3nakotlgi17b10fs11m0fakn2b"},
    TestItem{-542185444123566432, "-ao04k44bw9io", "3nakotlgi17b105413f1eqd5g1"},
    TestItem{-9223372036854775807, "-5hg4ck9jd4u37", "3nakotlgi17b0rvbtseswp7hvi"},
    TestItem{-9223372036854775807 - 1, "-5hg4ck9jd4u38", "3nakotlgi17b0rvbtseswp7hvh"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<34>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2j", "2j"},
    TestItem{194, "5o", "5o"},
    TestItem{6427, "5j1", "5j1"},
    TestItem{12345312, "983be", "983be"},
    TestItem{33312234991, "lj61a9d", "lj61a9d"},
    TestItem{901369000000, "h5gfpqt6", "h5gfpqt6"},
    TestItem{1579542123461234, "q0h4r6ex5k", "q0h4r6ex5k"},
    TestItem{123345345123123321, "1ppgamjm260h", "1ppgamjm260h"},
    TestItem{9223372036854775806, "3tdtk1v8j6tpo", "3tdtk1v8j6tpo"},
    TestItem{9223372036854775807, "3tdtk1v8j6tpp", "3tdtk1v8j6tpp"},
    TestItem{-1, "-1", "1ppi6nphe0ckbctn7qwp6qrh9h"},
    TestItem{-409, "-c1", "1ppi6nphe0ckbctn7qwp6qrgvh"},
    TestItem{-1412, "-17i", "1ppi6nphe0ckbctn7qwp6qrg20"},
    TestItem{-8166, "-726", "1ppi6nphe0ckbctn7qwp6qra7c"},
    TestItem{-90000001, "-1xbsmt", "1ppi6nphe0ckbctn7qwp4rfmkn"},
    TestItem{-790301243, "-hdddk7", "1ppi6nphe0ckbctn7qwonde3nb"},
    TestItem{-8999099123732, "-51bdh0ja4", "1ppi6nphe0ckbctn7lvdr9qvxe"},
    TestItem{-542185444123566432, "-7oloakblghpq", "1ppi6nphe0ckbclwk2m4t5axhq"},
    TestItem{-9223372036854775807, "-3tdtk1v8j6tpp", "1ppi6nphe0ckb909c6urw7klhr"},
    TestItem{-9223372036854775807 - 1, "-3tdtk1v8j6tpq", "1ppi6nphe0ckb909c6urw7klhq"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<35>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2h", "2h"},
    TestItem{194, "5j", "5j"},
    TestItem{6427, "58m", "58m"},
    TestItem{12345312, "87ws7", "87ws7"},
    TestItem{33312234991, "i48vq7l", "i48vq7l"},
    TestItem{901369000000, "e0brdnk0", "e0brdnk0"},
    TestItem{1579542123461234, "k1f6psc609", "k1f6psc609"},
    TestItem{123345345123123321, "19oyhfllc98g", "19oyhfllc98g"},
    TestItem{9223372036854775806, "2pijmikexrxp6", "2pijmikexrxp6"},
    TestItem{9223372036854775807, "2pijmikexrxp7", "2pijmikexrxp7"},
    TestItem{-1, "-1", "try5wbbiprfp7r727m0oyq2wa"},
    TestItem{-409, "-bo", "try5wbbiprfp7r727m0oyq2km"},
    TestItem{-1412, "-15c", "try5wbbiprfp7r727m0oyq1qy"},
    TestItem{-8166, "-6nb", "try5wbbiprfp7r727m0oypv90"},
    TestItem{-90000001, "-1oy4dl", "try5wbbiprfp7r727m0n9qxip"},
    TestItem{-790301243, "-f1mnui", "try5wbbiprfp7r727m09x3e1s"},
    TestItem{-8999099123732, "-3yuf0oqaw", "try5wbbiprfp7r723m59y1ble"},
    TestItem{-542185444123566432, "-5lj5lu0x1f57", "try5wbbiprfp7lki205o1omr4"},
    TestItem{-9223372036854775807, "-2pijmikexrxp7", "try5wbbiprfp51nhk3fa0x474"},
    TestItem{-9223372036854775807 - 1, "-2pijmikexrxp8", "try5wbbiprfp51nhk3fa0x473"},
  }};
}

template <>
__host__ __device__ constexpr cuda::std::array<TestItem, 22> get_test_items<36>()
{
  return {{
    TestItem{0, "0", "0"},
    TestItem{1, "1", "1"},
    TestItem{87, "2f", "2f"},
    TestItem{194, "5e", "5e"},
    TestItem{6427, "4yj", "4yj"},
    TestItem{12345312, "7clpc", "7clpc"},
    TestItem{33312234991, "fax8t0v", "fax8t0v"},
    TestItem{901369000000, "bi2ztvls", "bi2ztvls"},
    TestItem{1579542123461234, "fjwfmtqmpe", "fjwfmtqmpe"},
    TestItem{123345345123123321, "xqi9xajvddl", "xqi9xajvddl"},
    TestItem{9223372036854775806, "1y2p0ij32e8e6", "1y2p0ij32e8e6"},
    TestItem{9223372036854775807, "1y2p0ij32e8e7", "1y2p0ij32e8e7"},
    TestItem{-1, "-1", "f5lxx1zz5pnorynqglhzmsp33"},
    TestItem{-409, "-bd", "f5lxx1zz5pnorynqglhzmsorr"},
    TestItem{-1412, "-138", "f5lxx1zz5pnorynqglhzmsnzw"},
    TestItem{-8166, "-6au", "f5lxx1zz5pnorynqglhzmsisa"},
    TestItem{-90000001, "-1hl0g1", "f5lxx1zz5pnorynqglhy57on3"},
    TestItem{-790301243, "-d2iwcb", "f5lxx1zz5pnorynqglhmk9sqt"},
    TestItem{-8999099123732, "-36u4n9nv8", "f5lxx1zz5pnorynqdenuzj17w"},
    TestItem{-542185444123566432, "-44akp6hluf6o", "f5lxx1zz5pnorujfvwbi0y9wg"},
    TestItem{-9223372036854775807, "-1y2p0ij32e8e7", "f5lxx1zz5pnoq0l1g2ywkegox"},
    TestItem{-9223372036854775807 - 1, "-1y2p0ij32e8e8", "f5lxx1zz5pnoq0l1g2ywkegow"},
  }};
}

template <class T, int Base>
__host__ __device__ constexpr void test_to_chars(const TestItem& item)
{
  constexpr cuda::std::size_t buff_size = 150;

  static_assert(cuda::std::is_same_v<
                cuda::std::to_chars_result,
                decltype(cuda::std::to_chars(cuda::std::declval<char*>(), cuda::std::declval<char*>(), T{}, int{}))>);
  static_assert(noexcept(cuda::std::to_chars(cuda::std::declval<char*>(), cuda::std::declval<char*>(), T{}, int{})));

  char buff[buff_size + 1]{};
  char* buff_start = buff + 1;

  const auto value    = static_cast<T>(item.val);
  const char* ref_str = (cuda::std::is_signed_v<T>) ? item.str_signed : item.str_unsigned;
  const auto ref_len  = cuda::std::strlen(ref_str);

  // Check valid buffer size
  {
    const auto result = cuda::std::to_chars(buff_start, buff_start + buff_size, value, Base);
    assert(result.ec == cuda::std::errc{});
    assert(result.ptr == buff_start + ref_len);

    // Compare with reference string
    assert(cuda::std::strncmp(buff_start, ref_str, buff_size) == 0);

    // Check that the operation did not underflow the buffer
    assert(buff[0] == '\0');
  }

  // Check too small buffer
  {
    const auto result = cuda::std::to_chars(buff_start, buff_start + ref_len - 1, value, Base);
    assert(result.ec == cuda::std::errc::value_too_large);
    assert(result.ptr == buff_start + ref_len - 1);
  }

  // Check zero buffer
  {
    const auto result = cuda::std::to_chars(buff_start, buff_start, value, Base);
    assert(result.ec == cuda::std::errc::value_too_large);
    assert(result.ptr == buff_start);
  }
}

template <int Base>
__host__ __device__ constexpr void test_base()
{
  constexpr auto items = get_test_items<Base>();

  for (const auto& item : items)
  {
    test_to_chars<__int128_t, Base>(item);
    test_to_chars<__uint128_t, Base>(item);
  }
}

template <int... Base>
__host__ __device__ constexpr void test_helper(cuda::std::integer_sequence<int, Base...>)
{
  (test_base<Base + first_base>(), ...);
}

__host__ __device__ constexpr bool test()
{
  test_helper(cuda::std::make_integer_sequence<int, last_base - first_base + 1>{});

  return true;
}

#endif // _CCCL_HAS_INT128()

int main(int, char**)
{
#if _CCCL_HAS_INT128()
  test();
  static_assert(test());
#endif // _CCCL_HAS_INT128()
  return 0;
}
