//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES.
//
//===----------------------------------------------------------------------===//

// <cuda/std/chrono>

// time_point

// template <cuda/std/class Clock, class Duration1, class Rep2, class Period2>
//   time_point<cuda/std/Clock, typename common_type<Duration1, duration<Rep2, Period2>>::type>
//   operator-(const time_point<cuda/std/Clock, Duration1>& lhs, const duration<Rep2, Period2>& rhs);

#include <cuda/std/cassert>
#include <cuda/std/chrono>
#include <cuda/std/cstdint>

#include "test_macros.h"

template <class D>
__host__ __device__ void test2739() // LWG2739
{
  using TimePoint = cuda::std::chrono::time_point<cuda::std::chrono::system_clock>;
  using Dur       = cuda::std::chrono::duration<D>;
  const Dur d(5);
  TimePoint t0 = cuda::std::chrono::system_clock::from_time_t(200);
  TimePoint t1 = t0 - d;
  assert(t1 < t0);
}

__host__ __device__ constexpr bool test()
{
  using Clock     = cuda::std::chrono::system_clock;
  using Duration1 = cuda::std::chrono::milliseconds;
  using Duration2 = cuda::std::chrono::microseconds;

  cuda::std::chrono::time_point<Clock, Duration1> t1(Duration1(3));
  cuda::std::chrono::time_point<Clock, Duration2> t2 = t1 - Duration2(5);
  assert(t2.time_since_epoch() == Duration2(2995));

  return true;
}

int main(int, char**)
{
  test();
  static_assert(test());

  test2739<cuda::std::int32_t>();
  test2739<cuda::std::uint32_t>();

  return 0;
}
