import './HelpModal.scss';

import variables from '_variables.module.scss';
import TooltipIcon from 'components/common/TooltipIcon';
import React, { FunctionComponent, useEffect, useState } from 'react';
import { Button, Col, Container, Modal, ModalDialog, Row } from 'react-bootstrap';
import Draggable from 'react-draggable';
import styled from 'styled-components';

import HelpModalContentContainer from '../containers/HelpModalContentContainer';

interface ModalProps {
  /** Function to be executed when the cancel button is clicked  */
  handleCancel: () => void;
  /** Function to be executed when the submit button is clicked */
  handleSubmit: () => void;
  /** External state var to track whether or not this modal is being shown. */
  show: boolean;
}

/**
 * This wrapper component allows this dialog to be dragged by clicking and dragging the modal header.
 */
const DraggableModalDialog = (props: any) => {
  return (
    <Draggable handle=".modal-title">
      <ModalDialog {...props} />
    </Draggable>
  );
};

const DraggableTitle = styled(Modal.Title)`
  width: 100%;
  cursor: pointer;
  color: white;
`;

const ModalHeader = styled(Modal.Header)`
  background-color: ${variables.primaryColor};
  padding: 5px;
`;

const StyledTooltip = styled(TooltipIcon)`
  margin-left: 5px;
  margin-bottom: 5px;
`;

/**
 * Help Component used to display an interface with help text and the ability to create a mailto ticket.
 * @param props customize the component with custom text, and an operation to take when the component is closed.
 */
const HelpModal: FunctionComponent<ModalProps> = ({ handleCancel, handleSubmit, show }) => {
  const [mailto, setMailto] = useState<string | undefined>(undefined);
  const [updatingMailto, setUpdatingMailto] = useState<boolean>(false);
  /**
   * Chrome has added some protection such that mailto links generated by JS cannot be fired without direct user input.
   * In addition, Chrome caches mailto link values so updating the href after initial load will not be ignored unless the component
   * with the href is removed from the dom completely. The following two useEffects handle this by removing the Button with the href from the DOM entirely whenever the href changes.
   */
  useEffect(() => {
    setUpdatingMailto(true);
  }, [mailto]);
  useEffect(() => {
    setUpdatingMailto(false);
  }, [updatingMailto]);
  return (
    <Container>
      <Modal
        dialogAs={DraggableModalDialog}
        show={show}
        onHide={handleCancel}
        dialogClassName="help-modal"
      >
        <ModalHeader closeButton closeVariant="white">
          <DraggableTitle>
            <Row style={{ height: '40px' }}>
              <Col md="auto">
                <p>Help Desk</p>
              </Col>
              <Col md="auto">
                <StyledTooltip
                  toolTipId="help-toolTip"
                  toolTip="Click and drag to move this popup"
                />
              </Col>
            </Row>
          </DraggableTitle>
        </ModalHeader>

        <Modal.Body>
          <HelpModalContentContainer setMailto={setMailto} />
        </Modal.Body>

        <Modal.Footer>
          {updatingMailto ? (
            <Button>Submit</Button>
          ) : (
            <Button
              href={mailto}
              onClick={() => {
                handleSubmit();
              }}
            >
              Submit
            </Button>
          )}
          <Button variant="secondary" onClick={handleCancel}>
            Cancel
          </Button>
        </Modal.Footer>
      </Modal>
    </Container>
  );
};

export default HelpModal;
