/*
Copyright 2020 The Rook Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

// Package cluster to manage a Ceph cluster.
package cluster

import (
	"context"

	"github.com/pkg/errors"
	"github.com/rook/rook/pkg/clusterd"
	"github.com/rook/rook/pkg/operator/k8sutil"
	v1 "k8s.io/api/core/v1"
	kerrors "k8s.io/apimachinery/pkg/api/errors"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// populateConfigOverrideConfigMap creates the "rook-config-override" config map
// Its content allows modifying Ceph configuration flags
func populateConfigOverrideConfigMap(clusterdContext *clusterd.Context, namespace string, ownerInfo *k8sutil.OwnerInfo, clusterMetadata metav1.ObjectMeta) error {
	ctx := context.TODO()

	existingCM, err := clusterdContext.Clientset.CoreV1().ConfigMaps(namespace).Get(ctx, k8sutil.ConfigOverrideName, metav1.GetOptions{})
	if err != nil {
		if !kerrors.IsNotFound(err) {
			logger.Warningf("failed to get cm %q to check labels and annotations", k8sutil.ConfigOverrideName)
			return nil
		}

		labels := map[string]string{}
		annotations := map[string]string{}
		initRequiredMetadata(clusterMetadata, labels, annotations)

		// Create the configmap since it doesn't exist yet
		placeholderConfig := map[string]string{
			k8sutil.ConfigOverrideVal: "",
		}
		cm := &v1.ConfigMap{
			ObjectMeta: metav1.ObjectMeta{
				Name:        k8sutil.ConfigOverrideName,
				Namespace:   namespace,
				Labels:      labels,
				Annotations: annotations,
			},
			Data: placeholderConfig,
		}

		err := ownerInfo.SetControllerReference(cm)
		if err != nil {
			return errors.Wrapf(err, "failed to set owner reference to override configmap %q", cm.Name)
		}
		_, err = clusterdContext.Clientset.CoreV1().ConfigMaps(namespace).Create(ctx, cm, metav1.CreateOptions{})
		if err != nil {
			return errors.Wrapf(err, "failed to create override configmap %s", namespace)
		}
		logger.Infof("created placeholder configmap for ceph overrides %q", cm.Name)
		return nil
	}

	// Ensure the annotations and labels are initialized
	if existingCM.Annotations == nil {
		existingCM.Annotations = map[string]string{}
	}
	if existingCM.Labels == nil {
		existingCM.Labels = map[string]string{}
	}

	// Add recommended labels and annotations to the existing configmap if it doesn't have any yet
	updateRequired := initRequiredMetadata(clusterMetadata, existingCM.Labels, existingCM.Annotations)
	if updateRequired {
		_, err = clusterdContext.Clientset.CoreV1().ConfigMaps(namespace).Update(ctx, existingCM, metav1.UpdateOptions{})
		if err != nil {
			logger.Warningf("failed to add recommended labels and annotations to configmap %q. %v", existingCM.Name, err)
		} else {
			logger.Infof("added expected labels and annotations to configmap %q", existingCM.Name)
		}
	}
	return nil
}

func initRequiredMetadata(metadata metav1.ObjectMeta, labels, annotations map[string]string) bool {
	// Add the helm labels and annotations in case the user wants to install the cluster helm chart to start managing it
	releaseNameAttr := "meta.helm.sh/release-name"
	chartName, ok := metadata.Annotations[releaseNameAttr]
	if !ok {
		logger.Debug("cluster helm chart is not configured, not adding helm annotations to configmap")
		return false
	}
	if _, ok := annotations[releaseNameAttr]; ok {
		logger.Debug("cluster helm chart helm annotations already added to configmap")
		return false
	}

	logger.Infof("adding helm chart name %q annotation to configmap", chartName)
	labels["app.kubernetes.io/managed-by"] = "Helm"
	annotations[releaseNameAttr] = chartName
	annotations["meta.helm.sh/release-namespace"] = metadata.Namespace
	return true
}
