/*
 * Copyright 2012-2013 the original author or authors
 * @license MIT, see LICENSE.txt for details
 *
 * @author Scott Andrews
 */

(function (define, global, document) {
	'use strict';

	define(function (require) {

		var when, UrlBuilder;

		when = require('when');
		UrlBuilder = require('../UrlBuilder');

		// consider abstracting this into a util module
		function clearProperty(scope, propertyName) {
			try {
				delete scope[propertyName];
			}
			catch (e) {
				// IE doesn't like to delete properties on the window object
				if (propertyName in scope) {
					scope[propertyName] = undefined;
				}
			}
		}

		function cleanupScriptNode(response) {
			if (response.raw && response.raw.parentNode) {
				response.raw.parentNode.removeChild(response.raw);
			}
		}

		function registerCallback(prefix, resolver, response) {
			var name;

			do {
				name = prefix + Math.floor(new Date().getTime() * Math.random());
			}
			while (name in global);

			global[name] = function jsonpCallback(data) {
				response.entity = data;
				clearProperty(global, name);
				cleanupScriptNode(response);
				if (!response.request.canceled) {
					resolver.resolve(response);
				}
			};

			return name;
		}

		/**
		 * Executes the request as JSONP.
		 *
		 * @param {string} request.path the URL to load
		 * @param {Object} [request.params] parameters to bind to the path
		 * @param {string} [request.callback.param='callback'] the parameter name for which the callback function name is the value
		 * @param {string} [request.callback.prefix='jsonp'] prefix for the callback function, as the callback is attached to the window object, a unique, unobtrusive prefix is desired
		 *
		 * @returns {Promise<Response>}
		 */
		function jsonp(request) {
			var d, callbackName, callbackParams, script, firstScript, response;

			response = {
				request: request
			};

			if (request.canceled) {
				response.error = 'precanceled';
				return when.reject(response);
			}

			d = when.defer();
			request.callback = request.callback || {};
			callbackName = registerCallback(request.callback.prefix || 'jsonp', d.resolver, response);
			callbackParams = {};
			callbackParams[request.callback.param || 'callback'] = callbackName;

			request.canceled = false;
			request.cancel = function cancel() {
				request.canceled = true;
				cleanupScriptNode(response);
				d.reject(response);
			};

			script = document.createElement('script');
			script.type = 'text/javascript';
			script.async = true;
			script.src = new UrlBuilder(request.path, request.params).build(callbackParams);

			script.onerror = function () {
				if (global[callbackName]) {
					response.error = 'loaderror';
					clearProperty(global, callbackName);
					cleanupScriptNode(response);
					d.reject(response);
				}
			};
			script.onload = script.onreadystatechange = function (e) {
				// script tag load callbacks are completely non-standard
				if ((e && (e.type === 'load' || e.type === 'error')) || script.readyState === 'loaded') {
					script.onerror(e);
				}
			};

			response.raw = script;
			firstScript = document.getElementsByTagName('script')[0];
			firstScript.parentNode.insertBefore(script, firstScript);

			return d.promise;
		}

		jsonp.chain = function (interceptor, config) {
			return interceptor(jsonp, config);
		};

		return jsonp;

	});

}(
	typeof define === 'function' && define.amd ? define : function (factory) { module.exports = factory(require); },
	this,
	this.document
	// Boilerplate for AMD and Node
));
