/*
 * Copyright 2012-2013 the original author or authors
 * @license MIT, see LICENSE.txt for details
 *
 * @author Scott Andrews
 */

(function (buster, define) {
	'use strict';

	var assert, refute, fail;

	assert = buster.assertions.assert;
	refute = buster.assertions.refute;
	fail = buster.assertions.fail;

	define('rest/interceptor/hateoas-test', function (require) {

		var hateoas, rest, when, supports;

		hateoas = require('rest/interceptor/hateoas');
		rest = require('rest');
		when = require('when');

		supports = {
			'Object.defineProperty': (function () {
				try {
					var obj = {};
					Object.defineProperty(obj, 'test', { enumerable: false, configurable: true, value: true });
					return obj.test;
				}
				catch (e) {
					return false;
				}
			}()),
			'ES5 getters': (function () {
				try {
					var obj = {};
					Object.defineProperty(obj, 'test', { get: function () { return true; } });
					return obj.test;
				}
				catch (e) {
					return false;
				}
			}())
		};

		buster.testCase('rest/interceptor/hateoas', {
			requiresSupportFor: { 'Object.defineProperty': supports['Object.defineProperty'] },

			'should parse links in the entity': function () {
				var client, body, parent, self;

				parent = { rel: 'parent', href: '/' };
				self = { rel: 'self', href: '/resource' };

				body = { links: [ parent, self ]};
				client = hateoas(function () { return { entity: body }; });

				return client().then(function (response) {
					assert.same(parent, response.entity._links.parentLink);
					assert.same(self, response.entity._links.selfLink);
				}).otherwise(fail);
			},
			'should parse links in the entity into the entity': function () {
				var client, body, parent, self;

				parent = { rel: 'parent', href: '/' };
				self = { rel: 'self', href: '/resource' };

				body = { links: [ parent, self ]};
				client = hateoas(function () { return { entity: body }; }, { target: '' });

				return client().then(function (response) {
					assert.same(parent, response.entity.parentLink);
					assert.same(self, response.entity.selfLink);
				}).otherwise(fail);
			},
			'should create a client for the related resource': function () {
				var client, body, parent, self;

				parent = { rel: 'parent', href: '/' };
				self = { rel: 'self', href: '/resource' };

				body = { links: [ parent, self ]};
				client = hateoas(function () { return { entity: body }; });

				return client().then(function (response) {
					var parentClient = response.entity._links.clientFor('parent', function (request) { return { request: request }; });
					return parentClient().then(function (response) {
						assert.same(parent.href, response.request.path);
					});
				}).otherwise(fail);
			},
			'should fetch a related resource': {
				requiresSupportFor: { 'ES5 getters': supports['ES5 getters'] },
				'': function () {
					var client, parentClient;

					parentClient = function (request) {
						return request.path === '/' ?
							{ request: request, entity: { links: [ { rel: 'self', href: '/' }, { rel: 'child', href: '/resource' } ] } } :
							{ request: request, entity: { links: [ { rel: 'self', href: '/resource' }, { rel: 'parent', href: '/' } ] } };
					};
					client = hateoas(parentClient);

					return client({ path: '/' }).then(function (response) {
						assert.same('/', response.request.path);
						assert.same('/', response.entity._links.selfLink.href);
						return response.entity._links.child.then(function (response) {
							assert.same('/resource', response.request.path);
							assert.same('/resource', response.entity._links.selfLink.href);
						});
					}).otherwise(fail);
				}
			},
			'should have the default client as the parent by default': function () {
				assert.same(rest, hateoas().skip());
			},
			'should support interceptor chaining': function () {
				assert(typeof hateoas().chain === 'function');
			}
		});

	});

}(
	this.buster || require('buster'),
	typeof define === 'function' && define.amd ? define : function (id, factory) {
		var packageName = id.split(/[\/\-]/)[0], pathToRoot = id.replace(/[^\/]+/g, '..');
		pathToRoot = pathToRoot.length > 2 ? pathToRoot.substr(3) : pathToRoot;
		factory(function (moduleId) {
			return require(moduleId.indexOf(packageName) === 0 ? pathToRoot + moduleId.substr(packageName.length) : moduleId);
		});
	}
	// Boilerplate for AMD and Node
));
